/*
    Falling Block Game
    Copyright (C) 1999-2002 Jared Krinke <http://derajdezine.vze.com/>


    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    This application is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this distribution; if not, write to:
    Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307  USA

    Jared Krinke

    Deraj DeZine
    http://derajdezine.vze.com/
*/

#include "fbg.h"
#include <time.h>
#include <math.h>
#include <stdlib.h>
#include <string>
#include <SDL/SDL.h>
using namespace std;

fbgGame::fbgGame(const char* argv0, const string& newTheme, bool newLight, int startLevel, int newbtype, const string& newPrefix, const char* musicFile) {
	renderer.setGame(this);
	light = newLight;
	dropBlockID = NULL;
	removeLineGainID = NULL;
	lightLineGainRedrawID = NULL;
	if (newTheme != "") theme = newTheme;

	// Setup next and current blocks
	srand(time(NULL));
	setupBlocks();
	nextBlockIndex = rand()%blockSet.size();
	cycleBlocks();

	downHeld = false;leftHeld = false;rightHeld = false;
	wantsToMove = false;
	lineGain = NULL;
	prefix = newPrefix;

	downStart = -1;lines = 0;level = startLevel;score = 0;
	curBlockAdjust = 40.0f;
	setState(GAMEPLAY);

	btype = newbtype;
	if (btype >= 0) lines = 25;
	for (int row=0; row < 18; row++) {
		for (int col=0; col < 10; col++)  {
			if (btype >= 0 && row >= 18-btype*2) matrix[row][col] = rand()%2 ? 0 : rand()%(blockSet.size());
			else matrix[row][col] = 0;
		}
	}

	// Init PhysicsFS
	PHYSFS_init(argv0);
	PHYSFS_addToSearchPath((prefix+PHYSFS_getDirSeparator()+"data"+PHYSFS_getDirSeparator()).c_str(), 1);
#ifdef WIN32
	PHYSFS_addToSearchPath((string(PHYSFS_getBaseDir())+"data\\").c_str(), 1);
	PHYSFS_setWriteDir((string(PHYSFS_getUserDir())+"fbg\\data\\").c_str());
	PHYSFS_addToSearchPath((string(PHYSFS_getUserDir())+"fbg\\data\\").c_str(), 1);
#else
	PHYSFS_addToSearchPath((prefix+"/data/").c_str(), 1);
	PHYSFS_setWriteDir((string(PHYSFS_getUserDir())+".fbg/data/").c_str());
	PHYSFS_addToSearchPath((string(PHYSFS_getUserDir())+".fbg/data/").c_str(), 1);
	PHYSFS_addToSearchPath(FBGDATADIR "/data/", 1);
	PHYSFS_addToSearchPath((string(PHYSFS_getBaseDir())+"data/").c_str(), 1);
	PHYSFS_addToSearchPath("/usr/local/games/fbg/data/", 1);
	PHYSFS_addToSearchPath("/usr/games/fbg/data/", 1);
#endif

	// Load in .pk3 files
	char** list = PHYSFS_enumerateFiles("");
	for (int i=0; list[i] != NULL; i++) {
		int end;
		for (end=0; list[i][end] != '\0'; end++);
		if (list[i][end-1] == '3' && list[i][end-2] == 'k' && list[i][end-3] == 'p' && list[i][end-4] == '.') {
			PHYSFS_addToSearchPath((string(PHYSFS_getRealDir(list[i]))+list[i]).c_str(), 1);
		}
	}
	PHYSFS_freeList(list);

	// Music
#ifdef ENABLE_SOUND
	initSound();
	music = NULL;
	musicID = NULL;
	if (musicFile != NULL && PHYSFS_exists((string("music/")+musicFile).c_str())) initMusic(string("music/")+musicFile);
#endif
}

void fbgGame::cycleBlocks() {
	curBlock = blockSet[nextBlockIndex];
	nextBlockIndex = rand()%blockSet.size();
}

void fbgGame::setupBlocks() {
	// FIXME ugly code
	{bool tmp[16]={	false, true , true , false,
			false, true , true , false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 0, tmp);
	blockSet.push_back(tmp2);}
	{bool tmp[16]={	true , true , true , true ,
			false, false, false, false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 1, tmp);
	blockSet.push_back(tmp2);}
	{bool tmp[16]={	true , true , true , false,
			true , false, false, false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 2, tmp);
	blockSet.push_back(tmp2);}
	{bool tmp[16]={	true , true , true , false,
			false, false, true , false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 3, tmp);
	blockSet.push_back(tmp2);}
	{bool tmp[16]={	false, true , true , false,
			true , true , false, false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 4, tmp);
	blockSet.push_back(tmp2);}
	{bool tmp[16]={	true , true , false, false,
			false, true , true , false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 5, tmp);
	blockSet.push_back(tmp2);}
	{bool tmp[16]={	true , true , true , false,
			false, true , false, false,
			false, false, false, false,
			false, false, false, false};
	fbgBlock tmp2(this, 6, tmp);
	blockSet.push_back(tmp2);}
}

void fbgGame::putBlockInMatrix(const fbgBlock& theBlock) {
	for (int row=0; row < 4; row++) {
		for (int col=0; col < 4; col++) {
			if (theBlock.getMatrix(row, col)) matrix[theBlock.getPosY()+row][theBlock.getPosX()+col] = theBlock.getIndex()+1;
		}
	}
}
void fbgGame::finishCurBlock() {
	// Scoring for individual block
	if (downStart >= 1) {
		score += curBlock.getPosY()-downStart;
	}
	downStart = -1;

	// Cycle current, next blocks
	putBlockInMatrix(curBlock);
	cycleBlocks();
	downHeld = false;	// So next block doesn't drop by accident
	// Check for full lines
	lineGain = new fbgLineGain();

	for (int row=17; row >= 0; row--) {
		int col;
		for (col=0; col < 10; col++) {
			if (matrix[row][col] <= 0) break;
		}
		if (col == 10) {
			lineGain->lineCount++;
			lineGain->linePos[lineGain->lineCount-1] = row;
			for (int c=0; c < 10; c++) lineGain->lineMatrix[lineGain->lineCount-1][c] = matrix[row][c];
		}
	}

	// Remove lines from matrix
	int gap = 0; // Space between where we're copying from/to
	for (int row=17; row >= 0; row--) {
		if (gap > 0) {
			for (int col=0; col < 10; col++) {
				if (row+gap <= 17) matrix[row+gap][col] = matrix[row][col];
			}
		}
		if (row == lineGain->linePos[0] || row == lineGain->linePos[1] || row == lineGain->linePos[2] || row == lineGain->linePos[3]) gap++;
	}
	// Clear the lines we never got to due to the gap
	for (int row=gap-1; row >= 0; row--) {
		for (int col=0; col < 10; col++) matrix[row][col] = 0;
	}

	// Adjust Score, Line count
	if (btype >= 0) {
		lines -= lineGain->lineCount;
		if (lines < 0) lines = 0;
	}
	else lines += lineGain->lineCount;
	if (lineGain->lineCount == 1) score += 40*level;
	else if (lineGain->lineCount == 2) score += 100*level;
	else if (lineGain->lineCount == 3) score += 300*level;
	else if (lineGain->lineCount == 4) score += 1200*level;
	if (btype < 0 && level < lines/10 && lines > 0 && level < 20) lineGain->levelGained = ++level;

	if (lineGain->lineCount <= 0) {
		delete lineGain;
		lineGain = NULL;
	}
	else {
		lastLine = SDL_GetTicks();
		setPaused(true);
		SDL_CreateThread(&fbgGame::removeLineGain, NULL);
		if (light) SDL_CreateThread(&fbgGame::lightLineGainRedraw, NULL);
	}

	// Check for Game Over
	if (!curBlock.checkBlockPosition()) {
		putBlockInMatrix(curBlock);
		setState(GAMEOVER);
		fbgGame::issue(EVT_REDRAW);
	}
}

void fbgGame::verifyCurBlockAdjust() {
	fbgBlock tmp = curBlock;
	if (!tmp.moveBlockUp()) curBlockAdjust = 0.0f;
}

void fbgGame::setNextDrop(const Uint32& newDrop) {
	nextDrop = newDrop;
	resetDropTimer();
}
void fbgGame::setLastDrop(const Uint32& newDrop) {
	lastDrop = newDrop;
	setNextDrop(lastDrop + fbgLevelDelays[getDownHeld() ? 20 : getLevel()]);
}

// Game Logic
Uint32 fbgGame::dropBlock(Uint32 interval, void* param) {
	fbgGame::issue(EVT_DROP);
	fbgGame::issue(EVT_REDRAW);

	return 0;
}
int fbgGame::removeLineGain(void* params) {
	SDL_Delay(LINESPEED);
	fbgGame::issue(EVT_REMOVELINEGAIN);

	return 0;
}
int fbgGame::lightLineGainRedraw(void* param) {
	Uint32 entrance = SDL_GetTicks();
	while (SDL_GetTicks() < entrance + LINESPEED) {
		fbgGame::issue(EVT_REDRAW);
		SDL_Delay(LIGHTLINEGAINREDRAWSPEED);
	}

	return 0;
}

void fbgGame::resetDropTimer() {
	if (dropBlockID != NULL) SDL_RemoveTimer(dropBlockID);
	if (!getPaused()) {
		Sint32 interval = Sint32(getNextDrop())-Sint32(SDL_GetTicks());
		dropBlockID = SDL_AddTimer(Uint32(interval > 0 ? interval : 1), &fbgGame::dropBlock, (void*)(this));
	}
}

fbgGame::~fbgGame() {
	if (dropBlockID) SDL_RemoveTimer(dropBlockID);
	if (removeLineGainID) SDL_WaitThread(removeLineGainID, NULL);
	if (lightLineGainRedrawID) SDL_WaitThread(lightLineGainRedrawID, NULL);

#ifdef ENABLE_SOUND
	stopMusic();
	stopSound();
#endif

	PHYSFS_deinit();
	renderer.exit();
}

inline void fbgGame::issue(const int& eventCode) {
	SDL_Event event;
	event.type = SDL_USEREVENT;
	event.user.code = eventCode;
	SDL_PushEvent(&event);
}

void fbgGame::processEvent(const SDL_Event& event) {
	switch (getState()) {
	case BEGINWAIT:
		switch (event.type) {
		case SDL_KEYUP:
			switch (event.key.keysym.sym) {
			case SDLK_ESCAPE:
				setState(QUIT);
			break;
			default:
				setState(GAMEPLAY);
				setLastDrop(SDL_GetTicks());
				renderer.draw();
			break;
			}
		break;
		case SDL_VIDEOEXPOSE:
			renderer.draw();
		break;
		case SDL_QUIT:
			setState(QUIT);
		break;
		}
	break;
	case GAMEPLAY:
		switch (event.type) {
		case SDL_USEREVENT:
			switch (event.user.code) {
			case EVT_REDRAW:
				if (light) renderer.draw();	// Draw Scene (only necessary with the light renderer)
			break;
			case EVT_DROP:
				if (!getPaused()) {
					if (!getCurBlock()->moveBlockDown()) finishCurBlock();
					setCurBlockAdjust(40.0f);
					verifyCurBlockAdjust();
					setLastDrop(getNextDrop());
					// If the user was holding down an arrow trying to move under another block, move it now
					if (getWantsToMove()) {
						if (getRightHeld()) getCurBlock()->moveBlockRight();
						else if (getLeftHeld()) getCurBlock()->moveBlockLeft();
						verifyCurBlockAdjust();
					}
				}
			break;
			case EVT_REMOVELINEGAIN:
				delete lineGain;
				lineGain = NULL;
				if (btype >= 0 && lines == 0) {
					setState(GAMEOVER);
				}
				else {
					setLastDrop(SDL_GetTicks());
					setPaused(false);
				}
				fbgGame::issue(EVT_REDRAW);
//				if (light) SDL_RemoveTimer(lightLineGainRedrawID);
			break;
			}
		break;
		case SDL_VIDEOEXPOSE:
			renderer.draw();
		break;
		case SDL_KEYUP:
			switch (event.key.keysym.sym) {
				case SDLK_DOWN:
					if (!getPaused()) {
						if (getDownHeld()) {
							setCurBlockAdjust(getSmoothAdjust());
							setNextDrop(getLastDrop()+fbgLevelDelays[getLevel()]);
						}
						setDownHeld(false);
					}
				break;
				case SDLK_LEFT:
					if (!getPaused()) {
						setLeftHeld(false);
						setWantsToMove(false);
					}
				break;
				case SDLK_RIGHT:
					if (!getPaused()) {
						setRightHeld(false);
						setWantsToMove(false);
					}
				break;
			}
		break;
		case SDL_KEYDOWN:
			switch (event.key.keysym.sym) {
				case SDLK_ESCAPE:
					setState(QUIT);
				break;
				case SDLK_DOWN:
					if (!getPaused()) {
						if (!getDownHeld()) {
							setCurBlockAdjust(getSmoothAdjust());
							if (Sint32(getNextDrop()) - Sint32(SDL_GetTicks()) > fbgLevelDelays[20]) setNextDrop(SDL_GetTicks()+fbgLevelDelays[20]);
						}
						setDownHeld(true);
					}
				break;
				case SDLK_RIGHT:
					if (!getPaused()) {
						setRightHeld(true);
						if (!getCurBlock()->moveBlockRight()) setWantsToMove(true);
						else {
							verifyCurBlockAdjust();
							fbgGame::issue(EVT_REDRAW);
						}
					}
				break;
				case SDLK_LEFT:
					if (!getPaused()) {
						setLeftHeld(true);
						if (!getCurBlock()->moveBlockLeft()) setWantsToMove(true);
						else {
							verifyCurBlockAdjust();
							fbgGame::issue(EVT_REDRAW);
						}
					}
				break;
				case SDLK_COMMA:
					if (!getPaused()) {
						getCurBlock()->rotateBlockLeft();
						verifyCurBlockAdjust();
						fbgGame::issue(EVT_REDRAW);
					}
				break;
				case SDLK_PERIOD:
					if (!getPaused()) {
						getCurBlock()->rotateBlockRight();
						verifyCurBlockAdjust();
						fbgGame::issue(EVT_REDRAW);
					}
				break;
			}
		break;
		case SDL_QUIT:
			setState(QUIT);
		break;
		}
	break;
	default:
		switch (event.type) {
		case SDL_USEREVENT:
			switch (event.user.code) {
			case EVT_REDRAW:
				if (light) renderer.draw();	// Draw Scene (only necessary with the light renderer)
			break;
			}
		break;
		case SDL_KEYUP:
			switch (event.key.keysym.sym) {
			case SDLK_ESCAPE:
				setState(QUIT);
			break;
			}
		break;
		case SDL_VIDEOEXPOSE:
			renderer.draw();
		break;
		case SDL_QUIT:
			setState(QUIT);
		break;
		}
	break;
	}
}
void fbgGame::gameplayLoop() {
	if (getState() == BEGINWAIT) {
		SDL_Event event;
		while (getState() == BEGINWAIT) {
			SDL_WaitEvent(&event);
			setLastDrop(SDL_GetTicks());
			processEvent(event);
		}
	}
	// Main loop
	if (light) {
		SDL_Event event;
		while (getState() != QUIT) {
			SDL_WaitEvent(&event);
			processEvent(event);
		}
	}
	else {
		SDL_Event event;
		while (getState() != QUIT) {
			renderer.draw();
			// Event Handling
			while (SDL_PollEvent(&event)) {
				processEvent(event);
			}
		}
	}
}

float fbgGame::getSmoothAdjust() {
		float ret = getCurBlockAdjust()-((float(SDL_GetTicks())-getLastDrop())/(getNextDrop()-getLastDrop()))*getCurBlockAdjust();
		if (ret > 40.0f) ret = 40.0f;
		else if (ret < 0.0f) ret = 0.0f;
		return ret;
}

PHYSFS_file* fbgGame::loadThemeFile(const string& file) {
	if (theme != "" && PHYSFS_exists((string("themes/")+theme+"/"+file).c_str())) return PHYSFS_openRead((string("themes/")+theme+"/"+file).c_str());
	else return PHYSFS_openRead((string("themes/default/")+file).c_str());
}
