
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: meta_info.c 2833 2007-09-24 15:38:46Z mschwerin $
 *
 */
#include "config.h"

#include "xine.h"

#include "codeset.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "meta_info.h"
#include "oxine.h"
#include "utils.h"

extern oxine_t *oxine;

static xine_stream_t *stream = NULL;


char *
meta_info_get_from_stream (meta_info_t type, xine_stream_t * stream)
{
    char *meta = NULL;
    const char *xmeta = xine_get_meta_info (stream, type);

    if (xmeta) {
        /* Strip whitespace from the beginning of the string. */
        while ((xmeta[0] == ' ')
               || (xmeta[0] == '\t')
               || (xmeta[0] == '\n')
               || (xmeta[0] == '\r')) {
            xmeta = (char *) (xmeta + 1);
        }

        /* Convert '_' to ' '. */
        for (meta = (char *) xmeta; meta[0] != '\0'; meta++) {
            if (meta[0] == '_') {
                meta[0] = ' ';
            }
        }

        /* Convert the string to our local codeset. */
        char *codeset = get_system_encoding ();
        recode_t *xr = recode_init ("utf8", codeset);
        ho_free (codeset);
        meta = recode (xr, xmeta);
        recode_done (xr);
    }

    return meta;
}


char *
meta_info_get (meta_info_t type, const char *mrl)
{
    if (!mrl) {
        return NULL;
    }

    char *meta = NULL;
    if (xine_open (stream, mrl) == 1) {
        meta = meta_info_get_from_stream (type, stream);
        xine_close (stream);
    }
    else {
        odk_log_stream_error (stream, __FILE__, __LINE__,
                              _("Could not open '%s': %s!"), mrl);
    }

    return meta;
}


int
meta_info_get_playback_length (const char *mrl)
{
    int playback_length = 0;

    if (starts_with (mrl, "v4l:")
        || starts_with (mrl, "dvb:")
        || starts_with (mrl, "vdr:")) {
        /* Nothing to do, we return a playback length of zero. */
    }
#ifdef HAVE_IMAGE_PLAYBACK
    else if (is_file_image (mrl)) {
        if (config_get_bool ("image.slideshow.enable")) {
            playback_length = config_get_number ("image.slideshow.delay");
        }
    }
#endif
    else if (xine_open (stream, mrl) == 1) {
        int pos;
        int time;
        int length;
        if (xine_get_pos_length (stream, &pos, &time, &length) == 1) {
            playback_length = length / 1000;
        }
        xine_close (stream);
    }
    else {
        odk_log_stream_error (stream, __FILE__, __LINE__,
                              _("Could not open '%s': %s!"), mrl);
    }

    return playback_length;
}


void
meta_info_init (xine_t * xine)
{
    assert (xine);

    stream = xine_stream_new (oxine->xine, NULL, NULL);
    if (!stream) {
        error (_("Could not create stream!"));
    }
}


void
meta_info_free (void)
{
    if (stream) {
        xine_dispose (stream);
        stream = NULL;
    }
}


#ifdef HAVE_LIBEXIF

#include <libexif/exif-data.h>


char *
exif_info_get (exif_info_t type, const char *mrl)
{
    int i = 0;
    int j = 0;
    char *exif = NULL;
    char *filename = mrl_get_filename (mrl);

    if (!is_file_image (filename)) {
        return exif;
    }

    ExifData *edata = exif_data_new_from_file (filename);
    if (!edata) {
        error (_("Could not find EXIF information in '%s'!"), mrl);
        goto out_free;
    }

    switch (type) {
    case EXIF_INFO_SORT:
    case EXIF_INFO_TIME:
    case EXIF_INFO_DATE:
        for (i = 0; i < EXIF_IFD_COUNT; i++) {
            ExifContent *content = edata->ifd[i];

            if (!content || !content->count)
                continue;

            for (j = 0; j < content->count; j++) {
                ExifEntry *e = content->entries[j];
                struct tm tm = { 0, };
                char *data;

                if (!content->entries[j])
                    continue;

                if ((e->tag != EXIF_TAG_DATE_TIME) &&
                    (e->tag != EXIF_TAG_DATE_TIME_ORIGINAL) &&
                    (e->tag != EXIF_TAG_DATE_TIME_DIGITIZED))
                    continue;

                if (e->data == NULL)
                    continue;

                if (strlen ((char *) e->data) < 10)
                    continue;

                data = ho_strdup ((char *) e->data);

                data[4] = data[7] = data[10] = '\0';

                tm.tm_year = atoi (data) - 1900;
                tm.tm_mon = atoi (data + 5) - 1;
                tm.tm_mday = atoi (data + 8);
                tm.tm_hour = 0;
                tm.tm_min = 0;
                tm.tm_sec = 0;
                tm.tm_isdst = -1;

                if (strlen ((char *) e->data) > 10) {
                    data[13] = data[16] = '\0';
                    tm.tm_hour = atoi (data + 11);
                    tm.tm_min = atoi (data + 14);
                    tm.tm_sec = atoi (data + 17);
                }
                ho_free (data);

                time_t time = mktime (&tm);
                struct tm *brokentime = localtime (&time);

                const char *format = NULL;
                if (type == EXIF_INFO_SORT) {
                    format = "%Y-%m-%d %T";
                }
                else if (type == EXIF_INFO_TIME) {
                    format = config_get_string ("gui.time_format");
                }
                else {
                    format = config_get_string ("gui.date_format");
                }

                char time_str[128];
                strftime (time_str, 128, format, brokentime);
                exif = ho_strdup (time_str);

                goto out_free;
            }
        }
        break;
    default:
        break;
    }

  out_free:
    exif_data_unref (edata);
    ho_free (filename);

    return exif;
}


#endif /* HAVE_LIBEXIF */
