
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_checkbox.c 2072 2007-05-18 16:21:34Z mschwerin $
 */
#include "config.h"

#include <assert.h>

#include "heap.h"
#include "i18n.h"
#include "odk.h"
#include "otk.h"

typedef struct {
    otk_widget_t widget;

    bool is_checked;

    /// Callback that is called whenever the checkbox is clicked.
    otk_cb_t cb;

    /// User-defined data for the above callback.
    void *cb_data;
} otk_checkbox_t;


static void
checkbox_destroy (otk_widget_t * this)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_CHECKBOX))
        return;

    ho_free (checkbox);
}


static void
checkbox_draw (otk_widget_t * this)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_CHECKBOX))
        return;

    int palette = 0;
    if (this->is_focused) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_FOCUSED);
    }
    else {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
    }
    int background_color = palette + OSD_TEXT_PALETTE_BACKGROUND;
    int foreground_color = palette + OSD_TEXT_PALETTE_FOREGROUND;

    if (this->is_focused) {
        odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                       background_color, true);
    }
    {
        odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                       foreground_color, false);
    }
    if (checkbox->is_checked) {
        odk_draw_vector (this->odk, this->x, this->y, this->w, this->h,
                         OSD_VECTOR_CHECK, foreground_color);
    }

    this->need_repaint = false;
}


static void
checkbox_event_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_CHECKBOX))
        return;

    if ((ev->source.key != OXINE_KEY_ACTIVATE)
        && (ev->source.button != OXINE_MOUSE_BUTTON_LEFT))
        return;

    checkbox->is_checked = checkbox->is_checked ? false : true;
    checkbox->widget.need_repaint = true;

    if (checkbox->cb) {
        checkbox->cb (checkbox->cb_data);
    }

    ev->source.key = OXINE_KEY_NULL;
    ev->source.button = OXINE_MOUSE_BUTTON_NULL;
}


otk_widget_t *
otk_checkbox_new (otk_t * otk, int x, int y, int w, bool is_checked,
                  otk_cb_t cb, void *cb_data)
{
    otk_checkbox_t *checkbox = ho_new (otk_checkbox_t);

    otk_widget_constructor ((otk_widget_t *) checkbox, otk,
                            OTK_WIDGET_CHECKBOX, x, y, w, w);

    checkbox->widget.draw = checkbox_draw;
    checkbox->widget.destroy = checkbox_destroy;

    checkbox->widget.key_handler = checkbox_event_handler;
    checkbox->widget.button_handler = checkbox_event_handler;

    checkbox->is_checked = is_checked;

    checkbox->cb = cb;
    checkbox->cb_data = cb_data;

    return (otk_widget_t *) checkbox;
}
