
/*
 * Copyright (C) 2005 Miguel Freitas
 * Copyright (C) 2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_scrollbar.c 2468 2007-07-09 09:57:39Z mschwerin $
 */
#include "config.h"

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk.h"
#include "otk.h"


typedef struct {
    otk_widget_t widget;

    double position;
    double length;

    otk_widget_t *button_up;
    otk_widget_t *button_down;
    otk_widget_t *button_pgup;
    otk_widget_t *button_pgdown;

    otk_int_set_cb_t cb_click;

    void *cb_data;
} otk_scrollbar_t;


static void
scrollbar_destroy (otk_widget_t * this)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SCROLLBAR))
        return;

    ho_free (scrollbar);
}


static void
scrollbar_draw (otk_widget_t * this)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SCROLLBAR))
        return;

    int palette;
    if (this->is_focused) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_FOCUSED);
    }
    else {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
    }
    int background_color = palette + OSD_TEXT_PALETTE_BACKGROUND;
    int foreground_color = palette + OSD_TEXT_PALETTE_FOREGROUND;

    odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                   background_color, true);

    int x1 = this->x + 5;
    int y1 = this->y + 5 + (scrollbar->position * (this->h - 10)) / 100;
    int w1 = this->w - 10;
    int h1 = (scrollbar->length * (this->h - 10)) / 100;
    if (h1 < 10) {
        h1 = 10;
    }
    if ((y1 + h1) > (this->y + this->h - 5)) {
        y1 = this->y + this->h - 5 - h1;
    }
    odk_draw_rect (this->odk, x1, y1, w1, h1, 5, foreground_color, true);

    this->need_repaint = false;
}


static void
scrollbar_button_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SCROLLBAR))
        return;
    if (!scrollbar->cb_click)
        return;

    int position;
    oxine_button_id_t saved_button = ev->source.button;
    ev->source.button = OXINE_MOUSE_BUTTON_NULL;

    switch (saved_button) {
    case OXINE_MOUSE_BUTTON_LEFT:
        position = (ev->data.mouse.pos.y - this->y) * 100 / this->h;
        scrollbar->cb_click (scrollbar->cb_data, position);
        break;
    case OXINE_MOUSE_SCROLLWHEEL_UP:
        position = scrollbar->position - (this->h / 100);
        scrollbar->cb_click (scrollbar->cb_data, position);
        break;
    case OXINE_MOUSE_SCROLLWHEEL_DOWN:
        position = scrollbar->position + (this->h / 100);
        scrollbar->cb_click (scrollbar->cb_data, position);
        break;
    default:
        /* If we did not use this event we restore the event. */
        ev->source.button = saved_button;
        break;
    }
}


otk_widget_t *
otk_scrollbar_new (otk_t * otk, int x, int y, int w, int h,
                   otk_int_set_cb_t cb_click,
                   otk_cb_t cb_up, otk_cb_t cb_down,
                   otk_cb_t cb_pgup, otk_cb_t cb_pgdown, void *cb_data)
{
    otk_scrollbar_t *scrollbar = ho_new (otk_scrollbar_t);

    int bw = w;
    int bh = (w / 2) + 10;
    int bs = bh + 2;

    int ys = y + 1 * bs;
    int hs = h - 4 * bs;

    otk_widget_constructor ((otk_widget_t *) scrollbar, otk,
                            OTK_WIDGET_SCROLLBAR, x, ys, w, hs);

    scrollbar->widget.selectable = OTK_SELECTABLE_MOUSE;

    scrollbar->widget.draw = scrollbar_draw;
    scrollbar->widget.destroy = scrollbar_destroy;

    scrollbar->widget.button_handler = scrollbar_button_handler;

    scrollbar->cb_click = cb_click;
    scrollbar->cb_data = cb_data;
    scrollbar->position = 0;
    scrollbar->length = 100;

    int x1 = x;
    int y1 = y;
    int y2 = y1 + bs + hs + 2;
    int y3 = y2 + bs;
    int y4 = y3 + bs;
    int vw = w - 10;
    int vh = vw / 2;

    scrollbar->button_up =
        otk_vector_button_new (otk, x1, y1, bw, bh,
                               OSD_VECTOR_ARROW_UP_SIMPLE,
                               vw, vh, cb_up, cb_data);

    scrollbar->button_down =
        otk_vector_button_new (otk, x1, y2, bw, bh,
                               OSD_VECTOR_ARROW_DOWN_SIMPLE,
                               vw, vh, cb_down, cb_data);

    scrollbar->button_pgup =
        otk_vector_button_new (otk, x1, y3, bw, bh,
                               OSD_VECTOR_ARROW_UP_DOUBLE,
                               vw, vh, cb_pgup, cb_data);

    scrollbar->button_pgdown =
        otk_vector_button_new (otk, x1, y4, bw, bh,
                               OSD_VECTOR_ARROW_DOWN_DOUBLE,
                               vw, vh, cb_pgdown, cb_data);

    return (otk_widget_t *) scrollbar;
}


void
otk_scrollbar_set (otk_widget_t * this, double position, double length)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SCROLLBAR))
        return;

    scrollbar->position = position;
    if (scrollbar->position < 0.0) {
        scrollbar->position = 0.0;
    }
    if (scrollbar->position > 100.0) {
        scrollbar->position = 100.0;
    }

    scrollbar->length = length;
    if (scrollbar->length < 0.0) {
        scrollbar->length = 0.0;
    }
    if (scrollbar->length > 100.0) {
        scrollbar->length = 100.0;
    }

    bool e1 = !(scrollbar->position == 0.0);
    otk_widget_set_enabled (scrollbar->button_up, e1);
    otk_widget_set_enabled (scrollbar->button_pgup, e1);

    bool e2 = !(((scrollbar->position + scrollbar->length) == 100.0)
                || (scrollbar->length == 100.0));
    otk_widget_set_enabled (scrollbar->button_down, e2);
    otk_widget_set_enabled (scrollbar->button_pgdown, e2);

    this->need_repaint = true;
}
