// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package secrets

import (
	"bytes"
	"context"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/mock"

	"github.com/go-chi/chi"
	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
)

func TestHandleUpdate(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	secrets := mock.NewMockGlobalSecretStore(controller)
	secrets.EXPECT().FindName(gomock.Any(), dummySecret.Namespace, dummySecret.Name).Return(dummySecret, nil)
	secrets.EXPECT().Update(gomock.Any(), gomock.Any()).Return(nil)

	c := new(chi.Context)
	c.URLParams.Add("namespace", "octocat")
	c.URLParams.Add("name", "github_password")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(dummySecret)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(secrets).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusOK; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(core.Secret), dummySecretScrubbed
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestHandleUpdate_ValidationError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	secrets := mock.NewMockGlobalSecretStore(controller)
	secrets.EXPECT().FindName(gomock.Any(), dummySecret.Namespace, dummySecret.Name).Return(&core.Secret{Name: "github_password"}, nil)

	c := new(chi.Context)
	c.URLParams.Add("namespace", "octocat")
	c.URLParams.Add("name", "github_password")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(&core.Secret{Data: ""})

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(secrets).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusBadRequest; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), &errors.Error{Message: "Invalid Secret Value"}
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestHandleUpdate_BadRequest(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	c := new(chi.Context)
	c.URLParams.Add("namespace", "octocat")
	c.URLParams.Add("name", "github_password")

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", nil)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(nil).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusBadRequest; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), &errors.Error{Message: "EOF"}
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestHandleUpdate_SecretNotFound(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	secrets := mock.NewMockGlobalSecretStore(controller)
	secrets.EXPECT().FindName(gomock.Any(), dummySecret.Namespace, dummySecret.Name).Return(nil, errors.ErrNotFound)

	c := new(chi.Context)
	c.URLParams.Add("namespace", "octocat")
	c.URLParams.Add("name", "github_password")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(&core.Secret{})

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(secrets).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusNotFound; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestHandleUpdate_UpdateError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	secrets := mock.NewMockGlobalSecretStore(controller)
	secrets.EXPECT().FindName(gomock.Any(), dummySecret.Namespace, dummySecret.Name).Return(&core.Secret{Name: "github_password"}, nil)
	secrets.EXPECT().Update(gomock.Any(), gomock.Any()).Return(errors.ErrNotFound)

	c := new(chi.Context)
	c.URLParams.Add("namespace", "octocat")
	c.URLParams.Add("name", "github_password")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(&core.Secret{Data: "password"})

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(secrets).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusInternalServerError; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}
