/***************************************************************************
 *                                                                         *
 *   gtrm.cpp       (begin: Jun 25 2004)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh}@cs.uni-duesseldorf.de                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "gtrm.h"
#include "opturtree.h"
#include "brent.h"
#include "model.h"

/* compute 1 PAM rate matrix, its eigensystem, and the inverse matrix thereof */

//=========================================================================
//=========================================================================
void GTRM::createRateParameters () {
	/*
	double sumGenPam_ = tsAG_ + tsCT_ + tvAC_ + tvAT_ + tvCG_ + tvGT_;
	tsAG_ /=  sumGenPam_;
	tsCT_ /=  sumGenPam_;
	tvAC_ /=  sumGenPam_;
	tvAT_ /=  sumGenPam_;
	tvCG_ /=  sumGenPam_;
	tvGT_ /=  sumGenPam_;
	*/
	//tvGT_ = 0.1;
	unitRateMat_[BASE_A][BASE_C] = tvAC_;
	unitRateMat_[BASE_C][BASE_A] = tvAC_;

	unitRateMat_[BASE_A][BASE_G] = tsAG_;
	unitRateMat_[BASE_G][BASE_A] = tsAG_;

	unitRateMat_[BASE_A][BASE_T] = tvAT_;
	unitRateMat_[BASE_T][BASE_A] = tvAT_;

	unitRateMat_[BASE_C][BASE_G] = tvCG_;
	unitRateMat_[BASE_G][BASE_C] = tvCG_;

	unitRateMat_[BASE_C][BASE_T] = tsCT_;
	unitRateMat_[BASE_T][BASE_C] = tsCT_;

	unitRateMat_[BASE_G][BASE_T] = tvGT_;
	unitRateMat_[BASE_T][BASE_G] = tvGT_;

	unitRateMat_[BASE_A][BASE_A] = 0.0;
	unitRateMat_[BASE_C][BASE_C] = 0.0;
	unitRateMat_[BASE_G][BASE_G] = 0.0;
	unitRateMat_[BASE_T][BASE_T] = 0.0;
}

//=========================================================================
//=========================================================================
void GTRM::init () {
	createRateParameters ();
	tranprobmat(nState_);
} /* tranprobmat */

void GTRM::reInit () {
	createRateParameters ();
	tranprobmat(nState_);
}

//----------------------------------------------------------------------------------------------------------------------------------------
double GTRM::cmpNegLogLi (double value) {
	if (isActPam_ == TS_AG)
		tsAG_ = value;
	if (isActPam_ == TS_CT)
		tsCT_ = value;
	if (isActPam_ == TV_AC)
		tvAC_ = value;
	if (isActPam_ == TV_AT)
		tvAT_ = value;
	if (isActPam_ == TV_CG)
		tvCG_ = value;
	if (isActPam_ == TV_GT)
		tvGT_ = value;

	double oriTsAG_, oriTsCT_, oriTvAC_, oriTvAT_, oriTvCG_, oriTvGT_;
	oriTsAG_ = tsAG_;
	oriTsCT_ = tsCT_;
	oriTvAC_ = tvAC_;
	oriTvAT_ = tvAT_;
	oriTvCG_ = tvCG_;
	oriTvGT_ = tvGT_;

	reInit ();

	tsAG_ = oriTsAG_;
	tsCT_ = oriTsCT_;
	tvAC_ = oriTvAC_;
	tvAT_ = oriTvAT_;
	tvCG_ = oriTvCG_;
	tvGT_ = oriTvGT_;


	opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}


/**
	the target function which needs to be optimized
*/
double GTRM::targetFunk(double x[]) {
	unpackData(x);
	reInit();
	opt_urtree.cmpLiNd ();
	double logLi_ = opt_urtree.getLogLi ();
	//cout << "  logli = " << logLi_ << endl;
	//cout << "logli = " << logLi_ << endl;
	return -logLi_;	
}

/**
	return the number of dimensions
*/
int GTRM::getNDim() {
	return 5; // five parameters
}

/**
	pack the data to the vector x
*/
void GTRM::packData( double x[], double lower[], double upper[], bool bound_check[]) {
	x[1] = tvAC_;
	x[2] = tsAG_;
	x[3] = tvAT_;
	x[4] = tvCG_;
	x[5] = tsCT_;
	for (int i = 1; i <= 5; i++) {
		lower[i] = MIN_GEN_PAM;
		upper[i] = MAX_GEN_PAM;
		bound_check[i] = true;
	}
}

/**
	unpack data from vector x
*/
void GTRM::unpackData(double x[]) {
	tvAC_ = x[1];
	tsAG_ = x[2];
	tvAT_ = x[3];
	tvCG_ = x[4];
	tsCT_ = x[5];
}


//--------------------------------------------------------------------
//optimize the tsTvRatio basing Brent method
bool GTRM::optPam () {
	double fx_, error_;
	double oriTsAG_ = tsAG_;
	double oriTsCT_ = tsCT_;
	double oriTvAC_ = tvAC_;
	double oriTvAT_ = tvAT_;
	double oriTvCG_ = tvCG_;
	double oriTvGT_ = tvGT_;


	if (isMasterProc())
		std::cout <<"Optimizing general reversible model parameters ..." << endl;
	std::cout.precision (5);

	Brent::turnOnOptedPam ();
	//double logLi_ = opt_urtree.cmpLogLi ();
	//  std::cout <<"Log likelihood: " << logLi_ << endl;

	if (mymodel.pam_brent == 2) {
		// optimize with Brent method

		isActPam_ = TV_AC;
		if (isMasterProc()) {
			cout << "    1. Transversion rate from A to C = ";
			cout.flush();
		}
		tvAC_= optOneDim (MIN_GEN_PAM, tvAC_, MAX_GEN_PAM,
		                  EPS_MODEL_PAM_ERROR, &fx_, &error_);
		if (isMasterProc())
			cout << tvAC_ << endl;

		reInit ();

		//------------------------------/------------------------------/------------------------------/------------------------------/------------------------------/------------------------------
		isActPam_ = TS_AG;
		if (isMasterProc()) {
			cout << "    2. Transition   rate from A to G = ";
			cout.flush();
		}
		tsAG_= optOneDim (MIN_GEN_PAM, tsAG_, MAX_GEN_PAM,
		                  EPS_MODEL_PAM_ERROR, &fx_, &error_);
		if (isMasterProc())
			cout << tsAG_ << endl;
		reInit ();

		//------------------------------/------------------------------/------------------------------/------------------------------/------------------------------/------------------------------
		isActPam_ = TV_AT;
		if (isMasterProc()) {
			cout << "    3. Transversion rate from A to T = ";
			cout.flush();
		}
		tvAT_= optOneDim (MIN_GEN_PAM, tvAT_, MAX_GEN_PAM,
		                  EPS_MODEL_PAM_ERROR, &fx_, &error_);
		if (isMasterProc())
			cout << tvAT_ << endl;
		reInit ();

		//------------------------------/------------------------------/------------------------------/------------------------------/------------------------------/------------------------------
		isActPam_ = TV_CG;
		if (isMasterProc()) {
			cout << "    4. Transversion rate from C to G = ";
			cout.flush();
		}
		tvCG_= optOneDim (MIN_GEN_PAM, tvCG_, MAX_GEN_PAM,
		                  EPS_MODEL_PAM_ERROR, &fx_, &error_);
		if (isMasterProc())
			cout << tvCG_ << endl;
		reInit ();

		//------------------------------/------------------------------/------------------------------/------------------------------/------------------------------/------------------------------

		isActPam_ = TS_CT;

		if (isMasterProc()) {
			cout << "    5. Transition   rate from C to T = ";
			cout.flush();
		}
		tsCT_= optOneDim (MIN_GEN_PAM, tsCT_, MAX_GEN_PAM,
		                  EPS_MODEL_PAM_ERROR, &fx_, &error_);
		if (isMasterProc())
			cout << tsCT_ << endl;
		if (isMasterProc())
			cout << "    6. Transversion rate from G to T =  " << tvGT_  << endl;

		reInit ();
	} else { // BFGS algorithm
		int ndim = getNDim();
		double *variables = new double[ndim+1];
		double *upper_bound = new double[ndim+1];
		double *lower_bound = new double[ndim+1];
		bool *bound_check = new bool[ndim+1];

		if (isMasterProc()) {
			cout <<"   Optimizing GTR parameters by BFGS..." << endl;
		}
		// by BFGS algorithm
		//int ndim = getNDim();
		packData(variables, lower_bound, upper_bound, bound_check);
		//optMultiDim(variables, ndim, EPS_MODEL_PAM_ERROR);
		optMultiDim(variables, ndim, lower_bound, upper_bound, bound_check, EPS_MODEL_FUNC_ERROR);
		unpackData(variables);
		
		reInit();

		delete bound_check;
		delete lower_bound;
		delete upper_bound;
		delete variables;
		if (isMasterProc()) {
			cout << "    1. Transversion rate from A to C = " << tvAC_ << endl;
			cout << "    2. Transition   rate from A to G = " << tsAG_ << endl;
			cout << "    3. Transversion rate from A to T = " << tvAT_ << endl;
			cout << "    4. Transversion rate from C to G = " << tvCG_ << endl;
			cout << "    5. Transition   rate from C to T = " << tsCT_ << endl;
			cout << "    6. Transversion rate from G to T =  " << tvGT_  << endl;
		}


	}
	opt_urtree.cmpLiNd ();
	//logLi_ = opt_urtree.getLogLi ();

	Brent::turnOffOptedPam ();

	// copy parameters to mymodel
	//mymodel.
	return ( fabs (tsAG_ - oriTsAG_) > EPS_MODEL_PAM_ERROR ||
	         fabs (tsCT_ - oriTsCT_) > EPS_MODEL_PAM_ERROR ||
	         fabs (tvAC_ - oriTvAC_) > EPS_MODEL_PAM_ERROR ||
	         fabs (tvAT_ - oriTvAT_) > EPS_MODEL_PAM_ERROR ||
	         fabs (tvCG_ - oriTvCG_) > EPS_MODEL_PAM_ERROR ||
	         fabs (tvGT_ - oriTvGT_) > EPS_MODEL_PAM_ERROR  );
}


