/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef LIBR_H
#define LIBR_H

#include "br.h"
#include "lind.h"

/**
	class to compute log-likelihood of tree
	on a branch
*/
class LiBr : public Br<double> {
public:

	/**
		the constructor
	*/
	LiBr ();

	/**
		smaLiNd, smaNdNo_ ===================greNdNo, greLiNd
	*/
	LiNd *smaLiNd_;

	LiNd *greLiNd_;

	/**
		log-likelihood difference when apply NNI
		for this branch
	*/
	double score_;

	/**
		1 if should swap on the branch based on NNI
	*/
	int isSwapLChi_;

	/**
		TODO
	*/
	int isCan_;

	/**
		TODO
	*/
	int isSwap_;

	/**
		print information 
	*/
	void writeInf ();

	/**
		copy the base br into this likelihood branch
	*/
	void copyBase (Br<double> &br);

	/**
		set the new tail LiNd
		@param newTailLiNd new tail LiNd
	*/
	void setTailLiNd (LiNd *newTailLiNd);

	/**
		@param existNdNo the node ID at the opposite
		@return the remaining liNd of this branch 
		opposite to existNdNo
	*/
	LiNd &getRemLiNd (int existNdNo);

	/**
		@return the small liNd of this branch
	*/
	LiNd &getSmaLiNd ();

	/**
		@return the great liNd of this branch
	*/
	LiNd &getGreLiNd ();

	/**
		@return the tail liNd of this branch
	*/
	LiNd &getTailLiNd ();

	/**
		@return the head liNd of this branch
	*/
	LiNd &getHeadLiNd ();

	
	/**
		@param ndNo node ID
		@return the LiNd corresponding to ndNo
	*/
	LiNd &getLiNd (int ndNo);

	/**
		@return 1 if this is an outgroup node
	*/
	int isOutGrp ();

	/**
		create the external descendant for all liNds of this branch
	*/
	void createExDesLiNd ();

	/**
		set the two liNds to be NOT cmped
	*/
	void openLiNd ();

	/**
		set the tail liNd to be Not cmped
	*/
	void openTailLiNd ();

	/**
		set the head liNd to be Not cmped
	*/
	void openHeadLiNd ();

	/**
		compute the likelihood for liNd of this branch
		@param ndNo node ID
	*/
	void cmpLiNd (int ndNo);


	/**
		compute the likelihood for tail liNd of this branch
	*/
	void cmpTailLiNd ();

	/**
		compute the likelihood for head liNd of this branch
	*/
	void cmpHeadLiNd ();

	/**
		reCompute the likelihood for tail liNd of this branch
	*/
	void reCmpTailLiNd ();

	/**
		reCompute the likelihood for head liNd of this branch
	*/
	void reCmpHeadLiNd ();

	/**
		compute the likelihood along this branch
	*/
	double cmpLi (int ptnNo, int rateNo, int parBaseNo, int parNdNo);

	/***********************************************************************
	* compute the probalibity change matrix times base frequency
	***********************************************************************/	
	
	/**
		compute the array of probability of changing matrix 
		times frequencies between all pairs of states
		after a period of brLen / subRate_
		@param brLen branch length (or evolutional time)
		@param probMat (OUT) the prob. change matrix
	*/
	void cmpProbChangeFrq (const double brLen, DMat20 &probMat);
	
	/**
		compute the array of probability of changing matrix 
		times frequencies between all pairs of states
		for heterogenous omega codon-based model
		after a period of brLen / subRate_
		@param brLen branch length (or evolutional time)
		@param classNo omega class 
		@param probMat (OUT) the prob. change matrix
	*/
	void cmpProbChangeFrq (const double brLen, const int classNo, DMat20 &probMat);

	
	/**
		compute the array of probability of changing matrix 
		and the 1st and 2nd derivatives
		times frequencies between all pairs of states
		after a period of brLen / subRate_
		@param brLen branch length (or evolutional time)
		@param probMat (OUT) the prob. change matrix
		@param derv1 (OUT) 1st of P(t)
		@param derv2 (OUT) 2nd of P(t)
	*/
	void cmpProbChangeDerivativesFrq (const double brLen, DMat20 &probMat, DMat20 &derv1, DMat20 &derv2);
	
	/**
		compute the array of probability of changing matrix 
		and the 1st and 2nd derivatives
		times frequencies between all pairs of states
		for heterogenous omega codon-based model
		after a period of brLen / subRate_
		@param brLen branch length (or evolutional time)
		@param classNo omega class 
		@param probMat (OUT) the prob. change matrix
		@param derv1 (OUT) 1st of P(t)
		@param derv2 (OUT) 2nd of P(t)
	*/
	void cmpProbChangeDerivativesFrq (const double brLen, int classNo, DMat20 &probMat,
        DMat20 &derv1, DMat20 &derv2);

	/***********************************************************************
	* compute likelihood 
	***********************************************************************/
	
	//-----------------------------------------------
	//-  UNIFORM and SITE-SPECIFIC rate	
	//-----------------------------------------------

	/**
		compute the log likelihood of this tree 
		using uniform rate or specific rate
		@return tree log-likelihood
	*/
	double cmpLogLiUniformRate ();

	/**
		compute the log likelihood derivative 
		using uniform rate or specific rate
		@param calc_logli true if you want to compute log-likelihood
		@param logli_derv1 (OUT) 1st derivative of log-likelihood
		@param logli_derv2 (OUT) 2nd derivative of log-likelihood
		@return log-likelihood if calc_logli==true
	*/
	double cmpLogLiDerivativeUniformRate (double &logli_derv1, double &logli_derv2, bool calc_logli);
	
	//-----------------------------------------------
	//-  GAMMA rate	+ YANG CODON MODEL
	//-----------------------------------------------
	
	/**
		compute the log likelihood of this tree 
		using gamma rate or codon model
		@return tree log-likelihood
	*/
	double cmpLogLiGammaRate ();

	/**
		compute the log likelihood of this tree 
		using gamma rate or codon model
		@param calc_logli true if you want to compute log-likelihood
		@param logli_derv1 (OUT) 1st derivative of log-likelihood
		@param logli_derv2 (OUT) 2nd derivative of log-likelihood
		@return log-likelihood if calc_logli==true
	*/
	double cmpLogLiDerivativeGammaRate (double &logli_derv1, double &logli_derv2, bool calc_logli);
	
	
	
	/***********************************************************************
	* general computing likelihood 
	***********************************************************************/

	inline bool isSiteSpec();

	
	/**
		compute the log-likelihood 
		@return tree log-likelihood
	*/
	double cmpLogLi ();

	/**
		compute the log-likelihood 
		@param calc_logli true if you want to compute log-likelihood
		@param logli_derv1 (OUT) 1st derivative of log-likelihood
		@param logli_derv2 (OUT) 2nd derivative of log-likelihood
		@return log-likelihood if calc_logli==true
	*/
	double cmpLogLiDerivatives (double &logli_derv1, double &logli_derv2, bool calc_logli);
	
	
	/**
		empirical bayes method to infer sites under
		positive selection using codon model (Yang 94)
	*/
	void empiricalBayesAnalysis(ostream &out);

	/**
		computing the rate for each site based on gamma distributed rates
	*/
	void empiricalBayesGammaRate (ostream &out);

	/**
		set memory for this liBr
	*/
	void setLimit ();

	/**
		release all memory of this class
	*/
	void release ();

	/**
		the destructor
	*/
	virtual ~LiBr ();
};

#endif
