/* Copyright 2020 The Chromium OS Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

#ifndef __ZEPHYR_GPIO_MAP_H
#define __ZEPHYR_GPIO_MAP_H

#include <devicetree.h>
#include <gpio_signal.h>

#include "extpower.h"
#include "lid_switch.h"
#include "power_button.h"

/*
 * Without https://github.com/zephyrproject-rtos/zephyr/pull/29282, we need
 * to manually link GPIO_ defines that platform/ec code expects to the
 * enum gpio_signal values that are generated by device tree bindings.
 *
 * Note we only need to create aliases for GPIOs that are referenced in common
 * platform/ec code.
 */
#define GPIO_TEMP_SENSOR_POWER GPIO_PG_EC_DSW_PWROK

/* Helper macros for generating CROS_EC_GPIO_INTERRUPTS */
#ifdef CONFIG_PLATFORM_EC_POWERSEQ
#define POWER_SIGNAL_INT(gpio, edge) \
	GPIO_INT(gpio, edge, power_signal_interrupt)
#define AP_PROCHOT_INT(gpio, edge) \
	GPIO_INT(gpio, edge, throttle_ap_prochot_input_interrupt)
#define POWER_BUTTON_INT(gpio, edge) \
	GPIO_INT(gpio, edge, power_button_interrupt)
#else
#define POWER_SIGNAL_INT(gpio, edge)
#define AP_PROCHOT_INT(gpio, edge)
#define POWER_BUTTON_INT(gpio, edge)
#endif

#ifdef CONFIG_PLATFORM_EC_USBC
#define TCPC_ALERT_INT(gpio, edge) GPIO_INT(gpio, edge, tcpc_alert_event)
#define PPC_INT(gpio, edge) GPIO_INT(gpio, edge, ppc_interrupt)
#define BC12_INT(gpio, edge) GPIO_INT(gpio, edge, bc12_interrupt)
#else
#define TCPC_ALERT_INT(gpio, edge)
#define PPC_INT(gpio, edge)
#define BC12_INT(gpio, edge)
#endif

#ifdef CONFIG_PLATFORM_EC_ALS_TCS3400
#define TCS3400_INT(gpio, edge) GPIO_INT(gpio, edge, tcs3400_interrupt)
#else
#define TCS3400_INT(gpio, edge)
#endif

#ifdef CONFIG_PLATFORM_EC_ACCELGYRO_BMI260
#define BMI260_INT(gpio, edge) GPIO_INT(gpio, edge, bmi260_interrupt)
#else
#define BMI260_INT(gpio, edge)
#endif

#ifdef CONFIG_PLATFORM_EC_GMR_TABLET_MODE
#define GMR_TABLET_MODE_INT(gpio, edge) GPIO_INT(gpio, edge, \
						 gmr_tablet_switch_isr)
#define GMR_TABLET_MODE_GPIO_L	GPIO_TABLET_MODE_L
#else
#define GMR_TABLET_MODE_INT(gpio, edge)
#endif

#define GPIO_EC_BATT_PRES_ODL GPIO_BATT_PRES_ODL

/*
 * Set EC_CROS_GPIO_INTERRUPTS to a space-separated list of GPIO_INT items.
 *
 * Each GPIO_INT requires three parameters:
 *   gpio_signal - The enum gpio_signal for the interrupt gpio
 *   interrupt_flags - The interrupt-related flags (e.g. GPIO_INT_EDGE_BOTH)
 *   handler - The platform/ec interrupt handler.
 *
 * Ensure that this files includes all necessary headers to declare all
 * referenced handler functions.
 *
 * For example, one could use the follow definition:
 * #define EC_CROS_GPIO_INTERRUPTS \
 *   GPIO_INT(NAMED_GPIO(h1_ec_pwr_btn_odl), GPIO_INT_EDGE_BOTH, button_print)
 */
#define EC_CROS_GPIO_INTERRUPTS                                            \
	BMI260_INT(GPIO_EC_IMU_INT_L, GPIO_INT_EDGE_FALLING)               \
	GMR_TABLET_MODE_INT(GPIO_TABLET_MODE_L, GPIO_INT_EDGE_BOTH)        \
	GPIO_INT(GPIO_AC_PRESENT, GPIO_INT_EDGE_BOTH, extpower_interrupt)  \
	GPIO_INT(GPIO_LID_OPEN, GPIO_INT_EDGE_BOTH, lid_interrupt)         \
	GPIO_INT(GPIO_EC_WP_L, GPIO_INT_EDGE_BOTH, switch_interrupt)       \
	POWER_SIGNAL_INT(GPIO_PCH_SLP_S0_L, GPIO_INT_EDGE_BOTH)            \
	POWER_SIGNAL_INT(GPIO_PCH_SLP_S3_L, GPIO_INT_EDGE_BOTH)            \
	POWER_SIGNAL_INT(GPIO_PCH_SLP_SUS_L, GPIO_INT_EDGE_BOTH)           \
	POWER_SIGNAL_INT(GPIO_PG_EC_DSW_PWROK, GPIO_INT_EDGE_BOTH)         \
	POWER_SIGNAL_INT(GPIO_PG_EC_RSMRST_ODL, GPIO_INT_EDGE_BOTH)        \
	POWER_SIGNAL_INT(GPIO_PCH_DSW_PWROK, GPIO_INT_EDGE_BOTH)           \
	POWER_SIGNAL_INT(GPIO_PG_EC_ALL_SYS_PWRGD, GPIO_INT_EDGE_BOTH)     \
	POWER_BUTTON_INT(GPIO_POWER_BUTTON_L, GPIO_INT_EDGE_BOTH)          \
	TCPC_ALERT_INT(GPIO_USB_C0_TCPC_INT_ODL, GPIO_INT_EDGE_BOTH)       \
	TCPC_ALERT_INT(GPIO_USB_C1_TCPC_INT_ODL, GPIO_INT_EDGE_BOTH)       \
	TCS3400_INT(GPIO_EC_ALS_RGB_INT_L, GPIO_INT_EDGE_FALLING)          \
	PPC_INT(GPIO_USB_C0_PPC_INT_ODL, GPIO_INT_EDGE_BOTH)               \
	PPC_INT(GPIO_USB_C1_PPC_INT_ODL, GPIO_INT_EDGE_BOTH)               \
	BC12_INT(GPIO_USB_C0_BC12_INT_ODL, GPIO_INT_EDGE_BOTH)             \
	BC12_INT(GPIO_USB_C1_MIX_INT_ODL, GPIO_INT_EDGE_BOTH)              \
	AP_PROCHOT_INT(GPIO_EC_PROCHOT_IN_L, GPIO_INT_EDGE_BOTH)           \
	GPIO_INT(GPIO_VOLUME_DOWN_L, GPIO_INT_EDGE_BOTH, button_interrupt) \
	GPIO_INT(GPIO_VOLUME_UP_L, GPIO_INT_EDGE_BOTH, button_interrupt)

#endif /* __ZEPHYR_GPIO_MAP_H */
