/*
 * scdtools - Tools for Scdaemon and OpenPGP smartcards
 * Copyright (C) 2014,2015,2016,2017 Damien Goutte-Gattat
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>

#include <gcrypt.h>

#include "gpg-util.h"

/*
 * Check that a socket exists at the specified location.
 *
 * Return 0 if the socket does exist, or a gpg_error_t error code.
 */
static gpg_error_t
check_socket(const char *socket_name)
{
    struct stat st;
    gpg_error_t e;

    if ( stat(socket_name, &st) == -1 )
        e = gpg_err_code_from_errno(errno);
    else if ( ! S_ISSOCK(st.st_mode) )
        e = gpg_err_code_from_errno(ENOTSOCK);
    else
        e = GPG_ERR_NO_ERROR;

    return e;
}

/*
 * Convenience function: it does the same as check_socket above
 * but accept a printf-like format string to construct the path
 * before testing it.
 */
static gpg_error_t
check_socket_fmt(char *buffer, size_t len, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    vsnprintf(buffer, len, fmt, ap);
    va_end(ap);

    return check_socket(buffer);
}

/*
 * Find the socket of GnuPG Agent. First look for the GPG_AGENT_INFO
 * environment variable, if unset look for the socket at some
 * standard locations, if not found use gpg-connect-agent.
 *
 * Return 0 if the socket is found, or a gpg_error_t error code.
 */
static gpg_error_t
get_agent_socket_name(char *buffer, size_t len)
{
    char *env_info = NULL;
    gpg_error_t e = GPG_ERR_NO_AGENT;

    /* First look for GPG_AGENT_INFO in the environment
     * (GnuPG < 2.1). */
    if ( (env_info = getenv("GPG_AGENT_INFO")) ) {
        unsigned proto;
        char fmt[32];

        /*
         * GPG_AGENT_INFO should be of the form
         * "path_to_socket:x:y", where x is the PID
         * of the GnuPG Agent process and y is the
         * protocol version (currently always 1).
         */
        snprintf(fmt, sizeof(fmt), "%%%lus:%%*u:%%i", len);
        if ( sscanf(env_info, fmt, buffer, &proto) != 2 )
            e = gcry_error(GPG_ERR_INV_VALUE);
        else if ( proto != 1 )
            e = gcry_error(GPG_ERR_UNSUPPORTED_PROTOCOL);
        else
            e = check_socket(buffer);

        /*
         * If the GPG_AGENT_INFO variable exists, it must
         * point to a suitable socket; otherwise, abort
         * here without looking elsewhere.
         */
        if ( e )
            return e;
    }

    /* If GNUPGHOME is defined, try looking for a socket in
     * the specified directory. */
    if ( e && (env_info = getenv("GNUPGHOME")) )
        e = check_socket_fmt(buffer, len, "%s/S.gpg-agent", env_info);

    /* Without GNUPGHOME, try looking in standard directories. */
    if ( e && ! env_info ) {
        static const char *prefixes[] = { "/run", "/var/run", NULL };
        struct passwd *pwd;
        int i;

        if ( ! (env_info = getenv("HOME")) && (pwd = getpwuid(getuid())) )
            env_info = pwd->pw_dir;

        if ( env_info )
            e = check_socket_fmt(buffer, len, "%s/.gnupg/S.gpg-agent",
                    env_info);

        /* If no socket was found in GnuPG's home directory,
         * try looking under [/var]/run (for GnuPG >= 2.1.13). */
        for ( i = 0; e && prefixes[i]; i++ )
            e = check_socket_fmt(buffer, len, "%s/user/%u/gnupg/S.gpg-agent",
                    prefixes[i], getuid());
    }

    /*
     * Finally, try asking the agent directly. This methold could
     * replace all the methods above, but since it requires spawning
     * a new process, we use it only as a fallback. We expect that
     * most of the time, the agent should already be running when our
     * programs are called, and thus the socket would already exist
     * and be found by the above code.
     */
    if ( e ) {
        FILE *f;

        f = popen(GPG_CONNECT_AGENT_PATH " 'GETINFO socket_name' /bye", "r");
        if ( f ) {
            char fmt[32];

            snprintf(fmt, sizeof(fmt), "D %%%lus\nOK\n", len);
            if ( fscanf(f, fmt, buffer) == 1 )
                e = check_socket(buffer);

            pclose(f);
        }
    }

    return e;
}


/*
 * Establish a connection with a running GnuPG Agent.
 *
 * @param ctx       A pointer to a assuan_context_t object that will
 *                  be initialized by this function.
 * @param init_env  If true, environment variables needed for pinentry
 *                  will be passed to the agent.
 *
 * @return 0 if the connection was established, or a gpg_error_t
 *         error code.
 */
gpg_error_t
connect_to_agent(assuan_context_t *ctx, int init_env)
{
    char socket_name[255];
    gpg_error_t e;

    if ( (e = get_agent_socket_name(socket_name, sizeof(socket_name))) )
        return e;

    if ( ! (e = assuan_new(ctx)) ) {
        if ( ! (e = assuan_socket_connect(*ctx, socket_name, ASSUAN_INVALID_PID, 0)) ) {
            if ( init_env && (e = init_agent_environment(*ctx)) )
                assuan_release(*ctx);
        }
        else
            assuan_release(*ctx);
    }

    return e;
}

struct pstring {
    size_t  len;
    char   *buffer;
};

/*
 * Callback for the below function.
 */
static gpg_error_t
socket_name_cb(void *arg, const void *line, size_t len)
{
    struct pstring *p = (struct pstring *)arg;

    if ( len > p->len - 1 )
        return gcry_error(GPG_ERR_BUFFER_TOO_SHORT);

    strncpy(p->buffer, line, p->len);

    return 0;
}

/*
 * Find the socket of Scdaemon. Look for the socket in some
 * standard locations, and if not found ask to the agent.
 *
 * @return 0 if Scdaemon's socket was found, or a gpg_error_t
 *         error code otherwise.
 */
static gpg_error_t
get_scd_socket_name(char *buffer, size_t len)
{
    char *env_info = NULL;
    gpg_error_t e = GPG_ERR_NO_SCDAEMON;

    /*
     * This function roughly follows the same logic as the
     * get_agent_socket_name above:
     * - first look for $GNUPGHOME/S.scdaemon;
     * - then and if GNUPGHOME is not set, look in
     *   GnuPG's default home directory and under [/var]/run;
     * - as a fallback, send a request to the GnuPG Agent.
     */

    if ( (env_info = getenv("GNUPGHOME")) )
        e = check_socket_fmt(buffer, len, "%s/S.scdaemon", env_info);

    if ( e && ! env_info ) {
        static const char *prefixes[] = { "/run", "/var/run", NULL };
        struct passwd *pwd;
        int i;

        if ( ! (env_info = getenv("HOME")) && (pwd = getpwuid(getuid())) )
            env_info = pwd->pw_dir;

        if ( env_info )
            e = check_socket_fmt(buffer, len, "%s/.gnupg/S.scdaemon", env_info);

        for ( i = 0; e && prefixes[i]; i++ )
            e = check_socket_fmt(buffer, len, "%s/user/%u/gnupg/S.scdaemon",
                    prefixes[i], getuid());
    }

    if ( e ) {
        assuan_context_t ctx;

        if ( ! (e = connect_to_agent(&ctx, 0)) ) {
            if ( ! (e = assuan_transact(ctx, "GETINFO scd_running",
                            NULL, NULL, NULL, NULL, NULL, NULL)) ) {
                struct pstring p;

                p.len = len;
                p.buffer = buffer;
                e = assuan_transact(ctx, "SCD GETINFO socket_name",
                        socket_name_cb, &p, NULL, NULL, NULL, NULL);

                if ( ! e )
                    e = check_socket(buffer);
            }
        }

        assuan_release(ctx);
    }

    return e;
}

/*
 * Establish a connection with a running Scdaemon.
 *
 * @param ctx A pointer to a assuan_context_t object that will be
 *            initialized by this function.
 *
 * @return 0 if the connection was established, or a gpg_error_t
 *         error code.
 */
gpg_error_t
connect_to_scdaemon(assuan_context_t *ctx)
{
    char socket_name[255];
    gpg_error_t e;

    if ( (e = get_scd_socket_name(socket_name, sizeof(socket_name))) )
        return e;


    if ( ! (e = assuan_new(ctx)) ) {
        if ( ! (e = assuan_socket_connect(*ctx, socket_name, ASSUAN_INVALID_PID, 0)) )
            e = gcry_error(GPG_ERR_NO_ERROR);
        else
            assuan_release(*ctx);
    }

    return e;
}

/*
 * Pass to the GnuPG Agent the environment variables needed for
 * pinentry.
 *
 * @param ctx The assuan_context_t object used to communicate with
 *            a running GnuPG Agent.
 *
 * @return 0 if all environment variables were passed successfully,
 *         or a gpg_error_t error code.
 */
gpg_error_t
init_agent_environment(assuan_context_t ctx)
{
    char command[64], *value;
    gpg_error_t e;

    struct {
        const char *name;
        const char *option;
    } *var, env_vars[] = {
        { "GPG_TTY",            "ttyname" },
        { "TERM",               "ttytype" },
        { "DISPLAY",            "display" },
        { "XAUTHORITY",         "xauthority" },
        { "PINENTRY_USER_DATA", "pinentry-user-data" },
        { NULL,                 NULL }
    };

    for ( var = env_vars, e = 0; var->name && e == 0; var++ ) {
        if ( (value = getenv(var->name)) ) {
            if ( (unsigned) snprintf(command, sizeof(command), "OPTION %s=%s",
                        var->option, value) >= sizeof(command) )
                e = gcry_error(GPG_ERR_TOO_LARGE);

            e = assuan_transact(ctx, command, NULL, NULL, NULL, NULL,
                    NULL, NULL);
        }
    }

    return e;
}
