/*  sectormappings.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef SECTORMAPPINGS_H_
#define SECTORMAPPINGS_H_

#include "integralfamily.h"
#include <set>
#include <map>
#include "sector.h"
#include "functions.h"

// forward declarations
namespace YAML {
class Node;
class Emitter;
}

namespace Reduze {

class Sector;
class SectorGraph;
class SectorGL;

/// zero sectors, sector relations and sector symmetries
/** one-to-one mappings based on the permutation symmetries of IntegralFamily
 ** are not stored explicitly
 **  */
class SectorMappings {
public:

	// constructors

	/// constructs empty sector mappings
	SectorMappings(const IntegralFamily* ic);

	/// constructs empty sector mappings
	SectorMappings(const CrossedIntegralFamily* ic);

	// getters

	inline const IntegralFamily* integralfamily() const {
		return integralfamily_;
	}
	inline const std::set<int>& zero_sectors() const {
		return zero_sectors_;
	}
	inline const std::set<int>& sectors_without_graph() const {
		return sectors_without_graph_;
	}
	inline const std::map<int, Sector>& sector_relations() const {
		return sector_relations_;
	}
	inline const std::map<int, GiNaC::exmap>& sector_relation_shifts() const {
		return sector_relation_shifts_;
	}
	inline const std::map<int, std::list<GiNaC::exmap> >& sector_symmetries() const {
		return sector_symmetries_;
	}

	/* terminology:
	 * The permutation symmetries of the integral family define for each sector
	 * a class of equivalent sectors. The minimal sector (with minimal sector id)
	 * of such a class is called the "minimal equivalent" or just "equivalent"
	 *
	 * If shift relations (with |det|=1) are defined then we call the sector
	 * to which a sector can be shifted the "shift targed sector" of just the "shifted".
	 * The shifted sector is not automatically the "minimal equivalent".
	 *
	 * Sectors, which cannot be modified by perm. symmetries or shifts
	 * anymore, we call reduced sectors
	 */

	/// using all available information to check whether this sector is zero
	static bool is_zero(const Sector& sector);

	// shift targets

	/**  Returns all sectors of this family which are minimal w.r.t global permutation
	 **  symmetries and for which no shift to any lower sector exists.
	 **  Sectors might be zero and don't need to have a graph **/
	void find_shift_targets(std::set<int>&) const;
	/** returns all sectors of this family which are minimal w.r.t global permutation
	 ** symmetries and for which no shift to any lower sector exists and which are not
	 ** marked as unphysical (as sectors without a graph)
	 ** The returned sectors might be zeros sectors **/
	void find_shift_targets_with_graph(std::list<SectorGL>& shift_targets,
			bool minimize_by_twists, bool construct_minimal_graphs) const;
	/** shift targets of all sectors with number of loops in the range [l_min,l_max] **/
	static size_t find_shift_targets_with_graph(std::map<int, std::map<int,
			std::list<SectorGL> > >& secbytbyloop, int l_min, int l_max,
			bool minimize_by_twists, bool construct_minimal_graphs);

	/// returns the minimal equivalent version of given integral
	/** chooses minimal equivalent crossing, employs permutation symmetries,
	 ** zero sectors are mapped to ID = 0 sector of same integral family */
	static INT get_equivalent_or_unique_zero(const INT&);

	/// applies all shifts and permutations to find a minimal equivalent sector
	/** for crossed sectors without a shift the minimal equivalent crossing is
	 ** selected, recurse flag controls whether to continue after first shift
	 ** (relevant for chains of shifts), zero sectors are mapped to ID = 0
	 ** sector **/
	static Sector get_shifted_sector(const Sector&, bool recurse = true);

	/// inserts shift
	void insert_shift(int from_id, const Sector& to, const GiNaC::exmap& shift);
	/// deletes shift
	void remove_shift(int from_id);
	/// inserts symmetry shift (maps sector onto itself)
	void insert_symmetry_shifts(int sector_id,
			const std::list<GiNaC::exmap>& shifts);
    /// constructs explicite permutations of propagators for a shift
	/** returns false if shift fails to map sector 'from' to sector 'to',
	 ** if propagator i in 'from' is mapped to propagator j in 'to',
	 ** perm[i] = j is set, other entries are set to perm[*] = -1 */
    // \todo this method could be used to speed up apply_shift()
	static bool find_permutations_for_shift(const Sector& from,
			const Sector& to, const GiNaC::exmap& shift,
			std::vector<int>& perm );
	/// tests if the shift is valid, throws runtime_error if not
	void verify_shift(int from_id, const Sector& to,
			const GiNaC::exmap& shift);

	// graphs

	/// get the graphs of the sectors (this function is only efficient if the sectors do have a graph)
	static void get_graphs(std::list<SectorGL>& sectorgraphs,
			const std::set<Sector>& sectors, bool minimize_by_twists,
			bool construct_minimal_graphs, bool verbose);

	//	/// helper to find substitutions needed to implement a shift relation
	//	/** permutes integral to equivalent form such that shift can be applied **/
	//	GiNaC::exmap find_shift_for_sector_relation(INT& integral) const;


	// zero sectors
	/// insert zero sectors directly
	void insert_zero_sectors(const std::set<int>& sector_ids);
	/// test whether given sectors (keys in map) are zero and add them together with the values (sectors in the list) to the zero_sectors_
	// all sectors must belong to this integral family
	int find_zero_sectors(
			const std::map<Sector, std::list<Sector> >& sectors_by_rep,
			bool allow_longtest);

	// output

	static void print_sectortree_dot(const std::set<Sector>& sectors,
			const std::string& filename);

	void read(const YAML::Node&);
	friend YAML::Emitter & operator<<(YAML::Emitter&, const SectorMappings&);

	/// throw if the mappings are not equivalent
	/** equivalent means that they are equal up to the explicit shifts in the
	 ** sector relations and the symmetry shifts **/
	void verify_is_equivalent(const SectorMappings& other) const;
	// \todo remove me, just a hack to make comparing sector mapping files work
	static bool no_explicit_shift_verification_;

private:
	/// searches the sectors for graphs
	/** sectors must belong to SectorMappings integral family
	 ** adds sectors with vanishing U-polynomial to zero_sectors_
	 ** adds sectors for which no graph has been found to sectors_without_graph_ */
	void find_graphs(std::list<SectorGraph>& sectorgraphs, const std::set<
			Sector>& sectors, bool minimize_by_twists,
			bool construct_minimal_graphs, bool verbose);

	/// finds shifts of the sectors in 'sectors' to the sectors in 'minimal' and to 'sectors' itself
	/**
	 ** All sectors in 'minimal' (from arbitrary families) are assumed to be lower
	 ** than the sectors in 'sector' (from this family).
	 ** New minimal equivalent sectors of 'sectors' are added to 'minimal' and the map
	 ** 'new_targets_with_equiv' is set to contain the pairs of all minimal new target from 'sectors'
	 ** with a list of equivalent sectors  from 'sectors'.
	 ** if find_zero_sectors is true then the sectors for which a shift to a lower
	 ** zero-sector of another family has been found is added to the zero_sectors */
	void find_sector_shifts(
	/**/std::set<SectorGL>& minimal, const std::set<SectorGL>& sectors,
			std::map<SectorGL, std::list<SectorGL> >& new_targets_with_equiv,
			bool minimize_target_crossings, bool find_zero_sectors,
			bool allow_crossings, bool allow_general_crossings, std::map<
					Sector, std::list<std::map<int, int> > >& cached_node_perm,
			bool verbose);

	/** Finds the symmetry shifts: these are shifts which transform the momenta
	 ** from one leg to another leg using the automorphism group of the graphs
	 ** and multi-edge permutations.
	 ** The returned shifts have |det| = 1.
	 ** The shifts my contain permutation of external legs if they are crossings
	 ** equivalent to the identity. **/
	void find_symmetry_shifts(const std::set<SectorGL>& sectors,
			bool examine_twists);

	/// performs a small reduction to test if corner integrals reduce to zero
	/** returns true if corner integrals are found to reduce to zero */
	bool search_corner_reduction_to_zero(int sector_id, bool allow_longtest =
			true) const;

	/// helper method for read()
	void read_sector_relations(const YAML::Node&);

private:
	/// the integral family
	const IntegralFamily* integralfamily_;
	/// zero sector ids
	std::set<int> zero_sectors_;
	/// sectors for which no graph can be drawn
	/** sectors which are trivially zero b.c. they don't have enough independent
	 * propagators are not included */
	std::set<int> sectors_without_graph_;
	/// linear sector relations
	/** source sector id -> target sector **/
	std::map<int, Sector> sector_relations_;
	/// shifts needed to actually employ sector_relations_
	/** source sector id  -> shift to apply to source sector */
	std::map<int, GiNaC::exmap> sector_relation_shifts_;
	/// map of sector ids which are invariant under the shifts
	std::map<int, std::list<GiNaC::exmap> > sector_symmetries_;

	friend class SetupSectorMappings;
};

} // namespace Reduze

#endif /* SECTORMAPPINGS_H_ */
