//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
// -------------------------------------------------------------------
//
// GEANT4 Class file
//
// File name:     G4RayleighAngularGenerator
//
// Author:        V. Ivanchenko using design of existing 
//                interface 
// 
// Creation date: 31 May 2012
//
// Modifications: 
//
// Class Description: Class are tabulated data according modified formula
//                    modified fit formulas from Dermott E. Cullen, 
//                    Nucl. Instrum. Meth. Phys. Res. B v.101, (4),499-510. 
//                    
//
// Class for Rayleigh scattering generation
//
// Class Description: End 
// -------------------------------------------------------------------
//
//    

#include "G4RayleighAngularGenerator.hh"
#include "G4PhysicalConstants.hh"
#include "G4SystemOfUnits.hh"
#include "Randomize.hh"
#include "G4Log.hh"
#include "G4Exp.hh"

using namespace std;

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4RayleighAngularGenerator::G4RayleighAngularGenerator()
  : G4VEmAngularDistribution("CullenGenerator")
{
  G4double x = cm/(h_Planck*c_light);
  fFactor = 0.5*x*x;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4RayleighAngularGenerator::~G4RayleighAngularGenerator() 
{}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....

G4ThreeVector& 
G4RayleighAngularGenerator::SampleDirection(const G4DynamicParticle* dp,
					    G4double, G4int Z,
					    const G4Material*)
{
  G4double ekin = dp->GetKineticEnergy();
  G4double xx = fFactor*ekin*ekin;

  G4double n0 = PP6[Z] - 1.0;
  G4double n1 = PP7[Z] - 1.0;
  G4double n2 = PP8[Z] - 1.0;
  G4double b0 = PP3[Z];
  G4double b1 = PP4[Z];
  G4double b2 = PP5[Z];

  static const G4double numlim = 0.02;
  G4double x  = 2.*xx*b0;
  G4double w0 = (x < numlim) ? n0*x*(1.0 - 0.5*(n0 - 1.0)*x*(1.0 - (n0 - 2.0)*x/3.))
    : 1.0 - G4Exp(-n0*G4Log(1.0 + x)); 

  x  = 2.*xx*b1;
  G4double w1 = (x < numlim) ? n1*x*(1.0 - 0.5*(n1 - 1.0)*x*(1.0 - (n1 - 2.0)*x/3.))
    : 1.0 - G4Exp(-n1*G4Log(1.0 + x));
 
  x  = 2.*xx*b2;
  G4double w2 = (x < numlim) ? n2*x*(1.0 - 0.5*(n2 - 1.0)*x*(1.0 - (n2 - 2.0)*x/3.))
    : 1.0 - G4Exp(-n2*G4Log(1.0 + x));

  G4double x0= w0*PP0[Z]/(b0*n0);
  G4double x1= w1*PP1[Z]/(b1*n1);
  G4double x2= w2*PP2[Z]/(b2*n2);

  G4double cost;
  do {
    G4double w = w0;
    G4double n = n0;
    G4double b = b0;
  
    x = G4UniformRand()*(x0 + x1 + x2);
    if(x > x0) {
      x -= x0;
      if(x <= x1 ) {
	w = w1;
	n = n1;
	b = b1;
      } else { 
	w = w2;
	n = n2;
	b = b2;
      }
    }
    n = 1.0/n;

    // sampling of angle
    G4double y = w*G4UniformRand();
    if(y < numlim) { x = y*n*( 1. + 0.5*(n + 1.)*y*(1. - (n + 2.)*y/3.)); }   
    else           { x = G4Exp(-n*G4Log(1. - y)) - 1.0; }
    cost = 1.0 - x/(b*xx);
  } while (2*G4UniformRand() > 1.0 + cost*cost || cost < -1.0);

  G4double phi  = twopi*G4UniformRand();
  G4double sint = sqrt((1. - cost)*(1.0 + cost));
  fLocalDirection.set(sint*cos(phi),sint*sin(phi),cost);
  fLocalDirection.rotateUz(dp->GetMomentumDirection());

  return fLocalDirection;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....
const G4double
  G4RayleighAngularGenerator::PP0[101] = {0,
  -2.477067493e-06, 3.308728278e+00, 5.214590000e+00, 1.018853220e+01, 3.668594783e+00, 1.540100708e+01, 3.709326016e+00, 3.694224007e+00, 3.545302543e+00, 3.487912341e+00,  // 1-10
  3.407539119e+00, 3.387258718e+00, 3.230544865e+00, 3.013493175e+00, 2.572039815e+00, 1.792301548e+00, 1.701680300e+00, 1.601363893e+02, 2.968330000e+02, 3.065649816e+02,  // 11-20
  3.731860000e+02, 4.201934230e+02, 4.612920338e+02, 4.861184287e+02, 3.101874240e+02, 3.351026594e+02, 6.367362521e+02, 3.888087526e+02, 7.588301541e+02, 8.158625388e+02,  // 21-30
  4.773566612e+02, 5.084376201e+02, 5.413677298e+02, 5.744903249e+02, 3.743351693e+00, 1.009910000e+01, 6.816358014e+02, 7.187067112e+02, 7.580649444e+02, 4.414330698e+00,  // 31-40
  2.972874756e+00, 2.740289935e+00, 2.507487315e+00, 9.657410000e+02, 1.011090000e+03, 2.130702941e+00, 6.533143308e+00, 1.037000000e+03, 2.288698564e+00, 4.244459075e+00,  // 41-50
  6.127527681e+01, 3.792700000e+00, 4.798110886e+00, 1.312500008e+03, 1.889593891e+00, 2.404540000e+03, 3.352713210e+01, 3.721985093e+01, 1.800704938e+02, 1.752112119e+03,  // 51-60
  1.194981340e+02, 2.167522747e+02, 3.364590000e+03, 6.311743615e+01, 2.506979318e+02, 8.928490874e+01, 2.021000018e+03, 1.288577408e+02, 3.286469325e+03, 1.437863346e+02,  // 61-70
  1.431016078e+02, 1.359161828e+02, 3.670456078e+02, 3.934040000e+02, 3.175509489e+02, 3.262959885e+03, 3.021668154e+03, 3.252098480e+02, 3.477962301e+02, 3.363519614e+02,  // 71-80
  3.340399767e+02, 3.011038509e+03, 3.485497941e+02, 3.657731951e+02, 3.251994160e+03, 3.987097887e+02, 3.891269496e+03, 4.223467334e+02, 4.009941534e+02, 1.380789588e+00,  // 81-90
  2.791527615e+00, 3.755804160e+03, 4.863705847e+02, 5.464276392e+02, 7.089644571e+02, 7.445117785e+02, 2.037086948e+00, 1.065528514e+03, 1.133135197e+00, 3.269427358e+03 // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP1[101] = {0,
  1.000002477e+00, 6.912717221e-01, 3.772400000e+00, 3.673669324e+00, 1.199955472e+01, 3.658311128e+00, 2.130780048e+01, 2.833297837e+01, 3.748184285e+01, 4.656971845e+01,  // 1-10
  5.759246095e+01, 6.861274506e+01, 8.376139862e+01, 9.799999987e+01, 1.104279602e+02, 1.279999999e+02, 1.439999998e+02, 1.620000000e+02, 6.155750000e+01, 9.069690514e+01,  // 11-20
  6.540840000e+01, 6.120116992e+01, 6.540487635e+01, 8.701154587e+01, 3.120000000e+02, 3.380000000e+02, 8.971041364e+01, 3.920000000e+02, 7.927032750e+01, 8.129623999e+01,  // 21-30
  4.800000000e+02, 5.120000000e+02, 5.440000000e+02, 5.780000000e+02, 6.119999952e+02, 6.479930000e+02, 6.839999993e+02, 7.219999975e+02, 7.600000000e+02, 7.999965590e+02,  // 31-40
  8.399999990e+02, 8.819999639e+02, 9.239993520e+02, 9.680000000e+02, 1.012000000e+03, 1.055869297e+03, 1.102904035e+03, 1.150000007e+02, 1.199999994e+03, 1.249999690e+03,  // 41-50
  1.300000000e+03, 1.352000000e+03, 1.400242211e+03, 1.455000001e+02, 1.511110411e+03, 7.298520000e+02, 1.624000000e+03, 1.644780150e+03, 1.739999988e+03, 1.800000000e+03,  // 51-60
  1.859999972e+03, 1.705247726e+03, 6.031510000e+02, 2.047999979e+03, 2.099526330e+03, 2.088715091e+03, 2.240000003e+02, 2.183146437e+03, 2.380000102e+02, 2.449999497e+03,  // 61-70
  2.519999640e+03, 2.591999669e+03, 2.317935577e+03, 2.344600000e+03, 2.811999969e+03, 2.378704485e+03, 2.614304773e+03, 3.041994341e+03, 3.065168794e+03, 3.092144649e+03,  // 71-80
  3.279999955e+03, 3.362000000e+03, 3.443996426e+03, 3.190843165e+03, 3.610058546e+02, 3.299292785e+03, 3.283959104e+03, 3.507767814e+03, 3.729069400e+03, 4.049999721e+03,  // 81-90
  4.139900637e+03, 4.761958553e+02, 3.838629680e+03, 3.871572368e+03, 4.511999370e+03, 3.863489473e+03, 4.703865705e+03, 4.801999723e+03, 4.899867488e+03, 4.999989627e+03  // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP2[101] = {0,
  0.000000000e+00, 0.000000000e+00, 1.300910000e-02, 2.137798478e+00, 9.331850497e+00, 1.694068179e+01, 2.398287351e+01, 3.197279763e+01, 3.997285461e+01, 4.994236921e+01,  // 1-10
  5.999999993e+01, 7.199999622e+01, 8.200805651e+01, 9.498650695e+01, 1.120000000e+02, 1.262076986e+02, 1.432983199e+02, 1.863610653e+00, 2.609270000e+00, 2.738113251e+00,  // 11-20
  2.405300000e+00, 2.605407111e+00, 2.303089878e+00, 2.870025399e+00, 2.812575977e+00, 2.897340596e+00, 2.553334297e+00, 3.191247422e+00, 2.899518400e+00, 2.841221189e+00,  // 21-30
  3.643338776e+00, 3.562379948e+00, 3.632270214e+00, 3.509675162e+00, 6.092566531e+02, 6.379080000e+02, 3.364199322e+00, 3.293291322e+00, 2.935055634e+00, 7.955891103e+02,  // 31-40
  8.380271263e+02, 8.792597462e+02, 9.224931607e+02, 2.258920000e+00, 1.909930000e+00, 1.058000000e+03, 1.099562822e+03, 1.151999999e+03, 1.198711308e+03, 1.245755851e+03,  // 41-50
  1.239724724e+03, 1.348210000e+03, 1.403959678e+03, 1.457999992e+03, 1.511999996e+03, 1.608510000e+00, 1.591472868e+03, 1.681999999e+03, 1.560929519e+03, 4.788788135e+01,  // 51-60
  1.741501894e+03, 1.922000000e+03, 1.259160000e+00, 1.984882585e+03, 1.874775739e+03, 2.178000000e+03, 2.243999981e+03, 2.311995822e+03, 1.236530665e+03, 2.306214168e+03,  // 61-70
  2.377898752e+03, 2.456084148e+03, 2.644018816e+03, 2.738000000e+03, 2.495449082e+03, 1.343356300e+02, 2.930270735e+02, 2.716795811e+03, 2.828034975e+03, 2.971503390e+03,  // 71-80
  2.946960068e+03, 3.509614912e+02, 3.096453780e+03, 3.499383640e+03, 3.611999985e+03, 3.697997426e+03, 3.937713997e+02, 3.813885452e+03, 3.790936446e+03, 4.048619490e+03,  // 81-90
  4.138307836e+03, 4.231999985e+03, 4.323999735e+03, 4.417999993e+03, 3.804036173e+03, 4.607998748e+03, 4.703097208e+03, 3.736471763e+03, 4.899999376e+03, 1.730583016e+03 // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP3[101] = {0,
  1.325913011e-16, 4.871957764e-16, 1.950420000e-15, 3.995936773e-16, 4.732524874e-17, 5.469297930e-16, 2.375713373e-17, 1.810463731e-17, 1.336380641e-17, 1.052686533e-17,  // 1-10
  8.030444002e-18, 6.678303405e-18, 5.358234684e-18, 4.335725896e-18, 3.306051079e-18, 1.819679449e-18, 1.228045928e-18, 2.430163362e-16, 1.869430000e-15, 9.951691390e-16,  // 11-20
  1.805410000e-15, 1.568852030e-15, 1.464264654e-15, 4.939313750e-16, 2.361373681e-16, 2.319629775e-16, 4.429138015e-16, 2.283559073e-16, 3.210934122e-16, 2.876037822e-16,  // 21-30
  2.121922892e-16, 2.064234936e-16, 1.996479886e-16, 1.946742248e-16, 8.778017543e-19, 3.460900000e-18, 1.809840754e-16, 1.783125511e-16, 1.722338339e-16, 9.822789565e-19,  // 31-40
  4.605105770e-19, 3.816125471e-19, 3.017537476e-19, 1.666200000e-16, 1.688410000e-16, 1.839402998e-19, 1.246431543e-18, 9.807308372e-17, 1.932399654e-19, 6.865448303e-19,  // 41-50
  9.418934098e-18, 6.540360000e-19, 6.723352605e-19, 1.147797684e-16, 7.975389174e-20, 1.232720000e-15, 5.621737854e-18, 6.015501574e-18, 1.770592320e-17, 1.646956996e-16,  // 51-60
  1.285032193e-17, 1.897941915e-17, 8.382250000e-16, 8.047052107e-18, 1.953322654e-17, 1.048054550e-17, 1.547617945e-15, 1.315449339e-17, 2.966877317e-16, 1.381146328e-17,  // 61-70
  1.351358940e-17, 1.262062535e-17, 2.238342977e-17, 2.364690000e-17, 1.901461934e-17, 2.019852016e-16, 1.258732966e-16, 1.866062124e-17, 1.933265996e-17, 1.869942002e-17,  // 71-80
  1.842791671e-17, 1.006061314e-16, 1.845331171e-17, 1.903270841e-17, 9.716982760e-17, 2.023698895e-17, 1.689153339e-15, 2.047894466e-17, 1.932138801e-17, 1.971060876e-20,  // 81-90
  1.861376691e-19, 8.994366946e-17, 2.168855529e-17, 2.359136120e-17, 2.975661335e-17, 3.035334920e-17, 1.119247839e-19, 4.012940590e-17, 1.467159462e-20, 9.769616225e-17  // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP4[101] = {0,
  1.105926652e-15, 2.260014406e-16, 1.568360000e-16, 7.604704395e-17, 2.874727277e-16, 3.180531750e-17, 3.632331994e-16, 2.720099679e-16, 1.197719851e-16, 1.079438057e-16,  // 1-10
  2.381852183e-15, 3.734413372e-15, 8.391867563e-17, 4.824902206e-16, 1.501401964e-16, 3.852331215e-16, 3.990595815e-16, 3.293279270e-16, 1.384700000e-17, 3.625981107e-17,  // 11-20
  1.379630000e-17, 9.139141296e-18, 1.872408780e-17, 2.203851185e-17, 2.682848134e-16, 2.769206296e-16, 2.418950951e-17, 2.841108885e-16, 1.400094487e-17, 1.387082935e-17,  // 21-30
  5.210034592e-16, 7.614492538e-16, 8.948178869e-16, 9.788323391e-16, 8.671211693e-16, 8.778680000e-16, 1.077256234e-15, 1.679640775e-15, 1.420854572e-15, 7.623342657e-16,  // 31-40
  6.465293663e-16, 3.854995461e-16, 4.894995857e-16, 2.432900000e-16, 5.828990000e-16, 1.609458606e-16, 2.173584371e-16, 1.795526101e-17, 9.530729189e-16, 2.344176851e-16,  // 41-50
  3.105865310e-16, 4.197600000e-16, 1.326321431e-16, 1.773261410e-17, 1.494954852e-16, 6.910460000e-17, 9.139538226e-16, 1.600844678e-16, 7.627976232e-16, 6.798418289e-16,  // 51-60
  7.097022036e-16, 1.115122840e-16, 4.610210000e-17, 7.337639645e-16, 3.348557496e-16, 1.736562533e-16, 1.915871639e-17, 1.931338908e-16, 1.974415886e-17, 2.054336404e-16,  // 61-70
  2.003109343e-16, 1.934498511e-16, 1.319008056e-15, 9.450540000e-16, 1.087303394e-16, 1.665148262e-15, 1.202328448e-15, 1.128331221e-15, 8.531572269e-16, 7.408838005e-16,  // 71-80
  1.158662950e-15, 1.374180788e-15, 8.566677765e-16, 9.492940536e-17, 1.870350513e-17, 9.491622217e-17, 8.928257029e-17, 9.336061303e-17, 9.334736660e-17, 4.790157134e-16,  // 81-90
  1.305399034e-16, 2.169990868e-17, 8.372799536e-17, 8.658930267e-17, 8.911500996e-16, 8.860103256e-17, 4.406306122e-16, 6.993638266e-16, 1.349886337e-16, 3.034949291e-16  // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP5[101] ={0,
 6.894128600e-17, 2.232031723e-17, 2.477820000e-17, 2.177967522e-15, 1.284829196e-15, 1.823111356e-16, 1.555489343e-16, 1.335924448e-16, 2.114259423e-16, 1.667587557e-16,  // 1-10
 5.490626359e-17, 6.721220587e-17, 2.797861959e-15, 1.006989227e-16, 4.970209967e-16, 2.167639187e-16, 2.319436556e-16, 1.141095767e-18, 1.660590000e-18, 1.909325187e-18,  // 11-20
 1.137340000e-18, 1.172666277e-18, 9.208052526e-19, 1.423664003e-18, 9.502898022e-19, 9.157542056e-19, 1.055368898e-18, 9.370568491e-19, 1.041778157e-18, 9.638088116e-19,  // 21-30
 1.000486545e-18, 8.890635605e-19, 8.866911360e-19, 8.345076503e-19, 1.886536486e-16, 1.813120000e-16, 6.659892285e-19, 6.086335346e-19, 4.868786350e-19, 1.636754844e-16,  // 31-40
 1.673069643e-16, 1.638573610e-16, 1.618391268e-16, 2.686910000e-19, 1.813800000e-19, 2.102345989e-16, 1.318039263e-16, 2.077093130e-16, 1.588911975e-16, 1.515458297e-16,  // 41-50
 1.080904532e-16, 1.490120000e-16, 2.463071192e-16, 5.939133658e-16, 3.046264790e-16, 1.142460000e-19, 1.562245161e-16, 9.001027117e-16, 1.078667444e-16, 7.294347496e-18,  // 51-60
 1.358154769e-16, 8.509795172e-16, 4.722000000e-20, 1.653557513e-16, 9.741638336e-17, 8.121141361e-16, 2.060180430e-16, 1.293242400e-15, 2.817685189e-15, 1.295140057e-15,  // 61-70
 1.325219469e-15, 1.361391001e-15, 1.166226381e-16, 1.078650000e-16, 8.154952090e-16, 1.184682586e-17, 1.781757642e-17, 1.042830884e-16, 9.653409394e-17, 9.620084329e-17,  // 71-80
 9.927188930e-17, 1.933919795e-17, 8.642866554e-17, 1.300494398e-15, 1.573438944e-15, 1.595960704e-15, 1.963662451e-17, 1.824165751e-15, 1.536213562e-15, 1.230473199e-16,  // 81-90
 4.609653396e-16, 1.503608663e-15, 9.130139990e-16, 8.950731342e-16, 8.843355368e-17, 9.314765075e-16, 1.311786759e-16, 9.206849025e-17, 3.169463543e-16, 2.154753688e-15  // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP6[101] = {0,
  5.245179284e+00, 3.941069622e+00, 7.595470000e+00, 1.556100268e+01, 3.958715820e+00, 1.006439332e+01, 3.947506567e+00, 3.911947451e+00, 3.960123395e+00, 3.995405309e+00,  // 1-10
  4.165369684e+00, 4.154784462e+00, 4.243106173e+00, 4.326072224e+00, 4.520737868e+00, 5.804915543e+00, 7.148125826e+00, 1.979899376e+00, 1.619050000e+00, 3.020461557e+00,  // 11-20
  1.570400000e+00, 1.595930653e+00, 1.592295792e+00, 3.539697537e+00, 1.823123604e+00, 1.823008934e+00, 3.108054801e+00, 1.802053625e+00, 3.315832610e+00, 3.438022820e+00,  // 21-30
  1.788357076e+00, 1.779735777e+00, 1.782694477e+00, 1.778454849e+00, 4.288820945e+00, 2.573830000e+00, 1.754405892e+00, 1.749703489e+00, 1.739317589e+00, 3.536923800e+00,  // 31-40
  4.992480078e+00, 5.353762609e+00, 5.983872882e+00, 1.659260000e+00, 1.637810000e+00, 7.594207497e+00, 2.760872072e+00, 2.996427458e+00, 6.765238189e+00, 3.358446915e+00,  // 41-50
  1.895332668e+00, 3.235160000e+00, 3.117412878e+00, 2.561242590e+00, 1.080139229e+01, 1.513050000e+00, 1.944567866e+00, 1.922906186e+00, 1.732857547e+00, 1.666739664e+00,  // 51-60
  1.735584396e+00, 1.700970330e+00, 1.493940000e+00, 1.789491004e+00, 1.668966729e+00, 1.722315051e+00, 1.564155879e+00, 1.679770510e+00, 1.660734791e+00, 1.650941092e+00,  // 61-70
  1.646594633e+00, 1.646361209e+00, 1.596899505e+00, 1.575670000e+00, 1.595800331e+00, 1.598706510e+00, 2.267574396e+00, 1.571022960e+00, 1.556511213e+00, 1.547150547e+00,  // 71-80
  1.539905529e+00, 2.445886590e+00, 1.521666588e+00, 1.514607927e+00, 2.418992922e+00, 1.489341220e+00, 1.445770175e+00, 1.477753418e+00, 1.473313627e+00, 1.034799940e+01,  // 81-90
  2.720221569e+00, 2.423181552e+00, 1.433800031e+00, 1.424989075e+00, 1.398959178e+00, 1.397503726e+00, 2.862849596e+00, 1.377034589e+00, 9.782799214e+00, 1.348300032e+00  // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP7[101] ={0,
 3.999571467e+00, 3.991128282e+00, 3.855930000e+00, 3.954881255e+00, 1.209804865e+01, 3.984648146e+00, 1.159847718e+01, 1.208416029e+01, 7.793410191e+00, 7.050783546e+00,  // 1-10
 1.860829199e+00, 1.805395915e+00, 5.601066041e+00, 1.285317248e+01, 2.982331588e+00, 1.387872806e+01, 1.234603360e+01, 1.341537351e+01, 9.537270000e+00, 4.399859528e+00,  // 11-20
 7.887250000e+00, 1.010638153e+01, 5.125426065e+00, 4.529968522e+00, 1.295225813e+01, 1.125545587e+01, 3.469375224e+00, 9.019427119e+00, 4.422801993e+00, 4.231377923e+00,  // 21-30
 4.487670198e+00, 3.377601937e+00, 3.060634263e+00, 2.916013798e+00, 3.277148894e+00, 3.204930000e+00, 2.941802865e+00, 2.233026370e+00, 2.626492823e+00, 4.442039033e+00,  // 31-40
 4.924387077e+00, 7.722486735e+00, 6.373463850e+00, 1.126070000e+01, 4.865950000e+00, 1.666437783e+00, 1.165792401e+01, 1.832865314e+00, 3.197863589e+00, 1.108841018e+01,  // 41-50
 8.586125523e+00, 6.676730000e+00, 1.759586815e+00, 1.779094293e+00, 1.606067697e+00, 1.716220000e+00, 3.714794374e+00, 1.684962847e+00, 3.694058142e+00, 3.883036642e+00,  // 51-60
 3.616573408e+00, 2.616627730e+00, 1.724970000e+00, 3.214294756e+00, 6.048180604e+00, 1.644134456e+00, 1.645297046e+00, 1.638457907e+00, 1.632603379e+00, 1.628448343e+00,  // 61-70
 1.632630475e+00, 1.635873458e+00, 1.825704957e+00, 2.491660000e+00, 2.578788354e+00, 1.564697536e+00, 1.957756845e+00, 1.753512815e+00, 2.248102578e+00, 2.608650253e+00,  // 71-80
 1.775440582e+00, 1.577663847e+00, 2.371177051e+00, 2.470000483e+00, 1.508411013e+00, 2.464240407e+00, 2.506389292e+00, 2.470932595e+00, 2.409396196e+00, 4.479899308e+00,  // 81-90
 1.455793655e+00, 1.441149845e+00, 2.479382371e+00, 2.453619326e+00, 2.383185233e+00, 2.503014659e+00, 4.272318275e+00, 2.713400801e+00, 1.391715512e+00, 2.623357264e+00  // 91-100
};

const G4double
  G4RayleighAngularGenerator::PP8[101] = {0,
  3.999988492e+00, 3.007894874e+00, 5.946860000e+00, 7.152304101e+00, 6.130363514e+00, 1.208178632e+01, 1.037802325e+01, 9.199988173e+00, 1.172397112e+01, 1.207012659e+01,  // 1-10
  1.115856859e+01, 7.827803195e+00, 2.641126114e+00, 4.312456794e+00, 1.198258924e+01, 2.176176802e+00, 2.039327000e+00, 7.141859323e+00, 5.280700000e+00, 4.469270053e+00,  // 11-20
  5.878090000e+00, 5.532784283e+00, 5.963337902e+00, 4.314887550e+00, 5.841459409e+00, 5.679735328e+00, 4.361061890e+00, 5.202026076e+00, 4.135817334e+00, 4.133917331e+00,  // 21-30
  4.551995537e+00, 4.677756013e+00, 4.528279561e+00, 4.498802330e+00, 1.774563576e+00, 1.862200000e+00, 4.622134764e+00, 4.695173427e+00, 5.128574495e+00, 1.782470135e+00,  // 31-40
  1.727732750e+00, 1.724150271e+00, 1.709518222e+00, 6.334160000e+00, 7.926650000e+00, 1.159184448e+01, 1.895174961e+00, 1.154752444e+01, 1.645989249e+00, 1.687530363e+00,  // 41-50
  2.409664359e+00, 1.654010000e+00, 1.134686144e+01, 4.793719850e+00, 9.632614374e+00, 7.137710000e+00, 1.705278976e+00, 3.589745289e+00, 2.650211489e+00, 1.847594729e+00,  // 51-60
  2.057911711e+00, 2.980049185e+00, 1.086000000e+01, 1.654657393e+00, 2.823690929e+00, 2.581116822e+00, 1.824902805e+00, 1.610400116e+00, 1.584768888e+00, 1.561632819e+00,  // 61-70
  1.536232590e+00, 1.520253937e+00, 2.701107185e+00, 2.786000000e+00, 2.677490471e+00, 1.613510248e+00, 1.574877874e+00, 2.478321094e+00, 2.578973774e+00, 2.530183017e+00,  // 71-80
  2.454130254e+00, 1.522182263e+00, 2.585541261e+00, 1.743698112e+00, 1.507618128e+00, 1.517673206e+00, 1.487198678e+00, 1.482431696e+00, 1.808266084e+00, 1.468787657e+00,  // 81-90
  4.518186694e+00, 1.690796964e+00, 2.458974369e+00, 2.416442577e+00, 2.505591266e+00, 2.315596171e+00, 1.421452131e+00, 2.563262017e+00, 5.323940741e+00, 2.366288037e+00  // 91-100
};
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo.... ....oooOO0OOooo....
