// Copyright © 2019, Oracle and/or its affiliates.
package ociauth

// Do not edit this file. This is based on standard OCI GO SDK format

// Claim is a representation of a JWT claim
type InternalClaim struct {
	Issuer string `json:"issuer"`
	Key    string `json:"key"`
	Value  string `json:"value"`
}

// These constants define types of principal
const (
	PrincipalTypeUser     = "user"
	PrincipalTypeInstance = "instance"
)

// This constant defines the Principal type key
var (
	ClaimPrincipalType = "ptype"
)

// Claims represents a collection of JWT claims
type InternalClaims map[string][]InternalClaim

// FromClaims takes in a list of claims and coverts it to InternalClaims
func FromClaims(claimList []Claim) InternalClaims {
	outputMap := make(map[string][]InternalClaim)

	for _, item := range claimList {
		internalClaim := FromClaim(item)
		outputMap[internalClaim.Key] = append(outputMap[internalClaim.Key], internalClaim)

	}

	return outputMap
}

// FromClaims takes in a claim and coverts it to an InternalClaim
func FromClaim(claim Claim) InternalClaim {
	return InternalClaim{
		Issuer: *claim.Issuer,
		Key:    *claim.Key,
		Value:  *claim.Value,
	}
}

// GetSingleClaim returns single claim given a claim type.
func (c InternalClaims) GetSingleClaim(key string) InternalClaim {
	claims := c[key]
	if len(claims) > 0 {
		return claims[0]
	}
	return InternalClaim{}
}

// GetString returns the claim value given a claim type.
func (c InternalClaims) GetString(key string) string {
	claim := c.GetSingleClaim(key)
	return claim.Value
}
