<?php # (jEdit options) :folding=explicit:collapseFolds=1:
/*****************************************************************************
    This is a framework for a model-view-controller (MVC) architecture.
    The actual control systems are driven by public_html/index.php
    The work is done by "delegate" scripts in pages/
    
    Model       all data is stored in $_SESSION
    View        generated by display() in pages/xxx.php; takes no actions
    Controller  handlers in pages/xxx.php; change application state
    
{{{ Data structure of a single "page", stored in $_SESSION['pages']

    delegate        the name of a PHP script with display() and one or more
                    handlers, relative to MP_BASE_DIR/pages/
    context         an array of info for use by display()
    handlers[]      an array of arrays. First key is event ID.
                    Second keys (i.e. for each handler) are:
        funcName    the name of an event handler in the current delegate
        funcArgs    array of arguments to be passed to the handler (maybe empty)
}}}
{{{ Division of responsibilities

    index.php VS xxx.php in pages/
        index.php is creates the framework; loads and saves the session; etc.
        xxx.php defines a display() function and zero or more event handlers.
    
    display() vs. event handlers
        display() creates an HTML pages and registers handlers for events.
        It MUST NOT change the application "state" or do any work that shouldn't
        be repeated multiple times.
        Event handlers do not create any HTML output, but are responsible for
        doing work (either themselves or by launching a background job) and for
        guiding application flow by changing the current delegate page using
        pageGoto(), pageCall(), and pageReturn().
}}}

*****************************************************************************/
// Someone else MUST have defined this before including us!
if(!defined('MP_BASE_DIR')) die("MP_BASE_DIR is not defined.");
require_once(MP_BASE_DIR.'/lib/sessions.php');

// These functions are for use in event handlers only.
#{{{ pageGoto - sets the current delegate without changing page stack depth
############################################################################
/**
* $pageName is the path to the delegate script, relative to pages/
* $context is information used by the delegate's display() function
*   to specify e.g. which model to display, etc.
* This function swaps out the current delegate for the specified one.
*/
function pageGoto($pageName, $context = null)
{
    pageReturn();                   // pops the old delegate
    pageCall($pageName, $context);  // pushes the new delegate
}
#}}}########################################################################

#{{{ pageCall - sets a new delegate as a "subroutine call" of the current one
############################################################################
/**
* Like pageGoto(), but the new delegate is pushed onto a stack above
* the old one, so it can be popped off by pageReturn().
* Return values/handlers are not supported; see index.php for the rationale.
*/
function pageCall($pageName, $context = null)
{
    $newPage = array(
        'delegate' => $pageName,
        'context' => $context,
        'handlers' => array()
    );
    $_SESSION['pages'][] = $newPage; // equiv. to array_push()
}
#}}}########################################################################

#{{{ pageReturn - returns to the delegate that called the current one
############################################################################
/**
* Undoes the actions of pageCall() and returns to the old delegate.
* Return values/handlers are not supported; see index.php for the rationale.
*/
function pageReturn()
{
    if(count($_SESSION['pages']) >= 1)
        array_pop($_SESSION['pages']);
}
#}}}########################################################################

#{{{ get/setContext - communicate with display() by changing its $context
############################################################################
function getContext()
{
    $page = end($_SESSION['pages']);
    return $page['context'];
}

function setContext($context_or_key, $value = null)
{
    end($_SESSION['pages']);
    $key = key($_SESSION['pages']);
    $page =& $_SESSION['pages'][$key];
    
    if(func_num_args() == 1 && is_array($context_or_key))
        $page['context'] = $context_or_key;
    else
        $page['context'][$context_or_key] = $value;
}
#}}}########################################################################


// These functions are for use in display() only.
#{{{ makeEventURL - creates a URL for <a> tags that launches an event via GET
############################################################################
/**
* Returns a string like
*   "index.php?session=123ABC&eventID=456789"
*/
function makeEventURL($funcName /* and other function args, as desired */)
{
    $funcArgs = func_get_args();
    array_shift($funcArgs); // remove $funcName
    $id = addEventHandler($funcName, $funcArgs);
    // What's the difference b/t this and $_SERVER[SCRIPT_NAME] ? None I can see.
    // We use basename() to get "index.php" instead of the full path,
    // which is subject to corruption with URL forwarding thru kinemage.
    return basename($_SERVER['PHP_SELF'])."?$_SESSION[sessTag]&eventID=$id";
}
#}}}########################################################################

#{{{ makeEventForm - creates a <form> that launches an event via POST
############################################################################
/**
* Returns a string like
*   <form method='post' enctype='multipart/form-data' action='index.php'>
*   <input type='hidden' name='session' value='123ABC'>
*   <input type='hidden' name='eventID' value='456789'>
*/
function makeEventForm($funcName /* and other function args, as desired */)
{
    $funcArgs = func_get_args();
    array_shift($funcArgs); // remove $funcName
    $id = addEventHandler($funcName, $funcArgs);
    $s = "<form method='post' ";
    // This is needed for forms that have file uploads,
    // but it doesn't do any harm for other forms (?)
    $s .= "enctype='multipart/form-data' ";
    // Better to do this via the button's onclick= handler
    //if($onSubmit)       $s .= "onsubmit='return $onSubmit' ";
    // What's the difference b/t this and $_SERVER[SCRIPT_NAME] ? None I can see.
    // We use basename() to get "index.php" instead of the full path,
    // which is subject to corruption with URL forwarding thru kinemage.
    $s .= "action='".basename($_SERVER['PHP_SELF'])."'>\n";
    $s .= postSessionID();
    $s .= "<input type='hidden' name='eventID' value='$id'>\n";
    return $s;
}
#}}}########################################################################


// These functions are utilities for the driving script (index.php)
#{{{ addEventHandler - registers a handler for the current page delegate
############################################################################
/**
* Returns the eventID for the handler.
*/
function addEventHandler($funcName, $funcArgs)
{
    end($_SESSION['pages']);
    $key = key($_SESSION['pages']);
    $page =& $_SESSION['pages'][$key];
    
    $eid = ++$_SESSION['currEventID']; // using preinc works even when the var is unset
    if($_SESSION['currEventID'] >= 1<<30) $_SESSION['currEventID'] = 1;
    
    $page['handlers'][$eid] = array(
        'funcName' => $funcName,
        'funcArgs' => $funcArgs
    );
    
    return $eid;
}
#}}}########################################################################

#{{{ clearEventHandlers - removes existing handlers and reseeds the starting ID
############################################################################
/**
* Removes all registered event handlers for the current delegate.
*/
function clearEventHandlers()
{
    end($_SESSION['pages']);
    $key = key($_SESSION['pages']);
    $page =& $_SESSION['pages'][$key];
    
    $page['handlers'] = array();
}
#}}}########################################################################

#{{{ makeDelegateObject - instantiates the current delegate class
############################################################################
/**
* Instantiates the current delegate class. In order for this to work,
* the class name must be the same as the file name, with ".php" stripped off
* and "_delegate" appended. Thus, for "pages/reduce_setup.php", we get:
*   class reduce_setup_delegate extends BasicDelegate { ... }
*/
function makeDelegateObject()
{
    $page = end($_SESSION['pages']); // not a ref; read only
    if(! $page['delegate']) mpControllerDie("No page delegate defined"); // else cryptic error from require_once()
    
    // FUNKY: Must use require_once() b/c UI delegate (below) may be the same,
    // or may be different, and we can't redefine classes or functions.
    $page_file = MP_BASE_DIR."/pages/$page[delegate]";
    if(!file_exists($page_file)) mpControllerDie("Cannot find page delegate file at $page_file");
    require_once($page_file);
    
    $page_class = basename($page['delegate'], '.php') . '_delegate';
    if(! class_exists($page_class)) mpControllerDie("Class $page_class not defined in $page_file");
    // Now we instantiate a variably-named class:
    $delegate = new $page_class();
    
    // Now we set elements of $context as instance vars, using variable variables
    // I'm not sure this is safe in MolProbity, as we didn't design for it.
    //if(is_array($page['context'])) foreach($page['context'] as $k => $v)
    //    $delegate->$k = $v;
    
    return $delegate;
}
#}}}########################################################################

#{{{ mpControllerDie - prints helpful error msg and wipes the session
############################################################################
/**
* For unrecoverable errors (like misnamed target pages), this is easier than
* having to kill the browser and come back in!
*/
function mpControllerDie($msg = "")
{
    echo "<html><body><b>$msg</b><p><pre>";
    echo '$_REQUEST = ';
    var_export($_REQUEST);
    echo "\n\n\n";
    echo '$_SESSION = ';
    var_export($_SESSION);
    echo "</pre></body></html>";
    //session_destroy(); # only necessary if using cookies for sessID
    die($msg); // in case this is logged somewhere?
}
#}}}########################################################################

#{{{ a_function_definition - sumary_statement_goes_here
############################################################################
/**
* Documentation for this function.
*/
//function someFunctionName() {}
#}}}########################################################################


// This is the base class that all delegate classes should extend.
// It implements certain common event handlers that work with core/mpNavBar().
class BasicDelegate {

    /** All page classes must implement display() to render a web page. */
    function display()
    { die("All page classes must override display()"); }
    
    function pageHeader($title, $active = "none", $refresh = "")
    { return mpPageHeader($title, $active, $refresh, $this->headContent()); }
    
    #{{{ headContent - scripts, etc that should appear in <HEAD>
    ############################################################################
    function headContent()
    {
        $s = "";
        return $s;
    }
    #}}}########################################################################
    
    function pageFooter()
    { return mpPageFooter(); }

    //{{{ onGoto, onCall, onReturn
    /**
    * $arg is the name of the page to go to.
    */
    function onGoto($page, $ctx = null)
    { pageGoto($page, $ctx); }
    
    /**
    * $arg is the name of the page to call.
    */
    function onCall($page, $ctx = null)
    { pageCall($page, $ctx); }
    
    function onReturn()
    { pageReturn(); }
    //}}}

    function onNavBarGoto($arg) { $this->onGoto($arg); }
    function onNavBarCall($arg) { $this->onCall($arg); }
}
?>
