<?php # (jEdit options) :folding=explicit:collapseFolds=1:
/*****************************************************************************
    Implements lab notebook functions for MolProbity.
    
    The labbook data structure is an associative array where
    the first index is an integer "entry number", and
    the seconds index is one of the following:
    
    'title'     a one-line title for this entry
    'ctime'     the time the entry was created (seconds since the Epoch)
    'modtime'   the time the entry was created (seconds since the Epoch)
    'model'     which model this relates to, or the empty string/undefined
                This may also be a bar-separated list (see below).
    'keywords'  a bar-separated (|) list of keywords for classifying entries
                Use explode() and implode() to convert to/from an array.
                Stored as a string b/c they're easier to transmit in forms.
                Common keywords:
                    auto        Automatically generated by MP3 code
    'thumbnail' a filename from public_html/img/ to represent this entry.
                The image must be square, but clients will specify the size.
                Typical size is 20 x 20 pixels.
    'entry'     the user's comments, with HTML formatting
    
*****************************************************************************/
require_once(MP_BASE_DIR.'/lib/core.php');
require_once(MP_BASE_DIR.'/lib/timezones.php');
require_once(MP_BASE_DIR.'/lib/strings.php');

#{{{ openLabbook - loads labbook from disk and returns array
############################################################################
function openLabbook()
{
    $file = $_SESSION['dataDir']."/".MP_DIR_SYSTEM."/labbook";
    
    // Read in notebook data, if present
    if($fp = @fopen($file, "rb"))
    {
        // Without this, we read the wrong number of bytes and so particularly
        // additions to the book will be truncated!
        clearstatcache();
        $bookData = fread($fp, filesize($file));
        $book = unserialize($bookData);
        if(!$book) echo "Unable to unserialize '$bookData'";
        @fclose($fp);
    }
    else $book = array(); // empty array
    
    return $book;
}
#}}}########################################################################

#{{{ saveLabbook - writes labbook array to disk
############################################################################
/**
* Returns false on failure
*/
function saveLabbook($bookData)
{
    saveLabbookHTML($bookData);
    
    // Write the notebook data
    $file = $_SESSION['dataDir']."/".MP_DIR_SYSTEM."/labbook";
    if($fp = @fopen($file, "wb"))
    {
        $r = fwrite($fp, serialize($bookData));
        @fclose($fp);
        return $r;
    }
    else return false;
}
#}}}########################################################################

#{{{ saveLabbookHTML - writes human-readable labbook data to disk
############################################################################
/**
* This function is called automatically by saveLabbook().
* The output won't be ideal, because paths to linked items will be broken.
*/
function saveLabbookHTML($bookData)
{
    $file = $_SESSION['dataDir']."/".MP_DIR_SYSTEM."/labbook.html";

    // Write the notebook data
    if($fp = @fopen($file, "wb"))
    {
        fwrite($fp, mpPageHeader("Lab notebook"));
        fwrite($fp, "<a name='top'>\n<ul>\n");
        foreach($bookData as $num => $entry)
        {
            $title = $entry['title'];
            if($title == "") $title = "(no title)";
            fwrite($fp, "<li><a href='#entry$num'</a>$title [".formatDayTime($entry['modtime'])."]</li>\n");
        }
        fwrite($fp, "</ul>\n<br clear='all' />\n");
        foreach($bookData as $num => $entry)
        {
            fwrite($fp, "<hr>\n");
            fwrite($fp, "<a name='entry$num'>\n");
            fwrite($fp, formatLabbookEntry($entry));
            fwrite($fp, "</a>\n");
            fwrite($fp, "<p><a href='#top'>Top</a>\n");
        }
        fwrite($fp, mpPageFooter());

        @fclose($fp);
        return true;
    }
    else return false;
}
#}}}########################################################################

#{{{ newLabbookEntry - creates a new entry
############################################################################
function newLabbookEntry($model = "", $keywords = "")
{
    return array(
        'ctime'     => time(),
        'modtime'   => time(),
        'model'     => $model,
        'keywords'  => $keywords,
        'thumbnail' => 'clear_1x1.gif',
    );
}
#}}}########################################################################

#{{{ addLabbookEntry - all-in-one automatic entry for background jobs
############################################################################
/**
* Returns the entry number of the new entry.
*/
function addLabbookEntry($title, $text, $model = "", $keywords = "", $thumbnail = null)
{
    $labbook = openLabbook();
    $entry = newLabbookEntry($model, $keywords);
    $entry['title'] = $title;
    $entry['entry'] = $text;
    if($thumbnail) $entry['thumbnail'] = $thumbnail; // else it stays the default
    $entryNum = count($labbook);
    $labbook[ $entryNum ] = $entry;
    saveLabbook($labbook);
    
    return $entryNum;
}
#}}}########################################################################

#{{{ formatLabbookEntry - returns HTML rendering of one entry
############################################################################
/**
* $entry - one of the entries from the labbook, i.e. $labbook[N]
*   This should be a single array with keys 'title', 'ctime', etc.
*
* Returns a formatted chunk of HTML that you can insert in your pages.
*/
function formatLabbookEntry($entry)
{
    $s = "";
    $s .= "<b>".$entry['title']."</b>\n";
    $s .= "<br><table border=0 cellpadding=0 cellspacing=3>\n";
    $s .= "<tr align=left valign=top><td><small>Entry begun:</small></td><td><small>".formatDayTime($entry['ctime'])."</small></td></tr>\n";
    $s .= "<tr align=left valign=top><td><small>Last modified:</small></td><td><small>".formatDayTime($entry['modtime'])."</small></td></tr>\n";
    // At the moment, these fields are more for internal use than for the user.
    //if($entry['model'] != "")
    //    $s .= "<tr align=left valign=top><td><small>Model:</small></td><td><small>".$entry['model']."</small></td></tr>\n";
    //if($entry['keywords'] != "")
    //    $s .= "<tr align=left valign=top><td><small>Keywords:</small></td><td><small>".$entry['keywords']."</small></td></tr>\n";
    $s .= "</table>";
    $s .= "<p>".$entry['entry']."\n";
    
    return $s;
}
#}}}########################################################################

#{{{ formEditLabbook - produces a form for creating/editing an entry
############################################################################
/**
* $entry - one of the entries from the labbook, i.e. $labbook[N]
*   This should be a single array with keys 'title', 'ctime', etc.
*   For a new entry, this should be newLabbookEntry().
* $width, $height - the size of the text entry area, in characters.
*
* Returns a formatted chunk of HTML that you can insert in your pages,
* inside a <FORM> element. No submit button is provided.
*
* When submitted, the form will define an array called 'labbookEntry'
* with keys 'title', 'ctime', etc. Look for it in $_REQUEST.
*/
function formEditLabbook($entry, $width = 90, $height = 30)
{
    // HTML entities are interpretted once by the browser inside the form,
    // so title and entry have to be "double protected" to save correctly.
    $s = "";
    $s .= "<input type='text' size='$width' name='labbookEntry[title]' value='".htmlspecialchars($entry['title'])."'>\n";
    $s .= "<input type='hidden' name='labbookEntry[ctime]' value='$entry[ctime]'>\n";
    $s .= "<input type='hidden' name='labbookEntry[modtime]' value='$entry[modtime]'>\n";
    $s .= "<input type='hidden' name='labbookEntry[model]' value='$entry[model]'>\n";
    $s .= "<input type='hidden' name='labbookEntry[keywords]' value='$entry[keywords]'>\n";
    $s .= "<input type='hidden' name='labbookEntry[thumbnail]' value='$entry[thumbnail]'>\n";
    
    $s .= "<br><table border=0 cellpadding=0 cellspacing=3>\n";
    $s .= "<tr align=left valign=top><td>Entry begun:</td><td>".formatDayTime($entry['ctime'])."</td></tr>\n";
    $s .= "<tr align=left valign=top><td>Last modified:</td><td>".formatDayTime($entry['modtime'])."</td></tr>\n";
    $s .= "</table>";
    
    if($entry['model'] != "")
        $s .= "<br>Model: ".implode(", ", explode("|", $entry['model']))."\n";
    if($entry['keywords'] != "")
        $s .= "<br>Keywords: ".implode(", ", explode("|", $entry['keywords']))."\n";

    $s .= "<br><textarea name='labbookEntry[entry]' cols='$width' rows='$height'>".htmlspecialchars($entry['entry'])."</textarea>\n";
    
    return $s;
}
#}}}########################################################################

#{{{ a_function_definition - sumary_statement_goes_here
############################################################################
/**
* Documentation for this function.
*/
//function someFunctionName() {}
#}}}########################################################################
?>
