//**************************************************************************************************
//                                           PrcBase.hpp                                           *
//                                          -------------                                          *
// Description : Provides a wrapper for the wxProcess class adding a bit of extra desirable        *
//               functionality.                                                                    *
// Started     : 2004-01-29                                                                        *
// Last Update : 2020-09-30                                                                        *
// Copyright   : (C) 2004-2020 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#ifndef PRCBASE_HPP
#define PRCBASE_HPP

// Application Includes

#include "TypeDefs.hpp"
#include "SysScan.hpp"
#include "utility/TextCtrl.hpp"
#include "utility/StrUtils.hpp"

// wxWidgets Includes

#include <wx/process.h>

// Local Constant Declarations

#define DEF_LOG_FILE  "./prcbase.log"  // Default log file path and name

//**************************************************************************************************

class PrcBase : protected wxProcess
{
  private :

    wxFileName    m_ofnNameBin;  // The file name of the binary to be executed
    wxFileName    m_ofnNameLog;  // The log file name for the textual process output
    wxString      m_osArgLst;    // The argument list appended to the binary name
    int           m_iFlags;      // Flags which may be used in the function wxExecute( )
    wxExecuteEnv  m_oEnv;        // Structure containing working directory and environment variables
    wxString      m_osErrMsg;    // Error message
    int           m_iPid;        // The process ID number
    int           m_iExitCode;   // The process exit code

  protected :

    bool  bLogOutput( void );

  public :

    explicit  PrcBase( int iFlags=wxPROCESS_REDIRECT );
             ~PrcBase( );

    bool  bBinExists( void );

    bool  bIsExec( void ) { return( m_iPid>0 ? true : false ); }
    bool  bIsOk  ( void ) { return( m_iExitCode==0 && m_osErrMsg.IsEmpty( ) ); }

    bool  bSetBinFile( const wxString & rosFileName );
    bool  bSetArgLst ( const wxString & rosArgLst   );
    bool  bSetCwd    ( const wxString & rosCwd      );
    bool  bSetLogFile( const wxString & rosFileName );

    void  SetErrMsg( const wxString & rosErrMsg ) { if(m_osErrMsg.IsEmpty()) m_osErrMsg=rosErrMsg; }
    void  ClrErrMsg( void )                       { m_osErrMsg.Empty( ); }

    const wxFileName & rofnGetBinFile ( void ) { return( m_ofnNameBin  ); }
    const wxString   &  rosGetArgLst  ( void ) { return( m_osArgLst    ); }
    const wxString   &  rosGetCwd     ( void ) { return( m_oEnv.cwd    ); }
    const wxFileName & rofnGetLogFile ( void ) { return( m_ofnNameLog  ); }
    const wxString   &  rosGetErrMsg  ( void ) { return( m_osErrMsg    ); }
          int             iGetExitCode( void ) { return( m_iExitCode   ); }

    const wxString & rosEscSpaceChrs( const wxString & rosPath );

    virtual  bool  bExecAsync( void );
    virtual  bool  bExecSync ( void );
    virtual  bool  bKill     ( void );

    bool  bDelLogFile( void );

    void  PrintIO ( TextCtrl & roTxtCtl );
    void  PrintCmd( TextCtrl & roTxtCtl );
    void  PrintRsp( TextCtrl & roTxtCtl );

    // The following functions are intended for debugging
    virtual  void  Print( const wxString & rosPrefix="  " );
             void  PrintOutput( void );

    // Event handlers
    void  OnTerminate( int iPid, int iStatus );
};

//**************************************************************************************************

#endif // PRCBASE_HPP
