/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#define _GNU_SOURCE
#define _FILE_OFFSET_BITS 64

#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/uio.h>

#include <glib.h>

#include "rbccommand.h"
#include "rbcprivate.h"


#define VENDOR_ID   "Linux   "
#define PRODUCT_ID  "Endpoint        "
#define PRODUCT_REV "0001"


typedef enum
{
  INQUIRY         = 0x12,
  MODE_SELECT     = 0x15,
  MODE_SENSE      = 0x1a,
  READ            = 0x28,
  READ_CAPACITY   = 0x25,
  START_STOP_UNIT = 0x1b,
  TEST_UNIT_READY = 0x00,
  VERIFY          = 0x2f,
  WRITE           = 0x2a,
  WRITE_BUFFER    = 0x3b
} Command;

typedef enum
{
  GOOD                 = 0x0,
  CHECK_CONDITION      = 0x2,
  CONDITION_MET        = 0x4,
  BUSY                 = 0x8,
  RESERVATION_CONFLICT = 0x18,
  COMMAND_TERMINATED   = 0x22
} Status;

typedef enum
{
  NO_SENSE        = 0x00,
  RECOVERED_ERROR = 0x01,
  NOT_READY       = 0x02,
  MEDIUM_ERROR    = 0x03,
  ILLEGAL_REQUEST = 0x04,
  UNIT_ATTENTION  = 0x05,
  DATA_PROTECT    = 0x06,
  BLANK_CHECK     = 0x08,
  COPY_ABORTED    = 0x0a,
  ABORTED_COMMAND = 0x0b,
  VOLUME_OVERFLOW = 0x0d,
  MISCOMPARE      = 0x0e
} SenseKey;


typedef gboolean (*CommandHandler) (RBCDisk   *disk,
				    guint8    *cmd,
				    RBCBuffer *buffer,
				    guint8    *sense);


static gboolean handle_inquiry         (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_mode_select     (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_mode_sense      (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_read            (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_read_capacity   (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_start_stop_unit (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_test_unit_ready (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_verify          (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_write           (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);
static gboolean handle_write_buffer    (RBCDisk   *disk,
					guint8    *cmd,
					RBCBuffer *buffer,
					guint8    *sense);


static inline void
fill_sense_buffer (guint8   *sense,
                   Status    status,
                   SenseKey  key,
		   gint      asc,
		   gint      asq)
{
  g_return_if_fail (sense != NULL);

  sense[0] = status;
  sense[1] = key;
  sense[2] = asc;
  sense[3] = asq;
}


gboolean
rbc_disk_command (RBCDisk   *disk,
		  guint8    *cmd,
		  RBCBuffer *buffer,
		  guint8    *sense)
{
  CommandHandler handler;
  RBCBuffer      sg_buffer;
  gboolean       process_sg = TRUE;
  gboolean       no_empty = TRUE;

  g_return_val_if_fail (disk != NULL, FALSE);
  g_return_val_if_fail (cmd != NULL, FALSE);
  g_return_val_if_fail (buffer != NULL, FALSE);
  g_return_val_if_fail (sense != NULL, FALSE);

  switch (cmd[0])
    {
    case INQUIRY:
      handler = handle_inquiry;
      break;

    case MODE_SELECT:
      handler = handle_mode_select;
      no_empty = FALSE;
      break;

    case MODE_SENSE:
      handler = handle_mode_sense;
      no_empty = FALSE;
      break;

    case READ:
      handler = handle_read;
      process_sg = FALSE;
      break;

    case READ_CAPACITY:
      handler = handle_read_capacity;
      break;

    case START_STOP_UNIT:
      handler = handle_start_stop_unit;
      no_empty = FALSE;
      break;

    case TEST_UNIT_READY:
      handler = handle_test_unit_ready;
      no_empty = FALSE;
      break;
   
    case VERIFY:
      handler = handle_verify; 
      process_sg = FALSE;
      break;

    case WRITE:
      handler = handle_write;
      process_sg = FALSE;
      break;

    case WRITE_BUFFER:
      handler = handle_write_buffer;
      no_empty = FALSE;
      break;

    default:
      fill_sense_buffer (sense, CHECK_CONDITION, ILLEGAL_REQUEST, 0x20, 0);
      return FALSE;
      break;
    }

  if (no_empty && buffer->type == RBC_BUFFER_EMPTY)
    {
      fill_sense_buffer (sense, CHECK_CONDITION, ILLEGAL_REQUEST, 0x20, 0);
      return FALSE;
    }

  if (process_sg)
    {
      if (buffer->type == RBC_BUFFER_PAGE_TABLE)
	{
	  RBCPage *page = buffer->data;

	  rbc_buffer_fill_from_page (&sg_buffer, page);
	  buffer = &sg_buffer;
	}
    }

  return handler (disk, cmd, buffer, sense);
}

static gboolean
handle_inquiry (RBCDisk   *disk,
		guint8    *cmd,
		RBCBuffer *buffer,
		guint8    *sense)
{
  guint8 *data = buffer->data;

  memset (buffer->data, 0, buffer->length);

  data[0] = 0xe;

  memcpy (&data[8],  VENDOR_ID,   8);
  memcpy (&data[16], PRODUCT_ID,  16);
  memcpy (&data[32], PRODUCT_REV, 4);

  return TRUE;
}

static gboolean
handle_mode_select (RBCDisk   *disk,
		    guint8    *cmd,
		    RBCBuffer *buffer,
		    guint8    *sense)
{
  /* TODO: implement */

  fill_sense_buffer (sense, CHECK_CONDITION, ILLEGAL_REQUEST, 0x20, 0);
  return FALSE;
}

static gboolean
handle_mode_sense (RBCDisk   *disk,
		   guint8    *cmd,
		   RBCBuffer *buffer,
		   guint8    *sense)
{
  /* TODO: implement */

  fill_sense_buffer (sense, CHECK_CONDITION, ILLEGAL_REQUEST, 0x20, 0);
  return FALSE;
}

static gboolean
handle_read (RBCDisk   *disk,
	     guint8    *cmd,
	     RBCBuffer *buffer,
	     guint8    *sense)
{
  gsize block;
  gsize num;
  off_t offset;
  gsize length;

  block = (cmd[2] << 24) + (cmd[3] << 16) + (cmd[4] << 8) + cmd[5];
  num = (cmd[7] << 8) + cmd[8];

  if (block + num > disk->capacity)
    {
      fill_sense_buffer (sense, CHECK_CONDITION, ILLEGAL_REQUEST, 0x21, 0);
      return FALSE;
    }

  if (num == 0)
    return TRUE;

  offset = (off_t) block * RBC_BLOCK_SIZE;
  length = num * RBC_BLOCK_SIZE;

  if (buffer->type == RBC_BUFFER_PAGE_TABLE)
    {
      lseek (disk->fd, offset, SEEK_SET);
      readv (disk->fd, buffer->data, buffer->length);
    }
  else
    pread (disk->fd, buffer->data, length, offset);

  return TRUE;
}

static gboolean
handle_read_capacity (RBCDisk   *disk,
		      guint8    *cmd,
		      RBCBuffer *buffer,
		      guint8    *sense)
{
  guint8 *data = buffer->data;
  gsize capacity = disk->capacity - 1;

  memset (buffer->data, 0, buffer->length);

  data[0] = (capacity >> 24);
  data[1] = (capacity >> 16) & 0xff;
  data[2] = (capacity >> 8) & 0xff;
  data[3] = capacity & 0xff;
  data[6] = (RBC_BLOCK_SIZE >> 8) & 0xff;
  data[7] = RBC_BLOCK_SIZE & 0xff;

  return TRUE;
}

static gboolean
handle_start_stop_unit (RBCDisk   *disk,
			guint8    *cmd,
			RBCBuffer *buffer,
			guint8    *sense)
{
  /* Power condition requests are unsupported, ignore */
  return TRUE;
}

static gboolean
handle_test_unit_ready (RBCDisk   *disk,
			guint8    *cmd,
			RBCBuffer *buffer,
			guint8    *sense)
{
  /* We're always ready */
  return TRUE;
}

static gboolean
handle_verify (RBCDisk   *disk,
	       guint8    *cmd,
	       RBCBuffer *buffer,
	       guint8    *sense)
{
  /* TODO: Implement */
  return TRUE;
}

static gboolean
handle_write (RBCDisk   *disk,
	      guint8    *cmd,
	      RBCBuffer *buffer,
	      guint8    *sense)
{
  gsize block;
  gsize num;
  off_t offset;
  gsize length;

  block = (cmd[2] << 24) + (cmd[3] << 16) + (cmd[4] << 8) + cmd[5];
  num = (cmd[7] << 8) + cmd[8];

  if (block + num > disk->capacity)
    {
      fill_sense_buffer (sense, CHECK_CONDITION, ILLEGAL_REQUEST, 0x21, 0);
      return FALSE;
    }

  if (num == 0)
    return TRUE;

  offset = (off_t) block * RBC_BLOCK_SIZE;
  length = num * RBC_BLOCK_SIZE;

  if (buffer->type == RBC_BUFFER_PAGE_TABLE)
    {
      lseek (disk->fd, offset, SEEK_SET);
      writev (disk->fd, buffer->data, buffer->length);
    }
  else
    pwrite (disk->fd, buffer->data, length, offset);

  return TRUE;
}

static gboolean 
handle_write_buffer (RBCDisk   *disk,
		     guint8    *cmd,
		     RBCBuffer *buffer,
		     guint8    *sense)
{
  /* Unsupported, always succeed */
  return TRUE;
}
