/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <string.h>
#include <limits.h>

#include <glib.h>

#include <libraw1394/raw1394.h>

#include <libsbp2/sbp2login.h>

#include "app.h"
#include "util.h"
#include "wire.h"


/* Keep this updated with the biggest wire message structure */
#define MAX_MESSAGE_LENGTH (sizeof (WireLogin))


typedef struct _WireLoginID WireLoginID;

typedef struct _WireBuffer WireBuffer;

struct _WireBuffer
{
  gpointer  message;

  gchar    *buffer;
  gsize     length;
};


static gboolean read_data        (GIOChannel   *channel,
				  WireBuffer   *wire_buffer);
static gboolean write_data       (GIOChannel   *channel,
                                  GIOCondition  condition,
				  WireBuffer   *wire_buffer);

static void     dispatch_message (EndpointApp  *app,
                                  gpointer      message);


static inline WireMessageType
get_message_type (gpointer message)
{
  return ((WireHeader *) message)->type;
}

static inline gpointer
alloc_message (WireMessageType type)
{
  WireHeader *header;

  header = g_malloc0 (MAX_MESSAGE_LENGTH); /* valgrind safe */
  header->type = type;

  return header;
}

static inline gpointer
alloc_message_login_ID (WireMessageType type,
                        guint           login_ID)
{
  WireLoginID *message;

  message = alloc_message (type);
  message->login_ID = login_ID;

  return message;
}


gboolean
wire_init (void)
{
  return PIPE_BUF >= MAX_MESSAGE_LENGTH;
}

gboolean
wire_message_send (GIOChannel  *channel,
                   gpointer     message,
		   EndpointApp *app)
{
  WireBuffer *wire_buffer;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);
  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->context != NULL, FALSE);

  wire_buffer = g_new0 (WireBuffer, 1);

  wire_buffer->message = message;
  wire_buffer->buffer = message;
  wire_buffer->length = MAX_MESSAGE_LENGTH;

  /* util_io_add_watch (app->context, channel,
                     G_IO_OUT | G_IO_ERR | G_IO_HUP,
		     (GIOFunc) write_data, wire_buffer); */

  return write_data (channel, 0, wire_buffer);
}

gboolean
wire_message_process (GIOChannel   *channel,
                      GIOCondition  condition,
		      EndpointApp  *app)
{
  WireBuffer *buffer;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->wire_funcs != NULL, FALSE);

  if (condition & G_IO_ERR || condition & G_IO_HUP)
    return FALSE;

  buffer = g_new0 (WireBuffer, 1);

  buffer->message = g_new (guint8, MAX_MESSAGE_LENGTH);
  buffer->buffer = buffer->message;
  buffer->length = MAX_MESSAGE_LENGTH;

  if (!read_data (channel, buffer))
    return FALSE;

  dispatch_message (app, buffer->message);

  g_free (buffer->message);
  g_free (buffer);

  return TRUE;
}

gpointer
wire_message_copy (gpointer message)
{
  return g_memdup (message, MAX_MESSAGE_LENGTH);
}

gpointer
wire_message_login (guint         login_ID,
                    gint          port,
                    nodeid_t      node,
		    SBP2Pointer  *pointer,
		    SBP2LoginORB *orb)
{
  WireLogin *message;

  g_return_val_if_fail (pointer != NULL, NULL);
  g_return_val_if_fail (orb != NULL, NULL);

  message = alloc_message (WIRE_LOGIN);
  message->login_ID = login_ID;

  message->node = node;

  memcpy (&message->pointer, pointer, sizeof (SBP2Pointer));
  memcpy (&message->orb, orb, sizeof (SBP2LoginORB));

  return message;
}

gpointer
wire_message_reconnect (guint    login_ID,
                        nodeid_t node)
{
  WireReconnect *message;

  message = alloc_message (WIRE_RECONNECT);
  message->login_ID = login_ID;

  message->node = node;

  return message;
}

gpointer 
wire_message_logout (guint login_ID)
{
  return alloc_message_login_ID (WIRE_LOGOUT, login_ID);
}

gpointer 
wire_message_worker_active (guint login_ID)
{
  return alloc_message_login_ID (WIRE_WORKER_ACTIVE, login_ID);
}

gpointer 
wire_message_worker_died (guint login_ID)
{
  return alloc_message_login_ID (WIRE_WORKER_DIED, login_ID);
}

static gboolean
read_data (GIOChannel *channel,
           WireBuffer *wire_buffer)
{
  gchar     *buffer;
  gsize      length;
  gsize      bytes;
  GIOStatus  status;
  GError    *error = NULL;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (wire_buffer != NULL, FALSE);

  buffer = wire_buffer->buffer;
  length = wire_buffer->length;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (buffer != NULL, FALSE);

  while (length > 0)
    {
      do
	{
	  bytes = 0;
	  status = g_io_channel_read_chars (channel,
	                                    buffer, length,
					    &bytes, &error);
	}
      while (status == G_IO_STATUS_AGAIN);

      if (status != G_IO_STATUS_NORMAL)
	{
	  if (error)
	    {
	      g_warning ("wire read data: error: %s", error->message);
	      g_error_free (error);
	    }
	  else
	    {
	      g_warning ("wire read data: error");
	    }

	  return FALSE;
	}

      if (bytes == 0)
	{
	  g_warning ("wire read data: unexpected EOF");
	  return FALSE;
	}

      length -= bytes;
      buffer += bytes;
    }

  return TRUE;
}

static gboolean
write_data (GIOChannel   *channel,
            GIOCondition  condition,
	    WireBuffer   *wire_buffer)
{
  gsize      bytes;
  GIOStatus  status;
  GError    *error = NULL;

  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (wire_buffer != NULL, FALSE);
  g_return_val_if_fail (wire_buffer->message != NULL, FALSE);
  g_return_val_if_fail (wire_buffer->buffer != NULL, FALSE);
  g_return_val_if_fail (wire_buffer->length > 0, FALSE);

  if (condition & G_IO_ERR || condition & G_IO_HUP)
    goto free_and_stop;

  bytes = 0;
  status = g_io_channel_write_chars (channel,
				     wire_buffer->buffer, wire_buffer->length,
				     &bytes, &error);

  if (status == G_IO_STATUS_AGAIN)
    return TRUE;

  if (status != G_IO_STATUS_NORMAL)
    {
      if (error)
	{
	  g_warning ("wire write data: error: %s", error->message);
	  g_error_free (error);
	}
      else
	g_warning ("wire write data: error");

      goto free_and_stop;
    }

  wire_buffer->length -= bytes;
  wire_buffer->buffer += bytes;

  if (wire_buffer->length > 0)
    return TRUE;

free_and_stop:
  g_free (wire_buffer->message);
  g_free (wire_buffer);

  return FALSE;
}

static void
dispatch_message (EndpointApp *app,
                  gpointer     message)
{
  WireMsgFuncs *funcs;

  g_return_if_fail (app != NULL);
  g_return_if_fail (app->wire_funcs != NULL);
  g_return_if_fail (message != NULL);

  funcs = app->wire_funcs;

  switch (get_message_type (message))
    {
    case WIRE_LOGIN:
      g_return_if_fail (funcs->login != NULL);

      funcs->login (app, message);
      break;

    case WIRE_RECONNECT:
      g_return_if_fail (funcs->reconnect != NULL);

      funcs->reconnect (app, message);
      break;

    case WIRE_LOGOUT:
      g_return_if_fail (funcs->logout != NULL);

      funcs->logout (app, message);
      break;

    case WIRE_WORKER_ACTIVE:
      g_return_if_fail (funcs->worker_active != NULL);

      funcs->worker_active (app, message);
      break;

    case WIRE_WORKER_DIED:
      g_return_if_fail (funcs->worker_died != NULL);

      funcs->worker_died (app, message);
      break;

    default:
      g_assert_not_reached ();
      break;
    }
}
