#include "bool.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include "util.h"
#include "shared.h"
#include "rcfile.h"
#include "main.h"
#include "exec.h"

static bool mksettings(FILE *fp);
static int getsetting(FILE *fp, char **argument);
static int getsetting_check(char *buf);
static char **set_multiple(int *numargs, char *arg);
static bool setboolean(bool *v, const char *arg);
static void boolean_err(const char *keyword, int line);

bool get_rcfile_settings(char *filename)
{
	FILE *fp;
	bool ret;
	if (!filename)
		return FALSE; /* no rcfile */
	if (!(fp = fopen(filename, "r"))) {
		if (!set.rcfile_must_exist)
			return FALSE;
		notice_err("cannot open rc file '%s'", filename);
		return TRUE;
	}
	ret = mksettings(fp);
	if (EOF==fclose(fp)) {
		notice_err("cannot close rc file '%s'", filename);
		return TRUE;
	}
	return ret;
}
	
enum {
	RC_INVALID,	RC_EOF,		RC_FATAL,	RC_COMMENT,
	RC_PORT,	RC_FATAL_ADMIN,	RC_BIND,	RC_MAXCON,
	RC_MAXFROMONE,	RC_DEBUG,	RC_COMMANDLOGFILE,
	RC_LOGFILE,	RC_KEEPALIVE,	RC_ONCONNECT,	RC_ONDISCONNECT,
	RC_HOSTMASK,	RC_AUTHFILE,	RC_PIDFILE,	RC_COMMAND,
	RC_COMMANDON,	RC_COMMANDOFF,	RC_EXTERNAL,	RC_PRE_COMMANDON,
	RC_POST_COMMANDON,		RC_MAXREDIALS,	RC_DISCONNECT_WAIT,
	RC_STAYDROPPED, RC_WAITPIPE,	RC_ONCONNECT_WAIT
};

bool mksettings(FILE *fp)
{
	int ret;
	char *arg;
	int line = 0;
	bool error = FALSE;
	bool bad;
	
	if (RC_FATAL==(ret = getsetting(fp, &arg)))
		return TRUE;
	while (ret!=RC_EOF) {
		++line;
		if (ret==RC_INVALID) {
			notice("invalid keyword in rc file line %d\n", line);
			error = TRUE;
		} else if (!arg) {
			switch(ret) {
			case RC_COMMENT: break;
			case RC_DEBUG:
#if 0
				notice("warning: 'debug' is deprecated, "
						"use 'debug true' in rc file "
						"line %d\n", line);
#endif
				set.debug = TRUE;
	      			break;
    			default:
				notice("keyword requires argument in rc file "
						"line %d\n", line);
				error = TRUE;
			       	break;
			}
		} else {
			switch(ret) {
		 	case RC_PORT:
				if (mystrtous(arg, &set.port)) {
					notice_err("invalid port argument in "
							"rc file line %d",
							line);
					error = TRUE;
				}
				free(arg);
	      			break;
			case RC_FATAL_ADMIN:
				notice("warning: the 'fatal admin' rc file "
						"option has been removed\n");
				break;
    			case RC_BIND:
				if (set.bind_ips)
					free(set.bind_ips[0]);
				free(set.bind_ips);
				if (set_bind_ips(arg)) {
					free(arg);
					error = TRUE;
				}
	      			break;
			case RC_MAXCON:
				if (mystrtoi(arg, &set.maxcons)) {
					notice_err("invalid maxcon argument in "
							"rc file line %d", 
							line);
					error = TRUE;
				}
				free(arg);
	      			break;
    			case RC_MAXFROMONE:
				notice("warning: the 'max_from_one' rc file "
						"option has been removed\n");
				break;
#if 0
				if (mystrtoi(arg, &set.max_from_one)) {
					notice_err("invalid max_from_one "
							"argument in rc file "
							"line %d", line);
					error = TRUE;
				}
				free(arg);
#endif
				break;
			case RC_MAXREDIALS:
				if (mystrtoi(arg, &set.max_redials)) {
					notice_err("invalid redials argument "
							"in rc file line %d",
							line);
					error = TRUE;
				}
				free(arg);
				break;
		    	case RC_KEEPALIVE: 
				if (mystrtol(arg, &set.keep_alive_timeout)) {
					notice_err("invalid keepalive argument "
							"in rc file line %d",
							line);
					error = TRUE;
				}
				free(arg);
				break;
			case RC_DISCONNECT_WAIT:
				if (mystrtol(arg, &set.disconnect_wait)) {
					notice_err("invalid disconnect_wait "
							"argument in rc file "
							"line %d", line);
					error = TRUE;
				}
				free(arg);
				break;
			case RC_ONCONNECT:
				free(set.onconnect);
				set.onconnect = arg;
	      			break;
    			case RC_ONDISCONNECT:
				free(set.ondisconnect);
				set.ondisconnect = arg;
				break;
			case RC_HOSTMASK:
				free(set.hmdata);
				if (set_hostmasks(arg)) {
					free(arg);
					error = TRUE;
				}
	      			break;
    			case RC_AUTHFILE:
				free(set.authfile);
				bad = realfile(&set.authfile, arg, 0);
				free(arg);
				if (bad)
					return TRUE;
				break;
			case RC_COMMANDLOGFILE:
				free(set.command_logfile);
				bad = realfile(&set.command_logfile, arg,
						MY_CREATE);
				free(arg);
				if (bad)
					return TRUE;
				break;
			case RC_LOGFILE:
				free(set.logfile);
				bad = realfile(&set.logfile, arg, MY_CREATE);
				free(arg);
				if (bad)
					return TRUE;
				break;
			case RC_PIDFILE: 
				free(set.pidfile);
				bad = realfile(&set.pidfile, arg,
						MY_CREATE | MY_ALLREAD);
				free(arg);
				if (bad)
					return TRUE;
				break;
			case RC_COMMAND:
				notice("warning: 'command' is deprecated, "
						"use 'commandon' in rc file "
						"line %d\n", line);
				free(set.commandon);
				set.commandon = arg;
				break;
			case RC_COMMANDON:
				free(set.commandon);
				set.commandon = arg;
				break;
			case RC_COMMANDOFF:
				free(set.commandoff);
				set.commandoff = arg;
				break;
			case RC_EXTERNAL:
				free(set.external);
				bad = realfile(&set.external, arg, 0);
				free(arg);
				if (bad)
					return TRUE;
				break;
			case RC_PRE_COMMANDON:
				free(set.pre_commandon);
				set.pre_commandon = arg;
				break;
			case RC_POST_COMMANDON:
				free(set.post_commandon);
				set.post_commandon = arg;
				break;
			case RC_STAYDROPPED:
				if (setboolean(&set.staydropped, arg)) {
					boolean_err("staydropped", line);
					error = TRUE;
				}
				break;
			case RC_WAITPIPE:
				if (setboolean(&set.waitpipe, arg)) {
					boolean_err("waitpipe", line);
					error = TRUE;
				}
				break;
			case RC_ONCONNECT_WAIT:
				if (setboolean(&set.onconnect_wait, arg)) {
					boolean_err("onconnect_wait", line);
					error = TRUE;
				}
				if (set.onconnect_wait && !set.onconnect)
					set.onconnect_wait = FALSE;
				break;
			case RC_DEBUG:
				if (setboolean(&set.debug, arg)) {
					boolean_err("debug", line);
					error = TRUE;
				}
				break;
			default:
				notice("error in rc file line %d\n", line);
				notice_debug("bug in mksettings\n");
		      		break;
			}
		}
		if (RC_FATAL==(ret = getsetting(fp, &arg)))
			return TRUE;
	}
	if (error)
		return TRUE; /* having printed all other errors */
	return FALSE;
}

int getsetting(FILE *fp, char **argument)
{
	char command[MAXLINE];
	char *sep;
	char *start;
	char *end;
	int ret;
	int c;

	if (!fgets(command, MAXLINE, fp))
		return RC_EOF;
	end = strchr(command, '\n');
	if (!end) {
		while((c = getc(fp))!=EOF && c!='\n');
		return RC_INVALID; /* too long */
	}
	/* Remove leading spaces */
	for (start=command; *start==' ' || *start=='\t'; start++);
	
	/* Remove \n and trailing spaces */
	*end = '\0';

	/* Some editors make it easy to leave a space at the end of the
	 * line.
	 *
	 * XXX: Since the user may wish to use a file which actually
	 * ends in a space, we can't do this. */
#if 0
	while(end > command && *--end==' ')
		*end='\0';
#endif
	/* Comments start with # or are blank lines */
	if (start[0]=='#' || start[0]=='\0') {
		*argument = NULL; /* XXX */
		return RC_COMMENT;
	}
	sep = strchr(start, ' ');
	if (sep) {
		*sep = '\0';
		if (alloc_copy(argument, sep+1))
                        return RC_FATAL;
        } else {
                *argument = NULL;
	}
	ret = getsetting_check(start);
	return ret;
}

int getsetting_check(char *buf)
{
	int i;
	struct list {
		char *descr;
		int tag;
	} parse[] = {
		{ "port", 	RC_PORT },
		{ "fatal",	RC_FATAL_ADMIN },
		{ "bind", 	RC_BIND},
		{ "maxcon", 	RC_MAXCON },
		{ "max_from_one", RC_MAXFROMONE },
		{ "redials", 	RC_MAXREDIALS },
		{ "debug", 	RC_DEBUG },
		{ "keepalive", 	RC_KEEPALIVE },
		{ "onconnect", 	RC_ONCONNECT },
		{ "ondisconnect", RC_ONDISCONNECT },
		{ "hostmask", 	RC_HOSTMASK },
		{ "authfile", 	RC_AUTHFILE },
		{ "command_logfile", RC_COMMANDLOGFILE },
		{ "logfile", 	RC_LOGFILE },
		{ "pidfile",  	RC_PIDFILE },
		{ "command", 	RC_COMMAND },
		{ "commandon", 	RC_COMMANDON },
		{ "commandoff", RC_COMMANDOFF },
		{ "external", 	RC_EXTERNAL },
		{ "pre_commandon", RC_PRE_COMMANDON },
		{ "post_commandon", RC_POST_COMMANDON },
		{ "disconnect_wait", RC_DISCONNECT_WAIT },
		{ "staydropped", RC_STAYDROPPED },
		{ "waitpipe",	RC_WAITPIPE },
		{ "onconnect_wait", RC_ONCONNECT_WAIT },
		{ NULL, -1 }
	};
	for (i=0; parse[i].descr; ++i) {
		if (0==strcasecmp(buf, parse[i].descr))
			return parse[i].tag;
	}
	return RC_INVALID;
}

bool setboolean(bool *v, const char *arg)
{
	if (0==strcasecmp("true", arg) || 0==strcasecmp("on", arg))
		*v = TRUE;
	else if (0==strcasecmp("false", arg) || 0==strcasecmp("off", arg))
		*v = FALSE;
	else
		return TRUE; /* invalid argument */
	return FALSE;
}

void boolean_err(const char *keyword, int line)
{
	notice("invalid argument to boolean %s in rc file line %d\n", keyword,
			line);
	return;
}

bool set_bind_ips(char *arg)
{
	set.bind_ips = set_multiple(&set.num_ips, arg);
	if (!set.bind_ips)
		return TRUE;
	return FALSE;
}

bool set_hostmasks(char *arg)
{
	char **hostmasks;
	int ret;
	hostmasks = set_multiple(&set.num_hms, arg);
	if (!hostmasks)
		return TRUE;
	ret = makehostmasks(hostmasks);
	free(hostmasks[0]);
	free(hostmasks);
	return ret;
}

/* returns pointer to the space and/or comma delimited argments in arg.
 * (Which is modified).
 *
 * If numargs is non-NULL, the number of items is set.
 *
 * e.g. for "192.168.1.1 127.0.0.1" or "192.168.1.1,127.0.0.1"
 * 	set.bind_ips[0] points to "192.168.1.1"
 * 	set.bind_ips[1] points to "127.0.0.1"
 * 	set.bind_ips[2] points to NULL
 * 	*numargs = 2
 * these are pointing inside the arg buffer.
 *
 * We allow things such as "foo, bar", "foo,,bar", "foo bar "
 *
 */
char **set_multiple(int *numargs, char *arg)
{
	size_t len;
	int maxargs = 16;
	char **pt;
	char *curr = arg;
	int i;

	if (!(pt = xmalloc(sizeof(*pt) * (maxargs+1))))
		return NULL;
	for (i=0; *curr; ++i) {
		if ((i+1) > maxargs) {
			maxargs *= 2;
			if (!(pt = realloc(pt, sizeof(*pt) * (maxargs+1)))) {
				notice_err("malloc failed");
				free(pt);
				return NULL;
			}
		}

		pt[i] = curr;

		len = strcspn(curr, " ,\0");
		curr += len;
		if (*curr)
			*curr++ = '\0';
		/* Skip over any extra spaces or commas. */
		while (*curr && (*curr==' ' || *curr==',')) 
			++curr;
	}
	pt[i] = NULL;
	if (numargs)
		*numargs = i;
	return pt;
}
