/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2005 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "ui2_includes.h"
#include "ui2_typedefs.h"
#include "ui2_tooltip.h"

#include "ui2_main.h"
#include "ui2_skin.h"
#include "ui2_widget.h"
#include "ui2_text.h"
#include "ui2_util.h"
#include "ui_pixbuf_ops.h"


#define TOOLTIP_KEY "tooltip_window"
#define MESSAGE_KEY "tooltip_message"
#define TIPDATA_KEY "tooltip_data"
#define TOOLTIP_DELAY 500


typedef struct _UITooltipData UITooltipData;
struct _UITooltipData {
	UIData *ui;
	WidgetData *widget;
	gchar *message;
	gint delay_id;
};


static UIData *ui_tooltip_new(UIData *ui);


static SkinData *ui_tooltip_skin_default(void)
{
	SkinData *skin;
	TextData *text;
	GdkPixbuf *pixbuf;
	gint w, h;

	skin = skin_new();

	pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, 16, 16);
	w = gdk_pixbuf_get_width(pixbuf);
	h = gdk_pixbuf_get_height(pixbuf);
	pixbuf_draw_rect_fill(pixbuf, 0, 0, w, h, 220, 220, 220, 255);
	pixbuf_draw_rect(pixbuf, 0, 0, w, h, 0, 0, 0, 255, 1, 1, 1, 1);

	skin->real_overlay = util_size_pixbuf(pixbuf, TRUE);
	skin->has_border = TRUE;
	skin->border_left = 1;
	skin->border_right = 1;
	skin->border_top = 1;
	skin->border_bottom = 1;
	skin->width = 26;
	skin->height = 24;

	util_size(&skin->width);
	util_size(&skin->height);
	util_size(&skin->border_left);
	util_size(&skin->border_right);
	util_size(&skin->border_bottom);
	util_size(&skin->border_top);

	skin->width_def = skin->width_min = skin->width_max = skin->width;
	skin->height_def = skin->height_min = skin->height_max = skin->height;

	text = text_new_from_x("Sans 10", 3, 3, 20, TRUE, 0);
	text_set_color(text, 0, 0, 0, 255);
	text_register(skin, text, MESSAGE_KEY, NULL);

	return skin;
}

static gint ui_tooltip_back_cb(UIData *ui, GdkPixbuf *pixbuf, gpointer data)
{
	UIData *tip = data;
	gint x, y;

	if (!GTK_WIDGET_VISIBLE(ui->display) || !GTK_WIDGET_REALIZED(ui->display)) return FALSE;

	if (ui_geometry_get(tip, &x, &y, NULL, NULL))
		{
		util_pixbuf_fill_from_root_window(pixbuf, x, y, TRUE);
		return TRUE;
		}

	return FALSE;
}

static gint ui_tooltip_motion_cb(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
	UIData *tip = data;

	if (GTK_WIDGET_VISIBLE(tip->window)) gtk_widget_hide(tip->window);

	return TRUE;
}

static void ui_tooltip_display(UITooltipData *tt)
{
	UIData *ui;
	gint px, py;
	gint pw, ph;
	gint wx, wy;
	gint ww, wh;
	gint tx, ty;
	gint tw, th;
	WidgetData *wd;
	TextData *td;
	gint old_width;
	gint new_width;

	ui = tt->ui;

	if (!tt->widget || !tt->message) return;

	if (!ui->skin)
		{
		if (!ui_skin_load(ui, ui->parent->skin_path, "skindata_tooltip"))
			{
			ui_skin_load(ui, NULL, "skindata_tooltip");
			}
		if (!ui->skin || !skin_widget_get_by_key(ui->skin, MESSAGE_KEY, text_type_id()))
			{
			ui_skin_set(ui, ui_tooltip_skin_default(), NULL, "skindata_tooltip");
			}
		}

	if (!ui->skin || !ui->parent->skin) return;

	/* validate widget still exists */
	if (g_list_find(ui->parent->skin->widget_list, tt->widget) == NULL) return;

	if (!ui_geometry_get(ui->parent, &px, &py, &pw, &ph) ||
	    !ui_widget_get_geometry(tt->widget, &wx, &wy, &ww, &wh)) return;

	if (!ui_geometry_get(ui, &tx, &ty, &tw, &th)) return;

	wd = skin_widget_get_by_key(ui->skin, MESSAGE_KEY, text_type_id());
	if (!wd) return;

	if (!ui_widget_get_geometry(wd, NULL, NULL, &old_width, NULL)) return;

	td = wd->widget;
	new_width = font_string_length(td->font, tt->message);

	tx = px + wx + ww;
	ty = py + wy + wh;

	tw += new_width - old_width;
	skin_resize(ui, tw, th);

	text_set_text(MESSAGE_KEY, ui, tt->message);
	ui_widget_set_data(wd, "data", tt->message);

	gtk_window_move(GTK_WINDOW(ui->window), tx, ty);

	if (!GTK_WIDGET_VISIBLE(ui->window)) gtk_widget_show(ui->window);
}

static gboolean ui_tooltip_delay_cb(gpointer data)
{
	UITooltipData *tt = data;

	ui_tooltip_display(tt);

	tt->delay_id = -1;
	return FALSE;
}

static void ui_tooltip_reschedule(UITooltipData *tt, gint enable)
{
	if (tt->delay_id != -1)
		{
		g_source_remove(tt->delay_id);
		tt->delay_id = -1;
		}
	if (enable)
		{
		tt->delay_id = g_timeout_add(TOOLTIP_DELAY, ui_tooltip_delay_cb, tt);
		}
}

void ui_tooltip_show(UIData *ui, WidgetData *widget, const gchar *message)
{
	UITooltipData *tt;
	UIData *tip;

	tip = ui_group_get_child(ui, TOOLTIP_KEY);
	if (!widget || !message)
		{
		if (tip)
			{
			if (GTK_WIDGET_VISIBLE(tip->window)) gtk_widget_hide(tip->window);

			tt = g_object_get_data(G_OBJECT(tip->window), TIPDATA_KEY);
			tt->widget = NULL;
			g_free(tt->message);
			tt->message = NULL;
			ui_tooltip_reschedule(tt, FALSE);
			}

		return;
		}
	if (!tip) tip = ui_tooltip_new(ui);

	tt = g_object_get_data(G_OBJECT(tip->window), TIPDATA_KEY);
	if (!tt) return;

	if (tt->widget == widget && tt->message && strcmp(tt->message, message) == 0) return;

	tt->widget = widget;
	g_free(tt->message);
	tt->message = g_strdup(message);

	ui_tooltip_reschedule(tt, TRUE);
}

static void ui_tooltip_destroy_cb(GtkWidget *widget, gpointer data)
{
	UITooltipData *tt = data;

	ui_tooltip_reschedule(tt, FALSE);

	g_free(tt->message);
	g_free(tt);
}

static UIData *ui_tooltip_new(UIData *ui)
{
	UIData *tip;
	GtkWidget *window;
	UITooltipData *tt;

	window = gtk_window_new(GTK_WINDOW_POPUP);
	tip = ui_new_into_container(ui->class, TOOLTIP_KEY, window);
	tip->window = window;

	ui_moveable_set(tip, FALSE);
	ui_focus_set(tip, FALSE);

	text_register_key(MESSAGE_KEY, tip, NULL, NULL);

	ui_set_back_callback(tip, ui_tooltip_back_cb, tip);

	ui_group_set_child(ui, tip);
	tip->skin_func = ui->skin_func;
	tip->skin_data = ui->skin_data;

	g_signal_connect(G_OBJECT(tip->display), "motion_notify_event",
			 G_CALLBACK(ui_tooltip_motion_cb), tip);

	tt = g_new0(UITooltipData, 1);
	tt->ui = tip;
	tt->widget = NULL;
	tt->message = NULL;
	tt->delay_id = -1;

	g_object_set_data(G_OBJECT(window), TIPDATA_KEY, tt);
	g_signal_connect(G_OBJECT(window), "destroy",
			G_CALLBACK(ui_tooltip_destroy_cb), tt);

	return tip;
}


