/*
 *  svaf_read.c: load SVAF models from files or memory buffers.
 *
 *  Copyright (c) 2006, Michael C. Martin
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - Neither the name 'Bumbershoot Software' nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 *  CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 *  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 *  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 *  THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "svaf.h"

/* IFF tags for SVAF files */
#define FORM 0x464f524d
#define SVAF 0x53564146
#define PNTS 0x504e5453
#define NRML 0x4e524d4c
#define MTRL 0x4d54524c
#define TRIS 0x54524953
#define TFAN 0x5446414e
#define TSTP 0x54535450
#define QUAD 0x51554144
#define QSTP 0x51535450

static unsigned int
read_uint32 (FILE *f)
{
	int i, result;
	result = 0;
	for (i = 0; i < 4; i++)
	{
		result <<= 8;
		result += fgetc(f);
	}
	return result;
}

static unsigned int
decode_uint32 (const unsigned char *b)
{
	int i, result;
	result = 0;
	for (i = 0; i < 4; i++)
	{
		result <<= 8;
		result += *b++;
	}
	return result;
}

static unsigned int
decode_uint16 (const unsigned char *b)
{
	int i, result;
	result = 0;
	for (i = 0; i < 2; i++)
	{
		result <<= 8;
		result += *b++;
	}
	return result;
}

static float
decode_float (const unsigned char *b)
{
	union { float f; unsigned int i; } bits;
	bits.i = decode_uint32 (b);
	return bits.f;
}

static unsigned int *
read_indices (const unsigned char *p, int count)
{
	unsigned int *result = (unsigned int *)malloc(sizeof(unsigned int)*count);
	int i;
	for (i = 0; i < count; i++)
	{
		result[i] = decode_uint16 (p);
		p += 2;
	}
	return result;
}

svaf_model *
svaf_ReadBuffer (const unsigned char *buffer, int len, const char *name)
{
	const unsigned char *i, *end;
	int num_cmds, cmd_index, j;
	svaf_model *result;

	if (decode_uint32 (buffer) == FORM)
	{
		len = decode_uint32 (buffer+4) - 4;
		buffer += 12;
	}

	end = buffer + len;

	/* Pass 1: Count the number of commands */

	num_cmds = 0;
	for (i = buffer; i < end; )
	{
		int id = decode_uint32 (i);
		int chunk_len = decode_uint32 (i+4);
		
		if ((id == MTRL) || (id == TRIS) || (id == TSTP) ||
		    (id == TFAN) || (id == QUAD) || (id == QSTP))
			num_cmds++;

		i += 8+chunk_len;
	}

	/* Allocate memory */

	result = (svaf_model *)malloc (sizeof(svaf_model));
	result->vertices = NULL;
	result->normals = NULL;
	result->num_commands = num_cmds;
	result->commands = (svaf_command*)malloc (sizeof(svaf_command) * num_cmds);
	for (j = 0; j < num_cmds; j++)
		result->commands[j].tag = SVAF_CMD_NONE;

	/* Pass 2: Read in data */

	cmd_index = 0;
	for (i = buffer; i < end; )
	{
		int id = decode_uint32 (i);
		int chunk_len = decode_uint32 (i+4);
		i += 8;
		switch (id)
		{
		case PNTS:
		{
			const unsigned char *x;
			int p_index, num = chunk_len/4;
			if (result->vertices)
			{
				fprintf (stderr, "%s: Multiple PNTS chunks\n", name);
				svaf_FreeModel(result);
				return NULL;
			}

			if (chunk_len % 12)
			{
				fprintf (stderr, "%s: Invalid PNTS chunk length\n", name);
				svaf_FreeModel(result);
				return NULL;
			}

			if (result->normals && (result->num_vertices != num/3))
			{
				fprintf (stderr, "%s: PNTS and NRML chunks inconsistent\n", name);
				svaf_FreeModel(result);
				return NULL;
			}

			result->num_vertices = num / 3;
			result->vertices = (GLfloat *)malloc (sizeof (GLfloat) * num);

			x = i;
			for (p_index = 0; p_index < num; p_index++)
			{
				result->vertices[p_index] = decode_float(x);
				x += 4;
			}
			break;
		}
		case NRML:
		{
			const unsigned char *x;
			int n_index, num = chunk_len/4;
			if (result->normals)
			{
				fprintf (stderr, "%s: Multiple NRML chunks\n", name);
				svaf_FreeModel(result);
				return NULL;
			}

			if (chunk_len % 12)
			{
				fprintf (stderr, "%s: Invalid NRML chunk length\n", name);
				svaf_FreeModel(result);
				return NULL;
			}

			if (result->vertices && (result->num_vertices != num/3))
			{
				fprintf (stderr, "%s: PNTS and NRML chunks inconsistent\n", name);
				svaf_FreeModel(result);
				return NULL;
			}

			result->num_vertices = num / 3;
			result->normals = (GLfloat *)malloc (sizeof (GLfloat) * num);

			x = i;
			for (n_index = 0; n_index < num; n_index++)
			{
				result->normals[n_index] = decode_float(x);
				x += 4;
			}

			/* Normalize normals */
			for (n_index = 0; n_index < num; n_index += 3)
			{
				float x, y, z;
				double norm;
				
				x = result->normals[n_index];
				y = result->normals[n_index+1];
				z = result->normals[n_index+2];
				norm = sqrt (x*x+y*y+z*z);
				result->normals[n_index] = x / norm;
				result->normals[n_index+1] = y / norm;
				result->normals[n_index+2] = z / norm;
			}			    
			break;
		}
		case MTRL:
		{
			if (chunk_len != 52)
			{
				fprintf (stderr, "%s: Invalid MTRL chunk\n", name);
				svaf_FreeModel (result);
				return NULL;
			}
			result->commands[cmd_index].tag = SVAF_CMD_MATERIAL;
			result->commands[cmd_index].command.material.diffuse[0]  = decode_float(i);
			result->commands[cmd_index].command.material.diffuse[1]  = decode_float(i+ 4);
			result->commands[cmd_index].command.material.diffuse[2]  = decode_float(i+ 8);
			result->commands[cmd_index].command.material.diffuse[3]  = decode_float(i+12);
			result->commands[cmd_index].command.material.specular[0] = decode_float(i+16);
			result->commands[cmd_index].command.material.specular[1] = decode_float(i+20);
			result->commands[cmd_index].command.material.specular[2] = decode_float(i+24);
			result->commands[cmd_index].command.material.specular[3] = decode_float(i+28);
			result->commands[cmd_index].command.material.emissive[0] = decode_float(i+32);
			result->commands[cmd_index].command.material.emissive[1] = decode_float(i+36);
			result->commands[cmd_index].command.material.emissive[2] = decode_float(i+40);
			result->commands[cmd_index].command.material.emissive[3] = decode_float(i+44);
			result->commands[cmd_index].command.material.shininess   = decode_float(i+48);
			cmd_index++;
			break;
		}
		case TRIS:
		{
			result->commands[cmd_index].tag = SVAF_CMD_ELEMENTS;
			result->commands[cmd_index].command.elements.type = GL_TRIANGLES;
			result->commands[cmd_index].command.elements.count = chunk_len / 2;
			result->commands[cmd_index].command.elements.indices = read_indices (i, chunk_len / 2);
			cmd_index++;
			break;
		}
		case TFAN:
		{
			result->commands[cmd_index].tag = SVAF_CMD_ELEMENTS;
			result->commands[cmd_index].command.elements.type = GL_TRIANGLE_FAN;
			result->commands[cmd_index].command.elements.count = chunk_len / 2;
			result->commands[cmd_index].command.elements.indices = read_indices (i, chunk_len / 2);
			cmd_index++;
			break;
		}
		case TSTP:
		{
			result->commands[cmd_index].tag = SVAF_CMD_ELEMENTS;
			result->commands[cmd_index].command.elements.type = GL_TRIANGLE_STRIP;
			result->commands[cmd_index].command.elements.count = chunk_len / 2;
			result->commands[cmd_index].command.elements.indices = read_indices (i, chunk_len / 2);
			cmd_index++;
			break;
		}
		case QUAD:
		{
			result->commands[cmd_index].tag = SVAF_CMD_ELEMENTS;
			result->commands[cmd_index].command.elements.type = GL_QUADS;
			result->commands[cmd_index].command.elements.count = chunk_len / 2;
			result->commands[cmd_index].command.elements.indices = read_indices (i, chunk_len / 2);
			cmd_index++;
			break;
		}
		case QSTP:
		{
			result->commands[cmd_index].tag = SVAF_CMD_ELEMENTS;
			result->commands[cmd_index].command.elements.type = GL_QUAD_STRIP;
			result->commands[cmd_index].command.elements.count = chunk_len / 2;
			result->commands[cmd_index].command.elements.indices = read_indices (i, chunk_len / 2);
			cmd_index++;
			break;
		}
		default:
			printf ("%s: Unknown chunk \"%c%c%c%c\" (%d bytes)\n", name,
				(id >> 24) & 0xff, 
				(id >> 16) & 0xff, 
				(id >> 8) & 0xff, 
				id & 0xff, chunk_len);
			break;
		}
		i += chunk_len;
	}
	return result;
}

svaf_model *
svaf_ReadFile (const char *fname)
{
	FILE *f = fopen (fname, "rb");
	int len;
	unsigned char *buffer;
	svaf_model *result;

	if (!f)
	{
		// fprintf (stderr, "Could not open %s\n", fname);
		return NULL;
	}
	if (read_uint32 (f) != FORM)
	{
		fprintf (stderr, "%s is not a SVAF file\n", fname);
		fclose (f);
		return NULL;
	}
	len = read_uint32 (f) - 4;
	if (read_uint32 (f) != SVAF)
	{
		fprintf (stderr, "%s is not a SVAF file\n", fname);
		fclose (f);
		return NULL;
	}

	buffer = (unsigned char *)malloc (len);
	if (fread (buffer, 1, len, f) != len)
	{
		fprintf (stderr, "%s had unexpected EOF\n", fname);
		fclose (f);
		free (buffer);
		return NULL;
	}
	fclose(f);

	result = svaf_ReadBuffer (buffer, len, fname);

	free (buffer);
	return result;
}

void
svaf_FreeModel (svaf_model *m)
{
	if (m)
	{
		if (m->vertices)
			free (m->vertices);
		if (m->normals)
			free (m->normals);
		if (m->commands)
		{
			int i;
			for (i = 0; i < m->num_commands; i++)
			{
				if ((m->commands[i].tag == SVAF_CMD_ELEMENTS) && m->commands[i].command.elements.indices)
					free (m->commands[i].command.elements.indices);
			}
			free (m->commands);
		}
		free (m);
	}
}
