/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildbindingnode.go
// functions to generate NodeJS-bindings of a library's API.
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"fmt"
	"io"
	"path"
	"log"
	"os"
	"strings"
)

// BuildBindingNode builds NodeJS-bindings of a library's API
func BuildBindingNode(component ComponentDefinition, outputFolder string, indentString string) error {
	namespace := component.NameSpace
	libraryname := component.LibraryName
	baseName := component.BaseName;

	NodeAddOnImplName := path.Join(outputFolder, baseName + "_nodeaddon.cc");
	log.Printf("Creating \"%s\"", NodeAddOnImplName)
	nodeaddonfile, err := os.Create(NodeAddOnImplName)
	if err != nil {
		log.Fatal(err)
	}
	WriteLicenseHeader(nodeaddonfile, component,
		fmt.Sprintf("This is an autogenerated C++ Implementation file for the Node addon class \n of %s", libraryname),
		true)

	NodeWrapperHeaderName := path.Join(outputFolder, baseName + "_nodewrapper.h")
	log.Printf("Creating \"%s\"", NodeWrapperHeaderName)
	nodewrapperhfile, err := os.Create(NodeWrapperHeaderName)
	if err != nil {
		log.Fatal(err)
	}
	WriteLicenseHeader(nodewrapperhfile, component,
		fmt.Sprintf("This is an autogenerated C++ Header file for the Node wrapper class \n of %s", libraryname),
		true)

	NodeWrapperImplName := path.Join(outputFolder, baseName + "_nodewrapper.cc")
	log.Printf("Creating \"%s\"", NodeWrapperImplName)
	nodewrapperccfile, err := os.Create(NodeWrapperImplName)
	if err != nil {
		log.Fatal(err)
	}
	WriteLicenseHeader(nodewrapperccfile, component,
		fmt.Sprintf("This is an autogenerated C++ Implementation file for the Node wrapper class \n of %s", libraryname),
		true)

	err = buildNodeAddOnImplementation(component, nodeaddonfile, namespace, baseName)
	if err != nil {
		log.Fatal(err)
	}

	NodeBindingGypName := path.Join(outputFolder, "binding.gyp")
	log.Printf("Creating \"%s\"", NodeBindingGypName)
	bindinggypfile, err := os.Create(NodeBindingGypName)
	if err != nil {
		log.Fatal(err)
	}
	buildNodeBindingGyp (component, bindinggypfile, indentString);
	
	return buildNodeWrapperClass(component, nodewrapperhfile, nodewrapperccfile, namespace, baseName)
}

func buildNodeAddOnImplementation(component ComponentDefinition, w io.Writer, NameSpace string, BaseName string) error {

	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "#include <node.h>\n")
	fmt.Fprintf(w, "#include \"%s_nodewrapper.h\"\n", BaseName)
	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "using namespace v8;\n")
	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "void Load%s (const FunctionCallbackInfo<Value>& args)\n", NameSpace)
	fmt.Fprintf(w, "{\n")
	fmt.Fprintf(w, "    Isolate* isolate = args.GetIsolate();\n")
	fmt.Fprintf(w, "    HandleScope scope(isolate);\n")
	fmt.Fprintf(w, "    args.GetReturnValue().Set (C%sWrapper::NewInstance());\n", NameSpace)
	fmt.Fprintf(w, "}\n")
	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "void InitAll(Handle<Object> exports, Handle<Object> module)\n")
	fmt.Fprintf(w, "{\n")
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		fmt.Fprintf(w, "    C%s%s::Init();\n", NameSpace, class.ClassName)
	}
	fmt.Fprintf(w, "    C%sWrapper::Init();\n", NameSpace)
	fmt.Fprintf(w, "    NODE_SET_METHOD(module, \"exports\", Load%s);\n", NameSpace)
	fmt.Fprintf(w, "}\n")
	fmt.Fprintf(w, "\n")

	fmt.Fprintf(w, "NODE_MODULE(%s_nodeaddon, InitAll)\n", strings.ToLower(NameSpace))
	fmt.Fprintf(w, "\n")

	return nil
}

func writeNodeMethodImplementation(method ComponentDefinitionMethod, implw io.Writer, NameSpace string, ClassName string, isGlobal bool) error {

	returndeclaration := ""
	inputdeclaration := ""
	inputcheck := ""
	returncode := ""
	functioncode := "";
	requiresInitCall := false;

	spacing := "        "

	callParameters := ""
	initCallParameters := ""
	

	returnParamCount := 0;
	
	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k];
		if ((param.ParamPass == "out") || (param.ParamPass == "return")) {
			returnParamCount = returnParamCount + 1;
		}		
	}
	
	if (returnParamCount > 1) {
		inputdeclaration = inputdeclaration + fmt.Sprintf ("%sLocal<Object> outObject = Object::New(isolate);\n", spacing);
	}


	for k := 0; k < len(method.Params); k++ {
	
		initCallParameter := "";
		callParameter := "";
	
		param := method.Params[k]
		switch param.ParamPass {
		case "in":

			inputcheckfunction := ""

			switch param.ParamType {
			case "uint8":
				inputcheckfunction = "IsUint32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sunsigned char n%s = (unsigned char) args[%d]->Uint32Value ();\n", spacing, param.ParamName, k)
				callParameter = "n" + param.ParamName;
				initCallParameter = callParameter;

			case "uint16":
				inputcheckfunction = "IsUint32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sunsigned short n%s = (unsigned short) args[%d]->Uint32Value ();\n", spacing, param.ParamName, k)
				callParameter = "n" + param.ParamName
				initCallParameter = callParameter;

			case "uint32":
				inputcheckfunction = "IsUint32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sunsigned int n%s = (unsigned int) args[%d]->IntegerValue ();\n", spacing, param.ParamName, k)
				callParameter = "n" + param.ParamName
				initCallParameter = callParameter;

			case "uint64":
				inputcheckfunction = "IsString"

				inputdeclaration = inputdeclaration + fmt.Sprintf("%sv8::String::Utf8Value sutf8%s (args[%d]->ToString());\n", spacing, param.ParamName, k)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sstd::string s%s = *sutf8%s;\n", spacing, param.ParamName, param.ParamName)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%suint64_t n%s = stoull (s%s);\n", spacing, param.ParamName, param.ParamName)
				callParameter = "n" + param.ParamName
				initCallParameter = callParameter;
			
			case "pointer":
				inputcheckfunction = "IsString"

				inputdeclaration = inputdeclaration + fmt.Sprintf("%sv8::String::Utf8Value sutf8%s (args[%d]->ToString());\n", spacing, param.ParamName, k)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sstd::string s%s = *sutf8%s;\n", spacing, param.ParamName, param.ParamName)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%suint64_t n%s = stoull (s%s);\n", spacing, param.ParamName, param.ParamName)
				callParameter = "(void*) n" + param.ParamName
				initCallParameter = callParameter;

			case "int8":
				inputcheckfunction = "IsInt32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%s char n%s = (char) args[%d]->Int32Value ();\n", spacing, param.ParamName, k)
				callParameter = "n" + param.ParamName;
				initCallParameter = callParameter;

			case "int16":
				inputcheckfunction = "IsInt32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%s short n%s = (short) args[%d]->Int32Value ();\n", spacing, param.ParamName, k)
				callParameter = "n" + param.ParamName
				initCallParameter = callParameter;

			case "int32":
				inputcheckfunction = "IsInt32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%s int n%s = (int) args[%d]->IntegerValue ();\n", spacing, param.ParamName, k)
				callParameter = "n" + param.ParamName
				initCallParameter = callParameter;

			case "int64":
				inputcheckfunction = "IsString"

				inputdeclaration = inputdeclaration + fmt.Sprintf("%sv8::String::Utf8Value sutf8%s (args[%d]->ToString());\n", spacing, param.ParamName, k)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sstd::string s%s = *sutf8%s;\n", spacing, param.ParamName, param.ParamName)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sint64_t n%s = stoll (s%s);\n", spacing, param.ParamName, param.ParamName)
				callParameter = "n" + param.ParamName
				initCallParameter = callParameter;

			case "string":
				inputcheckfunction = "IsString"

				inputdeclaration = inputdeclaration + fmt.Sprintf("%sv8::String::Utf8Value sutf8%s (args[%d]->ToString());\n", spacing, param.ParamName, k)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sstd::string s%s = *sutf8%s;\n", spacing, param.ParamName, param.ParamName)
				callParameter = "s" + param.ParamName + ".c_str()"
				initCallParameter = callParameter;

			case "basicarray":
				callParameter = "0, nullptr";
				initCallParameter = callParameter;

			case "structarray":
				callParameter = "0, nullptr";
				initCallParameter = callParameter;

			case "functiontype":
				callParameter = "nullptr";
				initCallParameter = callParameter;
				
			case "bool":
				inputcheckfunction = "IsBoolean"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sbool b%s = args[%d]->BooleanValue ();\n", spacing, param.ParamName, k)
				callParameter = "b" + param.ParamName
				initCallParameter = callParameter;

			case "single":
				inputcheckfunction = "IsNumber"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sfloat f%s = (float) args[%d]->NumberValue ();\n", spacing, param.ParamName, k)
				callParameter = "f" + param.ParamName
				initCallParameter = callParameter;

			case "double":
				inputcheckfunction = "IsNumber"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sdouble d%s = (double) args[%d]->NumberValue ();\n", spacing, param.ParamName, k)
				callParameter = "d" + param.ParamName
				initCallParameter = callParameter;

			case "enum":
				inputcheckfunction = "IsUint32"
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sunsigned int e%s = (unsigned int) args[%d]->IntegerValue ();\n", spacing, param.ParamName, k)
				callParameter = fmt.Sprintf("(e%s%s) e%s", NameSpace, param.ParamClass, param.ParamName)
				initCallParameter = callParameter;

			case "struct":
				inputcheckfunction = "IsObject"
				
				inputdeclaration = inputdeclaration + fmt.Sprintf("%ss%s%s s%s = convertObjectTo%s%s(isolate, args[%d]);\n", spacing, NameSpace, param.ParamClass, param.ParamName, NameSpace, param.ParamClass, k)
				
				callParameter = fmt.Sprintf ("&s%s", param.ParamName);
				initCallParameter = callParameter;

			case "class", "optionalclass":
				inputcheckfunction = "IsObject"

				inputdeclaration = inputdeclaration + fmt.Sprintf("%sLocal<Object> obj%s = args[%d]->ToObject(isolate->GetCurrentContext()).ToLocalChecked ();\n", spacing, param.ParamName, k)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sC%s%s * instance%s = ObjectWrap::Unwrap<C%s%s>(obj%s);\n", spacing, NameSpace, param.ParamClass, param.ParamName, NameSpace, param.ParamClass, param.ParamName)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%sif (instance%s == nullptr)\n", spacing, param.ParamName)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%s    throw std::runtime_error(\"Invalid Object parameter %d (%s)\");\n", spacing, k, param.ParamName)
				inputdeclaration = inputdeclaration + fmt.Sprintf("%s%sHandle h%s = instance%s->getHandle ( obj%s );\n", spacing, NameSpace, param.ParamName, param.ParamName, param.ParamName)

				callParameter = "h" + param.ParamName
				initCallParameter = callParameter;

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)

			}

			if inputcheckfunction != "" {
				inputcheck = inputcheck + fmt.Sprintf("%sif (!args[%d]->%s()) {\n", spacing, k, inputcheckfunction)
				inputcheck = inputcheck + fmt.Sprintf("%s    throw std::runtime_error (\"Expected %s parameter %d (%s)\");\n", spacing, param.ParamType, k, param.ParamName)
				inputcheck = inputcheck + fmt.Sprintf("%s}\n", spacing)
			}

		case "out", "return":
		
			var argsvalue string;
			if (returnParamCount > 1) {				
				argsvalue = fmt.Sprintf ("outObject->Set (String::NewFromUtf8 (isolate, \"%s\"), ", param.ParamName);
				
			} else {
				argsvalue = "args.GetReturnValue().Set (";
			}
				

			switch param.ParamType {
			case "uint8":
				returndeclaration = returndeclaration + fmt.Sprintf("%sunsigned char nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::NewFromUnsigned (isolate, nReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "uint16":
				returndeclaration = returndeclaration + fmt.Sprintf("%sunsigned short nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::NewFromUnsigned (isolate, nReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "uint32":
				returndeclaration = returndeclaration + fmt.Sprintf("%sunsigned int nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::NewFromUnsigned (isolate, nReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "uint64":
				returndeclaration = returndeclaration + fmt.Sprintf("%suint64_t nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sString::NewFromUtf8 (isolate, std::to_string (nReturn%s).c_str()));\n", spacing, argsvalue, param.ParamName)
			
			case "pointer":
				returndeclaration = returndeclaration + fmt.Sprintf("%suint64_t nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sString::NewFromUtf8 (isolate, std::to_string (nReturn%s).c_str()));\n", spacing, argsvalue, param.ParamName)

			case "int8":
				returndeclaration = returndeclaration + fmt.Sprintf("%schar nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::New (isolate, nReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "int16":
				returndeclaration = returndeclaration + fmt.Sprintf("%s short nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::New (isolate, nReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "int32":
				returndeclaration = returndeclaration + fmt.Sprintf("%s int nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::New (isolate, nReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "int64":
				returndeclaration = returndeclaration + fmt.Sprintf("%s int64_t nReturn%s = 0;\n", spacing, param.ParamName)
				callParameter = "&nReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sString::NewFromUtf8 (isolate, std::to_string (nReturn%s).c_str() ));\n", spacing, argsvalue, param.ParamName)

			case "string":
				requiresInitCall = true;
			
				returndeclaration = returndeclaration + fmt.Sprintf("%sunsigned int bytesNeeded%s = 0;\n", spacing, param.ParamName)
				returndeclaration = returndeclaration + fmt.Sprintf("%sunsigned int bytesWritten%s = 0;\n", spacing, param.ParamName)
				initCallParameter = fmt.Sprintf("0, &bytesNeeded%s, nullptr", param.ParamName);

				functioncode = functioncode + fmt.Sprintf("%sstd::vector<char> buffer%s;\n", spacing, param.ParamName)
				functioncode = functioncode + fmt.Sprintf("%sbuffer%s.resize(bytesNeeded%s);\n", spacing, param.ParamName, param.ParamName)

				callParameter = fmt.Sprintf("bytesNeeded%s, &bytesWritten%s, &buffer%s[0]", param.ParamName, param.ParamName, param.ParamName)

				returncode = returncode + fmt.Sprintf("%s%sString::NewFromUtf8 (isolate, &buffer%s[0]));\n", spacing, argsvalue, param.ParamName);
			
			case "bool":
				returndeclaration = returndeclaration + fmt.Sprintf("%sbool bReturn%s = false;\n", spacing, param.ParamName)
				callParameter = "&bReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sBoolean::New (isolate, bReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "single":
				returndeclaration = returndeclaration + fmt.Sprintf("%sfloat fReturn%s = 0.0f;\n", spacing, param.ParamName)
				callParameter = "&fReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sNumber::New (isolate, (double) fReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "double":
				returndeclaration = returndeclaration + fmt.Sprintf("%sdouble dReturn%s = 0.0;\n", spacing, param.ParamName)
				callParameter = "&dReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sNumber::New (isolate, dReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "enum":
				returndeclaration = returndeclaration + fmt.Sprintf("%se%s%s eReturn%s;\n", spacing, NameSpace, param.ParamClass, param.ParamName)
				callParameter = "&eReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%s%sInteger::New (isolate, (int) eReturn%s));\n", spacing, argsvalue, param.ParamName)

			case "struct":
				returndeclaration = returndeclaration + fmt.Sprintf("%ss%s%s sReturn%s;\n", spacing, NameSpace, param.ParamClass, param.ParamName)
				callParameter = "&sReturn" + param.ParamName
				initCallParameter = callParameter;
				
				returncode = returncode + fmt.Sprintf("%s%sconvert%s%sToObject (isolate, sReturn%s));\n", spacing, argsvalue, NameSpace, param.ParamClass, param.ParamName)

			case "basicarray":
				callParameter = "0, nullptr, nullptr";
				initCallParameter = callParameter;

			case "structarray":
				callParameter = "0, nullptr, nullptr";
				initCallParameter = callParameter;

			case "functiontype":
				callParameter = "nullptr";
				initCallParameter = callParameter;
				
			case "class", "optionalclass":
				returndeclaration = returndeclaration + fmt.Sprintf("%s%sHandle hReturn%s = nullptr;\n", spacing, NameSpace, param.ParamName)
				callParameter = "&hReturn" + param.ParamName
				initCallParameter = callParameter;

				returncode = returncode + fmt.Sprintf("%sLocal<Object> instanceObj%s = C%s%s::NewInstance (args.Holder(), hReturn%s);\n", spacing, param.ParamName, NameSpace, param.ParamClass, param.ParamName)
				returncode = returncode + fmt.Sprintf("%s%sinstanceObj%s);\n", spacing, argsvalue, param.ParamName)

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)

			}
		}
		
		if callParameters != "" {
			callParameters = callParameters + ", " + callParameter;
		} else {
			callParameters = callParameters + callParameter;
		}

		if initCallParameters != "" {
			initCallParameters = initCallParameters + ", " + initCallParameter;
		} else {
			initCallParameters = initCallParameters + initCallParameter;
		}
		
	}

	if (returnParamCount > 1) {
		returncode = returncode + fmt.Sprintf ("%sargs.GetReturnValue().Set (outObject);\n", spacing);
	}
	
	
	fmt.Fprintf(implw, "\n")

	fmt.Fprintf(implw, "void C%s%s::%s (const FunctionCallbackInfo<Value>& args) \n", NameSpace, ClassName, method.MethodName)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    Isolate* isolate = args.GetIsolate();\n")
	fmt.Fprintf(implw, "    HandleScope scope(isolate);\n")
	fmt.Fprintf(implw, "    try {\n")

	fmt.Fprintf(implw, inputcheck)
	fmt.Fprintf(implw, inputdeclaration)
	
	fmt.Fprintf(implw, returndeclaration)

	fmt.Fprintf(implw, "%ss%sDynamicWrapperTable * wrapperTable = C%sBaseClass::getDynamicWrapperTable (args.Holder());\n", spacing, NameSpace, NameSpace)
	fmt.Fprintf(implw, "%sif (wrapperTable == nullptr)\n", spacing)
	fmt.Fprintf(implw, "%s    throw std::runtime_error (\"Could not get wrapper table for %s method %s.\");\n", spacing, NameSpace, method.MethodName)

	if isGlobal {
		fmt.Fprintf(implw, "%sif (wrapperTable->m_%s == nullptr)\n", spacing, method.MethodName)
		fmt.Fprintf(implw, "%s    throw std::runtime_error (\"Could not call %s method %s.\");\n", spacing, NameSpace, method.MethodName)
	} else {
		fmt.Fprintf(implw, "%sif (wrapperTable->m_%s_%s == nullptr)\n", spacing, ClassName, method.MethodName)
		fmt.Fprintf(implw, "%s    throw std::runtime_error (\"Could not call %s method %s::%s.\");\n", spacing, NameSpace, ClassName, method.MethodName)
	}
	
		
	if (!isGlobal) {
		fmt.Fprintf(implw, "%s%sHandle instanceHandle = C%sBaseClass::getHandle (args.Holder());\n", spacing, NameSpace, NameSpace);
	}
	
	if (requiresInitCall) {
	
		if isGlobal {
			fmt.Fprintf(implw, "%s%sResult initErrorCode = wrapperTable->m_%s (%s);\n", spacing, NameSpace, method.MethodName, initCallParameters)

		} else {
			if initCallParameters != "" {
				initCallParameters = ", " + initCallParameters
			}
			fmt.Fprintf(implw, "%s%sResult initErrorCode = wrapperTable->m_%s_%s (instanceHandle%s);\n", spacing, NameSpace, ClassName, method.MethodName, initCallParameters)
		}
	
	
		if (isGlobal) {
			fmt.Fprintf(implw, "%sCheckError (isolate, wrapperTable, nullptr, initErrorCode);\n", spacing)
		} else {
			fmt.Fprintf(implw, "%sCheckError (isolate, wrapperTable, instanceHandle, initErrorCode);\n", spacing)
		}
	}
	
	fmt.Fprintf(implw, functioncode)
	
	if isGlobal {
		fmt.Fprintf(implw, "%s%sResult errorCode = wrapperTable->m_%s (%s);\n", spacing, NameSpace, method.MethodName, callParameters)

	} else {
		if callParameters != "" {
			callParameters = ", " + callParameters
		}
		fmt.Fprintf(implw, "%s%sResult errorCode = wrapperTable->m_%s_%s (instanceHandle%s);\n", spacing, NameSpace, ClassName, method.MethodName, callParameters)
	}
	
	
	if (isGlobal) {
		fmt.Fprintf(implw, "%sCheckError (isolate, wrapperTable, nullptr, errorCode);\n", spacing)
	} else {
		fmt.Fprintf(implw, "%sCheckError (isolate, wrapperTable, instanceHandle, errorCode);\n", spacing)
	}

	fmt.Fprintf(implw, returncode)

	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "    } catch (std::exception & E) {\n")
	fmt.Fprintf(implw, "        RaiseError (isolate, E.what());\n")
	fmt.Fprintf(implw, "    }\n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")

	return nil
}

func buildNodeStructConversion(structdefinition ComponentDefinitionStruct, implw io.Writer, NameSpace string) error {

	hasRowVariable := false;
	hasColumnVariable := false;
	
	for i := 0; i < len(structdefinition.Members); i++ {
	
		member := structdefinition.Members[i];
		if (member.Rows > 0) {			
			hasRowVariable = true;
		
			if (member.Columns > 0) {			
				hasColumnVariable = true;			
			}
		}
		
	}
	

	fmt.Fprintf(implw, "/*************************************************************************************************************************\n")
	fmt.Fprintf(implw, " Class s%s%s Conversion\n", NameSpace, structdefinition.Name)
	fmt.Fprintf(implw, "**************************************************************************************************************************/\n")
	
	fmt.Fprintf(implw, "");
	fmt.Fprintf(implw, "s%s%s convertObjectTo%s%s (Isolate* isolate, const Local<Value> & pParamValue)\n", NameSpace, structdefinition.Name, NameSpace, structdefinition.Name);
	fmt.Fprintf(implw, "{\n");
	fmt.Fprintf(implw, "  s%s%s s%s;\n", NameSpace, structdefinition.Name, structdefinition.Name);
	fmt.Fprintf(implw, "  Local<Context> context = isolate->GetCurrentContext();\n");	
	
	if (hasRowVariable) {
		fmt.Fprintf(implw, "  int rowIndex;\n");	
	}
	if (hasColumnVariable) {
		fmt.Fprintf(implw, "  int columnIndex;\n");	
	}
	
	fmt.Fprintf(implw, "\n");
	
	for i := 0; i < len(structdefinition.Members); i++ {
	
		member := structdefinition.Members[i];
		defaultValue, err := GetCMemberDefaultValue (member.Type, member.Class, NameSpace);
		if err != nil {
			return err;
		}
		
		defaultValueAssignment := " = " + defaultValue;
		if (member.Type == "enum") {
			defaultValueAssignment = ".m_code = " + defaultValue;
		}
		
		if (member.Rows > 0) {
			if (member.Columns > 0) {										
				fmt.Fprintf(implw, "  for (columnIndex = 0; columnIndex < %d; columnIndex++)\n", member.Columns);
				fmt.Fprintf(implw, "    for (rowIndex = 0; rowIndex < %d; rowIndex++)\n", member.Rows);
				fmt.Fprintf(implw, "      s%s.m_%s[columnIndex][rowIndex]%s;\n", structdefinition.Name, member.Name, defaultValueAssignment);
			} else {
				fmt.Fprintf(implw, "  for (rowIndex = 0; rowIndex < %d; rowIndex++)\n", member.Rows);
				fmt.Fprintf(implw, "    s%s.m_%s[rowIndex]%s;\n", structdefinition.Name, member.Name, defaultValueAssignment);
			}
		} else {
		
			fmt.Fprintf(implw, "  s%s.m_%s%s;\n", structdefinition.Name, member.Name, defaultValueAssignment);
		}
		
		
	}
	
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "  if (pParamValue->IsObject ()) {\n");
	fmt.Fprintf(implw, "    MaybeLocal<Object> maybeObject = pParamValue->ToObject(context);\n");
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "    if (!maybeObject.IsEmpty ()) {\n");
	fmt.Fprintf(implw, "      Local<Object> obj = maybeObject.ToLocalChecked();\n");
	fmt.Fprintf(implw, "\n");

	for i := 0; i < len(structdefinition.Members); i++ {

		member := structdefinition.Members[i];		
		fmt.Fprintf(implw, "      // %s Member\n", member.Name);

		fmt.Fprintf(implw, "      MaybeLocal<Value> maybeVal%s = obj->Get(context, String::NewFromUtf8(isolate, \"%s\"));\n", member.Name, member.Name);
		fmt.Fprintf(implw, "      if (!maybeVal%s.IsEmpty ()) {\n", member.Name);
		fmt.Fprintf(implw, "        Local<Value> val%s = maybeVal%s.ToLocalChecked ();\n", member.Name, member.Name);
		
		valueTypeCall := "";
		assignmentOperator := " = ";
		switch (member.Type) {
			case "uint8", "uint16", "uint32":
				valueTypeCall = "Uint32Value";				
			case "int8", "int16", "int32":
				valueTypeCall = "Int32Value";		
			case "enum":
				valueTypeCall = "Int32Value";
				assignmentOperator = ".m_code = ";
				
			case "uint64", "int64":
				valueTypeCall = "IntegerValue";				
			case "pointer":
				valueTypeCall = "IntegerValue";				
				assignmentOperator = " = (void *)";
			case "bool":
				valueTypeCall = "BooleanValue";				
			case "single":
				assignmentOperator = " = (float)";
				valueTypeCall = "NumberValue";
			case "double":
				valueTypeCall = "NumberValue";
		}
		
		
		if (member.Rows > 0) {

			fmt.Fprintf(implw, "        if (val%s->IsArray ()) {\n", member.Name);
			fmt.Fprintf(implw, "          Local<Array> array%s = Local<Array>::Cast(val%s);\n", member.Name, member.Name);
			
			if (member.Columns > 0) {
					
				fmt.Fprintf(implw, "          for (int colIndex = 0; colIndex < %d; colIndex++) {\n", member.Columns);
				fmt.Fprintf(implw, "            MaybeLocal<Value> mlocalCol = array%s->Get(context, colIndex);\n", member.Name);
				fmt.Fprintf(implw, "            Local<Value> localCol;\n");
				fmt.Fprintf(implw, "            if (mlocalCol.ToLocal (&localCol)) {\n");
				fmt.Fprintf(implw, "        	  if (localCol->IsArray ()) {\n");
				fmt.Fprintf(implw, "                Local<Array> localColArray = Local<Array>::Cast(localCol);\n");
				fmt.Fprintf(implw, "                for (int rowIndex = 0; rowIndex < %d; rowIndex++) {\n", member.Rows);
				fmt.Fprintf(implw, "                  MaybeLocal<Value> mlocalValue = localColArray->Get(context, rowIndex);\n");
				fmt.Fprintf(implw, "                  Local<Value> localValue;\n");
  				fmt.Fprintf(implw, "                  if (mlocalValue.ToLocal (&localValue)) {\n");
				fmt.Fprintf(implw, "                    if (localValue->IsNumber ()) {\n");
				fmt.Fprintf(implw, "                      MaybeLocal<Number> localNumber = localValue->ToNumber(context);\n");
				fmt.Fprintf(implw, "                      s%s.m_%s[colIndex][rowIndex]%slocalNumber.ToLocalChecked()->%s ();\n", structdefinition.Name, member.Name, assignmentOperator, valueTypeCall);
				fmt.Fprintf(implw, "                    } else {\n");
				fmt.Fprintf(implw, "                      isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s array entry is not a number\" )));\n", member.Name);
				fmt.Fprintf(implw, "                    }\n");
				fmt.Fprintf(implw, "                  } else {\n");
				fmt.Fprintf(implw, "                    isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s array entry is invalid\" )));\n", member.Name);
				fmt.Fprintf(implw, "                  }\n");					
				fmt.Fprintf(implw, "                }\n");
				fmt.Fprintf(implw, "              } else {\n");
				fmt.Fprintf(implw, "                isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s array entry is not an array\" )));\n", member.Name);
				fmt.Fprintf(implw, "              }\n");
				fmt.Fprintf(implw, "            } else {\n");
				fmt.Fprintf(implw, "              isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s array entry is invalid\" )));\n", member.Name);
				fmt.Fprintf(implw, "            }\n");
				fmt.Fprintf(implw, "          }\n");
				
				
					
			} else {

				fmt.Fprintf(implw, "          for (int rowIndex = 0; rowIndex < %d; rowIndex++) {\n", member.Rows);
				fmt.Fprintf(implw, "            MaybeLocal<Value> mlocalValue = array%s->Get(context, rowIndex);\n", member.Name);
				fmt.Fprintf(implw, "            Local<Value> localValue;\n");
  				fmt.Fprintf(implw, "            if (mlocalValue.ToLocal (&localValue)) {\n");
				fmt.Fprintf(implw, "              if (localValue->IsNumber ()) {\n");
				fmt.Fprintf(implw, "                MaybeLocal<Number> localNumber = localValue->ToNumber(context);\n");
				fmt.Fprintf(implw, "                s%s.m_%s[rowIndex]%slocalNumber.ToLocalChecked()->%s ();\n", structdefinition.Name, member.Name, assignmentOperator, valueTypeCall);
				fmt.Fprintf(implw, "              } else {\n");
				fmt.Fprintf(implw, "                isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s array entry is not a number\" )));\n", member.Name);
				fmt.Fprintf(implw, "              }\n");					
				fmt.Fprintf(implw, "            } else {\n");
				fmt.Fprintf(implw, "              isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s array entry is invalid\" )));\n", member.Name);
				fmt.Fprintf(implw, "            }\n");					
				fmt.Fprintf(implw, "          }\n");
			
			}

			fmt.Fprintf(implw, "        } else {\n");	
			fmt.Fprintf(implw, "          isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s member is not an array\" )));\n", member.Name);
			fmt.Fprintf(implw, "        }\n");			
			
			
		} else {
			fmt.Fprintf(implw, "        if (val%s->IsNumber ()) {\n", member.Name);
			fmt.Fprintf(implw, "          MaybeLocal<Number> localVal%s = val%s->ToNumber(context);\n", member.Name, member.Name);
			fmt.Fprintf(implw, "          s%s.m_%s%slocalVal%s.ToLocalChecked()->%s ();\n", structdefinition.Name, member.Name, assignmentOperator, member.Name, valueTypeCall);
			fmt.Fprintf(implw, "        } else {\n");	
			fmt.Fprintf(implw, "          isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s member is not a number\" )));\n", member.Name);
			fmt.Fprintf(implw, "        }\n");			
		}	
		
		fmt.Fprintf(implw, "      } else {\n");
		fmt.Fprintf(implw, "        isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"%s member not found in object\" )));\n", member.Name);
		fmt.Fprintf(implw, "      }\n");
		fmt.Fprintf(implw, "\n");
		
	}
							
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "    } else {\n");
	fmt.Fprintf(implw, "      isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"invalid object passed.\" )));\n");
	fmt.Fprintf(implw, "    }\n");
	fmt.Fprintf(implw, "  } else {\n");
	fmt.Fprintf(implw, "    isolate->ThrowException(Exception::TypeError (String::NewFromUtf8(isolate, \"expected object parameter.\" )));\n");
	fmt.Fprintf(implw, "  }\n");
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "  return s%s;\n", structdefinition.Name);	
	fmt.Fprintf(implw, "}\n");
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "\n");
	
	
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "Local<Object> convert%s%sToObject (Isolate* isolate, s%s%s s%s)\n", NameSpace, structdefinition.Name, NameSpace, structdefinition.Name, structdefinition.Name);
	fmt.Fprintf(implw, "{\n");
	fmt.Fprintf(implw, "  Local<Object> returnInstance = Object::New(isolate);\n");
	
	for i := 0; i < len(structdefinition.Members); i++ {
	
		member := structdefinition.Members[i];		

		conversionCall := "";
		conversionValue := fmt.Sprintf ("s%s.m_%s", structdefinition.Name, member.Name);
		conversionPostfix := "";
		switch (member.Type) {
			case "uint8", "uint16", "uint32":
				conversionCall = "Integer::NewFromUnsigned";
			case "int8", "int16", "int32":
				conversionCall = "Integer::New";
			case "enum":
				conversionCall = "Integer::New";
				conversionPostfix = ".m_code";
			case "uint64", "int64":
				conversionCall = "String::NewFromUtf8";
				conversionValue = "std::to_string (" + conversionValue;
				conversionPostfix = ").c_str()";
			case "pointer":
				conversionCall = "String::NewFromUtf8";
				conversionValue = "std::to_string ((uint_ptr) " + conversionValue;
				conversionPostfix = ").c_str()";
			case "bool":
				conversionCall = "Boolean::New";
			case "single":
				conversionCall = "Number::New";				
				conversionValue = "(double) " + conversionValue;
			case "double":
				conversionCall = "Number::New";
				
		}
	
		if (member.Rows > 0) {
			
			if (member.Columns > 0) {


				fmt.Fprintf(implw, "  Local<Array> new%s = Array::New (isolate, %d);\n", member.Name, member.Columns);
				fmt.Fprintf(implw, "  for (int colIndex = 0; colIndex < %d; colIndex++) {\n", member.Columns);
				fmt.Fprintf(implw, "    Local<Array> colArray = Array::New (isolate, %d);\n", member.Rows);
				fmt.Fprintf(implw, "    for (int rowIndex = 0; rowIndex < %d; rowIndex++) {\n", member.Rows);	
				fmt.Fprintf(implw, "      colArray->Set (rowIndex, %s (isolate, %s[colIndex][rowIndex]%s));\n", conversionCall, conversionValue, conversionPostfix);
				fmt.Fprintf(implw, "    }\n");
				fmt.Fprintf(implw, "    new%s->Set (colIndex, colArray);\n", member.Name);
				fmt.Fprintf(implw, "  }\n");
				fmt.Fprintf(implw, "  returnInstance->Set (String::NewFromUtf8 (isolate, \"%s\"), new%s);\n", member.Name, member.Name);
				fmt.Fprintf(implw, "\n");

			} else {

				fmt.Fprintf(implw, "  Local<Array> new%s = Array::New (isolate, %d);\n", member.Name, member.Rows);
				fmt.Fprintf(implw, "  for (int rowIndex = 0; rowIndex < %d; rowIndex++) {\n", member.Rows);	
				fmt.Fprintf(implw, "    new%s->Set (rowIndex, %s (isolate, %s[rowIndex]%s));\n", member.Name, conversionCall, conversionValue, conversionPostfix);
				fmt.Fprintf(implw, "  }\n");
				fmt.Fprintf(implw, "  returnInstance->Set (String::NewFromUtf8 (isolate, \"%s\"), new%s);\n", member.Name, member.Name);
				fmt.Fprintf(implw, "\n");

			}
			
		} else {
			fmt.Fprintf(implw, "  returnInstance->Set (String::NewFromUtf8 (isolate, \"%s\"), %s (isolate, %s%s));\n", member.Name, conversionCall, conversionValue, conversionPostfix);
		}
		
	}
	
	fmt.Fprintf(implw, "\n");
	fmt.Fprintf(implw, "  return returnInstance;\n");	
	fmt.Fprintf(implw, "}\n");
	fmt.Fprintf(implw, "\n");
	
	 	

	return nil;
	
}




func buildNodeWrapperClass(component ComponentDefinition, w io.Writer, implw io.Writer, NameSpace string, BaseName string) error {

	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "#ifndef %s_NODEWRAPPER_H\n", strings.ToUpper(NameSpace))
	fmt.Fprintf(w, "#define %s_NODEWRAPPER_H\n", strings.ToUpper(NameSpace))
	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "#include \"%s_dynamic.h\"\n", strings.ToLower(NameSpace))
	fmt.Fprintf(w, "#include <node.h>\n")
	fmt.Fprintf(w, "#include <node_object_wrap.h>\n")
	fmt.Fprintf(w, "#include <string>\n")
	fmt.Fprintf(w, "\n")

	fmt.Fprintf(w, "#define NODEWRAPPER_FIELDCOUNT 4\n")
	fmt.Fprintf(w, "#define NODEWRAPPER_TABLEINDEX 2\n")
	fmt.Fprintf(w, "#define NODEWRAPPER_HANDLEINDEX 3\n")

	fmt.Fprintf(w, "/*************************************************************************************************************************\n")
	fmt.Fprintf(w, " Forward declarations \n")
	fmt.Fprintf(w, "**************************************************************************************************************************/\n")

	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "class C%sWrapper;\n", NameSpace)
	fmt.Fprintf(w, "\n")

	fmt.Fprintf(w, "/*************************************************************************************************************************\n")
	fmt.Fprintf(w, " Class C%sBaseClass \n", NameSpace)
	fmt.Fprintf(w, "**************************************************************************************************************************/\n")
	fmt.Fprintf(w, "\n")

	fmt.Fprintf(w, "class C%sBaseClass : public node::ObjectWrap {\n", NameSpace)
	fmt.Fprintf(w, "private:\n")
	fmt.Fprintf(w, "protected:\n")
	fmt.Fprintf(w, "public:\n")
	fmt.Fprintf(w, "    C%sBaseClass ();\n", NameSpace)
	fmt.Fprintf(w, "    static void RaiseError (v8::Isolate * isolate, std::string Message);\n")
	fmt.Fprintf(w, "    static void CheckError (v8::Isolate * isolate, s%sDynamicWrapperTable * sWrapperTable, %sHandle pInstance, %sResult errorCode);\n", NameSpace, NameSpace, NameSpace)
	fmt.Fprintf(w, "    static void setHandle (%sHandle pHandle);\n", NameSpace)
	fmt.Fprintf(w, "    static %sHandle getHandle (v8::Handle<v8::Object> objecthandle);\n", NameSpace)
	fmt.Fprintf(w, "    static s%sDynamicWrapperTable * getDynamicWrapperTable (v8::Handle<v8::Object> objecthandle);\n", NameSpace)
	fmt.Fprintf(w, "};\n")
	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "\n")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		fmt.Fprintf(w, "/*************************************************************************************************************************\n")
		fmt.Fprintf(w, " Class C%s%s \n", NameSpace, class.ClassName)
		fmt.Fprintf(w, "**************************************************************************************************************************/\n")
		fmt.Fprintf(w, "class C%s%s : public C%sBaseClass {\n", NameSpace, class.ClassName, NameSpace)
		fmt.Fprintf(w, "private:\n")
		fmt.Fprintf(w, "    static void New(const v8::FunctionCallbackInfo<v8::Value>& args);\n")
		fmt.Fprintf(w, "    static v8::Persistent<v8::Function> constructor;\n")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			fmt.Fprintf(w, "    static void %s (const v8::FunctionCallbackInfo<v8::Value>& args);\n", method.MethodName)
		}

		fmt.Fprintf(w, "\n")
		fmt.Fprintf(w, "public:\n")
		fmt.Fprintf(w, "    C%s%s ();\n", NameSpace, class.ClassName)
		fmt.Fprintf(w, "    ~C%s%s ();\n", NameSpace, class.ClassName)
		fmt.Fprintf(w, "    \n")
		fmt.Fprintf(w, "    static void Init();\n")
		fmt.Fprintf(w, "    static v8::Local<v8::Object> NewInstance(v8::Local<v8::Object>, %sHandle pHandle);\n", NameSpace)
		fmt.Fprintf(w, "    \n")
		fmt.Fprintf(w, "};\n")
		fmt.Fprintf(w, "\n")

	}

	fmt.Fprintf(w, "/*************************************************************************************************************************\n")
	fmt.Fprintf(w, " Class C%sWrapper \n", NameSpace)
	fmt.Fprintf(w, "**************************************************************************************************************************/\n")

	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "class C%sWrapper : public C%sBaseClass {\n", NameSpace, NameSpace)
	fmt.Fprintf(w, "private:\n")
	fmt.Fprintf(w, "    static void New(const v8::FunctionCallbackInfo<v8::Value>& args);\n")
	fmt.Fprintf(w, "    static v8::Persistent<v8::Function> constructor;\n")

	global := component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		fmt.Fprintf(w, "    static void %s(const v8::FunctionCallbackInfo<v8::Value>& args);\n", method.MethodName)
	}

	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "public:\n")
	fmt.Fprintf(w, "    C%sWrapper ();\n", NameSpace)
	fmt.Fprintf(w, "    ~C%sWrapper();\n", NameSpace)
	fmt.Fprintf(w, "    static void Init();\n")
	fmt.Fprintf(w, "    static v8::Local<v8::Object> NewInstance();\n")
	fmt.Fprintf(w, "};\n")

	fmt.Fprintf(w, "\n")
	fmt.Fprintf(w, "#endif // %s_NODEWRAPPER_H\n", strings.ToUpper(NameSpace))
	fmt.Fprintf(w, "\n")

	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "#include <node.h>\n")
	fmt.Fprintf(implw, "#include \"%s_nodewrapper.h\"\n", BaseName)
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "using namespace v8;\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "Persistent<Function> C%sWrapper::constructor;\n", NameSpace)
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		fmt.Fprintf(implw, "Persistent<Function> C%s%s::constructor;\n", NameSpace, class.ClassName)
	}
	fmt.Fprintf(implw, "\n")
	


	for i := 0; i < len(component.Structs); i++ {
		structdefinition := component.Structs[i];
		err := buildNodeStructConversion (structdefinition, implw, NameSpace);
		if (err != nil) {
			return err;
		}
	}
	
	

	fmt.Fprintf(implw, "/*************************************************************************************************************************\n")
	fmt.Fprintf(implw, " Class C%sBaseClass Implementation\n", NameSpace)
	fmt.Fprintf(implw, "**************************************************************************************************************************/\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "C%sBaseClass::C%sBaseClass ()\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "    : node::ObjectWrap ()\n")
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "void C%sBaseClass::RaiseError (v8::Isolate * isolate, std::string Message)\n", NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    if (isolate != nullptr) {	\n")
	fmt.Fprintf(implw, "        isolate->ThrowException(Exception::TypeError(\n")
	fmt.Fprintf(implw, "            String::NewFromUtf8(isolate, Message.c_str() )));\n")
	fmt.Fprintf(implw, "    }\n")
	fmt.Fprintf(implw, "}\n")

	fmt.Fprintf(implw, "void C%sBaseClass::CheckError (v8::Isolate * isolate, s%sDynamicWrapperTable * sWrapperTable, %sHandle pInstance, %sResult errorCode)\n", NameSpace, NameSpace, NameSpace, NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    if (errorCode != 0) {	\n")
	fmt.Fprintf(implw, "      std::string sMessage;\n")
	
	if len (component.Global.ErrorMethod) > 0 {
		
		fmt.Fprintf(implw, "      if ((sWrapperTable != nullptr) && (pInstance != nullptr)) {\n")
		fmt.Fprintf(implw, "        if (sWrapperTable->m_%s != nullptr) {\n", component.Global.ErrorMethod)
		fmt.Fprintf(implw, "          uint32_t neededChars = 0;\n")
		fmt.Fprintf(implw, "          bool hasLastError = 0;\n")
		fmt.Fprintf(implw, "          if (sWrapperTable->m_%s (pInstance, 0, &neededChars, nullptr, &hasLastError) == 0) {\n", component.Global.ErrorMethod)
		fmt.Fprintf(implw, "            uint32_t dummyChars = 0;\n")
		fmt.Fprintf(implw, "            std::vector<char> Buffer;\n")
		fmt.Fprintf(implw, "            Buffer.resize (neededChars + 2);\n")
		fmt.Fprintf(implw, "            if (sWrapperTable->m_%s (pInstance, neededChars + 1, &dummyChars, Buffer.data(), &hasLastError) == 0) {\n", component.Global.ErrorMethod)
		fmt.Fprintf(implw, "              Buffer[neededChars + 1] = 0;\n")
		fmt.Fprintf(implw, "              sMessage = std::string (\": \") + std::string (&Buffer[0]);\n")
		fmt.Fprintf(implw, "            }\n")
		fmt.Fprintf(implw, "          }\n")
		fmt.Fprintf(implw, "        }\n")
		fmt.Fprintf(implw, "      }\n")
		
	}
	fmt.Fprintf(implw, "      throw std::runtime_error (\"%s Error\" + sMessage + \" (\" + std::to_string (errorCode) + \")\");\n", NameSpace)
	fmt.Fprintf(implw, "    }\n")
	fmt.Fprintf(implw, "}\n")

	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "void C%sBaseClass::setHandle (%sHandle pHandle)\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    \n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "%sHandle C%sBaseClass::getHandle (v8::Handle<v8::Object> objecthandle)\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    auto Field = objecthandle->GetInternalField (NODEWRAPPER_HANDLEINDEX);\n")
	fmt.Fprintf(implw, "    v8::Local<v8::External> externalField = Field.As<v8::External> ();\n")
	fmt.Fprintf(implw, "    return (%sHandle *) externalField->Value();\n", NameSpace)
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "s%sDynamicWrapperTable * C%sBaseClass::getDynamicWrapperTable (v8::Handle<v8::Object> objecthandle)\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    auto Field = objecthandle->GetInternalField (NODEWRAPPER_TABLEINDEX);\n")
	fmt.Fprintf(implw, "    v8::Local<v8::External> externalField = Field.As<v8::External> ();\n")
	fmt.Fprintf(implw, "    return (s%sDynamicWrapperTable *) externalField->Value();\n", NameSpace)
	fmt.Fprintf(implw, "}\n")

	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "\n")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		fmt.Fprintf(implw, "/*************************************************************************************************************************\n")
		fmt.Fprintf(implw, " Class C%s%s Implementation\n", NameSpace, class.ClassName)
		fmt.Fprintf(implw, "**************************************************************************************************************************/\n")
		fmt.Fprintf(implw, "\n")

		fmt.Fprintf(implw, "C%s%s::C%s%s ()\n", NameSpace, class.ClassName, NameSpace, class.ClassName)
		fmt.Fprintf(implw, "    : C%sBaseClass ()\n", NameSpace)
		fmt.Fprintf(implw, "{\n")
		fmt.Fprintf(implw, "}\n")
		fmt.Fprintf(implw, "\n")
		fmt.Fprintf(implw, "C%s%s::~C%s%s()\n", NameSpace, class.ClassName, NameSpace, class.ClassName)
		fmt.Fprintf(implw, "{\n")
		fmt.Fprintf(implw, "}\n")
		fmt.Fprintf(implw, "\n")
		fmt.Fprintf(implw, "void C%s%s::Init()\n", NameSpace, class.ClassName)
		fmt.Fprintf(implw, "{\n")
		fmt.Fprintf(implw, "    Isolate* isolate = Isolate::GetCurrent();\n")
		fmt.Fprintf(implw, "\n")
		fmt.Fprintf(implw, "    // Prepare constructor template\n")
		fmt.Fprintf(implw, "    Local<FunctionTemplate> tpl = FunctionTemplate::New(isolate, New);\n")
		fmt.Fprintf(implw, "    tpl->SetClassName(String::NewFromUtf8(isolate, \"%s%s\"));\n", NameSpace, class.ClassName)
		fmt.Fprintf(implw, "    tpl->InstanceTemplate()->SetInternalFieldCount(NODEWRAPPER_FIELDCOUNT);\n")
		fmt.Fprintf(implw, "\n")
		fmt.Fprintf(implw, "    // Prototype\n")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			fmt.Fprintf(implw, "    NODE_SET_PROTOTYPE_METHOD(tpl, \"%s\", %s);\n", method.MethodName, method.MethodName)
		}

		fmt.Fprintf(implw, "    constructor.Reset(isolate, tpl->GetFunction());\n")
		fmt.Fprintf(implw, "\n")
		fmt.Fprintf(implw, "}\n")
		fmt.Fprintf(implw, "\n")

		fmt.Fprintf(implw, "void C%s%s::New(const FunctionCallbackInfo<Value>& args)\n", NameSpace, class.ClassName)
		fmt.Fprintf(implw, "{\n")
		fmt.Fprintf(implw, "    Isolate* isolate = args.GetIsolate();\n")
		fmt.Fprintf(implw, "    HandleScope scope(isolate);\n")
		fmt.Fprintf(implw, "\n")
		fmt.Fprintf(implw, "    if (args.IsConstructCall()) {\n")
		fmt.Fprintf(implw, "        C%sBaseClass * holderObj = ObjectWrap::Unwrap<C%sBaseClass>(args.Holder());\n", NameSpace, NameSpace)
		fmt.Fprintf(implw, "        C%s%s * %sInstance = new C%s%s ();\n", NameSpace, class.ClassName, strings.ToLower(class.ClassName), NameSpace, class.ClassName)

		//fmt.Fprintf (implw, "        Local<Object> localHolderObj = args.Holder();\n");
		//fmt.Fprintf (implw, "        %sInstance->m_pWrapperObject.Reset (localHolderObj);\n", strings.ToLower (class.ClassName));

		fmt.Fprintf(implw, "        %sInstance->Wrap(args.This());\n", strings.ToLower(class.ClassName))
		fmt.Fprintf(implw, "        args.GetReturnValue().Set(args.This());\n")
		fmt.Fprintf(implw, "    } else {\n")
		fmt.Fprintf(implw, "        RaiseError (isolate, \"%s%s: Invalid call to Constructor\");\n", NameSpace, class.ClassName)
		fmt.Fprintf(implw, "    }\n")
		fmt.Fprintf(implw, "}\n")
		fmt.Fprintf(implw, "\n")

		fmt.Fprintf(implw, "Local<Object> C%s%s::NewInstance(Local<Object> pParent, %sHandle pHandle)\n", NameSpace, class.ClassName, NameSpace)
		fmt.Fprintf(implw, "{\n")
		fmt.Fprintf(implw, "    Isolate* isolate = Isolate::GetCurrent();\n")
		fmt.Fprintf(implw, "    HandleScope scope(isolate);\n")
		fmt.Fprintf(implw, "    Local<Function> cons = Local<Function>::New(isolate, constructor);\n")
		fmt.Fprintf(implw, "    Local<Object> instance;\n");
		fmt.Fprintf(implw, "    if (cons->NewInstance(isolate->GetCurrentContext()).ToLocal (&instance)) {\n")
		fmt.Fprintf(implw, "      instance->SetInternalField (NODEWRAPPER_TABLEINDEX, External::New (isolate, C%sBaseClass::getDynamicWrapperTable (pParent)));\n", NameSpace)
		fmt.Fprintf(implw, "      instance->SetInternalField (NODEWRAPPER_HANDLEINDEX, External::New (isolate, pHandle));\n")
		fmt.Fprintf(implw, "    }\n");
		fmt.Fprintf(implw, "    return instance;\n")
		fmt.Fprintf(implw, "}\n")
		fmt.Fprintf(implw, "\n")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writeNodeMethodImplementation(method, implw, NameSpace, class.ClassName, false)
			if err != nil {
				return err
			}
		}
	}

	fmt.Fprintf(implw, "/*************************************************************************************************************************\n")
	fmt.Fprintf(implw, " Class C%sWrapper Implementation\n", NameSpace)
	fmt.Fprintf(implw, "**************************************************************************************************************************/\n")
	fmt.Fprintf(implw, "\n")

	fmt.Fprintf(implw, "C%sWrapper::C%sWrapper()\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "    : C%sBaseClass ()\n", NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "C%sWrapper::~C%sWrapper()\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "void C%sWrapper::Init()\n", NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    Isolate* isolate = Isolate::GetCurrent();\n")
	fmt.Fprintf(implw, "    // Prepare constructor template\n")
	fmt.Fprintf(implw, "    Local<FunctionTemplate> tpl = FunctionTemplate::New(isolate, New);\n")
	fmt.Fprintf(implw, "    tpl->SetClassName(String::NewFromUtf8(isolate, \"%sWrapper\"));\n", NameSpace)
	fmt.Fprintf(implw, "    tpl->InstanceTemplate()->SetInternalFieldCount(NODEWRAPPER_FIELDCOUNT);\n")
	fmt.Fprintf(implw, "    \n")
	fmt.Fprintf(implw, "    // Prototype\n")
	fmt.Fprintf(implw, "    \n")
	global = component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		fmt.Fprintf(implw, "    NODE_SET_PROTOTYPE_METHOD(tpl, \"%s\", %s);\n", method.MethodName, method.MethodName)
	}

	fmt.Fprintf(implw, "    constructor.Reset(isolate, tpl->GetFunction());\n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "void C%sWrapper::New(const FunctionCallbackInfo<Value>& args)\n", NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    Isolate* isolate = args.GetIsolate();\n")
	fmt.Fprintf(implw, "    HandleScope scope(isolate);\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "    try {\n")
	fmt.Fprintf(implw, "        if (args.IsConstructCall()) {\n")

	fmt.Fprintf(implw, "            // Get Library Name as Argument\n")
	fmt.Fprintf(implw, "            std::string sLibraryName = \"%s.dll\";\n", BaseName)
	fmt.Fprintf(implw, "            if (args[0]->IsString()) {\n")
	fmt.Fprintf(implw, "                v8::String::Utf8Value stringParam (args[0]->ToString());\n")
	fmt.Fprintf(implw, "                sLibraryName = *stringParam;\n")
	fmt.Fprintf(implw, "            }\n")

	fmt.Fprintf(implw, "            // Invoked as constructor: `new %sWrapper(...)`\n", NameSpace)
	fmt.Fprintf(implw, "            C%sWrapper* obj = new C%sWrapper ();\n", NameSpace, NameSpace)

	fmt.Fprintf(implw, "            Local<Object> newObject = args.This();\n")
	fmt.Fprintf(implw, "            std::auto_ptr<s%sDynamicWrapperTable> wrapperTable ( new s%sDynamicWrapperTable );\n", NameSpace, NameSpace)
	fmt.Fprintf(implw, "            CheckError (isolate, nullptr, nullptr, Load%sWrapperTable (wrapperTable.get(), sLibraryName.c_str()));\n", NameSpace)
	fmt.Fprintf(implw, "            newObject->SetInternalField (NODEWRAPPER_TABLEINDEX, External::New (isolate, wrapperTable.release ()));\n")

	// write out enums
	for i := 0; i < len(component.Enums); i++ {
		enum := component.Enums[i];		
		for j := 0; j < len(enum.Options); j++ {							
			option := enum.Options[j];
			fmt.Fprintf (implw, "            newObject->Set (String::NewFromUtf8(isolate, \"e%s_%s\"), Integer::New(isolate, %d));\n", enum.Name, option.Name, option.Value);
		}		
	}

	fmt.Fprintf(implw, "            obj->Wrap(newObject);\n")
	fmt.Fprintf(implw, "            args.GetReturnValue().Set(newObject);\n")
	fmt.Fprintf(implw, "        } else {\n")
	fmt.Fprintf(implw, "            // Invoked as plain function `%sWrapper(...)`, turn into construct call.\n", NameSpace)
	fmt.Fprintf(implw, "            const int argc = 1;\n")
	fmt.Fprintf(implw, "            Local<Value> argv[argc] = { args[0] };\n")
	fmt.Fprintf(implw, "            Local<Function> cons = Local<Function>::New(isolate, constructor);\n")
	fmt.Fprintf(implw, "            args.GetReturnValue().Set(cons->NewInstance(isolate->GetCurrentContext(), argc, argv).ToLocalChecked());\n")
	fmt.Fprintf(implw, "        }\n")
	fmt.Fprintf(implw, "    } catch (std::exception & E) {\n")
	fmt.Fprintf(implw, "        RaiseError (isolate, E.what());\n")
	fmt.Fprintf(implw, "    }\n")
	fmt.Fprintf(implw, "}\n")
	fmt.Fprintf(implw, "\n")
	fmt.Fprintf(implw, "Local<Object> C%sWrapper::NewInstance ()\n", NameSpace)
	fmt.Fprintf(implw, "{\n")
	fmt.Fprintf(implw, "    Isolate* isolate = Isolate::GetCurrent();\n")
	fmt.Fprintf(implw, "    HandleScope scope(isolate);\n")
	fmt.Fprintf(implw, "    Local<Function> cons = Local<Function>::New(isolate, constructor);\n")
	fmt.Fprintf(implw, "    Local<Object> instance;\n"); 
	fmt.Fprintf(implw, "    cons->NewInstance(isolate->GetCurrentContext()).ToLocal(&instance);\n")
	fmt.Fprintf(implw, "    return instance;\n")
	fmt.Fprintf(implw, "}\n")

	fmt.Fprintf(implw, "\n")

	global = component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err := writeNodeMethodImplementation(method, implw, NameSpace, "Wrapper", true)
		if err != nil {
			return err
		}
	}

	fmt.Fprintf(implw, "\n")

	return nil
}


func buildNodeBindingGyp(component ComponentDefinition, w io.Writer, indentString string) error {

	BaseName := component.BaseName;

	fmt.Fprintf(w, "{\n")
	fmt.Fprintf(w, "%s\"targets\": [\n", indentString)
	fmt.Fprintf(w, "%s%s{\n", indentString, indentString)
	fmt.Fprintf(w, "%s%s%s\"target_name\": \"%s_nodeaddon\",\n", indentString, indentString, indentString, BaseName)
	fmt.Fprintf(w, "%s%s%s\"sources\": [ \"%s_nodeaddon.cc\", \"%s_nodewrapper.cc\", \"%s_dynamic.cc\" ],\n", indentString, indentString, indentString, BaseName, BaseName, BaseName)
	fmt.Fprintf(w, "%s%s%s\"cflags\": [ \"-fexceptions \" ],\n", indentString, indentString, indentString )
	fmt.Fprintf(w, "%s%s%s\"cflags_cc\": [ \"-fexceptions \" ],\n", indentString, indentString, indentString )
	fmt.Fprintf(w, "%s%s%s\"msvs_settings\": {\n", indentString, indentString, indentString)
	fmt.Fprintf(w, "%s%s%s%s\"VCCLCompilerTool\": { \"ExceptionHandling\": 1 }\n", indentString, indentString, indentString, indentString)
	fmt.Fprintf(w, "%s%s%s},\n", indentString, indentString, indentString)
	fmt.Fprintf(w, "%s%s%s\"conditions\": [\n", indentString, indentString, indentString)
	fmt.Fprintf(w, "%s%s%s%s[\"OS=='win'\", {	\"defines\": [ \"_HAS_EXCEPTIONS=1\" ] }]\n", indentString, indentString, indentString, indentString)
	fmt.Fprintf(w, "%s%s%s]\n", indentString, indentString, indentString)
				
	fmt.Fprintf(w, "%s%s}\n", indentString, indentString)
	fmt.Fprintf(w, "%s]\n", indentString)
	fmt.Fprintf(w, "}\n")
	fmt.Fprintf(w, "\n")
	
	return nil;
  
}
