#include "uocompiler.h"
#include <stdlib.h>
#include <stdio.h> /* rename */
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "str_ulong.h"
#include "uostr.h"
#include "uoio.h"
#include "mtalib.h"

/* quoting bernstein:
 "A program delivers a mail message in six steps. 
  - First, it chdir()s to the maildir directory."
	[done at startup -- uwe]
 "- Second, it stat()s the name tmp/time.pid.host, where time
	is the number of seconds since the beginning of 1970 GMT,
	pid is the program's process ID, and host is the host name.
  - Third, if stat() returned anything other than ENOENT, the
	program sleeps for two seconds, updates time, and tries the
	stat() again, a limited number of times. 
  - Fourth, the program creates tmp/time.pid.host.
  - Fifth, the program NFS-writes the message to the file. 
  - Sixth, the program link()s the file to new/time.pid.host. At
	that instant the message has been successfully delivered.
  The delivery program is required to start a 24-hour timer before
  creating tmp/time.pid.host, and to abort the delivery if the
  timer expires. Upon error, timeout, or normal completion,
  the delivery program may attempt to unlink() tmp/time.pid.host.
  NFS-writing means 
	(1) as usual, checking the number of bytes returned from
		each write() call;
	(2) calling fsync() and checking its return value; 
	(3) calling close() and checking its return value. (Standard
		<F9NFS implementations handle fsync() incorrectly but make
		up for it by abusing close().)"
 Unix is funny. Microsoft must have had a hard time to invent 
 something worse.

 We actually use time.pid_count.host as djb suggested on the web.

 We do not implement the timer stuff. This should be done at 
 a higher level.
*/

/* workaround for aged systems */
#ifndef HAVE_FSYNC
#ifdef O_SYNC
#define NEED_O_SYNC
#define fsync(x) 0
#else
#define fsync(x) sync()
#endif
#endif

struct priv {
	int count;
	char *fname;
	uoio_t o;
	int goterrs;
};

static mta_ret_t mta_maildir_open P__((mta_t **m, const char *path, const char *from, const char *to));
static mta_ret_t mta_maildir_close P__((mta_t *mta));
static mta_ret_t mta_maildir_write P__((mta_t *mta, const char *, size_t));
static mta_ret_t mta_maildir_abort P__((mta_t *mta));
mta_t mta_maildir={
	mta_maildir_open,
	mta_maildir_write,
	mta_maildir_abort,
	mta_maildir_close,
	0
};


static mta_ret_t
mta_maildir_open(mta_t **m, const char *path, const char *from, const char *to)
{
	mta_t *mta;
	int e;
	struct priv *priv;
	static int lfd=0;
	int maxloops=5;
	int fd=-1;
	uostr_t tmpfname;tmpfname.data=0;
	(void)from;
	(void)to;

	if (!mta_hostname && MTA_SUCCESS!=mta_set_hostname(0)) return MTA_DEFER;

	priv=malloc(sizeof(struct priv));
	if (!priv) {errno=ENOMEM; mta_set_err("out of memory","(#4.3.0)",0); return 0; }
	memset(priv,0,sizeof(*priv));

	mta=malloc(sizeof(mta_t));
	if (!mta) {e=ENOMEM; mta_set_err("out of memory","(#4.3.0)",0); goto cleanup1; }
	memset(mta,0,sizeof(*mta));
	mta->mta_open=mta_maildir.mta_open;
	mta->mta_close=mta_maildir.mta_close;
	mta->mta_write=mta_maildir.mta_write;
	mta->mta_abort=mta_maildir.mta_abort;


	/* count them */
	
#define X(y) do {if (!(y)) {e=ENOMEM;mta_set_err("out of memory","(#4.3.0)",0); goto cleanup_close;} } while (0)
	while (1) {
		time_t now=time(0);
		struct stat st;
		char numbuf[STR_ULONG];
		if (maxloops--==0) { e=errno; goto cleanup_close; }
		if (path) {
			X(uostr_dup_cstr(&tmpfname,path));
			X(uostr_dup_char(&tmpfname,'/'));
			X(uostr_dup_cstr(&tmpfname,"tmp/"));
		} else X(uostr_dup_cstr(&tmpfname,"tmp/"));
		str_ulong(numbuf,now); X(uostr_add_cstr(&tmpfname,numbuf));
		X(uostr_add_char(&tmpfname,'.'));
		str_ulong(numbuf,(unsigned long) getpid()); X(uostr_add_cstr(&tmpfname,numbuf)); 
		X(uostr_add_char(&tmpfname,'_'));
		str_ulong(numbuf,(unsigned long) lfd++); X(uostr_add_cstr(&tmpfname,numbuf)); 
		X(uostr_add_char(&tmpfname,'.'));
		X(uostr_add_cstr(&tmpfname,mta_hostname));
		X(uostr_0(&tmpfname));
		if (stat(tmpfname.data,&st)==-1 && errno==ENOENT)
			break;
		sleep(2);
	}

	fd=open(tmpfname.data,O_WRONLY|O_CREAT|O_EXCL
#ifdef NEED_O_SYNC
			|O_SYNC
#endif
			,0600);
	if (fd==-1) { e=errno; mta_set_err("open error: ","(#4.3.0)",1); goto cleanup_close; }
	priv->fname=strdup(tmpfname.data);
	if (!priv->fname) {close(fd);e=ENOMEM;mta_set_err("out of memory","(#4.3.0)",0); goto cleanup_close;}
	uoio_assign_w(&priv->o,fd,write,0);
	/* heya */
	mta->priv=priv;
	*m=mta;
	return MTA_SUCCESS;

  cleanup_close: close(fd);
  	uostr_freedata(&tmpfname);
	free(mta);
  cleanup1: free(priv);
	errno=e;
	return MTA_DEFER;
}

static mta_ret_t 
mta_maildir_abort(mta_t *mta)
{
	struct priv *priv=mta->priv;
	close(priv->o.fd);
	uoio_destroy(&priv->o);
	unlink(priv->fname);
	free(priv->fname);
	free(priv);
	free(mta);
	return MTA_SUCCESS;
}

static mta_ret_t
mta_maildir_write(mta_t *mta,const char *s,size_t len)
{
	struct priv *priv=mta->priv;
	if (priv->goterrs) {mta_set_err("I/O error","(#4.3.0)",0); return MTA_DEFER;}
	priv->goterrs=(-1==uoio_write_mem(&priv->o,s,len));
	if (priv->goterrs) {mta_set_err("I/O error","(#4.3.0)",errno); return MTA_DEFER;}
	return MTA_SUCCESS;
}

static mta_ret_t
mta_maildir_close(mta_t *mta)
{
	struct priv *priv=mta->priv;
	if (-1==uoio_flush(&priv->o)) { int e=errno; mta_maildir_abort(mta); 
		mta_set_err("I/O error","(#4.3.0)",e);errno=e; return MTA_DEFER; 
	} else if (-1==fsync(priv->o.fd)) { int e=errno; mta_maildir_abort(mta); 
		mta_set_err("I/O error: fsync failed","(#4.3.0)",e); errno=e; return -1; 
	} else if (-1==close(priv->o.fd)) { int e=errno; mta_maildir_abort(mta); 
		mta_set_err("I/O error: close failed","(#4.3.0)",e); errno=e; return -1;
	} else {
		uostr_t newfname;newfname.data=0;
		/* ok, tmp file is written */
		if (!uostr_dup_cstr(&newfname,priv->fname)) {int e=errno;mta_maildir_abort(mta);
	       	mta_errtxt="out of memory"; mta_hcmssc="(#4.3.0)"; errno=e; return -1; }
		if (!uostr_add_char(&newfname,0)) {int e=errno;mta_maildir_abort(mta);uostr_freedata(&newfname);
	       	mta_errtxt="out of memory"; mta_hcmssc="(#4.3.0)"; errno=e; return -1; }
		/* be nice to a user who changes tmp/X to /.../tmp/X */
		{char *p=strrchr(newfname.data,'/'); memcpy(p-3,"new",3); }
		if (-1==link(priv->fname,newfname.data)) {
			int e=errno; mta_maildir_abort(mta); uostr_freedata(&newfname); 
			mta_set_err("link failed", "(#4.3.0)", e); errno=e; return -1; }
		unlink(priv->fname);
		free(priv->fname);
		mta_set_err(newfname.data,0,0);
		uoio_destroy(&priv->o);
		free(priv);
		free(mta);

		uostr_freedata(&newfname);
		return MTA_SUCCESS;
	}
}
