/* xmpdvmkpkg.c
	$Id: xmpdvmkpkg.c,v 1.5 2001/09/01 19:01:12 gwiley Exp $
   Glen Wiley <gwiley@ieee.org>

	This file contains the GUI elements of xmpdvmkpkg - utilities
	are elsewhere.

	uses std. X resources, command line parms
	-background <color>
	-foreground <color>
	-title <window title bar text>
	etc.

	exit with non-zero on initialization error

Copyright (c)1999,2000,2001 Glen Wiley

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sub-license, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/

#if HAVE_CONFIG_H
 #include "config.h"
#endif

#include <errno.h>
#if HAVE_FCNTL_H
 #include <fcntl.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <time.h>
#if HAVE_UNISTD_H
 #include <unistd.h>
#endif
#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/FileSB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include "../version.h"
#include "../pdv.h"
#include "xmpdvmkpkg.h"

/*-------------------- constants */

const int TXENTRYWIDTH  = 30;   /* width of file entry lines */
const int INITAPPWIDTH  = 620;
const int INITAPPHEIGHT = 470;
const char *APPNAME     = "xmpdvmkpkg";
String  g_fallbackres[] =
{
	(const String) "*background: lavender",
	(const String) "*foreground: black",
	(const String) "*iconName: Xmpdvmkpkg",
	NULL
};

/*-------------------- global vars */

char                  *g_prg;
XtAppContext          g_context;
Widget                g_toplevel;
Widget                g_statusw  = NULL;
XmStringCharSet       g_char_set = XmSTRING_DEFAULT_CHARSET;
struct pdvspec_st     g_pdvspec;
struct specwidgets_st g_specwidgets;

/*-------------------- functions */

void   buildmainapp(void);
Widget buildmenu(Widget form);
void   buildscreen(Widget form, Widget menubar);

void   quitappCB(Widget w,     void *client_data, void *call_data);
void   open_menuCB(Widget w,   void *client_data, void *call_data);
void   close_menuCB(Widget w,  void *client_data, void *call_data);
void   save_menuCB(Widget w,   void *client_data, void *call_data);
void   saveas_menuCB(Widget w, void *client_data, void *call_data);
void   mkpkg_menuCB(Widget w,  void *client_data, void *call_data);
void   about_menuCB(Widget w,  void *client_data, void *call_data);

void   helptextCB(Widget w,  void *client_data, void *call_data);
void   openfileCB(Widget w,  void *client_data, void *call_data);
void   savefileCB(Widget w,  void *client_data, void *call_data);
void   cancelfsbCB(Widget w, void *client_data, void *call_data);
void   makepackage(void);
Boolean jumpstart(XtPointer filename);

/*---------------------------------------- main */
int
main(int argc, char *argv[])
{
	Arg  al[10];
	int  ac        = 0;

	g_prg = argv[0];

	/* stage/clear some of our global vars */

	g_pdvspec.fn_spec    = NULL;
	g_pdvspec.fn_helpmsg = NULL;
	g_pdvspec.fn_agrmsg  = NULL;
	g_pdvspec.fn_output  = NULL;
	g_pdvspec.fn_payld   = NULL;
	g_pdvspec.fn_pdvstub = NULL;
	g_pdvspec.fn_exec    = NULL;
	g_pdvspec.fn_pkg     = NULL;
	g_pdvspec.compress   = NULL;
	g_pdvspec.tar        = NULL;

	/* setup application X environment */

	ac         = 0;
	g_toplevel = XtOpenApplication(&g_context, APPNAME, NULL, 0, &argc, argv
	 , g_fallbackres, topLevelShellWidgetClass, NULL, 0);

	XtSetArg(al[ac], XmNwidth,  INITAPPWIDTH); ac++;
	XtSetArg(al[ac], XmNheight, INITAPPHEIGHT); ac++;
	XtSetValues(g_toplevel, al, ac);

	/* setup window and application */

	buildmainapp();

	/* if a spec file was specified on the command line the setup a
	   work procedure to read it in once the interface is up */

	if(argc > 1)
	{
		XtAppAddWorkProc(g_context, jumpstart, (XtPointer) argv[1]);
	}

	/* let's go... */

	XtRealizeWidget(g_toplevel);
	XtAppMainLoop(g_context);

	return 0;
} /* main */

/*---------------------------------------- buildmainapp
  construct main window, widgets and perform layout
*/
void
buildmainapp(void)
{
	int    ac;
	Arg    args[20];
	Widget form;
	Widget menubar;

	ac   = 0;
	form = XmCreateForm(g_toplevel, (String) "mainform", args, ac);
	XtManageChild(form);

	menubar = buildmenu(form);

	buildscreen(form, menubar);

	return;
} /* buildmainapp */

/*---------------------------------------- build menu
	this constructs the main pulldown menu
	form = parent form widget for the menu
	returns the menubar that was created (so that other widgets can
	attach to it)
*/
Widget
buildmenu(Widget form)
{
	Arg       args[20];
	int       ac;
	Widget    w;
	Widget    menuw;
	Widget    mitemw;
	Widget    menubarw;
	XmString  lblstr;

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment,   XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNleftAttachment,  XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
	menubarw = XmCreateMenuBar(form, (String) "MenuBar", args, ac);

	/*-------------------- file menu */

	menuw  = XmCreatePulldownMenu(menubarw, "FilePullDown", NULL, 0);
	lblstr = XmStringCreateLocalized("File");
	w      = XtVaCreateManagedWidget("File", xmCascadeButtonWidgetClass
	 , menubarw
	 , XmNsubMenuId,   menuw
	 , XmNlabelString, lblstr
	 , XmNmnemonic,    'F'
	 , NULL);
	XmStringFree(lblstr);

	/* File-->Open */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'O'); ac++;
	mitemw = XtCreateManagedWidget("Open", xmPushButtonGadgetClass, menuw
	 , args, ac);
	XtAddCallback(mitemw, XmNactivateCallback, open_menuCB, NULL);

	/* File-->Close */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'C'); ac++;
	mitemw = XtCreateManagedWidget("Close", xmPushButtonGadgetClass, menuw
	 , args, ac);
	XtAddCallback(mitemw, XmNactivateCallback, close_menuCB, NULL);

	/* File-->Save */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'S'); ac++;
	mitemw = XtCreateManagedWidget("Save", xmPushButtonGadgetClass, menuw
	 , args, ac);
	XtAddCallback(mitemw, XmNactivateCallback, save_menuCB, NULL);

	/* File-->Save As */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'A'); ac++;
	mitemw = XtCreateManagedWidget("Save As", xmPushButtonGadgetClass, menuw
	 , args, ac);
	XtAddCallback(mitemw, XmNactivateCallback, saveas_menuCB, NULL);

	XtVaCreateManagedWidget("sep", xmSeparatorGadgetClass, menuw, NULL);

	/* File-->Make Package */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'M'); ac++;
	mitemw = XtCreateManagedWidget("Make Package", xmPushButtonGadgetClass
	 , menuw, args, ac);
	XtAddCallback(mitemw, XmNactivateCallback, mkpkg_menuCB, NULL);

	XtVaCreateManagedWidget("sep", xmSeparatorGadgetClass, menuw, NULL);

	/* File-->Exit */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'x'); ac++;
	mitemw = XtCreateManagedWidget("Exit", xmPushButtonGadgetClass, menuw, args
	 , ac);
	XtAddCallback(mitemw, XmNactivateCallback, quitappCB, NULL);

	/*-------------------- help menu */

	menuw  = XmCreatePulldownMenu(menubarw, "HelpPullDown", NULL, 0);
	lblstr = XmStringCreateLocalized("Help");
	w      = XtVaCreateManagedWidget("Help", xmCascadeButtonWidgetClass
	 , menubarw
	 , XmNsubMenuId,   menuw
	 , XmNlabelString, lblstr
	 , XmNmnemonic,    'H'
	 , NULL);
	XmStringFree(lblstr);
	XtVaSetValues(menubarw, XmNmenuHelpWidget, w, NULL);

	/* Help-->About */
	ac = 0;
	XtSetArg(args[ac], XmNmnemonic, 'A'); ac++;
	mitemw = XtCreateManagedWidget("About", xmPushButtonGadgetClass, menuw
	 , args, ac);
	XtAddCallback(mitemw, XmNactivateCallback, about_menuCB, NULL);

	XtManageChild(menubarw);

	return menubarw;
} /* buildmenu */

/*---------------------------------------- quitappCB */
void
quitappCB(Widget w, void *client_data, void *call_data)
{
	exit(0);
} /* quitappCB */

/*---------------------------------------- about_menuCB */
void
about_menuCB(Widget w, void *client_data, void *call_data)
{
	static   Widget dialog = NULL;
	int      ac;
	Arg      args[5];
	char     cmsg[250];
	XmString msg;
	XmString title;
	Widget   button;

	if(dialog == NULL)
	{
		sprintf(cmsg, 
		 "xpdvmkpkg\n\n"
		 "%s\n\n"
		 "%s\n\n"
		 "X11/Motif based utility for creating PDV \n"
		 "self-extracting/executing packages."
		 , g_versionstr, g_homepage);
		title = XmStringCreateLocalized("About xmpdvmkpkg");
		msg   = XmStringCreateLocalized(cmsg);

		ac = 0;
		XtSetArg(args[ac], XmNmessageString, msg); ac++;
		XtSetArg(args[ac], XmNdialogTitle,   title); ac++;
		dialog = XmCreateInformationDialog(g_toplevel, "About", args, ac);

		XmStringFree(title);
		XmStringFree(msg);

		button = XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON);
		XtUnmanageChild(button);
		button = XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON);
		XtUnmanageChild(button);
	}

	XtManageChild(dialog);
	XtPopup(XtParent(dialog), XtGrabNone);

	return;
} /* about_menuCB */

/*---------------------------------------- open_menuCB
	this callback is invoked from the file menu, open menu item
*/
void
open_menuCB(Widget w, void *client_data, void *call_data)
{
	Arg      args[10];
	int      ac = 0;
	XmString pat;
	XmString title;
	Widget   hlpb;
	static Widget fsdlgw = NULL;

	if(fsdlgw == NULL)
	{
		pat   = XmStringCreateLocalized("*.pdv");
		title = XmStringCreateLocalized("Open PDV Spec File");

		XtSetArg(args[ac], XmNpattern,      pat); ac++;
		XtSetArg(args[ac], XmNdialogTitle,  title); ac++;
		fsdlgw = XmCreateFileSelectionDialog(g_toplevel, "openfile", args, ac);

		XmStringFree(pat);
		XmStringFree(title);

		XtAddCallback(fsdlgw, XmNokCallback,     openfileCB, fsdlgw);
		XtAddCallback(fsdlgw, XmNcancelCallback, cancelfsbCB, fsdlgw);

		hlpb = XmFileSelectionBoxGetChild(fsdlgw, XmDIALOG_HELP_BUTTON);
		XtUnmanageChild(hlpb);

	}
	XtManageChild(fsdlgw);

	return;
} /* open_menuCB */

/*---------------------------------------- close_menuCB
	this callback is invoked from the file menu, close item
*/
void
close_menuCB(Widget w, void *client_data, void *call_data)
{
	freespecmembers(&g_pdvspec);
	spectowidgets();
	xu_SetMainTitle("", "xpdvmkpkg", g_toplevel);
	statusmsg("spec file closed");

} /* close_menuCB */

/*---------------------------------------- save_menuCB
	this callback is invoked from the file menu, save item
*/
void
save_menuCB(Widget w, void *client_data, void *call_data)
{
	/* if we have no filename then launch the saveas dialog */

	if(g_pdvspec.fn_spec == NULL)
		saveas_menuCB(w, "save", call_data);
	else
		savefileCB(w, NULL, call_data);

} /* save_menuCB */

/*---------------------------------------- saveas_menuCB
	this callback is invoked from the file menu, saveas item
*/
void
saveas_menuCB(Widget w, void *client_data , void *call_data)
{
	Arg      args[10];
	int      ac = 0;
	XmString pat;
	XmString title;
	XmString deffn;
	Widget   hlpb;
	static Widget fsdlgw = NULL;

	if(fsdlgw == NULL)
	{
		pat   = XmStringCreateLocalized("*.pdv");
		title = XmStringCreateLocalized("Save PDV Spec File As");

		XtSetArg(args[ac], XmNpattern,     pat); ac++;
		XtSetArg(args[ac], XmNdialogTitle, title); ac++;
		fsdlgw = XmCreateFileSelectionDialog(g_toplevel, "saveasfile", args, ac);

		XmStringFree(pat);
		XmStringFree(title);

		XtAddCallback(fsdlgw, XmNokCallback,     savefileCB, fsdlgw);
		XtAddCallback(fsdlgw, XmNcancelCallback, cancelfsbCB, fsdlgw);

		hlpb = XmFileSelectionBoxGetChild(fsdlgw, XmDIALOG_HELP_BUTTON);
		XtUnmanageChild(hlpb);

	}

	deffn = XmStringCreateLocalized((g_pdvspec.fn_spec 
	 ? g_pdvspec.fn_spec : ".pdv"));
	XtVaSetValues(fsdlgw, XmNtextString, deffn, NULL);
	XmStringFree(deffn);

	XtManageChild(fsdlgw);

	return;
} /* saveas_menuCB */

/*---------------------------------------- mkpkg_menuCB
	this callback is invoked from the edit menu, warrior types menu item
*/
void
mkpkg_menuCB(Widget w, void *client_data , void *call_data)
{
	makepackage();
} /* mkpkg_menuCB */

/*---------------------------------------- buildscreen
	build the data entry screen - this includes all of the elements used
	to get/display package info, sets up all of the callbacks etc.
	attaches below the menubar widget within form
*/
void
buildscreen(Widget form, Widget menubar)
{
	int      ac;
	int      wn;
	Arg      args[20];
	XmString brz;
	Widget   lblw;
	Widget   ctlw;
	Widget   frame;
	Widget   rcatts;
	/* the following array of structures lets me build most of the
	   widgets in a simple loop - I love Motif */
	struct attrwidgs_st
	{
		char   *lbl;
		char   *wname;
		Widget *w;
		char   *butname;
	} attrwidgs[] =
	 {
	   "Package File Name",   "fnpkg",    &(g_specwidgets.tx_fnpkg),   "pbpkg"
	 , "Payload File Name",   "fnpayld",  &(g_specwidgets.tx_fnpayld), "pbpay"
	 , "Help File Name",      "fnhelp",   &(g_specwidgets.tx_fnhelp),  "pbhlp"
	 , "Agreement File Name", "fnagrmsg", &(g_specwidgets.tx_fnagrmsg),"pbagr" 
	 , "Execute Command",     "execcmd",  &(g_specwidgets.tx_exec),     NULL
	 , "Output File",         "fnoutput", &(g_specwidgets.tx_fnoutput), NULL
	 , NULL, NULL, NULL, False
	};

	rcatts = XtVaCreateWidget("rc_attributes", xmRowColumnWidgetClass, form
	 , XmNpacking,         XmPACK_COLUMN
	 , XmNnumColumns,      6
	 , XmNorientation,     XmHORIZONTAL
	 , XmNisAligned,       True
	 , XmNentryAlignment,  XmALIGNMENT_END
	 , XmNtopAttachment,   XmATTACH_WIDGET
	 , XmNtopWidget,       menubar
	 , XmNtopOffset,       20
	 , XmNleftAttachment,  XmATTACH_FORM
	 , XmNrightAttachment, XmATTACH_FORM
	 , NULL);

	/*-------------------- build the row/column widget and its children */

	wn = 0;
	while(attrwidgs[wn].lbl != NULL)
	{
		XtVaCreateManagedWidget(attrwidgs[wn].lbl, xmLabelGadgetClass
		 , rcatts, NULL);

		*attrwidgs[wn].w = XtVaCreateManagedWidget(attrwidgs[wn].wname
		 , xmTextWidgetClass, rcatts
		 , XmNcolumns, TXENTRYWIDTH
		 , NULL);

		XtAddCallback(*attrwidgs[wn].w, XmNfocusCallback, helptextCB
		 , attrwidgs[wn].wname);
		XtAddCallback(*attrwidgs[wn].w, XmNlosingFocusCallback, helptextCB
		 , "clear");

		wn++;
	}

	XtManageChild(rcatts);

	/*-------------------- compress toggle */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment,  XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNtopWidget,      rcatts); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNset,            False); ac++;
#if XmVersion >= 2001
	XtSetArg(args[ac], XmNtoggleMode,     XmTOGGLE_BOOLEAN); ac++;
#endif
	g_specwidgets.tb_compress = ctlw = XtCreateManagedWidget(
	 "Filter Payload Through compress", xmToggleButtonWidgetClass, form
	 , args, ac);

	XtAddCallback(ctlw, XmNvalueChangedCallback, helptextCB, "compresstb");

	/*-------------------- tar toggle */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment,  XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNtopWidget,      ctlw); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); ac++;
	XtSetArg(args[ac], XmNleftWidget,     ctlw); ac++;
	XtSetArg(args[ac], XmNset,            False); ac++;
#if XmVersion >= 2001
	XtSetArg(args[ac], XmNtoggleMode,     XmTOGGLE_BOOLEAN); ac++;
#endif
	g_specwidgets.tb_tar = ctlw = XtCreateManagedWidget(
	 "Filter Payload Through tar", xmToggleButtonWidgetClass, form, args, ac);

	XtAddCallback(ctlw, XmNvalueChangedCallback, helptextCB, "tartb");

	/*-------------------- the help text area */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment,    XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNtopWidget,        ctlw); ac++;
	XtSetArg(args[ac], XmNleftAttachment,   XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNrightAttachment,  XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNshadowType,       XmSHADOW_OUT); ac++;
	frame = XtCreateManagedWidget("help", xmFrameWidgetClass, form, args, ac);

	ac = 0;
	XtSetArg(args[ac], XmNchildType,  XmFRAME_TITLE_CHILD); ac++;
	lblw = XtCreateManagedWidget("Field Help"
	 , xmLabelGadgetClass, frame, args, ac);

	ac = 0;
	XtSetArg(args[ac], XmNeditable,              False); ac++;
	XtSetArg(args[ac], XmNeditMode,              XmMULTI_LINE_EDIT); ac++;
	XtSetArg(args[ac], XmNwordWrap,              True); ac++;
	XtSetArg(args[ac], XmNcursorPositionVisible, False); ac++;
	XtSetArg(args[ac], XmNtraversalOn,           False); ac++;
	XtSetArg(args[ac], XmNnavigationType,        XmNONE); ac++;
	XtSetArg(args[ac], XmNrows,                  7); ac++;
	ctlw = XtCreateManagedWidget("helptext"
	 , xmTextWidgetClass, frame, args, ac);

	/* this populates a static data member that keeps track of which
	   widget displays help text */

	helptextCB(ctlw, NULL, NULL);

	/*-------------------- the status line */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment,    XmATTACH_WIDGET); ac++;
	XtSetArg(args[ac], XmNtopWidget,        frame); ac++;
	XtSetArg(args[ac], XmNleftAttachment,   XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNrightAttachment,  XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNshadowType,       XmSHADOW_OUT); ac++;
	frame = XtCreateManagedWidget("status", xmFrameWidgetClass, form, args, ac);

	ac = 0;
	XtSetArg(args[ac], XmNchildType,  XmFRAME_TITLE_CHILD); ac++;
	lblw = XtCreateManagedWidget("Status"
	 , xmLabelGadgetClass, frame, args, ac);

	ac = 0;
	XtSetArg(args[ac], XmNeditable,              False); ac++;
	XtSetArg(args[ac], XmNeditMode,              XmMULTI_LINE_EDIT); ac++;
	XtSetArg(args[ac], XmNwordWrap,              True); ac++;
	XtSetArg(args[ac], XmNcursorPositionVisible, False); ac++;
	XtSetArg(args[ac], XmNtraversalOn,           False); ac++;
	XtSetArg(args[ac], XmNnavigationType,        XmNONE); ac++;
	XtSetArg(args[ac], XmNrows,                  2); ac++;
	g_statusw = ctlw = XtCreateManagedWidget("statustext"
	 , xmTextWidgetClass, frame, args, ac);

	return;
} /* buildscreen */

/*---------------------------------------- helptextCB
	this callback is used to present help text to the user based on which
	field he is in.

	this should be called with NULL as client_data and the help widget
	to be updated in w
*/
void
helptextCB(Widget w, void *client_data, void *call_data)
{
	static Widget helptextw ;
	char *id;
	char *msg;

	if(client_data == NULL)
	{
		helptextw = w;
		return;
	}

	id = (char *) client_data;

	if(strcmp(id, "fnpkg") == 0)
	{
		msg = "This is the filename that you will distribute as the executable "
		      "package - it will contain the payload.  This file is created "
				"when you select 'Make Package' from the 'File' menu";
	}
	else if(strcmp(id, "fnpayld") == 0)
	{
		msg = "This is the file that will be included as the payload of the "
		      "package - it is the file that you want delivered.  In most "
				"cases this will be a tar (maybe a compressed tar) that will "
				"then be filtered by pdv during extraction.  If you need to "
				"package more files then it is recommended that you use a tar "
				"for the payload and select the tar filter checkbox below.";
	}
	else if(strcmp(id, "fnhelp") == 0)
	{
		msg = "The contents of this file will be regurgitated if the user "
		      "executes your package with '-h' on the command line.  "
				"It should contain some helpful, meaningful info - like what "
				"exactly it is they are about to execute/install.";
	}
	else if(strcmp(id, "fnagrmsg") == 0)
	{
		msg = "The contents of this file will be displayed to the user"
		      " prior to extracting the payload.  The user will be prompted"
				" to enter 'yes' to accept this agreement, the payload will only"
				" be extracted after the user answers yes.  This is an excellent"
				" place for a copyright message or license agreement.";
	}
	else if(strcmp(id, "fnoutput") == 0)
	{
		msg = "This is the name of the file that will be written when the"
				" end user executes your package.  If you supply a tar and"
				" indicate that it should be filtered through tar then no"
				" output file is written - this value is ignored.";
	}
	else if(strcmp(id, "execcmd") == 0)
	{
		msg = "If this field is valued then the command specified here will"
				" be executed once the package is extracted.  This can be useful"
				" for calling an installation command.  For instance, if you"
				" set the output filename to be foo, you might want to run "
				" 'rpm -i foo' or 'pkgadd -d ./foo'.";
	}
	else if(strcmp(id, "compresstb") == 0)
	{
		msg = "If you are supplying a file for the payload that was produced"
				" by the 'compress' command, then you can have pdv filter the"
				" file through 'uncompress' when it creates it or filters it"
				" through tar.  If you expect to execute your output file then"
				" you MUST make the compressed file (the payload) executable"
				" via the chmod command.";
	}
	else if(strcmp(id, "tartb") == 0)
	{
		msg = "If this is checked, pdv assumes that the payload is a tar"
				" file that needs to be filtered through 'tar -xf -'.  This"
				" works well with compressed files (set the compress filter)"
				" to allow packaging of an arbitrary number of files via pdv."
				"  This is also good for packaging directories etc., you should"
				" refer to the tar manpage for details on its usage.  The most"
				" common form is 'tar -cf foo.tar foo' where foo is the name"
				" of the directory to be packaged, and foo.tar is the file"
				" that will be used as the payload.";
	}
	else
	{
		msg = "";
	}

	XmTextSetString(helptextw, msg);

	return;
} /* helptextCB */

/*---------------------------------------- openfileCB
	this is called via the OK button on the open file selection dialog
	client_data should hold the fileselection box widget so that we
	can unmanage it

	if w == NULL then we got called by a work proc to try to jump start
	the interface, client_data is a filename to read
*/
void
openfileCB(Widget w, void *client_data, void *call_data)
{
	int  res;
	char *fn;
	char *basefn;
	char *badfield;
	XmFileSelectionBoxCallbackStruct *cbs;

	if(w != NULL)
	{
		cbs = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &fn);
	}
	else
		fn = (char *) client_data;

	if(fn != NULL && *fn != '\0')
	{
		freespecmembers(&g_pdvspec);
		res = readspec(fn, &g_pdvspec, &badfield);
		if(res != 0)
		{
			if(res == ENOMSG)
				statusmsg("spec file %s is corrupted or improperly formatted, "
				 "field %s", fn, (badfield ? badfield : "unknown"));
			else
				statusmsg("error reading spec file %s: %s", fn
				 , strerror(res));
		}
		else
		{
			basefn = strrchr(fn, '/');
			if(basefn == NULL || *basefn == '\0')
				basefn = fn;
			else
				basefn++;

			xu_SetMainTitle(basefn, "xpdvmkpkg: ", g_toplevel);

			spectowidgets();

			statusmsg("spec file %s successfully read", basefn);
		}
	}
	else
		statusmsg("no spec file to read");

	/* if we were called as a jumpstart then we should not free the filename
	   or unmanage the dialog (there is none!) */

	if(w != NULL)
	{
		if(fn)
			XtFree(fn);

		XtUnmanageChild(*((Widget *) client_data));
	}

	return;
} /* openfileCB */

/*---------------------------------------- cancelfsbCB
	handle the cancel button on file selection dialogs
	this is meant to be used as a generic cancel operation
	client_data must contain the widget so that we can unmanage it
*/
void
cancelfsbCB(Widget w, void *client_data, void *call_data)
{
	XtUnmanageChild(*((Widget *) client_data));

	statusmsg("operation cancelled");

	return;
} /* cancelfsbCB */

/*---------------------------------------- savefileCB
	this is called via the OK button on the save as file selection dialog
	and from the save option on the file menu
	client_data should hold the fileselection box widget so that we
	can unmanage it, or NULL if no dialog was involved
*/
void
savefileCB(Widget w, void *client_data, void *call_data)
{
	int  res;
	char *fn;
	char *basefn;
	XmFileSelectionBoxCallbackStruct *cbs;

	cbs = (XmFileSelectionBoxCallbackStruct *) call_data;

	if(client_data == NULL)
		fn = strdup(g_pdvspec.fn_spec);
	else
		XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &fn);

	if(fn != NULL && *fn != '\0')
	{
		widgetstospec(&g_pdvspec);
		g_pdvspec.fn_spec = strdup(fn);

		res = writespec(&g_pdvspec);
		if(res != 0)
		{
			statusmsg("error writing spec file %s: %s\n", fn, strerror(res));
		}
		else
		{
			basefn = strrchr(fn, '/');
			if(basefn == NULL || *basefn == '\0')
				basefn = fn;
			else
				basefn++;

			xu_SetMainTitle(basefn, "xpdvmkpkg: ", g_toplevel);
			statusmsg("spec file successfully saved: %s", fn);
		}
	}
	else
		statusmsg("no spec filename supplied");

	if(fn)
	{
		if(client_data == NULL)
			free(fn);
		else
			XtFree(fn);
	}

	if(client_data != NULL)
		XtUnmanageChild(*((Widget *) client_data));

	return;
} /* savefileCB */

/*---------------------------------------- makepackage
	this provides a X based wrapper around the mkpkg utilities
	used by the command line client
*/
void
makepackage(void)
{
	int    allswell = 1;
	int    result;
	char   *fn;
	struct payload_st pld;
	struct stat pldstat;

	/* make sure that we have a legitimate stub */

	fn = findpdv(g_prg, g_pdvspec.fn_pdvstub, 0);
	if(g_pdvspec.fn_pdvstub != NULL)
		free(g_pdvspec.fn_pdvstub);
	g_pdvspec.fn_pdvstub = fn;

	if(g_pdvspec.fn_pdvstub == NULL || access(g_pdvspec.fn_pdvstub, R_OK) != 0)
	{
		statusmsg("unable to find pdv stub, the pdv executable must be in "
		 "$PATH");
		allswell = 0;
	}

	/* make sure we can get to the payload */

	if(allswell)
	{
		if(g_pdvspec.fn_payld == NULL || access(g_pdvspec.fn_payld, R_OK) != 0)
		{
			statusmsg("unable to read payload file %s"
			 , (g_pdvspec.fn_payld ? g_pdvspec.fn_payld : "(nil)" ));
			allswell = 0;
		}
	}

	/* convert tar, compress flags, output filename and exec cmd to payload */

	if(allswell)
	{
		result = spectopayload(&g_pdvspec, &pld);
		if(result != 0)
		{
			statusmsg("error converting spec to payload: %s", strerror(result));
			allswell = 0;
		}
	}

	/* pick up payload file mode */

	if(allswell)
	{
		result = stat(g_pdvspec.fn_payld, &pldstat);
		if(result != 0)
		{
			statusmsg("error converting spec to payload: %s", strerror(result));
			allswell = 0;
		}
		else
		{
			pld.payldmode = pldstat.st_mode;
		}
	}

	/* perform some sanity checks on our spec */

	if(allswell)
	{
		if(g_pdvspec.tar[0] == 'n' 
		 && (g_pdvspec.fn_output == NULL || g_pdvspec.fn_output[0] == '\0'))
		{
			statusmsg("error: an output filename must be specified if payload "
			 "is not a tar");
			allswell = 0;
		}
	}

	/* read the help message file as needed */

	if(allswell)
	{
		if(g_pdvspec.fn_helpmsg != NULL && g_pdvspec.fn_helpmsg[0] != '\0')
		{
			result = readMsgFile(g_pdvspec.fn_helpmsg, &(pld.hlpmsg));
			if(result != 0)
			{
				statusmsg("error reading help message file: %s", strerror(result));
				allswell = 0;
			}
		}
	}

	/* read the agreement message file as needed */

	if(allswell)
	{
		if(g_pdvspec.fn_agrmsg != NULL && g_pdvspec.fn_agrmsg[0] != '\0')
		{
			result = readMsgFile(g_pdvspec.fn_agrmsg, &(pld.agrmsg));
			if(result != 0)
			{
				statusmsg("error reading agreement message file: %s"
				 , strerror(result));
				allswell = 0;
			}
		}
	}

	/* create the new package file */

	if(allswell)
	{
		result = newpkgfile(g_pdvspec.fn_pdvstub, g_pdvspec.fn_pkg
		 , g_pdvspec.fn_payld, &pld, 0);
		if(result != 0)
		{
			statusmsg("error creating package %s, %s", g_pdvspec.fn_pkg
			 , strerror(result));
			allswell = 0;
		}
	}

	if(allswell)
		statusmsg("created PDV package %s", g_pdvspec.fn_pkg);

	return;
} /* makepackage */

/*---------------------------------------- jumpstart
  accepts a filename to jump-start the interface with - we expect that
  this was specified on the command line so we must wait until the
  interface is up and running before we try to read the spec file
*/
Boolean 
jumpstart(XtPointer filename)
{
	openfileCB(NULL, filename, NULL);

	return True;
} /* jumpstart */

/* xmpdvmkpkg.c */
