/***************************************************************************
 *	Copyright (C) 2005 by Karye												*
 *	karye@users.sourceforge.net												*
 *																			*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.										*
 *																			*
 *	This program is distributed in the hope that it will be useful,			*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.							*
 *																			*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the							*
 *	Free Software Foundation, Inc.,											*
 *	59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.				*
 ***************************************************************************/

#include <cassert>            // for assert
#include <klocalizedstring.h>  // for i18n
#include <qaction.h>           // for QAction
#include <qapplication.h>      // for QApplication, qApp
#include <qbytearray.h>        // for QByteArray
#include <qdebug.h>            // for QDebug
#include <qglobal.h>           // for qgetenv, qDebug
#include <qmenu.h>             // for QMenu
#include <qthread.h>           // for QThread
#include <qwidget.h>           // for QWidget
#include <QMainWindow>
#include <utility>             // for move

#include "common.h"            // for EmergeSingleton, LINE_INFO, SignalistS...
#include "emerge.h"            // for Emerge
#include "signalist.h"         // for Signalist
#include "systemtray.h"

SystemTray* SystemTray::s_instance = nullptr;

/**
 * @class SystemTray
 * @short Singleton object that creates the kuroo systemtray icon and actions.
 *
 * Displays kuroo icon in systemtray and switches to "busy" icon when kuroo is busy.
 * The user can disable the systemtray icon in settings.
 */
SystemTray::SystemTray(QMainWindow* parent) : KStatusNotifierItem(parent)
{
	s_instance = this;

	qDebug() << LINE_INFO << "KDE_FULL_SESSION" << qgetenv("KDE_FULL_SESSION") << "XDG_CURRENT_DESKTOP" << qgetenv("XDG_CURRENT_DESKTOP");

	//Rearranged some things trying to get rid of 'QSystemTrayIcon::setVisible: No Icon set' message (didn't work)
	setIconByName( QStringLiteral("kuroo") );
	setCategory( KStatusNotifierItem::ApplicationStatus );
	setStatus( KStatusNotifierItem::Active );
	setToolTipTitle( i18n("Kuroo - Portage frontend") );
	//setting associated widget as recommended in http://techbase.kde.org/Development/Tutorials/PortToKStatusNotifierItem
	//since KF6 this method changed name and takes a QWindow instead of QWidget, given that the Kuroo that initializes it
	//is a KXmlGuiWindow which inherits from QMainWindow this seemed like the plausible way to handle it
	setAssociatedWindow( parent->windowHandle() );

	contextMenu()->addAction( i18n("&Configure Kuroo..."), this, &SystemTray::slotPreferences );
	m_menuPause = contextMenu()->addAction( i18n("Pause Emerge"), this, &SystemTray::slotPause );
	m_menuUnpause = contextMenu()->addAction( i18n("Unpause Emerge"), this, &SystemTray::slotUnpause );

	m_menuPause->setEnabled( false );
	m_menuUnpause->setEnabled( false );

	connect( SignalistSingleton::Instance(), &Signalist::signalKurooBusy, this, &SystemTray::slotBusy );
}

SystemTray::~SystemTray()
= default;

/*void SystemTray::activate()
{
	slotBusy( false );
	setStatus( KStatusNotificationItem::Active );
	//show();
}

void SystemTray::inactivate()
{
	setStatus( KStatusNotifierItem::Passive );
	//hide();
}*/

void SystemTray::slotPreferences()
{
	Q_EMIT signalPreferences();
}

void SystemTray::slotPause()
{
	if( EmergeSingleton::Instance()->canPause() ) {
		m_menuPause->setEnabled( false );
		m_menuUnpause->setEnabled( true );
		EmergeSingleton::Instance()->slotPause();
	}
}

void SystemTray::slotUnpause()
{
	if( EmergeSingleton::Instance()->canPause() ) {
		m_menuPause->setEnabled( true );
		m_menuUnpause->setEnabled( false );
		EmergeSingleton::Instance()->slotUnpause();
	}
}

/**
 * Show busy kuroo icon.
 * @param	kuroo state = busy or ready
 */
void SystemTray::slotBusy( bool busy )
{
	assert(QThread::currentThread() == qApp->thread());
	if ( busy ) {
		setIconByName( QStringLiteral("kuroo_emerging") );
		if( EmergeSingleton::Instance()->isPaused() && EmergeSingleton::Instance()->canPause() ) {
			m_menuUnpause->setEnabled( true );
		} else if( !EmergeSingleton::Instance()->isPaused() && EmergeSingleton::Instance()->canPause() ) {
			m_menuPause->setEnabled( true );
		}
	}
	else {
		setIconByName( QStringLiteral("kuroo") );
		if( EmergeSingleton::Instance()->canPause() && EmergeSingleton::Instance()->isPaused() ) {
			m_menuUnpause->setEnabled( true );
		} else {
			m_menuPause->setEnabled( false );
			m_menuUnpause->setEnabled( false );
		}
	}
}

