/*
  Copyright(C) 2005-2006 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/* ioperms.c

   verify file access permissions
*/

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include <memalloc.h>
#include <myerrno.h>

#include "ioperms.h"

static gid_t *gid_list=NULL;

static int gid_nbr=0;

void free_gid_list()
{
  XFREE(gid_list);
}

int is_file_accessible(const char *file, struct stat *stat_buf)
{
  struct stat *stat_data=NULL;

  myerrno=MENONE;

  if(!stat_buf)
    stat_data=XCALLOC(struct stat, 1);
  else
    stat_data=stat_buf;

  if(!stat(file,stat_data))
    {
      if(!stat_buf)
	XFREE(stat_data);
      stat_data=NULL;
	  
      return 1;
    };

  switch(errno)
    {
    case ENOENT: myerrno=MENOENT; break;
    case ENOTDIR: myerrno=MENOTDIR; break;
    case ELOOP: myerrno=MELOOP; break;
    case EFAULT: myerrno=MEFAULT; break;
    case EACCES: myerrno=MEACCESS; break;
    case ENAMETOOLONG: myerrno=MENAMETOOLONG; break;
    case ENOMEM: myerrno=MENOMEM; break;
    default:
      myerrno=MEUNKNOWN;
    };

  return 0;
}

/* get_process_group_ids

   Return a list of the group IDs assigned to the current process, and, if arg 1 is
   not NULL, set the variable pointed by arg 1 to the number of IDs in the list.

   On success, return a pointer to a NULL terminated list of gid_t.
   On failure, return NULL, and depending on the error, arg 1 is set to -1 or the 
   number of IDs that should have been in the list, and myerrno is set to the 
   appropriate error code. 
*/
gid_t *get_process_group_ids(int *gid_nbr)
{
  long ngroups_max=0;
  int alt_gid_nbr=0;
  gid_t *list=NULL;

  myerrno=MENONE;

  ngroups_max=sysconf(_SC_NGROUPS_MAX);
  list=XCALLOC(gid_t,ngroups_max+1);

  if(!gid_nbr)
    gid_nbr=&alt_gid_nbr;

  if(!list)
    {
      *gid_nbr=getgroups(0,NULL);
      (*gid_nbr)++;
      myerrno=MENOMEM;
      return NULL;
    };

  *gid_nbr=getgroups(ngroups_max,list);

  if(*gid_nbr==-1)
    {
      XFREE(list);
      switch(errno)
	{
	case EFAULT: myerrno=MEFAULT; break;
	case EINVAL: myerrno=MEINVAL; break;
	default:
	  myerrno=MEUNKNOWN; break;
	};
    }
  else
    {
      list[*gid_nbr]=getgid();
      *gid_nbr++;
    };

  return list;
}
  
int is_file_readable(const char *file)
{
  unsigned int i;
  struct stat stat_buf;
  
  myerrno=MENONE;

  if(!file)
    {
      myerrno=MEINVAL;
      return -1;
    };

  if(!is_file_accessible(file,&stat_buf))
    return -1;

  if((stat_buf.st_mode & S_IRUSR) &&
     stat_buf.st_uid==getuid())
    return 1;

  if(stat_buf.st_mode & S_IRGRP)
    {
      if(!gid_list)
	{
	  gid_list=get_process_group_ids(&gid_nbr);
	  if(!gid_list)
	    return -1;
#ifdef HAVE_ATEXIT
	  atexit(free_gid_list);
#endif /* HAVE_ATEXIT */
	};
  
      for(i=0;i<gid_nbr;i++)
	if(stat_buf.st_gid==gid_list[i])
	  return 1;
    };

  if(stat_buf.st_mode & S_IROTH)
    return 1;
  
  return 0;
}

int is_file_writable(const char *file)
{
  unsigned int i;
  struct stat stat_buf;
  
  myerrno=MENONE;

  if(!file)
    {
      myerrno=MEINVAL;
      return -1;
    };

  if(!is_file_accessible(file,&stat_buf))
    return -1;

  if((stat_buf.st_mode & S_IWUSR) &&
     stat_buf.st_uid==getuid())
    return 1;

  if(stat_buf.st_mode & S_IWGRP)
    {
      if(!gid_list)
	{
	  gid_list=get_process_group_ids(&gid_nbr);
	  if(!gid_list)
	    return -1;
	  atexit(free_gid_list);
	};
  
      for(i=0;i<gid_nbr;i++)
	if(stat_buf.st_gid==gid_list[i])
	  return 1;
    };

  if(stat_buf.st_mode & S_IWOTH)
    return 1;
  
  return 0;
}

int is_file_executable(const char *file)
{
  unsigned int i;
  struct stat stat_buf;
  
  myerrno=MENONE;

  if(!file)
    {
      myerrno=MEINVAL;
      return -1;
    };

  if(!is_file_accessible(file,&stat_buf))
    return -1;

  if((stat_buf.st_mode & S_IXUSR) &&
     stat_buf.st_uid==getuid())
    return 1;

  if(stat_buf.st_mode & S_IXGRP)
    {
      if(!gid_list)
	{
	  gid_list=get_process_group_ids(&gid_nbr);
	  if(!gid_list)
	    return -1;
	  atexit(free_gid_list);
	};
  
      for(i=0;i<gid_nbr;i++)
	if(stat_buf.st_gid==gid_list[i])
	  return 1;
    };

  if(stat_buf.st_mode & S_IXOTH)
    return 1;
  
  return 0;
}
