/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  mamory.c

  Command line interface for libmamory.so functions
*/

#include <common/common.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_VFS_H
#include <sys/vfs.h>
#else
#ifdef HAVE_SYS_MOUNT_H /* macOS X have statfs prototype in sys/mount.h */
#include <sys/param.h> /* required for openBSD 3.4 */
#include <sys/mount.h>
#endif /* HAVE_MOUNT_H */
#endif /* HAVE_VFS_H */
#include <sys/types.h>

#include <mamory/mamory.h>
#include <scapi/scapi.h>

s_Options *SetOptions(s_Options *Options)
{
  unsigned char *Home=NULL;
  unsigned char *path=NULL;
  s_Context *Context;
  
  Options=InitOptions("Mamory","Pierre Maziere","2002-2007",
		      LIB_VER_MAJ,LIB_VER_MIN,LIB_VER_MISC);
  SetOptionsUsage(Options,
		  "mamory [options] GamesSrc \n"
		  "\t GamesSrc can be :\n"
		  "\t\t- a XMAME or SDLMAME executable\n"
		  "\t\t- a Roms Directory\n"
		  "\t\t- a file generated by -listinfo xmame option\n"
		  "\t\t- a file generated by -listxml xmame option\n");

  
  Context=InitContext(NULL,NULL,NULL,0,0,0);
  Home=getenv("HOME");
  path=XCALLOC(unsigned char,strlen(Home)+1+strlen(".mamory/mamory.conf")+1);
  strncpy(path,Home,strlen(Home));
  strcat(path,"/");
  strcat(path,".mamory/mamory.conf");
//  SetLocalConfFile(Context,path);
  XFREE(path);


  AddOption(Options,Context,"--help",NO_ARG);
  SetOptionShortName(Options,Context,"--help","-h");
  SetOptionHelp(Context,"-h",NULL,
		"Show this help.");

#ifndef NDEBUG
  AddOption(Options,Context,"--debug",NO_ARG);
  SetOptionShortName(Options,Context,"--debug","-D");
  SetOptionHelp(Context,"-D",NULL,
		"Display debugging messages.");
#endif

  AddOption(Options,Context,"--output",FREE_ARG);
  SetOptionShortName(Options,Context,"--output","-o");
  SetOptionHelp(Context,"-o","file",
		"Redirect stdout to file.");

  AddOption(Options,Context,"--storage-mode",SET_ARG);
  SetOptionShortName(Options,Context,"--storage-mode","-m");
  SetOptionConfFileName(Options,Context,"--storage-mode","storage_mode",NULL);
  AddOptionArg(Context,"-m","merge");
  AddOptionArg(Context,"-m","full");
  AddOptionArg(Context,"-m","split");
  SetOptionDefaultValue(Context,"-m","split");
  SetOptionHelp(Context,"-m",NULL,
		"\
Set the roms storage mode: merge(original and clones in the same zip file), \
full(original and clones in separated zip files with all required roms for \
each), split(original and clones in separated zip files without duplicating \
roms in clones).");

  AddOption(Options,Context,"--list-games",FREE_ARG);
  SetOptionShortName(Options,Context,"--list-games","-l");
  SetOptionHelp(Context,"-l","file",
		"Save the games from GamesSrcs into file.");
/*
  AddOption(Options,Context,"--same-crc-roms",FREE_ARG);
  SetOptionShortName(Options,Context,"--same-crc-roms","-c");
  SetOptionHelp(Context,"-c","file",
		"Save the list of roms with the same CRC into file");

  AddOption(Options,Context,"--same-name-roms",FREE_ARG);
  SetOptionShortName(Options,Context,"--same-name-roms","-n");
  SetOptionHelp(Context,"-n","file",
		"Save the list of roms with the same name into file");
*/

  AddOption(Options,Context,"--additional-sources",FREE_ARG);
  SetOptionShortName(Options,Context,"--additional-sources","-a");
  SetOptionFlags(Context,"-a",SCAPI_MULTIPLE_ARGS);
  SetOptionSeparator(Context,"-a",":");
  SetOptionHelp(Context,"-a","GamesSrc1:GamesSrc2:...",
		"\
Add games sources to GamesSrc. These sources will not be modified.");

  AddOption(Options,Context,"--ignore-nodump",NO_ARG);
  SetOptionShortName(Options,Context,"--ignore-nodump","-i");
  SetOptionHelp(Context,"-i",NULL,
		"Ignore roms whose CRC is NULL.");

  AddOption(Options,Context,"--backup-path",FREE_ARG);
  SetOptionShortName(Options,Context,"--backup-path","-b");
  SetOptionConfFileName(Options,Context,"--backup-path","backup_path",NULL);
  SetOptionHelp(Context,"-b","BackUpPath",
		"\
Backup each modified game in the mamory-roms_backup-YYYYMMDDhhmmss \
directory created in BackUpPath.");

  AddOption(Options,Context,"--diff-with",FREE_ARG);
  SetOptionShortName(Options,Context,"--diff-with","-d");
//  SetOptionFlags(Context,"-d",SCAPI_MULTIPLE_ARGS);
//  SetOptionSeparator(Context,"-d",":");
  SetOptionHelp(Context,"-d","DiffGamesSrc",
		"\
Diff DiffGamesSrc romsets with GamesSrcs romsets. The games and roms \
added, suppressed or changed are printed on stdout.");

  AddOption(Options,Context,"--synchronize",NO_ARG);
  SetOptionShortName(Options,Context,"--synchronize","-s");
  SetOptionHelp(Context,"-s",NULL,
		"\
Synchronize GamesSrcs romsets with DiffGamesSrc romsets.");

  AddOption(Options,Context,"--remove-incomplete-games",NO_ARG);
  SetOptionShortName(Options,Context,"--remove-incomplete-games","-r");
  SetOptionHelp(Context,"-r",NULL,
		"\
If the -s option is set, delete each games whose rom set is incomplete. \
In the merge storage mode, it means that all roms from the clones and the \
original game have to be available, otherwhise the whole set will be deleted");

  AddOption(Options,Context,"--test",NO_ARG);
  SetOptionShortName(Options,Context,"--test","-t");
  SetOptionHelp(Context,"-t",NULL,
		"\
If the -s option is set, simulate the synchronization without modifying \
any romsets.");

  AddOption(Options,Context,"--export-diffset-using",FREE_ARG);
  SetOptionShortName(Options,Context,"--export-diffset-using","-e");
  SetOptionFlags(Context,"-e",SCAPI_MULTIPLE_ARGS);
  SetOptionSeparator(Context,"-e",":");
  SetOptionHelp(Context,"-e","RomsDir1:RomsDir2:...",
		"\
Export the roms missing between GamesSrcs and DiffGamesSrc in \
./mamory_diffset, using romsets from RomsDirs.");

  AddContext(Options,Context);

  return Options;
};

long int GetFreeDiskSpace(unsigned char *path)
{
  struct statfs buf_statfs;
  
  if(statfs(path,&buf_statfs)!=-1)
    return(buf_statfs.f_bsize*buf_statfs.f_bavail);
  else
    return -1;
}

int main (int argc,char **argv)
{

  unsigned char **optionvalue=NULL;
  unsigned int StorageMode=STMODE_SPLIT;
  unsigned int i,j;
  FILE *fp=NULL,*Output=NULL;  
  struct stat stat_buf; 
  unsigned char Buffer[BUFSIZ]={0};

  s_Options *Options=NULL;

  s_GamesList **GamesListSources=NULL;
  unsigned int nbrsources=0;
  s_GamesList *MergedSourcesGamesList=NULL;

  s_GamesList **GamesListDests=NULL;
  unsigned int nbrdests=0;
  s_GamesList *MergedDestsGamesList=NULL;

  s_GamesList **GamesListExportSources=NULL;
  unsigned int nbrexportsources=0;
  s_GamesList *MergedExportSourcesGamesList=NULL;

  s_GamesList *GamesList=NULL;

  s_Diff *Diff=NULL;
  unsigned char TempDir[]="/tmp";
  unsigned char *BackUpPath=NULL;
  unsigned int OptionFlags=OPTION_NONE;

// If you may need to uncomment and modify this for your own applications

//  Libmamory_out=stdout;
//  Libmamory_err=stderr;
//  Libmamory_OUTPUT_On=TRUE;
//  Libmamory_WARNING_On=TRUE;
//  Libmamory_ERROR_On=TRUE;
  Libmamory_HEADER_OUTPUT_On=FALSE;
//  Libmamory_HEADER_WARNING_On=TRUE;
//  Libmamory_HEADER_ERROR_On=TRUE;
  Libmamory_HEADER_OUTPUT_LIBNAME_On=FALSE;
//  Libmamory_HEADER_WARNING_LIBNAME_On=TRUE;
//  Libmamory_HEADER_ERROR_LIBNAME_On=TRUE;

  Libscapi_OUTPUT_On=FALSE;
    
  Options=SetOptions(Options);
  if((ParseOptions(Options,argc,argv)==NULL || argc==1) ||
     Options->NbrUnknownTokens!=1 ||
     GetOption(Options,"-h",&optionvalue))
    {
      DisplayOptionsHelp(Options,32,46);
      FreeOptions(Options);
      exit(0);
    };


#ifndef NDEBUG
  if(!GetOption(Options,"-D",&optionvalue))
      Libmamory_DEBUG_On=FALSE;
#endif  

  fprintf(stderr,"Mamory v%i.%i.%i (c) 2002-2007 Pierre Mazire\n\n",
	  LIB_VER_MAJ,LIB_VER_MIN,LIB_VER_MISC);

  if(GetOption(Options,"-i",&optionvalue))
    OptionFlags|=OPTION_IGNORE_MISSING_NO_DUMP_ROMS;

  if(GetOption(Options,"-r",&optionvalue))
    OptionFlags|=OPTION_DELETE_INCOMPLETE_GAMES;

  if(GetOption(Options,"-m",&optionvalue))
    {
      if(strcmp(optionvalue[0],"merge")==0)
	StorageMode=STMODE_MERGE;
      if(strcmp(optionvalue[0],"split")==0)
	StorageMode=STMODE_SPLIT;
      if(strcmp(optionvalue[0],"full")==0)
	StorageMode=STMODE_FULL ;
    }
  else
    StorageMode=STMODE_SPLIT;

  for(i=0;i<1/*Options->NbrUnknownTokens*/;i++)
    {
      if(stat(Options->UnknownTokens[i],&stat_buf))
	{
	  fprintf(stderr,"\n*** Mamory : %s '%s'\n",
		  strerror(errno),
		  Options->UnknownTokens[i]);
	  continue;
	}
      GamesList=GetGamesList(Options->UnknownTokens[i],TempDir);
      ConvertGamesListToSTMODE(GamesList,StorageMode);
      if(GamesList!=NULL)
	{
	  fprintf(stderr,"\tNbr of RomContents: %i\n",
		  GamesList->NbrRomContents);
	  //fprintf(stderr,"\tNbr of Samples: %i\n",
	  //	    GamesList->NbrSoundSamples);
	  if(GamesList->NbrUnknownTokens!=0)
	    {
	      fprintf(stderr,"\n*** Mamory : Nbr of Unknown Tokens: %i\n",
		      GamesList->NbrUnknownTokens);
	      fprintf(stderr,"\n*** Mamory : '%s' is not a valid source\n",
		      Options->UnknownTokens[i]);
	      FreeGamesList(GamesList);
	      GamesList=NULL;
	      continue;
	    };
	  GamesListSources=XREALLOC(GamesListSources,s_GamesList*,
				    nbrsources+1);
	  GamesListSources[nbrsources]=GamesList;
	  MergedSourcesGamesList=MergeGamesLists(MergedSourcesGamesList,
						 GamesListSources[nbrsources]);
	  nbrsources++;
	}
      else
	fprintf(stderr,"\n*** Mamory : '%s' is not a valid source\n",
		Options->UnknownTokens[i]);
    };

  if(nbrsources!=0)
    { 
      if((j=GetOption(Options,"--additional-sources",&optionvalue)))
	{	  
	  for(i=0;i<j;i++)
	    {
	      if(stat(optionvalue[i],&stat_buf))
		{
		  fprintf(stderr,"\n*** Mamory : %s '%s'\n",
			  strerror(errno),
			  optionvalue[i]);
		  continue;
		}
	      GamesList=GetGamesList(optionvalue[i],TempDir);
	      ConvertGamesListToSTMODE(GamesList,StorageMode);
	      if(GamesList!=NULL)
		{
		  fprintf(stderr,"\tNbr of RomContents: %i\n",
			  GamesList->NbrRomContents);
		  //fprintf(stderr,"\tNbr of Samples: %i\n",
		  //	    GamesList->NbrSoundSamples);
		  if(GamesList->NbrUnknownTokens!=0)
		    {
		      fprintf(stderr,
			      "\n*** Mamory : Nbr of Unknown Tokens: %i\n",
			      GamesList->NbrUnknownTokens);
		      fprintf(stderr,
			      "\n*** Mamory : '%s' is not a valid source\n",
			      optionvalue[i]);
		      FreeGamesList(GamesList);
		      GamesList=NULL;
		      continue;
		    };
		  GamesListSources=XREALLOC(GamesListSources,s_GamesList*,
					    nbrsources+1);
		  GamesListSources[nbrsources]=GamesList;
		  MergedSourcesGamesList=
		    MergeGamesLists(MergedSourcesGamesList,
				    GamesListSources[nbrsources]);
		  nbrsources++;
		}
	      else
		fprintf(stderr,"\n*** Mamory : '%s' not valid\n",optionvalue[i]);
	    };
	};
      
      if(GetOption(Options,"--output",&optionvalue))
	Output=fopen(optionvalue[0],"w");
      else
	Output=stdout;
      
      if(GetOption(Options,"--list-games",&optionvalue))
	{
	  fp=fopen(optionvalue[0],"w");	  
	  fprintGamesListMamoryHeader(fp,MergedSourcesGamesList);
	  
	  for(i=0;i<MergedSourcesGamesList->NbrGames;i++)
	    fprintGameInfo(fp,MergedSourcesGamesList,MergedSourcesGamesList->Games[i]);
	  fclose(fp);
	};
      /*      
	      if(GetOption(Options,"--same-crc-roms",&optionvalue))
	      {
	      fp=fopen(optionvalue[0],"w");
	      GetSameCRCRoms(fp,SourcesMergedGamesList);
	      fclose(fp);
	      };
	      
	      if(GetOption(Options,"--same-name-roms",&optionvalue))
	      {
	      fp=fopen(optionvalue[0],"w");
	      GetSameNameRoms(fp,SourcesMergedGamesList);
	      fclose(fp);
	      };	  
      */    
      if(GetOption(Options,"--backup-path",&optionvalue))
	{
	  BackUpPath=XSTRDUP(optionvalue[0]);
	  if (BackUpPath!=NULL)
	    {
	      if(!stat(BackUpPath,&stat_buf))
		{
		  if(!S_ISDIR(stat_buf.st_mode))
		    {
		      fprintf(stderr,
			      "\n*** Mamory : '%s' is not a directory\n",
			      BackUpPath);
		      XFREE(BackUpPath);
		    }
		  else
		    {
		      if(stat_buf.st_uid==getuid())
			{
			  if((stat_buf.st_mode&S_IWUSR)!=S_IWUSR)
			    {
			      fprintf(stderr,
				      "\n*** Mamory : '%s' is not a writable directory\n",
				      BackUpPath);
			      XFREE(BackUpPath);
			    };
			}
		      else
			{
			  if(stat_buf.st_gid==getgid())
			    {
			      if((stat_buf.st_mode&S_IWGRP)!=S_IWGRP)
				{
				  fprintf(stderr,
					  "\n*** Mamory : '%s' is not a writable directory\n",
					  BackUpPath);
				  XFREE(BackUpPath);
				}
			    }
			  else
			    {
			      if((stat_buf.st_mode&S_IWOTH)!=S_IWOTH)
				{
				  fprintf(stderr,
					  "\n*** Mamory : '%s' is not a writable directory\n",
					  BackUpPath);
				  XFREE(BackUpPath);
				};
			    };
			};
		    };
		}
	      else
		{
		  fprintf(stderr,
			  "\n*** Mamory : '%s' is not a valid directory\n",
			  BackUpPath);
		  XFREE(BackUpPath);
		};
	    };
	};
      
      if((j=(GetOption(Options,"--diff-with",&optionvalue))))
	{
	  for(i=0;i<j;i++)
	    {
	      if(stat(optionvalue[i],&stat_buf))
		{
		  fprintf(stderr,"\n*** Mamory : %s '%s'\n",
			  strerror(errno),
			  optionvalue[i]);
		  continue;
		}
	      GamesList=GetGamesList(optionvalue[i],TempDir);
	      ConvertGamesListToSTMODE(GamesList,StorageMode);
	      if(GamesList!=NULL)
		{
		  fprintf(stderr,"\tNbr of RomContents: %i\n",
			  GamesList->NbrRomContents);
		  //fprintf(stderr,"\tNbr of Samples: %i\n",
		  //	    GamesList->NbrSoundSamples);
		  if(GamesList->NbrUnknownTokens!=0)
		    {
		      fprintf(stderr,
			      "\n*** Mamory : Nbr of Unknown Tokens: %i\n",
			      GamesList->NbrUnknownTokens);
		      fprintf(stderr,
			      "\n*** Mamory : '%s' is not a valid source\n",
			      optionvalue[i]);
		      FreeGamesList(GamesList);
		      GamesList=NULL;
		      continue;
		    };
		  GamesListDests=XREALLOC(GamesListDests,s_GamesList*,
					  nbrdests+1);
		  GamesListDests[nbrdests]=GamesList;
		  MergedDestsGamesList=
		    MergeGamesLists(MergedDestsGamesList,
				    GamesListDests[nbrdests]);
		  nbrdests++;
		}
	      else
		fprintf(stderr,"\n*** Mamory : '%s' not valid\n",optionvalue[i]);
	    };

	  if(nbrdests!=0)
	    {
	      fprintf(stderr,"\n");
	      printf("--- Starting Roms Analysis\n");
	      Diff=DiffGamesLists(MergedSourcesGamesList,MergedDestsGamesList,
				  OptionFlags);
	      printf("--- End of Roms Analysis\n");
	      fprintDiff(Output,Diff);
	    }
	  else
	    Diff=NULL;

	  if(Diff!=NULL &&
	     (j=GetOption(Options,"--export-diffset-using",&optionvalue)))
	    {

	      mkdir("./mamory_diffset",S_IRWXU|S_IRWXG);
	      GamesListExportSources=XREALLOC(GamesListExportSources,
					      s_GamesList*,
					      nbrexportsources+1);

	      GamesListExportSources[nbrexportsources]=
		GetGamesList("./mamory_diffset",TempDir);
	      ConvertGamesListToSTMODE(GamesListExportSources[nbrexportsources],
				       StorageMode);
	      MergedExportSourcesGamesList=
		MergeGamesLists(MergedExportSourcesGamesList,
				GamesListExportSources[nbrexportsources]);
	      nbrexportsources++;

	      for(i=0;i<j;i++)
		{
		  if(stat(optionvalue[i],&stat_buf))
		    {
		      fprintf(stderr,"\n*** Mamory : %s '%s'\n",
			      strerror(errno),
			      optionvalue[i]);
		      continue;
		    }
		  GamesList=GetGamesList(optionvalue[i],TempDir);
		  ConvertGamesListToSTMODE(GamesList,StorageMode);
		  if(GamesList!=NULL)
		    {
		      fprintf(stderr,"\tNbr of RomContents: %i\n",
			      GamesList->NbrRomContents);
		      //fprintf(stderr,"\tNbr of Samples: %i\n",
		      //	    GamesList->NbrSoundSamples);
		      if(GamesList->NbrUnknownTokens!=0)
			{
			  fprintf(stderr,
				  "\n*** Mamory : Nbr of Unknown Tokens: %i\n",
				  GamesList->NbrUnknownTokens);
			  fprintf(stderr,
				  "\n*** Mamory : '%s' is not a valid source\n",
				  optionvalue[i]);
			  FreeGamesList(GamesList);
			  GamesList=NULL;
			  continue;
			};
		      GamesListExportSources=XREALLOC(GamesListExportSources,
						      s_GamesList*,
						      nbrexportsources+1);
		      GamesListExportSources[nbrexportsources]=GamesList;
		      MergedExportSourcesGamesList=
			MergeGamesLists(MergedExportSourcesGamesList,
					GamesListExportSources[nbrexportsources]);
		      nbrexportsources++;
		    }
		  else
		    fprintf(stderr,"\n*** Mamory : '%s' not valid\n",optionvalue[i]);
		};

	      if(nbrexportsources>1)
		{
		  GamesList=MISSINGtoGamesList(Diff);
		  FreeDiff(Diff);
		  Diff=DiffGamesLists(MergedExportSourcesGamesList,GamesList,
				      OptionFlags);
		  if(GetOption(Options,"--test",&optionvalue))
		    synchronize(Diff,NULL,1);
		  else
		    synchronize(Diff,NULL,0);	
		  FreeGamesList(GamesList);
		}
	      else
		{
		  fprintf(stderr,"\n*** Mamory : no valid sources for exporting.");
		  fprintf(stderr,"\n*** Mamory : diffset export failed.");
		};
	    };

	  if(GetOption(Options,"--synchronize",&optionvalue))
	    {
	     // StripDiff(Diff);
	      if(GetOption(Options,"--test",&optionvalue))
		j=GetSyncDiskSpaceNeed(Diff,BackUpPath,1);
	      else
		j=GetSyncDiskSpaceNeed(Diff,BackUpPath,0);

	      if(j==TRUE)
		{
		  printf("\n**********************************************************************\n");
		  if(BackUpPath==NULL)
		    printf("WARNING : BACKUP NOT SELECTED OR NOT POSSIBLE\n");
		  printf("Storage Mode: ");
		  switch(StorageMode)
		    {
		    case STMODE_MERGE: 
		      printf("MERGE (Original and clones roms in one slot)\n"); 
		      break;
		    case STMODE_SPLIT:
		      printf("SPLIT (Original and clones exclusive roms in their respective slots)\n");
		      break;
		    case STMODE_FULL:
		      printf("FULL (Original and Clones required roms in their respective slots)\n");
		      break;
		    };
		  if(GetOption(Options,"--test",&optionvalue))
		    printf("For this test, you will need :\n");
		  else
		    printf("You will need :\n");
		  printf("\t %lu bytes in '%s' (free: %llu bytes)\n",
			 Diff->MainNeededSize,Diff->GamesList1->Sources[0]->Source,
			 GetFreeDiskSpace(Diff->GamesList1->Sources[0]->Source));
		  if(BackUpPath==NULL)
		    printf("\t %lu bytes in '%s' (free: %lu bytes)\n",
			   Diff->TempNeededSize,"/tmp",
			   GetFreeDiskSpace("/tmp"));
		  else
		    printf("\t %lu bytes in '%s' (free: %lu bytes)\n",
			   Diff->BackUpNeededSize,BackUpPath,
			   GetFreeDiskSpace(BackUpPath));
		  printf("\n**********************************************************************\n");
		  printf("\nDo you want to continue (Y/N) ?");
		  fflush(stdout);
		  fflush(stdin);
		  fgets(Buffer,3,stdin);
		  if(strncmp(Buffer,"Y",1)!=0 && strncmp(Buffer,"y",1)!=0)
		    abort();
		  if(GetOption(Options,"--test",&optionvalue))
		    synchronize(Diff,BackUpPath,1);
		  else
		    synchronize(Diff,BackUpPath,0);
		  
		  fprintf(Output,"\nAfter Synchronization, there is:\n");
		  fprintDiff(Output,Diff);
		};
	    };
	};

      FreeDiff(Diff);
      Diff=NULL;
      FreeGamesList(MergedExportSourcesGamesList);
      MergedExportSourcesGamesList=NULL;
      for(i=0;i<nbrexportsources;i++)
	{
	  FreeGamesList(GamesListExportSources[i]);
	  GamesListExportSources[i]=NULL;
	};
      XFREE(GamesListExportSources);
      GamesListExportSources=NULL; 
      nbrexportsources=0;

      FreeGamesList(MergedDestsGamesList);
      MergedDestsGamesList=NULL;
      for (i=0;i<nbrdests;i++)
	{
	  FreeGamesList(GamesListDests[i]);
	  GamesListDests[i]=NULL;
	};
      XFREE(GamesListDests);
      GamesListDests=NULL;
      nbrdests=0;

      FreeGamesList(MergedSourcesGamesList);
      MergedSourcesGamesList=NULL;
      for (i=0;i<nbrsources;i++)
	{
	  FreeGamesList(GamesListSources[i]);
	  GamesListSources[i]=NULL;
	};
      XFREE(GamesListSources);
      GamesListSources=NULL;
      nbrsources=0;
    }
  else
    fprintf(stderr,"\n*** Mamory : no valid sources\n");
  
  if(Output!=NULL && Output!=stdout)
    fclose(Output);

  XFREE(BackUpPath);
  FreeOptions(Options);
  return EXIT_SUCCESS;
};
