/********************************************************************************************************
 * QRNA - Comparative analysis of biological sequences 
 *         with pair hidden Markov models, pair stochastic context-free
 *        grammars, and probabilistic evolutionary  models.
 *       
 * Version 2.0.0 (JUN 2003)
 *
 * Copyright (C) 2000-2003 Howard Hughes Medical Institute/Washington University School of Medicine
 * All Rights Reserved
 * 
 *     This source code is distributed under the terms of the
 *     GNU General Public License. See the files COPYING and LICENSE
 *     for details.
 ***********************************************************************************************************/

/*
 * rnacomp.h
 * 
 * Header file for API for RNA matrix routines.  Used in parsing alignment
 * into matrix and later reading in matrix.
 *
 * Robert J. Klein
 * February 25, 2002
 */

#ifndef _RNAMAT_H
#define _RNAMAT_H

#include "squid.h"
#include "msa.h"
#include "structs.h"

#define RNA_ALPHABET_SIZE Alphabet_size
#define RNAPAIR_ALPHABET "AAAACCCCGGGGUUUU"
#define RNAPAIR_ALPHABET2 "ACGUACGUACGUACGU"

/* Returns true if pos. C of seq B of msa A is a gap as defined by isgap(c) 
   from squid */
#define is_rna_gap(A, B, C) (isgap(A->aseq[B][C]))

/* Returns true if pos. C of seq B of msa A is an uppercase A, C, G, T, or U */
#define is_defined_rna_nucleotide(A, B, C) (A->aseq[B][C] == 'A' || A->aseq[B][C] == 'C' || A->aseq[B][C] == 'G' || A->aseq[B][C] == 'T' || A->aseq[B][C] == 'U')
#define is_rna_nucleotide(X) (X == 'A' || X == 'C' || X == 'G' || X == 'T' || X == 'U')

/*
 * Maps to index of matrix, using binary representation of
 * nucleotides (unsorted).
 *
 * See lab book 7, p. 3-4 for details of mapping function
 */
#define matrix_index(X,Y) ((X>Y) ? X*(X+1)/2+Y: Y*(Y+1)/2+X)



extern fullmat_t     *AllocFullMatrix (int L);
extern fullcondmat_t *AllocFullCondMatrix (int L);
extern matrix_t      *AllocMatrix (int L);
extern condmatrix_t  *AllocCondMatrix (int L);
extern void           CalculateConditionals (FILE *fp, fullmat_t *fullmat);
extern void           CalculateJoint (FILE *fp, fullmat_t *fullmat, double *background_nt, int log2oods);
extern void           CheckSingleProb(double *psingle, int size);
extern void           CheckSingleLog2Prob(double *psingle, int size);
extern void           CountMatrixER (MSA *msa, fullmat_t *fullmat, double *background_nt, int cutoff_perc);
extern fullmat_t     *CopyFullMatrix (fullmat_t *fullmat);
extern void           CopyMatrix (matrix_t *matcopy, matrix_t *mat);
extern void           DLog2(double *vec, int n);
extern void           FreeFullMatrix (fullmat_t *fullmat);
extern void           FreeFullCondMatrix (fullcondmat_t *fullmat);
extern void           FreeMatrix (matrix_t *mat);
extern void           FreeCondMatrix (condmatrix_t *mat);
extern void           Log2Matrix (matrix_t *mat);
extern void           Log2toOddsMatrix (matrix_t *mat, double *background_nt);
extern FILE          *MatFileOpen (char *deflt, char *matdir, char *matfile);
extern int            numbered_basepair (char c, char d);
extern int            numbered_nucleotide (char c);
extern void           PrintCondMatrix (FILE *fp, condmatrix_t *condmat, int ispaircond, int islog2, char *title); 
extern void           PrintFullMatrix (FILE *fp, fullmat_t *fullmat);
extern void           PrintFullCondMatrix (FILE *fp, fullcondmat_t *fullcondmat, int ispaircond, int islog2);
extern int           *rjk_KHS2ct(char *ss, int len);
extern fullmat_t     *ReadMatrix(FILE *matfp);

    
#endif
  
