//! A compiled-in copy of the full X.509 root certificates trusted by Mozilla.
//!
//! You should generally prefer to use [`webpki-roots`] when using [`rustls`] or [`webpki`] as it is
//! more space efficient and convenient for that use.
//!
//! This library is suitable for use in applications that can always be recompiled and instantly deployed.
//! For applications that are deployed to end-users and cannot be recompiled, or which need certification
//! before deployment, consider a library that uses the platform native certificate verifier such as
//! [`rustls-platform-verifier`]. This has the additional benefit of supporting OS provided CA constraints
//! and revocation data.
//!
//! [`webpki-roots`]: https://docs.rs/webpki-roots
//! [`webpki`]: https://docs.rs/rustls-webpki
//! [`rustls`]: https://docs.rs/rustls
//! [`rustls-platform-verifier`]: https://docs.rs/rustls-platform-verifier
//
// This library is automatically generated from the Mozilla
// IncludedCACertificateReportPEMCSV report via ccadb.org. Don't edit it.
//
// The generation is done deterministically so you can verify it
// yourself by inspecting and re-running the generation process.

#![no_std]
#![forbid(unsafe_code, unstable_features)]
#![deny(
    elided_lifetimes_in_paths,
    trivial_casts,
    trivial_numeric_casts,
    unused_import_braces,
    unused_extern_crates,
    unused_qualifications
)]

use pki_types::CertificateDer;

pub const TLS_SERVER_ROOT_CERTS: &[CertificateDer<'static>] = &[
   // "DigiCert TLS ECC P384 Root G5"
   CertificateDer::from_slice(b"\x30\x82\x02\x19\x30\x82\x01\x9F\xA0\x03\x02\x01\x02\x02\x10\x09\xE0\x93\x65\xAC\xF7\xD9\xC8\xB9\x3E\x1C\x0B\x04\x2A\x2E\xF3\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x17\x30\x15\x06\x03\x55\x04\x0A\x13\x0E\x44\x69\x67\x69\x43\x65\x72\x74\x2C\x20\x49\x6E\x63\x2E\x31\x26\x30\x24\x06\x03\x55\x04\x03\x13\x1D\x44\x69\x67\x69\x43\x65\x72\x74\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x50\x33\x38\x34\x20\x52\x6F\x6F\x74\x20\x47\x35\x30\x1E\x17\x0D\x32\x31\x30\x31\x31\x35\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x31\x31\x34\x32\x33\x35\x39\x35\x39\x5A\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x17\x30\x15\x06\x03\x55\x04\x0A\x13\x0E\x44\x69\x67\x69\x43\x65\x72\x74\x2C\x20\x49\x6E\x63\x2E\x31\x26\x30\x24\x06\x03\x55\x04\x03\x13\x1D\x44\x69\x67\x69\x43\x65\x72\x74\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x50\x33\x38\x34\x20\x52\x6F\x6F\x74\x20\x47\x35\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xC1\x44\xA1\xCF\x11\x97\x50\x9A\xDE\x23\x82\x35\x07\xCD\xD0\xCB\x18\x9D\xD2\xF1\x7F\x77\x35\x4F\x3B\xDD\x94\x72\x52\xED\xC2\x3B\xF8\xEC\xFA\x7B\x6B\x58\x20\xEC\x99\xAE\xC9\xFC\x68\xB3\x75\xB9\xDB\x09\xEC\xC8\x13\xF5\x4E\xC6\x0A\x1D\x66\x30\x4C\xBB\x1F\x47\x0A\x3C\x61\x10\x42\x29\x7C\xA5\x08\x0E\xE0\x22\xE9\xD3\x35\x68\xCE\x9B\x63\x9F\x84\xB5\x99\x4D\x58\xA0\x8E\xF5\x54\xE7\x95\xC9\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC1\x51\x45\x50\x59\xAB\x3E\xE7\x2C\x5A\xFA\x20\x22\x12\x07\x80\x88\x7C\x11\x6A\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\x89\x6A\x8D\x47\xE7\xEC\xFC\x6E\x55\x03\xD9\x67\x6C\x26\x4E\x83\xC6\xFD\xC9\xFB\x2B\x13\xBC\xB7\x7A\x8C\xB4\x65\xD2\x69\x69\x63\x13\x63\x3B\x26\x50\x2E\x01\xA1\x79\x06\x91\x9D\x48\xBF\xC2\xBE\x02\x30\x47\xC3\x15\x7B\xB1\xA0\x91\x99\x49\x93\xA8\x3C\x7C\xE8\x46\x06\x8B\x2C\xF2\x31\x00\x94\x9D\x62\xC8\x89\xBD\x19\x84\x14\xE9\xA5\xFB\x01\xB8\x0D\x76\x43\x8C\x2E\x53\xCB\x7C\xDF\x0C\x17\x96\x50"),
   // "D-TRUST BR Root CA 2 2023"
   CertificateDer::from_slice(b"\x30\x82\x05\xA9\x30\x82\x03\x91\xA0\x03\x02\x01\x02\x02\x10\x73\x3B\x30\x04\x48\x5B\xD9\x4D\x78\x2E\x73\x4B\xC9\xA1\xDC\x66\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x42\x52\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x20\x32\x30\x32\x33\x30\x1E\x17\x0D\x32\x33\x30\x35\x30\x39\x30\x38\x35\x36\x33\x31\x5A\x17\x0D\x33\x38\x30\x35\x30\x39\x30\x38\x35\x36\x33\x30\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x42\x52\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x20\x32\x30\x32\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xAE\xFF\x09\x59\x91\x80\x0A\x4A\x68\xE6\x24\x3F\xB8\xA7\xE4\xC8\x3A\x0A\x3A\x16\xCD\xC9\x23\x61\xA0\x93\x71\xF2\xAB\x8B\x73\x8F\xA0\x67\x65\x60\xD2\x54\x6B\x63\x51\x6F\x49\x33\xE0\x72\x07\x13\x7D\x38\xCD\x06\x92\x07\x29\x52\x6B\x4E\x77\x6C\x04\xD3\x95\xFA\xDD\x4C\x8C\xD9\x5D\xC1\x61\x7D\x4B\xE7\x28\xB3\x44\x81\x7B\x51\xAF\xDD\x33\xB1\x68\x7C\xD6\x4E\x4C\xFE\x2B\x68\xB9\xCA\x66\x69\xC4\xEC\x5E\x57\x7F\xF7\x0D\xC7\x9C\x36\x36\xE5\x07\x60\xAC\xC0\x4C\xEA\x08\x6C\xEF\x06\x7C\x4F\x5B\x28\x7A\x08\xFC\x93\x5D\x9B\xF6\x9C\xB4\x8B\x86\xBA\x21\xB9\xF4\xF0\xE8\x59\x5A\x28\xA1\x34\x84\x1A\x25\x91\xB6\xB5\x8F\xEF\xB2\xF9\x80\xFA\xF9\x3D\x3C\x11\x72\xD8\xE3\x2F\x86\x76\xC5\x79\x2C\xC1\xA9\x90\x93\x46\x98\x67\xCB\x83\x6A\xA0\x50\x23\xA7\x3B\xF6\x81\x39\xE0\xED\xF0\xB9\xBF\x65\xF1\xD8\xCB\x7A\xFB\xEF\x73\x03\xCE\x00\xF4\x7D\xD7\xE0\x5D\x3B\x66\xB8\xDC\x8E\xBA\x83\xCB\x87\x76\x03\xFC\x25\xD9\xE7\x23\x6F\x06\xFD\x67\xF3\xE0\xFF\x84\xBC\x47\xBF\xB5\x16\x18\x46\x69\x14\xCC\x05\xF7\xDB\xD3\x49\xAC\x6B\xCC\xAB\xE4\xB5\x0B\x43\x24\x5E\x4B\x6B\x4D\x67\xDF\xD6\xB5\x3E\x4F\x78\x1F\x94\x71\x24\xEA\xDE\x70\xFC\xF1\x93\xFE\x9E\x93\x5A\xE4\x94\x5A\x97\x54\x0C\x35\x7B\x5F\x6C\xEE\x00\x1F\x24\xEC\x03\xBA\x02\xF5\x76\xF4\x9F\xD4\x9A\xED\x85\x2C\x38\x22\x2F\xC7\xD8\x2F\x76\x11\x4F\xFD\x6C\x5C\xE8\xF5\x8E\x27\x87\x7F\x19\x4A\x21\x47\x90\x1D\x79\x8D\x1C\x5B\xF8\xCF\x4A\x85\xE4\xED\xB3\x5B\x8D\xBE\xC4\x64\x28\x5D\x41\xC4\x6E\xAC\x38\x5A\x4F\x23\x74\x74\xA9\x12\xC3\xF6\xD2\xB9\x11\x15\x33\x07\x91\xD8\x3B\x37\x3A\x63\x30\x06\xD1\xC5\x22\x36\x28\x62\x23\x10\xE0\x46\xCC\x97\xAC\xD6\x2B\x5D\x64\x24\xD5\xEE\x1C\x0E\xDE\xFB\x08\x5A\x75\x2A\xF6\x63\x6D\xCE\x0B\x42\xBE\xD1\xBA\x70\x1C\x9C\x21\xE5\x0F\x31\x69\x17\xD7\xFC\x0A\xB4\xDE\xED\x80\x9C\xCB\x92\xB4\x8B\xF5\xDE\x59\xA2\x58\x09\xA5\x63\x47\x0B\xE1\x41\x32\x34\x41\xD9\x9A\xB1\xD9\xA8\xB0\x1B\x5A\xDE\x0D\x0D\xF4\xE2\xB2\x5D\x35\x80\xB9\x81\xD4\x84\x69\x91\x02\xCB\x75\xD0\x8D\xC5\xB5\x3D\x09\x91\x09\x8F\x14\xA1\x14\x74\x79\x3E\xD6\xC9\x15\x1D\xA4\x59\x59\x22\xDC\xF6\x8A\x45\x3D\x3C\x12\xD6\x3E\x5D\x32\x2F\x02\x03\x01\x00\x01\xA3\x81\x8E\x30\x81\x8B\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x67\x90\xF0\xD6\xDE\xB5\x18\xD5\x46\x29\x7E\x5C\xAB\xF8\x9E\x08\xBC\x64\x95\x10\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x49\x06\x03\x55\x1D\x1F\x04\x42\x30\x40\x30\x3E\xA0\x3C\xA0\x3A\x86\x38\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x63\x72\x6C\x2F\x64\x2D\x74\x72\x75\x73\x74\x5F\x62\x72\x5F\x72\x6F\x6F\x74\x5F\x63\x61\x5F\x32\x5F\x32\x30\x32\x33\x2E\x63\x72\x6C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x03\x82\x02\x01\x00\x34\xF7\xB3\x77\x53\xDB\x30\x16\xB9\x2D\xA5\x21\xF1\x40\x21\x75\xEB\xEB\x48\x16\x81\x3D\x73\xE0\x9E\x27\x2A\xEB\x77\xA9\x13\xA4\x6A\x0A\x5A\x5A\x14\x33\x3D\x68\x1F\x81\xAE\x69\xFD\x8C\x9F\x65\x6C\x34\x42\xD9\x2D\xD0\x7F\x78\x16\xB1\x3A\xAC\x23\x31\xAD\x5E\x7F\xAE\xE7\xAE\x2B\xFA\xBA\xFC\x3C\x97\x95\x40\x93\x5F\xC3\x2D\x03\xA3\xED\xA4\x6F\x53\xD7\xFA\x40\x0E\x30\xF5\x00\x20\x2C\x00\x4C\x8C\x3B\xB4\xA3\x1F\xB6\xBF\x91\x32\xAB\xAF\x92\x98\xD3\x16\xE6\xD4\xD1\x54\x5C\x43\x5B\x2E\xAE\xEF\x57\x2A\xA8\xB4\x6F\xA4\xEF\x0D\x56\x14\xDA\x21\xAB\x20\x76\x9E\x03\xFC\x26\xB8\x9E\x3F\x3E\x03\x26\xE6\x4C\xDB\x9D\x5F\x42\x84\x3D\x45\x03\x03\x1C\x59\x88\xCA\xDC\x2E\x61\x24\x5A\xA4\xEA\x27\x0B\x73\x12\xBE\x52\xB3\x0A\xCF\x32\x17\xE2\x1E\x87\x1A\x16\x95\x48\x6D\x5A\xE0\xD0\xCF\x09\x92\x26\x66\x91\xD8\xA3\x61\x0E\xAA\x81\x81\x7F\xE8\x52\x82\xD1\x42\xE7\xE0\x1D\x18\xFA\xA4\x85\x36\xE7\x86\xE0\x0D\xEB\xBC\xD4\xC9\xD6\x3C\x43\xF1\x5D\x49\x6E\x7E\x81\x9B\x69\xB5\x89\x62\x8F\x88\x52\xD8\xD7\xFE\x27\xC1\x23\xC5\xCB\x2B\x02\xBB\xB1\x5F\xFE\xFB\x43\x85\x03\x46\xBE\x5D\xC6\xCA\x21\x26\xFF\xD7\x02\x9E\x74\x4A\xDC\xF8\x13\x15\xB1\x81\x57\x36\xCB\x65\x5C\xD1\x1D\x31\x77\xE9\x25\xC3\xC3\xB2\x32\x37\xD5\xF1\x98\x09\xE4\x6D\x63\x80\x08\xAB\x06\x92\x81\xD4\xE9\x70\x8F\xA7\x3F\xB2\xED\x86\x8C\x82\x6A\x35\xC8\x42\x5A\x82\xD1\x52\x1A\x45\x0F\x15\xA5\x00\xF0\x94\x7B\x65\x27\x57\x39\x43\xCF\x7C\x7F\xE6\xBD\x35\xB3\x7B\xF1\x19\x4C\xDE\x3A\x96\xCF\xE9\x76\xEE\x03\xE7\xC2\x43\x52\x3C\x6A\x81\xE8\xC1\x5A\x80\xBD\x11\x5D\x93\x6B\xFB\xC7\xE6\x64\x3F\xBB\x69\x1C\xE9\xDD\x25\x8B\xAF\x74\xC9\x54\x40\xCA\xCB\x93\x13\x0A\xED\xFB\x66\x92\x11\xCA\xF5\xC0\xFA\xD8\x83\x55\x03\x7C\xD3\xC5\x22\x46\x75\x70\x6B\x79\x48\x06\x2A\x82\x9A\xBF\xE6\xEB\x16\x0E\x22\x45\x01\xBC\xDD\x36\x94\x34\xA9\x35\x26\x8A\xD7\x97\xB9\xEE\x08\x72\xBF\x34\x92\x70\x83\x80\xAB\x38\xAA\x59\x68\xDD\x40\xA4\x18\x90\xB2\xF3\xD5\x03\xCA\x26\xCA\xEF\xD5\xC7\xE0\x8F\x53\x8E\xF0\x00\xE3\xA8\xED\x9F\xF9\xAD\x77\xE0\x2B\x63\x4F\x9E\xC3\xEE\x37\xBB\x78\x09\x84\x9E\xB9\x6E\xFB\x29\x99\x90\xE8\x80\xD3\x9F\x24"),
   // "TrustAsia TLS RSA Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\x80\x30\x82\x03\x68\xA0\x03\x02\x01\x02\x02\x14\x1C\x18\xD8\xCF\xE5\x53\x3F\x22\x35\x46\x53\x54\x24\x3C\x6C\x47\xD1\x5C\x4A\x9C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x58\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x32\x34\x30\x35\x31\x35\x30\x35\x34\x31\x35\x37\x5A\x17\x0D\x34\x34\x30\x35\x31\x35\x30\x35\x34\x31\x35\x36\x5A\x30\x58\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC3\x16\xB8\x1B\x6A\xA4\x44\x73\xE5\xD6\x4E\xF4\xB9\xCF\x5B\x0B\xC1\xD1\x9A\x81\xF5\x63\xB0\x8F\x43\xC1\xBB\x08\x5A\x1A\x7A\xE1\x07\x76\x26\x1F\x8F\x69\x56\xBE\xFE\x36\x60\xD0\x0C\x83\xCD\x94\xEA\xE7\xC5\x2D\x5C\x2F\x05\x16\x02\x9E\x0A\xA8\x2F\xE5\x60\x26\x54\x96\xF8\x98\x40\x1D\xAC\xAE\x9D\x74\x67\x2F\x54\xFB\x94\x63\x98\xE3\x26\x4A\x83\xC6\x95\xB6\x09\x43\x50\xCD\x0D\x7D\xDE\x44\x0E\x60\x12\x4F\x5B\x35\xBD\xBF\x99\x38\x6D\x7D\x6C\xDA\xE2\x68\x73\x1F\xF9\x31\xA5\x19\x10\x73\x05\x2A\xC3\x32\x19\x85\xEA\x34\xAA\xDD\x42\x1E\x30\x8D\x3F\x9E\xB5\x1E\x61\xD5\x6F\xBD\x00\x72\x72\xAD\x12\x3F\xAA\xA6\x49\x73\xF2\x86\x15\x95\x0C\x10\x5F\x51\x64\xCE\xFF\x77\xB8\xC9\x6D\xAC\xE5\xD5\x98\xF1\x99\x2E\x6C\xE3\xC9\x44\xF9\xB5\x43\x27\x08\x4D\xF6\x7E\xDE\x44\x79\xBB\xB2\x8C\x1C\xDA\xD3\x4B\x6C\x2E\xD6\xC3\x78\xB7\x4C\xD5\xA4\xE4\xDA\xDC\x8A\x8E\x0E\xFF\x0F\xC3\xA6\x60\x26\x28\xCF\x36\xBF\xFA\x57\x0A\xEC\x5B\x3F\xE9\x30\x16\x7B\xC4\xDB\xC4\xD4\x64\xA0\x30\xFB\xE5\xFD\x1D\x71\x92\xDD\x29\x8F\x41\x58\xDE\x00\xAD\x3A\xFD\x3D\x7C\x1A\xA8\xB1\x17\xF0\x4E\x34\x78\x58\x25\xD6\x85\x21\xEB\x79\x1D\xD0\xDC\xAB\xCF\x62\x3C\xB0\xC7\x97\x8B\xD6\xD0\x9F\xD3\xFE\x3C\xDE\xC5\xE3\xFC\x3A\x83\x70\x84\x21\x1D\x09\xC2\xA1\xFC\xBB\x28\x21\x65\x53\x60\x7A\x90\x6D\x96\x38\x5B\xFF\xF1\xD7\x7A\x5B\x6D\xD3\xC9\x70\x49\x4A\xBA\x1D\x3A\xD0\x50\xDA\x32\x20\x19\xE4\x8B\x3F\xEB\xD5\x16\x26\x37\x3C\xFB\x75\x9E\xB0\x07\x70\xB8\x14\x60\x77\xDC\xF6\x0F\x5C\x52\x0A\xBC\x5D\x6D\x8D\x51\x0A\xDA\xC5\x18\xC8\x9B\x6D\xDC\xB0\x83\xB3\x7F\xA3\x4E\x78\x4C\x98\x25\xAB\xF2\x7E\x2E\x20\x5E\x82\x15\xA6\xD6\xD8\x8F\xAC\xE5\xCD\x32\x86\xC8\xF9\xE4\xDA\x89\xE2\x3B\x3E\x93\xC5\x13\x6A\xFF\xC7\xF7\x18\xFC\x67\xB5\xEF\x18\x2C\x54\xAB\x40\xD3\x6C\x4D\xED\xC5\xC8\xB7\x38\x4F\x64\xC4\x94\x4C\x4F\xA0\xCD\xAE\xA5\xC6\x27\x25\x75\x27\x6D\xC6\xDA\x9C\x8E\xB0\x05\x99\xD5\x28\x11\xC5\x21\xD4\xB8\xFC\xC7\x96\xAB\x4A\x64\x59\x61\x6B\x13\xFC\x4B\xCC\x92\xA6\x0F\xA4\x51\xBF\x0E\x58\x90\xD9\x82\x83\x39\xA6\x5F\xB3\xFA\x6E\x92\xB3\xCD\x33\x0E\x85\xA8\xF1\x29\xD3\xAB\x59\xAD\xD6\xDB\xDE\xD4\x2B\x81\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB8\x07\x91\x79\x5C\x06\xF4\x46\xFD\x7B\x59\xCA\x5A\x26\x91\xA7\x45\x2B\xF8\x53\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x21\x9B\x6A\x05\x20\x5D\x18\x16\xA7\x12\xA4\xF7\x47\x3F\xCD\xCA\x3B\xAE\x8E\xC0\x82\xCE\xDC\x48\x25\x78\x17\x6C\xE0\xE0\x70\xC4\xD6\x96\xD9\xD9\xF6\xBF\x7A\x9C\x13\xBB\x53\x95\x92\xFF\xCF\x92\xE0\xF3\xC5\x27\x35\x57\xA8\x3B\xA2\x19\xEB\xF8\x8A\xA3\x04\xDE\x69\x8B\xE4\x3C\x3C\xBD\xE5\xAF\xD0\x12\x1B\xE5\x89\x6B\x73\x9B\x6F\x25\x02\x4B\x90\x82\xAF\x40\xC2\xAD\xBA\x9A\x60\x24\x5A\x81\x4D\x05\x18\xA3\xE2\x33\x79\x7A\x0B\x1F\x93\xED\xEC\x39\x4A\xF5\x13\x04\xEC\x8D\x74\x9D\x81\x0A\xA8\x6F\xE8\x76\xFE\x8B\x4F\x2B\x69\x12\x86\x0A\x34\x36\xDD\x82\x9B\x6F\x4F\xE0\x73\xDD\x49\x7F\x28\x3E\xC9\x3B\xA3\x57\x04\xD8\x93\xD9\x3C\x29\xDB\x12\x10\x68\xE1\xC4\x89\xB8\x06\x20\xE7\x1B\x7A\xBC\xB1\x93\x61\x9A\x19\x40\x11\xB5\x6A\x85\x57\x01\xF1\xB6\x5A\x2D\x26\x73\x27\x96\xF9\xB9\x3F\x5B\xD0\x88\x9D\xE4\xA0\xFD\xBA\x4E\x95\x09\x83\x7F\x77\xDC\x51\x05\x94\x9D\x20\x71\x76\xB1\xEC\xFA\x89\xB5\x9C\x2E\x85\xF1\xB9\x5B\x24\x5A\xB1\x09\xAB\xDA\xCA\x1A\x07\xCD\x86\xDF\xDB\x69\xDB\xB4\x48\x18\x00\x2D\xBC\xA2\xC4\x89\x45\x23\xA5\x0E\xE1\x44\x65\x3E\x8A\xC1\x6A\x30\x1D\xE2\x60\xF8\x3A\xAA\x87\x09\x42\xB9\x81\x92\xDC\x25\x88\x98\x89\xF1\x7C\x35\xF1\x62\xCA\xEE\x79\xF8\x19\x2F\xFE\x5F\xF8\xDB\x87\xCE\xEA\xA8\x0F\x0C\xA6\xE8\x0B\xA0\x3C\xE0\x6B\x94\x98\x9C\x7F\xAD\x06\xE6\x1C\xEA\x42\x10\x5F\xE2\x26\x15\x3C\x37\x39\xF7\xD7\xBC\xEE\xB8\xE5\xEF\x03\xEE\xBD\x22\xFB\x28\x86\x34\x58\xC4\x5A\x7F\x61\xA2\x78\x0F\x5E\xF3\xCA\x9D\xBC\x1B\x3C\xA7\xC8\x2D\xF6\xA7\x22\x11\xCA\x03\xD8\xE7\x67\x0A\x8A\x0E\xCB\x35\x8E\x34\x39\xD8\xC8\xEA\x8D\x9F\x64\xE0\x37\xB0\x6C\xD7\xC5\x8F\xC1\x90\x61\x26\xDB\x24\x8C\x1E\x4A\xC0\x88\xDF\x35\x90\xCE\x3F\x02\xF4\x64\x61\xDE\xC7\xC8\x07\x95\xDE\x32\x18\x3A\x8F\xA2\x42\x40\x24\xE5\xD6\x33\x5D\x7C\xAE\xEF\xB1\x4D\x4F\xD4\x57\x90\x35\x6A\xDC\xAE\x04\x97\x49\x89\x34\x97\x2E\x30\x04\xE7\x98\xF7\xDB\x0B\x01\x90\xC1\x1F\x0A\x3F\xF8\xC2\xFE\x4E\xFA\xDB\x9A\x73\x73\x16\xBC\x05\xB8\x79\xD8\x59\xAF\x06\xE7\x3F\x67\x38\x39\xB1\x7C\xAB\x94\xE0\x29\x14\xA6\x28\xF2\xDF\x6D\xE2\x9A\xDB\x54\x43\xF8\x47\x58\xA3\x5D\x74\x15\x9C\xC1\xAB\xB0\xC7\x34"),
   // "D-TRUST EV Root CA 1 2020"
   CertificateDer::from_slice(b"\x30\x82\x02\xDB\x30\x82\x02\x60\xA0\x03\x02\x01\x02\x02\x10\x5F\x02\x41\xD7\x7A\x87\x7C\x4C\x03\xA3\xAC\x96\x8D\xFB\xFF\xD0\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x20\x32\x30\x32\x30\x30\x1E\x17\x0D\x32\x30\x30\x32\x31\x31\x31\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x35\x30\x32\x31\x31\x30\x39\x35\x39\x35\x39\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x20\x32\x30\x32\x30\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xF1\x0B\xDD\x86\x43\x20\x19\xDF\x97\x85\xE8\x22\x4A\x9B\xCF\x9D\x98\xBF\xB4\x05\x26\xC9\xCB\xE3\xA6\xD2\x8F\xC5\x9E\x78\x7B\x31\x89\xA9\x89\xAD\x27\x3C\x65\x10\x82\xFC\xDF\xC3\x9D\x4E\xF0\x33\x23\xC4\xD2\x32\xF5\x1C\xB0\xDF\x33\x17\x5D\xC5\xF0\xB1\x8A\xF9\xEF\xB9\xB7\x14\xCA\x29\x4A\xC2\x0F\xA9\x7F\x75\x65\x49\x2A\x30\x67\xF4\x64\xF7\xD6\x1A\x77\xDA\xC3\xC2\x97\x61\x42\x7B\x49\xAD\xA3\x82\x01\x0D\x30\x82\x01\x09\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x7F\x10\x01\x16\x37\x3A\xA4\x28\xE4\x50\xF8\xA4\xF7\xEC\x6B\x32\xB6\xFE\xE9\x8B\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x81\xC6\x06\x03\x55\x1D\x1F\x04\x81\xBE\x30\x81\xBB\x30\x3E\xA0\x3C\xA0\x3A\x86\x38\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x63\x72\x6C\x2F\x64\x2D\x74\x72\x75\x73\x74\x5F\x65\x76\x5F\x72\x6F\x6F\x74\x5F\x63\x61\x5F\x31\x5F\x32\x30\x32\x30\x2E\x63\x72\x6C\x30\x79\xA0\x77\xA0\x75\x86\x73\x6C\x64\x61\x70\x3A\x2F\x2F\x64\x69\x72\x65\x63\x74\x6F\x72\x79\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x43\x4E\x3D\x44\x2D\x54\x52\x55\x53\x54\x25\x32\x30\x45\x56\x25\x32\x30\x52\x6F\x6F\x74\x25\x32\x30\x43\x41\x25\x32\x30\x31\x25\x32\x30\x32\x30\x32\x30\x2C\x4F\x3D\x44\x2D\x54\x72\x75\x73\x74\x25\x32\x30\x47\x6D\x62\x48\x2C\x43\x3D\x44\x45\x3F\x63\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x72\x65\x76\x6F\x63\x61\x74\x69\x6F\x6E\x6C\x69\x73\x74\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\xCA\x3C\xC6\x2A\x75\xC2\x5E\x75\x62\x39\x36\x00\x60\x5A\x8B\xC1\x93\x99\xCC\xD9\xDB\x41\x3B\x3B\x87\x99\x17\x3B\xD5\xCC\x4F\xCA\x22\xF7\xA0\x80\xCB\xF9\xB4\xB1\x1B\x56\xF5\x72\xD2\xFC\x19\xD1\x02\x31\x00\x91\xF7\x30\x93\x3F\x10\x46\x2B\x71\xA4\xD0\x3B\x44\x9B\xC0\x29\x02\x05\xB2\x41\x77\x51\xF3\x79\x5A\x9E\x8E\x14\xA0\x4E\x42\xD2\x5B\x81\xF3\x34\x6A\x03\xE7\x22\x38\x50\x5B\xED\x19\x4F\x43\x16"),
   // "COMODO Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x04\x1D\x30\x82\x03\x05\xA0\x03\x02\x01\x02\x02\x10\x4E\x81\x2D\x8A\x82\x65\xE0\x0B\x02\xEE\x3E\x35\x02\x46\xE5\x3D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x81\x81\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x1B\x30\x19\x06\x03\x55\x04\x08\x13\x12\x47\x72\x65\x61\x74\x65\x72\x20\x4D\x61\x6E\x63\x68\x65\x73\x74\x65\x72\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x53\x61\x6C\x66\x6F\x72\x64\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x41\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x27\x30\x25\x06\x03\x55\x04\x03\x13\x1E\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x30\x36\x31\x32\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x32\x39\x31\x32\x33\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x81\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x1B\x30\x19\x06\x03\x55\x04\x08\x13\x12\x47\x72\x65\x61\x74\x65\x72\x20\x4D\x61\x6E\x63\x68\x65\x73\x74\x65\x72\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x53\x61\x6C\x66\x6F\x72\x64\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x41\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x27\x30\x25\x06\x03\x55\x04\x03\x13\x1E\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xD0\x40\x8B\x8B\x72\xE3\x91\x1B\xF7\x51\xC1\x1B\x54\x04\x98\xD3\xA9\xBF\xC1\xE6\x8A\x5D\x3B\x87\xFB\xBB\x88\xCE\x0D\xE3\x2F\x3F\x06\x96\xF0\xA2\x29\x50\x99\xAE\xDB\x3B\xA1\x57\xB0\x74\x51\x71\xCD\xED\x42\x91\x4D\x41\xFE\xA9\xC8\xD8\x6A\x86\x77\x44\xBB\x59\x66\x97\x50\x5E\xB4\xD4\x2C\x70\x44\xCF\xDA\x37\x95\x42\x69\x3C\x30\xC4\x71\xB3\x52\xF0\x21\x4D\xA1\xD8\xBA\x39\x7C\x1C\x9E\xA3\x24\x9D\xF2\x83\x16\x98\xAA\x16\x7C\x43\x9B\x15\x5B\xB7\xAE\x34\x91\xFE\xD4\x62\x26\x18\x46\x9A\x3F\xEB\xC1\xF9\xF1\x90\x57\xEB\xAC\x7A\x0D\x8B\xDB\x72\x30\x6A\x66\xD5\xE0\x46\xA3\x70\xDC\x68\xD9\xFF\x04\x48\x89\x77\xDE\xB5\xE9\xFB\x67\x6D\x41\xE9\xBC\x39\xBD\x32\xD9\x62\x02\xF1\xB1\xA8\x3D\x6E\x37\x9C\xE2\x2F\xE2\xD3\xA2\x26\x8B\xC6\xB8\x55\x43\x88\xE1\x23\x3E\xA5\xD2\x24\x39\x6A\x47\xAB\x00\xD4\xA1\xB3\xA9\x25\xFE\x0D\x3F\xA7\x1D\xBA\xD3\x51\xC1\x0B\xA4\xDA\xAC\x38\xEF\x55\x50\x24\x05\x65\x46\x93\x34\x4F\x2D\x8D\xAD\xC6\xD4\x21\x19\xD2\x8E\xCA\x05\x61\x71\x07\x73\x47\xE5\x8A\x19\x12\xBD\x04\x4D\xCE\x4E\x9C\xA5\x48\xAC\xBB\x26\xF7\x02\x03\x01\x00\x01\xA3\x81\x8E\x30\x81\x8B\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x0B\x58\xE5\x8B\xC6\x4C\x15\x37\xA4\x40\xA9\x30\xA9\x21\xBE\x47\x36\x5A\x56\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x49\x06\x03\x55\x1D\x1F\x04\x42\x30\x40\x30\x3E\xA0\x3C\xA0\x3A\x86\x38\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x63\x6F\x6D\x6F\x64\x6F\x63\x61\x2E\x63\x6F\x6D\x2F\x43\x4F\x4D\x4F\x44\x4F\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x2E\x63\x72\x6C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x3E\x98\x9E\x9B\xF6\x1B\xE9\xD7\x39\xB7\x78\xAE\x1D\x72\x18\x49\xD3\x87\xE4\x43\x82\xEB\x3F\xC9\xAA\xF5\xA8\xB5\xEF\x55\x7C\x21\x52\x65\xF9\xD5\x0D\xE1\x6C\xF4\x3E\x8C\x93\x73\x91\x2E\x02\xC4\x4E\x07\x71\x6F\xC0\x8F\x38\x61\x08\xA8\x1E\x81\x0A\xC0\x2F\x20\x2F\x41\x8B\x91\xDC\x48\x45\xBC\xF1\xC6\xDE\xBA\x76\x6B\x33\xC8\x00\x2D\x31\x46\x4C\xED\xE7\x9D\xCF\x88\x94\xFF\x33\xC0\x56\xE8\x24\x86\x26\xB8\xD8\x38\x38\xDF\x2A\x6B\xDD\x12\xCC\xC7\x3F\x47\x17\x4C\xA2\xC2\x06\x96\x09\xD6\xDB\xFE\x3F\x3C\x46\x41\xDF\x58\xE2\x56\x0F\x3C\x3B\xC1\x1C\x93\x35\xD9\x38\x52\xAC\xEE\xC8\xEC\x2E\x30\x4E\x94\x35\xB4\x24\x1F\x4B\x78\x69\xDA\xF2\x02\x38\xCC\x95\x52\x93\xF0\x70\x25\x59\x9C\x20\x67\xC4\xEE\xF9\x8B\x57\x61\xF4\x92\x76\x7D\x3F\x84\x8D\x55\xB7\xE8\xE5\xAC\xD5\xF1\xF5\x19\x56\xA6\x5A\xFB\x90\x1C\xAF\x93\xEB\xE5\x1C\xD4\x67\x97\x5D\x04\x0E\xBE\x0B\x83\xA6\x17\x83\xB9\x30\x12\xA0\xC5\x33\x15\x05\xB9\x0D\xFB\xC7\x05\x76\xE3\xD8\x4A\x8D\xFC\x34\x17\xA3\xC6\x21\x28\xBE\x30\x45\x31\x1E\xC7\x78\xBE\x58\x61\x38\xAC\x3B\xE2\x01\x65"),
   // "emSign Root CA - C1"
   CertificateDer::from_slice(b"\x30\x82\x03\x73\x30\x82\x02\x5B\xA0\x03\x02\x01\x02\x02\x0B\x00\xAE\xCF\x00\xBA\xC4\xCF\x32\xF8\x43\xB2\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x56\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x13\x0B\x65\x4D\x75\x64\x68\x72\x61\x20\x49\x6E\x63\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x65\x6D\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x43\x31\x30\x1E\x17\x0D\x31\x38\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x17\x0D\x34\x33\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x30\x56\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x13\x0B\x65\x4D\x75\x64\x68\x72\x61\x20\x49\x6E\x63\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x65\x6D\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x43\x31\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xCF\xEB\xA9\xB9\xF1\x99\x05\xCC\xD8\x28\x21\x4A\xF3\x73\x34\x51\x84\x56\x10\xF5\xA0\x4F\x2C\x12\xE3\xFA\x13\x9A\x27\xD0\xCF\xF9\x79\x1A\x74\x5F\x1D\x79\x39\xFC\x5B\xF8\x70\x8E\xE0\x92\x52\xF7\xE4\x25\xF9\x54\x83\xD9\x1D\xD3\xC8\x5A\x85\x3F\x5E\xC7\xB6\x07\xEE\x3E\xC0\xCE\x9A\xAF\xAC\x56\x42\x2A\x39\x25\x70\xD6\xBF\xB5\x7B\x36\xAD\xAC\xF6\x73\xDC\xCD\xD7\x1D\x8A\x83\xA5\xFB\x2B\x90\x15\x37\x6B\x1C\x26\x47\xDC\x3B\x29\x56\x93\x6A\xB3\xC1\x6A\x3A\x9D\x3D\xF5\xC1\x97\x38\x58\x05\x8B\x1C\x11\xE3\xE4\xB4\xB8\x5D\x85\x1D\x83\xFE\x78\x5F\x0B\x45\x68\x18\x48\xA5\x46\x73\x34\x3B\xFE\x0F\xC8\x76\xBB\xC7\x18\xF3\x05\xD1\x86\xF3\x85\xED\xE7\xB9\xD9\x32\xAD\x55\x88\xCE\xA6\xB6\x91\xB0\x4F\xAC\x7E\x15\x23\x96\xF6\x3F\xF0\x20\x34\x16\xDE\x0A\xC6\xC4\x04\x45\x79\x7F\xA7\xFD\xBE\xD2\xA9\xA5\xAF\x9C\xC5\x23\x2A\xF7\x3C\x21\x6C\xBD\xAF\x8F\x4E\xC5\x3A\xB2\xF3\x34\x12\xFC\xDF\x80\x1A\x49\xA4\xD4\xA9\x95\xF7\x9E\x89\x5E\xA2\x89\xAC\x94\xCB\xA8\x68\x9B\xAF\x8A\x65\x27\xCD\x89\xEE\xDD\x8C\xB5\x6B\x29\x70\x43\xA0\x69\x0B\xE4\xB9\x0F\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xFE\xA1\xE0\x70\x1E\x2A\x03\x39\x52\x5A\x42\xBE\x5C\x91\x85\x7A\x18\xAA\x4D\xB5\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\xC2\x4A\x56\xFA\x15\x21\x7B\x28\xA2\xE9\xE5\x1D\xFB\xF8\x2D\xC4\x39\x96\x41\x4C\x3B\x27\x2C\xC4\x6C\x18\x15\x80\xC6\xAC\xAF\x47\x59\x2F\x26\x0B\xE3\x36\xB0\xEF\x3B\xFE\x43\x97\x49\x32\x99\x12\x15\x5B\xDF\x11\x29\xFF\xAB\x53\xF8\xBB\xC1\x78\x0F\xAC\x9C\x53\xAF\x57\xBD\x68\x8C\x3D\x69\x33\xF0\xA3\xA0\x23\x63\x3B\x64\x67\x22\x44\xAD\xD5\x71\xCB\x56\x2A\x78\x92\xA3\x4F\x12\x31\x36\x36\xE2\xDE\xFE\x00\xC4\xA3\x60\x0F\x27\xAD\xA0\xB0\x8A\xB5\x36\x7A\x52\xA1\xBD\x27\xF4\x20\x27\x62\xE8\x4D\x94\x24\x13\xE4\x0A\x04\xE9\x3C\xAB\x2E\xC8\x43\x09\x4A\xC6\x61\x04\xE5\x49\x34\x7E\xD3\xC4\xC8\xF5\x0F\xC0\xAA\xE9\xBA\x54\x5E\xF3\x63\x2B\x4F\x4F\x50\xD4\xFE\xB9\x7B\x99\x8C\x3D\xC0\x2E\xBC\x02\x2B\xD3\xC4\x40\xE4\x8A\x07\x31\x1E\x9B\xCE\x26\x99\x13\xFB\x11\xEA\x9A\x22\x0C\x11\x19\xC7\x5E\x1B\x81\x50\x30\xC8\x96\x12\x6E\xE7\xCB\x41\x7F\x91\x3B\xA2\x47\xB7\x54\x80\x1B\xDC\x00\xCC\x9A\x90\xEA\xC3\xC3\x50\x06\x62\x0C\x30\xC0\x15\x48\xA7\xA8\x59\x7C\xE1\xAE\x22\xA2\xE2\x0A\x7A\x0F\xFA\x62\xAB\x52\x4C\xE1\xF1\xDF\xCA\xBE\x83\x0D\x42"),
   // "COMODO ECC Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x02\x89\x30\x82\x02\x0F\xA0\x03\x02\x01\x02\x02\x10\x1F\x47\xAF\xAA\x62\x00\x70\x50\x54\x4C\x01\x9E\x9B\x63\x99\x2A\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x81\x85\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x1B\x30\x19\x06\x03\x55\x04\x08\x13\x12\x47\x72\x65\x61\x74\x65\x72\x20\x4D\x61\x6E\x63\x68\x65\x73\x74\x65\x72\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x53\x61\x6C\x66\x6F\x72\x64\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x41\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x43\x4F\x4D\x4F\x44\x4F\x20\x45\x43\x43\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x30\x38\x30\x33\x30\x36\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x38\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x85\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x1B\x30\x19\x06\x03\x55\x04\x08\x13\x12\x47\x72\x65\x61\x74\x65\x72\x20\x4D\x61\x6E\x63\x68\x65\x73\x74\x65\x72\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x53\x61\x6C\x66\x6F\x72\x64\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x41\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x43\x4F\x4D\x4F\x44\x4F\x20\x45\x43\x43\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x03\x47\x7B\x2F\x75\xC9\x82\x15\x85\xFB\x75\xE4\x91\x16\xD4\xAB\x62\x99\xF5\x3E\x52\x0B\x06\xCE\x41\x00\x7F\x97\xE1\x0A\x24\x3C\x1D\x01\x04\xEE\x3D\xD2\x8D\x09\x97\x0C\xE0\x75\xE4\xFA\xFB\x77\x8A\x2A\xF5\x03\x60\x4B\x36\x8B\x16\x23\x16\xAD\x09\x71\xF4\x4A\xF4\x28\x50\xB4\xFE\x88\x1C\x6E\x3F\x6C\x2F\x2F\x09\x59\x5B\xA5\x5B\x0B\x33\x99\xE2\xC3\x3D\x89\xF9\x6A\x2C\xEF\xB2\xD3\x06\xE9\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x75\x71\xA7\x19\x48\x19\xBC\x9D\x9D\xEA\x41\x47\xDF\x94\xC4\x48\x77\x99\xD3\x79\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xEF\x03\x5B\x7A\xAC\xB7\x78\x0A\x72\xB7\x88\xDF\xFF\xB5\x46\x14\x09\x0A\xFA\xA0\xE6\x7D\x08\xC6\x1A\x87\xBD\x18\xA8\x73\xBD\x26\xCA\x60\x0C\x9D\xCE\x99\x9F\xCF\x5C\x0F\x30\xE1\xBE\x14\x31\xEA\x02\x30\x14\xF4\x93\x3C\x49\xA7\x33\x7A\x90\x46\x47\xB3\x63\x7D\x13\x9B\x4E\xB7\x6F\x18\x37\x80\x53\xFE\xDD\x20\xE0\x35\x9A\x36\xD1\xC7\x01\xB9\xE6\xDC\xDD\xF3\xFF\x1D\x2C\x3A\x16\x57\xD9\x92\x39\xD6"),
   // "GlobalSign"
   CertificateDer::from_slice(b"\x30\x82\x02\x1E\x30\x82\x01\xA4\xA0\x03\x02\x01\x02\x02\x11\x60\x59\x49\xE0\x26\x2E\xBB\x55\xF9\x0A\x77\x8A\x71\xF9\x4A\xD8\x6C\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x50\x31\x24\x30\x22\x06\x03\x55\x04\x0B\x13\x1B\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x35\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x1E\x17\x0D\x31\x32\x31\x31\x31\x33\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x39\x30\x33\x31\x34\x30\x37\x5A\x30\x50\x31\x24\x30\x22\x06\x03\x55\x04\x0B\x13\x1B\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x35\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x47\x45\x0E\x96\xFB\x7D\x5D\xBF\xE9\x39\xD1\x21\xF8\x9F\x0B\xB6\xD5\x7B\x1E\x92\x3A\x48\x59\x1C\xF0\x62\x31\x2D\xC0\x7A\x28\xFE\x1A\xA7\x5C\xB3\xB6\xCC\x97\xE7\x45\xD4\x58\xFA\xD1\x77\x6D\x43\xA2\xC0\x87\x65\x34\x0A\x1F\x7A\xDD\xEB\x3C\x33\xA1\xC5\x9D\x4D\xA4\x6F\x41\x95\x38\x7F\xC9\x1E\x84\xEB\xD1\x9E\x49\x92\x87\x94\x87\x0C\x3A\x85\x4A\x66\x9F\x9D\x59\x93\x4D\x97\x61\x06\x86\x4A\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x3D\xE6\x29\x48\x9B\xEA\x07\xCA\x21\x44\x4A\x26\xDE\x6E\xDE\xD2\x83\xD0\x9F\x59\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xE5\x69\x12\xC9\x6E\xDB\xC6\x31\xBA\x09\x41\xE1\x97\xF8\xFB\xFD\x9A\xE2\x7D\x12\xC9\xED\x7C\x64\xD3\xCB\x05\x25\x8B\x56\xD9\xA0\xE7\x5E\x5D\x4E\x0B\x83\x9C\x5B\x76\x29\xA0\x09\x26\x21\x6A\x62\x02\x30\x71\xD2\xB5\x8F\x5C\xEA\x3B\xE1\x78\x09\x85\xA8\x75\x92\x3B\xC8\x5C\xFD\x48\xEF\x0D\x74\x22\xA8\x08\xE2\x6E\xC5\x49\xCE\xC7\x0C\xBC\xA7\x61\x69\xF1\xF7\x3B\xE1\x2A\xCB\xF9\x2B\xF3\x66\x90\x37"),
   // "Amazon Root CA 3"
   CertificateDer::from_slice(b"\x30\x82\x01\xB6\x30\x82\x01\x5B\xA0\x03\x02\x01\x02\x02\x13\x06\x6C\x9F\xD5\x74\x97\x36\x66\x3F\x3B\x0B\x9A\xD9\xE8\x9E\x76\x03\xF2\x4A\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x30\x1E\x17\x0D\x31\x35\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x30\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x30\x59\x30\x13\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x08\x2A\x86\x48\xCE\x3D\x03\x01\x07\x03\x42\x00\x04\x29\x97\xA7\xC6\x41\x7F\xC0\x0D\x9B\xE8\x01\x1B\x56\xC6\xF2\x52\xA5\xBA\x2D\xB2\x12\xE8\xD2\x2E\xD7\xFA\xC9\xC5\xD8\xAA\x6D\x1F\x73\x81\x3B\x3B\x98\x6B\x39\x7C\x33\xA5\xC5\x4E\x86\x8E\x80\x17\x68\x62\x45\x57\x7D\x44\x58\x1D\xB3\x37\xE5\x67\x08\xEB\x66\xDE\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xAB\xB6\xDB\xD7\x06\x9E\x37\xAC\x30\x86\x07\x91\x70\xC7\x9C\xC4\x19\xB1\x78\xC0\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x49\x00\x30\x46\x02\x21\x00\xE0\x85\x92\xA3\x17\xB7\x8D\xF9\x2B\x06\xA5\x93\xAC\x1A\x98\x68\x61\x72\xFA\xE1\xA1\xD0\xFB\x1C\x78\x60\xA6\x43\x99\xC5\xB8\xC4\x02\x21\x00\x9C\x02\xEF\xF1\x94\x9C\xB3\x96\xF9\xEB\xC6\x2A\xF8\xB6\x2C\xFE\x3A\x90\x14\x16\xD7\x8C\x63\x24\x48\x1C\xDF\x30\x7D\xD5\x68\x3B"),
   // "QuoVadis Root CA 3"
   CertificateDer::from_slice(b"\x30\x82\x06\x9D\x30\x82\x04\x85\xA0\x03\x02\x01\x02\x02\x02\x05\xC6\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x30\x1E\x17\x0D\x30\x36\x31\x31\x32\x34\x31\x39\x31\x31\x32\x33\x5A\x17\x0D\x33\x31\x31\x31\x32\x34\x31\x39\x30\x36\x34\x34\x5A\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xCC\x57\x42\x16\x54\x9C\xE6\x98\xD3\xD3\x4D\xEE\xFE\xED\xC7\x9F\x43\x39\x4A\x65\xB3\xE8\x16\x88\x34\xDB\x0D\x59\x91\x74\xCF\x92\xB8\x04\x40\xAD\x02\x4B\x31\xAB\xBC\x8D\x91\x68\xD8\x20\x0E\x1A\x01\xE2\x1A\x7B\x4E\x17\x5D\xE2\x8A\xB7\x3F\x99\x1A\xCD\xEB\x61\xAB\xC2\x65\xA6\x1F\xB7\xB7\xBD\xB7\x8F\xFC\xFD\x70\x8F\x0B\xA0\x67\xBE\x01\xA2\x59\xCF\x71\xE6\x0F\x29\x76\xFF\xB1\x56\x79\x45\x2B\x1F\x9E\x7A\x54\xE8\xA3\x29\x35\x68\xA4\x01\x4F\x0F\xA4\x2E\x37\xEF\x1B\xBF\xE3\x8F\x10\xA8\x72\xAB\x58\x57\xE7\x54\x86\xC8\xC9\xF3\x5B\xDA\x2C\xDA\x5D\x8E\x6E\x3C\xA3\x3E\xDA\xFB\x82\xE5\xDD\xF2\x5C\xB2\x05\x33\x6F\x8A\x36\xCE\xD0\x13\x4E\xFF\xBF\x4A\x0C\x34\x4C\xA6\xC3\x21\xBD\x50\x04\x55\xEB\xB1\xBB\x9D\xFB\x45\x1E\x64\x15\xDE\x55\x01\x8C\x02\x76\xB5\xCB\xA1\x3F\x42\x69\xBC\x2F\xBD\x68\x43\x16\x56\x89\x2A\x37\x61\x91\xFD\xA6\xAE\x4E\xC0\xCB\x14\x65\x94\x37\x4B\x92\x06\xEF\x04\xD0\xC8\x9C\x88\xDB\x0B\x7B\x81\xAF\xB1\x3D\x2A\xC4\x65\x3A\x78\xB6\xEE\xDC\x80\xB1\xD2\xD3\x99\x9C\x3A\xEE\x6B\x5A\x6B\xB3\x8D\xB7\xD5\xCE\x9C\xC2\xBE\xA5\x4B\x2F\x16\xB1\x9E\x68\x3B\x06\x6F\xAE\x7D\x9F\xF8\xDE\xEC\xCC\x29\xA7\x98\xA3\x25\x43\x2F\xEF\xF1\x5F\x26\xE1\x88\x4D\xF8\x5E\x6E\xD7\xD9\x14\x6E\x19\x33\x69\xA7\x3B\x84\x89\x93\xC4\x53\x55\x13\xA1\x51\x78\x40\xF8\xB8\xC9\xA2\xEE\x7B\xBA\x52\x42\x83\x9E\x14\xED\x05\x52\x5A\x59\x56\xA7\x97\xFC\x9D\x3F\x0A\x29\xD8\xDC\x4F\x91\x0E\x13\xBC\xDE\x95\xA4\xDF\x8B\x99\xBE\xAC\x9B\x33\x88\xEF\xB5\x81\xAF\x1B\xC6\x22\x53\xC8\xF6\xC7\xEE\x97\x14\xB0\xC5\x7C\x78\x52\xC8\xF0\xCE\x6E\x77\x60\x84\xA6\xE9\x2A\x76\x20\xED\x58\x01\x17\x30\x93\xE9\x1A\x8B\xE0\x73\x63\xD9\x6A\x92\x94\x49\x4E\xB4\xAD\x4A\x85\xC4\xA3\x22\x30\xFC\x09\xED\x68\x22\x73\xA6\x88\x0C\x55\x21\x58\xC5\xE1\x3A\x9F\x2A\xDD\xCA\xE1\x90\xE0\xD9\x73\xAB\x6C\x80\xB8\xE8\x0B\x64\x93\xA0\x9C\x8C\x19\xFF\xB3\xD2\x0C\xEC\x91\x26\x87\x8A\xB3\xA2\xE1\x70\x8F\x2C\x0A\xE5\xCD\x6D\x68\x51\xEB\xDA\x3F\x05\x7F\x8B\x32\xE6\x13\x5C\x6B\xFE\x5F\x40\xE2\x22\xC8\xB4\xB4\x64\x4F\xD6\xBA\x7D\x48\x3E\xA8\x69\x0C\xD7\xBB\x86\x71\xC9\x73\xB8\x3F\x3B\x9D\x25\x4B\xDA\xFF\x40\xEB\x02\x03\x01\x00\x01\xA3\x82\x01\x95\x30\x82\x01\x91\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x81\xE1\x06\x03\x55\x1D\x20\x04\x81\xD9\x30\x81\xD6\x30\x81\xD3\x06\x09\x2B\x06\x01\x04\x01\xBE\x58\x00\x03\x30\x81\xC5\x30\x81\x93\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x02\x30\x81\x86\x1A\x81\x83\x41\x6E\x79\x20\x75\x73\x65\x20\x6F\x66\x20\x74\x68\x69\x73\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x63\x6F\x6E\x73\x74\x69\x74\x75\x74\x65\x73\x20\x61\x63\x63\x65\x70\x74\x61\x6E\x63\x65\x20\x6F\x66\x20\x74\x68\x65\x20\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x50\x6F\x6C\x69\x63\x79\x20\x2F\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x50\x72\x61\x63\x74\x69\x63\x65\x20\x53\x74\x61\x74\x65\x6D\x65\x6E\x74\x2E\x30\x2D\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x01\x16\x21\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x71\x75\x6F\x76\x61\x64\x69\x73\x67\x6C\x6F\x62\x61\x6C\x2E\x63\x6F\x6D\x2F\x63\x70\x73\x30\x0B\x06\x03\x55\x1D\x0F\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF2\xC0\x13\xE0\x82\x43\x3E\xFB\xEE\x2F\x67\x32\x96\x35\x5C\xDB\xB8\xCB\x02\xD0\x30\x6E\x06\x03\x55\x1D\x23\x04\x67\x30\x65\x80\x14\xF2\xC0\x13\xE0\x82\x43\x3E\xFB\xEE\x2F\x67\x32\x96\x35\x5C\xDB\xB8\xCB\x02\xD0\xA1\x49\xA4\x47\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x82\x02\x05\xC6\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00\x4F\xAD\xA0\x2C\x4C\xFA\xC0\xF2\x6F\xF7\x66\x55\xAB\x23\x34\xEE\xE7\x29\xDA\xC3\x5B\xB6\xB0\x83\xD9\xD0\xD0\xE2\x21\xFB\xF3\x60\xA7\x3B\x5D\x60\x53\x27\xA2\x9B\xF6\x08\x22\x2A\xE7\xBF\xA0\x72\xE5\x9C\x24\x6A\x31\xB1\x90\x7A\x27\xDB\x84\x11\x89\x27\xA6\x77\x5A\x38\xD7\xBF\xAC\x86\xFC\xEE\x5D\x83\xBC\x06\xC6\xD1\x77\x6B\x0F\x6D\x24\x2F\x4B\x7A\x6C\xA7\x07\x96\xCA\xE3\x84\x9F\xAD\x88\x8B\x1D\xAB\x16\x8D\x5B\x66\x17\xD9\x16\xF4\x8B\x80\xD2\xDD\xF8\xB2\x76\xC3\xFC\x38\x13\xAA\x0C\xDE\x42\x69\x2B\x6E\xF3\x3C\xEB\x80\x27\xDB\xF5\xA6\x44\x0D\x9F\x5A\x55\x59\x0B\xD5\x0D\x52\x48\xC5\xAE\x9F\xF2\x2F\x80\xC5\xEA\x32\x50\x35\x12\x97\x2E\xC1\xE1\xFF\xF1\x23\x88\x51\x38\x9F\xF2\x66\x56\x76\xE7\x0F\x51\x97\xA5\x52\x0C\x4D\x49\x51\x95\x36\x3D\xBF\xA2\x4B\x0C\x10\x1D\x86\x99\x4C\xAA\xF3\x72\x11\x93\xE4\xEA\xF6\x9B\xDA\xA8\x5D\xA7\x4D\xB7\x9E\x02\xAE\x73\x00\xC8\xDA\x23\x03\xE8\xF9\xEA\x19\x74\x62\x00\x94\xCB\x22\x20\xBE\x94\xA7\x59\xB5\x82\x6A\xBE\x99\x79\x7A\xA9\xF2\x4A\x24\x52\xF7\x74\xFD\xBA\x4E\xE6\xA8\x1D\x02\x6E\xB1\x0D\x80\x44\xC1\xAE\xD3\x23\x37\x5F\xBB\x85\x7C\x2B\x92\x2E\xE8\x7E\xA5\x8B\xDD\x99\xE1\xBF\x27\x6F\x2D\x5D\xAA\x7B\x87\xFE\x0A\xDD\x4B\xFC\x8E\xF5\x26\xE4\x6E\x70\x42\x6E\x33\xEC\x31\x9E\x7B\x93\xC1\xE4\xC9\x69\x1A\x3D\xC0\x6B\x4E\x22\x6D\xEE\xAB\x58\x4D\xC6\xD0\x41\xC1\x2B\xEA\x4F\x12\x87\x5E\xEB\x45\xD8\x6C\xF5\x98\x02\xD3\xA0\xD8\x55\x8A\x06\x99\x19\xA2\xA0\x77\xD1\x30\x9E\xAC\xCC\x75\xEE\x83\xF5\xB0\x62\x39\xCF\x6C\x57\xE2\x4C\xD2\x91\x0B\x0E\x75\x28\x1B\x9A\xBF\xFD\x1A\x43\xF1\xCA\x77\xFB\x3B\x8F\x61\xB8\x69\x28\x16\x42\x04\x5E\x70\x2A\x1C\x21\xD8\x8F\xE1\xBD\x23\x5B\x2D\x74\x40\x92\xD9\x63\x19\x0D\x73\xDD\x69\xBC\x62\x47\xBC\xE0\x74\x2B\xB2\xEB\x7D\xBE\x41\x1B\xB5\xC0\x46\xC5\xA1\x22\xCB\x5F\x4E\xC1\x28\x92\xDE\x18\xBA\xD5\x2A\x28\xBB\x11\x8B\x17\x93\x98\x99\x60\x94\x5C\x23\xCF\x5A\x27\x97\x5E\x0B\x05\x06\x93\x37\x1E\x3B\x69\x36\xEB\xA9\x9E\x61\x1D\x8F\x32\xDA\x8E\x0C\xD6\x74\x3E\x7B\x09\x24\xDA\x01\x77\x47\xC4\x3B\xCD\x34\x8C\x99\xF5\xCA\xE1\x25\x61\x33\xB2\x59\x1B\xE2\x6E\xD7\x37\x57\xB6\x0D\xA9\x12\xDA"),
   // "SwissSign RSA TLS Root CA 2022 - 1"
   CertificateDer::from_slice(b"\x30\x82\x05\x93\x30\x82\x03\x7B\xA0\x03\x02\x01\x02\x02\x14\x43\xFA\x0C\x5F\x4E\x1B\x80\x18\x44\xEF\xD1\xB4\x4F\x35\x1F\x44\xF4\x80\xED\xCB\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x41\x47\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x52\x53\x41\x20\x54\x4C\x53\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x32\x20\x2D\x20\x31\x30\x1E\x17\x0D\x32\x32\x30\x36\x30\x38\x31\x31\x30\x38\x32\x32\x5A\x17\x0D\x34\x37\x30\x36\x30\x38\x31\x31\x30\x38\x32\x32\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x41\x47\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x52\x53\x41\x20\x54\x4C\x53\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x32\x20\x2D\x20\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xCB\x2A\x68\xE2\x0B\xC3\x57\xBC\x35\x63\xBC\x70\xA5\x3B\xF3\x8C\x3C\x4E\x57\x96\x6E\xC3\x4E\x36\xA4\xF6\x02\xCA\x1E\xAA\xAE\xB8\xDE\xA8\xAF\x1D\x76\xDA\xBA\x35\xD0\x91\x70\x07\xDF\xB3\x06\xF2\x8A\xF2\x2E\x55\x51\x7B\xBB\x2C\x24\xCB\x7F\x92\x26\x80\xA3\xB4\x94\xF6\x82\xA1\xA4\xE8\xFA\x75\x1D\x59\xF3\x07\x6A\x61\x64\xE2\xC6\x8C\x95\xAF\xA3\xBB\x8E\x6F\x56\xCF\x71\xCC\x5E\x81\x61\x0D\x6D\xF2\xAB\x02\x2E\xA4\x97\xE5\x71\xFC\x8A\xB0\x91\x20\x5B\x9C\x74\x52\x6D\xAE\x15\x27\x59\x78\xF2\x09\xCA\x65\x0E\x7F\xCB\xF4\xEB\xE7\xDC\xA9\x4C\x77\xF6\x2B\x16\x04\x95\xAE\x9C\x71\xA5\x3F\x2A\xDA\x41\x42\xE7\x3C\x84\x10\xF4\xE1\x3D\x8C\x6B\xE2\x2B\x91\x47\x55\x4F\xB8\x56\xBE\x45\xDE\x22\x51\x4D\x4E\x28\xD9\x5F\x19\x41\x06\x8F\x0E\x4D\x06\xE0\x70\x40\x23\x01\x6A\xE4\xCB\x13\x9B\x73\xAC\x4D\x14\x48\x92\x2D\xFE\x6D\xA7\xF8\x87\x6B\x79\x75\xE1\xBE\x10\xB1\xAA\x88\x40\x59\x54\xD7\xCF\xC4\xD0\x9B\x44\xB3\x38\x69\x64\x8C\x81\xD1\x23\x7E\xAA\x39\x3C\x3B\x0F\x9F\x4A\x7B\x82\xCA\x6B\x6F\xCA\x22\x3E\x31\xD0\xB0\xD0\x2A\x1C\x92\x8A\x8F\xD8\x19\x9C\x47\xE4\x3E\x0C\xB9\xC2\xCD\xBE\x41\x0C\xF8\xA4\x47\x05\xDB\xC1\x17\x30\x38\x3A\x69\xDC\xCD\xC3\x69\x23\xFD\x9A\x0F\x02\xCE\x10\x6A\xCE\xCA\xF8\xB9\x29\xA3\x36\x89\x86\xAE\x0B\xC0\x4F\x63\xB9\x06\x59\x49\x5E\x0E\xC1\x69\xB3\x0A\xF3\x77\x7E\x2E\x9D\x8C\xB3\x27\x98\xD2\x99\x8D\x25\xA7\x1F\x86\xB3\xA6\x54\x70\x38\xFC\x7D\x5D\xE8\x4F\x83\x0C\xD1\x93\xE5\x12\xE4\x54\xDA\x3E\xF2\xAD\x3A\xDE\x3E\x3C\x45\xF0\x28\x0F\x06\xB9\xE1\xDB\x97\x7B\x99\x45\x9E\xDD\xFE\x95\x59\x04\x2F\x75\x3F\xD3\xAE\x89\x99\x86\xAC\x14\xB4\xA8\x84\xFA\xC8\x5D\x3B\x1B\x58\x93\xC1\x17\x94\x55\xC8\x0B\xE3\x82\x79\x84\x9F\xF3\x00\x84\x34\xEE\xDC\x31\xD5\x8F\xF2\xFA\x4F\x96\x4C\x06\xAA\x78\xFB\xDE\x64\xA2\x23\xCD\x1F\x3E\xC5\x8C\xBC\x37\x54\x0E\xBB\x5A\x72\x55\xEF\xC8\x5B\xB5\x72\xF8\x78\xDF\x37\x20\x4C\x57\x91\x73\x92\x73\xAC\x18\x77\x43\x82\x20\x69\xEC\xE9\xAC\x29\x46\xE5\x0B\x4E\xF8\x37\x73\x89\x96\x8A\x1C\x6D\xBD\xEF\xBE\xD8\xB6\xF4\xCA\xC0\xFD\x47\xF0\xAE\x0B\x58\x20\xC5\xC8\x1D\x36\xAE\x97\x8D\x50\x83\x26\x24\x29\xF7\x9D\x3B\x0F\x05\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x6F\x8E\x62\x8B\x93\x43\xB0\xE1\x40\xF6\xA7\xC3\xFD\xF1\x0F\xB8\x0F\x15\x38\xA5\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x6F\x8E\x62\x8B\x93\x43\xB0\xE1\x40\xF6\xA7\xC3\xFD\xF1\x0F\xB8\x0F\x15\x38\xA5\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\xAC\x2C\x29\x41\x7D\xFA\x5C\xF5\x1A\x95\x18\xBF\x2C\xA9\x8A\xA9\x24\x54\x75\xF5\xB8\x40\xAB\xCB\xA8\x24\x51\x2B\x18\x3F\x63\xA9\xAE\x98\x56\x2B\x05\x43\x22\xA3\xB7\xD7\x46\x9E\xC0\x2A\x12\x3D\x8E\x96\x96\x40\xDF\x0C\x33\x8B\x6B\x37\x91\x3A\x95\xBB\x39\x29\x6D\xC0\x02\x6C\x8A\x94\x0B\x07\x02\x4D\x18\x3E\xFB\xFB\x7B\xF5\x76\x3D\x9B\xF6\x5E\x30\x06\x58\x33\x1E\xAA\x78\xD5\xE6\x54\x04\x3A\xB2\x82\x09\x8D\xCE\x16\x33\x59\x45\x28\xF1\xA5\xA3\x97\x0E\x43\x23\xFD\x0B\x20\x80\x90\xFF\xE3\x26\xCF\xB8\x64\x91\xE5\x05\x8F\x13\xA0\x76\x0D\xD7\x37\x0C\x10\x88\x96\xF4\x3E\xBE\x95\xBD\xF1\xC3\x7D\xF0\xA3\xC3\x79\x47\x0B\x5C\x92\x15\x63\xED\x52\x75\x8A\xE7\x46\x69\xCB\x51\x55\x0B\x2A\x4C\xF5\xF2\x64\x4F\xA9\x5C\xFF\x67\x32\x8E\x55\x2D\x32\x82\x1C\x80\x2F\x6A\x91\xF8\xCB\xBC\x7E\x18\xA2\x26\xA8\x2E\xA3\x53\x28\x87\xED\x57\xE5\x65\x7A\x4E\x00\x4A\x5B\x4E\x53\xC9\x62\x36\xBD\xC2\x8E\x5B\xEB\xCC\x6E\x27\x81\x18\x59\x8B\x44\x63\x9F\xD5\x0C\x65\xF4\x29\x65\x7F\x91\x2C\xE5\x7F\x7E\xE8\x89\xCF\x8F\x20\xCB\x6D\x07\x42\x11\x51\x26\x32\x8A\x2E\x3A\x47\x13\xB8\x8D\xBD\x47\x0D\x09\xF0\x16\xA4\xED\x96\x86\x2E\x19\xD8\xBE\x8C\x3A\xE8\x45\x2E\x11\xBA\xAE\x5A\xE7\xB9\xBF\xB1\xCC\x8F\xE0\xA0\xFF\xB8\xB3\xD1\x85\x7B\x79\x66\xA3\x39\xB5\x3B\x66\xD8\x40\xBE\xCF\xB7\x67\x8B\x48\xC9\x19\x25\x55\xFC\xBD\x8D\xCF\x5E\xDA\x4E\xA6\xF2\x69\xCE\xFD\x7F\x4C\x77\xD0\xC1\x46\x35\x98\x5C\x23\x9B\x02\x45\x43\x94\x5A\xDD\xBC\x47\xAD\x22\xFE\xBA\x5E\x2F\x91\x29\x29\x86\x7B\x21\xDE\x6E\x64\xB7\xCB\x0D\x8F\x37\x5B\xA3\x08\x6A\xEB\xF4\xDD\x02\x8F\x50\x03\x02\xB1\xB8\x37\x68\x96\x50\xEB\xB8\x5F\xD4\x28\x8A\xA5\x22\x0C\x8A\x84\xF0\x59\x2E\xD5\x37\xD1\x61\xE5\x42\x73\x58\x2A\x81\xF7\x76\xDB\xE2\xE2\x4D\x0D\x5F\xF6\xB7\xBE\x05\xB4\xAE\x4E\x0D\xDE\x16\x3D\x03\x81\xB3\x26\x5E\x4B\xB8\x4B\x00\xCF\xFF\x8C\x17\xBA\x6C\x60\x2D\x27\x87\x37\x24\xE6\x7A\x60\x2F\xB5\xD3\x83\x04\xAA\x4F\x43\x75\xA2\xC1\x83\xB2\x27\x98\x2B\xB1\x0E\x80\xBA\xC0\x85\x5E\x42\xB9\xDF\xB1\x60\x91\xD3\xEB\x18\x7E\x70\x78\xAE\x76\x83\xBE\x71\x5A\xD0\x90\xE3\xCA\xC1\x16\x25\x67\x4A\xF0\xB6\x7B\xBA\xE1\x9C\xD9"),
   // "Amazon Root CA 2"
   CertificateDer::from_slice(b"\x30\x82\x05\x41\x30\x82\x03\x29\xA0\x03\x02\x01\x02\x02\x13\x06\x6C\x9F\xD2\x96\x35\x86\x9F\x0A\x0F\xE5\x86\x78\xF8\x5B\x26\xBB\x8A\x37\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x1E\x17\x0D\x31\x35\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x30\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xAD\x96\x9F\x2D\x9C\x4A\x4C\x4A\x81\x79\x51\x99\xEC\x8A\xCB\x6B\x60\x51\x13\xBC\x4D\x6D\x06\xFC\xB0\x08\x8D\xDD\x19\x10\x6A\xC7\x26\x0C\x35\xD8\xC0\x6F\x20\x84\xE9\x94\xB1\x9B\x85\x03\xC3\x5B\xDB\x4A\xE8\xC8\xF8\x90\x76\xD9\x5B\x4F\xE3\x4C\xE8\x06\x36\x4D\xCC\x9A\xAC\x3D\x0C\x90\x2B\x92\xD4\x06\x19\x60\xAC\x37\x44\x79\x85\x81\x82\xAD\x5A\x37\xE0\x0D\xCC\x9D\xA6\x4C\x52\x76\xEA\x43\x9D\xB7\x04\xD1\x50\xF6\x55\xE0\xD5\xD2\xA6\x49\x85\xE9\x37\xE9\xCA\x7E\xAE\x5C\x95\x4D\x48\x9A\x3F\xAE\x20\x5A\x6D\x88\x95\xD9\x34\xB8\x52\x1A\x43\x90\xB0\xBF\x6C\x05\xB9\xB6\x78\xB7\xEA\xD0\xE4\x3A\x3C\x12\x53\x62\xFF\x4A\xF2\x7B\xBE\x35\x05\xA9\x12\x34\xE3\xF3\x64\x74\x62\x2C\x3D\x00\x49\x5A\x28\xFE\x32\x44\xBB\x87\xDD\x65\x27\x02\x71\x3B\xDA\x4A\xF7\x1F\xDA\xCD\xF7\x21\x55\x90\x4F\x0F\xEC\xAE\x82\xE1\x9F\x6B\xD9\x45\xD3\xBB\xF0\x5F\x87\xED\x3C\x2C\x39\x86\xDA\x3F\xDE\xEC\x72\x55\xEB\x79\xA3\xAD\xDB\xDD\x7C\xB0\xBA\x1C\xCE\xFC\xDE\x4F\x35\x76\xCF\x0F\xF8\x78\x1F\x6A\x36\x51\x46\x27\x61\x5B\xE9\x9E\xCF\xF0\xA2\x55\x7D\x7C\x25\x8A\x6F\x2F\xB4\xC5\xCF\x84\x2E\x2B\xFD\x0D\x51\x10\x6C\xFB\x5F\x1B\xBC\x1B\x7E\xC5\xAE\x3B\x98\x01\x31\x92\xFF\x0B\x57\xF4\x9A\xB2\xB9\x57\xE9\xAB\xEF\x0D\x76\xD1\xF0\xEE\xF4\xCE\x86\xA7\xE0\x6E\xE9\xB4\x69\xA1\xDF\x69\xF6\x33\xC6\x69\x2E\x97\x13\x9E\xA5\x87\xB0\x57\x10\x81\x37\xC9\x53\xB3\xBB\x7F\xF6\x92\xD1\x9C\xD0\x18\xF4\x92\x6E\xDA\x83\x4F\xA6\x63\x99\x4C\xA5\xFB\x5E\xEF\x21\x64\x7A\x20\x5F\x6C\x64\x85\x15\xCB\x37\xE9\x62\x0C\x0B\x2A\x16\xDC\x01\x2E\x32\xDA\x3E\x4B\xF5\x9E\x3A\xF6\x17\x40\x94\xEF\x9E\x91\x08\x86\xFA\xBE\x63\xA8\x5A\x33\xEC\xCB\x74\x43\x95\xF9\x6C\x69\x52\x36\xC7\x29\x6F\xFC\x55\x03\x5C\x1F\xFB\x9F\xBD\x47\xEB\xE7\x49\x47\x95\x0B\x4E\x89\x22\x09\x49\xE0\xF5\x61\x1E\xF1\xBF\x2E\x8A\x72\x6E\x80\x59\xFF\x57\x3A\xF9\x75\x32\xA3\x4E\x5F\xEC\xED\x28\x62\xD9\x4D\x73\xF2\xCC\x81\x17\x60\xED\xCD\xEB\xDC\xDB\xA7\xCA\xC5\x7E\x02\xBD\xF2\x54\x08\x54\xFD\xB4\x2D\x09\x2C\x17\x54\x4A\x98\xD1\x54\xE1\x51\x67\x08\xD2\xED\x6E\x7E\x6F\x3F\xD2\x2D\x81\x59\x29\x66\xCB\x90\x39\x95\x11\x1E\x74\x27\xFE\xDD\xEB\xAF\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB0\x0C\xF0\x4C\x30\xF4\x05\x58\x02\x48\xFD\x33\xE5\x52\xAF\x4B\x84\xE3\x66\x52\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\xAA\xA8\x80\x8F\x0E\x78\xA3\xE0\xA2\xD4\xCD\xE6\xF5\x98\x7A\x3B\xEA\x00\x03\xB0\x97\x0E\x93\xBC\x5A\xA8\xF6\x2C\x8C\x72\x87\xA9\xB1\xFC\x7F\x73\xFD\x63\x71\x78\xA5\x87\x59\xCF\x30\xE1\x0D\x10\xB2\x13\x5A\x6D\x82\xF5\x6A\xE6\x80\x9F\xA0\x05\x0B\x68\xE4\x47\x6B\xC7\x6A\xDF\xB6\xFD\x77\x32\x72\xE5\x18\xFA\x09\xF4\xA0\x93\x2C\x5D\xD2\x8C\x75\x85\x76\x65\x90\x0C\x03\x79\xB7\x31\x23\x63\xAD\x78\x83\x09\x86\x68\x84\xCA\xFF\xF9\xCF\x26\x9A\x92\x79\xE7\xCD\x4B\xC5\xE7\x61\xA7\x17\xCB\xF3\xA9\x12\x93\x93\x6B\xA7\xE8\x2F\x53\x92\xC4\x60\x58\xB0\xCC\x02\x51\x18\x5B\x85\x8D\x62\x59\x63\xB6\xAD\xB4\xDE\x9A\xFB\x26\xF7\x00\x27\xC0\x5D\x55\x37\x74\x99\xC9\x50\x7F\xE3\x59\x2E\x44\xE3\x2C\x25\xEE\xEC\x4C\x32\x77\xB4\x9F\x1A\xE9\x4B\x5D\x20\xC5\xDA\xFD\x1C\x87\x16\xC6\x43\xE8\xD4\xBB\x26\x9A\x45\x70\x5E\xA9\x0B\x37\x53\xE2\x46\x7B\x27\xFD\xE0\x46\xF2\x89\xB7\xCC\x42\xB6\xCB\x28\x26\x6E\xD9\xA5\xC9\x3A\xC8\x41\x13\x60\xF7\x50\x8C\x15\xAE\xB2\x6D\x1A\x15\x1A\x57\x78\xE6\x92\x2A\xD9\x65\x90\x82\x3F\x6C\x02\xAF\xAE\x12\x3A\x27\x96\x36\x04\xD7\x1D\xA2\x80\x63\xA9\x9B\xF1\xE5\xBA\xB4\x7C\x14\xB0\x4E\xC9\xB1\x1F\x74\x5F\x38\xF6\x51\xEA\x9B\xFA\x2C\xA2\x11\xD4\xA9\x2D\x27\x1A\x45\xB1\xAF\xB2\x4E\x71\x0D\xC0\x58\x46\xD6\x69\x06\xCB\x53\xCB\xB3\xFE\x6B\x41\xCD\x41\x7E\x7D\x4C\x0F\x7C\x72\x79\x7A\x59\xCD\x5E\x4A\x0E\xAC\x9B\xA9\x98\x73\x79\x7C\xB4\xF4\xCC\xB9\xB8\x07\x0C\xB2\x74\x5C\xB8\xC7\x6F\x88\xA1\x90\xA7\xF4\xAA\xF9\xBF\x67\x3A\xF4\x1A\x15\x62\x1E\xB7\x9F\xBE\x3D\xB1\x29\xAF\x67\xA1\x12\xF2\x58\x10\x19\x53\x03\x30\x1B\xB8\x1A\x89\xF6\x9C\xBD\x97\x03\x8E\xA3\x09\xF3\x1D\x8B\x21\xF1\xB4\xDF\xE4\x1C\xD1\x9F\x65\x02\x06\xEA\x5C\xD6\x13\xB3\x84\xEF\xA2\xA5\x5C\x8C\x77\x29\xA7\x68\xC0\x6B\xAE\x40\xD2\xA8\xB4\xEA\xCD\xF0\x8D\x4B\x38\x9C\x19\x9A\x1B\x28\x54\xB8\x89\x90\xEF\xCA\x75\x81\x3E\x1E\xF2\x64\x24\xC7\x18\xAF\x4E\xFF\x47\x9E\x07\xF6\x35\x65\xA4\xD3\x0A\x56\xFF\xF5\x17\x64\x6C\xEF\xA8\x22\x25\x49\x93\xB6\xDF\x00\x17\xDA\x58\x7E\x5D\xEE\xC5\x1B\xB0\xD1\xD1\x5F\x21\x10\xC7\xF9\xF3\xBA\x02\x0A\x27\x07\xC5\xF1\xD6\xC7\xD3\xE0\xFB\x09\x60\x6C"),
   // "SSL.com EV Root Certification Authority ECC"
   CertificateDer::from_slice(b"\x30\x82\x02\x94\x30\x82\x02\x1A\xA0\x03\x02\x01\x02\x02\x08\x2C\x29\x9C\x5B\x16\xED\x05\x95\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x7F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x34\x30\x32\x06\x03\x55\x04\x03\x0C\x2B\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x45\x43\x43\x30\x1E\x17\x0D\x31\x36\x30\x32\x31\x32\x31\x38\x31\x35\x32\x33\x5A\x17\x0D\x34\x31\x30\x32\x31\x32\x31\x38\x31\x35\x32\x33\x5A\x30\x7F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x34\x30\x32\x06\x03\x55\x04\x03\x0C\x2B\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x45\x43\x43\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xAA\x12\x47\x90\x98\x1B\xFB\xEF\xC3\x40\x07\x83\x20\x4E\xF1\x30\x82\xA2\x06\xD1\xF2\x92\x86\x61\xF2\xF6\x21\x68\xCA\x00\xC4\xC7\xEA\x43\x00\x54\x86\xDC\xFD\x1F\xDF\x00\xB8\x41\x62\x5C\xDC\x70\x16\x32\xDE\x1F\x99\xD4\xCC\xC5\x07\xC8\x08\x1F\x61\x16\x07\x51\x3D\x7D\x5C\x07\x53\xE3\x35\x38\x8C\xDF\xCD\x9F\xD9\x2E\x0D\x4A\xB6\x19\x2E\x5A\x70\x5A\x06\xED\xBE\xF0\xA1\xB0\xCA\xD0\x09\x29\xA3\x63\x30\x61\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x5B\xCA\x5E\xE5\xDE\xD2\x81\xAA\xCD\xA8\x2D\x64\x51\xB6\xD9\x72\x9B\x97\xE6\x4F\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x5B\xCA\x5E\xE5\xDE\xD2\x81\xAA\xCD\xA8\x2D\x64\x51\xB6\xD9\x72\x9B\x97\xE6\x4F\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x68\x00\x30\x65\x02\x31\x00\x8A\xE6\x40\x89\x37\xEB\xE9\xD5\x13\xD9\xCA\xD4\x6B\x24\xF3\xB0\x3D\x87\x46\x58\x1A\xEC\xB1\xDF\x6F\xFB\x56\xBA\x70\x6B\xC7\x38\xCC\xE8\xB1\x8C\x4F\x0F\xF7\xF1\x67\x76\x0E\x83\xD0\x1E\x51\x8F\x02\x30\x3D\xF6\x23\x28\x26\x4C\xC6\x60\x87\x93\x26\x9B\xB2\x35\x1E\xBA\xD6\xF7\x3C\xD1\x1C\xCE\xFA\x25\x3C\xA6\x1A\x81\x15\x5B\xF3\x12\x0F\x6C\xEE\x65\x8A\xC9\x87\xA8\xF9\x07\xE0\x62\x9A\x8C\x5C\x4A"),
   // "Telia Root CA v2"
   CertificateDer::from_slice(b"\x30\x82\x05\x74\x30\x82\x03\x5C\xA0\x03\x02\x01\x02\x02\x0F\x01\x67\x5F\x27\xD6\xFE\x7A\xE3\xE4\xAC\xBE\x09\x5B\x05\x9E\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x44\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x49\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x0C\x11\x54\x65\x6C\x69\x61\x20\x46\x69\x6E\x6C\x61\x6E\x64\x20\x4F\x79\x6A\x31\x19\x30\x17\x06\x03\x55\x04\x03\x0C\x10\x54\x65\x6C\x69\x61\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x76\x32\x30\x1E\x17\x0D\x31\x38\x31\x31\x32\x39\x31\x31\x35\x35\x35\x34\x5A\x17\x0D\x34\x33\x31\x31\x32\x39\x31\x31\x35\x35\x35\x34\x5A\x30\x44\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x49\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x0C\x11\x54\x65\x6C\x69\x61\x20\x46\x69\x6E\x6C\x61\x6E\x64\x20\x4F\x79\x6A\x31\x19\x30\x17\x06\x03\x55\x04\x03\x0C\x10\x54\x65\x6C\x69\x61\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x76\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB2\xD0\x3F\x07\xBC\xE2\x7B\xD0\x6B\x99\xF8\xE2\x77\x69\xE7\xCE\x9D\xA4\x03\xBC\x82\x6D\xA1\xFE\x81\x65\x1F\x4C\x27\xAC\x8E\x00\xBA\x16\x7B\xEB\x30\x6A\x00\xC0\xB3\x74\x68\x7E\xB2\xAF\xC7\xD5\x62\xB3\x7A\x3F\x50\xCA\x8C\x36\x44\x24\x63\xD2\x36\xE9\x0C\x85\xF6\x43\x76\xD5\x4C\xA1\x60\x72\x67\xE2\x28\x33\xA5\xCB\x31\xB8\x3A\x22\x23\x34\xB8\x7D\xBD\x56\x22\x40\x9D\xEA\xF4\x7B\x03\xAD\x68\xFC\xB2\x81\x4F\x98\xD0\x74\xEA\x8D\xE5\x7D\xCD\x63\xC3\xA3\xF6\xDE\x92\xC2\x58\x19\xE0\x96\xBB\xC5\xC4\xA9\x3D\xA5\x74\x96\xFE\xAF\xF9\x89\xAA\xBD\x95\x17\x54\xD8\x78\x44\xF1\x0C\x77\x15\x92\xE0\x98\x42\xA7\xA4\xD6\xAA\x20\x92\xCD\xC1\xA0\xB3\x96\xB2\x3A\x84\x42\x8D\x7D\xD5\x95\xE4\xD6\xDB\xE9\x62\xC4\x58\xB3\x79\xC5\x8C\xD3\x35\x33\x83\x9F\x75\xA1\x52\x27\x61\x38\xF1\x59\x3D\x8E\x50\xE0\xBD\x79\x3C\xE7\x6C\x96\xFE\x5E\xD9\x02\x65\xB4\x8E\x5C\xD0\x11\x34\xDF\x5D\xBF\x52\xA7\x81\x00\xC3\x7F\x99\x45\x99\x15\xD5\x17\xC8\x0A\x53\xEC\x63\xF3\x99\x7D\xCC\x69\x12\x86\xC2\x17\xF0\x01\x9E\xBF\x84\xBC\xD1\x52\xCB\x1B\x92\x66\xCE\xA4\x53\xE5\xA1\xBF\xC4\xDB\x09\xD6\xE6\x89\x56\x2B\xC8\xE3\x7C\xDE\xE3\xFF\x89\xE5\x35\x6E\x28\xE8\x6C\x0B\x23\x51\xA9\x25\x05\xEB\x48\xF8\xDD\xB1\xCA\xFA\x6C\x08\x51\xEF\xB7\x18\x6C\x44\xCA\x26\xE1\x73\xC6\x89\x06\x81\xE5\x8A\xAC\xB0\xE2\x29\xC6\xB9\x24\xB3\x6B\x44\x11\xF4\xA5\x43\xC2\x4C\x43\xE5\x70\x36\x8C\xB6\x33\x57\x7A\x95\x2E\x82\xA0\xF4\x5C\x10\xB3\x61\x83\xF6\x02\x05\x86\x2E\x7C\x2D\x6C\xDC\x03\x46\x6E\x35\x93\xD5\x7A\x95\x2F\xDE\x20\xD8\x5B\x7E\x94\x90\x04\x6A\xBA\x59\x3D\x04\x05\x75\x9D\x37\xA2\x0E\x2E\x3D\xEB\xC1\xA4\x52\x83\xFE\xD0\x6B\xD4\x66\x8E\xDC\xC6\xE9\x12\x4E\x1D\x2A\x57\xAA\x10\xBC\x7C\x5E\x82\x7D\xA6\xA6\xC9\xF2\x2D\xB9\xF5\x17\x27\xAD\xD1\x0E\x89\x54\x2B\x95\xFA\xC0\xAD\x1D\x98\x14\x78\x33\x42\x86\x0A\xA9\x73\xB5\xFB\x74\x0D\xB7\x1B\x30\x19\xC4\x5A\x0E\x1C\x27\xB7\xDA\x18\xD0\xFF\x8A\xC8\x05\xBA\xF1\xAA\x1C\xA2\x37\xB7\xE6\x48\xA4\x46\x2C\x94\xEA\xA8\x76\x62\x47\x8B\x10\x53\x07\x48\x57\x6C\xE2\x92\x4D\xB6\xAE\x05\xCB\xDC\xC1\x4A\x5E\x8F\xAC\x3D\x19\x4E\xC2\xED\x60\x75\x2B\xDB\xC1\xCA\x42\xD5\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x72\xAC\xE4\x33\x79\xAA\x45\x87\xF6\xFD\xAC\x1D\x9E\xD6\xC7\x2F\x86\xD8\x24\x39\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x72\xAC\xE4\x33\x79\xAA\x45\x87\xF6\xFD\xAC\x1D\x9E\xD6\xC7\x2F\x86\xD8\x24\x39\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\xA0\x3B\x59\xA7\x09\x94\x3E\x36\x84\xD2\x7E\x2F\x39\xA5\x96\x97\xFA\x11\xAD\xFC\x67\xF3\x71\x09\xF2\xB2\x89\x84\x67\x44\xAF\xB9\xEF\xED\x96\xEC\x9C\x64\xDB\x32\x30\x6F\x67\x9A\xAC\x7E\x5F\xB2\xAB\x01\x36\x7E\x81\xFA\xE4\x84\x5E\xD2\xAC\x36\xE0\x6B\x62\xC5\x7D\x4B\x0E\x82\x6D\xD2\x76\x62\xD1\xFE\x97\xF8\x9F\x30\x7C\x18\xF9\xB4\x52\x77\x82\x1D\x76\xDB\xD3\x1D\xA9\xF0\xC1\x9A\x00\xBD\x6D\x75\xD8\x7D\xE7\xFA\xC7\x38\xA3\x9C\x70\xE8\x46\x79\x03\xAF\x2E\x74\xDB\x75\xF8\x6E\x53\x0C\x03\xC8\x99\x1A\x89\x35\x19\x3C\xD3\xC9\x54\x7C\xA8\xF0\x2C\xE6\x6E\x07\x79\x6F\x6A\xE1\xE6\xEA\x91\x82\x69\x0A\x1D\xC3\x7E\x59\xA2\x9E\x6B\x46\x15\x98\x5B\xD3\xAF\x46\x1D\x62\xC8\xCE\x80\x52\x49\x11\x3F\xC9\x04\x12\xC3\x13\x7C\x3F\x3B\x8A\x96\xDB\x3C\xA0\x1E\x0A\xB4\x8B\x54\xB2\x24\x67\x0D\xEF\x82\xCB\xBE\x3C\x7D\xD1\xE2\x7F\xAE\x16\xD6\x56\x58\xB9\xDA\x20\xB1\x83\x15\xA1\xEF\x8A\x4D\x32\x6F\x41\x2F\x13\x52\x82\x94\xD7\x1A\xC1\x78\xA2\x51\xDD\x2B\x70\x6D\xB7\x1A\xF9\xF7\xB0\xE0\x67\x97\x56\xDB\x7C\x61\x53\x09\x03\x28\x02\x40\xC7\xB3\xD8\xFD\x9C\x70\x6A\xC6\x28\xC3\x85\xE9\xE2\xED\x1A\x93\xA0\xDE\x4B\x98\xA2\x84\x3E\x05\x77\x01\x96\x3D\xFB\xB4\x20\x0F\x9C\x72\x02\x7A\x12\x2F\xD5\xA3\xBA\x51\x78\xAF\x2A\x2B\x44\x65\x4E\xB5\xFD\x0A\xE8\xC1\xCD\x79\x87\x61\x2B\xDE\x80\x57\x45\xBF\x67\xF1\x9B\x91\x5E\xA5\xA4\xEC\x59\x48\x10\x0D\x38\xC7\xB0\xFA\xC3\x44\x6D\x04\xF5\x78\x50\x1C\x92\x96\x5B\xDA\xF5\xB8\x2E\xBA\x5B\xCF\xE5\xF0\x6A\x9D\x4B\x2F\x58\x73\x2D\x4F\x2D\xC4\x1C\x3E\xF4\xB3\x3F\xAB\x15\x0E\x3B\x19\x41\x8A\xA4\xC1\x57\x12\x66\x71\x4C\xFA\x53\xE3\x57\xEB\x62\x95\x09\x9E\x54\xDD\xD1\xC2\x3C\x57\x3C\xBD\x38\xAD\x98\x64\xB7\xB8\x03\x9A\x53\x56\x60\x5D\xB3\xD8\x42\x1B\x5C\x4B\x12\x8A\x1C\xEB\xEB\x7D\xC6\x7A\x69\xC7\x27\x7F\xA4\xF8\x8B\xF2\xE4\x94\x66\x87\x4B\xE9\x94\x07\x09\x12\x79\x8A\xB2\xEB\x74\x04\xDC\xCE\xF4\x44\x59\xE0\x16\xCA\xC5\x2C\x58\xD7\x3C\x7B\xCF\x62\x86\x6A\x50\x7D\x35\x36\x66\xA7\xFB\x37\xE7\x28\xC7\xD8\xD0\xAD\xA5\x69\x94\x8F\xE8\xC1\xDF\x24\xF8\x1B\x07\x31\x87\x81\xD8\x5D\xF6\xE8\x28\xD8\x4A\x52\x80\xAC\x13\xEE\x50\x14\x1E\x98\xC7"),
   // "Izenpe.com"
   CertificateDer::from_slice(b"\x30\x82\x05\xF1\x30\x82\x03\xD9\xA0\x03\x02\x01\x02\x02\x10\x00\xB0\xB7\x5A\x16\x48\x5F\xBF\xE1\xCB\xF5\x8B\xD7\x19\xE6\x7D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x38\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x0C\x0B\x49\x5A\x45\x4E\x50\x45\x20\x53\x2E\x41\x2E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x0C\x0A\x49\x7A\x65\x6E\x70\x65\x2E\x63\x6F\x6D\x30\x1E\x17\x0D\x30\x37\x31\x32\x31\x33\x31\x33\x30\x38\x32\x38\x5A\x17\x0D\x33\x37\x31\x32\x31\x33\x30\x38\x32\x37\x32\x35\x5A\x30\x38\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x0C\x0B\x49\x5A\x45\x4E\x50\x45\x20\x53\x2E\x41\x2E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x0C\x0A\x49\x7A\x65\x6E\x70\x65\x2E\x63\x6F\x6D\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC9\xD3\x7A\xCA\x0F\x1E\xAC\xA7\x86\xE8\x16\x65\x6A\xB1\xC2\x1B\x45\x32\x71\x95\xD9\xFE\x10\x5B\xCC\xAF\xE7\xA5\x79\x01\x8F\x89\xC3\xCA\xF2\x55\x71\xF7\x77\xBE\x77\x94\xF3\x72\xA4\x2C\x44\xD8\x9E\x92\x9B\x14\x3A\xA1\xE7\x24\x90\x0A\x0A\x56\x8E\xC5\xD8\x26\x94\xE1\xD9\x48\xE1\x2D\x3E\xDA\x0A\x72\xDD\xA3\x99\x15\xDA\x81\xA2\x87\xF4\x7B\x6E\x26\x77\x89\x58\xAD\xD6\xEB\x0C\xB2\x41\x7A\x73\x6E\x6D\xDB\x7A\x78\x41\xE9\x08\x88\x12\x7E\x87\x2E\x66\x11\x63\x6C\x54\xFB\x3C\x9D\x72\xC0\xBC\x2E\xFF\xC2\xB7\xDD\x0D\x76\xE3\x3A\xD7\xF7\xB4\x68\xBE\xA2\xF5\xE3\x81\x6E\xC1\x46\x6F\x5D\x8D\xE0\x4D\xC6\x54\x55\x89\x1A\x33\x31\x0A\xB1\x57\xB9\xA3\x8A\x98\xC3\xEC\x3B\x34\xC5\x95\x41\x69\x7E\x75\xC2\x3C\x20\xC5\x61\xBA\x51\x47\xA0\x20\x90\x93\xA1\x90\x4B\xF3\x4E\x7C\x85\x45\x54\x9A\xD1\x05\x26\x41\xB0\xB5\x4D\x1D\x33\xBE\xC4\x03\xC8\x25\x7C\xC1\x70\xDB\x3B\xF4\x09\x2D\x54\x27\x48\xAC\x2F\xE1\xC4\xAC\x3E\xC8\xCB\x92\x4C\x53\x39\x37\x23\xEC\xD3\x01\xF9\xE0\x09\x44\x4D\x4D\x64\xC0\xE1\x0D\x5A\x87\x22\xBC\xAD\x1B\xA3\xFE\x26\xB5\x15\xF3\xA7\xFC\x84\x19\xE9\xEC\xA1\x88\xB4\x44\x69\x84\x83\xF3\x89\xD1\x74\x06\xA9\xCC\x0B\xD6\xC2\xDE\x27\x85\x50\x26\xCA\x17\xB8\xC9\x7A\x87\x56\x2C\x1A\x01\x1E\x6C\xBE\x13\xAD\x10\xAC\xB5\x24\xF5\x38\x91\xA1\xD6\x4B\xDA\xF1\xBB\xD2\xDE\x47\xB5\xF1\xBC\x81\xF6\x59\x6B\xCF\x19\x53\xE9\x8D\x15\xCB\x4A\xCB\xA9\x6F\x44\xE5\x1B\x41\xCF\xE1\x86\xA7\xCA\xD0\x6A\x9F\xBC\x4C\x8D\x06\x33\x5A\xA2\x85\xE5\x90\x35\xA0\x62\x5C\x16\x4E\xF0\xE3\xA2\xFA\x03\x1A\xB4\x2C\x71\xB3\x58\x2C\xDE\x7B\x0B\xDB\x1A\x0F\xEB\xDE\x21\x1F\x06\x77\x06\x03\xB0\xC9\xEF\x99\xFC\xC0\xB9\x4F\x0B\x86\x28\xFE\xD2\xB9\xEA\xE3\xDA\xA5\xC3\x47\x69\x12\xE0\xDB\xF0\xF6\x19\x8B\xED\x7B\x70\xD7\x02\xD6\xED\x87\x18\x28\x2C\x04\x24\x4C\x77\xE4\x48\x8A\x1A\xC6\x3B\x9A\xD4\x0F\xCA\xFA\x75\xD2\x01\x40\x5A\x8D\x79\xBF\x8B\xCF\x4B\xCF\xAA\x16\xC1\x95\xE4\xAD\x4C\x8A\x3E\x17\x91\xD4\xB1\x62\xE5\x82\xE5\x80\x04\xA4\x03\x7E\x8D\xBF\xDA\x7F\xA2\x0F\x97\x4F\x0C\xD3\x0D\xFB\xD7\xD1\xE5\x72\x7E\x1C\xC8\x77\xFF\x5B\x9A\x0F\xB7\xAE\x05\x46\xE5\xF1\xA8\x16\xEC\x47\xA4\x17\x02\x03\x01\x00\x01\xA3\x81\xF6\x30\x81\xF3\x30\x81\xB0\x06\x03\x55\x1D\x11\x04\x81\xA8\x30\x81\xA5\x81\x0F\x69\x6E\x66\x6F\x40\x69\x7A\x65\x6E\x70\x65\x2E\x63\x6F\x6D\xA4\x81\x91\x30\x81\x8E\x31\x47\x30\x45\x06\x03\x55\x04\x0A\x0C\x3E\x49\x5A\x45\x4E\x50\x45\x20\x53\x2E\x41\x2E\x20\x2D\x20\x43\x49\x46\x20\x41\x30\x31\x33\x33\x37\x32\x36\x30\x2D\x52\x4D\x65\x72\x63\x2E\x56\x69\x74\x6F\x72\x69\x61\x2D\x47\x61\x73\x74\x65\x69\x7A\x20\x54\x31\x30\x35\x35\x20\x46\x36\x32\x20\x53\x38\x31\x43\x30\x41\x06\x03\x55\x04\x09\x0C\x3A\x41\x76\x64\x61\x20\x64\x65\x6C\x20\x4D\x65\x64\x69\x74\x65\x72\x72\x61\x6E\x65\x6F\x20\x45\x74\x6F\x72\x62\x69\x64\x65\x61\x20\x31\x34\x20\x2D\x20\x30\x31\x30\x31\x30\x20\x56\x69\x74\x6F\x72\x69\x61\x2D\x47\x61\x73\x74\x65\x69\x7A\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x1D\x1C\x65\x0E\xA8\xF2\x25\x7B\xB4\x91\xCF\xE4\xB1\xB1\xE6\xBD\x55\x74\x6C\x05\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x78\xA6\x0C\x16\x4A\x9F\x4C\x88\x3A\xC0\xCB\x0E\xA5\x16\x7D\x9F\xB9\x48\x5F\x18\x8F\x0D\x62\x36\xF6\xCD\x19\x6B\xAC\xAB\xD5\xF6\x91\x7D\xAE\x71\xF3\x3F\xB3\x0E\x78\x85\x9B\x95\xA4\x27\x21\x47\x42\x4A\x7C\x48\x3A\xF5\x45\x7C\xB3\x0C\x8E\x51\x78\xAC\x95\x13\xDE\xC6\xFD\x7D\xB8\x1A\x90\x4C\xAB\x92\x03\xC7\xED\x42\x01\xCE\x0F\xD8\xB1\xFA\xA2\x92\xE1\x60\x6D\xAE\x7A\x6B\x09\xAA\xC6\x29\xEE\x68\x49\x67\x30\x80\x24\x7A\x31\x16\x39\x5B\x7E\xF1\x1C\x2E\xDD\x6C\x09\xAD\xF2\x31\xC1\x82\x4E\xB9\xBB\xF9\xBE\xBF\x2A\x85\x3F\xC0\x40\xA3\x3A\x59\xFC\x59\x4B\x3C\x28\x24\xDB\xB4\x15\x75\xAE\x0D\x88\xBA\x2E\x73\xC0\xBD\x58\x87\xE5\x42\xF2\xEB\x5E\xEE\x1E\x30\x22\x99\xCB\x37\xD1\xC4\x21\x6C\x81\xEC\xBE\x6D\x26\xE6\x1C\xE4\x42\x20\x9E\x47\xB0\xAC\x83\x59\x70\x2C\x35\xD6\xAF\x36\x34\xB4\xCD\x3B\xF8\x32\xA8\xEF\xE3\x78\x89\xFB\x8D\x45\x2C\xDA\x9C\xB8\x7E\x40\x1C\x61\xE7\x3E\xA2\x92\x2C\x4B\xF2\xCD\xFA\x98\xB6\x29\xFF\xF3\xF2\x7B\xA9\x1F\x2E\xA0\x93\x57\x2B\xDE\x85\x03\xF9\x69\x37\xCB\x9E\x78\x6A\x05\xB4\xC5\x31\x78\x89\xEC\x7A\xA7\x85\xE1\xB9\x7B\x3C\xDE\xBE\x1E\x79\x84\xCE\x9F\x70\x0E\x59\xC2\x35\x2E\x90\x2A\x31\xD9\xE4\x45\x7A\x41\xA4\x2E\x13\x9B\x34\x0E\x66\x7B\x49\xAB\x64\x97\xD0\x46\xC3\x79\x9D\x72\x50\x63\xA6\x98\x5B\x06\xBD\x48\x6D\xD8\x39\x83\x70\xE8\x35\xF0\x05\xD1\xAA\xBC\xE3\xDB\xC8\x02\xEA\x7C\xFD\x82\xDA\xC2\x5B\x52\x35\xAE\x98\x3A\xAD\xBA\x35\x93\x23\xA7\x1F\x48\xDD\x35\x46\x98\xB2\x10\x68\xE4\xA5\x31\xC2\x0A\x58\x2E\x19\x81\x10\xC9\x50\x75\xFC\xEA\x5A\x16\xCE\x11\xD7\xEE\xEF\x50\x88\x2D\x61\xFF\x3F\x42\x73\x05\x94\x43\xD5\x8E\x3C\x4E\x01\x3A\x19\xA5\x1F\x46\x4E\x77\xD0\x5D\xE5\x81\x22\x21\x87\xFE\x94\x7D\x84\xD8\x93\xAD\xD6\x68\x43\x48\xB2\xDB\xEB\x73\x24\xE7\x91\x7F\x54\xA4\xB6\x80\x3E\x9D\xA3\x3C\x4C\x72\xC2\x57\xC4\xA0\xD4\xCC\x38\x27\xCE\xD5\x06\x9E\xA2\x48\xD9\xE9\x9F\xCE\x82\x70\x36\x93\x9A\x3B\xDF\x96\x21\xE3\x59\xB7\x0C\xDA\x91\x37\xF0\xFD\x59\x5A\xB3\x99\xC8\x69\x6C\x43\x26\x01\x35\x63\x60\x55\x89\x03\x3A\x75\xD8\xBA\x4A\xD9\x54\xFF\xEE\xDE\x80\xD8\x2D\xD1\x38\xD5\x5E\x2D\x0B\x98\x7D\x3E\x6C\xDB\xFC\x26\x88\xC7"),
   // "GlobalSign"
   CertificateDer::from_slice(b"\x30\x82\x05\x83\x30\x82\x03\x6B\xA0\x03\x02\x01\x02\x02\x0E\x45\xE6\xBB\x03\x83\x33\xC3\x85\x65\x48\xE6\xFF\x45\x51\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x4C\x31\x20\x30\x1E\x06\x03\x55\x04\x0B\x13\x17\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x36\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x1E\x17\x0D\x31\x34\x31\x32\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x34\x31\x32\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x30\x4C\x31\x20\x30\x1E\x06\x03\x55\x04\x0B\x13\x17\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x36\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x95\x07\xE8\x73\xCA\x66\xF9\xEC\x14\xCA\x7B\x3C\xF7\x0D\x08\xF1\xB4\x45\x0B\x2C\x82\xB4\x48\xC6\xEB\x5B\x3C\xAE\x83\xB8\x41\x92\x33\x14\xA4\x6F\x7F\xE9\x2A\xCC\xC6\xB0\x88\x6B\xC5\xB6\x89\xD1\xC6\xB2\xFF\x14\xCE\x51\x14\x21\xEC\x4A\xDD\x1B\x5A\xC6\xD6\x87\xEE\x4D\x3A\x15\x06\xED\x64\x66\x0B\x92\x80\xCA\x44\xDE\x73\x94\x4E\xF3\xA7\x89\x7F\x4F\x78\x63\x08\xC8\x12\x50\x6D\x42\x66\x2F\x4D\xB9\x79\x28\x4D\x52\x1A\x8A\x1A\x80\xB7\x19\x81\x0E\x7E\xC4\x8A\xBC\x64\x4C\x21\x1C\x43\x68\xD7\x3D\x3C\x8A\xC5\xB2\x66\xD5\x90\x9A\xB7\x31\x06\xC5\xBE\xE2\x6D\x32\x06\xA6\x1E\xF9\xB9\xEB\xAA\xA3\xB8\xBF\xBE\x82\x63\x50\xD0\xF0\x18\x89\xDF\xE4\x0F\x79\xF5\xEA\xA2\x1F\x2A\xD2\x70\x2E\x7B\xE7\xBC\x93\xBB\x6D\x53\xE2\x48\x7C\x8C\x10\x07\x38\xFF\x66\xB2\x77\x61\x7E\xE0\xEA\x8C\x3C\xAA\xB4\xA4\xF6\xF3\x95\x4A\x12\x07\x6D\xFD\x8C\xB2\x89\xCF\xD0\xA0\x61\x77\xC8\x58\x74\xB0\xD4\x23\x3A\xF7\x5D\x3A\xCA\xA2\xDB\x9D\x09\xDE\x5D\x44\x2D\x90\xF1\x81\xCD\x57\x92\xFA\x7E\xBC\x50\x04\x63\x34\xDF\x6B\x93\x18\xBE\x6B\x36\xB2\x39\xE4\xAC\x24\x36\xB7\xF0\xEF\xB6\x1C\x13\x57\x93\xB6\xDE\xB2\xF8\xE2\x85\xB7\x73\xA2\xB8\x35\xAA\x45\xF2\xE0\x9D\x36\xA1\x6F\x54\x8A\xF1\x72\x56\x6E\x2E\x88\xC5\x51\x42\x44\x15\x94\xEE\xA3\xC5\x38\x96\x9B\x4E\x4E\x5A\x0B\x47\xF3\x06\x36\x49\x77\x30\xBC\x71\x37\xE5\xA6\xEC\x21\x08\x75\xFC\xE6\x61\x16\x3F\x77\xD5\xD9\x91\x97\x84\x0A\x6C\xD4\x02\x4D\x74\xC0\x14\xED\xFD\x39\xFB\x83\xF2\x5E\x14\xA1\x04\xB0\x0B\xE9\xFE\xEE\x8F\xE1\x6E\x0B\xB2\x08\xB3\x61\x66\x09\x6A\xB1\x06\x3A\x65\x96\x59\xC0\xF0\x35\xFD\xC9\xDA\x28\x8D\x1A\x11\x87\x70\x81\x0A\xA8\x9A\x75\x1D\x9E\x3A\x86\x05\x00\x9E\xDB\x80\xD6\x25\xF9\xDC\x05\x9E\x27\x59\x4C\x76\x39\x5B\xEA\xF9\xA5\xA1\xD8\x83\x0F\xD1\xFF\xDF\x30\x11\xF9\x85\xCF\x33\x48\xF5\xCA\x6D\x64\x14\x2C\x7A\x58\x4F\xD3\x4B\x08\x49\xC5\x95\x64\x1A\x63\x0E\x79\x3D\xF5\xB3\x8C\xCA\x58\xAD\x9C\x42\x45\x79\x6E\x0E\x87\x19\x5C\x54\xB1\x65\xB6\xBF\x8C\x9B\xDC\x13\xE9\x0D\x6F\xB8\x2E\xDC\x67\x6E\xC9\x8B\x11\xB5\x84\x14\x8A\x00\x19\x70\x83\x79\x91\x97\x91\xD4\x1A\x27\xBF\x37\x1E\x32\x07\xD8\x14\x63\x3C\x28\x4C\xAF\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xAE\x6C\x05\xA3\x93\x13\xE2\xA2\xE7\xE2\xD7\x1C\xD6\xC7\xF0\x7F\xC8\x67\x53\xA0\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xAE\x6C\x05\xA3\x93\x13\xE2\xA2\xE7\xE2\xD7\x1C\xD6\xC7\xF0\x7F\xC8\x67\x53\xA0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x83\x25\xED\xE8\xD1\xFD\x95\x52\xCD\x9E\xC0\x04\xA0\x91\x69\xE6\x5C\xD0\x84\xDE\xDC\xAD\xA2\x4F\xE8\x47\x78\xD6\x65\x98\xA9\x5B\xA8\x3C\x87\x7C\x02\x8A\xD1\x6E\xB7\x16\x73\xE6\x5F\xC0\x54\x98\xD5\x74\xBE\xC1\xCD\xE2\x11\x91\xAD\x23\x18\x3D\xDD\xE1\x72\x44\x96\xB4\x95\x5E\xC0\x7B\x8E\x99\x78\x16\x43\x13\x56\x57\xB3\xA2\xB3\x3B\xB5\x77\xDC\x40\x72\xAC\xA3\xEB\x9B\x35\x3E\xB1\x08\x21\xA1\xE7\xC4\x43\x37\x79\x32\xBE\xB5\xE7\x9C\x2C\x4C\xBC\x43\x29\x99\x8E\x30\xD3\xAC\x21\xE0\xE3\x1D\xFA\xD8\x07\x33\x76\x54\x00\x22\x2A\xB9\x4D\x20\x2E\x70\x68\xDA\xE5\x53\xFC\x83\x5C\xD3\x9D\xF2\xFF\x44\x0C\x44\x66\xF2\xD2\xE3\xBD\x46\x00\x1A\x6D\x02\xBA\x25\x5D\x8D\xA1\x31\x51\xDD\x54\x46\x1C\x4D\xDB\x99\x96\xEF\x1A\x1C\x04\x5C\xA6\x15\xEF\x78\xE0\x79\xFE\x5D\xDB\x3E\xAA\x4C\x55\xFD\x9A\x15\xA9\x6F\xE1\xA6\xFB\xDF\x70\x30\xE9\xC3\xEE\x42\x46\xED\xC2\x93\x05\x89\xFA\x7D\x63\x7B\x3F\xD0\x71\x81\x7C\x00\xE8\x98\xAE\x0E\x78\x34\xC3\x25\xFB\xAF\x0A\x9F\x20\x6B\xDD\x3B\x13\x8F\x12\x8C\xE2\x41\x1A\x48\x7A\x73\xA0\x77\x69\xC7\xB6\x5C\x7F\x82\xC8\x1E\xFE\x58\x1B\x28\x2B\xA8\x6C\xAD\x5E\x6D\xC0\x05\xD2\x7B\xB7\xEB\x80\xFE\x25\x37\xFE\x02\x9B\x68\xAC\x42\x5D\xC3\xEE\xF5\xCC\xDC\xF0\x50\x75\xD2\x36\x69\x9C\xE6\x7B\x04\xDF\x6E\x06\x69\xB6\xDE\x0A\x09\x48\x59\x87\xEB\x7B\x14\x60\x7A\x64\xAA\x69\x43\xEF\x91\xC7\x4C\xEC\x18\xDD\x6C\xEF\x53\x2D\x8C\x99\xE1\x5E\xF2\x72\x3E\xCF\x54\xC8\xBD\x67\xEC\xA4\x0F\x4C\x45\xFF\xD3\xB9\x30\x23\x07\x4C\x8F\x10\xBF\x86\x96\xD9\x99\x5A\xB4\x99\x57\x1C\xA4\xCC\xBB\x15\x89\x53\xBA\x2C\x05\x0F\xE4\xC4\x9E\x19\xB1\x18\x34\xD5\x4C\x9D\xBA\xED\xF7\x1F\xAF\x24\x95\x04\x78\xA8\x03\xBB\xEE\x81\xE5\xDA\x5F\x7C\x8B\x4A\xA1\x90\x74\x25\xA7\xB3\x3E\x4B\xC8\x2C\x56\xBD\xC7\xC8\xEF\x38\xE2\x5C\x92\xF0\x79\xF7\x9C\x84\xBA\x74\x2D\x61\x01\x20\x7E\x7E\xD1\xF2\x4F\x07\x59\x5F\x8B\x2D\x43\x52\xEB\x46\x0C\x94\xE1\xF5\x66\x47\x79\x77\xD5\x54\x5B\x1F\xAD\x24\x37\xCB\x45\x5A\x4E\xA0\x44\x48\xC8\xD8\xB0\x99\xC5\x15\x84\x09\xF6\xD6\x49\x49\xC0\x65\xB8\xE6\x1A\x71\x6E\xA0\xA8\xF1\x82\xE8\x45\x3E\x6C\xD6\x02\xD7\x0A\x67\x83\x05\x5A\xC9\xA4\x10"),
   // "Starfield Root Certificate Authority - G2"
   CertificateDer::from_slice(b"\x30\x82\x03\xDD\x30\x82\x02\xC5\xA0\x03\x02\x01\x02\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x8F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x10\x30\x0E\x06\x03\x55\x04\x08\x13\x07\x41\x72\x69\x7A\x6F\x6E\x61\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x53\x63\x6F\x74\x74\x73\x64\x61\x6C\x65\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x32\x30\x30\x06\x03\x55\x04\x03\x13\x29\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x2D\x20\x47\x32\x30\x1E\x17\x0D\x30\x39\x30\x39\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x37\x31\x32\x33\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x8F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x10\x30\x0E\x06\x03\x55\x04\x08\x13\x07\x41\x72\x69\x7A\x6F\x6E\x61\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x53\x63\x6F\x74\x74\x73\x64\x61\x6C\x65\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x32\x30\x30\x06\x03\x55\x04\x03\x13\x29\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x2D\x20\x47\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xBD\xED\xC1\x03\xFC\xF6\x8F\xFC\x02\xB1\x6F\x5B\x9F\x48\xD9\x9D\x79\xE2\xA2\xB7\x03\x61\x56\x18\xC3\x47\xB6\xD7\xCA\x3D\x35\x2E\x89\x43\xF7\xA1\x69\x9B\xDE\x8A\x1A\xFD\x13\x20\x9C\xB4\x49\x77\x32\x29\x56\xFD\xB9\xEC\x8C\xDD\x22\xFA\x72\xDC\x27\x61\x97\xEE\xF6\x5A\x84\xEC\x6E\x19\xB9\x89\x2C\xDC\x84\x5B\xD5\x74\xFB\x6B\x5F\xC5\x89\xA5\x10\x52\x89\x46\x55\xF4\xB8\x75\x1C\xE6\x7F\xE4\x54\xAE\x4B\xF8\x55\x72\x57\x02\x19\xF8\x17\x71\x59\xEB\x1E\x28\x07\x74\xC5\x9D\x48\xBE\x6C\xB4\xF4\xA4\xB0\xF3\x64\x37\x79\x92\xC0\xEC\x46\x5E\x7F\xE1\x6D\x53\x4C\x62\xAF\xCD\x1F\x0B\x63\xBB\x3A\x9D\xFB\xFC\x79\x00\x98\x61\x74\xCF\x26\x82\x40\x63\xF3\xB2\x72\x6A\x19\x0D\x99\xCA\xD4\x0E\x75\xCC\x37\xFB\x8B\x89\xC1\x59\xF1\x62\x7F\x5F\xB3\x5F\x65\x30\xF8\xA7\xB7\x4D\x76\x5A\x1E\x76\x5E\x34\xC0\xE8\x96\x56\x99\x8A\xB3\xF0\x7F\xA4\xCD\xBD\xDC\x32\x31\x7C\x91\xCF\xE0\x5F\x11\xF8\x6B\xAA\x49\x5C\xD1\x99\x94\xD1\xA2\xE3\x63\x5B\x09\x76\xB5\x56\x62\xE1\x4B\x74\x1D\x96\xD4\x26\xD4\x08\x04\x59\xD0\x98\x0E\x0E\xE6\xDE\xFC\xC3\xEC\x1F\x90\xF1\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x7C\x0C\x32\x1F\xA7\xD9\x30\x7F\xC4\x7D\x68\xA3\x62\xA8\xA1\xCE\xAB\x07\x5B\x27\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x11\x59\xFA\x25\x4F\x03\x6F\x94\x99\x3B\x9A\x1F\x82\x85\x39\xD4\x76\x05\x94\x5E\xE1\x28\x93\x6D\x62\x5D\x09\xC2\xA0\xA8\xD4\xB0\x75\x38\xF1\x34\x6A\x9D\xE4\x9F\x8A\x86\x26\x51\xE6\x2C\xD1\xC6\x2D\x6E\x95\x20\x4A\x92\x01\xEC\xB8\x8A\x67\x7B\x31\xE2\x67\x2E\x8C\x95\x03\x26\x2E\x43\x9D\x4A\x31\xF6\x0E\xB5\x0C\xBB\xB7\xE2\x37\x7F\x22\xBA\x00\xA3\x0E\x7B\x52\xFB\x6B\xBB\x3B\xC4\xD3\x79\x51\x4E\xCD\x90\xF4\x67\x07\x19\xC8\x3C\x46\x7A\x0D\x01\x7D\xC5\x58\xE7\x6D\xE6\x85\x30\x17\x9A\x24\xC4\x10\xE0\x04\xF7\xE0\xF2\x7F\xD4\xAA\x0A\xFF\x42\x1D\x37\xED\x94\xE5\x64\x59\x12\x20\x77\x38\xD3\x32\x3E\x38\x81\x75\x96\x73\xFA\x68\x8F\xB1\xCB\xCE\x1F\xC5\xEC\xFA\x9C\x7E\xCF\x7E\xB1\xF1\x07\x2D\xB6\xFC\xBF\xCA\xA4\xBF\xD0\x97\x05\x4A\xBC\xEA\x18\x28\x02\x90\xBD\x54\x78\x09\x21\x71\xD3\xD1\x7D\x1D\xD9\x16\xB0\xA9\x61\x3D\xD0\x0A\x00\x22\xFC\xC7\x7B\xCB\x09\x64\x45\x0B\x3B\x40\x81\xF7\x7D\x7C\x32\xF5\x98\xCA\x58\x8E\x7D\x2A\xEE\x90\x59\x73\x64\xF9\x36\x74\x5E\x25\xA1\xF5\x66\x05\x2E\x7F\x39\x15\xA9\x2A\xFB\x50\x8B\x8E\x85\x69\xF4"),
   // "TunTrust Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\xB3\x30\x82\x03\x9B\xA0\x03\x02\x01\x02\x02\x14\x13\x02\xD5\xE2\x40\x4C\x92\x46\x86\x16\x67\x5D\xB4\xBB\xBB\xB2\x6B\x3E\xFC\x13\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x4E\x31\x37\x30\x35\x06\x03\x55\x04\x0A\x0C\x2E\x41\x67\x65\x6E\x63\x65\x20\x4E\x61\x74\x69\x6F\x6E\x61\x6C\x65\x20\x64\x65\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x45\x6C\x65\x63\x74\x72\x6F\x6E\x69\x71\x75\x65\x31\x19\x30\x17\x06\x03\x55\x04\x03\x0C\x10\x54\x75\x6E\x54\x72\x75\x73\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x39\x30\x34\x32\x36\x30\x38\x35\x37\x35\x36\x5A\x17\x0D\x34\x34\x30\x34\x32\x36\x30\x38\x35\x37\x35\x36\x5A\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x4E\x31\x37\x30\x35\x06\x03\x55\x04\x0A\x0C\x2E\x41\x67\x65\x6E\x63\x65\x20\x4E\x61\x74\x69\x6F\x6E\x61\x6C\x65\x20\x64\x65\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x45\x6C\x65\x63\x74\x72\x6F\x6E\x69\x71\x75\x65\x31\x19\x30\x17\x06\x03\x55\x04\x03\x0C\x10\x54\x75\x6E\x54\x72\x75\x73\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC3\xCD\xD3\xFC\xBD\x04\x53\xDD\x0C\x20\x3A\xD5\x88\x2E\x05\x4B\x41\xF5\x83\x82\x7E\xF7\x59\x9F\x9E\x9E\x63\xE8\x73\xDA\xF6\x06\xA9\x4F\x1F\xB4\xF9\x0B\x1F\x39\x8C\x9A\x20\xD0\x7E\x06\xD4\xEC\x34\xD9\x86\xBC\x75\x5B\x87\x88\xF0\xD2\xD9\xD4\xA3\x0A\xB2\x6C\x1B\xEB\x49\x2C\x3E\xAC\x5D\xD8\x94\x03\xA0\xEC\x34\xE5\x30\xC4\x35\x7D\xFB\x26\x4D\x1B\x6E\x30\x54\xD8\xF5\x80\x45\x9C\x39\xAD\x9C\xC9\x25\x04\x4D\x9A\x90\x3E\x4E\x40\x6E\x8A\x6B\xCD\x29\x67\xC6\xCC\x2D\xE0\x74\xE8\x05\x57\x0A\x48\x50\xFA\x7A\x43\xDA\x7E\xEC\x5B\x9A\x0E\x62\x76\xFE\xEA\x9D\x1D\x85\x72\xEC\x11\xBB\x35\xE8\x1F\x27\xBF\xC1\xA1\xC7\xBB\x48\x16\xDD\x56\xD7\xCC\x4E\xA0\xE1\xB9\xAC\xDB\xD5\x83\x19\x1A\x85\xD1\x94\x97\xD7\xCA\xA3\x65\x0B\xF3\x38\xF9\x02\xAE\xDD\xF6\x67\xCF\xC9\x3F\xF5\x8A\x2C\x47\x1A\x99\x6F\x05\x0D\xFD\xD0\x1D\x82\x31\xFC\x29\xCC\x00\x58\x97\x91\x4C\x80\x00\x1C\x33\x85\x96\x2F\xCB\x41\xC2\x8B\x10\x84\xC3\x09\x24\x89\x1F\xB5\x0F\xD9\xD9\x77\x47\x18\x92\x94\x60\x5C\xC7\x99\x03\x3C\xFE\xF7\x95\xA7\x7D\x50\xA1\x80\xC2\xA9\x83\xAD\x58\x96\x55\x21\xDB\x86\x59\xD4\xAF\xC6\xBC\xDD\x81\x6E\x07\xDB\x60\x62\xFE\xEC\x10\x6E\xDA\x68\x01\xF4\x83\x1B\xA9\x3E\xA2\x5B\x23\xD7\x64\xC6\xDF\xDC\xA2\x7D\xD8\x4B\xBA\x82\xD2\x51\xF8\x66\xBF\x06\x46\xE4\x79\x2A\x26\x36\x79\x8F\x1F\x4E\x99\x1D\xB2\x8F\x0C\x0E\x1C\xFF\xC9\x5D\xC0\xFD\x90\x10\xA6\xB1\x37\xF3\xCD\x3A\x24\x6E\xB4\x85\x90\xBF\x80\xB9\x0C\x8C\xD5\x9B\xD6\xC8\xF1\x56\x3F\x1A\x80\x89\x7A\xA9\xE2\x1B\x32\x51\x2C\x3E\xF2\xDF\x7B\xF6\x5D\x7A\x29\x19\x8E\xE5\xC8\xBD\x36\x71\x8B\x5D\x4C\xC2\x1D\x3F\xAD\x58\xA2\xCF\x3D\x70\x4D\xA6\x50\x98\x25\xDC\x23\xF9\xB8\x58\x41\x08\x71\xBF\x4F\xB8\x84\xA0\x8F\x00\x54\x15\xFC\x91\x6D\x58\xA7\x96\x3B\xEB\x4B\x96\x27\xCD\x6B\xA2\xA1\x86\xAC\x0D\x7C\x54\xE6\x66\x4C\x66\x5F\x90\xBE\x21\x9A\x02\x46\x2D\xE4\x83\xC2\x80\xB9\xCF\x4B\x3E\xE8\x7F\x3C\x01\xEC\x8F\x5E\xCD\x7F\xD2\x28\x42\x01\x95\x8A\xE2\x97\x3D\x10\x21\x7D\xF6\x9D\x1C\xC5\x34\xA1\xEC\x2C\x0E\x0A\x52\x2C\x12\x55\x70\x24\x3D\xCB\xC2\x14\x35\x43\x5D\x27\x4E\xBE\xC0\xBD\xAA\x7C\x96\xE7\xFC\x9E\x61\xAD\x44\xD3\x00\x97\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x06\x9A\x9B\x1F\x53\x7D\xF1\xF5\xA4\xC8\xD3\x86\x3E\xA1\x73\x59\xB4\xF7\x44\x21\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x06\x9A\x9B\x1F\x53\x7D\xF1\xF5\xA4\xC8\xD3\x86\x3E\xA1\x73\x59\xB4\xF7\x44\x21\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\xAA\x05\x6E\xB6\xDD\x15\xC9\xBF\xB3\xC6\x20\xF6\x06\x47\xB0\x86\x93\x25\xD3\x8D\xB9\xC8\x00\x3F\x97\xF5\x52\x27\x88\x71\xC9\x74\xFD\xEB\xCA\x64\xDB\x5B\xEF\x1E\x5D\xBA\xBF\xD1\xEB\xEE\x5C\x69\xBA\x16\xC8\xF3\xB9\x8F\xD3\x36\x2E\x40\x49\x07\x0D\x59\xDE\x8B\x10\xB0\x49\x05\xE2\xFF\x91\x3F\x4B\xB7\xDD\x02\x8E\xF8\x81\x28\x5C\xCC\xDC\x6D\xAF\x5F\x14\x9C\x7D\x58\x78\x0D\xF6\x80\x09\xB9\xE9\x0E\x97\x29\x19\xB8\xB7\xEB\xF8\x16\xCB\x55\x12\xE4\xC6\x7D\xBB\xC4\xEC\xF8\xB5\x1C\x4E\x3E\x67\xBF\xC5\x5F\x1B\x6D\x6D\x47\x28\xAA\x04\x58\x61\xD6\x76\xBF\x22\x7F\xD0\x07\x6A\xA7\x64\x53\xF0\x97\x8D\x9D\x80\x3F\xBB\xC1\x07\xDB\x65\xAF\xE6\x9B\x32\x9A\xC3\x54\x93\xC4\x1C\x08\xC3\x44\xFB\x7B\x63\x11\x43\xD1\x6A\x1A\x61\x6A\x79\x6D\x90\x4F\x29\x8E\x47\x05\xC1\x12\x69\x69\xD6\xC6\x36\x31\xE1\xFC\xFA\x80\xBA\x5C\x4F\xC4\xEB\xB7\x32\xAC\xF8\x75\x61\x17\xD7\x10\x19\xB9\xF1\xD2\x09\xEF\x7A\x42\x9D\x5B\x5A\x0B\xD4\xC6\x95\x4E\x2A\xCE\xFF\x07\xD7\x4F\x7E\x18\x06\x88\xF1\x19\xB5\xD9\x98\xBB\xAE\x71\xC4\x1C\xE7\x74\x59\x58\xEF\x0C\x89\xCF\x8B\x1F\x75\x93\x1A\x04\x14\x92\x48\x50\xA9\xEB\x57\x29\x00\x16\xE3\x36\x1C\xC8\xF8\xBF\xF0\x33\xD5\x41\x0F\xC4\xCC\x3C\xDD\xE9\x33\x43\x01\x91\x10\x2B\x1E\xD1\xB9\x5D\xCD\x32\x19\x8B\x8F\x8C\x20\x77\xD7\x22\xC4\x42\xDC\x84\x16\x9B\x25\x6D\xE8\xB4\x55\x71\x7F\xB0\x7C\xB3\xD3\x71\x49\xB9\xCF\x52\xA4\x04\x3F\xDC\x3D\xA0\xBB\xAF\x33\x9E\x0A\x30\x60\x8E\xDB\x9D\x5D\x94\xA8\xBD\x60\xE7\x62\x80\x76\x81\x83\x0C\x8C\xCC\x30\x46\x49\xE2\x0C\xD2\xA8\xAF\xEB\x61\x71\xEF\xE7\x22\x62\xA9\xF7\x5C\x64\x6C\x9F\x16\x8C\x67\x36\x27\x45\xF5\x09\x7B\xBF\xF6\x10\x0A\xF1\xB0\x8D\x54\x43\x8C\x04\xBA\xA3\x3F\xEF\xE2\x35\xC7\xF9\x74\xE0\x6F\x34\x41\xD0\xBF\x73\x65\x57\x20\xF9\x9B\x67\x7A\x66\x68\x24\x4E\x80\x65\xBD\x10\x99\x06\x59\xF2\x65\xAF\xB8\xC6\x47\xBB\xFD\x90\x78\x8B\x41\x73\x2E\xAF\x55\x1F\xDC\x3B\x92\x72\x6E\x84\xD3\xD0\x61\x4C\x0D\xCC\x76\x57\xE2\x2D\x85\x22\x15\x36\x0D\xEB\x01\x9D\xEB\xD8\xEB\xC4\x84\x99\xFB\xC0\x0C\xCC\x32\xE8\xE3\x77\xDA\x83\x44\x8B\x9E\x55\x28\xC0\x8B\x58\xD3\x90\x3E\x4E\x1B\x00\xF1\x15\xAD\x83\x2B\x9A"),
   // "SSL.com EV Root Certification Authority RSA R2"
   CertificateDer::from_slice(b"\x30\x82\x05\xEB\x30\x82\x03\xD3\xA0\x03\x02\x01\x02\x02\x08\x56\xB6\x29\xCD\x34\xBC\x78\xF6\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x37\x30\x35\x06\x03\x55\x04\x03\x0C\x2E\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x52\x53\x41\x20\x52\x32\x30\x1E\x17\x0D\x31\x37\x30\x35\x33\x31\x31\x38\x31\x34\x33\x37\x5A\x17\x0D\x34\x32\x30\x35\x33\x30\x31\x38\x31\x34\x33\x37\x5A\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x37\x30\x35\x06\x03\x55\x04\x03\x0C\x2E\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x52\x53\x41\x20\x52\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x8F\x36\x65\x40\xE1\xD6\x4D\xC0\xD7\xB4\xE9\x46\xDA\x6B\xEA\x33\x47\xCD\x4C\xF9\x7D\x7D\xBE\xBD\x2D\x3D\xF0\xDB\x78\xE1\x86\xA5\xD9\xBA\x09\x57\x68\xED\x57\x3E\xA0\xD0\x08\x41\x83\xE7\x28\x41\x24\x1F\xE3\x72\x15\xD0\x01\x1A\xFB\x5E\x70\x23\xB2\xCB\x9F\x39\xE3\xCF\xC5\x4E\xC6\x92\x6D\x26\xC6\x7B\xBB\xB3\xDA\x27\x9D\x0A\x86\xE9\x81\x37\x05\xFE\xF0\x71\x71\xEC\xC3\x1C\xE9\x63\xA2\x17\x14\x9D\xEF\x1B\x67\xD3\x85\x55\x02\x02\xD6\x49\xC9\xCC\x5A\xE1\xB1\xF7\x6F\x32\x9F\xC9\xD4\x3B\x88\x41\xA8\x9C\xBD\xCB\xAB\xDB\x6D\x7B\x09\x1F\xA2\x4C\x72\x90\xDA\x2B\x08\xFC\xCF\x3C\x54\xCE\x67\x0F\xA8\xCF\x5D\x96\x19\x0B\xC4\xE3\x72\xEB\xAD\xD1\x7D\x1D\x27\xEF\x92\xEB\x10\xBF\x5B\xEB\x3B\xAF\xCF\x80\xDD\xC1\xD2\x96\x04\x5B\x7A\x7E\xA4\xA9\x3C\x38\x76\xA4\x62\x8E\xA0\x39\x5E\xEA\x77\xCF\x5D\x00\x59\x8F\x66\x2C\x3E\x07\xA2\xA3\x05\x26\x11\x69\x97\xEA\x85\xB7\x0F\x96\x0B\x4B\xC8\x40\xE1\x50\xBA\x2E\x8A\xCB\xF7\x0F\x9A\x22\xE7\x7F\x9A\x37\x13\xCD\xF2\x4D\x13\x6B\x21\xD1\xC0\xCC\x22\xF2\xA1\x46\xF6\x44\x69\x9C\xCA\x61\x35\x07\x00\x6F\xD6\x61\x08\x11\xEA\xBA\xB8\xF6\xE9\xB3\x60\xE5\x4D\xB9\xEC\x9F\x14\x66\xC9\x57\x58\xDB\xCD\x87\x69\xF8\x8A\x86\x12\x03\x47\xBF\x66\x13\x76\xAC\x77\x7D\x34\x24\x85\x83\xCD\xD7\xAA\x9C\x90\x1A\x9F\x21\x2C\x7F\x78\xB7\x64\xB8\xD8\xE8\xA6\xF4\x78\xB3\x55\xCB\x84\xD2\x32\xC4\x78\xAE\xA3\x8F\x61\xDD\xCE\x08\x53\xAD\xEC\x88\xFC\x15\xE4\x9A\x0D\xE6\x9F\x1A\x77\xCE\x4C\x8F\xB8\x14\x15\x3D\x62\x9C\x86\x38\x06\x00\x66\x12\xE4\x59\x76\x5A\x53\xC0\x02\x98\xA2\x10\x2B\x68\x44\x7B\x8E\x79\xCE\x33\x4A\x76\xAA\x5B\x81\x16\x1B\xB5\x8A\xD8\xD0\x00\x7B\x5E\x62\xB4\x09\xD6\x86\x63\x0E\xA6\x05\x95\x49\xBA\x28\x8B\x88\x93\xB2\x34\x1C\xD8\xA4\x55\x6E\xB7\x1C\xD0\xDE\x99\x55\x3B\x23\xF4\x22\xE0\xF9\x29\x66\x26\xEC\x20\x50\x77\xDB\x4A\x0B\x8F\xBE\xE5\x02\x60\x70\x41\x5E\xD4\xAE\x50\x39\x22\x14\x26\xCB\xB2\x3B\x73\x74\x55\x47\x07\x79\x81\x39\xA8\x30\x13\x44\xE5\x04\x8A\xAE\x96\x13\x25\x42\x0F\xB9\x53\xC4\x9B\xFC\xCD\xE4\x1C\xDE\x3C\xFA\xAB\xD6\x06\x4A\x1F\x67\xA6\x98\x30\x1C\xDD\x2C\xDB\xDC\x18\x95\x57\x66\xC6\xFF\x5C\x8B\x56\xF5\x77\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xF9\x60\xBB\xD4\xE3\xD5\x34\xF6\xB8\xF5\x06\x80\x25\xA7\x73\xDB\x46\x69\xA8\x9E\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF9\x60\xBB\xD4\xE3\xD5\x34\xF6\xB8\xF5\x06\x80\x25\xA7\x73\xDB\x46\x69\xA8\x9E\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x56\xB3\x8E\xCB\x0A\x9D\x49\x8E\xBF\xA4\xC4\x91\xBB\x66\x17\x05\x51\x98\x75\xFB\xE5\x50\x2C\x7A\x9E\xF1\x14\xFA\xAB\xD3\x8A\x3E\xFF\x91\x29\x8F\x63\x8B\xD8\xB4\xA9\x54\x01\x0D\xBE\x93\x86\x2F\xF9\x4A\x6D\xC7\x5E\xF5\x57\xF9\xCA\x55\x1C\x12\xBE\x47\x0F\x36\xC5\xDF\x6A\xB7\xDB\x75\xC2\x47\x25\x7F\xB9\xF1\x63\xF8\x68\x2D\x55\x04\xD1\xF2\x8D\xB0\xA4\xCF\xBC\x3C\x5E\x1F\x78\xE7\xA5\xA0\x20\x70\xB0\x04\xC5\xB7\xF7\x72\xA7\xDE\x22\x0D\xBD\x33\x25\x46\x8C\x64\x92\x26\xE3\x3E\x2E\x63\x96\xDA\x9B\x8C\x3D\xF8\x18\x09\xD7\x03\xCC\x7D\x86\x82\xE0\xCA\x04\x07\x51\x50\xD7\xFF\x92\xD5\x0C\xEF\xDA\x86\x9F\x99\xD7\xEB\xB7\xAF\x68\xE2\x39\x26\x94\xBA\x68\xB7\xBF\x83\xD3\xEA\x7A\x67\x3D\x62\x67\xAE\x25\xE5\x72\xE8\xE2\xE4\xEC\xAE\x12\xF6\x4B\x2B\x3C\x9F\xE9\xB0\x40\xF3\x38\x54\xB3\xFD\xB7\x68\xC8\xDA\xC6\x8F\x51\x3C\xB2\xFB\x91\xDC\x1C\xE7\x9B\x9D\xE1\xB7\x0D\x72\x8F\xE2\xA4\xC4\xA9\x78\xF9\xEB\x14\xAC\xC6\x43\x05\xC2\x65\x39\x28\x18\x02\xC3\x82\xB2\x9D\x05\xBE\x65\xED\x96\x5F\x65\x74\x3C\xFB\x09\x35\x2E\x7B\x9C\x13\xFD\x1B\x0F\x5D\xC7\x6D\x81\x3A\x56\x0F\xCC\x3B\xE1\xAF\x02\x2F\x22\xAC\x46\xCA\x46\x3C\xA0\x1C\x4C\xD6\x44\xB4\x5E\x2E\x5C\x15\x66\x09\xE1\x26\x29\xFE\xC6\x52\x61\xBA\xB1\x73\xFF\xC3\x0C\x9C\xE5\x6C\x6A\x94\x3F\x14\xCA\x40\x16\x95\x84\xF3\x59\xA9\xAC\x5F\x4C\x61\x93\x6D\xD1\x3B\xCC\xA2\x95\x0C\x22\xA6\x67\x67\x44\x2E\xB9\xD9\xD2\x8A\x41\xB3\x66\x0B\x5A\xFB\x7D\x23\xA5\xF2\x1A\xB0\xFF\xDE\x9B\x83\x94\x2E\xD1\x3F\xDF\x92\xB7\x91\xAF\x05\x3B\x65\xC7\xA0\x6C\xB1\xCD\x62\x12\xC3\x90\x1B\xE3\x25\xCE\x34\xBC\x6F\x77\x76\xB1\x10\xC3\xF7\x05\x1A\xC0\xD6\xAF\x74\x62\x48\x17\x77\x92\x69\x90\x61\x1C\xDE\x95\x80\x74\x54\x8F\x18\x1C\xC3\xF3\x03\xD0\xBF\xA4\x43\x75\x86\x53\x18\x7A\x0A\x2E\x09\x1C\x36\x9F\x91\xFD\x82\x8A\x22\x4B\xD1\x0E\x50\x25\xDD\xCB\x03\x0C\x17\xC9\x83\x00\x08\x4E\x35\x4D\x8A\x8B\xED\xF0\x02\x94\x66\x2C\x44\x7F\xCB\x95\x27\x96\x17\xAD\x09\x30\xAC\xB6\x71\x17\x6E\x8B\x17\xF6\x1C\x09\xD4\x2D\x3B\x98\xA5\x71\xD3\x54\x13\xD9\x60\xF3\xF5\x4B\x66\x4F\xFA\xF1\xEE\x20\x12\x8D\xB4\xAC\x57\xB1\x45\x63\xA1\xAC\x76\xA9\xC2\xFB"),
   // "IdenTrust Public Sector Root CA 1"
   CertificateDer::from_slice(b"\x30\x82\x05\x66\x30\x82\x03\x4E\xA0\x03\x02\x01\x02\x02\x10\x0A\x01\x42\x80\x00\x00\x01\x45\x23\xCF\x46\x7C\x00\x00\x00\x02\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x31\x2A\x30\x28\x06\x03\x55\x04\x03\x13\x21\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x20\x50\x75\x62\x6C\x69\x63\x20\x53\x65\x63\x74\x6F\x72\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x30\x1E\x17\x0D\x31\x34\x30\x31\x31\x36\x31\x37\x35\x33\x33\x32\x5A\x17\x0D\x33\x34\x30\x31\x31\x36\x31\x37\x35\x33\x33\x32\x5A\x30\x4D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x31\x2A\x30\x28\x06\x03\x55\x04\x03\x13\x21\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x20\x50\x75\x62\x6C\x69\x63\x20\x53\x65\x63\x74\x6F\x72\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB6\x22\x94\xFC\xA4\x48\xAF\xE8\x47\x6B\x0A\xFB\x27\x76\xE4\xF2\x3F\x8A\x3B\x7A\x4A\x2C\x31\x2A\x8C\x8D\xB0\xA9\xC3\x31\x6B\xA8\x77\x76\x84\x26\xB6\xAC\x81\x42\x0D\x08\xEB\x55\x58\xBB\x7A\xF8\xBC\x65\x7D\xF2\xA0\x6D\x8B\xA8\x47\xE9\x62\x76\x1E\x11\xEE\x08\x14\xD1\xB2\x44\x16\xF4\xEA\xD0\xFA\x1E\x2F\x5E\xDB\xCB\x73\x41\xAE\xBC\x00\xB0\x4A\x2B\x40\xB2\xAC\xE1\x3B\x4B\xC2\x2D\x9D\xE4\xA1\x9B\xEC\x1A\x3A\x1E\xF0\x08\xB3\xD0\xE4\x24\x35\x07\x9F\x9C\xB4\xC9\x52\x6D\xDB\x07\xCA\x8F\xB5\x5B\xF0\x83\xF3\x4F\xC7\x2D\xA5\xC8\xAD\xCB\x95\x20\xA4\x31\x28\x57\x58\x5A\xE4\x8D\x1B\x9A\xAB\x9E\x0D\x0C\xF2\x0A\x33\x39\x22\x39\x0A\x97\x2E\xF3\x53\x77\xB9\x44\x45\xFD\x84\xCB\x36\x20\x81\x59\x2D\x9A\x6F\x6D\x48\x48\x61\xCA\x4C\xDF\x53\xD1\xAF\x52\xBC\x44\x9F\xAB\x2F\x6B\x83\x72\xEF\x75\x80\xDA\x06\x33\x1B\x5D\xC8\xDA\x63\xC6\x4D\xCD\xAC\x66\x31\xCD\xD1\xDE\x3E\x87\x10\x36\xE1\xB9\xA4\x7A\xEF\x60\x50\xB2\xCB\xCA\xA6\x56\xE0\x37\xAF\xAB\x34\x13\x39\x25\xE8\x39\x66\xE4\x98\x7A\xAA\x12\x98\x9C\x59\x66\x86\x3E\xAD\xF1\xB0\xCA\x3E\x06\x0F\x7B\xF0\x11\x4B\x37\xA0\x44\x6D\x7B\xCB\xA8\x8C\x71\xF4\xD5\xB5\x91\x36\xCC\xF0\x15\xC6\x2B\xDE\x51\x17\xB1\x97\x4C\x50\x3D\xB1\x95\x59\x7C\x05\x7D\x2D\x21\xD5\x00\xBF\x01\x67\xA2\x5E\x7B\xA6\x5C\xF2\xF7\x22\xF1\x90\x0D\x93\xDB\xAA\x44\x51\x66\xCC\x7D\x76\x03\xEB\x6A\xA8\x2A\x38\x19\x97\x76\x0D\x6B\x8A\x61\xF9\xBC\xF6\xEE\x76\xFD\x70\x2B\xDD\x29\x3C\xF8\x0A\x1E\x5B\x42\x1C\x8B\x56\x2F\x55\x1B\x1C\xA1\x2E\xB5\xC7\x16\xE6\xF8\xAA\x3C\x92\x8E\x69\xB6\x01\xC1\xB5\x86\x9D\x89\x0F\x0B\x38\x94\x54\xE8\xEA\xDC\x9E\x3D\x25\xBC\x53\x26\xED\xD5\xAB\x39\xAA\xC5\x40\x4C\x54\xAB\xB2\xB4\xD9\xD9\xF8\xD7\x72\xDB\x1C\xBC\x6D\xBD\x65\x5F\xEF\x88\x35\x2A\x66\x2F\xEE\xF6\xB3\x65\xF0\x33\x8D\x7C\x98\x41\x69\x46\x0F\x43\x1C\x69\xFA\x9B\xB5\xD0\x61\x6A\xCD\xCA\x4B\xD9\x4C\x90\x46\xAB\x15\x59\xA1\x47\x54\x29\x2E\x83\x28\x5F\x1C\xC2\xA2\xAB\x72\x17\x00\x06\x8E\x45\xEC\x8B\xE2\x33\x3D\x7F\xDA\x19\x44\xE4\x62\x72\xC3\xDF\x22\xC6\xF2\x56\xD4\xDD\x5F\x95\x72\xED\x6D\x5F\xF7\x48\x03\x5B\xFD\xC5\x2A\xA0\xF6\x73\x23\x84\x10\x1B\x01\xE7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE3\x71\xE0\x9E\xD8\xA7\x42\xD9\xDB\x71\x91\x6B\x94\x93\xEB\xC3\xA3\xD1\x14\xA3\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x47\xFA\xDD\x0A\xB0\x11\x91\x38\xAD\x4D\x5D\xF7\xE5\x0E\x97\x54\x19\x82\x48\x87\x54\x8C\xAA\x64\x99\xD8\x5A\xFE\x88\x01\xC5\x58\xA5\x99\xB1\x23\x54\x23\xB7\x6A\x1D\x20\x57\xE5\x01\x62\x41\x17\xD3\x09\xDB\x75\xCB\x6E\x54\x90\x75\xFE\x1A\x9F\x81\x0A\xC2\xDD\xD7\xF7\x09\xD0\x5B\x72\x15\xE4\x1E\x09\x6A\x3D\x33\xF3\x21\x9A\xE6\x15\x7E\xAD\x51\xD5\x0D\x10\xED\x7D\x42\xC0\x8F\xEE\xC0\x9A\x08\xD5\x41\xD6\x5C\x0E\x21\x69\x6E\x80\x61\x0E\x15\xC0\xB8\xCF\xC5\x49\x12\x52\xCC\xBE\x3A\xCC\xD4\x2E\x38\x05\xDE\x35\xFD\x1F\x6F\xB8\x80\x68\x98\x3D\x4D\xA0\xCA\x40\x65\xD2\x73\x7C\xF5\x8B\xD9\x0A\x95\x3F\xD8\x3F\x23\x6D\x1A\xD1\x2A\x24\x19\xD9\x85\xB3\x17\xEF\x78\x6E\xA9\x58\xD1\x23\xD3\xC7\x13\xED\x72\x25\x7F\x5D\xB1\x73\x70\xD0\x7F\x06\x97\x09\x84\x29\x80\x61\x1D\xFA\x5E\xFF\x73\xAC\xA0\xE3\x89\xB8\x1C\x71\x15\xC6\xDE\x31\x7F\x12\xDC\xE1\x6D\x9B\xAF\xE7\xE8\x9F\x75\x78\x4C\xAB\x46\x3B\x9A\xCE\xBF\x05\x18\x5D\x4D\x15\x3C\x16\x9A\x19\x50\x04\x9A\xB2\x9A\x6F\x65\x8B\x52\x5F\x3C\x58\x04\x28\x25\xC0\x66\x61\x31\x7E\xB9\xE0\x75\xB9\x1A\xA8\x81\xD6\x72\x17\xB3\xC5\x03\x31\x35\x11\x78\x78\xA2\xE0\xE9\x30\x8C\x7F\x80\xDF\x58\xDF\x3C\xBA\x27\x96\xE2\x80\x34\x6D\xE3\x98\xD3\x64\x27\xAC\x48\x7E\x28\x77\x5C\xC6\x25\x61\x25\xF8\x85\x0C\x65\xFA\xC4\x32\x2F\xA5\x98\x05\xE4\xF8\x0B\x67\x16\x16\xC6\x82\xB8\x32\x19\xF9\xF9\xB9\x79\xDC\x1F\xCD\xEB\xAF\xAB\x0E\xDD\x1B\xDB\x45\xE4\x7A\xE7\x02\xE2\x95\x5D\xFC\x69\xF0\x53\x69\x61\x95\x75\x79\x0B\x5E\x55\xE6\x38\x1C\x94\xA9\x59\x33\x9E\xC8\x71\x74\x79\x7F\x51\x89\xB6\xC8\x6A\xB8\x30\xC8\x6A\x38\xC3\x6E\x9E\xE1\x37\x16\xEA\x05\x62\x4C\x5B\x12\x47\xED\xA7\xB4\xB3\x58\x56\xC7\x49\xF3\x7F\x12\x68\x09\x31\x71\xF0\x6D\xF8\x4E\x47\xFB\xD6\x85\xEE\xC5\x58\x40\x19\xA4\x1D\xA7\xF9\x4B\x43\x37\xDC\x68\x5A\x4F\xCF\xEB\xC2\x64\x74\xDE\xB4\x15\xD9\xF4\x54\x54\x1A\x2F\x1C\xD7\x97\x71\x54\x90\x8E\xD9\x20\x9D\x53\x2B\x7F\xAB\x8F\xE2\xEA\x30\xBC\x50\x37\xEF\xF1\x47\xB5\x7D\x7C\x2C\x04\xEC\x68\x9D\xB4\x49\x44\x10\xF4\x72\x4B\x1C\x64\xE7\xFC\xE6\x6B\x90\xDD\x69\x7D\x69\xFD\x00\x56\xA5\xB7\xAC\xB6\xAD\xB7\xCA\x3E\x01\xEF\x9C"),
   // "vTrus ECC Root CA"
   CertificateDer::from_slice(b"\x30\x82\x02\x0F\x30\x82\x01\x95\xA0\x03\x02\x01\x02\x02\x14\x6E\x6A\xBC\x59\xAA\x53\xBE\x98\x39\x67\xA2\xD2\x6B\xA4\x3B\xE6\x6D\x1C\xD6\xDA\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x1C\x30\x1A\x06\x03\x55\x04\x0A\x13\x13\x69\x54\x72\x75\x73\x43\x68\x69\x6E\x61\x20\x43\x6F\x2E\x2C\x4C\x74\x64\x2E\x31\x1A\x30\x18\x06\x03\x55\x04\x03\x13\x11\x76\x54\x72\x75\x73\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x38\x30\x37\x33\x31\x30\x37\x32\x36\x34\x34\x5A\x17\x0D\x34\x33\x30\x37\x33\x31\x30\x37\x32\x36\x34\x34\x5A\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x1C\x30\x1A\x06\x03\x55\x04\x0A\x13\x13\x69\x54\x72\x75\x73\x43\x68\x69\x6E\x61\x20\x43\x6F\x2E\x2C\x4C\x74\x64\x2E\x31\x1A\x30\x18\x06\x03\x55\x04\x03\x13\x11\x76\x54\x72\x75\x73\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x65\x50\x4A\xAE\x8C\x79\x96\x4A\xAA\x1C\x08\xC3\xA3\xA2\xCD\xFE\x59\x56\x41\x77\xFD\x26\x94\x42\xBB\x1D\xCD\x08\xDB\x73\xB2\x5B\x75\xF3\xCF\x9C\x4E\x82\xF4\xBF\xF8\x61\x26\x85\x6C\xD6\x85\x5B\x72\x70\xD2\xFD\xDB\x62\xB4\xDF\x53\x8B\xBD\xB1\x44\x58\x62\x42\x09\xC7\xFA\x7F\x5B\x10\xE7\xFE\x40\xFD\xC0\xD8\xC3\x2B\x32\xE7\x70\xA6\xB7\xA6\x20\x55\x1D\x7B\x80\x5D\x4B\x8F\x67\x4C\xF1\x10\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x98\x39\xCD\xBE\xD8\xB2\x8C\xF7\xB2\xAB\xE1\xAD\x24\xAF\x7B\x7C\xA1\xDB\x1F\xCF\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x57\x9D\xDD\x56\xF1\xC7\xE3\xE9\xB8\x49\x50\x6B\x9B\x69\xC3\x6F\xEC\xC3\x7D\x25\xE4\x57\x95\x13\x40\x9B\x52\xD3\x3B\xF3\x40\x19\xBC\x26\xC7\x2D\x06\x9E\xB5\x7B\x36\x9F\xF5\x25\xD4\x63\x6B\x00\x02\x31\x00\xE9\xD3\xC6\x9E\x56\x9A\x2A\xCC\xA1\xDA\x3F\xC8\x66\x2B\xD3\x58\x9C\x20\x85\xFA\xAB\x91\x8A\x70\x70\x11\x38\x60\x64\x0B\x62\x09\x91\x58\x00\xF9\x4D\xFB\x34\x68\xDA\x09\xAD\x21\x06\x18\x94\xCE"),
   // "DigiCert Global Root G3"
   CertificateDer::from_slice(b"\x30\x82\x02\x3F\x30\x82\x01\xC5\xA0\x03\x02\x01\x02\x02\x10\x05\x55\x56\xBC\xF2\x5E\xA4\x35\x35\xC3\xA4\x0F\xD5\xAB\x45\x72\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x44\x69\x67\x69\x43\x65\x72\x74\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x33\x30\x1E\x17\x0D\x31\x33\x30\x38\x30\x31\x31\x32\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x35\x31\x32\x30\x30\x30\x30\x5A\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x44\x69\x67\x69\x43\x65\x72\x74\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x33\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xDD\xA7\xD9\xBB\x8A\xB8\x0B\xFB\x0B\x7F\x21\xD2\xF0\xBE\xBE\x73\xF3\x33\x5D\x1A\xBC\x34\xEA\xDE\xC6\x9B\xBC\xD0\x95\xF6\xF0\xCC\xD0\x0B\xBA\x61\x5B\x51\x46\x7E\x9E\x2D\x9F\xEE\x8E\x63\x0C\x17\xEC\x07\x70\xF5\xCF\x84\x2E\x40\x83\x9C\xE8\x3F\x41\x6D\x3B\xAD\xD3\xA4\x14\x59\x36\x78\x9D\x03\x43\xEE\x10\x13\x6C\x72\xDE\xAE\x88\xA7\xA1\x6B\xB5\x43\xCE\x67\xDC\x23\xFF\x03\x1C\xA3\xE2\x3E\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB3\xDB\x48\xA4\xF9\xA1\xC5\xD8\xAE\x36\x41\xCC\x11\x63\x69\x62\x29\xBC\x4B\xC6\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xAD\xBC\xF2\x6C\x3F\x12\x4A\xD1\x2D\x39\xC3\x0A\x09\x97\x73\xF4\x88\x36\x8C\x88\x27\xBB\xE6\x88\x8D\x50\x85\xA7\x63\xF9\x9E\x32\xDE\x66\x93\x0F\xF1\xCC\xB1\x09\x8F\xDD\x6C\xAB\xFA\x6B\x7F\xA0\x02\x30\x39\x66\x5B\xC2\x64\x8D\xB8\x9E\x50\xDC\xA8\xD5\x49\xA2\xED\xC7\xDC\xD1\x49\x7F\x17\x01\xB8\xC8\x86\x8F\x4E\x8C\x88\x2B\xA8\x9A\xA9\x8A\xC5\xD1\x00\xBD\xF8\x54\xE2\x9A\xE5\x5B\x7C\xB3\x27\x17"),
   // "SSL.com Root Certification Authority ECC"
   CertificateDer::from_slice(b"\x30\x82\x02\x8D\x30\x82\x02\x14\xA0\x03\x02\x01\x02\x02\x08\x75\xE6\xDF\xCB\xC1\x68\x5B\xA8\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x7C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x0C\x28\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x45\x43\x43\x30\x1E\x17\x0D\x31\x36\x30\x32\x31\x32\x31\x38\x31\x34\x30\x33\x5A\x17\x0D\x34\x31\x30\x32\x31\x32\x31\x38\x31\x34\x30\x33\x5A\x30\x7C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x0C\x28\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x45\x43\x43\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x45\x6E\xA9\x50\xC4\xA6\x23\x36\x9E\x5F\x28\x8D\x17\xCB\x96\x22\x64\x3F\xDC\x7A\x8E\x1D\xCC\x08\xB3\xA2\x71\x24\xBA\x8E\x49\xB9\x04\x1B\x47\x96\x58\xAB\x2D\x95\xC8\xED\x9E\x08\x35\xC8\x27\xEB\x89\x8C\x53\x58\xEB\x62\x8A\xFE\xF0\x5B\x0F\x6B\x31\x52\x63\x41\x3B\x89\xCD\xEC\xEC\xB6\x8D\x19\xD3\x34\x07\xDC\xBB\xC6\x06\x7F\xC2\x45\x95\xEC\xCB\x7F\xA8\x23\xE0\x09\xE9\x81\xFA\xF3\x47\xD3\xA3\x63\x30\x61\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x82\xD1\x85\x73\x30\xE7\x35\x04\xD3\x8E\x02\x92\xFB\xE5\xA4\xD1\xC4\x21\xE8\xCD\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x82\xD1\x85\x73\x30\xE7\x35\x04\xD3\x8E\x02\x92\xFB\xE5\xA4\xD1\xC4\x21\xE8\xCD\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x67\x00\x30\x64\x02\x30\x6F\xE7\xEB\x59\x11\xA4\x60\xCF\x61\xB0\x96\x7B\xED\x05\xF9\x2F\x13\x91\xDC\xED\xE5\xFC\x50\x6B\x11\x46\x46\xB3\x1C\x21\x00\x62\xBB\xBE\xC3\xE7\xE8\xCD\x07\x99\xF9\x0D\x0B\x5D\x72\x3E\xC4\xAA\x02\x30\x1F\xBC\xBA\x0B\xE2\x30\x24\xFB\x7C\x6D\x80\x55\x0A\x99\x3E\x80\x0D\x33\xE5\x66\xA3\xB3\xA3\xBB\xA5\xD5\x8B\x8F\x09\x2C\xA6\x5D\x7E\xE2\xF0\x07\x08\x68\x6D\xD2\x7C\x69\x6E\x5F\xDF\xE5\x6A\x65"),
   // "GTS Root R4"
   CertificateDer::from_slice(b"\x30\x82\x02\x09\x30\x82\x01\x8E\xA0\x03\x02\x01\x02\x02\x0D\x02\x03\xE5\xC0\x68\xEF\x63\x1A\x9C\x72\x90\x50\x52\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x34\x30\x1E\x17\x0D\x31\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x34\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xF3\x74\x73\xA7\x68\x8B\x60\xAE\x43\xB8\x35\xC5\x81\x30\x7B\x4B\x49\x9D\xFB\xC1\x61\xCE\xE6\xDE\x46\xBD\x6B\xD5\x61\x18\x35\xAE\x40\xDD\x73\xF7\x89\x91\x30\x5A\xEB\x3C\xEE\x85\x7C\xA2\x40\x76\x3B\xA9\xC6\xB8\x47\xD8\x2A\xE7\x92\x91\x6A\x73\xE9\xB1\x72\x39\x9F\x29\x9F\xA2\x98\xD3\x5F\x5E\x58\x86\x65\x0F\xA1\x84\x65\x06\xD1\xDC\x8B\xC9\xC7\x73\xC8\x8C\x6A\x2F\xE5\xC4\xAB\xD1\x1D\x8A\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x80\x4C\xD6\xEB\x74\xFF\x49\x36\xA3\xD5\xD8\xFC\xB5\x3E\xC5\x6A\xF0\x94\x1D\x8C\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\xE8\x40\xFF\x83\xDE\x03\xF4\x9F\xAE\x1D\x7A\xA7\x2E\xB9\xAF\x4F\xF6\x83\x1D\x0E\x2D\x85\x01\x1D\xD1\xD9\x6A\xEC\x0F\xC2\xAF\xC7\x5E\x56\x5E\x5C\xD5\x1C\x58\x22\x28\x0B\xF7\x30\xB6\x2F\xB1\x7C\x02\x31\x00\xF0\x61\x3C\xA7\xF4\xA0\x82\xE3\x21\xD5\x84\x1D\x73\x86\x9C\x2D\xAF\xCA\x34\x9B\xF1\x9F\xB9\x23\x36\xE2\xBC\x60\x03\x9D\x80\xB3\x9A\x56\xC8\xE1\xE2\xBB\x14\x79\xCA\xCD\x21\xD4\x94\xB5\x49\x43"),
   // "GTS Root R3"
   CertificateDer::from_slice(b"\x30\x82\x02\x09\x30\x82\x01\x8E\xA0\x03\x02\x01\x02\x02\x0D\x02\x03\xE5\xB8\x82\xEB\x20\xF8\x25\x27\x6D\x3D\x66\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x33\x30\x1E\x17\x0D\x31\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x33\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x1F\x4F\x33\x87\x33\x29\x8A\xA1\x84\xDE\xCB\xC7\x21\x58\x41\x89\xEA\x56\x9D\x2B\x4B\x85\xC6\x1D\x4C\x27\xBC\x7F\x26\x51\x72\x6F\xE2\x9F\xD6\xA3\xCA\xCC\x45\x14\x46\x8B\xAD\xEF\x7E\x86\x8C\xEC\xB1\x7E\x2F\xFF\xA9\x71\x9D\x18\x84\x45\x04\x41\x55\x6E\x2B\xEA\x26\x7F\xBB\x90\x01\xE3\x4B\x19\xBA\xE4\x54\x96\x45\x09\xB1\xD5\x6C\x91\x44\xAD\x84\x13\x8E\x9A\x8C\x0D\x80\x0C\x32\xF6\xE0\x27\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC1\xF1\x26\xBA\xA0\x2D\xAE\x85\x81\xCF\xD3\xF1\x2A\x12\xBD\xB8\x0A\x67\xFD\xBC\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\xF6\xE1\x20\x95\x14\x7B\x54\xA3\x90\x16\x11\xBF\x84\xC8\xEA\x6F\x6B\x17\x9E\x1E\x46\x98\x20\x9B\x9F\xD3\x0D\xD9\xAC\xD3\x2F\xCD\x7C\xF8\x5B\x2E\x55\xBB\xBF\xDD\x92\xF7\xA4\x0C\xDC\x31\xE1\xA2\x02\x31\x00\xFC\x97\x66\x66\xE5\x43\x16\x13\x83\xDD\xC7\xDF\x2F\xBE\x14\x38\xED\x01\xCE\xB1\x17\x1A\x11\x75\xE9\xBD\x03\x8F\x26\x7E\x84\xE5\xC9\x60\xA6\x95\xD7\x54\x59\xB7\xE7\x11\x2C\x89\xD4\xB9\xEE\x17"),
   // "Microsoft ECC Root Certificate Authority 2017"
   CertificateDer::from_slice(b"\x30\x82\x02\x59\x30\x82\x01\xDF\xA0\x03\x02\x01\x02\x02\x10\x66\xF2\x3D\xAF\x87\xDE\x8B\xB1\x4A\xEA\x0C\x57\x31\x01\xC2\xEC\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x32\x30\x31\x37\x30\x1E\x17\x0D\x31\x39\x31\x32\x31\x38\x32\x33\x30\x36\x34\x35\x5A\x17\x0D\x34\x32\x30\x37\x31\x38\x32\x33\x31\x36\x30\x34\x5A\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x32\x30\x31\x37\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xD4\xBC\x3D\x02\x42\x75\x41\x13\x23\xCD\x80\x04\x86\x02\x51\x2F\x6A\xA8\x81\x62\x0B\x65\xCC\xF6\xCA\x9D\x1E\x6F\x4A\x66\x51\xA2\x03\xD9\x9D\x91\xFA\xB6\x16\xB1\x8C\x6E\xDE\x7C\xCD\xDB\x79\xA6\x2F\xCE\xBB\xCE\x71\x2F\xE5\xA5\xAB\x28\xEC\x63\x04\x66\x99\xF8\xFA\xF2\x93\x10\x05\xE1\x81\x28\x42\xE3\xC6\x68\xF4\xE6\x1B\x84\x60\x4A\x89\xAF\xED\x79\x0F\x3B\xCE\xF1\xF6\x44\xF5\x01\x78\xC0\xA3\x54\x30\x52\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC8\xCB\x99\x72\x70\x52\x0C\xF8\xE6\xBE\xB2\x04\x57\x29\x2A\xCF\x42\x10\xED\x35\x30\x10\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x15\x01\x04\x03\x02\x01\x00\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x58\xF2\x4D\xEA\x0C\xF9\x5F\x5E\xEE\x60\x29\xCB\x3A\xF2\xDB\xD6\x32\x84\x19\x3F\x7C\xD5\x2F\xC2\xB1\xCC\x93\xAE\x50\xBB\x09\x32\xC6\xC6\xED\x7E\xC9\x36\x94\x12\xE4\x68\x85\x06\xA2\x1B\xD0\x2F\x02\x31\x00\x99\xE9\x16\xB4\x0E\xFA\x56\x48\xD4\xA4\x30\x16\x91\x78\xDB\x54\x8C\x65\x01\x8A\xE7\x50\x66\xC2\x31\xB7\x39\xBA\xB8\x1A\x22\x07\x4E\xFC\x6B\x54\x16\x20\xFF\x2B\xB5\xE7\x4C\x0C\x4D\xA6\x4F\x73"),
   // "DigiCert TLS RSA4096 Root G5"
   CertificateDer::from_slice(b"\x30\x82\x05\x66\x30\x82\x03\x4E\xA0\x03\x02\x01\x02\x02\x10\x08\xF9\xB4\x78\xA8\xFA\x7E\xDA\x6A\x33\x37\x89\xDE\x7C\xCF\x8A\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x4D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x17\x30\x15\x06\x03\x55\x04\x0A\x13\x0E\x44\x69\x67\x69\x43\x65\x72\x74\x2C\x20\x49\x6E\x63\x2E\x31\x25\x30\x23\x06\x03\x55\x04\x03\x13\x1C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x54\x4C\x53\x20\x52\x53\x41\x34\x30\x39\x36\x20\x52\x6F\x6F\x74\x20\x47\x35\x30\x1E\x17\x0D\x32\x31\x30\x31\x31\x35\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x31\x31\x34\x32\x33\x35\x39\x35\x39\x5A\x30\x4D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x17\x30\x15\x06\x03\x55\x04\x0A\x13\x0E\x44\x69\x67\x69\x43\x65\x72\x74\x2C\x20\x49\x6E\x63\x2E\x31\x25\x30\x23\x06\x03\x55\x04\x03\x13\x1C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x54\x4C\x53\x20\x52\x53\x41\x34\x30\x39\x36\x20\x52\x6F\x6F\x74\x20\x47\x35\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB3\xD0\xF4\xC9\x79\x11\x9D\xFD\xFC\x66\x81\xE7\xCC\xD5\xE4\xBC\xEC\x81\x3E\x6A\x35\x8E\x2E\xB7\xE7\xDE\xAF\xF9\x07\x4D\xCF\x30\x9D\xEA\x09\x0B\x99\xBD\x6C\x57\xDA\x18\x4A\xB8\x78\xAC\x3A\x39\xA8\xA6\x48\xAC\x2E\x72\xE5\xBD\xEB\xF1\x1A\xCD\xE7\xA4\x03\xA9\x3F\x11\xB4\xD8\x2F\x89\x16\xFB\x94\x01\x3D\xBB\x2F\xF8\x13\x05\xA1\x78\x1C\x8E\x28\xE0\x45\xE0\x83\xF4\x59\x1B\x95\xB3\xAE\x7E\x03\x45\xE5\xBE\xC2\x42\xFE\xEE\xF2\x3C\xB6\x85\x13\x98\x32\x9D\x16\xA8\x29\xC2\x0B\x1C\x38\xDC\x9F\x31\x77\x5C\xBF\x27\xA3\xFC\x27\xAC\xB7\x2B\xBD\x74\x9B\x17\x2D\xF2\x81\xDA\x5D\xB0\xE1\x23\x17\x3E\x88\x4A\x12\x23\xD0\xEA\xCF\x9D\xDE\x03\x17\xB1\x42\x4A\xA0\x16\x4C\xA4\x6D\x93\xE9\x3F\x3A\xEE\x3A\x7C\x9D\x58\x9D\xF4\x4E\x8F\xFC\x3B\x23\xC8\x6D\xB8\xE2\x05\xDA\xCC\xEB\xEC\xC3\x31\xF4\xD7\xA7\x29\x54\x80\xCF\x44\x5B\x4C\x6F\x30\x9E\xF3\xCC\xDD\x1F\x94\x43\x9D\x4D\x7F\x70\x70\x0D\xD4\x3A\xD1\x37\xF0\x6C\x9D\x9B\xC0\x14\x93\x58\xEF\xCD\x41\x38\x75\xBC\x13\x03\x95\x7C\x7F\xE3\x5C\xE9\xD5\x0D\xD5\xE2\x7C\x10\x62\xAA\x6B\xF0\x3D\x76\xF3\x3F\xA3\xE8\xB0\xC1\xFD\xEF\xAA\x57\x4D\xAC\x86\xA7\x18\xB4\x29\xC1\x2C\x0E\xBF\x64\xBE\x29\x8C\xD8\x02\x2D\xCD\x5C\x2F\xF2\x7F\xEF\x15\xF4\x0C\x15\xAC\x0A\xB0\xF1\xD3\x0D\x4F\x6A\x4D\x77\x97\x01\xA0\xF1\x66\xB7\xB7\xCE\xEF\xCE\xEC\xEC\xA5\x75\xCA\xAC\xE3\xE1\x63\xF7\xB8\xA1\x04\xC8\xBC\x7B\x3F\x5D\x2D\x16\x22\x56\xED\x48\x49\xFE\xA7\x2F\x79\x30\x25\x9B\xBA\x6B\x2D\x3F\x9D\x3B\xC4\x17\xE7\x1D\x2E\xFB\xF2\xCF\xA6\xFC\xE3\x14\x2C\x96\x98\x21\x8C\xB4\x91\xE9\x19\x60\x83\xF2\x30\x2B\x06\x73\x50\xD5\x98\x3B\x06\xE9\xC7\x8A\x0C\x60\x8C\x28\xF8\x52\x9B\x6E\xE1\xF6\x4D\xBB\x06\x24\x9B\xD7\x2B\x26\x3F\xFD\x2A\x2F\x71\xF5\xD6\x24\xBE\x7F\x31\x9E\x0F\x6D\xE8\x8F\x4F\x4D\xA3\x3F\xFF\x35\xEA\xDF\x49\x5E\x41\x8F\x86\xF9\xF1\x77\x79\x4B\x1B\xB4\xA3\x5E\x2F\xFB\x46\x02\xD0\x66\x13\x5E\x5E\x85\x4F\xCE\xD8\x70\x88\x7B\xCE\x01\xB5\x96\x97\xD7\xCD\x7D\xFD\x82\xF8\xC2\x24\xC1\xCA\x01\x39\x4F\x8D\xA2\xC1\x14\x40\x1F\x9C\x66\xD5\x0C\x09\x46\xD6\xF2\xD0\xD1\x48\x76\x56\x3A\x43\xCB\xB6\x0A\x11\x39\xBA\x8C\x13\x6C\x06\xB5\x9E\xCF\xEB\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x51\x33\x1C\xED\x36\x40\xAF\x17\xD3\x25\xCD\x69\x68\xF2\xAF\x4E\x23\x3E\xB3\x41\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x60\xA6\xAF\x5B\x5F\x57\xDA\x89\xDB\x4B\x50\xA9\xC4\x23\x35\x21\xFF\xD0\x61\x30\x84\x91\xB7\x3F\x10\xCF\x25\x8E\xC9\xBF\x46\x34\xD9\xC1\x21\x26\x1C\x70\x19\x72\x1E\xA3\xC9\x87\xFE\xA9\x43\x64\x96\x3A\xC8\x53\x04\x0A\xB6\x41\xBB\xC4\x47\x00\xD9\x9F\x18\x18\x3B\xB2\x0E\xF3\x34\xEA\x24\xF7\xDD\xAF\x20\x60\xAE\x92\x28\x5F\x36\xE7\x5D\xE4\xDE\xC7\x3C\xDB\x50\x39\xAD\xBB\x3D\x28\x4D\x96\x7C\x76\xC6\x5B\xF4\xC1\xDB\x14\xA5\xAB\x19\x62\x07\x18\x40\x5F\x97\x91\xDC\x9C\xC7\xAB\xB5\x51\x0D\xE6\x69\x53\x55\xCC\x39\x7D\xDA\xC5\x11\x55\x72\xC5\x3B\x8B\x89\xF8\x34\x2D\xA4\x17\xE5\x17\xE6\x99\x7D\x30\x88\x21\x37\xCD\x30\x17\x3D\xB8\xF2\xBC\xA8\x75\xA0\x43\xDC\x3E\x89\x4B\x90\xAE\x6D\x03\xE0\x1C\xA3\xA0\x96\x09\xBB\x7D\xA3\xB7\x2A\x10\x44\x4B\x46\x07\x34\x63\xED\x31\xB9\x04\xEE\xA3\x9B\x9A\xAE\xE6\x31\x78\xF4\xEA\x24\x61\x3B\xAB\x58\x64\xFF\xBB\x87\x27\x62\x25\x81\xDF\xDC\xA1\x2F\xF6\xED\xA7\xFF\x7A\x8F\x51\x2E\x30\xF8\xA4\x01\xD2\x85\x39\x5F\x01\x99\x96\x6F\x5A\x5B\x70\x19\x46\xFE\x86\x60\x3E\xAD\x80\x10\x09\xDD\x39\x25\x2F\x58\x7F\xBB\xD2\x74\xF0\xF7\x46\x1F\x46\x39\x4A\xD8\x53\xD0\xF3\x2E\x3B\x71\xA5\xD4\x6F\xFC\xF3\x67\xE4\x07\x8F\xDD\x26\x19\xE1\x8D\x5B\xFA\xA3\x93\x11\x9B\xE9\xC8\x3A\xC3\x55\x68\x9A\x92\xE1\x52\x76\x38\xE8\xE1\xBA\xBD\xFB\x4F\xD5\xEF\xB3\xE7\x48\x83\x31\xF0\x82\x21\xE3\xB6\xBE\xA7\xAB\x6F\xEF\x9F\xDF\x4C\xCF\x01\xB8\x62\x6A\x23\x3D\xE7\x09\x4D\x80\x1B\x7B\x30\xA4\xC3\xDD\x07\x7F\x34\xBE\xA4\x26\xB2\xF6\x41\xE8\x09\x1D\xE3\x20\x98\xAA\x37\x4F\xFF\xF7\xF1\xE2\x29\x70\x31\x47\x3F\x74\xD0\x14\x16\xFA\x21\x8A\x02\xD5\x8A\x09\x94\x77\x2E\xF2\x59\x28\x8B\x7C\x50\x92\x0A\x66\x78\x38\x83\x75\xC4\xB5\x5A\xA8\x11\xC6\xE5\xC1\x9D\x66\x55\xCF\x53\xC4\xAF\xD7\x75\x85\xA9\x42\x13\x56\xEC\x21\x77\x81\x93\x5A\x0C\xEA\x96\xD9\x49\xCA\xA1\x08\xF2\x97\x3B\x6D\x9B\x04\x18\x24\x44\x8E\x7C\x01\xF2\xDC\x25\xD8\x5E\x86\x9A\xB1\x39\xDB\xF5\x91\x32\x6A\xD1\xA6\x70\x8A\xA2\xF7\xDE\xA4\x45\x85\x26\xA8\x1E\x8C\x5D\x29\x5B\xC8\x4B\xD8\x9A\x6A\x03\x5E\x70\xF2\x85\x4F\x6C\x4B\x68\x2F\xCA\x54\xF6\x8C\xDA\x32\xFE\xC3\x6B\x83\x3F\x38\xC6\x7E"),
   // "Microsec e-Szigno Root CA 2009"
   CertificateDer::from_slice(b"\x30\x82\x04\x0A\x30\x82\x02\xF2\xA0\x03\x02\x01\x02\x02\x09\x00\xC2\x7E\x43\x04\x4E\x47\x3F\x19\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x55\x31\x11\x30\x0F\x06\x03\x55\x04\x07\x0C\x08\x42\x75\x64\x61\x70\x65\x73\x74\x31\x16\x30\x14\x06\x03\x55\x04\x0A\x0C\x0D\x4D\x69\x63\x72\x6F\x73\x65\x63\x20\x4C\x74\x64\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x4D\x69\x63\x72\x6F\x73\x65\x63\x20\x65\x2D\x53\x7A\x69\x67\x6E\x6F\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x30\x39\x31\x1F\x30\x1D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x09\x01\x16\x10\x69\x6E\x66\x6F\x40\x65\x2D\x73\x7A\x69\x67\x6E\x6F\x2E\x68\x75\x30\x1E\x17\x0D\x30\x39\x30\x36\x31\x36\x31\x31\x33\x30\x31\x38\x5A\x17\x0D\x32\x39\x31\x32\x33\x30\x31\x31\x33\x30\x31\x38\x5A\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x55\x31\x11\x30\x0F\x06\x03\x55\x04\x07\x0C\x08\x42\x75\x64\x61\x70\x65\x73\x74\x31\x16\x30\x14\x06\x03\x55\x04\x0A\x0C\x0D\x4D\x69\x63\x72\x6F\x73\x65\x63\x20\x4C\x74\x64\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x4D\x69\x63\x72\x6F\x73\x65\x63\x20\x65\x2D\x53\x7A\x69\x67\x6E\x6F\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x30\x39\x31\x1F\x30\x1D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x09\x01\x16\x10\x69\x6E\x66\x6F\x40\x65\x2D\x73\x7A\x69\x67\x6E\x6F\x2E\x68\x75\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xE9\xF8\x8F\xF3\x63\xAD\xDA\x86\xD8\xA7\xE0\x42\xFB\xCF\x91\xDE\xA6\x26\xF8\x99\xA5\x63\x70\xAD\x9B\xAE\xCA\x33\x40\x7D\x6D\x96\x6E\xA1\x0E\x44\xEE\xE1\x13\x9D\x94\x42\x52\x9A\xBD\x75\x85\x74\x2C\xA8\x0E\x1D\x93\xB6\x18\xB7\x8C\x2C\xA8\xCF\xFB\x5C\x71\xB9\xDA\xEC\xFE\xE8\x7E\x8F\xE4\x2F\x1D\xB2\xA8\x75\x87\xD8\xB7\xA1\xE5\x3B\xCF\x99\x4A\x46\xD0\x83\x19\x7D\xC0\xA1\x12\x1C\x95\x6D\x4A\xF4\xD8\xC7\xA5\x4D\x33\x2E\x85\x39\x40\x75\x7E\x14\x7C\x80\x12\x98\x50\xC7\x41\x67\xB8\xA0\x80\x61\x54\xA6\x6C\x4E\x1F\xE0\x9D\x0E\x07\xE9\xC9\xBA\x33\xE7\xFE\xC0\x55\x28\x2C\x02\x80\xA7\x19\xF5\x9E\xDC\x55\x53\x03\x97\x7B\x07\x48\xFF\x99\xFB\x37\x8A\x24\xC4\x59\xCC\x50\x10\x63\x8E\xAA\xA9\x1A\xB0\x84\x1A\x86\xF9\x5F\xBB\xB1\x50\x6E\xA4\xD1\x0A\xCC\xD5\x71\x7E\x1F\xA7\x1B\x7C\xF5\x53\x6E\x22\x5F\xCB\x2B\xE6\xD4\x7C\x5D\xAE\xD6\xC2\xC6\x4C\xE5\x05\x01\xD9\xED\x57\xFC\xC1\x23\x79\xFC\xFA\xC8\x24\x83\x95\xF3\xB5\x6A\x51\x01\xD0\x77\xD6\xE9\x12\xA1\xF9\x1A\x83\xFB\x82\x1B\xB9\xB0\x97\xF4\x76\x06\x33\x43\x49\xA0\xFF\x0B\xB5\xFA\xB5\x02\x03\x01\x00\x01\xA3\x81\x80\x30\x7E\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xCB\x0F\xC6\xDF\x42\x43\xCC\x3D\xCB\xB5\x48\x23\xA1\x1A\x7A\xA6\x2A\xBB\x34\x68\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xCB\x0F\xC6\xDF\x42\x43\xCC\x3D\xCB\xB5\x48\x23\xA1\x1A\x7A\xA6\x2A\xBB\x34\x68\x30\x1B\x06\x03\x55\x1D\x11\x04\x14\x30\x12\x81\x10\x69\x6E\x66\x6F\x40\x65\x2D\x73\x7A\x69\x67\x6E\x6F\x2E\x68\x75\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\xC9\xD1\x0E\x5E\x2E\xD5\xCC\xB3\x7C\x3E\xCB\xFC\x3D\xFF\x0D\x28\x95\x93\x04\xC8\xBF\xDA\xCD\x79\xB8\x43\x90\xF0\xA4\xBE\xEF\xF2\xEF\x21\x98\xBC\xD4\xD4\x5D\x06\xF6\xEE\x42\xEC\x30\x6C\xA0\xAA\xA9\xCA\xF1\xAF\x8A\xFA\x3F\x0B\x73\x6A\x3E\xEA\x2E\x40\x7E\x1F\xAE\x54\x61\x79\xEB\x2E\x08\x37\xD7\x23\xF3\x8C\x9F\xBE\x1D\xB1\xE1\xA4\x75\xDB\xA0\xE2\x54\x14\xB1\xBA\x1C\x29\xA4\x18\xF6\x12\xBA\xA2\x14\x14\xE3\x31\x35\xC8\x40\xFF\xB7\xE0\x05\x76\x57\xC1\x1C\x59\xF2\xF8\xBF\xE4\xED\x25\x62\x5C\x84\xF0\x7E\x7E\x1F\xB3\xBE\xF9\xB7\x21\x11\xCC\x03\x01\x56\x70\xA7\x10\x92\x1E\x1B\x34\x81\x1E\xAD\x9C\x1A\xC3\x04\x3C\xED\x02\x61\xD6\x1E\x06\xF3\x5F\x3A\x87\xF2\x2B\xF1\x45\x87\xE5\x3D\xAC\xD1\xC7\x57\x84\xBD\x6B\xAE\xDC\xD8\xF9\xB6\x1B\x62\x70\x0B\x3D\x36\xC9\x42\xF2\x32\xD7\x7A\x61\xE6\xD2\xDB\x3D\xCF\xC8\xA9\xC9\x9B\xDC\xDB\x58\x44\xD7\x6F\x38\xAF\x7F\x78\xD3\xA3\xAD\x1A\x75\xBA\x1C\xC1\x36\x7C\x8F\x1E\x6D\x1C\xC3\x75\x46\xAE\x35\x05\xA6\xF6\x5C\x3D\x21\xEE\x56\xF0\xC9\x82\x22\x2D\x7A\x54\xAB\x70\xC3\x7D\x22\x65\x82\x70\x96"),
   // "DigiCert Assured ID Root CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xB7\x30\x82\x02\x9F\xA0\x03\x02\x01\x02\x02\x10\x0C\xE7\xE0\xE5\x17\xD8\x46\xFE\x8F\xE5\x60\xFC\x1B\xF0\x30\x39\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x44\x69\x67\x69\x43\x65\x72\x74\x20\x41\x73\x73\x75\x72\x65\x64\x20\x49\x44\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x30\x36\x31\x31\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x31\x31\x31\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x44\x69\x67\x69\x43\x65\x72\x74\x20\x41\x73\x73\x75\x72\x65\x64\x20\x49\x44\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xAD\x0E\x15\xCE\xE4\x43\x80\x5C\xB1\x87\xF3\xB7\x60\xF9\x71\x12\xA5\xAE\xDC\x26\x94\x88\xAA\xF4\xCE\xF5\x20\x39\x28\x58\x60\x0C\xF8\x80\xDA\xA9\x15\x95\x32\x61\x3C\xB5\xB1\x28\x84\x8A\x8A\xDC\x9F\x0A\x0C\x83\x17\x7A\x8F\x90\xAC\x8A\xE7\x79\x53\x5C\x31\x84\x2A\xF6\x0F\x98\x32\x36\x76\xCC\xDE\xDD\x3C\xA8\xA2\xEF\x6A\xFB\x21\xF2\x52\x61\xDF\x9F\x20\xD7\x1F\xE2\xB1\xD9\xFE\x18\x64\xD2\x12\x5B\x5F\xF9\x58\x18\x35\xBC\x47\xCD\xA1\x36\xF9\x6B\x7F\xD4\xB0\x38\x3E\xC1\x1B\xC3\x8C\x33\xD9\xD8\x2F\x18\xFE\x28\x0F\xB3\xA7\x83\xD6\xC3\x6E\x44\xC0\x61\x35\x96\x16\xFE\x59\x9C\x8B\x76\x6D\xD7\xF1\xA2\x4B\x0D\x2B\xFF\x0B\x72\xDA\x9E\x60\xD0\x8E\x90\x35\xC6\x78\x55\x87\x20\xA1\xCF\xE5\x6D\x0A\xC8\x49\x7C\x31\x98\x33\x6C\x22\xE9\x87\xD0\x32\x5A\xA2\xBA\x13\x82\x11\xED\x39\x17\x9D\x99\x3A\x72\xA1\xE6\xFA\xA4\xD9\xD5\x17\x31\x75\xAE\x85\x7D\x22\xAE\x3F\x01\x46\x86\xF6\x28\x79\xC8\xB1\xDA\xE4\x57\x17\xC4\x7E\x1C\x0E\xB0\xB4\x92\xA6\x56\xB3\xBD\xB2\x97\xED\xAA\xA7\xF0\xB7\xC5\xA8\x3F\x95\x16\xD0\xFF\xA1\x96\xEB\x08\x5F\x18\x77\x4F\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x45\xEB\xA2\xAF\xF4\x92\xCB\x82\x31\x2D\x51\x8B\xA7\xA7\x21\x9D\xF3\x6D\xC8\x0F\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x45\xEB\xA2\xAF\xF4\x92\xCB\x82\x31\x2D\x51\x8B\xA7\xA7\x21\x9D\xF3\x6D\xC8\x0F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\xA2\x0E\xBC\xDF\xE2\xED\xF0\xE3\x72\x73\x7A\x64\x94\xBF\xF7\x72\x66\xD8\x32\xE4\x42\x75\x62\xAE\x87\xEB\xF2\xD5\xD9\xDE\x56\xB3\x9F\xCC\xCE\x14\x28\xB9\x0D\x97\x60\x5C\x12\x4C\x58\xE4\xD3\x3D\x83\x49\x45\x58\x97\x35\x69\x1A\xA8\x47\xEA\x56\xC6\x79\xAB\x12\xD8\x67\x81\x84\xDF\x7F\x09\x3C\x94\xE6\xB8\x26\x2C\x20\xBD\x3D\xB3\x28\x89\xF7\x5F\xFF\x22\xE2\x97\x84\x1F\xE9\x65\xEF\x87\xE0\xDF\xC1\x67\x49\xB3\x5D\xEB\xB2\x09\x2A\xEB\x26\xED\x78\xBE\x7D\x3F\x2B\xF3\xB7\x26\x35\x6D\x5F\x89\x01\xB6\x49\x5B\x9F\x01\x05\x9B\xAB\x3D\x25\xC1\xCC\xB6\x7F\xC2\xF1\x6F\x86\xC6\xFA\x64\x68\xEB\x81\x2D\x94\xEB\x42\xB7\xFA\x8C\x1E\xDD\x62\xF1\xBE\x50\x67\xB7\x6C\xBD\xF3\xF1\x1F\x6B\x0C\x36\x07\x16\x7F\x37\x7C\xA9\x5B\x6D\x7A\xF1\x12\x46\x60\x83\xD7\x27\x04\xBE\x4B\xCE\x97\xBE\xC3\x67\x2A\x68\x11\xDF\x80\xE7\x0C\x33\x66\xBF\x13\x0D\x14\x6E\xF3\x7F\x1F\x63\x10\x1E\xFA\x8D\x1B\x25\x6D\x6C\x8F\xA5\xB7\x61\x01\xB1\xD2\xA3\x26\xA1\x10\x71\x9D\xAD\xE2\xC3\xF9\xC3\x99\x51\xB7\x2B\x07\x08\xCE\x2E\xE6\x50\xB2\xA7\xFA\x0A\x45\x2F\xA2\xF0\xF2"),
   // "SecureSign Root CA12"
   CertificateDer::from_slice(b"\x30\x82\x03\x72\x30\x82\x02\x5A\xA0\x03\x02\x01\x02\x02\x14\x66\xF9\xC7\xC1\xAF\xEC\xC2\x51\xB4\xED\x53\x97\xE6\xE6\x82\xC3\x2B\x1C\x90\x16\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x13\x1A\x43\x79\x62\x65\x72\x74\x72\x75\x73\x74\x20\x4A\x61\x70\x61\x6E\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x13\x14\x53\x65\x63\x75\x72\x65\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x32\x30\x1E\x17\x0D\x32\x30\x30\x34\x30\x38\x30\x35\x33\x36\x34\x36\x5A\x17\x0D\x34\x30\x30\x34\x30\x38\x30\x35\x33\x36\x34\x36\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x13\x1A\x43\x79\x62\x65\x72\x74\x72\x75\x73\x74\x20\x4A\x61\x70\x61\x6E\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x13\x14\x53\x65\x63\x75\x72\x65\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xBA\x39\xC1\x37\x7A\x68\x45\x2B\x14\xB4\xEB\xE4\x13\xEB\x57\x75\x23\x4D\x8F\x24\x2D\x16\xE8\xAE\x8E\xC9\x7D\xA4\x57\x3B\x2A\x76\x25\x33\x83\x6C\xEA\x32\x8A\x94\x9B\x4E\x3C\x96\xE4\xFD\x51\xBF\x99\xC9\x93\x7E\xBF\xF9\xAD\xA7\xB2\x48\x2B\x07\x1C\x27\xF5\x4C\xBC\x70\x12\x77\xA4\x85\x54\xB5\xFD\x90\x7A\xE4\xA3\xE4\x51\x58\x03\xCD\x10\x79\x79\xEE\x6B\x93\x1F\x64\x8E\x6B\x64\xAB\xA3\x13\xE3\x71\xFE\x7D\xAB\x9C\xDD\x27\x53\x37\xB3\xAA\x18\xC2\x59\x26\xEC\x5B\x1F\xD2\xE6\x65\x7C\xEF\x93\xBD\xD8\x58\x5C\x0B\xC0\xE3\x65\x6F\x3C\xC7\xCA\x59\xE3\xFE\x6E\x5F\xAC\x83\xBE\xFD\x5D\x25\x4E\x2A\x29\x3B\xD6\x0B\xAB\x17\x32\x78\xA4\xE1\x3E\x94\x46\xBE\x62\x6E\x9B\xDE\x46\xA8\xB1\x16\xE7\x85\x6E\xF4\x08\x40\x45\x11\xA0\x9E\x54\x44\x84\xF7\xD8\x36\xCE\xF5\x50\x47\xDC\x2C\x30\x9B\xEE\xC0\xF5\x96\xD2\xFE\x09\x86\xC7\x06\x59\xAE\x4F\xAE\x8E\x11\x98\x7B\xF3\x0B\x52\xAA\x62\x26\xAA\x21\xDF\x8E\x25\x33\x79\x97\x16\x49\x8D\xF5\x3E\xD5\x47\x9F\x37\x31\x49\x33\x72\x05\x4D\x0C\xB6\x55\x8C\xF1\x57\x8F\x8A\x87\xD1\xAD\xC5\x11\x12\x39\xA0\xAD\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x57\x34\xF3\x74\xCF\x04\x4B\xD5\x25\xE6\xF1\x40\xB6\x2C\x4C\xD9\x2D\xE9\xA0\xAD\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x3E\xBB\xDB\x17\x16\xD2\xF2\x14\x01\x20\x2C\x38\x83\x4B\xAD\xBE\xCA\x85\x7A\x9A\xB6\x9B\x6B\xA6\xE1\xFC\xA5\x3A\xAC\xAD\xB4\x28\x3A\xAF\xD7\x01\x83\x49\x2B\x63\xA2\xDD\x9A\x64\x0E\x98\x5C\x6F\xDD\x8E\xBB\x8A\x54\x22\x2D\x4A\x13\xF3\xAE\x40\x43\xDB\x4F\x91\xB7\x86\x1A\xEC\x00\xB4\x41\x81\xA4\x4F\xFA\x6A\x8B\x88\xB3\x76\x08\x72\x2A\x49\x40\xC3\xD3\xC3\x85\x89\x98\x10\xA5\x9D\x6F\x19\xB7\xBB\xCF\x7A\x65\x55\xDB\x37\xEB\x3C\x8A\x72\x32\x97\x1E\x9A\x29\x3E\xAD\x8D\xE6\xA3\x1B\x6D\xF5\x75\x1A\xE6\xB0\x68\xB9\x5B\xA2\xEE\x69\x47\x27\x35\xA1\x86\x99\x80\xF3\x33\x4B\xE1\x6B\xA4\x26\xC3\xEF\x74\x59\x6C\x7A\xA2\x64\xB6\x1E\x44\xC3\x50\xE0\x0F\x39\x3D\xA9\x33\xF1\xA5\xF3\xD2\xBD\x62\x84\xAC\x8E\x1C\xA9\xCD\x5A\xBD\x37\x3B\x6E\x0A\x22\xB4\xF4\x15\xE7\x91\x58\xC5\x3A\x44\xD3\x95\x28\xD9\xC0\x65\xE9\x72\xCA\xD0\x0F\xBD\x1F\xB3\x15\xD9\xA9\xE3\xA4\x47\x09\x9E\xE0\xCB\x37\xFB\xFD\xBD\x97\xD5\xBE\x18\x1A\x69\xA2\x39\x81\xD9\x1A\xF5\xAB\x7F\xC8\xE3\xE2\x67\x0B\x9D\xF4\x0C\xEA\x54\xDF\xD2\xB2\xAF\xB1\x22\xF1\x20\xDF\xBC\x44\x1C"),
   // "TWCA CYBER Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\x8D\x30\x82\x03\x75\xA0\x03\x02\x01\x02\x02\x10\x40\x01\x34\x8C\xC2\x00\x00\x00\x00\x00\x00\x00\x01\x3C\xF2\xC6\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x50\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x54\x41\x49\x57\x41\x4E\x2D\x43\x41\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x13\x07\x52\x6F\x6F\x74\x20\x43\x41\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x54\x57\x43\x41\x20\x43\x59\x42\x45\x52\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x32\x32\x31\x31\x32\x32\x30\x36\x35\x34\x32\x39\x5A\x17\x0D\x34\x37\x31\x31\x32\x32\x31\x35\x35\x39\x35\x39\x5A\x30\x50\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x54\x41\x49\x57\x41\x4E\x2D\x43\x41\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x13\x07\x52\x6F\x6F\x74\x20\x43\x41\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x54\x57\x43\x41\x20\x43\x59\x42\x45\x52\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC6\xF8\xCA\x1E\xD9\x09\x20\x7E\x1D\x6C\x4E\xCE\x8F\xE3\x47\x33\x44\x9C\xC7\xC9\x69\xAA\x3A\x5B\x78\xEE\x70\xD2\x92\xF8\x04\xB3\x52\x52\x1D\x67\x72\x28\xA1\xDF\x8B\x5D\x95\x0A\xFE\xEA\xCD\xED\xF7\x29\xCE\xF0\x6F\x7F\xAC\xCD\x3D\xEF\xB3\x1C\x45\x6A\xF7\x28\x90\xF1\x61\x57\xC5\x0C\xC4\xA3\x50\x5D\xDE\xD4\xB5\xCB\x19\xCA\x80\xB9\x75\xCE\x29\xCE\xD2\x85\x22\xEC\x02\x63\xCC\x44\x30\x20\xDA\xEA\x91\x5B\x56\xE6\x1D\x1C\xD5\x9D\x66\xC7\x3F\xDF\x86\xCA\x4B\x53\xC4\xD9\x8D\xB2\x1D\xEA\xF8\xDC\x27\x53\xA3\x47\xE1\x61\xCC\x7D\xB5\xB0\xF8\xEE\x73\x91\xC5\xCE\x73\x6F\xCE\xEE\x10\x1F\x1A\x06\xCF\xE9\x27\x60\xC5\x4F\x19\xE4\xEB\xCE\x22\x26\x45\xD7\x60\x99\xDD\xCE\x4F\x37\xE0\x7F\xE7\x63\xAD\xB0\xB8\x59\xB8\xD0\x06\x68\x35\x60\xD3\x36\xAE\x71\x43\x04\xF1\x69\x65\x78\x7C\xF3\x1F\xF3\xCA\x28\x9F\x5A\x20\x95\x66\xB4\xCD\xB7\xEE\x8F\x78\xA4\x45\x18\xE9\x26\x2F\x8D\x9B\x29\x28\xB1\xA4\xB7\x3A\x6D\xB9\xD4\x1C\x38\x72\x45\x58\xB1\x5E\xEB\xF0\x28\x9B\xB7\x82\xCA\xFD\xCF\xD6\x33\x0F\x9F\xFB\x97\x9E\xB1\x1C\x9C\x9E\xEA\x5F\x5E\xDB\xAA\xDD\x54\xE9\x30\x21\x28\x6D\x8E\x79\xF3\x75\x92\x8C\x26\xFE\xDC\xC5\xF6\xC3\xB0\xDF\x44\x59\x43\xA3\xB6\x03\x28\xF6\x08\x30\xAA\x0D\x33\xE1\xEF\x9C\xA9\x07\x22\xE3\x59\x5B\x40\x8F\xDA\x88\xB7\x69\x08\xA8\xB7\x23\x2E\x44\x09\x59\x37\x5B\xC7\xE3\x17\xF2\x22\xEB\x6E\x39\x52\xC5\xDE\x54\xA7\x98\xC9\x4B\x20\x95\xDC\x46\x89\x5F\xB4\x12\xF9\x85\x29\x8E\xEB\xC8\x27\x15\x20\xC0\x4B\xD4\xCC\x7C\x0C\x6C\x34\x0C\x26\x9B\x26\x31\xA6\x3C\xA7\xF6\xD9\xD0\x4B\xA2\x64\xFF\x3B\x99\x41\x72\xC1\xE0\x70\x97\xF1\x24\xBB\x2B\xC4\x74\x22\xB1\xAC\x6B\x22\x32\x24\xD3\x78\x2A\xC0\xC0\xA1\x2F\xF1\x52\x05\xC9\x3F\xEF\x76\x66\xE2\x45\xD8\x0D\x3D\xAD\x95\xC8\xC7\x89\x26\xC8\x0F\xAE\xA7\x03\x2E\xFB\xC1\x5F\xFA\x20\xE1\x70\xAD\xB0\x65\x20\x37\x33\x60\xB0\xD5\xAF\xD7\x0C\x1C\xC2\x90\x70\xD7\x4A\x18\xBC\x7E\x01\xB0\xB0\xEB\x15\x1E\x44\x06\xCD\xA4\x4F\xE8\x0C\xD1\xC3\x20\x10\xE1\x54\x65\x9E\xB6\x51\xD0\x1A\x76\x6B\x42\x5A\x58\x76\x34\xEA\xB7\x37\x19\xAE\x2E\x75\xF9\x96\xE5\xC1\x59\xF7\x94\x57\x29\x25\x8D\x3A\x4C\xAB\x4D\x9A\x41\xD0\x5F\x26\x03\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x9D\x85\x61\x14\x7C\xC1\x62\x6F\x97\x68\xE4\x4F\x37\x40\xE1\xAD\xE0\x0D\x56\x37\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x9D\x85\x61\x14\x7C\xC1\x62\x6F\x97\x68\xE4\x4F\x37\x40\xE1\xAD\xE0\x0D\x56\x37\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x64\x8F\x7A\xC4\x62\x0E\xB5\x88\xCC\xB8\xC7\x86\x0E\xA1\x4A\x16\xCD\x70\x0B\xB7\xA7\x85\x0B\xB3\x76\xB6\x0F\xA7\xFF\x08\x8B\x0B\x25\xCF\xA8\xD4\x83\x75\x2A\xB8\x96\x88\xB6\xFB\xDF\x2D\x2D\xB4\x69\x53\x21\x35\x57\xD6\x89\x4D\x73\xBF\x69\x8F\x70\xA3\x61\xCC\x9A\xDB\x1E\x9A\xE0\x20\xF8\x6C\xBB\x9B\x22\x9D\x5D\x84\x31\x9A\x2C\x8A\xDD\x6A\xA1\xD7\x28\x69\xCA\xFE\x76\x55\x7A\x46\x67\xEB\xCC\x43\x88\x16\xA2\x03\xD6\xB9\x17\xF8\x19\x6C\x6D\x23\x02\x7F\xF1\x5F\xD0\x0A\x29\x23\x3B\xD1\xAA\x0A\xED\xA9\x17\x26\x54\x0A\x4D\xC2\xA5\x4D\xF8\xC5\xFD\xB8\x81\xCF\x2B\x2C\x78\xA3\x67\x4C\xA9\x07\x9A\xF3\xDF\x5E\xFB\x7C\xF5\x89\xCD\x74\x97\x61\x10\x6A\x07\x2B\x81\x5A\xD2\x8E\xB7\xE7\x20\xD1\x20\x6E\x24\xA8\x84\x27\xA1\x57\xAC\xAA\x55\x58\x2F\xDC\xD9\xCA\xFA\x68\x04\x9E\xED\x44\x24\xF9\x74\x40\x3B\x23\x33\xAB\x83\x5A\x18\x26\x42\xB6\x6D\x54\xB5\x16\x60\x30\x6C\xB1\xA0\xF8\xB8\x41\xA0\x5D\x49\x49\xD2\x65\x05\x3A\xEA\xFE\x9D\x61\xBC\x86\xD9\xBF\xDE\xD3\xBA\x3A\xB1\x7F\x7E\x92\x34\x8E\xC9\x00\x6E\xDC\x98\xBD\xDC\xEC\x80\x05\xAD\x02\x3D\xDF\x65\xED\x0B\x03\xF7\xF7\x16\x84\x04\x31\xBA\x93\x94\xD8\xF2\x12\xF8\x8A\xE3\xBF\x42\xAF\xA7\xD4\xCD\x11\x17\x16\xC8\x42\x1D\x14\xA8\x42\xF6\xD2\x40\x86\xA0\x4F\x23\xCA\x96\x45\x56\x60\x06\xCD\xB7\x55\x01\xA6\x01\x94\x65\xFE\x6E\x05\x09\xBA\xB4\xA4\xAA\xE2\xEF\x58\xBE\xBD\x27\x56\xD8\xEF\x73\x71\x5B\x44\x33\xF2\x9A\x72\xEA\xB0\x5E\x3E\x6E\xA9\x52\x5B\xEC\x70\x6D\xB5\x87\x8F\x37\x5E\x3C\x8C\x9C\xCE\xE4\xF0\xCE\x0C\x67\x41\xCC\xCE\xF6\x80\xAB\x4E\xCC\x4C\x56\xF5\xC1\x61\x59\x93\xB4\x3E\xA6\xDA\xB8\x37\x12\x9F\x2A\x32\xE3\x8B\xB8\x21\xEC\xC3\x2B\x65\x0C\xEF\x22\xDE\x88\x29\x3B\x4C\xD7\xFA\xFE\xB7\xE1\x47\xBE\x9C\x3E\x3E\x83\xFB\x51\x5D\xF5\x68\xF7\x2E\x21\x85\xDC\xBF\xF1\x5A\xE2\x7C\xD7\xC5\xE4\x83\xC1\x6A\xEB\xBA\x80\x5A\xDE\x5C\x2D\x70\x76\xF8\xC8\xE5\x87\x87\xCA\xA0\x9D\xA1\xE5\x22\x12\x27\x0F\x44\x3D\x1D\x6C\xEA\xD4\xC2\x8B\x2F\x6F\x79\xAB\x7F\x50\xA6\xC4\x19\xA7\xA1\x7A\xB7\x96\xF9\xC1\x1F\x62\x5A\xA2\x43\x07\x40\x5E\x26\xC6\xAC\xED\xAE\x70\x16\xC5\xAA\xCA\x72\x8A\x4D\xB0\xCF\x01\x8B\x03\x3F\x6E\xD7"),
   // "HARICA TLS ECC Root CA 2021"
   CertificateDer::from_slice(b"\x30\x82\x02\x54\x30\x82\x01\xDB\xA0\x03\x02\x01\x02\x02\x10\x67\x74\x9D\x8D\x77\xD8\x3B\x6A\xDB\x22\xF4\xFF\x59\xE2\xBF\xCE\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x6C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x37\x30\x35\x06\x03\x55\x04\x0A\x0C\x2E\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x41\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x48\x41\x52\x49\x43\x41\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x31\x30\x1E\x17\x0D\x32\x31\x30\x32\x31\x39\x31\x31\x30\x31\x31\x30\x5A\x17\x0D\x34\x35\x30\x32\x31\x33\x31\x31\x30\x31\x30\x39\x5A\x30\x6C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x37\x30\x35\x06\x03\x55\x04\x0A\x0C\x2E\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x41\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x48\x41\x52\x49\x43\x41\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x31\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x38\x08\xFE\xB1\xA0\x96\xD2\x7A\xAC\xAF\x49\x3A\xD0\xC0\xE0\xC3\x3B\x28\xAA\xF1\x72\x6D\x65\x00\x47\x88\x84\xFC\x9A\x26\x6B\xAA\x4B\xBA\x6C\x04\x0A\x88\x5E\x17\xF2\x55\x87\xFC\x30\xB0\x34\xE2\x34\x58\x57\x1A\x84\x53\xE9\x30\xD9\xA9\xF2\x96\x74\xC3\x51\x1F\x58\x49\x31\xCC\x98\x4E\x60\x11\x87\x75\xD3\x72\x94\x90\x4F\x9B\x10\x25\x2A\xA8\x78\x2D\xBE\x90\x41\x58\x90\x15\x72\xA7\xA1\xB7\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC9\x1B\x53\x81\x12\xFE\x04\xD5\x16\xD1\xAA\xBC\x9A\x6F\xB7\xA0\x95\x19\x6E\xCA\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x67\x00\x30\x64\x02\x30\x11\xDE\xAE\xF8\xDC\x4E\x88\xB0\xA9\xF0\x22\xAD\xC2\x51\x40\xEF\x60\x71\x2D\xEE\x8F\x02\xC4\x5D\x03\x70\x49\xA4\x92\xEA\xC5\x14\x88\x70\xA6\xD3\x0D\xB0\xAA\xCA\x2C\x40\x9C\xFB\xE9\x82\x6E\x9A\x02\x30\x2B\x47\x9A\x07\xC6\xD1\xC2\x81\x7C\xCA\x0B\x96\x18\x41\x1B\xA3\xF4\x30\x09\x9E\xB5\x23\x28\x0D\x9F\x14\xB6\x3C\x53\xA2\x4C\x06\x69\x7D\xFA\x6C\x91\xC6\x2A\x49\x45\xE6\xEC\xB7\x13\xE1\x3A\x6C"),
   // "emSign Root CA - G1"
   CertificateDer::from_slice(b"\x30\x82\x03\x94\x30\x82\x02\x7C\xA0\x03\x02\x01\x02\x02\x0A\x31\xF5\xE4\x62\x0C\x6C\x58\xED\xD6\xD8\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x67\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x49\x4E\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x65\x4D\x75\x64\x68\x72\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x65\x6D\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x47\x31\x30\x1E\x17\x0D\x31\x38\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x17\x0D\x34\x33\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x30\x67\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x49\x4E\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x65\x4D\x75\x64\x68\x72\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x65\x6D\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x47\x31\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\x93\x4B\xBB\xE9\x66\x8A\xEE\x9D\x5B\xD5\x34\x93\xD0\x1B\x1E\xC3\xE7\x9E\xB8\x64\x33\x7F\x63\x78\x68\xB4\xCD\x2E\x71\x75\xD7\x9B\x20\xC6\x4D\x29\xBC\xB6\x68\x60\x8A\xF7\x21\x9A\x56\x35\x5A\xF3\x76\xBD\xD8\xCD\x9A\xFF\x93\x56\x4B\xA5\x59\x06\xA1\x93\x34\x29\xDD\x16\x34\x75\x4E\xF2\x81\xB4\xC7\x96\x4E\xAD\x19\x15\x52\x4A\xFE\x3C\x70\x75\x70\xCD\xAF\x2B\xAB\x15\x9A\x33\x3C\xAA\xB3\x8B\xAA\xCD\x43\xFD\xF5\xEA\x70\xFF\xED\xCF\x11\x3B\x94\xCE\x4E\x32\x16\xD3\x23\x40\x2A\x77\xB3\xAF\x3C\x01\x2C\x6C\xED\x99\x2C\x8B\xD9\x4E\x69\x98\xB2\xF7\x8F\x41\xB0\x32\x78\x61\xD6\x0D\x5F\xC3\xFA\xA2\x40\x92\x1D\x5C\x17\xE6\x70\x3E\x35\xE7\xA2\xB7\xC2\x62\xE2\xAB\xA4\x38\x4C\xB5\x39\x35\x6F\xEA\x03\x69\xFA\x3A\x54\x68\x85\x6D\xD6\xF2\x2F\x43\x55\x1E\x91\x0D\x0E\xD8\xD5\x6A\xA4\x96\xD1\x13\x3C\x2C\x78\x50\xE8\x3A\x92\xD2\x17\x56\xE5\x35\x1A\x40\x1C\x3E\x8D\x2C\xED\x39\xDF\x42\xE0\x83\x41\x74\xDF\xA3\xCD\xC2\x86\x60\x48\x68\xE3\x69\x0B\x54\x00\x8B\xE4\x76\x69\x21\x0D\x79\x4E\x34\x08\x5E\x14\xC2\xCC\xB1\xB7\xAD\xD7\x7C\x70\x8A\xC7\x85\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xFB\xEF\x0D\x86\x9E\xB0\xE3\xDD\xA9\xB9\xF1\x21\x17\x7F\x3E\xFC\xF0\x77\x2B\x1A\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x59\xFF\xF2\x8C\xF5\x87\x7D\x71\x3D\xA3\x9F\x1B\x5B\xD1\xDA\xF8\xD3\x9C\x6B\x36\xBD\x9B\xA9\x61\xEB\xDE\x16\x2C\x74\x3D\x9E\xE6\x75\xDA\xD7\xBA\xA7\xBC\x42\x17\xE7\x3D\x91\xEB\xE5\x7D\xDD\x3E\x9C\xF1\xCF\x92\xAC\x6C\x48\xCC\xC2\x22\x3F\x69\x3B\xC5\xB6\x15\x2F\xA3\x35\xC6\x68\x2A\x1C\x57\xAF\x39\xEF\x8D\xD0\x35\xC3\x18\x0C\x7B\x00\x56\x1C\xCD\x8B\x19\x74\xDE\xBE\x0F\x12\xE0\xD0\xAA\xA1\x3F\x02\x34\xB1\x70\xCE\x9D\x18\xD6\x08\x03\x09\x46\xEE\x60\xE0\x7E\xB6\xC4\x49\x04\x51\x7D\x70\x60\xBC\xAA\xB2\xFF\x79\x72\x7A\xA6\x1D\x3D\x5F\x2A\xF8\xCA\xE2\xFD\x39\xB7\x47\xB9\xEB\x7E\xDF\x04\x23\xAF\xFA\x9C\x06\x07\xE9\xFB\x63\x93\x80\x40\xB5\xC6\x6C\x0A\x31\x28\xCE\x0C\x9F\xCF\xB3\x23\x35\x80\x41\x8D\x6C\xC4\x37\x7B\x81\x2F\x80\xA1\x40\x42\x85\xE9\xD9\x38\x8D\xE8\xA1\x53\xCD\x01\xBF\x69\xE8\x5A\x06\xF2\x45\x0B\x90\xFA\xAE\xE1\xBF\x9D\xF2\xAE\x57\x3C\xA5\xAE\xB2\x56\xF4\x8B\x65\x40\xE9\xFD\x31\x81\x2C\xF4\x39\x09\xD8\xEE\x6B\xA7\xB4\xA6\x1D\x15\xA5\x98\xF7\x01\x81\xD8\x85\x7D\xF3\x51\x5C\x71\x88\xDE\xBA\xCC\x1F\x80\x7E\x4A"),
   // "Secure Global CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xBC\x30\x82\x02\xA4\xA0\x03\x02\x01\x02\x02\x10\x07\x56\x22\xA4\xE8\xD4\x8A\x89\x4D\xF4\x13\xC8\xF0\xF8\xEA\xA5\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x4A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x20\x30\x1E\x06\x03\x55\x04\x0A\x13\x17\x53\x65\x63\x75\x72\x65\x54\x72\x75\x73\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x53\x65\x63\x75\x72\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x43\x41\x30\x1E\x17\x0D\x30\x36\x31\x31\x30\x37\x31\x39\x34\x32\x32\x38\x5A\x17\x0D\x32\x39\x31\x32\x33\x31\x31\x39\x35\x32\x30\x36\x5A\x30\x4A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x20\x30\x1E\x06\x03\x55\x04\x0A\x13\x17\x53\x65\x63\x75\x72\x65\x54\x72\x75\x73\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x53\x65\x63\x75\x72\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xAF\x35\x2E\xD8\xAC\x6C\x55\x69\x06\x71\xE5\x13\x68\x24\xB3\x4F\xD8\xCC\x21\x47\xF8\xF1\x60\x38\x89\x89\x03\xE9\xBD\xEA\x5E\x46\x53\x09\xDC\x5C\xF5\x5A\xE8\xF7\x45\x2A\x02\xEB\x31\x61\xD7\x29\x33\x4C\xCE\xC7\x7C\x0A\x37\x7E\x0F\xBA\x32\x98\xE1\x1D\x97\xAF\x8F\xC7\xDC\xC9\x38\x96\xF3\xDB\x1A\xFC\x51\xED\x68\xC6\xD0\x6E\xA4\x7C\x24\xD1\xAE\x42\xC8\x96\x50\x63\x2E\xE0\xFE\x75\xFE\x98\xA7\x5F\x49\x2E\x95\xE3\x39\x33\x64\x8E\x1E\xA4\x5F\x90\xD2\x67\x3C\xB2\xD9\xFE\x41\xB9\x55\xA7\x09\x8E\x72\x05\x1E\x8B\xDD\x44\x85\x82\x42\xD0\x49\xC0\x1D\x60\xF0\xD1\x17\x2C\x95\xEB\xF6\xA5\xC1\x92\xA3\xC5\xC2\xA7\x08\x60\x0D\x60\x04\x10\x96\x79\x9E\x16\x34\xE6\xA9\xB6\xFA\x25\x45\x39\xC8\x1E\x65\xF9\x93\xF5\xAA\xF1\x52\xDC\x99\x98\x3D\xA5\x86\x1A\x0C\x35\x33\xFA\x4B\xA5\x04\x06\x15\x1C\x31\x80\xEF\xAA\x18\x6B\xC2\x7B\xD7\xDA\xCE\xF9\x33\x20\xD5\xF5\xBD\x6A\x33\x2D\x81\x04\xFB\xB0\x5C\xD4\x9C\xA3\xE2\x5C\x1D\xE3\xA9\x42\x75\x5E\x7B\xD4\x77\xEF\x39\x54\xBA\xC9\x0A\x18\x1B\x12\x99\x49\x2F\x88\x4B\xFD\x50\x62\xD1\x73\xE7\x8F\x7A\x43\x02\x03\x01\x00\x01\xA3\x81\x9D\x30\x81\x9A\x30\x13\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x14\x02\x04\x06\x1E\x04\x00\x43\x00\x41\x30\x0B\x06\x03\x55\x1D\x0F\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xAF\x44\x04\xC2\x41\x7E\x48\x83\xDB\x4E\x39\x02\xEC\xEC\x84\x7A\xE6\xCE\xC9\xA4\x30\x34\x06\x03\x55\x1D\x1F\x04\x2D\x30\x2B\x30\x29\xA0\x27\xA0\x25\x86\x23\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x73\x65\x63\x75\x72\x65\x74\x72\x75\x73\x74\x2E\x63\x6F\x6D\x2F\x53\x47\x43\x41\x2E\x63\x72\x6C\x30\x10\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x15\x01\x04\x03\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x63\x1A\x08\x40\x7D\xA4\x5E\x53\x0D\x77\xD8\x7A\xAE\x1F\x0D\x0B\x51\x16\x03\xEF\x18\x7C\xC8\xE3\xAF\x6A\x58\x93\x14\x60\x91\xB2\x84\xDC\x88\x4E\xBE\x39\x8A\x3A\xF3\xE6\x82\x89\x5D\x01\x37\xB3\xAB\x24\xA4\x15\x0E\x92\x35\x5A\x4A\x44\x5E\x4E\x57\xFA\x75\xCE\x1F\x48\xCE\x66\xF4\x3C\x40\x26\x92\x98\x6C\x1B\xEE\x24\x46\x0C\x17\xB3\x52\xA5\xDB\xA5\x91\x91\xCF\x37\xD3\x6F\xE7\x27\x08\x3A\x4E\x19\x1F\x3A\xA7\x58\x5C\x17\xCF\x79\x3F\x8B\xE4\xA7\xD3\x26\x23\x9D\x26\x0F\x58\x69\xFC\x47\x7E\xB2\xD0\x8D\x8B\x93\xBF\x29\x4F\x43\x69\x74\x76\x67\x4B\xCF\x07\x8C\xE6\x02\xF7\xB5\xE1\xB4\x43\xB5\x4B\x2D\x14\x9F\xF9\xDC\x26\x0D\xBF\xA6\x47\x74\x06\xD8\x88\xD1\x3A\x29\x30\x84\xCE\xD2\x39\x80\x62\x1B\xA8\xC7\x57\x49\xBC\x6A\x55\x51\x67\x15\x4A\xBE\x35\x07\xE4\xD5\x75\x98\x37\x79\x30\x14\xDB\x29\x9D\x6C\xC5\x69\xCC\x47\x55\xA2\x30\xF7\xCC\x5C\x7F\xC2\xC3\x98\x1C\x6B\x4E\x16\x80\xEB\x7A\x78\x65\x45\xA2\x00\x1A\xAF\x0C\x0D\x55\x64\x34\x48\xB8\x92\xB9\xF1\xB4\x50\x29\xF2\x4F\x23\x1F\xDA\x6C\xAC\x1F\x44\xE1\xDD\x23\x78\x51\x5B\xC7\x16"),
   // "DigiCert Global Root CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xAF\x30\x82\x02\x97\xA0\x03\x02\x01\x02\x02\x10\x08\x3B\xE0\x56\x90\x42\x46\xB1\xA1\x75\x6A\xC9\x59\x91\xC7\x4A\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x44\x69\x67\x69\x43\x65\x72\x74\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x30\x36\x31\x31\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x31\x31\x31\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x44\x69\x67\x69\x43\x65\x72\x74\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xE2\x3B\xE1\x11\x72\xDE\xA8\xA4\xD3\xA3\x57\xAA\x50\xA2\x8F\x0B\x77\x90\xC9\xA2\xA5\xEE\x12\xCE\x96\x5B\x01\x09\x20\xCC\x01\x93\xA7\x4E\x30\xB7\x53\xF7\x43\xC4\x69\x00\x57\x9D\xE2\x8D\x22\xDD\x87\x06\x40\x00\x81\x09\xCE\xCE\x1B\x83\xBF\xDF\xCD\x3B\x71\x46\xE2\xD6\x66\xC7\x05\xB3\x76\x27\x16\x8F\x7B\x9E\x1E\x95\x7D\xEE\xB7\x48\xA3\x08\xDA\xD6\xAF\x7A\x0C\x39\x06\x65\x7F\x4A\x5D\x1F\xBC\x17\xF8\xAB\xBE\xEE\x28\xD7\x74\x7F\x7A\x78\x99\x59\x85\x68\x6E\x5C\x23\x32\x4B\xBF\x4E\xC0\xE8\x5A\x6D\xE3\x70\xBF\x77\x10\xBF\xFC\x01\xF6\x85\xD9\xA8\x44\x10\x58\x32\xA9\x75\x18\xD5\xD1\xA2\xBE\x47\xE2\x27\x6A\xF4\x9A\x33\xF8\x49\x08\x60\x8B\xD4\x5F\xB4\x3A\x84\xBF\xA1\xAA\x4A\x4C\x7D\x3E\xCF\x4F\x5F\x6C\x76\x5E\xA0\x4B\x37\x91\x9E\xDC\x22\xE6\x6D\xCE\x14\x1A\x8E\x6A\xCB\xFE\xCD\xB3\x14\x64\x17\xC7\x5B\x29\x9E\x32\xBF\xF2\xEE\xFA\xD3\x0B\x42\xD4\xAB\xB7\x41\x32\xDA\x0C\xD4\xEF\xF8\x81\xD5\xBB\x8D\x58\x3F\xB5\x1B\xE8\x49\x28\xA2\x70\xDA\x31\x04\xDD\xF7\xB2\x16\xF2\x4C\x0A\x4E\x07\xA8\xED\x4A\x3D\x5E\xB5\x7F\xA3\x90\xC3\xAF\x27\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x03\xDE\x50\x35\x56\xD1\x4C\xBB\x66\xF0\xA3\xE2\x1B\x1B\xC3\x97\xB2\x3D\xD1\x55\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x03\xDE\x50\x35\x56\xD1\x4C\xBB\x66\xF0\xA3\xE2\x1B\x1B\xC3\x97\xB2\x3D\xD1\x55\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\xCB\x9C\x37\xAA\x48\x13\x12\x0A\xFA\xDD\x44\x9C\x4F\x52\xB0\xF4\xDF\xAE\x04\xF5\x79\x79\x08\xA3\x24\x18\xFC\x4B\x2B\x84\xC0\x2D\xB9\xD5\xC7\xFE\xF4\xC1\x1F\x58\xCB\xB8\x6D\x9C\x7A\x74\xE7\x98\x29\xAB\x11\xB5\xE3\x70\xA0\xA1\xCD\x4C\x88\x99\x93\x8C\x91\x70\xE2\xAB\x0F\x1C\xBE\x93\xA9\xFF\x63\xD5\xE4\x07\x60\xD3\xA3\xBF\x9D\x5B\x09\xF1\xD5\x8E\xE3\x53\xF4\x8E\x63\xFA\x3F\xA7\xDB\xB4\x66\xDF\x62\x66\xD6\xD1\x6E\x41\x8D\xF2\x2D\xB5\xEA\x77\x4A\x9F\x9D\x58\xE2\x2B\x59\xC0\x40\x23\xED\x2D\x28\x82\x45\x3E\x79\x54\x92\x26\x98\xE0\x80\x48\xA8\x37\xEF\xF0\xD6\x79\x60\x16\xDE\xAC\xE8\x0E\xCD\x6E\xAC\x44\x17\x38\x2F\x49\xDA\xE1\x45\x3E\x2A\xB9\x36\x53\xCF\x3A\x50\x06\xF7\x2E\xE8\xC4\x57\x49\x6C\x61\x21\x18\xD5\x04\xAD\x78\x3C\x2C\x3A\x80\x6B\xA7\xEB\xAF\x15\x14\xE9\xD8\x89\xC1\xB9\x38\x6C\xE2\x91\x6C\x8A\xFF\x64\xB9\x77\x25\x57\x30\xC0\x1B\x24\xA3\xE1\xDC\xE9\xDF\x47\x7C\xB5\xB4\x24\x08\x05\x30\xEC\x2D\xBD\x0B\xBF\x45\xBF\x50\xB9\xA9\xF3\xEB\x98\x01\x12\xAD\xC8\x88\xC6\x98\x34\x5F\x8D\x0A\x3C\xC6\xE9\xD5\x95\x95\x6D\xDE"),
   // "Hellenic Academic and Research Institutions ECC RootCA 2015"
   CertificateDer::from_slice(b"\x30\x82\x02\xC3\x30\x82\x02\x4A\xA0\x03\x02\x01\x02\x02\x01\x00\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x81\xAA\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x0F\x30\x0D\x06\x03\x55\x04\x07\x13\x06\x41\x74\x68\x65\x6E\x73\x31\x44\x30\x42\x06\x03\x55\x04\x0A\x13\x3B\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x65\x72\x74\x2E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x44\x30\x42\x06\x03\x55\x04\x03\x13\x3B\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x43\x41\x20\x32\x30\x31\x35\x30\x1E\x17\x0D\x31\x35\x30\x37\x30\x37\x31\x30\x33\x37\x31\x32\x5A\x17\x0D\x34\x30\x30\x36\x33\x30\x31\x30\x33\x37\x31\x32\x5A\x30\x81\xAA\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x0F\x30\x0D\x06\x03\x55\x04\x07\x13\x06\x41\x74\x68\x65\x6E\x73\x31\x44\x30\x42\x06\x03\x55\x04\x0A\x13\x3B\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x65\x72\x74\x2E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x44\x30\x42\x06\x03\x55\x04\x03\x13\x3B\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x43\x41\x20\x32\x30\x31\x35\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x92\xA0\x41\xE8\x4B\x82\x84\x5C\xE2\xF8\x31\x11\x99\x86\x64\x4E\x09\x25\x2F\x9D\x41\x2F\x0A\xAE\x35\x4F\x74\x95\xB2\x51\x64\x6B\x8D\x6B\xE6\x3F\x70\x95\xF0\x05\x44\x47\xA6\x72\x38\x50\x76\x95\x02\x5A\x8E\xAE\x28\x9E\xF9\x2D\x4E\x99\xEF\x2C\x48\x6F\x4C\x25\x29\xE8\xD1\x71\x5B\xDF\x1D\xC1\x75\x37\xB4\xD7\xFA\x7B\x7A\x42\x9C\x6A\x0A\x56\x5A\x7C\x69\x0B\xAA\x80\x09\x24\x6C\x7E\xC1\x46\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB4\x22\x0B\x82\x99\x24\x01\x0E\x9C\xBB\xE4\x0E\xFD\xBF\xFB\x97\x20\x93\x99\x2A\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x67\x00\x30\x64\x02\x30\x67\xCE\x16\x62\x38\xA2\xAC\x62\x45\xA7\xA9\x95\x24\xC0\x1A\x27\x9C\x32\x3B\xC0\xC0\xD5\xBA\xA9\xE7\xF8\x04\x43\x53\x85\xEE\x52\x21\xDE\x9D\xF5\x25\x83\x3E\x9E\x58\x4B\x2F\xD7\x67\x13\x0E\x21\x02\x30\x05\xE1\x75\x01\xDE\x68\xED\x2A\x1F\x4D\x4C\x09\x08\x0D\xEC\x4B\xAD\x64\x17\x28\xE7\x75\xCE\x45\x65\x72\x21\x17\xCB\x22\x41\x0E\x8C\x13\x98\x38\x9A\x54\x6D\x9B\xCA\xE2\x7C\xEA\x02\x58\x22\x91"),
   // "Go Daddy Root Certificate Authority - G2"
   CertificateDer::from_slice(b"\x30\x82\x03\xC5\x30\x82\x02\xAD\xA0\x03\x02\x01\x02\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x83\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x10\x30\x0E\x06\x03\x55\x04\x08\x13\x07\x41\x72\x69\x7A\x6F\x6E\x61\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x53\x63\x6F\x74\x74\x73\x64\x61\x6C\x65\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x47\x6F\x44\x61\x64\x64\x79\x2E\x63\x6F\x6D\x2C\x20\x49\x6E\x63\x2E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x13\x28\x47\x6F\x20\x44\x61\x64\x64\x79\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x2D\x20\x47\x32\x30\x1E\x17\x0D\x30\x39\x30\x39\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x37\x31\x32\x33\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x83\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x10\x30\x0E\x06\x03\x55\x04\x08\x13\x07\x41\x72\x69\x7A\x6F\x6E\x61\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x53\x63\x6F\x74\x74\x73\x64\x61\x6C\x65\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x47\x6F\x44\x61\x64\x64\x79\x2E\x63\x6F\x6D\x2C\x20\x49\x6E\x63\x2E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x13\x28\x47\x6F\x20\x44\x61\x64\x64\x79\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x2D\x20\x47\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xBF\x71\x62\x08\xF1\xFA\x59\x34\xF7\x1B\xC9\x18\xA3\xF7\x80\x49\x58\xE9\x22\x83\x13\xA6\xC5\x20\x43\x01\x3B\x84\xF1\xE6\x85\x49\x9F\x27\xEA\xF6\x84\x1B\x4E\xA0\xB4\xDB\x70\x98\xC7\x32\x01\xB1\x05\x3E\x07\x4E\xEE\xF4\xFA\x4F\x2F\x59\x30\x22\xE7\xAB\x19\x56\x6B\xE2\x80\x07\xFC\xF3\x16\x75\x80\x39\x51\x7B\xE5\xF9\x35\xB6\x74\x4E\xA9\x8D\x82\x13\xE4\xB6\x3F\xA9\x03\x83\xFA\xA2\xBE\x8A\x15\x6A\x7F\xDE\x0B\xC3\xB6\x19\x14\x05\xCA\xEA\xC3\xA8\x04\x94\x3B\x46\x7C\x32\x0D\xF3\x00\x66\x22\xC8\x8D\x69\x6D\x36\x8C\x11\x18\xB7\xD3\xB2\x1C\x60\xB4\x38\xFA\x02\x8C\xCE\xD3\xDD\x46\x07\xDE\x0A\x3E\xEB\x5D\x7C\xC8\x7C\xFB\xB0\x2B\x53\xA4\x92\x62\x69\x51\x25\x05\x61\x1A\x44\x81\x8C\x2C\xA9\x43\x96\x23\xDF\xAC\x3A\x81\x9A\x0E\x29\xC5\x1C\xA9\xE9\x5D\x1E\xB6\x9E\x9E\x30\x0A\x39\xCE\xF1\x88\x80\xFB\x4B\x5D\xCC\x32\xEC\x85\x62\x43\x25\x34\x02\x56\x27\x01\x91\xB4\x3B\x70\x2A\x3F\x6E\xB1\xE8\x9C\x88\x01\x7D\x9F\xD4\xF9\xDB\x53\x6D\x60\x9D\xBF\x2C\xE7\x58\xAB\xB8\x5F\x46\xFC\xCE\xC4\x1B\x03\x3C\x09\xEB\x49\x31\x5C\x69\x46\xB3\xE0\x47\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x3A\x9A\x85\x07\x10\x67\x28\xB6\xEF\xF6\xBD\x05\x41\x6E\x20\xC1\x94\xDA\x0F\xDE\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x99\xDB\x5D\x79\xD5\xF9\x97\x59\x67\x03\x61\xF1\x7E\x3B\x06\x31\x75\x2D\xA1\x20\x8E\x4F\x65\x87\xB4\xF7\xA6\x9C\xBC\xD8\xE9\x2F\xD0\xDB\x5A\xEE\xCF\x74\x8C\x73\xB4\x38\x42\xDA\x05\x7B\xF8\x02\x75\xB8\xFD\xA5\xB1\xD7\xAE\xF6\xD7\xDE\x13\xCB\x53\x10\x7E\x8A\x46\xD1\x97\xFA\xB7\x2E\x2B\x11\xAB\x90\xB0\x27\x80\xF9\xE8\x9F\x5A\xE9\x37\x9F\xAB\xE4\xDF\x6C\xB3\x85\x17\x9D\x3D\xD9\x24\x4F\x79\x91\x35\xD6\x5F\x04\xEB\x80\x83\xAB\x9A\x02\x2D\xB5\x10\xF4\xD8\x90\xC7\x04\x73\x40\xED\x72\x25\xA0\xA9\x9F\xEC\x9E\xAB\x68\x12\x99\x57\xC6\x8F\x12\x3A\x09\xA4\xBD\x44\xFD\x06\x15\x37\xC1\x9B\xE4\x32\xA3\xED\x38\xE8\xD8\x64\xF3\x2C\x7E\x14\xFC\x02\xEA\x9F\xCD\xFF\x07\x68\x17\xDB\x22\x90\x38\x2D\x7A\x8D\xD1\x54\xF1\x69\xE3\x5F\x33\xCA\x7A\x3D\x7B\x0A\xE3\xCA\x7F\x5F\x39\xE5\xE2\x75\xBA\xC5\x76\x18\x33\xCE\x2C\xF0\x2F\x4C\xAD\xF7\xB1\xE7\xCE\x4F\xA8\xC4\x9B\x4A\x54\x06\xC5\x7F\x7D\xD5\x08\x0F\xE2\x1C\xFE\x7E\x17\xB8\xAC\x5E\xF6\xD4\x16\xB2\x43\x09\x0C\x4D\xF6\xA7\x6B\xB4\x99\x84\x65\xCA\x7A\x88\xE2\xE2\x44\xBE\x5C\xF7\xEA\x1C\xF5"),
   // "TUBITAK Kamu SM SSL Kok Sertifikasi - Surum 1"
   CertificateDer::from_slice(b"\x30\x82\x04\x63\x30\x82\x03\x4B\xA0\x03\x02\x01\x02\x02\x01\x01\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\xD2\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x52\x31\x18\x30\x16\x06\x03\x55\x04\x07\x13\x0F\x47\x65\x62\x7A\x65\x20\x2D\x20\x4B\x6F\x63\x61\x65\x6C\x69\x31\x42\x30\x40\x06\x03\x55\x04\x0A\x13\x39\x54\x75\x72\x6B\x69\x79\x65\x20\x42\x69\x6C\x69\x6D\x73\x65\x6C\x20\x76\x65\x20\x54\x65\x6B\x6E\x6F\x6C\x6F\x6A\x69\x6B\x20\x41\x72\x61\x73\x74\x69\x72\x6D\x61\x20\x4B\x75\x72\x75\x6D\x75\x20\x2D\x20\x54\x55\x42\x49\x54\x41\x4B\x31\x2D\x30\x2B\x06\x03\x55\x04\x0B\x13\x24\x4B\x61\x6D\x75\x20\x53\x65\x72\x74\x69\x66\x69\x6B\x61\x73\x79\x6F\x6E\x20\x4D\x65\x72\x6B\x65\x7A\x69\x20\x2D\x20\x4B\x61\x6D\x75\x20\x53\x4D\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x54\x55\x42\x49\x54\x41\x4B\x20\x4B\x61\x6D\x75\x20\x53\x4D\x20\x53\x53\x4C\x20\x4B\x6F\x6B\x20\x53\x65\x72\x74\x69\x66\x69\x6B\x61\x73\x69\x20\x2D\x20\x53\x75\x72\x75\x6D\x20\x31\x30\x1E\x17\x0D\x31\x33\x31\x31\x32\x35\x30\x38\x32\x35\x35\x35\x5A\x17\x0D\x34\x33\x31\x30\x32\x35\x30\x38\x32\x35\x35\x35\x5A\x30\x81\xD2\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x52\x31\x18\x30\x16\x06\x03\x55\x04\x07\x13\x0F\x47\x65\x62\x7A\x65\x20\x2D\x20\x4B\x6F\x63\x61\x65\x6C\x69\x31\x42\x30\x40\x06\x03\x55\x04\x0A\x13\x39\x54\x75\x72\x6B\x69\x79\x65\x20\x42\x69\x6C\x69\x6D\x73\x65\x6C\x20\x76\x65\x20\x54\x65\x6B\x6E\x6F\x6C\x6F\x6A\x69\x6B\x20\x41\x72\x61\x73\x74\x69\x72\x6D\x61\x20\x4B\x75\x72\x75\x6D\x75\x20\x2D\x20\x54\x55\x42\x49\x54\x41\x4B\x31\x2D\x30\x2B\x06\x03\x55\x04\x0B\x13\x24\x4B\x61\x6D\x75\x20\x53\x65\x72\x74\x69\x66\x69\x6B\x61\x73\x79\x6F\x6E\x20\x4D\x65\x72\x6B\x65\x7A\x69\x20\x2D\x20\x4B\x61\x6D\x75\x20\x53\x4D\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x54\x55\x42\x49\x54\x41\x4B\x20\x4B\x61\x6D\x75\x20\x53\x4D\x20\x53\x53\x4C\x20\x4B\x6F\x6B\x20\x53\x65\x72\x74\x69\x66\x69\x6B\x61\x73\x69\x20\x2D\x20\x53\x75\x72\x75\x6D\x20\x31\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xAF\x75\x30\x33\xAA\xBB\x6B\xD3\x99\x2C\x12\x37\x84\xD9\x8D\x7B\x97\x80\xD3\x6E\xE7\xFF\x9B\x50\x95\x3E\x90\x95\x56\x42\xD7\x19\x7C\x26\x84\x8D\x92\xFA\x01\x1D\x3A\x0F\xE2\x64\x38\xB7\x8C\xBC\xE8\x88\xF9\x8B\x24\xAB\x2E\xA3\xF5\x37\xE4\x40\x8E\x18\x25\x79\x83\x75\x1F\x3B\xFF\x6C\xA8\xC5\xC6\x56\xF8\xB4\xED\x8A\x44\xA3\xAB\x6C\x4C\xFC\x1D\xD0\xDC\xEF\x68\xBD\xCF\xE4\xAA\xCE\xF0\x55\xF7\xA2\x34\xD4\x83\x6B\x37\x7C\x1C\xC2\xFE\xB5\x03\xEC\x57\xCE\xBC\xB4\xB5\xC5\xED\x00\x0F\x53\x37\x2A\x4D\xF4\x4F\x0C\x83\xFB\x86\xCF\xCB\xFE\x8C\x4E\xBD\x87\xF9\xA7\x8B\x21\x57\x9C\x7A\xDF\x03\x67\x89\x2C\x9D\x97\x61\xA7\x10\xB8\x55\x90\x7F\x0E\x2D\x27\x38\x74\xDF\xE7\xFD\xDA\x4E\x12\xE3\x4D\x15\x22\x02\xC8\xE0\xE0\xFC\x0F\xAD\x8A\xD7\xC9\x54\x50\xCC\x3B\x0F\xCA\x16\x80\x84\xD0\x51\x56\xC3\x8E\x56\x7F\x89\x22\x33\x2F\xE6\x85\x0A\xBD\xA5\xA8\x1B\x36\xDE\xD3\xDC\x2C\x6D\x3B\xC7\x13\xBD\x59\x23\x2C\xE6\xE5\xA4\xF7\xD8\x0B\xED\xEA\x90\x40\x44\xA8\x95\xBB\x93\xD5\xD0\x80\x34\xB6\x46\x78\x0E\x1F\x00\x93\x46\xE1\xEE\xE9\xF9\xEC\x4F\x17\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x65\x3F\xC7\x8A\x86\xC6\x3C\xDD\x3C\x54\x5C\x35\xF8\x3A\xED\x52\x0C\x47\x57\xC8\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x2A\x3F\xE1\xF1\x32\x8E\xAE\xE1\x98\x5C\x4B\x5E\xCF\x6B\x1E\x6A\x09\xD2\x22\xA9\x12\xC7\x5E\x57\x7D\x73\x56\x64\x80\x84\x7A\x93\xE4\x09\xB9\x10\xCD\x9F\x2A\x27\xE1\x00\x77\xBE\x48\xC8\x35\xA8\x81\x9F\xE4\xB8\x2C\xC9\x7F\x0E\xB0\xD2\x4B\x37\x5D\xEA\xB9\xD5\x0B\x5E\x34\xBD\xF4\x73\x29\xC3\xED\x26\x15\x9C\x7E\x08\x53\x8A\x58\x8D\xD0\x4B\x28\xDF\xC1\xB3\xDF\x20\xF3\xF9\xE3\xE3\x3A\xDF\xCC\x9C\x94\xD8\x4E\x4F\xC3\x6B\x17\xB7\xF7\x72\xE8\xAD\x66\x33\xB5\x25\x53\xAB\xE0\xF8\x4C\xA9\x9D\xFD\xF2\x0D\xBA\xAE\xB9\xD9\xAA\xC6\x6B\xF9\x93\xBB\xAE\xAB\xB8\x97\x3C\x03\x1A\xBA\x43\xC6\x96\xB9\x45\x72\x38\xB3\xA7\xA1\x96\x3D\x91\x7B\x7E\xC0\x21\x53\x4C\x87\xED\xF2\x0B\x54\x95\x51\x93\xD5\x22\xA5\x0D\x8A\xF1\x93\x0E\x3E\x54\x0E\xB0\xD8\xC9\x4E\xDC\xF2\x31\x32\x56\xEA\x64\xF9\xEA\xB5\x9D\x16\x66\x42\x72\xF3\x7F\xD3\xB1\x31\x43\xFC\xA4\x8E\x17\xF1\x6D\x23\xAB\x94\x66\xF8\xAD\xFB\x0F\x08\x6E\x26\x2D\x7F\x17\x07\x09\xB2\x8C\xFB\x50\xC0\x9F\x96\x8D\xCF\xB6\xFD\x00\x9D\x5A\x14\x9A\xBF\x02\x44\xF5\xC1\xC2\x9F\x22\x5E\xA2\x0F\xA1\xE3"),
   // "D-TRUST Root Class 3 CA 2 2009"
   CertificateDer::from_slice(b"\x30\x82\x04\x33\x30\x82\x03\x1B\xA0\x03\x02\x01\x02\x02\x03\x09\x83\xF3\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x0C\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x44\x2D\x54\x52\x55\x53\x54\x20\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x33\x20\x43\x41\x20\x32\x20\x32\x30\x30\x39\x30\x1E\x17\x0D\x30\x39\x31\x31\x30\x35\x30\x38\x33\x35\x35\x38\x5A\x17\x0D\x32\x39\x31\x31\x30\x35\x30\x38\x33\x35\x35\x38\x5A\x30\x4D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x0C\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x44\x2D\x54\x52\x55\x53\x54\x20\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x33\x20\x43\x41\x20\x32\x20\x32\x30\x30\x39\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xD3\xB2\x4A\xCF\x7A\x47\xEF\x75\x9B\x23\xFA\x3A\x2F\xD6\x50\x45\x89\x35\x3A\xC6\x6B\xDB\xFE\xDB\x00\x68\xA8\xE0\x03\x11\x1D\x37\x50\x08\x9F\x4D\x4A\x68\x94\x35\xB3\x53\xD1\x94\x63\xA7\x20\x56\xAF\xDE\x51\x78\xEC\x2A\x3D\xF3\x48\x48\x50\x3E\x0A\xDF\x46\x55\x8B\x27\x6D\xC3\x10\x4D\x0D\x91\x52\x43\xD8\x87\xE0\x5D\x4E\x36\xB5\x21\xCA\x5F\x39\x40\x04\x5F\x5B\x7E\xCC\xA3\xC6\x2B\xA9\x40\x1E\xD9\x36\x84\xD6\x48\xF3\x92\x1E\x34\x46\x20\x24\xC1\xA4\x51\x8E\x4A\x1A\xEF\x50\x3F\x69\x5D\x19\x7F\x45\xC3\xC7\x01\x8F\x51\xC9\x23\xE8\x72\xAE\xB4\xBC\x56\x09\x7F\x12\xCB\x1C\xB1\xAF\x29\x90\x0A\xC9\x55\xCC\x0F\xD3\xB4\x1A\xED\x47\x35\x5A\x4A\xED\x9C\x73\x04\x21\xD0\xAA\xBD\x0C\x13\xB5\x00\xCA\x26\x6C\xC4\x6B\x0C\x94\x5A\x95\x94\xDA\x50\x9A\xF1\xFF\xA5\x2B\x66\x31\xA4\xC9\x38\xA0\xDF\x1D\x1F\xB8\x09\x2E\xF3\xA7\xE8\x67\x52\xAB\x95\x1F\xE0\x46\x3E\xD8\xA4\xC3\xCA\x5A\xC5\x31\x80\xE8\x48\x9A\x9F\x94\x69\xFE\x19\xDD\xD8\x73\x7C\x81\xCA\x96\xDE\x8E\xED\xB3\x32\x05\x65\x84\x34\xE6\xE6\xFD\x57\x10\xB5\x5F\x76\xBF\x2F\xB0\x10\x0D\xC5\x02\x03\x01\x00\x01\xA3\x82\x01\x1A\x30\x82\x01\x16\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xFD\xDA\x14\xC4\x9F\x30\xDE\x21\xBD\x1E\x42\x39\xFC\xAB\x63\x23\x49\xE0\xF1\x84\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x81\xD3\x06\x03\x55\x1D\x1F\x04\x81\xCB\x30\x81\xC8\x30\x81\x80\xA0\x7E\xA0\x7C\x86\x7A\x6C\x64\x61\x70\x3A\x2F\x2F\x64\x69\x72\x65\x63\x74\x6F\x72\x79\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x43\x4E\x3D\x44\x2D\x54\x52\x55\x53\x54\x25\x32\x30\x52\x6F\x6F\x74\x25\x32\x30\x43\x6C\x61\x73\x73\x25\x32\x30\x33\x25\x32\x30\x43\x41\x25\x32\x30\x32\x25\x32\x30\x32\x30\x30\x39\x2C\x4F\x3D\x44\x2D\x54\x72\x75\x73\x74\x25\x32\x30\x47\x6D\x62\x48\x2C\x43\x3D\x44\x45\x3F\x63\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x72\x65\x76\x6F\x63\x61\x74\x69\x6F\x6E\x6C\x69\x73\x74\x30\x43\xA0\x41\xA0\x3F\x86\x3D\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x63\x72\x6C\x2F\x64\x2D\x74\x72\x75\x73\x74\x5F\x72\x6F\x6F\x74\x5F\x63\x6C\x61\x73\x73\x5F\x33\x5F\x63\x61\x5F\x32\x5F\x32\x30\x30\x39\x2E\x63\x72\x6C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x7F\x97\xDB\x30\xC8\xDF\xA4\x9C\x7D\x21\x7A\x80\x70\xCE\x14\x12\x69\x88\x14\x95\x60\x44\x01\xAC\xB2\xE9\x30\x4F\x9B\x50\xC2\x66\xD8\x7E\x8D\x30\xB5\x70\x31\xE9\xE2\x69\xC7\xF3\x70\xDB\x20\x15\x86\xD0\x0D\xF0\xBE\xAC\x01\x75\x84\xCE\x7E\x9F\x4D\xBF\xB7\x60\x3B\x9C\xF3\xCA\x1D\xE2\x5E\x68\xD8\xA3\x9D\x97\xE5\x40\x60\xD2\x36\x21\xFE\xD0\xB4\xB8\x17\xDA\x74\xA3\x7F\xD4\xDF\xB0\x98\x02\xAC\x6F\x6B\x6B\x2C\x25\x24\x72\xA1\x65\xEE\x25\x5A\xE5\xE6\x32\xE7\xF2\xDF\xAB\x49\xFA\xF3\x90\x69\x23\xDB\x04\xD9\xE7\x5C\x58\xFC\x65\xD4\x97\xBE\xCC\xFC\x2E\x0A\xCC\x25\x2A\x35\x04\xF8\x60\x91\x15\x75\x3D\x41\xFF\x23\x1F\x19\xC8\x6C\xEB\x82\x53\x04\xA6\xE4\x4C\x22\x4D\x8D\x8C\xBA\xCE\x5B\x73\xEC\x64\x54\x50\x6D\xD1\x9C\x55\xFB\x69\xC3\x36\xC3\x8C\xBC\x3C\x85\xA6\x6B\x0A\x26\x0D\xE0\x93\x98\x60\xAE\x7E\xC6\x24\x97\x8A\x61\x5F\x91\x8E\x66\x92\x09\x87\x36\xCD\x8B\x9B\x2D\x3E\xF6\x51\xD4\x50\xD4\x59\x28\xBD\x83\xF2\xCC\x28\x7B\x53\x86\x6D\xD8\x26\x88\x70\xD7\xEA\x91\xCD\x3E\xB9\xCA\xC0\x90\x6E\x5A\xC6\x5E\x74\x65\xD7\x5C\xFE\xA3\xE2"),
   // "SecureSign Root CA14"
   CertificateDer::from_slice(b"\x30\x82\x05\x72\x30\x82\x03\x5A\xA0\x03\x02\x01\x02\x02\x14\x64\xDB\x5A\x0C\x20\x4E\xE8\xD7\x29\x77\xC8\x50\x27\xA2\x5A\x27\xDD\x2D\xF2\xCB\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x13\x1A\x43\x79\x62\x65\x72\x74\x72\x75\x73\x74\x20\x4A\x61\x70\x61\x6E\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x13\x14\x53\x65\x63\x75\x72\x65\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x34\x30\x1E\x17\x0D\x32\x30\x30\x34\x30\x38\x30\x37\x30\x36\x31\x39\x5A\x17\x0D\x34\x35\x30\x34\x30\x38\x30\x37\x30\x36\x31\x39\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x13\x1A\x43\x79\x62\x65\x72\x74\x72\x75\x73\x74\x20\x4A\x61\x70\x61\x6E\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x13\x14\x53\x65\x63\x75\x72\x65\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x34\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC5\xD2\x7A\xA1\xD6\x8A\xBF\x16\x31\xD0\x98\xD1\x3A\x94\xFC\x5A\xB8\x6E\x22\xC1\x62\xF7\xA7\x0A\x27\xEF\x50\xF6\x2E\xB1\x9E\x68\x12\xF0\x6C\x24\x63\x39\xF1\xF0\xDF\x10\xC6\xDE\xB7\x52\x20\xD5\x52\x5B\x42\x99\x9E\xF3\xA0\xBE\x52\x1F\x5F\xCC\x67\x6D\xA7\x2E\x50\xA2\xC1\x97\x8D\xB6\xF8\x95\xF5\xB0\xBA\xDC\x9D\xE0\xBE\xCB\xDF\xF7\x38\xF2\x47\xF5\xA6\x9A\x92\x95\x2A\x62\x59\x50\x0B\xA2\xB1\x35\xE7\x65\xB2\x61\xB2\xEA\x92\x71\x69\xE4\x29\xF0\x4F\x81\x81\x04\x3C\xB2\xA5\x5B\xD4\xC5\xA8\x59\x67\x7B\x55\x1C\x49\xAB\x7A\x9D\xC2\xE7\x73\x4D\xEF\xCD\x09\xC2\xC4\x57\x12\xDB\x01\x0E\x23\x79\x09\x07\x3B\xA2\xE8\xFC\x8A\xCF\x8F\xC0\x46\x24\x9C\x38\x27\xE0\x83\x9D\x1B\xA0\xBF\x78\x15\x10\xEB\x86\x4E\x0A\x5A\xFD\xDF\xDA\x2C\x82\x7E\xEE\xCA\xF6\x29\xE1\xFA\x71\xA1\xF7\x88\x68\x9C\x9C\xF0\x8D\xBE\x0F\x49\x91\xD8\xEA\x3A\xF9\xFD\xD0\x68\x71\xDB\xE9\xB5\x2B\x4E\x82\x92\x6F\x66\x1F\xE0\xF0\xDC\x4C\xEC\xCA\xD1\xEA\xBA\x74\x06\xF9\xB3\x84\x90\x94\xD1\x5F\x8E\x73\x19\x10\x5D\x02\xE5\x70\xA5\xC0\x10\xD0\x10\x7C\x6F\xC5\x58\x49\xB4\xB0\x6E\x9A\xDA\x7D\x95\xF5\xCC\xDA\x02\xAF\xB8\x2C\x7D\x79\x8F\xBE\x43\xF1\xF9\x28\x28\x8D\x09\x43\xF8\x08\xDD\x6B\xC8\x8B\x2C\x24\xB1\x8D\x52\x07\xBD\x78\x9B\xCB\xCA\x68\xB2\xA4\xDD\x0C\x4C\x79\x60\xC6\x99\xD1\x93\xF1\x30\x1A\x07\xD3\xAE\x22\xC2\xEA\xCE\xF1\x84\x09\xCC\xE0\x14\x6E\x7F\x3F\x7E\xD2\x82\x85\xAC\xDC\xA9\x16\x4E\x85\xA0\x60\xCB\xF6\x9C\xD7\xC8\xB3\x8E\xED\xC6\x9B\x98\x75\x0D\x55\xE8\x5F\xE5\x95\x8B\x02\xA4\xAE\x43\x29\x28\x11\xA4\xE6\x12\x30\x01\x4B\x75\x6B\x1E\x66\x9D\x79\x2F\xA5\x76\x2F\x1D\x40\xB4\x6D\xC9\x7D\x79\x08\xEC\xD1\x6A\xB6\x5D\x2A\xB2\xA5\x66\xBD\x6B\x85\xF4\x74\x56\xC3\xF5\xE7\x75\x52\x28\x2C\xA5\xFF\x66\x47\xA5\xD4\xFE\xFE\x9E\x54\xBF\x65\x7E\x01\xD6\x30\x8F\xA5\x36\x9C\xA2\x50\x1C\xEE\x38\x80\x01\x48\xC6\xC7\x74\xF4\xC6\xAC\xC3\x40\x49\x16\x61\x74\x2C\xAF\x8C\x6F\x35\xED\x7B\x18\x00\x5B\x36\x3C\x9C\x50\x0D\xCA\x92\x33\x10\xF1\x26\x49\x6D\xDF\x75\x24\x37\x82\x22\xD7\xE8\x96\xFD\x15\x4B\x02\x96\x3E\x07\x72\x95\x7E\xAB\x3D\x4C\x2E\xD7\xCA\xF0\xDF\xE0\x58\x3F\x2D\x2F\x04\x9A\x38\xA3\x01\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x06\x93\xA3\x0A\x5E\x28\x69\x37\xAA\x61\x1D\xEB\xEB\xFC\x2D\x6F\x23\xE4\xF3\xA0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x96\x80\x72\x09\x06\x7E\x9C\xCC\x93\x04\x16\xBB\xA0\x3A\x8D\x92\x4E\xB7\x11\x1A\x0A\x71\x71\x10\xCD\x04\xAD\x7F\xA5\x45\x50\x10\x66\x4E\x4A\x41\xA2\x03\xD9\x11\x4F\x7A\x37\xB9\x4B\xE2\xC6\x8F\x32\x66\x75\x25\xFB\xEB\xCE\x3F\x03\x29\x26\x8D\xB8\x16\x1D\xF6\x1F\x33\x6E\x48\xE6\xE8\xF8\x57\xB2\x1B\x79\xDF\x3B\x87\x0A\xE2\x64\xBA\x00\xCA\x6C\xEF\x7E\xD0\x23\xEB\x78\x8F\xFF\x64\x9B\x34\x37\x9F\x35\x65\xA2\xA4\x00\x3D\x12\x23\x96\x58\x5D\xCA\x63\x87\xC6\xA3\x07\x88\x4D\xE7\x69\x76\x8A\x53\xCD\xF1\x4F\xEC\x42\xF2\x93\xE3\x99\xA4\x37\x3C\x87\xB8\x62\xDB\xF0\xEC\x1F\x37\x3F\x37\x5F\x43\xCC\x51\x9D\xB5\xF0\x97\xC2\xB7\x85\x6A\x68\x0B\x44\x1E\xE5\x51\xEE\x93\xCE\x4B\x6E\x86\xC1\xD2\x0C\x24\x59\x36\x1A\x9F\x2C\x91\x8F\xE3\x18\xDB\x94\x95\x0A\xED\x91\xAA\x0E\x99\xDC\x96\x53\xE3\x61\x83\xC6\x16\xBA\x23\xBA\xDC\xDD\x7E\x1A\xC6\x7B\x42\xB6\xD9\x5A\x05\xDC\x9A\x5F\xD5\xDF\xB8\xDA\x47\x7D\xDA\x38\xDB\xAC\x39\xD5\x1E\x6B\x6C\x2A\x17\x8C\x61\xCD\xB1\x6D\x72\x01\xC3\xC3\x20\x00\x62\x68\x16\x31\xD5\x76\xAA\x86\xBB\x0E\xAA\x9E\xC6\xF9\xF0\xD9\xF8\x0D\x21\x02\xE4\xC5\x28\x16\x59\x11\xB9\xD9\x69\x73\x2A\x92\x78\xB8\x92\x57\x9B\x08\xF2\x3A\xE5\x2F\x95\xB0\x58\xB7\x6B\x20\x14\x6D\x14\xEF\x0A\xBC\x7E\xD8\x55\xD8\x88\xDA\x2F\xFA\x19\xA5\xFB\x8B\xE0\x7F\x39\xF5\x72\x2B\x85\xC4\x2C\xAC\xEF\x19\x45\x92\x4C\xB3\x61\x07\xDC\x4D\x1F\x6E\xD2\x81\x13\x5C\x9A\xF3\x12\x67\x83\xCF\x9B\x3F\x8B\x9F\x9D\xA4\xB9\xA8\x96\x03\x7A\xC5\xEE\x20\xDE\x33\xDA\x2F\x9E\x1A\x7A\x74\x1E\xE1\xEE\xCC\x5A\x3A\x04\xDD\xB3\x1A\x04\xA8\x14\x63\xAC\xB7\x47\x12\x83\x9A\x6C\xF5\xE6\xE9\x15\x15\x91\x1A\x84\x19\x0E\x94\x44\xE7\x12\x8E\x25\x5B\x80\x67\x19\xDC\x63\x93\x10\x0B\x65\x2E\x8A\xFA\x09\x9A\x4E\xDA\x86\x28\x7D\xAA\x61\x35\xD8\x0E\xA7\x28\x1A\xBB\x52\xE0\x78\xF8\x6C\xBA\x6C\xB0\x6E\xB9\x87\x5E\xE9\x99\x35\x37\xF1\x3D\x64\x2B\xA9\xA0\x34\x93\xCF\x63\x2F\xD5\x81\xDF\xAE\x63\x27\xA5\x1E\x4E\x8D\xDC\x29\x78\x59\xF8\xF9\xA1\x20\x8C\xA7\x26\x40\x6E\x82\x72\xCD\x78\xB2\xC8\x8F\x3C\x1E\x73\xE7\xC1\x1F\xBF\xCF\xCE\xA5\x2A\x9B\xDB\x44\x64\x32\xA0\xBB\x7F\x5C\x25\x13\x48\xB5\x7F\x92"),
   // "GlobalSign Root R46"
   CertificateDer::from_slice(b"\x30\x82\x05\x5A\x30\x82\x03\x42\xA0\x03\x02\x01\x02\x02\x12\x11\xD2\xBB\xB9\xD7\x23\x18\x9E\x40\x5F\x0A\x9D\x2D\xD0\xDF\x25\x67\xD1\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x46\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x45\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x6E\x76\x2D\x73\x61\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x52\x34\x36\x30\x1E\x17\x0D\x31\x39\x30\x33\x32\x30\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x33\x32\x30\x30\x30\x30\x30\x30\x30\x5A\x30\x46\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x45\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x6E\x76\x2D\x73\x61\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x52\x34\x36\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xAC\xAC\x74\x32\xE8\xB3\x65\xE5\xBA\xED\x43\x26\x1D\xA6\x89\x0D\x45\xBA\x29\x88\xB2\xA4\x1D\x63\xDD\xD3\xC1\x2C\x09\x57\x89\x39\xA1\x55\xE9\x67\x34\x77\x0C\x6E\xE4\x55\x1D\x52\x25\xD2\x13\x6B\x5E\xE1\x1D\xA9\xB7\x7D\x89\x32\x5F\x0D\x9E\x9F\x2C\x7A\x63\x60\x40\x1F\xA6\xB0\xB6\x78\x8F\x99\x54\x96\x08\x58\xAE\xE4\x06\xBC\x62\x05\x02\x16\xBF\xAF\xA8\x23\x03\xB6\x94\x0F\xBC\x6E\x6C\xC2\xCB\xD5\xA6\xBB\x0C\xE9\xF6\xC1\x02\xFB\x21\xDE\x66\xDD\x17\xAB\x74\x42\xEF\xF0\x74\x2F\x25\xF4\xEA\x6B\x55\x5B\x90\xDB\x9D\xDF\x5E\x87\x0A\x40\xFB\xAD\x19\x6B\xFB\xF7\xCA\x60\x88\xDE\xDA\xC1\x8F\xD6\xAE\xD5\x7F\xD4\x3C\x83\xEE\xD7\x16\x4C\x83\x45\x33\x6B\x27\xD0\x86\xD0\x1C\x2D\x6B\xF3\xAB\x7D\xF1\x85\xA9\xF5\x28\xD2\xAD\xEF\xF3\x84\x4B\x1C\x87\xFC\x13\xA3\x3A\x72\xA2\x5A\x11\x2B\xD6\x27\x71\x27\xED\x81\x2D\x6D\x66\x81\x92\x87\xB4\x1B\x58\x7A\xCC\x3F\x0A\xFA\x46\x4F\x4D\x78\x5C\xF8\x2B\x48\xE3\x04\x84\xCB\x5D\xF6\xB4\x6A\xB3\x65\xFC\x42\x9E\x51\x26\x23\x20\xCB\x3D\x14\xF9\x81\xED\x65\x16\x00\x4F\x1A\x64\x97\x66\x08\xCF\x8C\x7B\xE3\x2B\xC0\x9D\xF9\x14\xF2\x1B\xF1\x56\x6A\x16\xBF\x2C\x85\x85\xCD\x78\x38\x9A\xEB\x42\x6A\x02\x34\x18\x83\x17\x4E\x94\x56\xF8\xB6\x82\xB5\xF3\x96\xDD\x3D\xF3\xBE\x7F\x20\x77\x3E\x7B\x19\x23\x6B\x2C\xD4\x72\x73\x43\x57\x7D\xE0\xF8\xD7\x69\x4F\x17\x36\x04\xF9\xC0\x90\x60\x37\x45\xDE\xE6\x0C\xD8\x74\x8D\xAE\x9C\xA2\x6D\x74\x5D\x42\xBE\x06\xF5\xD9\x64\x6E\x02\x10\xAC\x89\xB0\x4C\x3B\x07\x4D\x40\x7E\x24\xC5\x8A\x98\x82\x79\x8E\xA4\xA7\x82\x20\x8D\x23\xFA\x27\x71\xC9\xDF\xC6\x41\x74\xA0\x4D\xF6\x91\x16\xDC\x46\x8C\x5F\x29\x63\x31\x59\x71\x0C\xD8\x6F\xC2\xB6\x32\x7D\xFB\xE6\x5D\x53\xA6\x7E\x15\xFC\xBB\x75\x7C\x5D\xEC\xF8\xF6\x17\x1C\xEC\xC7\x6B\x19\xCB\xF3\x7B\xF0\x2B\x07\xA5\xD9\x6C\x79\x54\x76\x6C\x9D\x1C\xA6\x6E\x0E\xE9\x79\x0C\xA8\x23\x6A\xA3\xDF\x1B\x30\x31\x9F\xB1\x54\x7B\xFE\x6A\xCB\x66\xAA\xDC\x65\xD0\xA2\x9E\x4A\x9A\x07\x21\x6B\x81\x8F\xDB\xC4\x59\xFA\xDE\x22\xC0\x04\x9C\xE3\xAA\x5B\x36\x93\xE8\x3D\xBD\x7A\xA1\x9D\x0B\x76\xB1\x0B\xC7\x9D\xFD\xCF\x98\xA8\x06\xC2\xF8\x2A\xA3\xA1\x83\xA0\xB7\x25\x72\xA5\x02\xE3\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x03\x5C\xAB\x73\x81\x87\xA8\xCC\xB0\xA6\xD5\x94\xE2\x36\x96\x49\xFF\x05\x99\x2C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x7C\x78\xEC\xF6\x02\x2C\xBB\x5B\x7E\x92\x2B\x5D\x39\xDC\xBE\xD8\x1D\xA2\x42\x33\x4D\xF9\xEF\xA4\x2A\x3B\x44\x69\x1E\xAC\xD9\x45\xA3\x4E\x3C\xA7\xD8\x24\x51\xB2\x54\x1C\x93\x4E\xC4\xEF\x7B\x93\x85\x60\x26\xEA\x09\x48\xE0\xF5\xBB\xC7\xE9\x68\xD2\xBB\x6A\x31\x71\xCC\x79\xAE\x11\xA8\xF0\x99\xFD\xE5\x1F\xBC\x2F\xA8\xCC\x57\xEB\x76\xC4\x21\xA6\x47\x53\x55\x4D\x68\xBF\x05\xA4\xEE\xD7\x26\xAB\x62\xDA\x43\x37\x4B\xE2\xC6\xB5\xE5\xB2\x83\x19\x3A\xC7\xD3\xDB\x4D\x9E\x08\x7A\xF3\xEE\xCF\x3E\x62\xFB\xAC\xE8\x60\xCC\xD1\xC7\xA1\x5C\x83\x45\xC4\x45\xCC\xF3\x17\x6B\x14\xC9\x04\x02\x3E\xD2\x24\xA6\x79\xE9\x1E\xCE\xA2\xE7\xC1\x59\x15\x9F\x1D\xE2\x4B\x9A\x3E\x9F\x76\x08\x2D\x6B\xD8\xBA\x57\x14\xDA\x83\xEA\xFE\x8C\x55\xE9\xD0\x4E\xA9\xCC\x77\x31\xB1\x44\x11\x7A\x5C\xB1\x3E\xD3\x14\x45\x15\x18\x62\x24\x13\xD2\xCB\x4D\xCE\x5C\x83\xC1\x36\xF2\x10\xB5\x0E\x88\x6D\xB8\xE1\x56\x9F\x89\xDE\x96\x66\x39\x47\x64\x2C\x6E\x4D\xAE\x62\x7B\xBF\x60\x74\x19\xB8\x56\xAC\x92\xAC\x16\x32\xED\xAD\x68\x55\xFE\x98\xBA\xD3\x34\xDE\xF4\xC9\x61\xC3\x0E\x86\xF6\x4B\x84\x60\xEE\x0D\x7B\xB5\x32\x58\x79\x91\x55\x2C\x81\x43\xB3\x74\x1F\x7A\xAA\x25\x9E\x1D\xD7\xA1\x8B\xB9\xCD\x42\x2E\x04\xA4\x66\x83\x4D\x89\x35\xB6\x6C\xA8\x36\x4A\x79\x21\x78\x22\xD0\x42\xBC\xD1\x40\x31\x90\xA1\xBE\x04\xCF\xCA\x67\xED\xF5\xF0\x80\xD3\x60\xC9\x83\x2A\x22\x05\xD0\x07\x3B\x52\xBF\x0C\x9E\xAA\x2B\xF9\xBB\xE6\x1F\x8F\x25\xBA\x85\x8D\x17\x1E\x02\xFE\x5D\x50\x04\x57\xCF\xFE\x2D\xBC\xEF\x5C\xC0\x1A\xAB\xB6\x9F\x24\xC6\xDF\x73\x68\x48\x90\x2C\x14\xF4\x3F\x52\x1A\xE4\xD2\xCB\x14\xC3\x61\x69\xCF\xE2\xF9\x18\xC5\xBA\x33\x9F\x14\xA3\x04\x5D\xB9\x71\xF7\xB5\x94\xD8\xF6\x33\xC1\x5A\xC1\x34\x8B\x7C\x9B\xDD\x93\x3A\xE7\x13\xA2\x70\x61\x9F\xAF\x8F\xEB\xD8\xC5\x75\xF8\x33\x66\xD4\x74\x67\x3A\x37\x77\x9C\xE7\xDD\xA4\x0F\x76\x43\x66\x8A\x43\xF2\x9F\xFB\x0C\x42\x78\x63\xD1\xE2\x0F\x6F\x7B\xD4\xA1\x3D\x74\x97\x85\xB7\x48\x39\x41\xD6\x20\xFC\xD0\x3A\xB3\xFA\xE8\x6F\xC4\x8A\xBA\x71\x37\xBE\x8B\x97\xB1\x78\x31\x4F\xB3\xE7\xB6\x03\x13\xCE\x54\x9D\xAE\x25\x59\xCC\x7F\x35\x5F\x08\xF7\x40\x45\x31\x78\x2A\x7A"),
   // "USERTrust ECC Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x02\x8F\x30\x82\x02\x15\xA0\x03\x02\x01\x02\x02\x10\x5C\x8B\x99\xC5\x5A\x94\xC5\xD2\x71\x56\xDE\xCD\x89\x80\xCC\x26\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x81\x88\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x08\x13\x0A\x4E\x65\x77\x20\x4A\x65\x72\x73\x65\x79\x31\x14\x30\x12\x06\x03\x55\x04\x07\x13\x0B\x4A\x65\x72\x73\x65\x79\x20\x43\x69\x74\x79\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x54\x68\x65\x20\x55\x53\x45\x52\x54\x52\x55\x53\x54\x20\x4E\x65\x74\x77\x6F\x72\x6B\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x13\x25\x55\x53\x45\x52\x54\x72\x75\x73\x74\x20\x45\x43\x43\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x30\x30\x32\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x38\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x88\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x08\x13\x0A\x4E\x65\x77\x20\x4A\x65\x72\x73\x65\x79\x31\x14\x30\x12\x06\x03\x55\x04\x07\x13\x0B\x4A\x65\x72\x73\x65\x79\x20\x43\x69\x74\x79\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x54\x68\x65\x20\x55\x53\x45\x52\x54\x52\x55\x53\x54\x20\x4E\x65\x74\x77\x6F\x72\x6B\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x13\x25\x55\x53\x45\x52\x54\x72\x75\x73\x74\x20\x45\x43\x43\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x1A\xAC\x54\x5A\xA9\xF9\x68\x23\xE7\x7A\xD5\x24\x6F\x53\xC6\x5A\xD8\x4B\xAB\xC6\xD5\xB6\xD1\xE6\x73\x71\xAE\xDD\x9C\xD6\x0C\x61\xFD\xDB\xA0\x89\x03\xB8\x05\x14\xEC\x57\xCE\xEE\x5D\x3F\xE2\x21\xB3\xCE\xF7\xD4\x8A\x79\xE0\xA3\x83\x7E\x2D\x97\xD0\x61\xC4\xF1\x99\xDC\x25\x91\x63\xAB\x7F\x30\xA3\xB4\x70\xE2\xC7\xA1\x33\x9C\xF3\xBF\x2E\x5C\x53\xB1\x5F\xB3\x7D\x32\x7F\x8A\x34\xE3\x79\x79\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x3A\xE1\x09\x86\xD4\xCF\x19\xC2\x96\x76\x74\x49\x76\xDC\xE0\x35\xC6\x63\x63\x9A\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x36\x67\xA1\x16\x08\xDC\xE4\x97\x00\x41\x1D\x4E\xBE\xE1\x63\x01\xCF\x3B\xAA\x42\x11\x64\xA0\x9D\x94\x39\x02\x11\x79\x5C\x7B\x1D\xFA\x64\xB9\xEE\x16\x42\xB3\xBF\x8A\xC2\x09\xC4\xEC\xE4\xB1\x4D\x02\x31\x00\xE9\x2A\x61\x47\x8C\x52\x4A\x4B\x4E\x18\x70\xF6\xD6\x44\xD6\x6E\xF5\x83\xBA\x6D\x58\xBD\x24\xD9\x56\x48\xEA\xEF\xC4\xA2\x46\x81\x88\x6A\x3A\x46\xD1\xA9\x9B\x4D\xC9\x61\xDA\xD1\x5D\x57\x6A\x18"),
   // "Security Communication RootCA2"
   CertificateDer::from_slice(b"\x30\x82\x03\x77\x30\x82\x02\x5F\xA0\x03\x02\x01\x02\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x5D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x45\x43\x4F\x4D\x20\x54\x72\x75\x73\x74\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x43\x4F\x2E\x2C\x4C\x54\x44\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x53\x65\x63\x75\x72\x69\x74\x79\x20\x43\x6F\x6D\x6D\x75\x6E\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x43\x41\x32\x30\x1E\x17\x0D\x30\x39\x30\x35\x32\x39\x30\x35\x30\x30\x33\x39\x5A\x17\x0D\x32\x39\x30\x35\x32\x39\x30\x35\x30\x30\x33\x39\x5A\x30\x5D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x45\x43\x4F\x4D\x20\x54\x72\x75\x73\x74\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x43\x4F\x2E\x2C\x4C\x54\x44\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x53\x65\x63\x75\x72\x69\x74\x79\x20\x43\x6F\x6D\x6D\x75\x6E\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x43\x41\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xD0\x15\x39\x52\xB1\x52\xB3\xBA\xC5\x59\x82\xC4\x5D\x52\xAE\x3A\x43\x65\x80\x4B\xC7\xF2\x96\xBC\xDB\x36\x97\xD6\xA6\x64\x8C\xA8\x5E\xF0\xE3\x0A\x1C\xF7\xDF\x97\x3D\x4B\xAE\xF6\x5D\xEC\x21\xB5\x41\xAB\xCD\xB9\x7E\x76\x9F\xBE\xF9\x3E\x36\x34\xA0\x3B\xC1\xF6\x31\x11\x45\x74\x93\x3D\x57\x80\xC5\xF9\x89\x99\xCA\xE5\xAB\x6A\xD4\xB5\xDA\x41\x90\x10\xC1\xD6\xD6\x42\x89\xC2\xBF\xF4\x38\x12\x95\x4C\x54\x05\xF7\x36\xE4\x45\x83\x7B\x14\x65\xD6\xDC\x0C\x4D\xD1\xDE\x7E\x0C\xAB\x3B\xC4\x15\xBE\x3A\x56\xA6\x5A\x6F\x76\x69\x52\xA9\x7A\xB9\xC8\xEB\x6A\x9A\x5D\x52\xD0\x2D\x0A\x6B\x35\x16\x09\x10\x84\xD0\x6A\xCA\x3A\x06\x00\x37\x47\xE4\x7E\x57\x4F\x3F\x8B\xEB\x67\xB8\x88\xAA\xC5\xBE\x53\x55\xB2\x91\xC4\x7D\xB9\xB0\x85\x19\x06\x78\x2E\xDB\x61\x1A\xFA\x85\xF5\x4A\x91\xA1\xE7\x16\xD5\x8E\xA2\x39\xDF\x94\xB8\x70\x1F\x28\x3F\x8B\xFC\x40\x5E\x63\x83\x3C\x83\x2A\x1A\x99\x6B\xCF\xDE\x59\x6A\x3B\xFC\x6F\x16\xD7\x1F\xFD\x4A\x10\xEB\x4E\x82\x16\x3A\xAC\x27\x0C\x53\xF1\xAD\xD5\x24\xB0\x6B\x03\x50\xC1\x2D\x3C\x16\xDD\x44\x34\x27\x1A\x75\xFB\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x0A\x85\xA9\x77\x65\x05\x98\x7C\x40\x81\xF8\x0F\x97\x2C\x38\xF1\x0A\xEC\x3C\xCF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x4C\x3A\xA3\x44\xAC\xB9\x45\xB1\xC7\x93\x7E\xC8\x0B\x0A\x42\xDF\x64\xEA\x1C\xEE\x59\x6C\x08\xBA\x89\x5F\x6A\xCA\x4A\x95\x9E\x7A\x8F\x07\xC5\xDA\x45\x72\x82\x71\x0E\x3A\xD2\xCC\x6F\xA7\xB4\xA1\x23\xBB\xF6\x24\x9F\xCB\x17\xFE\x8C\xA6\xCE\xC2\xD2\xDB\xCC\x8D\xFC\x71\xFC\x03\x29\xC1\x6C\x5D\x33\x5F\x64\xB6\x65\x3B\x89\x6F\x18\x76\x78\xF5\xDC\xA2\x48\x1F\x19\x3F\x8E\x93\xEB\xF1\xFA\x17\xEE\xCD\x4E\xE3\x04\x12\x55\xD6\xE5\xE4\xDD\xFB\x3E\x05\x7C\xE2\x1D\x5E\xC6\xA7\xBC\x97\x4F\x68\x3A\xF5\xE9\x2E\x0A\x43\xB6\xAF\x57\x5C\x62\x68\x7C\xB7\xFD\xA3\x8A\x84\xA0\xAC\x62\xBE\x2B\x09\x87\x34\xF0\x6A\x01\xBB\x9B\x29\x56\x3C\xFE\x00\x37\xCF\x23\x6C\xF1\x4E\xAA\xB6\x74\x46\x12\x6C\x91\xEE\x34\xD5\xEC\x9A\x91\xE7\x44\xBE\x90\x31\x72\xD5\x49\x02\xF6\x02\xE5\xF4\x1F\xEB\x7C\xD9\x96\x55\xA9\xFF\xEC\x8A\xF9\x99\x47\xFF\x35\x5A\x02\xAA\x04\xCB\x8A\x5B\x87\x71\x29\x91\xBD\xA4\xB4\x7A\x0D\xBD\x9A\xF5\x57\x23\x00\x07\x21\x17\x3F\x4A\x39\xD1\x05\x49\x0B\xA7\xB6\x37\x81\xA5\x5D\x8C\xAA\x33\x5E\x81\x28\x7C\xA7\x7D\x27\xEB\x00\xAE\x8D\x37"),
   // "COMODO RSA Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x05\xD8\x30\x82\x03\xC0\xA0\x03\x02\x01\x02\x02\x10\x4C\xAA\xF9\xCA\xDB\x63\x6F\xE0\x1F\xF7\x4E\xD8\x5B\x03\x86\x9D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x81\x85\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x1B\x30\x19\x06\x03\x55\x04\x08\x13\x12\x47\x72\x65\x61\x74\x65\x72\x20\x4D\x61\x6E\x63\x68\x65\x73\x74\x65\x72\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x53\x61\x6C\x66\x6F\x72\x64\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x41\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x43\x4F\x4D\x4F\x44\x4F\x20\x52\x53\x41\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x30\x30\x31\x31\x39\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x38\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x85\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x1B\x30\x19\x06\x03\x55\x04\x08\x13\x12\x47\x72\x65\x61\x74\x65\x72\x20\x4D\x61\x6E\x63\x68\x65\x73\x74\x65\x72\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x53\x61\x6C\x66\x6F\x72\x64\x31\x1A\x30\x18\x06\x03\x55\x04\x0A\x13\x11\x43\x4F\x4D\x4F\x44\x4F\x20\x43\x41\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x43\x4F\x4D\x4F\x44\x4F\x20\x52\x53\x41\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x91\xE8\x54\x92\xD2\x0A\x56\xB1\xAC\x0D\x24\xDD\xC5\xCF\x44\x67\x74\x99\x2B\x37\xA3\x7D\x23\x70\x00\x71\xBC\x53\xDF\xC4\xFA\x2A\x12\x8F\x4B\x7F\x10\x56\xBD\x9F\x70\x72\xB7\x61\x7F\xC9\x4B\x0F\x17\xA7\x3D\xE3\xB0\x04\x61\xEE\xFF\x11\x97\xC7\xF4\x86\x3E\x0A\xFA\x3E\x5C\xF9\x93\xE6\x34\x7A\xD9\x14\x6B\xE7\x9C\xB3\x85\xA0\x82\x7A\x76\xAF\x71\x90\xD7\xEC\xFD\x0D\xFA\x9C\x6C\xFA\xDF\xB0\x82\xF4\x14\x7E\xF9\xBE\xC4\xA6\x2F\x4F\x7F\x99\x7F\xB5\xFC\x67\x43\x72\xBD\x0C\x00\xD6\x89\xEB\x6B\x2C\xD3\xED\x8F\x98\x1C\x14\xAB\x7E\xE5\xE3\x6E\xFC\xD8\xA8\xE4\x92\x24\xDA\x43\x6B\x62\xB8\x55\xFD\xEA\xC1\xBC\x6C\xB6\x8B\xF3\x0E\x8D\x9A\xE4\x9B\x6C\x69\x99\xF8\x78\x48\x30\x45\xD5\xAD\xE1\x0D\x3C\x45\x60\xFC\x32\x96\x51\x27\xBC\x67\xC3\xCA\x2E\xB6\x6B\xEA\x46\xC7\xC7\x20\xA0\xB1\x1F\x65\xDE\x48\x08\xBA\xA4\x4E\xA9\xF2\x83\x46\x37\x84\xEB\xE8\xCC\x81\x48\x43\x67\x4E\x72\x2A\x9B\x5C\xBD\x4C\x1B\x28\x8A\x5C\x22\x7B\xB4\xAB\x98\xD9\xEE\xE0\x51\x83\xC3\x09\x46\x4E\x6D\x3E\x99\xFA\x95\x17\xDA\x7C\x33\x57\x41\x3C\x8D\x51\xED\x0B\xB6\x5C\xAF\x2C\x63\x1A\xDF\x57\xC8\x3F\xBC\xE9\x5D\xC4\x9B\xAF\x45\x99\xE2\xA3\x5A\x24\xB4\xBA\xA9\x56\x3D\xCF\x6F\xAA\xFF\x49\x58\xBE\xF0\xA8\xFF\xF4\xB8\xAD\xE9\x37\xFB\xBA\xB8\xF4\x0B\x3A\xF9\xE8\x43\x42\x1E\x89\xD8\x84\xCB\x13\xF1\xD9\xBB\xE1\x89\x60\xB8\x8C\x28\x56\xAC\x14\x1D\x9C\x0A\xE7\x71\xEB\xCF\x0E\xDD\x3D\xA9\x96\xA1\x48\xBD\x3C\xF7\xAF\xB5\x0D\x22\x4C\xC0\x11\x81\xEC\x56\x3B\xF6\xD3\xA2\xE2\x5B\xB7\xB2\x04\x22\x52\x95\x80\x93\x69\xE8\x8E\x4C\x65\xF1\x91\x03\x2D\x70\x74\x02\xEA\x8B\x67\x15\x29\x69\x52\x02\xBB\xD7\xDF\x50\x6A\x55\x46\xBF\xA0\xA3\x28\x61\x7F\x70\xD0\xC3\xA2\xAA\x2C\x21\xAA\x47\xCE\x28\x9C\x06\x45\x76\xBF\x82\x18\x27\xB4\xD5\xAE\xB4\xCB\x50\xE6\x6B\xF4\x4C\x86\x71\x30\xE9\xA6\xDF\x16\x86\xE0\xD8\xFF\x40\xDD\xFB\xD0\x42\x88\x7F\xA3\x33\x3A\x2E\x5C\x1E\x41\x11\x81\x63\xCE\x18\x71\x6B\x2B\xEC\xA6\x8A\xB7\x31\x5C\x3A\x6A\x47\xE0\xC3\x79\x59\xD6\x20\x1A\xAF\xF2\x6A\x98\xAA\x72\xBC\x57\x4A\xD2\x4B\x9D\xBB\x10\xFC\xB0\x4C\x41\xE5\xED\x1D\x3D\x5E\x28\x9D\x9C\xCC\xBF\xB3\x51\xDA\xA7\x47\xE5\x84\x53\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xBB\xAF\x7E\x02\x3D\xFA\xA6\xF1\x3C\x84\x8E\xAD\xEE\x38\x98\xEC\xD9\x32\x32\xD4\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x0A\xF1\xD5\x46\x84\xB7\xAE\x51\xBB\x6C\xB2\x4D\x41\x14\x00\x93\x4C\x9C\xCB\xE5\xC0\x54\xCF\xA0\x25\x8E\x02\xF9\xFD\xB0\xA2\x0D\xF5\x20\x98\x3C\x13\x2D\xAC\x56\xA2\xB0\xD6\x7E\x11\x92\xE9\x2E\xBA\x9E\x2E\x9A\x72\xB1\xBD\x19\x44\x6C\x61\x35\xA2\x9A\xB4\x16\x12\x69\x5A\x8C\xE1\xD7\x3E\xA4\x1A\xE8\x2F\x03\xF4\xAE\x61\x1D\x10\x1B\x2A\xA4\x8B\x7A\xC5\xFE\x05\xA6\xE1\xC0\xD6\xC8\xFE\x9E\xAE\x8F\x2B\xBA\x3D\x99\xF8\xD8\x73\x09\x58\x46\x6E\xA6\x9C\xF4\xD7\x27\xD3\x95\xDA\x37\x83\x72\x1C\xD3\x73\xE0\xA2\x47\x99\x03\x38\x5D\xD5\x49\x79\x00\x29\x1C\xC7\xEC\x9B\x20\x1C\x07\x24\x69\x57\x78\xB2\x39\xFC\x3A\x84\xA0\xB5\x9C\x7C\x8D\xBF\x2E\x93\x62\x27\xB7\x39\xDA\x17\x18\xAE\xBD\x3C\x09\x68\xFF\x84\x9B\x3C\xD5\xD6\x0B\x03\xE3\x57\x9E\x14\xF7\xD1\xEB\x4F\xC8\xBD\x87\x23\xB7\xB6\x49\x43\x79\x85\x5C\xBA\xEB\x92\x0B\xA1\xC6\xE8\x68\xA8\x4C\x16\xB1\x1A\x99\x0A\xE8\x53\x2C\x92\xBB\xA1\x09\x18\x75\x0C\x65\xA8\x7B\xCB\x23\xB7\x1A\xC2\x28\x85\xC3\x1B\xFF\xD0\x2B\x62\xEF\xA4\x7B\x09\x91\x98\x67\x8C\x14\x01\xCD\x68\x06\x6A\x63\x21\x75\x03\x80\x88\x8A\x6E\x81\xC6\x85\xF2\xA9\xA4\x2D\xE7\xF4\xA5\x24\x10\x47\x83\xCA\xCD\xF4\x8D\x79\x58\xB1\x06\x9B\xE7\x1A\x2A\xD9\x9D\x01\xD7\x94\x7D\xED\x03\x4A\xCA\xF0\xDB\xE8\xA9\x01\x3E\xF5\x56\x99\xC9\x1E\x8E\x49\x3D\xBB\xE5\x09\xB9\xE0\x4F\x49\x92\x3D\x16\x82\x40\xCC\xCC\x59\xC6\xE6\x3A\xED\x12\x2E\x69\x3C\x6C\x95\xB1\xFD\xAA\x1D\x7B\x7F\x86\xBE\x1E\x0E\x32\x46\xFB\xFB\x13\x8F\x75\x7F\x4C\x8B\x4B\x46\x63\xFE\x00\x34\x40\x70\xC1\xC3\xB9\xA1\xDD\xA6\x70\xE2\x04\xB3\x41\xBC\xE9\x80\x91\xEA\x64\x9C\x7A\xE1\x22\x03\xA9\x9C\x6E\x6F\x0E\x65\x4F\x6C\x87\x87\x5E\xF3\x6E\xA0\xF9\x75\xA5\x9B\x40\xE8\x53\xB2\x27\x9D\x4A\xB9\xC0\x77\x21\x8D\xFF\x87\xF2\xDE\xBC\x8C\xEF\x17\xDF\xB7\x49\x0B\xD1\xF2\x6E\x30\x0B\x1A\x0E\x4E\x76\xED\x11\xFC\xF5\xE9\x56\xB2\x7D\xBF\xC7\x6D\x0A\x93\x8C\xA5\xD0\xC0\xB6\x1D\xBE\x3A\x4E\x94\xA2\xD7\x6E\x6C\x0B\xC2\x8A\x7C\xFA\x20\xF3\xC4\xE4\xE5\xCD\x0D\xA8\xCB\x91\x92\xB1\x7C\x85\xEC\xB5\x14\x69\x66\x0E\x82\xE7\xCD\xCE\xC8\x2D\xA6\x51\x7F\x21\xC1\x35\x53\x85\x06\x4A\x5D\x9F\xAD\xBB\x1B\x5F\x74"),
   // "DigiCert Trusted Root G4"
   CertificateDer::from_slice(b"\x30\x82\x05\x90\x30\x82\x03\x78\xA0\x03\x02\x01\x02\x02\x10\x05\x9B\x1B\x57\x9E\x8E\x21\x32\xE2\x39\x07\xBD\xA7\x77\x75\x5C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x62\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x21\x30\x1F\x06\x03\x55\x04\x03\x13\x18\x44\x69\x67\x69\x43\x65\x72\x74\x20\x54\x72\x75\x73\x74\x65\x64\x20\x52\x6F\x6F\x74\x20\x47\x34\x30\x1E\x17\x0D\x31\x33\x30\x38\x30\x31\x31\x32\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x35\x31\x32\x30\x30\x30\x30\x5A\x30\x62\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x21\x30\x1F\x06\x03\x55\x04\x03\x13\x18\x44\x69\x67\x69\x43\x65\x72\x74\x20\x54\x72\x75\x73\x74\x65\x64\x20\x52\x6F\x6F\x74\x20\x47\x34\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xBF\xE6\x90\x73\x68\xDE\xBB\xE4\x5D\x4A\x3C\x30\x22\x30\x69\x33\xEC\xC2\xA7\x25\x2E\xC9\x21\x3D\xF2\x8A\xD8\x59\xC2\xE1\x29\xA7\x3D\x58\xAB\x76\x9A\xCD\xAE\x7B\x1B\x84\x0D\xC4\x30\x1F\xF3\x1B\xA4\x38\x16\xEB\x56\xC6\x97\x6D\x1D\xAB\xB2\x79\xF2\xCA\x11\xD2\xE4\x5F\xD6\x05\x3C\x52\x0F\x52\x1F\xC6\x9E\x15\xA5\x7E\xBE\x9F\xA9\x57\x16\x59\x55\x72\xAF\x68\x93\x70\xC2\xB2\xBA\x75\x99\x6A\x73\x32\x94\xD1\x10\x44\x10\x2E\xDF\x82\xF3\x07\x84\xE6\x74\x3B\x6D\x71\xE2\x2D\x0C\x1B\xEE\x20\xD5\xC9\x20\x1D\x63\x29\x2D\xCE\xEC\x5E\x4E\xC8\x93\xF8\x21\x61\x9B\x34\xEB\x05\xC6\x5E\xEC\x5B\x1A\xBC\xEB\xC9\xCF\xCD\xAC\x34\x40\x5F\xB1\x7A\x66\xEE\x77\xC8\x48\xA8\x66\x57\x57\x9F\x54\x58\x8E\x0C\x2B\xB7\x4F\xA7\x30\xD9\x56\xEE\xCA\x7B\x5D\xE3\xAD\xC9\x4F\x5E\xE5\x35\xE7\x31\xCB\xDA\x93\x5E\xDC\x8E\x8F\x80\xDA\xB6\x91\x98\x40\x90\x79\xC3\x78\xC7\xB6\xB1\xC4\xB5\x6A\x18\x38\x03\x10\x8D\xD8\xD4\x37\xA4\x2E\x05\x7D\x88\xF5\x82\x3E\x10\x91\x70\xAB\x55\x82\x41\x32\xD7\xDB\x04\x73\x2A\x6E\x91\x01\x7C\x21\x4C\xD4\xBC\xAE\x1B\x03\x75\x5D\x78\x66\xD9\x3A\x31\x44\x9A\x33\x40\xBF\x08\xD7\x5A\x49\xA4\xC2\xE6\xA9\xA0\x67\xDD\xA4\x27\xBC\xA1\x4F\x39\xB5\x11\x58\x17\xF7\x24\x5C\x46\x8F\x64\xF7\xC1\x69\x88\x76\x98\x76\x3D\x59\x5D\x42\x76\x87\x89\x97\x69\x7A\x48\xF0\xE0\xA2\x12\x1B\x66\x9A\x74\xCA\xDE\x4B\x1E\xE7\x0E\x63\xAE\xE6\xD4\xEF\x92\x92\x3A\x9E\x3D\xDC\x00\xE4\x45\x25\x89\xB6\x9A\x44\x19\x2B\x7E\xC0\x94\xB4\xD2\x61\x6D\xEB\x33\xD9\xC5\xDF\x4B\x04\x00\xCC\x7D\x1C\x95\xC3\x8F\xF7\x21\xB2\xB2\x11\xB7\xBB\x7F\xF2\xD5\x8C\x70\x2C\x41\x60\xAA\xB1\x63\x18\x44\x95\x1A\x76\x62\x7E\xF6\x80\xB0\xFB\xE8\x64\xA6\x33\xD1\x89\x07\xE1\xBD\xB7\xE6\x43\xA4\x18\xB8\xA6\x77\x01\xE1\x0F\x94\x0C\x21\x1D\xB2\x54\x29\x25\x89\x6C\xE5\x0E\x52\x51\x47\x74\xBE\x26\xAC\xB6\x41\x75\xDE\x7A\xAC\x5F\x8D\x3F\xC9\xBC\xD3\x41\x11\x12\x5B\xE5\x10\x50\xEB\x31\xC5\xCA\x72\x16\x22\x09\xDF\x7C\x4C\x75\x3F\x63\xEC\x21\x5F\xC4\x20\x51\x6B\x6F\xB1\xAB\x86\x8B\x4F\xC2\xD6\x45\x5F\x9D\x20\xFC\xA1\x1E\xC5\xC0\x8F\xA2\xB1\x7E\x0A\x26\x99\xF5\xE4\x69\x2F\x98\x1D\x2D\xF5\xD9\xA9\xB2\x1D\xE5\x1B\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xEC\xD7\xE3\x82\xD2\x71\x5D\x64\x4C\xDF\x2E\x67\x3F\xE7\xBA\x98\xAE\x1C\x0F\x4F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\xBB\x61\xD9\x7D\xA9\x6C\xBE\x17\xC4\x91\x1B\xC3\xA1\xA2\x00\x8D\xE3\x64\x68\x0F\x56\xCF\x77\xAE\x70\xF9\xFD\x9A\x4A\x99\xB9\xC9\x78\x5C\x0C\x0C\x5F\xE4\xE6\x14\x29\x56\x0B\x36\x49\x5D\x44\x63\xE0\xAD\x9C\x96\x18\x66\x1B\x23\x0D\x3D\x79\xE9\x6D\x6B\xD6\x54\xF8\xD2\x3C\xC1\x43\x40\xAE\x1D\x50\xF5\x52\xFC\x90\x3B\xBB\x98\x99\x69\x6B\xC7\xC1\xA7\xA8\x68\xA4\x27\xDC\x9D\xF9\x27\xAE\x30\x85\xB9\xF6\x67\x4D\x3A\x3E\x8F\x59\x39\x22\x53\x44\xEB\xC8\x5D\x03\xCA\xED\x50\x7A\x7D\x62\x21\x0A\x80\xC8\x73\x66\xD1\xA0\x05\x60\x5F\xE8\xA5\xB4\xA7\xAF\xA8\xF7\x6D\x35\x9C\x7C\x5A\x8A\xD6\xA2\x38\x99\xF3\x78\x8B\xF4\x4D\xD2\x20\x0B\xDE\x04\xEE\x8C\x9B\x47\x81\x72\x0D\xC0\x14\x32\xEF\x30\x59\x2E\xAE\xE0\x71\xF2\x56\xE4\x6A\x97\x6F\x92\x50\x6D\x96\x8D\x68\x7A\x9A\xB2\x36\x14\x7A\x06\xF2\x24\xB9\x09\x11\x50\xD7\x08\xB1\xB8\x89\x7A\x84\x23\x61\x42\x29\xE5\xA3\xCD\xA2\x20\x41\xD7\xD1\x9C\x64\xD9\xEA\x26\xA1\x8B\x14\xD7\x4C\x19\xB2\x50\x41\x71\x3D\x3F\x4D\x70\x23\x86\x0C\x4A\xDC\x81\xD2\xCC\x32\x94\x84\x0D\x08\x09\x97\x1C\x4F\xC0\xEE\x6B\x20\x74\x30\xD2\xE0\x39\x34\x10\x85\x21\x15\x01\x08\xE8\x55\x32\xDE\x71\x49\xD9\x28\x17\x50\x4D\xE6\xBE\x4D\xD1\x75\xAC\xD0\xCA\xFB\x41\xB8\x43\xA5\xAA\xD3\xC3\x05\x44\x4F\x2C\x36\x9B\xE2\xFA\xE2\x45\xB8\x23\x53\x6C\x06\x6F\x67\x55\x7F\x46\xB5\x4C\x3F\x6E\x28\x5A\x79\x26\xD2\xA4\xA8\x62\x97\xD2\x1E\xE2\xED\x4A\x8B\xBC\x1B\xFD\x47\x4A\x0D\xDF\x67\x66\x7E\xB2\x5B\x41\xD0\x3B\xE4\xF4\x3B\xF4\x04\x63\xE9\xEF\xC2\x54\x00\x51\xA0\x8A\x2A\xC9\xCE\x78\xCC\xD5\xEA\x87\x04\x18\xB3\xCE\xAF\x49\x88\xAF\xF3\x92\x99\xB6\xB3\xE6\x61\x0F\xD2\x85\x00\xE7\x50\x1A\xE4\x1B\x95\x9D\x19\xA1\xB9\x9C\xB1\x9B\xB1\x00\x1E\xEF\xD0\x0F\x4F\x42\x6C\xC9\x0A\xBC\xEE\x43\xFA\x3A\x71\xA5\xC8\x4D\x26\xA5\x35\xFD\x89\x5D\xBC\x85\x62\x1D\x32\xD2\xA0\x2B\x54\xED\x9A\x57\xC1\xDB\xFA\x10\xCF\x19\xB7\x8B\x4A\x1B\x8F\x01\xB6\x27\x95\x53\xE8\xB6\x89\x6D\x5B\xBC\x68\xD4\x23\xE8\x8B\x51\xA2\x56\xF9\xF0\xA6\x80\xA0\xD6\x1E\xB3\xBC\x0F\x0F\x53\x75\x29\xAA\xEA\x13\x77\xE4\xDE\x8C\x81\x21\xAD\x07\x10\x47\x11\xAD\x87\x3D\x07\xD1\x75\xBC\xCF\xF3\x66\x7E"),
   // "AC RAIZ FNMT-RCM SERVIDORES SEGUROS"
   CertificateDer::from_slice(b"\x30\x82\x02\x6E\x30\x82\x01\xF3\xA0\x03\x02\x01\x02\x02\x10\x62\xF6\x32\x6C\xE5\xC4\xE3\x68\x5C\x1B\x62\xDD\x9C\x2E\x9D\x95\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x78\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x31\x0E\x30\x0C\x06\x03\x55\x04\x0B\x0C\x05\x43\x65\x72\x65\x73\x31\x18\x30\x16\x06\x03\x55\x04\x61\x0C\x0F\x56\x41\x54\x45\x53\x2D\x51\x32\x38\x32\x36\x30\x30\x34\x4A\x31\x2C\x30\x2A\x06\x03\x55\x04\x03\x0C\x23\x41\x43\x20\x52\x41\x49\x5A\x20\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x20\x53\x45\x52\x56\x49\x44\x4F\x52\x45\x53\x20\x53\x45\x47\x55\x52\x4F\x53\x30\x1E\x17\x0D\x31\x38\x31\x32\x32\x30\x30\x39\x33\x37\x33\x33\x5A\x17\x0D\x34\x33\x31\x32\x32\x30\x30\x39\x33\x37\x33\x33\x5A\x30\x78\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x31\x0E\x30\x0C\x06\x03\x55\x04\x0B\x0C\x05\x43\x65\x72\x65\x73\x31\x18\x30\x16\x06\x03\x55\x04\x61\x0C\x0F\x56\x41\x54\x45\x53\x2D\x51\x32\x38\x32\x36\x30\x30\x34\x4A\x31\x2C\x30\x2A\x06\x03\x55\x04\x03\x0C\x23\x41\x43\x20\x52\x41\x49\x5A\x20\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x20\x53\x45\x52\x56\x49\x44\x4F\x52\x45\x53\x20\x53\x45\x47\x55\x52\x4F\x53\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xF6\xBA\x57\x53\xC8\xCA\xAB\xDF\x36\x4A\x52\x21\xE4\x97\xD2\x83\x67\x9E\xF0\x65\x51\xD0\x5E\x87\xC7\x47\xB1\x59\xF2\x57\x47\x9B\x00\x02\x93\x44\x17\x69\xDB\x42\xC7\xB1\xB2\x3A\x18\x0E\xB4\x5D\x8C\xB3\x66\x5D\xA1\x34\xF9\x36\x2C\x49\xDB\xF3\x46\xFC\xB3\x44\x69\x44\x13\x66\xFD\xD7\xC5\xFD\xAF\x36\x4D\xCE\x03\x4D\x07\x71\xCF\xAF\x6A\x05\xD2\xA2\x43\x5A\x0A\x52\x6F\x01\x03\x4E\x8E\x8B\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x01\xB9\x2F\xEF\xBF\x11\x86\x60\xF2\x4F\xD0\x41\x6E\xAB\x73\x1F\xE7\xD2\x6E\x49\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\xAE\x4A\xE3\x2B\x40\xC3\x74\x11\xF2\x95\xAD\x16\x23\xDE\x4E\x0C\x1A\xE6\x5D\xA5\x24\x5E\x6B\x44\x7B\xFC\x38\xE2\x4F\xCB\x9C\x45\x17\x11\x4C\x14\x27\x26\x55\x39\x75\x4A\x03\xCC\x13\x90\x9F\x92\x02\x31\x00\xFA\x4A\x6C\x60\x88\x73\xF3\xEE\xB8\x98\x62\xA9\xCE\x2B\xC2\xD9\x8A\xA6\x70\x31\x1D\xAF\xB0\x94\x4C\xEB\x4F\xC6\xE3\xD1\xF3\x62\xA7\x3C\xFF\x93\x2E\x07\x5C\x49\x01\x67\x69\x12\x02\x72\xBF\xE7"),
   // "Trustwave Global ECC P384 Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x02\x9D\x30\x82\x02\x24\xA0\x03\x02\x01\x02\x02\x0C\x08\xBD\x85\x97\x6C\x99\x27\xA4\x80\x68\x47\x3B\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x81\x91\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x08\x13\x08\x49\x6C\x6C\x69\x6E\x6F\x69\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x43\x68\x69\x63\x61\x67\x6F\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x48\x6F\x6C\x64\x69\x6E\x67\x73\x2C\x20\x49\x6E\x63\x2E\x31\x3A\x30\x38\x06\x03\x55\x04\x03\x13\x31\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x45\x43\x43\x20\x50\x33\x38\x34\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x37\x30\x38\x32\x33\x31\x39\x33\x36\x34\x33\x5A\x17\x0D\x34\x32\x30\x38\x32\x33\x31\x39\x33\x36\x34\x33\x5A\x30\x81\x91\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x08\x13\x08\x49\x6C\x6C\x69\x6E\x6F\x69\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x43\x68\x69\x63\x61\x67\x6F\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x48\x6F\x6C\x64\x69\x6E\x67\x73\x2C\x20\x49\x6E\x63\x2E\x31\x3A\x30\x38\x06\x03\x55\x04\x03\x13\x31\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x45\x43\x43\x20\x50\x33\x38\x34\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x6B\xDA\x0D\x75\x35\x08\x31\x47\x05\xAE\x45\x99\x55\xF1\x11\x13\x2E\x4A\xF8\x10\x31\x23\xA3\x7E\x83\xD3\x7F\x28\x08\x3A\x26\x1A\x3A\xCF\x97\x82\x1F\x80\xB7\x27\x09\x8F\xD1\x8E\x30\xC4\x0A\x9B\x0E\xAC\x58\x04\xAB\xF7\x36\x7D\x94\x23\xA4\x9B\x0A\x8A\x8B\xAB\xEB\xFD\x39\x25\x66\xF1\x5E\xFE\x8C\xAE\x8D\x41\x79\x9D\x09\x60\xCE\x28\xA9\xD3\x8A\x6D\xF3\xD6\x45\xD4\xF2\x98\x84\x38\x65\xA0\xA3\x43\x30\x41\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0F\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x05\x03\x03\x07\x06\x00\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x55\xA9\x84\x89\xD2\xC1\x32\xBD\x18\xCB\x6C\xA6\x07\x4E\xC8\xE7\x9D\xBE\x82\x90\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x67\x00\x30\x64\x02\x30\x37\x01\x92\x97\x45\x12\x7E\xA0\xF3\x3E\xAD\x19\x3A\x72\xDD\xF4\x50\x93\x03\x12\xBE\x44\xD2\x4F\x41\xA4\x8C\x9C\x9D\x1F\xA3\xF6\xC2\x92\xE7\x48\x14\xFE\x4E\x9B\xA5\x91\x57\xAE\xC6\x37\x72\xBB\x02\x30\x67\x25\x0A\xB1\x0C\x5E\xEE\xA9\x63\x92\x6F\xE5\x90\x0B\xFE\x66\x22\xCA\x47\xFD\x8A\x31\xF7\x83\xFE\x7A\xBF\x10\xBE\x18\x2B\x1E\x8F\xF6\x29\x1E\x94\x59\xEF\x8E\x21\x37\xCB\x51\x98\xA5\x6E\x4B"),
   // "Actalis Authentication Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\xBB\x30\x82\x03\xA3\xA0\x03\x02\x01\x02\x02\x08\x57\x0A\x11\x97\x42\xC4\xE3\xCC\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x6B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x49\x54\x31\x0E\x30\x0C\x06\x03\x55\x04\x07\x0C\x05\x4D\x69\x6C\x61\x6E\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x0C\x1A\x41\x63\x74\x61\x6C\x69\x73\x20\x53\x2E\x70\x2E\x41\x2E\x2F\x30\x33\x33\x35\x38\x35\x32\x30\x39\x36\x37\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x41\x63\x74\x61\x6C\x69\x73\x20\x41\x75\x74\x68\x65\x6E\x74\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x31\x30\x39\x32\x32\x31\x31\x32\x32\x30\x32\x5A\x17\x0D\x33\x30\x30\x39\x32\x32\x31\x31\x32\x32\x30\x32\x5A\x30\x6B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x49\x54\x31\x0E\x30\x0C\x06\x03\x55\x04\x07\x0C\x05\x4D\x69\x6C\x61\x6E\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x0C\x1A\x41\x63\x74\x61\x6C\x69\x73\x20\x53\x2E\x70\x2E\x41\x2E\x2F\x30\x33\x33\x35\x38\x35\x32\x30\x39\x36\x37\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x41\x63\x74\x61\x6C\x69\x73\x20\x41\x75\x74\x68\x65\x6E\x74\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA7\xC6\xC4\xA5\x29\xA4\x2C\xEF\xE5\x18\xC5\xB0\x50\xA3\x6F\x51\x3B\x9F\x0A\x5A\xC9\xC2\x48\x38\x0A\xC2\x1C\xA0\x18\x7F\x91\xB5\x87\xB9\x40\x3F\xDD\x1D\x68\x1F\x08\x83\xD5\x2D\x1E\x88\xA0\xF8\x8F\x56\x8F\x6D\x99\x02\x92\x90\x16\xD5\x5F\x08\x6C\x89\xD7\xE1\xAC\xBC\x20\xC2\xB1\xE0\x83\x51\x8A\x69\x4D\x00\x96\x5A\x6F\x2F\xC0\x44\x7E\xA3\x0E\xE4\x91\xCD\x58\xEE\xDC\xFB\xC7\x1E\x45\x47\xDD\x27\xB9\x08\x01\x9F\xA6\x21\x1D\xF5\x41\x2D\x2F\x4C\xFD\x28\xAD\xE0\x8A\xAD\x22\xB4\x56\x65\x8E\x86\x54\x8F\x93\x43\x29\xDE\x39\x46\x78\xA3\x30\x23\xBA\xCD\xF0\x7D\x13\x57\xC0\x5D\xD2\x83\x6B\x48\x4C\xC4\xAB\x9F\x80\x5A\x5B\x3A\xBD\xC9\xA7\x22\x3F\x80\x27\x33\x5B\x0E\xB7\x8A\x0C\x5D\x07\x37\x08\xCB\x6C\xD2\x7A\x47\x22\x44\x35\xC5\xCC\xCC\x2E\x8E\xDD\x2A\xED\xB7\x7D\x66\x0D\x5F\x61\x51\x22\x55\x1B\xE3\x46\xE3\xE3\x3D\xD0\x35\x62\x9A\xDB\xAF\x14\xC8\x5B\xA1\xCC\x89\x1B\xE1\x30\x26\xFC\xA0\x9B\x1F\x81\xA7\x47\x1F\x04\xEB\xA3\x39\x92\x06\x9F\x99\xD3\xBF\xD3\xEA\x4F\x50\x9C\x19\xFE\x96\x87\x1E\x3C\x65\xF6\xA3\x18\x24\x83\x86\x10\xE7\x54\x3E\xA8\x3A\x76\x24\x4F\x81\x21\xC5\xE3\x0F\x02\xF8\x93\x94\x47\x20\xBB\xFE\xD4\x0E\xD3\x68\xB9\xDD\xC4\x7A\x84\x82\xE3\x53\x54\x79\xDD\xDB\x9C\xD2\xF2\x07\x9B\x2E\xB6\xBC\x3E\xED\x85\x6D\xEF\x25\x11\xF2\x97\x1A\x42\x61\xF7\x4A\x97\xE8\x8B\xB1\x10\x07\xFA\x65\x81\xB2\xA2\x39\xCF\xF7\x3C\xFF\x18\xFB\xC6\xF1\x5A\x8B\x59\xE2\x02\xAC\x7B\x92\xD0\x4E\x14\x4F\x59\x45\xF6\x0C\x5E\x28\x5F\xB0\xE8\x3F\x45\xCF\xCF\xAF\x9B\x6F\xFB\x84\xD3\x77\x5A\x95\x6F\xAC\x94\x84\x9E\xEE\xBC\xC0\x4A\x8F\x4A\x93\xF8\x44\x21\xE2\x31\x45\x61\x50\x4E\x10\xD8\xE3\x35\x7C\x4C\x19\xB4\xDE\x05\xBF\xA3\x06\x9F\xC8\xB5\xCD\xE4\x1F\xD7\x17\x06\x0D\x7A\x95\x74\x55\x0D\x68\x1A\xFC\x10\x1B\x62\x64\x9D\x6D\xE0\x95\xA0\xC3\x94\x07\x57\x0D\x14\xE6\xBD\x05\xFB\xB8\x9F\xE6\xDF\x8B\xE2\xC6\xE7\x7E\x96\xF6\x53\xC5\x80\x34\x50\x28\x58\xF0\x12\x50\x71\x17\x30\xBA\xE6\x78\x63\xBC\xF4\xB2\xAD\x9B\x2B\xB2\xFE\xE1\x39\x8C\x5E\xBA\x0B\x20\x94\xDE\x7B\x83\xB8\xFF\xE3\x56\x8D\xB7\x11\xE9\x3B\x8C\xF2\xB1\xC1\x5D\x9D\xA4\x0B\x4C\x2B\xD9\xB2\x18\xF5\xB5\x9F\x4B\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x52\xD8\x88\x3A\xC8\x9F\x78\x66\xED\x89\xF3\x7B\x38\x70\x94\xC9\x02\x02\x36\xD0\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x52\xD8\x88\x3A\xC8\x9F\x78\x66\xED\x89\xF3\x7B\x38\x70\x94\xC9\x02\x02\x36\xD0\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x0B\x7B\x72\x87\xC0\x60\xA6\x49\x4C\x88\x58\xE6\x1D\x88\xF7\x14\x64\x48\xA6\xD8\x58\x0A\x0E\x4F\x13\x35\xDF\x35\x1D\xD4\xED\x06\x31\xC8\x81\x3E\x6A\xD5\xDD\x3B\x1A\x32\xEE\x90\x3D\x11\xD2\x2E\xF4\x8E\xC3\x63\x2E\x23\x66\xB0\x67\xBE\x6F\xB6\xC0\x13\x39\x60\xAA\xA2\x34\x25\x93\x75\x52\xDE\xA7\x9D\xAD\x0E\x87\x89\x52\x71\x6A\x16\x3C\x19\x1D\x83\xF8\x9A\x29\x65\xBE\xF4\x3F\x9A\xD9\xF0\xF3\x5A\x87\x21\x71\x80\x4D\xCB\xE0\x38\x9B\x3F\xBB\xFA\xE0\x30\x4D\xCF\x86\xD3\x65\x10\x19\x18\xD1\x97\x02\xB1\x2B\x72\x42\x68\xAC\xA0\xBD\x4E\x5A\xDA\x18\xBF\x6B\x98\x81\xD0\xFD\x9A\xBE\x5E\x15\x48\xCD\x11\x15\xB9\xC0\x29\x5C\xB4\xE8\x88\xF7\x3E\x36\xAE\xB7\x62\xFD\x1E\x62\xDE\x70\x78\x10\x1C\x48\x5B\xDA\xBC\xA4\x38\xBA\x67\xED\x55\x3E\x5E\x57\xDF\xD4\x03\x40\x4C\x81\xA4\xD2\x4F\x63\xA7\x09\x42\x09\x14\xFC\x00\xA9\xC2\x80\x73\x4F\x2E\xC0\x40\xD9\x11\x7B\x48\xEA\x7A\x02\xC0\xD3\xEB\x28\x01\x26\x58\x74\xC1\xC0\x73\x22\x6D\x93\x95\xFD\x39\x7D\xBB\x2A\xE3\xF6\x82\xE3\x2C\x97\x5F\x4E\x1F\x91\x94\xFA\xFE\x2C\xA3\xD8\x76\x1A\xB8\x4D\xB2\x38\x4F\x9B\xFA\x1D\x48\x60\x79\x26\xE2\xF3\xFD\xA9\xD0\x9A\xE8\x70\x8F\x49\x7A\xD6\xE5\xBD\x0A\x0E\xDB\x2D\xF3\x8D\xBF\xEB\xE3\xA4\x7D\xCB\xC7\x95\x71\xE8\xDA\xA3\x7C\xC5\xC2\xF8\x74\x92\x04\x1B\x86\xAC\xA4\x22\x53\x40\xB6\xAC\xFE\x4C\x76\xCF\xFB\x94\x32\xC0\x35\x9F\x76\x3F\x6E\xE5\x90\x6E\xA0\xA6\x26\xA2\xB8\x2C\xBE\xD1\x2B\x85\xFD\xA7\x68\xC8\xBA\x01\x2B\xB1\x6C\x74\x1D\xB8\x73\x95\xE7\xEE\xB7\xC7\x25\xF0\x00\x4C\x00\xB2\x7E\xB6\x0B\x8B\x1C\xF3\xC0\x50\x9E\x25\xB9\xE0\x08\xDE\x36\x66\xFF\x37\xA5\xD1\xBB\x54\x64\x2C\xC9\x27\xB5\x4B\x92\x7E\x65\xFF\xD3\x2D\xE1\xB9\x4E\xBC\x7F\xA4\x41\x21\x90\x41\x77\xA6\x39\x1F\xEA\x9E\xE3\x9F\xD0\x66\x6F\x05\xEC\xAA\x76\x7E\xBF\x6B\x16\xA0\xEB\xB5\xC7\xFC\x92\x54\x2F\x2B\x11\x27\x25\x37\x78\x4C\x51\x6A\xB0\xF3\xCC\x58\x5D\x14\xF1\x6A\x48\x15\xFF\xC2\x07\xB6\xB1\x8D\x0F\x8E\x5C\x50\x46\xB3\x3D\xBF\x01\x98\x4F\xB2\x59\x54\x47\x3E\x34\x7B\x78\x6D\x56\x93\x2E\x73\xEA\x66\x28\x78\xCD\x1D\x14\xBF\xA0\x8F\x2F\x2E\xB8\x2E\x8E\xF2\x14\x8A\xCC\xE9\xB5\x7C\xFB\x6C\x9D\x0C\xA5\xE1\x96"),
   // "Starfield Services Root Certificate Authority - G2"
   CertificateDer::from_slice(b"\x30\x82\x03\xEF\x30\x82\x02\xD7\xA0\x03\x02\x01\x02\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x98\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x10\x30\x0E\x06\x03\x55\x04\x08\x13\x07\x41\x72\x69\x7A\x6F\x6E\x61\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x53\x63\x6F\x74\x74\x73\x64\x61\x6C\x65\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x3B\x30\x39\x06\x03\x55\x04\x03\x13\x32\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x2D\x20\x47\x32\x30\x1E\x17\x0D\x30\x39\x30\x39\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x37\x31\x32\x33\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x98\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x10\x30\x0E\x06\x03\x55\x04\x08\x13\x07\x41\x72\x69\x7A\x6F\x6E\x61\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x53\x63\x6F\x74\x74\x73\x64\x61\x6C\x65\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x3B\x30\x39\x06\x03\x55\x04\x03\x13\x32\x53\x74\x61\x72\x66\x69\x65\x6C\x64\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x2D\x20\x47\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xD5\x0C\x3A\xC4\x2A\xF9\x4E\xE2\xF5\xBE\x19\x97\x5F\x8E\x88\x53\xB1\x1F\x3F\xCB\xCF\x9F\x20\x13\x6D\x29\x3A\xC8\x0F\x7D\x3C\xF7\x6B\x76\x38\x63\xD9\x36\x60\xA8\x9B\x5E\x5C\x00\x80\xB2\x2F\x59\x7F\xF6\x87\xF9\x25\x43\x86\xE7\x69\x1B\x52\x9A\x90\xE1\x71\xE3\xD8\x2D\x0D\x4E\x6F\xF6\xC8\x49\xD9\xB6\xF3\x1A\x56\xAE\x2B\xB6\x74\x14\xEB\xCF\xFB\x26\xE3\x1A\xBA\x1D\x96\x2E\x6A\x3B\x58\x94\x89\x47\x56\xFF\x25\xA0\x93\x70\x53\x83\xDA\x84\x74\x14\xC3\x67\x9E\x04\x68\x3A\xDF\x8E\x40\x5A\x1D\x4A\x4E\xCF\x43\x91\x3B\xE7\x56\xD6\x00\x70\xCB\x52\xEE\x7B\x7D\xAE\x3A\xE7\xBC\x31\xF9\x45\xF6\xC2\x60\xCF\x13\x59\x02\x2B\x80\xCC\x34\x47\xDF\xB9\xDE\x90\x65\x6D\x02\xCF\x2C\x91\xA6\xA6\xE7\xDE\x85\x18\x49\x7C\x66\x4E\xA3\x3A\x6D\xA9\xB5\xEE\x34\x2E\xBA\x0D\x03\xB8\x33\xDF\x47\xEB\xB1\x6B\x8D\x25\xD9\x9B\xCE\x81\xD1\x45\x46\x32\x96\x70\x87\xDE\x02\x0E\x49\x43\x85\xB6\x6C\x73\xBB\x64\xEA\x61\x41\xAC\xC9\xD4\x54\xDF\x87\x2F\xC7\x22\xB2\x26\xCC\x9F\x59\x54\x68\x9F\xFC\xBE\x2A\x2F\xC4\x55\x1C\x75\x40\x60\x17\x85\x02\x55\x39\x8B\x7F\x05\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x9C\x5F\x00\xDF\xAA\x01\xD7\x30\x2B\x38\x88\xA2\xB8\x6D\x4A\x9C\xF2\x11\x91\x83\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x4B\x36\xA6\x84\x77\x69\xDD\x3B\x19\x9F\x67\x23\x08\x6F\x0E\x61\xC9\xFD\x84\xDC\x5F\xD8\x36\x81\xCD\xD8\x1B\x41\x2D\x9F\x60\xDD\xC7\x1A\x68\xD9\xD1\x6E\x86\xE1\x88\x23\xCF\x13\xDE\x43\xCF\xE2\x34\xB3\x04\x9D\x1F\x29\xD5\xBF\xF8\x5E\xC8\xD5\xC1\xBD\xEE\x92\x6F\x32\x74\xF2\x91\x82\x2F\xBD\x82\x42\x7A\xAD\x2A\xB7\x20\x7D\x4D\xBC\x7A\x55\x12\xC2\x15\xEA\xBD\xF7\x6A\x95\x2E\x6C\x74\x9F\xCF\x1C\xB4\xF2\xC5\x01\xA3\x85\xD0\x72\x3E\xAD\x73\xAB\x0B\x9B\x75\x0C\x6D\x45\xB7\x8E\x94\xAC\x96\x37\xB5\xA0\xD0\x8F\x15\x47\x0E\xE3\xE8\x83\xDD\x8F\xFD\xEF\x41\x01\x77\xCC\x27\xA9\x62\x85\x33\xF2\x37\x08\xEF\x71\xCF\x77\x06\xDE\xC8\x19\x1D\x88\x40\xCF\x7D\x46\x1D\xFF\x1E\xC7\xE1\xCE\xFF\x23\xDB\xC6\xFA\x8D\x55\x4E\xA9\x02\xE7\x47\x11\x46\x3E\xF4\xFD\xBD\x7B\x29\x26\xBB\xA9\x61\x62\x37\x28\xB6\x2D\x2A\xF6\x10\x86\x64\xC9\x70\xA7\xD2\xAD\xB7\x29\x70\x79\xEA\x3C\xDA\x63\x25\x9F\xFD\x68\xB7\x30\xEC\x70\xFB\x75\x8A\xB7\x6D\x60\x67\xB2\x1E\xC8\xB9\xE9\xD8\xA8\x6F\x02\x8B\x67\x0D\x4D\x26\x57\x71\xDA\x20\xFC\xC1\x4A\x50\x8D\xB1\x28\xBA"),
   // "BJCA Global Root CA2"
   CertificateDer::from_slice(b"\x30\x82\x02\x25\x30\x82\x01\xAB\xA0\x03\x02\x01\x02\x02\x10\x2C\x17\x08\x7D\x64\x2A\xC0\xFE\x85\x18\x59\x06\xCF\xB4\x4A\xEB\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x54\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x26\x30\x24\x06\x03\x55\x04\x0A\x0C\x1D\x42\x45\x49\x4A\x49\x4E\x47\x20\x43\x45\x52\x54\x49\x46\x49\x43\x41\x54\x45\x20\x41\x55\x54\x48\x4F\x52\x49\x54\x59\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x0C\x14\x42\x4A\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x32\x30\x1E\x17\x0D\x31\x39\x31\x32\x31\x39\x30\x33\x31\x38\x32\x31\x5A\x17\x0D\x34\x34\x31\x32\x31\x32\x30\x33\x31\x38\x32\x31\x5A\x30\x54\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x26\x30\x24\x06\x03\x55\x04\x0A\x0C\x1D\x42\x45\x49\x4A\x49\x4E\x47\x20\x43\x45\x52\x54\x49\x46\x49\x43\x41\x54\x45\x20\x41\x55\x54\x48\x4F\x52\x49\x54\x59\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x0C\x14\x42\x4A\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x32\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x9D\xCB\x80\x91\x8D\x53\x67\xB5\xB9\x50\xB1\x03\xF8\xE5\x49\x1F\x41\x22\x09\xB0\x51\x52\x58\xD6\x2B\x34\x8F\xC5\x12\x46\x14\xC5\x8B\x2F\x2C\x84\xFF\x2C\x6E\xA8\xD5\xF1\x09\xE3\x03\x21\x14\xC4\x43\x3D\x7C\xC1\x2C\xC4\x4B\x6A\x4A\xCD\xE9\x87\xE0\x7D\xF6\x22\xBE\xFA\x4A\x51\xB8\x30\x8A\xFD\xE1\xDE\x18\x12\x0A\xF6\x47\xB7\xE7\x17\xBF\x27\x8A\xD4\x41\x4C\x96\x3C\x60\x96\xC1\xFD\x15\x1C\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD2\x4A\xB1\x51\x7F\x06\xF0\xD1\x82\x1F\x4E\x6E\x5F\xAB\x83\xFC\x48\xD4\xB0\x91\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x1A\xBC\x5B\xD7\xFE\xA9\xD2\x54\x0E\x4A\x5D\xD2\x6D\xB1\x40\xDC\xF4\x43\xD5\xD2\x4A\x99\x19\x12\x56\x80\xF7\x83\x34\xE1\x35\x4E\x48\x6D\x04\x0F\x57\x31\x30\x30\x2D\xB1\xAA\x9D\x03\x38\xDB\x06\x02\x31\x00\xCB\xCC\x87\x53\xCB\x7A\xDF\x20\x51\x73\x90\xC0\xA8\x5B\x61\xD0\xC5\x50\x39\xFD\x85\xFE\xC1\xE3\x78\xF8\xA6\xD6\x4B\xBD\x9B\x87\x8F\x0F\xE5\xD6\x53\x96\xAB\x3C\xC8\x40\xDA\x61\xF7\x53\xA3\xF7"),
   // "Telekom Security TLS ECC Root 2020"
   CertificateDer::from_slice(b"\x30\x82\x02\x42\x30\x82\x01\xC9\xA0\x03\x02\x01\x02\x02\x10\x36\x3A\x96\x8C\xC9\x5C\xB2\x58\xCD\xD0\x01\x5D\xC5\xE5\x57\x00\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x63\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x27\x30\x25\x06\x03\x55\x04\x0A\x0C\x1E\x44\x65\x75\x74\x73\x63\x68\x65\x20\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x47\x6D\x62\x48\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x0C\x22\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x32\x30\x32\x30\x30\x1E\x17\x0D\x32\x30\x30\x38\x32\x35\x30\x37\x34\x38\x32\x30\x5A\x17\x0D\x34\x35\x30\x38\x32\x35\x32\x33\x35\x39\x35\x39\x5A\x30\x63\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x27\x30\x25\x06\x03\x55\x04\x0A\x0C\x1E\x44\x65\x75\x74\x73\x63\x68\x65\x20\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x47\x6D\x62\x48\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x0C\x22\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x32\x30\x32\x30\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xCE\xBF\xFE\x57\xA8\xBF\xD5\xAA\xF7\x10\x9A\xCD\xBC\xD1\x11\xA2\xBD\x67\x42\xCC\x90\xEB\x15\x18\x90\xD9\xA2\xCD\x0C\x2A\x25\xEB\x3E\x4F\xCE\xB5\xD2\x8F\x0F\xF3\x35\xDA\x43\x8B\x02\x80\xBE\x6F\x51\x24\x1D\x0F\x6B\x2B\xCA\x9F\xC2\x6F\x50\x32\xE5\x37\x20\xB6\x20\xFF\x88\x0D\x0F\x6D\x49\xBB\xDB\x06\xA4\x87\x90\x92\x94\xF4\x09\xD0\xCF\x7F\xC8\x80\x0B\xC1\x97\xB3\xBB\x35\x27\xC9\xC2\x1B\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE3\x72\xCC\x6E\x95\x99\x47\xB1\xE6\xB3\x61\x4C\xD1\xCB\xAB\xE3\xBA\xCD\xDE\x9F\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x67\x00\x30\x64\x02\x30\x75\x52\x8B\xB7\xA4\x10\x4F\xAE\x4A\x10\x8B\xB2\x84\x5B\x42\xE1\xE6\x2A\x36\x02\xDA\xA0\x6E\x19\x3F\x25\xBF\xDA\x59\x32\x8E\xE4\xFB\x90\xDC\x93\x64\xCE\xAD\xB4\x41\x47\x60\xE2\xCF\xA7\xCB\x1E\x02\x30\x37\x41\x8C\x66\xDF\x41\x6B\xD6\x83\x00\x41\xFD\x2F\x5A\xF7\x50\xB4\x67\xD1\x2C\xA8\x71\xD7\x43\xCA\x9C\x27\x24\x91\x83\x48\x0D\xCF\xCD\xF7\x54\x81\xAF\xEC\x7F\xE4\x67\xDB\xB8\x90\xEE\xDD\x25"),
   // "Autoridad de Certificacion Firmaprofesional CIF A62634068"
   CertificateDer::from_slice(b"\x30\x82\x06\x14\x30\x82\x03\xFC\xA0\x03\x02\x01\x02\x02\x08\x1B\x70\xE9\xD2\xFF\xAE\x6C\x71\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x42\x30\x40\x06\x03\x55\x04\x03\x0C\x39\x41\x75\x74\x6F\x72\x69\x64\x61\x64\x20\x64\x65\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x63\x69\x6F\x6E\x20\x46\x69\x72\x6D\x61\x70\x72\x6F\x66\x65\x73\x69\x6F\x6E\x61\x6C\x20\x43\x49\x46\x20\x41\x36\x32\x36\x33\x34\x30\x36\x38\x30\x1E\x17\x0D\x31\x34\x30\x39\x32\x33\x31\x35\x32\x32\x30\x37\x5A\x17\x0D\x33\x36\x30\x35\x30\x35\x31\x35\x32\x32\x30\x37\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x42\x30\x40\x06\x03\x55\x04\x03\x0C\x39\x41\x75\x74\x6F\x72\x69\x64\x61\x64\x20\x64\x65\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x63\x69\x6F\x6E\x20\x46\x69\x72\x6D\x61\x70\x72\x6F\x66\x65\x73\x69\x6F\x6E\x61\x6C\x20\x43\x49\x46\x20\x41\x36\x32\x36\x33\x34\x30\x36\x38\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xCA\x96\x6B\x8E\xEA\xF8\xFB\xF1\xA2\x35\xE0\x7F\x4C\xDA\xE0\xC3\x52\xD7\x7D\xB6\x10\xC8\x02\x5E\xB3\x43\x2A\xC4\x4F\x6A\xB2\xCA\x1C\x5D\x28\x9A\x78\x11\x1A\x69\x59\x57\xAF\xB5\x20\x42\xE4\x8B\x0F\xE6\xDF\x5B\xA6\x03\x92\x2F\xF5\x11\xE4\x62\xD7\x32\x71\x38\xD9\x04\x0C\x71\xAB\x3D\x51\x7E\x0F\x07\xDF\x63\x05\x5C\xE9\xBF\x94\x6F\xC1\x29\x82\xC0\xB4\xDA\x51\xB0\xC1\x3C\xBB\xAD\x37\x4A\x5C\xCA\xF1\x4B\x36\x0E\x24\xAB\xBF\xC3\x84\x77\xFD\xA8\x50\xF4\xB1\xE7\xC6\x2F\xD2\x2D\x59\x8D\x7A\x0A\x4E\x96\x69\x52\x02\xAA\x36\x98\xEC\xFC\xFA\x14\x83\x0C\x37\x1F\xC9\x92\x37\x7F\xD7\x81\x2D\xE5\xC4\xB9\xE0\x3E\x34\xFE\x67\xF4\x3E\x66\xD1\xD3\xF4\x40\xCF\x5E\x62\x34\x0F\x70\x06\x3E\x20\x18\x5A\xCE\xF7\x72\x1B\x25\x6C\x93\x74\x14\x93\xA3\x73\xB1\x0E\xAA\x87\x10\x23\x59\x5F\x20\x05\x19\x47\xED\x68\x8E\x92\x12\xCA\x5D\xFC\xD6\x2B\xB2\x92\x3C\x20\xCF\xE1\x5F\xAF\x20\xBE\xA0\x76\x7F\x76\xE5\xEC\x1A\x86\x61\x33\x3E\xE7\x7B\xB4\x3F\xA0\x0F\x8E\xA2\xB9\x6A\x6F\xB9\x87\x26\x6F\x41\x6C\x88\xA6\x50\xFD\x6A\x63\x0B\xF5\x93\x16\x1B\x19\x8F\xB2\xED\x9B\x9B\xC9\x90\xF5\x01\x0C\xDF\x19\x3D\x0F\x3E\x38\x23\xC9\x2F\x8F\x0C\xD1\x02\xFE\x1B\x55\xD6\x4E\xD0\x8D\x3C\xAF\x4F\xA4\xF3\xFE\xAF\x2A\xD3\x05\x9D\x79\x08\xA1\xCB\x57\x31\xB4\x9C\xC8\x90\xB2\x67\xF4\x18\x16\x93\x3A\xFC\x47\xD8\xD1\x78\x96\x31\x1F\xBA\x2B\x0C\x5F\x5D\x99\xAD\x63\x89\x5A\x24\x20\x76\xD8\xDF\xFD\xAB\x4E\xA6\x22\xAA\x9D\x5E\xE6\x27\x8A\x7D\x68\x29\xA3\xE7\x8A\xB8\xDA\x11\xBB\x17\x2D\x99\x9D\x13\x24\x46\xF7\xC5\xE2\xD8\x9F\x8E\x7F\xC7\x8F\x74\x6D\x5A\xB2\xE8\x72\xF5\xAC\xEE\x24\x10\xAD\x2F\x14\xDA\xFF\x2D\x9A\x46\x71\x47\xBE\x42\xDF\xBB\x01\xDB\xF4\x7F\xD3\x28\x8F\x31\x59\x5B\xD3\xC9\x02\xA6\xB4\x52\xCA\x6E\x97\xFB\x43\xC5\x08\x26\x6F\x8A\xF4\xBB\xFD\x9F\x28\xAA\x0D\xD5\x45\xF3\x13\x3A\x1D\xD8\xC0\x78\x8F\x41\x67\x3C\x1E\x94\x64\xAE\x7B\x0B\xC5\xE8\xD9\x01\x88\x39\x1A\x97\x86\x64\x41\xD5\x3B\x87\x0C\x6E\xFA\x0F\xC6\xBD\x48\x14\xBF\x39\x4D\xD4\x9E\x41\xB6\x8F\x96\x1D\x63\x96\x93\xD9\x95\x06\x78\x31\x68\x9E\x37\x06\x3B\x80\x89\x45\x61\x39\x23\xC7\x1B\x44\xA3\x15\xE5\x1C\xF8\x92\x30\xBB\x02\x03\x01\x00\x01\xA3\x81\xEF\x30\x81\xEC\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x65\xCD\xEB\xAB\x35\x1E\x00\x3E\x7E\xD5\x74\xC0\x1C\xB4\x73\x47\x0E\x1A\x64\x2F\x30\x12\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x08\x30\x06\x01\x01\xFF\x02\x01\x01\x30\x81\xA6\x06\x03\x55\x1D\x20\x04\x81\x9E\x30\x81\x9B\x30\x81\x98\x06\x04\x55\x1D\x20\x00\x30\x81\x8F\x30\x2F\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x01\x16\x23\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x66\x69\x72\x6D\x61\x70\x72\x6F\x66\x65\x73\x69\x6F\x6E\x61\x6C\x2E\x63\x6F\x6D\x2F\x63\x70\x73\x30\x5C\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x02\x30\x50\x1E\x4E\x00\x50\x00\x61\x00\x73\x00\x65\x00\x6F\x00\x20\x00\x64\x00\x65\x00\x20\x00\x6C\x00\x61\x00\x20\x00\x42\x00\x6F\x00\x6E\x00\x61\x00\x6E\x00\x6F\x00\x76\x00\x61\x00\x20\x00\x34\x00\x37\x00\x20\x00\x42\x00\x61\x00\x72\x00\x63\x00\x65\x00\x6C\x00\x6F\x00\x6E\x00\x61\x00\x20\x00\x30\x00\x38\x00\x30\x00\x31\x00\x37\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x74\x87\x28\x02\x2B\x77\x1F\x66\x89\x64\xED\x8F\x74\x2E\x46\x1C\xBB\xA8\xF8\xF8\x0B\x1D\x83\xB6\x3A\xA7\xE8\x45\x8A\x07\xB7\xE0\x3E\x20\xCB\xE1\x08\xDB\x13\x08\xF8\x28\xA1\x35\xB2\x80\xB3\x0B\x51\xC0\xD3\x56\x9A\x8D\x33\x45\x49\xAF\x49\xF0\xE0\x3D\x07\x7A\x45\x13\x5A\xFF\xC8\x97\xD8\xD3\x18\x2C\x7D\x96\xF8\xDD\xA2\x65\x43\x70\x93\x90\x15\xBA\x90\xDF\xE8\x19\xB0\xDB\x2C\x8A\x60\x0F\xB7\x6F\x94\x07\x1E\x1D\xA6\xC9\x85\xF6\xBD\x34\xF8\x40\x78\x62\x10\x70\x3A\xBE\x7D\x4B\x39\x81\xA9\x10\xD4\x96\x41\xBB\xF8\x5F\x1C\x0B\x1D\x08\xF2\xB1\xB0\x89\x7A\xF2\xF7\xA0\xE0\xC4\x8F\x8B\x78\xB5\x3B\x58\xA5\x23\x8E\x4F\x55\xFE\x36\x3B\xE0\x0C\xB7\xCA\x2A\x30\x41\x20\xB4\x80\xCD\xAE\xFC\x76\x66\x73\xA8\xAE\x6E\xE1\x7C\xDA\x03\xE8\x94\x20\xE6\x22\xA3\xD0\x1F\x90\x5D\x20\x53\x14\x26\x57\xDA\x54\x97\xDF\x16\x44\x10\x01\x1E\x88\x66\x8F\x72\x38\x93\xDD\x20\xB7\x34\xBE\xD7\xF1\xEE\x63\x8E\x47\x79\x28\x06\xFC\xF3\x59\x45\x25\x60\x22\x33\x1B\xA3\x5F\xA8\xBA\x2A\xDA\x1A\x3D\xCD\x40\xEA\x8C\xEE\x05\x15\x95\xD5\xA5\x2C\x20\x2F\xA7\x98\x28\xEE\x45\xFC\xF1\xB8\x88\x00\x2C\x8F\x42\xDA\x51\xD5\x9C\xE5\x13\x68\x71\x45\x43\x8B\x9E\x0B\x21\x3C\x4B\x5C\x05\xDC\x1A\x9F\x98\x8E\xDA\xBD\x22\x9E\x72\xCD\xAD\x0A\xCB\xCC\xA3\x67\x9B\x28\x74\xC4\x9B\xD7\x1A\x3C\x04\x58\xA6\x82\x9D\xAD\xC7\x7B\x6F\xFF\x80\x96\xE9\xF8\x8D\x6A\xBD\x18\x90\x1D\xFF\x49\x1A\x90\x52\x37\x93\x2F\x3C\x02\x5D\x82\x76\x0B\x51\xE7\x16\xC7\x57\xF8\x38\xF9\xA7\xCD\x9B\x22\x54\xEF\x63\xB0\x15\x6D\x53\x65\x03\x4A\x5E\x4A\xA0\xB2\xA7\x8E\x49\x00\x59\x38\xD5\xC7\xF4\x80\x64\xF5\x6E\x95\x50\xB8\x11\x7E\x15\x70\x38\x4A\xB0\x7F\xD0\xC4\x32\x70\xC0\x19\xFF\xC9\x38\x2D\x14\x2C\x66\xF4\x42\x44\xE6\x55\x76\x1B\x80\x15\x57\xFF\xC0\xA7\xA7\xAA\x39\xAA\xD8\xD3\x70\xD0\x2E\xBA\xEB\x94\x6A\xFA\x5F\x34\x86\xE7\x62\xB5\xFD\x8A\xF0\x30\x85\x94\xC9\xAF\x24\x02\x2F\x6F\xD6\xDD\x67\xFE\xE3\xB0\x55\x4F\x04\x98\x4F\xA4\x41\x56\xE2\x93\xD0\x6A\xE8\xD6\xF3\xFB\x65\xE0\xCE\x75\xC4\x31\x59\x0C\xEE\x82\xC8\x0C\x60\x33\x4A\x19\xBA\x84\x67\x27\x0F\xBC\x42\x5D\xBD\x24\x54\x0D\xEC\x1D\x70\x06\x5F\xA4\xBC\xFA\x20\x7C\x55"),
   // "TWCA Global Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\x41\x30\x82\x03\x29\xA0\x03\x02\x01\x02\x02\x02\x0C\xBE\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x54\x41\x49\x57\x41\x4E\x2D\x43\x41\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x13\x07\x52\x6F\x6F\x74\x20\x43\x41\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x54\x57\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x32\x30\x36\x32\x37\x30\x36\x32\x38\x33\x33\x5A\x17\x0D\x33\x30\x31\x32\x33\x31\x31\x35\x35\x39\x35\x39\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x54\x41\x49\x57\x41\x4E\x2D\x43\x41\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x13\x07\x52\x6F\x6F\x74\x20\x43\x41\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x54\x57\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB0\x05\xDB\xC8\xEB\x8C\xC4\x6E\x8A\x21\xEF\x8E\x4D\x9C\x71\x0A\x1F\x52\x70\xED\x6D\x82\x9C\x97\xC5\xD7\x4C\x4E\x45\x49\xCB\x40\x42\xB5\x12\x34\x6C\x19\xC2\x74\xA4\x31\x5F\x85\x02\x97\xEC\x43\x33\x0A\x53\xD2\x9C\x8C\x8E\xB7\xB8\x79\xDB\x2B\xD5\x6A\xF2\x8E\x66\xC4\xEE\x2B\x01\x07\x92\xD4\xB3\xD0\x02\xDF\x50\xF6\x55\xAF\x66\x0E\xCB\xE0\x47\x60\x2F\x2B\x32\x39\x35\x52\x3A\x28\x83\xF8\x7B\x16\xC6\x18\xB8\x62\xD6\x47\x25\x91\xCE\xF0\x19\x12\x4D\xAD\x63\xF5\xD3\x3F\x75\x5F\x29\xF0\xA1\x30\x1C\x2A\xA0\x98\xA6\x15\xBD\xEE\xFD\x19\x36\xF0\xE2\x91\x43\x8F\xFA\xCA\xD6\x10\x27\x49\x4C\xEF\xDD\xC1\xF1\x85\x70\x9B\xCA\xEA\xA8\x5A\x43\xFC\x6D\x86\x6F\x73\xE9\x37\x45\xA9\xF0\x36\xC7\xCC\x88\x75\x1E\xBB\x6C\x06\xFF\x9B\x6B\x3E\x17\xEC\x61\xAA\x71\x7C\xC6\x1D\xA2\xF7\x49\xE9\x15\xB5\x3C\xD6\xA1\x61\xF5\x11\xF7\x05\x6F\x1D\xFD\x11\xBE\xD0\x30\x07\xC2\x29\xB0\x09\x4E\x26\xDC\xE3\xA2\xA8\x91\x6A\x1F\xC2\x91\x45\x88\x5C\xE5\x98\xB8\x71\xA5\x15\x19\xC9\x7C\x75\x11\xCC\x70\x74\x4F\x2D\x9B\x1D\x91\x44\xFD\x56\x28\xA0\xFE\xBB\x86\x6A\xC8\xFA\x5C\x0B\x58\xDC\xC6\x4B\x76\xC8\xAB\x22\xD9\x73\x0F\xA5\xF4\x5A\x02\x89\x3F\x4F\x9E\x22\x82\xEE\xA2\x74\x53\x2A\x3D\x53\x27\x69\x1D\x6C\x8E\x32\x2C\x64\x00\x26\x63\x61\x36\x4E\xA3\x46\xB7\x3F\x7D\xB3\x2D\xAC\x6D\x90\xA2\x95\xA2\xCE\xCF\xDA\x82\xE7\x07\x34\x19\x96\xE9\xB8\x21\xAA\x29\x7E\xA6\x38\xBE\x8E\x29\x4A\x21\x66\x79\x1F\xB3\xC3\xB5\x09\x67\xDE\xD6\xD4\x07\x46\xF3\x2A\xDA\xE6\x22\x37\x60\xCB\x81\xB6\x0F\xA0\x0F\xE9\xC8\x95\x7F\xBF\x55\x91\x05\x7A\xCF\x3D\x15\xC0\x6F\xDE\x09\x94\x01\x83\xD7\x34\x1B\xCC\x40\xA5\xF0\xB8\x9B\x67\xD5\x98\x91\x3B\xA7\x84\x78\x95\x26\xA4\x5A\x08\xF8\x2B\x74\xB4\x00\x04\x3C\xDF\xB8\x14\x8E\xE8\xDF\xA9\x8D\x6C\x67\x92\x33\x1D\xC0\xB7\xD2\xEC\x92\xC8\xBE\x09\xBF\x2C\x29\x05\x6F\x02\x6B\x9E\xEF\xBC\xBF\x2A\xBC\x5B\xC0\x50\x8F\x41\x70\x71\x87\xB2\x4D\xB7\x04\xA9\x84\xA3\x32\xAF\xAE\xEE\x6B\x17\x8B\xB2\xB1\xFE\x6C\xE1\x90\x8C\x88\xA8\x97\x48\xCE\xC8\x4D\xCB\xF3\x06\xCF\x5F\x6A\x0A\x42\xB1\x1E\x1E\x77\x2F\x8E\xA0\xE6\x92\x0E\x06\xFC\x05\x22\xD2\x26\xE1\x31\x51\x7D\x32\xDC\x0F\x02\x03\x01\x00\x01\xA3\x23\x30\x21\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x5F\x34\x81\x76\xEF\x96\x1D\xD5\xE5\xB5\xD9\x02\x63\x84\x16\xC1\xAE\xA0\x70\x51\xA7\xF7\x4C\x47\x35\xC8\x0B\xD7\x28\x3D\x89\x71\xD9\xAA\x33\x41\xEA\x14\x1B\x6C\x21\x00\xC0\x6C\x42\x19\x7E\x9F\x69\x5B\x20\x42\xDF\xA2\xD2\xDA\xC4\x7C\x97\x4B\x8D\xB0\xE8\xAC\xC8\xEE\xA5\x69\x04\x99\x0A\x92\xA6\xAB\x27\x2E\x1A\x4D\x81\xBF\x84\xD4\x70\x1E\xAD\x47\xFE\xFD\x4A\x9D\x33\xE0\xF2\xB9\xC4\x45\x08\x21\x0A\xDA\x69\x69\x73\x72\x0D\xBE\x34\xFE\x94\x8B\xAD\xC3\x1E\x35\xD7\xA2\x83\xEF\xE5\x38\xC7\xA5\x85\x1F\xAB\xCF\x34\xEC\x3F\x28\xFE\x0C\xF1\x57\x86\x4E\xC9\x55\xF7\x1C\xD4\xD8\xA5\x7D\x06\x7A\x6F\xD5\xDF\x10\xDF\x81\x4E\x21\x65\xB1\xB6\xE1\x17\x79\x95\x45\x06\xCE\x5F\xCC\xDC\x46\x89\x63\x68\x44\x8D\x93\xF4\x64\x70\xA0\x3D\x9D\x28\x05\xC3\x39\x70\xB8\x62\x7B\x20\xFD\xE4\xDB\xE9\x08\xA1\xB8\x9E\x3D\x09\xC7\x4F\xFB\x2C\xF8\x93\x76\x41\xDE\x52\xE0\xE1\x57\xD2\x9D\x03\xBC\x77\x9E\xFE\x9E\x29\x5E\xF7\xC1\x51\x60\x1F\xDE\xDA\x0B\xB2\x2D\x75\xB7\x43\x48\x93\xE7\xF6\x79\xC6\x84\x5D\x80\x59\x60\x94\xFC\x78\x98\x8F\x3C\x93\x51\xED\x40\x90\x07\xDF\x64\x63\x24\xCB\x4E\x71\x05\xA1\xD7\x94\x1A\x88\x32\xF1\x22\x74\x22\xAE\xA5\xA6\xD8\x12\x69\x4C\x60\xA3\x02\xEE\x2B\xEC\xD4\x63\x92\x0B\x5E\xBE\x2F\x76\x6B\xA3\xB6\x26\xBC\x8F\x03\xD8\x0A\xF2\x4C\x64\x46\xBD\x39\x62\xE5\x96\xEB\x34\x63\x11\x28\xCC\x95\xF1\xAD\xEF\xEF\xDC\x80\x58\x48\xE9\x4B\xB8\xEA\x65\xAC\xE9\xFC\x80\xB5\xB5\xC8\x45\xF9\xAC\xC1\x9F\xD9\xB9\xEA\x62\x88\x8E\xC4\xF1\x4B\x83\x12\xAD\xE6\x8B\x84\xD6\x9E\xC2\xEB\x83\x18\x9F\x6A\xBB\x1B\x24\x60\x33\x70\xCC\xEC\xF7\x32\xF3\x5C\xD9\x79\x7D\xEF\x9E\xA4\xFE\xC9\x23\xC3\x24\xEE\x15\x92\xB1\x3D\x91\x4F\x26\x86\xBD\x66\x73\x24\x13\xEA\xA4\xAE\x63\xC1\xAD\x7D\x84\x03\x3C\x10\x78\x86\x1B\x79\xE3\xC4\xF3\xF2\x04\x95\x20\xAE\x23\x82\xC4\xB3\x3A\x00\x62\xBF\xE6\x36\x24\xE1\x57\xBA\xC7\x1E\x90\x75\xD5\x5F\x3F\x95\x61\x2B\xC1\x3B\xCD\xE5\xB3\x68\x61\xD0\x46\x26\xA9\x21\x52\x69\x2D\xEB\x2E\xC7\xEB\x77\xCE\xA6\x3A\xB5\x03\x33\x4F\x76\xD1\xE7\x5C\x54\x01\x5D\xCB\x78\xF4\xC9\x0C\xBF\xCF\x12\x8E\x17\x2D\x23\x68\x94\xE7\xAB\xFE\xA9\xB2\x2B\x06\xD0\x04\xCD"),
   // "Hongkong Post Root CA 3"
   CertificateDer::from_slice(b"\x30\x82\x05\xCF\x30\x82\x03\xB7\xA0\x03\x02\x01\x02\x02\x14\x08\x16\x5F\x8A\x4C\xA5\xEC\x00\xC9\x93\x40\xDF\xC4\xC6\xAE\x23\xB8\x1C\x5A\xA4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x6F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x4B\x31\x12\x30\x10\x06\x03\x55\x04\x08\x13\x09\x48\x6F\x6E\x67\x20\x4B\x6F\x6E\x67\x31\x12\x30\x10\x06\x03\x55\x04\x07\x13\x09\x48\x6F\x6E\x67\x20\x4B\x6F\x6E\x67\x31\x16\x30\x14\x06\x03\x55\x04\x0A\x13\x0D\x48\x6F\x6E\x67\x6B\x6F\x6E\x67\x20\x50\x6F\x73\x74\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x48\x6F\x6E\x67\x6B\x6F\x6E\x67\x20\x50\x6F\x73\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x30\x1E\x17\x0D\x31\x37\x30\x36\x30\x33\x30\x32\x32\x39\x34\x36\x5A\x17\x0D\x34\x32\x30\x36\x30\x33\x30\x32\x32\x39\x34\x36\x5A\x30\x6F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x4B\x31\x12\x30\x10\x06\x03\x55\x04\x08\x13\x09\x48\x6F\x6E\x67\x20\x4B\x6F\x6E\x67\x31\x12\x30\x10\x06\x03\x55\x04\x07\x13\x09\x48\x6F\x6E\x67\x20\x4B\x6F\x6E\x67\x31\x16\x30\x14\x06\x03\x55\x04\x0A\x13\x0D\x48\x6F\x6E\x67\x6B\x6F\x6E\x67\x20\x50\x6F\x73\x74\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x48\x6F\x6E\x67\x6B\x6F\x6E\x67\x20\x50\x6F\x73\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB3\x88\xD7\xEA\xCE\x0F\x20\x4E\xBE\xE6\xD6\x03\x6D\xEE\x59\xFC\xC2\x57\xDF\x29\x68\xA1\x83\x0E\x3E\x68\xC7\x68\x58\x9C\x1C\x60\x4B\x89\x43\x0C\xB9\xD4\x15\xB2\xEE\xC1\x4E\x75\xE9\xB5\xA7\xEF\xE5\xE9\x35\x99\xE4\xCC\x1C\xE7\x4B\x5F\x8D\x33\x30\x20\x33\x53\xD9\xA6\xBB\xD5\x3E\x13\x8E\xE9\x1F\x87\x49\xAD\x50\x2D\x50\xCA\x18\xBE\x01\x58\xA2\x13\x70\x96\xBB\x89\x88\x56\x80\x5C\xF8\xBD\x2C\x3C\xE1\x4C\x57\x88\xBB\xD3\xB9\x95\xEF\xCB\xC7\xF6\xDA\x31\x74\x28\xA6\xE6\x54\x89\xF5\x41\x31\xCA\xE5\x26\x1A\xCD\x82\xE0\x70\xDA\x3B\x29\xBB\xD5\x03\xF5\x99\xBA\x55\xF5\x64\xD1\x60\x0E\xB3\x89\x49\xB8\x8A\x2F\x05\xD2\x84\x45\x28\x7C\x8F\x68\x50\x12\x78\xFC\x0B\xB5\x53\xCB\xC2\x98\x1C\x84\xA3\x9E\xB0\xBE\x23\xA4\xDA\xDC\xC8\x2B\x1E\xDA\x6E\x45\x1E\x89\x98\xDA\xF9\x00\x2E\x06\xE9\x0C\x3B\x70\xD5\x50\x25\x88\x99\xCB\xCD\x73\x60\xF7\xD5\xFF\x35\x67\xC5\xA1\xBC\x5E\xAB\xCD\x4A\xB8\x45\xEB\xC8\x68\x1E\x0D\x0D\x14\x46\x12\xE3\xD2\x64\x62\x8A\x42\x98\xBC\xB4\xC6\x08\x08\xF8\xFD\xA8\x4C\x64\x9C\x76\x01\xBD\x2F\xA9\x6C\x33\x0F\xD8\x3F\x28\xB8\x3C\x69\x01\x42\x86\x7E\x69\xC1\xC9\x06\xCA\xE5\x7A\x46\x65\xE9\xC2\xD6\x50\x41\x2E\x3F\xB7\xE4\xED\x6C\xD7\xBF\x26\x01\x11\xA2\x16\x29\x4A\x6B\x34\x06\x90\xEC\x13\xD2\xB6\xFB\x6A\x76\xD2\x3C\xED\xF0\xD6\x2D\xDD\xE1\x15\xEC\xA3\x9B\x2F\x2C\xC9\x3E\x2B\xE4\x69\x3B\xFF\x72\x25\xB1\x36\x86\x5B\xC7\x7F\x6B\x8B\x55\x1B\x4A\xC5\x20\x61\x3D\xAE\xCB\x50\xE1\x08\x3A\xBE\xB0\x8F\x63\x41\x53\x30\x08\x59\x3C\x98\x1D\x77\xBA\x63\x91\x7A\xCA\x10\x50\x60\xBF\xF0\xD7\xBC\x95\x87\x8F\x97\xC5\xFE\x97\x6A\x01\x94\xA3\x7C\x5B\x85\x1D\x2A\x39\x3A\xD0\x54\xA1\xD1\x39\x71\x9D\xFD\x21\xF9\xB5\x7B\xF0\xE2\xE0\x02\x8F\x6E\x96\x24\x25\x2C\xA0\x1E\x2C\xA8\xC4\x89\xA7\xEF\xED\x99\x06\x2F\xB6\x0A\x4C\x4F\xDB\xA2\xCC\x37\x1A\xAF\x47\x85\x2D\x8A\x5F\xC4\x34\x34\x4C\x00\xFD\x18\x93\x67\x13\xD1\x37\xE6\x48\xB4\x8B\x06\xC5\x57\x7B\x19\x86\x0A\x79\xCB\x00\xC9\x52\xAF\x42\xFF\x37\x8F\xE1\xA3\x1E\x7A\x3D\x50\xAB\x63\x06\xE7\x15\xB5\x3F\xB6\x45\x37\x94\x37\xB1\x7E\xF2\x48\xC3\x7F\xC5\x75\xFE\x97\x8D\x45\x8F\x1A\xA7\x1A\x72\x28\x1A\x40\x0F\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x17\x9D\xCD\x1E\x8B\xD6\x39\x2B\x70\xD3\x5C\xD4\xA0\xB8\x1F\xB0\x00\xFC\xC5\x61\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x17\x9D\xCD\x1E\x8B\xD6\x39\x2B\x70\xD3\x5C\xD4\xA0\xB8\x1F\xB0\x00\xFC\xC5\x61\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x56\xD5\x7B\x6E\xE6\x22\x01\xD2\x42\x9B\x18\xD5\x0E\xD7\x66\x23\x5C\xE3\xFE\xA0\xC7\x92\xD2\xE9\x94\xAD\x4B\xA2\xC6\xEC\x12\x7C\x74\xD5\x48\xD2\x59\x14\x99\xC0\xEB\xB9\xD1\xEB\xF4\x48\x30\x5B\xAD\xA7\x57\x73\x99\xA9\xD3\xE5\xB7\xD1\x2E\x59\x24\x58\xDC\x68\x2E\x2E\x62\xD8\x6A\xE4\x70\x0B\x2D\x20\x50\x20\xA4\x32\x95\xD1\x00\x98\xBB\xD3\xFD\xF7\x32\xF2\x49\xAE\xC6\x7A\xE0\x47\xBE\x6E\xCE\xCB\xA3\x72\x3A\x2D\x69\x5D\xCB\xC8\xE8\x45\x39\xD4\xFA\x42\xC1\x11\x4C\x77\x5D\x92\xFB\x6A\xFF\x58\x44\xE5\xEB\x81\x9E\xAF\xA0\x99\xAD\xBE\xA9\x01\x66\xCB\x38\x1D\x3C\xDF\x43\x1F\xF4\x4D\x6E\xB4\xBA\x17\x46\xFC\x7D\xFD\x87\x81\x79\x6A\x0D\x33\x0F\xFA\x2F\xF8\x14\xB9\x80\xB3\x5D\x4D\xAA\x97\xE1\xF9\xE4\x18\xC5\xF8\xD5\x38\x8C\x26\x3C\xFD\xF2\x28\xE2\xEE\x5A\x49\x88\x2C\xDF\x79\x3D\x8E\x9E\x90\x3C\xBD\x41\x4A\x3A\xDD\x5B\xF6\x9A\xB4\xCE\x3F\x25\x30\x7F\x32\x7D\xA2\x03\x94\xD0\xDC\x7A\xA1\x52\xDE\x6E\x93\x8D\x18\x26\xFD\x55\xAC\xBD\x8F\x9B\xD2\xCF\xAF\xE7\x86\x2C\xCB\x1F\x09\x6F\xA3\x6F\xA9\x84\xD4\x73\xBF\x4D\xA1\x74\x1B\x4E\x23\x60\xF2\xCC\x0E\xAA\x7F\xA4\x9C\x4C\x25\xA8\xB2\x66\x3B\x38\xFF\xD9\x94\x30\xF6\x72\x84\xBE\x68\x55\x10\x0F\xC6\x73\x2C\x16\x69\x93\x07\xFE\xB1\x45\xED\xBB\xA2\x55\x6A\xB0\xDA\xB5\x4A\x02\x25\x27\x85\xD7\xB7\xB7\x86\x44\x16\x89\x6C\x80\x2B\x3E\x97\xA9\x9C\xD5\x7E\x55\x4C\xC6\xDE\x45\x10\x1C\xEA\xE9\x3B\x9F\x03\x53\xEE\xEE\x7A\x01\x02\x16\x78\xD4\xE8\xC2\xBE\x46\x76\x88\x13\x3F\x22\xBB\x48\x12\x1D\x52\x00\xB4\x02\x7E\x21\x1A\x1E\x9C\x25\xF4\xF3\x3D\x5E\x1E\xD2\x1C\xF9\xB3\x2D\xB6\xF7\x37\x5C\xC6\xCB\x21\x4E\xB0\xF7\x99\x47\x18\x85\xC1\x2B\xBA\x55\xAE\x06\xEA\xD0\x07\xB2\xDC\xAB\xD0\x82\x96\x75\xCE\xD2\x50\xFE\x99\xE7\xCF\x2F\x9F\xE7\x76\xD1\x61\x2A\xFB\x21\xBB\x31\xD0\xAA\x9F\x47\xA4\xB2\x22\xCA\x16\x3A\x50\x57\xC4\x5B\x43\x67\xC5\x65\x62\x03\x49\x01\xEB\x43\xD9\xD8\xF8\x9E\xAD\xCF\xB1\x63\x0E\x45\xF4\xA0\x5A\x2C\x9B\x2D\xC5\xA6\xC0\xAD\xA8\x47\xF4\x27\x4C\x38\x0D\x2E\x1B\x49\x3B\x52\xF4\xE8\x88\x83\x2B\x54\x28\xD4\xF2\x35\x52\xB4\x32\x83\x62\x69\x64\x0C\x91\x9C\x9F\x97\xEA\x74\x16\xFD\x1F\x11\x06\x9A\x9B\xF4"),
   // "Certum Trusted Network CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xBB\x30\x82\x02\xA3\xA0\x03\x02\x01\x02\x02\x03\x04\x44\xC0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x7E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x55\x6E\x69\x7A\x65\x74\x6F\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x43\x65\x72\x74\x75\x6D\x20\x54\x72\x75\x73\x74\x65\x64\x20\x4E\x65\x74\x77\x6F\x72\x6B\x20\x43\x41\x30\x1E\x17\x0D\x30\x38\x31\x30\x32\x32\x31\x32\x30\x37\x33\x37\x5A\x17\x0D\x32\x39\x31\x32\x33\x31\x31\x32\x30\x37\x33\x37\x5A\x30\x7E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x55\x6E\x69\x7A\x65\x74\x6F\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x43\x65\x72\x74\x75\x6D\x20\x54\x72\x75\x73\x74\x65\x64\x20\x4E\x65\x74\x77\x6F\x72\x6B\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xE3\xFB\x7D\xA3\x72\xBA\xC2\xF0\xC9\x14\x87\xF5\x6B\x01\x4E\xE1\x6E\x40\x07\xBA\x6D\x27\x5D\x7F\xF7\x5B\x2D\xB3\x5A\xC7\x51\x5F\xAB\xA4\x32\xA6\x61\x87\xB6\x6E\x0F\x86\xD2\x30\x02\x97\xF8\xD7\x69\x57\xA1\x18\x39\x5D\x6A\x64\x79\xC6\x01\x59\xAC\x3C\x31\x4A\x38\x7C\xD2\x04\xD2\x4B\x28\xE8\x20\x5F\x3B\x07\xA2\xCC\x4D\x73\xDB\xF3\xAE\x4F\xC7\x56\xD5\x5A\xA7\x96\x89\xFA\xF3\xAB\x68\xD4\x23\x86\x59\x27\xCF\x09\x27\xBC\xAC\x6E\x72\x83\x1C\x30\x72\xDF\xE0\xA2\xE9\xD2\xE1\x74\x75\x19\xBD\x2A\x9E\x7B\x15\x54\x04\x1B\xD7\x43\x39\xAD\x55\x28\xC5\xE2\x1A\xBB\xF4\xC0\xE4\xAE\x38\x49\x33\xCC\x76\x85\x9F\x39\x45\xD2\xA4\x9E\xF2\x12\x8C\x51\xF8\x7C\xE4\x2D\x7F\xF5\xAC\x5F\xEB\x16\x9F\xB1\x2D\xD1\xBA\xCC\x91\x42\x77\x4C\x25\xC9\x90\x38\x6F\xDB\xF0\xCC\xFB\x8E\x1E\x97\x59\x3E\xD5\x60\x4E\xE6\x05\x28\xED\x49\x79\x13\x4B\xBA\x48\xDB\x2F\xF9\x72\xD3\x39\xCA\xFE\x1F\xD8\x34\x72\xF5\xB4\x40\xCF\x31\x01\xC3\xEC\xDE\x11\x2D\x17\x5D\x1F\xB8\x50\xD1\x5E\x19\xA7\x69\xDE\x07\x33\x28\xCA\x50\x95\xF9\xA7\x54\xCB\x54\x86\x50\x45\xA9\xF9\x49\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x08\x76\xCD\xCB\x07\xFF\x24\xF6\xC5\xCD\xED\xBB\x90\xBC\xE2\x84\x37\x46\x75\xF7\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\xA6\xA8\xAD\x22\xCE\x01\x3D\xA6\xA3\xFF\x62\xD0\x48\x9D\x8B\x5E\x72\xB0\x78\x44\xE3\xDC\x1C\xAF\x09\xFD\x23\x48\xFA\xBD\x2A\xC4\xB9\x55\x04\xB5\x10\xA3\x8D\x27\xDE\x0B\x82\x63\xD0\xEE\xDE\x0C\x37\x79\x41\x5B\x22\xB2\xB0\x9A\x41\x5C\xA6\x70\xE0\xD4\xD0\x77\xCB\x23\xD3\x00\xE0\x6C\x56\x2F\xE1\x69\x0D\x0D\xD9\xAA\xBF\x21\x81\x50\xD9\x06\xA5\xA8\xFF\x95\x37\xD0\xAA\xFE\xE2\xB3\xF5\x99\x2D\x45\x84\x8A\xE5\x42\x09\xD7\x74\x02\x2F\xF7\x89\xD8\x99\xE9\xBC\x27\xD4\x47\x8D\xBA\x0D\x46\x1C\x77\xCF\x14\xA4\x1C\xB9\xA4\x31\xC4\x9C\x28\x74\x03\x34\xFF\x33\x19\x26\xA5\xE9\x0D\x74\xB7\x3E\x97\xC6\x76\xE8\x27\x96\xA3\x66\xDD\xE1\xAE\xF2\x41\x5B\xCA\x98\x56\x83\x73\x70\xE4\x86\x1A\xD2\x31\x41\xBA\x2F\xBE\x2D\x13\x5A\x76\x6F\x4E\xE8\x4E\x81\x0E\x3F\x5B\x03\x22\xA0\x12\xBE\x66\x58\x11\x4A\xCB\x03\xC4\xB4\x2A\x2A\x2D\x96\x17\xE0\x39\x54\xBC\x48\xD3\x76\x27\x9D\x9A\x2D\x06\xA6\xC9\xEC\x39\xD2\xAB\xDB\x9F\x9A\x0B\x27\x02\x35\x29\xB1\x40\x95\xE7\xF9\xE8\x9C\x55\x88\x19\x46\xD6\xB7\x34\xF5\x7E\xCE\x39\x9A\xD9\x38\xF1\x51\xF7\x4F\x2C"),
   // "CFCA EV ROOT"
   CertificateDer::from_slice(b"\x30\x82\x05\x8D\x30\x82\x03\x75\xA0\x03\x02\x01\x02\x02\x04\x18\x4A\xCC\xD6\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x56\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x30\x30\x2E\x06\x03\x55\x04\x0A\x0C\x27\x43\x68\x69\x6E\x61\x20\x46\x69\x6E\x61\x6E\x63\x69\x61\x6C\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x15\x30\x13\x06\x03\x55\x04\x03\x0C\x0C\x43\x46\x43\x41\x20\x45\x56\x20\x52\x4F\x4F\x54\x30\x1E\x17\x0D\x31\x32\x30\x38\x30\x38\x30\x33\x30\x37\x30\x31\x5A\x17\x0D\x32\x39\x31\x32\x33\x31\x30\x33\x30\x37\x30\x31\x5A\x30\x56\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x30\x30\x2E\x06\x03\x55\x04\x0A\x0C\x27\x43\x68\x69\x6E\x61\x20\x46\x69\x6E\x61\x6E\x63\x69\x61\x6C\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x15\x30\x13\x06\x03\x55\x04\x03\x0C\x0C\x43\x46\x43\x41\x20\x45\x56\x20\x52\x4F\x4F\x54\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD7\x5D\x6B\xCD\x10\x3F\x1F\x05\x59\xD5\x05\x4D\x37\xB1\x0E\xEC\x98\x2B\x8E\x15\x1D\xFA\x93\x4B\x17\x82\x21\x71\x10\x52\xD7\x51\x64\x70\x16\xC2\x55\x69\x4D\x8E\x15\x6D\x9F\xBF\x0C\x1B\xC2\xE0\xA3\x67\xD6\x0C\xAC\xCF\x22\xAE\xAF\x77\x54\x2A\x4B\x4C\x8A\x53\x52\x7A\xC3\xEE\x2E\xDE\xB3\x71\x25\xC1\xE9\x5D\x3D\xEE\xA1\x2F\xA3\xF7\x2A\x3C\xC9\x23\x1D\x6A\xAB\x1D\xA1\xA7\xF1\xF3\xEC\xA0\xD5\x44\xCF\x15\xCF\x72\x2F\x1D\x63\x97\xE8\x99\xF9\xFD\x93\xA4\x54\x80\x4C\x52\xD4\x52\xAB\x2E\x49\xDF\x90\xCD\xB8\x5F\xBE\x3F\xDE\xA1\xCA\x4D\x20\xD4\x25\xE8\x84\x29\x53\xB7\xB1\x88\x1F\xFF\xFA\xDA\x90\x9F\x0A\xA9\x2D\x41\x3F\xB1\xF1\x18\x29\xEE\x16\x59\x2C\x34\x49\x1A\xA8\x06\xD7\xA8\x88\xD2\x03\x72\x7A\x32\xE2\xEA\x68\x4D\x6E\x2C\x96\x65\x7B\xCA\x59\xFA\xF2\xE2\xDD\xEE\x30\x2C\xFB\xCC\x46\xAC\xC4\x63\xEB\x6F\x7F\x36\x2B\x34\x73\x12\x94\x7F\xDF\xCC\x26\x9E\xF1\x72\x5D\x50\x65\x59\x8F\x69\xB3\x87\x5E\x32\x6F\xC3\x18\x8A\xB5\x95\x8F\xB0\x7A\x37\xDE\x5A\x45\x3B\xC7\x36\xE1\xEF\x67\xD1\x39\xD3\x97\x5B\x73\x62\x19\x48\x2D\x87\x1C\x06\xFB\x74\x98\x20\x49\x73\xF0\x05\xD2\x1B\xB1\xA0\xA3\xB7\x1B\x70\xD3\x88\x69\xB9\x5A\xD6\x38\xF4\x62\xDC\x25\x8B\x78\xBF\xF8\xE8\x7E\xB8\x5C\xC9\x95\x4F\x5F\xA7\x2D\xB9\x20\x6B\xCF\x6B\xDD\xF5\x0D\xF4\x82\xB7\xF4\xB2\x66\x2E\x10\x28\xF6\x97\x5A\x7B\x96\x16\x8F\x01\x19\x2D\x6C\x6E\x7F\x39\x58\x06\x64\x83\x01\x83\x83\xC3\x4D\x92\xDD\x32\xC6\x87\xA4\x37\xE9\x16\xCE\xAA\x2D\x68\xAF\x0A\x81\x65\x3A\x70\xC1\x9B\xAD\x4D\x6D\x54\xCA\x2A\x2D\x4B\x85\x1B\xB3\x80\xE6\x70\x45\x0D\x6B\x5E\x35\xF0\x7F\x3B\xB8\x9C\xE4\x04\x70\x89\x12\x25\x93\xDA\x0A\x99\x22\x60\x6A\x63\x60\x4E\x76\x06\x98\x4E\xBD\x83\xAD\x1D\x58\x8A\x25\x85\xD2\xC7\x65\x1E\x2D\x8E\xC6\xDF\xB6\xC6\xE1\x7F\x8A\x04\x21\x15\x29\x74\xF0\x3E\x9C\x90\x9D\x0C\x2E\xF1\x8A\x3E\x5A\xAA\x0C\x09\x1E\xC7\xD5\x3C\xA3\xED\x97\xC3\x1E\x34\xFA\x38\xF9\x08\x0E\xE3\xC0\x5D\x2B\x83\xD1\x56\x6A\xC9\xB6\xA8\x54\x53\x2E\x78\x32\x67\x3D\x82\x7F\x74\xD0\xFB\xE1\xB6\x05\x60\xB9\x70\xDB\x8E\x0B\xF9\x13\x58\x6F\x71\x60\x10\x52\x10\xB9\xC1\x41\x09\xEF\x72\x1F\x67\x31\x78\xFF\x96\x05\x8D\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xE3\xFE\x2D\xFD\x28\xD0\x0B\xB5\xBA\xB6\xA2\xC4\xBF\x06\xAA\x05\x8C\x93\xFB\x2F\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE3\xFE\x2D\xFD\x28\xD0\x0B\xB5\xBA\xB6\xA2\xC4\xBF\x06\xAA\x05\x8C\x93\xFB\x2F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x25\xC6\xBA\x6B\xEB\x87\xCB\xDE\x82\x39\x96\x3D\xF0\x44\xA7\x6B\x84\x73\x03\xDE\x9D\x2B\x4F\xBA\x20\x7F\xBC\x78\xB2\xCF\x97\xB0\x1B\x9C\xF3\xD7\x79\x2E\xF5\x48\xB6\xD2\xFB\x17\x88\xE6\xD3\x7A\x3F\xED\x53\x13\xD0\xE2\x2F\x6A\x79\xCB\x00\x23\x28\xE6\x1E\x37\x57\x35\x89\x84\xC2\x76\x4F\x34\x36\xAD\x67\xC3\xCE\x41\x06\x88\xC5\xF7\xEE\xD8\x1A\xB8\xD6\x0B\x7F\x50\xFF\x93\xAA\x17\x4B\x8C\xEC\xED\x52\x60\xB2\xA4\x06\xEA\x4E\xEB\xF4\x6B\x19\xFD\xEB\xF5\x1A\xE0\x25\x2A\x9A\xDC\xC7\x41\x36\xF7\xC8\x74\x05\x84\x39\x95\x39\xD6\x0B\x3B\xA4\x27\xFA\x08\xD8\x5C\x1E\xF8\x04\x60\x52\x11\x28\x28\x03\xFF\xEF\x53\x66\x00\xA5\x4A\x34\x16\x66\x7C\xFD\x09\xA4\xAE\x9E\x67\x1A\x6F\x41\x0B\x6B\x06\x13\x9B\x8F\x86\x71\x05\xB4\x2F\x8D\x89\x66\x33\x29\x76\x54\x9A\x11\xF8\x27\xFA\xB2\x3F\x91\xE0\xCE\x0D\x1B\xF3\x30\x1A\xAD\xBF\x22\x5D\x1B\xD3\xBF\x25\x05\x4D\xE1\x92\x1A\x7F\x99\x9F\x3C\x44\x93\xCA\xD4\x40\x49\x6C\x80\x87\xD7\x04\x3A\xC3\x32\x52\x35\x0E\x56\xF8\xA5\xDD\x7D\xC4\x8B\x0D\x11\x1F\x53\xCB\x1E\xB2\x17\xB6\x68\x77\x5A\xE0\xD4\xCB\xC8\x07\xAE\xF5\x3A\x2E\x8E\x37\xB7\xD0\x01\x4B\x43\x29\x77\x8C\x39\x97\x8F\x82\x5A\xF8\x51\xE5\x89\xA0\x18\xE7\x68\x7F\x5D\x0A\x2E\xFB\xA3\x47\x0E\x3D\xA6\x23\x7A\xC6\x01\xC7\x8F\xC8\x5E\xBF\x6D\x80\x56\xBE\x8A\x24\xBA\x33\xEA\x9F\xE1\x32\x11\x9E\xF1\xD2\x4F\x80\xF6\x1B\x40\xAF\x38\x9E\x11\x50\x79\x73\x12\x12\xCD\xE6\x6C\x9D\x2C\x88\x72\x3C\x30\x81\x06\x91\x22\xEA\x59\xAD\xDA\x19\x2E\x22\xC2\x8D\xB9\x8C\x87\xE0\x66\xBC\x73\x23\x5F\x21\x64\x63\x80\x48\xF5\xA0\x3C\x18\x3D\x94\xC8\x48\x41\x1D\x40\xBA\x5E\xFE\xFE\x56\x39\xA1\xC8\xCF\x5E\x9E\x19\x64\x46\x10\xDA\x17\x91\xB7\x05\x80\xAC\x8B\x99\x92\x7D\xE7\xA2\xD8\x07\x0B\x36\x27\xE7\x48\x79\x60\x8A\xC3\xD7\x13\x5C\xF8\x72\x40\xDF\x4A\xCB\xCF\x99\x00\x0A\x00\x0B\x11\x95\xDA\x56\x45\x03\x88\x0A\x9F\x67\xD0\xD5\x79\xB1\xA8\x8D\x40\x6D\x0D\xC2\x7A\x40\xFA\xF3\x5F\x64\x47\x92\xCB\x53\xB9\xBB\x59\xCE\x4F\xFD\xD0\x15\x53\x01\xD8\xDF\xEB\xD9\xE6\x76\xEF\xD0\x23\xBB\x3B\xA9\x79\xB3\xD5\x02\x29\xCD\x89\xA3\x96\x0F\x4A\x35\xE7\x4E\x42\xC0\x75\xCD\x07\xCF\xE6\x2C\xEB\x7B\x2E"),
   // "IdenTrust Commercial Root CA 1"
   CertificateDer::from_slice(b"\x30\x82\x05\x60\x30\x82\x03\x48\xA0\x03\x02\x01\x02\x02\x10\x0A\x01\x42\x80\x00\x00\x01\x45\x23\xC8\x44\xB5\x00\x00\x00\x02\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x31\x27\x30\x25\x06\x03\x55\x04\x03\x13\x1E\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x20\x43\x6F\x6D\x6D\x65\x72\x63\x69\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x30\x1E\x17\x0D\x31\x34\x30\x31\x31\x36\x31\x38\x31\x32\x32\x33\x5A\x17\x0D\x33\x34\x30\x31\x31\x36\x31\x38\x31\x32\x32\x33\x5A\x30\x4A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x31\x27\x30\x25\x06\x03\x55\x04\x03\x13\x1E\x49\x64\x65\x6E\x54\x72\x75\x73\x74\x20\x43\x6F\x6D\x6D\x65\x72\x63\x69\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA7\x50\x19\xDE\x3F\x99\x3D\xD4\x33\x46\xF1\x6F\x51\x61\x82\xB2\xA9\x4F\x8F\x67\x89\x5D\x84\xD9\x53\xDD\x0C\x28\xD9\xD7\xF0\xFF\xAE\x95\x43\x72\x99\xF9\xB5\x5D\x7C\x8A\xC1\x42\xE1\x31\x50\x74\xD1\x81\x0D\x7C\xCD\x9B\x21\xAB\x43\xE2\xAC\xAD\x5E\x86\x6E\xF3\x09\x8A\x1F\x5A\x32\xBD\xA2\xEB\x94\xF9\xE8\x5C\x0A\xEC\xFF\x98\xD2\xAF\x71\xB3\xB4\x53\x9F\x4E\x87\xEF\x92\xBC\xBD\xEC\x4F\x32\x30\x88\x4B\x17\x5E\x57\xC4\x53\xC2\xF6\x02\x97\x8D\xD9\x62\x2B\xBF\x24\x1F\x62\x8D\xDF\xC3\xB8\x29\x4B\x49\x78\x3C\x93\x60\x88\x22\xFC\x99\xDA\x36\xC8\xC2\xA2\xD4\x2C\x54\x00\x67\x35\x6E\x73\xBF\x02\x58\xF0\xA4\xDD\xE5\xB0\xA2\x26\x7A\xCA\xE0\x36\xA5\x19\x16\xF5\xFD\xB7\xEF\xAE\x3F\x40\xF5\x6D\x5A\x04\xFD\xCE\x34\xCA\x24\xDC\x74\x23\x1B\x5D\x33\x13\x12\x5D\xC4\x01\x25\xF6\x30\xDD\x02\x5D\x9F\xE0\xD5\x47\xBD\xB4\xEB\x1B\xA1\xBB\x49\x49\xD8\x9F\x5B\x02\xF3\x8A\xE4\x24\x90\xE4\x62\x4F\x4F\xC1\xAF\x8B\x0E\x74\x17\xA8\xD1\x72\x88\x6A\x7A\x01\x49\xCC\xB4\x46\x79\xC6\x17\xB1\xDA\x98\x1E\x07\x59\xFA\x75\x21\x85\x65\xDD\x90\x56\xCE\xFB\xAB\xA5\x60\x9D\xC4\x9D\xF9\x52\xB0\x8B\xBD\x87\xF9\x8F\x2B\x23\x0A\x23\x76\x3B\xF7\x33\xE1\xC9\x00\xF3\x69\xF9\x4B\xA2\xE0\x4E\xBC\x7E\x93\x39\x84\x07\xF7\x44\x70\x7E\xFE\x07\x5A\xE5\xB1\xAC\xD1\x18\xCC\xF2\x35\xE5\x49\x49\x08\xCA\x56\xC9\x3D\xFB\x0F\x18\x7D\x8B\x3B\xC1\x13\xC2\x4D\x8F\xC9\x4F\x0E\x37\xE9\x1F\xA1\x0E\x6A\xDF\x62\x2E\xCB\x35\x06\x51\x79\x2C\xC8\x25\x38\xF4\xFA\x4B\xA7\x89\x5C\x9C\xD2\xE3\x0D\x39\x86\x4A\x74\x7C\xD5\x59\x87\xC2\x3F\x4E\x0C\x5C\x52\xF4\x3D\xF7\x52\x82\xF1\xEA\xA3\xAC\xFD\x49\x34\x1A\x28\xF3\x41\x88\x3A\x13\xEE\xE8\xDE\xFF\x99\x1D\x5F\xBA\xCB\xE8\x1E\xF2\xB9\x50\x60\xC0\x31\xD3\x73\xE5\xEF\xBE\xA0\xED\x33\x0B\x74\xBE\x20\x20\xC4\x67\x6C\xF0\x08\x03\x7A\x55\x80\x7F\x46\x4E\x96\xA7\xF4\x1E\x3E\xE1\xF6\xD8\x09\xE1\x33\x64\x2B\x63\xD7\x32\x5E\x9F\xF9\xC0\x7B\x0F\x78\x6F\x97\xBC\x93\x9A\xF9\x9C\x12\x90\x78\x7A\x80\x87\x15\xD7\x72\x74\x9C\x55\x74\x78\xB1\xBA\xE1\x6E\x70\x04\xBA\x4F\xA0\xBA\x68\xC3\x7B\xFF\x31\xF0\x73\x3D\x3D\x94\x2A\xB1\x0B\x41\x0E\xA0\xFE\x4D\x88\x65\x6B\x79\x33\xB4\xD7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xED\x44\x19\xC0\xD3\xF0\x06\x8B\xEE\xA4\x7B\xBE\x42\xE7\x26\x54\xC8\x8E\x36\x76\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x0D\xAE\x90\x32\xF6\xA6\x4B\x7C\x44\x76\x19\x61\x1E\x27\x28\xCD\x5E\x54\xEF\x25\xBC\xE3\x08\x90\xF9\x29\xD7\xAE\x68\x08\xE1\x94\x00\x58\xEF\x2E\x2E\x7E\x53\x52\x8C\xB6\x5C\x07\xEA\x88\xBA\x99\x8B\x50\x94\xD7\x82\x80\xDF\x61\x09\x00\x93\xAD\x0D\x14\xE6\xCE\xC1\xF2\x37\x94\x78\xB0\x5F\x9C\xB3\xA2\x73\xB8\x8F\x05\x93\x38\xCD\x8D\x3E\xB0\xB8\xFB\xC0\xCF\xB1\xF2\xEC\x2D\x2D\x1B\xCC\xEC\xAA\x9A\xB3\xAA\x60\x82\x1B\x2D\x3B\xC3\x84\x3D\x57\x8A\x96\x1E\x9C\x75\xB8\xD3\x30\xCD\x60\x08\x83\x90\xD3\x8E\x54\xF1\x4D\x66\xC0\x5D\x74\x03\x40\xA3\xEE\x85\x7E\xC2\x1F\x77\x9C\x06\xE8\xC1\xA7\x18\x5D\x52\x95\xED\xC9\xDD\x25\x9E\x6D\xFA\xA9\xED\xA3\x3A\x34\xD0\x59\x7B\xDA\xED\x50\xF3\x35\xBF\xED\xEB\x14\x4D\x31\xC7\x60\xF4\xDA\xF1\x87\x9C\xE2\x48\xE2\xC6\xC5\x37\xFB\x06\x10\xFA\x75\x59\x66\x31\x47\x29\xDA\x76\x9A\x1C\xE9\x82\xAE\xEF\x9A\xB9\x51\xF7\x88\x23\x9A\x69\x95\x62\x3C\xE5\x55\x80\x36\xD7\x54\x02\xFF\xF1\xB9\x5D\xCE\xD4\x23\x6F\xD8\x45\x84\x4A\x5B\x65\xEF\x89\x0C\xDD\x14\xA7\x20\xCB\x18\xA5\x25\xB4\x0D\xF9\x01\xF0\xA2\xD2\xF4\x00\xC8\x74\x8E\xA1\x2A\x48\x8E\x65\xDB\x13\xC4\xE2\x25\x17\x7D\xEB\xBE\x87\x5B\x17\x20\x54\x51\x93\x4A\x53\x03\x0B\xEC\x5D\xCA\x33\xED\x62\xFD\x45\xC7\x2F\x5B\xDC\x58\xA0\x80\x39\xE6\xFA\xD7\xFE\x13\x14\xA6\xED\x3D\x94\x4A\x42\x74\xD4\xC3\x77\x59\x73\xCD\x8F\x46\xBE\x55\x38\xEF\xFA\xE8\x91\x32\xEA\x97\x58\x04\x22\xDE\x38\xC3\xCC\xBC\x6D\xC9\x33\x3A\x6A\x0A\x69\x3F\xA0\xC8\xEA\x72\x8F\x8C\x63\x86\x23\xBD\x6D\x3C\x96\x9E\x95\xE0\x49\x4C\xAA\xA2\xB9\x2A\x1B\x9C\x36\x81\x78\xED\xC3\xE8\x46\xE2\x26\x59\x44\x75\x1E\xD9\x75\x89\x51\xCD\x10\x84\x9D\x61\x60\xCB\x5D\xF9\x97\x22\x4D\x8E\x98\xE6\xE3\x7F\xF6\x5B\xBB\xAE\xCD\xCA\x4A\x81\x6B\x5E\x0B\xF3\x51\xE1\x74\x2B\xE9\x7E\x27\xA7\xD9\x99\x49\x4E\xF8\xA5\x80\xDB\x25\x0F\x1C\x63\x62\x8A\xC9\x33\x67\x6B\x3C\x10\x83\xC6\xAD\xDE\xA8\xCD\x16\x8E\x8D\xF0\x07\x37\x71\x9F\xF2\xAB\xFC\x41\xF5\xC1\x8B\xEC\x00\x37\x5D\x09\xE5\x4E\x80\xEF\xFA\xB1\x5C\x38\x06\xA5\x1B\x4A\xE1\xDC\x38\x2D\x3C\xDC\xAB\x1F\x90\x1A\xD5\x4A\x9C\xEE\xD1\x70\x6C\xCC\xEE\xF4\x57\xF8\x18\xBA\x84\x6E\x87"),
   // "SwissSign Gold CA - G2"
   CertificateDer::from_slice(b"\x30\x82\x05\xBA\x30\x82\x03\xA2\xA0\x03\x02\x01\x02\x02\x09\x00\xBB\x40\x1C\x43\xF5\x5E\x4F\xB0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x41\x47\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x13\x16\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x47\x6F\x6C\x64\x20\x43\x41\x20\x2D\x20\x47\x32\x30\x1E\x17\x0D\x30\x36\x31\x30\x32\x35\x30\x38\x33\x30\x33\x35\x5A\x17\x0D\x33\x36\x31\x30\x32\x35\x30\x38\x33\x30\x33\x35\x5A\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x41\x47\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x13\x16\x53\x77\x69\x73\x73\x53\x69\x67\x6E\x20\x47\x6F\x6C\x64\x20\x43\x41\x20\x2D\x20\x47\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xAF\xE4\xEE\x7E\x8B\x24\x0E\x12\x6E\xA9\x50\x2D\x16\x44\x3B\x92\x92\x5C\xCA\xB8\x5D\x84\x92\x42\x13\x2A\xBC\x65\x57\x82\x40\x3E\x57\x24\xCD\x50\x8B\x25\x2A\xB7\x6F\xFC\xEF\xA2\xD0\xC0\x1F\x02\x24\x4A\x13\x96\x8F\x23\x13\xE6\x28\x58\x00\xA3\x47\xC7\x06\xA7\x84\x23\x2B\xBB\xBD\x96\x2B\x7F\x55\xCC\x8B\xC1\x57\x1F\x0E\x62\x65\x0F\xDD\x3D\x56\x8A\x73\xDA\xAE\x7E\x6D\xBA\x81\x1C\x7E\x42\x8C\x20\x35\xD9\x43\x4D\x84\xFA\x84\xDB\x52\x2C\xF3\x0E\x27\x77\x0B\x6B\xBF\x11\x2F\x72\x78\x9F\x2E\xD8\x3E\xE6\x18\x37\x5A\x2A\x72\xF9\xDA\x62\x90\x92\x95\xCA\x1F\x9C\xE9\xB3\x3C\x2B\xCB\xF3\x01\x13\xBF\x5A\xCF\xC1\xB5\x0A\x60\xBD\xDD\xB5\x99\x64\x53\xB8\xA0\x96\xB3\x6F\xE2\x26\x77\x91\x8C\xE0\x62\x10\x02\x9F\x34\x0F\xA4\xD5\x92\x33\x51\xDE\xBE\x8D\xBA\x84\x7A\x60\x3C\x6A\xDB\x9F\x2B\xEC\xDE\xDE\x01\x3F\x6E\x4D\xE5\x50\x86\xCB\xB4\xAF\xED\x44\x40\xC5\xCA\x5A\x8C\xDA\xD2\x2B\x7C\xA8\xEE\xBE\xA6\xE5\x0A\xAA\x0E\xA5\xDF\x05\x52\xB7\x55\xC7\x22\x5D\x32\x6A\x97\x97\x63\x13\xDB\xC9\xDB\x79\x36\x7B\x85\x3A\x4A\xC5\x52\x89\xF9\x24\xE7\x9D\x77\xA9\x82\xFF\x55\x1C\xA5\x71\x69\x2B\xD1\x02\x24\xF2\xB3\x26\xD4\x6B\xDA\x04\x55\xE5\xC1\x0A\xC7\x6D\x30\x37\x90\x2A\xE4\x9E\x14\x33\x5E\x16\x17\x55\xC5\x5B\xB5\xCB\x34\x89\x92\xF1\x9D\x26\x8F\xA1\x07\xD4\xC6\xB2\x78\x50\xDB\x0C\x0C\x0B\x7C\x0B\x8C\x41\xD7\xB9\xE9\xDD\x8C\x88\xF7\xA3\x4D\xB2\x32\xCC\xD8\x17\xDA\xCD\xB7\xCE\x66\x9D\xD4\xFD\x5E\xFF\xBD\x97\x3E\x29\x75\xE7\x7E\xA7\x62\x58\xAF\x25\x34\xA5\x41\xC7\x3D\xBC\x0D\x50\xCA\x03\x03\x0F\x08\x5A\x1F\x95\x73\x78\x62\xBF\xAF\x72\x14\x69\x0E\xA5\xE5\x03\x0E\x78\x8E\x26\x28\x42\xF0\x07\x0B\x62\x20\x10\x67\x39\x46\xFA\xA9\x03\xCC\x04\x38\x7A\x66\xEF\x20\x83\xB5\x8C\x4A\x56\x8E\x91\x00\xFC\x8E\x5C\x82\xDE\x88\xA0\xC3\xE2\x68\x6E\x7D\x8D\xEF\x3C\xDD\x65\xF4\x5D\xAC\x51\xEF\x24\x80\xAE\xAA\x56\x97\x6F\xF9\xAD\x7D\xDA\x61\x3F\x98\x77\x3C\xA5\x91\xB6\x1C\x8C\x26\xDA\x65\xA2\x09\x6D\xC1\xE2\x54\xE3\xB9\xCA\x4C\x4C\x80\x8F\x77\x7B\x60\x9A\x1E\xDF\xB6\xF2\x48\x1E\x0E\xBA\x4E\x54\x6D\x98\xE0\xE1\xA2\x1A\xA2\x77\x50\xCF\xC4\x63\x92\xEC\x47\x19\x9D\xEB\xE6\x6B\xCE\xC1\x02\x03\x01\x00\x01\xA3\x81\xAC\x30\x81\xA9\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x5B\x25\x7B\x96\xA4\x65\x51\x7E\xB8\x39\xF3\xC0\x78\x66\x5E\xE8\x3A\xE7\xF0\xEE\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x5B\x25\x7B\x96\xA4\x65\x51\x7E\xB8\x39\xF3\xC0\x78\x66\x5E\xE8\x3A\xE7\xF0\xEE\x30\x46\x06\x03\x55\x1D\x20\x04\x3F\x30\x3D\x30\x3B\x06\x09\x60\x85\x74\x01\x59\x01\x02\x01\x01\x30\x2E\x30\x2C\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x01\x16\x20\x68\x74\x74\x70\x3A\x2F\x2F\x72\x65\x70\x6F\x73\x69\x74\x6F\x72\x79\x2E\x73\x77\x69\x73\x73\x73\x69\x67\x6E\x2E\x63\x6F\x6D\x2F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00\x27\xBA\xE3\x94\x7C\xF1\xAE\xC0\xDE\x17\xE6\xE5\xD8\xD5\xF5\x54\xB0\x83\xF4\xBB\xCD\x5E\x05\x7B\x4F\x9F\x75\x66\xAF\x3C\xE8\x56\x7E\xFC\x72\x78\x38\x03\xD9\x2B\x62\x1B\x00\xB9\xF8\xE9\x60\xCD\xCC\xCE\x51\x8A\xC7\x50\x31\x6E\xE1\x4A\x7E\x18\x2F\x69\x59\xB6\x3D\x64\x81\x2B\xE3\x83\x84\xE6\x22\x87\x8E\x7D\xE0\xEE\x02\x99\x61\xB8\x1E\xF4\xB8\x2B\x88\x12\x16\x84\xC2\x31\x93\x38\x96\x31\xA6\xB9\x3B\x53\x3F\xC3\x24\x93\x56\x5B\x69\x92\xEC\xC5\xC1\xBB\x38\x00\xE3\xEC\x17\xA9\xB8\xDC\xC7\x7C\x01\x83\x9F\x32\x47\xBA\x52\x22\x34\x1D\x32\x7A\x09\x56\xA7\x7C\x25\x36\xA9\x3D\x4B\xDA\xC0\x82\x6F\x0A\xBB\x12\xC8\x87\x4B\x27\x11\xF9\x1E\x2D\xC7\x93\x3F\x9E\xDB\x5F\x26\x6B\x52\xD9\x2E\x8A\xF1\x14\xC6\x44\x8D\x15\xA9\xB7\xBF\xBD\xDE\xA6\x1A\xEE\xAE\x2D\xFB\x48\x77\x17\xFE\xBB\xEC\xAF\x18\xF5\x2A\x51\xF0\x39\x84\x97\x95\x6C\x6E\x1B\xC3\x2B\xC4\x74\x60\x79\x25\xB0\x0A\x27\xDF\xDF\x5E\xD2\x39\xCF\x45\x7D\x42\x4B\xDF\xB3\x2C\x1E\xC5\xC6\x5D\xCA\x55\x3A\xA0\x9C\x69\x9A\x8F\xDA\xEF\xB2\xB0\x3C\x9F\x87\x6C\x12\x2B\x65\x70\x15\x52\x31\x1A\x24\xCF\x6F\x31\x23\x50\x1F\x8C\x4F\x8F\x23\xC3\x74\x41\x63\x1C\x55\xA8\x14\xDD\x3E\xE0\x51\x50\xCF\xF1\x1B\x30\x56\x0E\x92\xB0\x82\x85\xD8\x83\xCB\x22\x64\xBC\x2D\xB8\x25\xD5\x54\xA2\xB8\x06\xEA\xAD\x92\xA4\x24\xA0\xC1\x86\xB5\x4A\x13\x6A\x47\xCF\x2E\x0B\x56\x95\x54\xCB\xCE\x9A\xDB\x6A\xB4\xA6\xB2\xDB\x41\x08\x86\x27\x77\xF7\x6A\xA0\x42\x6C\x0B\x38\xCE\xD7\x75\x50\x32\x92\xC2\xDF\x2B\x30\x22\x48\xD0\xD5\x41\x38\x25\x5D\xA4\xE9\x5D\x9F\xC6\x94\x75\xD0\x45\xFD\x30\x97\x43\x8F\x90\xAB\x0A\xC7\x86\x73\x60\x4A\x69\x2D\xDE\xA5\x78\xD7\x06\xDA\x6A\x9E\x4B\x3E\x77\x3A\x20\x13\x22\x01\xD0\xBF\x68\x9E\x63\x60\x6B\x35\x4D\x0B\x6D\xBA\xA1\x3D\xC0\x93\xE0\x7F\x23\xB3\x55\xAD\x72\x25\x4E\x46\xF9\xD2\x16\xEF\xB0\x64\xC1\x01\x9E\xE9\xCA\xA0\x6A\x98\x0E\xCF\xD8\x60\xF2\x2F\x49\xB8\xE4\x42\xE1\x38\x35\x16\xF4\xC8\x6E\x4F\xF7\x81\x56\xE8\xBA\xA3\xBE\x23\xAF\xAE\xFD\x6F\x03\xE0\x02\x3B\x30\x76\xFA\x1B\x6D\x41\xCF\x01\xB1\xE9\xB8\xC9\x66\xF4\xDB\x26\xF3\x3A\xA4\x74\xF2\x49\x24\x5B\xC9\xB0\xD0\x57\xC1\xFA\x3E\x7A\xE1\x97\xC9"),
   // "certSIGN ROOT CA G2"
   CertificateDer::from_slice(b"\x30\x82\x05\x47\x30\x82\x03\x2F\xA0\x03\x02\x01\x02\x02\x09\x11\x00\x34\xB6\x4E\xC6\x36\x2D\x36\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x41\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x52\x4F\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x13\x0B\x43\x45\x52\x54\x53\x49\x47\x4E\x20\x53\x41\x31\x1C\x30\x1A\x06\x03\x55\x04\x0B\x13\x13\x63\x65\x72\x74\x53\x49\x47\x4E\x20\x52\x4F\x4F\x54\x20\x43\x41\x20\x47\x32\x30\x1E\x17\x0D\x31\x37\x30\x32\x30\x36\x30\x39\x32\x37\x33\x35\x5A\x17\x0D\x34\x32\x30\x32\x30\x36\x30\x39\x32\x37\x33\x35\x5A\x30\x41\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x52\x4F\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x13\x0B\x43\x45\x52\x54\x53\x49\x47\x4E\x20\x53\x41\x31\x1C\x30\x1A\x06\x03\x55\x04\x0B\x13\x13\x63\x65\x72\x74\x53\x49\x47\x4E\x20\x52\x4F\x4F\x54\x20\x43\x41\x20\x47\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC0\xC5\x75\x19\x91\x7D\x44\x74\x74\x87\xFE\x0E\x3B\x96\xDC\xD8\x01\x16\xCC\xEE\x63\x91\xE7\x0B\x6F\xCE\x3B\x0A\x69\x1A\x7C\xC2\xE3\xAF\x82\x8E\x86\xD7\x5E\x8F\x57\xEB\xD3\x21\x59\xFD\x39\x37\x42\x30\xBE\x50\xEA\xB6\x0F\xA9\x88\xD8\x2E\x2D\x69\x21\xE7\xD1\x37\x18\x4E\x7D\x91\xD5\x16\x5F\x6B\x5B\x00\xC2\x39\x43\x0D\x36\x85\x52\xB9\x53\x65\x0F\x1D\x42\xE5\x8F\xCF\x05\xD3\xEE\xDC\x0C\x1A\xD9\xB8\x8B\x78\x22\x67\xE4\x69\xB0\x68\xC5\x3C\xE4\x6C\x5A\x46\xE7\xCD\xC7\xFA\xEF\xC4\xEC\x4B\xBD\x6A\xA4\xAC\xFD\xCC\x28\x51\xEF\x92\xB4\x29\xAB\xAB\x35\x9A\x4C\xE4\xC4\x08\xC6\x26\xCC\xF8\x69\x9F\xE4\x9C\xF0\x29\xD3\x5C\xF9\xC6\x16\x25\x9E\x23\xC3\x20\xC1\x3D\x0F\x3F\x38\x40\xB0\xFE\x82\x44\x38\xAA\x5A\x1A\x8A\x6B\x63\x58\x38\xB4\x15\xD3\xB6\x11\x69\x7B\x1E\x54\xEE\x8C\x1A\x22\xAC\x72\x97\x3F\x23\x59\x9B\xC9\x22\x84\xC1\x07\x4F\xCC\x7F\xE2\x57\xCA\x12\x70\xBB\xA6\x65\xF3\x69\x75\x63\xBD\x95\xFB\x1B\x97\xCD\xE4\xA8\xAF\xF6\xD1\x4E\xA8\xD9\x8A\x71\x24\xCD\x36\x3D\xBC\x96\xC4\xF1\x6C\xA9\xAE\xE5\xCF\x0D\x6E\x28\x0D\xB0\x0E\xB5\xCA\x51\x7B\x78\x14\xC3\x20\x2F\x7F\xFB\x14\x55\xE1\x11\x99\xFD\xD5\x0A\xA1\x9E\x02\xE3\x62\x5F\xEB\x35\x4B\x2C\xB8\x72\xE8\x3E\x3D\x4F\xAC\x2C\xBB\x2E\x86\xE2\xA3\x76\x8F\xE5\x93\x2A\xCF\xA5\xAB\xC8\x5C\x8D\x4B\x06\xFF\x12\x46\xAC\x78\xCB\x14\x07\x35\xE0\xA9\xDF\x8B\xE9\xAF\x15\x4F\x16\x89\x5B\xBD\xF6\x8D\xC6\x59\xAE\x88\x85\x0E\xC1\x89\xEB\x1F\x67\xC5\x45\x8E\xFF\x6D\x37\x36\x2B\x78\x66\x83\x91\x51\x2B\x3D\xFF\x51\x77\x76\x62\xA1\xEC\x67\x3E\x3E\x81\x83\xE0\x56\xA9\x50\x1F\x1F\x7A\x99\xAB\x63\xBF\x84\x17\x77\xF1\x0D\x3B\xDF\xF7\x9C\x61\xB3\x35\x98\x8A\x3A\xB2\xEC\x3C\x1A\x37\x3F\x7E\x8F\x92\xCF\xD9\x12\x14\x64\xDA\x10\x02\x15\x41\xFF\x4F\xC4\xEB\x1C\xA3\xC9\xFA\x99\xF7\x46\xE9\xE1\x18\xD9\xB1\xB8\x32\x2D\xCB\x14\x0C\x50\xD8\x83\x65\x83\xEE\xB9\x5C\xCF\xCB\x05\x5A\x4C\xFA\x19\x97\x6B\xD6\x5D\x13\xD3\xC2\x5C\x54\xBC\x32\x73\xA0\x78\xF5\xF1\x6D\x1E\xCB\x9F\xA5\xA6\x9F\x22\xDC\xD1\x51\x9E\x82\x79\x64\x60\x29\x13\x3E\xA3\xFD\x4F\x72\x6A\xAB\xE2\xD4\xE5\xB8\x24\x55\x2C\x44\x4B\x8A\x88\x44\x9C\xCA\x84\xD3\x2A\x3B\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x82\x21\x2D\x66\xC6\xD7\xA0\xE0\x15\xEB\xCE\x4C\x09\x77\xC4\x60\x9E\x54\x6E\x03\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x60\xDE\x1A\xB8\xE7\xF2\x60\x82\xD5\x03\x33\x81\xCB\x06\x8A\xF1\x22\x49\xE9\xE8\xEA\x91\x7F\xC6\x33\x5E\x68\x19\x03\x86\x3B\x43\x01\xCF\x07\x70\xE4\x08\x1E\x65\x85\x91\xE6\x11\x22\xB7\xF5\x02\x23\x8E\xAE\xB9\x1E\x7D\x1F\x7E\x6C\xE6\xBD\x25\xD5\x95\x1A\xF2\x05\xA6\xAF\x85\x02\x6F\xAE\xF8\xD6\x31\xFF\x25\xC9\x4A\xC8\xC7\x8A\xA9\xD9\x9F\x4B\x49\x9B\x11\x57\x99\x92\x43\x11\xDE\xB6\x33\xA4\xCC\xD7\x8D\x64\x7D\xD4\xCD\x3C\x28\x2C\xB4\x9A\x96\xEA\x4D\xF5\xC4\x44\xC4\x25\xAA\x20\x80\xD8\x29\x55\xF7\xE0\x41\xFC\x06\x26\xFF\xB9\x36\xF5\x43\x14\x03\x66\x78\xE1\x11\xB1\xDA\x20\x5F\x46\x00\x78\x00\x21\xA5\x1E\x00\x28\x61\x78\x6F\xA8\x01\x01\x8F\x9D\x34\x9A\xFF\xF4\x38\x90\xFB\xB8\xD1\xB3\x72\x06\xC9\x71\xE6\x81\xC5\x79\xED\x0B\xA6\x79\xF2\x13\x0B\x9C\xF7\x5D\x0E\x7B\x24\x93\xB4\x48\xDB\x86\x5F\xDE\x50\x86\x78\xE7\x40\xE6\x31\xA8\x90\x76\x70\x61\xAF\x9C\x37\x2C\x11\xB5\x82\xB7\xAA\xAE\x24\x34\x5B\x72\x0C\x69\x0D\xCD\x59\x9F\xF6\x71\xAF\x9C\x0B\xD1\x0A\x38\xF9\x06\x22\x83\x53\x25\x0C\xFC\x51\xC4\xE6\xBE\xE2\x39\x95\x0B\x24\xAD\xAF\xD1\x95\xE4\x96\xD7\x74\x64\x6B\x71\x4E\x02\x3C\xAA\x85\xF3\x20\xA3\x43\x39\x76\x5B\x6C\x50\xFE\x9A\x9C\x14\x1E\x65\x14\x8A\x15\xBD\xA3\x82\x45\x5A\x49\x56\x6A\xD2\x9C\xB1\x63\x32\xE5\x61\xE0\x53\x22\x0E\xA7\x0A\x49\xEA\xCB\x7E\x1F\xA8\xE2\x62\x80\xF6\x10\x45\x52\x98\x06\x18\xDE\xA5\xCD\x2F\x7F\xAA\xD4\xE9\x3E\x08\x72\xEC\x23\x03\x02\x3C\xA6\xAA\xD8\xBC\x67\x74\x3D\x14\x17\xFB\x54\x4B\x17\xE3\xD3\x79\x3D\x6D\x6B\x49\xC9\x28\x0E\x2E\x74\x50\xBF\x0C\xD9\x46\x3A\x10\x86\xC9\xA7\x3F\xE9\xA0\xEC\x7F\xEB\xA5\x77\x58\x69\x71\xE6\x83\x0A\x37\xF2\x86\x49\x6A\xBE\x79\x08\x90\xF6\x02\x16\x64\x3E\xE5\xDA\x4C\x7E\x0C\x34\xC9\xF9\x5F\xB6\xB3\x28\x51\xA7\xA7\x2B\xAA\x49\xFA\x8D\x65\x29\x4E\xE3\x6B\x13\xA7\x94\xA3\x2D\x51\x6D\x78\x0C\x44\xCB\xDF\xDE\x08\x6F\xCE\xA3\x64\xAB\xD3\x95\x84\xD4\xB9\x52\x54\x72\x7B\x96\x25\xCC\xBC\x69\xE3\x48\x6E\x0D\xD0\xC7\x9D\x27\x9A\xAA\xF8\x13\x92\xDD\x1E\xDF\x63\x9F\x35\xA9\x16\x36\xEC\x8C\xB8\x83\xF4\x3D\x89\x8F\xCD\xB4\x17\x5E\xD7\xB3\x17\x41\x10\x5D\x27\x73\x60\x85\x57\x49\x22\x07"),
   // "ISRG Root X2"
   CertificateDer::from_slice(b"\x30\x82\x02\x1B\x30\x82\x01\xA1\xA0\x03\x02\x01\x02\x02\x10\x41\xD2\x9D\xD1\x72\xEA\xEE\xA7\x80\xC1\x2C\x6C\xE9\x2F\x87\x52\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x4F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x29\x30\x27\x06\x03\x55\x04\x0A\x13\x20\x49\x6E\x74\x65\x72\x6E\x65\x74\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x47\x72\x6F\x75\x70\x31\x15\x30\x13\x06\x03\x55\x04\x03\x13\x0C\x49\x53\x52\x47\x20\x52\x6F\x6F\x74\x20\x58\x32\x30\x1E\x17\x0D\x32\x30\x30\x39\x30\x34\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x30\x30\x39\x31\x37\x31\x36\x30\x30\x30\x30\x5A\x30\x4F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x29\x30\x27\x06\x03\x55\x04\x0A\x13\x20\x49\x6E\x74\x65\x72\x6E\x65\x74\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x47\x72\x6F\x75\x70\x31\x15\x30\x13\x06\x03\x55\x04\x03\x13\x0C\x49\x53\x52\x47\x20\x52\x6F\x6F\x74\x20\x58\x32\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xCD\x9B\xD5\x9F\x80\x83\x0A\xEC\x09\x4A\xF3\x16\x4A\x3E\x5C\xCF\x77\xAC\xDE\x67\x05\x0D\x1D\x07\xB6\xDC\x16\xFB\x5A\x8B\x14\xDB\xE2\x71\x60\xC4\xBA\x45\x95\x11\x89\x8E\xEA\x06\xDF\xF7\x2A\x16\x1C\xA4\xB9\xC5\xC5\x32\xE0\x03\xE0\x1E\x82\x18\x38\x8B\xD7\x45\xD8\x0A\x6A\x6E\xE6\x00\x77\xFB\x02\x51\x7D\x22\xD8\x0A\x6E\x9A\x5B\x77\xDF\xF0\xFA\x41\xEC\x39\xDC\x75\xCA\x68\x07\x0C\x1F\xEA\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x7C\x42\x96\xAE\xDE\x4B\x48\x3B\xFA\x92\xF8\x9E\x8C\xCF\x6D\x8B\xA9\x72\x37\x95\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x7B\x79\x4E\x46\x50\x84\xC2\x44\x87\x46\x1B\x45\x70\xFF\x58\x99\xDE\xF4\xFD\xA4\xD2\x55\xA6\x20\x2D\x74\xD6\x34\xBC\x41\xA3\x50\x5F\x01\x27\x56\xB4\xBE\x27\x75\x06\xAF\x12\x2E\x75\x98\x8D\xFC\x02\x31\x00\x8B\xF5\x77\x6C\xD4\xC8\x65\xAA\xE0\x0B\x2C\xEE\x14\x9D\x27\x37\xA4\xF9\x53\xA5\x51\xE4\x29\x83\xD7\xF8\x90\x31\x5B\x42\x9F\x0A\xF5\xFE\xAE\x00\x68\xE7\x8C\x49\x0F\xB6\x6F\x5B\x5B\x15\xF2\xE7"),
   // "Certum EC-384 CA"
   CertificateDer::from_slice(b"\x30\x82\x02\x65\x30\x82\x01\xEB\xA0\x03\x02\x01\x02\x02\x10\x78\x8F\x27\x5C\x81\x12\x52\x20\xA5\x04\xD0\x2D\xDD\xBA\x73\xF4\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x74\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x41\x73\x73\x65\x63\x6F\x20\x44\x61\x74\x61\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x43\x65\x72\x74\x75\x6D\x20\x45\x43\x2D\x33\x38\x34\x20\x43\x41\x30\x1E\x17\x0D\x31\x38\x30\x33\x32\x36\x30\x37\x32\x34\x35\x34\x5A\x17\x0D\x34\x33\x30\x33\x32\x36\x30\x37\x32\x34\x35\x34\x5A\x30\x74\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x41\x73\x73\x65\x63\x6F\x20\x44\x61\x74\x61\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x43\x65\x72\x74\x75\x6D\x20\x45\x43\x2D\x33\x38\x34\x20\x43\x41\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xC4\x28\x8E\xAB\x18\x5B\x6A\xBE\x6E\x64\x37\x63\xE4\xCD\xEC\xAB\x3A\xF7\xCC\xA1\xB8\x0E\x82\x49\xD7\x86\x29\x9F\xA1\x94\xF2\xE3\x60\x78\x98\x81\x78\x06\x4D\xF2\xEC\x9A\x0E\x57\x60\x83\x9F\xB4\xE6\x17\x2F\x1A\xB3\x5D\x02\x5B\x89\x23\x3C\xC2\x11\x05\x2A\xA7\x88\x13\x18\xF3\x50\x84\xD7\xBD\x34\x2C\x27\x89\x55\xFF\xCE\x4C\xE7\xDF\xA6\x1F\x28\xC4\xF0\x54\xC3\xB9\x7C\xB7\x53\xAD\xEB\xC2\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x8D\x06\x66\x74\x24\x76\x3A\xF3\x89\xF7\xBC\xD6\xBD\x47\x7D\x2F\xBC\x10\x5F\x4B\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x03\x55\x2D\xA6\xE6\x18\xC4\x7C\xEF\xC9\x50\x6E\xC1\x27\x0F\x9C\x87\xAF\x6E\xD5\x1B\x08\x18\xBD\x92\x29\xC1\xEF\x94\x91\x78\xD2\x3A\x1C\x55\x89\x62\xE5\x1B\x09\x1E\xBA\x64\x6B\xF1\x76\xB4\xD4\x02\x31\x00\xB4\x42\x84\x99\xFF\xAB\xE7\x9E\xFB\x91\x97\x27\x5D\xDC\xB0\x5B\x30\x71\xCE\x5E\x38\x1A\x6A\xD9\x25\xE7\xEA\xF7\x61\x92\x56\xF8\xEA\xDA\x36\xC2\x87\x65\x96\x2E\x72\x25\x2F\x7F\xDF\xC3\x13\xC9"),
   // "OISTE WISeKey Global Root GB CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xB5\x30\x82\x02\x9D\xA0\x03\x02\x01\x02\x02\x10\x76\xB1\x20\x52\x74\xF0\x85\x87\x46\xB3\xF8\x23\x1A\xF6\xC2\xC0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x6D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x10\x30\x0E\x06\x03\x55\x04\x0A\x13\x07\x57\x49\x53\x65\x4B\x65\x79\x31\x22\x30\x20\x06\x03\x55\x04\x0B\x13\x19\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x20\x45\x6E\x64\x6F\x72\x73\x65\x64\x31\x28\x30\x26\x06\x03\x55\x04\x03\x13\x1F\x4F\x49\x53\x54\x45\x20\x57\x49\x53\x65\x4B\x65\x79\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x42\x20\x43\x41\x30\x1E\x17\x0D\x31\x34\x31\x32\x30\x31\x31\x35\x30\x30\x33\x32\x5A\x17\x0D\x33\x39\x31\x32\x30\x31\x31\x35\x31\x30\x33\x31\x5A\x30\x6D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x10\x30\x0E\x06\x03\x55\x04\x0A\x13\x07\x57\x49\x53\x65\x4B\x65\x79\x31\x22\x30\x20\x06\x03\x55\x04\x0B\x13\x19\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x20\x45\x6E\x64\x6F\x72\x73\x65\x64\x31\x28\x30\x26\x06\x03\x55\x04\x03\x13\x1F\x4F\x49\x53\x54\x45\x20\x57\x49\x53\x65\x4B\x65\x79\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x42\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xD8\x17\xB7\x1C\x4A\x24\x2A\xD6\x97\xB1\xCA\xE2\x1E\xFB\x7D\x38\xEF\x98\xF5\xB2\x39\x98\x4E\x27\xB8\x11\x5D\x7B\xD2\x25\x94\x88\x82\x15\x26\x6A\x1B\x31\xBB\xA8\x5B\x21\x21\x2B\xD8\x0F\x4E\x9F\x5A\xF1\xB1\x5A\xE4\x79\xD6\x32\x23\x2B\xE1\x53\xCC\x99\x45\x5C\x7B\x4F\xAD\xBC\xBF\x87\x4A\x0B\x4B\x97\x5A\xA8\xF6\x48\xEC\x7D\x7B\x0D\xCD\x21\x06\xDF\x9E\x15\xFD\x41\x8A\x48\xB7\x20\xF4\xA1\x7A\x1B\x57\xD4\x5D\x50\xFF\xBA\x67\xD8\x23\x99\x1F\xC8\x3F\xE3\xDE\xFF\x6F\x5B\x77\xB1\x6B\x6E\xB8\xC9\x64\xF7\xE1\xCA\x41\x46\x0E\x29\x71\xD0\xB9\x23\xFC\xC9\x81\x5F\x4E\xF7\x6F\xDF\xBF\x84\xAD\x73\x64\xBB\xB7\x42\x8E\x69\xF6\xD4\x76\x1D\x7E\x9D\xA7\xB8\x57\x8A\x51\x67\x72\xD7\xD4\xA8\xB8\x95\x54\x40\x73\x03\xF6\xEA\xF4\xEB\xFE\x28\x42\x77\x3F\x9D\x23\x1B\xB2\xB6\x3D\x80\x14\x07\x4C\x2E\x4F\xF7\xD5\x0A\x16\x0D\xBD\x66\x43\x37\x7E\x23\x43\x79\xC3\x40\x86\xF5\x4C\x29\xDA\x8E\x9A\xAD\x0D\xA5\x04\x87\x88\x1E\x85\xE3\xE9\x53\xD5\x9B\xC8\x8B\x03\x63\x78\xEB\xE0\x19\x4A\x6E\xBB\x2F\x6B\x33\x64\x58\x93\xAD\x69\xBF\x8F\x1B\xEF\x82\x48\xC7\x02\x03\x01\x00\x01\xA3\x51\x30\x4F\x30\x0B\x06\x03\x55\x1D\x0F\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x35\x0F\xC8\x36\x63\x5E\xE2\xA3\xEC\xF9\x3B\x66\x15\xCE\x51\x52\xE3\x91\x9A\x3D\x30\x10\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x15\x01\x04\x03\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x40\x4C\xFB\x87\xB2\x99\x81\x90\x7E\x9D\xC5\xB0\xB0\x26\xCD\x88\x7B\x2B\x32\x8D\x6E\xB8\x21\x71\x58\x97\x7D\xAE\x37\x14\xAF\x3E\xE7\xF7\x9A\xE2\x7D\xF6\x71\x98\x99\x04\xAA\x43\x74\x78\xA3\xE3\x49\x61\x3E\x73\x8C\x4D\x94\xE0\xF9\x71\xC4\xB6\x16\x0E\x53\x78\x1F\xD6\xA2\x87\x2F\x02\x39\x81\x29\x3C\xAF\x15\x98\x21\x30\xFE\x28\x90\x00\x8C\xD1\xE1\xCB\xFA\x5E\xC8\xFD\xF8\x10\x46\x3B\xA2\x78\x42\x91\x17\x74\x55\x0A\xDE\x50\x67\x4D\x66\xD1\xA7\xFF\xFD\xD9\xC0\xB5\xA8\xA3\x8A\xCE\x66\xF5\x0F\x43\xCD\xA7\x2B\x57\x7B\x63\x46\x6A\xAA\x2E\x52\xD8\xF4\xED\xE1\x6D\xAD\x29\x90\x78\x48\xBA\xE1\x23\xAA\xA3\x89\xEC\xB5\xAB\x96\xC0\xB4\x4B\xA2\x1D\x97\x9E\x7A\xF2\x6E\x40\x71\xDF\x68\xF1\x65\x4D\xCE\x7C\x05\xDF\x53\x65\xA9\xA5\xF0\xB1\x97\x04\x70\x15\x46\x03\x98\xD4\xD2\xBF\x54\xB4\xA0\x58\x7D\x52\x6F\xDA\x56\x26\x62\xD4\xD8\xDB\x89\x31\x6F\x1C\xF0\x22\xC2\xD3\x62\x1C\x35\xCD\x4C\x69\x15\x54\x1A\x90\x98\xDE\xEB\x1E\x5F\xCA\x77\xC7\xCB\x8E\x3D\x43\x69\x9C\x9A\x58\xD0\x24\x3B\xDF\x1B\x40\x96\x7E\x35\xAD\x81\xC7\x4E\x71\xBA\x88\x13"),
   // "NetLock Arany (Class Gold) Főtanúsítvány"
   CertificateDer::from_slice(b"\x30\x82\x04\x15\x30\x82\x02\xFD\xA0\x03\x02\x01\x02\x02\x06\x49\x41\x2C\xE4\x00\x10\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\xA7\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x55\x31\x11\x30\x0F\x06\x03\x55\x04\x07\x0C\x08\x42\x75\x64\x61\x70\x65\x73\x74\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x0C\x0C\x4E\x65\x74\x4C\x6F\x63\x6B\x20\x4B\x66\x74\x2E\x31\x37\x30\x35\x06\x03\x55\x04\x0B\x0C\x2E\x54\x61\x6E\xC3\xBA\x73\xC3\xAD\x74\x76\xC3\xA1\x6E\x79\x6B\x69\x61\x64\xC3\xB3\x6B\x20\x28\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x53\x65\x72\x76\x69\x63\x65\x73\x29\x31\x35\x30\x33\x06\x03\x55\x04\x03\x0C\x2C\x4E\x65\x74\x4C\x6F\x63\x6B\x20\x41\x72\x61\x6E\x79\x20\x28\x43\x6C\x61\x73\x73\x20\x47\x6F\x6C\x64\x29\x20\x46\xC5\x91\x74\x61\x6E\xC3\xBA\x73\xC3\xAD\x74\x76\xC3\xA1\x6E\x79\x30\x1E\x17\x0D\x30\x38\x31\x32\x31\x31\x31\x35\x30\x38\x32\x31\x5A\x17\x0D\x32\x38\x31\x32\x30\x36\x31\x35\x30\x38\x32\x31\x5A\x30\x81\xA7\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x55\x31\x11\x30\x0F\x06\x03\x55\x04\x07\x0C\x08\x42\x75\x64\x61\x70\x65\x73\x74\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x0C\x0C\x4E\x65\x74\x4C\x6F\x63\x6B\x20\x4B\x66\x74\x2E\x31\x37\x30\x35\x06\x03\x55\x04\x0B\x0C\x2E\x54\x61\x6E\xC3\xBA\x73\xC3\xAD\x74\x76\xC3\xA1\x6E\x79\x6B\x69\x61\x64\xC3\xB3\x6B\x20\x28\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x53\x65\x72\x76\x69\x63\x65\x73\x29\x31\x35\x30\x33\x06\x03\x55\x04\x03\x0C\x2C\x4E\x65\x74\x4C\x6F\x63\x6B\x20\x41\x72\x61\x6E\x79\x20\x28\x43\x6C\x61\x73\x73\x20\x47\x6F\x6C\x64\x29\x20\x46\xC5\x91\x74\x61\x6E\xC3\xBA\x73\xC3\xAD\x74\x76\xC3\xA1\x6E\x79\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xC4\x24\x5E\x73\xBE\x4B\x6D\x14\xC3\xA1\xF4\xE3\x97\x90\x6E\xD2\x30\x45\x1E\x3C\xEE\x67\xD9\x64\xE0\x1A\x8A\x7F\xCA\x30\xCA\x83\xE3\x20\xC1\xE3\xF4\x3A\xD3\x94\x5F\x1A\x7C\x5B\x6D\xBF\x30\x4F\x84\x27\xF6\x9F\x1F\x49\xBC\xC6\x99\x0A\x90\xF2\x0F\xF5\x7F\x43\x84\x37\x63\x51\x8B\x7A\xA5\x70\xFC\x7A\x58\xCD\x8E\x9B\xED\xC3\x46\x6C\x84\x70\x5D\xDA\xF3\x01\x90\x23\xFC\x4E\x30\xA9\x7E\xE1\x27\x63\xE7\xED\x64\x3C\xA0\xB8\xC9\x33\x63\xFE\x16\x90\xFF\xB0\xB8\xFD\xD7\xA8\xC0\xC0\x94\x43\x0B\xB6\xD5\x59\xA6\x9E\x56\xD0\x24\x1F\x70\x79\xAF\xDB\x39\x54\x0D\x65\x75\xD9\x15\x41\x94\x01\xAF\x5E\xEC\xF6\x8D\xF1\xFF\xAD\x64\xFE\x20\x9A\xD7\x5C\xEB\xFE\xA6\x1F\x08\x64\xA3\x8B\x76\x55\xAD\x1E\x3B\x28\x60\x2E\x87\x25\xE8\xAA\xAF\x1F\xC6\x64\x46\x20\xB7\x70\x7F\x3C\xDE\x48\xDB\x96\x53\xB7\x39\x77\xE4\x1A\xE2\xC7\x16\x84\x76\x97\x5B\x2F\xBB\x19\x15\x85\xF8\x69\x85\xF5\x99\xA7\xA9\xF2\x34\xA7\xA9\xB6\xA6\x03\xFC\x6F\x86\x3D\x54\x7C\x76\x04\x9B\x6B\xF9\x40\x5D\x00\x34\xC7\x2E\x99\x75\x9D\xE5\x88\x03\xAA\x4D\xF8\x03\xD2\x42\x76\xC0\x1B\x02\x03\x00\xA8\x8B\xA3\x45\x30\x43\x30\x12\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x08\x30\x06\x01\x01\xFF\x02\x01\x04\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xCC\xFA\x67\x93\xF0\xB6\xB8\xD0\xA5\xC0\x1E\xF3\x53\xFD\x8C\x53\xDF\x83\xD7\x96\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\xAB\x7F\xEE\x1C\x16\xA9\x9C\x3C\x51\x00\xA0\xC0\x11\x08\x05\xA7\x99\xE6\x6F\x01\x88\x54\x61\x6E\xF1\xB9\x18\xAD\x4A\xAD\xFE\x81\x40\x23\x94\x2F\xFB\x75\x7C\x2F\x28\x4B\x62\x24\x81\x82\x0B\xF5\x61\xF1\x1C\x6E\xB8\x61\x38\xEB\x81\xFA\x62\xA1\x3B\x5A\x62\xD3\x94\x65\xC4\xE1\xE6\x6D\x82\xF8\x2F\x25\x70\xB2\x21\x26\xC1\x72\x51\x1F\x8C\x2C\xC3\x84\x90\xC3\x5A\x8F\xBA\xCF\xF4\xA7\x65\xA5\xEB\x98\xD1\xFB\x05\xB2\x46\x75\x15\x23\x6A\x6F\x85\x63\x30\x80\xF0\xD5\x9E\x1F\x29\x1C\xC2\x6C\xB0\x50\x59\x5D\x90\x5B\x3B\xA8\x0D\x30\xCF\xBF\x7D\x7F\xCE\xF1\x9D\x83\xBD\xC9\x46\x6E\x20\xA6\xF9\x61\x51\xBA\x21\x2F\x7B\xBE\xA5\x15\x63\xA1\xD4\x95\x87\xF1\x9E\xB9\xF3\x89\xF3\x3D\x85\xB8\xB8\xDB\xBE\xB5\xB9\x29\xF9\xDA\x37\x05\x00\x49\x94\x03\x84\x44\xE7\xBF\x43\x31\xCF\x75\x8B\x25\xD1\xF4\xA6\x64\xF5\x92\xF6\xAB\x05\xEB\x3D\xE9\xA5\x0B\x36\x62\xDA\xCC\x06\x5F\x36\x8B\xB6\x5E\x31\xB8\x2A\xFB\x5E\xF6\x71\xDF\x44\x26\x9E\xC4\xE6\x0D\x91\xB4\x2E\x75\x95\x80\x51\x6A\x4B\x30\xA6\xB0\x62\xA1\x93\xF1\x9B\xD8\xCE\xC4\x63\x75\x3F\x59\x47\xB1"),
   // "DigiCert High Assurance EV Root CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xC5\x30\x82\x02\xAD\xA0\x03\x02\x01\x02\x02\x10\x02\xAC\x5C\x26\x6A\x0B\x40\x9B\x8F\x0B\x79\xF2\xAE\x46\x25\x77\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x6C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x44\x69\x67\x69\x43\x65\x72\x74\x20\x48\x69\x67\x68\x20\x41\x73\x73\x75\x72\x61\x6E\x63\x65\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x30\x36\x31\x31\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x31\x31\x31\x31\x30\x30\x30\x30\x30\x30\x30\x5A\x30\x6C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x44\x69\x67\x69\x43\x65\x72\x74\x20\x48\x69\x67\x68\x20\x41\x73\x73\x75\x72\x61\x6E\x63\x65\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xC6\xCC\xE5\x73\xE6\xFB\xD4\xBB\xE5\x2D\x2D\x32\xA6\xDF\xE5\x81\x3F\xC9\xCD\x25\x49\xB6\x71\x2A\xC3\xD5\x94\x34\x67\xA2\x0A\x1C\xB0\x5F\x69\xA6\x40\xB1\xC4\xB7\xB2\x8F\xD0\x98\xA4\xA9\x41\x59\x3A\xD3\xDC\x94\xD6\x3C\xDB\x74\x38\xA4\x4A\xCC\x4D\x25\x82\xF7\x4A\xA5\x53\x12\x38\xEE\xF3\x49\x6D\x71\x91\x7E\x63\xB6\xAB\xA6\x5F\xC3\xA4\x84\xF8\x4F\x62\x51\xBE\xF8\xC5\xEC\xDB\x38\x92\xE3\x06\xE5\x08\x91\x0C\xC4\x28\x41\x55\xFB\xCB\x5A\x89\x15\x7E\x71\xE8\x35\xBF\x4D\x72\x09\x3D\xBE\x3A\x38\x50\x5B\x77\x31\x1B\x8D\xB3\xC7\x24\x45\x9A\xA7\xAC\x6D\x00\x14\x5A\x04\xB7\xBA\x13\xEB\x51\x0A\x98\x41\x41\x22\x4E\x65\x61\x87\x81\x41\x50\xA6\x79\x5C\x89\xDE\x19\x4A\x57\xD5\x2E\xE6\x5D\x1C\x53\x2C\x7E\x98\xCD\x1A\x06\x16\xA4\x68\x73\xD0\x34\x04\x13\x5C\xA1\x71\xD3\x5A\x7C\x55\xDB\x5E\x64\xE1\x37\x87\x30\x56\x04\xE5\x11\xB4\x29\x80\x12\xF1\x79\x39\x88\xA2\x02\x11\x7C\x27\x66\xB7\x88\xB7\x78\xF2\xCA\x0A\xA8\x38\xAB\x0A\x64\xC2\xBF\x66\x5D\x95\x84\xC1\xA1\x25\x1E\x87\x5D\x1A\x50\x0B\x20\x12\xCC\x41\xBB\x6E\x0B\x51\x38\xB8\x4B\xCB\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB1\x3E\xC3\x69\x03\xF8\xBF\x47\x01\xD4\x98\x26\x1A\x08\x02\xEF\x63\x64\x2B\xC3\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xB1\x3E\xC3\x69\x03\xF8\xBF\x47\x01\xD4\x98\x26\x1A\x08\x02\xEF\x63\x64\x2B\xC3\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x1C\x1A\x06\x97\xDC\xD7\x9C\x9F\x3C\x88\x66\x06\x08\x57\x21\xDB\x21\x47\xF8\x2A\x67\xAA\xBF\x18\x32\x76\x40\x10\x57\xC1\x8A\xF3\x7A\xD9\x11\x65\x8E\x35\xFA\x9E\xFC\x45\xB5\x9E\xD9\x4C\x31\x4B\xB8\x91\xE8\x43\x2C\x8E\xB3\x78\xCE\xDB\xE3\x53\x79\x71\xD6\xE5\x21\x94\x01\xDA\x55\x87\x9A\x24\x64\xF6\x8A\x66\xCC\xDE\x9C\x37\xCD\xA8\x34\xB1\x69\x9B\x23\xC8\x9E\x78\x22\x2B\x70\x43\xE3\x55\x47\x31\x61\x19\xEF\x58\xC5\x85\x2F\x4E\x30\xF6\xA0\x31\x16\x23\xC8\xE7\xE2\x65\x16\x33\xCB\xBF\x1A\x1B\xA0\x3D\xF8\xCA\x5E\x8B\x31\x8B\x60\x08\x89\x2D\x0C\x06\x5C\x52\xB7\xC4\xF9\x0A\x98\xD1\x15\x5F\x9F\x12\xBE\x7C\x36\x63\x38\xBD\x44\xA4\x7F\xE4\x26\x2B\x0A\xC4\x97\x69\x0D\xE9\x8C\xE2\xC0\x10\x57\xB8\xC8\x76\x12\x91\x55\xF2\x48\x69\xD8\xBC\x2A\x02\x5B\x0F\x44\xD4\x20\x31\xDB\xF4\xBA\x70\x26\x5D\x90\x60\x9E\xBC\x4B\x17\x09\x2F\xB4\xCB\x1E\x43\x68\xC9\x07\x27\xC1\xD2\x5C\xF7\xEA\x21\xB9\x68\x12\x9C\x3C\x9C\xBF\x9E\xFC\x80\x5C\x9B\x63\xCD\xEC\x47\xAA\x25\x27\x67\xA0\x37\xF3\x00\x82\x7D\x54\xD7\xA9\xF8\xE9\x2E\x13\xA3\x77\xE8\x1F\x4A"),
   // "Certainly Root R1"
   CertificateDer::from_slice(b"\x30\x82\x05\x47\x30\x82\x03\x2F\xA0\x03\x02\x01\x02\x02\x11\x00\x8E\x0F\xF9\x4B\x90\x71\x68\x65\x33\x54\xF4\xD4\x44\x39\xB7\xE0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x3D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x31\x1A\x30\x18\x06\x03\x55\x04\x03\x13\x11\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x20\x52\x6F\x6F\x74\x20\x52\x31\x30\x1E\x17\x0D\x32\x31\x30\x34\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x34\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x30\x3D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x31\x1A\x30\x18\x06\x03\x55\x04\x03\x13\x11\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x20\x52\x6F\x6F\x74\x20\x52\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD0\x36\xD4\x1F\xEA\xDD\xAB\xE4\xD1\xB6\xE6\xFB\x22\xC0\xDD\x13\x0D\x6A\x7B\x22\x13\x1C\x97\x3C\x68\x63\x66\x32\x9C\x03\xB5\x8D\xA4\x81\x83\xDA\x78\x30\x11\xCF\xDC\xB2\x2B\xBE\x92\xBF\x8E\xE4\xC4\x13\xBE\xA4\x68\x4C\xDA\x02\x68\x16\x74\xBE\xB2\xDD\x04\xE4\x6B\x2A\xDD\x37\x1F\x60\x2C\xDB\xF5\xF7\xA1\x7C\x95\xB7\x0C\x70\x86\x2E\xF1\x3A\xEF\x52\xF7\xCC\xD3\x9B\xF9\x8B\xBE\x0E\xDF\x31\xB7\x9D\x68\x5C\x92\xA6\xF5\xE5\xF3\x0A\x34\xB5\xFF\x7B\xA2\xE4\x87\xA1\xC6\xAF\x17\x00\xEF\x03\x91\xED\xA9\x1C\x4E\x71\x3D\xD2\x8B\x6C\x89\xF4\x78\x86\xE6\x6A\x49\xA0\xCE\xB5\xD2\xB0\xAB\x9B\xF6\xF4\xD4\x2E\xE3\x72\xF9\x36\xC6\xEB\x15\xB7\x25\x8C\x3A\xFC\x25\x0D\xB3\x22\x73\x21\x74\xC8\x4A\x96\x61\x92\xF5\x2F\x0B\x18\xA5\xF4\xAD\xE2\xEE\x41\xBD\x01\x79\xFA\x96\x8C\x8D\x17\x02\x30\xB4\xF9\xAF\x78\x1A\x8C\xB4\x36\x10\x10\x07\x05\x70\xD0\xF4\x31\x90\x8A\x51\xC5\x86\x26\x79\xB2\x11\x88\x5E\xC5\xF0\x0A\x54\xCD\x49\xA6\xBF\x02\x9C\xD2\x44\xA7\xED\xE3\x78\xEF\x46\x5E\x6D\x71\xD1\x79\x70\x1C\x46\x5F\x51\xE9\xC9\x37\xDC\x5F\x7E\x69\x7B\x41\xDF\x34\x45\xE0\x3B\x84\xF4\xA1\x8A\x0A\x36\x9E\x37\xCC\x62\x52\xE1\x89\x0D\x28\xF9\x7A\x23\xB1\x0D\x3D\x3D\x9A\xFD\x9D\x81\xEF\x2C\x90\xC0\x7B\x44\x4E\xBB\x49\xE0\x0E\x4A\x56\x92\xBC\xCB\xB5\xDD\x79\x17\x89\x91\xDE\x61\x89\x74\x92\xA8\xE3\x32\x85\xBE\x4E\x85\xA4\x4B\x59\xCB\x2B\xC5\x78\x8E\x71\x54\xD0\x02\x37\x99\x8C\xE5\x49\xEA\xE0\x54\x72\xA4\x11\x06\x2F\x0B\x8C\xC1\x5B\xBE\xB5\xA1\xB0\x53\x6E\x9C\xB8\x60\x91\x1F\x59\x6B\xF9\x2D\xF4\x94\x0A\x97\xB5\xEC\xC5\x76\x03\x54\x1B\x65\x52\xBA\x4C\x92\x56\x51\x35\xA0\x40\xD8\x29\xDB\xAE\x52\x76\x3B\x2D\x30\x40\x9B\x8A\xD0\x42\x56\xB4\xB7\x88\x01\xA4\x87\x3B\x53\x96\xCD\xA3\x16\x8F\xF3\x66\xAA\x17\xB1\xC7\x60\xE0\xC1\x43\x05\x0C\xEE\x9B\x5B\x60\x6F\x06\x5C\x87\x5B\x27\xF9\x40\x11\x9E\x9C\x33\xC1\xB7\xE5\x35\x57\x05\x7F\x27\xCE\x17\x20\x8C\x1C\xFC\xF1\xFB\xDA\x31\x29\x49\xED\xF5\x0B\x84\xA7\x4F\xC1\xF6\x4E\xC2\x28\x9C\xFA\xEE\xE0\xAF\x07\xFB\x33\x11\x7A\x21\x4F\x0B\x21\x10\xB6\x40\x3A\xAB\x22\x3A\x04\x9C\x8B\x9B\x84\x86\x72\x9A\xD2\xA7\xA5\xC4\xB4\x75\x91\xA9\x2B\x23\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE0\xAA\x3F\x25\x8D\x9F\x44\x5C\xC1\x3A\xE8\x2E\xAE\x77\x4C\x84\x3E\x67\x0C\xF4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\xB9\x57\xAF\xB8\x12\xDA\x57\x83\x8F\x68\x0B\x33\x1D\x03\x53\x55\xF4\x95\x70\xE4\x2B\x3D\xB0\x39\xEB\xFA\x89\x62\xFD\xF7\xD6\x18\x04\x2F\x21\x34\xDD\xF1\x68\xF0\xD5\x96\x5A\xDE\xC2\x80\xA3\xC1\x8D\xC6\x6A\xF7\x59\x77\xAE\x15\x64\xCF\x5B\x79\x05\x77\x66\xEA\x8C\xD3\x6B\x0D\xDD\xF1\x59\x2C\xC1\x33\xA5\x30\x80\x15\x45\x07\x45\x1A\x31\x22\xB6\x92\x00\xAB\x99\x4D\x3A\x8F\x77\xAF\xA9\x22\xCA\x2F\x63\xCA\x15\xD6\xC7\xC6\xF0\x3D\x6C\xFC\x1C\x0D\x98\x10\x61\x9E\x11\xA2\x22\xD7\x0A\xF2\x91\x7A\x6B\x39\x0E\x2F\x30\xC3\x36\x49\x9F\xE0\xE9\x0F\x02\x44\x50\x37\x94\x55\x7D\xEA\x9F\xF6\x3B\xBA\x94\xA5\x4C\xE9\xBC\x3E\x51\xB4\xE8\xCA\x92\x36\x54\x6D\x5C\x25\x28\xDA\xDD\xAD\x14\xFD\xD3\xEE\xE2\x22\x05\xEB\xD0\xF2\xB7\x68\x12\xD7\x5A\x8A\x41\x1A\xC6\x92\xA5\x5A\x3B\x63\x45\x4F\xBF\xE1\x3A\x77\x22\x2F\x5C\xBF\x46\xF9\x5A\x03\x85\x13\x42\x5F\xCA\xDE\x53\xD7\x62\xB5\xA6\x35\x04\xC2\x47\xFF\x99\xFD\x84\xDF\x5C\xCE\xE9\x5E\x80\x28\x41\xF2\x7D\xE7\x1E\x90\xD8\x4F\x76\x3E\x82\x3C\x0D\xFC\xA5\x03\xFA\x7B\x1A\xD9\x45\x1E\x60\xDA\xC4\x8E\xF9\xFC\x2B\xC9\x7B\x95\xC5\x2A\xFF\xAA\x89\xDF\x82\x31\x0F\x72\xFF\x0C\x27\xD7\x0A\x1E\x56\x00\x50\x1E\x0C\x90\xC1\x96\xB5\xD8\x14\x85\xBB\xA7\x0D\x16\xC1\xF8\x07\x24\x1B\xBA\x85\xA1\x1A\x05\x09\x80\xBA\x95\x63\xC9\x3A\xEC\x25\x9F\x7F\x9D\xBA\xA4\x47\x15\x9B\x44\x70\xF1\x6A\x4B\xD6\x38\x5E\x43\xF3\x18\x7E\x50\x6E\xE9\x5A\x28\xE6\x65\xE6\x77\x1B\x3A\xFD\x1D\xBE\x03\x26\xA3\xDB\xD4\xE1\xBB\x7E\x96\x27\x2B\x1D\xEE\xA4\xFB\xDA\x25\x54\x13\x03\xDE\x39\xC6\xC3\x1F\x4D\x90\xEC\x8F\x1B\x4A\xD2\x1C\xED\x85\x95\x38\x50\x79\x46\xD6\xC1\x90\x50\x31\xA9\x5C\x9A\x6E\x1D\xF5\x33\x56\x8B\xA7\x99\xD2\xF2\xC8\x2C\x33\x93\x92\x30\xC7\x4E\x8C\x65\x33\x10\x64\x17\xFD\x24\x17\x96\xD1\x8D\xC2\x3A\x6A\x2B\xEB\x13\x8B\x44\xF2\x21\xF3\x4A\x1A\xB7\x77\x5F\xD7\xED\x88\xA4\x72\xE5\x39\x1F\x95\x9D\xBE\x67\xC1\x70\x11\x3D\xBB\xF4\xF8\x49\xB7\xE3\x26\x97\x3A\x9F\xD2\x5F\x7C\xFB\xC0\x99\x7C\x39\x29\xE0\x7B\x1D\xBF\x0D\xA7\x8F\xD2\x29\x34\x6E\x24\x15\xCB\xDE\x90\x5E\xBF\x1A\xC4\x66\xEA\xC2\xE6\xBA\x39\x5F\x8A\x99\xA9\x41\x59\x07\xB0\x2C\xAF"),
   // "Sectigo Public Server Authentication Root R46"
   CertificateDer::from_slice(b"\x30\x82\x05\x8A\x30\x82\x03\x72\xA0\x03\x02\x01\x02\x02\x10\x75\x8D\xFD\x8B\xAE\x7C\x07\x00\xFA\xA9\x25\xA7\xE1\xC7\xAD\x14\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x5F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x13\x0F\x53\x65\x63\x74\x69\x67\x6F\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x53\x65\x63\x74\x69\x67\x6F\x20\x50\x75\x62\x6C\x69\x63\x20\x53\x65\x72\x76\x65\x72\x20\x41\x75\x74\x68\x65\x6E\x74\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x52\x34\x36\x30\x1E\x17\x0D\x32\x31\x30\x33\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x33\x32\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x5F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x13\x0F\x53\x65\x63\x74\x69\x67\x6F\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x53\x65\x63\x74\x69\x67\x6F\x20\x50\x75\x62\x6C\x69\x63\x20\x53\x65\x72\x76\x65\x72\x20\x41\x75\x74\x68\x65\x6E\x74\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x52\x34\x36\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x93\xBE\xD5\x36\x52\x75\xD8\x01\x23\xA0\x1C\x47\x42\x49\xEE\x63\xB6\xB7\x21\xFD\xC4\x95\xD5\x48\x2B\x26\x7C\x14\x53\x10\xDA\x79\xFD\x2B\xB7\x2D\xA4\xD4\x2C\xFA\xEA\x32\xDD\x49\xC2\xB9\xBD\x0F\x48\x3D\x7B\x5A\x98\x54\xAF\x9E\x5D\x31\x74\x4F\x07\xFC\x50\x21\xDD\xA4\xCF\x68\x4F\x1B\x12\x63\x6D\x25\x99\x4C\x2A\x99\xF3\x48\x30\x61\xFA\x81\x7C\x1E\xA7\x08\x4A\xDC\x3E\x2B\x1C\x1F\x18\x4C\x71\xAA\x35\x8C\xAD\xF8\x6E\xE8\x3B\x4A\xD9\xE5\x94\x02\xD6\x89\x84\x13\xAA\x6D\xC8\x4F\x33\xCC\x50\x96\x37\x92\x33\xDC\x5F\x88\xE7\x9F\x54\xD9\x48\xF0\x98\x43\xD6\x66\xFD\x9F\x17\x38\x43\xC5\x01\x51\x0B\xD7\xE3\x23\x0F\x14\x5D\x5B\x14\xE7\x4B\xBE\xDD\xF4\xC8\xDA\x03\x37\xD1\xD6\x39\xA1\x21\x51\x30\x83\xB0\x6D\xD7\x30\x4E\x96\x5B\x91\xF0\x70\x24\xAB\xBF\x45\x81\x64\x43\x0D\xBD\x21\x3A\x2F\x3C\xE9\x9E\x0D\xCB\x20\xB5\x42\x27\xCC\xDA\x6F\x9B\xEE\x64\x30\x90\x39\xCD\x93\x65\x81\x21\x31\xB5\x23\x50\x33\x37\x22\xE3\x38\xED\xF8\x31\x30\xCC\x45\xFE\x62\xF9\xD1\x5D\x32\x79\x42\x87\xDF\x6A\xCC\x56\x19\x40\x4D\xCE\xAA\xBB\xF9\xB5\x76\x49\x94\xF1\x27\xF8\x91\xA5\x83\xE5\x06\xB3\x63\x0E\x80\xDC\xE0\x12\x55\x80\xA6\x3B\x66\xB4\x39\x87\x2D\xC8\xF0\xD0\xD1\x14\xE9\xE4\x0D\x4D\x0E\xF6\x5D\x57\x72\xC5\x3B\x1C\x47\x56\x9D\xE2\xD5\xFB\x81\x61\x8C\xCC\x4D\x80\x90\x34\x5B\xB7\xD7\x14\x75\xDC\xD8\x04\x48\x9F\xC0\xC1\x28\x88\xB4\xE9\x1C\xCA\xA7\xB1\xF1\x56\xB7\x7B\x49\x4C\x59\xE5\x20\x15\xA8\x84\x02\x29\xFA\x38\x94\x69\x9A\x49\x06\x8F\xCD\x1F\x79\x14\x17\x12\x0C\x83\x7A\xDE\x1F\xB1\x97\xEE\xF9\x97\x78\x28\xA4\xC8\x44\x92\xE9\x7D\x26\x05\xA6\x58\x72\x9B\x79\x13\xD8\x11\x5F\xAE\xC5\x38\x62\x34\x68\xB2\x86\x30\x8E\xF8\x90\x61\x9E\x32\x6C\xF5\x07\x36\xCD\xA2\x4C\x6E\xEC\x8A\x36\xED\xF2\xE6\x99\x15\x44\x70\xC3\x7C\xBC\x9C\x39\xC0\xB4\xE1\x6B\xF7\x83\x25\x23\x57\xD9\x12\x80\xE5\x49\xF0\x75\x0F\xEF\x8D\xEB\x1C\x9B\x54\x28\xB4\x21\x3C\xFC\x7C\x0A\xFF\xEF\x7B\x6B\x75\xFF\x8B\x1D\xA0\x19\x05\xAB\xFA\xF8\x2B\x81\x42\xE8\x38\xBA\xBB\xFB\xAA\xFD\x3D\xE0\xF3\xCA\xDF\x4E\x97\x97\x29\xED\xF3\x18\x56\xE9\xA5\x96\xAC\xBD\xC3\x90\x98\xB2\xE0\xF9\xA2\xD4\xA6\x47\x43\x7C\x6D\xCF\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x56\x73\x58\x64\x95\xF9\x92\x1A\xB0\x12\x2A\x04\x62\x79\xA1\x40\x15\x88\x21\x49\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x2F\x5C\x99\x3C\xFC\x06\x5E\x8C\x94\x2E\x70\xEA\xD2\x32\x31\x8D\xB4\xF0\x51\xD5\xBC\x0A\xF3\x64\x9F\x07\x5E\xD5\xC1\x73\x68\x64\x7A\xA2\xB9\x0E\xE8\xF9\x5D\x85\x2D\xA8\x37\x45\xAA\x28\xF4\x96\x05\x50\x60\xA9\x49\x7E\x9F\xE2\x99\x36\x29\x13\x44\x47\x6A\x9D\x55\x20\x3C\xD8\x9B\xF1\x03\x32\xBA\xDA\x40\xA1\x73\xEA\x83\xA1\xB7\x44\xA6\x0E\x99\x01\x9B\xE4\xBC\x7F\xBE\x13\x94\x7E\xCA\xA6\x1E\x76\x80\x36\x3D\x84\x06\x8B\x33\x26\x65\x6D\xCA\x7E\x9E\xFE\x1F\x8C\x58\x38\x7B\x1A\x83\xB1\x0F\xBC\x17\x11\xBB\xE6\x06\xCC\x63\xFA\x81\xF2\x81\x4C\xDA\x0B\x10\x6B\xA1\xFA\xD5\x28\xA5\xCF\x06\x40\x16\xFF\x7B\x7D\x18\x5E\x39\x12\xA4\x53\x9E\x7E\x32\x42\x10\xA6\x21\x91\xA9\x1C\x4E\x17\x7C\x84\xBC\x9F\x8C\xD1\xE8\xDF\xE6\x51\xB9\x36\x47\x3F\x90\xB9\xC7\xBC\x02\xDC\x5B\x1C\x4F\x0E\x48\xC1\x25\x83\x9C\x0A\x3F\x9E\xB1\x03\x33\x12\x1A\x27\xAC\xF7\x22\x6C\x24\xD1\x01\x41\xF8\x58\x03\xFE\x25\x68\x22\x1F\x9A\x5A\x3C\x7C\x6C\x9E\x75\x48\xF3\x81\xF1\x66\x67\x6E\x4C\x82\xC0\xEE\xBA\x57\x0E\x18\xEF\x2E\x9A\xF7\x12\xD8\xA0\x6B\xE9\x05\xA5\xA1\xE9\x68\xF8\xBC\x4C\x3F\x12\x1E\x45\xE8\x52\xC0\xA3\xBF\x12\x27\x79\xB9\xCC\x31\x3C\xC3\xF6\x3A\x22\x16\x03\xA0\xC9\x8F\x66\xA4\x5B\xA2\x4D\xD6\x81\x25\x06\xE9\x76\xA4\x00\x0A\x3E\xCB\xCD\x35\x9B\xE0\xE1\x38\xCB\x60\x53\x86\x28\x42\x41\x1C\x44\x57\xE8\xA8\xAD\xAB\x45\xE3\x25\x10\xBC\xDB\x3E\x65\x41\xFB\x1B\xA6\x97\x0F\xEB\xB9\x74\x79\xF9\x1E\xBC\x1D\x57\x0D\x47\xAF\xC3\x2F\x9F\x87\x46\xA7\xEB\x26\x5A\x0F\x56\x63\xB5\x62\x60\x6E\x00\xFB\xE3\x27\x11\x22\xE7\xFE\x99\x8F\x34\xF5\xB9\xE8\xC3\x91\x72\xBD\xD8\xC3\x1E\xB9\x2E\xF2\x91\x44\x51\xD0\x57\xCD\x0C\x34\xD5\x48\x21\xBF\xDB\x13\xF1\x66\x25\x43\x52\xD2\x70\x22\x36\xCD\x9F\xC4\x1C\x75\x20\xAD\x63\x72\x63\x06\x0F\x0E\x27\xCE\xD2\x6A\x0D\xBC\xB5\x39\x1A\xE9\xD1\x76\x7A\xD1\x5C\xE4\xE7\x49\x49\x2D\x55\x37\x68\xF0\x1A\x3A\x98\x3E\x54\x17\x87\x54\xE9\xA6\x27\x50\x89\x7B\x20\x2F\x3F\xFF\xBF\xA1\x8B\x4A\x47\x98\xFF\x2B\x7B\x49\x3E\xC3\x29\x46\x60\x18\x42\xAB\x33\x29\xBA\xC0\x29\xB9\x13\x89\xD3\x88\x8A\x39\x41\x3B\xC9\xFD\xA6\xED\x1F\xF4\x60\x63\xDF\xD2\x2D\x55\x01\x8B"),
   // "DigiCert Assured ID Root G2"
   CertificateDer::from_slice(b"\x30\x82\x03\x96\x30\x82\x02\x7E\xA0\x03\x02\x01\x02\x02\x10\x0B\x93\x1C\x3A\xD6\x39\x67\xEA\x67\x23\xBF\xC3\xAF\x9A\xF4\x4B\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x44\x69\x67\x69\x43\x65\x72\x74\x20\x41\x73\x73\x75\x72\x65\x64\x20\x49\x44\x20\x52\x6F\x6F\x74\x20\x47\x32\x30\x1E\x17\x0D\x31\x33\x30\x38\x30\x31\x31\x32\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x35\x31\x32\x30\x30\x30\x30\x5A\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x44\x69\x67\x69\x43\x65\x72\x74\x20\x41\x73\x73\x75\x72\x65\x64\x20\x49\x44\x20\x52\x6F\x6F\x74\x20\x47\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xD9\xE7\x28\x2F\x52\x3F\x36\x72\x49\x88\x93\x34\xF3\xF8\x6A\x1E\x31\x54\x80\x9F\xAD\x54\x41\xB5\x47\xDF\x96\xA8\xD4\xAF\x80\x2D\xB9\x0A\xCF\x75\xFD\x89\xA5\x7D\x24\xFA\xE3\x22\x0C\x2B\xBC\x95\x17\x0B\x33\xBF\x19\x4D\x41\x06\x90\x00\xBD\x0C\x4D\x10\xFE\x07\xB5\xE7\x1C\x6E\x22\x55\x31\x65\x97\xBD\xD3\x17\xD2\x1E\x62\xF3\xDB\xEA\x6C\x50\x8C\x3F\x84\x0C\x96\xCF\xB7\xCB\x03\xE0\xCA\x6D\xA1\x14\x4C\x1B\x89\xDD\xED\x00\xB0\x52\x7C\xAF\x91\x6C\xB1\x38\x13\xD1\xE9\x12\x08\xC0\x00\xB0\x1C\x2B\x11\xDA\x77\x70\x36\x9B\xAE\xCE\x79\x87\xDC\x82\x70\xE6\x09\x74\x70\x55\x69\xAF\xA3\x68\x9F\xBF\xDD\xB6\x79\xB3\xF2\x9D\x70\x29\x55\xF4\xAB\xFF\x95\x61\xF3\xC9\x40\x6F\x1D\xD1\xBE\x93\xBB\xD3\x88\x2A\xBB\x9D\xBF\x72\x5A\x56\x71\x3B\x3F\xD4\xF3\xD1\x0A\xFE\x28\xEF\xA3\xEE\xD9\x99\xAF\x03\xD3\x8F\x60\xB7\xF2\x92\xA1\xB1\xBD\x89\x89\x1F\x30\xCD\xC3\xA6\x2E\x62\x33\xAE\x16\x02\x77\x44\x5A\xE7\x81\x0A\x3C\xA7\x44\x2E\x79\xB8\x3F\x04\xBC\x5C\xA0\x87\xE1\x1B\xAF\x51\x8E\xCD\xEC\x2C\xFA\xF8\xFE\x6D\xF0\x3A\x7C\xAA\x8B\xE4\x67\x95\x31\x8D\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xCE\xC3\x4A\xB9\x99\x55\xF2\xB8\xDB\x60\xBF\xA9\x7E\xBD\x56\xB5\x97\x36\xA7\xD6\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\xCA\xA5\x55\x8C\xE3\xC8\x41\x6E\x69\x27\xA7\x75\x11\xEF\x3C\x86\x36\x6F\xD2\x9D\xC6\x78\x38\x1D\x69\x96\xA2\x92\x69\x2E\x38\x6C\x9B\x7D\x04\xD4\x89\xA5\xB1\x31\x37\x8A\xC9\x21\xCC\xAB\x6C\xCD\x8B\x1C\x9A\xD6\xBF\x48\xD2\x32\x66\xC1\x8A\xC0\xF3\x2F\x3A\xEF\xC0\xE3\xD4\x91\x86\xD1\x50\xE3\x03\xDB\x73\x77\x6F\x4A\x39\x53\xED\xDE\x26\xC7\xB5\x7D\xAF\x2B\x42\xD1\x75\x62\xE3\x4A\x2B\x02\xC7\x50\x4B\xE0\x69\xE2\x96\x6C\x0E\x44\x66\x10\x44\x8F\xAD\x05\xEB\xF8\x79\xAC\xA6\x1B\xE8\x37\x34\x9D\x53\xC9\x61\xAA\xA2\x52\xAF\x4A\x70\x16\x86\xC2\x3A\xC8\xB1\x13\x70\x36\xD8\xCF\xEE\xF4\x0A\x34\xD5\x5B\x4C\xFD\x07\x9C\xA2\xBA\xD9\x01\x72\x5C\xF3\x4D\xC1\xDD\x0E\xB1\x1C\x0D\xC4\x63\xBE\xAD\xF4\x14\xFB\x89\xEC\xA2\x41\x0E\x4C\xCC\xC8\x57\x40\xD0\x6E\x03\xAA\xCD\x0C\x8E\x89\x99\x99\x6C\xF0\x3C\x30\xAF\x38\xDF\x6F\xBC\xA3\xBE\x29\x20\x27\xAB\x74\xFF\x13\x22\x78\xDE\x97\x52\x55\x1E\x83\xB5\x54\x20\x03\xEE\xAE\xC0\x4F\x56\xDE\x37\xCC\xC3\x7F\xAA\x04\x27\xBB\xD3\x77\xB8\x62\xDB\x17\x7C\x9C\x28\x22\x13\x73\x6C\xCF\x26\xF5\x8A\x29\xE7"),
   // "DigiCert Assured ID Root G3"
   CertificateDer::from_slice(b"\x30\x82\x02\x46\x30\x82\x01\xCD\xA0\x03\x02\x01\x02\x02\x10\x0B\xA1\x5A\xFA\x1D\xDF\xA0\xB5\x49\x44\xAF\xCD\x24\xA0\x6C\xEC\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x44\x69\x67\x69\x43\x65\x72\x74\x20\x41\x73\x73\x75\x72\x65\x64\x20\x49\x44\x20\x52\x6F\x6F\x74\x20\x47\x33\x30\x1E\x17\x0D\x31\x33\x30\x38\x30\x31\x31\x32\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x35\x31\x32\x30\x30\x30\x30\x5A\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x44\x69\x67\x69\x43\x65\x72\x74\x20\x41\x73\x73\x75\x72\x65\x64\x20\x49\x44\x20\x52\x6F\x6F\x74\x20\x47\x33\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x19\xE7\xBC\xAC\x44\x65\xED\xCD\xB8\x3F\x58\xFB\x8D\xB1\x57\xA9\x44\x2D\x05\x15\xF2\xEF\x0B\xFF\x10\x74\x9F\xB5\x62\x52\x5F\x66\x7E\x1F\xE5\xDC\x1B\x45\x79\x0B\xCC\xC6\x53\x0A\x9D\x8D\x5D\x02\xD9\xA9\x59\xDE\x02\x5A\xF6\x95\x2A\x0E\x8D\x38\x4A\x8A\x49\xC6\xBC\xC6\x03\x38\x07\x5F\x55\xDA\x7E\x09\x6E\xE2\x7F\x5E\xD0\x45\x20\x0F\x59\x76\x10\xD6\xA0\x24\xF0\x2D\xDE\x36\xF2\x6C\x29\x39\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xCB\xD0\xBD\xA9\xE1\x98\x05\x51\xA1\x4D\x37\xA2\x83\x79\xCE\x8D\x1D\x2A\xE4\x84\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x67\x00\x30\x64\x02\x30\x25\xA4\x81\x45\x02\x6B\x12\x4B\x75\x74\x4F\xC8\x23\xE3\x70\xF2\x75\x72\xDE\x7C\x89\xF0\xCF\x91\x72\x61\x9E\x5E\x10\x92\x59\x56\xB9\x83\xC7\x10\xE7\x38\xE9\x58\x26\x36\x7D\xD5\xE4\x34\x86\x39\x02\x30\x7C\x36\x53\xF0\x30\xE5\x62\x63\x3A\x99\xE2\xB6\xA3\x3B\x9B\x34\xFA\x1E\xDA\x10\x92\x71\x5E\x91\x13\xA7\xDD\xA4\x6E\x92\xCC\x32\xD6\xF5\x21\x66\xC7\x2F\xEA\x96\x63\x6A\x65\x45\x92\x95\x01\xB4"),
   // "Atos TrustedRoot Root CA RSA TLS 2021"
   CertificateDer::from_slice(b"\x30\x82\x05\x64\x30\x82\x03\x4C\xA0\x03\x02\x01\x02\x02\x10\x53\xD5\xCF\xE6\x19\x93\x0B\xFB\x2B\x05\x12\xD8\xC2\x2A\xA2\xA4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x4C\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x0C\x25\x41\x74\x6F\x73\x20\x54\x72\x75\x73\x74\x65\x64\x52\x6F\x6F\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x52\x53\x41\x20\x54\x4C\x53\x20\x32\x30\x32\x31\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x74\x6F\x73\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x30\x1E\x17\x0D\x32\x31\x30\x34\x32\x32\x30\x39\x32\x31\x31\x30\x5A\x17\x0D\x34\x31\x30\x34\x31\x37\x30\x39\x32\x31\x30\x39\x5A\x30\x4C\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x0C\x25\x41\x74\x6F\x73\x20\x54\x72\x75\x73\x74\x65\x64\x52\x6F\x6F\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x52\x53\x41\x20\x54\x4C\x53\x20\x32\x30\x32\x31\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x74\x6F\x73\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB6\x80\x0E\xC4\x79\xBD\x05\x8C\x7D\xB0\xA3\x9D\x4D\x22\x4D\xCB\xF0\x41\x97\x4D\x59\xE0\xD1\xFE\x56\x8C\x97\xF2\xD7\xBD\x8F\x6C\xB7\x23\x8F\x5F\xD5\xC4\xD8\x41\xCB\xF2\x02\x1E\x71\xE5\xE9\xF6\x5E\xCB\x08\x2A\x5E\x30\xF2\x2D\x66\xC7\x84\x1B\x64\x57\x38\x9D\x75\x2D\x56\xC6\x2F\x61\xEF\x96\xFC\x20\x46\xBD\xEB\xD4\x7B\x3F\x3F\x7C\x47\x38\x04\xA9\x1B\xAA\x52\xDF\x13\x37\xD3\x15\x15\x4E\xBD\x5F\x7C\xAF\xAD\x63\xC7\x79\xDC\x08\x7B\xD5\xA0\xE5\xF7\x5B\x75\xAC\x80\x55\x99\x92\x61\x9B\xCD\x2A\x17\x7D\xDB\x8F\xF4\xB5\x6A\xEA\x17\x4A\x64\x28\x66\x15\x29\x6C\x02\xF1\x6B\xD5\xBA\xA3\x33\xDC\x5A\x67\xA7\x05\xE2\xBF\x65\xB6\x16\xB0\x10\xED\xCD\x50\x33\xC9\x70\x50\xEC\x19\x8E\xB0\xC7\xF2\x74\x5B\x6B\x44\xC6\x7D\x96\xB9\x98\x08\x59\x66\xDE\x29\x01\x9B\xF4\x2A\x6D\xD3\x15\x3A\x90\x6A\x67\xF1\xB4\x6B\x66\xD9\x21\xEB\xCA\xD9\x62\x7C\x46\x10\x5C\xDE\x75\x49\x67\x9E\x42\xF9\xFE\x75\xA9\xA3\xAD\xFF\x76\x0A\x67\x40\xE3\xC5\xF7\x8D\xC7\x85\x9A\x59\x9E\x62\x9A\x6A\xED\x45\x87\x98\x67\xB2\xD5\x4A\x3C\xD7\xB4\x3B\x00\x0D\xC0\x8F\x1F\xE1\x40\xC4\xAE\x6C\x21\xDC\x49\x7E\x7E\xCA\xB2\x8D\x6D\xB6\xBF\x93\x2F\xA1\x5C\x3E\x8F\xCA\xED\x80\x8E\x58\xE1\xDB\x57\xCF\x85\x36\x38\xB2\x71\xA4\x09\x8C\x92\x89\x08\x88\x48\xF1\x40\x63\x18\xB2\x5B\x8C\x5A\xE3\xC3\xD3\x17\xAA\xAB\x19\xA3\x2C\x1B\xE4\xD5\xC6\xE2\x66\x7A\xD7\x82\x19\xA6\x3B\x16\x2C\x2F\x71\x87\x5F\x45\x9E\x95\x73\x93\xC2\x42\x81\x21\x13\x96\xD7\x9D\xBB\x93\x68\x15\xFA\x9D\xA4\x1D\x8C\xF2\x81\xE0\x58\x06\xBD\xC9\xB6\xE3\xF6\x89\x5D\x89\xF9\xAC\x44\xA1\xCB\x6B\xFA\x16\xF1\xC7\x50\x3D\x24\xDA\xF7\xC3\xE4\x87\xD5\x56\xF1\x4F\x90\x30\xFA\x45\x09\x59\xDA\x34\xCE\xE0\x13\x1C\x04\x7C\x00\xD4\x9B\x86\xA4\x40\xBC\xD9\xDC\x4C\x57\x7E\xAE\xB7\x33\xB6\x5E\x76\xE1\x65\x8B\x66\xDF\x8D\xCA\xD7\x98\xAF\xCE\x36\x98\x8C\x9C\x83\x99\x03\x70\xF3\xAF\x74\xED\xC6\x0E\x36\xE7\xBD\xEC\xC1\x73\xA7\x94\x5A\xCB\x92\x64\x82\xA6\x00\xC1\x70\xA1\x6E\x2C\x29\xE1\x58\x57\xEC\x5A\x7C\x99\x6B\x25\xA4\x90\x3A\x80\xF4\x20\x9D\x9A\xCE\xC7\x2D\xF9\xB2\x4B\x29\x95\x83\xE9\x35\x8D\xA7\x49\x48\xA7\x0F\x4C\x19\x91\xD0\xF5\xBF\x10\xE0\x71\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x74\x49\x99\xD1\xFF\xB4\x7A\x68\x45\x75\xC3\x7E\xB4\xDC\xCC\xCE\x39\x33\xDA\x08\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x23\x43\x53\x24\x62\x5C\x6D\xFD\x3E\xC2\xCF\x55\x00\x6C\xC5\x56\x88\xB9\x0E\xDD\x3A\xE2\x25\x0D\x95\x4A\x97\xCA\x80\x89\xEE\x2A\xCD\x65\xF8\xDB\x16\xE0\x09\x92\xE0\x18\xC7\x78\x98\xBB\xF3\xEC\x42\x52\xFB\xA9\xA4\x82\xD7\x4D\xD8\x8A\xFC\xE4\x4E\xFD\xAB\x90\xC4\x38\x75\x32\x84\x9F\xFF\xB3\xB0\x2B\x02\x33\x36\xC0\x10\x90\x6F\x1D\x9C\xAF\xE1\x69\x93\xEC\xA3\x45\x2F\x14\x9F\xF5\x4C\x2A\x65\x43\x72\x0C\xF7\xC3\xF8\x95\x8B\x14\xF3\x85\x20\x62\xDD\x54\x53\xDD\x2C\xDC\x18\x95\x69\x4F\x83\x47\x70\x40\x33\x58\x77\x12\x0C\xA2\xEB\x52\x31\x1E\x4C\xC9\xA8\xCE\xC5\xEF\xC3\xD1\xAD\xE0\x6B\x03\x00\x34\x26\xB4\x54\x21\x35\x97\x01\xDC\x5F\x1B\xF1\x7C\xE7\x55\xFA\x2D\x68\x77\x7B\xD3\x69\xCC\xD3\x0E\x6B\xBA\x4D\x76\x44\xD6\xC2\x15\x9A\x26\xEC\xB0\xC5\xF5\xBB\xD1\x7A\x74\xC2\x6C\xCD\xC5\xB5\x5E\xF6\x4C\xE6\x5B\x2D\x81\xDB\xB3\xB7\x3A\x97\x9E\xED\xCF\x46\xB2\x50\x3D\x84\x60\x99\x71\xB5\x33\xB5\x57\x45\xE6\x42\x47\x75\x6A\x0E\xB0\x08\x0C\xAE\xBD\xDE\xF7\xBB\x0F\x58\x3D\x8F\x03\x31\xE8\x3D\x82\x50\xCA\x2F\x5E\x0C\x5D\xB4\x97\xBE\x20\x34\x07\xF4\xC4\x12\xE1\xEE\xD7\xB0\xD9\x59\x2D\x69\xF7\x31\x04\xF4\xF2\xF9\xAB\xF9\x13\x31\xF8\x01\x77\x0E\x3D\x42\x23\x26\xCC\x9A\x72\x67\x51\x21\x7A\xCC\x3C\x85\xA8\xEA\x21\x6A\x3B\xDB\x5A\x3C\xA5\x34\x9E\x9A\xC0\x2C\xDF\x80\x9C\x29\xE0\xDF\x77\x94\xD1\xA2\x80\x42\xFF\x6A\x4C\x5B\x11\xD0\xF5\xCD\xA2\xBE\xAE\xCC\x51\x5C\xC3\xD5\x54\x7B\x0C\xAE\xD6\xB9\x06\x77\x80\xE2\xEF\x07\x1A\x68\xCC\x59\x51\xAD\x7E\x5C\x67\x6B\xB9\xDB\xE2\x07\x42\x5B\xB8\x01\x05\x58\x39\x4D\xE4\xBB\x98\xA3\xB1\x32\xEC\xD9\xA3\xD6\x6F\x94\x23\xFF\x3B\xB7\x29\x65\xE6\x07\xE9\xEF\xB6\x19\xEA\xE7\xC2\x38\x1D\x32\x88\x90\x3C\x13\x2B\x6E\xCC\xEF\xAB\x77\x06\x34\x77\x84\x4F\x72\xE4\x81\x84\xF9\xB9\x74\x34\xDE\x76\x4F\x92\x2A\x53\xB1\x25\x39\xDB\x3C\xFF\xE5\x3E\xA6\x0E\xE5\x6B\x9E\xFF\xDB\xEC\x2F\x74\x83\xDF\x8E\xB4\xB3\xA9\xDE\x14\x4D\xFF\x31\xA3\x45\x73\x24\xFA\x95\x29\xCC\x12\x97\x04\xA2\x38\xB6\x8D\xB0\xF0\x37\xFC\xC8\x21\x7F\x3F\xB3\x24\x1B\x3D\x8B\x6E\xCC\x4D\xB0\x16\x0D\x96\x1D\x83\x1F\x46\xC0\x9B\xBD\x43\x99\xE7\xC4\x96\x2E\xCE\x5F\xC9"),
   // "OISTE WISeKey Global Root GC CA"
   CertificateDer::from_slice(b"\x30\x82\x02\x69\x30\x82\x01\xEF\xA0\x03\x02\x01\x02\x02\x10\x21\x2A\x56\x0C\xAE\xDA\x0C\xAB\x40\x45\xBF\x2B\xA2\x2D\x3A\xEA\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x6D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x10\x30\x0E\x06\x03\x55\x04\x0A\x13\x07\x57\x49\x53\x65\x4B\x65\x79\x31\x22\x30\x20\x06\x03\x55\x04\x0B\x13\x19\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x20\x45\x6E\x64\x6F\x72\x73\x65\x64\x31\x28\x30\x26\x06\x03\x55\x04\x03\x13\x1F\x4F\x49\x53\x54\x45\x20\x57\x49\x53\x65\x4B\x65\x79\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x43\x20\x43\x41\x30\x1E\x17\x0D\x31\x37\x30\x35\x30\x39\x30\x39\x34\x38\x33\x34\x5A\x17\x0D\x34\x32\x30\x35\x30\x39\x30\x39\x35\x38\x33\x33\x5A\x30\x6D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x10\x30\x0E\x06\x03\x55\x04\x0A\x13\x07\x57\x49\x53\x65\x4B\x65\x79\x31\x22\x30\x20\x06\x03\x55\x04\x0B\x13\x19\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x20\x45\x6E\x64\x6F\x72\x73\x65\x64\x31\x28\x30\x26\x06\x03\x55\x04\x03\x13\x1F\x4F\x49\x53\x54\x45\x20\x57\x49\x53\x65\x4B\x65\x79\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x43\x20\x43\x41\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x4C\xE9\x50\xC0\xC6\x0F\x72\x18\xBC\xD8\xF1\xBA\xB3\x89\xE2\x79\x4A\xA3\x16\xA7\x6B\x54\x24\xDB\x51\xFF\xEA\xF4\x09\x24\xC3\x0B\x22\x9F\xCB\x6A\x27\x82\x81\x0D\xD2\xC0\xAF\x31\xE4\x74\x82\x6E\xCA\x25\xD9\x8C\x75\x9D\xF1\xDB\xD0\x9A\xA2\x4B\x21\x7E\x16\xA7\x63\x90\xD2\x39\xD4\xB1\x87\x78\x5F\x18\x96\x0F\x50\x1B\x35\x37\x0F\x6A\xC6\xDC\xD9\x13\x4D\xA4\x8E\x90\x37\xE6\xBD\x5B\x31\x91\xA3\x54\x30\x52\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x48\x87\x14\xAC\xE3\xC3\x9E\x90\x60\x3A\xD7\xCA\x89\xEE\xD3\xAD\x8C\xB4\x50\x66\x30\x10\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x15\x01\x04\x03\x02\x01\x00\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x26\xC7\x69\x5B\xDC\xD5\xE7\xB2\xE7\xC8\x0C\x8C\x8C\xC3\xDD\x79\x8C\x1B\x63\xD5\xC9\x52\x94\x4E\x4D\x82\x4A\x73\x1E\xB2\x80\x84\xA9\x25\xC0\x4C\x5A\x6D\x49\x29\x60\x78\x13\xE2\x7E\x48\xEB\x64\x02\x31\x00\xDB\x34\x20\x32\x08\xFF\x9A\x49\x02\xB6\x88\xDE\x14\xAF\x5D\x6C\x99\x71\x8D\x1A\x3F\x8B\xD7\xE0\xA2\x36\x86\x1C\x07\x82\x3A\x76\x53\xFD\xC2\xA2\xED\xEF\x7B\xB0\x80\x4F\x58\x0F\x4B\x53\x39\xBD"),
   // "SSL.com Root Certification Authority RSA"
   CertificateDer::from_slice(b"\x30\x82\x05\xDD\x30\x82\x03\xC5\xA0\x03\x02\x01\x02\x02\x08\x7B\x2C\x9B\xD3\x16\x80\x32\x99\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x7C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x0C\x28\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x52\x53\x41\x30\x1E\x17\x0D\x31\x36\x30\x32\x31\x32\x31\x37\x33\x39\x33\x39\x5A\x17\x0D\x34\x31\x30\x32\x31\x32\x31\x37\x33\x39\x33\x39\x5A\x30\x7C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0E\x30\x0C\x06\x03\x55\x04\x08\x0C\x05\x54\x65\x78\x61\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x48\x6F\x75\x73\x74\x6F\x6E\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x0C\x28\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x52\x53\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xF9\x0F\xDD\xA3\x2B\x7D\xCB\xD0\x2A\xFE\xEC\x67\x85\xA6\xE7\x2E\x1B\xBA\x77\xE1\xE3\xF5\xAF\xA4\xEC\xFA\x4A\x5D\x91\xC4\x57\x47\x6B\x18\x77\x6B\x76\xF2\xFD\x93\xE4\x3D\x0F\xC2\x16\x9E\x0B\x66\xC3\x56\x94\x9E\x17\x83\x85\xCE\x56\xEF\xF2\x16\xFD\x00\x62\xF5\x22\x09\x54\xE8\x65\x17\x4E\x41\xB9\xE0\x4F\x46\x97\xAA\x1B\xC8\xB8\x6E\x62\x5E\x69\xB1\x5F\xDB\x2A\x02\x7E\xFC\x6C\xCA\xF3\x41\xD8\xED\xD0\xE8\xFC\x3F\x61\x48\xED\xB0\x03\x14\x1D\x10\x0E\x4B\x19\xE0\xBB\x4E\xEC\x86\x65\xFF\x36\xF3\x5E\x67\x02\x0B\x9D\x86\x55\x61\xFD\x7A\x38\xED\xFE\xE2\x19\x00\xB7\x6F\xA1\x50\x62\x75\x74\x3C\xA0\xFA\xC8\x25\x92\xB4\x6E\x7A\x22\xC7\xF8\x1E\xA1\xE3\xB2\xDD\x91\x31\xAB\x2B\x1D\x04\xFF\xA5\x4A\x04\x37\xE9\x85\xA4\x33\x2B\xFD\xE2\xD6\x55\x34\x7C\x19\xA4\x4A\x68\xC7\xB2\xA8\xD3\xB7\xCA\xA1\x93\x88\xEB\xC1\x97\xBC\x8C\xF9\x1D\xD9\x22\x84\x24\x74\xC7\x04\x3D\x6A\xA9\x29\x93\xCC\xEB\xB8\x5B\xE1\xFE\x5F\x25\xAA\x34\x58\xC8\xC1\x23\x54\x9D\x1B\x98\x11\xC3\x38\x9C\x7E\x3D\x86\x6C\xA5\x0F\x40\x86\x7C\x02\xF4\x5C\x02\x4F\x28\xCB\xAE\x71\x9F\x0F\x3A\xC8\x33\xFE\x11\x25\x35\xEA\xFC\xBA\xC5\x60\x3D\xD9\x7C\x18\xD5\xB2\xA9\xD3\x75\x78\x03\x72\x22\xCA\x3A\xC3\x1F\xEF\x2C\xE5\x2E\xA9\xFA\x9E\x2C\xB6\x51\x46\xFD\xAF\x03\xD6\xEA\x60\x68\xEA\x85\x16\x36\x6B\x85\xE9\x1E\xC0\xB3\xDD\xC4\x24\xDC\x80\x2A\x81\x41\x6D\x94\x3E\xC8\xE0\xC9\x81\x41\x00\x9E\x5E\xBF\x7F\xC5\x08\x98\xA2\x18\x2C\x42\x40\xB3\xF9\x6F\x38\x27\x4B\x4E\x80\xF4\x3D\x81\x47\xE0\x88\x7C\xEA\x1C\xCE\xB5\x75\x5C\x51\x2E\x1C\x2B\x7F\x1A\x72\x28\xE7\x00\xB5\xD1\x74\xC6\xD7\xE4\x9F\xAD\x07\x93\xB6\x53\x35\x35\xFC\x37\xE4\xC3\xF6\x5D\x16\xBE\x21\x73\xDE\x92\x0A\xF8\xA0\x63\x6A\xBC\x96\x92\x6A\x3E\xF8\xBC\x65\x55\x9B\xDE\xF5\x0D\x89\x26\x04\xFC\x25\x1A\xA6\x25\x69\xCB\xC2\x6D\xCA\x7C\xE2\x59\x5F\x97\xAC\xEB\xEF\x2E\xC8\xBC\xD7\x1B\x59\x3C\x2B\xCC\xF2\x19\xC8\x93\x6B\x27\x63\x19\xCF\xFC\xE9\x26\xF8\xCA\x71\x9B\x7F\x93\xFE\x34\x67\x84\x4E\x99\xEB\xFC\xB3\x78\x09\x33\x70\xBA\x66\xA6\x76\xED\x1B\x73\xEB\x1A\xA5\x0D\xC4\x22\x13\x20\x94\x56\x0A\x4E\x2C\x6C\x4E\xB1\xFD\xCF\x9C\x09\xBA\xA2\x33\xED\x87\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xDD\x04\x09\x07\xA2\xF5\x7A\x7D\x52\x53\x12\x92\x95\xEE\x38\x80\x25\x0D\xA6\x59\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xDD\x04\x09\x07\xA2\xF5\x7A\x7D\x52\x53\x12\x92\x95\xEE\x38\x80\x25\x0D\xA6\x59\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x20\x18\x11\x94\x29\xFB\x26\x9D\x1C\x1E\x1E\x70\x61\xF1\x95\x72\x93\x71\x24\xAD\x68\x93\x58\x8E\x32\xAF\x1B\xB3\x70\x03\xFC\x25\x2B\x74\x85\x90\x3D\x78\x6A\xF4\xB9\x8B\xA5\x97\x3B\xB5\x18\x91\xBB\x1E\xA7\xF9\x40\x5B\x91\xF9\x55\x99\xAF\x1E\x11\xD0\x5C\x1D\xA7\x66\xE3\xB1\x94\x07\x0C\x32\x39\xA6\xEA\x1B\xB0\x79\xD8\x1D\x9C\x70\x44\xE3\x8A\xDD\xC4\xF9\x95\x1F\x8A\x38\x43\x3F\x01\x85\xA5\x47\xA7\x3D\x46\xB2\xBC\xE5\x22\x68\xF7\x7B\x9C\xD8\x2C\x3E\x0A\x21\xC8\x2D\x33\xAC\xBF\xC5\x81\x99\x31\x74\xC1\x75\x71\xC5\xBE\xB1\xF0\x23\x45\xF4\x9D\x6B\xFC\x19\x63\x9D\xA3\xBC\x04\xC6\x18\x0B\x25\xBB\x53\x89\x0F\xB3\x80\x50\xDE\x45\xEE\x44\x7F\xAB\x94\x78\x64\x98\xD3\xF6\x28\xDD\x87\xD8\x70\x65\x74\xFB\x0E\xB9\x13\xEB\xA7\x0F\x61\xA9\x32\x96\xCC\xDE\xBB\xED\x63\x4C\x18\xBB\xA9\x40\xF7\xA0\x54\x6E\x20\x88\x71\x75\x18\xEA\x7A\xB4\x34\x72\xE0\x23\x27\x77\x5C\xB6\x90\xEA\x86\x25\x40\xAB\xEF\x33\x0F\xCB\x9F\x82\xBE\xA2\x20\xFB\xF6\xB5\x2D\x1A\xE6\xC2\x85\xB1\x74\x0F\xFB\xC8\x65\x02\xA4\x52\x01\x47\xDD\x49\x22\xC1\xBF\xD8\xEB\x6B\xAC\x7E\xDE\xEC\x63\x33\x15\xB7\x23\x08\x8F\xC6\x0F\x8D\x41\x5A\xDD\x8E\xC5\xB9\x8F\xE5\x45\x3F\x78\xDB\xBA\xD2\x1B\x40\xB1\xFE\x71\x4D\x3F\xE0\x81\xA2\xBA\x5E\xB4\xEC\x15\xE0\x93\xDD\x08\x1F\x7E\xE1\x55\x99\x0B\x21\xDE\x93\x9E\x0A\xFB\xE6\xA3\x49\xBD\x36\x30\xFE\xE7\x77\xB2\xA0\x75\x97\xB5\x2D\x81\x88\x17\x65\x20\xF7\xDA\x90\x00\x9F\xC9\x52\xCC\x32\xCA\x35\x7C\xF5\x3D\x0F\xD8\x2B\xD7\xF5\x26\x6C\xC9\x06\x34\x96\x16\xEA\x70\x59\x1A\x32\x79\x79\x0B\xB6\x88\x7F\x0F\x52\x48\x3D\xBF\x6C\xD8\xA2\x44\x2E\xD1\x4E\xB7\x72\x58\xD3\x89\x13\x95\xFE\x44\xAB\xF8\xD7\x8B\x1B\x6E\x9C\xBC\x2C\xA0\x5B\xD5\x6A\x00\xAF\x5F\x37\xE1\xD5\xFA\x10\x0B\x98\x9C\x86\xE7\x26\x8F\xCE\xF0\xEC\x6E\x8A\x57\x0B\x80\xE3\x4E\xB2\xC0\xA0\x63\x61\x90\xBA\x55\x68\x37\x74\x6A\xB6\x92\xDB\x9F\xA1\x86\x22\xB6\x65\x27\x0E\xEC\xB6\x9F\x42\x60\xE4\x67\xC2\xB5\xDA\x41\x0B\xC4\xD3\x8B\x61\x1B\xBC\xFA\x1F\x91\x2B\xD7\x44\x07\x5E\xBA\x29\xAC\xD9\xC5\xE9\xEF\x53\x48\x5A\xEB\x80\xF1\x28\x58\x21\xCD\xB0\x06\x55\xFB\x27\x3F\x53\x90\x70\xA9\x04\x1E\x57\x27\xB9"),
   // "QuoVadis Root CA 2"
   CertificateDer::from_slice(b"\x30\x82\x05\xB7\x30\x82\x03\x9F\xA0\x03\x02\x01\x02\x02\x02\x05\x09\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x1E\x17\x0D\x30\x36\x31\x31\x32\x34\x31\x38\x32\x37\x30\x30\x5A\x17\x0D\x33\x31\x31\x31\x32\x34\x31\x38\x32\x33\x33\x33\x5A\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x9A\x18\xCA\x4B\x94\x0D\x00\x2D\xAF\x03\x29\x8A\xF0\x0F\x81\xC8\xAE\x4C\x19\x85\x1D\x08\x9F\xAB\x29\x44\x85\xF3\x2F\x81\xAD\x32\x1E\x90\x46\xBF\xA3\x86\x26\x1A\x1E\xFE\x7E\x1C\x18\x3A\x5C\x9C\x60\x17\x2A\x3A\x74\x83\x33\x30\x7D\x61\x54\x11\xCB\xED\xAB\xE0\xE6\xD2\xA2\x7E\xF5\x6B\x6F\x18\xB7\x0A\x0B\x2D\xFD\xE9\x3E\xEF\x0A\xC6\xB3\x10\xE9\xDC\xC2\x46\x17\xF8\x5D\xFD\xA4\xDA\xFF\x9E\x49\x5A\x9C\xE6\x33\xE6\x24\x96\xF7\x3F\xBA\x5B\x2B\x1C\x7A\x35\xC2\xD6\x67\xFE\xAB\x66\x50\x8B\x6D\x28\x60\x2B\xEF\xD7\x60\xC3\xC7\x93\xBC\x8D\x36\x91\xF3\x7F\xF8\xDB\x11\x13\xC4\x9C\x77\x76\xC1\xAE\xB7\x02\x6A\x81\x7A\xA9\x45\x83\xE2\x05\xE6\xB9\x56\xC1\x94\x37\x8F\x48\x71\x63\x22\xEC\x17\x65\x07\x95\x8A\x4B\xDF\x8F\xC6\x5A\x0A\xE5\xB0\xE3\x5F\x5E\x6B\x11\xAB\x0C\xF9\x85\xEB\x44\xE9\xF8\x04\x73\xF2\xE9\xFE\x5C\x98\x8C\xF5\x73\xAF\x6B\xB4\x7E\xCD\xD4\x5C\x02\x2B\x4C\x39\xE1\xB2\x95\x95\x2D\x42\x87\xD7\xD5\xB3\x90\x43\xB7\x6C\x13\xF1\xDE\xDD\xF6\xC4\xF8\x89\x3F\xD1\x75\xF5\x92\xC3\x91\xD5\x8A\x88\xD0\x90\xEC\xDC\x6D\xDE\x89\xC2\x65\x71\x96\x8B\x0D\x03\xFD\x9C\xBF\x5B\x16\xAC\x92\xDB\xEA\xFE\x79\x7C\xAD\xEB\xAF\xF7\x16\xCB\xDB\xCD\x25\x2B\xE5\x1F\xFB\x9A\x9F\xE2\x51\xCC\x3A\x53\x0C\x48\xE6\x0E\xBD\xC9\xB4\x76\x06\x52\xE6\x11\x13\x85\x72\x63\x03\x04\xE0\x04\x36\x2B\x20\x19\x02\xE8\x74\xA7\x1F\xB6\xC9\x56\x66\xF0\x75\x25\xDC\x67\xC1\x0E\x61\x60\x88\xB3\x3E\xD1\xA8\xFC\xA3\xDA\x1D\xB0\xD1\xB1\x23\x54\xDF\x44\x76\x6D\xED\x41\xD8\xC1\xB2\x22\xB6\x53\x1C\xDF\x35\x1D\xDC\xA1\x77\x2A\x31\xE4\x2D\xF5\xE5\xE5\xDB\xC8\xE0\xFF\xE5\x80\xD7\x0B\x63\xA0\xFF\x33\xA1\x0F\xBA\x2C\x15\x15\xEA\x97\xB3\xD2\xA2\xB5\xBE\xF2\x8C\x96\x1E\x1A\x8F\x1D\x6C\xA4\x61\x37\xB9\x86\x73\x33\xD7\x97\x96\x9E\x23\x7D\x82\xA4\x4C\x81\xE2\xA1\xD1\xBA\x67\x5F\x95\x07\xA3\x27\x11\xEE\x16\x10\x7B\xBC\x45\x4A\x4C\xB2\x04\xD2\xAB\xEF\xD5\xFD\x0C\x51\xCE\x50\x6A\x08\x31\xF9\x91\xDA\x0C\x8F\x64\x5C\x03\xC3\x3A\x8B\x20\x3F\x6E\x8D\x67\x3D\x3A\xD6\xFE\x7D\x5B\x88\xC9\x5E\xFB\xCC\x61\xDC\x8B\x33\x77\xD3\x44\x32\x35\x09\x62\x04\x92\x16\x10\xD8\x9E\x27\x47\xFB\x3B\x21\xE3\xF8\xEB\x1D\x5B\x02\x03\x01\x00\x01\xA3\x81\xB0\x30\x81\xAD\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0B\x06\x03\x55\x1D\x0F\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x1A\x84\x62\xBC\x48\x4C\x33\x25\x04\xD4\xEE\xD0\xF6\x03\xC4\x19\x46\xD1\x94\x6B\x30\x6E\x06\x03\x55\x1D\x23\x04\x67\x30\x65\x80\x14\x1A\x84\x62\xBC\x48\x4C\x33\x25\x04\xD4\xEE\xD0\xF6\x03\xC4\x19\x46\xD1\x94\x6B\xA1\x49\xA4\x47\x30\x45\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x13\x12\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x82\x02\x05\x09\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00\x3E\x0A\x16\x4D\x9F\x06\x5B\xA8\xAE\x71\x5D\x2F\x05\x2F\x67\xE6\x13\x45\x83\xC4\x36\xF6\xF3\xC0\x26\x0C\x0D\xB5\x47\x64\x5D\xF8\xB4\x72\xC9\x46\xA5\x03\x18\x27\x55\x89\x78\x7D\x76\xEA\x96\x34\x80\x17\x20\xDC\xE7\x83\xF8\x8D\xFC\x07\xB8\xDA\x5F\x4D\x2E\x67\xB2\x84\xFD\xD9\x44\xFC\x77\x50\x81\xE6\x7C\xB4\xC9\x0D\x0B\x72\x53\xF8\x76\x07\x07\x41\x47\x96\x0C\xFB\xE0\x82\x26\x93\x55\x8C\xFE\x22\x1F\x60\x65\x7C\x5F\xE7\x26\xB3\xF7\x32\x90\x98\x50\xD4\x37\x71\x55\xF6\x92\x21\x78\xF7\x95\x79\xFA\xF8\x2D\x26\x87\x66\x56\x30\x77\xA6\x37\x78\x33\x52\x10\x58\xAE\x3F\x61\x8E\xF2\x6A\xB1\xEF\x18\x7E\x4A\x59\x63\xCA\x8D\xA2\x56\xD5\xA7\x2F\xBC\x56\x1F\xCF\x39\xC1\xE2\xFB\x0A\xA8\x15\x2C\x7D\x4D\x7A\x63\xC6\x6C\x97\x44\x3C\xD2\x6F\xC3\x4A\x17\x0A\xF8\x90\xD2\x57\xA2\x19\x51\xA5\x2D\x97\x41\xDA\x07\x4F\xA9\x50\xDA\x90\x8D\x94\x46\xE1\x3E\xF0\x94\xFD\x10\x00\x38\xF5\x3B\xE8\x40\xE1\xB4\x6E\x56\x1A\x20\xCC\x6F\x58\x8D\xED\x2E\x45\x8F\xD6\xE9\x93\x3F\xE7\xB1\x2C\xDF\x3A\xD6\x22\x8C\xDC\x84\xBB\x22\x6F\xD0\xF8\xE4\xC6\x39\xE9\x04\x88\x3C\xC3\xBA\xEB\x55\x7A\x6D\x80\x99\x24\xF5\x6C\x01\xFB\xF8\x97\xB0\x94\x5B\xEB\xFD\xD2\x6F\xF1\x77\x68\x0D\x35\x64\x23\xAC\xB8\x55\xA1\x03\xD1\x4D\x42\x19\xDC\xF8\x75\x59\x56\xA3\xF9\xA8\x49\x79\xF8\xAF\x0E\xB9\x11\xA0\x7C\xB7\x6A\xED\x34\xD0\xB6\x26\x62\x38\x1A\x87\x0C\xF8\xE8\xFD\x2E\xD3\x90\x7F\x07\x91\x2A\x1D\xD6\x7E\x5C\x85\x83\x99\xB0\x38\x08\x3F\xE9\x5E\xF9\x35\x07\xE4\xC9\x62\x6E\x57\x7F\xA7\x50\x95\xF7\xBA\xC8\x9B\xE6\x8E\xA2\x01\xC5\xD6\x66\xBF\x79\x61\xF3\x3C\x1C\xE1\xB9\x82\x5C\x5D\xA0\xC3\xE9\xD8\x48\xBD\x19\xA2\x11\x14\x19\x6E\xB2\x86\x1B\x68\x3E\x48\x37\x1A\x88\xB7\x5D\x96\x5E\x9C\xC7\xEF\x27\x62\x08\xE2\x91\x19\x5C\xD2\xF1\x21\xDD\xBA\x17\x42\x82\x97\x71\x81\x53\x31\xA9\x9F\xF6\x7D\x62\xBF\x72\xE1\xA3\x93\x1D\xCC\x8A\x26\x5A\x09\x38\xD0\xCE\xD7\x0D\x80\x16\xB4\x78\xA5\x3A\x87\x4C\x8D\x8A\xA5\xD5\x46\x97\xF2\x2C\x10\xB9\xBC\x54\x22\xC0\x01\x50\x69\x43\x9E\xF4\xB2\xEF\x6D\xF8\xEC\xDA\xF1\xE3\xB1\xEF\xDF\x91\x8F\x54\x2A\x0B\x25\xC1\x26\x19\xC4\x52\x10\x05\x65\xD5\x82\x10\xEA\xC2\x31\xCD\x2E"),
   // "emSign ECC Root CA - G3"
   CertificateDer::from_slice(b"\x30\x82\x02\x4E\x30\x82\x01\xD3\xA0\x03\x02\x01\x02\x02\x0A\x3C\xF6\x07\xA9\x68\x70\x0E\xDA\x8B\x84\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x6B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x49\x4E\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x65\x4D\x75\x64\x68\x72\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x65\x6D\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x47\x33\x30\x1E\x17\x0D\x31\x38\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x17\x0D\x34\x33\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x30\x6B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x49\x4E\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x65\x4D\x75\x64\x68\x72\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x65\x6D\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x47\x33\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x23\xA5\x0C\xB8\x2D\x12\xF5\x28\xF3\xB1\xB2\xDD\xE2\x02\x12\x80\x9E\x39\x5F\x49\x4D\x9F\xC9\x25\x34\x59\x74\xEC\xBB\x06\x1C\xE7\xC0\x72\xAF\xE8\xAE\x2F\xE1\x41\x54\x87\x14\xA8\x4A\xB2\xE8\x7C\x82\xE6\x5B\x6A\xB5\xDC\xB3\x75\xCE\x8B\x06\xD0\x86\x23\xBF\x46\xD5\x8E\x0F\x3F\x04\xF4\xD7\x1C\x92\x7E\xF6\xA5\x63\xC2\xF5\x5F\x8E\x2E\x4F\xA1\x18\x19\x02\x2B\x32\x0A\x82\x64\x7D\x16\x93\xD1\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x7C\x5D\x02\x84\x13\xD4\xCC\x8A\x9B\x81\xCE\x17\x1C\x2E\x29\x1E\x9C\x48\x63\x42\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\xBE\xF3\x61\xCF\x02\x10\x1D\x64\x95\x07\xB8\x18\x6E\x88\x85\x05\x2F\x83\x08\x17\x90\xCA\x1F\x8A\x4C\xE8\x0D\x1B\x7A\xB1\xAD\xD5\x81\x09\x47\xEF\x3B\xAC\x08\x04\x7C\x5C\x99\xB1\xED\x47\x07\xD2\x02\x31\x00\x9D\xBA\x55\xFC\xA9\x4A\xE8\xED\xED\xE6\x76\x01\x42\x7B\xC8\xF8\x60\xD9\x8D\x51\x8B\x55\x3B\xFB\x8C\x7B\xEB\x65\x09\xC3\xF8\x96\xCD\x47\xA8\x82\xF2\x16\x55\x77\x24\x7E\x12\x10\x95\x04\x2C\xA3"),
   // "QuoVadis Root CA 3 G3"
   CertificateDer::from_slice(b"\x30\x82\x05\x60\x30\x82\x03\x48\xA0\x03\x02\x01\x02\x02\x14\x2E\xF5\x9B\x02\x28\xA7\xDB\x7A\xFF\xD5\xA3\xA9\xEE\xBD\x03\xA0\xCF\x12\x6A\x1D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x13\x15\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x20\x47\x33\x30\x1E\x17\x0D\x31\x32\x30\x31\x31\x32\x32\x30\x32\x36\x33\x32\x5A\x17\x0D\x34\x32\x30\x31\x31\x32\x32\x30\x32\x36\x33\x32\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x13\x15\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x33\x20\x47\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB3\xCB\x0E\x10\x67\x8E\xEA\x14\x97\xA7\x32\x2A\x0A\x56\x36\x7F\x68\x4C\xC7\xB3\x6F\x3A\x23\x14\x91\xFF\x19\x7F\xA5\xCA\xAC\xEE\xB3\x76\x9D\x7A\xE9\x8B\x1B\xAB\x6B\x31\xDB\xFA\x0B\x53\x4C\xAF\xC5\xA5\x1A\x79\x3C\x8A\x4C\xFF\xAC\xDF\x25\xDE\x4E\xD9\x82\x32\x0B\x44\xDE\xCA\xDB\x8C\xAC\xA3\x6E\x16\x83\x3B\xA6\x64\x4B\x32\x89\xFB\x16\x16\x38\x7E\xEB\x43\xE2\xD3\x74\x4A\xC2\x62\x0A\x73\x0A\xDD\x49\xB3\x57\xD2\xB0\x0A\x85\x9D\x71\x3C\xDE\xA3\xCB\xC0\x32\xF3\x01\x39\x20\x43\x1B\x35\xD1\x53\xB3\xB1\xEE\xC5\x93\x69\x82\x3E\x16\xB5\x28\x46\xA1\xDE\xEA\x89\x09\xED\x43\xB8\x05\x46\x8A\x86\xF5\x59\x47\xBE\x1B\x6F\x01\x21\x10\xB9\xFD\xA9\xD2\x28\xCA\x10\x39\x09\xCA\x13\x36\xCF\x9C\xAD\xAD\x40\x74\x79\x2B\x02\x3F\x34\xFF\xFA\x20\x69\x7D\xD3\xEE\x61\xF5\xBA\xB3\xE7\x30\xD0\x37\x23\x86\x72\x61\x45\x29\x48\x59\x68\x6F\x77\xA6\x2E\x81\xBE\x07\x4D\x6F\xAF\xCE\xC4\x45\x13\x91\x14\x70\x06\x8F\x1F\x9F\xF8\x87\x69\xB1\x0E\xEF\xC3\x89\x19\xEB\xEA\x1C\x61\xFC\x7A\x6C\x8A\xDC\xD6\x03\x0B\x9E\x26\xBA\x12\xDD\xD4\x54\x39\xAB\x26\xA3\x33\xEA\x75\x81\xDA\x2D\xCD\x0F\x4F\xE4\x03\xD1\xEF\x15\x97\x1B\x6B\x90\xC5\x02\x90\x93\x66\x02\x21\xB1\x47\xDE\x8B\x9A\x4A\x80\xB9\x55\x8F\xB5\xA2\x2F\xC0\xD6\x33\x67\xDA\x7E\xC4\xA7\xB4\x04\x44\xEB\x47\xFB\xE6\x58\xB9\xF7\x0C\xF0\x7B\x2B\xB1\xC0\x70\x29\xC3\x40\x62\x2D\x3B\x48\x69\xDC\x23\x3C\x48\xEB\x7B\x09\x79\xA9\x6D\xDA\xA8\x30\x98\xCF\x80\x72\x03\x88\xA6\x5B\x46\xAE\x72\x79\x7C\x08\x03\x21\x65\xAE\xB7\xE1\x1C\xA5\xB1\x2A\xA2\x31\xDE\x66\x04\xF7\xC0\x74\xE8\x71\xDE\xFF\x3D\x59\xCC\x96\x26\x12\x8B\x85\x95\x57\x1A\xAB\x6B\x75\x0B\x44\x3D\x11\x28\x3C\x7B\x61\xB7\xE2\x8F\x67\x4F\xE5\xEC\x3C\x4C\x60\x80\x69\x57\x38\x1E\x01\x5B\x8D\x55\xE8\xC7\xDF\xC0\xCC\x77\x23\x34\x49\x75\x7C\xF6\x98\x11\xEB\x2D\xDE\xED\x41\x2E\x14\x05\x02\x7F\xE0\xFE\x20\xEB\x35\xE7\x11\xAC\x22\xCE\x57\x3D\xDE\xC9\x30\x6D\x10\x03\x85\xCD\xF1\xFF\x8C\x16\xB5\xC1\xB2\x3E\x88\x6C\x60\x7F\x90\x4F\x95\xF7\xF6\x2D\xAD\x01\x39\x07\x04\xFA\x75\x80\x7D\xBF\x49\x50\xED\xEF\xC9\xC4\x7C\x1C\xEB\x80\x7E\xDB\xB6\xD0\xDD\x13\xFE\xC9\xD3\x9C\xD7\xB2\x97\xA9\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC6\x17\xD0\xBC\xA8\xEA\x02\x43\xF2\x1B\x06\x99\x5D\x2B\x90\x20\xB9\xD7\x9C\xE4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x34\x61\xD9\x56\xB5\x12\x87\x55\x4D\xDD\xA3\x35\x31\x46\xBB\xA4\x07\x72\xBC\x5F\x61\x62\xE8\xA5\xFB\x0B\x37\xB1\x3C\xB6\xB3\xFA\x29\x9D\x7F\x02\xF5\xA4\xC9\xA8\x93\xB7\x7A\x71\x28\x69\x8F\x73\xE1\x52\x90\xDA\xD5\xBE\x3A\xE5\xB7\x76\x6A\x56\x80\x21\xDF\x5D\xE6\xE9\x3A\x9E\xE5\x3E\xF6\xA2\x69\xC7\x2A\x0A\xB0\x18\x47\xDC\x20\x70\x7D\x52\xA3\x3E\x59\x7C\xC1\xBA\xC9\xC8\x15\x40\x61\xCA\x72\xD6\x70\xAC\xD2\xB7\xF0\x1C\xE4\x86\x29\xF0\xCE\xEF\x68\x63\xD0\xB5\x20\x8A\x15\x61\x9A\x7E\x86\x98\xB4\xC9\xC2\x76\xFB\xCC\xBA\x30\x16\xCC\xA3\x61\xC6\x74\x13\xE5\x6B\xEF\xA3\x15\xEA\x03\xFE\x13\x8B\x64\xE4\xD3\xC1\xD2\xE8\x84\xFB\x49\xD1\x10\x4D\x79\x66\xEB\xAA\xFD\xF4\x8D\x31\x1E\x70\x14\xAD\xDC\xDE\x67\x13\x4C\x81\x15\x61\xBC\xB7\xD9\x91\x77\x71\x19\x81\x60\xBB\xF0\x58\xA5\xB5\x9C\x0B\xF7\x8F\x22\x55\x27\xC0\x4B\x01\x6D\x3B\x99\x0D\xD4\x1D\x9B\x63\x67\x2F\xD0\xEE\x0D\xCA\x66\xBC\x94\x4F\xA6\xAD\xED\xFC\xEE\x63\xAC\x57\x3F\x65\x25\xCF\xB2\x86\x8F\xD0\x08\xFF\xB8\x76\x14\x6E\xDE\xE5\x27\xEC\xAB\x78\xB5\x53\xB9\xB6\x3F\xE8\x20\xF9\xD2\xA8\xBE\x61\x46\xCA\x87\x8C\x84\xF3\xF9\xF1\xA0\x68\x9B\x22\x1E\x81\x26\x9B\x10\x04\x91\x71\xC0\x06\x1F\xDC\xA0\xD3\xB9\x56\xA7\xE3\x98\x2D\x7F\x83\x9D\xDF\x8C\x2B\x9C\x32\x8E\x32\x94\xF0\x01\x3C\x22\x2A\x9F\x43\xC2\x2E\xC3\x98\x39\x07\x38\x7B\xFC\x5E\x00\x42\x1F\xF3\x32\x26\x79\x83\x84\xF6\xE5\xF0\xC1\x51\x12\xC0\x0B\x1E\x04\x23\x0C\x54\xA5\x4C\x2F\x49\xC5\x4A\xD1\xB6\x6E\x60\x0D\x6B\xFC\x6B\x8B\x85\x24\x64\xB7\x89\x0E\xAB\x25\x47\x5B\x3C\xCF\x7E\x49\xBD\xC7\xE9\x0A\xC6\xDA\xF7\x7E\x0E\x17\x08\xD3\x48\x97\xD0\x71\x92\xF0\x0F\x39\x3E\x34\x6A\x1C\x7D\xD8\xF2\x22\xAE\xBB\x69\xF4\x33\xB4\xA6\x48\x55\xD1\x0F\x0E\x26\xE8\xEC\xB6\x0B\x2D\xA7\x85\x35\xCD\xFD\x59\xC8\x9F\xD1\xCD\x3E\x5A\x29\x34\xB9\x3D\x84\xCE\xB1\x65\xD4\x59\x91\x91\x56\x75\x21\xC1\x77\x9E\xF9\x7A\xE1\x60\x9D\xD3\xAD\x04\x18\xF4\x7C\xEB\x5E\x93\x8F\x53\x4A\x22\x29\xF8\x48\x2B\x3E\x4D\x86\xAC\x5B\x7F\xCB\x06\x99\x59\x60\xD8\x58\x65\x95\x8D\x44\xD1\xF7\x7F\x7E\x27\x7F\x7D\xAE\x80\xF5\x07\x4C\xB6\x3E\x9C\x71\x54\x99\x04\x4B\xFD\x58\xF9\x98\xF4"),
   // "NAVER Global Root Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x05\xA2\x30\x82\x03\x8A\xA0\x03\x02\x01\x02\x02\x14\x01\x94\x30\x1E\xA2\x0B\xDD\xF5\xC5\x33\x2A\xB1\x43\x44\x71\xF8\xD6\x50\x4D\x0D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x69\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4B\x52\x31\x26\x30\x24\x06\x03\x55\x04\x0A\x0C\x1D\x4E\x41\x56\x45\x52\x20\x42\x55\x53\x49\x4E\x45\x53\x53\x20\x50\x4C\x41\x54\x46\x4F\x52\x4D\x20\x43\x6F\x72\x70\x2E\x31\x32\x30\x30\x06\x03\x55\x04\x03\x0C\x29\x4E\x41\x56\x45\x52\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x37\x30\x38\x31\x38\x30\x38\x35\x38\x34\x32\x5A\x17\x0D\x33\x37\x30\x38\x31\x38\x32\x33\x35\x39\x35\x39\x5A\x30\x69\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4B\x52\x31\x26\x30\x24\x06\x03\x55\x04\x0A\x0C\x1D\x4E\x41\x56\x45\x52\x20\x42\x55\x53\x49\x4E\x45\x53\x53\x20\x50\x4C\x41\x54\x46\x4F\x52\x4D\x20\x43\x6F\x72\x70\x2E\x31\x32\x30\x30\x06\x03\x55\x04\x03\x0C\x29\x4E\x41\x56\x45\x52\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB6\xD4\xF1\x93\x5C\xB5\x40\x89\x0A\xAB\x0D\x90\x5B\x50\x63\xAE\x90\x94\x74\x17\x45\x72\xD6\x7B\x65\x5A\x29\x4B\xA7\x56\xA0\x4B\xB8\x2F\x42\x75\xE9\xD9\x7B\x24\x5A\x31\x65\xAB\x17\x17\xD1\x33\x3A\xD9\x11\xDC\x40\x36\x87\xDF\xC7\x6A\xE9\x26\x5E\x59\x8A\x77\xE3\xE8\x48\x9C\x31\x16\xFA\x3E\x91\xB1\xCA\xC9\xA3\xE2\x9F\xCE\x21\x53\xA3\x02\x36\x30\xCB\x52\x02\xE5\xDA\x32\x5D\xC3\xC5\xE6\xF9\xEE\x11\xC7\x8B\xC9\x44\x1E\x84\x93\x18\x4A\xB4\x9F\xE5\x12\x64\x69\xD0\x26\x85\x62\x01\xB6\xC9\x02\x1D\xBE\x83\x51\xBB\x5C\xDA\xF8\xAD\x15\x6A\x99\xF7\x92\x54\xF7\x34\x5B\xE9\xBF\xEA\x29\x81\x12\xD4\x53\x91\x96\xB3\x91\x5A\xDD\xFE\x90\x73\x28\xFB\x30\x46\xB5\xCA\x08\x07\xC7\x71\x72\xC9\x66\xD3\x34\x97\xF6\x8C\xF4\x18\x4A\xE1\xD0\x3D\x5A\x45\xB6\x69\xA7\x29\xFB\x23\xCE\x88\xD8\x12\x9C\x00\x48\xA8\xA6\x0F\xB3\x3B\x92\x8D\x71\x0E\x74\xC5\x8B\xC8\x4C\xF9\xF4\x9B\x8E\xB8\x3C\x69\xED\x6F\x3B\x50\x2F\x58\xED\xC4\xB0\xD0\x1C\x1B\x6A\x0C\xE2\xBC\x44\xAA\xD8\xCD\x14\x5D\x94\x78\x61\xBF\x0E\x6E\xDA\x2A\xBC\x2F\x0C\x0B\x71\xA6\xB3\x16\x3F\x9C\xE6\xF9\xCC\x9F\x53\x35\xE2\x03\xA0\xA0\x18\xBF\xBB\xF1\xBE\xF4\xD6\x8C\x87\x0D\x42\xF7\x06\xB9\xF1\x6D\xED\x04\x94\xA8\xFE\xB6\xD3\x06\xC6\x40\x61\xDF\x9D\x9D\xF3\x54\x76\xCE\x53\x3A\x01\xA6\x92\x41\xEC\x04\xA3\x8F\x0D\xA2\xD5\x09\xCA\xD6\xCB\x9A\xF1\xEF\x43\x5D\xC0\xAB\xA5\x41\xCF\x5C\x53\x70\x70\xC9\x88\xA6\x2D\xD4\x6B\x61\x73\x50\x26\x86\x61\x0E\x5F\x1B\xC2\x2B\xE2\x8C\xD5\xBB\x9D\xC1\x03\x42\xBA\x94\xDA\x5F\xA9\xB0\xCA\xCC\x4D\x0A\xEF\x47\x69\x03\x2F\x22\xFB\xF1\x28\xCE\xBF\x5D\x50\x65\xA8\x90\x6D\xB3\x74\xB0\x08\xC7\xAC\xA8\xD1\xEB\x3E\x9C\xFC\x5D\x1A\x83\x2E\x2B\xCB\xB5\xF3\x44\x9D\x3A\xA7\x17\x61\x96\xA2\x71\xD3\x70\x96\x15\x4D\xB7\x4C\x73\xEE\x19\x5C\xC5\x5B\x3E\x41\xFE\xAC\x75\x60\x3B\x1B\x63\xCE\x00\xDD\xDA\x08\x90\x62\xB4\xE5\x2D\xEE\x48\xA7\x6B\x17\x99\x54\xBE\x87\x4A\xE3\xA9\x5E\x04\x4C\xEB\x10\x6D\x54\xD6\xEF\xF1\xE8\xF2\x62\x16\xCB\x80\x6B\xED\x3D\xED\xF5\x1F\x30\xA5\xAE\x4B\xC9\x13\xED\x8A\x01\x01\xC9\xB8\x51\x58\xC0\x66\x3A\xB1\x66\x4B\xC4\xD5\x31\x02\x62\xE9\x74\x84\x0C\xDB\x4D\x46\x2D\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD2\x9F\x88\xDF\xA1\xCD\x2C\xBD\xEC\xF5\x3B\x01\x01\x93\x33\x27\xB2\xEB\x60\x4B\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x32\xCA\x80\xB3\x9D\x3D\x54\x06\xDD\xD2\xD2\x2E\xF0\xA4\x01\x21\x0B\x67\x48\xCA\x6D\x8E\xE0\xC8\xAA\x0D\xAA\x8D\x21\x57\x8F\xC6\x3E\x7A\xCA\xDB\x51\xD4\x52\xB3\xD4\x96\x84\xA5\x58\x60\x7F\xE5\x0B\x8E\x1F\xF5\xDC\x0A\x15\x81\xE5\x3B\xB6\xB7\x22\x2F\x09\x9C\x13\x16\xB1\x6C\x0C\x35\x08\x6D\xAB\x63\x72\xED\xDC\xBE\xEC\xC7\x57\xE6\x30\x20\x71\xD6\xD7\x10\xC1\x13\x55\x01\x8C\x2A\x43\xE4\x41\xF1\xCF\x3A\x7A\x53\x92\xCE\xA2\x03\x05\x0D\x38\xDF\x02\xBB\x10\x2E\xD9\x3B\xD2\x9B\x7A\xC0\xA1\xA6\xF8\xB5\x31\xE6\xF4\x75\xC9\xB9\x53\x99\x75\x47\x22\x5A\x14\x15\xC7\x78\x1B\xB6\x9D\xE9\x0C\xF8\x1B\x76\xF1\x85\x84\xDE\xA1\xDA\x12\xEF\xA4\xE2\x10\x97\x7A\x78\xDE\x0C\x51\x97\xA8\x21\x40\x8B\x86\xBD\x0D\xF0\x5E\x4E\x4B\x36\xBB\x3B\x20\x1F\x8A\x42\x56\xE1\x0B\x1A\xBF\x7B\xD0\x22\x43\x2C\x44\x8C\xFB\xE5\x2A\xB4\x6C\x1C\x1C\xBA\x94\xE0\x13\x7E\x21\xE6\x9A\xC2\xCB\xC5\x42\x64\xB4\x1E\x94\x7B\x08\x25\xC8\x71\xCC\x87\x45\x57\x85\xD3\x9F\x29\x62\x22\x83\x51\x97\x00\x18\x97\x77\x6A\x98\x92\xC9\x7C\x60\x6C\xDF\x6C\x7D\x4A\xE4\x70\x4C\xC2\x9E\xB8\x1D\xF7\xD0\x34\xC7\x0F\xCC\xFB\xA7\xFF\x03\xBE\xAD\x70\x90\xDA\x0B\xDD\xC8\x6D\x97\x5F\x9A\x7F\x09\x32\x41\xFD\xCD\xA2\xCC\x5A\x6D\x4C\xF2\xAA\x49\xFE\x66\xF8\xE9\xD8\x35\xEB\x0E\x28\x1E\xEE\x48\x2F\x3A\xD0\x79\x09\x38\x7C\xA6\x22\x82\x93\x95\xD0\x03\xBE\xBE\x02\xA0\x05\xDD\x20\x22\xE3\x6F\x1D\x88\x34\x60\xC6\xE6\x0A\xB9\x09\x75\x0B\xF0\x07\xE8\x69\x96\x35\xC7\xFB\x23\x81\x8E\x38\x39\xB8\x45\x2B\x43\x78\xA2\xD1\x2C\x14\xFF\x0D\x28\x72\x72\x95\x9B\x5E\x09\xDB\x89\x44\x98\xAA\xA1\x49\xBB\x71\x52\xF2\xBF\xF6\xFF\x27\xA1\x36\xAF\xB8\xB6\x77\x88\xDD\x3A\xA4\x6D\x9B\x34\x90\xDC\x14\x5D\x30\xBF\xB7\xEB\x17\xE4\x87\xB7\x71\xD0\xA1\xD7\x77\x15\xD4\x42\xD7\xF2\xF3\x31\x99\x5D\x9B\xDD\x16\x6D\x3F\xEA\x06\x23\xF8\x46\xA2\x22\xED\x93\xF6\xDD\x9A\xE6\x2A\x87\xB1\x98\x54\xF1\x22\xF7\x6B\x45\xE3\xE2\x8E\x76\x1D\x9A\x8D\xC4\x06\x8D\x36\xB7\x14\xF3\x9D\x54\x69\xB7\x8E\x3C\xD5\xA4\x6D\x93\x81\xB7\xAD\xF6\xBD\x64\x7B\xC2\xC9\x68\x39\xA0\x92\x9C\xCD\x34\x86\x91\x90\xFA\x64\x51\x9D\xFE\xFE\xEB\xA5\xF5\x75\xDE\x89\xF7\x72"),
   // "vTrus Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\x56\x30\x82\x03\x3E\xA0\x03\x02\x01\x02\x02\x14\x43\xE3\x71\x13\xD8\xB3\x59\x14\x5D\xB7\xCE\x8C\xFD\x35\xFD\x6F\xBC\x05\x8D\x45\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x43\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x1C\x30\x1A\x06\x03\x55\x04\x0A\x13\x13\x69\x54\x72\x75\x73\x43\x68\x69\x6E\x61\x20\x43\x6F\x2E\x2C\x4C\x74\x64\x2E\x31\x16\x30\x14\x06\x03\x55\x04\x03\x13\x0D\x76\x54\x72\x75\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x38\x30\x37\x33\x31\x30\x37\x32\x34\x30\x35\x5A\x17\x0D\x34\x33\x30\x37\x33\x31\x30\x37\x32\x34\x30\x35\x5A\x30\x43\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x1C\x30\x1A\x06\x03\x55\x04\x0A\x13\x13\x69\x54\x72\x75\x73\x43\x68\x69\x6E\x61\x20\x43\x6F\x2E\x2C\x4C\x74\x64\x2E\x31\x16\x30\x14\x06\x03\x55\x04\x03\x13\x0D\x76\x54\x72\x75\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xBD\x55\x7C\x61\xD3\xB8\x1D\x04\x62\x05\xA0\xAE\x6C\xB7\x70\xB4\x41\xEA\x4B\x03\x5E\x10\x3F\x90\x5A\x1C\x8B\x3B\xB0\x66\x8B\x6C\x48\xA6\x1C\x22\xBA\xD5\x40\x92\xEE\x33\xB2\x23\x59\xC9\x8E\xBC\x58\xDA\x8B\x9E\xD0\x19\xF2\x2F\x59\xC6\x8C\x63\x5A\xBA\x9F\xA3\x0B\xB0\xB3\x9A\x5C\xBA\x11\xB8\x12\xE9\x0C\xBB\xCF\x6E\x6C\x80\x87\x29\x14\x03\x2C\x8D\x24\x9A\xC8\x64\x83\xB5\x6A\xAC\x13\x2C\x33\xF1\x9F\xDC\x2C\x61\x3C\x1A\x3F\x70\x55\x9B\xAD\x00\x52\x7F\xCF\x04\xB9\xFE\x36\xFA\x9C\xC0\x16\xAE\x62\xFE\x96\x4C\x43\x7E\x55\x14\xBE\x1A\xB3\xD2\x6D\xC2\xAF\x76\x66\x95\x6B\x2A\xB0\x94\x77\x85\x5E\x04\x0F\x62\x1D\x63\x75\xF7\x6B\xE7\xCB\x5B\x9A\x70\xEC\x3E\x67\x05\xF0\xFE\x07\x08\x80\xCF\x28\xDB\x05\xC6\x14\x27\x2F\x86\x7D\xF0\x27\xDE\xFF\xE6\x7E\x33\x48\xE7\x0B\x1E\x58\xD1\x27\x2B\x53\x0E\x57\x4A\x65\xD7\xFB\xA2\x80\x60\xFC\x4C\xBC\x35\x53\x01\x6A\x97\x72\x82\xAF\xF1\x1D\x70\xE8\x9C\xF5\xEF\x5E\xC2\x6C\xC7\x47\x7E\x5A\x94\x85\x26\x4D\x3B\xBA\xEB\x4C\xE8\xB0\x09\xC2\x65\xC2\x9D\x9D\x09\x9B\x4E\xB5\x97\x05\xAC\xF5\x06\xA0\xF7\x36\x05\x7E\xF4\x90\xB2\x6B\xC4\xB4\xF9\x64\xEA\xE9\x1A\x0A\xC8\x0D\xA8\xED\x27\xC9\xD4\xE7\xB3\xB9\xAB\x82\x22\x90\x27\x3D\x2A\xE8\x7C\x90\xEF\xBC\x4F\xFD\xE2\x0A\x24\xA7\xDE\x65\x24\xA4\x5D\xEA\xC0\x76\x30\xD3\x77\x50\xF8\x0D\x04\x9B\x94\x36\x01\x73\xCA\x06\x58\xA6\xD3\x3B\xDC\xFA\x04\x46\x13\x55\x8A\xC9\x44\x47\xB8\x51\x39\x1A\x2E\xE8\x34\xE2\x79\xCB\x59\x4A\x0A\x7F\xBC\xA6\xEF\x1F\x03\x67\x6A\x59\x2B\x25\x62\x93\xD9\x53\x19\x66\x3C\x27\x62\x29\x86\x4D\xA4\x6B\xEE\xFF\xD4\x4E\xBA\xD5\xB4\xE2\x8E\x48\x5A\x00\x19\x09\xF1\x05\xD9\xCE\x91\xB1\xF7\xEB\xE9\x39\x4F\xF6\x6F\x04\x43\x9A\x55\xF5\x3E\x05\x14\xBD\xBF\xB3\x59\xB4\xD8\x8E\x33\x84\xA3\x90\x52\xAA\xB3\x02\x95\x60\xF9\x0C\x4C\x68\xF9\xEE\xD5\x17\x0D\xF8\x71\x57\xB5\x25\xE4\x29\xEE\x65\x5D\xAF\xD1\xEE\x3C\x17\x0B\x5A\x43\xC5\xA5\x86\xEA\x24\x9E\xE2\x05\x07\xDC\x34\x42\x12\x91\xD6\x39\x74\xAE\x4C\x41\x82\xDB\xF2\xA6\x48\xD1\xB3\x9B\xF3\x33\xAA\xF3\xA6\xC0\xC5\x4E\xF5\xF4\x9D\x76\x63\xE6\x02\xC6\x22\x4B\xC1\x95\x3F\x50\x64\x2C\x54\xE5\xB6\xF0\x3C\x29\xCF\x57\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x54\x62\x70\x63\xF1\x75\x84\x43\x58\x8E\xD1\x16\x20\xB1\xC6\xAC\x1A\xBC\xF6\x89\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x29\xBA\x92\x49\xA7\xAD\xF0\xF1\x70\xC3\xE4\x97\xF0\x9F\xA9\x25\xD5\x6B\x9E\x34\xFE\xE6\x1A\x64\xF6\x3A\x6B\x52\xB2\x10\x78\x1A\x9F\x4C\xDA\x8A\xDA\xEC\x1C\x37\x52\xE0\x42\x4B\xFB\x6C\x76\xCA\x24\x0B\x39\x12\x15\x9D\x9F\x11\x2D\xFC\x79\x64\xDC\xE0\xE0\xF5\xDD\xE0\x57\xC9\xA5\xB2\x76\x70\x50\xA4\xFE\xB7\x0A\x70\xD5\xA0\x34\xF1\x75\xD7\x4D\x49\xBA\x11\xD1\xB3\xD8\xEC\x82\xFF\xEB\x0E\xC4\xBF\x64\x2D\x7D\x63\x6E\x17\x78\xEC\x5D\x7C\x88\xC8\xEB\x8E\x57\x76\xD9\x59\x04\xFA\xBC\x52\x1F\x45\xAC\xF0\x7A\x80\xEC\xEC\x6F\x76\xAE\x91\xDB\x10\x8E\x04\xDC\x92\xDF\xA0\xF6\xE6\xAE\x49\xD3\xC1\x6C\x12\x1B\xCC\x29\xAA\xF9\x08\xA5\xE2\x37\x14\xCA\xB1\xB8\x66\xEF\x1A\x82\xE4\xF0\xF8\xF1\xA7\x16\x69\xB7\xDB\xA9\x61\x3C\x9F\xF5\x31\xCB\xE4\x00\x46\xC2\x2F\x74\xB1\xB1\xD7\x81\xEE\xA8\x26\x95\xBC\x88\xAF\x4C\x35\x07\x2A\x02\xCA\x78\x14\x6D\x47\x2B\x40\x56\xE9\xCB\x2A\x60\xA1\x67\x03\xA0\xCE\x8C\xBC\xB0\x72\x67\xC4\x31\xCE\xDB\x34\xE5\x25\x03\x60\x25\x7B\x71\x98\xE4\xC0\x1B\x2B\x5F\x74\x42\xD2\x4B\xC5\x59\x08\x07\x87\xBE\xC5\xC3\x7F\xE7\x96\xD9\xE1\xDC\x28\x97\xD6\x8F\x05\xE3\xF5\x9B\x4E\xCA\x1D\x50\x47\x05\x53\xB0\xCA\x39\xE7\x85\xA0\x89\xC1\x05\x3B\x01\x37\xD3\x3F\x49\xE2\x77\xEB\x23\xC8\x88\x66\x3B\x3D\x39\x76\x21\x46\xF1\xEC\x5F\x23\xB8\xEB\xA2\x66\x75\x74\xC1\x40\xF7\xD8\x68\x9A\x93\xE2\x2D\xA9\x2E\xBD\x1C\xA3\x1E\xC8\x74\xC6\xA4\x2D\x7A\x20\xAB\x3B\xB8\xB0\x46\xFD\x6F\xDD\x5F\x52\x55\x75\x62\xF0\x97\xA0\x7C\xD7\x38\xFD\x25\xDF\xCD\xA0\x9B\x10\xCF\x8B\xB8\x38\x5E\x5E\xC5\xB4\xA6\x02\x36\xA1\x1E\x5F\x1C\xCF\xE2\x96\x9D\x29\xAA\xFD\x98\xAE\x52\xE1\xF3\x41\x52\xFB\xA9\x2E\x72\x96\x9F\x27\xE3\xAA\x73\x7D\xF8\x1A\x23\x66\x7B\x3B\xAB\x65\xB0\x32\x01\x4B\x15\x3E\x3D\xA2\x4F\x0C\x2B\x35\xA2\xC6\xD9\x67\x12\x35\x30\xCD\x76\x2E\x16\xB3\x99\x9E\x4D\x4F\x4E\x2D\x3B\x34\x43\xE1\x9A\x0E\x0D\xA4\x66\x97\xBA\xD2\x1C\x4A\x4C\x2C\x2A\x8B\x8B\x81\x4F\x71\x1A\xA9\xDD\x5C\x7B\x7B\x08\xC5\x00\x0D\x37\x40\xE3\x7C\x7B\x54\x5F\x2F\x85\x5F\x76\xF6\xF7\xA7\xB0\x1C\x57\x56\xC1\x72\xE8\xAD\xA2\xAF\x8D\x33\x49\xBA\x1F\x8A\xDC\xE6\x74\x7C\x60\x86\x6F\x87\x97\x7B"),
   // "QuoVadis Root CA 1 G3"
   CertificateDer::from_slice(b"\x30\x82\x05\x60\x30\x82\x03\x48\xA0\x03\x02\x01\x02\x02\x14\x78\x58\x5F\x2E\xAD\x2C\x19\x4B\xE3\x37\x07\x35\x34\x13\x28\xB5\x96\xD4\x65\x93\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x13\x15\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x20\x47\x33\x30\x1E\x17\x0D\x31\x32\x30\x31\x31\x32\x31\x37\x32\x37\x34\x34\x5A\x17\x0D\x34\x32\x30\x31\x31\x32\x31\x37\x32\x37\x34\x34\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x13\x15\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x20\x47\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA0\xBE\x50\x10\x8E\xE9\xF2\x6C\x40\xB4\x04\x9C\x85\xB9\x31\xCA\xDC\x2D\xE4\x11\xA9\x04\x3C\x1B\x55\xC1\xE7\x58\x30\x1D\x24\xB4\xC3\xEF\x85\xDE\x8C\x2C\xE1\xC1\x3D\xDF\x82\xE6\x4F\xAD\x47\x87\x6C\xEC\x5B\x49\xC1\x4A\xD5\xBB\x8F\xEC\x87\xAC\x7F\x82\x9A\x86\xEC\x3D\x03\x99\x52\x01\xD2\x35\x9E\xAC\xDA\xF0\x53\xC9\x66\x3C\xD4\xAC\x02\x01\xDA\x24\xD3\x3B\xA8\x02\x46\xAF\xA4\x1C\xE3\xF8\x73\x58\x76\xB7\xF6\x0E\x90\x0D\xB5\xF0\xCF\xCC\xFA\xF9\xC6\x4C\xE5\xC3\x86\x30\x0A\x8D\x17\x7E\x35\xEB\xC5\xDF\xBB\x0E\x9C\xC0\x8D\x87\xE3\x88\x38\x85\x67\xFA\x3E\xC7\xAB\xE0\x13\x9C\x05\x18\x98\xCF\x93\xF5\xB1\x92\xB4\xFC\x23\xD3\xCF\xD5\xC4\x27\x49\xE0\x9E\x3C\x9B\x08\xA3\x8B\x5D\x2A\x21\xE0\xFC\x39\xAA\x53\xDA\x7D\x7E\xCF\x1A\x09\x53\xBC\x5D\x05\x04\xCF\xA1\x4A\x8F\x8B\x76\x82\x0D\xA1\xF8\xD2\xC7\x14\x77\x5B\x90\x36\x07\x81\x9B\x3E\x06\xFA\x52\x5E\x63\xC5\xA6\x00\xFE\xA5\xE9\x52\x1B\x52\xB5\x92\x39\x72\x03\x09\x62\xBD\xB0\x60\x16\x6E\xA6\xDD\x25\xC2\x03\x66\xDD\xF3\x04\xD1\x40\xE2\x4E\x8B\x86\xF4\x6F\xE5\x83\xA0\x27\x84\x5E\x04\xC1\xF5\x90\xBD\x30\x3D\xC4\xEF\xA8\x69\xBC\x38\x9B\xA4\xA4\x96\xD1\x62\xDA\x69\xC0\x01\x96\xAE\xCB\xC4\x51\x34\xEA\x0C\xAA\xFF\x21\x8E\x59\x8F\x4A\x5C\xE4\x61\x9A\xA7\xD2\xE9\x2A\x78\x8D\x51\x3D\x3A\x15\xEE\xA2\x59\x8E\xA9\x5C\xDE\xC5\xF9\x90\x22\xE5\x88\x45\x71\xDD\x91\x99\x6C\x7A\x9F\x3D\x3D\x98\x7C\x5E\xF6\xBE\x16\x68\xA0\x5E\xAE\x0B\x23\xFC\x5A\x0F\xAA\x22\x76\x2D\xC9\xA1\x10\x1D\xE4\xD3\x44\x23\x90\x88\x9F\xC6\x2A\xE6\xD7\xF5\x9A\xB3\x58\x1E\x2F\x30\x89\x08\x1B\x54\xA2\xB5\x98\x23\xEC\x08\x77\x1C\x95\x5D\x61\xD1\xCB\x89\x9C\x5F\xA2\x4A\x91\x9A\xEF\x21\xAA\x49\x16\x08\xA8\xBD\x61\x28\x31\xC9\x74\xAD\x85\xF6\xD9\xC5\xB1\x8B\xD1\xE5\x10\x32\x4D\x5F\x8B\x20\x3A\x3C\x49\x1F\x33\x85\x59\x0D\xDB\xCB\x09\x75\x43\x69\x73\xFB\x6B\x71\x7D\xF0\xDF\xC4\x4C\x7D\xC6\xA3\x2E\xC8\x95\x79\xCB\x73\xA2\x8E\x4E\x4D\x24\xFB\x5E\xE4\x04\xBE\x72\x1B\xA6\x27\x2D\x49\x5A\x99\x7A\xD7\x5C\x09\x20\xB7\x7F\x94\xB9\x4F\xF1\x0D\x1C\x5E\x88\x42\x1B\x11\xB7\xE7\x91\xDB\x9E\x6C\xF4\x6A\xDF\x8C\x06\x98\x03\xAD\xCC\x28\xEF\xA5\x47\xF3\x53\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xA3\x97\xD6\xF3\x5E\xA2\x10\xE1\xAB\x45\x9F\x3C\x17\x64\x3C\xEE\x01\x70\x9C\xCC\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x18\xFA\x5B\x75\xFC\x3E\x7A\xC7\x5F\x77\xC7\xCA\xDF\xCF\x5F\xC3\x12\xC4\x40\x5D\xD4\x32\xAA\xB8\x6A\xD7\xD5\x15\x15\x46\x98\x23\xA5\xE6\x90\x5B\x18\x99\x4C\xE3\xAD\x42\xA3\x82\x31\x36\x88\xCD\xE9\xFB\xC4\x04\x96\x48\x8B\x01\xC7\x8D\x01\xCF\x5B\x33\x06\x96\x46\x66\x74\x1D\x4F\xED\xC1\xB6\xB9\xB4\x0D\x61\xCC\x63\x7E\xD7\x2E\x77\x8C\x96\x1C\x2A\x23\x68\x6B\x85\x57\x76\x70\x33\x13\xFE\xE1\x4F\xA6\x23\x77\x18\xFA\x1A\x8C\xE8\xBD\x65\xC9\xCF\x3F\xF4\xC9\x17\xDC\xEB\xC7\xBC\xC0\x04\x2E\x2D\x46\x2F\x69\x66\xC3\x1B\x8F\xFE\xEC\x3E\xD3\xCA\x94\xBF\x76\x0A\x25\x0D\xA9\x7B\x02\x1C\xA9\xD0\x3B\x5F\x0B\xC0\x81\x3A\x3D\x64\xE1\xBF\xA7\x2D\x4E\xBD\x4D\xC4\xD8\x29\xC6\x22\x18\xD0\xC5\xAC\x72\x02\x82\x3F\xAA\x3A\xA2\x3A\x22\x97\x31\xDD\x08\x63\xC3\x75\x14\xB9\x60\x28\x2D\x5B\x68\xE0\x16\xA9\x66\x82\x23\x51\xF5\xEB\x53\xD8\x31\x9B\x7B\xE9\xB7\x9D\x4B\xEB\x88\x16\xCF\xF9\x5D\x38\x8A\x49\x30\x8F\xED\xF1\xEB\x19\xF4\x77\x1A\x31\x18\x4D\x67\x54\x6C\x2F\x6F\x65\xF9\xDB\x3D\xEC\x21\xEC\x5E\xF4\xF4\x8B\xCA\x60\x65\x54\xD1\x71\x64\xF4\xF9\xA6\xA3\x81\x33\x36\x33\x71\xF0\xA4\x78\x5F\x4E\xAD\x83\x21\xDE\x34\x49\x8D\xE8\x59\xAC\x9D\xF2\x76\x5A\x36\xF2\x13\xF4\xAF\xE0\x09\xC7\x61\x2A\x6C\xF7\xE0\x9D\xAE\xBB\x86\x4A\x28\x6F\x2E\xEE\xB4\x79\xCD\x90\x33\xC3\xB3\x76\xFA\xF5\xF0\x6C\x9D\x01\x90\xFA\x9E\x90\xF6\x9C\x72\xCF\x47\xDA\xC3\x1F\xE4\x35\x20\x53\xF2\x54\xD1\xDF\x61\x83\xA6\x02\xE2\x25\x38\xDE\x85\x32\x2D\x5E\x73\x90\x52\x5D\x42\xC4\xCE\x3D\x4B\xE1\xF9\x19\x84\x1D\xD5\xA2\x50\xCC\x41\xFB\x41\x14\xC3\xBD\xD6\xC9\x5A\xA3\x63\x66\x02\x80\xBD\x05\x3A\x3B\x47\x9C\xEC\x00\x26\x4C\xF5\x88\x51\xBF\xA8\x23\x7F\x18\x07\xB0\x0B\xED\x8B\x26\xA1\x64\xD3\x61\x4A\xEB\x5C\x9F\xDE\xB3\xAF\x67\x03\xB3\x1F\xDD\x6D\x5D\x69\x68\x69\xAB\x5E\x3A\xEC\x7C\x69\xBC\xC7\x3B\x85\x4E\x9E\x15\xB9\xB4\x15\x4F\xC3\x95\x7A\x58\xD7\xC9\x6C\xE9\x6C\xB9\xF3\x29\x63\x5E\xB4\x2C\xF0\x2D\x3D\xED\x5A\x65\xE0\xA9\x5B\x40\xC2\x48\x99\x81\x6D\x9E\x1F\x06\x2A\x3C\x12\xB4\x8B\x0F\x9B\xA2\x24\xF0\xA6\x8D\xD6\x7A\xE0\x4B\xB6\x64\x96\x63\x95\x84\xC2\x4A\xCD\x1C\x2E\x24\x87\x33\x60\xE5\xC3"),
   // "GTS Root R2"
   CertificateDer::from_slice(b"\x30\x82\x05\x57\x30\x82\x03\x3F\xA0\x03\x02\x01\x02\x02\x0D\x02\x03\xE5\xAE\xC5\x8D\x04\x25\x1A\xAB\x11\x25\xAA\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x32\x30\x1E\x17\x0D\x31\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xCE\xDE\xFD\xA6\xFB\xEC\xEC\x14\x34\x3C\x07\x06\x5A\x6C\x59\xF7\x19\x35\xDD\xF7\xC1\x9D\x55\xAA\xD3\xCD\x3B\xA4\x93\x72\xEF\x0A\xFA\x6D\x9D\xF6\xF0\x85\x80\x5B\xA1\x48\x52\x9F\x39\xC5\xB7\xEE\x28\xAC\xEF\xCB\x76\x68\x14\xB9\xDF\xAD\x01\x6C\x99\x1F\xC4\x22\x1D\x9F\xFE\x72\x77\xE0\x2C\x5B\xAF\xE4\x04\xBF\x4F\x72\xA0\x1A\x34\x98\xE8\x39\x68\xEC\x95\x25\x7B\x76\xA1\xE6\x69\xB9\x85\x19\xBD\x89\x8C\xFE\xAD\xED\x36\xEA\x73\xBC\xFF\x83\xE2\xCB\x7D\xC1\xD2\xCE\x4A\xB3\x8D\x05\x9E\x8B\x49\x93\xDF\xC1\x5B\xD0\x6E\x5E\xF0\x2E\x30\x2E\x82\xFC\xFA\xBC\xB4\x17\x0A\x48\xE5\x88\x9B\xC5\x9B\x6B\xDE\xB0\xCA\xB4\x03\xF0\xDA\xF4\x90\xB8\x65\x64\xF7\x5C\x4C\xAD\xE8\x7E\x66\x5E\x99\xD7\xB8\xC2\x3E\xC8\xD0\x13\x9D\xAD\xEE\xE4\x45\x7B\x89\x55\xF7\x8A\x1F\x62\x52\x84\x12\xB3\xC2\x40\x97\xE3\x8A\x1F\x47\x91\xA6\x74\x5A\xD2\xF8\xB1\x63\x28\x10\xB8\xB3\x09\xB8\x56\x77\x40\xA2\x26\x98\x79\xC6\xFE\xDF\x25\xEE\x3E\xE5\xA0\x7F\xD4\x61\x0F\x51\x4B\x3C\x3F\x8C\xDA\xE1\x70\x74\xD8\xC2\x68\xA1\xF9\xC1\x0C\xE9\xA1\xE2\x7F\xBB\x55\x3C\x76\x06\xEE\x6A\x4E\xCC\x92\x88\x30\x4D\x9A\xBD\x4F\x0B\x48\x9A\x84\xB5\x98\xA3\xD5\xFB\x73\xC1\x57\x61\xDD\x28\x56\x75\x13\xAE\x87\x8E\xE7\x0C\x51\x09\x10\x75\x88\x4C\xBC\x8D\xF9\x7B\x3C\xD4\x22\x48\x1F\x2A\xDC\xEB\x6B\xBB\x44\xB1\xCB\x33\x71\x32\x46\xAF\xAD\x4A\xF1\x8C\xE8\x74\x3A\xAC\xE7\x1A\x22\x73\x80\xD2\x30\xF7\x25\x42\xC7\x22\x3B\x3B\x12\xAD\x96\x2E\xC6\xC3\x76\x07\xAA\x20\xB7\x35\x49\x57\xE9\x92\x49\xE8\x76\x16\x72\x31\x67\x2B\x96\x7E\x8A\xA3\xC7\x94\x56\x22\xBF\x6A\x4B\x7E\x01\x21\xB2\x23\x32\xDF\xE4\x9A\x44\x6D\x59\x5B\x5D\xF5\x00\xA0\x1C\x9B\xC6\x78\x97\x8D\x90\xFF\x9B\xC8\xAA\xB4\xAF\x11\x51\x39\x5E\xD9\xFB\x67\xAD\xD5\x5B\x11\x9D\x32\x9A\x1B\xBD\xD5\xBA\x5B\xA5\xC9\xCB\x25\x69\x53\x55\x27\x5C\xE0\xCA\x36\xCB\x88\x61\xFB\x1E\xB7\xD0\xCB\xEE\x16\xFB\xD3\xA6\x4C\xDE\x92\xA5\xD4\xE2\xDF\xF5\x06\x54\xDE\x2E\x9D\x4B\xB4\x93\x30\xAA\x81\xCE\xDD\x1A\xDC\x51\x73\x0D\x4F\x70\xE9\xE5\xB6\x16\x21\x19\x79\xB2\xE6\x89\x0B\x75\x64\xCA\xD5\xAB\xBC\x09\xC1\x18\xA1\xFF\xD4\x54\xA1\x85\x3C\xFD\x14\x24\x03\xB2\x87\xD3\xA4\xB7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xBB\xFF\xCA\x8E\x23\x9F\x4F\x99\xCA\xDB\xE2\x68\xA6\xA5\x15\x27\x17\x1E\xD9\x0E\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x1F\xCA\xCE\xDD\xC7\xBE\xA1\x9F\xD9\x27\x4C\x0B\xDC\x17\x98\x11\x6A\x88\xDE\x3D\xE6\x71\x56\x72\xB2\x9E\x1A\x4E\x9C\xD5\x2B\x98\x24\x5D\x9B\x6B\x7B\xB0\x33\x82\x09\xBD\xDF\x25\x46\xEA\x98\x9E\xB6\x1B\xFE\x83\x3C\xD2\x62\x61\xC1\x04\xED\xCE\xE0\xC5\xC9\xC8\x13\x13\x55\xE7\xA8\x63\xAD\x8C\x7B\x01\xFE\x77\x30\xE1\xCE\x68\x9B\x05\xF8\x12\xEE\x79\x31\xA0\x41\x45\x35\x28\x0A\x71\xA4\x24\x4F\x8C\xDC\x3C\x82\x07\x5F\x66\xDC\x7D\x10\xFE\x0C\x61\xB3\x05\x95\xEE\xE1\xAE\x81\x0F\xA8\xF8\xC7\x8F\x4D\xA8\x23\x02\x26\x6B\x1D\x83\x52\x55\xCE\xB5\x2F\x00\xCA\x80\x40\xE0\xE1\x74\xAC\x60\xF5\x87\x80\x9D\xAE\x36\x64\x91\x5D\xB0\x68\x18\xEA\x8A\x61\xC9\x77\xA8\x97\xC4\xC9\xC7\xA5\xFC\x55\x4B\xF3\xF0\x7F\xB9\x65\x3D\x27\x68\xD0\xCC\x6B\xFA\x53\x9D\xE1\x91\x1A\xC9\x5D\x1A\x96\x6D\x32\x87\xED\x03\x20\xC8\x02\xCE\x5A\xBE\xD9\xEA\xFD\xB2\x4D\xC4\x2F\x1B\xDF\x5F\x7A\xF5\xF8\x8B\xC6\xEE\x31\x3A\x25\x51\x55\x67\x8D\x64\x32\x7B\xE9\x9E\xC3\x82\xBA\x2A\x2D\xE9\x1E\xB4\xE0\x48\x06\xA2\xFC\x67\xAF\x1F\x22\x02\x73\xFB\x20\x0A\xAF\x9D\x54\x4B\xA1\xCD\xFF\x60\x47\xB0\x3F\x5D\xEF\x1B\x56\xBD\x97\x21\x96\x2D\x0A\xD1\x5E\x9D\x38\x02\x47\x6C\xB9\xF4\xF6\x23\x25\xB8\xA0\x6A\x9A\x2B\x77\x08\xFA\xC4\xB1\x28\x90\x26\x58\x08\x3C\xE2\x7E\xAA\xD7\x3D\x6F\xBA\x31\x88\x0A\x05\xEB\x27\xB5\xA1\x49\xEE\xA0\x45\x54\x7B\xE6\x27\x65\x99\x20\x21\xA8\xA3\xBC\xFB\x18\x96\xBB\x52\x6F\x0C\xED\x83\x51\x4C\xE9\x59\xE2\x20\x60\xC5\xC2\x65\x92\x82\x8C\xF3\x10\x1F\x0E\x8A\x97\xBE\x77\x82\x6D\x3F\x8F\x1D\x5D\xBC\x49\x27\xBD\xCC\x4F\x0F\xE1\xCE\x76\x86\x04\x23\xC5\xC0\x8C\x12\x5B\xFD\xDB\x84\xA0\x24\xF1\x48\xFF\x64\x7C\xD0\xBE\x5C\x16\xD1\xEF\x99\xAD\xC0\x1F\xFB\xCB\xAE\xBC\x38\x22\x06\x26\x64\xDA\xDA\x97\x0E\x3F\x28\x15\x44\xA8\x4F\x00\xCA\xF0\x9A\xCC\xCF\x74\x6A\xB4\x3E\x3C\xEB\x95\xEC\xB5\xD3\x5A\xD8\x81\x99\xE9\x43\x18\x37\xEB\xB3\xBB\xD1\x58\x62\x41\xF3\x66\xD2\x8F\xAA\x78\x95\x54\x20\xC3\x5A\x2E\x74\x2B\xD5\xD1\xBE\x18\x69\xC0\xAC\xD5\xA4\xCF\x39\xBA\x51\x84\x03\x65\xE9\x62\xC0\x62\xFE\xD8\x4D\x55\x96\xE2\xD0\x11\xFA\x48\x34\x11\xEC\x9E\xED\x05\x1D\xE4\xC8\xD6\x1D\x86\xCB"),
   // "D-TRUST EV Root CA 2 2023"
   CertificateDer::from_slice(b"\x30\x82\x05\xA9\x30\x82\x03\x91\xA0\x03\x02\x01\x02\x02\x10\x69\x26\x09\x7E\x80\x4B\x4C\xA0\xA7\x8C\x78\x62\x53\x5F\x5A\x6F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x20\x32\x30\x32\x33\x30\x1E\x17\x0D\x32\x33\x30\x35\x30\x39\x30\x39\x31\x30\x33\x33\x5A\x17\x0D\x33\x38\x30\x35\x30\x39\x30\x39\x31\x30\x33\x32\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x45\x56\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x20\x32\x30\x32\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD8\x8E\xA3\x89\x80\x0B\xB2\x57\x52\xDC\xA9\x53\x4C\x37\xB9\x7F\x63\x17\x13\xEF\xA7\x5B\x23\x5B\x69\x75\xB0\x99\x0A\x17\xC1\x8B\xC4\xDB\xA8\xE0\xCC\x31\xBA\xC2\xF2\xCD\x5D\xE9\xB7\xF8\x1D\xAF\x6A\xC4\x95\x87\xD7\x47\xC9\x95\xD8\x82\x04\x50\x3D\x81\x08\xFF\xE4\x3D\xB3\xB1\xD6\xC5\xB2\xFD\x88\x09\xDB\x9C\x84\xEC\x25\x17\x14\x87\x7F\x30\x78\x9B\x6A\x58\xC9\xB6\x73\x28\x3C\x34\xF7\x99\xF7\x7F\xD3\xA6\xF8\x1C\x45\x7C\xAD\x2C\x8C\x94\x3F\xD8\x67\x10\x53\x7E\x22\xCD\x4E\x25\x51\xF0\x25\x24\x35\x11\x5E\x10\xC6\xEC\x87\x66\x89\x81\x68\xBA\xCC\x2B\x9D\x47\x73\x1F\xBD\xCD\x91\xA4\x72\x6A\x9C\xA2\x1B\x18\xA0\x6F\xEC\x50\xF4\x7D\x40\xC2\xA8\x30\xCF\xBD\x73\xC8\x13\x2B\x10\x13\x1E\x8B\x9A\xA8\x3A\x94\x73\xD3\x18\x69\x0A\x4A\xFF\xC1\x01\x03\xFF\x79\x7F\xB5\x48\x7F\x7B\xEE\xE8\x29\x6F\x36\x4C\x95\x61\x86\xD8\xF9\xA2\x73\x8A\xEE\xAE\x2F\x96\xEE\x68\xCD\x3D\x4D\x28\x42\xF9\x45\x2B\x32\x1B\x46\x55\x16\x6A\xA6\x4B\x29\xF9\xBB\x95\x56\xBF\x46\x1D\xEC\x1D\x93\x1D\xC0\x65\xB2\x1F\xA1\x43\xAE\x56\x9E\xA0\xB1\x8F\x6B\x12\xB7\x60\x6D\x78\x0B\xCA\x8A\x5C\xED\x1E\x96\x0E\x83\xA6\x48\x95\x8D\x3B\xA3\x21\xC4\xAE\x58\xC6\x00\xB2\x84\xB4\x23\xA4\x96\x86\x35\xB8\xD8\x9E\xD8\xAC\x34\x49\x98\x63\x95\xC5\xCB\x6D\x48\x47\xE2\xF2\x2E\x18\x1E\xD0\x31\xAB\xDD\x74\xEC\xF9\xDC\x8C\xB8\x1C\x8E\x68\x23\xBA\xD0\xF3\x50\xDC\xCF\x65\x8F\x73\x3A\x32\xC7\x7C\xFE\xCA\x82\x22\x4F\xBE\x8E\x62\x47\x66\xE5\xCD\x87\xE2\xE8\xD5\x0F\x18\x9F\xE5\x04\x72\x4B\x46\x3C\x10\xF2\x44\xC2\x64\x56\x71\x4E\x75\xE8\x9C\xC9\x26\x74\xC5\x7D\x59\xD1\x0A\x5B\x0F\x6D\xFE\x9E\x75\x1C\x18\xC6\x1A\x3A\x7C\xD8\x0D\x04\xCC\xCD\xB7\x45\x65\x7A\xB1\x8F\xB8\xAE\x84\x48\x3E\xB3\x7A\x4D\xA8\x03\xE2\xE2\x7E\x01\x16\x59\x68\x18\x43\x33\xB0\xD2\xDC\xB0\x1A\x43\x35\xEE\xA5\xDA\xA9\x46\x5C\xAE\x86\x81\x41\x01\x4A\x74\x26\xEC\x9F\x06\xBF\xC2\x05\x37\x64\x75\x78\x29\x68\xFD\xC5\xF5\xEB\xFE\x47\xF9\xE4\x85\xB0\xE1\x7B\x31\x9D\xA6\x7F\x72\xA3\xB9\xC4\x2C\x2E\xCC\x99\x57\x0E\x21\x0C\x45\x01\x94\x65\xEB\x65\x09\xC6\x63\x22\x0B\x33\x49\x92\x48\x3C\xFC\xCD\xCE\xB0\x3E\x8E\x9E\x8B\xF8\xFE\x49\xC5\x35\x72\x47\x02\x03\x01\x00\x01\xA3\x81\x8E\x30\x81\x8B\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xAA\xFC\x91\x10\x1B\x87\x91\x5F\x16\xB9\xBF\x4F\x4B\x91\x5E\x00\x1C\xB1\x32\x80\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x49\x06\x03\x55\x1D\x1F\x04\x42\x30\x40\x30\x3E\xA0\x3C\xA0\x3A\x86\x38\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x63\x72\x6C\x2F\x64\x2D\x74\x72\x75\x73\x74\x5F\x65\x76\x5F\x72\x6F\x6F\x74\x5F\x63\x61\x5F\x32\x5F\x32\x30\x32\x33\x2E\x63\x72\x6C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x03\x82\x02\x01\x00\x93\xCB\xA5\x1F\x99\x11\xEC\x9A\x0D\x5F\x2C\x15\x93\xC6\x3F\xBE\x10\x8D\x78\x42\xF0\x6E\x90\x47\x47\x8E\xA3\x92\x32\x8D\x70\x8F\xF6\x5B\x8D\xBE\x89\xCE\x47\x01\x6A\x1B\x20\x20\x89\x5B\xC8\x82\x10\x6C\xE0\xE7\x99\xAA\x6B\xC6\x2A\xA0\x63\x35\x91\x6A\x85\x25\xAD\x17\x38\xA5\x9B\x7E\x50\xF2\x76\xEA\x85\x05\x2A\x27\x41\x2B\xB1\x81\xD1\xA2\xF6\x40\x75\xA9\x0E\xCB\xF1\x55\x48\xD8\xEC\xD1\xEC\xB3\xE8\xCE\x14\xA1\x35\xEC\xC2\x5E\x35\x1A\xAB\xA6\x16\x01\x06\x8E\xEA\xDC\x2F\xA3\x8A\xCA\x2C\x91\xEB\x52\x8E\x5F\x0C\x9B\x17\xCF\xCB\x73\x07\x19\xC4\x6A\xC2\x73\x54\xEF\x7C\x43\x52\x63\xC1\x11\xCA\xC2\x45\xB1\xF4\x3B\x53\xF5\x69\xAE\x3C\xE3\xA5\xDE\xAC\xE8\x54\xB7\xB2\x91\xFD\xAC\xA9\x1F\xF2\x87\xE4\x17\xC6\x49\xA8\x7C\xD8\x0A\x41\xF4\xF2\x3E\xE7\x77\x34\x04\x52\xDD\xE8\x81\xF2\x4D\x2F\x54\x45\x9D\x15\xE1\x4F\xCC\xE5\xDE\x34\x57\x10\xC9\x23\x72\x17\x70\x8D\x50\x70\x1F\x56\x6C\xCC\xB9\xFF\x3A\x5A\x4F\x63\x7A\xC3\x6E\x65\x07\x1D\x84\xA1\xFF\xA9\x0C\x63\x89\x6D\xB2\x40\x88\x39\xD7\x1F\x77\x68\xB5\xFC\x9C\xD5\xD6\x67\x69\x5B\xA8\x74\xDB\xFC\x89\xF6\x1B\x32\xF7\xA4\x24\xA6\x76\xB7\x47\x53\xEF\x8D\x49\x8F\xA9\xB6\x83\x5A\xA5\x96\x90\x45\x61\xF5\xDE\x03\x4F\x26\x0F\xA8\x8B\xF0\x03\x96\xB0\xAC\x15\xD0\x71\x5A\x6A\x7B\x94\xE6\x70\x93\xDA\xF1\x69\xE0\xB2\x62\x4D\x9E\x8F\xFF\x89\x9D\x9B\x5D\xCD\x45\xE9\x94\x02\x22\x8D\xE0\x35\x7F\xE8\xF1\x04\x79\x71\x6C\x54\x83\xF8\x33\xB9\x05\x32\x1B\x58\x55\x11\x4F\xD0\xE5\x27\x47\x71\xEC\xED\xDA\x67\xD6\x62\xA6\x4B\x4D\x0F\x69\xA2\xC9\xBC\xEC\x22\x4B\x94\xC7\x68\x94\x17\x7E\xE2\x8E\x28\x3E\xB6\xC6\xEA\xF5\x34\x6C\x9F\x37\x88\x07\x38\xDB\x86\x71\xFA\xCD\x95\x48\x43\x6E\xA3\x4F\x82\x87\xD7\x34\x98\x6E\x4B\x93\x79\x60\x75\x69\x0F\xF0\x1A\xD5\x53\xFA\x21\x0C\xC2\x3F\xE9\x3F\x1F\x18\x8C\x92\x5D\x78\xA7\x76\x67\x19\xBB\xB2\xEA\x7F\xE9\x70\x09\x56\x56\xA3\xB0\x0C\x0B\x2D\x36\x5E\xC5\xE9\xC4\xD5\x83\xCB\x86\x17\x97\x2C\x6C\x13\x6F\x87\x5A\xAF\x49\xA6\x1D\xDB\xCD\x38\x04\x2E\x5F\xE2\x4A\x35\x0E\x2D\x4B\xF8\xA2\x24\x04\x8D\xD8\xE1\x63\x5E\x02\x92\x34\xDA\x98\x61\x5C\x1C\x6F\x58\x76\x64\xB3\xFC\x02\xB8\xF5\x9D\x0A"),
   // "Amazon Root CA 1"
   CertificateDer::from_slice(b"\x30\x82\x03\x41\x30\x82\x02\x29\xA0\x03\x02\x01\x02\x02\x13\x06\x6C\x9F\xCF\x99\xBF\x8C\x0A\x39\xE2\xF0\x78\x8A\x43\xE6\x96\x36\x5B\xCA\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x30\x1E\x17\x0D\x31\x35\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x37\x30\x30\x30\x30\x30\x30\x5A\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xB2\x78\x80\x71\xCA\x78\xD5\xE3\x71\xAF\x47\x80\x50\x74\x7D\x6E\xD8\xD7\x88\x76\xF4\x99\x68\xF7\x58\x21\x60\xF9\x74\x84\x01\x2F\xAC\x02\x2D\x86\xD3\xA0\x43\x7A\x4E\xB2\xA4\xD0\x36\xBA\x01\xBE\x8D\xDB\x48\xC8\x07\x17\x36\x4C\xF4\xEE\x88\x23\xC7\x3E\xEB\x37\xF5\xB5\x19\xF8\x49\x68\xB0\xDE\xD7\xB9\x76\x38\x1D\x61\x9E\xA4\xFE\x82\x36\xA5\xE5\x4A\x56\xE4\x45\xE1\xF9\xFD\xB4\x16\xFA\x74\xDA\x9C\x9B\x35\x39\x2F\xFA\xB0\x20\x50\x06\x6C\x7A\xD0\x80\xB2\xA6\xF9\xAF\xEC\x47\x19\x8F\x50\x38\x07\xDC\xA2\x87\x39\x58\xF8\xBA\xD5\xA9\xF9\x48\x67\x30\x96\xEE\x94\x78\x5E\x6F\x89\xA3\x51\xC0\x30\x86\x66\xA1\x45\x66\xBA\x54\xEB\xA3\xC3\x91\xF9\x48\xDC\xFF\xD1\xE8\x30\x2D\x7D\x2D\x74\x70\x35\xD7\x88\x24\xF7\x9E\xC4\x59\x6E\xBB\x73\x87\x17\xF2\x32\x46\x28\xB8\x43\xFA\xB7\x1D\xAA\xCA\xB4\xF2\x9F\x24\x0E\x2D\x4B\xF7\x71\x5C\x5E\x69\xFF\xEA\x95\x02\xCB\x38\x8A\xAE\x50\x38\x6F\xDB\xFB\x2D\x62\x1B\xC5\xC7\x1E\x54\xE1\x77\xE0\x67\xC8\x0F\x9C\x87\x23\xD6\x3F\x40\x20\x7F\x20\x80\xC4\x80\x4C\x3E\x3B\x24\x26\x8E\x04\xAE\x6C\x9A\xC8\xAA\x0D\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x84\x18\xCC\x85\x34\xEC\xBC\x0C\x94\x94\x2E\x08\x59\x9C\xC7\xB2\x10\x4E\x0A\x08\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x98\xF2\x37\x5A\x41\x90\xA1\x1A\xC5\x76\x51\x28\x20\x36\x23\x0E\xAE\xE6\x28\xBB\xAA\xF8\x94\xAE\x48\xA4\x30\x7F\x1B\xFC\x24\x8D\x4B\xB4\xC8\xA1\x97\xF6\xB6\xF1\x7A\x70\xC8\x53\x93\xCC\x08\x28\xE3\x98\x25\xCF\x23\xA4\xF9\xDE\x21\xD3\x7C\x85\x09\xAD\x4E\x9A\x75\x3A\xC2\x0B\x6A\x89\x78\x76\x44\x47\x18\x65\x6C\x8D\x41\x8E\x3B\x7F\x9A\xCB\xF4\xB5\xA7\x50\xD7\x05\x2C\x37\xE8\x03\x4B\xAD\xE9\x61\xA0\x02\x6E\xF5\xF2\xF0\xC5\xB2\xED\x5B\xB7\xDC\xFA\x94\x5C\x77\x9E\x13\xA5\x7F\x52\xAD\x95\xF2\xF8\x93\x3B\xDE\x8B\x5C\x5B\xCA\x5A\x52\x5B\x60\xAF\x14\xF7\x4B\xEF\xA3\xFB\x9F\x40\x95\x6D\x31\x54\xFC\x42\xD3\xC7\x46\x1F\x23\xAD\xD9\x0F\x48\x70\x9A\xD9\x75\x78\x71\xD1\x72\x43\x34\x75\x6E\x57\x59\xC2\x02\x5C\x26\x60\x29\xCF\x23\x19\x16\x8E\x88\x43\xA5\xD4\xE4\xCB\x08\xFB\x23\x11\x43\xE8\x43\x29\x72\x62\xA1\xA9\x5D\x5E\x08\xD4\x90\xAE\xB8\xD8\xCE\x14\xC2\xD0\x55\xF2\x86\xF6\xC4\x93\x43\x77\x66\x61\xC0\xB9\xE8\x41\xD7\x97\x78\x60\x03\x6E\x4A\x72\xAE\xA5\xD1\x7D\xBA\x10\x9E\x86\x6C\x1B\x8A\xB9\x59\x33\xF8\xEB\xC4\x90\xBE\xF1\xB9"),
   // "SSL.com TLS RSA Root CA 2022"
   CertificateDer::from_slice(b"\x30\x82\x05\x89\x30\x82\x03\x71\xA0\x03\x02\x01\x02\x02\x10\x6F\xBE\xDA\xAD\x73\xBD\x08\x40\xE2\x8B\x4D\xBE\xD4\xF7\x5B\x91\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x32\x30\x1E\x17\x0D\x32\x32\x30\x38\x32\x35\x31\x36\x33\x34\x32\x32\x5A\x17\x0D\x34\x36\x30\x38\x31\x39\x31\x36\x33\x34\x32\x31\x5A\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD0\xA4\x09\x72\x4F\x40\x88\x12\x61\x3E\x35\x23\x9E\xEE\xF6\x74\xCF\x2F\x7B\x58\x3D\xCE\x3C\x0D\x10\x28\x90\x2F\x97\xF7\x8C\x48\xD8\xA0\xD8\x25\xB1\x4C\xB0\x11\x4C\x17\x73\x50\xD0\x22\x4A\x63\xBB\x81\xD3\x29\x6E\xD5\xB5\x09\x3E\x26\x18\x7F\xB2\x12\x7F\x93\x98\xB7\xAF\xF0\x36\xBF\xF2\xEE\x18\x9E\x9C\x3B\x52\xC5\x47\x19\x5D\x74\xF3\x64\x66\xD5\x5D\xC7\x68\xB4\xBF\x1B\x1C\x06\xA3\xBC\x8F\x40\x23\xB6\x1E\xC6\x84\xBD\x51\xC4\x1B\x39\xC1\x95\xD2\x29\xEC\x4B\xAE\x7B\x2D\xBF\x39\xFD\xB4\x62\xDE\x96\x7B\x41\xC6\x9C\xA0\xE0\x06\x72\xFB\xF0\x07\x97\x09\x39\x81\x74\xAF\xF7\x34\x59\x11\x57\x0A\xC2\x5B\xC1\x24\xF4\x31\x73\x30\x82\xC6\x9D\xBA\x02\xF7\x3E\x7C\x44\x5F\x83\x0D\xF3\xF1\xDD\x20\x69\x16\x09\x50\xE2\xD4\x55\xB6\xE0\x80\x72\x76\x6E\x4C\x47\xB7\x75\x55\x59\xB4\x53\x74\xD9\x94\xC6\x41\xAD\x58\x8A\x31\x66\x0F\x1E\xA2\x1B\x29\x40\x4E\x2F\xDF\x7B\xE6\x16\x2C\x2D\xFC\xBF\xEC\xF3\xB4\xFA\xBE\x18\xF6\x9B\x49\xD4\xEE\x05\x6E\xD9\x34\xF3\x9C\xF1\xEC\x01\x8B\xD1\x20\xC6\x0F\xA0\xB5\xBC\x17\x4E\x48\x7B\x51\xC2\xFC\xE9\x5C\x69\x37\x47\x66\xB3\x68\xF8\x15\x28\xF0\xB9\xD3\xA4\x15\xCC\x5A\x4F\xBA\x52\x70\xA3\x12\x45\xDD\xC6\xBA\x4E\xFB\xC2\xD0\xF7\xA8\x52\x27\x6D\x6E\x79\xB5\x8C\xFC\x7B\x8C\xC1\x16\x4C\xEE\x80\x7F\xBE\xF0\x76\xBE\x41\x53\x12\x33\xAE\x5A\x38\x42\xAB\xD7\x0F\x3E\x41\x8D\x76\x07\x32\xD5\xAB\x89\xF6\x4E\x67\xD9\xB1\x42\x75\x23\x6E\xF3\xCD\x42\xB2\xFC\x55\xF5\x53\x87\x17\x3B\xC0\x33\x58\xF1\x52\xD2\xF9\x80\xA4\xF0\xE8\xF0\x3B\x8B\x38\xCC\xA4\xC6\x90\x7F\x0F\x9C\xFD\x8B\xD1\xA3\xCF\xDA\x83\xA7\x69\xC9\x50\x36\xD5\x5C\x05\xD2\x0A\x41\x74\xDB\x63\x11\x37\xC1\xA5\xA0\x96\x4B\x1E\x8C\x16\x12\x77\xAE\x94\x34\x7B\x1E\x7F\xC2\x66\x00\xE4\xAA\x83\xEA\x8A\x90\xAD\xCE\x36\x44\x4D\xD1\x51\xE9\xBC\x1F\xF3\x6A\x05\xFD\xC0\x74\x1F\x25\x19\x40\x51\x6E\xEA\x82\x51\x40\xDF\x9B\xB9\x08\x2A\x06\x02\xD5\x23\x1C\x13\xD6\xE9\xDB\xDB\xC6\xB0\x7A\xCB\x7B\x27\x9B\xFB\xE0\xD5\x46\x24\xED\x10\x4B\x63\x4B\xA5\x05\x8F\xBA\xB8\x1D\x2B\xA6\xFA\x91\xE2\x92\x52\xBD\xEC\xEB\x67\x97\x6D\x9A\x2D\x9F\x81\x32\x05\x67\x32\xFB\x48\x08\x3F\xD9\x25\xB8\x04\x25\x2F\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xFB\x2E\x37\xEE\xE3\x84\x7A\x27\x2E\xCD\x19\x35\xB1\x33\x7C\xFF\xD4\x44\x42\xB9\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xFB\x2E\x37\xEE\xE3\x84\x7A\x27\x2E\xCD\x19\x35\xB1\x33\x7C\xFF\xD4\x44\x42\xB9\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x8D\x89\x6D\x84\x45\x18\xF1\x4F\xB3\xA0\xEF\x68\xA4\xC0\x1D\xAC\x30\xBC\x67\x66\xB0\x9A\xCD\xB6\xAB\x22\x19\x66\xD3\x3B\x41\xB5\x10\x9D\x10\xBA\x72\x6E\x29\x24\x20\x1C\x01\x99\x62\xD3\x96\xE0\xE2\xFB\x0C\x42\xD7\xE1\x5A\xC4\x96\x4D\x54\xCD\x8F\xCA\x43\x53\xFD\x2A\xB8\xEA\xF8\x65\xCA\x01\xC2\xAD\x60\x68\x06\x9F\x39\x1A\x51\xD9\xE0\x8D\x26\xF9\x0B\x4E\xA5\x53\x25\x7A\x23\xA4\x1C\xCE\x08\x1B\xDF\x47\x88\xB2\xAD\x3E\xE0\x27\x87\x8B\x49\x8C\x1F\xA9\x47\x58\x7B\x96\xF2\x88\x1D\x18\xAE\xB3\xD1\xA6\x0A\x94\xFA\xDB\xD3\xE5\x38\x0A\x6B\x79\x12\x33\xFB\x4A\x59\x37\x16\x40\x0E\xBB\xDE\xF5\x89\x0C\xF1\x6C\xD3\xF7\x51\x6B\x5E\x35\xF5\xDB\xC0\x26\xEA\x12\x73\x4E\xA9\x91\x90\xA6\x17\xC3\x6C\x2F\x38\xD4\xA3\x72\x94\x43\x2C\x62\xE1\x4E\x5C\x32\x3D\xBD\x4C\x7D\x19\x47\xA2\xC3\x49\xE7\x96\x3F\x8F\x9A\xD3\x3B\xE4\x11\xD8\x8B\x03\xDC\xF6\xB6\x60\x55\x18\xA6\x81\x51\xF3\xE1\xA8\x15\x6A\xEB\xE0\x0B\xF0\x14\x31\xD6\xB9\x8C\x45\x3A\xA8\x10\xD8\xF0\xB9\x27\xEB\xF7\xCB\x7A\xEF\x05\x72\x96\xB5\xC4\x8F\x96\x73\xC4\xE8\x56\x73\x9C\xBC\x69\x51\x63\xBC\xEF\x67\x1C\x43\x1A\x5F\x77\x19\x1F\x18\xF8\x1C\x25\x29\xF9\x49\x99\x29\xB6\x92\x3D\xA2\x83\x37\xB1\x20\x91\xA8\x9B\x30\xE9\x6A\x6C\xB4\x23\x93\x65\x04\xAB\x11\xF3\x0E\x1D\x53\x24\x49\x53\x1D\xA1\x3F\x9D\x48\x92\x11\xE2\x7D\x0D\x4F\xF5\xD7\xBD\xA2\x58\x3E\x78\x9D\x1E\x1F\x2B\xFE\x21\xBB\x1A\x13\xB6\xB1\x28\x64\xFD\xB0\x02\x00\xC7\x6C\x80\xA2\xBD\x16\x50\x20\x0F\x72\x81\x5F\xCC\x94\xFF\xBB\x99\xE6\xBA\x90\xCB\xEA\xF9\xC6\x0C\xC2\xAE\xC5\x19\xCE\x33\xA1\x6B\x5C\xBB\x7E\x7C\x34\x57\x17\xAD\xF0\x3F\xAE\xCD\xEA\xAF\x99\xEC\x2C\x54\x7E\x8C\xCE\x2E\x12\x56\x48\xEF\x17\x3B\x3F\x4A\x5E\x60\xD2\xDC\x74\x36\xBC\xA5\x43\x63\xCB\x0F\x5B\xA3\x02\x56\x09\x9E\x24\x2C\xE1\x86\x81\x8C\xFE\xAB\x17\x2C\xFA\xC8\xE2\x32\x1A\x3A\xFF\x85\x08\xC9\x83\x9F\xF2\x4A\x48\x10\x54\x77\x37\xED\xA2\xBC\x40\xBE\xE4\x10\x74\xF7\xE4\x5B\xBB\xB9\xF3\x89\xF9\x8F\x41\xD8\xC7\xE4\x50\x90\x35\x80\x3E\x1C\xB8\x4D\x90\xD3\xD4\xF7\xC3\xB0\xA1\x7E\x84\xCA\x77\x92\x31\x2C\xB8\x90\xB1\x82\x7A\x74\x4E\x9B\x13\x26\xB4\xD5\x50\x66\x54\x78\xAE\x60"),
   // "QuoVadis Root CA 2 G3"
   CertificateDer::from_slice(b"\x30\x82\x05\x60\x30\x82\x03\x48\xA0\x03\x02\x01\x02\x02\x14\x44\x57\x34\x24\x5B\x81\x89\x9B\x35\xF2\xCE\xB8\x2B\x3B\x5B\xA7\x26\xF0\x75\x28\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x13\x15\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x20\x47\x33\x30\x1E\x17\x0D\x31\x32\x30\x31\x31\x32\x31\x38\x35\x39\x33\x32\x5A\x17\x0D\x34\x32\x30\x31\x31\x32\x31\x38\x35\x39\x33\x32\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x13\x15\x51\x75\x6F\x56\x61\x64\x69\x73\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x20\x47\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA1\xAE\x25\xB2\x01\x18\xDC\x57\x88\x3F\x46\xEB\xF9\xAF\xE2\xEB\x23\x71\xE2\x9A\xD1\x61\x66\x21\x5F\xAA\xAF\x27\x51\xE5\x6E\x1B\x16\xD4\x2D\x7D\x50\xB0\x53\x77\xBD\x78\x3A\x60\xE2\x64\x02\x9B\x7C\x86\x9B\xD6\x1A\x8E\xAD\xFF\x1F\x15\x7F\xD5\x95\x1E\x12\xCB\xE6\x14\x84\x04\xC1\xDF\x36\xB3\x16\x9F\x8A\xE3\xC9\xDB\x98\x34\xCE\xD8\x33\x17\x28\x46\xFC\xA7\xC9\xF0\xD2\xB4\xD5\x4D\x09\x72\x49\xF9\xF2\x87\xE3\xA9\xDA\x7D\xA1\x7D\x6B\xB2\x3A\x25\xA9\x6D\x52\x44\xAC\xF8\xBE\x6E\xFB\xDC\xA6\x73\x91\x90\x61\xA6\x03\x14\x20\xF2\xE7\x87\xA3\x88\xAD\xAD\xA0\x8C\xFF\xA6\x0B\x25\x52\x25\xE7\x16\x01\xD5\xCB\xB8\x35\x81\x0C\xA3\x3B\xF0\xE1\xE1\xFC\x5A\x5D\xCE\x80\x71\x6D\xF8\x49\xAB\x3E\x3B\xBA\xB8\xD7\x80\x01\xFB\xA5\xEB\x5B\xB3\xC5\x5E\x60\x2A\x31\xA0\xAF\x37\xE8\x20\x3A\x9F\xA8\x32\x2C\x0C\xCC\x09\x1D\xD3\x9E\x8E\x5D\xBC\x4C\x98\xEE\xC5\x1A\x68\x7B\xEC\x53\xA6\xE9\x14\x35\xA3\xDF\xCD\x80\x9F\x0C\x48\xFB\x1C\xF4\xF1\xBF\x4A\xB8\xFA\xD5\x8C\x71\x4A\xC7\x1F\xAD\xFE\x41\x9A\xB3\x83\x5D\xF2\x84\x56\xEF\xA5\x57\x43\xCE\x29\xAD\x8C\xAB\x55\xBF\xC4\xFB\x5B\x01\xDD\x23\x21\xA1\x58\x00\x8E\xC3\xD0\x6A\x13\xED\x13\xE3\x12\x2B\x80\xDC\x67\xE6\x95\xB2\xCD\x1E\x22\x6E\x2A\xF8\x41\xD4\xF2\xCA\x14\x07\x8D\x8A\x55\x12\xC6\x69\xF5\xB8\x86\x68\x2F\x53\x5E\xB0\xD2\xAA\x21\xC1\x98\xE6\x30\xE3\x67\x55\xC7\x9B\x6E\xAC\x19\xA8\x55\xA6\x45\x06\xD0\x23\x3A\xDB\xEB\x65\x5D\x2A\x11\x11\xF0\x3B\x4F\xCA\x6D\xF4\x34\xC4\x71\xE4\xFF\x00\x5A\xF6\x5C\xAE\x23\x60\x85\x73\xF1\xE4\x10\xB1\x25\xAE\xD5\x92\xBB\x13\xC1\x0C\xE0\x39\xDA\xB4\x39\x57\xB5\xAB\x35\xAA\x72\x21\x3B\x83\x35\xE7\x31\xDF\x7A\x21\x6E\xB8\x32\x08\x7D\x1D\x32\x91\x15\x4A\x62\x72\xCF\xE3\x77\xA1\xBC\xD5\x11\x1B\x76\x01\x67\x08\xE0\x41\x0B\xC3\xEB\x15\x6E\xF8\xA4\x19\xD9\xA2\xAB\xAF\xE2\x27\x52\x56\x2B\x02\x8A\x2C\x14\x24\xF9\xBF\x42\x02\xBF\x26\xC8\xC6\x8F\xE0\x6E\x38\x7D\x53\x2D\xE5\xED\x98\xB3\x95\x63\x68\x7F\xF9\x35\xF4\xDF\x88\xC5\x60\x35\x92\xC0\x7C\x69\x1C\x61\x95\x16\xD0\xEB\xDE\x0B\xAF\x3E\x04\x10\x45\x65\x58\x50\x38\xAF\x48\xF2\x59\xB6\x16\xF2\x3C\x0D\x90\x02\xC6\x70\x2E\x01\xAD\x3C\x15\xD7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xED\xE7\x6F\x76\x5A\xBF\x60\xEC\x49\x5B\xC6\xA5\x77\xBB\x72\x16\x71\x9B\xC4\x3D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x91\xDF\x80\x3F\x43\x09\x7E\x71\xC2\xF7\xEB\xB3\x88\x8F\xE1\x51\xB2\xBC\x3D\x75\xF9\x28\x5D\xC8\xBC\x99\x9B\x7B\x5D\xAA\xE5\xCA\xE1\x0A\xF7\xE8\xB2\xD3\x9F\xDD\x67\x31\x7E\xBA\x01\xAA\xC7\x6A\x41\x3B\x90\xD4\x08\x5C\xB2\x60\x6A\x90\xF0\xC8\xCE\x03\x62\xF9\x8B\xED\xFB\x6E\x2A\xDC\x06\x4D\x3C\x29\x0F\x89\x16\x8A\x58\x4C\x48\x0F\xE8\x84\x61\xEA\x3C\x72\xA6\x77\xE4\x42\xAE\x88\xA3\x43\x58\x79\x7E\xAE\xCA\xA5\x53\x0D\xA9\x3D\x70\xBD\x20\x19\x61\xA4\x6C\x38\xFC\x43\x32\xE1\xC1\x47\xFF\xF8\xEC\xF1\x11\x22\x32\x96\x9C\xC2\xF6\x5B\x69\x96\x7B\x20\x0C\x43\x41\x9A\x5B\xF6\x59\x19\x88\xDE\x55\x88\x37\x51\x0B\x78\x5C\x0A\x1E\xA3\x42\xFD\xC7\x9D\x88\x0F\xC0\xF2\x78\x02\x24\x54\x93\xAF\x89\x87\x88\xC9\x4A\x80\x1D\xEA\xD0\x6E\x3E\x61\x2E\x36\xBB\x35\x0E\x27\x96\xFD\x66\x34\x3B\x61\x72\x73\xF1\x16\x5C\x47\x06\x54\x49\x00\x7A\x58\x12\xB0\x0A\xEF\x85\xFD\xB1\xB8\x33\x75\x6A\x93\x1C\x12\xE6\x60\x5E\x6F\x1D\x7F\xC9\x1F\x23\xCB\x84\x61\x9F\x1E\x82\x44\xF9\x5F\xAD\x62\x55\x24\x9A\x52\x98\xED\x51\xE7\xA1\x7E\x97\x3A\xE6\x2F\x1F\x11\xDA\x53\x80\x2C\x85\x9E\xAB\x35\x10\xDB\x22\x5F\x6A\xC5\x5E\x97\x53\xF2\x32\x02\x09\x30\xA3\x58\xF0\x0D\x01\xD5\x72\xC6\xB1\x7C\x69\x7B\xC3\xF5\x36\x45\xCC\x61\x6E\x5E\x4C\x94\xC5\x5E\xAE\xE8\x0E\x5E\x8B\xBF\xF7\xCD\xE0\xED\xA1\x0E\x1B\x33\xEE\x54\x18\xFE\x0F\xBE\xEF\x7E\x84\x6B\x43\xE3\x70\x98\xDB\x5D\x75\xB2\x0D\x59\x07\x85\x15\x23\x39\xD6\xF1\xDF\xA9\x26\x0F\xD6\x48\xC7\xB3\xA6\x22\xF5\x33\x37\x5A\x95\x47\x9F\x7B\xBA\x18\x15\x6F\xFF\xD6\x14\x64\x83\x49\xD2\x0A\x67\x21\xDB\x0F\x35\x63\x60\x28\x22\xE3\xB1\x95\x83\xCD\x85\xA6\xDD\x2F\x0F\xE7\x67\x52\x6E\xBB\x2F\x85\x7C\xF5\x4A\x73\xE7\xC5\x3E\xC0\xBD\x21\x12\x05\x3F\xFC\xB7\x03\x49\x02\x5B\xC8\x25\xE6\xE2\x54\x38\xF5\x79\x87\x8C\x1D\x53\xB2\x4E\x85\x7B\x06\x38\xC7\x2C\xF8\xF8\xB0\x72\x8D\x25\xE5\x77\x52\xF4\x03\x1C\x48\xA6\x50\x5F\x88\x20\x30\x6E\xF2\x82\x43\xAB\x3D\x97\x84\xE7\x53\xFB\x21\xC1\x4F\x0F\x22\x9A\x86\xB8\x59\x2A\xF6\x47\x3D\x19\x88\x2D\xE8\x85\xE1\x9E\xEC\x85\x08\x6A\xB1\x6C\x34\xC9\x1D\xEC\x48\x2B\x3B\x78\xED\x66\xC4\x8E\x79\x69\x83\xDE\x7F\x8C"),
   // "T-TeleSec GlobalRoot Class 2"
   CertificateDer::from_slice(b"\x30\x82\x03\xC3\x30\x82\x02\xAB\xA0\x03\x02\x01\x02\x02\x01\x01\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x2B\x30\x29\x06\x03\x55\x04\x0A\x0C\x22\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x45\x6E\x74\x65\x72\x70\x72\x69\x73\x65\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x47\x6D\x62\x48\x31\x1F\x30\x1D\x06\x03\x55\x04\x0B\x0C\x16\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x54\x72\x75\x73\x74\x20\x43\x65\x6E\x74\x65\x72\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x54\x2D\x54\x65\x6C\x65\x53\x65\x63\x20\x47\x6C\x6F\x62\x61\x6C\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x32\x30\x1E\x17\x0D\x30\x38\x31\x30\x30\x31\x31\x30\x34\x30\x31\x34\x5A\x17\x0D\x33\x33\x31\x30\x30\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x2B\x30\x29\x06\x03\x55\x04\x0A\x0C\x22\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x45\x6E\x74\x65\x72\x70\x72\x69\x73\x65\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x47\x6D\x62\x48\x31\x1F\x30\x1D\x06\x03\x55\x04\x0B\x0C\x16\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x54\x72\x75\x73\x74\x20\x43\x65\x6E\x74\x65\x72\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x54\x2D\x54\x65\x6C\x65\x53\x65\x63\x20\x47\x6C\x6F\x62\x61\x6C\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xAA\x5F\xDA\x1B\x5F\xE8\x73\x91\xE5\xDA\x5C\xF4\xA2\xE6\x47\xE5\xF3\x68\x55\x60\x05\x1D\x02\xA4\xB3\x9B\x59\xF3\x1E\x8A\xAF\x34\xAD\xFC\x0D\xC2\xD9\x48\x19\xEE\x69\x8F\xC9\x20\xFC\x21\xAA\x07\x19\xED\xB0\x5C\xAC\x65\xC7\x5F\xED\x02\x7C\x7B\x7C\x2D\x1B\xD6\xBA\xB9\x80\xC2\x18\x82\x16\x84\xFA\x66\xB0\x08\xC6\x54\x23\x81\xE4\xCD\xB9\x49\x3F\xF6\x4F\x6E\x37\x48\x28\x38\x0F\xC5\xBE\xE7\x68\x70\xFD\x39\x97\x4D\xD2\xC7\x98\x91\x50\xAA\xC4\x44\xB3\x23\x7D\x39\x47\xE9\x52\x62\xD6\x12\x93\x5E\xB7\x31\x96\x42\x05\xFB\x76\xA7\x1E\xA3\xF5\xC2\xFC\xE9\x7A\xC5\x6C\xA9\x71\x4F\xEA\xCB\x78\xBC\x60\xAF\xC7\xDE\xF4\xD9\xCB\xBE\x7E\x33\xA5\x6E\x94\x83\xF0\x34\xFA\x21\xAB\xEA\x8E\x72\xA0\x3F\xA4\xDE\x30\x5B\xEF\x86\x4D\x6A\x95\x5B\x43\x44\xA8\x10\x15\x1C\xE5\x01\x57\xC5\x98\xF1\xE6\x06\x28\x91\xAA\x20\xC5\xB7\x53\x26\x51\x43\xB2\x0B\x11\x95\x58\xE1\xC0\x0F\x76\xD9\xC0\x8D\x7C\x81\xF3\x72\x70\x9E\x6F\xFE\x1A\x8E\xD9\x5F\x35\xC6\xB2\x6F\x34\x7C\xBE\x48\x4F\xE2\x5A\x39\xD7\xD8\x9D\x78\x9E\x9F\x86\x3E\x03\x5E\x19\x8B\x44\xA2\xD5\xC7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xBF\x59\x20\x36\x00\x79\xA0\xA0\x22\x6B\x8C\xD5\xF2\x61\xD2\xB8\x2C\xCB\x82\x4A\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x31\x03\xA2\x61\x0B\x1F\x74\xE8\x72\x36\xC6\x6D\xF9\x4D\x9E\xFA\x22\xA8\xE1\x81\x56\xCF\xCD\xBB\x9F\xEA\xAB\x91\x19\x38\xAF\xAA\x7C\x15\x4D\xF3\xB6\xA3\x8D\xA5\xF4\x8E\xF6\x44\xA9\xA7\xE8\x21\x95\xAD\x3E\x00\x62\x16\x88\xF0\x02\xBA\xFC\x61\x23\xE6\x33\x9B\x30\x7A\x6B\x36\x62\x7B\xAD\x04\x23\x84\x58\x65\xE2\xDB\x2B\x8A\xE7\x25\x53\x37\x62\x53\x5F\xBC\xDA\x01\x62\x29\xA2\xA6\x27\x71\xE6\x3A\x22\x7E\xC1\x6F\x1D\x95\x70\x20\x4A\x07\x34\xDF\xEA\xFF\x15\x80\xE5\xBA\xD7\x7A\xD8\x5B\x75\x7C\x05\x7A\x29\x47\x7E\x40\xA8\x31\x13\x77\xCD\x40\x3B\xB4\x51\x47\x7A\x2E\x11\xE3\x47\x11\xDE\x9D\x66\xD0\x8B\xD5\x54\x66\xFA\x83\x55\xEA\x7C\xC2\x29\x89\x1B\xE9\x6F\xB3\xCE\xE2\x05\x84\xC9\x2F\x3E\x78\x85\x62\x6E\xC9\x5F\xC1\x78\x63\x74\x58\xC0\x48\x18\x0C\x99\x39\xEB\xA4\xCC\x1A\xB5\x79\x5A\x8D\x15\x9C\xD8\x14\x0D\xF6\x7A\x07\x57\xC7\x22\x83\x05\x2D\x3C\x9B\x25\x26\x3D\x18\xB3\xA9\x43\x7C\xC8\xC8\xAB\x64\x8F\x0E\xA3\xBF\x9C\x1B\x9D\x30\xDB\xDA\xD0\x19\x2E\xAA\x3C\xF1\xFB\x33\x80\x76\xE4\xCD\xAD\x19\x4F\x05\x27\x8E\x13\xA1\x6E\xC2"),
   // "Trustwave Global ECC P256 Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x02\x60\x30\x82\x02\x07\xA0\x03\x02\x01\x02\x02\x0C\x0D\x6A\x5F\x08\x3F\x28\x5C\x3E\x51\x95\xDF\x5D\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x81\x91\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x08\x13\x08\x49\x6C\x6C\x69\x6E\x6F\x69\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x43\x68\x69\x63\x61\x67\x6F\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x48\x6F\x6C\x64\x69\x6E\x67\x73\x2C\x20\x49\x6E\x63\x2E\x31\x3A\x30\x38\x06\x03\x55\x04\x03\x13\x31\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x45\x43\x43\x20\x50\x32\x35\x36\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x37\x30\x38\x32\x33\x31\x39\x33\x35\x31\x30\x5A\x17\x0D\x34\x32\x30\x38\x32\x33\x31\x39\x33\x35\x31\x30\x5A\x30\x81\x91\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x08\x13\x08\x49\x6C\x6C\x69\x6E\x6F\x69\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x13\x07\x43\x68\x69\x63\x61\x67\x6F\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x48\x6F\x6C\x64\x69\x6E\x67\x73\x2C\x20\x49\x6E\x63\x2E\x31\x3A\x30\x38\x06\x03\x55\x04\x03\x13\x31\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x45\x43\x43\x20\x50\x32\x35\x36\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x59\x30\x13\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x08\x2A\x86\x48\xCE\x3D\x03\x01\x07\x03\x42\x00\x04\x7E\xFB\x6C\xE6\x23\xE3\x73\x32\x08\xCA\x60\xE6\x53\x9C\xBA\x74\x8D\x18\xB0\x78\x90\x52\x80\xDD\x38\xC0\x4A\x1D\xD1\xA8\xCC\x93\xA4\x97\x06\x38\xCA\x0D\x15\x62\xC6\x8E\x01\x2A\x65\x9D\xAA\xDF\x34\x91\x2E\x81\xC1\xE4\x33\x92\x31\xC4\xFD\x09\x3A\xA6\x3F\xAD\xA3\x43\x30\x41\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0F\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x05\x03\x03\x07\x06\x00\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xA3\x41\x06\xAC\x90\x6D\xD1\x4A\xEB\x75\xA5\x4A\x10\x99\xB3\xB1\xA1\x8B\x4A\xF7\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x47\x00\x30\x44\x02\x20\x07\xE6\x54\xDA\x0E\xA0\x5A\xB2\xAE\x11\x9F\x87\xC5\xB6\xFF\x69\xDE\x25\xBE\xF8\xA0\xB7\x08\xF3\x44\xCE\x2A\xDF\x08\x21\x0C\x37\x02\x20\x2D\x26\x03\xA0\x05\xBD\x6B\xD1\xF6\x5C\xF8\x65\xCC\x86\x6D\xB3\x9C\x34\x48\x63\x84\x09\xC5\x8D\x77\x1A\xE2\xCC\x9C\xE1\x74\x7B"),
   // "ISRG Root X1"
   CertificateDer::from_slice(b"\x30\x82\x05\x6B\x30\x82\x03\x53\xA0\x03\x02\x01\x02\x02\x11\x00\x82\x10\xCF\xB0\xD2\x40\xE3\x59\x44\x63\xE0\xBB\x63\x82\x8B\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x29\x30\x27\x06\x03\x55\x04\x0A\x13\x20\x49\x6E\x74\x65\x72\x6E\x65\x74\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x47\x72\x6F\x75\x70\x31\x15\x30\x13\x06\x03\x55\x04\x03\x13\x0C\x49\x53\x52\x47\x20\x52\x6F\x6F\x74\x20\x58\x31\x30\x1E\x17\x0D\x31\x35\x30\x36\x30\x34\x31\x31\x30\x34\x33\x38\x5A\x17\x0D\x33\x35\x30\x36\x30\x34\x31\x31\x30\x34\x33\x38\x5A\x30\x4F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x29\x30\x27\x06\x03\x55\x04\x0A\x13\x20\x49\x6E\x74\x65\x72\x6E\x65\x74\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x47\x72\x6F\x75\x70\x31\x15\x30\x13\x06\x03\x55\x04\x03\x13\x0C\x49\x53\x52\x47\x20\x52\x6F\x6F\x74\x20\x58\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xAD\xE8\x24\x73\xF4\x14\x37\xF3\x9B\x9E\x2B\x57\x28\x1C\x87\xBE\xDC\xB7\xDF\x38\x90\x8C\x6E\x3C\xE6\x57\xA0\x78\xF7\x75\xC2\xA2\xFE\xF5\x6A\x6E\xF6\x00\x4F\x28\xDB\xDE\x68\x86\x6C\x44\x93\xB6\xB1\x63\xFD\x14\x12\x6B\xBF\x1F\xD2\xEA\x31\x9B\x21\x7E\xD1\x33\x3C\xBA\x48\xF5\xDD\x79\xDF\xB3\xB8\xFF\x12\xF1\x21\x9A\x4B\xC1\x8A\x86\x71\x69\x4A\x66\x66\x6C\x8F\x7E\x3C\x70\xBF\xAD\x29\x22\x06\xF3\xE4\xC0\xE6\x80\xAE\xE2\x4B\x8F\xB7\x99\x7E\x94\x03\x9F\xD3\x47\x97\x7C\x99\x48\x23\x53\xE8\x38\xAE\x4F\x0A\x6F\x83\x2E\xD1\x49\x57\x8C\x80\x74\xB6\xDA\x2F\xD0\x38\x8D\x7B\x03\x70\x21\x1B\x75\xF2\x30\x3C\xFA\x8F\xAE\xDD\xDA\x63\xAB\xEB\x16\x4F\xC2\x8E\x11\x4B\x7E\xCF\x0B\xE8\xFF\xB5\x77\x2E\xF4\xB2\x7B\x4A\xE0\x4C\x12\x25\x0C\x70\x8D\x03\x29\xA0\xE1\x53\x24\xEC\x13\xD9\xEE\x19\xBF\x10\xB3\x4A\x8C\x3F\x89\xA3\x61\x51\xDE\xAC\x87\x07\x94\xF4\x63\x71\xEC\x2E\xE2\x6F\x5B\x98\x81\xE1\x89\x5C\x34\x79\x6C\x76\xEF\x3B\x90\x62\x79\xE6\xDB\xA4\x9A\x2F\x26\xC5\xD0\x10\xE1\x0E\xDE\xD9\x10\x8E\x16\xFB\xB7\xF7\xA8\xF7\xC7\xE5\x02\x07\x98\x8F\x36\x08\x95\xE7\xE2\x37\x96\x0D\x36\x75\x9E\xFB\x0E\x72\xB1\x1D\x9B\xBC\x03\xF9\x49\x05\xD8\x81\xDD\x05\xB4\x2A\xD6\x41\xE9\xAC\x01\x76\x95\x0A\x0F\xD8\xDF\xD5\xBD\x12\x1F\x35\x2F\x28\x17\x6C\xD2\x98\xC1\xA8\x09\x64\x77\x6E\x47\x37\xBA\xCE\xAC\x59\x5E\x68\x9D\x7F\x72\xD6\x89\xC5\x06\x41\x29\x3E\x59\x3E\xDD\x26\xF5\x24\xC9\x11\xA7\x5A\xA3\x4C\x40\x1F\x46\xA1\x99\xB5\xA7\x3A\x51\x6E\x86\x3B\x9E\x7D\x72\xA7\x12\x05\x78\x59\xED\x3E\x51\x78\x15\x0B\x03\x8F\x8D\xD0\x2F\x05\xB2\x3E\x7B\x4A\x1C\x4B\x73\x05\x12\xFC\xC6\xEA\xE0\x50\x13\x7C\x43\x93\x74\xB3\xCA\x74\xE7\x8E\x1F\x01\x08\xD0\x30\xD4\x5B\x71\x36\xB4\x07\xBA\xC1\x30\x30\x5C\x48\xB7\x82\x3B\x98\xA6\x7D\x60\x8A\xA2\xA3\x29\x82\xCC\xBA\xBD\x83\x04\x1B\xA2\x83\x03\x41\xA1\xD6\x05\xF1\x1B\xC2\xB6\xF0\xA8\x7C\x86\x3B\x46\xA8\x48\x2A\x88\xDC\x76\x9A\x76\xBF\x1F\x6A\xA5\x3D\x19\x8F\xEB\x38\xF3\x64\xDE\xC8\x2B\x0D\x0A\x28\xFF\xF7\xDB\xE2\x15\x42\xD4\x22\xD0\x27\x5D\xE1\x79\xFE\x18\xE7\x70\x88\xAD\x4E\xE6\xD9\x8B\x3A\xC6\xDD\x27\x51\x6E\xFF\xBC\x64\xF5\x33\x43\x4F\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x79\xB4\x59\xE6\x7B\xB6\xE5\xE4\x01\x73\x80\x08\x88\xC8\x1A\x58\xF6\xE9\x9B\x6E\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x55\x1F\x58\xA9\xBC\xB2\xA8\x50\xD0\x0C\xB1\xD8\x1A\x69\x20\x27\x29\x08\xAC\x61\x75\x5C\x8A\x6E\xF8\x82\xE5\x69\x2F\xD5\xF6\x56\x4B\xB9\xB8\x73\x10\x59\xD3\x21\x97\x7E\xE7\x4C\x71\xFB\xB2\xD2\x60\xAD\x39\xA8\x0B\xEA\x17\x21\x56\x85\xF1\x50\x0E\x59\xEB\xCE\xE0\x59\xE9\xBA\xC9\x15\xEF\x86\x9D\x8F\x84\x80\xF6\xE4\xE9\x91\x90\xDC\x17\x9B\x62\x1B\x45\xF0\x66\x95\xD2\x7C\x6F\xC2\xEA\x3B\xEF\x1F\xCF\xCB\xD6\xAE\x27\xF1\xA9\xB0\xC8\xAE\xFD\x7D\x7E\x9A\xFA\x22\x04\xEB\xFF\xD9\x7F\xEA\x91\x2B\x22\xB1\x17\x0E\x8F\xF2\x8A\x34\x5B\x58\xD8\xFC\x01\xC9\x54\xB9\xB8\x26\xCC\x8A\x88\x33\x89\x4C\x2D\x84\x3C\x82\xDF\xEE\x96\x57\x05\xBA\x2C\xBB\xF7\xC4\xB7\xC7\x4E\x3B\x82\xBE\x31\xC8\x22\x73\x73\x92\xD1\xC2\x80\xA4\x39\x39\x10\x33\x23\x82\x4C\x3C\x9F\x86\xB2\x55\x98\x1D\xBE\x29\x86\x8C\x22\x9B\x9E\xE2\x6B\x3B\x57\x3A\x82\x70\x4D\xDC\x09\xC7\x89\xCB\x0A\x07\x4D\x6C\xE8\x5D\x8E\xC9\xEF\xCE\xAB\xC7\xBB\xB5\x2B\x4E\x45\xD6\x4A\xD0\x26\xCC\xE5\x72\xCA\x08\x6A\xA5\x95\xE3\x15\xA1\xF7\xA4\xED\xC9\x2C\x5F\xA5\xFB\xFF\xAC\x28\x02\x2E\xBE\xD7\x7B\xBB\xE3\x71\x7B\x90\x16\xD3\x07\x5E\x46\x53\x7C\x37\x07\x42\x8C\xD3\xC4\x96\x9C\xD5\x99\xB5\x2A\xE0\x95\x1A\x80\x48\xAE\x4C\x39\x07\xCE\xCC\x47\xA4\x52\x95\x2B\xBA\xB8\xFB\xAD\xD2\x33\x53\x7D\xE5\x1D\x4D\x6D\xD5\xA1\xB1\xC7\x42\x6F\xE6\x40\x27\x35\x5C\xA3\x28\xB7\x07\x8D\xE7\x8D\x33\x90\xE7\x23\x9F\xFB\x50\x9C\x79\x6C\x46\xD5\xB4\x15\xB3\x96\x6E\x7E\x9B\x0C\x96\x3A\xB8\x52\x2D\x3F\xD6\x5B\xE1\xFB\x08\xC2\x84\xFE\x24\xA8\xA3\x89\xDA\xAC\x6A\xE1\x18\x2A\xB1\xA8\x43\x61\x5B\xD3\x1F\xDC\x3B\x8D\x76\xF2\x2D\xE8\x8D\x75\xDF\x17\x33\x6C\x3D\x53\xFB\x7B\xCB\x41\x5F\xFF\xDC\xA2\xD0\x61\x38\xE1\x96\xB8\xAC\x5D\x8B\x37\xD7\x75\xD5\x33\xC0\x99\x11\xAE\x9D\x41\xC1\x72\x75\x84\xBE\x02\x41\x42\x5F\x67\x24\x48\x94\xD1\x9B\x27\xBE\x07\x3F\xB9\xB8\x4F\x81\x74\x51\xE1\x7A\xB7\xED\x9D\x23\xE2\xBE\xE0\xD5\x28\x04\x13\x3C\x31\x03\x9E\xDD\x7A\x6C\x8F\xC6\x07\x18\xC6\x7F\xDE\x47\x8E\x3F\x28\x9E\x04\x06\xCF\xA5\x54\x34\x77\xBD\xEC\x89\x9B\xE9\x17\x43\xDF\x5B\xDB\x5F\xFE\x8E\x1E\x57\xA2\xCD\x40\x9D\x7E\x62\x22\xDA\xDE\x18\x27"),
   // "Trustwave Global Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x05\xDA\x30\x82\x03\xC2\xA0\x03\x02\x01\x02\x02\x0C\x05\xF7\x0E\x86\xDA\x49\xF3\x46\x35\x2E\xBA\xB2\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x88\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x08\x0C\x08\x49\x6C\x6C\x69\x6E\x6F\x69\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x43\x68\x69\x63\x61\x67\x6F\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x0C\x18\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x48\x6F\x6C\x64\x69\x6E\x67\x73\x2C\x20\x49\x6E\x63\x2E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x0C\x28\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x37\x30\x38\x32\x33\x31\x39\x33\x34\x31\x32\x5A\x17\x0D\x34\x32\x30\x38\x32\x33\x31\x39\x33\x34\x31\x32\x5A\x30\x81\x88\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x08\x0C\x08\x49\x6C\x6C\x69\x6E\x6F\x69\x73\x31\x10\x30\x0E\x06\x03\x55\x04\x07\x0C\x07\x43\x68\x69\x63\x61\x67\x6F\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x0C\x18\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x48\x6F\x6C\x64\x69\x6E\x67\x73\x2C\x20\x49\x6E\x63\x2E\x31\x31\x30\x2F\x06\x03\x55\x04\x03\x0C\x28\x54\x72\x75\x73\x74\x77\x61\x76\x65\x20\x47\x6C\x6F\x62\x61\x6C\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB9\x5D\x51\x28\x4B\x3C\x37\x92\xD1\x82\xCE\xBD\x1D\xBD\xCD\xDD\xB8\xAB\xCF\x0A\x3E\xE1\x5D\xE5\xDC\xAA\x09\xB9\x57\x02\x3E\xE6\x63\x61\xDF\xF2\x0F\x82\x63\xAE\xA3\xF7\xAC\x73\xD1\x7C\xE7\xB3\x0B\xAF\x08\x00\x09\x59\x7F\xCD\x29\x2A\x88\x93\x87\x17\x18\x80\xED\x88\xB2\xB4\xB6\x10\x1F\x2D\xD6\x5F\x55\xA2\x13\x5D\xD1\xC6\xEB\x06\x56\x89\x88\xFE\xAC\x32\x9D\xFD\x5C\xC3\x05\xC7\x6E\xEE\x86\x89\xBA\x88\x03\x9D\x72\x21\x86\x90\xAE\x8F\x03\xA5\xDC\x9F\x88\x28\xCB\xA3\x92\x49\x0F\xEC\xD0\x0F\xE2\x6D\x44\x4F\x80\x6A\xB2\xD4\xE7\xA0\x0A\x53\x01\xBA\x8E\x97\x91\x76\x6E\xBC\xFC\xD5\x6B\x36\xE6\x40\x88\xD6\x7B\x2F\x5F\x05\xE8\x2C\x6D\x11\xF3\xE7\xB2\xBE\x92\x44\x4C\xD2\x97\xA4\xFE\xD2\x72\x81\x43\x07\x9C\xE9\x11\x3E\xF5\x8B\x1A\x59\x7D\x1F\x68\x58\xDD\x04\x00\x2C\x96\xF3\x43\xB3\x7E\x98\x19\x74\xD9\x9C\x73\xD9\x18\xBE\x41\xC7\x34\x79\xD9\xF4\x62\xC2\x43\xB9\xB3\x27\xB0\x22\xCB\xF9\x3D\x52\xC7\x30\x47\xB3\xC9\x3E\xB8\x6A\xE2\xE7\xE8\x81\x70\x5E\x42\x8B\x4F\x26\xA5\xFE\x3A\xC2\x20\x6E\xBB\xF8\x16\x8E\xCD\x0C\xA9\xB4\x1B\x6C\x76\x10\xE1\x58\x79\x46\x3E\x54\xCE\x80\xA8\x57\x09\x37\x29\x1B\x99\x13\x8F\x0C\xC8\xD6\x2C\x1C\xFB\x05\xE8\x08\x95\x3D\x65\x46\xDC\xEE\xCD\x69\xE2\x4D\x8F\x87\x28\x4E\x34\x0B\x3E\xCF\x14\xD9\xBB\xDD\xB6\x50\x9A\xAD\x77\xD4\x19\xD6\xDA\x1A\x88\xC8\x4E\x1B\x27\x75\xD8\xB2\x08\xF1\xAE\x83\x30\xB9\x11\x0E\xCD\x87\xF0\x84\x8D\x15\x72\x7C\xA1\xEF\xCC\xF2\x88\x61\xBA\xF4\x69\xBB\x0C\x8C\x0B\x75\x57\x04\xB8\x4E\x2A\x14\x2E\x3D\x0F\x1C\x1E\x32\xA6\x62\x36\xEE\x66\xE2\x22\xB8\x05\x40\x63\x10\x22\xF3\x33\x1D\x74\x72\x8A\x2C\xF5\x39\x29\xA0\xD3\xE7\x1B\x80\x84\x2D\xC5\x3D\xE3\x4D\xB1\xFD\x1A\x6F\xBA\x65\x07\x3B\x58\xEC\x42\x45\x26\xFB\xD8\xDA\x25\x72\xC4\xF6\x00\xB1\x22\x79\xBD\xE3\x7C\x59\x62\x4A\x9C\x05\x6F\x3D\xCE\xE6\xD6\x47\x63\x99\xC6\x24\x6F\x72\x12\xC8\xAC\x7F\x90\xB4\x0B\x91\x70\xE8\xB7\xE6\x16\x10\x71\x17\xCE\xDE\x06\x4F\x48\x41\x7D\x35\x4A\xA3\x89\xF2\xC9\x4B\x7B\x41\x11\x6D\x67\xB7\x08\x98\x4C\xE5\x11\x19\xAE\x42\x80\xDC\xFB\x90\x05\xD4\xF8\x50\xCA\xBE\xE4\xAD\xC7\xC2\x94\xD7\x16\x9D\xE6\x17\x8F\xAF\x36\xFB\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x99\xE0\x19\x67\x0D\x62\xDB\x76\xB3\xDA\x3D\xB8\x5B\xE8\xFD\x42\xD2\x31\x0E\x87\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x98\x73\x70\xE2\xB0\xD3\xED\x39\xEC\x4C\x60\xD9\xA9\x12\x86\x17\x1E\x96\xD0\xE8\x54\x28\x3B\x64\x2D\x21\xA6\xF8\x9D\x56\x13\x6A\x48\x3D\x4F\xC7\x3E\x29\xDB\x6D\x58\x83\x54\x3D\x87\x7D\x23\x05\xD4\xE4\x1C\xDC\xE8\x38\x65\x86\xC5\x75\xA7\x5A\xDB\x35\x05\xBD\x77\xDE\xBB\x29\x37\x40\x05\x07\xC3\x94\x52\x9F\xCA\x64\xDD\xF1\x1B\x2B\xDC\x46\x0A\x10\x02\x31\xFD\x4A\x68\x0D\x07\x64\x90\xE6\x1E\xF5\x2A\xA1\xA8\xBB\x3C\x5D\xF9\xA3\x08\x0B\x11\x0C\xF1\x3F\x2D\x10\x94\x6F\xFE\xE2\x34\x87\x83\xD6\xCF\xE5\x1B\x35\x6D\xD2\x03\xE1\xB0\x0D\xA8\xA0\xAA\x46\x27\x82\x36\xA7\x15\xB6\x08\xA6\x42\x54\x57\xB6\x99\x5A\xE2\x0B\x79\x90\xD7\x57\x12\x51\x35\x19\x88\x41\x68\x25\xD4\x37\x17\x84\x15\xFB\x01\x72\xDC\x95\xDE\x52\x26\x20\x98\x26\xE2\x76\xF5\x27\x6F\xFA\x00\x3B\x4A\x61\xD9\x0D\xCB\x51\x93\x2A\xFD\x16\x06\x96\xA7\x23\x9A\x23\x48\xFE\x51\xBD\xB6\xC4\xB0\xB1\x54\xCE\xDE\x6C\x41\xAD\x16\x67\x7E\xDB\xFD\x38\xCD\xB9\x38\x4E\xB2\xC1\x60\xCB\x9D\x17\xDF\x58\x9E\x7A\x62\xB2\x26\x8F\x74\x95\x9B\xE4\x5B\x1D\xD2\x0F\xDD\x98\x1C\x9B\x59\xB9\x23\xD3\x31\xA0\xA6\xFF\x38\xDD\xCF\x20\x4F\xE9\x58\x56\x3A\x67\xC3\xD1\xF6\x99\x99\x9D\xBA\x36\xB6\x80\x2F\x88\x47\x4F\x86\xBF\x44\x3A\x80\xE4\x37\x1C\xA6\xBA\xEA\x97\x98\x11\xD0\x84\x62\x47\x64\x1E\xAA\xEE\x40\xBF\x34\xB1\x9C\x8F\x4E\xE1\xF2\x92\x4F\x1F\x8E\xF3\x9E\x97\xDE\xF3\xA6\x79\x6A\x89\x71\x4F\x4B\x27\x17\x48\xFE\xEC\xF4\x50\x0F\x4F\x49\x7D\xCC\x45\xE3\xBD\x7A\x40\xC5\x41\xDC\x61\x56\x27\x06\x69\xE5\x72\x41\x81\xD3\xB6\x01\x89\xA0\x2F\x3A\x72\x79\xFE\x3A\x30\xBF\x41\xEC\xC7\x62\x3E\x91\x4B\xC7\xD9\x31\x76\x42\xF9\xF7\x3C\x63\xEC\x26\x8C\x73\x0C\x7D\x1A\x1D\xEA\xA8\x7C\x87\xA8\xC2\x27\x7C\xE1\x33\x41\x0F\xCF\xCF\xFC\x00\xA0\x22\x80\x9E\x4A\xA7\x6F\x00\xB0\x41\x45\xB7\x22\xCA\x68\x48\xC5\x42\xA2\xAE\xDD\x1D\xF2\xE0\x6E\x4E\x05\x58\xB1\xC0\x90\x16\x2A\xA4\x3D\x10\x40\xBE\x8F\x62\x63\x83\xA9\x9C\x82\x7D\x2D\x02\xE9\x83\x30\x7C\xCB\x27\xC9\xFD\x1E\x66\x00\xB0\x2E\xD3\x21\x2F\x8E\x33\x16\x6C\x98\xED\x10\xA8\x07\xD6\xCC\x93\xCF\xDB\xD1\x69\x1C\xE4\xCA\xC9\xE0\xB6\x9C\xE9\xCE\x71\x71\xDE\x6C\x3F\x16\xA4\x79"),
   // "Buypass Class 2 Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\x59\x30\x82\x03\x41\xA0\x03\x02\x01\x02\x02\x01\x02\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4E\x4F\x31\x1D\x30\x1B\x06\x03\x55\x04\x0A\x0C\x14\x42\x75\x79\x70\x61\x73\x73\x20\x41\x53\x2D\x39\x38\x33\x31\x36\x33\x33\x32\x37\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x0C\x17\x42\x75\x79\x70\x61\x73\x73\x20\x43\x6C\x61\x73\x73\x20\x32\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x30\x31\x30\x32\x36\x30\x38\x33\x38\x30\x33\x5A\x17\x0D\x34\x30\x31\x30\x32\x36\x30\x38\x33\x38\x30\x33\x5A\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4E\x4F\x31\x1D\x30\x1B\x06\x03\x55\x04\x0A\x0C\x14\x42\x75\x79\x70\x61\x73\x73\x20\x41\x53\x2D\x39\x38\x33\x31\x36\x33\x33\x32\x37\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x0C\x17\x42\x75\x79\x70\x61\x73\x73\x20\x43\x6C\x61\x73\x73\x20\x32\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD7\xC7\x5E\xF7\xC1\x07\xD4\x77\xFB\x43\x21\xF4\xF4\xF5\x69\xE4\xEE\x32\x01\xDB\xA3\x86\x1F\xE4\x59\x0D\xBA\xE7\x75\x83\x52\xEB\xEA\x1C\x61\x15\x48\xBB\x1D\x07\xCA\x8C\xAE\xB0\xDC\x96\x9D\xEA\xC3\x60\x92\x86\x82\x28\x73\x9C\x56\x06\xFF\x4B\x64\xF0\x0C\x2A\x37\x49\xB5\xE5\xCF\x0C\x7C\xEE\xF1\x4A\xBB\x73\x30\x65\xF3\xD5\x2F\x83\xB6\x7E\xE3\xE7\xF5\x9E\xAB\x60\xF9\xD3\xF1\x9D\x92\x74\x8A\xE4\x1C\x96\xAC\x5B\x80\xE9\xB5\xF4\x31\x87\xA3\x51\xFC\xC7\x7E\xA1\x6F\x8E\x53\x77\xD4\x97\xC1\x55\x33\x92\x3E\x18\x2F\x75\xD4\xAD\x86\x49\xCB\x95\xAF\x54\x06\x6C\xD8\x06\x13\x8D\x5B\xFF\xE1\x26\x19\x59\xC0\x24\xBA\x81\x71\x79\x90\x44\x50\x68\x24\x94\x5F\xB8\xB3\x11\xF1\x29\x41\x61\xA3\x41\xCB\x23\x36\xD5\xC1\xF1\x32\x50\x10\x4E\x7F\xF4\x86\x93\xEC\x84\xD3\x8E\xBC\x4B\xBF\x5C\x01\x4E\x07\x3D\xDC\x14\x8A\x94\x0A\xA4\xEA\x73\xFB\x0B\x51\xE8\x13\x07\x18\xFA\x0E\xF1\x2B\xD1\x54\x15\x7D\x3C\xE1\xF7\xB4\x19\x42\x67\x62\x5E\x77\xE0\xA2\x55\xEC\xB6\xD9\x69\x17\xD5\x3A\xAF\x44\xED\x4A\xC5\x9E\xE4\x7A\x27\x7C\xE5\x75\xD7\xAA\xCB\x25\xE7\xDF\x6B\x0A\xDB\x0F\x4D\x93\x4E\xA8\xA0\xCD\x7B\x2E\xF2\x59\x01\x6A\xB7\x0D\xB8\x07\x81\x7E\x8B\x38\x1B\x38\xE6\x0A\x57\x99\x3D\xEE\x21\xE8\xA3\xF5\x0C\x16\xDD\x8B\xEC\x34\x8E\x9C\x2A\x1C\x00\x15\x17\x8D\x68\x83\xD2\x70\x9F\x18\x08\xCD\x11\x68\xD5\xC9\x6B\x52\xCD\xC4\x46\x8F\xDC\xB5\xF3\xD8\x57\x73\x1E\xE9\x94\x39\x04\xBF\xD3\xDE\x38\xDE\xB4\x53\xEC\x69\x1C\xA2\x7E\xC4\x8F\xE4\x1B\x70\xAD\xF2\xA2\xF9\xFB\xF7\x16\x64\x66\x69\x9F\x49\x51\xA2\xE2\x15\x18\x67\x06\x4A\x7F\xD5\x6C\xB5\x4D\xB3\x33\xE0\x61\xEB\x5D\xBE\xE9\x98\x0F\x32\xD7\x1D\x4B\x3C\x2E\x5A\x01\x52\x91\x09\xF2\xDF\xEA\x8D\xD8\x06\x40\x63\xAA\x11\xE4\xFE\xC3\x37\x9E\x14\x52\x3F\xF4\xE2\xCC\xF2\x61\x93\xD1\xFD\x67\x6B\xD7\x52\xAE\xBF\x68\xAB\x40\x43\xA0\x57\x35\x53\x78\xF0\x53\xF8\x61\x42\x07\x64\xC6\xD7\x6F\x9B\x4C\x38\x0D\x63\xAC\x62\xAF\x36\x8B\xA2\x73\x0A\x0D\xF5\x21\xBD\x74\xAA\x4D\xEA\x72\x03\x49\xDB\xC7\x5F\x1D\x62\x63\xC7\xFD\xDD\x91\xEC\x33\xEE\xF5\x6D\xB4\x6E\x30\x68\xDE\xC8\xD6\x26\xB0\x75\x5E\x7B\xB4\x07\x20\x98\xA1\x76\x32\xB8\x4D\x6C\x4F\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC9\x80\x77\xE0\x62\x92\x82\xF5\x46\x9C\xF3\xBA\xF7\x4C\xC3\xDE\xB8\xA3\xAD\x39\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x53\x5F\x21\xF5\xBA\xB0\x3A\x52\x39\x2C\x92\xB0\x6C\x00\xC9\xEF\xCE\x20\xEF\x06\xF2\x96\x9E\xE9\xA4\x74\x7F\x7A\x16\xFC\xB7\xF5\xB6\xFB\x15\x1B\x3F\xAB\xA6\xC0\x72\x5D\x10\xB1\x71\xEE\xBC\x4F\xE3\xAD\xAC\x03\x6D\x2E\x71\x2E\xAF\xC4\xE3\xAD\xA3\xBD\x0C\x11\xA7\xB4\xFF\x4A\xB2\x7B\x10\x10\x1F\xA7\x57\x41\xB2\xC0\xAE\xF4\x2C\x59\xD6\x47\x10\x88\xF3\x21\x51\x29\x30\xCA\x60\x86\xAF\x46\xAB\x1D\xED\x3A\x5B\xB0\x94\xDE\x44\xE3\x41\x08\xA2\xC1\xEC\x1D\xD6\xFD\x4F\xB6\xD6\x47\xD0\x14\x0B\xCA\xE6\xCA\xB5\x7B\x77\x7E\x41\x1F\x5E\x83\xC7\xB6\x8C\x39\x96\xB0\x3F\x96\x81\x41\x6F\x60\x90\xE2\xE8\xF9\xFB\x22\x71\xD9\x7D\xB3\x3D\x46\xBF\xB4\x84\xAF\x90\x1C\x0F\x8F\x12\x6A\xAF\xEF\xEE\x1E\x7A\xAE\x02\x4A\x8A\x17\x2B\x76\xFE\xAC\x54\x89\x24\x2C\x4F\x3F\xB6\xB2\xA7\x4E\x8C\xA8\x91\x97\xFB\x29\xC6\x7B\x5C\x2D\xB9\xCB\x66\xB6\xB7\xA8\x5B\x12\x51\x85\xB5\x09\x7E\x62\x78\x70\xFE\xA9\x6A\x60\xB6\x1D\x0E\x79\x0C\xFD\xCA\xEA\x24\x80\x72\xC3\x97\x3F\xF2\x77\xAB\x43\x22\x0A\xC7\xEB\xB6\x0C\x84\x82\x2C\x80\x6B\x41\x8A\x08\xC0\xEB\xA5\x6B\xDF\x99\x12\xCB\x8A\xD5\x5E\x80\x0C\x91\xE0\x26\x08\x36\x48\xC5\xFA\x38\x11\x35\xFF\x25\x83\x2D\xF2\x7A\xBF\xDA\xFD\x8E\xFE\xA5\xCB\x45\x2C\x1F\xC4\x88\x53\xAE\x77\x0E\xD9\x9A\x76\xC5\x8E\x2C\x1D\xA3\xBA\xD5\xEC\x32\xAE\xC0\xAA\xAC\xF7\xD1\x7A\x4D\xEB\xD4\x07\xE2\x48\xF7\x22\x8E\xB0\xA4\x9F\x6A\xCE\x8E\xB2\xB2\x60\xF4\xA3\x22\xD0\x23\xEB\x94\x5A\x7A\x69\xDD\x0F\xBF\x40\x57\xAC\x6B\x59\x50\xD9\xA3\x99\xE1\x6E\xFE\x8D\x01\x79\x27\x23\x15\xDE\x92\x9D\x7B\x09\x4D\x5A\xE7\x4B\x48\x30\x5A\x18\xE6\x0A\x6D\xE6\x8F\xE0\xD2\xBB\xE6\xDF\x7C\x6E\x21\x82\xC1\x68\x39\x4D\xB4\x98\x58\x66\x62\xCC\x4A\x90\x5E\xC3\xFA\x27\x04\xB1\x79\x15\x74\x99\xCC\xBE\xAD\x20\xDE\x26\x60\x1C\xEB\x56\x51\xA6\xA3\xEA\xE4\xA3\x3F\xA7\xFF\x61\xDC\xF1\x5A\x4D\x6C\x32\x23\x43\xEE\xAC\xA8\xEE\xEE\x4A\x12\x09\x3C\x5D\x71\xC2\xBE\x79\xFA\xC2\x87\x68\x1D\x0B\xFD\x5C\x69\xCC\x06\xD0\x9A\x7D\x54\x99\x2A\xC9\x39\x1A\x19\xAF\x4B\x2A\x43\xF3\x63\x5D\x5A\x58\xE2\x2F\xE3\x1D\xE4\xA9\xD6\xD0\x0A\xD0\x9E\xBF\xD7\x81\x09\xF1\xC9\xC7\x26\x0D\xAC\x98\x16\x56\xA0"),
   // "ACCVRAIZ1"
   CertificateDer::from_slice(b"\x30\x82\x07\xD3\x30\x82\x05\xBB\xA0\x03\x02\x01\x02\x02\x08\x5E\xC3\xB7\xA6\x43\x7F\xA4\xE0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x42\x31\x12\x30\x10\x06\x03\x55\x04\x03\x0C\x09\x41\x43\x43\x56\x52\x41\x49\x5A\x31\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x0C\x07\x50\x4B\x49\x41\x43\x43\x56\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x43\x43\x56\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x30\x1E\x17\x0D\x31\x31\x30\x35\x30\x35\x30\x39\x33\x37\x33\x37\x5A\x17\x0D\x33\x30\x31\x32\x33\x31\x30\x39\x33\x37\x33\x37\x5A\x30\x42\x31\x12\x30\x10\x06\x03\x55\x04\x03\x0C\x09\x41\x43\x43\x56\x52\x41\x49\x5A\x31\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x0C\x07\x50\x4B\x49\x41\x43\x43\x56\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x43\x43\x56\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x9B\xA9\xAB\xBF\x61\x4A\x97\xAF\x2F\x97\x66\x9A\x74\x5F\xD0\xD9\x96\xFD\xCF\xE2\xE4\x66\xEF\x1F\x1F\x47\x33\xC2\x44\xA3\xDF\x9A\xDE\x1F\xB5\x54\xDD\x15\x7C\x69\x35\x11\x6F\xBB\xC8\x0C\x8E\x6A\x18\x1E\xD8\x8F\xD9\x16\xBC\x10\x48\x36\x5C\xF0\x63\xB3\x90\x5A\x5C\x24\x37\xD7\xA3\xD6\xCB\x09\x71\xB9\xF1\x01\x72\x84\xB0\x7D\xDB\x4D\x80\xCD\xFC\xD3\x6F\xC9\xF8\xDA\xB6\x0E\x82\xD2\x45\x85\xA8\x1B\x68\xA8\x3D\xE8\xF4\x44\x6C\xBD\xA1\xC2\xCB\x03\xBE\x8C\x3E\x13\x00\x84\xDF\x4A\x48\xC0\xE3\x22\x0A\xE8\xE9\x37\xA7\x18\x4C\xB1\x09\x0D\x23\x56\x7F\x04\x4D\xD9\x17\x84\x18\xA5\xC8\xDA\x40\x94\x73\xEB\xCE\x0E\x57\x3C\x03\x81\x3A\x9D\x0A\xA1\x57\x43\x69\xAC\x57\x6D\x79\x90\x78\xE5\xB5\xB4\x3B\xD8\xBC\x4C\x8D\x28\xA1\xA7\xA3\xA7\xBA\x02\x4E\x25\xD1\x2A\xAE\xED\xAE\x03\x22\xB8\x6B\x20\x0F\x30\x28\x54\x95\x7F\xE0\xEE\xCE\x0A\x66\x9D\xD1\x40\x2D\x6E\x22\xAF\x9D\x1A\xC1\x05\x19\xD2\x6F\xC0\xF2\x9F\xF8\x7B\xB3\x02\x42\xFB\x50\xA9\x1D\x2D\x93\x0F\x23\xAB\xC6\xC1\x0F\x92\xFF\xD0\xA2\x15\xF5\x53\x09\x71\x1C\xFF\x45\x13\x84\xE6\x26\x5E\xF8\xE0\x88\x1C\x0A\xFC\x16\xB6\xA8\x73\x06\xB8\xF0\x63\x84\x02\xA0\xC6\x5A\xEC\xE7\x74\xDF\x70\xAE\xA3\x83\x25\xEA\xD6\xC7\x97\x87\x93\xA7\xC6\x8A\x8A\x33\x97\x60\x37\x10\x3E\x97\x3E\x6E\x29\x15\xD6\xA1\x0F\xD1\x88\x2C\x12\x9F\x6F\xAA\xA4\xC6\x42\xEB\x41\xA2\xE3\x95\x43\xD3\x01\x85\x6D\x8E\xBB\x3B\xF3\x23\x36\xC7\xFE\x3B\xE0\xA1\x25\x07\x48\xAB\xC9\x89\x74\xFF\x08\x8F\x80\xBF\xC0\x96\x65\xF3\xEE\xEC\x4B\x68\xBD\x9D\x88\xC3\x31\xB3\x40\xF1\xE8\xCF\xF6\x38\xBB\x9C\xE4\xD1\x7F\xD4\xE5\x58\x9B\x7C\xFA\xD4\xF3\x0E\x9B\x75\x91\xE4\xBA\x52\x2E\x19\x7E\xD1\xF5\xCD\x5A\x19\xFC\xBA\x06\xF6\xFB\x52\xA8\x4B\x99\x04\xDD\xF8\xF9\xB4\x8B\x50\xA3\x4E\x62\x89\xF0\x87\x24\xFA\x83\x42\xC1\x87\xFA\xD5\x2D\x29\x2A\x5A\x71\x7A\x64\x6A\xD7\x27\x60\x63\x0D\xDB\xCE\x49\xF5\x8D\x1F\x90\x89\x32\x17\xF8\x73\x43\xB8\xD2\x5A\x93\x86\x61\xD6\xE1\x75\x0A\xEA\x79\x66\x76\x88\x4F\x71\xEB\x04\x25\xD6\x0A\x5A\x7A\x93\xE5\xB9\x4B\x17\x40\x0F\xB1\xB6\xB9\xF5\xDE\x4F\xDC\xE0\xB3\xAC\x3B\x11\x70\x60\x84\x4A\x43\x6E\x99\x20\xC0\x29\x71\x0A\xC0\x65\x02\x03\x01\x00\x01\xA3\x82\x02\xCB\x30\x82\x02\xC7\x30\x7D\x06\x08\x2B\x06\x01\x05\x05\x07\x01\x01\x04\x71\x30\x6F\x30\x4C\x06\x08\x2B\x06\x01\x05\x05\x07\x30\x02\x86\x40\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x61\x63\x63\x76\x2E\x65\x73\x2F\x66\x69\x6C\x65\x61\x64\x6D\x69\x6E\x2F\x41\x72\x63\x68\x69\x76\x6F\x73\x2F\x63\x65\x72\x74\x69\x66\x69\x63\x61\x64\x6F\x73\x2F\x72\x61\x69\x7A\x61\x63\x63\x76\x31\x2E\x63\x72\x74\x30\x1F\x06\x08\x2B\x06\x01\x05\x05\x07\x30\x01\x86\x13\x68\x74\x74\x70\x3A\x2F\x2F\x6F\x63\x73\x70\x2E\x61\x63\x63\x76\x2E\x65\x73\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD2\x87\xB4\xE3\xDF\x37\x27\x93\x55\xF6\x56\xEA\x81\xE5\x36\xCC\x8C\x1E\x3F\xBD\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xD2\x87\xB4\xE3\xDF\x37\x27\x93\x55\xF6\x56\xEA\x81\xE5\x36\xCC\x8C\x1E\x3F\xBD\x30\x82\x01\x73\x06\x03\x55\x1D\x20\x04\x82\x01\x6A\x30\x82\x01\x66\x30\x82\x01\x62\x06\x04\x55\x1D\x20\x00\x30\x82\x01\x58\x30\x82\x01\x22\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x02\x30\x82\x01\x14\x1E\x82\x01\x10\x00\x41\x00\x75\x00\x74\x00\x6F\x00\x72\x00\x69\x00\x64\x00\x61\x00\x64\x00\x20\x00\x64\x00\x65\x00\x20\x00\x43\x00\x65\x00\x72\x00\x74\x00\x69\x00\x66\x00\x69\x00\x63\x00\x61\x00\x63\x00\x69\x00\xF3\x00\x6E\x00\x20\x00\x52\x00\x61\x00\xED\x00\x7A\x00\x20\x00\x64\x00\x65\x00\x20\x00\x6C\x00\x61\x00\x20\x00\x41\x00\x43\x00\x43\x00\x56\x00\x20\x00\x28\x00\x41\x00\x67\x00\x65\x00\x6E\x00\x63\x00\x69\x00\x61\x00\x20\x00\x64\x00\x65\x00\x20\x00\x54\x00\x65\x00\x63\x00\x6E\x00\x6F\x00\x6C\x00\x6F\x00\x67\x00\xED\x00\x61\x00\x20\x00\x79\x00\x20\x00\x43\x00\x65\x00\x72\x00\x74\x00\x69\x00\x66\x00\x69\x00\x63\x00\x61\x00\x63\x00\x69\x00\xF3\x00\x6E\x00\x20\x00\x45\x00\x6C\x00\x65\x00\x63\x00\x74\x00\x72\x00\xF3\x00\x6E\x00\x69\x00\x63\x00\x61\x00\x2C\x00\x20\x00\x43\x00\x49\x00\x46\x00\x20\x00\x51\x00\x34\x00\x36\x00\x30\x00\x31\x00\x31\x00\x35\x00\x36\x00\x45\x00\x29\x00\x2E\x00\x20\x00\x43\x00\x50\x00\x53\x00\x20\x00\x65\x00\x6E\x00\x20\x00\x68\x00\x74\x00\x74\x00\x70\x00\x3A\x00\x2F\x00\x2F\x00\x77\x00\x77\x00\x77\x00\x2E\x00\x61\x00\x63\x00\x63\x00\x76\x00\x2E\x00\x65\x00\x73\x30\x30\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x01\x16\x24\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x61\x63\x63\x76\x2E\x65\x73\x2F\x6C\x65\x67\x69\x73\x6C\x61\x63\x69\x6F\x6E\x5F\x63\x2E\x68\x74\x6D\x30\x55\x06\x03\x55\x1D\x1F\x04\x4E\x30\x4C\x30\x4A\xA0\x48\xA0\x46\x86\x44\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x61\x63\x63\x76\x2E\x65\x73\x2F\x66\x69\x6C\x65\x61\x64\x6D\x69\x6E\x2F\x41\x72\x63\x68\x69\x76\x6F\x73\x2F\x63\x65\x72\x74\x69\x66\x69\x63\x61\x64\x6F\x73\x2F\x72\x61\x69\x7A\x61\x63\x63\x76\x31\x5F\x64\x65\x72\x2E\x63\x72\x6C\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x17\x06\x03\x55\x1D\x11\x04\x10\x30\x0E\x81\x0C\x61\x63\x63\x76\x40\x61\x63\x63\x76\x2E\x65\x73\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00\x97\x31\x02\x9F\xE7\xFD\x43\x67\x48\x44\x14\xE4\x29\x87\xED\x4C\x28\x66\xD0\x8F\x35\xDA\x4D\x61\xB7\x4A\x97\x4D\xB5\xDB\x90\xE0\x05\x2E\x0E\xC6\x79\xD0\xF2\x97\x69\x0F\xBD\x04\x47\xD9\xBE\xDB\xB5\x29\xDA\x9B\xD9\xAE\xA9\x99\xD5\xD3\x3C\x30\x93\xF5\x8D\xA1\xA8\xFC\x06\x8D\x44\xF4\xCA\x16\x95\x7C\x33\xDC\x62\x8B\xA8\x37\xF8\x27\xD8\x09\x2D\x1B\xEF\xC8\x14\x27\x20\xA9\x64\x44\xFF\x2E\xD6\x75\xAA\x6C\x4D\x60\x40\x19\x49\x43\x54\x63\xDA\xE2\xCC\xBA\x66\xE5\x4F\x44\x7A\x5B\xD9\x6A\x81\x2B\x40\xD5\x7F\xF9\x01\x27\x58\x2C\xC8\xED\x48\x91\x7C\x3F\xA6\x00\xCF\xC4\x29\x73\x11\x36\xDE\x86\x19\x3E\x9D\xEE\x19\x8A\x1B\xD5\xB0\xED\x8E\x3D\x9C\x2A\xC0\x0D\xD8\x3D\x66\xE3\x3C\x0D\xBD\xD5\x94\x5C\xE2\xE2\xA7\x35\x1B\x04\x00\xF6\x3F\x5A\x8D\xEA\x43\xBD\x5F\x89\x1D\xA9\xC1\xB0\xCC\x99\xE2\x4D\x00\x0A\xDA\xC9\x27\x5B\xE7\x13\x90\x5C\xE4\xF5\x33\xA2\x55\x6D\xDC\xE0\x09\x4D\x2F\xB1\x26\x5B\x27\x75\x00\x09\xC4\x62\x77\x29\x08\x5F\x9E\x59\xAC\xB6\x7E\xAD\x9F\x54\x30\x22\x03\xC1\x1E\x71\x64\xFE\xF9\x38\x0A\x96\x18\xDD\x02\x14\xAC\x23\xCB\x06\x1C\x1E\xA4\x7D\x8D\x0D\xDE\x27\x41\xE8\xAD\xDA\x15\xB7\xB0\x23\xDD\x2B\xA8\xD3\xDA\x25\x87\xED\xE8\x55\x44\x4D\x88\xF4\x36\x7E\x84\x9A\x78\xAC\xF7\x0E\x56\x49\x0E\xD6\x33\x25\xD6\x84\x50\x42\x6C\x20\x12\x1D\x2A\xD5\xBE\xBC\xF2\x70\x81\xA4\x70\x60\xBE\x05\xB5\x9B\x9E\x04\x44\xBE\x61\x23\xAC\xE9\xA5\x24\x8C\x11\x80\x94\x5A\xA2\xA2\xB9\x49\xD2\xC1\xDC\xD1\xA7\xED\x31\x11\x2C\x9E\x19\xA6\xEE\xE1\x55\xE1\xC0\xEA\xCF\x0D\x84\xE4\x17\xB7\xA2\x7C\xA5\xDE\x55\x25\x06\xEE\xCC\xC0\x87\x5C\x40\xDA\xCC\x95\x3F\x55\xE0\x35\xC7\xB8\x84\xBE\xB4\x5D\xCD\x7A\x83\x01\x72\xEE\x87\xE6\x5F\x1D\xAE\xB5\x85\xC6\x26\xDF\xE6\xC1\x9A\xE9\x1E\x02\x47\x9F\x2A\xA8\x6D\xA9\x5B\xCF\xEC\x45\x77\x7F\x98\x27\x9A\x32\x5D\x2A\xE3\x84\xEE\xC5\x98\x66\x2F\x96\x20\x1D\xDD\xD8\xC3\x27\xD7\xB0\xF9\xFE\xD9\x7D\xCD\xD0\x9F\x8F\x0B\x14\x58\x51\x9F\x2F\x8B\xC3\x38\x2D\xDE\xE8\x8F\xD6\x8D\x87\xA4\xF5\x56\x43\x16\x99\x2C\xF4\xA4\x56\xB4\x34\xB8\x61\x37\xC9\xC2\x58\x80\x1B\xA0\x97\xA1\xFC\x59\x8D\xE9\x11\xF6\xD1\x0F\x4B\x55\x34\x46\x2A\x8B\x86\x3B"),
   // "OISTE Server Root RSA G1"
   CertificateDer::from_slice(b"\x30\x82\x05\x83\x30\x82\x03\x6B\xA0\x03\x02\x01\x02\x02\x10\x55\xA5\xD9\x67\x94\x28\xC6\xED\x0C\xFA\x27\xDD\x5B\x01\x4D\x18\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x4B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x0C\x10\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x31\x21\x30\x1F\x06\x03\x55\x04\x03\x0C\x18\x4F\x49\x53\x54\x45\x20\x53\x65\x72\x76\x65\x72\x20\x52\x6F\x6F\x74\x20\x52\x53\x41\x20\x47\x31\x30\x1E\x17\x0D\x32\x33\x30\x35\x33\x31\x31\x34\x33\x37\x31\x36\x5A\x17\x0D\x34\x38\x30\x35\x32\x34\x31\x34\x33\x37\x31\x35\x5A\x30\x4B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x0C\x10\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x31\x21\x30\x1F\x06\x03\x55\x04\x03\x0C\x18\x4F\x49\x53\x54\x45\x20\x53\x65\x72\x76\x65\x72\x20\x52\x6F\x6F\x74\x20\x52\x53\x41\x20\x47\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xAA\xAE\xF4\xAB\x82\xCF\xFB\xE5\x37\x0B\xE7\xD5\x96\xAD\x90\xE8\x4B\x29\xDC\x55\x60\xE3\xCC\xBC\xB3\xBC\x2D\x92\xB9\xE4\xA3\x7A\xF1\x81\xB4\x9E\x72\x72\x43\xDF\x3F\xAB\x0B\x26\xB4\xEE\x7B\x1A\x69\xFB\x28\xD0\x72\x5C\x4A\x6D\x69\x99\xF0\x63\x1E\x0C\xD2\xB1\xFF\xD6\x8C\x34\xD0\xEE\xDB\xAC\x48\xB9\xEA\xB0\x14\x8E\xD8\x07\xA9\x24\x98\xDD\xE9\x09\xBE\xA8\x22\x1B\x59\x39\xD1\x27\x87\xDC\x1C\xCD\xF8\xFB\xB3\xEB\xE9\x93\x78\xED\x0F\xCE\x37\x7C\x26\x77\x6E\xA1\xD8\x2C\x21\x4C\xE4\x8A\x4F\xC7\x13\x3C\x6E\xC7\xD5\x13\x97\xB2\xA8\xDB\x24\x69\x83\x56\xD3\x69\xCB\x82\x12\xBB\x9D\x1B\xF2\xF8\x34\xF2\x98\x2B\x2A\x8E\x04\x67\xF6\xE3\x87\xA1\x1D\xAD\x6E\xCE\x36\x74\x0E\x5E\x33\x3B\xCB\xDB\x51\x97\x94\x6A\x95\x3C\xCE\x18\x5A\x6E\x4B\xC6\xFC\x07\x8F\x2E\x1A\xB9\x4A\xF7\x64\x34\x29\xDC\xA6\x8D\x50\xE1\x8D\x8B\x4B\xE5\x48\x1B\x6E\x2E\x80\x10\x3F\xE4\x9F\x1B\x65\x3F\x11\xB4\xEA\x57\x69\x9F\xB4\x00\xEB\x85\x24\x99\x24\xF5\x21\x9D\x97\xAA\xFB\x34\x7F\x02\x6B\x15\x90\xAD\xBB\x9E\x5A\x19\x7F\xA4\x8C\xD8\xFA\x6D\x28\xFC\x38\xC7\xE3\x4C\xAD\x6A\xCE\xD9\x4E\x93\x92\x8E\xCC\x0C\x67\xBF\x0B\x4B\x96\xCE\x66\x67\x53\x68\xCB\x17\x11\x8E\x59\xF7\xAC\x9C\x1B\xB9\x8E\x68\x44\xB7\x18\xAF\xE6\xE5\x0F\x65\xDC\x95\x09\xB0\x93\x12\xB5\x1F\x3E\x94\xA5\xC7\x88\x75\x21\xB1\xDE\x09\x24\x2A\x4C\xE2\xBC\xEC\x4C\x67\x47\xC2\x29\x88\xB9\x0A\xBA\xF9\xC1\x74\xCE\x8C\x18\x26\x65\xDA\xF7\x6F\xC6\x8C\x7B\x68\x5C\x0B\xEE\x63\xC0\x5E\x4B\xF1\x4E\xCC\x9F\x2F\x0F\xE1\xE8\x9A\x7A\x93\xF1\xE0\xC8\xDB\xBF\x27\xE6\x65\x29\x7B\x36\xE0\x33\x15\x73\xF2\x9D\x6B\x84\x08\x68\x2B\x36\x07\x2B\x27\xCC\x78\xD8\x6A\x87\x0E\x47\x74\xF4\xAA\xA0\x13\x5D\x64\x7E\xF4\xDB\x14\xAE\xFB\x3A\xE4\x2F\xC1\x65\xE3\xB9\x7A\x40\x6C\xF0\x06\xB7\x7B\x28\x9B\xD7\xE1\x5F\x38\x73\x94\xAC\xD9\x70\x93\x2D\xDC\x84\xAF\x46\x1C\xA2\x7A\x2C\x3F\x81\x26\x42\xE7\xD4\xD8\xC5\x6C\x84\x66\x11\x8B\x77\x6B\x54\x1C\xA3\xB5\xD8\x10\xF0\xAE\x29\xF7\x67\x08\x88\x17\x5C\xB8\x97\x79\xCF\xEA\x2B\x2A\xEE\x58\x33\xE5\x6D\xE9\x29\xAA\x65\x01\x0D\x82\x13\xEC\x25\x0B\x5D\x2C\x40\x72\x15\x29\xD3\x90\x2C\xF7\x1A\x43\xD5\x6A\xF0\x69\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xF2\xC9\xC1\x0F\x0D\x63\x00\xBB\xEC\x45\x0E\x4A\x1F\xB5\xB1\xB3\x36\xCD\x0E\x8D\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF2\xC9\xC1\x0F\x0D\x63\x00\xBB\xEC\x45\x0E\x4A\x1F\xB5\xB1\xB3\x36\xCD\x0E\x8D\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x34\x67\x79\xB2\x3A\xC6\xE5\x3D\xF7\x23\x72\xB9\x09\xEF\x92\xAD\x27\x1F\xA0\x4E\x0A\xB2\xF5\xDA\x17\x0C\xA2\x85\xD2\x7E\x92\x51\xFD\x15\x65\xD7\x5C\x6B\x64\x16\xEE\x8A\x45\xCA\x0C\x43\x36\x44\x35\xD9\x7F\xFE\x79\x3A\x1C\xE8\xC6\xE4\x3D\x6B\x77\xD4\x21\x10\xE3\xF6\xF3\x20\x4E\xA9\xBE\x89\xF3\x1C\x9C\xA9\xDF\xBC\x30\x3A\x17\xD1\x32\x43\xD0\xAA\x8A\x72\x1C\x51\x28\x4C\xDD\x36\xC8\xE4\x2D\x67\x7D\x91\x87\x1C\x9D\xBC\xFC\xAB\x28\x96\x5E\x61\x5C\xB8\x22\x33\x18\x48\x16\x50\xEA\xCA\x2F\xE9\xA5\x49\xDC\x7F\x3C\xA4\x19\xBC\x36\xAD\x92\xE2\xB9\xF4\x4B\xD5\xEB\x08\xAD\xE7\x78\xFE\x17\xC0\x5D\x87\x77\xE8\x67\x77\x4F\x00\x66\xAF\xF4\xB1\x03\x3A\xBE\x12\x7C\x41\x35\xE5\xF4\xA6\x1B\x47\x8B\xCB\x79\xF7\xD6\xBF\x17\x6E\x4E\x65\xF0\xF8\xDA\x57\xC1\x94\x81\xE5\x7A\x56\x0D\xBB\x46\x7F\x6F\x91\xFD\x7D\xE6\x17\xE4\x81\x27\xBB\x05\x88\x56\xDD\x20\xA5\xF7\x98\x2D\x91\x19\x69\x31\x5F\x9B\x30\xF2\x99\xAD\x72\x40\x96\xCC\xD8\x77\x66\x9B\xB4\xD5\x0E\xB2\x10\xFE\x14\xAA\xC3\x80\x71\x9D\x3D\x8D\xE8\x7D\x14\x6C\x61\x64\x86\x46\xA6\xD7\x54\xC5\xB6\xD7\x90\x16\x46\xA5\x85\xCA\x9E\x3A\xE3\xE6\x13\x16\xB6\x15\x23\xCC\xA9\x29\x52\xFD\x00\xC6\xF6\x90\x8E\x56\x8F\x89\x08\xDD\x96\xAA\xE6\xD3\x6A\xA9\x86\x35\xF6\xD5\x45\x78\x42\x4A\x46\xFC\x03\xC8\x5E\xD8\x66\xF6\x45\x65\x24\xB4\xBE\x87\x7B\x55\x20\x9D\xF7\x9D\xB5\x2A\xFC\xB9\x62\x19\xCB\x6C\x3B\xAF\xD3\x6D\x38\x6C\xAB\x7B\xA6\x1E\x8D\xFC\xE9\x9E\xFE\x6B\x15\xB9\xDB\x82\x9A\xCB\x98\xDF\x73\xA1\x90\xA0\xA0\xC5\xE0\xE8\x01\xA8\xA3\x14\x9C\xC8\xC1\x9A\xAC\x15\x50\x33\x8D\xED\x7C\x2A\x8B\x73\x95\x40\x43\x26\xFC\x81\xA4\x2A\x5F\x39\x90\xB7\x27\xCB\x51\x77\xF8\x96\x93\x1E\xCF\xF2\x77\x7D\x1F\x46\x93\xA2\x59\x1E\x95\x44\xC5\x2D\x75\x64\xB0\xD6\xF9\xE0\x3C\x69\xEA\x04\xB5\x1C\x0B\xE2\x46\x44\x4D\x43\x3B\x97\x49\x71\x11\xBD\x24\xB6\xC2\xAD\x72\x54\x06\xFE\x6B\x18\xF9\x77\xDB\x29\x2C\x52\x9E\x6D\x77\x7B\x62\xFD\x0F\x4D\x8E\x98\x32\x30\x30\x71\x12\xD6\x25\x35\xE3\x9F\xF8\x6F\x9C\xB5\xEB\x6A\x1B\xAD\xEA\x10\xD3\x96\x16\x72\x06\x21\x25\xC6\x4C\xBC\x8F\x70\xBB\x0C\xE4\x5E\x22\x83\x2D\xD2\xBE\xFE\x85\x5B\xB4\xE4\xBD"),
   // "UCA Global G2 Root"
   CertificateDer::from_slice(b"\x30\x82\x05\x46\x30\x82\x03\x2E\xA0\x03\x02\x01\x02\x02\x10\x5D\xDF\xB1\xDA\x5A\xA3\xED\x5D\xBE\x5A\x65\x20\x65\x03\x90\xEF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x3D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x55\x6E\x69\x54\x72\x75\x73\x74\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x0C\x12\x55\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x47\x32\x20\x52\x6F\x6F\x74\x30\x1E\x17\x0D\x31\x36\x30\x33\x31\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x30\x31\x32\x33\x31\x30\x30\x30\x30\x30\x30\x5A\x30\x3D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x55\x6E\x69\x54\x72\x75\x73\x74\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x0C\x12\x55\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x47\x32\x20\x52\x6F\x6F\x74\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC5\xE6\x2B\x6F\x7C\xEF\x26\x05\x27\xA3\x81\x24\xDA\x6F\xCB\x01\xF9\x99\x9A\xA9\x32\xC2\x22\x87\x61\x41\x91\x3B\xCB\xC3\x68\x1B\x06\xC5\x4C\xA9\x2B\xC1\x67\x17\x22\x1D\x2B\xED\xF9\x29\x89\x93\xA2\x78\xBD\x92\x6B\xA0\xA3\x0D\xA2\x7E\xCA\x93\xB3\xA6\xD1\x8C\x35\xD5\x75\xF9\x17\xF6\xCF\x45\xC5\xE5\x7A\xEC\x77\x93\xA0\x8F\x23\xAE\x0E\x1A\x03\x7F\xBE\xD4\xD0\xED\x2E\x7B\xAB\x46\x23\x5B\xFF\x2C\xE6\x54\x7A\x94\xC0\x2A\x15\xF0\xC9\x8D\xB0\x7A\x3B\x24\xE1\xD7\x68\xE2\x31\x3C\x06\x33\x46\xB6\x54\x11\xA6\xA5\x2F\x22\x54\x2A\x58\x0D\x01\x02\xF1\xFA\x15\x51\x67\x6C\xC0\xFA\xD7\xB6\x1B\x7F\xD1\x56\x88\x2F\x1A\x3A\x8D\x3B\xBB\x82\x11\xE0\x47\x00\xD0\x52\x87\xAB\xFB\x86\x7E\x0F\x24\x6B\x40\x9D\x34\x67\xBC\x8D\xC7\x2D\x86\x6F\x79\x3E\x8E\xA9\x3C\x17\x4B\x7F\xB0\x99\xE3\xB0\x71\x60\xDC\x0B\xF5\x64\xC3\xCE\x43\xBC\x6D\x71\xB9\xD2\xDE\x27\x5B\x8A\xE8\xD8\xC6\xAE\xE1\x59\x7D\xCF\x28\x2D\x35\xB8\x95\x56\x1A\xF1\xB2\x58\x4B\xB7\x12\x37\xC8\x7C\xB3\xED\x4B\x80\xE1\x8D\xFA\x32\x23\xB6\x6F\xB7\x48\x95\x08\xB1\x44\x4E\x85\x8C\x3A\x02\x54\x20\x2F\xDF\xBF\x57\x4F\x3B\x3A\x90\x21\xD7\xC1\x26\x35\x54\x20\xEC\xC7\x3F\x47\xEC\xEF\x5A\xBF\x4B\x7A\xC1\xAD\x3B\x17\x50\x5C\x62\xD8\x0F\x4B\x4A\xDC\x2B\xFA\x6E\xBC\x73\x92\xCD\xEC\xC7\x50\xE8\x41\x96\xD7\xA9\x7E\x6D\xD8\xE9\x1D\x8F\x8A\xB5\xB9\x58\x92\xBA\x4A\x92\x2B\x0C\x56\xFD\x80\xEB\x08\xF0\x5E\x29\x6E\x1B\x1C\x0C\xAF\x8F\x93\x89\xAD\xDB\xBD\xA3\x9E\x21\xCA\x89\x19\xEC\xDF\xB5\xC3\x1A\xEB\x16\xFE\x78\x36\x4C\xD6\x6E\xD0\x3E\x17\x1C\x90\x17\x6B\x26\xBA\xFB\x7A\x2F\xBF\x11\x1C\x18\x0E\x2D\x73\x03\x8F\xA0\xE5\x35\xA0\x5A\xE2\x4C\x75\x1D\x71\xE1\x39\x38\x53\x78\x40\xCC\x83\x93\xD7\x0A\x9E\x9D\x5B\x8F\x8A\xE4\xE5\xE0\x48\xE4\x48\xB2\x47\xCD\x4E\x2A\x75\x2A\x7B\xF2\x22\xF6\xC9\xBE\x09\x91\x96\x57\x7A\x88\x88\xAC\xEE\x70\xAC\xF9\xDC\x29\xE3\x0C\x1C\x3B\x12\x4E\x44\xD6\xA7\x4E\xB0\x26\xC8\xF3\xD9\x1A\x97\x91\x68\xEA\xEF\x8D\x46\x06\xD2\x56\x45\x58\x9A\x3C\x0C\x0F\x83\xB8\x05\x25\xC3\x39\xCF\x3B\xA4\x34\x89\xB7\x79\x12\x2F\x47\xC5\xE7\xA9\x97\x69\xFC\xA6\x77\x67\xB5\xDF\x7B\xF1\x7A\x65\x15\xE4\x61\x56\x65\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x81\xC4\x8C\xCC\xF5\xE4\x30\xFF\xA5\x0C\x08\x5F\x8C\x15\x67\x21\x74\x01\xDF\xDF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x13\x65\x22\xF5\x8E\x2B\xAD\x44\xE4\xCB\xFF\xB9\x68\xE6\xC3\x80\x48\x3D\x04\x7B\xFA\x23\x2F\x7A\xED\x36\xDA\xB2\xCE\x6D\xF6\xE6\x9E\xE5\x5F\x58\x8F\xCB\x37\x32\xA1\xC8\x65\xB6\xAE\x38\x3D\x35\x1B\x3E\xBC\x3B\xB6\x04\xD0\xBC\xF9\x49\xF5\x9B\xF7\x85\xC5\x36\xB6\xCB\xBC\xF8\xC8\x39\xD5\xE4\x5F\x07\xBD\x15\x54\x97\x74\xCA\xCA\xED\x4F\xBA\xBA\x64\x76\x9F\x81\xB8\x84\x45\x49\x4C\x8D\x6F\xA2\xEB\xB1\xCC\xD1\xC3\x94\xDA\x44\xC2\xE6\xE2\xEA\x18\xE8\xA2\x1F\x27\x05\xBA\xD7\xE5\xD6\xA9\xCD\xDD\xEF\x76\x98\x8D\x00\x0E\xCD\x1B\xFA\x03\xB7\x8E\x80\x58\x0E\x27\x3F\x52\xFB\x94\xA2\xCA\x5E\x65\xC9\xD6\x84\xDA\xB9\x35\x71\xF3\x26\xC0\x4F\x77\xE6\x81\x27\xD2\x77\x3B\x9A\x14\x6F\x79\xF4\xF6\xD0\xE1\xD3\x94\xBA\xD0\x57\x51\xBD\x27\x05\x0D\xC1\xFD\xC8\x12\x30\xEE\x6F\x8D\x11\x2B\x08\x9D\xD4\xD4\xBF\x80\x45\x14\x9A\x88\x44\xDA\x30\xEA\xB4\xA7\xE3\xEE\xEF\x5B\x82\xD5\x3E\xD6\xAD\x78\x92\xDB\x5C\x3C\xF3\xD8\xAD\xFA\xB8\x6B\x7F\xC4\x36\x28\xB6\x02\x15\x8A\x54\x2C\x9C\xB0\x17\x73\x8E\xD0\x37\xA3\x14\x3C\x98\x95\x00\x0C\x29\x05\x5B\x9E\x49\x49\xB1\x5F\xC7\xE3\xCB\xCF\x27\x65\x8E\x35\x17\xB7\x57\xC8\x30\xD9\x41\x5B\xB9\x14\xB6\xE8\xC2\x0F\x94\x31\xA7\x94\x98\xCC\x6A\xEB\xB5\xE1\x27\xF5\x10\xA8\x01\xE8\x8E\x12\x62\xE8\x88\xCC\xB5\x7F\x46\x97\xC0\x9B\x10\x66\x38\x1A\x36\x46\x5F\x22\x68\x3D\xDF\xC9\xC6\x13\x27\xAB\x53\x06\xAC\xA2\x3C\x86\x06\x65\x6F\xB1\x7E\xB1\x29\x44\x9A\xA3\xBA\x49\x69\x28\x69\x8F\xD7\xE5\x5F\xAD\x04\x86\x64\x6F\x1A\xA0\x0C\xC5\x08\x62\xCE\x80\xA3\xD0\xF3\xEC\x68\xDE\xBE\x33\xC7\x17\x5B\x7F\x80\xC4\x4C\x4C\xB1\xA6\x84\x8A\xC3\x3B\xB8\x09\xCD\x14\x81\xBA\x18\xE3\x54\x57\x36\xFE\xDB\x2F\x7C\x47\xA1\x3A\x33\xC8\xF9\x58\x3B\x44\x4F\xB1\xCA\x02\x89\x04\x96\x28\x68\xC5\x4B\xB8\x26\x89\xBB\xD6\x33\x2F\x50\xD5\xFE\x9A\x89\xBA\x18\x32\x92\x54\xC6\x5B\xE0\x9D\xF9\x5E\xE5\x0D\x22\x9B\xF6\xDA\xE2\xC8\x21\xB2\x62\x21\xAA\x86\x40\xB2\x2E\x64\xD3\x5F\xC8\xE3\x7E\x11\x67\x45\x1F\x05\xFE\xE3\xA2\xEF\xB3\xA8\xB3\xF3\x7D\x8F\xF8\x0C\x1F\x22\x1F\x2D\x70\xB4\xB8\x01\x34\x76\x30\x00\xE5\x23\x78\xA7\x56\xD7\x50\x1F\x8A\xFB\x06\xF5\xC2\x19\xF0\xD0"),
   // "Hellenic Academic and Research Institutions RootCA 2015"
   CertificateDer::from_slice(b"\x30\x82\x06\x0B\x30\x82\x03\xF3\xA0\x03\x02\x01\x02\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\xA6\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x0F\x30\x0D\x06\x03\x55\x04\x07\x13\x06\x41\x74\x68\x65\x6E\x73\x31\x44\x30\x42\x06\x03\x55\x04\x0A\x13\x3B\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x65\x72\x74\x2E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x40\x30\x3E\x06\x03\x55\x04\x03\x13\x37\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x52\x6F\x6F\x74\x43\x41\x20\x32\x30\x31\x35\x30\x1E\x17\x0D\x31\x35\x30\x37\x30\x37\x31\x30\x31\x31\x32\x31\x5A\x17\x0D\x34\x30\x30\x36\x33\x30\x31\x30\x31\x31\x32\x31\x5A\x30\x81\xA6\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x0F\x30\x0D\x06\x03\x55\x04\x07\x13\x06\x41\x74\x68\x65\x6E\x73\x31\x44\x30\x42\x06\x03\x55\x04\x0A\x13\x3B\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x65\x72\x74\x2E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x40\x30\x3E\x06\x03\x55\x04\x03\x13\x37\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x52\x6F\x6F\x74\x43\x41\x20\x32\x30\x31\x35\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC2\xF8\xA9\x3F\x1B\x89\xFC\x3C\x3C\x04\x5D\x3D\x90\x36\xB0\x91\x3A\x79\x3C\x66\x5A\xEF\x6D\x39\x01\x49\x1A\xB4\xB7\xCF\x7F\x4D\x23\x53\xB7\x90\x00\xE3\x13\x2A\x28\xA6\x31\xF1\x91\x00\xE3\x28\xEC\xAE\x21\x41\xCE\x1F\xDA\xFD\x7D\x12\x5B\x01\x83\x0F\xB9\xB0\x5F\x99\xE1\xF2\x12\x83\x80\x4D\x06\x3E\xDF\xAC\xAF\xE7\xA1\x88\x6B\x31\xAF\xF0\x8B\xD0\x18\x33\xB8\xDB\x45\x6A\x34\xF4\x02\x80\x24\x28\x0A\x02\x15\x95\x5E\x76\x2A\x0D\x99\x3A\x14\x5B\xF6\xCB\xCB\x53\xBC\x13\x4D\x01\x88\x37\x94\x25\x1B\x42\xBC\x22\xD8\x8E\xA3\x96\x5E\x3A\xD9\x32\xDB\x3E\xE8\xF0\x10\x65\xED\x74\xE1\x2F\xA7\x7C\xAF\x27\x34\xBB\x29\x7D\x9B\xB6\xCF\x09\xC8\xE5\xD3\x0A\xFC\x88\x65\x65\x74\x0A\xDC\x73\x1C\x5C\xCD\x40\xB1\x1C\xD4\xB6\x84\x8C\x4C\x50\xCF\x68\x8E\xA8\x59\xAE\xC2\x27\x4E\x82\xA2\x35\xDD\x14\xF4\x1F\xFF\xB2\x77\xD5\x87\x2F\xAA\x6E\x7D\x24\x27\xE7\xC6\xCB\x26\xE6\xE5\xFE\x67\x07\x63\xD8\x45\x0D\xDD\x3A\x59\x65\x39\x58\x7A\x92\x99\x72\x3D\x9C\x84\x5E\x88\x21\xB8\xD5\xF4\x2C\xFC\xD9\x70\x52\x4F\x78\xB8\xBD\x3C\x2B\x8B\x95\x98\xF5\xB3\xD1\x68\xCF\x20\x14\x7E\x4C\x5C\x5F\xE7\x8B\xE5\xF5\x35\x81\x19\x37\xD7\x11\x08\xB7\x66\xBE\xD3\x4A\xCE\x83\x57\x00\x3A\xC3\x81\xF8\x17\xCB\x92\x36\x5D\xD1\xA3\xD8\x75\x1B\xE1\x8B\x27\xEA\x7A\x48\x41\xFD\x45\x19\x06\xAD\x27\x99\x4E\xC1\x70\x47\xDD\xB5\x9F\x81\x53\x12\xE5\xB1\x8C\x48\x5D\x31\x43\x17\xE3\x8C\xC6\x7A\x63\x96\x4B\x29\x30\x4E\x84\x4E\x62\x19\x5E\x3C\xCE\x97\x90\xA5\x7F\x01\xEB\x9D\xE0\xF8\x8B\x89\xDD\x25\x98\x3D\x92\xB6\x7E\xEF\xD9\xF1\x51\x51\x7D\x2D\x26\xC8\x69\x59\x61\xE0\xAC\x6A\xB8\x2A\x36\x11\x04\x7A\x50\xBD\x32\x84\xBE\x2F\xDC\x72\xD5\xD7\x1D\x16\x47\xE4\x47\x66\x20\x3F\xF4\x96\xC5\xAF\x8E\x01\x7A\xA5\x0F\x7A\x64\xF5\x0D\x18\x87\xD9\xAE\x88\xD5\xFA\x84\xC1\x3A\xC0\x69\x28\x2D\xF2\x0D\x68\x51\xAA\xE3\xA5\x77\xC6\xA4\x90\x0E\xA1\x37\x8B\x31\x23\x47\xC1\x09\x08\xEB\x6E\xF7\x78\x9B\xD7\x82\xFC\x84\x20\x99\x49\x19\xB6\x12\x46\xB1\xFB\x45\x55\x16\xA9\xA3\x65\xAC\x9C\x07\x0F\xEA\x6B\xDC\x1F\x2E\x06\x72\xEC\x86\x88\x12\xE4\x2D\xDB\x5F\x05\x2F\xE4\xF0\x03\xD3\x26\x33\xE7\x80\xC2\xCD\x42\xA1\x17\x34\x0B\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x71\x15\x67\xC8\xC8\xC9\xBD\x75\x5D\x72\xD0\x38\x18\x6A\x9D\xF3\x71\x24\x54\x0B\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x75\xBB\x6D\x54\x4B\xAA\x10\x58\x46\x34\xF2\x62\xD7\x16\x36\x5D\x08\x5E\xD5\x6C\xC8\x87\xBD\xB4\x2E\x46\xF2\x31\xF8\x7C\xEA\x42\xB5\x93\x16\x55\xDC\xA1\x0C\x12\xA0\xDA\x61\x7E\x0F\x58\x58\x73\x64\x72\xC7\xE8\x45\x8E\xDC\xA9\xF2\x26\x3F\xC6\x79\x8C\xB1\x53\x08\x33\x81\xB0\x56\x13\xBE\xE6\x51\x5C\xD8\x9B\x0A\x4F\x4B\x9C\x56\x53\x02\xE9\x4F\xF6\x0D\x60\xEA\x4D\x42\x55\xE8\x7C\x1B\x21\x21\xD3\x1B\x3A\xCC\x77\xF2\xB8\x90\xF1\x68\xC7\xF9\x5A\xFE\xFA\x2D\xF4\xBF\xC9\xF5\x45\x1B\xCE\x38\x10\x2A\x37\x8A\x79\xA3\xB4\xE3\x09\x6C\x85\x86\x93\xFF\x89\x96\x27\x78\x81\x8F\x67\xE3\x46\x74\x54\x8E\xD9\x0D\x69\xE2\x4A\xF4\x4D\x74\x03\xFF\xB2\x77\xED\x95\x67\x97\xE4\xB1\xC5\xAB\xBF\x6A\x23\xE8\xD4\x94\xE2\x44\x28\x62\xC4\x4B\xE2\xF0\xD8\xE2\x29\x6B\x1A\x70\x7E\x24\x61\x93\x7B\x4F\x03\x32\x25\x0D\x45\x24\x2B\x96\xB4\x46\x6A\xBF\x4A\x0B\xF7\x9A\x8F\xC1\xAC\x1A\xC5\x67\xF3\x6F\x34\xD2\xFA\x73\x63\x8C\xEF\x16\xB0\xA8\xA4\x46\x2A\xF8\xEB\x12\xEC\x72\xB4\xEF\xF8\x2B\x7E\x8C\x52\xC0\x8B\x84\x54\xF9\x2F\x3E\xE3\x55\xA8\xDC\x66\xB1\xD9\xE1\x5F\xD8\xB3\x8C\x59\x34\x59\xA4\xAB\x4F\x6C\xBB\x1F\x18\xDB\x75\xAB\xD8\xCB\x92\xCD\x94\x38\x61\x0E\x07\x06\x1F\x4B\x46\x10\xF1\x15\xBE\x8D\x85\x5C\x3B\x4A\x2B\x81\x79\x0F\xB4\x69\x9F\x49\x50\x97\x4D\xF7\x0E\x56\x5D\xC0\x95\x6A\xC2\x36\xC3\x1B\x68\xC9\xF5\x2A\xDC\x47\x9A\xBE\xB2\xCE\xC5\x25\xE8\xFA\x03\xB9\xDA\xF9\x16\x6E\x91\x84\xF5\x1C\x28\xC8\xFC\x26\xCC\xD7\x1C\x90\x56\xA7\x5F\x6F\x3A\x04\xBC\xCD\x78\x89\x0B\x8E\x0F\x2F\xA3\xAA\x4F\xA2\x1B\x12\x3D\x16\x08\x40\x0F\xF1\x46\x4C\xD7\xAA\x7B\x08\xC1\x0A\xF5\x6D\x27\xDE\x02\x8F\xCA\xC3\xB5\x2B\xCA\xE9\xEB\xC8\x21\x53\x38\xA5\xCC\x3B\xD8\x77\x37\x30\xA2\x4F\xD9\x6F\xD1\xF2\x40\xAD\x41\x7A\x17\xC5\xD6\x4A\x35\x89\xB7\x41\xD5\x7C\x86\x7F\x55\x4D\x83\x4A\xA5\x73\x20\xC0\x3A\xAF\x90\xF1\x9A\x24\x8E\xD9\x8E\x71\xCA\x7B\xB8\x86\xDA\xB2\x8F\x99\x3E\x1D\x13\x0D\x12\x11\xEE\xD4\xAB\xF0\xE9\x15\x76\x02\xE4\xE0\xDF\xAA\x20\x1E\x5B\x61\x85\x64\x40\xA9\x90\x97\x0D\xAD\x53\xD2\x5A\x1D\x87\x6A\x00\x97\x65\x62\xB4\xBE\x6F\x6A\xA7\xF5\x2C\x42\xED\x32\xAD\xB6\x21\x9E\xBE\xBC"),
   // "SZAFIR ROOT CA2"
   CertificateDer::from_slice(b"\x30\x82\x03\x72\x30\x82\x02\x5A\xA0\x03\x02\x01\x02\x02\x14\x3E\x8A\x5D\x07\xEC\x55\xD2\x32\xD5\xB7\xE3\xB6\x5F\x01\xEB\x2D\xDC\xE4\xD6\xE4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x28\x30\x26\x06\x03\x55\x04\x0A\x0C\x1F\x4B\x72\x61\x6A\x6F\x77\x61\x20\x49\x7A\x62\x61\x20\x52\x6F\x7A\x6C\x69\x63\x7A\x65\x6E\x69\x6F\x77\x61\x20\x53\x2E\x41\x2E\x31\x18\x30\x16\x06\x03\x55\x04\x03\x0C\x0F\x53\x5A\x41\x46\x49\x52\x20\x52\x4F\x4F\x54\x20\x43\x41\x32\x30\x1E\x17\x0D\x31\x35\x31\x30\x31\x39\x30\x37\x34\x33\x33\x30\x5A\x17\x0D\x33\x35\x31\x30\x31\x39\x30\x37\x34\x33\x33\x30\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x28\x30\x26\x06\x03\x55\x04\x0A\x0C\x1F\x4B\x72\x61\x6A\x6F\x77\x61\x20\x49\x7A\x62\x61\x20\x52\x6F\x7A\x6C\x69\x63\x7A\x65\x6E\x69\x6F\x77\x61\x20\x53\x2E\x41\x2E\x31\x18\x30\x16\x06\x03\x55\x04\x03\x0C\x0F\x53\x5A\x41\x46\x49\x52\x20\x52\x4F\x4F\x54\x20\x43\x41\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xB7\xBC\x3E\x50\xA8\x4B\xCD\x40\xB5\xCE\x61\xE7\x96\xCA\xB4\xA1\xDA\x0C\x22\xB0\xFA\xB5\x7B\x76\x00\x77\x8C\x0B\xCF\x7D\xA8\x86\xCC\x26\x51\xE4\x20\x3D\x85\x0C\xD6\x58\xE3\xE7\xF4\x2A\x18\x9D\xDA\xD1\xAE\x26\xEE\xEB\x53\xDC\xF4\x90\xD6\x13\x4A\x0C\x90\x3C\xC3\xF4\xDA\xD2\x8E\x0D\x92\x3A\xDC\xB1\xB1\xFF\x38\xDE\xC3\xBA\x2D\x5F\x80\xB9\x02\xBD\x4A\x9D\x1B\x0F\xB4\xC3\xC2\xC1\x67\x03\xDD\xDC\x1B\x9C\x3D\xB3\xB0\xDE\x00\x1E\xA8\x34\x47\xBB\x9A\xEB\xFE\x0B\x14\xBD\x36\x84\xDA\x0D\x20\xBF\xFA\x5B\xCB\xA9\x16\x20\xAD\x39\x60\xEE\x2F\x75\xB6\xE7\x97\x9C\xF9\x3E\xFD\x7E\x4D\x6F\x4D\x2F\xEF\x88\x0D\x6A\xFA\xDD\xF1\x3D\x6E\x20\xA5\xA0\x12\xB4\x4D\x70\xB9\xCE\xD7\x72\x3B\x89\x93\xA7\x80\x84\x1C\x27\x49\x72\x49\xB5\xFF\x3B\x95\x9E\xC1\xCC\xC8\x01\xEC\xE8\x0E\x8A\x0A\x96\xE7\xB3\xA6\x87\xE5\xD6\xF9\x05\x2B\x0D\x97\x40\x70\x3C\xBA\xAC\x75\x5A\x9C\xD5\x4D\x9D\x02\x0A\xD2\x4B\x9B\x66\x4B\x46\x07\x17\x65\xAD\x9F\x6C\x88\x00\xDC\x22\x89\xE0\xE1\x64\xD4\x67\xBC\x31\x79\x61\x3C\xBB\xCA\x41\xCD\x5C\x6A\x00\xC8\x3C\x38\x8E\x58\xAF\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x2E\x16\xA9\x4A\x18\xB5\xCB\xCC\xF5\x6F\x50\xF3\x23\x5F\xF8\x5D\xE7\xAC\xF0\xC8\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\xB5\x73\xF8\x03\xDC\x59\x5B\x1D\x76\xE9\xA3\x2A\x7B\x90\x28\xB2\x4D\xC0\x33\x4F\xAA\x9A\xB1\xD4\xB8\xE4\x27\xFF\xA9\x96\x99\xCE\x46\xE0\x6D\x7C\x4C\xA2\x38\xA4\x06\x70\xF0\xF4\x41\x11\xEC\x3F\x47\x8D\x3F\x72\x87\xF9\x3B\xFD\xA4\x6F\x2B\x53\x00\xE0\xFF\x39\xB9\x6A\x07\x0E\xEB\x1D\x1C\xF6\xA2\x72\x90\xCB\x82\x3D\x11\x82\x8B\xD2\xBB\x9F\x2A\xAF\x21\xE6\x63\x86\x9D\x79\x19\xEF\xF7\xBB\x0C\x35\x90\xC3\x8A\xED\x4F\x0F\xF5\xCC\x12\xD9\xA4\x3E\xBB\xA0\xFC\x20\x95\x5F\x4F\x26\x2F\x11\x23\x83\x4E\x75\x07\x0F\xBF\x9B\xD1\xB4\x1D\xE9\x10\x04\xFE\xCA\x60\x8F\xA2\x4C\xB8\xAD\xCF\xE1\x90\x0F\xCD\xAE\x0A\xC7\x5D\x7B\xB7\x50\xD2\xD4\x61\xFA\xD5\x15\xDB\xD7\x9F\x87\x51\x54\xEB\xA5\xE3\xEB\xC9\x85\xA0\x25\x20\x37\xFB\x8E\xCE\x0C\x34\x84\xE1\x3C\x81\xB2\x77\x4E\x43\xA5\x88\x5F\x86\x67\xA1\x3D\xE6\xB4\x5C\x61\xB6\x3E\xDB\xFE\xB7\x28\xC5\xA2\x07\xAE\xB5\xCA\xCA\x8D\x2A\x12\xEF\x97\xED\xC2\x30\xA4\xC9\x2A\x7A\xFB\xF3\x4D\x23\x1B\x99\x33\x34\xA0\x2E\xF5\xA9\x0B\x3F\xD4\x5D\xE1\xCF\x84\x9F\xE2\x19\xC2\x5F\x8A\xD6\x20\x1E\xE3\x73\xB7"),
   // "GlobalSign"
   CertificateDer::from_slice(b"\x30\x82\x01\xDC\x30\x82\x01\x83\xA0\x03\x02\x01\x02\x02\x0D\x02\x03\xE5\x7E\xF5\x3F\x93\xFD\xA5\x09\x21\xB2\xA6\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x50\x31\x24\x30\x22\x06\x03\x55\x04\x0B\x13\x1B\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x34\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x1E\x17\x0D\x31\x32\x31\x31\x31\x33\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x39\x30\x33\x31\x34\x30\x37\x5A\x30\x50\x31\x24\x30\x22\x06\x03\x55\x04\x0B\x13\x1B\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x34\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x59\x30\x13\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x08\x2A\x86\x48\xCE\x3D\x03\x01\x07\x03\x42\x00\x04\xB8\xC6\x79\xD3\x8F\x6C\x25\x0E\x9F\x2E\x39\x19\x1C\x03\xA4\xAE\x9A\xE5\x39\x07\x09\x16\xCA\x63\xB1\xB9\x86\xF8\x8A\x57\xC1\x57\xCE\x42\xFA\x73\xA1\xF7\x65\x42\xFF\x1E\xC1\x00\xB2\x6E\x73\x0E\xFF\xC7\x21\xE5\x18\xA4\xAA\xD9\x71\x3F\xA8\xD4\xB9\xCE\x8C\x1D\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x54\xB0\x7B\xAD\x45\xB8\xE2\x40\x7F\xFB\x0A\x6E\xFB\xBE\x33\xC9\x3C\xA3\x84\xD5\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x47\x00\x30\x44\x02\x20\x22\x4F\x74\x72\xB9\x60\xAF\xF1\xE6\x9C\xA0\x16\x05\x50\x5F\xC3\x5E\x3B\x6E\x61\x74\xEF\xBE\x01\xC4\xBE\x18\x48\x59\x61\x82\x32\x02\x20\x26\x9D\x54\x63\x40\xDE\x37\x60\x50\xCF\xC8\xD8\xED\x9D\x82\xAE\x37\x98\xBC\xA3\x8F\x4C\x4C\xA9\x34\x2B\x6C\xEF\xFB\x95\x9B\x26"),
   // "Atos TrustedRoot Root CA ECC TLS 2021"
   CertificateDer::from_slice(b"\x30\x82\x02\x15\x30\x82\x01\x9B\xA0\x03\x02\x01\x02\x02\x10\x3D\x98\x3B\xA6\x66\x3D\x90\x63\xF7\x7E\x26\x57\x38\x04\xEF\x00\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x4C\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x0C\x25\x41\x74\x6F\x73\x20\x54\x72\x75\x73\x74\x65\x64\x52\x6F\x6F\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x45\x43\x43\x20\x54\x4C\x53\x20\x32\x30\x32\x31\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x74\x6F\x73\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x30\x1E\x17\x0D\x32\x31\x30\x34\x32\x32\x30\x39\x32\x36\x32\x33\x5A\x17\x0D\x34\x31\x30\x34\x31\x37\x30\x39\x32\x36\x32\x32\x5A\x30\x4C\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x0C\x25\x41\x74\x6F\x73\x20\x54\x72\x75\x73\x74\x65\x64\x52\x6F\x6F\x74\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x45\x43\x43\x20\x54\x4C\x53\x20\x32\x30\x32\x31\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x74\x6F\x73\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x96\x86\x58\x28\x37\x0A\x67\xD0\xA0\xDE\x24\x19\x19\xE1\xE4\x05\x07\x1F\x97\xED\xE8\x64\x82\xB9\xF6\xC4\x71\x50\xCE\x8A\x0C\xFF\xD7\xB5\x76\xBB\xA1\x6C\x93\x6C\x83\xA2\x68\x6E\xA5\xD9\xBE\x2C\x88\x95\x41\xCD\x5D\xDD\xB1\xCA\x83\x63\x83\xCC\xC0\xBE\x74\xD9\xE0\x9D\xA4\xEE\x4A\x4E\x56\xE0\x98\x29\x41\x93\x52\x10\xD5\x24\x38\x02\x32\x67\xF1\x94\x12\x6F\xEF\xD7\xC5\xDE\x2E\xFD\x19\x80\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x76\x28\x25\xD6\x7D\xE0\x66\x9A\x7A\x09\xB2\x6A\x3B\x8E\x33\xD7\x36\xD3\x4F\xA2\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x5B\x99\x29\xF3\x9C\x31\xB6\x89\x6B\x6C\xD6\xBD\x77\xE1\x7C\xE7\x51\x7E\xB8\x3A\xCD\xA3\x36\x5F\x7C\xF7\x3C\x77\x3E\xE4\x50\xAD\xA8\xE7\xD2\x59\x0C\x26\x8E\x30\x3B\x6E\x08\x2A\xC2\xA7\x5A\xC8\x02\x31\x00\x99\xE3\x0C\xE7\xA3\xC3\xAF\xD3\x49\x2E\x46\x82\x23\x66\x5D\xC9\x00\x14\x12\xFD\x38\xF4\xE1\x98\x6B\x77\x29\x7A\xDB\x24\xCF\x65\x40\xBF\xD2\xDC\x8C\x11\xE8\xF4\x7D\x7F\x20\x84\xA9\x42\xE4\x28"),
   // "Certainly Root E1"
   CertificateDer::from_slice(b"\x30\x82\x01\xF7\x30\x82\x01\x7D\xA0\x03\x02\x01\x02\x02\x10\x06\x25\x33\xB1\x47\x03\x33\x27\x5C\xF9\x8D\x9A\xB9\xBF\xCC\xF8\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x3D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x31\x1A\x30\x18\x06\x03\x55\x04\x03\x13\x11\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x20\x52\x6F\x6F\x74\x20\x45\x31\x30\x1E\x17\x0D\x32\x31\x30\x34\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x34\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x30\x3D\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x13\x09\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x31\x1A\x30\x18\x06\x03\x55\x04\x03\x13\x11\x43\x65\x72\x74\x61\x69\x6E\x6C\x79\x20\x52\x6F\x6F\x74\x20\x45\x31\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xDE\x6F\xF8\x7F\x1C\xDF\xED\xF9\x47\x87\x86\xB1\xA4\xC0\x8A\xF8\x82\x97\x80\xEA\x8F\xC8\x4A\x5E\x2A\x7D\x88\x68\xA7\x01\x62\x14\x91\x24\x7A\x5C\x9E\xA3\x17\x7D\x8A\x86\x21\x34\x18\x50\x1B\x10\xDE\xD0\x37\x4B\x26\xC7\x19\x60\x80\xE9\x34\xBD\x60\x19\x36\x40\xD6\x29\x87\x09\x3C\x91\x7A\xF6\xBC\x13\x23\xDD\x59\x4E\x04\x5E\xCF\xC8\x02\x1C\x18\x53\xC1\x31\xD8\xDA\x20\xE9\x44\x8D\xE4\x76\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF3\x28\x18\xCB\x64\x75\xEE\x29\x2A\xEB\xED\xAE\x23\x58\x38\x85\xEB\xC8\x22\x07\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xB1\x8E\x5A\x20\xC3\xB2\x19\x62\x4D\xDE\xB0\x4F\xDF\x6E\xD2\x70\x8A\xF1\x9F\x7E\x6A\x8C\xE6\xBA\xDE\x83\x69\xCA\x69\xB3\xA9\x05\xB5\x96\x92\x17\x87\xC2\xD2\xEA\xD0\x7B\xCE\xD8\x41\x5B\x7C\xAE\x02\x30\x46\xDE\xEA\xCB\x5D\x9A\xEC\x32\xC2\x65\x16\xB0\x4C\x30\x5C\x30\xF3\xDA\x4E\x73\x86\x06\xD8\xCE\x89\x04\x48\x37\x37\xF8\xDD\x33\x51\x9D\x70\xAF\x7B\x55\xD8\x01\x2E\x7D\x05\x64\x0E\x86\xB8\x91"),
   // "Certum Trusted Network CA 2"
   CertificateDer::from_slice(b"\x30\x82\x05\xD2\x30\x82\x03\xBA\xA0\x03\x02\x01\x02\x02\x10\x21\xD6\xD0\x4A\x4F\x25\x0F\xC9\x32\x37\xFC\xAA\x5E\x12\x8D\xE9\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x30\x81\x80\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x55\x6E\x69\x7A\x65\x74\x6F\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x43\x65\x72\x74\x75\x6D\x20\x54\x72\x75\x73\x74\x65\x64\x20\x4E\x65\x74\x77\x6F\x72\x6B\x20\x43\x41\x20\x32\x30\x22\x18\x0F\x32\x30\x31\x31\x31\x30\x30\x36\x30\x38\x33\x39\x35\x36\x5A\x18\x0F\x32\x30\x34\x36\x31\x30\x30\x36\x30\x38\x33\x39\x35\x36\x5A\x30\x81\x80\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x55\x6E\x69\x7A\x65\x74\x6F\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x24\x30\x22\x06\x03\x55\x04\x03\x13\x1B\x43\x65\x72\x74\x75\x6D\x20\x54\x72\x75\x73\x74\x65\x64\x20\x4E\x65\x74\x77\x6F\x72\x6B\x20\x43\x41\x20\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xBD\xF9\x78\xF8\xE6\xD5\x80\x0C\x64\x9D\x86\x1B\x96\x64\x67\x3F\x22\x3A\x1E\x75\x01\x7D\xEF\xFB\x5C\x67\x8C\xC9\xCC\x5C\x6B\xA9\x91\xE6\xB9\x42\xE5\x20\x4B\x9B\xDA\x9B\x7B\xB9\x99\x5D\xD9\x9B\x80\x4B\xD7\x84\x40\x2B\x27\xD3\xE8\xBA\x30\xBB\x3E\x09\x1A\xA7\x49\x95\xEF\x2B\x40\x24\xC2\x97\xC7\xA7\xEE\x9B\x25\xEF\xA8\x0A\x00\x97\x85\x5A\xAA\x9D\xDC\x29\xC9\xE2\x35\x07\xEB\x70\x4D\x4A\xD6\xC1\xB3\x56\xB8\xA1\x41\x38\x9B\xD1\xFB\x31\x7F\x8F\xE0\x5F\xE1\xB1\x3F\x0F\x8E\x16\x49\x60\xD7\x06\x8D\x18\xF9\xAA\x26\x10\xAB\x2A\xD3\xD0\xD1\x67\x8D\x1B\x46\xBE\x47\x30\xD5\x2E\x72\xD1\xC5\x63\xDA\xE7\x63\x79\x44\x7E\x4B\x63\x24\x89\x86\x2E\x34\x3F\x29\x4C\x52\x8B\x2A\xA7\xC0\xE2\x91\x28\x89\xB9\xC0\x5B\xF9\x1D\xD9\xE7\x27\xAD\xFF\x9A\x02\x97\xC1\xC6\x50\x92\x9B\x02\x2C\xBD\xA9\xB9\x34\x59\x0A\xBF\x84\x4A\xFF\xDF\xFE\xB3\x9F\xEB\xD9\x9E\xE0\x98\x23\xEC\xA6\x6B\x77\x16\x2A\xDB\xCC\xAD\x3B\x1C\xA4\x87\xDC\x46\x73\x5E\x19\x62\x68\x45\x57\xE4\x90\x82\x42\xBB\x42\xD6\xF0\x61\xE0\xC1\xA3\x3D\x66\xA3\x5D\xF4\x18\xEE\x88\xC9\x8D\x17\x45\x29\x99\x32\x75\x02\x31\xEE\x29\x26\xC8\x6B\x02\xE6\xB5\x62\x45\x7F\x37\x15\x5A\x23\x68\x89\xD4\x3E\xDE\x4E\x27\xB0\xF0\x40\x0C\xBC\x4D\x17\xCB\x4D\xA2\xB3\x1E\xD0\x06\x5A\xDD\xF6\x93\xCF\x57\x75\x99\xF5\xFA\x86\x1A\x67\x78\xB3\xBF\x96\xFE\x34\xDC\xBD\xE7\x52\x56\xE5\xB3\xE5\x75\x7B\xD7\x41\x91\x05\xDC\x5D\x69\xE3\x95\x0D\x43\xB9\xFC\x83\x96\x39\x95\x7B\x6C\x80\x5A\x4F\x13\x72\xC6\xD7\x7D\x29\x7A\x44\xBA\x52\xA4\x2A\xD5\x41\x46\x09\x20\xFE\x22\xA0\xB6\x5B\x30\x8D\xBC\x89\x0C\xD5\xD7\x70\xF8\x87\x52\xFD\xDA\xEF\xAC\x51\x2E\x07\xB3\x4E\xFE\xD0\x09\xDA\x70\xEF\x98\xFA\x56\xE6\x6D\xDB\xB5\x57\x4B\xDC\xE5\x2C\x25\x15\xC8\x9E\x2E\x78\x4E\xF8\xDA\x9C\x9E\x86\x2C\xCA\x57\xF3\x1A\xE5\xC8\x92\x8B\x1A\x82\x96\x7A\xC3\xBC\x50\x12\x69\xD8\x0E\x5A\x46\x8B\x3A\xEB\x26\xFA\x23\xC9\xB6\xB0\x81\xBE\x42\x00\xA4\xF8\xD6\xFE\x30\x2E\xC7\xD2\x46\xF6\xE5\x8E\x75\xFD\xF2\xCC\xB9\xD0\x87\x5B\xCC\x06\x10\x60\xBB\x83\x35\xB7\x5E\x67\xDE\x47\xEC\x99\x48\xF1\xA4\xA1\x15\xFE\xAD\x8C\x62\x8E\x39\x55\x4F\x39\x16\xB9\xB1\x63\x9D\xFF\xB7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB6\xA1\x54\x39\x02\xC3\xA0\x3F\x8E\x8A\xBC\xFA\xD4\xF8\x1C\xA6\xD1\x3A\x0E\xFD\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x03\x82\x02\x01\x00\x71\xA5\x0E\xCE\xE4\xE9\xBF\x3F\x38\xD5\x89\x5A\xC4\x02\x61\xFB\x4C\xC5\x14\x17\x2D\x8B\x4F\x53\x6B\x10\x17\xFC\x65\x84\xC7\x10\x49\x90\xDE\xDB\xC7\x26\x93\x88\x26\x6F\x70\xD6\x02\x5E\x39\xA0\xF7\x8F\xAB\x96\xB5\xA5\x13\x5C\x81\x14\x6D\x0E\x81\x82\x11\x1B\x8A\x4E\xC6\x4F\xA5\xDD\x62\x1E\x44\xDF\x09\x59\xF4\x5B\x77\x0B\x37\xE9\x8B\x20\xC6\xF8\x0A\x4E\x2E\x58\x1C\xEB\x33\xD0\xCF\x86\x60\xC9\xDA\xFB\x80\x2F\x9E\x4C\x60\x84\x78\x3D\x21\x64\xD6\xFB\x41\x1F\x18\x0F\xE7\xC9\x75\x71\xBD\xBD\x5C\xDE\x34\x87\x3E\x41\xB0\x0E\xF6\xB9\xD6\x3F\x09\x13\x96\x14\x2F\xDE\x9A\x1D\x5A\xB9\x56\xCE\x35\x3A\xB0\x5F\x70\x4D\x5E\xE3\x29\xF1\x23\x28\x72\x59\xB6\xAB\xC2\x8C\x66\x26\x1C\x77\x2C\x26\x76\x35\x8B\x28\xA7\x69\xA0\xF9\x3B\xF5\x23\xDD\x85\x10\x74\xC9\x90\x03\x56\x91\xE7\xAF\xBA\x47\xD4\x12\x97\x11\x22\xE3\xA2\x49\x94\x6C\xE7\xB7\x94\x4B\xBA\x2D\xA4\xDA\x33\x8B\x4C\xA6\x44\xFF\x5A\x3C\xC6\x1D\x64\xD8\xB5\x31\xE4\xA6\x3C\x7A\xA8\x57\x0B\xDB\xED\x61\x1A\xCB\xF1\xCE\x73\x77\x63\xA4\x87\x6F\x4C\x51\x38\xD6\xE4\x5F\xC7\x9F\xB6\x81\x2A\xE4\x85\x48\x79\x58\x5E\x3B\xF8\xDB\x02\x82\x67\xC1\x39\xDB\xC3\x74\x4B\x3D\x36\x1E\xF9\x29\x93\x88\x68\x5B\xA8\x44\x19\x21\xF0\xA7\xE8\x81\x0D\x2C\xE8\x93\x36\xB4\x37\xB2\xCA\xB0\x1B\x26\x7A\x9A\x25\x1F\x9A\x9A\x80\x9E\x4B\x2A\x3F\xFB\xA3\x9A\xFE\x73\x32\x71\xC2\x9E\xC6\x72\xE1\x8A\x68\x27\xF1\xE4\x0F\xB4\xC4\x4C\xA5\x61\x93\xF8\x97\x10\x07\x2A\x30\x25\xA9\xB9\xC8\x71\xB8\xEF\x68\xCC\x2D\x7E\xF5\xE0\x7E\x0F\x82\xA8\x6F\xB6\xBA\x6C\x83\x43\x77\xCD\x8A\x92\x17\xA1\x9E\x5B\x78\x16\x3D\x45\xE2\x33\x72\xDD\xE1\x66\xCA\x99\xD3\xC9\xC5\x26\xFD\x0D\x68\x04\x46\xAE\xB6\xD9\x9B\x8C\xBE\x19\xBE\xB1\xC6\xF2\x19\xE3\x5C\x02\xCA\x2C\xD8\x6F\x4A\x07\xD9\xC9\x35\xDA\x40\x75\xF2\xC4\xA7\x19\x6F\x9E\x42\x10\x98\x75\xE6\x95\x8B\x60\xBC\xED\xC5\x12\xD7\x8A\xCE\xD5\x98\x5C\x56\x96\x03\xC5\xEE\x77\x06\x35\xFF\xCF\xE4\xEE\x3F\x13\x61\xEE\xDB\xDA\x2D\x85\xF0\xCD\xAE\x9D\xB2\x18\x09\x45\xC3\x92\xA1\x72\x17\xFC\x47\xB6\xA0\x0B\x2C\xF1\xC4\xDE\x43\x68\x08\x6A\x5F\x3B\xF0\x76\x63\xFB\xCC\x06\x2C\xA6\xC6\xE2\x0E\xB5\xB9\xBE\x24\x8F"),
   // "emSign ECC Root CA - C3"
   CertificateDer::from_slice(b"\x30\x82\x02\x2B\x30\x82\x01\xB1\xA0\x03\x02\x01\x02\x02\x0A\x7B\x71\xB6\x82\x56\xB8\x12\x7C\x9C\xA8\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x13\x0B\x65\x4D\x75\x64\x68\x72\x61\x20\x49\x6E\x63\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x65\x6D\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x43\x33\x30\x1E\x17\x0D\x31\x38\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x17\x0D\x34\x33\x30\x32\x31\x38\x31\x38\x33\x30\x30\x30\x5A\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x0B\x13\x0A\x65\x6D\x53\x69\x67\x6E\x20\x50\x4B\x49\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x13\x0B\x65\x4D\x75\x64\x68\x72\x61\x20\x49\x6E\x63\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x65\x6D\x53\x69\x67\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x43\x33\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xFD\xA5\x61\xAE\x7B\x26\x10\x1D\xE9\xB7\x22\x30\xAE\x06\xF4\x81\xB3\xB1\x42\x71\x95\x39\xBC\xD3\x52\xE3\xAF\xAF\xF9\xF2\x97\x35\x92\x36\x46\x0E\x87\x95\x8D\xB9\x39\x5A\xE9\xBB\xDF\xD0\xFE\xC8\x07\x41\x3C\xBB\x55\x6F\x83\xA3\x6A\xFB\x62\xB0\x81\x89\x02\x70\x7D\x48\xC5\x4A\xE3\xE9\x22\x54\x22\x4D\x93\xBB\x42\x0C\xAF\x77\x9C\x23\xA6\x7D\xD7\x61\x11\xCE\x65\xC7\xF8\x7F\xFE\xF5\xF2\xA9\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xFB\x5A\x48\xD0\x80\x20\x40\xF2\xA8\xE9\x00\x07\x69\x19\x77\xA7\xE6\xC3\xF4\xCF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xB4\xD8\x2F\x02\x89\xFD\xB6\x4C\x62\xBA\x43\x4E\x13\x84\x72\xB5\xAE\xDD\x1C\xDE\xD6\xB5\xDC\x56\x8F\x58\x40\x5A\x2D\xDE\x20\x4C\x22\x83\xCA\x93\xA8\x7E\xEE\x12\x40\xC7\xD6\x87\x4F\xF8\xDF\x85\x02\x30\x1C\x14\x64\xE4\x7C\x96\x83\x11\x9C\xB0\xD1\x5A\x61\x4B\xA6\x0F\x49\xD3\x00\xFC\xA1\xFC\xE4\xA5\xFF\x7F\xAD\xD7\x30\xD0\xC7\x77\x7F\xBE\x81\x07\x55\x30\x50\x20\x14\xF5\x57\x38\x0A\xA8\x31\x51"),
   // "TrustAsia Global Root CA G4"
   CertificateDer::from_slice(b"\x30\x82\x02\x55\x30\x82\x01\xDC\xA0\x03\x02\x01\x02\x02\x14\x4F\x23\x64\xB8\x8E\x97\x63\x9E\xC6\x53\x81\xC1\x76\x4E\xCB\x2A\x74\x15\xD6\xD7\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x0C\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x47\x34\x30\x1E\x17\x0D\x32\x31\x30\x35\x32\x30\x30\x32\x31\x30\x32\x32\x5A\x17\x0D\x34\x36\x30\x35\x31\x39\x30\x32\x31\x30\x32\x32\x5A\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x0C\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x47\x34\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xF1\xB3\xCD\x38\xE4\x25\x43\xE5\xDE\x19\x09\xBB\x81\x79\xA2\x15\x5F\x15\x63\x01\xDE\xC2\xAB\xDD\xB3\xA6\x1B\x67\x4B\x80\x83\xAF\x99\xCB\xAC\x17\xDB\x2B\x96\xCA\x7C\x52\x55\xE2\x1A\xE1\x3D\x56\xF0\x2F\x16\x08\xFA\x15\xBC\x9B\xBB\x47\xE6\x3F\xEE\xA8\xE1\x4C\x8C\xF5\xD3\x36\xF9\x38\x5D\xAB\x70\x9A\x47\x0D\xE2\x81\x41\x06\xEB\x49\xF9\xB0\x29\xDD\x33\xEC\x50\xA5\x7F\x79\x29\xB8\x20\x98\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xA5\xBB\x4A\x97\xCE\xB3\x2B\x7F\xA4\x31\xDE\x97\x83\x59\x83\xA6\x6F\x71\xCB\xDE\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xA5\xBB\x4A\x97\xCE\xB3\x2B\x7F\xA4\x31\xDE\x97\x83\x59\x83\xA6\x6F\x71\xCB\xDE\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x67\x00\x30\x64\x02\x30\x5E\xF2\xEB\x06\xCC\x49\x31\x9F\x40\x00\x6D\xB7\x7E\x36\xF0\x4D\x11\x4F\xF3\xCB\x89\x3A\x2C\x78\x91\x50\xA3\x5B\xC0\xCA\x75\x26\xF2\xBF\x90\x5D\x0B\x82\x8C\x60\x28\x9F\xC6\x70\x9A\x68\xE4\xF1\x02\x30\x5C\x58\x0E\x56\x76\xCF\x58\xC3\xD7\x10\x8C\xBA\x8E\xAE\xE3\xBC\x64\x75\x47\xC5\x55\x90\xE3\xFD\xBA\x55\xEB\x07\xC4\x53\xAB\x37\xA9\xEE\x21\xB2\x21\x5B\x60\x8F\x3D\x32\xF1\xD5\x23\x94\xD6\x58"),
   // "e-Szigno Root CA 2017"
   CertificateDer::from_slice(b"\x30\x82\x02\x40\x30\x82\x01\xE5\xA0\x03\x02\x01\x02\x02\x0C\x01\x54\x48\xEF\x21\xFD\x97\x59\x0D\xF5\x04\x0A\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x30\x71\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x55\x31\x11\x30\x0F\x06\x03\x55\x04\x07\x0C\x08\x42\x75\x64\x61\x70\x65\x73\x74\x31\x16\x30\x14\x06\x03\x55\x04\x0A\x0C\x0D\x4D\x69\x63\x72\x6F\x73\x65\x63\x20\x4C\x74\x64\x2E\x31\x17\x30\x15\x06\x03\x55\x04\x61\x0C\x0E\x56\x41\x54\x48\x55\x2D\x32\x33\x35\x38\x34\x34\x39\x37\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x0C\x15\x65\x2D\x53\x7A\x69\x67\x6E\x6F\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x31\x37\x30\x1E\x17\x0D\x31\x37\x30\x38\x32\x32\x31\x32\x30\x37\x30\x36\x5A\x17\x0D\x34\x32\x30\x38\x32\x32\x31\x32\x30\x37\x30\x36\x5A\x30\x71\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x48\x55\x31\x11\x30\x0F\x06\x03\x55\x04\x07\x0C\x08\x42\x75\x64\x61\x70\x65\x73\x74\x31\x16\x30\x14\x06\x03\x55\x04\x0A\x0C\x0D\x4D\x69\x63\x72\x6F\x73\x65\x63\x20\x4C\x74\x64\x2E\x31\x17\x30\x15\x06\x03\x55\x04\x61\x0C\x0E\x56\x41\x54\x48\x55\x2D\x32\x33\x35\x38\x34\x34\x39\x37\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x0C\x15\x65\x2D\x53\x7A\x69\x67\x6E\x6F\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x31\x37\x30\x59\x30\x13\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x08\x2A\x86\x48\xCE\x3D\x03\x01\x07\x03\x42\x00\x04\x96\xDC\x3D\x8A\xD8\xB0\x7B\x6F\xC6\x27\xBE\x44\x90\xB1\xB3\x56\x15\x7B\x8E\x43\x24\x7D\x1A\x84\x59\xEE\x63\x68\xB2\xC6\x5E\x87\xD0\x15\x48\x1E\xA8\x90\xAD\xBD\x53\xA2\xDA\xDE\x3A\x90\xA6\x60\x5F\x68\x32\xB5\x86\x41\xDF\x87\x5B\x2C\x7B\xC5\xFE\x7C\x7A\xDA\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x87\x11\x15\x08\xD1\xAA\xC1\x78\x0C\xB1\xAF\xCE\xC6\xC9\x90\xEF\xBF\x30\x04\xC0\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x87\x11\x15\x08\xD1\xAA\xC1\x78\x0C\xB1\xAF\xCE\xC6\xC9\x90\xEF\xBF\x30\x04\xC0\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x02\x03\x49\x00\x30\x46\x02\x21\x00\xB5\x57\xDD\xD7\x8A\x55\x0B\x36\xE1\x86\x44\xFA\xD4\xD9\x68\x8D\xB8\xDC\x23\x8A\x8A\x0D\xD4\x2F\x7D\xEA\x73\xEC\xBF\x4D\x6C\xA8\x02\x21\x00\xCB\xA5\xB4\x12\xFA\xE7\xB5\xE8\xCF\x7E\x93\xFC\xF3\x35\x8F\x6F\x4E\x5A\x7C\xB4\xBC\x4E\xB2\xFC\x72\xAA\x5B\x59\xF9\xE7\xDC\x31"),
   // "FIRMAPROFESIONAL CA ROOT-A WEB"
   CertificateDer::from_slice(b"\x30\x82\x02\x7A\x30\x82\x02\x00\xA0\x03\x02\x01\x02\x02\x10\x31\x97\x21\xED\xAF\x89\x42\x7F\x35\x41\x87\xA1\x67\x56\x4C\x6D\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x6E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x1C\x30\x1A\x06\x03\x55\x04\x0A\x0C\x13\x46\x69\x72\x6D\x61\x70\x72\x6F\x66\x65\x73\x69\x6F\x6E\x61\x6C\x20\x53\x41\x31\x18\x30\x16\x06\x03\x55\x04\x61\x0C\x0F\x56\x41\x54\x45\x53\x2D\x41\x36\x32\x36\x33\x34\x30\x36\x38\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x46\x49\x52\x4D\x41\x50\x52\x4F\x46\x45\x53\x49\x4F\x4E\x41\x4C\x20\x43\x41\x20\x52\x4F\x4F\x54\x2D\x41\x20\x57\x45\x42\x30\x1E\x17\x0D\x32\x32\x30\x34\x30\x36\x30\x39\x30\x31\x33\x36\x5A\x17\x0D\x34\x37\x30\x33\x33\x31\x30\x39\x30\x31\x33\x36\x5A\x30\x6E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x1C\x30\x1A\x06\x03\x55\x04\x0A\x0C\x13\x46\x69\x72\x6D\x61\x70\x72\x6F\x66\x65\x73\x69\x6F\x6E\x61\x6C\x20\x53\x41\x31\x18\x30\x16\x06\x03\x55\x04\x61\x0C\x0F\x56\x41\x54\x45\x53\x2D\x41\x36\x32\x36\x33\x34\x30\x36\x38\x31\x27\x30\x25\x06\x03\x55\x04\x03\x0C\x1E\x46\x49\x52\x4D\x41\x50\x52\x4F\x46\x45\x53\x49\x4F\x4E\x41\x4C\x20\x43\x41\x20\x52\x4F\x4F\x54\x2D\x41\x20\x57\x45\x42\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x47\x53\xEA\x2C\x11\xA4\x77\xC7\x2A\xEA\xF3\xD6\x5F\x7B\xD3\x04\x91\x5C\xFA\x88\xC6\x22\xB9\x83\x10\x62\x77\x84\x33\x2D\xE9\x03\x88\xD4\xE0\x33\xF7\xED\x77\x2C\x4A\x60\xEA\xE4\x6F\xAD\x6D\xB4\xF8\x4C\x8A\xA4\xE4\x1F\xCA\xEA\x4F\x38\x4A\x2E\x82\x73\x2B\xC7\x66\x9B\x0A\x8C\x40\x9C\x7C\x8A\xF6\xF2\x39\x60\xB2\xDE\xCB\xEC\xB8\xE4\x6F\xEA\x9B\x5D\xB7\x53\x90\x18\x32\x55\xC5\x20\xB7\x94\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x93\xE1\x43\x63\x5C\x3C\x9D\xD6\x27\xF3\x52\xEC\x17\xB2\xA9\xAF\x2C\xF7\x76\xF8\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x93\xE1\x43\x63\x5C\x3C\x9D\xD6\x27\xF3\x52\xEC\x17\xB2\xA9\xAF\x2C\xF7\x76\xF8\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x1D\x7C\xA4\x7B\xC3\x89\x75\x33\xE1\x3B\xA9\x45\xBF\x46\xE9\xE9\xA1\xDD\xC9\x22\x16\xB7\x47\x11\x0B\xD8\x9A\xBA\xF1\xC8\x0B\x70\x50\x53\x02\x91\x70\x85\x59\xA9\x1E\xA4\xE6\xEA\x23\x31\xA0\x00\x02\x31\x00\xFD\xE2\xF8\xB3\xAF\x16\xB9\x1E\x73\xC4\x96\xE3\xC1\x30\x19\xD8\x7E\xE6\xC3\x97\xDE\x1C\x4F\xB8\x89\x2F\x33\xEB\x48\x0F\x19\xF7\x87\x46\x5D\x26\x90\xA5\x85\xC5\xB9\x7A\x94\x3E\x87\xA8\xBD\x00"),
   // "TWCA Root Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x03\x7B\x30\x82\x02\x63\xA0\x03\x02\x01\x02\x02\x01\x01\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x5F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x54\x41\x49\x57\x41\x4E\x2D\x43\x41\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x0C\x07\x52\x6F\x6F\x74\x20\x43\x41\x31\x2A\x30\x28\x06\x03\x55\x04\x03\x0C\x21\x54\x57\x43\x41\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x30\x38\x30\x38\x32\x38\x30\x37\x32\x34\x33\x33\x5A\x17\x0D\x33\x30\x31\x32\x33\x31\x31\x35\x35\x39\x35\x39\x5A\x30\x5F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x54\x41\x49\x57\x41\x4E\x2D\x43\x41\x31\x10\x30\x0E\x06\x03\x55\x04\x0B\x0C\x07\x52\x6F\x6F\x74\x20\x43\x41\x31\x2A\x30\x28\x06\x03\x55\x04\x03\x0C\x21\x54\x57\x43\x41\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xB0\x7E\x72\xB8\xA4\x03\x94\xE6\xA7\xDE\x09\x38\x91\x4A\x11\x40\x87\xA7\x7C\x59\x64\x14\x7B\xB5\x11\x10\xDD\xFE\xBF\xD5\xC0\xBB\x56\xE2\x85\x25\xF4\x35\x72\x0F\xF8\x53\xD0\x41\xE1\x44\x01\xC2\xB4\x1C\xC3\x31\x42\x16\x47\x85\x33\x22\x76\xB2\x0A\x6F\x0F\xE5\x25\x50\x4F\x85\x86\xBE\xBF\x98\x2E\x10\x67\x1E\xBE\x11\x05\x86\x05\x90\xC4\x59\xD0\x7C\x78\x10\xB0\x80\x5C\xB7\xE1\xC7\x2B\x75\xCB\x7C\x9F\xAE\xB5\xD1\x9D\x23\x37\x63\xA7\xDC\x42\xA2\x2D\x92\x04\x1B\x50\xC1\x7B\xB8\x3E\x1B\xC9\x56\x04\x8B\x2F\x52\x9B\xAD\xA9\x56\xE9\xC1\xFF\xAD\xA9\x58\x87\x30\xB6\x81\xF7\x97\x45\xFC\x19\x57\x3B\x2B\x6F\xE4\x47\xF4\x99\x45\xFE\x1D\xF1\xF8\x97\xA3\x88\x1D\x37\x1C\x5C\x8F\xE0\x76\x25\x9A\x50\xF8\xA0\x54\xFF\x44\x90\x76\x23\xD2\x32\xC6\xC3\xAB\x06\xBF\xFC\xFB\xBF\xF3\xAD\x7D\x92\x62\x02\x5B\x29\xD3\x35\xA3\x93\x9A\x43\x64\x60\x5D\xB2\xFA\x32\xFF\x3B\x04\xAF\x4D\x40\x6A\xF9\xC7\xE3\xEF\x23\xFD\x6B\xCB\xE5\x0F\x8B\x38\x0D\xEE\x0A\xFC\xFE\x0F\x98\x9F\x30\x31\xDD\x6C\x52\x65\xF9\x8B\x81\xBE\x22\xE1\x1C\x58\x03\xBA\x91\x1B\x89\x07\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x6A\x38\x5B\x26\x8D\xDE\x8B\x5A\xF2\x4F\x7A\x54\x83\x19\x18\xE3\x08\x35\xA6\xBA\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x3C\xD5\x77\x3D\xDA\xDF\x89\xBA\x87\x0C\x08\x54\x6A\x20\x50\x92\xBE\xB0\x41\x3D\xB9\x26\x64\x83\x0A\x2F\xE8\x40\xC0\x97\x28\x27\x82\x30\x4A\xC9\x93\xFF\x6A\xE7\xA6\x00\x7F\x89\x42\x9A\xD6\x11\xE5\x53\xCE\x2F\xCC\xF2\xDA\x05\xC4\xFE\xE2\x50\xC4\x3A\x86\x7D\xCC\xDA\x7E\x10\x09\x3B\x92\x35\x2A\x53\xB2\xFE\xEB\x2B\x05\xD9\x6C\x5D\xE6\xD0\xEF\xD3\x6A\x66\x9E\x15\x28\x85\x7A\xE8\x82\x00\xAC\x1E\xA7\x09\x69\x56\x42\xD3\x68\x51\x18\xBE\x54\x9A\xBF\x44\x41\xBA\x49\xBE\x20\xBA\x69\x5C\xEE\xB8\x77\xCD\xCE\x6C\x1F\xAD\x83\x96\x18\x7D\x0E\xB5\x14\x39\x84\xF1\x28\xE9\x2D\xA3\x9E\x7B\x1E\x7A\x72\x5A\x83\xB3\x79\x6F\xEF\xB4\xFC\xD0\x0A\xA5\x58\x4F\x46\xDF\xFB\x6D\x79\x59\xF2\x84\x22\x52\xAE\x0F\xCC\xFB\x7C\x3B\xE7\x6A\xCA\x47\x61\xC3\x7A\xF8\xD3\x92\x04\x1F\xB8\x20\x84\xE1\x36\x54\x16\xC7\x40\xDE\x3B\x8A\x73\xDC\xDF\xC6\x09\x4C\xDF\xEC\xDA\xFF\xD4\x53\x42\xA1\xC9\xF2\x62\x1D\x22\x83\x3C\x97\xC5\xF9\x19\x62\x27\xAC\x65\x22\xD7\xD3\x3C\xC6\xE5\x8E\xB2\x53\xCC\x49\xCE\xBC\x30\xFE\x7B\x0E\x33\x90\xFB\xED\xD2\x14\x91\x1F\x07\xAF"),
   // "GDCA TrustAUTH R5 ROOT"
   CertificateDer::from_slice(b"\x30\x82\x05\x88\x30\x82\x03\x70\xA0\x03\x02\x01\x02\x02\x08\x7D\x09\x97\xFE\xF0\x47\xEA\x7A\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x62\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x32\x30\x30\x06\x03\x55\x04\x0A\x0C\x29\x47\x55\x41\x4E\x47\x20\x44\x4F\x4E\x47\x20\x43\x45\x52\x54\x49\x46\x49\x43\x41\x54\x45\x20\x41\x55\x54\x48\x4F\x52\x49\x54\x59\x20\x43\x4F\x2E\x2C\x4C\x54\x44\x2E\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x0C\x16\x47\x44\x43\x41\x20\x54\x72\x75\x73\x74\x41\x55\x54\x48\x20\x52\x35\x20\x52\x4F\x4F\x54\x30\x1E\x17\x0D\x31\x34\x31\x31\x32\x36\x30\x35\x31\x33\x31\x35\x5A\x17\x0D\x34\x30\x31\x32\x33\x31\x31\x35\x35\x39\x35\x39\x5A\x30\x62\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x32\x30\x30\x06\x03\x55\x04\x0A\x0C\x29\x47\x55\x41\x4E\x47\x20\x44\x4F\x4E\x47\x20\x43\x45\x52\x54\x49\x46\x49\x43\x41\x54\x45\x20\x41\x55\x54\x48\x4F\x52\x49\x54\x59\x20\x43\x4F\x2E\x2C\x4C\x54\x44\x2E\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x0C\x16\x47\x44\x43\x41\x20\x54\x72\x75\x73\x74\x41\x55\x54\x48\x20\x52\x35\x20\x52\x4F\x4F\x54\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD9\xA3\x16\xF0\xC8\x74\x74\x77\x9B\xEF\x33\x0D\x3B\x06\x7E\x55\xFC\xB5\x60\x8F\x76\x86\x12\x42\x7D\x56\x66\x3E\x88\x82\xED\x72\x63\x0E\x9E\x8B\xDD\x34\x2C\x02\x51\x51\xC3\x19\xFD\x59\x54\x84\xC9\xF1\x6B\xB3\x4C\xB0\xE9\xE8\x46\x5D\x38\xC6\xA2\xA7\x2E\x11\x57\xBA\x82\x15\xA2\x9C\x8F\x6D\xB0\x99\x4A\x0A\xF2\xEB\x89\x70\x63\x4E\x79\xC4\xB7\x5B\xBD\xA2\x5D\xB1\xF2\x41\x02\x2B\xAD\xA9\x3A\xA3\xEC\x79\x0A\xEC\x5F\x3A\xE3\xFD\xEF\x80\x3C\xAD\x34\x9B\x1A\xAB\x88\x26\x7B\x56\xA2\x82\x86\x1F\xEB\x35\x89\x83\x7F\x5F\xAE\x29\x4E\x3D\xB6\x6E\xEC\xAE\xC1\xF0\x27\x9B\xAE\xE3\xF4\xEC\xEF\xAE\x7F\xF7\x86\x3D\x72\x7A\xEB\xA5\xFB\x59\x4E\xA7\xEB\x95\x8C\x22\x39\x79\xE1\x2D\x08\x8F\xCC\xBC\x91\xB8\x41\xF7\x14\xC1\x23\xA9\xC3\xAD\x9A\x45\x44\xB3\xB2\xD7\x2C\xCD\xC6\x29\xE2\x50\x10\xAE\x5C\xCB\x82\x8E\x17\x18\x36\x7D\x97\xE6\x88\x9A\xB0\x4D\x34\x09\xF4\x2C\xB9\x5A\x66\x2A\xB0\x17\x9B\x9E\x1E\x76\x9D\x4A\x66\x31\x41\xDF\x3F\xFB\xC5\x06\xEF\x1B\xB6\x7E\x1A\x46\x36\xF7\x64\x63\x3B\xE3\x39\x18\x23\xE7\x67\x75\x14\xD5\x75\x57\x92\x37\xBD\xBE\x6A\x1B\x26\x50\xF2\x36\x26\x06\x90\xC5\x70\x01\x64\x6D\x76\x66\xE1\x91\xDB\x6E\x07\xC0\x61\x80\x2E\xB2\x2E\x2F\x8C\x70\xA7\xD1\x3B\x3C\xB3\x91\xE4\x6E\xB6\xC4\x3B\x70\xF2\x6C\x92\x97\x09\xCD\x47\x7D\x18\xC0\xF3\xBB\x9E\x0F\xD6\x8B\xAE\x07\xB6\x5A\x0F\xCE\x0B\x0C\x47\xA7\xE5\x3E\xB8\xBD\x7D\xC7\x9B\x35\xA0\x61\x97\x3A\x41\x75\x17\xCC\x2B\x96\x77\x2A\x92\x21\x1E\xD9\x95\x76\x20\x67\x68\xCF\x0D\xBD\xDF\xD6\x1F\x09\x6A\x9A\xE2\xCC\x73\x71\xA4\x2F\x7D\x12\x80\xB7\x53\x30\x46\x5E\x4B\x54\x99\x0F\x67\xC9\xA5\xC8\xF2\x20\xC1\x82\xEC\x9D\x11\xDF\xC2\x02\xFB\x1A\x3B\xD1\xED\x20\x9A\xEF\x65\x64\x92\x10\x0D\x2A\xE2\xDE\x70\xF1\x18\x67\x82\x8C\x61\xDE\xB8\xBC\xD1\x2F\x9C\xFB\x0F\xD0\x2B\xED\x1B\x76\xB9\xE4\x39\x55\xF8\xF8\xA1\x1D\xB8\xAA\x80\x00\x4C\x82\xE7\xB2\x7F\x09\xB8\xBC\x30\xA0\x2F\x0D\xF5\x52\x9E\x8E\xF7\x92\xB3\x0A\x00\x1D\x00\x54\x97\x06\xE0\xB1\x07\xD9\xC7\x0F\x5C\x65\x7D\x3C\x6D\x59\x57\xE4\xED\xA5\x8D\xE9\x40\x53\x9F\x15\x4B\xA0\x71\xF6\x1A\x21\xE3\xDA\x70\x06\x21\x58\x14\x87\x85\x77\x79\xAA\x82\x79\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE2\xC9\x40\x9F\x4D\xCE\xE8\x9A\xA1\x7C\xCF\x0E\x3F\x65\xC5\x29\x88\x6A\x19\x51\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\xD1\x49\x57\xE0\xA7\xCC\x68\x58\xBA\x01\x0F\x2B\x19\xCD\x8D\xB0\x61\x45\xAC\x11\xED\x63\x50\x69\xF8\x1F\x7F\xBE\x16\x8F\xFD\x9D\xEB\x0B\xAA\x32\x47\x76\xD2\x67\x24\xED\xBD\x7C\x33\x32\x97\x2A\xC7\x05\x86\x66\x0D\x17\x7D\x14\x15\x1B\xD4\xEB\xFD\x1F\x9A\xF6\x5E\x97\x69\xB7\x1A\x25\xA4\x0A\xB3\x91\x3F\x5F\x36\xAC\x8B\xEC\x57\xA8\x3E\xE7\x81\x8A\x18\x57\x39\x85\x74\x1A\x42\xC7\xE9\x5B\x13\x5F\x8F\xF9\x08\xE9\x92\x74\x8D\xF5\x47\xD2\xAB\x3B\xD6\xFB\x78\x66\x4E\x36\x7D\xF9\xE9\x92\xE9\x04\xDE\xFD\x49\x63\xFC\x6D\xFB\x14\x71\x93\x67\x2F\x47\x4A\xB7\xB9\xFF\x1E\x2A\x73\x70\x46\x30\xBF\x5A\xF2\x2F\x79\xA5\xE1\x8D\x0C\xD9\xF9\xB2\x63\x37\x8C\x37\x65\x85\x70\x6A\x5C\x5B\x09\x72\xB9\xAD\x63\x3C\xB1\xDD\xF8\xFC\x32\xBF\x37\x86\xE4\xBB\x8E\x98\x27\x7E\xBA\x1F\x16\xE1\x70\x11\xF2\x03\xDF\x25\x62\x32\x27\x26\x18\x32\x84\x9F\xFF\x00\x3A\x13\xBA\x9A\x4D\xF4\x4F\xB8\x14\x70\x22\xB1\xCA\x2B\x90\xCE\x29\xC1\x70\xF4\x2F\x9D\x7F\xF2\x90\x1E\xD6\x5A\xDF\xB7\x46\xFC\xE6\x86\xFA\xCB\xE0\x20\x76\x7A\xBA\xA6\xCB\xF5\x7C\xDE\x62\xA5\xB1\x8B\xEE\xDE\x82\x66\x8A\x4E\x3A\x30\x1F\x3F\x80\xCB\xAD\x27\xBA\x0C\x5E\xD7\xD0\xB1\x56\xCA\x77\x71\xB2\xB5\x75\xA1\x50\xA9\x40\x43\x17\xC2\x28\xD9\xCF\x52\x8B\x5B\xC8\x63\xD4\x42\x3E\xA0\x33\x7A\x46\x2E\xF7\x0A\x20\x46\x54\x7E\x6A\x4F\x31\xF1\x81\x7E\x42\x74\x38\x65\x73\x27\xEE\xC6\x7C\xB8\x8E\xD7\xA5\x3A\xD7\x98\xA1\x9C\x8C\x10\x55\xD3\xDB\x4B\xEC\x40\x90\xF2\xCD\x6E\x57\xD2\x62\x0E\x7C\x57\x93\xB1\xA7\x6D\xCD\x9D\x83\xBB\x2A\xE7\xE5\xB6\x3B\x71\x58\xAD\xFD\xD1\x45\xBC\x5A\x91\xEE\x53\x15\x6F\xD3\x45\x09\x75\x6E\xBA\x90\x5D\x1E\x04\xCF\x37\xDF\x1E\xA8\x66\xB1\x8C\xE6\x20\x6A\xEF\xFC\x48\x4E\x74\x98\x42\xAF\x29\x6F\x2E\x6A\xC7\xFB\x7D\xD1\x66\x31\x22\xCC\x86\x00\x7E\x66\x83\x0C\x42\xF4\xBD\x34\x92\xC3\x1A\xEA\x4F\xCA\x7E\x72\x4D\x0B\x70\x8C\xA6\x48\xBB\xA6\xA1\x14\xF6\xFB\x58\x44\x99\x14\xAE\xAA\x0B\x93\x69\xA0\x29\x25\x4A\xA5\xCB\x2B\xDD\x8A\x66\x07\x16\x78\x15\x57\x71\x1B\xEC\xF5\x47\x84\xF3\x9E\x31\x37\x7A\xD5\x7F\x24\xAD\xE4\xBC\xFD\xFD\xCC\x6E\x83\xE8\x0C\xA8\xB7\x41\x6C\x07\xDD\xBD\x3C\x86\x97\x2F\xD2"),
   // "TrustAsia TLS ECC Root CA"
   CertificateDer::from_slice(b"\x30\x82\x02\x31\x30\x82\x01\xB7\xA0\x03\x02\x01\x02\x02\x14\x36\x74\xE1\x4D\x7C\x65\x13\xC9\xAC\x83\x55\x25\xA0\x3E\x52\x7E\x2F\x50\x68\xC7\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x58\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x32\x34\x30\x35\x31\x35\x30\x35\x34\x31\x35\x36\x5A\x17\x0D\x34\x34\x30\x35\x31\x35\x30\x35\x34\x31\x35\x35\x5A\x30\x58\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xB8\x7F\xA5\x5B\x3F\x01\x3E\x7D\xF0\x88\x6D\xAE\x29\x98\xE1\x9B\x5C\x53\x99\xDB\xF7\x08\xFF\xD5\x6A\xE0\x8E\xCB\x44\xA6\xF0\xC1\x8C\xBD\x4F\xD4\xCE\xD4\x88\x53\xE8\x5E\x57\xD7\x4E\xBE\x2C\x3F\xF3\x12\xA7\x00\xE9\x82\xE3\x2A\x5B\x32\x7C\x4B\x9B\x14\x29\x9E\xF8\x2D\x83\xB5\xEB\x8E\x31\x49\x3D\x26\x61\xE9\x1D\x72\x8B\x89\xB6\x0A\xBB\x9C\x33\x35\x0F\xDA\xEC\xDE\xA9\x4A\x1F\xC9\x33\xB1\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x2C\x85\x53\xBB\xB1\x43\xCD\x32\xEA\x9E\xA3\x87\xFE\xA2\x98\xA8\xA6\x93\xE9\x10\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x54\x47\xD7\xC3\x2D\x61\x86\x48\xF4\x79\x5A\x55\x0D\x35\x2F\x5F\x0D\xF6\x67\x6C\x77\x40\x1A\x46\xE7\xF8\x68\x5B\x4E\x27\x1D\xB8\x98\x7D\x7F\x93\xBF\x08\x4D\xC4\xDA\x45\x30\xA1\x0F\x5E\x4A\x78\x02\x31\x00\xA3\x91\x87\xF2\x11\x33\x83\xC3\xC1\x8A\x91\x3A\x4C\x2B\x50\xB1\xBD\x22\x94\x5D\x35\x89\x73\x83\xC5\x9B\x19\xFE\xB4\xA1\xE9\xD4\xA1\x66\xB6\x01\xA5\x36\xF9\xD8\x68\x61\x28\xB7\x74\xE1\xA2\x31"),
   // "SSL.com TLS ECC Root CA 2022"
   CertificateDer::from_slice(b"\x30\x82\x02\x3A\x30\x82\x01\xC0\xA0\x03\x02\x01\x02\x02\x10\x14\x03\xF5\xAB\xFB\x37\x8B\x17\x40\x5B\xE2\x43\xB2\xA5\xD1\xC4\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x32\x30\x1E\x17\x0D\x32\x32\x30\x38\x32\x35\x31\x36\x33\x33\x34\x38\x5A\x17\x0D\x34\x36\x30\x38\x31\x39\x31\x36\x33\x33\x34\x37\x5A\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x0C\x0F\x53\x53\x4C\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x53\x53\x4C\x2E\x63\x6F\x6D\x20\x54\x4C\x53\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x32\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x45\x29\x35\x73\xFA\xC2\xB8\x23\xCE\x14\x7D\xA8\xB1\x4D\xA0\x5B\x36\xEE\x2A\x2C\x53\xC3\x60\x09\x35\xB2\x24\x66\x26\x69\xC0\xB3\x95\xD6\x5D\x92\x40\x19\x0E\xC6\xA5\x13\x70\xF4\xEF\x12\x51\x28\x5D\xE7\xCC\xBD\xF9\x3C\x85\xC1\xCF\x94\x90\xC9\x2B\xCE\x92\x42\x58\x59\x67\xFD\x94\x27\x10\x64\x8C\x4F\x04\xB1\x4D\x49\xE4\x7B\x4F\x9B\xF5\xE7\x08\xF8\x03\x88\xF7\xA7\xC3\x92\x4B\x19\x54\x81\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x89\x8F\x2F\xA3\xE8\x2B\xA0\x14\x54\x7B\xF3\x56\xB8\x26\x5F\x67\x38\x0B\x9C\xD0\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x89\x8F\x2F\xA3\xE8\x2B\xA0\x14\x54\x7B\xF3\x56\xB8\x26\x5F\x67\x38\x0B\x9C\xD0\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x55\xE3\x22\x56\xE9\xD7\x92\x24\x58\x4F\x1E\x94\x32\x0F\x0C\x02\x36\xC2\xFD\xAC\x74\x32\x4E\xE1\xFB\x1C\x80\x88\xA3\xCC\xFB\xD7\xEB\x2B\xFF\x37\x7D\xF0\xED\xD7\x9E\x75\x6A\x35\x76\x52\x45\xE0\x02\x31\x00\xC7\x8D\x6F\x42\x20\x8F\xBE\xB6\x4D\x59\xED\x77\x4D\x29\xC4\x20\x20\x45\x64\x86\x3A\x50\xC6\xC4\xAD\x2D\x93\xF5\x18\x7D\x72\xED\xA9\xCF\xC4\xAC\x57\x36\x28\x08\x65\xDF\x3C\x79\x66\x7E\xA0\xEA"),
   // "Microsoft RSA Root Certificate Authority 2017"
   CertificateDer::from_slice(b"\x30\x82\x05\xA8\x30\x82\x03\x90\xA0\x03\x02\x01\x02\x02\x10\x1E\xD3\x97\x09\x5F\xD8\xB4\xB3\x47\x70\x1E\xAA\xBE\x7F\x45\xB3\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x32\x30\x31\x37\x30\x1E\x17\x0D\x31\x39\x31\x32\x31\x38\x32\x32\x35\x31\x32\x32\x5A\x17\x0D\x34\x32\x30\x37\x31\x38\x32\x33\x30\x30\x32\x33\x5A\x30\x65\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x4D\x69\x63\x72\x6F\x73\x6F\x66\x74\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x20\x32\x30\x31\x37\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xCA\x5B\xBE\x94\x33\x8C\x29\x95\x91\x16\x0A\x95\xBD\x47\x62\xC1\x89\xF3\x99\x36\xDF\x46\x90\xC9\xA5\xED\x78\x6A\x6F\x47\x91\x68\xF8\x27\x67\x50\x33\x1D\xA1\xA6\xFB\xE0\xE5\x43\xA3\x84\x02\x57\x01\x5D\x9C\x48\x40\x82\x53\x10\xBC\xBF\xC7\x3B\x68\x90\xB6\x82\x2D\xE5\xF4\x65\xD0\xCC\x6D\x19\xCC\x95\xF9\x7B\xAC\x4A\x94\xAD\x0E\xDE\x4B\x43\x1D\x87\x07\x92\x13\x90\x80\x83\x64\x35\x39\x04\xFC\xE5\xE9\x6C\xB3\xB6\x1F\x50\x94\x38\x65\x50\x5C\x17\x46\xB9\xB6\x85\xB5\x1C\xB5\x17\xE8\xD6\x45\x9D\xD8\xB2\x26\xB0\xCA\xC4\x70\x4A\xAE\x60\xA4\xDD\xB3\xD9\xEC\xFC\x3B\xD5\x57\x72\xBC\x3F\xC8\xC9\xB2\xDE\x4B\x6B\xF8\x23\x6C\x03\xC0\x05\xBD\x95\xC7\xCD\x73\x3B\x66\x80\x64\xE3\x1A\xAC\x2E\xF9\x47\x05\xF2\x06\xB6\x9B\x73\xF5\x78\x33\x5B\xC7\xA1\xFB\x27\x2A\xA1\xB4\x9A\x91\x8C\x91\xD3\x3A\x82\x3E\x76\x40\xB4\xCD\x52\x61\x51\x70\x28\x3F\xC5\xC5\x5A\xF2\xC9\x8C\x49\xBB\x14\x5B\x4D\xC8\xFF\x67\x4D\x4C\x12\x96\xAD\xF5\xFE\x78\xA8\x97\x87\xD7\xFD\x5E\x20\x80\xDC\xA1\x4B\x22\xFB\xD4\x89\xAD\xBA\xCE\x47\x97\x47\x55\x7B\x8F\x45\xC8\x67\x28\x84\x95\x1C\x68\x30\xEF\xEF\x49\xE0\x35\x7B\x64\xE7\x98\xB0\x94\xDA\x4D\x85\x3B\x3E\x55\xC4\x28\xAF\x57\xF3\x9E\x13\xDB\x46\x27\x9F\x1E\xA2\x5E\x44\x83\xA4\xA5\xCA\xD5\x13\xB3\x4B\x3F\xC4\xE3\xC2\xE6\x86\x61\xA4\x52\x30\xB9\x7A\x20\x4F\x6F\x0F\x38\x53\xCB\x33\x0C\x13\x2B\x8F\xD6\x9A\xBD\x2A\xC8\x2D\xB1\x1C\x7D\x4B\x51\xCA\x47\xD1\x48\x27\x72\x5D\x87\xEB\xD5\x45\xE6\x48\x65\x9D\xAF\x52\x90\xBA\x5B\xA2\x18\x65\x57\x12\x9F\x68\xB9\xD4\x15\x6B\x94\xC4\x69\x22\x98\xF4\x33\xE0\xED\xF9\x51\x8E\x41\x50\xC9\x34\x4F\x76\x90\xAC\xFC\x38\xC1\xD8\xE1\x7B\xB9\xE3\xE3\x94\xE1\x46\x69\xCB\x0E\x0A\x50\x6B\x13\xBA\xAC\x0F\x37\x5A\xB7\x12\xB5\x90\x81\x1E\x56\xAE\x57\x22\x86\xD9\xC9\xD2\xD1\xD7\x51\xE3\xAB\x3B\xC6\x55\xFD\x1E\x0E\xD3\x74\x0A\xD1\xDA\xAA\xEA\x69\xB8\x97\x28\x8F\x48\xC4\x07\xF8\x52\x43\x3A\xF4\xCA\x55\x35\x2C\xB0\xA6\x6A\xC0\x9C\xF9\xF2\x81\xE1\x12\x6A\xC0\x45\xD9\x67\xB3\xCE\xFF\x23\xA2\x89\x0A\x54\xD4\x14\xB9\x2A\xA8\xD7\xEC\xF9\xAB\xCD\x25\x58\x32\x79\x8F\x90\x5B\x98\x39\xC4\x08\x06\xC1\xAC\x7F\x0E\x3D\x00\xA5\x02\x03\x01\x00\x01\xA3\x54\x30\x52\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x09\xCB\x59\x7F\x86\xB2\x70\x8F\x1A\xC3\x39\xE3\xC0\xD9\xE9\xBF\xBB\x4D\xB2\x23\x30\x10\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x15\x01\x04\x03\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\xAC\xAF\x3E\x5D\xC2\x11\x96\x89\x8E\xA3\xE7\x92\xD6\x97\x15\xB8\x13\xA2\xA6\x42\x2E\x02\xCD\x16\x05\x59\x27\xCA\x20\xE8\xBA\xB8\xE8\x1A\xEC\x4D\xA8\x97\x56\xAE\x65\x43\xB1\x8F\x00\x9B\x52\xCD\x55\xCD\x53\x39\x6D\x62\x4C\x8B\x0D\x5B\x7C\x2E\x44\xBF\x83\x10\x8F\xF3\x53\x82\x80\xC3\x4F\x3A\xC7\x6E\x11\x3F\xE6\xE3\x16\x91\x84\xFB\x6D\x84\x7F\x34\x74\xAD\x89\xA7\xCE\xB9\xD7\xD7\x9F\x84\x64\x92\xBE\x95\xA1\xAD\x09\x53\x33\xDD\xEE\x0A\xEA\x4A\x51\x8E\x6F\x55\xAB\xBA\xB5\x94\x46\xAE\x8C\x7F\xD8\xA2\x50\x25\x65\x60\x80\x46\xDB\x33\x04\xAE\x6C\xB5\x98\x74\x54\x25\xDC\x93\xE4\xF8\xE3\x55\x15\x3D\xB8\x6D\xC3\x0A\xA4\x12\xC1\x69\x85\x6E\xDF\x64\xF1\x53\x99\xE1\x4A\x75\x20\x9D\x95\x0F\xE4\xD6\xDC\x03\xF1\x59\x18\xE8\x47\x89\xB2\x57\x5A\x94\xB6\xA9\xD8\x17\x2B\x17\x49\xE5\x76\xCB\xC1\x56\x99\x3A\x37\xB1\xFF\x69\x2C\x91\x91\x93\xE1\xDF\x4C\xA3\x37\x76\x4D\xA1\x9F\xF8\x6D\x1E\x1D\xD3\xFA\xEC\xFB\xF4\x45\x1D\x13\x6D\xCF\xF7\x59\xE5\x22\x27\x72\x2B\x86\xF3\x57\xBB\x30\xED\x24\x4D\xDC\x7D\x56\xBB\xA3\xB3\xF8\x34\x79\x89\xC1\xE0\xF2\x02\x61\xF7\xA6\xFC\x0F\xBB\x1C\x17\x0B\xAE\x41\xD9\x7C\xBD\x27\xA3\xFD\x2E\x3A\xD1\x93\x94\xB1\x73\x1D\x24\x8B\xAF\x5B\x20\x89\xAD\xB7\x67\x66\x79\xF5\x3A\xC6\xA6\x96\x33\xFE\x53\x92\xC8\x46\xB1\x11\x91\xC6\x99\x7F\x8F\xC9\xD6\x66\x31\x20\x41\x10\x87\x2D\x0C\xD6\xC1\xAF\x34\x98\xCA\x64\x83\xFB\x13\x57\xD1\xC1\xF0\x3C\x7A\x8C\xA5\xC1\xFD\x95\x21\xA0\x71\xC1\x93\x67\x71\x12\xEA\x8F\x88\x0A\x69\x19\x64\x99\x23\x56\xFB\xAC\x2A\x2E\x70\xBE\x66\xC4\x0C\x84\xEF\xE5\x8B\xF3\x93\x01\xF8\x6A\x90\x93\x67\x4B\xB2\x68\xA3\xB5\x62\x8F\xE9\x3F\x8C\x7A\x3B\x5E\x0F\xE7\x8C\xB8\xC6\x7C\xEF\x37\xFD\x74\xE2\xC8\x4F\x33\x72\xE1\x94\x39\x6D\xBD\x12\xAF\xBE\x0C\x4E\x70\x7C\x1B\x6F\x8D\xB3\x32\x93\x73\x44\x16\x6D\xE8\xF4\xF7\xE0\x95\x80\x8F\x96\x5D\x38\xA4\xF4\xAB\xDE\x0A\x30\x87\x93\xD8\x4D\x00\x71\x62\x45\x27\x4B\x3A\x42\x84\x5B\x7F\x65\xB7\x67\x34\x52\x2D\x9C\x16\x6B\xAA\xA8\xD8\x7B\xA3\x42\x4C\x71\xC7\x0C\xCA\x3E\x83\xE4\xA6\xEF\xB7\x01\x30\x5E\x51\xA3\x79\xF5\x70\x69\xA6\x41\x44\x0F\x86\xB0\x2C\x91\xC6\x3D\xEA\xAE\x0F\x84"),
   // "Sectigo Public Server Authentication Root E46"
   CertificateDer::from_slice(b"\x30\x82\x02\x3A\x30\x82\x01\xC1\xA0\x03\x02\x01\x02\x02\x10\x42\xF2\xCC\xDA\x1B\x69\x37\x44\x5F\x15\xFE\x75\x28\x10\xB8\xF4\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x5F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x13\x0F\x53\x65\x63\x74\x69\x67\x6F\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x53\x65\x63\x74\x69\x67\x6F\x20\x50\x75\x62\x6C\x69\x63\x20\x53\x65\x72\x76\x65\x72\x20\x41\x75\x74\x68\x65\x6E\x74\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x45\x34\x36\x30\x1E\x17\x0D\x32\x31\x30\x33\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x33\x32\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x5F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x42\x31\x18\x30\x16\x06\x03\x55\x04\x0A\x13\x0F\x53\x65\x63\x74\x69\x67\x6F\x20\x4C\x69\x6D\x69\x74\x65\x64\x31\x36\x30\x34\x06\x03\x55\x04\x03\x13\x2D\x53\x65\x63\x74\x69\x67\x6F\x20\x50\x75\x62\x6C\x69\x63\x20\x53\x65\x72\x76\x65\x72\x20\x41\x75\x74\x68\x65\x6E\x74\x69\x63\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x45\x34\x36\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x76\xFA\x99\xA9\x6E\x20\xED\xF9\xD7\x77\xE3\x07\x3B\xA8\xDB\x3D\x5F\x38\xE8\xAB\x55\xA6\x56\x4F\xD6\x48\xEA\xEC\x7F\x2D\xAA\xC3\xB2\xC5\x79\xEC\x99\x61\x7F\x10\x79\xC7\x02\x5A\xF9\x04\x37\xF5\x34\x35\x2B\x77\xCE\x7F\x20\x8F\x52\xA3\x00\x89\xEC\xD5\xA7\xA2\x6D\x5B\xE3\x4B\x92\x93\xA0\x80\xF5\x01\x94\xDC\xF0\x68\x07\x1E\xCD\xEE\xFE\x25\x52\xB5\x20\x43\x1C\x1B\xFE\xEB\x19\xCE\x43\xA3\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD1\x22\xDA\x4C\x59\xF1\x4B\x5F\x26\x38\xAA\x9D\xD6\xEE\xEB\x0D\xC3\xFB\xA9\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x67\x00\x30\x64\x02\x30\x27\xEE\xA4\x5A\xA8\x21\xBB\xE9\x47\x97\x94\x89\xA5\x74\x20\x6D\x79\x4F\xC8\xBD\x93\x5E\x58\x18\xFB\x2D\x1A\x00\x6A\xC9\xB8\x3D\xD0\xA4\x4F\x44\x47\x94\x01\x56\xA2\xF8\x33\x25\x0C\x42\xDF\xAA\x02\x30\x1D\xEA\xE1\x2E\x88\x2E\xE1\xF9\xA7\x1D\x02\x32\x4E\xF2\x9F\x6C\x55\x74\xE3\xAE\xAE\xFB\xA5\x1A\xEE\xED\xD2\xFC\xC2\x03\x11\xEB\x45\x5C\x60\x10\x3D\x5C\x7F\x99\x03\x5B\x6D\x54\x48\x01\x8A\x73"),
   // "DigiCert Global Root G2"
   CertificateDer::from_slice(b"\x30\x82\x03\x8E\x30\x82\x02\x76\xA0\x03\x02\x01\x02\x02\x10\x03\x3A\xF1\xE6\xA7\x11\xA9\xA0\xBB\x28\x64\xB1\x1D\x09\xFA\xE5\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x44\x69\x67\x69\x43\x65\x72\x74\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x32\x30\x1E\x17\x0D\x31\x33\x30\x38\x30\x31\x31\x32\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x35\x31\x32\x30\x30\x30\x30\x5A\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x69\x67\x69\x43\x65\x72\x74\x20\x49\x6E\x63\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x77\x77\x77\x2E\x64\x69\x67\x69\x63\x65\x72\x74\x2E\x63\x6F\x6D\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x13\x17\x44\x69\x67\x69\x43\x65\x72\x74\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x47\x32\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xBB\x37\xCD\x34\xDC\x7B\x6B\xC9\xB2\x68\x90\xAD\x4A\x75\xFF\x46\xBA\x21\x0A\x08\x8D\xF5\x19\x54\xC9\xFB\x88\xDB\xF3\xAE\xF2\x3A\x89\x91\x3C\x7A\xE6\xAB\x06\x1A\x6B\xCF\xAC\x2D\xE8\x5E\x09\x24\x44\xBA\x62\x9A\x7E\xD6\xA3\xA8\x7E\xE0\x54\x75\x20\x05\xAC\x50\xB7\x9C\x63\x1A\x6C\x30\xDC\xDA\x1F\x19\xB1\xD7\x1E\xDE\xFD\xD7\xE0\xCB\x94\x83\x37\xAE\xEC\x1F\x43\x4E\xDD\x7B\x2C\xD2\xBD\x2E\xA5\x2F\xE4\xA9\xB8\xAD\x3A\xD4\x99\xA4\xB6\x25\xE9\x9B\x6B\x00\x60\x92\x60\xFF\x4F\x21\x49\x18\xF7\x67\x90\xAB\x61\x06\x9C\x8F\xF2\xBA\xE9\xB4\xE9\x92\x32\x6B\xB5\xF3\x57\xE8\x5D\x1B\xCD\x8C\x1D\xAB\x95\x04\x95\x49\xF3\x35\x2D\x96\xE3\x49\x6D\xDD\x77\xE3\xFB\x49\x4B\xB4\xAC\x55\x07\xA9\x8F\x95\xB3\xB4\x23\xBB\x4C\x6D\x45\xF0\xF6\xA9\xB2\x95\x30\xB4\xFD\x4C\x55\x8C\x27\x4A\x57\x14\x7C\x82\x9D\xCD\x73\x92\xD3\x16\x4A\x06\x0C\x8C\x50\xD1\x8F\x1E\x09\xBE\x17\xA1\xE6\x21\xCA\xFD\x83\xE5\x10\xBC\x83\xA5\x0A\xC4\x67\x28\xF6\x73\x14\x14\x3D\x46\x76\xC3\x87\x14\x89\x21\x34\x4D\xAF\x0F\x45\x0C\xA6\x49\xA1\xBA\xBB\x9C\xC5\xB1\x33\x83\x29\x85\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x4E\x22\x54\x20\x18\x95\xE6\xE3\x6E\xE6\x0F\xFA\xFA\xB9\x12\xED\x06\x17\x8F\x39\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x60\x67\x28\x94\x6F\x0E\x48\x63\xEB\x31\xDD\xEA\x67\x18\xD5\x89\x7D\x3C\xC5\x8B\x4A\x7F\xE9\xBE\xDB\x2B\x17\xDF\xB0\x5F\x73\x77\x2A\x32\x13\x39\x81\x67\x42\x84\x23\xF2\x45\x67\x35\xEC\x88\xBF\xF8\x8F\xB0\x61\x0C\x34\xA4\xAE\x20\x4C\x84\xC6\xDB\xF8\x35\xE1\x76\xD9\xDF\xA6\x42\xBB\xC7\x44\x08\x86\x7F\x36\x74\x24\x5A\xDA\x6C\x0D\x14\x59\x35\xBD\xF2\x49\xDD\xB6\x1F\xC9\xB3\x0D\x47\x2A\x3D\x99\x2F\xBB\x5C\xBB\xB5\xD4\x20\xE1\x99\x5F\x53\x46\x15\xDB\x68\x9B\xF0\xF3\x30\xD5\x3E\x31\xE2\x8D\x84\x9E\xE3\x8A\xDA\xDA\x96\x3E\x35\x13\xA5\x5F\xF0\xF9\x70\x50\x70\x47\x41\x11\x57\x19\x4E\xC0\x8F\xAE\x06\xC4\x95\x13\x17\x2F\x1B\x25\x9F\x75\xF2\xB1\x8E\x99\xA1\x6F\x13\xB1\x41\x71\xFE\x88\x2A\xC8\x4F\x10\x20\x55\xD7\xF3\x14\x45\xE5\xE0\x44\xF4\xEA\x87\x95\x32\x93\x0E\xFE\x53\x46\xFA\x2C\x9D\xFF\x8B\x22\xB9\x4B\xD9\x09\x45\xA4\xDE\xA4\xB8\x9A\x58\xDD\x1B\x7D\x52\x9F\x8E\x59\x43\x88\x81\xA4\x9E\x26\xD5\x6F\xAD\xDD\x0D\xC6\x37\x7D\xED\x03\x92\x1B\xE5\x77\x5F\x76\xEE\x3C\x8D\xC4\x5D\x56\x5B\xA2\xD9\x66\x6E\xB3\x35\x37\xE5\x32\xB6"),
   // "GlobalSign"
   CertificateDer::from_slice(b"\x30\x82\x03\x5F\x30\x82\x02\x47\xA0\x03\x02\x01\x02\x02\x0B\x04\x00\x00\x00\x00\x01\x21\x58\x53\x08\xA2\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4C\x31\x20\x30\x1E\x06\x03\x55\x04\x0B\x13\x17\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x33\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x1E\x17\x0D\x30\x39\x30\x33\x31\x38\x31\x30\x30\x30\x30\x30\x5A\x17\x0D\x32\x39\x30\x33\x31\x38\x31\x30\x30\x30\x30\x30\x5A\x30\x4C\x31\x20\x30\x1E\x06\x03\x55\x04\x0B\x13\x17\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x52\x33\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x31\x13\x30\x11\x06\x03\x55\x04\x03\x13\x0A\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xCC\x25\x76\x90\x79\x06\x78\x22\x16\xF5\xC0\x83\xB6\x84\xCA\x28\x9E\xFD\x05\x76\x11\xC5\xAD\x88\x72\xFC\x46\x02\x43\xC7\xB2\x8A\x9D\x04\x5F\x24\xCB\x2E\x4B\xE1\x60\x82\x46\xE1\x52\xAB\x0C\x81\x47\x70\x6C\xDD\x64\xD1\xEB\xF5\x2C\xA3\x0F\x82\x3D\x0C\x2B\xAE\x97\xD7\xB6\x14\x86\x10\x79\xBB\x3B\x13\x80\x77\x8C\x08\xE1\x49\xD2\x6A\x62\x2F\x1F\x5E\xFA\x96\x68\xDF\x89\x27\x95\x38\x9F\x06\xD7\x3E\xC9\xCB\x26\x59\x0D\x73\xDE\xB0\xC8\xE9\x26\x0E\x83\x15\xC6\xEF\x5B\x8B\xD2\x04\x60\xCA\x49\xA6\x28\xF6\x69\x3B\xF6\xCB\xC8\x28\x91\xE5\x9D\x8A\x61\x57\x37\xAC\x74\x14\xDC\x74\xE0\x3A\xEE\x72\x2F\x2E\x9C\xFB\xD0\xBB\xBF\xF5\x3D\x00\xE1\x06\x33\xE8\x82\x2B\xAE\x53\xA6\x3A\x16\x73\x8C\xDD\x41\x0E\x20\x3A\xC0\xB4\xA7\xA1\xE9\xB2\x4F\x90\x2E\x32\x60\xE9\x57\xCB\xB9\x04\x92\x68\x68\xE5\x38\x26\x60\x75\xB2\x9F\x77\xFF\x91\x14\xEF\xAE\x20\x49\xFC\xAD\x40\x15\x48\xD1\x02\x31\x61\x19\x5E\xB8\x97\xEF\xAD\x77\xB7\x64\x9A\x7A\xBF\x5F\xC1\x13\xEF\x9B\x62\xFB\x0D\x6C\xE0\x54\x69\x16\xA9\x03\xDA\x6E\xE9\x83\x93\x71\x76\xC6\x69\x85\x82\x17\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x8F\xF0\x4B\x7F\xA8\x2E\x45\x24\xAE\x4D\x50\xFA\x63\x9A\x8B\xDE\xE2\xDD\x1B\xBC\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x4B\x40\xDB\xC0\x50\xAA\xFE\xC8\x0C\xEF\xF7\x96\x54\x45\x49\xBB\x96\x00\x09\x41\xAC\xB3\x13\x86\x86\x28\x07\x33\xCA\x6B\xE6\x74\xB9\xBA\x00\x2D\xAE\xA4\x0A\xD3\xF5\xF1\xF1\x0F\x8A\xBF\x73\x67\x4A\x83\xC7\x44\x7B\x78\xE0\xAF\x6E\x6C\x6F\x03\x29\x8E\x33\x39\x45\xC3\x8E\xE4\xB9\x57\x6C\xAA\xFC\x12\x96\xEC\x53\xC6\x2D\xE4\x24\x6C\xB9\x94\x63\xFB\xDC\x53\x68\x67\x56\x3E\x83\xB8\xCF\x35\x21\xC3\xC9\x68\xFE\xCE\xDA\xC2\x53\xAA\xCC\x90\x8A\xE9\xF0\x5D\x46\x8C\x95\xDD\x7A\x58\x28\x1A\x2F\x1D\xDE\xCD\x00\x37\x41\x8F\xED\x44\x6D\xD7\x53\x28\x97\x7E\xF3\x67\x04\x1E\x15\xD7\x8A\x96\xB4\xD3\xDE\x4C\x27\xA4\x4C\x1B\x73\x73\x76\xF4\x17\x99\xC2\x1F\x7A\x0E\xE3\x2D\x08\xAD\x0A\x1C\x2C\xFF\x3C\xAB\x55\x0E\x0F\x91\x7E\x36\xEB\xC3\x57\x49\xBE\xE1\x2E\x2D\x7C\x60\x8B\xC3\x41\x51\x13\x23\x9D\xCE\xF7\x32\x6B\x94\x01\xA8\x99\xE7\x2C\x33\x1F\x3A\x3B\x25\xD2\x86\x40\xCE\x3B\x2C\x86\x78\xC9\x61\x2F\x14\xBA\xEE\xDB\x55\x6F\xDF\x84\xEE\x05\x09\x4D\xBD\x28\xD8\x72\xCE\xD3\x62\x50\x65\x1E\xEB\x92\x97\x83\x31\xD9\xB3\xB5\xCA\x47\x58\x3F\x5F"),
   // "GlobalSign Root E46"
   CertificateDer::from_slice(b"\x30\x82\x02\x0B\x30\x82\x01\x91\xA0\x03\x02\x01\x02\x02\x12\x11\xD2\xBB\xBA\x33\x6E\xD4\xBC\xE6\x24\x68\xC5\x0D\x84\x1D\x98\xE8\x43\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x46\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x45\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x6E\x76\x2D\x73\x61\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x45\x34\x36\x30\x1E\x17\x0D\x31\x39\x30\x33\x32\x30\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x36\x30\x33\x32\x30\x30\x30\x30\x30\x30\x30\x5A\x30\x46\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x42\x45\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x13\x10\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x6E\x76\x2D\x73\x61\x31\x1C\x30\x1A\x06\x03\x55\x04\x03\x13\x13\x47\x6C\x6F\x62\x61\x6C\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x45\x34\x36\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x9C\x0E\xB1\xCF\xB7\xE8\x9E\x52\x77\x75\x34\xFA\xA5\x46\xA7\xAD\x32\x19\x32\xB4\x07\xA9\x27\xCA\x94\xBB\x0C\xD2\x0A\x10\xC7\xDA\x89\xB0\x97\x0C\x70\x13\x09\x01\x8E\xD8\xEA\x47\xEA\xBE\xB2\x80\x2B\xCD\xFC\x28\x0D\xDB\xAC\xBC\xA4\x86\x37\xED\x70\x08\x00\x75\xEA\x93\x0B\x7B\x2E\x52\x9C\x23\x68\x23\x06\x43\xEC\x92\x2F\x53\x84\xDB\xFB\x47\x14\x07\xE8\x5F\x94\x67\x5D\xC9\x7A\x81\x3C\x20\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x31\x0A\x90\x8F\xB6\xC6\x9D\xD2\x44\x4B\x80\xB5\xA2\xE6\x1F\xB1\x12\x4F\x1B\x95\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xDF\x54\x90\xED\x9B\xEF\x8B\x94\x02\x93\x17\x82\x99\xBE\xB3\x9E\x2C\xF6\x0B\x91\x8C\x9F\x4A\x14\xB1\xF6\x64\xBC\xBB\x68\x51\x13\x0C\x03\xF7\x15\x8B\x84\x60\xB9\x8B\xFF\x52\x8E\xE7\x8C\xBC\x1C\x02\x30\x3C\xF9\x11\xD4\x8C\x4E\xC0\xC1\x61\xC2\x15\x4C\xAA\xAB\x1D\x0B\x31\x5F\x3B\x1C\xE2\x00\x97\x44\x31\xE6\xFE\x73\x96\x2F\xDA\x96\xD3\xFE\x08\x07\xB3\x34\x89\xBC\x05\x9F\xF7\x1E\x86\xEE\x8B\x70"),
   // "UCA Extended Validation Root"
   CertificateDer::from_slice(b"\x30\x82\x05\x5A\x30\x82\x03\x42\xA0\x03\x02\x01\x02\x02\x10\x4F\xD2\x2B\x8F\xF5\x64\xC8\x33\x9E\x4F\x34\x58\x66\x23\x70\x60\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x55\x6E\x69\x54\x72\x75\x73\x74\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x55\x43\x41\x20\x45\x78\x74\x65\x6E\x64\x65\x64\x20\x56\x61\x6C\x69\x64\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x30\x1E\x17\x0D\x31\x35\x30\x33\x31\x33\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x31\x32\x33\x31\x30\x30\x30\x30\x30\x30\x5A\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x55\x6E\x69\x54\x72\x75\x73\x74\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x55\x43\x41\x20\x45\x78\x74\x65\x6E\x64\x65\x64\x20\x56\x61\x6C\x69\x64\x61\x74\x69\x6F\x6E\x20\x52\x6F\x6F\x74\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA9\x09\x07\x28\x13\x02\xB0\x99\xE0\x64\xAA\x1E\x43\x16\x7A\x73\xB1\x91\xA0\x75\x3E\xA8\xFA\xE3\x38\x00\x7A\xEC\x89\x6A\x20\x0F\x8B\xC5\xB0\x9B\x33\x03\x5A\x86\xC6\x58\x86\xD5\xC1\x85\xBB\x4F\xC6\x9C\x40\x4D\xCA\xBE\xEE\x69\x96\xB8\xAD\x81\x30\x9A\x7C\x92\x05\xEB\x05\x2B\x9A\x48\xD0\xB8\x76\x3E\x96\xC8\x20\xBB\xD2\xB0\xF1\x8F\xD8\xAC\x45\x46\xFF\xAA\x67\x60\xB4\x77\x7E\x6A\x1F\x3C\x1A\x52\x7A\x04\x3D\x07\x3C\x85\x0D\x84\xD0\x1F\x76\x0A\xF7\x6A\x14\xDF\x72\xE3\x34\x7C\x57\x4E\x56\x01\x3E\x79\xF1\xAA\x29\x3B\x6C\xFA\xF8\x8F\x6D\x4D\xC8\x35\xDF\xAE\xEB\xDC\x24\xEE\x79\x45\xA7\x85\xB6\x05\x88\xDE\x88\x5D\x25\x7C\x97\x64\x67\x09\xD9\xBF\x5A\x15\x05\x86\xF3\x09\x1E\xEC\x58\x32\x33\x11\xF3\x77\x64\xB0\x76\x1F\xE4\x10\x35\x17\x1B\xF2\x0E\xB1\x6C\xA4\x2A\xA3\x73\xFC\x09\x1F\x1E\x32\x19\x53\x11\xE7\xD9\xB3\x2C\x2E\x76\x2E\xA1\xA3\xDE\x7E\x6A\x88\x09\xE8\xF2\x07\x8A\xF8\xB2\xCD\x10\xE7\xE2\x73\x40\x93\xBB\x08\xD1\x3F\xE1\xFC\x0B\x94\xB3\x25\xEF\x7C\xA6\xD7\xD1\xAF\x9F\xFF\x96\x9A\xF5\x91\x7B\x98\x0B\x77\xD4\x7E\xE8\x07\xD2\x62\xB5\x95\x39\xE3\xF3\xF1\x6D\x0F\x0E\x65\x84\x8A\x63\x54\xC5\x80\xB6\xE0\x9E\x4B\x7D\x47\x26\xA7\x01\x08\x5D\xD1\x88\x9E\xD7\xC3\x32\x44\xFA\x82\x4A\x0A\x68\x54\x7F\x38\x53\x03\xCC\xA4\x00\x33\x64\x51\x59\x0B\xA3\x82\x91\x7A\x5E\xEC\x16\xC2\xF3\x2A\xE6\x62\xDA\x2A\xDB\x59\x62\x10\x25\x4A\x2A\x81\x0B\x47\x07\x43\x06\x70\x87\xD2\xFA\x93\x11\x29\x7A\x48\x4D\xEB\x94\xC7\x70\x4D\xAF\x67\xD5\x51\xB1\x80\x20\x01\x01\xB4\x7A\x08\xA6\x90\x7F\x4E\xE0\xEF\x07\x41\x87\xAF\x6A\xA5\x5E\x8B\xFB\xCF\x50\xB2\x9A\x54\xAF\xC3\x89\xBA\x58\x2D\xF5\x30\x98\xB1\x36\x72\x39\x7E\x49\x04\xFD\x29\xA7\x4C\x79\xE4\x05\x57\xDB\x94\xB9\x16\x53\x8D\x46\xB3\x1D\x95\x61\x57\x56\x7F\xAF\xF0\x16\x5B\x61\x58\x6F\x36\x50\x11\x0B\xD8\xAC\x2B\x95\x16\x1A\x0E\x1F\x08\xCD\x36\x34\x65\x10\x62\x66\xD5\x80\x5F\x14\x20\x5F\x2D\x0C\xA0\x78\x0A\x68\xD6\x2C\xD7\xE9\x6F\x2B\xD2\x4A\x05\x93\xFC\x9E\x6F\x6B\x67\xFF\x88\xF1\x4E\xA5\x69\x4A\x52\x37\x05\xEA\xC6\x16\x8D\xD2\xC4\x99\xD1\x82\x2B\x3B\xBA\x35\x75\xF7\x51\x51\x58\xF3\xC8\x07\xDD\xE4\xB4\x03\x7F\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD9\x74\x3A\xE4\x30\x3D\x0D\xF7\x12\xDC\x7E\x5A\x05\x9F\x1E\x34\x9A\xF7\xE1\x14\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x36\x8D\x97\xCC\x42\x15\x64\x29\x37\x9B\x26\x2C\xD6\xFB\xAE\x15\x69\x2C\x6B\x1A\x1A\xF7\x5F\xB6\xF9\x07\x4C\x59\xEA\xF3\xC9\xC8\xB9\xAE\xCC\xBA\x2E\x7A\xDC\xC0\xF5\xB0\x2D\xC0\x3B\xAF\x9F\x70\x05\x11\x6A\x9F\x25\x4F\x01\x29\x70\xE3\xE5\x0C\xE1\xEA\x5A\x7C\xDC\x49\xBB\xC1\x1E\x2A\x81\xF5\x16\x4B\x72\x91\xC8\xA2\x31\xB9\xAA\xDA\xFC\x9D\x1F\xF3\x5D\x40\x02\x13\xFC\x4E\x1C\x06\xCA\xB3\x14\x90\x54\x17\x19\x12\x1A\xF1\x1F\xD7\x0C\x69\x5A\xF6\x71\x78\xF4\x94\x7D\x91\x0B\x8E\xEC\x90\x54\x8E\xBC\x6F\xA1\x4C\xAB\xFC\x74\x64\xFD\x71\x9A\xF8\x41\x07\xA1\xCD\x91\xE4\x3C\x9A\xE0\x9B\x32\x39\x73\xAB\x2A\xD5\x69\xC8\x78\x91\x26\x31\x7D\xE2\xC7\x30\xF1\xFC\x14\x78\x77\x12\x0E\x13\xF4\xDD\x16\x94\xBF\x4B\x67\x7B\x70\x53\x85\xCA\xB0\xBB\xF3\x38\x4D\x2C\x90\x39\xC0\x0D\xC2\x5D\x6B\xE9\xE2\xE5\xD5\x88\x8D\xD6\x2C\xBF\xAB\x1B\xBE\xB5\x28\x87\x12\x17\x74\x6E\xFC\x7D\xFC\x8F\xD0\x87\x26\xB0\x1B\xFB\xB9\x6C\xAB\xE2\x9E\x3D\x15\xC1\x3B\x2E\x67\x02\x58\x91\x9F\xEF\xF8\x42\x1F\x2C\xB7\x68\xF5\x75\xAD\xCF\xB5\xF6\xFF\x11\x7D\xC2\xF0\x24\xA5\xAD\xD3\xFA\xA0\x3C\xA9\xFA\x5D\xDC\xA5\xA0\xEF\x44\xA4\xBE\xD6\xE8\xE5\xE4\x13\x96\x17\x7B\x06\x3E\x32\xED\xC7\xB7\x42\xBC\x76\xA3\xD8\x65\x38\x2B\x38\x35\x51\x21\x0E\x0E\x6F\x2E\x34\x13\x40\xE1\x2B\x67\x0C\x6D\x4A\x41\x30\x18\x23\x5A\x32\x55\x99\xC9\x17\xE0\x3C\xDE\xF6\xEC\x79\xAD\x2B\x58\x19\xA2\xAD\x2C\x22\x1A\x95\x8E\xBE\x96\x90\x5D\x42\x57\xC4\xF9\x14\x03\x35\x2B\x1C\x2D\x51\x57\x08\xA7\x3A\xDE\x3F\xE4\xC8\xB4\x03\x73\xC2\xC1\x26\x80\xBB\x0B\x42\x1F\xAD\x0D\xAF\x26\x72\xDA\xCC\xBE\xB3\xA3\x83\x58\x0D\x82\xC5\x1F\x46\x51\xE3\x9C\x18\xCC\x8D\x9B\x8D\xEC\x49\xEB\x75\x50\xD5\x8C\x28\x59\xCA\x74\x34\xDA\x8C\x0B\x21\xAB\x1E\xEA\x1B\xE5\xC7\xFD\x15\x3E\xC0\x17\xAA\xFB\x23\x6E\x26\x46\xCB\xFA\xF9\xB1\x72\x6B\x69\xCF\x22\x84\x0B\x62\x0F\xAC\xD9\x19\x00\x94\xA2\x76\x3C\xD4\x2D\x9A\xED\x04\x9E\x2D\x06\x62\x10\x37\x52\x1C\x85\x72\x1B\x27\xE5\xCC\xC6\x31\xEC\x37\xEC\x63\x59\x9B\x0B\x1D\x76\xCC\x7E\x32\x9A\x88\x95\x08\x36\x52\xBB\xDE\x76\x5F\x76\x49\x49\xAD\x7F\xBD\x65\x20\xB2\xC9\xC1\x2B\x76\x18\x76\x9F\x56\xB1"),
   // "Certigna Root CA"
   CertificateDer::from_slice(b"\x30\x82\x06\x5B\x30\x82\x04\x43\xA0\x03\x02\x01\x02\x02\x11\x00\xCA\xE9\x1B\x89\xF1\x55\x03\x0D\xA3\xE6\x41\x6D\xC4\xE3\xA6\xE1\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x52\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x44\x68\x69\x6D\x79\x6F\x74\x69\x73\x31\x1C\x30\x1A\x06\x03\x55\x04\x0B\x0C\x13\x30\x30\x30\x32\x20\x34\x38\x31\x34\x36\x33\x30\x38\x31\x30\x30\x30\x33\x36\x31\x19\x30\x17\x06\x03\x55\x04\x03\x0C\x10\x43\x65\x72\x74\x69\x67\x6E\x61\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x33\x31\x30\x30\x31\x30\x38\x33\x32\x32\x37\x5A\x17\x0D\x33\x33\x31\x30\x30\x31\x30\x38\x33\x32\x32\x37\x5A\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x52\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x44\x68\x69\x6D\x79\x6F\x74\x69\x73\x31\x1C\x30\x1A\x06\x03\x55\x04\x0B\x0C\x13\x30\x30\x30\x32\x20\x34\x38\x31\x34\x36\x33\x30\x38\x31\x30\x30\x30\x33\x36\x31\x19\x30\x17\x06\x03\x55\x04\x03\x0C\x10\x43\x65\x72\x74\x69\x67\x6E\x61\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xCD\x18\x39\x65\x1A\x59\xB1\xEA\x64\x16\x0E\x8C\x94\x24\x95\x7C\x83\xD3\xC5\x39\x26\xDC\x0C\xEF\x16\x57\x8D\xD7\xD8\xAC\xA3\x42\x7F\x82\xCA\xED\xCD\x5B\xDB\x0E\xB7\x2D\xED\x45\x08\x17\xB2\xD9\xB3\xCB\xD6\x17\x52\x72\x28\xDB\x8E\x4E\x9E\x8A\xB6\x0B\xF9\x9E\x84\x9A\x4D\x76\xDE\x22\x29\x5C\xD2\xB3\xD2\x06\x3E\x30\x39\xA9\x74\xA3\x92\x56\x1C\xA1\x6F\x4C\x0A\x20\x6D\x9F\x23\x7A\xB4\xC6\xDA\x2C\xE4\x1D\x2C\xDC\xB3\x28\xD0\x13\xF2\x4C\x4E\x02\x49\xA1\x54\x40\x9E\xE6\xE5\x05\xA0\x2D\x84\xC8\xFF\x98\x6C\xD0\xEB\x8A\x1A\x84\x08\x1E\xB7\x68\x23\xEE\x23\xD5\x70\xCE\x6D\x51\x69\x10\xEE\xA1\x7A\xC2\xD1\x22\x31\xC2\x82\x85\xD2\xF2\x55\x76\x50\x7C\x25\x7A\xC9\x84\x5C\x0B\xAC\xDD\x42\x4E\x2B\xE7\x82\xA2\x24\x89\xCB\x90\xB2\xD0\xEE\x23\xBA\x66\x4C\xBB\x62\xA4\xF9\x53\x5A\x64\x7B\x7C\x98\xFA\xA3\x48\x9E\x0F\x95\xAE\xA7\x18\xF4\x6A\xEC\x2E\x03\x45\xAF\xF0\x74\xF8\x2A\xCD\x7A\x5D\xD1\xBE\x44\x26\x32\x29\xF1\xF1\xF5\x6C\xCC\x7E\x02\x21\x0B\x9F\x6F\xA4\x3F\xBE\x9D\x53\xE2\xCF\x7D\xA9\x2C\x7C\x58\x1A\x97\xE1\x3D\x37\x37\x18\x66\x28\xD2\x40\xC5\x51\x8A\x8C\xC3\x2D\xCE\x53\x88\x24\x58\x64\x30\x16\xC5\xAA\xE0\xD6\x0A\xA6\x40\xDF\x78\xF6\xF5\x04\x7C\x69\x13\x84\xBC\xD1\xD1\xA7\x06\xCF\x01\xF7\x68\xC0\xA8\x57\xBB\x3A\x61\xAD\x04\x8C\x93\xE3\xAD\xFC\xF0\xDB\x44\x6D\x59\xDC\x49\x59\xAE\xAC\x9A\x99\x36\x30\x41\x7B\x76\x33\x22\x87\xA3\xC2\x92\x86\x6E\xF9\x70\xEE\xAE\x87\x87\x95\x1B\xC4\x7A\xBD\x31\xF3\xD4\xD2\xE5\x99\xFF\xBE\x48\xEC\x75\xF5\x78\x16\x1D\xA6\x70\xC1\x7F\x3C\x1B\xA1\x92\xFB\xCF\xC8\x3C\xD6\xC5\x93\x0A\x8F\xF5\x55\x3A\x76\x95\xCE\x59\x98\x8A\x09\x95\x77\x32\x9A\x83\xBA\x2C\x04\x3A\x97\xBD\xD4\x2F\xBE\xD7\x6C\x9B\xA2\xCA\x7D\x6D\x26\xC9\x55\xD5\xCF\xC3\x79\x52\x08\x09\x99\x07\x24\x2D\x64\x25\x6B\xA6\x21\x69\x9B\x6A\xDD\x74\x4D\x6B\x97\x7A\x41\xBD\xAB\x17\xF9\x90\x17\x48\x8F\x36\xF9\x2D\xD5\xC5\xDB\xEE\xAA\x85\x45\x41\xFA\xCD\x3A\x45\xB1\x68\xE6\x36\x4C\x9B\x90\x57\xEC\x23\xB9\x87\x08\xC2\xC4\x09\xF1\x97\x86\x2A\x28\x4D\xE2\x74\xC0\xDA\xC4\x8C\xDB\xDF\xE2\xA1\x17\x59\xCE\x24\x59\x74\x31\xDA\x7F\xFD\x30\x6D\xD9\xDC\xE1\x6A\xE1\xFC\x5F\x02\x03\x01\x00\x01\xA3\x82\x01\x1A\x30\x82\x01\x16\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x18\x87\x56\xE0\x6E\x77\xEE\x24\x35\x3C\x4E\x73\x9A\x1F\xD6\xE1\xE2\x79\x7E\x2B\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x18\x87\x56\xE0\x6E\x77\xEE\x24\x35\x3C\x4E\x73\x9A\x1F\xD6\xE1\xE2\x79\x7E\x2B\x30\x44\x06\x03\x55\x1D\x20\x04\x3D\x30\x3B\x30\x39\x06\x04\x55\x1D\x20\x00\x30\x31\x30\x2F\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x01\x16\x23\x68\x74\x74\x70\x73\x3A\x2F\x2F\x77\x77\x77\x77\x2E\x63\x65\x72\x74\x69\x67\x6E\x61\x2E\x66\x72\x2F\x61\x75\x74\x6F\x72\x69\x74\x65\x73\x2F\x30\x6D\x06\x03\x55\x1D\x1F\x04\x66\x30\x64\x30\x2F\xA0\x2D\xA0\x2B\x86\x29\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x63\x65\x72\x74\x69\x67\x6E\x61\x2E\x66\x72\x2F\x63\x65\x72\x74\x69\x67\x6E\x61\x72\x6F\x6F\x74\x63\x61\x2E\x63\x72\x6C\x30\x31\xA0\x2F\xA0\x2D\x86\x2B\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x64\x68\x69\x6D\x79\x6F\x74\x69\x73\x2E\x63\x6F\x6D\x2F\x63\x65\x72\x74\x69\x67\x6E\x61\x72\x6F\x6F\x74\x63\x61\x2E\x63\x72\x6C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x94\xB8\x9E\x4F\xF0\xE3\x95\x08\x22\xE7\xCD\x68\x41\xF7\x1C\x55\xD5\x7C\x00\xE2\x2D\x3A\x89\x5D\x68\x38\x2F\x51\x22\x0B\x4A\x8D\xCB\xE9\xBB\x5D\x3E\xBB\x5C\x3D\xB1\x28\xFE\xE4\x53\x55\x13\xCF\xA1\x90\x1B\x02\x1D\x5F\x66\x46\x09\x33\x28\xE1\x0D\x24\x97\x70\xD3\x10\x1F\xEA\x64\x57\x96\xBB\x5D\xDA\xE7\xC4\x8C\x4F\x4C\x64\x46\x1D\x5C\x87\xE3\x59\xDE\x42\xD1\x9B\xA8\x7E\xA6\x89\xDD\x8F\x1C\xC9\x30\x82\xED\x3B\x9C\xCD\xC0\xE9\x19\xE0\x6A\xD8\x02\x75\x37\xAB\xF7\x34\x28\x28\x91\xF2\x04\x0A\x4F\x35\xE3\x60\x26\x01\xFA\xD0\x11\x8C\xF9\x11\x6A\xEE\xAF\x3D\xC3\x50\xD3\x8F\x5F\x33\x79\x3C\x86\xA8\x73\x45\x90\x8C\x20\xB6\x72\x73\x17\x23\xBE\x07\x65\xE5\x78\x92\x0D\xBA\x01\xC0\xEB\x8C\x1C\x66\xBF\xAC\x86\x77\x01\x94\x0D\x9C\xE6\xE9\x39\x8D\x1F\xA6\x51\x8C\x99\x0C\x39\x77\xE1\xB4\x9B\xFA\x1C\x67\x57\x6F\x6A\x6A\x8E\xA9\x2B\x4C\x57\x79\x7A\x57\x22\xCF\xCD\x5F\x63\x46\x8D\x5C\x59\x3A\x86\xF8\x32\x47\x62\xA3\x67\x0D\x18\x91\xDC\xFB\xA6\x6B\xF5\x48\x61\x73\x23\x59\x8E\x02\xA7\xBC\x44\xEA\xF4\x49\x9D\xF1\x54\x58\xF9\x60\xAF\xDA\x18\xA4\x2F\x28\x45\xDC\x7A\xA0\x88\x86\x5D\xF3\x3B\xE7\xFF\x29\x35\x80\xFC\x64\x43\x94\xE6\xE3\x1C\x6F\xBE\xAD\x0E\x2A\x63\x99\x2B\xC9\x7E\x85\xF6\x71\xE8\x06\x03\x95\xFE\xDE\x8F\x48\x1C\x5A\xD4\x92\xE8\x2B\xEE\xE7\x31\xDB\xBA\x04\x6A\x87\x98\xE7\xC5\x5F\xEF\x7D\xA7\x22\xF7\x01\xD8\x4D\xF9\x89\xD0\x0E\x9A\x05\x59\xA4\x9E\x98\xD9\x6F\x2B\xCA\x70\xBE\x64\xC2\x55\xA3\xF4\xE9\xAF\xC3\x92\x29\xDC\x88\x16\x24\x99\x3C\x8D\x26\x98\xB6\x5B\xB7\xCC\xCE\xB7\x37\x07\xFD\x26\xD9\x98\x85\x24\xFF\x59\x23\x03\x9A\xED\x9D\x9D\xA8\xE4\x5E\x38\xCE\xD7\x52\x0D\x6F\xD2\x3F\x6D\xB1\x05\x6B\x49\xCE\x8A\x91\x46\x73\xF4\xF6\x2F\xF0\xA8\x73\x77\x0E\x65\xAC\xA1\x8D\x66\x52\x69\x7E\x4B\x68\x0C\xC7\x1E\x37\x27\x83\xA5\x8C\xC7\x02\xE4\x14\xCD\x49\x01\xB0\x73\xB3\xFD\xC6\x90\x3A\x6F\xD2\x6C\xED\x3B\xEE\xEC\x91\xBE\xA2\x43\x5D\x8B\x00\x4A\x66\x25\x44\x70\xDE\x40\x0F\xF8\x7C\x15\xF7\xA2\xCE\x3C\xD7\x5E\x13\x8C\x81\x17\x18\x17\xD1\xBD\xF1\x77\x10\x3A\xD4\x65\x39\xC1\x27\xAC\x57\x2C\x25\x54\xFF\xA2\xDA\x4F\x8A\x61\x39\x5E\xAE\x3D\x4A\x8C\xBD"),
   // "GTS Root R1"
   CertificateDer::from_slice(b"\x30\x82\x05\x57\x30\x82\x03\x3F\xA0\x03\x02\x01\x02\x02\x0D\x02\x03\xE5\x93\x6F\x31\xB0\x13\x49\x88\x6B\xA2\x17\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x31\x30\x1E\x17\x0D\x31\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x36\x30\x36\x32\x32\x30\x30\x30\x30\x30\x30\x5A\x30\x47\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x22\x30\x20\x06\x03\x55\x04\x0A\x13\x19\x47\x6F\x6F\x67\x6C\x65\x20\x54\x72\x75\x73\x74\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x4C\x4C\x43\x31\x14\x30\x12\x06\x03\x55\x04\x03\x13\x0B\x47\x54\x53\x20\x52\x6F\x6F\x74\x20\x52\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xB6\x11\x02\x8B\x1E\xE3\xA1\x77\x9B\x3B\xDC\xBF\x94\x3E\xB7\x95\xA7\x40\x3C\xA1\xFD\x82\xF9\x7D\x32\x06\x82\x71\xF6\xF6\x8C\x7F\xFB\xE8\xDB\xBC\x6A\x2E\x97\x97\xA3\x8C\x4B\xF9\x2B\xF6\xB1\xF9\xCE\x84\x1D\xB1\xF9\xC5\x97\xDE\xEF\xB9\xF2\xA3\xE9\xBC\x12\x89\x5E\xA7\xAA\x52\xAB\xF8\x23\x27\xCB\xA4\xB1\x9C\x63\xDB\xD7\x99\x7E\xF0\x0A\x5E\xEB\x68\xA6\xF4\xC6\x5A\x47\x0D\x4D\x10\x33\xE3\x4E\xB1\x13\xA3\xC8\x18\x6C\x4B\xEC\xFC\x09\x90\xDF\x9D\x64\x29\x25\x23\x07\xA1\xB4\xD2\x3D\x2E\x60\xE0\xCF\xD2\x09\x87\xBB\xCD\x48\xF0\x4D\xC2\xC2\x7A\x88\x8A\xBB\xBA\xCF\x59\x19\xD6\xAF\x8F\xB0\x07\xB0\x9E\x31\xF1\x82\xC1\xC0\xDF\x2E\xA6\x6D\x6C\x19\x0E\xB5\xD8\x7E\x26\x1A\x45\x03\x3D\xB0\x79\xA4\x94\x28\xAD\x0F\x7F\x26\xE5\xA8\x08\xFE\x96\xE8\x3C\x68\x94\x53\xEE\x83\x3A\x88\x2B\x15\x96\x09\xB2\xE0\x7A\x8C\x2E\x75\xD6\x9C\xEB\xA7\x56\x64\x8F\x96\x4F\x68\xAE\x3D\x97\xC2\x84\x8F\xC0\xBC\x40\xC0\x0B\x5C\xBD\xF6\x87\xB3\x35\x6C\xAC\x18\x50\x7F\x84\xE0\x4C\xCD\x92\xD3\x20\xE9\x33\xBC\x52\x99\xAF\x32\xB5\x29\xB3\x25\x2A\xB4\x48\xF9\x72\xE1\xCA\x64\xF7\xE6\x82\x10\x8D\xE8\x9D\xC2\x8A\x88\xFA\x38\x66\x8A\xFC\x63\xF9\x01\xF9\x78\xFD\x7B\x5C\x77\xFA\x76\x87\xFA\xEC\xDF\xB1\x0E\x79\x95\x57\xB4\xBD\x26\xEF\xD6\x01\xD1\xEB\x16\x0A\xBB\x8E\x0B\xB5\xC5\xC5\x8A\x55\xAB\xD3\xAC\xEA\x91\x4B\x29\xCC\x19\xA4\x32\x25\x4E\x2A\xF1\x65\x44\xD0\x02\xCE\xAA\xCE\x49\xB4\xEA\x9F\x7C\x83\xB0\x40\x7B\xE7\x43\xAB\xA7\x6C\xA3\x8F\x7D\x89\x81\xFA\x4C\xA5\xFF\xD5\x8E\xC3\xCE\x4B\xE0\xB5\xD8\xB3\x8E\x45\xCF\x76\xC0\xED\x40\x2B\xFD\x53\x0F\xB0\xA7\xD5\x3B\x0D\xB1\x8A\xA2\x03\xDE\x31\xAD\xCC\x77\xEA\x6F\x7B\x3E\xD6\xDF\x91\x22\x12\xE6\xBE\xFA\xD8\x32\xFC\x10\x63\x14\x51\x72\xDE\x5D\xD6\x16\x93\xBD\x29\x68\x33\xEF\x3A\x66\xEC\x07\x8A\x26\xDF\x13\xD7\x57\x65\x78\x27\xDE\x5E\x49\x14\x00\xA2\x00\x7F\x9A\xA8\x21\xB6\xA9\xB1\x95\xB0\xA5\xB9\x0D\x16\x11\xDA\xC7\x6C\x48\x3C\x40\xE0\x7E\x0D\x5A\xCD\x56\x3C\xD1\x97\x05\xB9\xCB\x4B\xED\x39\x4B\x9C\xC4\x3F\xD2\x55\x13\x6E\x24\xB0\xD6\x71\xFA\xF4\xC1\xBA\xCC\xED\x1B\xF5\xFE\x81\x41\xD8\x00\x98\x3D\x3A\xC8\xAE\x7A\x98\x37\x18\x05\x95\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE4\xAF\x2B\x26\x71\x1A\x2B\x48\x27\x85\x2F\x52\x66\x2C\xEF\xF0\x89\x13\x71\x3E\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x9F\xAA\x42\x26\xDB\x0B\x9B\xBE\xFF\x1E\x96\x92\x2E\x3E\xA2\x65\x4A\x6A\x98\xBA\x22\xCB\x7D\xC1\x3A\xD8\x82\x0A\x06\xC6\xF6\xA5\xDE\xC0\x4E\x87\x66\x79\xA1\xF9\xA6\x58\x9C\xAA\xF9\xB5\xE6\x60\xE7\xE0\xE8\xB1\x1E\x42\x41\x33\x0B\x37\x3D\xCE\x89\x70\x15\xCA\xB5\x24\xA8\xCF\x6B\xB5\xD2\x40\x21\x98\xCF\x22\x34\xCF\x3B\xC5\x22\x84\xE0\xC5\x0E\x8A\x7C\x5D\x88\xE4\x35\x24\xCE\x9B\x3E\x1A\x54\x1E\x6E\xDB\xB2\x87\xA7\xFC\xF3\xFA\x81\x55\x14\x62\x0A\x59\xA9\x22\x05\x31\x3E\x82\xD6\xEE\xDB\x57\x34\xBC\x33\x95\xD3\x17\x1B\xE8\x27\xA2\x8B\x7B\x4E\x26\x1A\x7A\x5A\x64\xB6\xD1\xAC\x37\xF1\xFD\xA0\xF3\x38\xEC\x72\xF0\x11\x75\x9D\xCB\x34\x52\x8D\xE6\x76\x6B\x17\xC6\xDF\x86\xAB\x27\x8E\x49\x2B\x75\x66\x81\x10\x21\xA6\xEA\x3E\xF4\xAE\x25\xFF\x7C\x15\xDE\xCE\x8C\x25\x3F\xCA\x62\x70\x0A\xF7\x2F\x09\x66\x07\xC8\x3F\x1C\xFC\xF0\xDB\x45\x30\xDF\x62\x88\xC1\xB5\x0F\x9D\xC3\x9F\x4A\xDE\x59\x59\x47\xC5\x87\x22\x36\xE6\x82\xA7\xED\x0A\xB9\xE2\x07\xA0\x8D\x7B\x7A\x4A\x3C\x71\xD2\xE2\x03\xA1\x1F\x32\x07\xDD\x1B\xE4\x42\xCE\x0C\x00\x45\x61\x80\xB5\x0B\x20\x59\x29\x78\xBD\xF9\x55\xCB\x63\xC5\x3C\x4C\xF4\xB6\xFF\xDB\x6A\x5F\x31\x6B\x99\x9E\x2C\xC1\x6B\x50\xA4\xD7\xE6\x18\x14\xBD\x85\x3F\x67\xAB\x46\x9F\xA0\xFF\x42\xA7\x3A\x7F\x5C\xCB\x5D\xB0\x70\x1D\x2B\x34\xF5\xD4\x76\x09\x0C\xEB\x78\x4C\x59\x05\xF3\x33\x42\xC3\x61\x15\x10\x1B\x77\x4D\xCE\x22\x8C\xD4\x85\xF2\x45\x7D\xB7\x53\xEA\xEF\x40\x5A\x94\x0A\x5C\x20\x5F\x4E\x40\x5D\x62\x22\x76\xDF\xFF\xCE\x61\xBD\x8C\x23\x78\xD2\x37\x02\xE0\x8E\xDE\xD1\x11\x37\x89\xF6\xBF\xED\x49\x07\x62\xAE\x92\xEC\x40\x1A\xAF\x14\x09\xD9\xD0\x4E\xB2\xA2\xF7\xBE\xEE\xEE\xD8\xFF\xDC\x1A\x2D\xDE\xB8\x36\x71\xE2\xFC\x79\xB7\x94\x25\xD1\x48\x73\x5B\xA1\x35\xE7\xB3\x99\x67\x75\xC1\x19\x3A\x2B\x47\x4E\xD3\x42\x8E\xFD\x31\xC8\x16\x66\xDA\xD2\x0C\x3C\xDB\xB3\x8E\xC9\xA1\x0D\x80\x0F\x7B\x16\x77\x14\xBF\xFF\xDB\x09\x94\xB2\x93\xBC\x20\x58\x15\xE9\xDB\x71\x43\xF3\xDE\x10\xC3\x00\xDC\xA8\x2A\x95\xB6\xC2\xD6\x3F\x90\x6B\x76\xDB\x6C\xFE\x8C\xBC\xF2\x70\x35\x0C\xDC\x99\x19\x35\xDC\xD7\xC8\x46\x63\xD5\x36\x71\xAE\x57\xFB\xB7\x82\x6D\xDC"),
   // "HARICA TLS RSA Root CA 2021"
   CertificateDer::from_slice(b"\x30\x82\x05\xA4\x30\x82\x03\x8C\xA0\x03\x02\x01\x02\x02\x10\x39\xCA\x93\x1C\xEF\x43\xF3\xC6\x8E\x93\xC7\xF4\x64\x89\x38\x7E\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x6C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x37\x30\x35\x06\x03\x55\x04\x0A\x0C\x2E\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x41\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x48\x41\x52\x49\x43\x41\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x31\x30\x1E\x17\x0D\x32\x31\x30\x32\x31\x39\x31\x30\x35\x35\x33\x38\x5A\x17\x0D\x34\x35\x30\x32\x31\x33\x31\x30\x35\x35\x33\x37\x5A\x30\x6C\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52\x31\x37\x30\x35\x06\x03\x55\x04\x0A\x0C\x2E\x48\x65\x6C\x6C\x65\x6E\x69\x63\x20\x41\x63\x61\x64\x65\x6D\x69\x63\x20\x61\x6E\x64\x20\x52\x65\x73\x65\x61\x72\x63\x68\x20\x49\x6E\x73\x74\x69\x74\x75\x74\x69\x6F\x6E\x73\x20\x43\x41\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x48\x41\x52\x49\x43\x41\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x32\x30\x32\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x8B\xC2\xE7\xAF\x65\x9B\x05\x67\x96\xC9\x0D\x24\xB9\xD0\x0E\x64\xFC\xCE\xE2\x24\x18\x2C\x84\x7F\x77\x51\xCB\x04\x11\x36\xB8\x5E\xED\x69\x71\xA7\x9E\xE4\x25\x09\x97\x67\xC1\x47\xC2\xCF\x91\x16\x36\x62\x3D\x38\x04\xE1\x51\x82\xFF\xAC\xD2\xB4\x69\xDD\x2E\xEC\x11\xA3\x45\xEE\x6B\x6B\x3B\x4C\xBF\x8C\x8D\xA4\x1E\x9D\x11\xB9\xE9\x38\xF9\x7A\x0E\x0C\x98\xE2\x23\x1D\xD1\x4E\x63\xD4\xE7\xB8\x41\x44\xFB\x6B\xAF\x6B\xDA\x1F\xD3\xC5\x91\x88\x5B\xA4\x89\x92\xD1\x81\xE6\x8C\x39\x58\xA0\xD6\x69\x43\xA9\xAD\x98\x52\x58\x6E\xDB\x0A\xFB\x6B\xCF\x68\xFA\xE3\xA4\x5E\x3A\x45\x73\x98\x07\xEA\x5F\x02\x72\xDE\x0C\xA5\xB3\x9F\xAE\xA9\x1D\xB7\x1D\xB3\xFC\x8A\x59\xE7\x6E\x72\x65\xAD\xF5\x30\x94\x23\x07\xF3\x82\x16\x4B\x35\x98\x9C\x53\xBB\x2F\xCA\xE4\x5A\xD9\xC7\x8D\x1D\xFC\x98\x99\xFB\x2C\xA4\x82\x6B\xF0\x2A\x1F\x8E\x0B\x5F\x71\x5C\x5C\xAE\x42\x7B\x29\x89\x81\xCB\x03\xA3\x99\xCA\x88\x9E\x0B\x40\x09\x41\x33\xDB\xE6\x58\x7A\xFD\xAE\x99\x70\xC0\x5A\x0F\xD6\x13\x86\x71\x2F\x76\x69\xFC\x90\xDD\xDB\x2D\x6E\xD1\xF2\x9B\xF5\x1A\x6B\x9E\x6F\x15\x8C\x7A\xF0\x4B\x28\xA0\x22\x38\x80\x24\x6C\x36\xA4\x3B\xF2\x30\x91\xF3\x78\x13\xCF\xC1\x3F\x35\xAB\xF1\x1D\x11\x23\xB5\x43\x22\x9E\x01\x92\xB7\x18\x02\xE5\x11\xD1\x82\xDB\x15\x00\xCC\x61\x37\xC1\x2A\x7C\x9A\xE1\xD0\xBA\xB3\x50\x46\xEE\x82\xAC\x9D\x31\xF8\xFB\x23\xE2\x03\x00\x48\x70\xA3\x09\x26\x79\x15\x53\x60\xF3\x38\x5C\xAD\x38\xEA\x81\x00\x63\x14\xB9\x33\x5E\xDD\x0B\xDB\xA0\x45\x07\x1A\x33\x09\xF8\x4D\xB4\xA7\x02\xA6\x69\xF4\xC2\x59\x05\x88\x65\x85\x56\xAE\x4B\xCB\xE0\xDE\x3C\x7D\x2D\x1A\xC8\xE9\xFB\x1F\xA3\x61\x4A\xD6\x2A\x13\xAD\x77\x4C\x1A\x18\x9B\x91\x0F\x58\xD8\x06\x54\xC5\x97\xF8\xAA\x3F\x20\x8A\xA6\x85\xA6\x77\xF6\xA6\xFC\x1C\xE2\xEE\x6E\x94\x33\x2A\x83\x50\x84\x0A\xE5\x4F\x86\xF8\x50\x45\x78\x00\x81\xEB\x5B\x68\xE3\x26\x8D\xCC\x7B\x5C\x51\xF4\x14\x2C\x40\xBE\x1A\x60\x1D\x7A\x72\x61\x1D\x1F\x63\x2D\x88\xAA\xCE\xA2\x45\x90\x08\xFC\x6B\xBE\xB3\x50\x2A\x5A\xFD\xA8\x48\x18\x46\xD6\x90\x40\x92\x90\x0A\x84\x5E\x68\x31\xF8\xEB\xED\x0D\xD3\x1D\xC6\x7D\x99\x18\x55\x56\x27\x65\x2E\x8D\x45\xC5\x24\xEC\xCE\xE3\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x0A\x48\x23\xA6\x60\xA4\x92\x0A\x33\xEA\x93\x5B\xC5\x57\xEA\x25\x4D\xBD\x12\xEE\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x3E\x90\x48\xAA\x6E\x62\x15\x25\x66\x7B\x0C\xD5\x8C\x8B\x89\x9D\xD7\xED\x4E\x07\xEF\x9C\xD0\x14\x5F\x5E\x50\xBD\x68\x96\x90\xA4\x14\x11\xAA\x68\x6D\x09\x35\x39\x40\x09\xDA\xF4\x09\x2C\x34\xA5\x7B\x59\x84\x49\x29\x97\x74\xC8\x07\x1E\x47\x6D\xF2\xCE\x1C\x50\x26\xE3\x9E\x3D\x40\x53\x3F\xF7\x7F\x96\x76\x10\xC5\x46\xA5\xD0\x20\x4B\x50\xF4\x35\x3B\x18\xF4\x55\x6A\x41\x1B\x47\x06\x68\x3C\xBB\x09\x08\x62\xD9\x5F\x55\x42\xAA\xAC\x53\x85\xAC\x95\x56\x36\x56\xAB\xE4\x05\x8C\xC5\xA8\xDA\x1F\xA3\x69\xBD\x53\x0F\xC4\xFF\xDC\xCA\xE3\x7E\xF2\x4C\x88\x86\x47\x46\x1A\xF3\x00\xF5\x80\x91\xA2\xDC\x43\x42\x94\x9B\x20\xF0\xD1\xCD\xB2\xEB\x2C\x53\xC2\x53\x78\x4A\x4F\x04\x94\x41\x9A\x8F\x27\x32\xC1\xE5\x49\x19\xBF\xF1\xF2\xC2\x8B\xA8\x0A\x39\x31\x28\xB4\x7D\x62\x36\x2C\x4D\xEC\x1F\x33\xB6\x7E\x77\x6D\x7E\x50\xF0\x9F\x0E\xD7\x11\x8F\xCF\x18\xC5\xE3\x27\xFE\x26\xEF\x05\x9D\xCF\xCF\x37\xC5\xD0\x7B\xDA\x3B\xB0\x16\x84\x0C\x3A\x93\xD6\xBE\x17\xDB\x0F\x3E\x0E\x19\x78\x09\xC7\xA9\x02\x72\x22\x4B\xF7\x37\x76\xBA\x75\xC4\x85\x03\x5A\x63\xD5\xB1\x75\x05\xC2\xB9\xBD\x94\xAD\x8C\x15\x99\xA7\x93\x7D\xF6\xC5\xF3\xAA\x74\xCF\x04\x85\x94\x98\x00\xF4\xE2\xF9\xCA\x24\x65\xBF\xE0\x62\xAF\xC8\xC5\xFA\xB2\xC9\x9E\x56\x48\xDA\x79\xFD\x96\x76\x15\xBE\xA3\x8E\x56\xC4\xB3\x34\xFC\xBE\x47\xF4\xC1\xB4\xA8\xFC\xD5\x30\x88\x68\xEE\xCB\xAE\xC9\x63\xC4\x76\xBE\xAC\x38\x18\xE1\x5E\x5C\xCF\xAE\x3A\x22\x51\xEB\xD1\x8B\xB3\xF3\x2B\x33\x07\x54\x87\xFA\xB4\xB2\x13\x7B\xBA\x53\x04\x62\x01\x9D\xF1\xC0\x4F\xEE\xE1\x3A\xD4\x8B\x20\x10\xFA\x02\x57\xE6\xEF\xC1\x0B\xB7\x90\x46\x9C\x19\x29\x8C\xDC\x6F\xA0\x4A\x69\x69\x94\xB7\x24\x65\xA0\xFF\xAC\x3F\xCE\x01\xFB\x21\x2E\xFD\x68\xF8\x9B\xF2\xA5\xCF\x31\x38\x5C\x15\xAA\xE6\x97\x00\xC1\xDF\x5A\xA5\xA7\x39\xAA\xE9\x84\x7F\x3C\x51\xA8\x3A\xD9\x94\x5B\x8C\xBF\x4F\x08\x71\xE5\xDB\xA8\x5C\xD4\xD2\xA6\xFE\x00\xA3\xC6\x16\xC7\x0F\xE8\x80\xCE\x1C\x28\x64\x74\x19\x08\xD3\x42\xE3\xCE\x00\x5D\x7F\xB1\xDC\x13\xB0\xE1\x05\xCB\xD1\x20\xAA\x86\x74\x9E\x39\xE7\x91\xFD\xFF\x5B\xD6\xF7\xAD\xA6\x2F\x03\x0B\x6D\xE3\x57\x54\xEB\x76\x53\x18\x8D\x11\x98\xBA"),
   // "TeliaSonera Root CA v1"
   CertificateDer::from_slice(b"\x30\x82\x05\x38\x30\x82\x03\x20\xA0\x03\x02\x01\x02\x02\x11\x00\x95\xBE\x16\xA0\xF7\x2E\x46\xF1\x7B\x39\x82\x72\xFA\x8B\xCD\x96\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x37\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x0C\x0B\x54\x65\x6C\x69\x61\x53\x6F\x6E\x65\x72\x61\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x0C\x16\x54\x65\x6C\x69\x61\x53\x6F\x6E\x65\x72\x61\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x76\x31\x30\x1E\x17\x0D\x30\x37\x31\x30\x31\x38\x31\x32\x30\x30\x35\x30\x5A\x17\x0D\x33\x32\x31\x30\x31\x38\x31\x32\x30\x30\x35\x30\x5A\x30\x37\x31\x14\x30\x12\x06\x03\x55\x04\x0A\x0C\x0B\x54\x65\x6C\x69\x61\x53\x6F\x6E\x65\x72\x61\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x0C\x16\x54\x65\x6C\x69\x61\x53\x6F\x6E\x65\x72\x61\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x76\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC2\xBE\xEB\x27\xF0\x21\xA3\xF3\x69\x26\x55\x7E\x9D\xC5\x55\x16\x91\x5C\xFD\xEF\x21\xBF\x53\x80\x7A\x2D\xD2\x91\x8C\x63\x31\xF0\xEC\x24\xF0\xC3\xA5\xD2\x72\x7C\x10\x6D\xF4\x37\xB7\xE5\xE6\x7C\x79\xEA\x8C\xB5\x82\x8B\xAE\x48\xB6\xAC\x00\xDC\x65\x75\xEC\x2A\x4D\x5F\xC1\x87\xF5\x20\x65\x2B\x81\xA8\x47\x3E\x89\x23\x95\x30\x16\x90\x7F\xE8\x57\x07\x48\xE7\x19\xAE\xBF\x45\x67\xB1\x37\x1B\x06\x2A\xFE\xDE\xF9\xAC\x7D\x83\xFB\x5E\xBA\xE4\x8F\x97\x67\xBE\x4B\x8E\x8D\x64\x07\x57\x38\x55\x69\x34\x36\x3D\x13\x48\xEF\x4F\xE2\xD3\x66\x1E\xA4\xCF\x1A\xB7\x5E\x36\x33\xD4\xB4\x06\xBD\x18\x01\xFD\x77\x84\x50\x00\x45\xF5\x8C\x5D\xE8\x23\xBC\x7E\xFE\x35\xE1\xED\x50\x7B\xA9\x30\x8D\x19\xD3\x09\x8E\x68\x67\x5D\xBF\x3C\x97\x18\x53\xBB\x29\x62\xC5\xCA\x5E\x72\xC1\xC7\x96\xD4\xDB\x2D\xA0\xB4\x1F\x69\x03\xEC\xEA\xE2\x50\xF1\x0C\x3C\xF0\xAC\xF3\x53\x2D\xF0\x1C\xF5\xED\x6C\x39\x39\x73\x80\x16\xC8\x52\xB0\x23\xCD\xE0\x3E\xDC\xDD\x3C\x47\xA0\xBB\x35\x8A\xE2\x98\x68\x8B\xBE\xE5\xBF\x72\xEE\xD2\xFA\xA5\xED\x12\xED\xFC\x98\x18\xA9\x26\x76\xDC\x28\x4B\x10\x20\x1C\xD3\x7F\x16\x77\x2D\xED\x6F\x80\xF7\x49\xBB\x53\x05\xBB\x5D\x68\xC7\xD4\xC8\x75\x16\x3F\x89\x5A\x8B\xF7\x17\x47\xD4\x4C\xF1\xD2\x89\x79\x3E\x4D\x3D\x98\xA8\x61\xDE\x3A\x1E\xD2\xF8\x5E\x03\xE0\xC1\xC9\x1C\x8C\xD3\x8D\x4D\xD3\x95\x36\xB3\x37\x5F\x63\x63\x9B\x33\x14\xF0\x2D\x26\x6B\x53\x7C\x89\x8C\x32\xC2\x6E\xEC\x3D\x21\x00\x39\xC9\xA1\x68\xE2\x50\x83\x2E\xB0\x3A\x2B\xF3\x36\xA0\xAC\x2F\xE4\x6F\x61\xC2\x51\x09\x39\x3E\x8B\x53\xB9\xBB\x67\xDA\xDC\x53\xB9\x76\x59\x36\x9D\x43\xE5\x20\xE0\x3D\x32\x60\x85\x22\x51\xB7\xC7\x33\xBB\xDD\x15\x2F\xA4\x78\xA6\x07\x7B\x81\x46\x36\x04\x86\xDD\x79\x35\xC7\x95\x2C\x3B\xB0\xA3\x17\x35\xE5\x73\x1F\xB4\x5C\x59\xEF\xDA\xEA\x10\x65\x7B\x7A\xD0\x7F\x9F\xB3\xB4\x2A\x37\x3B\x70\x8B\x9B\x5B\xB9\x2B\xB7\xEC\xB2\x51\x12\x97\x53\x29\x5A\xD4\xF0\x12\x10\xDC\x4F\x02\xBB\x12\x92\x2F\x62\xD4\x3F\x69\x43\x7C\x0D\xD6\xFC\x58\x75\x01\x88\x9D\x58\x16\x4B\xDE\xBA\x90\xFF\x47\x01\x89\x06\x6A\xF6\x5F\xB2\x90\x6A\xB3\x02\xA6\x02\x88\xBF\xB3\x47\x7E\x2A\xD9\xD5\xFA\x68\x78\x35\x4D\x02\x03\x01\x00\x01\xA3\x3F\x30\x3D\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0B\x06\x03\x55\x1D\x0F\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF0\x8F\x59\x38\x00\xB3\xF5\x8F\x9A\x96\x0C\xD5\xEB\xFA\x7B\xAA\x17\xE8\x13\x12\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00\xBE\xE4\x5C\x62\x4E\x24\xF4\x0C\x08\xFF\xF0\xD3\x0C\x68\xE4\x93\x49\x22\x3F\x44\x27\x6F\xBB\x6D\xDE\x83\x66\xCE\xA8\xCC\x0D\xFC\xF5\x9A\x06\xE5\x77\x14\x91\xEB\x9D\x41\x7B\x99\x2A\x84\xE5\xFF\xFC\x21\xC1\x5D\xF0\xE4\x1F\x57\xB7\x75\xA9\xA1\x5F\x02\x26\xFF\xD7\xC7\xF7\x4E\xDE\x4F\xF8\xF7\x1C\x46\xC0\x7A\x4F\x40\x2C\x22\x35\xF0\x19\xB1\xD0\x6B\x67\x2C\xB0\xA8\xE0\xC0\x40\x37\x35\xF6\x84\x5C\x5C\xE3\xAF\x42\x78\xFE\xA7\xC9\x0D\x50\xEA\x0D\x84\x76\xF6\x51\xEF\x83\x53\xC6\x7A\xFF\x0E\x56\x49\x2E\x8F\x7A\xD6\x0C\xE6\x27\x54\xE3\x4D\x0A\x60\x72\x62\xCD\x91\x07\xD6\xA5\xBF\xC8\x99\x6B\xED\xC4\x19\xE6\xAB\x4C\x11\x38\xC5\x6F\x31\xE2\x6E\x49\xC8\x3F\x76\x80\x26\x03\x26\x29\xE0\x36\xF6\xF6\x20\x53\xE3\x17\x70\x34\x17\x9D\x63\x68\x1E\x6B\xEC\xC3\x4D\x86\xB8\x13\x30\x2F\x5D\x46\x0D\x47\x43\xD5\x1B\xAA\x59\x0E\xB9\x5C\x8D\x06\x48\xAD\x74\x87\x5F\xC7\xFC\x31\x54\x41\x13\xE2\xC7\x21\x0E\x9E\xE0\x1E\x0D\xE1\xC0\x7B\x43\x85\x90\xC5\x8A\x58\xC6\x65\x0A\x78\x57\xF2\xC6\x23\x0F\x01\xD9\x20\x4B\xDE\x0F\xFB\x92\x85\x75\x2A\x5C\x73\x8D\x6D\x7B\x25\x91\xCA\xEE\x45\xAE\x06\x4B\x00\xCC\xD3\xB1\x59\x50\xDA\x3A\x88\x3B\x29\x43\x46\x5E\x97\x2B\x54\xCE\x53\x6F\x8D\x4A\xE7\x96\xFA\xBF\x71\x0E\x42\x8B\x7C\xFD\x28\xA0\xD0\x48\xCA\xDA\xC4\x81\x4C\xBB\xA2\x73\x93\x26\xC8\xEB\x0C\xD6\x26\x88\xB6\xC0\x24\xCF\xBB\xBD\x5B\xEB\x75\x7D\xE9\x08\x8E\x86\x33\x2C\x79\x77\x09\x69\xA5\x89\xFC\xB3\x70\x90\x87\x76\x8F\xD3\x22\xBB\x42\xCE\xBD\x73\x0B\x20\x26\x2A\xD0\x9B\x3D\x70\x1E\x24\x6C\xCD\x87\x76\xA9\x17\x96\xB7\xCF\x0D\x92\xFB\x8E\x18\xA9\x98\x49\xD1\x9E\xFE\x60\x44\x72\x21\xB9\x19\xED\xC2\xF5\x31\xF1\x39\x48\x88\x90\x24\x75\x54\x16\xAD\xCE\xF4\xF8\x69\x14\x64\x39\xFB\xA3\xB8\xBA\x70\x40\xC7\x27\x1C\xBF\xC4\x56\x53\xFA\x63\x65\xD0\xF3\x1C\x0E\x16\xF5\x6B\x86\x58\x4D\x18\xD4\xE4\x0D\x8E\xA5\x9D\x5B\x91\xDC\x76\x24\x50\x3F\xC6\x2A\xFB\xD9\xB7\x9C\xB5\xD6\xE6\xD0\xD9\xE8\x19\x8B\x15\x71\x48\xAD\xB7\xEA\xD8\x59\x88\xD4\x90\xBF\x16\xB3\xD9\xE9\xAC\x59\x61\x54\xC8\x1C\xBA\xCA\xC1\xCA\xE1\xB9\x20\x4C\x8F\x3A\x93\x89\xA5\xA0\xCC\xBF\xD3\xF6\x75\xA4\x75\x96\x6D\x56"),
   // "TrustAsia Global Root CA G3"
   CertificateDer::from_slice(b"\x30\x82\x05\xA5\x30\x82\x03\x8D\xA0\x03\x02\x01\x02\x02\x14\x64\xF6\x0E\x65\x77\x61\x6A\xAB\x3B\xB4\xEA\x85\x84\xBB\xB1\x89\xB8\x71\x93\x0F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x0C\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x47\x33\x30\x1E\x17\x0D\x32\x31\x30\x35\x32\x30\x30\x32\x31\x30\x31\x39\x5A\x17\x0D\x34\x36\x30\x35\x31\x39\x30\x32\x31\x30\x31\x39\x5A\x30\x5A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x0C\x1C\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x54\x65\x63\x68\x6E\x6F\x6C\x6F\x67\x69\x65\x73\x2C\x20\x49\x6E\x63\x2E\x31\x24\x30\x22\x06\x03\x55\x04\x03\x0C\x1B\x54\x72\x75\x73\x74\x41\x73\x69\x61\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x47\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xC0\x31\x82\x61\x92\xE4\x94\x1B\x0A\x2A\x65\xD0\xBE\x06\xA9\x87\x3B\x51\x12\xEA\x70\x41\xAE\xE2\xFB\x74\xEA\x0A\x8D\xB9\xB3\x4C\xDC\x8F\xB7\x13\x52\x4F\x54\x18\xE1\x2C\x73\x95\x91\xC5\x66\x3B\x6A\xCF\xAC\x63\x6D\x87\x53\xF0\xF7\xF1\x39\xB7\xA0\x43\x63\xB0\xC4\x03\x5D\x57\xA9\xE7\x44\xCE\xC4\xA1\x83\x65\xF6\x50\x3E\xB1\x7E\x16\xB8\x3A\x8A\x02\xD0\x96\x1F\x00\xCD\x05\x21\xEF\x06\x6D\xDD\x21\x9C\x19\x43\x45\xA1\xC5\xE8\x80\xCA\xC2\xAD\x40\x62\x17\x06\xC6\xAA\xBC\xF3\xD6\xE6\xFC\x50\x7E\x66\x42\x1F\x3C\x8B\xA6\x79\x79\x86\x40\x35\x9F\x20\xEF\x3F\xEB\x8B\x47\x1F\x8F\x8E\xC5\xD4\x8E\xB6\x2C\xC9\x44\x04\xE3\xD4\x43\x75\x3F\xD5\x3F\xAF\x1C\xCC\x7E\x46\x5F\xAC\xDF\x64\x10\x8A\xEF\x46\xF0\x90\xF0\x0F\x2D\xF4\x88\x0B\xB1\x29\xAA\xAF\x85\xAA\x49\x58\xA8\xBF\x63\xA0\x38\x91\xE6\xB3\xE6\x77\x68\xC4\xF9\x2A\x19\x84\xBB\x0E\xE1\xF5\xAF\x89\xEC\xA5\x2F\x50\x20\x74\x1E\x12\x41\x73\x1E\x24\xD9\xCA\xCE\x2C\xA1\x59\x35\xC0\xC8\x1D\x46\x27\x61\x5A\x8F\xF9\x4D\xD3\x72\x79\x66\x1E\x9F\x15\x90\x21\x2D\xFD\xED\x8B\x56\x70\x03\x4A\x49\x3E\x7F\x69\x31\x12\x69\xC7\x1E\x5C\xCA\x7A\x13\x8B\xE8\xE6\xF5\x60\x0F\xCC\x93\x2C\x84\x7F\xF1\xFC\x6A\xFC\x9B\x47\x9D\xDB\xAD\x88\x3D\xF3\x76\x75\x33\xD7\x4B\xA4\xC8\x8B\xF9\xF5\x43\x58\x4F\xCB\xC8\x03\x54\x8F\xA5\x85\x78\x04\x1A\xF3\x73\xF2\xD7\x87\x1D\x41\x9F\xE7\xD8\x17\xCE\x1A\x9C\x0F\x4A\xFC\xDC\x44\x68\x54\x68\xE2\x41\x3C\xFE\x2C\x84\x86\x37\x3C\xCD\x3F\x2F\xA2\xDB\xE7\xF7\x54\x03\x5F\x59\xD3\xF7\x91\x78\xC7\x8B\x77\x6A\x16\xE5\x49\x85\x90\x45\x72\x70\x2F\x91\x5D\xF8\x3E\x65\x40\x0B\x19\x99\xC9\x26\x20\x5A\x68\xC1\x35\xBF\x4F\xA7\x51\xF1\xD8\x11\x2B\x5B\xE0\x9A\x9E\x28\x3B\x0A\x3A\x0A\x1F\xC1\x81\xE5\x2E\xF0\xA6\xB9\x69\xA5\x88\x94\xE6\x6B\x13\x7F\xD1\x64\x3F\x3D\x9C\x70\x46\xE5\xA2\x85\x7B\x58\x84\x27\xDC\xC4\x80\x3E\x67\x9A\x9A\xC7\x9A\x31\x0E\x30\xEC\xE6\x17\x40\x95\xD9\x45\xED\x01\x96\xAA\xBF\x0C\xF3\x4B\xD1\x63\xF7\x13\x58\xC0\xB8\xF3\xFA\x67\xDD\x9B\x7D\x6D\x4A\xFF\x32\x4C\xB5\x25\x3B\xFF\x1C\x67\x0F\x85\x22\x59\x05\x91\x91\x41\x77\x81\xD0\x85\x4C\x87\x10\x71\xFF\x9E\x43\x1B\xAE\x95\x75\x2D\x81\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x40\xE4\xE4\xF2\x23\xEF\x38\xCA\xB0\xAE\x57\x7F\xF2\x21\x30\x16\x34\xDB\xBC\x92\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x40\xE4\xE4\xF2\x23\xEF\x38\xCA\xB0\xAE\x57\x7F\xF2\x21\x30\x16\x34\xDB\xBC\x92\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x26\x3B\x51\xE1\x4D\x38\xF3\x32\x18\xB4\xB4\x5E\xE1\x65\x5E\xC4\x94\x4F\xD4\xA7\x61\xA3\xF8\xC0\xCF\x33\x01\x02\xE9\xC3\xAA\x35\x0F\xF1\x94\x13\x77\x77\x35\x9E\x2D\x56\x51\x44\x6E\xE1\xC6\x2E\x28\x1E\xFF\xDA\xEC\x47\xCD\x97\x44\x17\xF7\xE0\x4C\xC2\xE1\x7C\x7C\x32\x7A\x66\xC8\x5A\xB6\x5C\x53\x45\x57\x5A\x45\xD4\x05\x99\x2F\x2E\x23\x55\xEE\x63\x68\xDF\xD3\x1B\x78\xA7\x12\x94\x06\x00\x75\x0D\x72\x84\xE9\x2E\xBC\x5A\x6A\xD5\xDE\x2F\x59\xC7\xA3\xEC\xD2\x87\x66\xDB\xB7\x54\xB5\x24\xAB\xF4\x43\x78\xDB\x4B\x04\xC4\x6F\xDD\xE6\x3E\x66\x3E\x29\xF2\x4B\x68\x71\x22\x87\xA0\xF8\xB1\x33\x63\x76\xE3\x0D\x85\x72\x44\x22\x55\x3F\x1C\x7C\xE9\xFC\xB8\x15\xE8\x52\xFA\xAA\x3E\xA3\x21\x39\x35\x74\x89\xA6\x6A\xC2\x39\xFA\x78\xCF\xB6\xAC\xE7\xE7\xD6\x56\xFF\x23\x92\x2E\x50\x0B\xA9\xB5\x07\x33\xF4\x38\x5F\xA4\x49\xA6\xCB\x65\x70\x76\xE8\x0A\x85\x80\x4B\x36\x3D\x33\xF7\x95\x54\x75\x25\xDA\xAC\xC4\x73\x82\x65\xE9\x52\xF5\x5C\xFD\x38\x95\x02\x6A\x69\x30\xC5\x1C\x0A\x57\x07\xAE\x22\xA4\x2C\xF9\xC5\x41\xB7\xB8\xEC\x9F\x4F\x48\x00\xF9\x01\x04\x55\xCC\xAC\xF9\x32\x31\xC4\x75\x95\x06\xA0\x7F\xD1\x8D\x27\xDD\xB3\xA9\xA4\x72\x87\xFE\x59\x8B\x9A\x7A\x74\x16\xDD\x16\xA5\x62\x29\xEB\x3A\x96\xDC\x8B\xA7\x68\x59\xD3\xEB\x77\x91\x39\xF8\xD7\xCB\xD9\x8F\x5F\x5A\x27\x01\x7D\x5D\x68\x19\x62\xD8\xC8\xCD\xF4\xB7\x72\x47\xBE\x5B\x97\xCE\xF2\xAD\xA2\x99\x93\xAD\x94\xCB\x93\xF6\x12\x09\x95\xB6\xAB\xD7\x3B\xD0\x3F\x11\xCB\x30\x16\x2E\x79\x80\xE4\x67\x81\x2D\x5D\xED\x70\x78\xB6\x60\x59\xAC\xE1\x5D\x45\x63\x8F\xC8\xDF\x72\x68\x5B\xEA\x1D\xB8\x01\xF1\x7E\xFB\xE7\x8A\xB3\xE3\x54\xA0\x38\x09\xE0\x3C\xDE\x42\xF2\xC2\xED\x2E\x9B\xF3\x1F\x35\xB6\x36\xD8\xE3\x80\xA1\x8B\xCD\x99\x64\x0F\xC2\xAA\xAB\xB1\xCA\xF5\x6F\x9E\x43\x8D\x84\x54\x99\xB3\x6E\xC0\x12\x66\xD8\x70\x10\xF1\x06\x35\x33\x43\xA8\x9C\x2E\xBA\x14\x31\xCE\x10\x7F\x1C\x86\xE3\x8F\xD2\xD5\xF8\x77\xEC\x9B\xAB\xF1\x2F\x63\xD9\x42\x5F\xE0\x67\x81\x64\x91\xF1\x97\x2F\xFC\x6E\x26\xF6\x33\xF8\xD3\xB5\xF8\xC4\x62\xAB\x31\x51\x25\x02\x7A\xF8\xDD\x6B\x65\xD5\x6D\x4D\x30\xC8\x65\xBA\x68\x14\x65\xAC\x27\x0B\x74\x8A\xF2\x87"),
   // "CA Disig Root R2"
   CertificateDer::from_slice(b"\x30\x82\x05\x69\x30\x82\x03\x51\xA0\x03\x02\x01\x02\x02\x09\x00\x92\xB8\x88\xDB\xB0\x8A\xC1\x63\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x52\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x53\x4B\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x42\x72\x61\x74\x69\x73\x6C\x61\x76\x61\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x44\x69\x73\x69\x67\x20\x61\x2E\x73\x2E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x43\x41\x20\x44\x69\x73\x69\x67\x20\x52\x6F\x6F\x74\x20\x52\x32\x30\x1E\x17\x0D\x31\x32\x30\x37\x31\x39\x30\x39\x31\x35\x33\x30\x5A\x17\x0D\x34\x32\x30\x37\x31\x39\x30\x39\x31\x35\x33\x30\x5A\x30\x52\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x53\x4B\x31\x13\x30\x11\x06\x03\x55\x04\x07\x13\x0A\x42\x72\x61\x74\x69\x73\x6C\x61\x76\x61\x31\x13\x30\x11\x06\x03\x55\x04\x0A\x13\x0A\x44\x69\x73\x69\x67\x20\x61\x2E\x73\x2E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x43\x41\x20\x44\x69\x73\x69\x67\x20\x52\x6F\x6F\x74\x20\x52\x32\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA2\xA3\xC4\x00\x09\xD6\x85\x5D\x2D\x6D\x14\xF6\xC2\xC3\x73\x9E\x35\xC2\x71\x55\x7E\x81\xFB\xAB\x46\x50\xE0\xC1\x7C\x49\x78\xE6\xAB\x79\x58\x3C\xDA\xFF\x7C\x1C\x9F\xD8\x97\x02\x78\x3E\x6B\x41\x04\xE9\x41\xBD\xBE\x03\x2C\x45\xF6\x2F\x64\xD4\xAB\x5D\xA3\x47\x3D\x64\x9B\xE9\x68\x9A\xC6\xCC\x1B\x3F\xBA\xBE\xB2\x8B\x34\x02\x2E\x98\x55\x19\xFC\x8C\x6F\xAA\x5F\xDA\x4C\xCE\x4D\x03\x21\xA3\xD8\xD2\x34\x93\x56\x96\xCB\x4C\x0C\x00\x16\x3C\x5F\x1A\xCD\xC8\xC7\x6C\xA6\xAD\xD3\x31\xA7\xBC\xE8\xE5\xE1\x66\xD6\xD2\xFB\x03\xB4\x41\x65\xC9\x10\xAE\x0E\x05\x63\xC6\x80\x6A\x69\x30\xFD\xD2\xEE\x90\xEF\x0D\x27\xDF\x9F\x95\x73\xF4\xE1\x25\xDA\x6C\x16\xDE\x41\x38\x34\xEA\x8B\xFC\xD1\xE8\x04\x14\x61\x2D\x41\x7E\xAC\xC7\x77\x4E\xCB\x51\x54\xFB\x5E\x92\x18\x1B\x04\x5A\x68\xC6\xC9\xC4\xFA\xB7\x13\xA0\x98\xB7\x11\x2B\xB7\xD6\x57\xCC\x7C\x9E\x17\xD1\xCB\x25\xFE\x86\x4E\x24\x2E\x56\x0C\x78\x4D\x9E\x01\x12\xA6\x2B\xA7\x01\x65\x6E\x7C\x62\x1D\x84\x84\xDF\xEA\xC0\x6B\xB5\xA5\x2A\x95\x83\xC3\x53\x11\x0C\x73\x1D\x0B\xB2\x46\x90\xD1\x42\x3A\xCE\x40\x6E\x95\xAD\xFF\xC6\x94\xAD\x6E\x97\x84\x8E\x7D\x6F\x9E\x8A\x80\x0D\x49\x6D\x73\xE2\x7B\x92\x1E\xC3\xF3\xC1\xF3\xEB\x2E\x05\x6F\xD9\x1B\xCF\x37\x76\x04\xC8\xB4\x5A\xE4\x17\xA7\xCB\xDD\x76\x1F\xD0\x19\x76\xE8\x2C\x05\xB3\xD6\x9C\x34\xD8\x96\xDC\x61\x87\x91\x05\xE4\x44\x08\x33\xC1\xDA\xB9\x08\x65\xD4\xAE\xB2\x36\x0D\xEB\xBA\x38\xBA\x0C\xE5\x9B\x9E\xEB\x8D\x66\xDD\x99\xCF\xD6\x89\x41\xF6\x04\x92\x8A\x29\x29\x6D\x6B\x3A\x1C\xE7\x75\x7D\x02\x71\x0E\xF3\xC0\xE7\xBD\xCB\x19\xDD\x9D\x60\xB2\xC2\x66\x60\xB6\xB1\x04\xEE\xC9\xE6\x86\xB9\x9A\x66\x40\xA8\xE7\x11\xED\x81\x45\x03\x8B\xF6\x67\x59\xE8\xC1\x06\x11\xBD\xDD\xCF\x80\x02\x4F\x65\x40\x78\x5C\x47\x50\xC8\x9B\xE6\x1F\x81\x7B\xE4\x44\xA8\x5B\x85\x9A\xE2\xDE\x5A\xD5\xC7\xF9\x3A\x44\x66\x4B\xE4\x32\x54\x7C\xE4\x6C\x9C\xB3\x0E\x3D\x17\xA2\xB2\x34\x12\xD6\x7E\xB2\xA8\x49\xBB\xD1\x7A\x28\x40\xBE\xA2\x16\x1F\xDF\xE4\x37\x1F\x11\x73\xFB\x90\x0A\x65\x43\xA2\x0D\x7C\xF8\x06\x01\x55\x33\x7D\xB0\x0D\xB8\xF4\xF5\xAE\xA5\x42\x57\x7C\x36\x11\x8C\x7B\x5E\xC4\x03\x9D\x8C\x79\x9D\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB5\x99\xF8\xAF\xB0\x94\xF5\xE3\x20\xD6\x0A\xAD\xCE\x4E\x56\xA4\x2E\x6E\x42\xED\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x26\x06\x5E\x70\xE7\x65\x33\xC8\x82\x6E\xD9\x9C\x17\x3A\x1B\x7A\x66\xB2\x01\xF6\x78\x3B\x69\x5E\x2F\xEA\xFF\x4E\xF9\x28\xC3\x98\x2A\x61\x4C\xB4\x24\x12\x8A\x7D\x6D\x11\x14\xF7\x9C\xB5\xCA\xE6\xBC\x9E\x27\x8E\x4C\x19\xC8\xA9\xBD\x7A\xC0\xD7\x36\x0E\x6D\x85\x72\x6E\xA8\xC6\xA2\x6D\xF6\xFA\x73\x63\x7F\xBC\x6E\x79\x08\x1C\x9D\x8A\x9F\x1A\x8A\x53\xA6\xD8\xBB\xD9\x35\x55\xB1\x11\xC5\xA9\x03\xB3\x56\x3B\xB9\x84\x93\x22\x5E\x7E\xC1\xF6\x12\x52\x8B\xEA\x2C\x67\xBC\xFE\x36\x4C\xF5\xB8\xCF\xD1\xB3\x49\x92\x3B\xD3\x29\x0E\x99\x1B\x96\xF7\x61\xB8\x3B\xC4\x2B\xB6\x78\x6C\xB4\x23\x6F\xF0\xFD\xD3\xB2\x5E\x75\x1F\x99\x95\xA8\xAC\xF6\xDA\xE1\xC5\x31\x7B\xFB\xD1\x46\xB3\xD2\xBC\x67\xB4\x62\x54\xBA\x09\xF7\x63\xB0\x93\xA2\x9A\xF9\xE9\x52\x2E\x8B\x60\x12\xAB\xFC\xF5\x60\x56\xEF\x10\x5C\x8B\xC4\x1A\x42\xDC\x83\x5B\x64\x0E\xCB\xB5\xBC\xD6\x4F\xC1\x7C\x3C\x6E\x8D\x13\x6D\xFB\x7B\xEB\x30\xD0\xDC\x4D\xAF\xC5\xD5\xB6\xA5\x4C\x5B\x71\xC9\xE8\x31\xBE\xE8\x38\x06\x48\xA1\x1A\xE2\xEA\xD2\xDE\x12\x39\x58\x1A\xFF\x80\x0E\x82\x75\xE6\xB7\xC9\x07\x6C\x0E\xEF\xFF\x38\xF1\x98\x71\xC4\xB7\x7F\x0E\x15\xD0\x25\x69\xBD\x22\x9D\x2B\xED\x05\xF6\x46\x47\xAC\xED\xC0\xF0\xD4\x3B\xE2\xEC\xEE\x96\x5B\x90\x13\x4E\x1E\x56\x3A\xEB\xB0\xEF\x96\xBB\x96\x23\x11\xBA\xF2\x43\x86\x74\x64\x95\xC8\x28\x75\xDF\x1D\x35\xBA\xD2\x37\x83\x38\x53\x38\x36\x3B\xCF\x6C\xE9\xF9\x6B\x0E\xD0\xFB\x04\xE8\x4F\x77\xD7\x65\x01\x78\x86\x0C\x7A\x3E\x21\x62\xF1\x7F\x63\x71\x0C\xC9\x9F\x44\xDB\xA8\x27\xA2\x75\xBE\x6E\x81\x3E\xD7\xC0\xEB\x1B\x98\x0F\x70\x5C\x34\xB2\x8A\xCC\xC0\x85\x18\xEB\x6E\x7A\xB3\xF7\x5A\xA1\x07\xBF\xA9\x42\x92\xF3\x60\x22\x97\xE4\x14\xA1\x07\x9B\x4E\x76\xC0\x8E\x7D\xFD\xA4\x25\xC7\x47\xED\xFF\x1F\x73\xAC\xCC\xC3\xA5\xE9\x6F\x0A\x8E\x9B\x65\xC2\x50\x85\xB5\xA3\xA0\x53\x12\xCC\x55\x87\x61\xF3\x81\xAE\x10\x46\x61\xBD\x44\x21\xB8\xC2\x3D\x74\xCF\x7E\x24\x35\xFA\x1C\x07\x0E\x9B\x3D\x22\xCA\xEF\x31\x2F\x8C\xAC\x12\xBD\xEF\x40\x28\xFC\x29\x67\x9F\xB2\x13\x4F\x66\x24\xC4\x53\x19\xE9\x1E\x29\x15\xEF\xE6\x6D\xB0\x7F\x2D\x67\xFD\xF3\x6C\x1B\x75\x46\xA3\xE5\x4A\x17\xE9\xA4\xD7\x0B"),
   // "Amazon Root CA 4"
   CertificateDer::from_slice(b"\x30\x82\x01\xF2\x30\x82\x01\x78\xA0\x03\x02\x01\x02\x02\x13\x06\x6C\x9F\xD7\xC1\xBB\x10\x4C\x29\x43\xE5\x71\x7B\x7B\x2C\xC8\x1A\xC1\x0E\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x34\x30\x1E\x17\x0D\x31\x35\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x34\x30\x30\x35\x32\x36\x30\x30\x30\x30\x30\x30\x5A\x30\x39\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x0F\x30\x0D\x06\x03\x55\x04\x0A\x13\x06\x41\x6D\x61\x7A\x6F\x6E\x31\x19\x30\x17\x06\x03\x55\x04\x03\x13\x10\x41\x6D\x61\x7A\x6F\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x34\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xD2\xAB\x8A\x37\x4F\xA3\x53\x0D\xFE\xC1\x8A\x7B\x4B\xA8\x7B\x46\x4B\x63\xB0\x62\xF6\x2D\x1B\xDB\x08\x71\x21\xD2\x00\xE8\x63\xBD\x9A\x27\xFB\xF0\x39\x6E\x5D\xEA\x3D\xA5\xC9\x81\xAA\xA3\x5B\x20\x98\x45\x5D\x16\xDB\xFD\xE8\x10\x6D\xE3\x9C\xE0\xE3\xBD\x5F\x84\x62\xF3\x70\x64\x33\xA0\xCB\x24\x2F\x70\xBA\x88\xA1\x2A\xA0\x75\xF8\x81\xAE\x62\x06\xC4\x81\xDB\x39\x6E\x29\xB0\x1E\xFA\x2E\x5C\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD3\xEC\xC7\x3A\x65\x6E\xCC\xE1\xDA\x76\x9A\x56\xFB\x9C\xF3\x86\x6D\x57\xE5\x81\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x3A\x8B\x21\xF1\xBD\x7E\x11\xAD\xD0\xEF\x58\x96\x2F\xD6\xEB\x9D\x7E\x90\x8D\x2B\xCF\x66\x55\xC3\x2C\xE3\x28\xA9\x70\x0A\x47\x0E\xF0\x37\x59\x12\xFF\x2D\x99\x94\x28\x4E\x2A\x4F\x35\x4D\x33\x5A\x02\x31\x00\xEA\x75\x00\x4E\x3B\xC4\x3A\x94\x12\x91\xC9\x58\x46\x9D\x21\x13\x72\xA7\x88\x9C\x8A\xE4\x4C\x4A\xDB\x96\xD4\xAC\x8B\x6B\x6B\x49\x12\x53\x33\xAD\xD7\xE4\xBE\x24\xFC\xB5\x0A\x76\xD4\xA5\xBC\x10"),
   // "Certigna"
   CertificateDer::from_slice(b"\x30\x82\x03\xA8\x30\x82\x02\x90\xA0\x03\x02\x01\x02\x02\x09\x00\xFE\xDC\xE3\x01\x0F\xC9\x48\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x34\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x52\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x44\x68\x69\x6D\x79\x6F\x74\x69\x73\x31\x11\x30\x0F\x06\x03\x55\x04\x03\x0C\x08\x43\x65\x72\x74\x69\x67\x6E\x61\x30\x1E\x17\x0D\x30\x37\x30\x36\x32\x39\x31\x35\x31\x33\x30\x35\x5A\x17\x0D\x32\x37\x30\x36\x32\x39\x31\x35\x31\x33\x30\x35\x5A\x30\x34\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x52\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x44\x68\x69\x6D\x79\x6F\x74\x69\x73\x31\x11\x30\x0F\x06\x03\x55\x04\x03\x0C\x08\x43\x65\x72\x74\x69\x67\x6E\x61\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xC8\x68\xF1\xC9\xD6\xD6\xB3\x34\x75\x26\x82\x1E\xEC\xB4\xBE\xEA\x5C\xE1\x26\xED\x11\x47\x61\xE1\xA2\x7C\x16\x78\x40\x21\xE4\x60\x9E\x5A\xC8\x63\xE1\xC4\xB1\x96\x92\xFF\x18\x6D\x69\x23\xE1\x2B\x62\xF7\xDD\xE2\x36\x2F\x91\x07\xB9\x48\xCF\x0E\xEC\x79\xB6\x2C\xE7\x34\x4B\x70\x08\x25\xA3\x3C\x87\x1B\x19\xF2\x81\x07\x0F\x38\x90\x19\xD3\x11\xFE\x86\xB4\xF2\xD1\x5E\x1E\x1E\x96\xCD\x80\x6C\xCE\x3B\x31\x93\xB6\xF2\xA0\xD0\xA9\x95\x12\x7D\xA5\x9A\xCC\x6B\xC8\x84\x56\x8A\x33\xA9\xE7\x22\x15\x53\x16\xF0\xCC\x17\xEC\x57\x5F\xE9\xA2\x0A\x98\x09\xDE\xE3\x5F\x9C\x6F\xDC\x48\xE3\x85\x0B\x15\x5A\xA6\xBA\x9F\xAC\x48\xE3\x09\xB2\xF7\xF4\x32\xDE\x5E\x34\xBE\x1C\x78\x5D\x42\x5B\xCE\x0E\x22\x8F\x4D\x90\xD7\x7D\x32\x18\xB3\x0B\x2C\x6A\xBF\x8E\x3F\x14\x11\x89\x20\x0E\x77\x14\xB5\x3D\x94\x08\x87\xF7\x25\x1E\xD5\xB2\x60\x00\xEC\x6F\x2A\x28\x25\x6E\x2A\x3E\x18\x63\x17\x25\x3F\x3E\x44\x20\x16\xF6\x26\xC8\x25\xAE\x05\x4A\xB4\xE7\x63\x2C\xF3\x8C\x16\x53\x7E\x5C\xFB\x11\x1A\x08\xC1\x46\x62\x9F\x22\xB8\xF1\xC2\x8D\x69\xDC\xFA\x3A\x58\x06\xDF\x02\x03\x01\x00\x01\xA3\x81\xBC\x30\x81\xB9\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x1A\xED\xFE\x41\x39\x90\xB4\x24\x59\xBE\x01\xF2\x52\xD5\x45\xF6\x5A\x39\xDC\x11\x30\x64\x06\x03\x55\x1D\x23\x04\x5D\x30\x5B\x80\x14\x1A\xED\xFE\x41\x39\x90\xB4\x24\x59\xBE\x01\xF2\x52\xD5\x45\xF6\x5A\x39\xDC\x11\xA1\x38\xA4\x36\x30\x34\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x46\x52\x31\x12\x30\x10\x06\x03\x55\x04\x0A\x0C\x09\x44\x68\x69\x6D\x79\x6F\x74\x69\x73\x31\x11\x30\x0F\x06\x03\x55\x04\x03\x0C\x08\x43\x65\x72\x74\x69\x67\x6E\x61\x82\x09\x00\xFE\xDC\xE3\x01\x0F\xC9\x48\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x11\x06\x09\x60\x86\x48\x01\x86\xF8\x42\x01\x01\x04\x04\x03\x02\x00\x07\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x85\x03\x1E\x92\x71\xF6\x42\xAF\xE1\xA3\x61\x9E\xEB\xF3\xC0\x0F\xF2\xA5\xD4\xDA\x95\xE6\xD6\xBE\x68\x36\x3D\x7E\x6E\x1F\x4C\x8A\xEF\xD1\x0F\x21\x6D\x5E\xA5\x52\x63\xCE\x12\xF8\xEF\x2A\xDA\x6F\xEB\x37\xFE\x13\x02\xC7\xCB\x3B\x3E\x22\x6B\xDA\x61\x2E\x7F\xD4\x72\x3D\xDD\x30\xE1\x1E\x4C\x40\x19\x8C\x0F\xD7\x9C\xD1\x83\x30\x7B\x98\x59\xDC\x7D\xC6\xB9\x0C\x29\x4C\xA1\x33\xA2\xEB\x67\x3A\x65\x84\xD3\x96\xE2\xED\x76\x45\x70\x8F\xB5\x2B\xDE\xF9\x23\xD6\x49\x6E\x3C\x14\xB5\xC6\x9F\x35\x1E\x50\xD0\xC1\x8F\x6A\x70\x44\x02\x62\xCB\xAE\x1D\x68\x41\xA7\xAA\x57\xE8\x53\xAA\x07\xD2\x06\xF6\xD5\x14\x06\x0B\x91\x03\x75\x2C\x6C\x72\xB5\x61\x95\x9A\x0D\x8B\xB9\x0D\xE7\xF5\xDF\x54\xCD\xDE\xE6\xD8\xD6\x09\x08\x97\x63\xE5\xC1\x2E\xB0\xB7\x44\x26\xC0\x26\xC0\xAF\x55\x30\x9E\x3B\xD5\x36\x2A\x19\x04\xF4\x5C\x1E\xFF\xCF\x2C\xB7\xFF\xD0\xFD\x87\x40\x11\xD5\x11\x23\xBB\x48\xC0\x21\xA9\xA4\x28\x2D\xFD\x15\xF8\xB0\x4E\x2B\xF4\x30\x5B\x21\xFC\x11\x91\x34\xBE\x41\xEF\x7B\x9D\x97\x75\xFF\x97\x95\xC0\x96\x58\x2F\xEA\xBB\x46\xD7\xBB\xE4\xD9\x2E"),
   // "D-TRUST BR Root CA 1 2020"
   CertificateDer::from_slice(b"\x30\x82\x02\xDB\x30\x82\x02\x60\xA0\x03\x02\x01\x02\x02\x10\x7C\xC9\x8F\x2B\x84\xD7\xDF\xEA\x0F\xC9\x65\x9A\xD3\x4B\x4D\x96\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x42\x52\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x20\x32\x30\x32\x30\x30\x1E\x17\x0D\x32\x30\x30\x32\x31\x31\x30\x39\x34\x35\x30\x30\x5A\x17\x0D\x33\x35\x30\x32\x31\x31\x30\x39\x34\x34\x35\x39\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x13\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x44\x2D\x54\x52\x55\x53\x54\x20\x42\x52\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x31\x20\x32\x30\x32\x30\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xC6\xCB\xC7\x28\xD1\xFB\x84\xF5\x9A\xEF\x42\x14\x20\xE1\x43\x6B\x6E\x75\xAD\xFC\x2B\x03\x84\xD4\x76\x93\x25\xD7\x59\x3B\x41\x65\x6B\x1E\xE6\x34\x2A\xBB\x74\xF6\x12\xCE\xE8\x6D\xE7\xAB\xE4\x3C\x4E\x3F\x44\x08\x8B\xCD\x16\x71\xCB\xBF\x92\x99\xF4\xA4\xD7\x3C\x50\x54\x52\x90\x85\x83\x78\x94\x67\x67\xA3\x1C\x09\x19\x3D\x75\x34\x85\xDE\xED\x60\x7D\xC7\x0C\xB4\x41\x52\xB9\x6E\xE5\xEE\x42\xA3\x82\x01\x0D\x30\x82\x01\x09\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x73\x91\x10\xAB\xFF\x55\xB3\x5A\x7C\x09\x25\xD5\xB2\xBA\x08\xA0\x6B\xAB\x1F\x6D\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x81\xC6\x06\x03\x55\x1D\x1F\x04\x81\xBE\x30\x81\xBB\x30\x3E\xA0\x3C\xA0\x3A\x86\x38\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x63\x72\x6C\x2F\x64\x2D\x74\x72\x75\x73\x74\x5F\x62\x72\x5F\x72\x6F\x6F\x74\x5F\x63\x61\x5F\x31\x5F\x32\x30\x32\x30\x2E\x63\x72\x6C\x30\x79\xA0\x77\xA0\x75\x86\x73\x6C\x64\x61\x70\x3A\x2F\x2F\x64\x69\x72\x65\x63\x74\x6F\x72\x79\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x43\x4E\x3D\x44\x2D\x54\x52\x55\x53\x54\x25\x32\x30\x42\x52\x25\x32\x30\x52\x6F\x6F\x74\x25\x32\x30\x43\x41\x25\x32\x30\x31\x25\x32\x30\x32\x30\x32\x30\x2C\x4F\x3D\x44\x2D\x54\x72\x75\x73\x74\x25\x32\x30\x47\x6D\x62\x48\x2C\x43\x3D\x44\x45\x3F\x63\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x72\x65\x76\x6F\x63\x61\x74\x69\x6F\x6E\x6C\x69\x73\x74\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\x94\x90\x2D\x13\xFA\xE1\x63\xF8\x61\x63\xE8\xAD\x85\x78\x54\x91\x9C\xB8\x93\x38\x3E\x1A\x41\xDA\x40\x16\x53\x42\x08\xCA\x2F\x8E\xF1\x3E\x81\x56\xC0\xAA\xD8\xED\x18\xC4\xB0\xAE\xF4\x3E\xFA\x26\x02\x31\x00\xF3\x28\xE2\xC6\xDB\x2B\x99\xFB\xB7\x51\xB8\x24\xA3\xA4\x94\x7A\x1A\x3F\xE6\x36\xE2\x03\x57\x33\x8A\x30\xCB\x82\xC7\xD6\x14\x11\xD5\x75\x63\x5B\x14\x95\x9C\x1F\x01\xCF\xD8\xD5\x72\xA7\x0F\x3B"),
   // "Security Communication ECC RootCA1"
   CertificateDer::from_slice(b"\x30\x82\x02\x38\x30\x82\x01\xBE\xA0\x03\x02\x01\x02\x02\x09\x00\xD6\x5D\x9B\xB3\x78\x81\x2E\xEB\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x45\x43\x4F\x4D\x20\x54\x72\x75\x73\x74\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x43\x4F\x2E\x2C\x4C\x54\x44\x2E\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x53\x65\x63\x75\x72\x69\x74\x79\x20\x43\x6F\x6D\x6D\x75\x6E\x69\x63\x61\x74\x69\x6F\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x43\x41\x31\x30\x1E\x17\x0D\x31\x36\x30\x36\x31\x36\x30\x35\x31\x35\x32\x38\x5A\x17\x0D\x33\x38\x30\x31\x31\x38\x30\x35\x31\x35\x32\x38\x5A\x30\x61\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x25\x30\x23\x06\x03\x55\x04\x0A\x13\x1C\x53\x45\x43\x4F\x4D\x20\x54\x72\x75\x73\x74\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x43\x4F\x2E\x2C\x4C\x54\x44\x2E\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x13\x22\x53\x65\x63\x75\x72\x69\x74\x79\x20\x43\x6F\x6D\x6D\x75\x6E\x69\x63\x61\x74\x69\x6F\x6E\x20\x45\x43\x43\x20\x52\x6F\x6F\x74\x43\x41\x31\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\xA4\xA5\x6F\x60\x03\x03\xC3\xBD\x31\xF4\xD3\x17\x9C\x2B\x84\x75\xAC\xE5\xFD\x3D\x57\x6E\xD7\x63\xBF\xE6\x04\x89\x92\x8E\x81\x9C\xE3\xE9\x47\x6E\xCA\x90\x12\xC8\x13\xE0\xA7\x9D\xF7\x65\x74\x1F\x6C\x10\xB2\xE8\xE4\xE9\xEF\x6D\x85\x32\x99\x44\xB1\x5E\xFD\xCC\x76\x10\xD8\x5B\xBD\xA2\xC6\xF9\xD6\x42\xE4\x57\x76\xDC\x90\xC2\x35\xA9\x4B\x88\x3C\x12\x47\x6D\x5C\xFF\x49\x4F\x1A\x4A\x50\xB1\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x86\x1C\xE7\xFE\x2D\xA5\x4A\x8B\x08\xFE\x28\x11\xFA\xBE\xA3\x66\xF8\x60\x59\x2F\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x30\x15\x5D\x42\x3D\xFC\xB6\xEE\xF7\x3B\xB1\x36\xE8\x9E\xF6\xC4\x46\x28\x49\x33\xD0\x58\x43\x2A\x63\x29\xCC\x4D\xB1\xB4\x7A\xA2\xB9\x0D\x38\xA5\x5D\x48\x2A\xFD\xCB\xB2\x73\x5D\xA3\x88\x08\xC7\x0C\x02\x31\x00\xC0\xAB\x2D\x0E\x6D\xED\x18\xA2\xDB\x53\xE9\x25\xDB\x55\x08\xE0\x50\xCC\xDF\x44\x61\x16\x82\xAB\x49\xB0\xB2\x81\xEC\x73\x87\x78\xB4\x4C\xB2\x62\x1B\x12\xFA\x16\x4D\x25\x4B\x63\xBD\x1E\x37\xD9"),
   // "SecureSign Root CA15"
   CertificateDer::from_slice(b"\x30\x82\x02\x23\x30\x82\x01\xA9\xA0\x03\x02\x01\x02\x02\x14\x16\x15\xC7\xC3\xD8\x49\xA7\xBE\x69\x0C\x8A\x88\xED\xF0\x70\xF9\xDD\xB7\x3E\x87\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x13\x1A\x43\x79\x62\x65\x72\x74\x72\x75\x73\x74\x20\x4A\x61\x70\x61\x6E\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x13\x14\x53\x65\x63\x75\x72\x65\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x35\x30\x1E\x17\x0D\x32\x30\x30\x34\x30\x38\x30\x38\x33\x32\x35\x36\x5A\x17\x0D\x34\x35\x30\x34\x30\x38\x30\x38\x33\x32\x35\x36\x5A\x30\x51\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4A\x50\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x13\x1A\x43\x79\x62\x65\x72\x74\x72\x75\x73\x74\x20\x4A\x61\x70\x61\x6E\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x13\x14\x53\x65\x63\x75\x72\x65\x53\x69\x67\x6E\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x35\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x0B\x50\x74\x8D\x64\x32\x99\x99\xB3\xD2\x60\x08\xB8\x22\x8E\x46\x74\x2C\x78\xC0\x2B\x44\x2D\x6D\x5F\x1D\xC9\xAE\x4B\x52\x20\x83\x3D\xB8\x14\x6D\x53\x87\x60\x9E\x5F\x6C\x85\xDB\x06\x14\x95\xE0\xC7\x28\xFF\x9D\x5F\xE4\xAA\xF1\xB3\x8B\x6D\xED\x4F\x2F\x4B\xC9\x4A\x94\x91\x64\x75\xFE\x01\xEC\xC1\xD8\xEB\x7A\x94\x78\x56\x18\x43\x5F\x6B\x81\xCB\xF6\xBC\xDA\xB4\x0C\xB6\x29\x93\x08\x69\x8F\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xEB\x41\xC8\xAE\xFC\xD5\x9E\x51\x48\xF5\xBD\x8B\xF4\x87\x20\x93\x41\x2B\xD3\xF4\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x68\x00\x30\x65\x02\x31\x00\xD9\x2E\x89\x7E\x5E\x4E\xA4\x11\x07\xBD\x59\xC2\x07\xDE\xAB\x32\x38\x53\x2A\x46\x44\x06\x17\x7A\xCE\x51\xE9\xE0\xFF\x66\x2D\x09\x4E\xE0\x4F\xF4\x05\xD1\x85\xF6\x35\x60\xDC\xF5\x72\xB3\x46\x7D\x02\x30\x44\x98\x46\x1A\x82\x85\x1E\x61\x69\x89\x4B\x07\x4B\x66\xB5\x9E\xAA\xBA\xA0\x1E\x41\xD9\x01\x74\x3A\x6E\x45\x3A\x89\x80\x19\x7B\x32\x98\x55\x63\xAB\xEB\x63\x6E\x93\x6D\xAB\x1B\x09\x60\x31\x4E"),
   // "USERTrust RSA Certification Authority"
   CertificateDer::from_slice(b"\x30\x82\x05\xDE\x30\x82\x03\xC6\xA0\x03\x02\x01\x02\x02\x10\x01\xFD\x6D\x30\xFC\xA3\xCA\x51\xA8\x1B\xBC\x64\x0E\x35\x03\x2D\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x81\x88\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x08\x13\x0A\x4E\x65\x77\x20\x4A\x65\x72\x73\x65\x79\x31\x14\x30\x12\x06\x03\x55\x04\x07\x13\x0B\x4A\x65\x72\x73\x65\x79\x20\x43\x69\x74\x79\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x54\x68\x65\x20\x55\x53\x45\x52\x54\x52\x55\x53\x54\x20\x4E\x65\x74\x77\x6F\x72\x6B\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x13\x25\x55\x53\x45\x52\x54\x72\x75\x73\x74\x20\x52\x53\x41\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x1E\x17\x0D\x31\x30\x30\x32\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x17\x0D\x33\x38\x30\x31\x31\x38\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x88\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x13\x30\x11\x06\x03\x55\x04\x08\x13\x0A\x4E\x65\x77\x20\x4A\x65\x72\x73\x65\x79\x31\x14\x30\x12\x06\x03\x55\x04\x07\x13\x0B\x4A\x65\x72\x73\x65\x79\x20\x43\x69\x74\x79\x31\x1E\x30\x1C\x06\x03\x55\x04\x0A\x13\x15\x54\x68\x65\x20\x55\x53\x45\x52\x54\x52\x55\x53\x54\x20\x4E\x65\x74\x77\x6F\x72\x6B\x31\x2E\x30\x2C\x06\x03\x55\x04\x03\x13\x25\x55\x53\x45\x52\x54\x72\x75\x73\x74\x20\x52\x53\x41\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\x80\x12\x65\x17\x36\x0E\xC3\xDB\x08\xB3\xD0\xAC\x57\x0D\x76\xED\xCD\x27\xD3\x4C\xAD\x50\x83\x61\xE2\xAA\x20\x4D\x09\x2D\x64\x09\xDC\xCE\x89\x9F\xCC\x3D\xA9\xEC\xF6\xCF\xC1\xDC\xF1\xD3\xB1\xD6\x7B\x37\x28\x11\x2B\x47\xDA\x39\xC6\xBC\x3A\x19\xB4\x5F\xA6\xBD\x7D\x9D\xA3\x63\x42\xB6\x76\xF2\xA9\x3B\x2B\x91\xF8\xE2\x6F\xD0\xEC\x16\x20\x90\x09\x3E\xE2\xE8\x74\xC9\x18\xB4\x91\xD4\x62\x64\xDB\x7F\xA3\x06\xF1\x88\x18\x6A\x90\x22\x3C\xBC\xFE\x13\xF0\x87\x14\x7B\xF6\xE4\x1F\x8E\xD4\xE4\x51\xC6\x11\x67\x46\x08\x51\xCB\x86\x14\x54\x3F\xBC\x33\xFE\x7E\x6C\x9C\xFF\x16\x9D\x18\xBD\x51\x8E\x35\xA6\xA7\x66\xC8\x72\x67\xDB\x21\x66\xB1\xD4\x9B\x78\x03\xC0\x50\x3A\xE8\xCC\xF0\xDC\xBC\x9E\x4C\xFE\xAF\x05\x96\x35\x1F\x57\x5A\xB7\xFF\xCE\xF9\x3D\xB7\x2C\xB6\xF6\x54\xDD\xC8\xE7\x12\x3A\x4D\xAE\x4C\x8A\xB7\x5C\x9A\xB4\xB7\x20\x3D\xCA\x7F\x22\x34\xAE\x7E\x3B\x68\x66\x01\x44\xE7\x01\x4E\x46\x53\x9B\x33\x60\xF7\x94\xBE\x53\x37\x90\x73\x43\xF3\x32\xC3\x53\xEF\xDB\xAA\xFE\x74\x4E\x69\xC7\x6B\x8C\x60\x93\xDE\xC4\xC7\x0C\xDF\xE1\x32\xAE\xCC\x93\x3B\x51\x78\x95\x67\x8B\xEE\x3D\x56\xFE\x0C\xD0\x69\x0F\x1B\x0F\xF3\x25\x26\x6B\x33\x6D\xF7\x6E\x47\xFA\x73\x43\xE5\x7E\x0E\xA5\x66\xB1\x29\x7C\x32\x84\x63\x55\x89\xC4\x0D\xC1\x93\x54\x30\x19\x13\xAC\xD3\x7D\x37\xA7\xEB\x5D\x3A\x6C\x35\x5C\xDB\x41\xD7\x12\xDA\xA9\x49\x0B\xDF\xD8\x80\x8A\x09\x93\x62\x8E\xB5\x66\xCF\x25\x88\xCD\x84\xB8\xB1\x3F\xA4\x39\x0F\xD9\x02\x9E\xEB\x12\x4C\x95\x7C\xF3\x6B\x05\xA9\x5E\x16\x83\xCC\xB8\x67\xE2\xE8\x13\x9D\xCC\x5B\x82\xD3\x4C\xB3\xED\x5B\xFF\xDE\xE5\x73\xAC\x23\x3B\x2D\x00\xBF\x35\x55\x74\x09\x49\xD8\x49\x58\x1A\x7F\x92\x36\xE6\x51\x92\x0E\xF3\x26\x7D\x1C\x4D\x17\xBC\xC9\xEC\x43\x26\xD0\xBF\x41\x5F\x40\xA9\x44\x44\xF4\x99\xE7\x57\x87\x9E\x50\x1F\x57\x54\xA8\x3E\xFD\x74\x63\x2F\xB1\x50\x65\x09\xE6\x58\x42\x2E\x43\x1A\x4C\xB4\xF0\x25\x47\x59\xFA\x04\x1E\x93\xD4\x26\x46\x4A\x50\x81\xB2\xDE\xBE\x78\xB7\xFC\x67\x15\xE1\xC9\x57\x84\x1E\x0F\x63\xD6\xE9\x62\xBA\xD6\x5F\x55\x2E\xEA\x5C\xC6\x28\x08\x04\x25\x39\xB8\x0E\x2B\xA9\xF2\x4C\x97\x1C\x07\x3F\x0D\x52\xF5\xED\xEF\x2F\x82\x0F\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x53\x79\xBF\x5A\xAA\x2B\x4A\xCF\x54\x80\xE1\xD8\x9B\xC0\x9D\xF2\xB2\x03\x66\xCB\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\x5C\xD4\x7C\x0D\xCF\xF7\x01\x7D\x41\x99\x65\x0C\x73\xC5\x52\x9F\xCB\xF8\xCF\x99\x06\x7F\x1B\xDA\x43\x15\x9F\x9E\x02\x55\x57\x96\x14\xF1\x52\x3C\x27\x87\x94\x28\xED\x1F\x3A\x01\x37\xA2\x76\xFC\x53\x50\xC0\x84\x9B\xC6\x6B\x4E\xBA\x8C\x21\x4F\xA2\x8E\x55\x62\x91\xF3\x69\x15\xD8\xBC\x88\xE3\xC4\xAA\x0B\xFD\xEF\xA8\xE9\x4B\x55\x2A\x06\x20\x6D\x55\x78\x29\x19\xEE\x5F\x30\x5C\x4B\x24\x11\x55\xFF\x24\x9A\x6E\x5E\x2A\x2B\xEE\x0B\x4D\x9F\x7F\xF7\x01\x38\x94\x14\x95\x43\x07\x09\xFB\x60\xA9\xEE\x1C\xAB\x12\x8C\xA0\x9A\x5E\xA7\x98\x6A\x59\x6D\x8B\x3F\x08\xFB\xC8\xD1\x45\xAF\x18\x15\x64\x90\x12\x0F\x73\x28\x2E\xC5\xE2\x24\x4E\xFC\x58\xEC\xF0\xF4\x45\xFE\x22\xB3\xEB\x2F\x8E\xD2\xD9\x45\x61\x05\xC1\x97\x6F\xA8\x76\x72\x8F\x8B\x8C\x36\xAF\xBF\x0D\x05\xCE\x71\x8D\xE6\xA6\x6F\x1F\x6C\xA6\x71\x62\xC5\xD8\xD0\x83\x72\x0C\xF1\x67\x11\x89\x0C\x9C\x13\x4C\x72\x34\xDF\xBC\xD5\x71\xDF\xAA\x71\xDD\xE1\xB9\x6C\x8C\x3C\x12\x5D\x65\xDA\xBD\x57\x12\xB6\x43\x6B\xFF\xE5\xDE\x4D\x66\x11\x51\xCF\x99\xAE\xEC\x17\xB6\xE8\x71\x91\x8C\xDE\x49\xFE\xDD\x35\x71\xA2\x15\x27\x94\x1C\xCF\x61\xE3\x26\xBB\x6F\xA3\x67\x25\x21\x5D\xE6\xDD\x1D\x0B\x2E\x68\x1B\x3B\x82\xAF\xEC\x83\x67\x85\xD4\x98\x51\x74\xB1\xB9\x99\x80\x89\xFF\x7F\x78\x19\x5C\x79\x4A\x60\x2E\x92\x40\xAE\x4C\x37\x2A\x2C\xC9\xC7\x62\xC8\x0E\x5D\xF7\x36\x5B\xCA\xE0\x25\x25\x01\xB4\xDD\x1A\x07\x9C\x77\x00\x3F\xD0\xDC\xD5\xEC\x3D\xD4\xFA\xBB\x3F\xCC\x85\xD6\x6F\x7F\xA9\x2D\xDF\xB9\x02\xF7\xF5\x97\x9A\xB5\x35\xDA\xC3\x67\xB0\x87\x4A\xA9\x28\x9E\x23\x8E\xFF\x5C\x27\x6B\xE1\xB0\x4F\xF3\x07\xEE\x00\x2E\xD4\x59\x87\xCB\x52\x41\x95\xEA\xF4\x47\xD7\xEE\x64\x41\x55\x7C\x8D\x59\x02\x95\xDD\x62\x9D\xC2\xB9\xEE\x5A\x28\x74\x84\xA5\x9B\xB7\x90\xC7\x0C\x07\xDF\xF5\x89\x36\x74\x32\xD6\x28\xC1\xB0\xB0\x0B\xE0\x9C\x4C\xC3\x1C\xD6\xFC\xE3\x69\xB5\x47\x46\x81\x2F\xA2\x82\xAB\xD3\x63\x44\x70\xC4\x8D\xFF\x2D\x33\xBA\xAD\x8F\x7B\xB5\x70\x88\xAE\x3E\x19\xCF\x40\x28\xD8\xFC\xC8\x90\xBB\x5D\x99\x22\xF5\x52\xE6\x58\xC5\x1F\x88\x31\x43\xEE\x88\x1D\xD7\xC6\x8E\x3C\x43\x6A\x1D\xA7\x18\xDE\x7D\x3D\x16\xF1\x62\xF9\xCA\x90\xA8\xFD"),
   // "certSIGN ROOT CA"
   CertificateDer::from_slice(b"\x30\x82\x03\x38\x30\x82\x02\x20\xA0\x03\x02\x01\x02\x02\x06\x20\x06\x05\x16\x70\x02\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x3B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x52\x4F\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x13\x08\x63\x65\x72\x74\x53\x49\x47\x4E\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x63\x65\x72\x74\x53\x49\x47\x4E\x20\x52\x4F\x4F\x54\x20\x43\x41\x30\x1E\x17\x0D\x30\x36\x30\x37\x30\x34\x31\x37\x32\x30\x30\x34\x5A\x17\x0D\x33\x31\x30\x37\x30\x34\x31\x37\x32\x30\x30\x34\x5A\x30\x3B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x52\x4F\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x13\x08\x63\x65\x72\x74\x53\x49\x47\x4E\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x13\x10\x63\x65\x72\x74\x53\x49\x47\x4E\x20\x52\x4F\x4F\x54\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xB7\x33\xB9\x7E\xC8\x25\x4A\x8E\xB5\xDB\xB4\x28\x1B\xAA\x57\x90\xE8\xD1\x22\xD3\x64\xBA\xD3\x93\xE8\xD4\xAC\x86\x61\x40\x6A\x60\x57\x68\x54\x84\x4D\xBC\x6A\x54\x02\x05\xFF\xDF\x9B\x9A\x2A\xAE\x5D\x07\x8F\x4A\xC3\x28\x7F\xEF\xFB\x2B\xFA\x79\xF1\xC7\xAD\xF0\x10\x53\x24\x90\x8B\x66\xC9\xA8\x88\xAB\xAF\x5A\xA3\x00\xE9\xBE\xBA\x46\xEE\x5B\x73\x7B\x2C\x17\x82\x81\x5E\x62\x2C\xA1\x02\x65\xB3\xBD\xC5\x2B\x00\x7E\xC4\xFC\x03\x33\x57\x0D\xED\xE2\xFA\xCE\x5D\x45\xD6\x38\xCD\x35\xB6\xB2\xC1\xD0\x9C\x81\x4A\xAA\xE4\xB2\x01\x5C\x1D\x8F\x5F\x99\xC4\xB1\xAD\xDB\x88\x21\xEB\x90\x08\x82\x80\xF3\x30\xA3\x43\xE6\x90\x82\xAE\x55\x28\x49\xED\x5B\xD7\xA9\x10\x38\x0E\xFE\x8F\x4C\x5B\x9B\x46\xEA\x41\xF5\xB0\x08\x74\xC3\xD0\x88\x33\xB6\x7C\xD7\x74\xDF\xDC\x84\xD1\x43\x0E\x75\x39\xA1\x25\x40\x28\xEA\x78\xCB\x0E\x2C\x2E\x39\x9D\x8C\x8B\x6E\x16\x1C\x2F\x26\x82\x10\xE2\xE3\x65\x94\x0A\x04\xC0\x5E\xF7\x5D\x5B\xF8\x10\xE2\xD0\xBA\x7A\x4B\xFB\xDE\x37\x00\x00\x1A\x5B\x28\xE3\xD2\x9C\x73\x3E\x32\x87\x98\xA1\xC9\x51\x2F\xD7\xDE\xAC\x33\xB3\x4F\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\xC6\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xE0\x8C\x9B\xDB\x25\x49\xB3\xF1\x7C\x86\xD6\xB2\x42\x87\x0B\xD0\x6B\xA0\xD9\xE4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x3E\xD2\x1C\x89\x2E\x35\xFC\xF8\x75\xDD\xE6\x7F\x65\x88\xF4\x72\x4C\xC9\x2C\xD7\x32\x4E\xF3\xDD\x19\x79\x47\xBD\x8E\x3B\x5B\x93\x0F\x50\x49\x24\x13\x6B\x14\x06\x72\xEF\x09\xD3\xA1\xA1\xE3\x40\x84\xC9\xE7\x18\x32\x74\x3C\x48\x6E\x0F\x9F\x4B\xD4\xF7\x1E\xD3\x93\x86\x64\x54\x97\x63\x72\x50\xD5\x55\xCF\xFA\x20\x93\x02\xA2\x9B\xC3\x23\x93\x4E\x16\x55\x76\xA0\x70\x79\x6D\xCD\x21\x1F\xCF\x2F\x2D\xBC\x19\xE3\x88\x31\xF8\x59\x1A\x81\x09\xC8\x97\xA6\x74\xC7\x60\xC4\x5B\xCC\x57\x8E\xB2\x75\xFD\x1B\x02\x09\xDB\x59\x6F\x72\x93\x69\xF7\x31\x41\xD6\x88\x38\xBF\x87\xB2\xBD\x16\x79\xF9\xAA\xE4\xBE\x88\x25\xDD\x61\x27\x23\x1C\xB5\x31\x07\x04\x36\xB4\x1A\x90\xBD\xA0\x74\x71\x50\x89\x6D\xBC\x14\xE3\x0F\x86\xAE\xF1\xAB\x3E\xC7\xA0\x09\xCC\xA3\x48\xD1\xE0\xDB\x64\xE7\x92\xB5\xCF\xAF\x72\x43\x70\x8B\xF9\xC3\x84\x3C\x13\xAA\x7E\x92\x9B\x57\x53\x93\xFA\x70\xC2\x91\x0E\x31\xF9\x9B\x67\x5D\xE9\x96\x38\x5E\x5F\xB3\x73\x4E\x88\x15\x67\xDE\x9E\x76\x10\x62\x20\xBE\x55\x69\x95\x43\x00\x39\x4D\xF6\xEE\xB0\x5A\x4E\x49\x44\x54\x58\x5F\x42\x83"),
   // "AC RAIZ FNMT-RCM"
   CertificateDer::from_slice(b"\x30\x82\x05\x83\x30\x82\x03\x6B\xA0\x03\x02\x01\x02\x02\x0F\x5D\x93\x8D\x30\x67\x36\xC8\x06\x1D\x1A\xC7\x54\x84\x69\x07\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x3B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x0C\x10\x41\x43\x20\x52\x41\x49\x5A\x20\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x30\x1E\x17\x0D\x30\x38\x31\x30\x32\x39\x31\x35\x35\x39\x35\x36\x5A\x17\x0D\x33\x30\x30\x31\x30\x31\x30\x30\x30\x30\x30\x30\x5A\x30\x3B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x11\x30\x0F\x06\x03\x55\x04\x0A\x0C\x08\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x31\x19\x30\x17\x06\x03\x55\x04\x0B\x0C\x10\x41\x43\x20\x52\x41\x49\x5A\x20\x46\x4E\x4D\x54\x2D\x52\x43\x4D\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xBA\x71\x80\x7A\x4C\x86\x6E\x7F\xC8\x13\x6D\xC0\xC6\x7D\x1C\x00\x97\x8F\x2C\x0C\x23\xBB\x10\x9A\x40\xA9\x1A\xB7\x87\x88\xF8\x9B\x56\x6A\xFB\xE6\x7B\x8E\x8B\x92\x8E\xA7\x25\x5D\x59\x11\xDB\x36\x2E\xB7\x51\x17\x1F\xA9\x08\x1F\x04\x17\x24\x58\xAA\x37\x4A\x18\xDF\xE5\x39\xD4\x57\xFD\xD7\xC1\x2C\x91\x01\x91\xE2\x22\xD4\x03\xC0\x58\xFC\x77\x47\xEC\x8F\x3E\x74\x43\xBA\xAC\x34\x8D\x4D\x38\x76\x67\x8E\xB0\xC8\x6F\x30\x33\x58\x71\x5C\xB4\xF5\x6B\x6E\xD4\x01\x50\xB8\x13\x7E\x6C\x4A\xA3\x49\xD1\x20\x19\xEE\xBC\xC0\x29\x18\x65\xA7\xDE\xFE\xEF\xDD\x0A\x90\x21\xE7\x1A\x67\x92\x42\x10\x98\x5F\x4F\x30\xBC\x3E\x1C\x45\xB4\x10\xD7\x68\x40\x14\xC0\x40\xFA\xE7\x77\x17\x7A\xE6\x0B\x8F\x65\x5B\x3C\xD9\x9A\x52\xDB\xB5\xBD\x9E\x46\xCF\x3D\xEB\x91\x05\x02\xC0\x96\xB2\x76\x4C\x4D\x10\x96\x3B\x92\xFA\x9C\x7F\x0F\x99\xDF\xBE\x23\x35\x45\x1E\x02\x5C\xFE\xB5\xA8\x9B\x99\x25\xDA\x5E\xF3\x22\xC3\x39\xF5\xE4\x2A\x2E\xD3\xC6\x1F\xC4\x6C\xAA\xC5\x1C\x6A\x01\x05\x4A\x2F\xD2\xC5\xC1\xA8\x34\x26\x5D\x66\xA5\xD2\x02\x21\xF9\x18\xB7\x06\xF5\x4E\x99\x6F\xA8\xAB\x4C\x51\xE8\xCF\x50\x18\xC5\x77\xC8\x39\x09\x2C\x49\x92\x32\x99\xA8\xBB\x17\x17\x79\xB0\x5A\xC5\xE6\xA3\xC4\x59\x65\x47\x35\x83\x5E\xA9\xE8\x35\x0B\x99\xBB\xE4\xCD\x20\xC6\x9B\x4A\x06\x39\xB5\x68\xFC\x22\xBA\xEE\x55\x8C\x2B\x4E\xEA\xF3\xB1\xE3\xFC\xB6\x99\x9A\xD5\x42\xFA\x71\x4D\x08\xCF\x87\x1E\x6A\x71\x7D\xF9\xD3\xB4\xE9\xA5\x71\x81\x7B\xC2\x4E\x47\x96\xA5\xF6\x76\x85\xA3\x28\x8F\xE9\x80\x6E\x81\x53\xA5\x6D\x5F\xB8\x48\xF9\xC2\xF9\x36\xA6\x2E\x49\xFF\xB8\x96\xC2\x8C\x07\xB3\x9B\x88\x58\xFC\xEB\x1B\x1C\xDE\x2D\x70\xE2\x97\x92\x30\xA1\x89\xE3\xBC\x55\xA8\x27\xD6\x4B\xED\x90\xAD\x8B\xFA\x63\x25\x59\x2D\xA8\x35\xDD\xCA\x97\x33\xBC\xE5\xCD\xC7\x9D\xD1\xEC\xEF\x5E\x0E\x4A\x90\x06\x26\x63\xAD\xB9\xD9\x35\x2D\x07\xBA\x76\x65\x2C\xAC\x57\x8F\x7D\xF4\x07\x94\xD7\x81\x02\x96\x5D\xA3\x07\x49\xD5\x7A\xD0\x57\xF9\x1B\xE7\x53\x46\x75\xAA\xB0\x79\x42\xCB\x68\x71\x08\xE9\x60\xBD\x39\x69\xCE\xF4\xAF\xC3\x56\x40\xC7\xAD\x52\xA2\x09\xE4\x6F\x86\x47\x8A\x1F\xEB\x28\x27\x5D\x83\x20\xAF\x04\xC9\x6C\x56\x9A\x8B\x46\xF5\x02\x03\x01\x00\x01\xA3\x81\x83\x30\x81\x80\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF7\x7D\xC5\xFD\xC4\xE8\x9A\x1B\x77\x64\xA7\xF5\x1D\xA0\xCC\xBF\x87\x60\x9A\x6D\x30\x3E\x06\x03\x55\x1D\x20\x04\x37\x30\x35\x30\x33\x06\x04\x55\x1D\x20\x00\x30\x2B\x30\x29\x06\x08\x2B\x06\x01\x05\x05\x07\x02\x01\x16\x1D\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x63\x65\x72\x74\x2E\x66\x6E\x6D\x74\x2E\x65\x73\x2F\x64\x70\x63\x73\x2F\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x07\x90\x4A\xDF\xF3\x23\x4E\xF0\xC3\x9C\x51\x65\x9B\x9C\x22\xA2\x8A\x0C\x85\xF3\x73\x29\x6B\x4D\xFE\x01\xE2\xA9\x0C\x63\x01\xBF\x04\x67\xA5\x9D\x98\x5F\xFD\x01\x13\xFA\xEC\x9A\x62\xE9\x86\xFE\xB6\x62\xD2\x6E\x4C\x94\xFB\xC0\x75\x45\x7C\x65\x0C\xF8\xB2\x37\xCF\xAC\x0F\xCF\x8D\x6F\xF9\x19\xF7\x8F\xEC\x1E\xF2\x70\x9E\xF0\xCA\xB8\xEF\xB7\xFF\x76\x37\x76\x5B\xF6\x6E\x88\xF3\xAF\x62\x32\x22\x93\x0D\x3A\x6A\x8E\x14\x66\x0C\x2D\x53\x74\x57\x65\x1E\xD5\xB2\xDD\x23\x81\x3B\xA5\x66\x23\x27\x67\x09\x8F\xE1\x77\xAA\x43\xCD\x65\x51\x08\xED\x51\x58\xFE\xE6\x39\xF9\xCB\x47\x84\xA4\x15\xF1\x76\xBB\xA4\xEE\xA4\x3B\xC4\x5F\xEF\xB2\x33\x96\x11\x18\xB7\xC9\x65\xBE\x18\xE1\xA3\xA4\xDC\xFA\x18\xF9\xD3\xBC\x13\x9B\x39\x7A\x34\xBA\xD3\x41\xFB\xFA\x32\x8A\x2A\xB7\x2B\x86\x0B\x69\x83\x38\xBE\xCD\x8A\x2E\x0B\x70\xAD\x8D\x26\x92\xEE\x1E\xF5\x01\x2B\x0A\xD9\xD6\x97\x9B\x6E\xE0\xA8\x19\x1C\x3A\x21\x8B\x0C\x1E\x40\xAD\x03\xE7\xDD\x66\x7E\xF5\xB9\x20\x0D\x03\xE8\x96\xF9\x82\x45\xD4\x39\xE0\xA0\x00\x5D\xD7\x98\xE6\x7D\x9E\x67\x73\xC3\x9A\x2A\xF7\xAB\x8B\xA1\x3A\x14\xEF\x34\xBC\x52\x0E\x89\x98\x9A\x04\x40\x84\x1D\x7E\x45\x69\x93\x57\xCE\xEB\xCE\xF8\x50\x7C\x4F\x1C\x6E\x04\x43\x9B\xF9\xD6\x3B\x23\x18\xE9\xEA\x8E\xD1\x4D\x46\x8D\xF1\x3B\xE4\x6A\xCA\xBA\xFB\x23\xB7\x9B\xFA\x99\x01\x29\x5A\x58\x5A\x2D\xE3\xF9\xD4\x6D\x0E\x26\xAD\xC1\x6E\x34\xBC\x32\xF8\x0C\x05\xFA\x65\xA3\xDB\x3B\x37\x83\x22\xE9\xD6\xDC\x72\x33\xFD\x5D\xF2\x20\xBD\x76\x3C\x23\xDA\x28\xF7\xF9\x1B\xEB\x59\x64\xD5\xDC\x5F\x72\x7E\x20\xFC\xCD\x89\xB5\x90\x67\x4D\x62\x7A\x3F\x4E\xAD\x1D\xC3\x39\xFE\x7A\xF4\x28\x16\xDF\x41\xF6\x48\x80\x05\xD7\x0F\x51\x79\xAC\x10\xAB\xD4\xEC\x03\x66\xE6\x6A\xB0\xBA\x31\x92\x42\x40\x6A\xBE\x3A\xD3\x72\xE1\x6A\x37\x55\xBC\xAC\x1D\x95\xB7\x69\x61\xF2\x43\x91\x74\xE6\xA0\xD3\x0A\x24\x46\xA1\x08\xAF\xD6\xDA\x45\x19\x96\xD4\x53\x1D\x5B\x84\x79\xF0\xC0\xF7\x47\xEF\x8B\x8F\xC5\x06\xAE\x9D\x4C\x62\x9D\xFF\x46\x04\xF8\xD3\xC9\xB6\x10\x25\x40\x75\xFE\x16\xAA\xC9\x4A\x60\x86\x2F\xBA\xEF\x30\x77\xE4\x54\xE2\xB8\x84\x99\x58\x80\xAA\x13\x8B\x51\x3A\x4F\x48\xF6\x8B\xB6\xB3"),
   // "Buypass Class 3 Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\x59\x30\x82\x03\x41\xA0\x03\x02\x01\x02\x02\x01\x02\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4E\x4F\x31\x1D\x30\x1B\x06\x03\x55\x04\x0A\x0C\x14\x42\x75\x79\x70\x61\x73\x73\x20\x41\x53\x2D\x39\x38\x33\x31\x36\x33\x33\x32\x37\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x0C\x17\x42\x75\x79\x70\x61\x73\x73\x20\x43\x6C\x61\x73\x73\x20\x33\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x30\x31\x30\x32\x36\x30\x38\x32\x38\x35\x38\x5A\x17\x0D\x34\x30\x31\x30\x32\x36\x30\x38\x32\x38\x35\x38\x5A\x30\x4E\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x4E\x4F\x31\x1D\x30\x1B\x06\x03\x55\x04\x0A\x0C\x14\x42\x75\x79\x70\x61\x73\x73\x20\x41\x53\x2D\x39\x38\x33\x31\x36\x33\x33\x32\x37\x31\x20\x30\x1E\x06\x03\x55\x04\x03\x0C\x17\x42\x75\x79\x70\x61\x73\x73\x20\x43\x6C\x61\x73\x73\x20\x33\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xA5\xDA\x0A\x95\x16\x50\xE3\x95\xF2\x5E\x9D\x76\x31\x06\x32\x7A\x9B\xF1\x10\x76\xB8\x00\x9A\xB5\x52\x36\xCD\x24\x47\xB0\x9F\x18\x64\xBC\x9A\xF6\xFA\xD5\x79\xD8\x90\x62\x4C\x22\x2F\xDE\x38\x3D\xD6\xE0\xA8\xE9\x1C\x2C\xDB\x78\x11\xE9\x8E\x68\x51\x15\x72\xC7\xF3\x33\x87\xE4\xA0\x5D\x0B\x5C\xE0\x57\x07\x2A\x30\xF5\xCD\xC4\x37\x77\x28\x4D\x18\x91\xE6\xBF\xD5\x52\xFD\x71\x2D\x70\x3E\xE7\xC6\xC4\x8A\xE3\xF0\x28\x0B\xF4\x76\x98\xA1\x8B\x87\x55\xB2\x3A\x13\xFC\xB7\x3E\x27\x37\x8E\x22\xE3\xA8\x4F\x2A\xEF\x60\xBB\x3D\xB7\x39\xC3\x0E\x01\x47\x99\x5D\x12\x4F\xDB\x43\xFA\x57\xA1\xED\xF9\x9D\xBE\x11\x47\x26\x5B\x13\x98\xAB\x5D\x16\x8A\xB0\x37\x1C\x57\x9D\x45\xFF\x88\x96\x36\xBF\xBB\xCA\x07\x7B\x6F\x87\x63\xD7\xD0\x32\x6A\xD6\x5D\x6C\x0C\xF1\xB3\x6E\x39\xE2\x6B\x31\x2E\x39\x00\x27\x14\xDE\x38\xC0\xEC\x19\x66\x86\x12\xE8\x9D\x72\x16\x13\x64\x52\xC7\xA9\x37\x1C\xFD\x82\x30\xED\x84\x18\x1D\xF4\xAE\x5C\xFF\x70\x13\x00\xEB\xB1\xF5\x33\x7A\x4B\xD6\x55\xF8\x05\x8D\x4B\x69\xB0\xF5\xB3\x28\x36\x5C\x14\xC4\x51\x73\x4D\x6B\x0B\xF1\x34\x07\xDB\x17\x39\xD7\xDC\x28\x7B\x6B\xF5\x9F\xF3\x2E\xC1\x4F\x17\x2A\x10\xF3\xCC\xCA\xE8\xEB\xFD\x6B\xAB\x2E\x9A\x9F\x2D\x82\x6E\x04\xD4\x52\x01\x93\x2D\x3D\x86\xFC\x7E\xFC\xDF\xEF\x42\x1D\xA6\x6B\xEF\xB9\x20\xC6\xF7\xBD\xA0\xA7\x95\xFD\xA7\xE6\x89\x24\xD8\xCC\x8C\x34\x6C\xE2\x23\x2F\xD9\x12\x1A\x21\xB9\x55\x91\x6F\x0B\x91\x79\x19\x0C\xAD\x40\x88\x0B\x70\xE2\x7A\xD2\x0E\xD8\x68\x48\xBB\x82\x13\x39\x10\x58\xE9\xD8\x2A\x07\xC6\x12\xDB\x58\xDB\xD2\x3B\x55\x10\x47\x05\x15\x67\x62\x7E\x18\x63\xA6\x46\x3F\x09\x0E\x54\x32\x5E\xBF\x0D\x62\x7A\x27\xEF\x80\xE8\xDB\xD9\x4B\x06\x5A\x37\x5A\x25\xD0\x08\x12\x77\xD4\x6F\x09\x50\x97\x3D\xC8\x1D\xC3\xDF\x8C\x45\x30\x56\xC6\xD3\x64\xAB\x66\xF3\xC0\x5E\x96\x9C\xC3\xC4\xEF\xC3\x7C\x6B\x8B\x3A\x79\x7F\xB3\x49\xCF\x3D\xE2\x89\x9F\xA0\x30\x4B\x85\xB9\x9C\x94\x24\x79\x8F\x7D\x6B\xA9\x45\x68\x0F\x2B\xD0\xF1\xDA\x1C\xCB\x69\xB8\xCA\x49\x62\x6D\xC8\xD0\x63\x62\xDD\x60\x0F\x58\xAA\x8F\xA1\xBC\x05\xA5\x66\xA2\xCF\x1B\x76\xB2\x84\x64\xB1\x4C\x39\x52\xC0\x30\xBA\xF0\x8C\x4B\x02\xB0\xB6\xB7\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x47\xB8\xCD\xFF\xE5\x6F\xEE\xF8\xB2\xEC\x2F\x4E\x0E\xF9\x25\xB0\x8E\x3C\x6B\xC3\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x00\x20\x23\x41\x35\x04\x90\xC2\x40\x62\x60\xEF\xE2\x35\x4C\xD7\x3F\xAC\xE2\x34\x90\xB8\xA1\x6F\x76\xFA\x16\x16\xA4\x48\x37\x2C\xE9\x90\xC2\xF2\x3C\xF8\x0A\x9F\xD8\x81\xE5\xBB\x5B\xDA\x25\x2C\xA4\xA7\x55\x71\x24\x32\xF6\xC8\x0B\xF2\xBC\x6A\xF8\x93\xAC\xB2\x07\xC2\x5F\x9F\xDB\xCC\xC8\x8A\xAA\xBE\x6A\x6F\xE1\x49\x10\xCC\x31\xD7\x80\xBB\xBB\xC8\xD8\xA2\x0E\x64\x57\xEA\xA2\xF5\xC2\xA9\x31\x15\xD2\x20\x6A\xEC\xFC\x22\x01\x28\xCF\x86\xB8\x80\x1E\xA9\xCC\x11\xA5\x3C\xF2\x16\xB3\x47\x9D\xFC\xD2\x80\x21\xC4\xCB\xD0\x47\x70\x41\xA1\xCA\x83\x19\x08\x2C\x6D\xF2\x5D\x77\x9C\x8A\x14\x13\xD4\x36\x1C\x92\xF0\xE5\x06\x37\xDC\xA6\xE6\x90\x9B\x38\x8F\x5C\x6B\x1B\x46\x86\x43\x42\x5F\x3E\x01\x07\x53\x54\x5D\x65\x7D\xF7\x8A\x73\xA1\x9A\x54\x5A\x1F\x29\x43\x14\x27\xC2\x85\x0F\xB5\x88\x7B\x1A\x3B\x94\xB7\x1D\x60\xA7\xB5\x9C\xE7\x29\x69\x57\x5A\x9B\x93\x7A\x43\x30\x1B\x03\xD7\x62\xC8\x40\xA6\xAA\xFC\x64\xE4\x4A\xD7\x91\x53\x01\xA8\x20\x88\x6E\x9C\x5F\x44\xB9\xCB\x60\x81\x34\xEC\x6F\xD3\x7D\xDA\x48\x5F\xEB\xB4\x90\xBC\x2D\xA9\x1C\x0B\xAC\x1C\xD5\xA2\x68\x20\x80\x04\xD6\xFC\xB1\x8F\x2F\xBB\x4A\x31\x0D\x4A\x86\x1C\xEB\xE2\x36\x29\x26\xF5\xDA\xD8\xC4\xF2\x75\x61\xCF\x7E\xAE\x76\x63\x4A\x7A\x40\x65\x93\x87\xF8\x1E\x80\x8C\x86\xE5\x86\xD6\x8F\x0E\xFC\x53\x2C\x60\xE8\x16\x61\x1A\xA2\x3E\x43\x7B\xCD\x39\x60\x54\x6A\xF5\xF2\x89\x26\x01\x68\x83\x48\xA2\x33\xE8\xC9\x04\x91\xB2\x11\x34\x11\x3E\xEA\xD0\x43\x19\x1F\x03\x93\x90\x0C\xFF\x51\x3D\x57\xF4\x41\x6E\xE1\xCB\xA0\xBE\xEB\xC9\x63\xCD\x6D\xCC\xE4\xF8\x36\xAA\x68\x9D\xED\xBD\x5D\x97\x70\x44\x0D\xB6\x0E\x35\xDC\xE1\x0C\x5D\xBB\xA0\x51\x94\xCB\x7E\x16\xEB\x11\x2F\xA3\x92\x45\xC8\x4C\x71\xD9\xBC\xC9\x99\x52\x57\x46\x2F\x50\xCF\xBD\x35\x69\xF4\x3D\x15\xCE\x06\xA5\x2C\x0F\x3E\xF6\x81\xBA\x94\xBB\xC3\xBB\xBF\x65\x78\xD2\x86\x79\xFF\x49\x3B\x1A\x83\x0C\xF0\xDE\x78\xEC\xC8\xF2\x4D\x4C\x1A\xDE\x82\x29\xF8\xC1\x5A\xDA\xED\xEE\xE6\x27\x5E\xE8\x45\xD0\x9D\x1C\x51\xA8\x68\xAB\x44\xE3\xD0\x8B\x6A\xE3\xF8\x3B\xBB\xDC\x4D\xD7\x64\xF2\x51\xBE\xE6\xAA\xAB\x5A\xE9\x31\xEE\x06\xBC\x73\xBF\x13\x62\x0A\x9F\xC7\xB9\x97"),
   // "D-TRUST Root Class 3 CA 2 EV 2009"
   CertificateDer::from_slice(b"\x30\x82\x04\x43\x30\x82\x03\x2B\xA0\x03\x02\x01\x02\x02\x03\x09\x83\xF4\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x50\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x0C\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x2A\x30\x28\x06\x03\x55\x04\x03\x0C\x21\x44\x2D\x54\x52\x55\x53\x54\x20\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x33\x20\x43\x41\x20\x32\x20\x45\x56\x20\x32\x30\x30\x39\x30\x1E\x17\x0D\x30\x39\x31\x31\x30\x35\x30\x38\x35\x30\x34\x36\x5A\x17\x0D\x32\x39\x31\x31\x30\x35\x30\x38\x35\x30\x34\x36\x5A\x30\x50\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x15\x30\x13\x06\x03\x55\x04\x0A\x0C\x0C\x44\x2D\x54\x72\x75\x73\x74\x20\x47\x6D\x62\x48\x31\x2A\x30\x28\x06\x03\x55\x04\x03\x0C\x21\x44\x2D\x54\x52\x55\x53\x54\x20\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x33\x20\x43\x41\x20\x32\x20\x45\x56\x20\x32\x30\x30\x39\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\x99\xF1\x84\x34\x70\xBA\x2F\xB7\x30\xA0\x8E\xBD\x7C\x04\xCF\xBE\x62\xBC\x99\xFD\x82\x97\xD2\x7A\x0A\x67\x96\x38\x09\xF6\x10\x4E\x95\x22\x73\x99\x8D\xDA\x15\x2D\xE7\x05\xFC\x19\x73\x22\xB7\x8E\x98\x00\xBC\x3C\x3D\xAC\xA1\x6C\xFB\xD6\x79\x25\x4B\xAD\xF0\xCC\x64\xDA\x88\x3E\x29\xB8\x0F\x09\xD3\x34\xDD\x33\xF5\x62\xD1\xE1\xCD\x19\xE9\xEE\x18\x4F\x4C\x58\xAE\xE2\x1E\xD6\x0C\x5B\x15\x5A\xD8\x3A\xB8\xC4\x18\x64\x1E\xE3\x33\xB2\xB5\x89\x77\x4E\x0C\xBF\xD9\x94\x6B\x13\x97\x6F\x12\xA3\xFE\x99\xA9\x04\xCC\x15\xEC\x60\x68\x36\xED\x08\x7B\xB7\xF5\xBF\x93\xED\x66\x31\x83\x8C\xC6\x71\x34\x87\x4E\x17\xEA\xAF\x8B\x91\x8D\x1C\x56\x41\xAE\x22\x37\x5E\x37\xF2\x1D\xD9\xD1\x2D\x0D\x2F\x69\x51\xA7\xBE\x66\xA6\x8A\x3A\x2A\xBD\xC7\x1A\xB1\xE1\x14\xF0\xBE\x3A\x1D\xB9\xCF\x5B\xB1\x6A\xFE\xB4\xB1\x46\x20\xA2\xFB\x1E\x3B\x70\xEF\x93\x98\x7D\x8C\x73\x96\xF2\xC5\xEF\x85\x70\xAD\x29\x26\xFC\x1E\x04\x3E\x1C\xA0\xD8\x0F\xCB\x52\x83\x62\x7C\xEE\x8B\x53\x95\x90\xA9\x57\xA2\xEA\x61\x05\xD8\xF9\x4D\xC4\x27\xFA\x6E\xAD\xED\xF9\xD7\x51\xF7\x6B\xA5\x02\x03\x01\x00\x01\xA3\x82\x01\x24\x30\x82\x01\x20\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xD3\x94\x8A\x4C\x62\x13\x2A\x19\x2E\xCC\xAF\x72\x8A\x7D\x36\xD7\x9A\x1C\xDC\x67\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x81\xDD\x06\x03\x55\x1D\x1F\x04\x81\xD5\x30\x81\xD2\x30\x81\x87\xA0\x81\x84\xA0\x81\x81\x86\x7F\x6C\x64\x61\x70\x3A\x2F\x2F\x64\x69\x72\x65\x63\x74\x6F\x72\x79\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x43\x4E\x3D\x44\x2D\x54\x52\x55\x53\x54\x25\x32\x30\x52\x6F\x6F\x74\x25\x32\x30\x43\x6C\x61\x73\x73\x25\x32\x30\x33\x25\x32\x30\x43\x41\x25\x32\x30\x32\x25\x32\x30\x45\x56\x25\x32\x30\x32\x30\x30\x39\x2C\x4F\x3D\x44\x2D\x54\x72\x75\x73\x74\x25\x32\x30\x47\x6D\x62\x48\x2C\x43\x3D\x44\x45\x3F\x63\x65\x72\x74\x69\x66\x69\x63\x61\x74\x65\x72\x65\x76\x6F\x63\x61\x74\x69\x6F\x6E\x6C\x69\x73\x74\x30\x46\xA0\x44\xA0\x42\x86\x40\x68\x74\x74\x70\x3A\x2F\x2F\x77\x77\x77\x2E\x64\x2D\x74\x72\x75\x73\x74\x2E\x6E\x65\x74\x2F\x63\x72\x6C\x2F\x64\x2D\x74\x72\x75\x73\x74\x5F\x72\x6F\x6F\x74\x5F\x63\x6C\x61\x73\x73\x5F\x33\x5F\x63\x61\x5F\x32\x5F\x65\x76\x5F\x32\x30\x30\x39\x2E\x63\x72\x6C\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x34\xED\x7B\x5A\x3C\xA4\x94\x88\xEF\x1A\x11\x75\x07\x2F\xB3\xFE\x3C\xFA\x1E\x51\x26\xEB\x87\xF6\x29\xDE\xE0\xF1\xD4\xC6\x24\x09\xE9\xC1\xCF\x55\x1B\xB4\x30\xD9\xCE\x1A\xFE\x06\x51\xA6\x15\xA4\x2D\xEF\xB2\x4B\xBF\x20\x28\x25\x49\xD1\xA6\x36\x77\x34\xE8\x64\xDF\x52\xB1\x11\xC7\x73\x7A\xCD\x39\x9E\xC2\xAD\x8C\x71\x21\xF2\x5A\x6B\xAF\xDF\x3C\x4E\x55\xAF\xB2\x84\x65\x14\x89\xB9\x77\xCB\x2A\x31\xBE\xCF\xA3\x6D\xCF\x6F\x48\x94\x32\x46\x6F\xE7\x71\x8C\xA0\xA6\x84\x19\x37\x07\xF2\x03\x45\x09\x2B\x86\x75\x7C\xDF\x5F\x69\x57\x00\xDB\x6E\xD8\xA6\x72\x22\x4B\x50\xD4\x75\x98\x56\xDF\xB7\x18\xFF\x43\x43\x50\xAE\x7A\x44\x7B\xF0\x79\x51\xD7\x43\x3D\xA7\xD3\x81\xD3\xF0\xC9\x4F\xB9\xDA\xC6\x97\x86\xD0\x82\xC3\xE4\x42\x6D\xFE\xB0\xE2\x64\x4E\x0E\x26\xE7\x40\x34\x26\xB5\x08\x89\xD7\x08\x63\x63\x38\x27\x75\x1E\x33\xEA\x6E\xA8\xDD\x9F\x99\x4F\x74\x4D\x81\x89\x80\x4B\xDD\x9A\x97\x29\x5C\x2F\xBE\x81\x41\xB9\x8C\xFF\xEA\x7D\x60\x06\x9E\xCD\xD7\x3D\xD3\x2E\xA3\x15\xBC\xA8\xE6\x26\xE5\x6F\xC3\xDC\xB8\x03\x21\xEA\x9F\x16\xF1\x2C\x54\xB5"),
   // "OISTE Server Root ECC G1"
   CertificateDer::from_slice(b"\x30\x82\x02\x35\x30\x82\x01\xBA\xA0\x03\x02\x01\x02\x02\x10\x23\xF9\xC3\xD6\x35\xAF\x8F\x28\x4B\x1F\xF0\x54\xEA\x7E\x97\x9D\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x30\x4B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x0C\x10\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x31\x21\x30\x1F\x06\x03\x55\x04\x03\x0C\x18\x4F\x49\x53\x54\x45\x20\x53\x65\x72\x76\x65\x72\x20\x52\x6F\x6F\x74\x20\x45\x43\x43\x20\x47\x31\x30\x1E\x17\x0D\x32\x33\x30\x35\x33\x31\x31\x34\x34\x32\x32\x38\x5A\x17\x0D\x34\x38\x30\x35\x32\x34\x31\x34\x34\x32\x32\x37\x5A\x30\x4B\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x48\x31\x19\x30\x17\x06\x03\x55\x04\x0A\x0C\x10\x4F\x49\x53\x54\x45\x20\x46\x6F\x75\x6E\x64\x61\x74\x69\x6F\x6E\x31\x21\x30\x1F\x06\x03\x55\x04\x03\x0C\x18\x4F\x49\x53\x54\x45\x20\x53\x65\x72\x76\x65\x72\x20\x52\x6F\x6F\x74\x20\x45\x43\x43\x20\x47\x31\x30\x76\x30\x10\x06\x07\x2A\x86\x48\xCE\x3D\x02\x01\x06\x05\x2B\x81\x04\x00\x22\x03\x62\x00\x04\x17\x2F\xFA\x12\xBC\xAC\x18\xF3\x0A\xF4\x44\xD6\x76\x42\x9E\xB3\xE8\x1F\xB7\x79\xA9\x58\xB6\xF8\x65\xD1\x3A\x21\x4F\xA8\xEB\xA3\xBE\xA4\x32\x72\xF3\xB6\x01\xC9\x2B\xFD\x77\x85\x6E\x53\xDD\xAD\xEA\xAA\x2E\x25\x92\xB6\xE9\x21\x11\xA8\xAF\xB5\x4C\x0B\xF3\x96\x60\x9A\x3B\xE7\xEA\x1A\x78\x2E\xB4\x3D\xE5\x28\xDE\x1C\x80\xBA\x5C\x6E\x0D\xDB\x19\xA5\xE3\x3F\x9C\x2A\xB8\x40\x4B\xDD\xE6\x4F\xA3\x63\x30\x61\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x37\x4D\x88\x65\xCF\xFC\x3D\x8A\xD5\xA3\xF1\x49\xC0\x4E\x0C\x10\x6F\x42\xB4\x9C\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x37\x4D\x88\x65\xCF\xFC\x3D\x8A\xD5\xA3\xF1\x49\xC0\x4E\x0C\x10\x6F\x42\xB4\x9C\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0A\x06\x08\x2A\x86\x48\xCE\x3D\x04\x03\x03\x03\x69\x00\x30\x66\x02\x31\x00\xA9\x2A\x30\x1D\xD0\xC2\x9F\x90\x51\x51\x40\x3E\x95\x54\x21\xCD\x16\x66\xF7\x53\x6C\x08\x16\x39\xD0\x12\x7C\x7F\x63\x1B\xDF\xE3\x38\x00\x39\xD9\x2D\x53\x20\x45\x0B\x1C\x60\x67\x31\x43\x25\xED\x02\x31\x00\x92\x89\xAE\xE9\x5C\x62\x83\x62\x61\xF9\x2D\x57\xAB\x56\xB9\x11\xDD\x25\xBE\x6A\x4E\x4A\x1A\x82\x6B\xDC\xCF\xD3\xBC\x4A\xB3\x3C\xD7\x2E\x9B\xDB\xF8\x28\x69\xBC\x6B\x2D\xEC\x31\xA1\x3A\xE3\x57"),
   // "Telekom Security TLS RSA Root 2023"
   CertificateDer::from_slice(b"\x30\x82\x05\xB3\x30\x82\x03\x9B\xA0\x03\x02\x01\x02\x02\x10\x21\x9C\x54\x2D\xE8\xF6\xEC\x71\x77\xFA\x4E\xE8\xC3\x70\x57\x97\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x30\x63\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x27\x30\x25\x06\x03\x55\x04\x0A\x0C\x1E\x44\x65\x75\x74\x73\x63\x68\x65\x20\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x47\x6D\x62\x48\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x0C\x22\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x32\x30\x32\x33\x30\x1E\x17\x0D\x32\x33\x30\x33\x32\x38\x31\x32\x31\x36\x34\x35\x5A\x17\x0D\x34\x38\x30\x33\x32\x37\x32\x33\x35\x39\x35\x39\x5A\x30\x63\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x27\x30\x25\x06\x03\x55\x04\x0A\x0C\x1E\x44\x65\x75\x74\x73\x63\x68\x65\x20\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x47\x6D\x62\x48\x31\x2B\x30\x29\x06\x03\x55\x04\x03\x0C\x22\x54\x65\x6C\x65\x6B\x6F\x6D\x20\x53\x65\x63\x75\x72\x69\x74\x79\x20\x54\x4C\x53\x20\x52\x53\x41\x20\x52\x6F\x6F\x74\x20\x32\x30\x32\x33\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xED\x35\xA1\x81\x80\xF3\xCB\x4A\x69\x5B\xC2\xFB\x51\x83\xAE\x26\xFD\xE1\x6E\xF3\x81\x12\x7D\x71\x40\xFF\x87\x75\x42\x29\x21\xED\x81\x52\x2C\xDF\x12\xC1\x19\x84\x89\xC1\xBD\xC5\x28\xD5\xD5\x4B\x6C\x44\xD6\x4C\xDB\x07\x96\x4A\x55\x7A\xCA\x36\x82\x04\x36\xA8\xA5\xFC\x27\xF6\x49\xF1\xD5\x72\x9E\x91\xF9\x23\xD6\x70\x7B\xBB\xF5\x9B\xC1\xEC\x93\xCF\x19\xEA\x65\x7E\x88\x70\xA0\x73\xFC\xF6\xFF\xB5\x56\x62\xE1\x73\x6A\x34\x98\x3E\x82\xB8\xAC\x95\x53\xF4\x01\xA0\x27\x07\x72\xA3\x00\x53\xA0\xE4\xB2\xAB\x83\x38\x57\x33\x25\x94\x9F\xBE\x48\x1D\x98\xE1\xA3\xBA\x9E\x5C\xCD\x04\x71\x51\x7D\x75\x78\xAB\xF3\x59\xAA\xC4\xE0\x60\xBE\x8F\x83\x52\xB8\x75\x1A\x41\x35\xED\xBC\xF3\x3A\x63\xE9\xA9\x14\x45\xD7\xE6\x52\xD1\x6E\xD2\xDE\xBC\xE3\xF5\x0B\x3B\xE6\xE0\xC4\xBD\x43\x64\x13\xA6\xCE\xF4\x98\x37\x6C\x8A\x95\xA8\x97\xC8\x47\x0F\xF0\x5E\x10\x8B\xE7\x1D\x1C\xFE\xB1\x3B\xA0\x05\x33\x68\x05\x41\x82\xC1\x03\x2B\x01\xC8\xE7\x8F\x4D\xAB\xE8\xB5\xF6\xCD\x6B\x44\xB5\xE7\xDD\x8B\xEC\xEA\x25\xB4\x00\x22\x57\x4D\xB0\xB1\xB2\x31\xC1\x16\xCE\xFF\xFD\x14\x84\xB7\x47\xFA\xB2\xF1\x70\xDE\xDB\x8B\x6C\x36\x58\xA4\x7C\xB3\x11\xD1\xC3\x77\x7F\x5F\xB6\x25\xE0\x0D\xC5\xD2\xB3\xF9\xB8\xB8\x77\xDB\x37\x71\x71\x47\xE3\x60\x18\x4F\x24\xB6\x75\x37\x78\xB9\xA3\x62\xAF\xBD\xC9\x72\x8E\x2F\xCC\xBB\xAE\xDB\xE4\x15\x52\x19\x07\x33\xFB\x6A\xB7\x2D\x4B\x90\x28\x82\x73\xFE\x18\x8B\x35\x8D\xDB\xA7\x04\x6A\xBE\xEA\xC1\x4D\x36\x3B\x16\x36\x91\x32\xEF\xB6\x40\x89\x91\x43\xE0\xF2\xA2\xAB\x04\x2E\xE6\xF2\x4C\x0E\x16\x34\x20\xAC\x87\xC1\x2D\x7E\xC9\x66\x47\x17\x14\x11\xA4\xF3\xF7\xA1\x24\x89\xAB\xD8\x1A\xC8\xA1\x5C\xB1\xA3\xF7\x8C\x6D\xC8\x01\xC9\x4F\xC9\xEC\xC4\xFC\xAC\x51\x33\xD1\xC8\x83\xD1\xC9\x9F\x1D\xD4\x47\x34\x29\x3E\xCB\xB0\x0E\xFA\x83\x0B\x28\x58\xE5\x29\xDC\x3F\x7C\xA8\x9F\xC9\xB6\x0A\xBB\xA6\xE8\x46\x16\x0F\x96\xE5\x7B\xE4\x6A\x7A\x48\x6D\x76\x98\x05\xA5\xDC\x6D\x1E\x42\x1E\x42\xDA\x1A\xE0\x52\xF7\xB5\x83\xC0\x1A\x7B\x78\x35\x2C\x38\xF5\x1F\xFD\x49\xA3\x2E\xD2\x59\x63\xBF\x80\xB0\x8C\x93\x73\xCB\x35\xA6\x99\x95\x22\x61\x65\x03\x60\xFB\x2F\x93\x4B\xFA\x9A\x9C\x80\x3B\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB6\xA7\x97\x82\x3D\x74\x85\x9B\xF7\x3C\x9F\x93\x9A\x95\x79\x75\x52\x8C\x6D\x47\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xB6\xA7\x97\x82\x3D\x74\x85\x9B\xF7\x3C\x9F\x93\x9A\x95\x79\x75\x52\x8C\x6D\x47\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0C\x05\x00\x03\x82\x02\x01\x00\xA8\xCC\x61\xA6\xBE\x75\x9E\x15\x50\xA4\x6B\xFB\xA8\x70\x45\x7C\xBA\x7E\xB1\x5A\xFC\x5B\x23\xFA\x0A\x77\xF8\x98\x71\x82\x0C\x6D\xE0\x5E\x46\xAA\x93\xF4\x1E\xA0\xC3\xE1\x93\xDB\x4B\xAD\xB2\xA6\x5D\xAB\xB0\xD4\x62\xCB\x5E\xBB\x66\xF5\x2D\xEE\x97\x40\x3C\x62\xEB\x5E\xD6\x14\xD6\x8C\xE2\x96\x8B\x41\x69\x93\x35\xE6\xB9\x99\x6B\x62\xB4\xA1\x17\x66\x34\xA6\x6B\x63\xC6\xB9\x4E\xF2\x22\xE9\x58\x0D\x56\x41\xD1\xFA\x0C\x4A\xF0\x33\xCD\x3B\xBB\x6D\x21\x3A\xAE\x8E\x72\xB5\xC3\x4A\xFB\xE9\x7D\xE5\xB1\x9B\x86\xEE\xE2\xE0\x7D\xB4\xF7\x32\xFD\x22\x84\xF1\x85\xC9\x37\x79\xE9\xB5\x3F\xBF\x5C\xE4\x74\xB2\x8F\x11\x62\x00\xDD\x18\x66\xA1\xD9\x7B\x23\x5F\xF1\x8E\xD5\x67\xE8\x54\xDA\x5B\x3A\x6B\x36\x6F\xF9\x81\xB1\x33\x47\x33\x77\x40\xF9\x52\xAA\xDD\xD4\x83\xCF\x85\x78\x99\x9A\x93\xB9\x73\x67\x42\x46\x11\x21\xEA\xFE\x0A\xA9\x1B\x1A\x65\x69\xB3\x8F\xAE\x16\xB6\xF6\x4B\x56\xB2\x2D\xF9\xA5\xC8\xEC\x3B\x62\xA3\xED\x6B\xD0\x4E\xD5\x40\x09\xA4\x1F\x98\xD7\x3A\xA5\x92\x59\x20\xE4\xB0\x7D\xCD\x5B\x73\x68\xBD\x6D\xC4\xA2\x13\x0E\x67\x19\xB8\x8D\x42\x7E\x6C\x0C\x9A\x6E\xA0\x24\x2D\xD5\x45\x1B\xDC\xC4\x02\x14\xFE\x85\x5B\x65\x97\xCA\x4E\x90\x50\x08\x7A\x42\x35\xF9\xEA\xC2\x66\xD4\xF8\x01\xAE\x1E\xB4\xBE\xC3\xA8\xEF\xFE\x76\x9A\xA2\xA6\x1F\x46\xF6\x84\xED\xFC\xDB\xCE\xC4\x02\xCE\x77\x48\x2C\x8C\xB2\xEC\xC3\x00\xA3\xEC\x2C\x55\x18\xC1\x7E\x19\xEE\xE1\x2F\xF2\xAD\x83\x9B\x9E\xAB\x19\xDF\xC6\x8A\x2F\x8C\x77\xE5\xB7\x05\xEC\x3B\xC1\xEC\xBE\x86\xB3\x86\xBC\xC0\xF7\xDC\xE7\xEA\x5B\xAE\xB2\xCC\xB5\x35\x86\x4B\xD0\xE2\x3F\xB6\xD8\xF8\x0E\x00\xEE\x5D\xE3\xF7\x8D\x58\xFF\xCF\x8B\x37\xE9\x63\x5F\x6E\xF7\x09\x71\x36\xC2\x12\x5D\x57\xF2\xC8\xB4\xCD\xF3\xEE\x02\xDF\x11\xDC\x6A\xB9\x57\x84\x1D\x59\x4D\x8C\xCE\xC8\x0E\x23\xC2\xB7\x26\x9A\x10\x14\x71\xFE\x93\xB2\x8A\xB8\x80\xF0\x0E\x10\x9E\xD3\xA8\x50\x0C\x37\x82\x2F\xEA\xE0\x8A\x9D\xE1\x2C\x39\xFF\xB5\xB4\x73\x00\xE4\xF7\x48\xA6\x73\xAC\xBF\xB2\xDE\x77\x04\x87\xB4\xA3\xCD\x9B\x35\x24\x37\xFA\x90\x93\x13\x81\x42\xC6\x98\x26\x75\x37\x66\x41\x10\xAC\xBB\xF5\x94\xE3\xC2\x31\x2B\xAD\xE7\x23\x56\xCC\x35\x25\x92\xB3\x50"),
   // "HiPKI Root CA - G1"
   CertificateDer::from_slice(b"\x30\x82\x05\x6A\x30\x82\x03\x52\xA0\x03\x02\x01\x02\x02\x10\x2D\xDD\xAC\xCE\x62\x97\x94\xA1\x43\xE8\xB0\xCD\x76\x6A\x5E\x60\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x4F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x0C\x1A\x43\x68\x75\x6E\x67\x68\x77\x61\x20\x54\x65\x6C\x65\x63\x6F\x6D\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x0C\x12\x48\x69\x50\x4B\x49\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x47\x31\x30\x1E\x17\x0D\x31\x39\x30\x32\x32\x32\x30\x39\x34\x36\x30\x34\x5A\x17\x0D\x33\x37\x31\x32\x33\x31\x31\x35\x35\x39\x35\x39\x5A\x30\x4F\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x54\x57\x31\x23\x30\x21\x06\x03\x55\x04\x0A\x0C\x1A\x43\x68\x75\x6E\x67\x68\x77\x61\x20\x54\x65\x6C\x65\x63\x6F\x6D\x20\x43\x6F\x2E\x2C\x20\x4C\x74\x64\x2E\x31\x1B\x30\x19\x06\x03\x55\x04\x03\x0C\x12\x48\x69\x50\x4B\x49\x20\x52\x6F\x6F\x74\x20\x43\x41\x20\x2D\x20\x47\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xF4\x1E\x7F\x52\x73\x32\x0C\x73\xE4\xBD\x13\x74\xA3\xD4\x30\xA8\xD0\xAE\x4B\xD8\xB6\xDF\x75\x47\x66\xF4\x7C\xE7\x39\x04\x1E\x6A\x70\x20\xD2\x5A\x47\x72\x67\x55\xF4\xA5\xE8\x9D\xD5\x1E\x21\xA1\xF0\x67\xBA\xCC\x21\x68\xBE\x44\x53\xBF\x8D\xF9\xE2\xDC\x2F\x55\xC8\x37\x3F\x1F\xA4\xC0\x9C\xB3\xE4\x77\x5C\xA0\x46\xFE\x77\xFA\x1A\xA0\x38\xEA\xED\x9A\x72\xDE\x2B\xBD\x94\x57\x3A\xBA\xEC\x79\xE7\x5F\x7D\x42\x64\x39\x7A\x26\x36\xF7\x24\xF0\xD5\x2F\xBA\x95\x98\x11\x66\xAD\x97\x35\xD6\x75\x01\x80\xE0\xAF\xF4\x84\x61\x8C\x0D\x1E\x5F\x7C\x87\x96\x5E\x41\xAF\xEB\x87\xEA\xF8\x5D\xF1\x2E\x88\x05\x3E\x4C\x22\xBB\xDA\x1F\x2A\xDD\x52\x46\x64\x39\xF3\x42\xCE\xD9\x9E\x0C\xB3\xB0\x77\x97\x64\x9C\xC0\xF4\xA3\x2E\x1F\x95\x07\xB0\x17\xDF\x30\xDB\x00\x18\x96\x4C\xA1\x81\x4B\xDD\x04\x6D\x53\xA3\x3D\xFC\x07\xAC\xD4\xC5\x37\x82\xEB\xE4\x95\x08\x19\x28\x82\xD2\x42\x3A\xA3\xD8\x53\xEC\x79\x89\x60\x48\x60\xC8\x72\x92\x50\xDC\x03\x8F\x83\x3F\xB2\x42\x57\x5A\xDB\x6A\xE9\x11\x97\xDD\x85\x28\xBC\x30\x4C\xAB\xE3\xC2\xB1\x45\x44\x47\x1F\xE0\x8A\x16\x07\x96\xD2\x21\x0F\x53\xC0\xED\xA9\x7E\xD4\x4E\xEC\x9B\x09\xEC\xAF\x42\xAC\x30\xD6\xBF\xD1\x10\x45\xE0\xA6\x16\xB2\xA5\xC5\xD3\x4F\x73\x94\x33\x71\x02\xA1\x6A\xA3\xD6\x33\x97\x4F\x21\x63\x1E\x5B\x8F\xD9\xC1\x5E\x45\x71\x77\x0F\x81\x5D\x5F\x21\x9A\xAD\x83\xCC\xFA\x5E\xD6\x8D\x23\x5F\x1B\x3D\x41\xAF\x20\x75\x66\x5A\x4A\xF6\x9F\xFB\xAB\x18\xF7\x71\xC0\xB6\x1D\x31\xEC\x3B\x20\xEB\xCB\xE2\xB8\xF5\xAE\x92\xB2\xF7\xE1\x84\x4B\xF2\xA2\xF2\x93\x9A\x22\x9E\xD3\x14\x6F\x36\x54\xBD\x1F\x5E\x59\x15\xB9\x73\xA8\xC1\x7C\x6F\x7B\x62\xE9\x16\x6C\x47\x5A\x65\xF3\x0E\x11\x9B\x46\xD9\xFD\x6D\xDC\xD6\x9C\xC0\xB4\x7D\xA5\xB0\xDD\x3F\x56\x6F\xA1\xF9\xF6\xE4\x12\x48\xFD\x06\x7F\x12\x57\xB6\xA9\x23\x4F\x5B\x03\xC3\xE0\x71\x2A\x23\xB7\xF7\xB0\xB1\x3B\xBC\x98\xBD\xD6\x98\xA8\x0C\x6B\xF6\x8E\x12\x67\xA6\xF2\xB2\x58\xE4\x02\x09\x13\x3C\xA9\xBB\x10\xB4\xD2\x30\x45\xF1\xEC\xF7\x00\x11\xDF\x65\xF8\xDC\x2B\x43\x55\xBF\x16\x97\xC4\x0F\xD5\x2C\x61\x84\xAA\x72\x86\xFE\xE6\x3A\x7E\xC2\x3F\x7D\xEE\xFC\x2F\x14\x3E\xE6\x85\xDD\x50\x6F\xB7\x49\xED\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xF2\x77\x17\xFA\x5E\xA8\xFE\xF6\x3D\x71\xD5\x68\xBA\xC9\x46\x0C\x38\xD8\xAF\xB0\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x50\x51\xF0\x75\xDC\x70\x04\xE3\xFF\xAA\x75\xD4\x71\xA2\xCB\x9E\x8F\xA8\xA9\xD3\xAF\x75\xC7\x54\xCF\x3A\x1C\x04\x99\x22\xAC\xC4\x11\xE2\xEF\x33\x4A\xA6\x23\x1D\x0E\x0D\x47\xD8\x37\xC7\x6F\xAF\x34\x7F\x4F\x81\x6B\x35\x4F\xE9\x72\xA5\x31\xE2\x78\xE7\xF7\x4E\x94\x18\x5B\x40\x7D\xCF\x6B\x21\x54\x86\xE6\x95\x7A\xFB\xC6\xCA\xEA\x9C\x48\x4E\x57\x09\x5D\x2F\xAC\xF4\xA5\xB4\x97\x33\x58\xD5\xAC\x79\xA9\xCC\x5F\xF9\x85\xFA\x52\xC5\x8D\xF8\x91\x14\xEB\x3A\x0D\x17\xD0\x52\xC2\x7B\xE3\xC2\x73\x8E\x46\x78\x06\x38\x2C\xE8\x5C\xDA\x66\xC4\xF4\xA4\xF0\x56\x19\x33\x29\x5A\x65\x92\x05\x47\x46\x4A\xAB\x84\xC3\x1E\x27\xA1\x1F\x11\x92\x99\x27\x75\x93\x0F\xBC\x36\x3B\x97\x57\x8F\x26\x5B\x0C\xBB\x9C\x0F\xD4\x6E\x30\x07\xD4\xDC\x5F\x36\x68\x66\x39\x83\x96\x27\x26\x8A\xC8\xC4\x39\xFE\x9A\x21\x6F\xD5\x72\x86\xE9\x7F\x62\xE5\x97\x4E\xD0\x24\xD0\x40\xB0\xD0\x75\x08\x8E\xBD\x68\xEE\x08\xD7\x6E\x7C\x10\x70\x46\x1B\x7C\xE0\x88\xB2\x9E\x72\x86\x99\x01\xE3\xBF\x9F\x49\x19\xB4\x25\xBE\x56\x65\xAE\x17\x63\xE5\x1E\xDF\xE8\xFF\x47\xA5\xBF\xE1\x26\x05\x84\xE4\xB0\xC0\xAF\xE7\x08\x99\xA8\x0C\x5E\x26\x80\x45\xD4\xF8\x68\x2F\x96\x8F\xAE\xE2\x4A\x1C\x9C\x16\x0C\x13\x6F\x38\x87\xF6\xBB\xC8\x34\x5F\x92\x03\x51\x79\x70\xA6\xDF\xCB\xF5\x99\x4D\x79\xCD\x4E\xBC\x57\x9F\x43\x4E\x6B\x2E\x2B\x18\xF8\x6A\x73\x8C\xBA\xC5\x35\xEF\x39\x6A\x41\x1E\xCF\x71\xA8\xA2\xB2\x86\x07\x5B\x3A\xC9\xE1\xEF\x3F\x65\x04\x80\x47\x32\x44\x70\x95\x4E\x31\x67\x6A\x74\x5B\x10\x45\x75\xEA\xB0\x9F\xD0\xE6\x35\xFE\x4E\x9F\x8B\xCC\x2B\x92\x45\x5B\x6E\x25\x60\x85\x46\xCD\xD1\xAA\xB0\x76\x66\x93\x77\x96\xBE\x83\xBE\x38\xB6\x24\x4E\x26\x0B\xCC\xED\x7A\x56\x1A\xE0\xE9\x5A\xC6\x64\xAD\x4C\x7A\x00\x48\x44\x2F\xB9\x40\xBB\x13\x3E\xBE\x15\x78\x9D\x85\x81\x4A\x2A\x57\xDE\xD5\x19\x43\xDA\xDB\xCA\x5B\x47\x86\x83\x0B\x3F\xB6\x0D\x76\x78\x73\x79\x22\x5E\xB1\x80\x1F\xCF\xBE\xD1\x3F\x56\x10\x98\x2B\x95\x87\xA1\x1F\x9D\x64\x14\x60\x39\x2C\xB3\x00\x55\x2E\xE4\xF5\xB3\x0E\x57\xC4\x91\x41\x00\x9C\x3F\xE8\xA5\xDF\xEA\xF6\xFF\xC8\xF0\xAD\x6D\x52\xA8\x17\xAB\x9B\x61\xFC\x12\x51\x35\xE4\x25\xFD\xAF\xAA\x6A\x86\x39"),
   // "SecureTrust CA"
   CertificateDer::from_slice(b"\x30\x82\x03\xB8\x30\x82\x02\xA0\xA0\x03\x02\x01\x02\x02\x10\x0C\xF0\x8E\x5C\x08\x16\xA5\xAD\x42\x7F\xF0\xEB\x27\x18\x59\xD0\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x20\x30\x1E\x06\x03\x55\x04\x0A\x13\x17\x53\x65\x63\x75\x72\x65\x54\x72\x75\x73\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x17\x30\x15\x06\x03\x55\x04\x03\x13\x0E\x53\x65\x63\x75\x72\x65\x54\x72\x75\x73\x74\x20\x43\x41\x30\x1E\x17\x0D\x30\x36\x31\x31\x30\x37\x31\x39\x33\x31\x31\x38\x5A\x17\x0D\x32\x39\x31\x32\x33\x31\x31\x39\x34\x30\x35\x35\x5A\x30\x48\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x55\x53\x31\x20\x30\x1E\x06\x03\x55\x04\x0A\x13\x17\x53\x65\x63\x75\x72\x65\x54\x72\x75\x73\x74\x20\x43\x6F\x72\x70\x6F\x72\x61\x74\x69\x6F\x6E\x31\x17\x30\x15\x06\x03\x55\x04\x03\x13\x0E\x53\x65\x63\x75\x72\x65\x54\x72\x75\x73\x74\x20\x43\x41\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xAB\xA4\x81\xE5\x95\xCD\xF5\xF6\x14\x8E\xC2\x4F\xCA\xD4\xE2\x78\x95\x58\x9C\x41\xE1\x0D\x99\x40\x24\x17\x39\x91\x33\x66\xE9\xBE\xE1\x83\xAF\x62\x5C\x89\xD1\xFC\x24\x5B\x61\xB3\xE0\x11\x11\x41\x1C\x1D\x6E\xF0\xB8\xBB\xF8\xDE\xA7\x81\xBA\xA6\x48\xC6\x9F\x1D\xBD\xBE\x8E\xA9\x41\x3E\xB8\x94\xED\x29\x1A\xD4\x8E\xD2\x03\x1D\x03\xEF\x6D\x0D\x67\x1C\x57\xD7\x06\xAD\xCA\xC8\xF5\xFE\x0E\xAF\x66\x25\x48\x04\x96\x0B\x5D\xA3\xBA\x16\xC3\x08\x4F\xD1\x46\xF8\x14\x5C\xF2\xC8\x5E\x01\x99\x6D\xFD\x88\xCC\x86\xA8\xC1\x6F\x31\x42\x6C\x52\x3E\x68\xCB\xF3\x19\x34\xDF\xBB\x87\x18\x56\x80\x26\xC4\xD0\xDC\xC0\x6F\xDF\xDE\xA0\xC2\x91\x16\xA0\x64\x11\x4B\x44\xBC\x1E\xF6\xE7\xFA\x63\xDE\x66\xAC\x76\xA4\x71\xA3\xEC\x36\x94\x68\x7A\x77\xA4\xB1\xE7\x0E\x2F\x81\x7A\xE2\xB5\x72\x86\xEF\xA2\x6B\x8B\xF0\x0F\xDB\xD3\x59\x3F\xBA\x72\xBC\x44\x24\x9C\xE3\x73\xB3\xF7\xAF\x57\x2F\x42\x26\x9D\xA9\x74\xBA\x00\x52\xF2\x4B\xCD\x53\x7C\x47\x0B\x36\x85\x0E\x66\xA9\x08\x97\x16\x34\x57\xC1\x66\xF7\x80\xE3\xED\x70\x54\xC7\x93\xE0\x2E\x28\x15\x59\x87\xBA\xBB\x02\x03\x01\x00\x01\xA3\x81\x9D\x30\x81\x9A\x30\x13\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x14\x02\x04\x06\x1E\x04\x00\x43\x00\x41\x30\x0B\x06\x03\x55\x1D\x0F\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x42\x32\xB6\x16\xFA\x04\xFD\xFE\x5D\x4B\x7A\xC3\xFD\xF7\x4C\x40\x1D\x5A\x43\xAF\x30\x34\x06\x03\x55\x1D\x1F\x04\x2D\x30\x2B\x30\x29\xA0\x27\xA0\x25\x86\x23\x68\x74\x74\x70\x3A\x2F\x2F\x63\x72\x6C\x2E\x73\x65\x63\x75\x72\x65\x74\x72\x75\x73\x74\x2E\x63\x6F\x6D\x2F\x53\x54\x43\x41\x2E\x63\x72\x6C\x30\x10\x06\x09\x2B\x06\x01\x04\x01\x82\x37\x15\x01\x04\x03\x02\x01\x00\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x30\xED\x4F\x4A\xE1\x58\x3A\x52\x72\x5B\xB5\xA6\xA3\x65\x18\xA6\xBB\x51\x3B\x77\xE9\x9D\xEA\xD3\x9F\x5C\xE0\x45\x65\x7B\x0D\xCA\x5B\xE2\x70\x50\xB2\x94\x05\x14\xAE\x49\xC7\x8D\x41\x07\x12\x73\x94\x7E\x0C\x23\x21\xFD\xBC\x10\x7F\x60\x10\x5A\x72\xF5\x98\x0E\xAC\xEC\xB9\x7F\xDD\x7A\x6F\x5D\xD3\x1C\xF4\xFF\x88\x05\x69\x42\xA9\x05\x71\xC8\xB7\xAC\x26\xE8\x2E\xB4\x8C\x6A\xFF\x71\xDC\xB8\xB1\xDF\x99\xBC\x7C\x21\x54\x2B\xE4\x58\xA2\xBB\x57\x29\xAE\x9E\xA9\xA3\x19\x26\x0F\x99\x2E\x08\xB0\xEF\xFD\x69\xCF\x99\x1A\x09\x8D\xE3\xA7\x9F\x2B\xC9\x36\x34\x7B\x24\xB3\x78\x4C\x95\x17\xA4\x06\x26\x1E\xB6\x64\x52\x36\x5F\x60\x67\xD9\x9C\xC5\x05\x74\x0B\xE7\x67\x23\xD2\x08\xFC\x88\xE9\xAE\x8B\x7F\xE1\x30\xF4\x37\x7E\xFD\xC6\x32\xDA\x2D\x9E\x44\x30\x30\x6C\xEE\x07\xDE\xD2\x34\xFC\xD2\xFF\x40\xF6\x4B\xF4\x66\x46\x06\x54\xA6\xF2\x32\x0A\x63\x26\x30\x6B\x9B\xD1\xDC\x8B\x47\xBA\xE1\xB9\xD5\x62\xD0\xA2\xA0\xF4\x67\x05\x78\x29\x63\x1A\x6F\x04\xD6\xF8\xC6\x4C\xA3\x9A\xB1\x37\xB4\x8D\xE5\x28\x4B\x1D\x9E\x2C\xC2\xB8\x68\xBC\xED\x02\xEE\x31"),
   // "Atos TrustedRoot 2011"
   CertificateDer::from_slice(b"\x30\x82\x03\x77\x30\x82\x02\x5F\xA0\x03\x02\x01\x02\x02\x08\x5C\x33\xCB\x62\x2C\x5F\xB3\x32\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x3C\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x0C\x15\x41\x74\x6F\x73\x20\x54\x72\x75\x73\x74\x65\x64\x52\x6F\x6F\x74\x20\x32\x30\x31\x31\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x74\x6F\x73\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x30\x1E\x17\x0D\x31\x31\x30\x37\x30\x37\x31\x34\x35\x38\x33\x30\x5A\x17\x0D\x33\x30\x31\x32\x33\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x3C\x31\x1E\x30\x1C\x06\x03\x55\x04\x03\x0C\x15\x41\x74\x6F\x73\x20\x54\x72\x75\x73\x74\x65\x64\x52\x6F\x6F\x74\x20\x32\x30\x31\x31\x31\x0D\x30\x0B\x06\x03\x55\x04\x0A\x0C\x04\x41\x74\x6F\x73\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\x95\x85\x3B\x97\x6F\x2A\x3B\x2E\x3B\xCF\xA6\xF3\x29\x35\xBE\xCF\x18\xAC\x3E\xAA\xD9\xF8\x4D\xA0\x3E\x1A\x47\xB9\xBC\x9A\xDF\xF2\xFE\xCC\x3E\x47\xE8\x7A\x96\xC2\x24\x8E\x35\xF4\xA9\x0C\xFC\x82\xFD\x6D\xC1\x72\x62\x27\xBD\xEA\x6B\xEB\xE7\x8A\xCC\x54\x3E\x90\x50\xCF\x80\xD4\x95\xFB\xE8\xB5\x82\xD4\x14\xC5\xB6\xA9\x55\x25\x57\xDB\xB1\x50\xF6\xB0\x60\x64\x59\x7A\x69\xCF\x03\xB7\x6F\x0D\xBE\xCA\x3E\x6F\x74\x72\xEA\xAA\x30\x2A\x73\x62\xBE\x49\x91\x61\xC8\x11\xFE\x0E\x03\x2A\xF7\x6A\x20\xDC\x02\x15\x0D\x5E\x15\x6A\xFC\xE3\x82\xC1\xB5\xC5\x9D\x64\x09\x6C\xA3\x59\x98\x07\x27\xC7\x1B\x96\x2B\x61\x74\x71\x6C\x43\xF1\xF7\x35\x89\x10\xE0\x9E\xEC\x55\xA1\x37\x22\xA2\x87\x04\x05\x2C\x47\x7D\xB4\x1C\xB9\x62\x29\x66\x28\xCA\xB7\xE1\x93\xF5\xA4\x94\x03\x99\xB9\x70\x85\xB5\xE6\x48\xEA\x8D\x50\xFC\xD9\xDE\xCC\x6F\x07\x0E\xDD\x0B\x72\x9D\x80\x30\x16\x07\x95\x3F\x28\x0E\xFD\xC5\x75\x4F\x53\xD6\x74\x9A\xB4\x24\x2E\x8E\x02\x91\xCF\x76\xC5\x9B\x1E\x55\x74\x9C\x78\x21\xB1\xF0\x2D\xF1\x0B\x9F\xC2\xD5\x96\x18\x1F\xF0\x54\x22\x7A\x8C\x07\x02\x03\x01\x00\x01\xA3\x7D\x30\x7B\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xA7\xA5\x06\xB1\x2C\xA6\x09\x60\xEE\xD1\x97\xE9\x70\xAE\xBC\x3B\x19\x6C\xDB\x21\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\xA7\xA5\x06\xB1\x2C\xA6\x09\x60\xEE\xD1\x97\xE9\x70\xAE\xBC\x3B\x19\x6C\xDB\x21\x30\x18\x06\x03\x55\x1D\x20\x04\x11\x30\x0F\x30\x0D\x06\x0B\x2B\x06\x01\x04\x01\xB0\x2D\x03\x04\x01\x01\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x26\x77\x34\xDB\x94\x48\x86\x2A\x41\x9D\x2C\x3E\x06\x90\x60\xC4\x8C\xAC\x0B\x54\xB8\x1F\xB9\x7B\xD3\x07\x39\xE4\xFA\x3E\x7B\xB2\x3D\x4E\xED\x9F\x23\xBD\x97\xF3\x6B\x5C\xEF\xEE\xFD\x40\xA6\xDF\xA1\x93\xA1\x0A\x86\xAC\xEF\x20\xD0\x79\x01\xBD\x78\xF7\x19\xD8\x24\x31\x34\x04\x01\xA6\xBA\x15\x9A\xC3\x27\xDC\xD8\x4F\x0F\xCC\x18\x63\xFF\x99\x0F\x0E\x91\x6B\x75\x16\xE1\x21\xFC\xD8\x26\xC7\x47\xB7\xA6\xCF\x58\x72\x71\x7E\xBA\xE1\x4D\x95\x47\x3B\xC9\xAF\x6D\xA1\xB4\xC1\xEC\x89\xF6\xB4\x0F\x38\xB5\xE2\x64\xDC\x25\xCF\xA6\xDB\xEB\x9A\x5C\x99\xA1\xC5\x08\xDE\xFD\xE6\xDA\xD5\xD6\x5A\x45\x0C\xC4\xB7\xC2\xB5\x14\xEF\xB4\x11\xFF\x0E\x15\xB5\xF5\xF5\xDB\xC6\xBD\xEB\x5A\xA7\xF0\x56\x22\xA9\x3C\x65\x54\xC6\x15\xA8\xBD\x86\x9E\xCD\x83\x96\x68\x7A\x71\x81\x89\xE1\x0B\xE1\xEA\x11\x1B\x68\x08\xCC\x69\x9E\xEC\x9E\x41\x9E\x44\x32\x26\x7A\xE2\x87\x0A\x71\x3D\xEB\xE4\x5A\xA4\xD2\xDB\xC5\xCD\xC6\xDE\x60\x7F\xB9\xF3\x4F\x44\x92\xEF\x2A\xB7\x18\x3E\xA7\x19\xD9\x0B\x7D\xB1\x37\x41\x42\xB0\xBA\x60\x1D\xF2\xFE\x09\x11\xB0\xF0\x87\x7B\xA7\x9D"),
   // "BJCA Global Root CA1"
   CertificateDer::from_slice(b"\x30\x82\x05\x74\x30\x82\x03\x5C\xA0\x03\x02\x01\x02\x02\x10\x55\x6F\x65\xE3\xB4\xD9\x90\x6A\x1B\x09\xD1\x6C\x3E\xC0\x6C\x20\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x54\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x26\x30\x24\x06\x03\x55\x04\x0A\x0C\x1D\x42\x45\x49\x4A\x49\x4E\x47\x20\x43\x45\x52\x54\x49\x46\x49\x43\x41\x54\x45\x20\x41\x55\x54\x48\x4F\x52\x49\x54\x59\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x0C\x14\x42\x4A\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x30\x1E\x17\x0D\x31\x39\x31\x32\x31\x39\x30\x33\x31\x36\x31\x37\x5A\x17\x0D\x34\x34\x31\x32\x31\x32\x30\x33\x31\x36\x31\x37\x5A\x30\x54\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x43\x4E\x31\x26\x30\x24\x06\x03\x55\x04\x0A\x0C\x1D\x42\x45\x49\x4A\x49\x4E\x47\x20\x43\x45\x52\x54\x49\x46\x49\x43\x41\x54\x45\x20\x41\x55\x54\x48\x4F\x52\x49\x54\x59\x31\x1D\x30\x1B\x06\x03\x55\x04\x03\x0C\x14\x42\x4A\x43\x41\x20\x47\x6C\x6F\x62\x61\x6C\x20\x52\x6F\x6F\x74\x20\x43\x41\x31\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xF1\x66\x08\xBD\xD9\xC5\x15\x61\xCB\x84\x04\x41\xA5\x69\x37\x77\x1D\xC1\xB0\x7B\xFA\xC3\x77\x48\x90\x13\x72\x64\xD1\xB8\x7C\x90\x35\x9D\x18\x79\x88\xE3\x97\x01\x3C\x47\x81\xF2\x0E\xA2\x98\x0D\x9E\x3F\x37\xE0\x19\xB2\x90\xF2\x46\x1C\x92\xB1\x3A\x61\xCE\xFA\xB7\x46\x9E\x03\x86\xD7\x33\x6E\xED\xF7\x45\x8C\x76\x37\xDE\x6E\x96\x91\xF7\xD7\x7E\x2B\x87\x17\xD5\x8B\x35\xEE\x84\x91\x72\x57\xDC\x60\xC3\xC3\xB9\xE7\xC7\x67\x24\x23\x4F\x63\x0A\x63\xF6\x66\x7D\x4B\x55\xA7\x3F\x78\x64\x49\x69\x12\x97\xE0\x4C\x0D\xD3\x09\xA0\x32\x30\x3A\xFA\x9F\xC0\xF2\x9C\xC5\x12\x2A\x2E\x1C\xB5\x04\x33\xDA\xA4\x38\x11\x6A\xDE\xC6\x18\xF6\x47\x3A\x22\x41\x87\x22\xFC\xC4\x89\x28\x54\xD8\x8C\xA5\x30\x0A\xF8\x17\x16\xCA\xAC\x37\xFD\x79\xA7\x91\x17\x78\x38\x99\xAD\x58\xED\xB2\xDE\xCC\x89\x7D\x03\x9C\xB3\x89\x65\xE7\xE3\x3B\xB1\x22\x86\x8F\x06\x6D\x78\x07\xFD\x91\x12\x7F\xB0\x6B\x1C\x89\x0D\xF9\xB8\xCB\x74\x5B\x07\xC2\xC8\xF4\x35\xD1\x64\x63\x7A\xE9\x6E\x9A\x28\xD6\x30\xBD\xE6\x1B\xDD\x15\xAF\x84\xEA\x9C\xC7\xCA\xF5\x0E\xEA\xF2\x5D\x29\x87\x8F\x69\x73\x39\xBE\x2E\x24\x6F\x45\x21\xAC\xC5\xD4\x69\x25\x06\x83\xAD\x7A\x48\x85\x13\x2C\x0D\x06\xB8\x6C\x79\x56\xFC\xA3\x67\x32\x81\xF5\x57\xA5\xCA\x57\x42\x69\xE9\x5C\x24\x61\xEF\xE2\x30\x18\x4E\x44\x98\x55\x6F\x7A\xC2\x93\xD8\x19\xB6\xDE\x7C\x47\x8A\x11\x4E\x49\x47\xDB\x28\x94\x02\x0B\x94\x4A\x2C\xF9\x12\xD0\x4F\xE8\x31\x7E\x6C\x7A\xBF\xA6\x3F\x9B\x39\x3D\x02\x16\xA3\x18\xB3\x67\xAC\x5B\x3F\x2C\x83\x2B\x67\x39\x81\x5C\xB9\x7E\x94\xD5\x64\xDD\x9E\x8F\x6E\xAE\xE8\x7C\x5B\xB4\xD7\x6A\x47\x48\xD7\x7E\xB3\xD4\x2D\x8E\x56\x76\x4E\xCF\x69\xF1\x6E\x44\x6C\xD4\x24\xEA\x8D\x24\xA1\x18\xBF\xBD\x57\xFE\xA9\x99\x35\xB5\xDB\x10\x77\xB8\x3D\x48\xBA\xD6\xC1\xE7\xF1\x23\x3E\xD7\xDF\x85\x9D\x27\x3C\xD4\x40\xBD\x0A\x0C\xBD\xF5\xE7\x8D\x25\xD6\x81\x74\x87\x46\xD4\x29\x75\xA2\x42\x6C\xF7\x73\x89\xE7\x7D\xBF\x7A\x4A\x1F\xD3\x22\xC9\x15\x55\xCF\xDF\x6F\x7C\x55\xD0\xA4\x8B\x07\x11\x37\x5F\x83\xA6\x26\x57\xA6\x01\x5B\x7E\xFE\x58\x68\x07\xA9\xE9\x7A\xD9\xB9\xE8\xFF\x50\x1F\xAB\xC2\xB4\xC0\xCE\xE8\xEA\xFD\x0F\xBD\x8D\x4D\xB8\xBC\x71\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xC5\xEF\xED\xCC\xD8\x8D\x21\xC6\x48\xE4\xE3\xD7\x14\x2E\xA7\x16\x93\xE5\x98\x01\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x52\x82\xAC\x21\x34\x1F\x23\xF2\xA2\xD8\xF9\xB8\xAF\x37\x36\x20\x89\xD1\x37\x03\xD6\x69\x9F\xB8\x61\x10\xBA\xA2\x31\x98\x59\x47\xE8\xD1\x0D\x25\x1E\x15\x41\x0C\xE0\x2A\x55\xD5\x57\x52\xCB\xF8\xE4\xC7\x69\xA3\x1D\x4D\x71\x02\x5E\x5F\x21\x45\x60\x48\x5C\x09\x8E\x49\x10\xC1\x04\xDC\xA9\x62\x6B\x02\xF0\x43\xC8\x4E\x9D\x38\x49\x74\xC9\x32\x70\x54\x6D\xC1\x47\xFC\x8E\xB4\x36\x9E\xD4\x9C\xBD\xDD\x20\xD6\x53\xC9\x18\xA9\xB5\x56\xB9\x76\x8B\x95\x67\x66\xEE\xBD\x98\xFE\xAE\xEF\xBE\x6E\xFB\x60\xF6\xFD\x59\xC6\x2A\x1B\x3F\x23\x4A\x94\x24\x30\x27\xC8\x89\xBC\xEB\x44\x24\x9A\xCB\x3D\xBE\x4F\xD5\x7A\xCE\x8E\x17\xCB\x62\xC1\xD9\xDE\x1E\x0E\x7A\xFF\x43\x86\x34\x52\xBC\x61\x3F\x3C\x5F\xBB\xD9\x76\xB4\x53\xBC\x97\xB3\xFE\x8A\x4C\x12\x2E\x2B\xF3\xD7\xCE\xE1\xA2\xFF\xDD\x7B\x70\xFB\x3B\xA1\x4D\xA4\x63\x02\xFD\x38\x97\x95\x3F\x05\x70\xA0\x6B\xDF\x62\x81\x43\x8B\xB4\x59\x0D\x4A\x8C\x54\x9C\xC5\xBB\x81\x9F\xCD\x7D\xA5\xEF\x0B\x25\x1E\x3A\x20\xDB\x1C\xFC\x1F\x98\x67\x02\x0A\xD4\x73\x44\x13\xDB\x51\x84\x1A\x55\x03\x56\xE0\x00\x7E\x74\x06\xFF\x38\xC4\x72\x1D\xD3\xA8\x3F\x68\x31\x5D\xD3\x09\xC7\x2E\x8C\x5B\x63\xE0\xE8\xDC\x1E\xD2\xEC\x61\x1E\xF2\xDE\xE5\xEF\xF6\x99\x76\x60\x2D\x1E\x94\x72\x71\xC6\x0B\x2A\x32\xC7\x92\x4E\xD5\x46\xD7\x1D\xF9\xA9\x19\x0A\xC8\xFA\x95\xCE\x6D\x23\x98\xAA\x0B\x38\xAD\x9A\x56\x0D\x6F\x8D\xF1\x31\x00\x88\xC1\x17\x9C\xCD\x19\x36\x35\xFE\x55\x53\xA0\xE0\x3C\x33\x5F\x96\x5E\xE2\x32\xE9\xDF\x33\xBB\x06\x4A\xA9\xD8\x84\x73\xCE\x77\xD2\xC6\xAC\x71\xE1\x5C\xA3\x1D\x0C\xBB\x0A\xDF\x5F\xE2\xA3\x71\xD8\xDA\x37\x5A\xA0\x78\x2B\xF4\xD4\x7D\xEB\x76\xED\xF2\x61\x70\xA5\x65\x9A\xD3\x89\x34\x18\xAB\xFB\x72\x3E\xD7\xB4\x3D\x79\x5C\xD8\x1F\xA1\x33\x7B\xD9\x82\x50\x0C\x93\x17\xAA\x6C\xDC\xC2\x82\xBB\x02\x57\x36\xAF\x98\x27\x2A\x39\x50\xE1\xB0\x89\xF5\x25\x97\x7E\x47\x68\x10\xB4\xEC\x73\xCA\xB3\x97\xD1\x24\xDC\xF6\x62\xA0\x28\xD3\xB5\xA3\xB8\x64\xB7\x88\x62\x42\xCF\x9D\x53\xCD\x99\xBE\x64\x68\x8F\x4F\x1E\x12\x48\xF7\xD2\x29\xC3\x98\x28\xCA\xF2\x32\x0B\x93\x8C\x29\x4F\x3C\x60\x32\xCD\x05\x96\x61\xEC\xF2\xAF\xFE\xB3\x70\x2C\x2E\xA6\xF2"),
   // "ANF Secure Server Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\xEF\x30\x82\x03\xD7\xA0\x03\x02\x01\x02\x02\x08\x0D\xD3\xE3\xBC\x6C\xF9\x6B\xB1\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x84\x31\x12\x30\x10\x06\x03\x55\x04\x05\x13\x09\x47\x36\x33\x32\x38\x37\x35\x31\x30\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x27\x30\x25\x06\x03\x55\x04\x0A\x13\x1E\x41\x4E\x46\x20\x41\x75\x74\x6F\x72\x69\x64\x61\x64\x20\x64\x65\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x63\x69\x6F\x6E\x31\x14\x30\x12\x06\x03\x55\x04\x0B\x13\x0B\x41\x4E\x46\x20\x43\x41\x20\x52\x61\x69\x7A\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x41\x4E\x46\x20\x53\x65\x63\x75\x72\x65\x20\x53\x65\x72\x76\x65\x72\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x39\x30\x39\x30\x34\x31\x30\x30\x30\x33\x38\x5A\x17\x0D\x33\x39\x30\x38\x33\x30\x31\x30\x30\x30\x33\x38\x5A\x30\x81\x84\x31\x12\x30\x10\x06\x03\x55\x04\x05\x13\x09\x47\x36\x33\x32\x38\x37\x35\x31\x30\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x27\x30\x25\x06\x03\x55\x04\x0A\x13\x1E\x41\x4E\x46\x20\x41\x75\x74\x6F\x72\x69\x64\x61\x64\x20\x64\x65\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x63\x69\x6F\x6E\x31\x14\x30\x12\x06\x03\x55\x04\x0B\x13\x0B\x41\x4E\x46\x20\x43\x41\x20\x52\x61\x69\x7A\x31\x22\x30\x20\x06\x03\x55\x04\x03\x13\x19\x41\x4E\x46\x20\x53\x65\x63\x75\x72\x65\x20\x53\x65\x72\x76\x65\x72\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xDB\xEB\x6B\x2B\xE6\x64\x54\x95\x82\x90\xA3\x72\xA4\x19\x01\x9D\x9C\x0B\x81\x5F\x73\x49\xBA\xA7\xAC\xF3\x04\x4E\x7B\x96\x0B\xEC\x11\xE0\x5B\xA6\x1C\xCE\x1B\xD2\x0D\x83\x1C\x2B\xB8\x9E\x1D\x7E\x45\x32\x60\x0F\x07\xE9\x77\x58\x7E\x9F\x6A\xC8\x61\x4E\xB6\x26\xC1\x4C\x8D\xFF\x4C\xEF\x34\xB2\x1F\x65\xD8\xB9\x78\xF5\xAD\xA9\x71\xB9\xEF\x4F\x58\x1D\xA5\xDE\x74\x20\x97\xA1\xED\x68\x4C\xDE\x92\x17\x4B\xBC\xAB\xFF\x65\x9A\x9E\xFB\x47\xD9\x57\x72\xF3\x09\xA1\xAE\x76\x44\x13\x6E\x9C\x2D\x44\x39\xBC\xF9\xC7\x3B\xA4\x58\x3D\x41\xBD\xB4\xC2\x49\xA3\xC8\x0D\xD2\x97\x2F\x07\x65\x52\x00\xA7\x6E\xC8\xAF\x68\xEC\xF4\x14\x96\xB6\x57\x1F\x56\xC3\x39\x9F\x2B\x6D\xE4\xF3\x3E\xF6\x35\x64\xDA\x0C\x1C\xA1\x84\x4B\x2F\x4B\x4B\xE2\x2C\x24\x9D\x6D\x93\x40\xEB\xB5\x23\x8E\x32\xCA\x6F\x45\xD3\xA8\x89\x7B\x1E\xCF\x1E\xFA\x5B\x43\x8B\xCD\xCD\xA8\x0F\x6A\xCA\x0C\x5E\xB9\x9E\x47\x8F\xF0\xD9\xB6\x0A\x0B\x58\x65\x17\x33\xB9\x23\xE4\x77\x19\x7D\xCB\x4A\x2E\x92\x7B\x4F\x2F\x10\x77\xB1\x8D\x2F\x68\x9C\x62\xCC\xE0\x50\xF8\xEC\x91\xA7\x54\x4C\x57\x09\xD5\x76\x63\xC5\xE8\x65\x1E\xEE\x6D\x6A\xCF\x09\x9D\xFA\x7C\x4F\xAD\x60\x08\xFD\x56\x99\x0F\x15\x2C\x7B\xA9\x80\xAB\x8C\x61\x8F\x4A\x07\x76\x42\xDE\x3D\xF4\xDD\xB2\x24\x33\x5B\xB8\xB5\xA3\x44\xC9\xAC\x7F\x77\x3C\x1D\x23\xEC\x82\xA9\xA6\xE2\xC8\x06\x4C\x02\xFE\xAC\x5C\x99\x99\x0B\x2F\x10\x8A\xA6\xF4\x7F\xD5\x87\x74\x0D\x59\x49\x45\xF6\xF0\x71\x5C\x39\x29\xD6\xBF\x4A\x23\x8B\xF5\x5F\x01\x63\xD2\x87\x73\x28\xB5\x4B\x0A\xF5\xF8\xAB\x82\x2C\x7E\x73\x25\x32\x1D\x0B\x63\x0A\x17\x81\x00\xFF\xB6\x76\x5E\xE7\xB4\xB1\x40\xCA\x21\xBB\xD5\x80\x51\xE5\x48\x52\x67\x2C\xD2\x61\x89\x07\x0D\x0F\xCE\x42\x77\xC0\x44\x73\x9C\x44\x50\xA0\xDB\x10\x0A\x2D\x95\x1C\x81\xAF\xE4\x1C\xE5\x14\x1E\xF1\x36\x41\x01\x02\x2F\x7D\x73\xA7\xDE\x42\xCC\x4C\xE9\x89\x0D\x56\xF7\x9F\x91\xD4\x03\xC6\x6C\xC9\x8F\xDB\xD8\x1C\xE0\x40\x98\x5D\x66\x99\x98\x80\x6E\x2D\xFF\x01\xC5\xCE\xCB\x46\x1F\xAC\x02\xC6\x43\xE6\xAE\xA2\x84\x3C\xC5\x4E\x1E\x3D\x6D\xC9\x14\x4C\xE3\x2E\x41\xBB\xCA\x39\xBF\x36\x3C\x2A\x19\xAA\x41\x87\x4E\xA5\xCE\x4B\x32\x79\xDD\x90\x49\x7F\x02\x03\x01\x00\x01\xA3\x63\x30\x61\x30\x1F\x06\x03\x55\x1D\x23\x04\x18\x30\x16\x80\x14\x9C\x5F\xD0\x6C\x63\xA3\x5F\x93\xCA\x93\x98\x08\xAD\x8C\x87\xA5\x2C\x5C\xC1\x37\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x9C\x5F\xD0\x6C\x63\xA3\x5F\x93\xCA\x93\x98\x08\xAD\x8C\x87\xA5\x2C\x5C\xC1\x37\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x86\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x02\x01\x00\x4E\x1E\xB9\x8A\xC6\xA0\x98\x3F\x6E\xC3\x69\xC0\x6A\x5C\x49\x52\xAC\xCB\x2B\x5D\x78\x38\xC1\xD5\x54\x84\x9F\x93\xF0\x87\x19\x3D\x2C\x66\x89\xEB\x0D\x42\xFC\xCC\xF0\x75\x85\x3F\x8B\xF4\x80\x5D\x79\xE5\x17\x67\xBD\x35\x82\xE2\xF2\x3C\x8E\x7D\x5B\x36\xCB\x5A\x80\x00\x29\xF2\xCE\x2B\x2C\xF1\x8F\xAA\x6D\x05\x93\x6C\x72\xC7\x56\xEB\xDF\x50\x23\x28\xE5\x45\x10\x3D\xE8\x67\xA3\xAF\x0E\x55\x0F\x90\x09\x62\xEF\x4B\x59\xA2\xF6\x53\xF1\xC0\x35\xE4\x2F\xC1\x24\xBD\x79\x2F\x4E\x20\x22\x3B\xFD\x1A\x20\xB0\xA4\x0E\x2C\x70\xED\x74\x3F\xB8\x13\x95\x06\x51\xC8\xE8\x87\x26\xCA\xA4\x5B\x6A\x16\x21\x92\xDD\x73\x60\x9E\x10\x18\xDE\x3C\x81\xEA\xE8\x18\xC3\x7C\x89\xF2\x8B\x50\x3E\xBD\x11\xE2\x15\x03\xA8\x36\x7D\x33\x01\x6C\x48\x15\xD7\x88\x90\x99\x04\xC5\xCC\xE6\x07\xF4\xBC\xF4\x90\xED\x13\xE2\xEA\x8B\xC3\x8F\xA3\x33\x0F\xC1\x29\x4C\x13\x4E\xDA\x15\x56\x71\x73\x72\x82\x50\xF6\x9A\x33\x7C\xA2\xB1\xA8\x1A\x34\x74\x65\x5C\xCE\xD1\xEB\xAB\x53\xE0\x1A\x80\xD8\xEA\x3A\x49\xE4\x26\x30\x9B\xE5\x1C\x8A\xA8\xA9\x15\x32\x86\x99\x92\x0A\x10\x23\x56\x12\xE0\xF6\xCE\x4C\xE2\xBB\xBE\xDB\x8D\x92\x73\x01\x66\x2F\x62\x3E\xB2\x72\x27\x45\x36\xED\x4D\x56\xE3\x97\x99\xFF\x3A\x35\x3E\xA5\x54\x4A\x52\x59\x4B\x60\xDB\xEE\xFE\x78\x11\x7F\x4A\xDC\x14\x79\x60\xB6\x6B\x64\x03\xDB\x15\x83\xE1\xA2\xBE\xF6\x23\x97\x50\xF0\x09\x33\x36\xA7\x71\x96\x25\xF3\xB9\x42\x7D\xDB\x38\x3F\x2C\x58\xAC\xE8\x42\xE1\x0E\xD8\xD3\x3B\x4C\x2E\x82\xE9\x83\x2E\x6B\x31\xD9\xDD\x47\x86\x4F\x6D\x97\x91\x2E\x4F\xE2\x28\x71\x35\x16\xD1\xF2\x73\xFE\x25\x2B\x07\x47\x24\x63\x27\xC8\xF8\xF6\xD9\x6B\xFC\x12\x31\x56\x08\xC0\x53\x42\xAF\x9C\xD0\x33\x7E\xFC\x06\xF0\x31\x44\x03\x14\xF1\x58\xEA\xF2\x6A\x0D\xA9\x11\xB2\x83\xBE\xC5\x1A\xBF\x07\xEA\x59\xDC\xA3\x88\x35\xEF\x9C\x76\x32\x3C\x4D\x06\x22\xCE\x15\xE5\xDD\x9E\xD8\x8F\xDA\xDE\xD2\xC4\x39\xE5\x17\x81\xCF\x38\x47\xEB\x7F\x88\x6D\x59\x1B\xDF\x9F\x42\x14\xAE\x7E\xCF\xA8\xB0\x66\x65\xDA\x37\xAF\x9F\xAA\x3D\xEA\x28\xB6\xDE\xD5\x31\x58\x16\x82\x5B\xEA\xBB\x19\x75\x02\x73\x1A\xCA\x48\x1A\x21\x93\x90\x0A\x8E\x93\x84\xA7\x7D\x3B\x23\x18\x92\x89\xA0\x8D\xAC"),
   // "T-TeleSec GlobalRoot Class 3"
   CertificateDer::from_slice(b"\x30\x82\x03\xC3\x30\x82\x02\xAB\xA0\x03\x02\x01\x02\x02\x01\x01\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x2B\x30\x29\x06\x03\x55\x04\x0A\x0C\x22\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x45\x6E\x74\x65\x72\x70\x72\x69\x73\x65\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x47\x6D\x62\x48\x31\x1F\x30\x1D\x06\x03\x55\x04\x0B\x0C\x16\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x54\x72\x75\x73\x74\x20\x43\x65\x6E\x74\x65\x72\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x54\x2D\x54\x65\x6C\x65\x53\x65\x63\x20\x47\x6C\x6F\x62\x61\x6C\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x33\x30\x1E\x17\x0D\x30\x38\x31\x30\x30\x31\x31\x30\x32\x39\x35\x36\x5A\x17\x0D\x33\x33\x31\x30\x30\x31\x32\x33\x35\x39\x35\x39\x5A\x30\x81\x82\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x44\x45\x31\x2B\x30\x29\x06\x03\x55\x04\x0A\x0C\x22\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x45\x6E\x74\x65\x72\x70\x72\x69\x73\x65\x20\x53\x65\x72\x76\x69\x63\x65\x73\x20\x47\x6D\x62\x48\x31\x1F\x30\x1D\x06\x03\x55\x04\x0B\x0C\x16\x54\x2D\x53\x79\x73\x74\x65\x6D\x73\x20\x54\x72\x75\x73\x74\x20\x43\x65\x6E\x74\x65\x72\x31\x25\x30\x23\x06\x03\x55\x04\x03\x0C\x1C\x54\x2D\x54\x65\x6C\x65\x53\x65\x63\x20\x47\x6C\x6F\x62\x61\x6C\x52\x6F\x6F\x74\x20\x43\x6C\x61\x73\x73\x20\x33\x30\x82\x01\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x01\x0F\x00\x30\x82\x01\x0A\x02\x82\x01\x01\x00\xBD\x75\x93\xF0\x62\x22\x6F\x24\xAE\xE0\x7A\x76\xAC\x7D\xBD\xD9\x24\xD5\xB8\xB7\xFC\xCD\xF0\x42\xE0\xEB\x78\x88\x56\x5E\x9B\x9A\x54\x1D\x4D\x0C\x8A\xF6\xD3\xCF\x70\xF4\x52\xB5\xD8\x93\x04\xE3\x46\x86\x71\x41\x4A\x2B\xF0\x2A\x2C\x55\x03\xD6\x48\xC3\xE0\x39\x38\xED\xF2\x5C\x3C\x3F\x44\xBC\x93\x3D\x61\xAB\x4E\xCD\x0D\xBE\xF0\x20\x27\x58\x0E\x44\x7F\x04\x1A\x87\xA5\xD7\x96\x14\x36\x90\xD0\x49\x7B\xA1\x75\xFB\x1A\x6B\x73\xB1\xF8\xCE\xA9\x09\x2C\xF2\x53\xD5\xC3\x14\x44\xB8\x86\xA5\xF6\x8B\x2B\x39\xDA\xA3\x33\x54\xD9\xFA\x72\x1A\xF7\x22\x15\x1C\x88\x91\x6B\x7F\x66\xE5\xC3\x6A\x80\xB0\x24\xF3\xDF\x86\x45\x88\xFD\x19\x7F\x75\x87\x1F\x1F\xB1\x1B\x0A\x73\x24\x5B\xB9\x65\xE0\x2C\x54\xC8\x60\xD3\x66\x17\x3F\xE1\xCC\x54\x33\x73\x91\x02\x3A\xA6\x7F\x7B\x76\x39\xA2\x1F\x96\xB6\x38\xAE\xB5\xC8\x93\x74\x1D\x9E\xB9\xB4\xE5\x60\x9D\x2F\x56\xD1\xE0\xEB\x5E\x5B\x4C\x12\x70\x0C\x6C\x44\x20\xAB\x11\xD8\xF4\x19\xF6\xD2\x9C\x52\x37\xE7\xFA\xB6\xC2\x31\x3B\x4A\xD4\x14\x99\xAD\xC7\x1A\xF5\x5D\x5F\xFA\x07\xB8\x7C\x0D\x1F\xD6\x83\x1E\xB3\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\xB5\x03\xF7\x76\x3B\x61\x82\x6A\x12\xAA\x18\x53\xEB\x03\x21\x94\xBF\xFE\xCE\xCA\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0B\x05\x00\x03\x82\x01\x01\x00\x56\x3D\xEF\x94\xD5\xBD\xDA\x73\xB2\x58\xBE\xAE\x90\xAD\x98\x27\x97\xFE\x01\xB1\xB0\x52\x00\xB8\x4D\xE4\x1B\x21\x74\x1B\x7E\xC0\xEE\x5E\x69\x2A\x25\xAF\x5C\xD6\x1D\xDA\xD2\x79\xC9\xF3\x97\x29\xE0\x86\x87\xDE\x04\x59\x0F\xF1\x59\xD4\x64\x85\x4B\x99\xAF\x25\x04\x1E\xC9\x46\xA9\x97\xDE\x82\xB2\x1B\x70\x9F\x9C\xF6\xAF\x71\x31\xDD\x7B\x05\xA5\x2C\xD3\xB9\xCA\x47\xF6\xCA\xF2\xF6\xE7\xAD\xB9\x48\x3F\xBC\x16\xB7\xC1\x6D\xF4\xEA\x09\xAF\xEC\xF3\xB5\xE7\x05\x9E\xA6\x1E\x8A\x53\x51\xD6\x93\x81\xCC\x74\x93\xF6\xB9\xDA\xA6\x25\x05\x74\x79\x5A\x7E\x40\x3E\x82\x4B\x26\x11\x30\x6E\xE1\x3F\x41\xC7\x47\x00\x35\xD5\xF5\xD3\xF7\x54\x3E\x81\x3D\xDA\x49\x6A\x9A\xB3\xEF\x10\x3D\xE6\xEB\x6F\xD1\xC8\x22\x47\xCB\xCC\xCF\x01\x31\x92\xD9\x18\xE3\x22\xBE\x09\x1E\x1A\x3E\x5A\xB2\xE4\x6B\x0C\x54\x7A\x7D\x43\x4E\xB8\x89\xA5\x7B\xD7\xA2\x3D\x96\x86\xCC\xF2\x26\x34\x2D\x6A\x92\x9D\x9A\x1A\xD0\x30\xE2\x5D\x4E\x04\xB0\x5F\x8B\x20\x7E\x77\xC1\x3D\x95\x82\xD1\x46\x9A\x3B\x3C\x78\xB8\x6F\xA1\xD0\x0D\x64\xA2\x78\x1E\x29\x4E\x93\xC3\xA4\x54\x14\x5B"),
   // "Certum Trusted Root CA"
   CertificateDer::from_slice(b"\x30\x82\x05\xC0\x30\x82\x03\xA8\xA0\x03\x02\x01\x02\x02\x10\x1E\xBF\x59\x50\xB8\xC9\x80\x37\x4C\x06\xF7\xEB\x55\x4F\xB5\xED\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x30\x7A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x41\x73\x73\x65\x63\x6F\x20\x44\x61\x74\x61\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x13\x16\x43\x65\x72\x74\x75\x6D\x20\x54\x72\x75\x73\x74\x65\x64\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x1E\x17\x0D\x31\x38\x30\x33\x31\x36\x31\x32\x31\x30\x31\x33\x5A\x17\x0D\x34\x33\x30\x33\x31\x36\x31\x32\x31\x30\x31\x33\x5A\x30\x7A\x31\x0B\x30\x09\x06\x03\x55\x04\x06\x13\x02\x50\x4C\x31\x21\x30\x1F\x06\x03\x55\x04\x0A\x13\x18\x41\x73\x73\x65\x63\x6F\x20\x44\x61\x74\x61\x20\x53\x79\x73\x74\x65\x6D\x73\x20\x53\x2E\x41\x2E\x31\x27\x30\x25\x06\x03\x55\x04\x0B\x13\x1E\x43\x65\x72\x74\x75\x6D\x20\x43\x65\x72\x74\x69\x66\x69\x63\x61\x74\x69\x6F\x6E\x20\x41\x75\x74\x68\x6F\x72\x69\x74\x79\x31\x1F\x30\x1D\x06\x03\x55\x04\x03\x13\x16\x43\x65\x72\x74\x75\x6D\x20\x54\x72\x75\x73\x74\x65\x64\x20\x52\x6F\x6F\x74\x20\x43\x41\x30\x82\x02\x22\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x01\x05\x00\x03\x82\x02\x0F\x00\x30\x82\x02\x0A\x02\x82\x02\x01\x00\xD1\x2D\x8E\xBB\xB7\x36\xEA\x6D\x37\x91\x9F\x4E\x93\xA7\x05\xE4\x29\x03\x25\xCE\x1C\x82\xF7\x7C\x99\x9F\x41\x06\xCD\xED\xA3\xBA\xC0\xDB\x09\x2C\xC1\x7C\xDF\x29\x7E\x4B\x65\x2F\x93\xA7\xD4\x01\x6B\x03\x28\x18\xA3\xD8\x9D\x05\xC1\x2A\xD8\x45\xF1\x91\xDE\xDF\x3B\xD0\x80\x02\x8C\xCF\x38\x0F\xEA\xA7\x5C\x78\x11\xA4\xC1\xC8\x85\x5C\x25\xD3\xD3\xB2\xE7\x25\xCF\x11\x54\x97\xAB\x35\xC0\x1E\x76\x1C\xEF\x00\x53\x9F\x39\xDC\x14\xA5\x2C\x22\x25\xB3\x72\x72\xFC\x8D\xB3\xE5\x3E\x08\x1E\x14\x2A\x37\x0B\x88\x3C\xCA\xB0\xF4\xC8\xC2\xA1\xAE\xBC\xC1\xBE\x29\x67\x55\xE2\xFC\xAD\x59\x5C\xFE\xBD\x57\x2C\xB0\x90\x8D\xC2\xED\x37\xB6\x7C\x99\x88\xB5\xD5\x03\x9A\x3D\x15\x0D\x3D\x3A\xA8\xA8\x45\xF0\x95\x4E\x25\x59\x1D\xCD\x98\x69\xBB\xD3\xCC\x32\xC9\x8D\xEF\x81\xFE\xAD\x7D\x89\xBB\xBA\x60\x13\xCA\x65\x95\x67\xA0\xF3\x19\xF6\x03\x56\xD4\x6A\xD3\x27\xE2\xA1\xAD\x83\xF0\x4A\x12\x22\x77\x1C\x05\x73\xE2\x19\x71\x42\xC0\xEC\x75\x46\x9A\x90\x58\xE0\x6A\x8E\x2B\xA5\x46\x30\x04\x8E\x19\xB2\x17\xE3\xBE\xA9\xBA\x7F\x56\xF1\x24\x03\xD7\xB2\x21\x28\x76\x0E\x36\x30\x4C\x79\xD5\x41\x9A\x9A\xA8\xB8\x35\xBA\x0C\x3A\xF2\x44\x1B\x20\x88\xF7\xC5\x25\xD7\x3D\xC6\xE3\x3E\x43\xDD\x87\xFE\xC4\xEA\xF5\x53\x3E\x4C\x65\xFF\x3B\x4A\xCB\x78\x5A\x6B\x17\x5F\x0D\xC7\xC3\x4F\x4E\x9A\x2A\xA2\xED\x57\x4D\x22\xE2\x46\x9A\x3F\x0F\x91\x34\x24\x7D\x55\xE3\x8C\x95\x37\xD3\x1A\xF0\x09\x2B\x2C\xD2\xC9\x8D\xB4\x0D\x00\xAB\x67\x29\x28\xD8\x01\xF5\x19\x04\xB6\x1D\xBE\x76\xFE\x72\x5C\xC4\x85\xCA\xD2\x80\x41\xDF\x05\xA8\xA3\xD5\x84\x90\x4F\x0B\xF3\xE0\x3F\x9B\x19\xD2\x37\x89\x3F\xF2\x7B\x52\x1C\x8C\xF6\xE1\xF7\x3C\x07\x97\x8C\x0E\xA2\x59\x81\x0C\xB2\x90\x3D\xD3\xE3\x59\x46\xED\x0F\xA9\xA7\xDE\x80\x6B\x5A\xAA\x07\xB6\x19\xCB\xBC\x57\xF3\x97\x21\x7A\x0C\xB1\x2B\x74\x3E\xEB\xDA\xA7\x67\x2D\x4C\xC4\x98\x9E\x36\x09\x76\x66\x66\xFC\x1A\x3F\xEA\x48\x54\x1C\xBE\x30\xBD\x80\x50\xBF\x7C\xB5\xCE\x00\xF6\x0C\x61\xD9\xE7\x24\x03\xE0\xE3\x01\x81\x0E\xBD\xD8\x85\x34\x88\xBD\xB2\x36\xA8\x7B\x5C\x08\xE5\x44\x80\x8C\x6F\xF8\x2F\xD5\x21\xCA\x1D\x1C\xD0\xFB\xC4\xB5\x87\xD1\x3A\x4E\xC7\x76\xB5\x35\x48\xB5\x02\x03\x01\x00\x01\xA3\x42\x30\x40\x30\x0F\x06\x03\x55\x1D\x13\x01\x01\xFF\x04\x05\x30\x03\x01\x01\xFF\x30\x1D\x06\x03\x55\x1D\x0E\x04\x16\x04\x14\x8C\xFB\x1C\x75\xBC\x02\xD3\x9F\x4E\x2E\x48\xD9\xF9\x60\x54\xAA\xC4\xB3\x4F\xFA\x30\x0E\x06\x03\x55\x1D\x0F\x01\x01\xFF\x04\x04\x03\x02\x01\x06\x30\x0D\x06\x09\x2A\x86\x48\x86\xF7\x0D\x01\x01\x0D\x05\x00\x03\x82\x02\x01\x00\x48\xA2\xD5\x00\x0B\x2E\xD0\x3F\xBC\x1C\xD5\xB5\x54\x49\x1E\x5A\x6B\xF4\xE4\xF2\xE0\x40\x37\xE0\xCC\x14\x7B\xB9\xC9\xFA\x35\xB5\x75\x17\x93\x6A\x05\x69\x85\x9C\xCD\x4F\x19\x78\x5B\x19\x81\xF3\x63\x3E\xC3\xCE\x5B\x8F\xF5\x2F\x5E\x01\x76\x13\x3F\x2C\x00\xB9\xCD\x96\x52\x39\x49\x6D\x04\x4E\xC5\xE9\x0F\x86\x0D\xE1\xFA\xB3\x5F\x82\x12\xF1\x3A\xCE\x66\x06\x24\x34\x2B\xE8\xCC\xCA\xE7\x69\xDC\x87\x9D\xC2\x34\xD7\x79\xD1\xD3\x77\xB8\xAA\x59\x58\xFE\x9D\x26\xFA\x38\x86\x3E\x9D\x8A\x87\x64\x57\xE5\x17\x3A\xE2\xF9\x8D\xB9\xE3\x33\x78\xC1\x90\xD8\xB8\xDD\xB7\x83\x51\xE4\xC4\xCC\x23\xD5\x06\x7C\xE6\x51\xD3\xCD\x34\x31\xC0\xF6\x46\xBB\x0B\xAD\xFC\x3D\x10\x05\x2A\x3B\x4A\x91\x25\xEE\x8C\xD4\x84\x87\x80\x2A\xBC\x09\x8C\xAA\x3A\x13\x5F\xE8\x34\x79\x50\xC1\x10\x19\xF9\xD3\x28\x1E\xD4\xD1\x51\x30\x29\xB3\xAE\x90\x67\xD6\x1F\x0A\x63\xB1\xC5\xA9\xC6\x42\x31\x63\x17\x94\xEF\x69\xCB\x2F\xFA\x8C\x14\x7D\xC4\x43\x18\x89\xD9\xF0\x32\x40\xE6\x80\xE2\x46\x5F\xE5\xE3\xC1\x00\x59\xA8\xF9\xE8\x20\xBC\x89\x2C\x0E\x47\x34\x0B\xEA\x57\xC2\x53\x36\xFC\xA7\xD4\xAF\x31\xCD\xFE\x02\xE5\x75\xFA\xB9\x27\x09\xF9\xF3\xF5\x3B\xCA\x7D\x9F\xA9\x22\xCB\x88\xC9\xAA\xD1\x47\x3D\x36\x77\xA8\x59\x64\x6B\x27\xCF\xEF\x27\xC1\xE3\x24\xB5\x86\xF7\xAE\x7E\x32\x4D\xB0\x79\x68\xD1\x39\xE8\x90\x58\xC3\x83\xBC\x0F\x2C\xD6\x97\xEB\xCE\x0C\xE1\x20\xC7\xDA\xB7\x3E\xC3\x3F\xBF\x2F\xDC\x34\xA4\xFB\x2B\x21\xCD\x67\x8F\x4B\xF4\xE3\xEA\xD4\x3F\xE7\x4F\xBA\xB9\xA5\x93\x45\x1C\x66\x1F\x21\xFA\x64\x5E\x6F\xE0\x76\x94\x32\xCB\x75\xF5\x6E\xE5\xF6\x8F\xC7\xB8\xA4\xCC\xA8\x96\x7D\x64\xFB\x24\x5A\x4A\x03\x6C\x6B\x38\xC6\xE8\x03\x43\x9A\xF7\x57\xB9\xB3\x29\x69\x93\x38\xF4\x03\xF2\xBB\xFB\x82\x6B\x07\x20\xD1\x52\x1F\x9A\x64\x02\x7B\x98\x66\xDB\x5C\x4D\x5A\x0F\xD0\x84\x95\xA0\x3C\x14\x43\x06\xCA\xCA\xDB\xB8\x41\x36\xDA\x6A\x44\x67\x87\xAF\xAF\xE3\x45\x11\x15\x69\x08\xB2\xBE\x16\x39\x97\x24\x6F\x12\x45\xD1\x67\x5D\x09\xA8\xC9\x15\xDA\xFA\xD2\xA6\x5F\x13\x61\x1F\xBF\x85\xAC\xB4\xAD\xAD\x05\x94\x08\x83\x1E\x75\x17\xD3\x71\x3B\x93\x50\x23\x59\xA0\xED\x3C\x91\x54\x9D\x76\x00\xC5\xC3\xB8\x38\xDB"),
];
