#include "message.h"

namespace ADNS {

	Message::Message()
	{
		mh = gcnew MessageHeader();
		Questions = gcnew List<QuestionHeader^>(0);
		Answer = gcnew RRSet();
		Authority = gcnew RRSet();
		Additional = gcnew RRSet();
		RegisteredTypes = gcnew List<RR_TYPE>() ;
		RegisteredRRParsers = gcnew List<GenRRDelegate^>();
		RegisteredRRPrinters = gcnew List<PrintRRDelegate^>();
		RegisteredRRCloners = gcnew List<CloneRRDelegate^>();
		RegisterResourceRecords();
	}

	Message::Message(array<Byte>^ packet)
	{
		UInt16 qdcount;
		UInt16 ancount;
		UInt16 nscount;
		UInt16 arcount;
		int i = 0;
		int pos = 0;
		int l = 0;
		int start = 0;
		int end = 0;
		UInt16 tmptype;
		UInt16 tmpclass;
		UInt32 tmpttl;
		UInt16 rdata_len;


		QuestionHeader^ qh;
		ResourceRecord^ rr;
		array<Byte>^ tmparray;

		Questions = gcnew List<QuestionHeader^>(0);
		Answer = gcnew RRSet();
		Authority = gcnew RRSet();
		Additional = gcnew RRSet();

		// Register all supported record types
		RegisteredTypes = gcnew List<RR_TYPE>() ;
		RegisteredRRParsers = gcnew List<GenRRDelegate^>();
		RegisteredRRPrinters = gcnew List<PrintRRDelegate^>();
		RegisteredRRCloners = gcnew List<CloneRRDelegate^>();
		RegisterResourceRecords();


		//parse packet header information
		mh = gcnew MessageHeader(packet);
		qdcount = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,4));
		ancount = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,6));
		nscount = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,8));
		arcount = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,10));

		//parse packet question section

		pos = 12;
		for (i = 0; i < qdcount; ++i)
		{
			qh = gcnew QuestionHeader();
			tmparray = ReadDomainFromPacket(packet,pos,l);
			if (l == -1)
			{
				throw gcnew Exception("Error:  unbounded domain name discovered at position " + Convert::ToString(pos) + " of incoming packet.");
				return;
			}
			qh->SetQuestionName(gcnew DOMAIN_NAME(tmparray));
			pos += l;
			tmptype = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;
			qh->SetQuestionClass((RR_CLASS) IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos)));
			pos += 2;
			Questions->Add(qh);
		}

		//parse packet answer section
		for (i = 0;i < ancount; ++i)
		{
			//We should be pointing at the start of a domain name
			tmparray = ReadDomainFromPacket(packet,pos,l);
			if ((l == -1) || (tmparray == nullptr))
			{
				throw gcnew Exception("Error:  unbounded domain name discovered at position " + Convert::ToString(pos) + " of incoming packet.");
				return;
			}
			pos += l;
			tmptype = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			if (!RRIsSupported(tmptype))
			{
				throw gcnew Exception("An error occurred parsing an incoming packet.  An Unsupported RR_TYPE was encounted.  Type Found was:  " + Convert::ToString(tmptype) + " at position " + Convert::ToString(pos) + " with " + Convert::ToString(ancount) + " answers found and processing #" + Convert::ToString(i) + " after processing " + Convert::ToString(qdcount) + " questions.");
				return;
				
			}
			pos +=2;
			tmpclass = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;
			tmpttl = IPAddress::NetworkToHostOrder((int) BitConverter::ToUInt32(packet,pos));
			pos += 4;
			rdata_len = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;

			//now, send all this info to a big'ol'parser function to get the right type of resource record.
			rr = ParseResourceRecord(tmparray,tmptype,tmpclass,tmpttl,rdata_len,packet,pos);

			Answer->Add(rr);
			pos += rdata_len;		
		}

		//parse packet authority section
		for (i = 0;i < nscount; ++i)
		{
			//We should be pointing at the start of a domain name
			tmparray = ReadDomainFromPacket(packet,pos,l);
			if (l == -1)
			{
				throw gcnew Exception("Error:  unbounded domain name discovered at position " + Convert::ToString(pos) + " of incoming packet.");
				return;
			}
			pos += l;
			tmptype = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			if (!RRIsSupported(tmptype))
			{
				throw gcnew Exception("An error occurred parsing an incoming packet.  An Unsupported RR_TYPE was encounted.  Type Found was:  " + Convert::ToString(tmptype) + " at position " + Convert::ToString(pos));
				return; //silently keep going.
				
			}
			pos +=2;
			tmpclass = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;
			tmpttl = IPAddress::NetworkToHostOrder((int) BitConverter::ToUInt32(packet,pos));
			pos += 4;
			rdata_len = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;

			//now, send all this info to a big'ol'parser function to get the right type of resource record.
			rr = ParseResourceRecord(tmparray,tmptype,tmpclass,tmpttl,rdata_len,packet,pos);

			Authority->Add(rr);
			pos += rdata_len;		
		}

		//parse packet additional section
		for (i = 0;i < arcount; ++i)
		{
			//We should be pointing at the start of a domain name
			tmparray = ReadDomainFromPacket(packet,pos,l);
			if (l == -1)
			{
				throw gcnew Exception("Error:  unbounded domain name discovered at position " + Convert::ToString(pos) + " of incoming packet.");
				return;
			}
			pos += l;
			tmptype = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			if (!RRIsSupported(tmptype))
			{
				throw gcnew Exception("An error occurred parsing an incoming packet.  An Unsupported RR_TYPE was encounted.  Type Found was:  " + Convert::ToString(tmptype) + " at position " + Convert::ToString(pos));
				return;
				
			}
			pos +=2;
			tmpclass = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;
			tmpttl = IPAddress::NetworkToHostOrder((int) BitConverter::ToUInt32(packet,pos));
			pos += 4;
			rdata_len = IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,pos));
			pos += 2;

			//now, send all this info to a big'ol'parser function to get the right type of resource record.
			rr = ParseResourceRecord(tmparray,tmptype,tmpclass,tmpttl,rdata_len,packet,pos);

			Additional->Add(rr);
			pos += rdata_len;		
		}
				

	}

	array<Byte>^ Message::ToWire()
	{
		array<Byte>^ output;
		int i;
		int pos;

		array<Byte>^ tmp;

		output = gcnew array<Byte>(12);
		
		//put all header information in.

		mh->ToWire()->CopyTo(output,0);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) Questions->Count))->CopyTo(output,4);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) Answer->Length()))->CopyTo(output,6);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) Authority->Length()))->CopyTo(output,8);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) Additional->Length()))->CopyTo(output,10);
		
		//put all questions in
		for (i = 0; i < Questions->Count; i++)
		{
			tmp = Questions[i]->ToWire();
			pos = output->Length;
			output->Resize(output,output->Length + tmp->Length);
			tmp->CopyTo(output,pos);

		}

		//put all answers in
		for (i = 0; i < Answer->Length(); i++)
		{
			tmp = Answer->set[i]->ToWire();
			pos = output->Length;
			output->Resize(output,output->Length + tmp->Length);
			tmp->CopyTo(output,pos);

		}		

		//put all authority records in
		for (i = 0; i < Authority->Length(); i++)
		{
			tmp = Authority->set[i]->ToWire();
			pos = output->Length;
			output->Resize(output,output->Length + tmp->Length);
			tmp->CopyTo(output,pos);

		}

		//put all additional records in
		for (i = 0; i < Additional->Length(); i++)
		{
			if (Additional->set[i]->rr_type != RR_TYPE::OPT)
			{
				tmp = Additional->set[i]->ToWire();
				pos = output->Length;
				output->Resize(output,output->Length + tmp->Length);
				tmp->CopyTo(output,pos);
			}
			else
			{
				tmp = safe_cast<OPT_RR^>(Additional->set[i])->ToWire();
				pos = output->Length;
				output->Resize(output,output->Length + tmp->Length);
				tmp->CopyTo(output,pos);
			}
		}

		return output;
	}

	ResourceRecord^ Message::ParseResourceRecord(array<Byte>^ domainname, UInt16 rr_type, UInt16 rr_class, UInt32 ttl, UInt16 rdata_len, array<Byte>^ packet, int rdata_start)
	{
		int ndx = -1;
		ResourceRecord^ output;
		ndx = RegisteredTypes->IndexOf((RR_TYPE) rr_type);
		if (ndx != -1)
		{
			output = RegisteredRRParsers[ndx](domainname, rr_type,  rr_class, ttl, rdata_len, packet, rdata_start);
		}
		else
		{
			throw gcnew Exception("Somehow a unsupported type got past the filter to GenerateResourceRecord!");
			output = nullptr;
		}

		return output;
	}

	ResourceRecord^ Message::CloneResourceRecord(ResourceRecord^ rr)
	{
		ResourceRecord^ output;
		int ndx = -1;

		if (rr == nullptr)
			return nullptr;

		ndx = RegisteredTypes->IndexOf(rr->rr_type);
		if (ndx != -1)
		{
			output = RegisteredRRCloners[ndx](rr);
		}
		else
		{
			throw gcnew Exception("Somehow a unsupported type got past the filter to GenerateResourceRecord!");
			output = nullptr;
		}

		return output;
	}

	String^ Message::PrintResourceRecord(ResourceRecord^ rr)
	{
		String^ output;
		int ndx = -1;
	
		if (rr == nullptr)
			return nullptr;
		
		ndx = RegisteredTypes->IndexOf(rr->rr_type);
		if (ndx != -1)
		{
			output = RegisteredRRPrinters[ndx](rr);
		}
		else
		{
			throw gcnew Exception("Somehow a unsupported type got past the filter to GenerateResourceRecord!");
			output = nullptr;
		}

		return output;
	}

	array<Byte>^ Message::SendMessageUDP(IPAddress^ DNSServer, int retries, int timeout)
	{
		// RFC1035 max. size of a UDP datagram is 512 bytes, but with OPT can get larger.

		array<Byte>^ response;
		int response_size;
		int i = 0;
		int attempts = 0;
		array<Byte>^ message;
		array<Byte>^ output;

		int received = 0;
		IPEndPoint^ destination = gcnew IPEndPoint(DNSServer,53);

		message = ToWire();

		//If there is an OPT record in the additional section, get the packet size from there
		//otherwise assume it's 512
		for (i = 0; i < Additional->Length(); ++i)
		{
			if (Additional->set[i]->rr_type == RR_TYPE::OPT)
				response_size = safe_cast<OPT_RR^>(Additional->set[i])->GetPayloadSize();
			else
				response_size = 512;
		}

		response = gcnew array<Byte>(response_size);

		for ( attempts = 0; attempts < retries; attempts++)
		{
			Socket^ socket = gcnew Socket(AddressFamily::InterNetwork, SocketType::Dgram, ProtocolType::Udp);
			socket->SetSocketOption(SocketOptionLevel::Socket, SocketOptionName::ReceiveTimeout, timeout);

			try
			{
				socket->SendTo(message, destination);
				received = socket->Receive(response);
				output = gcnew array<Byte>(received);
				response->Copy(response,0,output,0,received);
			}
			catch (SocketException^ ex)
			{
				continue; // next try
			}
			
			socket->Close();

			if (received > 0)
				break;

		}

		if (received <= 0)
			return nullptr;

		return response;
	}



	bool Message::RRIsSupported(UInt16 rr_type)
	{
		int ndx = -1;

		ndx = RegisteredTypes->IndexOf((RR_TYPE) rr_type);
		if (ndx != -1)
			return true;

		return false;
	}

	void Message::RegisterResourceRecords()
	{
		//Register A record
		RegisteredTypes->Add(RR_TYPE::A);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&A_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&A_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&A_RR::CloneRR));

		//Register AAAA record
		RegisteredTypes->Add(RR_TYPE::AAAA);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&AAAA_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&AAAA_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&AAAA_RR::CloneRR));

		//Register CNAME record
		RegisteredTypes->Add(RR_TYPE::CNAME);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&CNAME_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&CNAME_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&CNAME_RR::CloneRR));

		//Register DNSKEY record
		RegisteredTypes->Add(RR_TYPE::DNSKEY);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&DNSKEY_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&DNSKEY_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&DNSKEY_RR::CloneRR));

		//Register DS record
		RegisteredTypes->Add(RR_TYPE::DS);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&DS_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&DS_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&DS_RR::CloneRR));

		//Register HINFO record
		RegisteredTypes->Add(RR_TYPE::HINFO);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&HINFO_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&HINFO_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&HINFO_RR::CloneRR));

		//Register MB record
		RegisteredTypes->Add(RR_TYPE::MB);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&MB_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&MB_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&MB_RR::CloneRR));

		//Register MF record
		RegisteredTypes->Add(RR_TYPE::MF);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&MF_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&MF_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&MF_RR::CloneRR));

		//Register MG record
		RegisteredTypes->Add(RR_TYPE::MG);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&MG_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&MG_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&MG_RR::CloneRR));

		//Register MINFO record
		RegisteredTypes->Add(RR_TYPE::MINFO);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&MINFO_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&MINFO_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&MINFO_RR::CloneRR));

		//Register MR record
		RegisteredTypes->Add(RR_TYPE::MR);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&MR_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&MR_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&MR_RR::CloneRR));

		//Register MX record
		RegisteredTypes->Add(RR_TYPE::MX);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&MX_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&MX_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&MX_RR::CloneRR));

		//Register NS record
		RegisteredTypes->Add(RR_TYPE::NS);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&NS_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&NS_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&NS_RR::CloneRR));

		//Register NSEC record
		RegisteredTypes->Add(RR_TYPE::NSEC);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&NSEC_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&NSEC_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&NSEC_RR::CloneRR));

		//Register NSEC3 record
		RegisteredTypes->Add(RR_TYPE::NSEC3);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&NSEC3_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&NSEC3_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&NSEC3_RR::CloneRR));

		//Register NULL record
		RegisteredTypes->Add(RR_TYPE::NULL);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&NULL_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&NULL_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&NULL_RR::CloneRR));

		//Register OPT record
		RegisteredTypes->Add(RR_TYPE::OPT);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&OPT_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&OPT_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&OPT_RR::CloneRR));

		//Register PTR record
		RegisteredTypes->Add(RR_TYPE::PTR);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&PTR_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&PTR_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&PTR_RR::CloneRR));

		//Register RRSIG record
		RegisteredTypes->Add(RR_TYPE::RRSIG);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&RRSIG_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&RRSIG_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&RRSIG_RR::CloneRR));

		//Register SOA record
		RegisteredTypes->Add(RR_TYPE::SOA);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&SOA_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&SOA_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&SOA_RR::CloneRR));

		//Register TXT record
		RegisteredTypes->Add(RR_TYPE::TXT);
		RegisteredRRParsers->Add(gcnew GenRRDelegate(&TXT_RR::ParseResourceRecord));
		RegisteredRRPrinters->Add(gcnew PrintRRDelegate(&TXT_RR::PrintRR));
		RegisteredRRCloners->Add(gcnew CloneRRDelegate(&TXT_RR::CloneRR));

		return;
	}

}
