
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
/*
 * This file makes sure a prime given to us matches one of the known
 * approved primes for diffie-helman.
 *
 * It also checks if a prime is a safe prime for the case
 * where we don't match an approved prime.
 */
#include "seccomon.h"
#include "secitem.h"
#include "secerr.h"
#include "blapi.h"

/* IKE 1536 prime is: 2^1536 - 2^1472 - 1 + 2^64 * { [2^1406 pi] + 741804 } */
static const unsigned char prime_ike_1536[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2,
    0x21, 0x68, 0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
    0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE, 0xA6,
    0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
    0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D,
    0xF2, 0x5F, 0x14, 0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
    0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
    0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
    0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11,
    0x7C, 0x4B, 0x1F, 0xE6, 0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
    0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36,
    0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
    0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56,
    0x20, 0x85, 0x52, 0xBB, 0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
    0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04, 0xF1, 0x74, 0x6C, 0x08,
    0xCA, 0x23, 0x73, 0x27, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* IKE 2048 prime is: 2^2048 - 2^1984 - 1 + 2^64 * { [2^1918 pi] + 124476 */
static const unsigned char prime_ike_2048[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2,
    0x21, 0x68, 0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
    0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE, 0xA6,
    0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
    0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D,
    0xF2, 0x5F, 0x14, 0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
    0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
    0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
    0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11,
    0x7C, 0x4B, 0x1F, 0xE6, 0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
    0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36,
    0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
    0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56,
    0x20, 0x85, 0x52, 0xBB, 0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
    0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04, 0xF1, 0x74, 0x6C, 0x08,
    0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
    0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2,
    0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9,
    0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7C,
    0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
    0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAC, 0xAA, 0x68, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF
};

/* TLS 2048 prime is: 2^2048 - 2^1984 + {[2^1918 * e] + 560316 } * 2^64 - 1 */
static const unsigned char prime_tls_2048[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAD, 0xF8, 0x54, 0x58,
    0xA2, 0xBB, 0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 0x3D, 0x3C, 0xF1,
    0xD8, 0xB9, 0xC5, 0x83, 0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 0x41,
    0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9,
    0x7D, 0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 0xF6, 0x81, 0xB2, 0x02,
    0xAE, 0xC4, 0x61, 0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 0x65, 0x61,
    0x24, 0x33, 0xF5, 0x1F, 0x5F, 0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55,
    0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 0x5E, 0x7F, 0x57, 0xC9, 0x35,
    0x98, 0x4F, 0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 0xA6, 0x89, 0xDA,
    0xF3, 0xEF, 0xE8, 0x72, 0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 0x35,
    0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82,
    0xB3, 0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 0xB2, 0xC8, 0xE3, 0xFB,
    0xB9, 0x6A, 0xDA, 0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 0x42, 0xA3,
    0xDE, 0x39, 0x4D, 0xF4, 0xAE, 0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19,
    0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 0x70, 0x9E, 0x02, 0xFC, 0xE1,
    0xCD, 0xF7, 0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 0x34, 0x2F, 0x61,
    0x91, 0x72, 0xFE, 0x9C, 0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 0x32,
    0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73,
    0x3B, 0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 0xC5, 0x8E, 0xF1, 0x83,
    0x7D, 0x16, 0x83, 0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 0xEF, 0xFA,
    0x88, 0x6B, 0x42, 0x38, 0x61, 0x28, 0x5C, 0x97, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF
};

/* IKE 3072 prime is: 2^3072 - 2^3008 - 1 + 2^64 * { [2^2942 pi] + 1690314 } */
static const unsigned char prime_ike_3072[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2,
    0x21, 0x68, 0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
    0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE, 0xA6,
    0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
    0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D,
    0xF2, 0x5F, 0x14, 0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
    0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
    0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
    0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11,
    0x7C, 0x4B, 0x1F, 0xE6, 0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
    0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36,
    0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
    0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56,
    0x20, 0x85, 0x52, 0xBB, 0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
    0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04, 0xF1, 0x74, 0x6C, 0x08,
    0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
    0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2,
    0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9,
    0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7C,
    0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
    0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
    0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21, 0xAB, 0xDF, 0x1C, 0xBA, 0x64,
    0xEC, 0xFB, 0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A, 0xEA, 0x71, 0x57,
    0x5D, 0x06, 0x0C, 0x7D, 0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4, 0xC7,
    0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09, 0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0,
    0x4A, 0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26, 0x1A, 0xD2, 0xEE, 0x6B,
    0xF1, 0x2F, 0xFA, 0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76, 0x02, 0x73,
    0x3E, 0xC8, 0x6A, 0x64, 0x52, 0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
    0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D, 0x6C, 0x77, 0x09, 0x88, 0xC0,
    0xBA, 0xD9, 0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74, 0xE5, 0xAB, 0x31,
    0x43, 0xDB, 0x5B, 0xFC, 0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1, 0x20,
    0xA9, 0x3A, 0xD2, 0xCA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* TLS 3072 prime is: 2^3072 - 2^3008 + {[2^2942 * e] + 2625351} * 2^64 - 1 */
static const unsigned char prime_tls_3072[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAD, 0xF8, 0x54, 0x58,
    0xA2, 0xBB, 0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 0x3D, 0x3C, 0xF1,
    0xD8, 0xB9, 0xC5, 0x83, 0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 0x41,
    0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9,
    0x7D, 0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 0xF6, 0x81, 0xB2, 0x02,
    0xAE, 0xC4, 0x61, 0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 0x65, 0x61,
    0x24, 0x33, 0xF5, 0x1F, 0x5F, 0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55,
    0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 0x5E, 0x7F, 0x57, 0xC9, 0x35,
    0x98, 0x4F, 0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 0xA6, 0x89, 0xDA,
    0xF3, 0xEF, 0xE8, 0x72, 0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 0x35,
    0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82,
    0xB3, 0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 0xB2, 0xC8, 0xE3, 0xFB,
    0xB9, 0x6A, 0xDA, 0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 0x42, 0xA3,
    0xDE, 0x39, 0x4D, 0xF4, 0xAE, 0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19,
    0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 0x70, 0x9E, 0x02, 0xFC, 0xE1,
    0xCD, 0xF7, 0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 0x34, 0x2F, 0x61,
    0x91, 0x72, 0xFE, 0x9C, 0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 0x32,
    0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73,
    0x3B, 0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 0xC5, 0x8E, 0xF1, 0x83,
    0x7D, 0x16, 0x83, 0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 0xEF, 0xFA,
    0x88, 0x6B, 0x42, 0x38, 0x61, 0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B,
    0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 0xDE, 0xF9, 0x9C, 0x02, 0x38,
    0x61, 0xB4, 0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 0xD9, 0x1D, 0x26,
    0x91, 0xF7, 0xF7, 0xEE, 0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 0x1C,
    0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93,
    0xBC, 0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 0xE2, 0xD7, 0x4D, 0xD3,
    0x64, 0xF2, 0xE2, 0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 0x82, 0xAB,
    0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D,
    0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 0xDA, 0x1D, 0xBF, 0x9A, 0x42,
    0xD5, 0xC4, 0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 0x53, 0xDD, 0xEF,
    0x3C, 0x1B, 0x20, 0xEE, 0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 0x2B,
    0x66, 0xC6, 0x2E, 0x37, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* IKE 4096 prime is: 2^4096 - 2^4032 - 1 + 2^64 * { [2^3966 pi] + 240904 } */
static const unsigned char prime_ike_4096[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2,
    0x21, 0x68, 0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
    0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE, 0xA6,
    0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
    0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D,
    0xF2, 0x5F, 0x14, 0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
    0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
    0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
    0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11,
    0x7C, 0x4B, 0x1F, 0xE6, 0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
    0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36,
    0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
    0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56,
    0x20, 0x85, 0x52, 0xBB, 0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
    0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04, 0xF1, 0x74, 0x6C, 0x08,
    0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
    0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2,
    0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9,
    0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7C,
    0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
    0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
    0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21, 0xAB, 0xDF, 0x1C, 0xBA, 0x64,
    0xEC, 0xFB, 0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A, 0xEA, 0x71, 0x57,
    0x5D, 0x06, 0x0C, 0x7D, 0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4, 0xC7,
    0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09, 0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0,
    0x4A, 0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26, 0x1A, 0xD2, 0xEE, 0x6B,
    0xF1, 0x2F, 0xFA, 0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76, 0x02, 0x73,
    0x3E, 0xC8, 0x6A, 0x64, 0x52, 0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
    0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D, 0x6C, 0x77, 0x09, 0x88, 0xC0,
    0xBA, 0xD9, 0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74, 0xE5, 0xAB, 0x31,
    0x43, 0xDB, 0x5B, 0xFC, 0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1, 0x20,
    0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72, 0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7,
    0x88, 0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26, 0x99, 0xC3, 0x27, 0x18,
    0x6A, 0xF4, 0xE2, 0x3C, 0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15, 0x0B, 0xDA,
    0x25, 0x83, 0xE9, 0xCA, 0x2A, 0xD4, 0x4C, 0xE8, 0xDB, 0xBB, 0xC2, 0xDB,
    0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC, 0x14, 0x1F, 0xBE, 0xCA, 0xA6,
    0x28, 0x7C, 0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99, 0xB2, 0x96, 0x4F,
    0xA0, 0x90, 0xC3, 0xA2, 0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7, 0xED,
    0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2, 0xD7, 0xAF, 0xB8, 0x1B, 0xDD, 0x76,
    0x21, 0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27, 0xD5, 0xB0, 0x5A, 0xA9,
    0x93, 0xB4, 0xEA, 0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF, 0xB7, 0xDC,
    0x90, 0xA6, 0xC0, 0x8F, 0x4D, 0xF4, 0x35, 0xC9, 0x34, 0x06, 0x31, 0x99,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* TLS 4096 prime is: 2^4096 - 2^4032 + {[2^3966 * e] + 5736041} * 2^64 - 1 */
static const unsigned char prime_tls_4096[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAD, 0xF8, 0x54, 0x58,
    0xA2, 0xBB, 0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 0x3D, 0x3C, 0xF1,
    0xD8, 0xB9, 0xC5, 0x83, 0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 0x41,
    0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9,
    0x7D, 0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 0xF6, 0x81, 0xB2, 0x02,
    0xAE, 0xC4, 0x61, 0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 0x65, 0x61,
    0x24, 0x33, 0xF5, 0x1F, 0x5F, 0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55,
    0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 0x5E, 0x7F, 0x57, 0xC9, 0x35,
    0x98, 0x4F, 0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 0xA6, 0x89, 0xDA,
    0xF3, 0xEF, 0xE8, 0x72, 0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 0x35,
    0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82,
    0xB3, 0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 0xB2, 0xC8, 0xE3, 0xFB,
    0xB9, 0x6A, 0xDA, 0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 0x42, 0xA3,
    0xDE, 0x39, 0x4D, 0xF4, 0xAE, 0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19,
    0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 0x70, 0x9E, 0x02, 0xFC, 0xE1,
    0xCD, 0xF7, 0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 0x34, 0x2F, 0x61,
    0x91, 0x72, 0xFE, 0x9C, 0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 0x32,
    0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73,
    0x3B, 0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 0xC5, 0x8E, 0xF1, 0x83,
    0x7D, 0x16, 0x83, 0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 0xEF, 0xFA,
    0x88, 0x6B, 0x42, 0x38, 0x61, 0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B,
    0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 0xDE, 0xF9, 0x9C, 0x02, 0x38,
    0x61, 0xB4, 0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 0xD9, 0x1D, 0x26,
    0x91, 0xF7, 0xF7, 0xEE, 0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 0x1C,
    0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93,
    0xBC, 0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 0xE2, 0xD7, 0x4D, 0xD3,
    0x64, 0xF2, 0xE2, 0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 0x82, 0xAB,
    0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D,
    0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 0xDA, 0x1D, 0xBF, 0x9A, 0x42,
    0xD5, 0xC4, 0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 0x53, 0xDD, 0xEF,
    0x3C, 0x1B, 0x20, 0xEE, 0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 0x2B,
    0x66, 0x9E, 0x1E, 0xF1, 0x6E, 0x6F, 0x52, 0xC3, 0x16, 0x4D, 0xF4, 0xFB,
    0x79, 0x30, 0xE9, 0xE4, 0xE5, 0x88, 0x57, 0xB6, 0xAC, 0x7D, 0x5F, 0x42,
    0xD6, 0x9F, 0x6D, 0x18, 0x77, 0x63, 0xCF, 0x1D, 0x55, 0x03, 0x40, 0x04,
    0x87, 0xF5, 0x5B, 0xA5, 0x7E, 0x31, 0xCC, 0x7A, 0x71, 0x35, 0xC8, 0x86,
    0xEF, 0xB4, 0x31, 0x8A, 0xED, 0x6A, 0x1E, 0x01, 0x2D, 0x9E, 0x68, 0x32,
    0xA9, 0x07, 0x60, 0x0A, 0x91, 0x81, 0x30, 0xC4, 0x6D, 0xC7, 0x78, 0xF9,
    0x71, 0xAD, 0x00, 0x38, 0x09, 0x29, 0x99, 0xA3, 0x33, 0xCB, 0x8B, 0x7A,
    0x1A, 0x1D, 0xB9, 0x3D, 0x71, 0x40, 0x00, 0x3C, 0x2A, 0x4E, 0xCE, 0xA9,
    0xF9, 0x8D, 0x0A, 0xCC, 0x0A, 0x82, 0x91, 0xCD, 0xCE, 0xC9, 0x7D, 0xCF,
    0x8E, 0xC9, 0xB5, 0x5A, 0x7F, 0x88, 0xA4, 0x6B, 0x4D, 0xB5, 0xA8, 0x51,
    0xF4, 0x41, 0x82, 0xE1, 0xC6, 0x8A, 0x00, 0x7E, 0x5E, 0x65, 0x5F, 0x6A,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* IKE 6144 prime is: 2^6144 - 2^6080 - 1 + 2^64 * { [2^6014 pi] + 929484 } */
static const unsigned char prime_ike_6144[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2,
    0x21, 0x68, 0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
    0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE, 0xA6,
    0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
    0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D,
    0xF2, 0x5F, 0x14, 0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
    0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
    0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
    0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11,
    0x7C, 0x4B, 0x1F, 0xE6, 0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
    0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36,
    0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
    0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56,
    0x20, 0x85, 0x52, 0xBB, 0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
    0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04, 0xF1, 0x74, 0x6C, 0x08,
    0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
    0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2,
    0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9,
    0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7C,
    0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
    0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
    0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21, 0xAB, 0xDF, 0x1C, 0xBA, 0x64,
    0xEC, 0xFB, 0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A, 0xEA, 0x71, 0x57,
    0x5D, 0x06, 0x0C, 0x7D, 0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4, 0xC7,
    0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09, 0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0,
    0x4A, 0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26, 0x1A, 0xD2, 0xEE, 0x6B,
    0xF1, 0x2F, 0xFA, 0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76, 0x02, 0x73,
    0x3E, 0xC8, 0x6A, 0x64, 0x52, 0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
    0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D, 0x6C, 0x77, 0x09, 0x88, 0xC0,
    0xBA, 0xD9, 0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74, 0xE5, 0xAB, 0x31,
    0x43, 0xDB, 0x5B, 0xFC, 0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1, 0x20,
    0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72, 0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7,
    0x88, 0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26, 0x99, 0xC3, 0x27, 0x18,
    0x6A, 0xF4, 0xE2, 0x3C, 0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15, 0x0B, 0xDA,
    0x25, 0x83, 0xE9, 0xCA, 0x2A, 0xD4, 0x4C, 0xE8, 0xDB, 0xBB, 0xC2, 0xDB,
    0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC, 0x14, 0x1F, 0xBE, 0xCA, 0xA6,
    0x28, 0x7C, 0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99, 0xB2, 0x96, 0x4F,
    0xA0, 0x90, 0xC3, 0xA2, 0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7, 0xED,
    0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2, 0xD7, 0xAF, 0xB8, 0x1B, 0xDD, 0x76,
    0x21, 0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27, 0xD5, 0xB0, 0x5A, 0xA9,
    0x93, 0xB4, 0xEA, 0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF, 0xB7, 0xDC,
    0x90, 0xA6, 0xC0, 0x8F, 0x4D, 0xF4, 0x35, 0xC9, 0x34, 0x02, 0x84, 0x92,
    0x36, 0xC3, 0xFA, 0xB4, 0xD2, 0x7C, 0x70, 0x26, 0xC1, 0xD4, 0xDC, 0xB2,
    0x60, 0x26, 0x46, 0xDE, 0xC9, 0x75, 0x1E, 0x76, 0x3D, 0xBA, 0x37, 0xBD,
    0xF8, 0xFF, 0x94, 0x06, 0xAD, 0x9E, 0x53, 0x0E, 0xE5, 0xDB, 0x38, 0x2F,
    0x41, 0x30, 0x01, 0xAE, 0xB0, 0x6A, 0x53, 0xED, 0x90, 0x27, 0xD8, 0x31,
    0x17, 0x97, 0x27, 0xB0, 0x86, 0x5A, 0x89, 0x18, 0xDA, 0x3E, 0xDB, 0xEB,
    0xCF, 0x9B, 0x14, 0xED, 0x44, 0xCE, 0x6C, 0xBA, 0xCE, 0xD4, 0xBB, 0x1B,
    0xDB, 0x7F, 0x14, 0x47, 0xE6, 0xCC, 0x25, 0x4B, 0x33, 0x20, 0x51, 0x51,
    0x2B, 0xD7, 0xAF, 0x42, 0x6F, 0xB8, 0xF4, 0x01, 0x37, 0x8C, 0xD2, 0xBF,
    0x59, 0x83, 0xCA, 0x01, 0xC6, 0x4B, 0x92, 0xEC, 0xF0, 0x32, 0xEA, 0x15,
    0xD1, 0x72, 0x1D, 0x03, 0xF4, 0x82, 0xD7, 0xCE, 0x6E, 0x74, 0xFE, 0xF6,
    0xD5, 0x5E, 0x70, 0x2F, 0x46, 0x98, 0x0C, 0x82, 0xB5, 0xA8, 0x40, 0x31,
    0x90, 0x0B, 0x1C, 0x9E, 0x59, 0xE7, 0xC9, 0x7F, 0xBE, 0xC7, 0xE8, 0xF3,
    0x23, 0xA9, 0x7A, 0x7E, 0x36, 0xCC, 0x88, 0xBE, 0x0F, 0x1D, 0x45, 0xB7,
    0xFF, 0x58, 0x5A, 0xC5, 0x4B, 0xD4, 0x07, 0xB2, 0x2B, 0x41, 0x54, 0xAA,
    0xCC, 0x8F, 0x6D, 0x7E, 0xBF, 0x48, 0xE1, 0xD8, 0x14, 0xCC, 0x5E, 0xD2,
    0x0F, 0x80, 0x37, 0xE0, 0xA7, 0x97, 0x15, 0xEE, 0xF2, 0x9B, 0xE3, 0x28,
    0x06, 0xA1, 0xD5, 0x8B, 0xB7, 0xC5, 0xDA, 0x76, 0xF5, 0x50, 0xAA, 0x3D,
    0x8A, 0x1F, 0xBF, 0xF0, 0xEB, 0x19, 0xCC, 0xB1, 0xA3, 0x13, 0xD5, 0x5C,
    0xDA, 0x56, 0xC9, 0xEC, 0x2E, 0xF2, 0x96, 0x32, 0x38, 0x7F, 0xE8, 0xD7,
    0x6E, 0x3C, 0x04, 0x68, 0x04, 0x3E, 0x8F, 0x66, 0x3F, 0x48, 0x60, 0xEE,
    0x12, 0xBF, 0x2D, 0x5B, 0x0B, 0x74, 0x74, 0xD6, 0xE6, 0x94, 0xF9, 0x1E,
    0x6D, 0xCC, 0x40, 0x24, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* TLS 6144 prime is: 2^6144 - 2^6080 + {[2^6014 * e] + 15705020} * 2^64 - 1 */
static const unsigned char prime_tls_6144[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAD, 0xF8, 0x54, 0x58,
    0xA2, 0xBB, 0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 0x3D, 0x3C, 0xF1,
    0xD8, 0xB9, 0xC5, 0x83, 0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 0x41,
    0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9,
    0x7D, 0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 0xF6, 0x81, 0xB2, 0x02,
    0xAE, 0xC4, 0x61, 0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 0x65, 0x61,
    0x24, 0x33, 0xF5, 0x1F, 0x5F, 0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55,
    0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 0x5E, 0x7F, 0x57, 0xC9, 0x35,
    0x98, 0x4F, 0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 0xA6, 0x89, 0xDA,
    0xF3, 0xEF, 0xE8, 0x72, 0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 0x35,
    0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82,
    0xB3, 0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 0xB2, 0xC8, 0xE3, 0xFB,
    0xB9, 0x6A, 0xDA, 0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 0x42, 0xA3,
    0xDE, 0x39, 0x4D, 0xF4, 0xAE, 0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19,
    0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 0x70, 0x9E, 0x02, 0xFC, 0xE1,
    0xCD, 0xF7, 0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 0x34, 0x2F, 0x61,
    0x91, 0x72, 0xFE, 0x9C, 0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 0x32,
    0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73,
    0x3B, 0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 0xC5, 0x8E, 0xF1, 0x83,
    0x7D, 0x16, 0x83, 0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 0xEF, 0xFA,
    0x88, 0x6B, 0x42, 0x38, 0x61, 0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B,
    0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 0xDE, 0xF9, 0x9C, 0x02, 0x38,
    0x61, 0xB4, 0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 0xD9, 0x1D, 0x26,
    0x91, 0xF7, 0xF7, 0xEE, 0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 0x1C,
    0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93,
    0xBC, 0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 0xE2, 0xD7, 0x4D, 0xD3,
    0x64, 0xF2, 0xE2, 0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 0x82, 0xAB,
    0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D,
    0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 0xDA, 0x1D, 0xBF, 0x9A, 0x42,
    0xD5, 0xC4, 0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 0x53, 0xDD, 0xEF,
    0x3C, 0x1B, 0x20, 0xEE, 0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 0x2B,
    0x66, 0x9E, 0x1E, 0xF1, 0x6E, 0x6F, 0x52, 0xC3, 0x16, 0x4D, 0xF4, 0xFB,
    0x79, 0x30, 0xE9, 0xE4, 0xE5, 0x88, 0x57, 0xB6, 0xAC, 0x7D, 0x5F, 0x42,
    0xD6, 0x9F, 0x6D, 0x18, 0x77, 0x63, 0xCF, 0x1D, 0x55, 0x03, 0x40, 0x04,
    0x87, 0xF5, 0x5B, 0xA5, 0x7E, 0x31, 0xCC, 0x7A, 0x71, 0x35, 0xC8, 0x86,
    0xEF, 0xB4, 0x31, 0x8A, 0xED, 0x6A, 0x1E, 0x01, 0x2D, 0x9E, 0x68, 0x32,
    0xA9, 0x07, 0x60, 0x0A, 0x91, 0x81, 0x30, 0xC4, 0x6D, 0xC7, 0x78, 0xF9,
    0x71, 0xAD, 0x00, 0x38, 0x09, 0x29, 0x99, 0xA3, 0x33, 0xCB, 0x8B, 0x7A,
    0x1A, 0x1D, 0xB9, 0x3D, 0x71, 0x40, 0x00, 0x3C, 0x2A, 0x4E, 0xCE, 0xA9,
    0xF9, 0x8D, 0x0A, 0xCC, 0x0A, 0x82, 0x91, 0xCD, 0xCE, 0xC9, 0x7D, 0xCF,
    0x8E, 0xC9, 0xB5, 0x5A, 0x7F, 0x88, 0xA4, 0x6B, 0x4D, 0xB5, 0xA8, 0x51,
    0xF4, 0x41, 0x82, 0xE1, 0xC6, 0x8A, 0x00, 0x7E, 0x5E, 0x0D, 0xD9, 0x02,
    0x0B, 0xFD, 0x64, 0xB6, 0x45, 0x03, 0x6C, 0x7A, 0x4E, 0x67, 0x7D, 0x2C,
    0x38, 0x53, 0x2A, 0x3A, 0x23, 0xBA, 0x44, 0x42, 0xCA, 0xF5, 0x3E, 0xA6,
    0x3B, 0xB4, 0x54, 0x32, 0x9B, 0x76, 0x24, 0xC8, 0x91, 0x7B, 0xDD, 0x64,
    0xB1, 0xC0, 0xFD, 0x4C, 0xB3, 0x8E, 0x8C, 0x33, 0x4C, 0x70, 0x1C, 0x3A,
    0xCD, 0xAD, 0x06, 0x57, 0xFC, 0xCF, 0xEC, 0x71, 0x9B, 0x1F, 0x5C, 0x3E,
    0x4E, 0x46, 0x04, 0x1F, 0x38, 0x81, 0x47, 0xFB, 0x4C, 0xFD, 0xB4, 0x77,
    0xA5, 0x24, 0x71, 0xF7, 0xA9, 0xA9, 0x69, 0x10, 0xB8, 0x55, 0x32, 0x2E,
    0xDB, 0x63, 0x40, 0xD8, 0xA0, 0x0E, 0xF0, 0x92, 0x35, 0x05, 0x11, 0xE3,
    0x0A, 0xBE, 0xC1, 0xFF, 0xF9, 0xE3, 0xA2, 0x6E, 0x7F, 0xB2, 0x9F, 0x8C,
    0x18, 0x30, 0x23, 0xC3, 0x58, 0x7E, 0x38, 0xDA, 0x00, 0x77, 0xD9, 0xB4,
    0x76, 0x3E, 0x4E, 0x4B, 0x94, 0xB2, 0xBB, 0xC1, 0x94, 0xC6, 0x65, 0x1E,
    0x77, 0xCA, 0xF9, 0x92, 0xEE, 0xAA, 0xC0, 0x23, 0x2A, 0x28, 0x1B, 0xF6,
    0xB3, 0xA7, 0x39, 0xC1, 0x22, 0x61, 0x16, 0x82, 0x0A, 0xE8, 0xDB, 0x58,
    0x47, 0xA6, 0x7C, 0xBE, 0xF9, 0xC9, 0x09, 0x1B, 0x46, 0x2D, 0x53, 0x8C,
    0xD7, 0x2B, 0x03, 0x74, 0x6A, 0xE7, 0x7F, 0x5E, 0x62, 0x29, 0x2C, 0x31,
    0x15, 0x62, 0xA8, 0x46, 0x50, 0x5D, 0xC8, 0x2D, 0xB8, 0x54, 0x33, 0x8A,
    0xE4, 0x9F, 0x52, 0x35, 0xC9, 0x5B, 0x91, 0x17, 0x8C, 0xCF, 0x2D, 0xD5,
    0xCA, 0xCE, 0xF4, 0x03, 0xEC, 0x9D, 0x18, 0x10, 0xC6, 0x27, 0x2B, 0x04,
    0x5B, 0x3B, 0x71, 0xF9, 0xDC, 0x6B, 0x80, 0xD6, 0x3F, 0xDD, 0x4A, 0x8E,
    0x9A, 0xDB, 0x1E, 0x69, 0x62, 0xA6, 0x95, 0x26, 0xD4, 0x31, 0x61, 0xC1,
    0xA4, 0x1D, 0x57, 0x0D, 0x79, 0x38, 0xDA, 0xD4, 0xA4, 0x0E, 0x32, 0x9C,
    0xD0, 0xE4, 0x0E, 0x65, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

/* IKE 8192 prime is: 2^8192 - 2^8128 - 1 + 2^64 * { [2^8062 pi] + 4743158 } */
static const unsigned char prime_ike_8192[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC9, 0x0F, 0xDA, 0xA2,
    0x21, 0x68, 0xC2, 0x34, 0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
    0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74, 0x02, 0x0B, 0xBE, 0xA6,
    0x3B, 0x13, 0x9B, 0x22, 0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
    0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B, 0x30, 0x2B, 0x0A, 0x6D,
    0xF2, 0x5F, 0x14, 0x37, 0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
    0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6, 0xF4, 0x4C, 0x42, 0xE9,
    0xA6, 0x37, 0xED, 0x6B, 0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
    0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5, 0xAE, 0x9F, 0x24, 0x11,
    0x7C, 0x4B, 0x1F, 0xE6, 0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
    0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05, 0x98, 0xDA, 0x48, 0x36,
    0x1C, 0x55, 0xD3, 0x9A, 0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
    0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96, 0x1C, 0x62, 0xF3, 0x56,
    0x20, 0x85, 0x52, 0xBB, 0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
    0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04, 0xF1, 0x74, 0x6C, 0x08,
    0xCA, 0x18, 0x21, 0x7C, 0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
    0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03, 0x9B, 0x27, 0x83, 0xA2,
    0xEC, 0x07, 0xA2, 0x8F, 0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9,
    0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18, 0x39, 0x95, 0x49, 0x7C,
    0xEA, 0x95, 0x6A, 0xE5, 0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
    0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAA, 0xC4, 0x2D, 0xAD, 0x33, 0x17, 0x0D,
    0x04, 0x50, 0x7A, 0x33, 0xA8, 0x55, 0x21, 0xAB, 0xDF, 0x1C, 0xBA, 0x64,
    0xEC, 0xFB, 0x85, 0x04, 0x58, 0xDB, 0xEF, 0x0A, 0x8A, 0xEA, 0x71, 0x57,
    0x5D, 0x06, 0x0C, 0x7D, 0xB3, 0x97, 0x0F, 0x85, 0xA6, 0xE1, 0xE4, 0xC7,
    0xAB, 0xF5, 0xAE, 0x8C, 0xDB, 0x09, 0x33, 0xD7, 0x1E, 0x8C, 0x94, 0xE0,
    0x4A, 0x25, 0x61, 0x9D, 0xCE, 0xE3, 0xD2, 0x26, 0x1A, 0xD2, 0xEE, 0x6B,
    0xF1, 0x2F, 0xFA, 0x06, 0xD9, 0x8A, 0x08, 0x64, 0xD8, 0x76, 0x02, 0x73,
    0x3E, 0xC8, 0x6A, 0x64, 0x52, 0x1F, 0x2B, 0x18, 0x17, 0x7B, 0x20, 0x0C,
    0xBB, 0xE1, 0x17, 0x57, 0x7A, 0x61, 0x5D, 0x6C, 0x77, 0x09, 0x88, 0xC0,
    0xBA, 0xD9, 0x46, 0xE2, 0x08, 0xE2, 0x4F, 0xA0, 0x74, 0xE5, 0xAB, 0x31,
    0x43, 0xDB, 0x5B, 0xFC, 0xE0, 0xFD, 0x10, 0x8E, 0x4B, 0x82, 0xD1, 0x20,
    0xA9, 0x21, 0x08, 0x01, 0x1A, 0x72, 0x3C, 0x12, 0xA7, 0x87, 0xE6, 0xD7,
    0x88, 0x71, 0x9A, 0x10, 0xBD, 0xBA, 0x5B, 0x26, 0x99, 0xC3, 0x27, 0x18,
    0x6A, 0xF4, 0xE2, 0x3C, 0x1A, 0x94, 0x68, 0x34, 0xB6, 0x15, 0x0B, 0xDA,
    0x25, 0x83, 0xE9, 0xCA, 0x2A, 0xD4, 0x4C, 0xE8, 0xDB, 0xBB, 0xC2, 0xDB,
    0x04, 0xDE, 0x8E, 0xF9, 0x2E, 0x8E, 0xFC, 0x14, 0x1F, 0xBE, 0xCA, 0xA6,
    0x28, 0x7C, 0x59, 0x47, 0x4E, 0x6B, 0xC0, 0x5D, 0x99, 0xB2, 0x96, 0x4F,
    0xA0, 0x90, 0xC3, 0xA2, 0x23, 0x3B, 0xA1, 0x86, 0x51, 0x5B, 0xE7, 0xED,
    0x1F, 0x61, 0x29, 0x70, 0xCE, 0xE2, 0xD7, 0xAF, 0xB8, 0x1B, 0xDD, 0x76,
    0x21, 0x70, 0x48, 0x1C, 0xD0, 0x06, 0x91, 0x27, 0xD5, 0xB0, 0x5A, 0xA9,
    0x93, 0xB4, 0xEA, 0x98, 0x8D, 0x8F, 0xDD, 0xC1, 0x86, 0xFF, 0xB7, 0xDC,
    0x90, 0xA6, 0xC0, 0x8F, 0x4D, 0xF4, 0x35, 0xC9, 0x34, 0x02, 0x84, 0x92,
    0x36, 0xC3, 0xFA, 0xB4, 0xD2, 0x7C, 0x70, 0x26, 0xC1, 0xD4, 0xDC, 0xB2,
    0x60, 0x26, 0x46, 0xDE, 0xC9, 0x75, 0x1E, 0x76, 0x3D, 0xBA, 0x37, 0xBD,
    0xF8, 0xFF, 0x94, 0x06, 0xAD, 0x9E, 0x53, 0x0E, 0xE5, 0xDB, 0x38, 0x2F,
    0x41, 0x30, 0x01, 0xAE, 0xB0, 0x6A, 0x53, 0xED, 0x90, 0x27, 0xD8, 0x31,
    0x17, 0x97, 0x27, 0xB0, 0x86, 0x5A, 0x89, 0x18, 0xDA, 0x3E, 0xDB, 0xEB,
    0xCF, 0x9B, 0x14, 0xED, 0x44, 0xCE, 0x6C, 0xBA, 0xCE, 0xD4, 0xBB, 0x1B,
    0xDB, 0x7F, 0x14, 0x47, 0xE6, 0xCC, 0x25, 0x4B, 0x33, 0x20, 0x51, 0x51,
    0x2B, 0xD7, 0xAF, 0x42, 0x6F, 0xB8, 0xF4, 0x01, 0x37, 0x8C, 0xD2, 0xBF,
    0x59, 0x83, 0xCA, 0x01, 0xC6, 0x4B, 0x92, 0xEC, 0xF0, 0x32, 0xEA, 0x15,
    0xD1, 0x72, 0x1D, 0x03, 0xF4, 0x82, 0xD7, 0xCE, 0x6E, 0x74, 0xFE, 0xF6,
    0xD5, 0x5E, 0x70, 0x2F, 0x46, 0x98, 0x0C, 0x82, 0xB5, 0xA8, 0x40, 0x31,
    0x90, 0x0B, 0x1C, 0x9E, 0x59, 0xE7, 0xC9, 0x7F, 0xBE, 0xC7, 0xE8, 0xF3,
    0x23, 0xA9, 0x7A, 0x7E, 0x36, 0xCC, 0x88, 0xBE, 0x0F, 0x1D, 0x45, 0xB7,
    0xFF, 0x58, 0x5A, 0xC5, 0x4B, 0xD4, 0x07, 0xB2, 0x2B, 0x41, 0x54, 0xAA,
    0xCC, 0x8F, 0x6D, 0x7E, 0xBF, 0x48, 0xE1, 0xD8, 0x14, 0xCC, 0x5E, 0xD2,
    0x0F, 0x80, 0x37, 0xE0, 0xA7, 0x97, 0x15, 0xEE, 0xF2, 0x9B, 0xE3, 0x28,
    0x06, 0xA1, 0xD5, 0x8B, 0xB7, 0xC5, 0xDA, 0x76, 0xF5, 0x50, 0xAA, 0x3D,
    0x8A, 0x1F, 0xBF, 0xF0, 0xEB, 0x19, 0xCC, 0xB1, 0xA3, 0x13, 0xD5, 0x5C,
    0xDA, 0x56, 0xC9, 0xEC, 0x2E, 0xF2, 0x96, 0x32, 0x38, 0x7F, 0xE8, 0xD7,
    0x6E, 0x3C, 0x04, 0x68, 0x04, 0x3E, 0x8F, 0x66, 0x3F, 0x48, 0x60, 0xEE,
    0x12, 0xBF, 0x2D, 0x5B, 0x0B, 0x74, 0x74, 0xD6, 0xE6, 0x94, 0xF9, 0x1E,
    0x6D, 0xBE, 0x11, 0x59, 0x74, 0xA3, 0x92, 0x6F, 0x12, 0xFE, 0xE5, 0xE4,
    0x38, 0x77, 0x7C, 0xB6, 0xA9, 0x32, 0xDF, 0x8C, 0xD8, 0xBE, 0xC4, 0xD0,
    0x73, 0xB9, 0x31, 0xBA, 0x3B, 0xC8, 0x32, 0xB6, 0x8D, 0x9D, 0xD3, 0x00,
    0x74, 0x1F, 0xA7, 0xBF, 0x8A, 0xFC, 0x47, 0xED, 0x25, 0x76, 0xF6, 0x93,
    0x6B, 0xA4, 0x24, 0x66, 0x3A, 0xAB, 0x63, 0x9C, 0x5A, 0xE4, 0xF5, 0x68,
    0x34, 0x23, 0xB4, 0x74, 0x2B, 0xF1, 0xC9, 0x78, 0x23, 0x8F, 0x16, 0xCB,
    0xE3, 0x9D, 0x65, 0x2D, 0xE3, 0xFD, 0xB8, 0xBE, 0xFC, 0x84, 0x8A, 0xD9,
    0x22, 0x22, 0x2E, 0x04, 0xA4, 0x03, 0x7C, 0x07, 0x13, 0xEB, 0x57, 0xA8,
    0x1A, 0x23, 0xF0, 0xC7, 0x34, 0x73, 0xFC, 0x64, 0x6C, 0xEA, 0x30, 0x6B,
    0x4B, 0xCB, 0xC8, 0x86, 0x2F, 0x83, 0x85, 0xDD, 0xFA, 0x9D, 0x4B, 0x7F,
    0xA2, 0xC0, 0x87, 0xE8, 0x79, 0x68, 0x33, 0x03, 0xED, 0x5B, 0xDD, 0x3A,
    0x06, 0x2B, 0x3C, 0xF5, 0xB3, 0xA2, 0x78, 0xA6, 0x6D, 0x2A, 0x13, 0xF8,
    0x3F, 0x44, 0xF8, 0x2D, 0xDF, 0x31, 0x0E, 0xE0, 0x74, 0xAB, 0x6A, 0x36,
    0x45, 0x97, 0xE8, 0x99, 0xA0, 0x25, 0x5D, 0xC1, 0x64, 0xF3, 0x1C, 0xC5,
    0x08, 0x46, 0x85, 0x1D, 0xF9, 0xAB, 0x48, 0x19, 0x5D, 0xED, 0x7E, 0xA1,
    0xB1, 0xD5, 0x10, 0xBD, 0x7E, 0xE7, 0x4D, 0x73, 0xFA, 0xF3, 0x6B, 0xC3,
    0x1E, 0xCF, 0xA2, 0x68, 0x35, 0x90, 0x46, 0xF4, 0xEB, 0x87, 0x9F, 0x92,
    0x40, 0x09, 0x43, 0x8B, 0x48, 0x1C, 0x6C, 0xD7, 0x88, 0x9A, 0x00, 0x2E,
    0xD5, 0xEE, 0x38, 0x2B, 0xC9, 0x19, 0x0D, 0xA6, 0xFC, 0x02, 0x6E, 0x47,
    0x95, 0x58, 0xE4, 0x47, 0x56, 0x77, 0xE9, 0xAA, 0x9E, 0x30, 0x50, 0xE2,
    0x76, 0x56, 0x94, 0xDF, 0xC8, 0x1F, 0x56, 0xE8, 0x80, 0xB9, 0x6E, 0x71,
    0x60, 0xC9, 0x80, 0xDD, 0x98, 0xED, 0xD3, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF
};

/* TLS 8192 prime is: 2^8192 - 2^8128 + {[2^8062 * e] + 10965728} * 2^64 - 1 */
static const unsigned char prime_tls_8192[] = {
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xAD, 0xF8, 0x54, 0x58,
    0xA2, 0xBB, 0x4A, 0x9A, 0xAF, 0xDC, 0x56, 0x20, 0x27, 0x3D, 0x3C, 0xF1,
    0xD8, 0xB9, 0xC5, 0x83, 0xCE, 0x2D, 0x36, 0x95, 0xA9, 0xE1, 0x36, 0x41,
    0x14, 0x64, 0x33, 0xFB, 0xCC, 0x93, 0x9D, 0xCE, 0x24, 0x9B, 0x3E, 0xF9,
    0x7D, 0x2F, 0xE3, 0x63, 0x63, 0x0C, 0x75, 0xD8, 0xF6, 0x81, 0xB2, 0x02,
    0xAE, 0xC4, 0x61, 0x7A, 0xD3, 0xDF, 0x1E, 0xD5, 0xD5, 0xFD, 0x65, 0x61,
    0x24, 0x33, 0xF5, 0x1F, 0x5F, 0x06, 0x6E, 0xD0, 0x85, 0x63, 0x65, 0x55,
    0x3D, 0xED, 0x1A, 0xF3, 0xB5, 0x57, 0x13, 0x5E, 0x7F, 0x57, 0xC9, 0x35,
    0x98, 0x4F, 0x0C, 0x70, 0xE0, 0xE6, 0x8B, 0x77, 0xE2, 0xA6, 0x89, 0xDA,
    0xF3, 0xEF, 0xE8, 0x72, 0x1D, 0xF1, 0x58, 0xA1, 0x36, 0xAD, 0xE7, 0x35,
    0x30, 0xAC, 0xCA, 0x4F, 0x48, 0x3A, 0x79, 0x7A, 0xBC, 0x0A, 0xB1, 0x82,
    0xB3, 0x24, 0xFB, 0x61, 0xD1, 0x08, 0xA9, 0x4B, 0xB2, 0xC8, 0xE3, 0xFB,
    0xB9, 0x6A, 0xDA, 0xB7, 0x60, 0xD7, 0xF4, 0x68, 0x1D, 0x4F, 0x42, 0xA3,
    0xDE, 0x39, 0x4D, 0xF4, 0xAE, 0x56, 0xED, 0xE7, 0x63, 0x72, 0xBB, 0x19,
    0x0B, 0x07, 0xA7, 0xC8, 0xEE, 0x0A, 0x6D, 0x70, 0x9E, 0x02, 0xFC, 0xE1,
    0xCD, 0xF7, 0xE2, 0xEC, 0xC0, 0x34, 0x04, 0xCD, 0x28, 0x34, 0x2F, 0x61,
    0x91, 0x72, 0xFE, 0x9C, 0xE9, 0x85, 0x83, 0xFF, 0x8E, 0x4F, 0x12, 0x32,
    0xEE, 0xF2, 0x81, 0x83, 0xC3, 0xFE, 0x3B, 0x1B, 0x4C, 0x6F, 0xAD, 0x73,
    0x3B, 0xB5, 0xFC, 0xBC, 0x2E, 0xC2, 0x20, 0x05, 0xC5, 0x8E, 0xF1, 0x83,
    0x7D, 0x16, 0x83, 0xB2, 0xC6, 0xF3, 0x4A, 0x26, 0xC1, 0xB2, 0xEF, 0xFA,
    0x88, 0x6B, 0x42, 0x38, 0x61, 0x1F, 0xCF, 0xDC, 0xDE, 0x35, 0x5B, 0x3B,
    0x65, 0x19, 0x03, 0x5B, 0xBC, 0x34, 0xF4, 0xDE, 0xF9, 0x9C, 0x02, 0x38,
    0x61, 0xB4, 0x6F, 0xC9, 0xD6, 0xE6, 0xC9, 0x07, 0x7A, 0xD9, 0x1D, 0x26,
    0x91, 0xF7, 0xF7, 0xEE, 0x59, 0x8C, 0xB0, 0xFA, 0xC1, 0x86, 0xD9, 0x1C,
    0xAE, 0xFE, 0x13, 0x09, 0x85, 0x13, 0x92, 0x70, 0xB4, 0x13, 0x0C, 0x93,
    0xBC, 0x43, 0x79, 0x44, 0xF4, 0xFD, 0x44, 0x52, 0xE2, 0xD7, 0x4D, 0xD3,
    0x64, 0xF2, 0xE2, 0x1E, 0x71, 0xF5, 0x4B, 0xFF, 0x5C, 0xAE, 0x82, 0xAB,
    0x9C, 0x9D, 0xF6, 0x9E, 0xE8, 0x6D, 0x2B, 0xC5, 0x22, 0x36, 0x3A, 0x0D,
    0xAB, 0xC5, 0x21, 0x97, 0x9B, 0x0D, 0xEA, 0xDA, 0x1D, 0xBF, 0x9A, 0x42,
    0xD5, 0xC4, 0x48, 0x4E, 0x0A, 0xBC, 0xD0, 0x6B, 0xFA, 0x53, 0xDD, 0xEF,
    0x3C, 0x1B, 0x20, 0xEE, 0x3F, 0xD5, 0x9D, 0x7C, 0x25, 0xE4, 0x1D, 0x2B,
    0x66, 0x9E, 0x1E, 0xF1, 0x6E, 0x6F, 0x52, 0xC3, 0x16, 0x4D, 0xF4, 0xFB,
    0x79, 0x30, 0xE9, 0xE4, 0xE5, 0x88, 0x57, 0xB6, 0xAC, 0x7D, 0x5F, 0x42,
    0xD6, 0x9F, 0x6D, 0x18, 0x77, 0x63, 0xCF, 0x1D, 0x55, 0x03, 0x40, 0x04,
    0x87, 0xF5, 0x5B, 0xA5, 0x7E, 0x31, 0xCC, 0x7A, 0x71, 0x35, 0xC8, 0x86,
    0xEF, 0xB4, 0x31, 0x8A, 0xED, 0x6A, 0x1E, 0x01, 0x2D, 0x9E, 0x68, 0x32,
    0xA9, 0x07, 0x60, 0x0A, 0x91, 0x81, 0x30, 0xC4, 0x6D, 0xC7, 0x78, 0xF9,
    0x71, 0xAD, 0x00, 0x38, 0x09, 0x29, 0x99, 0xA3, 0x33, 0xCB, 0x8B, 0x7A,
    0x1A, 0x1D, 0xB9, 0x3D, 0x71, 0x40, 0x00, 0x3C, 0x2A, 0x4E, 0xCE, 0xA9,
    0xF9, 0x8D, 0x0A, 0xCC, 0x0A, 0x82, 0x91, 0xCD, 0xCE, 0xC9, 0x7D, 0xCF,
    0x8E, 0xC9, 0xB5, 0x5A, 0x7F, 0x88, 0xA4, 0x6B, 0x4D, 0xB5, 0xA8, 0x51,
    0xF4, 0x41, 0x82, 0xE1, 0xC6, 0x8A, 0x00, 0x7E, 0x5E, 0x0D, 0xD9, 0x02,
    0x0B, 0xFD, 0x64, 0xB6, 0x45, 0x03, 0x6C, 0x7A, 0x4E, 0x67, 0x7D, 0x2C,
    0x38, 0x53, 0x2A, 0x3A, 0x23, 0xBA, 0x44, 0x42, 0xCA, 0xF5, 0x3E, 0xA6,
    0x3B, 0xB4, 0x54, 0x32, 0x9B, 0x76, 0x24, 0xC8, 0x91, 0x7B, 0xDD, 0x64,
    0xB1, 0xC0, 0xFD, 0x4C, 0xB3, 0x8E, 0x8C, 0x33, 0x4C, 0x70, 0x1C, 0x3A,
    0xCD, 0xAD, 0x06, 0x57, 0xFC, 0xCF, 0xEC, 0x71, 0x9B, 0x1F, 0x5C, 0x3E,
    0x4E, 0x46, 0x04, 0x1F, 0x38, 0x81, 0x47, 0xFB, 0x4C, 0xFD, 0xB4, 0x77,
    0xA5, 0x24, 0x71, 0xF7, 0xA9, 0xA9, 0x69, 0x10, 0xB8, 0x55, 0x32, 0x2E,
    0xDB, 0x63, 0x40, 0xD8, 0xA0, 0x0E, 0xF0, 0x92, 0x35, 0x05, 0x11, 0xE3,
    0x0A, 0xBE, 0xC1, 0xFF, 0xF9, 0xE3, 0xA2, 0x6E, 0x7F, 0xB2, 0x9F, 0x8C,
    0x18, 0x30, 0x23, 0xC3, 0x58, 0x7E, 0x38, 0xDA, 0x00, 0x77, 0xD9, 0xB4,
    0x76, 0x3E, 0x4E, 0x4B, 0x94, 0xB2, 0xBB, 0xC1, 0x94, 0xC6, 0x65, 0x1E,
    0x77, 0xCA, 0xF9, 0x92, 0xEE, 0xAA, 0xC0, 0x23, 0x2A, 0x28, 0x1B, 0xF6,
    0xB3, 0xA7, 0x39, 0xC1, 0x22, 0x61, 0x16, 0x82, 0x0A, 0xE8, 0xDB, 0x58,
    0x47, 0xA6, 0x7C, 0xBE, 0xF9, 0xC9, 0x09, 0x1B, 0x46, 0x2D, 0x53, 0x8C,
    0xD7, 0x2B, 0x03, 0x74, 0x6A, 0xE7, 0x7F, 0x5E, 0x62, 0x29, 0x2C, 0x31,
    0x15, 0x62, 0xA8, 0x46, 0x50, 0x5D, 0xC8, 0x2D, 0xB8, 0x54, 0x33, 0x8A,
    0xE4, 0x9F, 0x52, 0x35, 0xC9, 0x5B, 0x91, 0x17, 0x8C, 0xCF, 0x2D, 0xD5,
    0xCA, 0xCE, 0xF4, 0x03, 0xEC, 0x9D, 0x18, 0x10, 0xC6, 0x27, 0x2B, 0x04,
    0x5B, 0x3B, 0x71, 0xF9, 0xDC, 0x6B, 0x80, 0xD6, 0x3F, 0xDD, 0x4A, 0x8E,
    0x9A, 0xDB, 0x1E, 0x69, 0x62, 0xA6, 0x95, 0x26, 0xD4, 0x31, 0x61, 0xC1,
    0xA4, 0x1D, 0x57, 0x0D, 0x79, 0x38, 0xDA, 0xD4, 0xA4, 0x0E, 0x32, 0x9C,
    0xCF, 0xF4, 0x6A, 0xAA, 0x36, 0xAD, 0x00, 0x4C, 0xF6, 0x00, 0xC8, 0x38,
    0x1E, 0x42, 0x5A, 0x31, 0xD9, 0x51, 0xAE, 0x64, 0xFD, 0xB2, 0x3F, 0xCE,
    0xC9, 0x50, 0x9D, 0x43, 0x68, 0x7F, 0xEB, 0x69, 0xED, 0xD1, 0xCC, 0x5E,
    0x0B, 0x8C, 0xC3, 0xBD, 0xF6, 0x4B, 0x10, 0xEF, 0x86, 0xB6, 0x31, 0x42,
    0xA3, 0xAB, 0x88, 0x29, 0x55, 0x5B, 0x2F, 0x74, 0x7C, 0x93, 0x26, 0x65,
    0xCB, 0x2C, 0x0F, 0x1C, 0xC0, 0x1B, 0xD7, 0x02, 0x29, 0x38, 0x88, 0x39,
    0xD2, 0xAF, 0x05, 0xE4, 0x54, 0x50, 0x4A, 0xC7, 0x8B, 0x75, 0x82, 0x82,
    0x28, 0x46, 0xC0, 0xBA, 0x35, 0xC3, 0x5F, 0x5C, 0x59, 0x16, 0x0C, 0xC0,
    0x46, 0xFD, 0x82, 0x51, 0x54, 0x1F, 0xC6, 0x8C, 0x9C, 0x86, 0xB0, 0x22,
    0xBB, 0x70, 0x99, 0x87, 0x6A, 0x46, 0x0E, 0x74, 0x51, 0xA8, 0xA9, 0x31,
    0x09, 0x70, 0x3F, 0xEE, 0x1C, 0x21, 0x7E, 0x6C, 0x38, 0x26, 0xE5, 0x2C,
    0x51, 0xAA, 0x69, 0x1E, 0x0E, 0x42, 0x3C, 0xFC, 0x99, 0xE9, 0xE3, 0x16,
    0x50, 0xC1, 0x21, 0x7B, 0x62, 0x48, 0x16, 0xCD, 0xAD, 0x9A, 0x95, 0xF9,
    0xD5, 0xB8, 0x01, 0x94, 0x88, 0xD9, 0xC0, 0xA0, 0xA1, 0xFE, 0x30, 0x75,
    0xA5, 0x77, 0xE2, 0x31, 0x83, 0xF8, 0x1D, 0x4A, 0x3F, 0x2F, 0xA4, 0x57,
    0x1E, 0xFC, 0x8C, 0xE0, 0xBA, 0x8A, 0x4F, 0xE8, 0xB6, 0x85, 0x5D, 0xFE,
    0x72, 0xB0, 0xA6, 0x6E, 0xDE, 0xD2, 0xFB, 0xAB, 0xFB, 0xE5, 0x8A, 0x30,
    0xFA, 0xFA, 0xBE, 0x1C, 0x5D, 0x71, 0xA8, 0x7E, 0x2F, 0x74, 0x1E, 0xF8,
    0xC1, 0xFE, 0x86, 0xFE, 0xA6, 0xBB, 0xFD, 0xE5, 0x30, 0x67, 0x7F, 0x0D,
    0x97, 0xD1, 0x1D, 0x49, 0xF7, 0xA8, 0x44, 0x3D, 0x08, 0x22, 0xE5, 0x06,
    0xA9, 0xF4, 0x61, 0x4E, 0x01, 0x1E, 0x2A, 0x94, 0x83, 0x8F, 0xF8, 0x8C,
    0xD6, 0x8C, 0x8B, 0xB7, 0xC5, 0xC6, 0x42, 0x4C, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF
};

/* q=(p-1)/2 for prime prime_ike_1536 */
static const unsigned char subprime_ike_1536_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xe4,
    0x87,
    0xed,
    0x51,
    0x10,
    0xb4,
    0x61,
    0x1a,
    0x62,
    0x63,
    0x31,
    0x45,
    0xc0,
    0x6e,
    0x0e,
    0x68,
    0x94,
    0x81,
    0x27,
    0x04,
    0x45,
    0x33,
    0xe6,
    0x3a,
    0x01,
    0x05,
    0xdf,
    0x53,
    0x1d,
    0x89,
    0xcd,
    0x91,
    0x28,
    0xa5,
    0x04,
    0x3c,
    0xc7,
    0x1a,
    0x02,
    0x6e,
    0xf7,
    0xca,
    0x8c,
    0xd9,
    0xe6,
    0x9d,
    0x21,
    0x8d,
    0x98,
    0x15,
    0x85,
    0x36,
    0xf9,
    0x2f,
    0x8a,
    0x1b,
    0xa7,
    0xf0,
    0x9a,
    0xb6,
    0xb6,
    0xa8,
    0xe1,
    0x22,
    0xf2,
    0x42,
    0xda,
    0xbb,
    0x31,
    0x2f,
    0x3f,
    0x63,
    0x7a,
    0x26,
    0x21,
    0x74,
    0xd3,
    0x1b,
    0xf6,
    0xb5,
    0x85,
    0xff,
    0xae,
    0x5b,
    0x7a,
    0x03,
    0x5b,
    0xf6,
    0xf7,
    0x1c,
    0x35,
    0xfd,
    0xad,
    0x44,
    0xcf,
    0xd2,
    0xd7,
    0x4f,
    0x92,
    0x08,
    0xbe,
    0x25,
    0x8f,
    0xf3,
    0x24,
    0x94,
    0x33,
    0x28,
    0xf6,
    0x72,
    0x2d,
    0x9e,
    0xe1,
    0x00,
    0x3e,
    0x5c,
    0x50,
    0xb1,
    0xdf,
    0x82,
    0xcc,
    0x6d,
    0x24,
    0x1b,
    0x0e,
    0x2a,
    0xe9,
    0xcd,
    0x34,
    0x8b,
    0x1f,
    0xd4,
    0x7e,
    0x92,
    0x67,
    0xaf,
    0xc1,
    0xb2,
    0xae,
    0x91,
    0xee,
    0x51,
    0xd6,
    0xcb,
    0x0e,
    0x31,
    0x79,
    0xab,
    0x10,
    0x42,
    0xa9,
    0x5d,
    0xcf,
    0x6a,
    0x94,
    0x83,
    0xb8,
    0x4b,
    0x4b,
    0x36,
    0xb3,
    0x86,
    0x1a,
    0xa7,
    0x25,
    0x5e,
    0x4c,
    0x02,
    0x78,
    0xba,
    0x36,
    0x04,
    0x65,
    0x11,
    0xb9,
    0x93,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_ike_2048 */
static const unsigned char subprime_ike_2048_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xe4,
    0x87,
    0xed,
    0x51,
    0x10,
    0xb4,
    0x61,
    0x1a,
    0x62,
    0x63,
    0x31,
    0x45,
    0xc0,
    0x6e,
    0x0e,
    0x68,
    0x94,
    0x81,
    0x27,
    0x04,
    0x45,
    0x33,
    0xe6,
    0x3a,
    0x01,
    0x05,
    0xdf,
    0x53,
    0x1d,
    0x89,
    0xcd,
    0x91,
    0x28,
    0xa5,
    0x04,
    0x3c,
    0xc7,
    0x1a,
    0x02,
    0x6e,
    0xf7,
    0xca,
    0x8c,
    0xd9,
    0xe6,
    0x9d,
    0x21,
    0x8d,
    0x98,
    0x15,
    0x85,
    0x36,
    0xf9,
    0x2f,
    0x8a,
    0x1b,
    0xa7,
    0xf0,
    0x9a,
    0xb6,
    0xb6,
    0xa8,
    0xe1,
    0x22,
    0xf2,
    0x42,
    0xda,
    0xbb,
    0x31,
    0x2f,
    0x3f,
    0x63,
    0x7a,
    0x26,
    0x21,
    0x74,
    0xd3,
    0x1b,
    0xf6,
    0xb5,
    0x85,
    0xff,
    0xae,
    0x5b,
    0x7a,
    0x03,
    0x5b,
    0xf6,
    0xf7,
    0x1c,
    0x35,
    0xfd,
    0xad,
    0x44,
    0xcf,
    0xd2,
    0xd7,
    0x4f,
    0x92,
    0x08,
    0xbe,
    0x25,
    0x8f,
    0xf3,
    0x24,
    0x94,
    0x33,
    0x28,
    0xf6,
    0x72,
    0x2d,
    0x9e,
    0xe1,
    0x00,
    0x3e,
    0x5c,
    0x50,
    0xb1,
    0xdf,
    0x82,
    0xcc,
    0x6d,
    0x24,
    0x1b,
    0x0e,
    0x2a,
    0xe9,
    0xcd,
    0x34,
    0x8b,
    0x1f,
    0xd4,
    0x7e,
    0x92,
    0x67,
    0xaf,
    0xc1,
    0xb2,
    0xae,
    0x91,
    0xee,
    0x51,
    0xd6,
    0xcb,
    0x0e,
    0x31,
    0x79,
    0xab,
    0x10,
    0x42,
    0xa9,
    0x5d,
    0xcf,
    0x6a,
    0x94,
    0x83,
    0xb8,
    0x4b,
    0x4b,
    0x36,
    0xb3,
    0x86,
    0x1a,
    0xa7,
    0x25,
    0x5e,
    0x4c,
    0x02,
    0x78,
    0xba,
    0x36,
    0x04,
    0x65,
    0x0c,
    0x10,
    0xbe,
    0x19,
    0x48,
    0x2f,
    0x23,
    0x17,
    0x1b,
    0x67,
    0x1d,
    0xf1,
    0xcf,
    0x3b,
    0x96,
    0x0c,
    0x07,
    0x43,
    0x01,
    0xcd,
    0x93,
    0xc1,
    0xd1,
    0x76,
    0x03,
    0xd1,
    0x47,
    0xda,
    0xe2,
    0xae,
    0xf8,
    0x37,
    0xa6,
    0x29,
    0x64,
    0xef,
    0x15,
    0xe5,
    0xfb,
    0x4a,
    0xac,
    0x0b,
    0x8c,
    0x1c,
    0xca,
    0xa4,
    0xbe,
    0x75,
    0x4a,
    0xb5,
    0x72,
    0x8a,
    0xe9,
    0x13,
    0x0c,
    0x4c,
    0x7d,
    0x02,
    0x88,
    0x0a,
    0xb9,
    0x47,
    0x2d,
    0x45,
    0x56,
    0x55,
    0x34,
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_tls_2048 */
static const unsigned char subprime_tls_2048_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xd6,
    0xfc,
    0x2a,
    0x2c,
    0x51,
    0x5d,
    0xa5,
    0x4d,
    0x57,
    0xee,
    0x2b,
    0x10,
    0x13,
    0x9e,
    0x9e,
    0x78,
    0xec,
    0x5c,
    0xe2,
    0xc1,
    0xe7,
    0x16,
    0x9b,
    0x4a,
    0xd4,
    0xf0,
    0x9b,
    0x20,
    0x8a,
    0x32,
    0x19,
    0xfd,
    0xe6,
    0x49,
    0xce,
    0xe7,
    0x12,
    0x4d,
    0x9f,
    0x7c,
    0xbe,
    0x97,
    0xf1,
    0xb1,
    0xb1,
    0x86,
    0x3a,
    0xec,
    0x7b,
    0x40,
    0xd9,
    0x01,
    0x57,
    0x62,
    0x30,
    0xbd,
    0x69,
    0xef,
    0x8f,
    0x6a,
    0xea,
    0xfe,
    0xb2,
    0xb0,
    0x92,
    0x19,
    0xfa,
    0x8f,
    0xaf,
    0x83,
    0x37,
    0x68,
    0x42,
    0xb1,
    0xb2,
    0xaa,
    0x9e,
    0xf6,
    0x8d,
    0x79,
    0xda,
    0xab,
    0x89,
    0xaf,
    0x3f,
    0xab,
    0xe4,
    0x9a,
    0xcc,
    0x27,
    0x86,
    0x38,
    0x70,
    0x73,
    0x45,
    0xbb,
    0xf1,
    0x53,
    0x44,
    0xed,
    0x79,
    0xf7,
    0xf4,
    0x39,
    0x0e,
    0xf8,
    0xac,
    0x50,
    0x9b,
    0x56,
    0xf3,
    0x9a,
    0x98,
    0x56,
    0x65,
    0x27,
    0xa4,
    0x1d,
    0x3c,
    0xbd,
    0x5e,
    0x05,
    0x58,
    0xc1,
    0x59,
    0x92,
    0x7d,
    0xb0,
    0xe8,
    0x84,
    0x54,
    0xa5,
    0xd9,
    0x64,
    0x71,
    0xfd,
    0xdc,
    0xb5,
    0x6d,
    0x5b,
    0xb0,
    0x6b,
    0xfa,
    0x34,
    0x0e,
    0xa7,
    0xa1,
    0x51,
    0xef,
    0x1c,
    0xa6,
    0xfa,
    0x57,
    0x2b,
    0x76,
    0xf3,
    0xb1,
    0xb9,
    0x5d,
    0x8c,
    0x85,
    0x83,
    0xd3,
    0xe4,
    0x77,
    0x05,
    0x36,
    0xb8,
    0x4f,
    0x01,
    0x7e,
    0x70,
    0xe6,
    0xfb,
    0xf1,
    0x76,
    0x60,
    0x1a,
    0x02,
    0x66,
    0x94,
    0x1a,
    0x17,
    0xb0,
    0xc8,
    0xb9,
    0x7f,
    0x4e,
    0x74,
    0xc2,
    0xc1,
    0xff,
    0xc7,
    0x27,
    0x89,
    0x19,
    0x77,
    0x79,
    0x40,
    0xc1,
    0xe1,
    0xff,
    0x1d,
    0x8d,
    0xa6,
    0x37,
    0xd6,
    0xb9,
    0x9d,
    0xda,
    0xfe,
    0x5e,
    0x17,
    0x61,
    0x10,
    0x02,
    0xe2,
    0xc7,
    0x78,
    0xc1,
    0xbe,
    0x8b,
    0x41,
    0xd9,
    0x63,
    0x79,
    0xa5,
    0x13,
    0x60,
    0xd9,
    0x77,
    0xfd,
    0x44,
    0x35,
    0xa1,
    0x1c,
    0x30,
    0x94,
    0x2e,
    0x4b,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_ike_3072 */
static const unsigned char subprime_ike_3072_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xe4,
    0x87,
    0xed,
    0x51,
    0x10,
    0xb4,
    0x61,
    0x1a,
    0x62,
    0x63,
    0x31,
    0x45,
    0xc0,
    0x6e,
    0x0e,
    0x68,
    0x94,
    0x81,
    0x27,
    0x04,
    0x45,
    0x33,
    0xe6,
    0x3a,
    0x01,
    0x05,
    0xdf,
    0x53,
    0x1d,
    0x89,
    0xcd,
    0x91,
    0x28,
    0xa5,
    0x04,
    0x3c,
    0xc7,
    0x1a,
    0x02,
    0x6e,
    0xf7,
    0xca,
    0x8c,
    0xd9,
    0xe6,
    0x9d,
    0x21,
    0x8d,
    0x98,
    0x15,
    0x85,
    0x36,
    0xf9,
    0x2f,
    0x8a,
    0x1b,
    0xa7,
    0xf0,
    0x9a,
    0xb6,
    0xb6,
    0xa8,
    0xe1,
    0x22,
    0xf2,
    0x42,
    0xda,
    0xbb,
    0x31,
    0x2f,
    0x3f,
    0x63,
    0x7a,
    0x26,
    0x21,
    0x74,
    0xd3,
    0x1b,
    0xf6,
    0xb5,
    0x85,
    0xff,
    0xae,
    0x5b,
    0x7a,
    0x03,
    0x5b,
    0xf6,
    0xf7,
    0x1c,
    0x35,
    0xfd,
    0xad,
    0x44,
    0xcf,
    0xd2,
    0xd7,
    0x4f,
    0x92,
    0x08,
    0xbe,
    0x25,
    0x8f,
    0xf3,
    0x24,
    0x94,
    0x33,
    0x28,
    0xf6,
    0x72,
    0x2d,
    0x9e,
    0xe1,
    0x00,
    0x3e,
    0x5c,
    0x50,
    0xb1,
    0xdf,
    0x82,
    0xcc,
    0x6d,
    0x24,
    0x1b,
    0x0e,
    0x2a,
    0xe9,
    0xcd,
    0x34,
    0x8b,
    0x1f,
    0xd4,
    0x7e,
    0x92,
    0x67,
    0xaf,
    0xc1,
    0xb2,
    0xae,
    0x91,
    0xee,
    0x51,
    0xd6,
    0xcb,
    0x0e,
    0x31,
    0x79,
    0xab,
    0x10,
    0x42,
    0xa9,
    0x5d,
    0xcf,
    0x6a,
    0x94,
    0x83,
    0xb8,
    0x4b,
    0x4b,
    0x36,
    0xb3,
    0x86,
    0x1a,
    0xa7,
    0x25,
    0x5e,
    0x4c,
    0x02,
    0x78,
    0xba,
    0x36,
    0x04,
    0x65,
    0x0c,
    0x10,
    0xbe,
    0x19,
    0x48,
    0x2f,
    0x23,
    0x17,
    0x1b,
    0x67,
    0x1d,
    0xf1,
    0xcf,
    0x3b,
    0x96,
    0x0c,
    0x07,
    0x43,
    0x01,
    0xcd,
    0x93,
    0xc1,
    0xd1,
    0x76,
    0x03,
    0xd1,
    0x47,
    0xda,
    0xe2,
    0xae,
    0xf8,
    0x37,
    0xa6,
    0x29,
    0x64,
    0xef,
    0x15,
    0xe5,
    0xfb,
    0x4a,
    0xac,
    0x0b,
    0x8c,
    0x1c,
    0xca,
    0xa4,
    0xbe,
    0x75,
    0x4a,
    0xb5,
    0x72,
    0x8a,
    0xe9,
    0x13,
    0x0c,
    0x4c,
    0x7d,
    0x02,
    0x88,
    0x0a,
    0xb9,
    0x47,
    0x2d,
    0x45,
    0x55,
    0x62,
    0x16,
    0xd6,
    0x99,
    0x8b,
    0x86,
    0x82,
    0x28,
    0x3d,
    0x19,
    0xd4,
    0x2a,
    0x90,
    0xd5,
    0xef,
    0x8e,
    0x5d,
    0x32,
    0x76,
    0x7d,
    0xc2,
    0x82,
    0x2c,
    0x6d,
    0xf7,
    0x85,
    0x45,
    0x75,
    0x38,
    0xab,
    0xae,
    0x83,
    0x06,
    0x3e,
    0xd9,
    0xcb,
    0x87,
    0xc2,
    0xd3,
    0x70,
    0xf2,
    0x63,
    0xd5,
    0xfa,
    0xd7,
    0x46,
    0x6d,
    0x84,
    0x99,
    0xeb,
    0x8f,
    0x46,
    0x4a,
    0x70,
    0x25,
    0x12,
    0xb0,
    0xce,
    0xe7,
    0x71,
    0xe9,
    0x13,
    0x0d,
    0x69,
    0x77,
    0x35,
    0xf8,
    0x97,
    0xfd,
    0x03,
    0x6c,
    0xc5,
    0x04,
    0x32,
    0x6c,
    0x3b,
    0x01,
    0x39,
    0x9f,
    0x64,
    0x35,
    0x32,
    0x29,
    0x0f,
    0x95,
    0x8c,
    0x0b,
    0xbd,
    0x90,
    0x06,
    0x5d,
    0xf0,
    0x8b,
    0xab,
    0xbd,
    0x30,
    0xae,
    0xb6,
    0x3b,
    0x84,
    0xc4,
    0x60,
    0x5d,
    0x6c,
    0xa3,
    0x71,
    0x04,
    0x71,
    0x27,
    0xd0,
    0x3a,
    0x72,
    0xd5,
    0x98,
    0xa1,
    0xed,
    0xad,
    0xfe,
    0x70,
    0x7e,
    0x88,
    0x47,
    0x25,
    0xc1,
    0x68,
    0x90,
    0x54,
    0x9d,
    0x69,
    0x65,
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_tls_3072 */
static const unsigned char subprime_tls_3072_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xd6,
    0xfc,
    0x2a,
    0x2c,
    0x51,
    0x5d,
    0xa5,
    0x4d,
    0x57,
    0xee,
    0x2b,
    0x10,
    0x13,
    0x9e,
    0x9e,
    0x78,
    0xec,
    0x5c,
    0xe2,
    0xc1,
    0xe7,
    0x16,
    0x9b,
    0x4a,
    0xd4,
    0xf0,
    0x9b,
    0x20,
    0x8a,
    0x32,
    0x19,
    0xfd,
    0xe6,
    0x49,
    0xce,
    0xe7,
    0x12,
    0x4d,
    0x9f,
    0x7c,
    0xbe,
    0x97,
    0xf1,
    0xb1,
    0xb1,
    0x86,
    0x3a,
    0xec,
    0x7b,
    0x40,
    0xd9,
    0x01,
    0x57,
    0x62,
    0x30,
    0xbd,
    0x69,
    0xef,
    0x8f,
    0x6a,
    0xea,
    0xfe,
    0xb2,
    0xb0,
    0x92,
    0x19,
    0xfa,
    0x8f,
    0xaf,
    0x83,
    0x37,
    0x68,
    0x42,
    0xb1,
    0xb2,
    0xaa,
    0x9e,
    0xf6,
    0x8d,
    0x79,
    0xda,
    0xab,
    0x89,
    0xaf,
    0x3f,
    0xab,
    0xe4,
    0x9a,
    0xcc,
    0x27,
    0x86,
    0x38,
    0x70,
    0x73,
    0x45,
    0xbb,
    0xf1,
    0x53,
    0x44,
    0xed,
    0x79,
    0xf7,
    0xf4,
    0x39,
    0x0e,
    0xf8,
    0xac,
    0x50,
    0x9b,
    0x56,
    0xf3,
    0x9a,
    0x98,
    0x56,
    0x65,
    0x27,
    0xa4,
    0x1d,
    0x3c,
    0xbd,
    0x5e,
    0x05,
    0x58,
    0xc1,
    0x59,
    0x92,
    0x7d,
    0xb0,
    0xe8,
    0x84,
    0x54,
    0xa5,
    0xd9,
    0x64,
    0x71,
    0xfd,
    0xdc,
    0xb5,
    0x6d,
    0x5b,
    0xb0,
    0x6b,
    0xfa,
    0x34,
    0x0e,
    0xa7,
    0xa1,
    0x51,
    0xef,
    0x1c,
    0xa6,
    0xfa,
    0x57,
    0x2b,
    0x76,
    0xf3,
    0xb1,
    0xb9,
    0x5d,
    0x8c,
    0x85,
    0x83,
    0xd3,
    0xe4,
    0x77,
    0x05,
    0x36,
    0xb8,
    0x4f,
    0x01,
    0x7e,
    0x70,
    0xe6,
    0xfb,
    0xf1,
    0x76,
    0x60,
    0x1a,
    0x02,
    0x66,
    0x94,
    0x1a,
    0x17,
    0xb0,
    0xc8,
    0xb9,
    0x7f,
    0x4e,
    0x74,
    0xc2,
    0xc1,
    0xff,
    0xc7,
    0x27,
    0x89,
    0x19,
    0x77,
    0x79,
    0x40,
    0xc1,
    0xe1,
    0xff,
    0x1d,
    0x8d,
    0xa6,
    0x37,
    0xd6,
    0xb9,
    0x9d,
    0xda,
    0xfe,
    0x5e,
    0x17,
    0x61,
    0x10,
    0x02,
    0xe2,
    0xc7,
    0x78,
    0xc1,
    0xbe,
    0x8b,
    0x41,
    0xd9,
    0x63,
    0x79,
    0xa5,
    0x13,
    0x60,
    0xd9,
    0x77,
    0xfd,
    0x44,
    0x35,
    0xa1,
    0x1c,
    0x30,
    0x8f,
    0xe7,
    0xee,
    0x6f,
    0x1a,
    0xad,
    0x9d,
    0xb2,
    0x8c,
    0x81,
    0xad,
    0xde,
    0x1a,
    0x7a,
    0x6f,
    0x7c,
    0xce,
    0x01,
    0x1c,
    0x30,
    0xda,
    0x37,
    0xe4,
    0xeb,
    0x73,
    0x64,
    0x83,
    0xbd,
    0x6c,
    0x8e,
    0x93,
    0x48,
    0xfb,
    0xfb,
    0xf7,
    0x2c,
    0xc6,
    0x58,
    0x7d,
    0x60,
    0xc3,
    0x6c,
    0x8e,
    0x57,
    0x7f,
    0x09,
    0x84,
    0xc2,
    0x89,
    0xc9,
    0x38,
    0x5a,
    0x09,
    0x86,
    0x49,
    0xde,
    0x21,
    0xbc,
    0xa2,
    0x7a,
    0x7e,
    0xa2,
    0x29,
    0x71,
    0x6b,
    0xa6,
    0xe9,
    0xb2,
    0x79,
    0x71,
    0x0f,
    0x38,
    0xfa,
    0xa5,
    0xff,
    0xae,
    0x57,
    0x41,
    0x55,
    0xce,
    0x4e,
    0xfb,
    0x4f,
    0x74,
    0x36,
    0x95,
    0xe2,
    0x91,
    0x1b,
    0x1d,
    0x06,
    0xd5,
    0xe2,
    0x90,
    0xcb,
    0xcd,
    0x86,
    0xf5,
    0x6d,
    0x0e,
    0xdf,
    0xcd,
    0x21,
    0x6a,
    0xe2,
    0x24,
    0x27,
    0x05,
    0x5e,
    0x68,
    0x35,
    0xfd,
    0x29,
    0xee,
    0xf7,
    0x9e,
    0x0d,
    0x90,
    0x77,
    0x1f,
    0xea,
    0xce,
    0xbe,
    0x12,
    0xf2,
    0x0e,
    0x95,
    0xb3,
    0x63,
    0x17,
    0x1b,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_ike_4096 */
static const unsigned char subprime_ike_4096_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xe4,
    0x87,
    0xed,
    0x51,
    0x10,
    0xb4,
    0x61,
    0x1a,
    0x62,
    0x63,
    0x31,
    0x45,
    0xc0,
    0x6e,
    0x0e,
    0x68,
    0x94,
    0x81,
    0x27,
    0x04,
    0x45,
    0x33,
    0xe6,
    0x3a,
    0x01,
    0x05,
    0xdf,
    0x53,
    0x1d,
    0x89,
    0xcd,
    0x91,
    0x28,
    0xa5,
    0x04,
    0x3c,
    0xc7,
    0x1a,
    0x02,
    0x6e,
    0xf7,
    0xca,
    0x8c,
    0xd9,
    0xe6,
    0x9d,
    0x21,
    0x8d,
    0x98,
    0x15,
    0x85,
    0x36,
    0xf9,
    0x2f,
    0x8a,
    0x1b,
    0xa7,
    0xf0,
    0x9a,
    0xb6,
    0xb6,
    0xa8,
    0xe1,
    0x22,
    0xf2,
    0x42,
    0xda,
    0xbb,
    0x31,
    0x2f,
    0x3f,
    0x63,
    0x7a,
    0x26,
    0x21,
    0x74,
    0xd3,
    0x1b,
    0xf6,
    0xb5,
    0x85,
    0xff,
    0xae,
    0x5b,
    0x7a,
    0x03,
    0x5b,
    0xf6,
    0xf7,
    0x1c,
    0x35,
    0xfd,
    0xad,
    0x44,
    0xcf,
    0xd2,
    0xd7,
    0x4f,
    0x92,
    0x08,
    0xbe,
    0x25,
    0x8f,
    0xf3,
    0x24,
    0x94,
    0x33,
    0x28,
    0xf6,
    0x72,
    0x2d,
    0x9e,
    0xe1,
    0x00,
    0x3e,
    0x5c,
    0x50,
    0xb1,
    0xdf,
    0x82,
    0xcc,
    0x6d,
    0x24,
    0x1b,
    0x0e,
    0x2a,
    0xe9,
    0xcd,
    0x34,
    0x8b,
    0x1f,
    0xd4,
    0x7e,
    0x92,
    0x67,
    0xaf,
    0xc1,
    0xb2,
    0xae,
    0x91,
    0xee,
    0x51,
    0xd6,
    0xcb,
    0x0e,
    0x31,
    0x79,
    0xab,
    0x10,
    0x42,
    0xa9,
    0x5d,
    0xcf,
    0x6a,
    0x94,
    0x83,
    0xb8,
    0x4b,
    0x4b,
    0x36,
    0xb3,
    0x86,
    0x1a,
    0xa7,
    0x25,
    0x5e,
    0x4c,
    0x02,
    0x78,
    0xba,
    0x36,
    0x04,
    0x65,
    0x0c,
    0x10,
    0xbe,
    0x19,
    0x48,
    0x2f,
    0x23,
    0x17,
    0x1b,
    0x67,
    0x1d,
    0xf1,
    0xcf,
    0x3b,
    0x96,
    0x0c,
    0x07,
    0x43,
    0x01,
    0xcd,
    0x93,
    0xc1,
    0xd1,
    0x76,
    0x03,
    0xd1,
    0x47,
    0xda,
    0xe2,
    0xae,
    0xf8,
    0x37,
    0xa6,
    0x29,
    0x64,
    0xef,
    0x15,
    0xe5,
    0xfb,
    0x4a,
    0xac,
    0x0b,
    0x8c,
    0x1c,
    0xca,
    0xa4,
    0xbe,
    0x75,
    0x4a,
    0xb5,
    0x72,
    0x8a,
    0xe9,
    0x13,
    0x0c,
    0x4c,
    0x7d,
    0x02,
    0x88,
    0x0a,
    0xb9,
    0x47,
    0x2d,
    0x45,
    0x55,
    0x62,
    0x16,
    0xd6,
    0x99,
    0x8b,
    0x86,
    0x82,
    0x28,
    0x3d,
    0x19,
    0xd4,
    0x2a,
    0x90,
    0xd5,
    0xef,
    0x8e,
    0x5d,
    0x32,
    0x76,
    0x7d,
    0xc2,
    0x82,
    0x2c,
    0x6d,
    0xf7,
    0x85,
    0x45,
    0x75,
    0x38,
    0xab,
    0xae,
    0x83,
    0x06,
    0x3e,
    0xd9,
    0xcb,
    0x87,
    0xc2,
    0xd3,
    0x70,
    0xf2,
    0x63,
    0xd5,
    0xfa,
    0xd7,
    0x46,
    0x6d,
    0x84,
    0x99,
    0xeb,
    0x8f,
    0x46,
    0x4a,
    0x70,
    0x25,
    0x12,
    0xb0,
    0xce,
    0xe7,
    0x71,
    0xe9,
    0x13,
    0x0d,
    0x69,
    0x77,
    0x35,
    0xf8,
    0x97,
    0xfd,
    0x03,
    0x6c,
    0xc5,
    0x04,
    0x32,
    0x6c,
    0x3b,
    0x01,
    0x39,
    0x9f,
    0x64,
    0x35,
    0x32,
    0x29,
    0x0f,
    0x95,
    0x8c,
    0x0b,
    0xbd,
    0x90,
    0x06,
    0x5d,
    0xf0,
    0x8b,
    0xab,
    0xbd,
    0x30,
    0xae,
    0xb6,
    0x3b,
    0x84,
    0xc4,
    0x60,
    0x5d,
    0x6c,
    0xa3,
    0x71,
    0x04,
    0x71,
    0x27,
    0xd0,
    0x3a,
    0x72,
    0xd5,
    0x98,
    0xa1,
    0xed,
    0xad,
    0xfe,
    0x70,
    0x7e,
    0x88,
    0x47,
    0x25,
    0xc1,
    0x68,
    0x90,
    0x54,
    0x90,
    0x84,
    0x00,
    0x8d,
    0x39,
    0x1e,
    0x09,
    0x53,
    0xc3,
    0xf3,
    0x6b,
    0xc4,
    0x38,
    0xcd,
    0x08,
    0x5e,
    0xdd,
    0x2d,
    0x93,
    0x4c,
    0xe1,
    0x93,
    0x8c,
    0x35,
    0x7a,
    0x71,
    0x1e,
    0x0d,
    0x4a,
    0x34,
    0x1a,
    0x5b,
    0x0a,
    0x85,
    0xed,
    0x12,
    0xc1,
    0xf4,
    0xe5,
    0x15,
    0x6a,
    0x26,
    0x74,
    0x6d,
    0xdd,
    0xe1,
    0x6d,
    0x82,
    0x6f,
    0x47,
    0x7c,
    0x97,
    0x47,
    0x7e,
    0x0a,
    0x0f,
    0xdf,
    0x65,
    0x53,
    0x14,
    0x3e,
    0x2c,
    0xa3,
    0xa7,
    0x35,
    0xe0,
    0x2e,
    0xcc,
    0xd9,
    0x4b,
    0x27,
    0xd0,
    0x48,
    0x61,
    0xd1,
    0x11,
    0x9d,
    0xd0,
    0xc3,
    0x28,
    0xad,
    0xf3,
    0xf6,
    0x8f,
    0xb0,
    0x94,
    0xb8,
    0x67,
    0x71,
    0x6b,
    0xd7,
    0xdc,
    0x0d,
    0xee,
    0xbb,
    0x10,
    0xb8,
    0x24,
    0x0e,
    0x68,
    0x03,
    0x48,
    0x93,
    0xea,
    0xd8,
    0x2d,
    0x54,
    0xc9,
    0xda,
    0x75,
    0x4c,
    0x46,
    0xc7,
    0xee,
    0xe0,
    0xc3,
    0x7f,
    0xdb,
    0xee,
    0x48,
    0x53,
    0x60,
    0x47,
    0xa6,
    0xfa,
    0x1a,
    0xe4,
    0x9a,
    0x03,
    0x18,
    0xcc,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_tls_4096 */
static const unsigned char subprime_tls_4096_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xd6,
    0xfc,
    0x2a,
    0x2c,
    0x51,
    0x5d,
    0xa5,
    0x4d,
    0x57,
    0xee,
    0x2b,
    0x10,
    0x13,
    0x9e,
    0x9e,
    0x78,
    0xec,
    0x5c,
    0xe2,
    0xc1,
    0xe7,
    0x16,
    0x9b,
    0x4a,
    0xd4,
    0xf0,
    0x9b,
    0x20,
    0x8a,
    0x32,
    0x19,
    0xfd,
    0xe6,
    0x49,
    0xce,
    0xe7,
    0x12,
    0x4d,
    0x9f,
    0x7c,
    0xbe,
    0x97,
    0xf1,
    0xb1,
    0xb1,
    0x86,
    0x3a,
    0xec,
    0x7b,
    0x40,
    0xd9,
    0x01,
    0x57,
    0x62,
    0x30,
    0xbd,
    0x69,
    0xef,
    0x8f,
    0x6a,
    0xea,
    0xfe,
    0xb2,
    0xb0,
    0x92,
    0x19,
    0xfa,
    0x8f,
    0xaf,
    0x83,
    0x37,
    0x68,
    0x42,
    0xb1,
    0xb2,
    0xaa,
    0x9e,
    0xf6,
    0x8d,
    0x79,
    0xda,
    0xab,
    0x89,
    0xaf,
    0x3f,
    0xab,
    0xe4,
    0x9a,
    0xcc,
    0x27,
    0x86,
    0x38,
    0x70,
    0x73,
    0x45,
    0xbb,
    0xf1,
    0x53,
    0x44,
    0xed,
    0x79,
    0xf7,
    0xf4,
    0x39,
    0x0e,
    0xf8,
    0xac,
    0x50,
    0x9b,
    0x56,
    0xf3,
    0x9a,
    0x98,
    0x56,
    0x65,
    0x27,
    0xa4,
    0x1d,
    0x3c,
    0xbd,
    0x5e,
    0x05,
    0x58,
    0xc1,
    0x59,
    0x92,
    0x7d,
    0xb0,
    0xe8,
    0x84,
    0x54,
    0xa5,
    0xd9,
    0x64,
    0x71,
    0xfd,
    0xdc,
    0xb5,
    0x6d,
    0x5b,
    0xb0,
    0x6b,
    0xfa,
    0x34,
    0x0e,
    0xa7,
    0xa1,
    0x51,
    0xef,
    0x1c,
    0xa6,
    0xfa,
    0x57,
    0x2b,
    0x76,
    0xf3,
    0xb1,
    0xb9,
    0x5d,
    0x8c,
    0x85,
    0x83,
    0xd3,
    0xe4,
    0x77,
    0x05,
    0x36,
    0xb8,
    0x4f,
    0x01,
    0x7e,
    0x70,
    0xe6,
    0xfb,
    0xf1,
    0x76,
    0x60,
    0x1a,
    0x02,
    0x66,
    0x94,
    0x1a,
    0x17,
    0xb0,
    0xc8,
    0xb9,
    0x7f,
    0x4e,
    0x74,
    0xc2,
    0xc1,
    0xff,
    0xc7,
    0x27,
    0x89,
    0x19,
    0x77,
    0x79,
    0x40,
    0xc1,
    0xe1,
    0xff,
    0x1d,
    0x8d,
    0xa6,
    0x37,
    0xd6,
    0xb9,
    0x9d,
    0xda,
    0xfe,
    0x5e,
    0x17,
    0x61,
    0x10,
    0x02,
    0xe2,
    0xc7,
    0x78,
    0xc1,
    0xbe,
    0x8b,
    0x41,
    0xd9,
    0x63,
    0x79,
    0xa5,
    0x13,
    0x60,
    0xd9,
    0x77,
    0xfd,
    0x44,
    0x35,
    0xa1,
    0x1c,
    0x30,
    0x8f,
    0xe7,
    0xee,
    0x6f,
    0x1a,
    0xad,
    0x9d,
    0xb2,
    0x8c,
    0x81,
    0xad,
    0xde,
    0x1a,
    0x7a,
    0x6f,
    0x7c,
    0xce,
    0x01,
    0x1c,
    0x30,
    0xda,
    0x37,
    0xe4,
    0xeb,
    0x73,
    0x64,
    0x83,
    0xbd,
    0x6c,
    0x8e,
    0x93,
    0x48,
    0xfb,
    0xfb,
    0xf7,
    0x2c,
    0xc6,
    0x58,
    0x7d,
    0x60,
    0xc3,
    0x6c,
    0x8e,
    0x57,
    0x7f,
    0x09,
    0x84,
    0xc2,
    0x89,
    0xc9,
    0x38,
    0x5a,
    0x09,
    0x86,
    0x49,
    0xde,
    0x21,
    0xbc,
    0xa2,
    0x7a,
    0x7e,
    0xa2,
    0x29,
    0x71,
    0x6b,
    0xa6,
    0xe9,
    0xb2,
    0x79,
    0x71,
    0x0f,
    0x38,
    0xfa,
    0xa5,
    0xff,
    0xae,
    0x57,
    0x41,
    0x55,
    0xce,
    0x4e,
    0xfb,
    0x4f,
    0x74,
    0x36,
    0x95,
    0xe2,
    0x91,
    0x1b,
    0x1d,
    0x06,
    0xd5,
    0xe2,
    0x90,
    0xcb,
    0xcd,
    0x86,
    0xf5,
    0x6d,
    0x0e,
    0xdf,
    0xcd,
    0x21,
    0x6a,
    0xe2,
    0x24,
    0x27,
    0x05,
    0x5e,
    0x68,
    0x35,
    0xfd,
    0x29,
    0xee,
    0xf7,
    0x9e,
    0x0d,
    0x90,
    0x77,
    0x1f,
    0xea,
    0xce,
    0xbe,
    0x12,
    0xf2,
    0x0e,
    0x95,
    0xb3,
    0x4f,
    0x0f,
    0x78,
    0xb7,
    0x37,
    0xa9,
    0x61,
    0x8b,
    0x26,
    0xfa,
    0x7d,
    0xbc,
    0x98,
    0x74,
    0xf2,
    0x72,
    0xc4,
    0x2b,
    0xdb,
    0x56,
    0x3e,
    0xaf,
    0xa1,
    0x6b,
    0x4f,
    0xb6,
    0x8c,
    0x3b,
    0xb1,
    0xe7,
    0x8e,
    0xaa,
    0x81,
    0xa0,
    0x02,
    0x43,
    0xfa,
    0xad,
    0xd2,
    0xbf,
    0x18,
    0xe6,
    0x3d,
    0x38,
    0x9a,
    0xe4,
    0x43,
    0x77,
    0xda,
    0x18,
    0xc5,
    0x76,
    0xb5,
    0x0f,
    0x00,
    0x96,
    0xcf,
    0x34,
    0x19,
    0x54,
    0x83,
    0xb0,
    0x05,
    0x48,
    0xc0,
    0x98,
    0x62,
    0x36,
    0xe3,
    0xbc,
    0x7c,
    0xb8,
    0xd6,
    0x80,
    0x1c,
    0x04,
    0x94,
    0xcc,
    0xd1,
    0x99,
    0xe5,
    0xc5,
    0xbd,
    0x0d,
    0x0e,
    0xdc,
    0x9e,
    0xb8,
    0xa0,
    0x00,
    0x1e,
    0x15,
    0x27,
    0x67,
    0x54,
    0xfc,
    0xc6,
    0x85,
    0x66,
    0x05,
    0x41,
    0x48,
    0xe6,
    0xe7,
    0x64,
    0xbe,
    0xe7,
    0xc7,
    0x64,
    0xda,
    0xad,
    0x3f,
    0xc4,
    0x52,
    0x35,
    0xa6,
    0xda,
    0xd4,
    0x28,
    0xfa,
    0x20,
    0xc1,
    0x70,
    0xe3,
    0x45,
    0x00,
    0x3f,
    0x2f,
    0x32,
    0xaf,
    0xb5,
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_ike_6144 */
static const unsigned char subprime_ike_6144_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xe4,
    0x87,
    0xed,
    0x51,
    0x10,
    0xb4,
    0x61,
    0x1a,
    0x62,
    0x63,
    0x31,
    0x45,
    0xc0,
    0x6e,
    0x0e,
    0x68,
    0x94,
    0x81,
    0x27,
    0x04,
    0x45,
    0x33,
    0xe6,
    0x3a,
    0x01,
    0x05,
    0xdf,
    0x53,
    0x1d,
    0x89,
    0xcd,
    0x91,
    0x28,
    0xa5,
    0x04,
    0x3c,
    0xc7,
    0x1a,
    0x02,
    0x6e,
    0xf7,
    0xca,
    0x8c,
    0xd9,
    0xe6,
    0x9d,
    0x21,
    0x8d,
    0x98,
    0x15,
    0x85,
    0x36,
    0xf9,
    0x2f,
    0x8a,
    0x1b,
    0xa7,
    0xf0,
    0x9a,
    0xb6,
    0xb6,
    0xa8,
    0xe1,
    0x22,
    0xf2,
    0x42,
    0xda,
    0xbb,
    0x31,
    0x2f,
    0x3f,
    0x63,
    0x7a,
    0x26,
    0x21,
    0x74,
    0xd3,
    0x1b,
    0xf6,
    0xb5,
    0x85,
    0xff,
    0xae,
    0x5b,
    0x7a,
    0x03,
    0x5b,
    0xf6,
    0xf7,
    0x1c,
    0x35,
    0xfd,
    0xad,
    0x44,
    0xcf,
    0xd2,
    0xd7,
    0x4f,
    0x92,
    0x08,
    0xbe,
    0x25,
    0x8f,
    0xf3,
    0x24,
    0x94,
    0x33,
    0x28,
    0xf6,
    0x72,
    0x2d,
    0x9e,
    0xe1,
    0x00,
    0x3e,
    0x5c,
    0x50,
    0xb1,
    0xdf,
    0x82,
    0xcc,
    0x6d,
    0x24,
    0x1b,
    0x0e,
    0x2a,
    0xe9,
    0xcd,
    0x34,
    0x8b,
    0x1f,
    0xd4,
    0x7e,
    0x92,
    0x67,
    0xaf,
    0xc1,
    0xb2,
    0xae,
    0x91,
    0xee,
    0x51,
    0xd6,
    0xcb,
    0x0e,
    0x31,
    0x79,
    0xab,
    0x10,
    0x42,
    0xa9,
    0x5d,
    0xcf,
    0x6a,
    0x94,
    0x83,
    0xb8,
    0x4b,
    0x4b,
    0x36,
    0xb3,
    0x86,
    0x1a,
    0xa7,
    0x25,
    0x5e,
    0x4c,
    0x02,
    0x78,
    0xba,
    0x36,
    0x04,
    0x65,
    0x0c,
    0x10,
    0xbe,
    0x19,
    0x48,
    0x2f,
    0x23,
    0x17,
    0x1b,
    0x67,
    0x1d,
    0xf1,
    0xcf,
    0x3b,
    0x96,
    0x0c,
    0x07,
    0x43,
    0x01,
    0xcd,
    0x93,
    0xc1,
    0xd1,
    0x76,
    0x03,
    0xd1,
    0x47,
    0xda,
    0xe2,
    0xae,
    0xf8,
    0x37,
    0xa6,
    0x29,
    0x64,
    0xef,
    0x15,
    0xe5,
    0xfb,
    0x4a,
    0xac,
    0x0b,
    0x8c,
    0x1c,
    0xca,
    0xa4,
    0xbe,
    0x75,
    0x4a,
    0xb5,
    0x72,
    0x8a,
    0xe9,
    0x13,
    0x0c,
    0x4c,
    0x7d,
    0x02,
    0x88,
    0x0a,
    0xb9,
    0x47,
    0x2d,
    0x45,
    0x55,
    0x62,
    0x16,
    0xd6,
    0x99,
    0x8b,
    0x86,
    0x82,
    0x28,
    0x3d,
    0x19,
    0xd4,
    0x2a,
    0x90,
    0xd5,
    0xef,
    0x8e,
    0x5d,
    0x32,
    0x76,
    0x7d,
    0xc2,
    0x82,
    0x2c,
    0x6d,
    0xf7,
    0x85,
    0x45,
    0x75,
    0x38,
    0xab,
    0xae,
    0x83,
    0x06,
    0x3e,
    0xd9,
    0xcb,
    0x87,
    0xc2,
    0xd3,
    0x70,
    0xf2,
    0x63,
    0xd5,
    0xfa,
    0xd7,
    0x46,
    0x6d,
    0x84,
    0x99,
    0xeb,
    0x8f,
    0x46,
    0x4a,
    0x70,
    0x25,
    0x12,
    0xb0,
    0xce,
    0xe7,
    0x71,
    0xe9,
    0x13,
    0x0d,
    0x69,
    0x77,
    0x35,
    0xf8,
    0x97,
    0xfd,
    0x03,
    0x6c,
    0xc5,
    0x04,
    0x32,
    0x6c,
    0x3b,
    0x01,
    0x39,
    0x9f,
    0x64,
    0x35,
    0x32,
    0x29,
    0x0f,
    0x95,
    0x8c,
    0x0b,
    0xbd,
    0x90,
    0x06,
    0x5d,
    0xf0,
    0x8b,
    0xab,
    0xbd,
    0x30,
    0xae,
    0xb6,
    0x3b,
    0x84,
    0xc4,
    0x60,
    0x5d,
    0x6c,
    0xa3,
    0x71,
    0x04,
    0x71,
    0x27,
    0xd0,
    0x3a,
    0x72,
    0xd5,
    0x98,
    0xa1,
    0xed,
    0xad,
    0xfe,
    0x70,
    0x7e,
    0x88,
    0x47,
    0x25,
    0xc1,
    0x68,
    0x90,
    0x54,
    0x90,
    0x84,
    0x00,
    0x8d,
    0x39,
    0x1e,
    0x09,
    0x53,
    0xc3,
    0xf3,
    0x6b,
    0xc4,
    0x38,
    0xcd,
    0x08,
    0x5e,
    0xdd,
    0x2d,
    0x93,
    0x4c,
    0xe1,
    0x93,
    0x8c,
    0x35,
    0x7a,
    0x71,
    0x1e,
    0x0d,
    0x4a,
    0x34,
    0x1a,
    0x5b,
    0x0a,
    0x85,
    0xed,
    0x12,
    0xc1,
    0xf4,
    0xe5,
    0x15,
    0x6a,
    0x26,
    0x74,
    0x6d,
    0xdd,
    0xe1,
    0x6d,
    0x82,
    0x6f,
    0x47,
    0x7c,
    0x97,
    0x47,
    0x7e,
    0x0a,
    0x0f,
    0xdf,
    0x65,
    0x53,
    0x14,
    0x3e,
    0x2c,
    0xa3,
    0xa7,
    0x35,
    0xe0,
    0x2e,
    0xcc,
    0xd9,
    0x4b,
    0x27,
    0xd0,
    0x48,
    0x61,
    0xd1,
    0x11,
    0x9d,
    0xd0,
    0xc3,
    0x28,
    0xad,
    0xf3,
    0xf6,
    0x8f,
    0xb0,
    0x94,
    0xb8,
    0x67,
    0x71,
    0x6b,
    0xd7,
    0xdc,
    0x0d,
    0xee,
    0xbb,
    0x10,
    0xb8,
    0x24,
    0x0e,
    0x68,
    0x03,
    0x48,
    0x93,
    0xea,
    0xd8,
    0x2d,
    0x54,
    0xc9,
    0xda,
    0x75,
    0x4c,
    0x46,
    0xc7,
    0xee,
    0xe0,
    0xc3,
    0x7f,
    0xdb,
    0xee,
    0x48,
    0x53,
    0x60,
    0x47,
    0xa6,
    0xfa,
    0x1a,
    0xe4,
    0x9a,
    0x01,
    0x42,
    0x49,
    0x1b,
    0x61,
    0xfd,
    0x5a,
    0x69,
    0x3e,
    0x38,
    0x13,
    0x60,
    0xea,
    0x6e,
    0x59,
    0x30,
    0x13,
    0x23,
    0x6f,
    0x64,
    0xba,
    0x8f,
    0x3b,
    0x1e,
    0xdd,
    0x1b,
    0xde,
    0xfc,
    0x7f,
    0xca,
    0x03,
    0x56,
    0xcf,
    0x29,
    0x87,
    0x72,
    0xed,
    0x9c,
    0x17,
    0xa0,
    0x98,
    0x00,
    0xd7,
    0x58,
    0x35,
    0x29,
    0xf6,
    0xc8,
    0x13,
    0xec,
    0x18,
    0x8b,
    0xcb,
    0x93,
    0xd8,
    0x43,
    0x2d,
    0x44,
    0x8c,
    0x6d,
    0x1f,
    0x6d,
    0xf5,
    0xe7,
    0xcd,
    0x8a,
    0x76,
    0xa2,
    0x67,
    0x36,
    0x5d,
    0x67,
    0x6a,
    0x5d,
    0x8d,
    0xed,
    0xbf,
    0x8a,
    0x23,
    0xf3,
    0x66,
    0x12,
    0xa5,
    0x99,
    0x90,
    0x28,
    0xa8,
    0x95,
    0xeb,
    0xd7,
    0xa1,
    0x37,
    0xdc,
    0x7a,
    0x00,
    0x9b,
    0xc6,
    0x69,
    0x5f,
    0xac,
    0xc1,
    0xe5,
    0x00,
    0xe3,
    0x25,
    0xc9,
    0x76,
    0x78,
    0x19,
    0x75,
    0x0a,
    0xe8,
    0xb9,
    0x0e,
    0x81,
    0xfa,
    0x41,
    0x6b,
    0xe7,
    0x37,
    0x3a,
    0x7f,
    0x7b,
    0x6a,
    0xaf,
    0x38,
    0x17,
    0xa3,
    0x4c,
    0x06,
    0x41,
    0x5a,
    0xd4,
    0x20,
    0x18,
    0xc8,
    0x05,
    0x8e,
    0x4f,
    0x2c,
    0xf3,
    0xe4,
    0xbf,
    0xdf,
    0x63,
    0xf4,
    0x79,
    0x91,
    0xd4,
    0xbd,
    0x3f,
    0x1b,
    0x66,
    0x44,
    0x5f,
    0x07,
    0x8e,
    0xa2,
    0xdb,
    0xff,
    0xac,
    0x2d,
    0x62,
    0xa5,
    0xea,
    0x03,
    0xd9,
    0x15,
    0xa0,
    0xaa,
    0x55,
    0x66,
    0x47,
    0xb6,
    0xbf,
    0x5f,
    0xa4,
    0x70,
    0xec,
    0x0a,
    0x66,
    0x2f,
    0x69,
    0x07,
    0xc0,
    0x1b,
    0xf0,
    0x53,
    0xcb,
    0x8a,
    0xf7,
    0x79,
    0x4d,
    0xf1,
    0x94,
    0x03,
    0x50,
    0xea,
    0xc5,
    0xdb,
    0xe2,
    0xed,
    0x3b,
    0x7a,
    0xa8,
    0x55,
    0x1e,
    0xc5,
    0x0f,
    0xdf,
    0xf8,
    0x75,
    0x8c,
    0xe6,
    0x58,
    0xd1,
    0x89,
    0xea,
    0xae,
    0x6d,
    0x2b,
    0x64,
    0xf6,
    0x17,
    0x79,
    0x4b,
    0x19,
    0x1c,
    0x3f,
    0xf4,
    0x6b,
    0xb7,
    0x1e,
    0x02,
    0x34,
    0x02,
    0x1f,
    0x47,
    0xb3,
    0x1f,
    0xa4,
    0x30,
    0x77,
    0x09,
    0x5f,
    0x96,
    0xad,
    0x85,
    0xba,
    0x3a,
    0x6b,
    0x73,
    0x4a,
    0x7c,
    0x8f,
    0x36,
    0xe6,
    0x20,
    0x12,
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_tls_6144 */
static const unsigned char subprime_tls_6144_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xd6,
    0xfc,
    0x2a,
    0x2c,
    0x51,
    0x5d,
    0xa5,
    0x4d,
    0x57,
    0xee,
    0x2b,
    0x10,
    0x13,
    0x9e,
    0x9e,
    0x78,
    0xec,
    0x5c,
    0xe2,
    0xc1,
    0xe7,
    0x16,
    0x9b,
    0x4a,
    0xd4,
    0xf0,
    0x9b,
    0x20,
    0x8a,
    0x32,
    0x19,
    0xfd,
    0xe6,
    0x49,
    0xce,
    0xe7,
    0x12,
    0x4d,
    0x9f,
    0x7c,
    0xbe,
    0x97,
    0xf1,
    0xb1,
    0xb1,
    0x86,
    0x3a,
    0xec,
    0x7b,
    0x40,
    0xd9,
    0x01,
    0x57,
    0x62,
    0x30,
    0xbd,
    0x69,
    0xef,
    0x8f,
    0x6a,
    0xea,
    0xfe,
    0xb2,
    0xb0,
    0x92,
    0x19,
    0xfa,
    0x8f,
    0xaf,
    0x83,
    0x37,
    0x68,
    0x42,
    0xb1,
    0xb2,
    0xaa,
    0x9e,
    0xf6,
    0x8d,
    0x79,
    0xda,
    0xab,
    0x89,
    0xaf,
    0x3f,
    0xab,
    0xe4,
    0x9a,
    0xcc,
    0x27,
    0x86,
    0x38,
    0x70,
    0x73,
    0x45,
    0xbb,
    0xf1,
    0x53,
    0x44,
    0xed,
    0x79,
    0xf7,
    0xf4,
    0x39,
    0x0e,
    0xf8,
    0xac,
    0x50,
    0x9b,
    0x56,
    0xf3,
    0x9a,
    0x98,
    0x56,
    0x65,
    0x27,
    0xa4,
    0x1d,
    0x3c,
    0xbd,
    0x5e,
    0x05,
    0x58,
    0xc1,
    0x59,
    0x92,
    0x7d,
    0xb0,
    0xe8,
    0x84,
    0x54,
    0xa5,
    0xd9,
    0x64,
    0x71,
    0xfd,
    0xdc,
    0xb5,
    0x6d,
    0x5b,
    0xb0,
    0x6b,
    0xfa,
    0x34,
    0x0e,
    0xa7,
    0xa1,
    0x51,
    0xef,
    0x1c,
    0xa6,
    0xfa,
    0x57,
    0x2b,
    0x76,
    0xf3,
    0xb1,
    0xb9,
    0x5d,
    0x8c,
    0x85,
    0x83,
    0xd3,
    0xe4,
    0x77,
    0x05,
    0x36,
    0xb8,
    0x4f,
    0x01,
    0x7e,
    0x70,
    0xe6,
    0xfb,
    0xf1,
    0x76,
    0x60,
    0x1a,
    0x02,
    0x66,
    0x94,
    0x1a,
    0x17,
    0xb0,
    0xc8,
    0xb9,
    0x7f,
    0x4e,
    0x74,
    0xc2,
    0xc1,
    0xff,
    0xc7,
    0x27,
    0x89,
    0x19,
    0x77,
    0x79,
    0x40,
    0xc1,
    0xe1,
    0xff,
    0x1d,
    0x8d,
    0xa6,
    0x37,
    0xd6,
    0xb9,
    0x9d,
    0xda,
    0xfe,
    0x5e,
    0x17,
    0x61,
    0x10,
    0x02,
    0xe2,
    0xc7,
    0x78,
    0xc1,
    0xbe,
    0x8b,
    0x41,
    0xd9,
    0x63,
    0x79,
    0xa5,
    0x13,
    0x60,
    0xd9,
    0x77,
    0xfd,
    0x44,
    0x35,
    0xa1,
    0x1c,
    0x30,
    0x8f,
    0xe7,
    0xee,
    0x6f,
    0x1a,
    0xad,
    0x9d,
    0xb2,
    0x8c,
    0x81,
    0xad,
    0xde,
    0x1a,
    0x7a,
    0x6f,
    0x7c,
    0xce,
    0x01,
    0x1c,
    0x30,
    0xda,
    0x37,
    0xe4,
    0xeb,
    0x73,
    0x64,
    0x83,
    0xbd,
    0x6c,
    0x8e,
    0x93,
    0x48,
    0xfb,
    0xfb,
    0xf7,
    0x2c,
    0xc6,
    0x58,
    0x7d,
    0x60,
    0xc3,
    0x6c,
    0x8e,
    0x57,
    0x7f,
    0x09,
    0x84,
    0xc2,
    0x89,
    0xc9,
    0x38,
    0x5a,
    0x09,
    0x86,
    0x49,
    0xde,
    0x21,
    0xbc,
    0xa2,
    0x7a,
    0x7e,
    0xa2,
    0x29,
    0x71,
    0x6b,
    0xa6,
    0xe9,
    0xb2,
    0x79,
    0x71,
    0x0f,
    0x38,
    0xfa,
    0xa5,
    0xff,
    0xae,
    0x57,
    0x41,
    0x55,
    0xce,
    0x4e,
    0xfb,
    0x4f,
    0x74,
    0x36,
    0x95,
    0xe2,
    0x91,
    0x1b,
    0x1d,
    0x06,
    0xd5,
    0xe2,
    0x90,
    0xcb,
    0xcd,
    0x86,
    0xf5,
    0x6d,
    0x0e,
    0xdf,
    0xcd,
    0x21,
    0x6a,
    0xe2,
    0x24,
    0x27,
    0x05,
    0x5e,
    0x68,
    0x35,
    0xfd,
    0x29,
    0xee,
    0xf7,
    0x9e,
    0x0d,
    0x90,
    0x77,
    0x1f,
    0xea,
    0xce,
    0xbe,
    0x12,
    0xf2,
    0x0e,
    0x95,
    0xb3,
    0x4f,
    0x0f,
    0x78,
    0xb7,
    0x37,
    0xa9,
    0x61,
    0x8b,
    0x26,
    0xfa,
    0x7d,
    0xbc,
    0x98,
    0x74,
    0xf2,
    0x72,
    0xc4,
    0x2b,
    0xdb,
    0x56,
    0x3e,
    0xaf,
    0xa1,
    0x6b,
    0x4f,
    0xb6,
    0x8c,
    0x3b,
    0xb1,
    0xe7,
    0x8e,
    0xaa,
    0x81,
    0xa0,
    0x02,
    0x43,
    0xfa,
    0xad,
    0xd2,
    0xbf,
    0x18,
    0xe6,
    0x3d,
    0x38,
    0x9a,
    0xe4,
    0x43,
    0x77,
    0xda,
    0x18,
    0xc5,
    0x76,
    0xb5,
    0x0f,
    0x00,
    0x96,
    0xcf,
    0x34,
    0x19,
    0x54,
    0x83,
    0xb0,
    0x05,
    0x48,
    0xc0,
    0x98,
    0x62,
    0x36,
    0xe3,
    0xbc,
    0x7c,
    0xb8,
    0xd6,
    0x80,
    0x1c,
    0x04,
    0x94,
    0xcc,
    0xd1,
    0x99,
    0xe5,
    0xc5,
    0xbd,
    0x0d,
    0x0e,
    0xdc,
    0x9e,
    0xb8,
    0xa0,
    0x00,
    0x1e,
    0x15,
    0x27,
    0x67,
    0x54,
    0xfc,
    0xc6,
    0x85,
    0x66,
    0x05,
    0x41,
    0x48,
    0xe6,
    0xe7,
    0x64,
    0xbe,
    0xe7,
    0xc7,
    0x64,
    0xda,
    0xad,
    0x3f,
    0xc4,
    0x52,
    0x35,
    0xa6,
    0xda,
    0xd4,
    0x28,
    0xfa,
    0x20,
    0xc1,
    0x70,
    0xe3,
    0x45,
    0x00,
    0x3f,
    0x2f,
    0x06,
    0xec,
    0x81,
    0x05,
    0xfe,
    0xb2,
    0x5b,
    0x22,
    0x81,
    0xb6,
    0x3d,
    0x27,
    0x33,
    0xbe,
    0x96,
    0x1c,
    0x29,
    0x95,
    0x1d,
    0x11,
    0xdd,
    0x22,
    0x21,
    0x65,
    0x7a,
    0x9f,
    0x53,
    0x1d,
    0xda,
    0x2a,
    0x19,
    0x4d,
    0xbb,
    0x12,
    0x64,
    0x48,
    0xbd,
    0xee,
    0xb2,
    0x58,
    0xe0,
    0x7e,
    0xa6,
    0x59,
    0xc7,
    0x46,
    0x19,
    0xa6,
    0x38,
    0x0e,
    0x1d,
    0x66,
    0xd6,
    0x83,
    0x2b,
    0xfe,
    0x67,
    0xf6,
    0x38,
    0xcd,
    0x8f,
    0xae,
    0x1f,
    0x27,
    0x23,
    0x02,
    0x0f,
    0x9c,
    0x40,
    0xa3,
    0xfd,
    0xa6,
    0x7e,
    0xda,
    0x3b,
    0xd2,
    0x92,
    0x38,
    0xfb,
    0xd4,
    0xd4,
    0xb4,
    0x88,
    0x5c,
    0x2a,
    0x99,
    0x17,
    0x6d,
    0xb1,
    0xa0,
    0x6c,
    0x50,
    0x07,
    0x78,
    0x49,
    0x1a,
    0x82,
    0x88,
    0xf1,
    0x85,
    0x5f,
    0x60,
    0xff,
    0xfc,
    0xf1,
    0xd1,
    0x37,
    0x3f,
    0xd9,
    0x4f,
    0xc6,
    0x0c,
    0x18,
    0x11,
    0xe1,
    0xac,
    0x3f,
    0x1c,
    0x6d,
    0x00,
    0x3b,
    0xec,
    0xda,
    0x3b,
    0x1f,
    0x27,
    0x25,
    0xca,
    0x59,
    0x5d,
    0xe0,
    0xca,
    0x63,
    0x32,
    0x8f,
    0x3b,
    0xe5,
    0x7c,
    0xc9,
    0x77,
    0x55,
    0x60,
    0x11,
    0x95,
    0x14,
    0x0d,
    0xfb,
    0x59,
    0xd3,
    0x9c,
    0xe0,
    0x91,
    0x30,
    0x8b,
    0x41,
    0x05,
    0x74,
    0x6d,
    0xac,
    0x23,
    0xd3,
    0x3e,
    0x5f,
    0x7c,
    0xe4,
    0x84,
    0x8d,
    0xa3,
    0x16,
    0xa9,
    0xc6,
    0x6b,
    0x95,
    0x81,
    0xba,
    0x35,
    0x73,
    0xbf,
    0xaf,
    0x31,
    0x14,
    0x96,
    0x18,
    0x8a,
    0xb1,
    0x54,
    0x23,
    0x28,
    0x2e,
    0xe4,
    0x16,
    0xdc,
    0x2a,
    0x19,
    0xc5,
    0x72,
    0x4f,
    0xa9,
    0x1a,
    0xe4,
    0xad,
    0xc8,
    0x8b,
    0xc6,
    0x67,
    0x96,
    0xea,
    0xe5,
    0x67,
    0x7a,
    0x01,
    0xf6,
    0x4e,
    0x8c,
    0x08,
    0x63,
    0x13,
    0x95,
    0x82,
    0x2d,
    0x9d,
    0xb8,
    0xfc,
    0xee,
    0x35,
    0xc0,
    0x6b,
    0x1f,
    0xee,
    0xa5,
    0x47,
    0x4d,
    0x6d,
    0x8f,
    0x34,
    0xb1,
    0x53,
    0x4a,
    0x93,
    0x6a,
    0x18,
    0xb0,
    0xe0,
    0xd2,
    0x0e,
    0xab,
    0x86,
    0xbc,
    0x9c,
    0x6d,
    0x6a,
    0x52,
    0x07,
    0x19,
    0x4e,
    0x68,
    0x72,
    0x07,
    0x32,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_ike_8192 */
static const unsigned char subprime_ike_8192_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xe4,
    0x87,
    0xed,
    0x51,
    0x10,
    0xb4,
    0x61,
    0x1a,
    0x62,
    0x63,
    0x31,
    0x45,
    0xc0,
    0x6e,
    0x0e,
    0x68,
    0x94,
    0x81,
    0x27,
    0x04,
    0x45,
    0x33,
    0xe6,
    0x3a,
    0x01,
    0x05,
    0xdf,
    0x53,
    0x1d,
    0x89,
    0xcd,
    0x91,
    0x28,
    0xa5,
    0x04,
    0x3c,
    0xc7,
    0x1a,
    0x02,
    0x6e,
    0xf7,
    0xca,
    0x8c,
    0xd9,
    0xe6,
    0x9d,
    0x21,
    0x8d,
    0x98,
    0x15,
    0x85,
    0x36,
    0xf9,
    0x2f,
    0x8a,
    0x1b,
    0xa7,
    0xf0,
    0x9a,
    0xb6,
    0xb6,
    0xa8,
    0xe1,
    0x22,
    0xf2,
    0x42,
    0xda,
    0xbb,
    0x31,
    0x2f,
    0x3f,
    0x63,
    0x7a,
    0x26,
    0x21,
    0x74,
    0xd3,
    0x1b,
    0xf6,
    0xb5,
    0x85,
    0xff,
    0xae,
    0x5b,
    0x7a,
    0x03,
    0x5b,
    0xf6,
    0xf7,
    0x1c,
    0x35,
    0xfd,
    0xad,
    0x44,
    0xcf,
    0xd2,
    0xd7,
    0x4f,
    0x92,
    0x08,
    0xbe,
    0x25,
    0x8f,
    0xf3,
    0x24,
    0x94,
    0x33,
    0x28,
    0xf6,
    0x72,
    0x2d,
    0x9e,
    0xe1,
    0x00,
    0x3e,
    0x5c,
    0x50,
    0xb1,
    0xdf,
    0x82,
    0xcc,
    0x6d,
    0x24,
    0x1b,
    0x0e,
    0x2a,
    0xe9,
    0xcd,
    0x34,
    0x8b,
    0x1f,
    0xd4,
    0x7e,
    0x92,
    0x67,
    0xaf,
    0xc1,
    0xb2,
    0xae,
    0x91,
    0xee,
    0x51,
    0xd6,
    0xcb,
    0x0e,
    0x31,
    0x79,
    0xab,
    0x10,
    0x42,
    0xa9,
    0x5d,
    0xcf,
    0x6a,
    0x94,
    0x83,
    0xb8,
    0x4b,
    0x4b,
    0x36,
    0xb3,
    0x86,
    0x1a,
    0xa7,
    0x25,
    0x5e,
    0x4c,
    0x02,
    0x78,
    0xba,
    0x36,
    0x04,
    0x65,
    0x0c,
    0x10,
    0xbe,
    0x19,
    0x48,
    0x2f,
    0x23,
    0x17,
    0x1b,
    0x67,
    0x1d,
    0xf1,
    0xcf,
    0x3b,
    0x96,
    0x0c,
    0x07,
    0x43,
    0x01,
    0xcd,
    0x93,
    0xc1,
    0xd1,
    0x76,
    0x03,
    0xd1,
    0x47,
    0xda,
    0xe2,
    0xae,
    0xf8,
    0x37,
    0xa6,
    0x29,
    0x64,
    0xef,
    0x15,
    0xe5,
    0xfb,
    0x4a,
    0xac,
    0x0b,
    0x8c,
    0x1c,
    0xca,
    0xa4,
    0xbe,
    0x75,
    0x4a,
    0xb5,
    0x72,
    0x8a,
    0xe9,
    0x13,
    0x0c,
    0x4c,
    0x7d,
    0x02,
    0x88,
    0x0a,
    0xb9,
    0x47,
    0x2d,
    0x45,
    0x55,
    0x62,
    0x16,
    0xd6,
    0x99,
    0x8b,
    0x86,
    0x82,
    0x28,
    0x3d,
    0x19,
    0xd4,
    0x2a,
    0x90,
    0xd5,
    0xef,
    0x8e,
    0x5d,
    0x32,
    0x76,
    0x7d,
    0xc2,
    0x82,
    0x2c,
    0x6d,
    0xf7,
    0x85,
    0x45,
    0x75,
    0x38,
    0xab,
    0xae,
    0x83,
    0x06,
    0x3e,
    0xd9,
    0xcb,
    0x87,
    0xc2,
    0xd3,
    0x70,
    0xf2,
    0x63,
    0xd5,
    0xfa,
    0xd7,
    0x46,
    0x6d,
    0x84,
    0x99,
    0xeb,
    0x8f,
    0x46,
    0x4a,
    0x70,
    0x25,
    0x12,
    0xb0,
    0xce,
    0xe7,
    0x71,
    0xe9,
    0x13,
    0x0d,
    0x69,
    0x77,
    0x35,
    0xf8,
    0x97,
    0xfd,
    0x03,
    0x6c,
    0xc5,
    0x04,
    0x32,
    0x6c,
    0x3b,
    0x01,
    0x39,
    0x9f,
    0x64,
    0x35,
    0x32,
    0x29,
    0x0f,
    0x95,
    0x8c,
    0x0b,
    0xbd,
    0x90,
    0x06,
    0x5d,
    0xf0,
    0x8b,
    0xab,
    0xbd,
    0x30,
    0xae,
    0xb6,
    0x3b,
    0x84,
    0xc4,
    0x60,
    0x5d,
    0x6c,
    0xa3,
    0x71,
    0x04,
    0x71,
    0x27,
    0xd0,
    0x3a,
    0x72,
    0xd5,
    0x98,
    0xa1,
    0xed,
    0xad,
    0xfe,
    0x70,
    0x7e,
    0x88,
    0x47,
    0x25,
    0xc1,
    0x68,
    0x90,
    0x54,
    0x90,
    0x84,
    0x00,
    0x8d,
    0x39,
    0x1e,
    0x09,
    0x53,
    0xc3,
    0xf3,
    0x6b,
    0xc4,
    0x38,
    0xcd,
    0x08,
    0x5e,
    0xdd,
    0x2d,
    0x93,
    0x4c,
    0xe1,
    0x93,
    0x8c,
    0x35,
    0x7a,
    0x71,
    0x1e,
    0x0d,
    0x4a,
    0x34,
    0x1a,
    0x5b,
    0x0a,
    0x85,
    0xed,
    0x12,
    0xc1,
    0xf4,
    0xe5,
    0x15,
    0x6a,
    0x26,
    0x74,
    0x6d,
    0xdd,
    0xe1,
    0x6d,
    0x82,
    0x6f,
    0x47,
    0x7c,
    0x97,
    0x47,
    0x7e,
    0x0a,
    0x0f,
    0xdf,
    0x65,
    0x53,
    0x14,
    0x3e,
    0x2c,
    0xa3,
    0xa7,
    0x35,
    0xe0,
    0x2e,
    0xcc,
    0xd9,
    0x4b,
    0x27,
    0xd0,
    0x48,
    0x61,
    0xd1,
    0x11,
    0x9d,
    0xd0,
    0xc3,
    0x28,
    0xad,
    0xf3,
    0xf6,
    0x8f,
    0xb0,
    0x94,
    0xb8,
    0x67,
    0x71,
    0x6b,
    0xd7,
    0xdc,
    0x0d,
    0xee,
    0xbb,
    0x10,
    0xb8,
    0x24,
    0x0e,
    0x68,
    0x03,
    0x48,
    0x93,
    0xea,
    0xd8,
    0x2d,
    0x54,
    0xc9,
    0xda,
    0x75,
    0x4c,
    0x46,
    0xc7,
    0xee,
    0xe0,
    0xc3,
    0x7f,
    0xdb,
    0xee,
    0x48,
    0x53,
    0x60,
    0x47,
    0xa6,
    0xfa,
    0x1a,
    0xe4,
    0x9a,
    0x01,
    0x42,
    0x49,
    0x1b,
    0x61,
    0xfd,
    0x5a,
    0x69,
    0x3e,
    0x38,
    0x13,
    0x60,
    0xea,
    0x6e,
    0x59,
    0x30,
    0x13,
    0x23,
    0x6f,
    0x64,
    0xba,
    0x8f,
    0x3b,
    0x1e,
    0xdd,
    0x1b,
    0xde,
    0xfc,
    0x7f,
    0xca,
    0x03,
    0x56,
    0xcf,
    0x29,
    0x87,
    0x72,
    0xed,
    0x9c,
    0x17,
    0xa0,
    0x98,
    0x00,
    0xd7,
    0x58,
    0x35,
    0x29,
    0xf6,
    0xc8,
    0x13,
    0xec,
    0x18,
    0x8b,
    0xcb,
    0x93,
    0xd8,
    0x43,
    0x2d,
    0x44,
    0x8c,
    0x6d,
    0x1f,
    0x6d,
    0xf5,
    0xe7,
    0xcd,
    0x8a,
    0x76,
    0xa2,
    0x67,
    0x36,
    0x5d,
    0x67,
    0x6a,
    0x5d,
    0x8d,
    0xed,
    0xbf,
    0x8a,
    0x23,
    0xf3,
    0x66,
    0x12,
    0xa5,
    0x99,
    0x90,
    0x28,
    0xa8,
    0x95,
    0xeb,
    0xd7,
    0xa1,
    0x37,
    0xdc,
    0x7a,
    0x00,
    0x9b,
    0xc6,
    0x69,
    0x5f,
    0xac,
    0xc1,
    0xe5,
    0x00,
    0xe3,
    0x25,
    0xc9,
    0x76,
    0x78,
    0x19,
    0x75,
    0x0a,
    0xe8,
    0xb9,
    0x0e,
    0x81,
    0xfa,
    0x41,
    0x6b,
    0xe7,
    0x37,
    0x3a,
    0x7f,
    0x7b,
    0x6a,
    0xaf,
    0x38,
    0x17,
    0xa3,
    0x4c,
    0x06,
    0x41,
    0x5a,
    0xd4,
    0x20,
    0x18,
    0xc8,
    0x05,
    0x8e,
    0x4f,
    0x2c,
    0xf3,
    0xe4,
    0xbf,
    0xdf,
    0x63,
    0xf4,
    0x79,
    0x91,
    0xd4,
    0xbd,
    0x3f,
    0x1b,
    0x66,
    0x44,
    0x5f,
    0x07,
    0x8e,
    0xa2,
    0xdb,
    0xff,
    0xac,
    0x2d,
    0x62,
    0xa5,
    0xea,
    0x03,
    0xd9,
    0x15,
    0xa0,
    0xaa,
    0x55,
    0x66,
    0x47,
    0xb6,
    0xbf,
    0x5f,
    0xa4,
    0x70,
    0xec,
    0x0a,
    0x66,
    0x2f,
    0x69,
    0x07,
    0xc0,
    0x1b,
    0xf0,
    0x53,
    0xcb,
    0x8a,
    0xf7,
    0x79,
    0x4d,
    0xf1,
    0x94,
    0x03,
    0x50,
    0xea,
    0xc5,
    0xdb,
    0xe2,
    0xed,
    0x3b,
    0x7a,
    0xa8,
    0x55,
    0x1e,
    0xc5,
    0x0f,
    0xdf,
    0xf8,
    0x75,
    0x8c,
    0xe6,
    0x58,
    0xd1,
    0x89,
    0xea,
    0xae,
    0x6d,
    0x2b,
    0x64,
    0xf6,
    0x17,
    0x79,
    0x4b,
    0x19,
    0x1c,
    0x3f,
    0xf4,
    0x6b,
    0xb7,
    0x1e,
    0x02,
    0x34,
    0x02,
    0x1f,
    0x47,
    0xb3,
    0x1f,
    0xa4,
    0x30,
    0x77,
    0x09,
    0x5f,
    0x96,
    0xad,
    0x85,
    0xba,
    0x3a,
    0x6b,
    0x73,
    0x4a,
    0x7c,
    0x8f,
    0x36,
    0xdf,
    0x08,
    0xac,
    0xba,
    0x51,
    0xc9,
    0x37,
    0x89,
    0x7f,
    0x72,
    0xf2,
    0x1c,
    0x3b,
    0xbe,
    0x5b,
    0x54,
    0x99,
    0x6f,
    0xc6,
    0x6c,
    0x5f,
    0x62,
    0x68,
    0x39,
    0xdc,
    0x98,
    0xdd,
    0x1d,
    0xe4,
    0x19,
    0x5b,
    0x46,
    0xce,
    0xe9,
    0x80,
    0x3a,
    0x0f,
    0xd3,
    0xdf,
    0xc5,
    0x7e,
    0x23,
    0xf6,
    0x92,
    0xbb,
    0x7b,
    0x49,
    0xb5,
    0xd2,
    0x12,
    0x33,
    0x1d,
    0x55,
    0xb1,
    0xce,
    0x2d,
    0x72,
    0x7a,
    0xb4,
    0x1a,
    0x11,
    0xda,
    0x3a,
    0x15,
    0xf8,
    0xe4,
    0xbc,
    0x11,
    0xc7,
    0x8b,
    0x65,
    0xf1,
    0xce,
    0xb2,
    0x96,
    0xf1,
    0xfe,
    0xdc,
    0x5f,
    0x7e,
    0x42,
    0x45,
    0x6c,
    0x91,
    0x11,
    0x17,
    0x02,
    0x52,
    0x01,
    0xbe,
    0x03,
    0x89,
    0xf5,
    0xab,
    0xd4,
    0x0d,
    0x11,
    0xf8,
    0x63,
    0x9a,
    0x39,
    0xfe,
    0x32,
    0x36,
    0x75,
    0x18,
    0x35,
    0xa5,
    0xe5,
    0xe4,
    0x43,
    0x17,
    0xc1,
    0xc2,
    0xee,
    0xfd,
    0x4e,
    0xa5,
    0xbf,
    0xd1,
    0x60,
    0x43,
    0xf4,
    0x3c,
    0xb4,
    0x19,
    0x81,
    0xf6,
    0xad,
    0xee,
    0x9d,
    0x03,
    0x15,
    0x9e,
    0x7a,
    0xd9,
    0xd1,
    0x3c,
    0x53,
    0x36,
    0x95,
    0x09,
    0xfc,
    0x1f,
    0xa2,
    0x7c,
    0x16,
    0xef,
    0x98,
    0x87,
    0x70,
    0x3a,
    0x55,
    0xb5,
    0x1b,
    0x22,
    0xcb,
    0xf4,
    0x4c,
    0xd0,
    0x12,
    0xae,
    0xe0,
    0xb2,
    0x79,
    0x8e,
    0x62,
    0x84,
    0x23,
    0x42,
    0x8e,
    0xfc,
    0xd5,
    0xa4,
    0x0c,
    0xae,
    0xf6,
    0xbf,
    0x50,
    0xd8,
    0xea,
    0x88,
    0x5e,
    0xbf,
    0x73,
    0xa6,
    0xb9,
    0xfd,
    0x79,
    0xb5,
    0xe1,
    0x8f,
    0x67,
    0xd1,
    0x34,
    0x1a,
    0xc8,
    0x23,
    0x7a,
    0x75,
    0xc3,
    0xcf,
    0xc9,
    0x20,
    0x04,
    0xa1,
    0xc5,
    0xa4,
    0x0e,
    0x36,
    0x6b,
    0xc4,
    0x4d,
    0x00,
    0x17,
    0x6a,
    0xf7,
    0x1c,
    0x15,
    0xe4,
    0x8c,
    0x86,
    0xd3,
    0x7e,
    0x01,
    0x37,
    0x23,
    0xca,
    0xac,
    0x72,
    0x23,
    0xab,
    0x3b,
    0xf4,
    0xd5,
    0x4f,
    0x18,
    0x28,
    0x71,
    0x3b,
    0x2b,
    0x4a,
    0x6f,
    0xe4,
    0x0f,
    0xab,
    0x74,
    0x40,
    0x5c,
    0xb7,
    0x38,
    0xb0,
    0x64,
    0xc0,
    0x6e,
    0xcc,
    0x76,
    0xe9,
    0xef,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

/* q=(p-1)/2 for prime prime_tls_8192 */
static const unsigned char subprime_tls_8192_data[] = {
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xd6,
    0xfc,
    0x2a,
    0x2c,
    0x51,
    0x5d,
    0xa5,
    0x4d,
    0x57,
    0xee,
    0x2b,
    0x10,
    0x13,
    0x9e,
    0x9e,
    0x78,
    0xec,
    0x5c,
    0xe2,
    0xc1,
    0xe7,
    0x16,
    0x9b,
    0x4a,
    0xd4,
    0xf0,
    0x9b,
    0x20,
    0x8a,
    0x32,
    0x19,
    0xfd,
    0xe6,
    0x49,
    0xce,
    0xe7,
    0x12,
    0x4d,
    0x9f,
    0x7c,
    0xbe,
    0x97,
    0xf1,
    0xb1,
    0xb1,
    0x86,
    0x3a,
    0xec,
    0x7b,
    0x40,
    0xd9,
    0x01,
    0x57,
    0x62,
    0x30,
    0xbd,
    0x69,
    0xef,
    0x8f,
    0x6a,
    0xea,
    0xfe,
    0xb2,
    0xb0,
    0x92,
    0x19,
    0xfa,
    0x8f,
    0xaf,
    0x83,
    0x37,
    0x68,
    0x42,
    0xb1,
    0xb2,
    0xaa,
    0x9e,
    0xf6,
    0x8d,
    0x79,
    0xda,
    0xab,
    0x89,
    0xaf,
    0x3f,
    0xab,
    0xe4,
    0x9a,
    0xcc,
    0x27,
    0x86,
    0x38,
    0x70,
    0x73,
    0x45,
    0xbb,
    0xf1,
    0x53,
    0x44,
    0xed,
    0x79,
    0xf7,
    0xf4,
    0x39,
    0x0e,
    0xf8,
    0xac,
    0x50,
    0x9b,
    0x56,
    0xf3,
    0x9a,
    0x98,
    0x56,
    0x65,
    0x27,
    0xa4,
    0x1d,
    0x3c,
    0xbd,
    0x5e,
    0x05,
    0x58,
    0xc1,
    0x59,
    0x92,
    0x7d,
    0xb0,
    0xe8,
    0x84,
    0x54,
    0xa5,
    0xd9,
    0x64,
    0x71,
    0xfd,
    0xdc,
    0xb5,
    0x6d,
    0x5b,
    0xb0,
    0x6b,
    0xfa,
    0x34,
    0x0e,
    0xa7,
    0xa1,
    0x51,
    0xef,
    0x1c,
    0xa6,
    0xfa,
    0x57,
    0x2b,
    0x76,
    0xf3,
    0xb1,
    0xb9,
    0x5d,
    0x8c,
    0x85,
    0x83,
    0xd3,
    0xe4,
    0x77,
    0x05,
    0x36,
    0xb8,
    0x4f,
    0x01,
    0x7e,
    0x70,
    0xe6,
    0xfb,
    0xf1,
    0x76,
    0x60,
    0x1a,
    0x02,
    0x66,
    0x94,
    0x1a,
    0x17,
    0xb0,
    0xc8,
    0xb9,
    0x7f,
    0x4e,
    0x74,
    0xc2,
    0xc1,
    0xff,
    0xc7,
    0x27,
    0x89,
    0x19,
    0x77,
    0x79,
    0x40,
    0xc1,
    0xe1,
    0xff,
    0x1d,
    0x8d,
    0xa6,
    0x37,
    0xd6,
    0xb9,
    0x9d,
    0xda,
    0xfe,
    0x5e,
    0x17,
    0x61,
    0x10,
    0x02,
    0xe2,
    0xc7,
    0x78,
    0xc1,
    0xbe,
    0x8b,
    0x41,
    0xd9,
    0x63,
    0x79,
    0xa5,
    0x13,
    0x60,
    0xd9,
    0x77,
    0xfd,
    0x44,
    0x35,
    0xa1,
    0x1c,
    0x30,
    0x8f,
    0xe7,
    0xee,
    0x6f,
    0x1a,
    0xad,
    0x9d,
    0xb2,
    0x8c,
    0x81,
    0xad,
    0xde,
    0x1a,
    0x7a,
    0x6f,
    0x7c,
    0xce,
    0x01,
    0x1c,
    0x30,
    0xda,
    0x37,
    0xe4,
    0xeb,
    0x73,
    0x64,
    0x83,
    0xbd,
    0x6c,
    0x8e,
    0x93,
    0x48,
    0xfb,
    0xfb,
    0xf7,
    0x2c,
    0xc6,
    0x58,
    0x7d,
    0x60,
    0xc3,
    0x6c,
    0x8e,
    0x57,
    0x7f,
    0x09,
    0x84,
    0xc2,
    0x89,
    0xc9,
    0x38,
    0x5a,
    0x09,
    0x86,
    0x49,
    0xde,
    0x21,
    0xbc,
    0xa2,
    0x7a,
    0x7e,
    0xa2,
    0x29,
    0x71,
    0x6b,
    0xa6,
    0xe9,
    0xb2,
    0x79,
    0x71,
    0x0f,
    0x38,
    0xfa,
    0xa5,
    0xff,
    0xae,
    0x57,
    0x41,
    0x55,
    0xce,
    0x4e,
    0xfb,
    0x4f,
    0x74,
    0x36,
    0x95,
    0xe2,
    0x91,
    0x1b,
    0x1d,
    0x06,
    0xd5,
    0xe2,
    0x90,
    0xcb,
    0xcd,
    0x86,
    0xf5,
    0x6d,
    0x0e,
    0xdf,
    0xcd,
    0x21,
    0x6a,
    0xe2,
    0x24,
    0x27,
    0x05,
    0x5e,
    0x68,
    0x35,
    0xfd,
    0x29,
    0xee,
    0xf7,
    0x9e,
    0x0d,
    0x90,
    0x77,
    0x1f,
    0xea,
    0xce,
    0xbe,
    0x12,
    0xf2,
    0x0e,
    0x95,
    0xb3,
    0x4f,
    0x0f,
    0x78,
    0xb7,
    0x37,
    0xa9,
    0x61,
    0x8b,
    0x26,
    0xfa,
    0x7d,
    0xbc,
    0x98,
    0x74,
    0xf2,
    0x72,
    0xc4,
    0x2b,
    0xdb,
    0x56,
    0x3e,
    0xaf,
    0xa1,
    0x6b,
    0x4f,
    0xb6,
    0x8c,
    0x3b,
    0xb1,
    0xe7,
    0x8e,
    0xaa,
    0x81,
    0xa0,
    0x02,
    0x43,
    0xfa,
    0xad,
    0xd2,
    0xbf,
    0x18,
    0xe6,
    0x3d,
    0x38,
    0x9a,
    0xe4,
    0x43,
    0x77,
    0xda,
    0x18,
    0xc5,
    0x76,
    0xb5,
    0x0f,
    0x00,
    0x96,
    0xcf,
    0x34,
    0x19,
    0x54,
    0x83,
    0xb0,
    0x05,
    0x48,
    0xc0,
    0x98,
    0x62,
    0x36,
    0xe3,
    0xbc,
    0x7c,
    0xb8,
    0xd6,
    0x80,
    0x1c,
    0x04,
    0x94,
    0xcc,
    0xd1,
    0x99,
    0xe5,
    0xc5,
    0xbd,
    0x0d,
    0x0e,
    0xdc,
    0x9e,
    0xb8,
    0xa0,
    0x00,
    0x1e,
    0x15,
    0x27,
    0x67,
    0x54,
    0xfc,
    0xc6,
    0x85,
    0x66,
    0x05,
    0x41,
    0x48,
    0xe6,
    0xe7,
    0x64,
    0xbe,
    0xe7,
    0xc7,
    0x64,
    0xda,
    0xad,
    0x3f,
    0xc4,
    0x52,
    0x35,
    0xa6,
    0xda,
    0xd4,
    0x28,
    0xfa,
    0x20,
    0xc1,
    0x70,
    0xe3,
    0x45,
    0x00,
    0x3f,
    0x2f,
    0x06,
    0xec,
    0x81,
    0x05,
    0xfe,
    0xb2,
    0x5b,
    0x22,
    0x81,
    0xb6,
    0x3d,
    0x27,
    0x33,
    0xbe,
    0x96,
    0x1c,
    0x29,
    0x95,
    0x1d,
    0x11,
    0xdd,
    0x22,
    0x21,
    0x65,
    0x7a,
    0x9f,
    0x53,
    0x1d,
    0xda,
    0x2a,
    0x19,
    0x4d,
    0xbb,
    0x12,
    0x64,
    0x48,
    0xbd,
    0xee,
    0xb2,
    0x58,
    0xe0,
    0x7e,
    0xa6,
    0x59,
    0xc7,
    0x46,
    0x19,
    0xa6,
    0x38,
    0x0e,
    0x1d,
    0x66,
    0xd6,
    0x83,
    0x2b,
    0xfe,
    0x67,
    0xf6,
    0x38,
    0xcd,
    0x8f,
    0xae,
    0x1f,
    0x27,
    0x23,
    0x02,
    0x0f,
    0x9c,
    0x40,
    0xa3,
    0xfd,
    0xa6,
    0x7e,
    0xda,
    0x3b,
    0xd2,
    0x92,
    0x38,
    0xfb,
    0xd4,
    0xd4,
    0xb4,
    0x88,
    0x5c,
    0x2a,
    0x99,
    0x17,
    0x6d,
    0xb1,
    0xa0,
    0x6c,
    0x50,
    0x07,
    0x78,
    0x49,
    0x1a,
    0x82,
    0x88,
    0xf1,
    0x85,
    0x5f,
    0x60,
    0xff,
    0xfc,
    0xf1,
    0xd1,
    0x37,
    0x3f,
    0xd9,
    0x4f,
    0xc6,
    0x0c,
    0x18,
    0x11,
    0xe1,
    0xac,
    0x3f,
    0x1c,
    0x6d,
    0x00,
    0x3b,
    0xec,
    0xda,
    0x3b,
    0x1f,
    0x27,
    0x25,
    0xca,
    0x59,
    0x5d,
    0xe0,
    0xca,
    0x63,
    0x32,
    0x8f,
    0x3b,
    0xe5,
    0x7c,
    0xc9,
    0x77,
    0x55,
    0x60,
    0x11,
    0x95,
    0x14,
    0x0d,
    0xfb,
    0x59,
    0xd3,
    0x9c,
    0xe0,
    0x91,
    0x30,
    0x8b,
    0x41,
    0x05,
    0x74,
    0x6d,
    0xac,
    0x23,
    0xd3,
    0x3e,
    0x5f,
    0x7c,
    0xe4,
    0x84,
    0x8d,
    0xa3,
    0x16,
    0xa9,
    0xc6,
    0x6b,
    0x95,
    0x81,
    0xba,
    0x35,
    0x73,
    0xbf,
    0xaf,
    0x31,
    0x14,
    0x96,
    0x18,
    0x8a,
    0xb1,
    0x54,
    0x23,
    0x28,
    0x2e,
    0xe4,
    0x16,
    0xdc,
    0x2a,
    0x19,
    0xc5,
    0x72,
    0x4f,
    0xa9,
    0x1a,
    0xe4,
    0xad,
    0xc8,
    0x8b,
    0xc6,
    0x67,
    0x96,
    0xea,
    0xe5,
    0x67,
    0x7a,
    0x01,
    0xf6,
    0x4e,
    0x8c,
    0x08,
    0x63,
    0x13,
    0x95,
    0x82,
    0x2d,
    0x9d,
    0xb8,
    0xfc,
    0xee,
    0x35,
    0xc0,
    0x6b,
    0x1f,
    0xee,
    0xa5,
    0x47,
    0x4d,
    0x6d,
    0x8f,
    0x34,
    0xb1,
    0x53,
    0x4a,
    0x93,
    0x6a,
    0x18,
    0xb0,
    0xe0,
    0xd2,
    0x0e,
    0xab,
    0x86,
    0xbc,
    0x9c,
    0x6d,
    0x6a,
    0x52,
    0x07,
    0x19,
    0x4e,
    0x67,
    0xfa,
    0x35,
    0x55,
    0x1b,
    0x56,
    0x80,
    0x26,
    0x7b,
    0x00,
    0x64,
    0x1c,
    0x0f,
    0x21,
    0x2d,
    0x18,
    0xec,
    0xa8,
    0xd7,
    0x32,
    0x7e,
    0xd9,
    0x1f,
    0xe7,
    0x64,
    0xa8,
    0x4e,
    0xa1,
    0xb4,
    0x3f,
    0xf5,
    0xb4,
    0xf6,
    0xe8,
    0xe6,
    0x2f,
    0x05,
    0xc6,
    0x61,
    0xde,
    0xfb,
    0x25,
    0x88,
    0x77,
    0xc3,
    0x5b,
    0x18,
    0xa1,
    0x51,
    0xd5,
    0xc4,
    0x14,
    0xaa,
    0xad,
    0x97,
    0xba,
    0x3e,
    0x49,
    0x93,
    0x32,
    0xe5,
    0x96,
    0x07,
    0x8e,
    0x60,
    0x0d,
    0xeb,
    0x81,
    0x14,
    0x9c,
    0x44,
    0x1c,
    0xe9,
    0x57,
    0x82,
    0xf2,
    0x2a,
    0x28,
    0x25,
    0x63,
    0xc5,
    0xba,
    0xc1,
    0x41,
    0x14,
    0x23,
    0x60,
    0x5d,
    0x1a,
    0xe1,
    0xaf,
    0xae,
    0x2c,
    0x8b,
    0x06,
    0x60,
    0x23,
    0x7e,
    0xc1,
    0x28,
    0xaa,
    0x0f,
    0xe3,
    0x46,
    0x4e,
    0x43,
    0x58,
    0x11,
    0x5d,
    0xb8,
    0x4c,
    0xc3,
    0xb5,
    0x23,
    0x07,
    0x3a,
    0x28,
    0xd4,
    0x54,
    0x98,
    0x84,
    0xb8,
    0x1f,
    0xf7,
    0x0e,
    0x10,
    0xbf,
    0x36,
    0x1c,
    0x13,
    0x72,
    0x96,
    0x28,
    0xd5,
    0x34,
    0x8f,
    0x07,
    0x21,
    0x1e,
    0x7e,
    0x4c,
    0xf4,
    0xf1,
    0x8b,
    0x28,
    0x60,
    0x90,
    0xbd,
    0xb1,
    0x24,
    0x0b,
    0x66,
    0xd6,
    0xcd,
    0x4a,
    0xfc,
    0xea,
    0xdc,
    0x00,
    0xca,
    0x44,
    0x6c,
    0xe0,
    0x50,
    0x50,
    0xff,
    0x18,
    0x3a,
    0xd2,
    0xbb,
    0xf1,
    0x18,
    0xc1,
    0xfc,
    0x0e,
    0xa5,
    0x1f,
    0x97,
    0xd2,
    0x2b,
    0x8f,
    0x7e,
    0x46,
    0x70,
    0x5d,
    0x45,
    0x27,
    0xf4,
    0x5b,
    0x42,
    0xae,
    0xff,
    0x39,
    0x58,
    0x53,
    0x37,
    0x6f,
    0x69,
    0x7d,
    0xd5,
    0xfd,
    0xf2,
    0xc5,
    0x18,
    0x7d,
    0x7d,
    0x5f,
    0x0e,
    0x2e,
    0xb8,
    0xd4,
    0x3f,
    0x17,
    0xba,
    0x0f,
    0x7c,
    0x60,
    0xff,
    0x43,
    0x7f,
    0x53,
    0x5d,
    0xfe,
    0xf2,
    0x98,
    0x33,
    0xbf,
    0x86,
    0xcb,
    0xe8,
    0x8e,
    0xa4,
    0xfb,
    0xd4,
    0x22,
    0x1e,
    0x84,
    0x11,
    0x72,
    0x83,
    0x54,
    0xfa,
    0x30,
    0xa7,
    0x00,
    0x8f,
    0x15,
    0x4a,
    0x41,
    0xc7,
    0xfc,
    0x46,
    0x6b,
    0x46,
    0x45,
    0xdb,
    0xe2,
    0xe3,
    0x21,
    0x26,
    0x7f,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
    0xff,
};

static const SECItem subprime_ike_1536 = { siBuffer,
                                           (unsigned char *)subprime_ike_1536_data,
                                           sizeof(subprime_ike_1536_data) };
static const SECItem subprime_ike_2048 = { siBuffer,
                                           (unsigned char *)subprime_ike_2048_data,
                                           sizeof(subprime_ike_2048_data) };
static const SECItem subprime_ike_3072 = { siBuffer,
                                           (unsigned char *)subprime_ike_3072_data,
                                           sizeof(subprime_ike_3072_data) };
static const SECItem subprime_ike_4096 = { siBuffer,
                                           (unsigned char *)subprime_ike_4096_data,
                                           sizeof(subprime_ike_4096_data) };
static const SECItem subprime_ike_6144 = { siBuffer,
                                           (unsigned char *)subprime_ike_6144_data,
                                           sizeof(subprime_ike_6144_data) };
static const SECItem subprime_ike_8192 = { siBuffer,
                                           (unsigned char *)subprime_ike_8192_data,
                                           sizeof(subprime_ike_8192_data) };
static const SECItem subprime_tls_2048 = { siBuffer,
                                           (unsigned char *)subprime_tls_2048_data,
                                           sizeof(subprime_tls_2048_data) };
static const SECItem subprime_tls_3072 = { siBuffer,
                                           (unsigned char *)subprime_tls_3072_data,
                                           sizeof(subprime_tls_3072_data) };
static const SECItem subprime_tls_4096 = { siBuffer,
                                           (unsigned char *)subprime_tls_4096_data,
                                           sizeof(subprime_tls_4096_data) };
static const SECItem subprime_tls_6144 = { siBuffer,
                                           (unsigned char *)subprime_tls_6144_data,
                                           sizeof(subprime_tls_6144_data) };
static const SECItem subprime_tls_8192 = { siBuffer,
                                           (unsigned char *)subprime_tls_8192_data,
                                           sizeof(subprime_tls_8192_data) };

/* generator for all the groups is 2 */
static const unsigned char generator_2_data[] = { 2 };

static const SECItem generator_2 = { siBuffer,
                                     (unsigned char *)generator_2_data,
                                     sizeof(generator_2_data) };

/*
 * verify that dhPrime matches one of our known primes
 */
const SECItem *
sftk_VerifyDH_Prime(SECItem *dhPrime, SECItem *g, PRBool isFIPS)
{
    /* use the length to decide which primes to check */
    switch (dhPrime->len) {
        case 1536 / PR_BITS_PER_BYTE:
            /* don't accept 1536 bit primes in FIPS mode */
            if (isFIPS) {
                break;
            }
            if (PORT_Memcmp(dhPrime->data, prime_ike_1536,
                            sizeof(prime_ike_1536)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_ike_1536;
            }
            break;
        case 2048 / PR_BITS_PER_BYTE:
            if (PORT_Memcmp(dhPrime->data, prime_tls_2048,
                            sizeof(prime_tls_2048)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_tls_2048;
            }
            if (PORT_Memcmp(dhPrime->data, prime_ike_2048,
                            sizeof(prime_ike_2048)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_ike_2048;
            }
            break;
        case 3072 / PR_BITS_PER_BYTE:
            if (PORT_Memcmp(dhPrime->data, prime_tls_3072,
                            sizeof(prime_tls_3072)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_tls_3072;
            }
            if (PORT_Memcmp(dhPrime->data, prime_ike_3072,
                            sizeof(prime_ike_3072)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_ike_3072;
            }
            break;
        case 4096 / PR_BITS_PER_BYTE:
            if (PORT_Memcmp(dhPrime->data, prime_tls_4096,
                            sizeof(prime_tls_4096)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_tls_4096;
            }
            if (PORT_Memcmp(dhPrime->data, prime_ike_4096,
                            sizeof(prime_ike_4096)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_ike_4096;
            }
            break;
        case 6144 / PR_BITS_PER_BYTE:
            if (PORT_Memcmp(dhPrime->data, prime_tls_6144,
                            sizeof(prime_tls_6144)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_tls_6144;
            }
            if (PORT_Memcmp(dhPrime->data, prime_ike_6144,
                            sizeof(prime_ike_6144)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_ike_6144;
            }
            break;
        case 8192 / PR_BITS_PER_BYTE:
            if (PORT_Memcmp(dhPrime->data, prime_tls_8192,
                            sizeof(prime_tls_8192)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_tls_8192;
            }
            if (PORT_Memcmp(dhPrime->data, prime_ike_8192,
                            sizeof(prime_ike_8192)) == 0) {
                if (g)
                    *g = generator_2;
                return &subprime_ike_8192;
            }
            break;
    }
    /* no match found, return an error */
    PORT_SetError(SEC_ERROR_INVALID_ARGS);
    return NULL;
}

/* Use the provided subPrime to see if dhPrime is a safe prime. We'll check
 * primality of those values later. */
SECStatus
sftk_IsSafePrime(const SECItem *dhPrime, const SECItem *dhSubPrime, PRBool *isSafe)
{
    int i;
    unsigned char carry = 0;
    int offset = 0, subPrimeLen = dhPrime->len;
    *isSafe = PR_FALSE;

    /* Both dhPrime and dhSubPrime should be odd */
    if (((dhPrime->data[dhPrime->len - 1] & 0x1) != 1) && ((dhSubPrime->data[dhSubPrime->len - 1] & 0x1) != 1)) {
        PORT_SetError(SEC_ERROR_INVALID_ARGS);
        return SECFailure;
    }

    /* subPrime is p-1/2, which means subPrime is 1 bit shorter than p.
     * It's length in bytes is the same unless the high byte of p == 1 or 0.
     */
    if (dhPrime->data[0] <= 1) {
        subPrimeLen--;
        offset++;
        carry = (dhPrime->data[0]) << 7;
    }

    /* if subprime len is not long enough it is not a strong prime */
    if (dhSubPrime->len != subPrimeLen) {
        return SECSuccess;
    }

    /* does the subprime match q == (p-1)/2 */
    for (i = 0; i < subPrimeLen; i++) {
        if (dhSubPrime->data[i] !=
            (carry | ((dhPrime->data[i + offset] >> 1) & 0x7f))) {
            return SECSuccess;
        }
        carry = ((dhPrime->data[i + offset] & 1) << 7) & 0x80;
    }
    /* subPrime for p claims to be q=(p-1)/2. So the caller thinks p
     * is a strong prime, just need to check primality of p and q to verify */
    *isSafe = PR_TRUE;
    return SECSuccess;
}
