// ---------------------------------------------------------------------------
// - Rmi.hpp                                                                 -
// - afnix:mth module - real matrix interface definitions                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_RMI_HPP
#define  AFNIX_RMI_HPP

#ifndef  AFNIX_RVI_HPP
#include "Rvi.hpp"
#endif

namespace afnix {

  /// This Rmi class is an abstract class that models the behavior of a
  /// real based matrix. Tha class defines the matrix size as well as the
  /// accessor and mutator methods.
  /// @author amaury darsch

  class Rmi : public virtual Object {
  public:
    /// the matrix row size
    t_long d_rsiz;
    /// the matrix column size
    t_long d_csiz;

  public:
    /// create a null matrix
    Rmi (void);

    /// create a square matrix by size
    /// @param size the matrix size
    Rmi (const t_long size);

    /// create a matrix by size
    /// @param rsiz the row size
    /// @param csiz the column size
    Rmi (const t_long rsiz, const t_long csiz);

    /// clear this matrix
    virtual void clear (void);

    /// @return the matrix row size
    virtual t_long getrsiz (void) const;

    /// @return the matrix column size
    virtual t_long getcsiz (void) const;

    /// @return true if the matrix is square
    virtual bool issquare (void) const;

    /// set a matrix by position
    /// @param row the row position
    /// @param col the column position
    /// @param val the value to set
    virtual void set (const t_long row, const t_long col, const t_real val) =0;

    /// get a matrix value by position
    /// @param row the row position
    /// @param col the column position
    virtual t_real get (const t_long row, const t_long col) const =0;

    /// compare two matrices upto a precision
    /// @param mx the matrix argument
    virtual bool cmp (const Rmi& x) const;

    /// @return the matrix frobenius norm
    virtual t_real norm (void) const;

    /// multiply a matrix with a vector and a scaling factor
    /// @param r the result vector
    /// @param x the vector argument
    /// @param s the scaling factor
    virtual Rvi& mul (Rvi& r, const Rvi& x, const t_real s) const;

  public:
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;
    
    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
    
  };
}

#endif
