#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

import math

from fenrirscreenreader.core.i18n import _


class adjustment_command:
    """Base class for speech and sound adjustment commands"""

    def __init__(self, section, setting, direction, step=0.1):
        self.section = section  # 'speech' or 'sound'
        self.setting = setting  # 'rate', 'pitch', 'volume'
        self.direction = direction  # 'inc' or 'dec'
        self.step = step

    def initialize(self, environment):
        self.env = environment

    def shutdown(self):
        pass

    def get_description(self):
        action = "Increase" if self.direction == "inc" else "Decrease"
        if self.section == "speech":
            return _(f"{action} the speech {self.setting}")
        else:
            return _(f"{action} the {self.section} {self.setting}")

    def run(self):
        if self.section == "sound" and self.setting == "volume":
            # Sound volume uses different method
            self._adjust_sound_volume()
        else:
            # Speech rate, pitch, volume use standard method
            self._adjust_speech_setting()

    def _adjust_speech_setting(self):
        """Adjust speech settings (rate, pitch, volume)"""
        value = self.env["runtime"]["SettingsManager"].get_setting_as_float(
            self.section, self.setting
        )

        # Apply adjustment with rounding
        if self.direction == "inc":
            value = round((math.ceil(10 * value) / 10) + self.step, 2)
            if value > 1.0:
                value = 1.0
        else:  # dec
            value = round((math.ceil(10 * value) / 10) - self.step, 2)
            if value < 0.0:
                value = 0.0

        # Set the new value
        self.env["runtime"]["SettingsManager"].set_setting(
            self.section, self.setting, str(value)
        )

        # Present feedback
        percentage = int(value * 100)
        if self.section == "speech":
            feedback = _("{0} percent speech {1}").format(
                percentage, self.setting
            )
        else:
            feedback = _("{0} percent {1} {2}").format(
                percentage, self.section, self.setting
            )

        self.env["runtime"]["OutputManager"].present_text(
            feedback, sound_icon="", interrupt=True
        )

    def _adjust_sound_volume(self):
        """Adjust sound volume using same logic as speech"""
        value = self.env["runtime"]["SettingsManager"].get_setting_as_float(
            self.section, self.setting
        )

        # Sound volume uses same math as speech settings
        if self.direction == "inc":
            value = round((math.ceil(10 * value) / 10) + self.step, 2)
            if value > 1.0:
                value = 1.0
        else:  # dec
            value = round((math.ceil(10 * value) / 10) - self.step, 2)
            if value < 0.0:
                value = 0.0

        # Set the new value
        self.env["runtime"]["SettingsManager"].set_setting(
            self.section, self.setting, str(value)
        )

        # Present feedback with appropriate sound icon
        percentage = int(value * 100)
        sound_icon = "SoundOn" if self.direction == "inc" else "SoundOff"
        feedback = _("{0} percent sound volume").format(percentage)
        self.env["runtime"]["OutputManager"].present_text(
            feedback, sound_icon=sound_icon, interrupt=True
        )

    def set_callback(self, callback):
        pass
