;;; Copyright (C) 2011 Team GPS.
;;; 
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

(ns twitter_clj.twitter
  (:import (twitter4j TwitterFactory TwitterException))
  (:import (twitter4j.conf ConfigurationBuilder))
  (:import (java.io File))
  (:require [twitter_clj.common :as common]
            [clojure.java.io :as io]
            [clojure.contrib.logging :as log]))

;; ==================================================
;; Global variables
;; ==================================================


;; ==================================================
;; Functions
;; ==================================================

(defn twitter-instance
  "Return an instance of Twitter class by using keys read from environment varibles:
    * GPSUSI_CONSUMER_KEY
    * GPSUSI_CONSUMER_SECRET
    * GPSUSI_ACCESS_TOKEN
    * GPSUSI_TOKEN_SECRET"
  []
  (let [env (System/getenv)
        [consumer-key
         consumer-secret
         access-token
         token-secret] (map #(get env %) ["GPSUSI_CONSUMER_KEY"
                                          "GPSUSI_CONSUMER_SECRET"
                                          "GPSUSI_ACCESS_TOKEN"
                                          "GPSUSI_TOKEN_SECRET"])]
    (when (some nil? [consumer-key consumer-secret access-token token-secret])
      (log/fatal "Twitter configuration not found")
      (throw (IllegalStateException.)))
    (let [conf (doto (ConfigurationBuilder.)
                 (.setOAuthConsumerKey       consumer-key)
                 (.setOAuthConsumerSecret    consumer-secret)
                 (.setOAuthAccessToken       access-token)
                 (.setOAuthAccessTokenSecret token-secret))]
     (.getInstance (TwitterFactory. (.build conf))))))


(defn log-message
  "Log a message to a file if the file does not exist.
   Return true if writting the message is successful;
   false otherwise (ex. the file exists)"
  [f str]
  (if (.exists f)
    (do
      (log/warn (format "Found an existing file: %s" f))
      false)
    (do
      (spit f str)
      true)))


(defn moves-file
  "Return a File instance into which moves tweets are recorded."
  [str]
  (let [m (re-find #"^\[\((\d+)\) .*\] <(\d+)s>" str)
        _ (assert m)
        nmove (Integer. (nth m 1))
        sec   (Integer. (nth m 2))
        file-name (format "analyses%03d_%05d.txt" nmove sec)]
    (io/file file-name)))


(defn moves-file-exists?
  [nmove]
  (let [re (re-pattern (format "analyses%03d_.*.txt$" nmove))]
    (some #(re-find re %) (.list (File. "./")))))


(defn log-move
  "Log a move tweet to a file."
  [str]
  (log-message (moves-file str) str))


(defn trancate-140-characters
  "Trancate a string str if it includes too many characters."
  [str]
  (if (< 140 (count str))
   (let [to-str (subs str 0 140)]
     (log/warn (format "Trancated characters from %s to %s" str to-str))
     to-str)
   str))


(defn post-twitter
  "Tweet a message."
  [logger str]
  (let [str (trancate-140-characters str)]
    (log/debug (format "Tweeting...  %s" str))
    (if (logger str)
      (try
        (if-not (:dry-run @common/options)
          (let [twitter (twitter-instance)
                status (.updateStatus twitter str)]
            (when (.isTruncated status)
              (log/warn "Message was truncated"))
            (log/debug (format "Tweeted: %s" (.getText status)))
            status))
        (catch TwitterException e
          (log/warn (format "Failed to tweet: %s %s" str e)))))))


(defn post-move
  "Tweet a move message."
  [str]
  (post-twitter log-move str))


(defn post-version
  "Tweet an engine version"
  [str]
  (letfn [(log-version [str]
            (let [f (io/file "tweet-version.txt")]
              (log-message f str)))]
    (if (empty? str)
      (log/warn "Empty version.")
      (post-twitter log-version str))))


(defn post-title
  "Tweet a title"
  []
  (letfn [(log-title [str]
            (let [f (io/file "tweet-title.txt")]
              (log-message f str)))]
  (let [f (io/file "title")]
    (if (.exists f)
      (post-twitter log-title (slurp f :encoding "EUC-JP"))
      (log/warn (format "Title file not found: %s" f))))))


(defn -main [& args]
  "To test a tweet."
  (post-move (first args)))
