"""Tests for file storage backend based on Swift"""

from cStringIO import StringIO

from twisted.internet.defer import inlineCallbacks, fail

from juju import errors
from juju.lib import testing

from juju.providers.openstack.tests import OpenStackTestMixin


class FileStorageTestCase(OpenStackTestMixin, testing.TestCase):

    def get_storage(self):
        provider = self.get_provider()
        storage = provider.get_file_storage()
        return storage

    def test_put_file(self):
        """A file can be put in the storage"""
        content = "some text"
        self.expect_swift_put("testing/object", content)
        self.mocker.replay()
        return self.get_storage().put("object", StringIO(content))

    def test_put_file_unicode(self):
        """A file with a unicode name is put in UTF-8 url encoded form"""
        content = "some text"
        self.expect_swift_put("testing/%C2%A7", content)
        self.mocker.replay()
        return self.get_storage().put(u"\xa7", StringIO(content))

    def test_put_file_create_container(self):
        """The container will be created if it doesn't exist yet"""
        content = "some text"
        self.expect_swift_put("testing/object", content, code=404)
        self.expect_swift_put_container("testing")
        self.expect_swift_put("testing/object", content)
        self.mocker.replay()
        return self.get_storage().put("object", StringIO(content))

    def test_put_file_unknown_error(self):
        """Unexpected errors from client propogate"""
        content = "some text"
        self._mock_swift("PUT", "testing/object", content)
        self.mocker.result(fail(ValueError("Something unexpected")))
        self.mocker.replay()
        deferred = self.get_storage().put("object", StringIO(content))
        return self.assertFailure(deferred, ValueError)

    @inlineCallbacks
    def test_get_url(self):
        """A url can be generated for any stored file."""
        self.mocker.replay()
        storage = self.get_storage()
        yield storage._swift._client.authenticate()
        url = storage.get_url("object")
        self.assertEqual(self.swift_url + "/testing/object", url)

    @inlineCallbacks
    def test_get_url_unicode(self):
        """A url can be generated for *any* stored file."""
        self.mocker.replay()
        storage = self.get_storage()
        yield storage._swift._client.authenticate()
        url = storage.get_url(u"\xa7")
        self.assertEqual(self.swift_url + "/testing/%C2%A7", url)

    @inlineCallbacks
    def test_get_file(self):
        """Retrieving a file returns a file-like object with the content"""
        content = "some text"
        self.expect_swift_get("testing/object", response=content)
        self.mocker.replay()
        result = yield self.get_storage().get("object")
        self.assertEqual(result.read(), content)

    @inlineCallbacks
    def test_get_file_unicode(self):
        """Retrieving a file with a unicode key uses a UTF-8 url"""
        content = "some text"
        self.expect_swift_get(u"testing/%C2%A7", response=content)
        self.mocker.replay()
        result = yield self.get_storage().get(u"\xa7")
        self.assertEqual(result.read(), content)

    def test_get_file_nonexistant(self):
        """Retrieving a nonexistant file raises a file not found error."""
        self.expect_swift_get(u"testing/missing", code=404)
        self.mocker.replay()
        deferred = self.get_storage().get("missing")
        return self.assertFailure(deferred, errors.FileNotFound)

    def test_get_file_error(self):
        """An error from the client results is attributed to the provider"""
        self.expect_swift_get(u"testing/unavailable", code=500)
        self.mocker.replay()
        deferred = self.get_storage().get("unavailable")
        return self.assertFailure(deferred, errors.ProviderInteractionError)
