#ifndef __CARDS_H
#define __CARDS_H
/*-------------------------------------------------------------------------
 * Copyright (c) 2002 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * cards
 *
 * Synopsis:
 *    routines for handling the MathWar cards
 *
 * $Id: cards.h,v 1.1 2002/02/10 21:44:17 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to 
 * Free Software Foundation, Inc. 
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
#include <gnome.h>

/*! \file cards.h

  \brief Define the card objects

  This module defines the card objects.  It also provides routines for 
  loading the cards from file, drawing the cards, and destroying the
  cards when we are done with them.

  \author Kenneth W. Sodemann <stufflehead@bigfoot.com>
  $Revision: 1.1 $
  $Date: 2002/02/10 21:44:17 $
*/

/*! 
 \def CARD_BACK_ONE
 The back of a card laying face down.  There are two distinct card backs.
 They are stained slightly differently.  This is just to break up the 
 uniformity.  Pass this value to the draw_card() function to draw card
 back #1.
*/
/*! 
 \def CARD_BACK_TWO
 Pass this value to the draw_card() function to draw card back #2.
*/
#define CARD_BACK_ONE -1
#define CARD_BACK_TWO -2

/*!
 \def CARD_WIDTH
 The width of the card, not taking the shadow into account.
*/
/*!
 \def CARD_HEIGHT
 The height of the card, not taking the shadow into account.
*/
#define CARD_WIDTH   125
#define CARD_HEIGHT  200

/*! \fn load_deck (void)
 
  \brief Load the deck of cards from disk.

  \retval TRUE
  The load worked.
  \retval FALSE
  The load failed for some reason.
*/
gboolean load_deck (void);

/*! \fn draw_card (GdkPixmap *pix, GdkGC *def_gc, GdkGC *shadow_gc,
                   gint card, gint x, gint y)
 
  \brief Draw the requested card at position x, y on the pixmap

  \param pix
  The pixmap to draw the cards on.
  \param def_gc
  The default graphic context used when drawing on the pixmap.
  \param shadow_gc
  The graphic context to use when drawing the card's shadow.
  \param card
  The face value of the card.  To draw a card back, use CARD_BACK_ONE
  or CARD_BACK_TWO.
  \param x
  The x coordinate where the upper left corner of the card should be.
  \param y
  The y coordinate where the upper left corner of the card should be.

  \return
  Nothing.
*/
void draw_card (GdkPixmap *pix, GdkGC *def_gc, GdkGC *shadow_gc,
                gint card, gint x, gint y);

/*! \fn destroy_deck (void)
 
  \brief Free the memory used by the deck.

  \return
  Nothing.
*/
void destroy_deck (void);

#endif
