/*	JPEG2000_Codestream_Info

HiRISE CVS ID: JPEG2000_Codestream_Info.java,v 1.32 2012/11/20 00:21:08 castalia Exp

Copyright (C) 2006-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Image_Tools;

import java.io.File;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.EOFException;

import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.MemoryCacheImageInputStream;
import java.nio.ByteOrder;

import java.util.Hashtable;

import PIRL.PVL.*;

//	Main dependencies.
import PIRL.Viewers.Parameter_View;
import java.util.Vector;
import java.util.Iterator;


/**	<i>JPEG2000_Codestream_Info</i> provides a set of PVL Parameters that
	describe the contents of a JPEG2000 codestream.
<p>
	The JPEG2000 codestream format is defined in Part 1 of the JPEG2000
	International Standard (ISO/IEC 15444-1:2004(E); "Information
	technology - JPEG 2000 image coding system: Core coding system").
	This format is composed entirely of a contiguous sequence of
	codestream segment. Each segment is prefaced with a marker that
	identifies the segement. The marker is followed by  length. Boxes may
	be nested within other boxes.
<p>
	@author		Bradford Castalia UA/PIRL
	@version	1.32
	@see		PIRL.Image_Tools.JP2_Info
	@see		PIRL.Image_Tools.JPEG2000_Info
*/
public class JPEG2000_Codestream_Info
	extends JPEG2000_Info
{
public static final String
	ID = "PIRL.Image_Tools.JPEG2000_Codestream_Info (1.32 2012/11/20 00:21:08)";

//!	The name of this Parameter Aggregate.
public static final String
	CODESTREAM						= "Codestream";

//!	Marker codes.
public static final int
	RESERVED_DELIMITER_MARKER_MIN	= 0xFF30,
	RESERVED_DELIMITER_MARKER_MAX	= 0xFF3F,

	SOC_MARKER						= 0xFF4F,
	SOT_MARKER						= 0xFF90,
	SOD_MARKER						= 0xFF93,
	EOC_MARKER						= 0xFFD9,
	SIZ_MARKER						= 0xFF51,
	COD_MARKER						= 0xFF52,
	COC_MARKER						= 0xFF53,
	RGN_MARKER						= 0xFF5E,
	QCD_MARKER						= 0xFF5C,
	QCC_MARKER						= 0xFF5D,
	POC_MARKER						= 0xFF5F,
	TLM_MARKER						= 0xFF55,
	PLM_MARKER						= 0xFF57,
	PLT_MARKER						= 0xFF58,
	PPM_MARKER						= 0xFF60,
	PPT_MARKER						= 0xFF61,
	SOP_MARKER						= 0xFF91,
	EPH_MARKER						= 0xFF92,
	CRG_MARKER						= 0xFF63,
	COM_MARKER						= 0xFF64;

//! Marker names.
public static final String
	//	Delimiting markers and marker segments.
	SOC_NAME						= "Start_of_Codestream",
	SOT_NAME						= "Start_of_Tile_Part",
	SOD_NAME						= "Start_of_Data",
	EOC_NAME						= "End_of_Codestream",

	//	Fixed information marker segments.
	SIZ_NAME						= "Size",

	//	Functional marker segments.
	COD_NAME						= "Coding_Style_Default",
	COC_NAME						= "Coding_Style_Component",
	RGN_NAME						= "Region_of_Interest",
	QCD_NAME						= "Quantization_Default",
	QCC_NAME						= "Quantization_Component",
	POC_NAME						= "Progression_Order_Change",

	//	Pointer marker segments.
	TLM_NAME						= "Tile_Lengths",
	PLM_NAME						= "Packet_Length_Main",
	PLT_NAME						= "Packet_Length_Tile",
	PPM_NAME						= "Packed_Packet_Main",
	PPT_NAME						= "Packed_Packet_Tile",

	//	In-bit-stream markers and marker segments.
	SOP_NAME						= "Start_of_Packet",
	EPH_NAME						= "End_of_Packet_Header",

	//	Information marker segments.
	CRG_NAME						= "Component_Registration",
	COM_NAME						= "Comment";

private static Hashtable
	Marker_Names					= new Hashtable ();
static
	{
	Marker_Names.put (new Integer (SOC_MARKER), SOC_NAME);
	Marker_Names.put (new Integer (SOT_MARKER), SOT_NAME);
	Marker_Names.put (new Integer (SOD_MARKER), SOD_NAME);
	Marker_Names.put (new Integer (EOC_MARKER), EOC_NAME);
	Marker_Names.put (new Integer (SIZ_MARKER), SIZ_NAME);
	Marker_Names.put (new Integer (COD_MARKER), COD_NAME);
	Marker_Names.put (new Integer (COC_MARKER), COC_NAME);
	Marker_Names.put (new Integer (RGN_MARKER), RGN_NAME);
	Marker_Names.put (new Integer (QCD_MARKER), QCD_NAME);
	Marker_Names.put (new Integer (QCC_MARKER), QCC_NAME);
	Marker_Names.put (new Integer (POC_MARKER), POC_NAME);
	Marker_Names.put (new Integer (TLM_MARKER), TLM_NAME);
	Marker_Names.put (new Integer (PLM_MARKER), PLM_NAME);
	Marker_Names.put (new Integer (PLT_MARKER), PLT_NAME);
	Marker_Names.put (new Integer (PPM_MARKER), PPM_NAME);
	Marker_Names.put (new Integer (PPT_MARKER), PPT_NAME);
	Marker_Names.put (new Integer (SOP_MARKER), SOP_NAME);
	Marker_Names.put (new Integer (EPH_MARKER), EPH_NAME);
	Marker_Names.put (new Integer (CRG_MARKER), CRG_NAME);
	Marker_Names.put (new Integer (COM_MARKER), COM_NAME);
	}

//!	Marker code parameter.
public static final String
	MARKER_PARAMETER						= "Marker";

//!	SOT parameters.
public static final String
	TILE_INDEX_PARAMETER					= "Tile_Index",
	TILE_PART_LENGTH_PARAMETER				= "Tile_Part_Length",
	TILE_PART_INDEX_PARAMETER				= "Tile_Part_Index",
	TOTAL_TILE_PARTS_PARAMETER				= "Total_Tile_Parts";

//!	SIZ parameters.
public static final String
	CAPABILITY_PARAMETER					= "Capability",
	REFERENCE_GRID_WIDTH_PARAMETER			= "Reference_Grid_Width",
	REFERENCE_GRID_HEIGHT_PARAMETER			= "Reference_Grid_Height",
	HORIZONTAL_IMAGE_OFFSET_PARAMETER		= "Horizontal_Image_Offset",
	VERTICAL_IMAGE_OFFSET_PARAMETER			= "Vertical_Image_Offset",
	TILE_WIDTH_PARAMETER					= "Tile_Width",
	TILE_HEIGHT_PARAMETER					= "Tile_Height",
	HORIZONTAL_TILE_OFFSET_PARAMETER		= "Horizontal_Tile_Offset",
	VERTICAL_TILE_OFFSET_PARAMETER			= "Vertical_Tile_Offset",
	HORIZONTAL_SAMPLE_SPACING_PARAMETER		= "Horizontal_Sample_Spacing",
	VERTICAL_SAMPLE_SPACING_PARAMETER		= "Vertical_Sample_Spacing";
	
//!	COD parameters.
public static final String
	CODING_STYLE_PARAMETER					= "Coding_Style",
	PROGRESSION_ORDER_PARAMETER				= "Progression_Order",
	TOTAL_QUALITY_LAYERS_PARAMETER			= "Total_Quality_Layers",
	MULTIPLE_COMPONENT_TRANSFORM_PARAMETER	= "Multiple_Component_Transform",
	TOTAL_RESOLUTION_LEVELS_PARAMETER		= "Total_Resolution_Levels",
	CODE_BLOCK_WIDTH_PARAMETER				= "Code_Block_Width",
	CODE_BLOCK_HEIGHT_PARAMETER				= "Code_Block_Height",
	CODE_BLOCK_STYLE_PARAMETER				= "Code_Block_Style",
	TRANSFORM_PARAMETER						= "Transform",
	PRECINCT_SIZE_PARAMETER					= "Precinct_Size";
//!	Coding style bit flag masks.
public static final int
	ENTROPY_CODER_FLAG						= 1 << 0,
	SOP_FLAG								= 1 << 1,
	EPH_FLAG								= 1 << 2;
//!	Progression order values.
public static final int
	LRCP_PROGRESSION_ORDER					= 0,
	RLCP_PROGRESSION_ORDER					= 1,
	RPCL_PROGRESSION_ORDER					= 2,
	PCRL_PROGRESSION_ORDER					= 3,
	CPRL_PROGRESSION_ORDER					= 4;
public static final String
	PROGRESSION_ORDERS[] =
		{
		"Layer-Resolution-Component-Position",
		"Resolution-Layer-Component-Position",
		"Resolution-Position-Component-Layer",
		"Position-Component-Resolution-Layer",
		"Component-Position-Resolution-Layer"
		};
//!	Code block style bit flag masks.
public static final int
	SELECTIVE_ARITHMETIC_BYPASS_FLAG		= 1 << 0,
	RESET_CONTEXT_PROBABILITIES				= 1 << 1,
	TERMINATION_FLAG						= 1 << 2,
	VERTICALLY_CAUSAL_CONTEXT_FLAG			= 1 << 3,
	PREDICTABLE_TERMINATION_FLAG			= 1 << 4,
	SEGMENTATION_SYMBOLS_FLAG				= 1 << 5;
//!	Transform values.
public static final int
	TRANSFORM_IRREVERSIBLE					= 0,
	TRANSFORM_REVERSIBLE					= 1;

//!	COC parameters.
public static final String
	COMPONENT_INDEX_PARAMETER				= "Component_Index";

//!	RGN parameters.
public static final String
	ROI_STYLE_PARAMETER						= "ROI_Style",
	IMPLICIT_SHIFT_PARAMETER				= "Implicit_Shift";

//!	QCD parameters.
public static final String
	QUANTIZATION_STYLE_PARAMETER			= "Quantization_Style",
	TOTAL_GUARD_BITS_PARAMETER				= "Total_Guard_Bits",
	STEP_SIZE_PARAMETER						= "Step_Size";
//!	Quantization style bit field masks.
public static final int
	NO_QUANTIZATION							= 0,
	QUANTIZATION_SCALAR_DERIVED				= 1,
	QUANTIZATION_SCALAR_EXPOUNDED			= 2;

//!	POC parameters.
public static final String
	LEVEL_INDEX_PARAMETER					= "Level_Index",
	LAYER_INDEX_PARAMETER					= "Layer_Index";

//!	TLM parameters.
public static final String
	INDEX_PARAMETER							= "Index",
	TILE_INDEX_SIZE_PARAMETER				= "Tile_Index_Size",
	TILE_PART_LENGTH_SIZE_PARAMETER			= "Tile_Part_Length_Size";

//!	PLM and PLT parameters.
public static final String
	PACKET_LENGTH_PARAMETER					= "Packet_Length",
	CONTINUATION_PARAMETER					= "Continuation";
//	PLM and PLT continuation state variables.
private Parameter
	PLM_Packet_Length_Parameter				= null,
	PLT_Packet_Length_Parameter				= null;
private long
	PLM_Packet_Length						= 0,
	PLT_Packet_Length						= 0,
	PLM_Packet_Length_Bytes_Remaining		= 0,
	PLT_Packet_Length_Bytes_Remaining		= 0;

//!	CRG parameters.
public static final String
	HORIZONTAL_COMPONENT_OFFSET_PARAMETER	= "Horizontal_Component_Offset",
	VERTICAL_COMPONENT_OFFSET_PARAMETER		= "Vertical_Component_Offset";

//!	COM parameters.
public static final String
	DATA_TYPE_PARAMETER						= "Data_Type",
	BINARY_DATA_PARAMETER					= "Binary_Data",
	TEXT_DATA_PARAMETER						= "Text_Data";
//!	Data type values.
public static final int
	DATA_TYPE_BINARY						= 0,
	DATA_TYPE_TEXT							= 1;

//------------------------------------------------------------------------------
//!	Codestream validity flags.
private int
	Codestream_Validity						= 0;

/**	{@link #Codestream_Validity() Codestream validity} bit mask
	for the SOC segment found flag.
*/
public static final int
	SOC_FOUND								= 1 << 0;
/**	{@link #Codestream_Validity() Codestream validity} bit mask
	for the SIZ segment found flag.
*/
public static final int
	SIZ_FOUND								= 1 << 1;
/**	{@link #Codestream_Validity() Codestream validity} bit mask
	for the COD segment found flag.
*/
public static final int
	COD_FOUND								= 1 << 2;
/**	{@link #Codestream_Validity() Codestream validity} bit mask
	for the QCD segment found flag.
*/
public static final int
	QCD_FOUND								= 1 << 3;
/**	{@link #Codestream_Validity() Codestream validity} bit mask
	for the EOC segment found flag.
*/
public static final int
	EOC_FOUND								= 1 << 4;
/**	{@link #Codestream_Validity() Codestream validity} value
	indicating that a complete codestream with all required segments
	were found.

	<b>N.B.</b>: The final EOC delimiter is not included as a
	required segment as it is usually not searched for.
*/
public static final int
	CODESTREAM_COMPLETE						= SOC_FOUND |
											  SIZ_FOUND |
											  COD_FOUND |
											  QCD_FOUND;

/**	Flag for missing EOC search through the codestream.
<p>
	If true and the End of Codestream ({@link #EOC_MARKER EOC}) is not
	found because the last tile-part segment has a zero length value, the
	remaining codestream data will be searched for the EOC marker. If
	false this search - which can be very time cosuming for a very large
	codestream segment - will not be done and the {@link
	#Codestream_Validity() codestream validity value} will not have the
	{@link #EOC_FOUND} flag set.
<p>
	By default the EOC search is disabled.
*/
public static boolean
	EOC_Search								= false;

/**	Flag whether a zero length tile-part was found in the codestream.
*/
private boolean
	Zero_Length_Tile_Part					= false;

//!	Image geometry information.
private long
	Reference_Grid_Width					= -1,
	Reference_Grid_Height					= -1,
	Image_Width								= -1,
	Image_Height							= -1,
	Horizontal_Image_Offset					= -1,
	Vertical_Image_Offset					= -1,
	Tile_Width								= -1,
	Tile_Height								= -1,
	Horizontal_Tile_Offset					= -1,
	Vertical_Tile_Offset					= -1;
	
private int[]
	Pixel_Height							= new int[0],
	Pixel_Width								= new int[0];

//!	Commonly accessed codestream info:
private int
	Total_Components						= -1,
	Total_Tiles								= -1,
	Total_Quality_Layers					= -1,
	Progression_Order						= -1,
	Total_Resolution_Levels					= -1,
	Transform								= -1;


//  DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_PARAMETERS		= 1 << 2,
	DEBUG_HELPERS			= 1 << 3,
	DEBUG_MAIN				= 1 << 4,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a JPEG2000_Codestream_Info for a named file.
<p>
	The inclusion of tile segments in the Parameters is controlled by the
	{@link JPEG2000_Info#Skip_Tiles_Default} flag.
<p>
	@param	filename	The filename from which to obtain the
		JPEG2000_Codestream_Info.
	@throws	IOException	If there was a problem reading the file.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	String	filename
	)
	throws IOException
{this (filename, Skip_Tiles_Default);}

/**	Construct a JPEG2000_Codestream_Info for a named file.
<p>
	@param	filename	The filename from which to obtain the
		JPEG2000_Codestream_Info.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	IOException	If there was a problem reading the file.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	String	filename,
	boolean	skip_tiles
	)
	throws IOException
{
super (filename);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info: " + filename);
Read (0, Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info");
}

/**	Construct a JPEG2000_Codestream_Info for a File.
<p>
	The inclusion of tile segments in the Parameters is controlled by the
	{@link JPEG2000_Info#Skip_Tiles_Default} flag.
<p>
	@param	file	The File from which to obtain the
		JPEG2000_Codestream_Info.
	@throws	IOException	If there was a problem reading the file.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	File	file
	)
	throws IOException
{this (file, Skip_Tiles_Default);}

/**	Construct a JPEG2000_Codestream_Info for a File.
<p>
	@param	file	The File from which to obtain the
		JPEG2000_Codestream_Info.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	IOException	If there was a problem reading the file.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	File	file,
	boolean	skip_tiles
	)
	throws IOException
{
super (file);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info: " + file);
Read (0, Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info");
}

/**	Construct a JPEG2000_Codestream_Info for an ImageInputStream.
<p>
	The inclusion of tile segments in the Parameters is controlled by the
	{@link JPEG2000_Info#Skip_Tiles_Default} flag.
<p>
	@param	image_input_stream	The ImageInputStream from which to obtain
		the JPEG2000_Codestream_Info.
	@throws	IOException	If there was a problem reading the file.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	ImageInputStream	image_input_stream
	)
	throws IOException
{this (image_input_stream, Skip_Tiles_Default);}

/**	Construct a JPEG2000_Codestream_Info for an ImageInputStream.
<p>
	@param	image_input_stream	The ImageInputStream from which to obtain
		the JPEG2000_Codestream_Info.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	IOException	If there was a problem reading the file.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	ImageInputStream	image_input_stream,
	boolean				skip_tiles
	)
	throws IOException
{
super (image_input_stream);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info: " + image_input_stream);
Read (0, Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info");
}

/**	Construct a JPEG2000_Codestream_Info for an ImageInputStream and
	Reads the stream up to an end position.
<p>
	@param	image_input_stream	The ImageInputStream containing a JPEG2000
		codestream. The stream is presumed to be positioned at the beginning
		of the codestream.
	@param	end_position	The byte position (exclusive) in the stream where
		the codestream ends.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	IOException	If a problem is encountered while reading the stream.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info
	(
	ImageInputStream	image_input_stream,
	long				end_position,
	boolean				skip_tiles
	)
	throws IOException
{
super (image_input_stream);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info: " + image_input_stream + '\n'
		+"    end_position " + end_position);
Read (end_position, Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info");
}

/**	Construct a an empty JPEG2000_Codestream_Info Object.
<p>
	Use one of the Source methods to assign a source file. Then use a
	Read method to obtain the information parameters.
<p>
	@see	JPEG2000_Info#Source(String)
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info ()
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">-< JPEG2000_Codestream_Info");
}

/*==============================================================================
	Accessors
*/
/**	Read JPEG2000 codestream information from the Source file.
<p>
	If no Source file has been assigned, nothing is done.
<p>
	The Source is read from its current position up to, but not
	including, the end position. If the end position is 0 the file is
	read until an End_of_Codestream segment is found, no more recognized
	segments are found, or the end of file. If an unrecognized marker is
	encountered the stream is repositioned back to the location where the
	marker was read.
<p>
	Any information Parameters currently present are removed. Then the
	file is scanned for JPEG2000 codestream marker segments. Each segment
	found is used to define the contents of a Parameter Group added to
	the base JPEG2000_Info Aggregate. The name of each Parameter Group is
	the name associated with the marker. The Group always contains the
	following parameters:
<p>
<dl>
<dt>Marker
	<dd>The marker code in numeric hexadecimal form.
<dt>^Position
	<dd>The position of the segment in the Source as a byte offset (from 0).
	<b>N.B.</b>: This parameter is not present if {@link
	JPEG2000_Info#Use_Data_Position(boolean) data position inclusion} is
	disabled.
<dt>Length
	<dd>The length of the segment in bytes.
</dl>
<p>
	The remaining Parameters in each Group depend on the segment type.
<p>
	When JP2 information boxes are nested, the corresponding Parameter
	Groups are also nested. JPEG2000 codestream (Contiguous_Codestream,
	jp2c) boxes contain a Codestream Parameter Group that is defined by a
	JPEG2000_Info object.
<p>
	<b>N.B.</b>: Unless {@link JPEG2000_Info#Throw_Illegal_State(boolean)
	illegal state detection} is disabled, a {@link #WARNING_PARAMETER}
	will be placed at the end of the Parameter set and further Source
	file scanning halted if the previous segment is invalid. The Value of
	the warning Parameter will be a String describing the problem.
	Possible sources of warning parameters are: The beginning of the
	codestream does not contain a valid Start_of_Codestream segment or
	the second segment is not a valid Size segment.
<p>
	@param	end_position	The location in the Source stream before which
		reading is to stop. If 0 reading stops at the first unrecognized
		segment marker or the end of file.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@return	This JPEG2000_Codestream_Info.
	@throws	IOException	If there was a problem reading the Source file.
	@see	JPEG2000_Info
*/
public JPEG2000_Codestream_Info Read
	(
	long	end_position,
	boolean	skip_tiles
	)
	throws IOException
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info.Read");
if (Image_Input_Stream == null)
	{
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("<<< JPEG2000_Codestream_Info.Read: Image_Input_Stream is null");
	return this;
	}
ByteOrder
	byte_order = Image_Input_Stream.getByteOrder ();
if (byte_order != ByteOrder.BIG_ENDIAN)
	Image_Input_Stream.setByteOrder (ByteOrder.BIG_ENDIAN);

//	Remove any and all current parameters.
Remove_All ();

//	Reset all codestream characterization values.
Reset_Codestream_Values ();

try
	{
	Stream_Position = Image_Input_Stream.getStreamPosition ();
	Add_Segments (this, end_position, skip_tiles);
	}
catch (IOException exception)
	{
	throw IO_Error
		("Unable to obtain JPEG2000 Codestream Info from Source"
			+ " at position " + Stream_Position,
		exception);
	}
catch (IllegalStateException exception)
	{
	try {Add (new Parameter (WARNING_PARAMETER)
			.Value (exception.getMessage ()));}
	catch (PVL_Exception except) {}
	}
finally
	{
	Image_Input_Stream.setByteOrder (byte_order);
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info.Read");
return this;
}

/**	Read JPEG2000 codestream information from the Source file.
<p>
	@return	This JPEG2000_Codestream_Info.
	@throws	IOException	If there was a problem reading the Source.
	@see	#Read(long, boolean)
*/
public JPEG2000_Codestream_Info Read ()
	throws IOException
{return Read (0, Skip_Tiles);}

/**	Test if the Source file contains a complete codestream.
<p>
	The Source must be {@link #Read(long, boolean) read} for this test to
	be valid.
<p>
	@return	true if the {@link #Codestream_Validity() Codestream
		Validity} value has all the {@link #CODESTREAM_COMPLETE} flags
		set; false otherwise.
*/
public boolean Is_Complete_Codestream ()
{return (Codestream_Validity & CODESTREAM_COMPLETE) == CODESTREAM_COMPLETE;}

/**	Get the codestream validity flags.
<p>
	@return	An int value holding the codestream validity flags. This
		will be 0 if the value is not yet known.
	@see	#SIZ_FOUND
	@see	#COD_FOUND
	@see	#QCD_FOUND
	@see	#EOC_FOUND
	@see	#CODESTREAM_COMPLETE
*/
public int Codestream_Validity ()
{return Codestream_Validity;}

/**	Get the total number of components identified in the codestream.
<p>
	This value corresponds to the value of the {@link
	#TOTAL_COMPONENTS_PARAMETER}.
<p>
	@return	The total number of components in the codestream. This will
		be -1 if the value is not yet known.
*/
public int Total_Components ()
{return Total_Components;}

/**	Get the width of the image.
<p>
	The image width .
<p>
	@return	The image width in reference grid units. This will be -1
		if the value is not yet known.
*/
public long Image_Width ()
{return Image_Width;}

/**	Get the height of the image.
<p>
	This value corresponds to the value of the {@link
	#REFERENCE_GRID_HEIGHT_PARAMETER}.
<p>
	@return	The image height in reference grid units. This will be -1
		if the value is not yet known.
*/
public long Image_Height ()
{return Image_Height;}

/**	Get the width of the image components reference grid.
<p>
	This value corresponds to the value of the {@link
	#REFERENCE_GRID_WIDTH_PARAMETER}.
<p>
	@return	The image components reference grid width. This will be -1
		if the value is not yet known.
*/
public long Reference_Grid_Width ()
{return Reference_Grid_Width;}

/**	Get the height of the image components reference grid.
<p>
	This value corresponds to the value of the {@link
	#REFERENCE_GRID_HEIGHT_PARAMETER}.
<p>
	@return	The image components reference grid height. This will be -1
		if the value is not yet known.
*/
public long Reference_Grid_Height ()
{return Reference_Grid_Height;}

/**	Get the horizontal offset of the image on the reference grid.
<p>
	The horizontal offset to the upper left corner of the image area
	will always be positive (to the right relative to the reference grid
	origin at the upper left corner of the reference grid). This value
	corresponds to the value of the {@link
	#HORIZONTAL_IMAGE_OFFSET_PARAMETER}.
<p>
	@return	The horizontal offset of the image on the reference grid.
		This will be -1 if the value is not yet known.
*/
public long Horizontal_Image_Offset ()
{return Horizontal_Image_Offset;}

/**	Get the vertical offset of the image on the reference grid.
<p>
	The vertical offset to the upper left corner of the image area will
	always be positive (downwards relative to the reference grid origin
	at the upper left corner of the reference grid). This value
	corresponds to the value of the {@link
	#VERTICAL_IMAGE_OFFSET_PARAMETER}.
<p>
	@return	The horizontal offset of the image on the reference grid.
		This will be -1 if the value is not yet known.
*/
public long Vertical_Image_Offset ()
{return Vertical_Image_Offset;}

/**	Get the width of a component's pixel.
<p>
	The width of a pixel is measured in reference grid units. Each
	component of the image may have a differenct pixel width. This
	value array corresponds to the array of the {@link
	#HORIZONTAL_SAMPLE_SPACING_PARAMETER}.
<p>
	@return	An int array of pixel width values in component order.
		This will be an empty array if the values are not yet known.
*/
public int[] Pixel_Width ()
{return Pixel_Width;}

/**	Get the height of a component's pixel.
<p>
	The height of a pixel is measured in reference grid units. Each
	component of the image may have a differenct pixel height. This
	value array corresponds to the array of the {@link
	#VERTICAL_SAMPLE_SPACING_PARAMETER}.
<p>
	@return	An int array of pixel height values in component order.
		This will be an empty array if the values are not yet known.
*/
public int[] Pixel_Height ()
{return Pixel_Height;}

/**	Get the tile width.
<p>
	This value corresponds to the value of the {@link
	#TILE_WIDTH_PARAMETER}.
<p>
	@return	The tile width measured in reference grid units. This will
		be -1 if the value is not yet known.
*/
public long Tile_Width ()
{return Tile_Width;}

/**	Get the tile height.
<p>
	This value corresponds to the value of the {@link
	#TILE_HEIGHT_PARAMETER}.
<p>
	@return	The tile height measured in reference grid units. This will
		be -1 if the value is not yet known.
*/
public long Tile_Height ()
{return Tile_Height;}

/**	Get the horizontal offset of the tiles on the reference grid.
<p>
	The horizontal offset to the upper left corner of the upper left
	tile will always be positive (to the right relative to the reference
	grid origin at the upper left corner of the reference grid) and no
	greather than the {@link #Horizontal_Image_Offset() horizontal image
	offset}. This value corresponds to the value of the {@link
	#HORIZONTAL_IMAGE_OFFSET_PARAMETER}.
<p>
	@return	The horizontal offset of the tiles on the reference grid.
		This will be -1 if the value is not yet known.
*/
public long Horizontal_Tile_Offset ()
{return Horizontal_Tile_Offset;}

/**	Get the vertical offset of the tiles on the reference grid.
<p>
	The vertical offset to the upper left corner of the upper left tile
	will always be positive (downwards relative to the reference grid
	origin at the upper left corner of the reference grid) and no
	greather than the {@link #Vertical_Image_Offset() vertical image
	offset}. This value corresponds to the value of the {@link
	#VERTICAL_IMAGE_OFFSET_PARAMETER}.
<p>
	@return	The horizontal offset of the tiles on the reference grid.
		This will be -1 if the value is not yet known.
*/
public long Vertical_Tile_Offset ()
{return Vertical_Tile_Offset;}

/**	Get the total tiles expected to be found in the codestream.
<p>
	@return	The total number of tiles expected, but not necessarily
		found, in the codestream. This will be -1 if the value is not
		yet known.
*/
public int Total_Tiles ()
{return Total_Tiles;}

/**	Test if a final tile-part of zero (unknown) length was found in the
	codestream.
<p>
	If a tile-part of zero length is found in the codestream it must be
	the last tile-part before the EOC marker. If {@link #EOC_Search} has
	been set the tile-part will be searched for the final EOC delimiter
	segment.
<p>
	@return	true if a zero length tile-part was found; false otherwise.
*/
public boolean Zero_Length_Tile_Part ()
{return Zero_Length_Tile_Part;}

/**	Get the total number of quality layers in the codestream.
<p>
	This value corresponds to the value of the {@link
	#TOTAL_QUALITY_LAYERS_PARAMETER}.
<p>
	@return	The total number of quality layers in the codestream. This
		will be -1 if the value is not yet known.
*/
public int Total_Quality_Layers ()
{return Total_Quality_Layers;}

/**	Get the codestream default progression order.
<p>
	Use the {@link #PROGRESSION_ORDERS} String array to obtain a
	description of the progression order code.
<p>
	This value corresponds to the value of the {@link
	#PROGRESSION_ORDER_PARAMETER}.
<p>
	@return	The codestream progression order code from the COD segment.
		The value may be one of {@link #LRCP_PROGRESSION_ORDER}, {@link
		#RLCP_PROGRESSION_ORDER}, {@link #RPCL_PROGRESSION_ORDER},
		{@link #PCRL_PROGRESSION_ORDER}, or {@link
		#CPRL_PROGRESSION_ORDER}. This will be -1 if the value is not
		yet known.
*/
public int Progression_Order ()
{return Progression_Order;}

/**	Get the total number of resolution levels.
<p>
	The value is obtained from the first coding style default (COD)
	segment encountered in the codestream. This is expected to be from
	the image and tile size (SIZ) segment. But if a tile header is
	independently parsed and it contains a COD segment then that will be
	used. <b>N.B.</b>: The value is never obtained from a coding style
	component (COC) segment. This value corresponds to the value of
	the {@link #TOTAL_RESOLUTION_LEVELS_PARAMETER}.
<p>
	The total number of resolution levels is one more than the maximum
	decomposition level.
<p>
	@return	The total number of resolution levels. This will be -1 if
		the value is not yet known.
*/
public int Total_Resolution_Levels ()
{return Total_Resolution_Levels;}

/**	Get the form of the codestream transformation.
<p>
	The value is obtained from the first coding style default (COD)
	segment encountered in the codestream. This is expected to be from
	the image and tile size (SIZ) segment. But if a tile header is
	independently parsed and it contains a COD segment then that will
	be used. <b>N.B.</b>: The value is never obtained from a coding
	style component (COC) segment. This value corresponds to the value
	of the {@link #TRANSFORM_PARAMETER}.
<p>
	@return	The codestream transformation indicator. This will be {@link
		#TRANSFORM_IRREVERSIBLE} (0) for the 9-7 irreversible filter; it
		will be {@link #TRANSFORM_REVERSIBLE} (1) for the 5-3 reversible
		filter. This will be -1 if the value is not yet known.
*/
public int Transform ()
{return Transform;}

private void Reset_Codestream_Values ()
{
Codestream_Validity		= 0;

Zero_Length_Tile_Part	= false;

Reference_Grid_Width	=
Reference_Grid_Height	=
Image_Width				=
Image_Height			=
Horizontal_Image_Offset	=
Vertical_Image_Offset	=
Tile_Width				=
Tile_Height				=
Horizontal_Tile_Offset	=
Vertical_Tile_Offset	= -1;

Total_Components		=
Total_Tiles				=
Total_Quality_Layers	=
Progression_Order		=
Total_Resolution_Levels	=
Transform				= -1;

Pixel_Width				= new int[0];
Pixel_Height			= new int[0];
}

/*==============================================================================
	Markers and Marker Segments
*/
/**	Assemble a Parameter Aggregate describing a sequence of codestream
	segments from a byte array.
<p>
	A sequence of codestream segments is parsed from the byte data. All
	segments found are parsed. {@link
	JPEG2000_Info#Use_Data_Position(boolean) Including data position}
	parameters is disabled as is {@link
	JPEG2000_Info#Throw_Illegal_State(boolean) illegal codestream segment
	organizaion detection}.
<p>
	@param	data	A byte array.
	@param	offset	The offset where the codestream segments are expected
		to start.
	@param	length	The maximum amount of byte data to be read. However,
		if the data array ends before length bytes after the starting
		offset, then only the available bytes will be read.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@return	A Parameter Aggregate containing Parameters describing the
		codestream segment contents. This will be null if the contents of
		a recognized segment were not available within the data length.
*/
public static Parameter Segments
	(
	byte[]	data,
	int		offset,
	int		length,
	boolean	skip_tiles
	)
{
if (data == null)
	return null;
JPEG2000_Codestream_Info
	info = new JPEG2000_Codestream_Info ();
info.Source (new MemoryCacheImageInputStream (new ByteArrayInputStream
	(data, offset, length)));
info.Source ().setByteOrder (ByteOrder.BIG_ENDIAN);
info.Use_Data_Position (false);
info.Throw_Illegal_State (false);
try {info.Read (0, skip_tiles);}
catch (Exception exception) {}
return (Parameter)info;
}

/**	Assemble a Parameter Aggregate describing a sequence of codestream
	segments from a byte array.
<p>
	The {@link #Skip_Tiles_Default} flag determines if tile segments
	will be skipped.
<p>
	@param	data	A byte array.
	@param	offset	The offset where the codestream segments are expected
		to start.
	@param	length	The maximum amount of byte data to be read. However,
		if the data array ends before length bytes after the starting
		offset, then only the available bytes will be read.
	@return	A Parameter Aggregate containing Parameters describing the
		codestream segment contents. This will be null if the contents of
		a recognized segment were not available within the data length.
	@see	#Segments(byte[], int, int, boolean)
*/
public static Parameter Segments
	(
	byte[]	data,
	int		offset,
	int		length
	)
{return Segments (data, offset, length, Skip_Tiles_Default);}

/**	Assemble a Parameter Aggregate describing a sequence of codestream
	segments from a byte array.
<p>
	@param	data	A byte array.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@return	A Parameter Aggregate containing Parameters describing the
		codestream segment contents. This will be null if the contents of
		a recognized segment were not available within the data length.
	@see	#Segments(byte[], int, int, boolean)
*/
public static Parameter Segments
	(
	byte[]	data,
	boolean	skip_tiles
	)
{return Segments (data, 0, data.length, skip_tiles);}

/**	Assemble a Parameter Aggregate describing a sequence of codestream
	segments from a byte array.
<p>
	The {@link #Skip_Tiles_Default} flag determines if tile segments
	will be skipped.
<p>
	@param	data	A byte array.
	@return	A Parameter Aggregate containing Parameters describing the
		codestream segment contents. This will be null if the contents of
		a recognized segment were not available within the data length.
	@see	#Segments(byte[], int, int, boolean)
*/
public static Parameter Segments
	(
	byte[]	data
	)
{return Segments (data, 0, data.length, Skip_Tiles_Default);}


private void Add_Segments
	(
	Parameter	container,
	long		end_position,
	boolean		skip_tiles
	)
	throws IOException, IllegalStateException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info.Add_Segments: To " + container.Name () + '\n'
		+"     from stream position " + Stream_Position + '\n'
		+"    until stream position " + end_position + '\n'
		+"    skip tiles: " + skip_tiles);
Parameter
	segment = null;
long
	total_tile_parts = 0;
if (end_position <= 0)
	end_position = Long.MAX_VALUE;
try
	{
	Get_Segments:
	while (Stream_Position < end_position &&
			(segment = Segment_Parameter ()) != null)
		{
		if (segment.Name ().equals (SOT_NAME))
			{
			//	Start of tile-part.
			long
				tile_part_end_position = 0;
			Parameter
				parameter;
			try
				{
				parameter = segment.Find (TILE_PART_LENGTH_PARAMETER);
				if (parameter != null)
					//	The length of the tile-part.
					tile_part_end_position = parameter.Value ().long_Data ();
				if (tile_part_end_position == 0)
					Zero_Length_Tile_Part = true;
				else
					tile_part_end_position =
						Stream_Position
							- segment.Find (LENGTH_PARAMETER)
								.Value ().long_Data ()
							+ tile_part_end_position;
				}
			catch (PVL_Exception exception)
				{
				//	Should never happen.
				segment = null;
				break;
				}
			if ((DEBUG & DEBUG_PARAMETERS) != 0)
				System.out.println
					("    tile-part " + total_tile_parts
						+ " end position = " + tile_part_end_position);

			if (skip_tiles)
				{
				if (Zero_Length_Tile_Part)
					//	Last tile-part before EOC, but unknown length.
					segment = Search_for_EOC (end_position);
				else
					{
					//	Move forward to the end of the tile-part.
					segment = null;	//	Don't include the tile-part segment.
					Stream_Position (tile_part_end_position);
					}
				break;
				}

			//	Add tile-part segments up to and including the Start_of_Data:
			++total_tile_parts;
			while (Stream_Position < end_position &&
					(parameter = Segment_Parameter ()) != null)
				{
				try {segment.Add (parameter);}
				catch (PVL_Exception exception)
					{
					//	Should never happen.
					segment = null;
					break Get_Segments;
					}
				if (parameter.Name ().equals (SOD_NAME))
					//	Start of bitstream data.
					break;
				if (tile_part_end_position != 0 &&
					tile_part_end_position <= Stream_Position)
					//	End of tile-part without SOD segment.
					break;
				}
			if (Zero_Length_Tile_Part)
				{
				//	Last tile-part before EOC, but unknown length.
				segment = Search_for_EOC (end_position);
				break;
				}
			else
				{
				if ((DEBUG & DEBUG_PARAMETERS) != 0)
					System.out.println
						("    Stream_Position after last tile-part segment = "
							+ Stream_Position);
				if (Stream_Position < tile_part_end_position)
					Stream_Position (tile_part_end_position);
				}
			}

		if (segment.Name ().equals (EOC_NAME))
			break;
		try {container.Add (segment);}
		catch (PVL_Exception exception) {}
		segment = null;
		}
	}
catch (EOFException exception)
	{
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    JPEG2000_Codestream_Info.Add_Segments: EOF at stream position "
				+ Stream_Position);
	}

if (! skip_tiles)
	{
	//	Total tile parts.
	try
		{
		container
			.Add (new Parameter (TOTAL_TILE_PARTS_PARAMETER)
			.Value (total_tile_parts)
			.Comments
				("\nTotal number of tile parts found in the codestream."));
		if (Total_Tiles > 0 &&
			Total_Tiles > total_tile_parts)
			container
				.Add (new Parameter (WARNING_PARAMETER)
				.Value
					("At least " + Total_Tiles + " tile parts were expected."));
		}
	catch (PVL_Exception exception) {}
	}

if (segment != null)
	{
	//	Final End of Codestream marker.
	try {container.Add (segment);}
	catch (PVL_Exception exception) {}
	}

if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info.Add_Segments");
}


private Parameter Segment_Parameter ()
	throws IOException, IllegalStateException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info.Segment_Parameter");
long
	segment_position = Stream_Position;
int
	marker = Read_Unsigned_short (MARKER_PARAMETER);
long
	segment_length =
		((marker == SOC_MARKER ||
		  marker == SOD_MARKER ||
		  marker == EPH_MARKER ||
		  marker == EOC_MARKER ||
		 (marker >= RESERVED_DELIMITER_MARKER_MIN &&
		  marker <= RESERVED_DELIMITER_MARKER_MAX)) ?
		//	Delimiting marker; no segment.
		2 :
		//	Segment length.
		Read_Unsigned_short (Marker_Name (marker) + ' ' + LENGTH_PARAMETER) + 2),
	end_position = segment_position + segment_length;

if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("    JPEG2000_Codestream_Info.Segment_Parameter: \""
			+ Marker_Name (marker)
			+ "\" (0x" + Integer.toHexString (marker) + ") -\n"
		+"    Position = " + segment_position + '\n'
		+"      Length = " + segment_length);
Parameter
	segment = new Parameter (Marker_Name (marker));
try
	{
	segment
		.Add (new Parameter (MARKER_PARAMETER)
			.Value (new Value (marker).Base (16)));
	if (Use_Data_Position)
		segment
			.Add (new Parameter (POSITION_PARAMETER)
				.Value (new Value (segment_position).Units ("byte offset")));
	segment
		.Add (new Parameter (LENGTH_PARAMETER)
			.Value (new Value (segment_length).Units ("bytes")));
	}
catch (PVL_Exception exception) {}

if (Codestream_Validity == 0)
	{
	//	The first marker must be the SOC.
	if (marker != SOC_MARKER)
		{
		if (Throw_Illegal_State)
			throw new IllegalStateException (ID + '\n'
				+ "Not a valid JPEG2000 codestream: No initial "
					+ Marker_Name (SOC_MARKER) + " marker.");
		}
	else
		Codestream_Validity = SOC_FOUND;
	}
else if (Codestream_Validity == SOC_FOUND)
	{
	//	The second marker must be the SIZ.
	if (marker != SIZ_MARKER)
		{
		if (Throw_Illegal_State)
			throw new IllegalStateException (ID + '\n'
				+ "Not a valid JPEG2000 codestream: Missing "
					+ Marker_Name (SIZ_MARKER) + " marker.");
		}
	else
		Codestream_Validity |= SIZ_FOUND;
	}
if (marker < RESERVED_DELIMITER_MARKER_MIN ||
	marker > RESERVED_DELIMITER_MARKER_MIN)
	{
	switch (marker)
		{
		//	Delimiting markers.
		case EOC_MARKER:
			Codestream_Validity |= EOC_FOUND;
		case SOC_MARKER:
		case SOD_MARKER:
		case EPH_MARKER:
			break;

		//	Segments.
		case SOT_MARKER:	SOT_Parameters (segment);	break;
		case SIZ_MARKER:	SIZ_Parameters (segment);	break;
		case COD_MARKER:	COD_Parameters (segment);
			Codestream_Validity |= COD_FOUND;
			break;
		case COC_MARKER:	COC_Parameters (segment, end_position);	break;
		case RGN_MARKER:	RGN_Parameters (segment, end_position);	break;
		case QCD_MARKER:	QCD_Parameters (segment, end_position);
			Codestream_Validity |= QCD_FOUND;
			break;
		case QCC_MARKER:	QCC_Parameters (segment, end_position);	break;
		case POC_MARKER:	POC_Parameters (segment, end_position);	break;
		case TLM_MARKER:	TLM_Parameters (segment, end_position);	break;
		case PLM_MARKER:	PLM_Parameters (segment, end_position);	break;
		case PLT_MARKER:	PLT_Parameters (segment, end_position);	break;
		case PPM_MARKER:	PPM_Parameters (segment, end_position);	break;
		case PPT_MARKER:	PPT_Parameters (segment, end_position);	break;
		case SOP_MARKER:	SOP_Parameters (segment);	break;
		case CRG_MARKER:	CRG_Parameters (segment, end_position);	break;
		case COM_MARKER:	COM_Parameters (segment, end_position);	break;
		}
	}

if (Stream_Position < end_position)
	Stream_Position  (end_position);
else
if (Throw_Illegal_State &&
	Stream_Position > end_position)
	throw new IllegalStateException (ID + '\n'
		+ "Invalid " + segment.Name () + " contents -\n"
		+ "  expected length " + segment_length
			+ " but used length " + (Stream_Position - segment_position));
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(((segment == null) ? "" : (segment.Description ()) + '\n')
		+"<<< JPEG2000_Codestream_Info.Segment_Parameter: " + segment);
return segment;
}


private void SOT_Parameters
	(
	Parameter	segment
	)
	throws IOException
{
try {
segment
	.Add (new Parameter (TILE_INDEX_PARAMETER)
		.Value (Read_Unsigned_short (segment.Name () + ' ' +
			TILE_INDEX_PARAMETER)))
	.Add (new Parameter (TILE_PART_LENGTH_PARAMETER)
		.Value (new Value (Read_Unsigned_int (segment.Name () + ' ' +
			TILE_PART_LENGTH_PARAMETER)).Units ("bytes")))
	.Add (new Parameter (TILE_PART_INDEX_PARAMETER)
		.Value (Read_Unsigned_byte (segment.Name () + ' ' +
			TILE_PART_INDEX_PARAMETER)))
	.Add (new Parameter (TOTAL_TILE_PARTS_PARAMETER)
		.Value (Read_Unsigned_byte (segment.Name () + ' ' +
			TOTAL_TILE_PARTS_PARAMETER)));
}
catch (PVL_Exception exception) {}
}


private void SIZ_Parameters
	(
	Parameter	segment
	)
	throws IOException
{
try {
segment
	.Add (new Parameter (CAPABILITY_PARAMETER)
		.Value (new Value (Read_Unsigned_short (segment.Name () + ' ' +
			CAPABILITY_PARAMETER))
			.Base (16)))
	.Add (new Parameter (REFERENCE_GRID_WIDTH_PARAMETER)
		.Value (Reference_Grid_Width = Read_Unsigned_int (segment.Name () + ' ' +
			REFERENCE_GRID_WIDTH_PARAMETER)))
	.Add (new Parameter (REFERENCE_GRID_HEIGHT_PARAMETER)
		.Value (Reference_Grid_Height = Read_Unsigned_int (segment.Name () + ' ' +
			REFERENCE_GRID_HEIGHT_PARAMETER)))
	.Add (new Parameter (HORIZONTAL_IMAGE_OFFSET_PARAMETER)
		.Value (Horizontal_Image_Offset = Read_Unsigned_int (segment.Name () + ' ' +
			HORIZONTAL_IMAGE_OFFSET_PARAMETER)))
	.Add (new Parameter (VERTICAL_IMAGE_OFFSET_PARAMETER)
		.Value (Vertical_Image_Offset = Read_Unsigned_int (segment.Name () + ' ' +
			VERTICAL_IMAGE_OFFSET_PARAMETER)))
	.Add (new Parameter (TILE_WIDTH_PARAMETER)
		.Value (Tile_Width = Read_Unsigned_int (segment.Name () + ' ' +
			TILE_WIDTH_PARAMETER)))
	.Add (new Parameter (TILE_HEIGHT_PARAMETER)
		.Value (Tile_Height = Read_Unsigned_int (segment.Name () + ' ' +
			TILE_HEIGHT_PARAMETER)))
	.Add (new Parameter (HORIZONTAL_TILE_OFFSET_PARAMETER)
		.Value (Horizontal_Tile_Offset = Read_Unsigned_int (segment.Name () + ' ' +
			HORIZONTAL_TILE_OFFSET_PARAMETER)))
	.Add (new Parameter (VERTICAL_TILE_OFFSET_PARAMETER)
		.Value (Vertical_Tile_Offset = Read_Unsigned_int (segment.Name () + ' ' +
			VERTICAL_TILE_OFFSET_PARAMETER)))
	.Add (new Parameter (TOTAL_COMPONENTS_PARAMETER)
		.Value (Total_Components =
			Read_Unsigned_short (segment.Name () + ' ' +
			TOTAL_COMPONENTS_PARAMETER)));

Image_Width  = Reference_Grid_Width  - Horizontal_Image_Offset;
Image_Height = Reference_Grid_Height - Vertical_Image_Offset;

Value
	Ssiz  = new Value ().Type (Value.SEQUENCE),
	XRsiz = new Value ().Type (Value.SEQUENCE),
	YRsiz = new Value ().Type (Value.SEQUENCE);
for (int component = 0;
		 component < Total_Components;
	   ++component)
	{
	Ssiz.Add (new Value (Read_byte (segment.Name () + ' ' +
		VALUE_BITS_PARAMETER + ' ' + component)));
	XRsiz.Add (new Value (Read_Unsigned_byte (segment.Name () + ' ' +
		HORIZONTAL_SAMPLE_SPACING_PARAMETER + ' ' + component)));
	YRsiz.Add (new Value (Read_Unsigned_byte (segment.Name () + ' ' +
		VERTICAL_SAMPLE_SPACING_PARAMETER + ' ' + component)));
	}
segment
	.Add (Value_Bits_Parameter (Ssiz))
	.Add (new Parameter (HORIZONTAL_SAMPLE_SPACING_PARAMETER)
		.Value (XRsiz))
	.Add (new Parameter (VERTICAL_SAMPLE_SPACING_PARAMETER)
		.Value (YRsiz));
Pixel_Width  = int_Array (XRsiz);
Pixel_Height = int_Array (YRsiz);

Total_Tiles = (int)(
	Math.ceil (((double)Reference_Grid_Width  - Horizontal_Tile_Offset)
		/ Tile_Width) *
	Math.ceil (((double)Reference_Grid_Height - Vertical_Tile_Offset)
		/ Tile_Height));
}
catch (PVL_Exception exception) {}
}


private void COD_Parameters
	(
	Parameter	segment
	)
	throws IOException
{
try {
//	Scod:
int
	Scod = Read_Unsigned_byte (segment.Name () + ' ' + CODING_STYLE_PARAMETER);
segment.Add (new Parameter (CODING_STYLE_PARAMETER)
	.Comments ("\nEntropy coder precincts:\n" +
		"  Precinct size " +
		(((Scod & ENTROPY_CODER_FLAG) == 0) ?
			"= 32768 x 32768\n" :
			("defined in the " + PRECINCT_SIZE_PARAMETER + " parameter.\n")) +
		(((Scod & SOP_FLAG) == 0) ? "  No " : "  ")
			+ "SOP marker segments used\n" +
		(((Scod & EPH_FLAG) == 0) ? "  No " : "  ")
			+ "EPH marker used")
	.Value (new Value (Scod)
		.Base (16)));

//	SGcod:
Progression_Order = Read_Unsigned_byte (segment.Name () + ' ' +
	PROGRESSION_ORDER_PARAMETER);
segment
	.Add (new Parameter (PROGRESSION_ORDER_PARAMETER)
		.Comments ("\nProgression order:\n  " +
			((Progression_Order < PROGRESSION_ORDERS.length) ?
				PROGRESSION_ORDERS[Progression_Order] : "Unknown"))
		.Value (Progression_Order))
	.Add (new Parameter (TOTAL_QUALITY_LAYERS_PARAMETER)
		.Value (Total_Quality_Layers = Read_Unsigned_short (segment.Name () + ' ' +
			TOTAL_QUALITY_LAYERS_PARAMETER)))
	.Add (new Parameter (MULTIPLE_COMPONENT_TRANSFORM_PARAMETER)
		.Value (Read_Unsigned_byte (segment.Name () + ' ' +
			MULTIPLE_COMPONENT_TRANSFORM_PARAMETER)));

//	SPcod:
Coding_Style_Parameters (segment, (Scod & ENTROPY_CODER_FLAG) != 0);

//	Set the default (highest level) resolution levels and transform type.
Parameter
	parameter;
if (Total_Resolution_Levels < 0 &&
	(parameter = segment.Find (TOTAL_RESOLUTION_LEVELS_PARAMETER)) != null)
	Total_Resolution_Levels = (int)(parameter.Value ().long_Data ());
if (Transform < 0 &&
	(parameter = segment.Find (TRANSFORM_PARAMETER)) != null)
	Transform = (int)(parameter.Value ().long_Data ());
}
catch (PVL_Exception exception) {}
}


private void COC_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
/*
	CAUTION: The size of the COMPONENT_INDEX_PARAMETER is dependent on
	the value of Total_Components. If Total_Components has not yet been
	set from a COD segment it is not possible to parse this segment.
*/
if (Total_Components < 0)
	{
	segment.Add (new Parameter (WARNING_PARAMETER)
		.Value ("Can not parse COC segment because total components unknown."));
	if (Use_Data_Position)
		Data_Position_Parameters (segment, end_position);
	else
		Data_Offset_Parameters (segment,
			0, end_position - Stream_Position);
	}
else
	{
	segment.Add (new Parameter (COMPONENT_INDEX_PARAMETER)
		.Value ((Total_Components < 257) ?
			Read_Unsigned_byte (segment.Name () + ' ' +
				COMPONENT_INDEX_PARAMETER) :
			Read_Unsigned_short (segment.Name () + ' ' +
				COMPONENT_INDEX_PARAMETER)));

	//	Scoc:
	int
		Scoc =
			Read_Unsigned_byte (segment.Name () + ' ' +
				CODING_STYLE_PARAMETER);
	segment.Add (new Parameter (CODING_STYLE_PARAMETER)
		.Comments ("\nEntropy coder precincts:\n" +
			"  Precinct size " +
			(((Scoc & ENTROPY_CODER_FLAG) == 0) ?
				"= 32768 x 32768" :
				("defined in the " + PRECINCT_SIZE_PARAMETER + " parameter.")))
		.Value (new Value (Scoc)
			.Base (16)));

	//	SPcoc:
	Coding_Style_Parameters (segment, (Scoc & ENTROPY_CODER_FLAG) != 0);
	}
}
catch (PVL_Exception exception) {}
}


private void Coding_Style_Parameters
	(
	Parameter	segment,
	boolean		has_precinct_sizes
	)
	throws IOException
{
try {
int
	levels,
	datum;
segment
	.Add (new Parameter (TOTAL_RESOLUTION_LEVELS_PARAMETER)
		.Value (levels = Read_Unsigned_byte (segment.Name () + ' ' +
			TOTAL_RESOLUTION_LEVELS_PARAMETER) + 1))
	.Add (new Parameter (CODE_BLOCK_WIDTH_PARAMETER)
		.Value (1 << ((datum = Read_Unsigned_byte (segment.Name () + ' ' +
			CODE_BLOCK_WIDTH_PARAMETER)) + 2))
		.Comments ("\nCode-block width exponent offset " + datum + '.'))
	.Add (new Parameter (CODE_BLOCK_HEIGHT_PARAMETER)
		.Value (1 << ((datum = Read_Unsigned_byte (segment.Name () + ' ' +
			CODE_BLOCK_HEIGHT_PARAMETER)) + 2))
		.Comments ("\nCode-block height exponent offset " + datum + '.'));

datum = Read_Unsigned_byte (segment.Name () + ' ' + CODE_BLOCK_STYLE_PARAMETER);
segment.Add (new Parameter (CODE_BLOCK_STYLE_PARAMETER)
	.Comments ("\nCode-block style:\n" +
		(((datum & SELECTIVE_ARITHMETIC_BYPASS_FLAG) == 0) ? "  No s" : "  S") +
			"elective arithmetic coding bypass.\n" +
		(((datum & RESET_CONTEXT_PROBABILITIES) == 0) ? "  No r" : "  R") +
			"eset of context probabilities on coding pass boundaries.\n" +
		(((datum & TERMINATION_FLAG) == 0) ? "  No t" : "  T") +
			"ermination on each coding pass.\n" +
		(((datum & VERTICALLY_CAUSAL_CONTEXT_FLAG) == 0) ? "  No v" : "  V") +
			"erticallly causal context.\n" +
		(((datum & PREDICTABLE_TERMINATION_FLAG) == 0) ? "  No p" : "  P") +
			"redictable termination.\n" +
		(((datum & SEGMENTATION_SYMBOLS_FLAG) == 0) ? "  No s" : "  S") +
			"egmentation symbols are used.")
	.Value (new Value (datum).Base (16)));

datum = Read_Unsigned_byte (segment.Name () + ' ' + TRANSFORM_PARAMETER);
segment.Add (new Parameter (TRANSFORM_PARAMETER)
	.Comments ("\nWavelet transformation used:\n  " +
		((datum == TRANSFORM_IRREVERSIBLE) ?
			"9-7 irreversible filter." :
		((datum == TRANSFORM_REVERSIBLE) ?
			"5-3 reversible filter." :
			"Unknown")))
	.Value (datum));

if (has_precinct_sizes)
	{
	Value
		list = new Value ().Type (Value.SEQUENCE),
		pair;
	for (int level = 0;
			 level < levels;
		   ++level)
		{
		pair = new Value ().Type (Value.SEQUENCE);
		datum = Read_Unsigned_byte (segment.Name () + ' ' +
			PRECINCT_SIZE_PARAMETER + ' ' + level);
		pair
			.Add (new Value (1 << (datum & 0x0F)))
			.Add (new Value (1 << ((datum & 0xF0) >> 4)));
		list.Add (pair);
		}
	segment.Add (new Parameter (PRECINCT_SIZE_PARAMETER)
		.Comments ("\nPrecinct (width, height) by resolution level.")
		.Value (list));
	}
}
catch (PVL_Exception exception) {}
}


private void RGN_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
/*
	CAUTION: The size of the COMPONENT_INDEX_PARAMETER is dependent on
	the value of Total_Components. If Total_Components has not yet been
	set from a COD segment it is not possible to parse this segment.
*/
if (Total_Components < 0)
	{
	segment.Add (new Parameter (WARNING_PARAMETER)
		.Value ("Can not parse RGN segment because total components unknown."));
	if (Use_Data_Position)
		Data_Position_Parameters (segment, end_position);
	else
		Data_Offset_Parameters (segment,
			0, end_position - Stream_Position);
	}
else
	segment
		.Add (new Parameter (COMPONENT_INDEX_PARAMETER)
			.Value ((Total_Components < 257) ?
				Read_Unsigned_byte (segment.Name () + ' ' +
					COMPONENT_INDEX_PARAMETER) :
				Read_Unsigned_short (segment.Name () + ' ' +
					COMPONENT_INDEX_PARAMETER)))
		.Add (new Parameter (ROI_STYLE_PARAMETER)
			.Value (Read_Unsigned_byte (segment.Name () + ' ' +
				ROI_STYLE_PARAMETER)))
		.Add (new Parameter (IMPLICIT_SHIFT_PARAMETER)
			.Value (Read_Unsigned_byte (segment.Name () + ' ' +
				IMPLICIT_SHIFT_PARAMETER)));
}
catch (PVL_Exception exception) {}
}


private void QCD_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{Quantization_Parameters (segment, end_position);}


private void QCC_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
/*
	CAUTION: The size of the COMPONENT_INDEX_PARAMETER is dependent on
	the value of Total_Components. If Total_Components has not yet been
	set from a COD segment it is not possible to parse this segment.
*/
if (Total_Components < 0)
	{
	segment.Add (new Parameter (WARNING_PARAMETER)
		.Value ("Can not parse QCC segment because total components unknown."));
	if (Use_Data_Position)
		Data_Position_Parameters (segment, end_position);
	else
		Data_Offset_Parameters (segment,
			0, end_position - Stream_Position);
	}
else
	{
	segment.Add (new Parameter (COMPONENT_INDEX_PARAMETER)
		.Value ((Total_Components < 257) ?
			Read_Unsigned_byte (segment.Name () + ' ' +
				COMPONENT_INDEX_PARAMETER) :
			Read_Unsigned_short (segment.Name () + ' ' +
				COMPONENT_INDEX_PARAMETER)));

	Quantization_Parameters (segment, end_position);
	}
}
catch (PVL_Exception exception) {}
}


private void Quantization_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
int
	datum = Read_Unsigned_byte (segment.Name () + ' ' +
		QUANTIZATION_STYLE_PARAMETER),
	style = datum & 3;
segment.Add (new Parameter (QUANTIZATION_STYLE_PARAMETER)
	.Comments ("\nQuantization:\n  " +
		((style == NO_QUANTIZATION) ? "None" :
		((style == QUANTIZATION_SCALAR_DERIVED) ?
			"Scalar derived; values signalled for N-subL LL sub-band only." :
		((style == QUANTIZATION_SCALAR_EXPOUNDED) ?
			"Scalar expounded; values signalled for each sub-band." :
			"Unknown"))))
	.Value (style));
segment.Add (new Parameter (TOTAL_GUARD_BITS_PARAMETER)
	.Value ((datum & 0xE0) >> 5));
Value
	list = new Value ().Type (Value.SEQUENCE),
	entry;
while (Stream_Position < end_position)
	{
	//	Exponent
	datum = Read_Unsigned_byte (segment.Name () + ' ' + STEP_SIZE_PARAMETER);
	if (style == NO_QUANTIZATION)
		entry = new Value ((datum & 0xF8) >> 3);
	else
		{
		//	Mantissa
		entry = new Value ().Type (Value.SEQUENCE);
		entry.Add (new Value ((datum & 0xF8) >> 3));
		datum <<= 8;
		datum |= Read_Unsigned_byte (segment.Name () + ' ' +
			STEP_SIZE_PARAMETER);
		entry.Add (new Value (datum & 0x07FF));
		}
	list.Add (entry);
	}
segment.Add (new Parameter (STEP_SIZE_PARAMETER)
	.Comments ("\n" + ((style == NO_QUANTIZATION) ?
		"Reversible transform dynamic range exponent" :
		"Irreversible transform quantization step size (exponent, mantissa)")
		+ " by sub-band.")
	.Value (list));
}
catch (PVL_Exception exception) {}
}


private void POC_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
/*
	CAUTION: The size of the COMPONENT_INDEX_PARAMETER is dependent on
	the value of Total_Components. If Total_Components has not yet been
	set from a COD segment it is not possible to parse this segment.
*/
if (Total_Components < 0)
	{
	segment.Add (new Parameter (WARNING_PARAMETER)
		.Value ("Can not parse POC segment because total components unknown."));
	if (Use_Data_Position)
		Data_Position_Parameters (segment, end_position);
	else
		Data_Offset_Parameters (segment,
			0, end_position - Stream_Position);
	}
else
	{
	Value
		levels = new Value ().Type (Value.SEQUENCE),
		components = new Value ().Type (Value.SEQUENCE),
		layers = new Value ().Type (Value.SEQUENCE),
		progression_orders = new Value ().Type (Value.SEQUENCE),
		levels_range,
		components_range;
	int
		datum;
	while (Stream_Position < end_position)
		{
		levels_range = new Value ().Type (Value.SEQUENCE)
			.Add (new Value (Read_Unsigned_byte (segment.Name () + ' ' +
				LEVEL_INDEX_PARAMETER)));
		components_range = new Value ().Type (Value.SEQUENCE)
			.Add (new Value ((Total_Components < 257) ?
				Read_Unsigned_byte (segment.Name () + ' ' +
					COMPONENT_INDEX_PARAMETER) :
				Read_Unsigned_short (segment.Name () + ' ' +
					COMPONENT_INDEX_PARAMETER)));
		layers.Add (new Value (Read_Unsigned_short (segment.Name () + ' ' +
				LAYER_INDEX_PARAMETER)));
		levels_range.Add (new Value (Read_Unsigned_byte (segment.Name () + ' ' +
				LEVEL_INDEX_PARAMETER)));
		levels.Add (levels_range);
		datum = (Total_Components < 257) ?
				Read_Unsigned_byte (segment.Name () + ' ' +
					COMPONENT_INDEX_PARAMETER) :
				Read_Unsigned_short (segment.Name () + ' ' +
					COMPONENT_INDEX_PARAMETER);
		if (datum == 0)
			datum = 256;
		components_range.Add (new Value (datum));
		components.Add (components_range);
		progression_orders.Add (new Value (Read_Unsigned_byte (segment.Name () + ' ' +
				PROGRESSION_ORDER_PARAMETER)));
		}
	String
		comment = "\nIndex (start, end] ranges.";
	segment
		.Add (new Parameter (LEVEL_INDEX_PARAMETER)
			.Comments (comment)
			.Value (levels))
		.Add (new Parameter (COMPONENT_INDEX_PARAMETER)
			.Comments (comment)
			.Value (components))
		.Add (new Parameter (LAYER_INDEX_PARAMETER)
			.Value (layers))
		.Add (new Parameter (PROGRESSION_ORDER_PARAMETER)
			.Value (progression_orders));
	}
}
catch (PVL_Exception exception) {}
}


private void TLM_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
segment.Add (new Parameter (INDEX_PARAMETER)
	.Value (Read_Unsigned_byte (segment.Name () + ' ' + INDEX_PARAMETER)));

int
	index_size,
	length_size = Read_Unsigned_byte (segment.Name () + ' ' +
		TILE_INDEX_SIZE_PARAMETER + " and " + TILE_PART_LENGTH_SIZE_PARAMETER);
length_size >>= 4;
index_size = length_size & 0x3;
length_size = (((length_size >> 2) & 0x3) + 1) * 2;
segment
	.Add (new Parameter (TILE_INDEX_SIZE_PARAMETER)
		.Value (new Value (index_size).Units ("bytes")))
	.Add (new Parameter (TILE_PART_LENGTH_SIZE_PARAMETER)
		.Value (new Value (length_size).Units ("bytes")));

Value
	indices = new Value ().Type (Value.SEQUENCE),
	lengths = new Value ().Type (Value.SEQUENCE).Units ("bytes");
while (Stream_Position < end_position)
	{
	if (index_size != 0)
		indices.Add (new Value ((index_size == 1) ?
			Read_Unsigned_byte  (segment.Name () + ' ' +
				TILE_INDEX_PARAMETER) :
			Read_Unsigned_short (segment.Name () + ' ' +
				TILE_INDEX_PARAMETER)));
	lengths.Add (new Value ((length_size == 2) ?
			Read_Unsigned_short (segment.Name () + ' ' +
				TILE_PART_LENGTH_PARAMETER) :
			Read_Unsigned_int   (segment.Name () + ' ' +
				TILE_PART_LENGTH_PARAMETER)));
	}
if (index_size != 0)
	segment.Add (new Parameter (TILE_INDEX_PARAMETER)
		.Value (indices));
segment.Add (new Parameter (TILE_PART_LENGTH_PARAMETER)
	.Value (lengths));
}
catch (PVL_Exception exception) {}
}


private void PLM_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
segment.Add (new Parameter (INDEX_PARAMETER)
	.Value (Read_Unsigned_byte (segment.Name () + ' ' + INDEX_PARAMETER)));

Value
	lengths;
if (PLM_Packet_Length_Parameter == null)
	//	First PLM segment; add the new packet lengths parameter and array.
	segment.Add (PLM_Packet_Length_Parameter =
		new Parameter (PACKET_LENGTH_PARAMETER)
			.Value (lengths =
				new Value ().Type (Value.SEQUENCE).Units ("bytes")));
else
	{
	//	Continuation PLM segment; append to the existing lengths array.
	lengths = PLM_Packet_Length_Parameter.Value ();
	segment.Add (new Parameter (CONTINUATION_PARAMETER)
		.Value ("true"));
	}
long
	datum;
while (Stream_Position < end_position)
	{
	if (PLM_Packet_Length_Bytes_Remaining == 0)
		PLM_Packet_Length_Bytes_Remaining =
			Read_Unsigned_byte (segment.Name () + " byte count");
	datum = Read_Unsigned_byte (segment.Name () + ' ' + PACKET_LENGTH_PARAMETER);
	--PLM_Packet_Length_Bytes_Remaining;
	PLM_Packet_Length <<= 7;
	PLM_Packet_Length |= datum & 0x7F;
	if (PLM_Packet_Length_Bytes_Remaining == 0)
		{
		lengths.Add (new Value (PLM_Packet_Length));
		PLM_Packet_Length = 0;
		}
	}
}
catch (PVL_Exception exception) {}
}


private void PLT_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
segment.Add (new Parameter (INDEX_PARAMETER)
	.Value (Read_Unsigned_byte (segment.Name () + ' ' + INDEX_PARAMETER)));

Value
	lengths;
if (PLT_Packet_Length_Parameter == null)
	//	First PLT segment; add the new packet lengths parameter and array.
	segment.Add (PLT_Packet_Length_Parameter =
		new Parameter (PACKET_LENGTH_PARAMETER)
			.Value (lengths =
				new Value ().Type (Value.SEQUENCE).Units ("bytes")));
else
	{
	//	Continuation PLT segment; append to the existing lengths array.
	lengths = PLT_Packet_Length_Parameter.Value ();
	segment.Add (new Parameter (CONTINUATION_PARAMETER)
		.Value ("true"));
	}
long
	datum;
while (Stream_Position < end_position)
	{
	datum = Read_Unsigned_byte (segment.Name () + ' ' + PACKET_LENGTH_PARAMETER);
	PLT_Packet_Length <<= 7;
	PLT_Packet_Length |= datum & 0x7F;
	if ((PLT_Packet_Length_Bytes_Remaining = (datum & 0x80)) == 0)
		{
		lengths.Add (new Value (PLT_Packet_Length));
		PLT_Packet_Length = 0;
		}
	}
}
catch (PVL_Exception exception) {}
}


private void PPM_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
segment.Add (new Parameter (INDEX_PARAMETER)
	.Value (Read_Unsigned_byte (segment.Name () + ' ' + INDEX_PARAMETER)));
if (Use_Data_Position)
	Data_Position_Parameters (segment, end_position);
else
	Data_Offset_Parameters (segment,
		1, end_position - Stream_Position);
}
catch (PVL_Exception exception) {}
}


private void PPT_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
segment.Add (new Parameter (INDEX_PARAMETER)
	.Value (Read_Unsigned_byte (segment.Name () + ' ' + INDEX_PARAMETER)));
if (Use_Data_Position)
	Data_Position_Parameters (segment, end_position);
else
	Data_Offset_Parameters (segment,
		1, end_position - Stream_Position);
}
catch (PVL_Exception exception) {}
}


private void SOP_Parameters
	(
	Parameter	segment
	)
	throws IOException
{
try {
segment.Add (new Parameter (INDEX_PARAMETER)
	.Value (Read_Unsigned_short (segment.Name () + ' ' + INDEX_PARAMETER)));
}
catch (PVL_Exception exception) {}
}


private void CRG_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
Value
	horizontal_offsets = new Value ().Type (Value.SEQUENCE),
	vertical_offsets = new Value ().Type (Value.SEQUENCE);
while (Stream_Position < end_position)
	{
	horizontal_offsets.Add (new Value (Read_Unsigned_short (segment.Name () + ' ' +
		HORIZONTAL_COMPONENT_OFFSET_PARAMETER)));
	vertical_offsets.Add (new Value (Read_Unsigned_short (segment.Name () + ' ' +
		VERTICAL_COMPONENT_OFFSET_PARAMETER)));
	}
segment
	.Add (new Parameter (HORIZONTAL_COMPONENT_OFFSET_PARAMETER)
		.Value (horizontal_offsets)
		.Comments ("\nOffset in units of 1/65536 of the "
			+ HORIZONTAL_SAMPLE_SPACING_PARAMETER + " by component."))
	.Add (new Parameter (VERTICAL_COMPONENT_OFFSET_PARAMETER)
		.Value (vertical_offsets)
		.Comments ("\nOffset in units of 1/65536 of the "
			+ VERTICAL_SAMPLE_SPACING_PARAMETER + " by component."));
}
catch (PVL_Exception exception) {}
}


private void COM_Parameters
	(
	Parameter	segment,
	long		end_position
	)
	throws IOException
{
try {
int
	datum;
segment.Add (new Parameter (DATA_TYPE_PARAMETER)
	.Value (datum = Read_Unsigned_short (segment.Name () + ' ' +
		DATA_TYPE_PARAMETER)));

if (datum == DATA_TYPE_TEXT)
	segment.Add (new Parameter (TEXT_DATA_PARAMETER)
		.Value (new Value (Read_String (segment.Name (), end_position))
			.Type (Value.TEXT)));
else
	{
	//	DATA_TYPE_BINARY
	Value
		sequence = new Value ().Type (Value.SEQUENCE);
	for (long index = Stream_Position ();
			  index < end_position;
			++index)
		sequence.Add (new Value (Read_Unsigned_byte (segment.Name () + ' ' +
			BINARY_DATA_PARAMETER)).Base (16));
	segment.Add (new Parameter (BINARY_DATA_PARAMETER)
		.Value (sequence));
	}
}
catch (PVL_Exception exception) {}
}


private Parameter Search_for_EOC
	(
	long		end_position
	)
	throws IOException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JPEG2000_Codestream_Info.Search_for_EOC:\n"
		+"    from position " + Stream_Position + '\n'
		+"      to position " + end_position);
if (! EOC_Search)
	{
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    EOC_Search disabled\n"
			+"<<< JPEG2000_Codestream_Info.Search_for_EOC");
	return null;
	}

Parameter
	segment = null;
while (Stream_Position < end_position)
	{
	if (Read_Unsigned_byte (EOC_NAME + " MSB") == (EOC_MARKER >>> 8) &&
		Read_Unsigned_byte (EOC_NAME + " LSB") == (EOC_MARKER & 0xFF))
		{
		if ((DEBUG & DEBUG_PARAMETERS) != 0)
			System.out.println
				("    EOC found at position " + (Stream_Position - 2));
		Codestream_Validity |= EOC_FOUND;
		segment = new Parameter (EOC_NAME);
		try
			{
			segment
				.Add (new Parameter (MARKER_PARAMETER)
					.Value (new Value (EOC_MARKER).Base (16)));
			if (Use_Data_Position)
				segment
					.Add (new Parameter (POSITION_PARAMETER)
						.Value (new Value (Stream_Position - 2)
						.Units ("byte offset")));
			segment
				.Add (new Parameter (LENGTH_PARAMETER)
					.Value (new Value (2).Units ("bytes")));
			}
		catch (PVL_Exception exception) {}
		break;
		}
	}
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JPEG2000_Codestream_Info.Search_for_EOC");
return segment;
}

/*==============================================================================
	Helpers
*/
public static String Marker_Name
	(
	int		marker
	)
	throws IOException
{
String
	string = (String)Marker_Names.get (new Integer (marker));
if (string == null)
	string = UNKNOWN;
return string;
}


private IOException IO_Error
	(
	String		message,
	Exception	exception
	)
{
if (message == null)
	message = ID;
else if (message.indexOf (ID) < 0)
	message = ID + '\n' + message;
if (exception != null)
	message += "\n" + exception.getMessage ();
IOException
	except = new IOException (message);
if (exception != null)
	{
	Throwable
		cause = exception.getCause ();
	if (cause != null)
		{
		try {except.initCause (cause);}
		catch (IllegalStateException e) {}
		}
	}
return except;
}


private IOException IO_Error
	(
	String		message
	)
{return IO_Error (message, null);}

/*==============================================================================
	Main
*/
//!	Exit status values.
public static final int
	SUCCESS			= 0,
	SYNTAX_ERROR	= 1,
	IO_EXCEPTION	= 2,
	NO_INFO			= 3,
	PVL_ERROR		= 4;

/**	Report the JPEG2000 codestream segments found in a raw codestream
	file.
<p>
	@param	arguments	The arguments String array.
	@see	#Usage()
*/
public static void main
	(
	String[]	arguments
	)
{
if ((DEBUG & DEBUG_MAIN) != 0)
	System.out.println
		("*** " + ID);
boolean
	skip_tiles = true,
	EOC_search = false,
	GUI = false;
String
	source = null;
Vector
	names = new Vector (),
	patterns = new Vector ();
for (int argument = 0;
		 argument < arguments.length;
	   ++argument)
	{
	if (arguments[argument].startsWith ("-") &&
		arguments[argument].length () > 1)
		{
		switch (arguments[argument].charAt (1))
			{
			case 'N':
			case 'n':
				int
					index = arguments[argument].indexOf ('_');
				if (index < 0)
					{
					System.out.println
						("Unknown option: " + arguments[argument]);
					Usage ();
					}
				switch (arguments[argument].charAt (++index))
					{
					case 'T':
					case 't':
						skip_tiles = true;
						break;
					case 'E':
					case 'e':
						EOC_search = false;
						break;
					default:
						System.out.println
							("Unknown option: " + arguments[argument]);
					}
				break;
			case 'T':
			case 't':
				skip_tiles = false;
				break;
			case 'E':
			case 'e':
				EOC_search = true;
				break;
			case 'F':
			case 'f':
				if (++argument == arguments.length)
					Usage ();
				names.add (arguments[argument]);
				break;
			case 'P':
			case 'p':
				if (++argument == arguments.length)
					Usage ();
				patterns.add (arguments[argument]);
				break;
			case 'G':
			case 'g':
				GUI = true;
				break;
			default:
				System.out.println
					("Unknown option: " + arguments[argument]);
			case 'H':
			case 'h':
				Usage ();
			}
		}
	else if (source == null)
		source = arguments[argument];
	else
		{
		System.err.println ("Only one source, please.");
		Usage ();
		}
	}
if (source == null)
	{
	System.err.println ("A source must be specified.");
	Usage ();
	}

//	Get the Info.
JPEG2000_Codestream_Info
	info = null;
JPEG2000_Codestream_Info.EOC_Search = EOC_search;
try {info = new JPEG2000_Codestream_Info (source, skip_tiles);}
catch (Exception exception)
	{
	System.out.println (exception.getMessage ());
	System.exit (IO_EXCEPTION);
	}

//	Report the Info.
if (info == null)
	{
	System.out.println ("No JPEG2000 Codestream Info");
	System.exit (NO_INFO);
	}

int
	validity = info.Codestream_Validity ();
boolean
	EOC_missing =
		((! info.Skip_Tiles ()) &&
		((validity & JPEG2000_Codestream_Info.EOC_FOUND) == 0)) ||
		(info.Zero_Length_Tile_Part () &&
		 JPEG2000_Codestream_Info.EOC_Search);
if (! info.Is_Complete_Codestream () ||
	EOC_missing)
	{
	System.out.println (">>> WARNING <<< Incomplete JPEG2000 codestream.\n");
	if ((validity & JPEG2000_Codestream_Info.SOC_FOUND) == 0)
		System.out.println ("    Start of Codestream marker not found.");
	if ((validity & JPEG2000_Codestream_Info.SIZ_FOUND) == 0)
		System.out.println ("    Size marker not found.");
	if ((validity & JPEG2000_Codestream_Info.COD_FOUND) == 0)
		System.out.println ("    Coding Style Default marker not found.");
	if ((validity & JPEG2000_Codestream_Info.QCD_FOUND) == 0)
		System.out.println ("    Quantization Default marker not found.");
	if (EOC_missing)
		System.out.println ("    End of Codestream marker not found.");
	System.out.println ();
	}

if (! names.isEmpty () ||
	! patterns.isEmpty ())
	{
	Parameter
		select = new Parameter (),
		parameter;
	Selector
		criteria;
	String
		name;
	int
		root_name_length = info.Name ().length () + 1;
	Iterator
		list;
	if (! names.isEmpty ())
		{
		criteria = new Selection ().Name (true).Specific (false);
		list = names.iterator ();
		while (list.hasNext ())
			{
			select.Name ((String)list.next ());
			parameter = null;
			while ((parameter = info.Find (select, criteria, parameter))
					!= null)
				{
				name = parameter.Name ();
				parameter.Name (parameter.Path_to_Name ()
					.substring (root_name_length) + name);
				try {parameter.Write ();}
				catch (Exception exception)
					{
					System.out.println
						("PVL error -\n" + exception.getMessage () + "\n");
					System.exit (PVL_ERROR);
					}
				parameter.Name (name);
				}
			}
		}
	if (! patterns.isEmpty ())
		{
		criteria = new Selection ().Name (true).Pattern_Match (true);
		list = patterns.iterator ();
		while (list.hasNext ())
			{
			select.Name ((String)list.next ());
			parameter = null;
			while ((parameter = info.Find (select, criteria, parameter))
					!= null)
				{
				name = parameter.Name ();
				parameter.Name (parameter.Path_to_Name ()
					.substring (root_name_length) + name);
				try {parameter.Write ();}
				catch (Exception exception)
					{
					System.out.println
						("PVL error -\n" + exception.getMessage () + "\n");
					System.exit (PVL_ERROR);
					}
				parameter.Name (name);
				}
			}
		}
	}
else if (GUI)
	{
	Parameter_View
		view = new Parameter_View (info);
	view.setVisible (true);
	}
else
	{
	System.out.println (info.Description ());
	System.exit (SUCCESS);
	}
}

/**	Print the command line usage syntax.
<p>
<h3>Usage</h3>
<p>
	<b>JPEG2000_Codestream_Info</b>
		[<b>-</b>[<b><u>N</u>o_</b>]<b><u>T</u>iles</b>]
		[<b>-<u>G</u>ui</b>]
		[<b>-<u>F</u>ind</b> &lt;<i>parameter</i>&gt;]
		[<b>-<u>P</u>attern</b> &lt;<i>regex</i>&gt;]
		&lt;<i>source</i>&gt;

<h3>Description</h3>
<p>
	The contents of the specified file are scanned for raw (not contained
	within a JP2 file) JPEG2000 codestream segments. The information
	found is reported in the form of PVL parameters containing all the
	data element values.

<h3>Options</h3>
<p>
<h4><b>-</b>[<b><u>N</u>o_</b>]<b><u>T</u>iles</b></h4>
	<p><blockquote>
	Tiles segment parameters are to be included (or not).
	The default is not to include tile segment parameters.
	</blockquote><p>
<h4><b>-<u>G</u>ui</b></h4>
	<p><blockquote>
	The information PVL is provided in a GUI viewer. The default is to
	provide a terminal listing.
	</blockquote><p>
<h4><b>-<u>F</u>ind</b> &lt;<i>parameter</i>&gt;</h4>
	<p><blockquote>
	The named parameter is listed. This option may be repeated to obtain
	a listing of all parameters specified.
	</blockquote><p>
<h4><b>-<u>P</u>attern</b> &lt;<i>regex</i>&gt;</h4>
	<p><blockquote>
	The parameters with names that match the regular expression are
	listed. This option may also be repeated.
	</blockquote><p>
<b>N.B.</b>This method always results in a System.exit with a status
of 1.
*/
public static void Usage ()
{
System.out.println
	(
	ID + '\n' +
	"Usage: JP2_Info [<Options>] <source>\n" +
	"  The contents of the specified file are scanned\n" +
	"  for raw (not in a JP2 file) JPEG2000 codestream segments.\n" +
	"  The information found is reported in the form of\n" +
	"  PVL parameters containing all the data element values.\n" +
	"\n" +
	"Options -\n" +
	"  -[No_]Tiles\n" +
	"    Tiles segment parameters are to be included (or not).\n" +
	"    The default is not to include tile segment parameters.\n" +
	"  -[No_]EOC_search\n" +
	"    Search a final tile-part of unknown length for the EOC marker.\n" +
	"    The default is not to search.\n" +
	"  -Gui\n" +
	"    The information PVL is provided in a GUI viewer.\n" +
	"    The default is to provide a terminal listing.\n" +
	"  -Find <parameter>\n" +
	"    The named parameter is listed. This option may be repeated\n" +
	"    to obtain a listing of all parameters specified.\n" +
	"  -Pattern <regex>\n" +
	"    The parameters with names that match the regular expression\n" +
	"    are listed. This option may also be repeated.\n" +
	"  -Help\n" +
	"    Print this usage description and exit.\n"
	);
System.exit (SYNTAX_ERROR);
}

}
