use ui;
use core:geometry;
use graphics;
use lang:bs:macro;

/**
 * The main presentation window.
 */
class Presenter extends Frame {
	// Painter.
	private PresentPainter render;

	// Key mappings.
	private Map<Key, fn()->void> keymap;

	// Mouse button pressed?
	private Bool showPointer;

	// Any digits entered?
	private Bool anyDigits;

	// Number being entered.
	private Nat number;

	// Source for the presentation. Used to reload the presentation.
	lang:Function? source;

	// Create.
	init(Presentation p) {
		init(p.title, Size(1024, 768)) {
			render(p);
		}

		initKeymap();

		cursorVisible = false;
		painter = render;

		create();
	}

	// Handle keypresses.
	Bool onKey(Bool pressed, Key key, Modifiers modifiers) {
		if (!pressed)
			return false;

		if (modifiers == Modifiers:none) {
			if (keymap.has(key)) {
				keymap.get(key).call();
				return true;
			}
		}

		false;
	}

	// Handle mouse clicks.
	Bool onClick(Bool pressed, Point pos, MouseButton button) {
		if (button != MouseButton:left)
			return false;

		showPointer = pressed;
		if (pressed) {
			render.showPointer(pos);
		} else {
			render.hidePointer();
		}
		repaint();
		true;
	}

	// Mouse movement.
	Bool onMouseMove(Point pos) {
		if (showPointer) {
			render.showPointer(pos);
			repaint();
		}
		true;
	}

	// Go to a specific slide.
	void goTo(Nat slide) {
		render.goTo(slide);
		repaint();
	}

private:
	// Initialize the keymap.
	void initKeymap() {
		putKeys([Key:esc, Key:q], &this.close());
		putKeys([Key:right, Key:down], &this.next());
		putKeys([Key:left, Key:up], &this.prev());
		keymap.put(Key:return, &this.enter());
		keymap.put(Key:f, &this.toggleFullscreen());
		keymap.put(Key:F5, &this.reload());
		keymap.put(Key:home, &this.first());
		keymap.put(Key:end, &this.last());
		keymap.put(Key:F8, &debug:threadSummary());

		var me = this;
		keymap.put(Key:num0, () => me.numKey(0));
		keymap.put(Key:num1, () => me.numKey(1));
		keymap.put(Key:num2, () => me.numKey(2));
		keymap.put(Key:num3, () => me.numKey(3));
		keymap.put(Key:num4, () => me.numKey(4));
		keymap.put(Key:num5, () => me.numKey(5));
		keymap.put(Key:num6, () => me.numKey(6));
		keymap.put(Key:num7, () => me.numKey(7));
		keymap.put(Key:num8, () => me.numKey(8));
		keymap.put(Key:num9, () => me.numKey(9));
	}

	// Add a series of keys for the same action.
	void putKeys(Key[] keys, fn()->void fn) {
		for (k in keys)
			keymap.put(k, fn);
	}

	// Toggle fullscreen.
	void toggleFullscreen() {
		fullscreen = !fullscreen;
	}

	// A number has been entered.
	void numKey(Nat num) {
		if (!anyDigits)
			number = 0;

		anyDigits = true;
		number = number * 10 + num;
	}

	// Called when "enter" has been pressed. Either advances one slide or moves to the slide previously entered.
	void enter() {
		if (anyDigits) {
			goTo(number);
			anyDigits = false;
		} else {
			next();
		}
	}

	// Advance to the next slide/animation step.
	void next() {
		render.next();
		repaint();
	}

	// Go back to a previous slide/animation step.
	void prev() {
		render.prev();
		repaint();
	}

	// Go to the first slide.
	void first() {
		render.first();
		repaint();
	}

	// Go to the last slide.
	void last() {
		render.last();
		repaint();
	}

	// Reload the presentation.
	void reload() {
		unless (source)
			return;

		doReload(source);
		if (fn = lang:pointer(source) as Fn<Presentation>) {
			render.presentation = fn.call();
			repaint();
		}
	}
}

// Helper for doing reload.
private void doReload(lang:NameLookup entity) on Compiler {
	// Find containing package.
	if (entity as lang:Package) {
		lang:bs:macro:reloadPkg(entity);
	} else if (parent = entity.parent) {
		// Try to find a parent package...
		doReload(parent);
	}
}


/**
 * Rendering of presentations.
 */
class PresentPainter extends Painter {
	// Current presentation.
	private Presentation p;

	// Currently playing animation?
	private Bool inAnimation;

	// Start of the currently playing animation.
	private Moment aniStart;

	// Length of the currently playing animation.
	private Duration aniLength;

	// Previous state, to keep slide animations consistent.
	private Presentation:Cursor? lastCursor;

	// Our cursor.
	private Presentation:Cursor cursor;

	// Show a pointer at some location?
	private Bool showPointer;

	// Show the pointer at this location (screen coordinates).
	private Point pointerPos;

	// Pointer brush.
	private Brush pointerBrush;

	// Size of the pointer.
	private Float pointerSize;

	// Create.
	init(Presentation p) {
		init() {
			p = p;
			cursor = p.begin;
			pointerBrush = SolidBrush(Color(1, 0, 0, 0.7));
			pointerSize = 10;
		}

		bgColor = black;
		animate();
	}

	// Draw things!
	Bool render(Size size, Graphics g) {
		// How is the animation going?
		Float aniPos = 1.0;
		if (inAnimation) {
			aniPos = (Moment() - aniStart) / aniLength;
			if (aniPos >= 1.0) {
				inAnimation = false;
				aniPos = 1.0;
			}
		}

		// Compute where the contents are supposed to be.
		Size content = p.size;
		Float scale = min(size.w / content.w, size.h / content.h);
		Point move = (size - content*scale) / 2;
		g.transform = scale(scale) * translate(move);
		g.push(Rect(content));

		p.draw(g, cursor, lastCursor, aniPos);

		g.pop();

		if (showPointer) {
			g.transform = Transform();
			var s = scale * pointerSize;
			g.fillOval(Rect(pointerPos - Size(s), Size(s*2)), pointerBrush);
		}

		inAnimation;
	}

	// Manage the pointer.
	void showPointer(Point at) {
		showPointer = true;
		pointerPos = at;
	}

	void hidePointer() {
		showPointer = false;
	}

	// Advance the animation.
	void next() {
		setCursor(p.next(cursor));
	}

	// Go back.
	void prev() {
		setCursor(p.prev(cursor));
	}

	// Go to specific slide.
	void goTo(Nat slide) {
		setCursor(p.at(slide));
	}

	// Go to the first slide.
	void first() {
		setCursor(p.at(0));
	}

	// Go to the last (interesting) slide.
	void last() {
		setCursor(p.prev(p.end));
	}

	// Set the presentation.
	assign presentation(Presentation pr) {
		p = pr;
	}

	// Set the cursor to a new location.
	private void setCursor(Presentation:Cursor c) {
		lastCursor = cursor;
		cursor = c;
		animate();
	}

	// Set us up for a new animation specified by the current cursor.
	private void animate() {
		var d = p.duration(cursor);

		if (d > 0 ms) {
			inAnimation = true;
			aniStart = Moment();
			aniLength = d;
		} else {
			inAnimation = false;
		}
	}
}

// Make sure the entire presentation library is compiled.
void ensureCompiled() {
	print("Compiling...");
	Moment start;
	compile(named{presentation});
	compile(named{layout});
	Moment end;
	print("Done in ${end - start}");
}

// Helper for showing a presentation.
void show(Presentation p) on Ui {
	ensureCompiled();
	Presenter w(p);
	w.waitForClose();
}

// Helper for showing a presentation, starts at the specified slide. Good when developing the presentation.
void show(Presentation p, Nat slide) on Ui {
	ensureCompiled();
	Presenter w(p);
	w.goTo(slide);
	w.waitForClose();
}

// Show a presentation in debug mode, allowing live-reloading changes to the presentation.
void showDebug(lang:Function presentation) on Ui {
	if (fn = lang:pointer(presentation) as Fn<Presentation>) {
		Presenter w(fn.call());
		w.source = presentation;
		w.waitForClose();
	} else {
		print("Error: Not a presentation function!");
	}
}

// Show a presentation in debug mode, allowing live-reloading changes to the presentation.
void showDebug(lang:Function presentation, Nat slide) on Ui {
	if (fn = lang:pointer(presentation) as Fn<Presentation>) {
		Presenter w(fn.call());
		w.source = presentation;
		w.goTo(slide);
		w.waitForClose();
	} else {
		print("Error: Not a presentation function!");
	}
}
