// @HEADER
// ************************************************************************
//
//                           Intrepid Package
//                 Copyright (2007) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Pavel Bochev  (pbboche@sandia.gov)
//                    Denis Ridzal  (dridzal@sandia.gov), or
//                    Kara Peterson (kjpeter@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file   Intrepid_CubatureDirectLineGaussDef.hpp
    \brief  Definition file for the Intrepid2::CubatureDirectLineGauss class.
    \author Created by P. Bochev, D. Ridzal, and D. Day.
*/

namespace Intrepid2 {

template <class Scalar, class ArrayPoint, class ArrayWeight>
CubatureDirectLineGauss<Scalar,ArrayPoint,ArrayWeight>::CubatureDirectLineGauss(const int degree) {
  this->degree_    = degree;
  this->dimension_ = 1;
  TEUCHOS_TEST_FOR_EXCEPTION((degree < 0) || (degree > INTREPID_CUBATURE_LINE_GAUSS_MAX_ENUM),
                     std::out_of_range,
                     ">>> ERROR (CubatureDirectLineGauss): No cubature rule implemented for the desired polynomial degree.");
} // end constructor



template <class Scalar, class ArrayPoint, class ArrayWeight>
const CubatureTemplate *  CubatureDirectLineGauss<Scalar,ArrayPoint,ArrayWeight>::exposeCubatureData() const {
  return cubature_data_;
}



template <class Scalar, class ArrayPoint, class ArrayWeight>
int CubatureDirectLineGauss<Scalar,ArrayPoint,ArrayWeight>::getMaxAccuracy() const {
  return INTREPID_CUBATURE_LINE_GAUSS_MAX_ENUM;
}



template <class Scalar, class ArrayPoint, class ArrayWeight>
const char* CubatureDirectLineGauss<Scalar,ArrayPoint,ArrayWeight>::getName() const {
  return cubature_name_;
} // end getName



template <class Scalar, class ArrayPoint, class ArrayWeight>
const char* CubatureDirectLineGauss<Scalar,ArrayPoint,ArrayWeight>::cubature_name_ = "INTREPID_CUBATURE_LINE_GAUSS";


//-------------------------------------------------------------------------------------//
//                          Definition of cubature templates                           //
//-------------------------------------------------------------------------------------//

/*
   Cubature templates for lines are defined the reference cell:

   Line -> (-1,0,0),(1,0,0)
*/

/*
   This static const member contains templates for Gauss(-Legendre) rules.
*/

template <class Scalar, class ArrayPoint, class ArrayWeight>
const CubatureTemplate CubatureDirectLineGauss<Scalar,ArrayPoint,ArrayWeight>::cubature_data_[INTREPID_CUBATURE_LINE_GAUSS_MAX_ENUM+1] =
{

  // Collection of Gauss rules on [-1,1]
  // The rule with array index i is exact for polynomials up to order i
  {
    1,
    {{0e0, 0.0, 0.0}},
    {2e0}
  },
  {
    1,
    {{0e0, 0.0, 0.0}},
    {2e0}
  },
  {
    2,
    {{5.773502691896257645091487805019574556476e-1, 0.0, 0.0},
     {-5.773502691896257645091487805019574556476e-1, 0.0, 0.0}},
    {1e0,
     1e0}
  },
  {
    2,
    {{5.773502691896257645091487805019574556476e-1, 0.0, 0.0},
     {-5.773502691896257645091487805019574556476e-1, 0.0, 0.0}},
    {1e0,
     1e0}
  },
  {
    3,
    {{7.745966692414833770358530799564799221666e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-7.745966692414833770358530799564799221666e-1, 0.0, 0.0}},
    {5.555555555555555555555555555555555555556e-1,
     8.888888888888888888888888888888888888889e-1,
     5.555555555555555555555555555555555555556e-1}
  },
  {
    3,
    {{7.745966692414833770358530799564799221666e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-7.745966692414833770358530799564799221666e-1, 0.0, 0.0}},
    {5.555555555555555555555555555555555555556e-1,
     8.888888888888888888888888888888888888889e-1,
     5.555555555555555555555555555555555555556e-1}
  },
  {
    4,
    {{8.611363115940525752239464888928095050957e-1, 0.0, 0.0},
     {3.399810435848562648026657591032446872006e-1, 0.0, 0.0},
     {-3.399810435848562648026657591032446872006e-1, 0.0, 0.0},
     {-8.611363115940525752239464888928095050957e-1, 0.0, 0.0}},
    {3.478548451374538573730639492219994072353e-1,
     6.521451548625461426269360507780005927647e-1,
     6.521451548625461426269360507780005927647e-1,
     3.478548451374538573730639492219994072353e-1}
  },
  {
    4,
    {{8.611363115940525752239464888928095050957e-1, 0.0, 0.0},
     {3.399810435848562648026657591032446872006e-1, 0.0, 0.0},
     {-3.399810435848562648026657591032446872006e-1, 0.0, 0.0},
     {-8.611363115940525752239464888928095050957e-1, 0.0, 0.0}},
    {3.478548451374538573730639492219994072353e-1,
     6.521451548625461426269360507780005927647e-1,
     6.521451548625461426269360507780005927647e-1,
     3.478548451374538573730639492219994072353e-1}
  },
  {
    5,
    {{9.061798459386639927976268782993929651257e-1, 0.0, 0.0},
     {5.384693101056830910363144207002088049673e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-5.384693101056830910363144207002088049673e-1, 0.0, 0.0},
     {-9.061798459386639927976268782993929651257e-1, 0.0, 0.0}},
    {2.369268850561890875142640407199173626433e-1,
     4.786286704993664680412915148356381929123e-1,
     5.688888888888888888888888888888888888889e-1,
     4.786286704993664680412915148356381929123e-1,
     2.369268850561890875142640407199173626433e-1}
  },
  {
    5,
    {{9.061798459386639927976268782993929651257e-1, 0.0, 0.0},
     {5.384693101056830910363144207002088049673e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-5.384693101056830910363144207002088049673e-1, 0.0, 0.0},
     {-9.061798459386639927976268782993929651257e-1, 0.0, 0.0}},
    {2.369268850561890875142640407199173626433e-1,
     4.786286704993664680412915148356381929123e-1,
     5.688888888888888888888888888888888888889e-1,
     4.786286704993664680412915148356381929123e-1,
     2.369268850561890875142640407199173626433e-1}
  },
  {
    6,
    {{9.324695142031520278123015544939946091348e-1, 0.0, 0.0},
     {6.612093864662645136613995950199053470064e-1, 0.0, 0.0},
     {2.386191860831969086305017216807119354186e-1, 0.0, 0.0},
     {-2.386191860831969086305017216807119354186e-1, 0.0, 0.0},
     {-6.612093864662645136613995950199053470064e-1, 0.0, 0.0},
     {-9.324695142031520278123015544939946091348e-1, 0.0, 0.0}},
    {1.713244923791703450402961421727328935268e-1,
     3.607615730481386075698335138377161116615e-1,
     4.679139345726910473898703439895509948117e-1,
     4.679139345726910473898703439895509948117e-1,
     3.607615730481386075698335138377161116615e-1,
     1.713244923791703450402961421727328935268e-1}
  },
  {
    6,
    {{9.324695142031520278123015544939946091348e-1, 0.0, 0.0},
     {6.612093864662645136613995950199053470064e-1, 0.0, 0.0},
     {2.386191860831969086305017216807119354186e-1, 0.0, 0.0},
     {-2.386191860831969086305017216807119354186e-1, 0.0, 0.0},
     {-6.612093864662645136613995950199053470064e-1, 0.0, 0.0},
     {-9.324695142031520278123015544939946091348e-1, 0.0, 0.0}},
    {1.713244923791703450402961421727328935268e-1,
     3.607615730481386075698335138377161116615e-1,
     4.679139345726910473898703439895509948117e-1,
     4.679139345726910473898703439895509948117e-1,
     3.607615730481386075698335138377161116615e-1,
     1.713244923791703450402961421727328935268e-1}
  },
  {
    7,
    {{9.491079123427585245261896840478512624008e-1, 0.0, 0.0},
     {7.415311855993944398638647732807884070741e-1, 0.0, 0.0},
     {4.058451513773971669066064120769614633474e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-4.058451513773971669066064120769614633474e-1, 0.0, 0.0},
     {-7.415311855993944398638647732807884070741e-1, 0.0, 0.0},
     {-9.491079123427585245261896840478512624008e-1, 0.0, 0.0}},
    {1.294849661688696932706114326790820183286e-1,
     2.797053914892766679014677714237795824869e-1,
     3.818300505051189449503697754889751338784e-1,
     4.179591836734693877551020408163265306122e-1,
     3.818300505051189449503697754889751338784e-1,
     2.797053914892766679014677714237795824869e-1,
     1.294849661688696932706114326790820183286e-1}
  },
  {
    7,
    {{9.491079123427585245261896840478512624008e-1, 0.0, 0.0},
     {7.415311855993944398638647732807884070741e-1, 0.0, 0.0},
     {4.058451513773971669066064120769614633474e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-4.058451513773971669066064120769614633474e-1, 0.0, 0.0},
     {-7.415311855993944398638647732807884070741e-1, 0.0, 0.0},
     {-9.491079123427585245261896840478512624008e-1, 0.0, 0.0}},
    {1.294849661688696932706114326790820183286e-1,
     2.797053914892766679014677714237795824869e-1,
     3.818300505051189449503697754889751338784e-1,
     4.179591836734693877551020408163265306122e-1,
     3.818300505051189449503697754889751338784e-1,
     2.797053914892766679014677714237795824869e-1,
     1.294849661688696932706114326790820183286e-1}
  },
  {
    8,
    {{9.602898564975362316835608685694729904282e-1, 0.0, 0.0},
     {7.966664774136267395915539364758304368372e-1, 0.0, 0.0},
     {5.25532409916328985817739049189246349042e-1, 0.0, 0.0},
     {1.834346424956498049394761423601839806668e-1, 0.0, 0.0},
     {-1.834346424956498049394761423601839806668e-1, 0.0, 0.0},
     {-5.25532409916328985817739049189246349042e-1, 0.0, 0.0},
     {-7.966664774136267395915539364758304368372e-1, 0.0, 0.0},
     {-9.602898564975362316835608685694729904282e-1, 0.0, 0.0}},
    {1.012285362903762591525313543099621901154e-1,
     2.223810344533744705443559944262408844301e-1,
     3.137066458778872873379622019866013132603e-1,
     3.626837833783619829651504492771956121941e-1,
     3.626837833783619829651504492771956121941e-1,
     3.137066458778872873379622019866013132603e-1,
     2.223810344533744705443559944262408844301e-1,
     1.012285362903762591525313543099621901154e-1}
  },
  {
    8,
    {{9.602898564975362316835608685694729904282e-1, 0.0, 0.0},
     {7.966664774136267395915539364758304368372e-1, 0.0, 0.0},
     {5.25532409916328985817739049189246349042e-1, 0.0, 0.0},
     {1.834346424956498049394761423601839806668e-1, 0.0, 0.0},
     {-1.834346424956498049394761423601839806668e-1, 0.0, 0.0},
     {-5.25532409916328985817739049189246349042e-1, 0.0, 0.0},
     {-7.966664774136267395915539364758304368372e-1, 0.0, 0.0},
     {-9.602898564975362316835608685694729904282e-1, 0.0, 0.0}},
    {1.012285362903762591525313543099621901154e-1,
     2.223810344533744705443559944262408844301e-1,
     3.137066458778872873379622019866013132603e-1,
     3.626837833783619829651504492771956121941e-1,
     3.626837833783619829651504492771956121941e-1,
     3.137066458778872873379622019866013132603e-1,
     2.223810344533744705443559944262408844301e-1,
     1.012285362903762591525313543099621901154e-1}
  },
  {
    9,
    {{9.681602395076260898355762029036728700494e-1, 0.0, 0.0},
     {8.360311073266357942994297880697348765441e-1, 0.0, 0.0},
     {6.133714327005903973087020393414741847857e-1, 0.0, 0.0},
     {3.24253423403808929038538014643336608572e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-3.24253423403808929038538014643336608572e-1, 0.0, 0.0},
     {-6.133714327005903973087020393414741847857e-1, 0.0, 0.0},
     {-8.360311073266357942994297880697348765441e-1, 0.0, 0.0},
     {-9.681602395076260898355762029036728700494e-1, 0.0, 0.0}},
    {8.127438836157441197189215811052365067566e-2,
     1.806481606948574040584720312429128095143e-1,
     2.606106964029354623187428694186328497718e-1,
     3.123470770400028400686304065844436655988e-1,
     3.302393550012597631645250692869740488788e-1,
     3.123470770400028400686304065844436655988e-1,
     2.606106964029354623187428694186328497718e-1,
     1.806481606948574040584720312429128095143e-1,
     8.127438836157441197189215811052365067566e-2}
  },
  {
    9,
    {{9.681602395076260898355762029036728700494e-1, 0.0, 0.0},
     {8.360311073266357942994297880697348765441e-1, 0.0, 0.0},
     {6.133714327005903973087020393414741847857e-1, 0.0, 0.0},
     {3.24253423403808929038538014643336608572e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-3.24253423403808929038538014643336608572e-1, 0.0, 0.0},
     {-6.133714327005903973087020393414741847857e-1, 0.0, 0.0},
     {-8.360311073266357942994297880697348765441e-1, 0.0, 0.0},
     {-9.681602395076260898355762029036728700494e-1, 0.0, 0.0}},
    {8.127438836157441197189215811052365067566e-2,
     1.806481606948574040584720312429128095143e-1,
     2.606106964029354623187428694186328497718e-1,
     3.123470770400028400686304065844436655988e-1,
     3.302393550012597631645250692869740488788e-1,
     3.123470770400028400686304065844436655988e-1,
     2.606106964029354623187428694186328497718e-1,
     1.806481606948574040584720312429128095143e-1,
     8.127438836157441197189215811052365067566e-2}
  },
  {
    10,
    {{9.739065285171717200779640120844520534283e-1, 0.0, 0.0},
     {8.650633666889845107320966884234930485275e-1, 0.0, 0.0},
     {6.794095682990244062343273651148735757693e-1, 0.0, 0.0},
     {4.333953941292471907992659431657841622001e-1, 0.0, 0.0},
     {1.488743389816312108848260011297199846176e-1, 0.0, 0.0},
     {-1.488743389816312108848260011297199846176e-1, 0.0, 0.0},
     {-4.333953941292471907992659431657841622001e-1, 0.0, 0.0},
     {-6.794095682990244062343273651148735757693e-1, 0.0, 0.0},
     {-8.650633666889845107320966884234930485275e-1, 0.0, 0.0},
     {-9.739065285171717200779640120844520534283e-1, 0.0, 0.0}},
    {6.667134430868813759356880989333179285786e-2,
     1.494513491505805931457763396576973324026e-1,
     2.190863625159820439955349342281631924588e-1,
     2.692667193099963550912269215694693528598e-1,
     2.95524224714752870173892994651338329421e-1,
     2.95524224714752870173892994651338329421e-1,
     2.692667193099963550912269215694693528598e-1,
     2.190863625159820439955349342281631924588e-1,
     1.494513491505805931457763396576973324026e-1,
     6.667134430868813759356880989333179285786e-2}
  },
  {
    10,
    {{9.739065285171717200779640120844520534283e-1, 0.0, 0.0},
     {8.650633666889845107320966884234930485275e-1, 0.0, 0.0},
     {6.794095682990244062343273651148735757693e-1, 0.0, 0.0},
     {4.333953941292471907992659431657841622001e-1, 0.0, 0.0},
     {1.488743389816312108848260011297199846176e-1, 0.0, 0.0},
     {-1.488743389816312108848260011297199846176e-1, 0.0, 0.0},
     {-4.333953941292471907992659431657841622001e-1, 0.0, 0.0},
     {-6.794095682990244062343273651148735757693e-1, 0.0, 0.0},
     {-8.650633666889845107320966884234930485275e-1, 0.0, 0.0},
     {-9.739065285171717200779640120844520534283e-1, 0.0, 0.0}},
    {6.667134430868813759356880989333179285786e-2,
     1.494513491505805931457763396576973324026e-1,
     2.190863625159820439955349342281631924588e-1,
     2.692667193099963550912269215694693528598e-1,
     2.95524224714752870173892994651338329421e-1,
     2.95524224714752870173892994651338329421e-1,
     2.692667193099963550912269215694693528598e-1,
     2.190863625159820439955349342281631924588e-1,
     1.494513491505805931457763396576973324026e-1,
     6.667134430868813759356880989333179285786e-2}
  },
  {
    11,
    {{9.782286581460569928039380011228573907714e-1, 0.0, 0.0},
     {8.870625997680952990751577693039272666317e-1, 0.0, 0.0},
     {7.301520055740493240934162520311534580496e-1, 0.0, 0.0},
     {5.190961292068118159257256694586095544802e-1, 0.0, 0.0},
     {2.695431559523449723315319854008615246796e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-2.695431559523449723315319854008615246796e-1, 0.0, 0.0},
     {-5.190961292068118159257256694586095544802e-1, 0.0, 0.0},
     {-7.301520055740493240934162520311534580496e-1, 0.0, 0.0},
     {-8.870625997680952990751577693039272666317e-1, 0.0, 0.0},
     {-9.782286581460569928039380011228573907714e-1, 0.0, 0.0}},
    {5.566856711617366648275372044254857872852e-2,
     1.255803694649046246346942992239401001976e-1,
     1.862902109277342514260976414316558916913e-1,
     2.331937645919904799185237048431751394318e-1,
     2.628045445102466621806888698905091953728e-1,
     2.72925086777900630714483528336342189156e-1,
     2.628045445102466621806888698905091953728e-1,
     2.331937645919904799185237048431751394318e-1,
     1.862902109277342514260976414316558916913e-1,
     1.255803694649046246346942992239401001976e-1,
     5.566856711617366648275372044254857872852e-2}
  },
  {
    11,
    {{9.782286581460569928039380011228573907714e-1, 0.0, 0.0},
     {8.870625997680952990751577693039272666317e-1, 0.0, 0.0},
     {7.301520055740493240934162520311534580496e-1, 0.0, 0.0},
     {5.190961292068118159257256694586095544802e-1, 0.0, 0.0},
     {2.695431559523449723315319854008615246796e-1, 0.0, 0.0},
     {0e0, 0.0, 0.0},
     {-2.695431559523449723315319854008615246796e-1, 0.0, 0.0},
     {-5.190961292068118159257256694586095544802e-1, 0.0, 0.0},
     {-7.301520055740493240934162520311534580496e-1, 0.0, 0.0},
     {-8.870625997680952990751577693039272666317e-1, 0.0, 0.0},
     {-9.782286581460569928039380011228573907714e-1, 0.0, 0.0}},
    {5.566856711617366648275372044254857872852e-2,
     1.255803694649046246346942992239401001976e-1,
     1.862902109277342514260976414316558916913e-1,
     2.331937645919904799185237048431751394318e-1,
     2.628045445102466621806888698905091953728e-1,
     2.72925086777900630714483528336342189156e-1,
     2.628045445102466621806888698905091953728e-1,
     2.331937645919904799185237048431751394318e-1,
     1.862902109277342514260976414316558916913e-1,
     1.255803694649046246346942992239401001976e-1,
     5.566856711617366648275372044254857872852e-2}
  },
  {
    12,
    { // points
     {.9815606342467192506905490901492808229602    ,0.0, 0.0},
     {.9041172563704748566784658661190961925376    ,0.0, 0.0},
     {.7699026741943046870368938332128180759849    ,0.0, 0.0},
     {.5873179542866174472967024189405342803691    ,0.0, 0.0},
     {.3678314989981801937526915366437175612564    ,0.0, 0.0},
     {.1252334085114689154724413694638531299834    ,0.0, 0.0},
     {-.1252334085114689154724413694638531299834   ,0.0, 0.0},
     {-.3678314989981801937526915366437175612564   ,0.0, 0.0},
     {-.5873179542866174472967024189405342803691   ,0.0, 0.0},
     {-.7699026741943046870368938332128180759849   ,0.0, 0.0},
     {-.9041172563704748566784658661190961925376   ,0.0, 0.0},
     {-.9815606342467192506905490901492808229602   ,0.0, 0.0}
    },
    { // weights
     .04717533638651182719461596148501706031703,
     .1069393259953184309602547181939962242146,
     .160078328543346226334652529543359071872,
     .2031674267230659217490644558097983765065,
     .2334925365383548087608498989248780562594,
     .2491470458134027850005624360429512108305,
      .2491470458134027850005624360429512108305,
      .2334925365383548087608498989248780562594,
      .2031674267230659217490644558097983765065,
      .160078328543346226334652529543359071872,
      .1069393259953184309602547181939962242146,
      .04717533638651182719461596148501706031703
    }
  },
  {
    12,
    { // points
     {.9815606342467192506905490901492808229602    ,0.0, 0.0},
     {.9041172563704748566784658661190961925376    ,0.0, 0.0},
     {.7699026741943046870368938332128180759849    ,0.0, 0.0},
     {.5873179542866174472967024189405342803691    ,0.0, 0.0},
     {.3678314989981801937526915366437175612564    ,0.0, 0.0},
     {.1252334085114689154724413694638531299834    ,0.0, 0.0},
     {-.1252334085114689154724413694638531299834   ,0.0, 0.0},
     {-.3678314989981801937526915366437175612564   ,0.0, 0.0},
     {-.5873179542866174472967024189405342803691   ,0.0, 0.0},
     {-.7699026741943046870368938332128180759849   ,0.0, 0.0},
     {-.9041172563704748566784658661190961925376   ,0.0, 0.0},
     {-.9815606342467192506905490901492808229602   ,0.0, 0.0}
    },
    { // weights
     .04717533638651182719461596148501706031703,
     .1069393259953184309602547181939962242146,
     .160078328543346226334652529543359071872,
     .2031674267230659217490644558097983765065,
     .2334925365383548087608498989248780562594,
     .2491470458134027850005624360429512108305,
      .2491470458134027850005624360429512108305,
      .2334925365383548087608498989248780562594,
      .2031674267230659217490644558097983765065,
      .160078328543346226334652529543359071872,
      .1069393259953184309602547181939962242146,
      .04717533638651182719461596148501706031703
    }
  },
  {
    13,
    { // points
     {.984183054718588149472829448807109611065     ,0.0, 0.0},
     {.9175983992229779652065478365007195123905    ,0.0, 0.0},
     {.8015780907333099127942064895828598903056    ,0.0, 0.0},
     {.6423493394403402206439846069955156500717    ,0.0, 0.0},
     {.4484927510364468528779128521276398678019    ,0.0, 0.0},
     {.2304583159551347940655281210979888352115    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.2304583159551347940655281210979888352115   ,0.0, 0.0},
     {-.4484927510364468528779128521276398678019   ,0.0, 0.0},
     {-.6423493394403402206439846069955156500717   ,0.0, 0.0},
     {-.8015780907333099127942064895828598903056   ,0.0, 0.0},
     {-.9175983992229779652065478365007195123905   ,0.0, 0.0},
     {-.984183054718588149472829448807109611065    ,0.0, 0.0}
    },
    { // weights
     .04048400476531587952002159220098606004199,
     .09212149983772844791442177595379712092368,
     .1388735102197872384636017768688714676219,
     .1781459807619457382800466919960979955128,
     .2078160475368885023125232193060527633866,
     .2262831802628972384120901860397766184348,
     .2325515532308739101945895152688359481566,
     .2262831802628972384120901860397766184348,
     .2078160475368885023125232193060527633866,
     .1781459807619457382800466919960979955128,
     .1388735102197872384636017768688714676219,
     .09212149983772844791442177595379712092368,
     .04048400476531587952002159220098606004199
    }
  },
  {
    13,
    { // points
     {.984183054718588149472829448807109611065     ,0.0, 0.0},
     {.9175983992229779652065478365007195123905    ,0.0, 0.0},
     {.8015780907333099127942064895828598903056    ,0.0, 0.0},
     {.6423493394403402206439846069955156500717    ,0.0, 0.0},
     {.4484927510364468528779128521276398678019    ,0.0, 0.0},
     {.2304583159551347940655281210979888352115    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.2304583159551347940655281210979888352115   ,0.0, 0.0},
     {-.4484927510364468528779128521276398678019   ,0.0, 0.0},
     {-.6423493394403402206439846069955156500717   ,0.0, 0.0},
     {-.8015780907333099127942064895828598903056   ,0.0, 0.0},
     {-.9175983992229779652065478365007195123905   ,0.0, 0.0},
     {-.984183054718588149472829448807109611065    ,0.0, 0.0}
    },
    { // weights
     .04048400476531587952002159220098606004199,
     .09212149983772844791442177595379712092368,
     .1388735102197872384636017768688714676219,
     .1781459807619457382800466919960979955128,
     .2078160475368885023125232193060527633866,
     .2262831802628972384120901860397766184348,
     .2325515532308739101945895152688359481566,
     .2262831802628972384120901860397766184348,
     .2078160475368885023125232193060527633866,
     .1781459807619457382800466919960979955128,
     .1388735102197872384636017768688714676219,
     .09212149983772844791442177595379712092368,
     .04048400476531587952002159220098606004199
    }
  },
  {
    14,
    { // points
     {.9862838086968123388415972667040528016761    ,0.0, 0.0},
     {.928434883663573517336391139377874264477     ,0.0, 0.0},
     {.8272013150697649931897947426503949610397    ,0.0, 0.0},
     {.6872929048116854701480198030193341375384    ,0.0, 0.0},
     {.5152486363581540919652907185511886623089    ,0.0, 0.0},
     {.3191123689278897604356718241684754668343    ,0.0, 0.0},
     {.108054948707343662066244650219834747612     ,0.0, 0.0},
     {-.108054948707343662066244650219834747612    ,0.0, 0.0},
     {-.3191123689278897604356718241684754668343   ,0.0, 0.0},
     {-.5152486363581540919652907185511886623089   ,0.0, 0.0},
     {-.6872929048116854701480198030193341375384   ,0.0, 0.0},
     {-.8272013150697649931897947426503949610397   ,0.0, 0.0},
     {-.928434883663573517336391139377874264477    ,0.0, 0.0},
     {-.9862838086968123388415972667040528016761   ,0.0, 0.0}
    },
    { // weights
     .03511946033175186303183287613819178061971,
     .0801580871597602098056332770628543095837,
     .1215185706879031846894148090724766259567,
     .1572031671581935345696019386238421566057,
     .1855383974779378137417165901251570362489,
     .2051984637212956039659240656612180557103,
     .215263853463157790195876443316260035275,
     .215263853463157790195876443316260035275,
     .2051984637212956039659240656612180557103,
     .1855383974779378137417165901251570362489,
     .1572031671581935345696019386238421566057,
     .1215185706879031846894148090724766259567,
     .0801580871597602098056332770628543095837,
     .03511946033175186303183287613819178061971
    }
  },
  {
    14,
    { // points
     {.9862838086968123388415972667040528016761    ,0.0, 0.0},
     {.928434883663573517336391139377874264477     ,0.0, 0.0},
     {.8272013150697649931897947426503949610397    ,0.0, 0.0},
     {.6872929048116854701480198030193341375384    ,0.0, 0.0},
     {.5152486363581540919652907185511886623089    ,0.0, 0.0},
     {.3191123689278897604356718241684754668343    ,0.0, 0.0},
     {.108054948707343662066244650219834747612     ,0.0, 0.0},
     {-.108054948707343662066244650219834747612    ,0.0, 0.0},
     {-.3191123689278897604356718241684754668343   ,0.0, 0.0},
     {-.5152486363581540919652907185511886623089   ,0.0, 0.0},
     {-.6872929048116854701480198030193341375384   ,0.0, 0.0},
     {-.8272013150697649931897947426503949610397   ,0.0, 0.0},
     {-.928434883663573517336391139377874264477    ,0.0, 0.0},
     {-.9862838086968123388415972667040528016761   ,0.0, 0.0}
    },
    { // weights
     .03511946033175186303183287613819178061971,
     .0801580871597602098056332770628543095837,
     .1215185706879031846894148090724766259567,
     .1572031671581935345696019386238421566057,
     .1855383974779378137417165901251570362489,
     .2051984637212956039659240656612180557103,
     .215263853463157790195876443316260035275,
     .215263853463157790195876443316260035275,
     .2051984637212956039659240656612180557103,
     .1855383974779378137417165901251570362489,
     .1572031671581935345696019386238421566057,
     .1215185706879031846894148090724766259567,
     .0801580871597602098056332770628543095837,
     .03511946033175186303183287613819178061971
    }
  },
  {
    15,
    { // points
     {.987992518020485428489565718586612581147     ,0.0, 0.0},
     {.937273392400705904307758947710209471244     ,0.0, 0.0},
     {.8482065834104272162006483207742168513663    ,0.0, 0.0},
     {.7244177313601700474161860546139380096309    ,0.0, 0.0},
     {.5709721726085388475372267372539106412384    ,0.0, 0.0},
     {.3941513470775633698972073709810454683628    ,0.0, 0.0},
     {.2011940939974345223006283033945962078128    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.2011940939974345223006283033945962078128   ,0.0, 0.0},
     {-.3941513470775633698972073709810454683628   ,0.0, 0.0},
     {-.5709721726085388475372267372539106412384   ,0.0, 0.0},
     {-.7244177313601700474161860546139380096309   ,0.0, 0.0},
     {-.8482065834104272162006483207742168513663   ,0.0, 0.0},
     {-.937273392400705904307758947710209471244    ,0.0, 0.0},
     {-.987992518020485428489565718586612581147    ,0.0, 0.0}
    },
    { // weights
     .03075324199611726835462839357720441772175,
     .07036604748810812470926741645066733846671,
     .1071592204671719350118695466858693034155,
     .1395706779261543144478047945110283225209,
     .1662692058169939335532008604812088111309,
     .1861610000155622110268005618664228245062,
     .1984314853271115764561183264438393248187,
     .2025782419255612728806201999675193148387,
     .1984314853271115764561183264438393248187,
     .1861610000155622110268005618664228245062,
     .1662692058169939335532008604812088111309,
     .1395706779261543144478047945110283225209,
     .1071592204671719350118695466858693034155,
     .07036604748810812470926741645066733846671,
     .03075324199611726835462839357720441772175
    }
  },
  {
    15,
    { // points
     {.987992518020485428489565718586612581147     ,0.0, 0.0},
     {.937273392400705904307758947710209471244     ,0.0, 0.0},
     {.8482065834104272162006483207742168513663    ,0.0, 0.0},
     {.7244177313601700474161860546139380096309    ,0.0, 0.0},
     {.5709721726085388475372267372539106412384    ,0.0, 0.0},
     {.3941513470775633698972073709810454683628    ,0.0, 0.0},
     {.2011940939974345223006283033945962078128    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.2011940939974345223006283033945962078128   ,0.0, 0.0},
     {-.3941513470775633698972073709810454683628   ,0.0, 0.0},
     {-.5709721726085388475372267372539106412384   ,0.0, 0.0},
     {-.7244177313601700474161860546139380096309   ,0.0, 0.0},
     {-.8482065834104272162006483207742168513663   ,0.0, 0.0},
     {-.937273392400705904307758947710209471244    ,0.0, 0.0},
     {-.987992518020485428489565718586612581147    ,0.0, 0.0}
    },
    { // weights
     .03075324199611726835462839357720441772175,
     .07036604748810812470926741645066733846671,
     .1071592204671719350118695466858693034155,
     .1395706779261543144478047945110283225209,
     .1662692058169939335532008604812088111309,
     .1861610000155622110268005618664228245062,
     .1984314853271115764561183264438393248187,
     .2025782419255612728806201999675193148387,
     .1984314853271115764561183264438393248187,
     .1861610000155622110268005618664228245062,
     .1662692058169939335532008604812088111309,
     .1395706779261543144478047945110283225209,
     .1071592204671719350118695466858693034155,
     .07036604748810812470926741645066733846671,
     .03075324199611726835462839357720441772175
    }
  },
  {
    16,
    { // points
     {.9894009349916499325961541734503326274263    ,0.0, 0.0},
     {.9445750230732325760779884155346083450911    ,0.0, 0.0},
     {.8656312023878317438804678977123931323873    ,0.0, 0.0},
     {.7554044083550030338951011948474422683538    ,0.0, 0.0},
     {.6178762444026437484466717640487910189919    ,0.0, 0.0},
     {.45801677765722738634241944298357757354      ,0.0, 0.0},
     {.2816035507792589132304605014604961064861    ,0.0, 0.0},
     {.09501250983763744018531933542495806313035   ,0.0, 0.0},
     {-.09501250983763744018531933542495806313035  ,0.0, 0.0},
     {-.2816035507792589132304605014604961064861   ,0.0, 0.0},
     {-.45801677765722738634241944298357757354     ,0.0, 0.0},
     {-.6178762444026437484466717640487910189919   ,0.0, 0.0},
     {-.7554044083550030338951011948474422683538   ,0.0, 0.0},
     {-.8656312023878317438804678977123931323873   ,0.0, 0.0},
     {-.9445750230732325760779884155346083450911   ,0.0, 0.0},
     {-.9894009349916499325961541734503326274263   ,0.0, 0.0}
    },
    { // weights
     .02715245941175409485178057245601810351227,
     .06225352393864789286284383699437769427499,
     .09515851168249278480992510760224622635526,
     .1246289712555338720524762821920164201449,
     .1495959888165767320815017305474785489705,
     .1691565193950025381893120790303599622116,
     .1826034150449235888667636679692199393836,
     .1894506104550684962853967232082831051469,
     .1894506104550684962853967232082831051469,
     .1826034150449235888667636679692199393836,
     .1691565193950025381893120790303599622116,
     .1495959888165767320815017305474785489705,
     .1246289712555338720524762821920164201449,
     .09515851168249278480992510760224622635526,
     .06225352393864789286284383699437769427499,
     .02715245941175409485178057245601810351227
    }
  },
  {
    16,
    { // points
     {.9894009349916499325961541734503326274263    ,0.0, 0.0},
     {.9445750230732325760779884155346083450911    ,0.0, 0.0},
     {.8656312023878317438804678977123931323873    ,0.0, 0.0},
     {.7554044083550030338951011948474422683538    ,0.0, 0.0},
     {.6178762444026437484466717640487910189919    ,0.0, 0.0},
     {.45801677765722738634241944298357757354      ,0.0, 0.0},
     {.2816035507792589132304605014604961064861    ,0.0, 0.0},
     {.09501250983763744018531933542495806313035   ,0.0, 0.0},
     {-.09501250983763744018531933542495806313035  ,0.0, 0.0},
     {-.2816035507792589132304605014604961064861   ,0.0, 0.0},
     {-.45801677765722738634241944298357757354     ,0.0, 0.0},
     {-.6178762444026437484466717640487910189919   ,0.0, 0.0},
     {-.7554044083550030338951011948474422683538   ,0.0, 0.0},
     {-.8656312023878317438804678977123931323873   ,0.0, 0.0},
     {-.9445750230732325760779884155346083450911   ,0.0, 0.0},
     {-.9894009349916499325961541734503326274263   ,0.0, 0.0}
    },
    { // weights
     .02715245941175409485178057245601810351227,
     .06225352393864789286284383699437769427499,
     .09515851168249278480992510760224622635526,
     .1246289712555338720524762821920164201449,
     .1495959888165767320815017305474785489705,
     .1691565193950025381893120790303599622116,
     .1826034150449235888667636679692199393836,
     .1894506104550684962853967232082831051469,
     .1894506104550684962853967232082831051469,
     .1826034150449235888667636679692199393836,
     .1691565193950025381893120790303599622116,
     .1495959888165767320815017305474785489705,
     .1246289712555338720524762821920164201449,
     .09515851168249278480992510760224622635526,
     .06225352393864789286284383699437769427499,
     .02715245941175409485178057245601810351227
    }
  },
  {
    17,
    { // points
     {.9905754753144173356754340199406652765078    ,0.0, 0.0},
     {.9506755217687677612227169578958030214434    ,0.0, 0.0},
     {.8802391537269859021229556944881556926234    ,0.0, 0.0},
     {.781514003896801406925230055520476050224     ,0.0, 0.0},
     {.6576711592166907658503022166430023351478    ,0.0, 0.0},
     {.5126905370864769678862465686295518745829    ,0.0, 0.0},
     {.3512317634538763152971855170953460050405    ,0.0, 0.0},
     {.1784841814958478558506774936540655574754    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.1784841814958478558506774936540655574754   ,0.0, 0.0},
     {-.3512317634538763152971855170953460050405   ,0.0, 0.0},
     {-.5126905370864769678862465686295518745829   ,0.0, 0.0},
     {-.6576711592166907658503022166430023351478   ,0.0, 0.0},
     {-.781514003896801406925230055520476050224    ,0.0, 0.0},
     {-.8802391537269859021229556944881556926234   ,0.0, 0.0},
     {-.9506755217687677612227169578958030214434   ,0.0, 0.0},
     {-.9905754753144173356754340199406652765078   ,0.0, 0.0}
    },
    { // weights
     .0241483028685479319601100262875653246917,
     .05545952937398720112944016535824466051285,
     .08503614831717918088353537019106207385049,
     .1118838471934039710947883856263559267358,
     .1351363684685254732863199817023501973721,
     .1540457610768102880814315948019586119405,
     .1680041021564500445099706637883231550212,
     .1765627053669926463252709901131972391509,
      .1794464703562065254582656442618856214488,
      .1765627053669926463252709901131972391509,
      .1680041021564500445099706637883231550212,
      .1540457610768102880814315948019586119405,
      .1351363684685254732863199817023501973721,
      .1118838471934039710947883856263559267358,
      .08503614831717918088353537019106207385049,
      .05545952937398720112944016535824466051285,
      .0241483028685479319601100262875653246917
    }
  },
  {
    17,
    { // points
     {.9905754753144173356754340199406652765078    ,0.0, 0.0},
     {.9506755217687677612227169578958030214434    ,0.0, 0.0},
     {.8802391537269859021229556944881556926234    ,0.0, 0.0},
     {.781514003896801406925230055520476050224     ,0.0, 0.0},
     {.6576711592166907658503022166430023351478    ,0.0, 0.0},
     {.5126905370864769678862465686295518745829    ,0.0, 0.0},
     {.3512317634538763152971855170953460050405    ,0.0, 0.0},
     {.1784841814958478558506774936540655574754    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.1784841814958478558506774936540655574754   ,0.0, 0.0},
     {-.3512317634538763152971855170953460050405   ,0.0, 0.0},
     {-.5126905370864769678862465686295518745829   ,0.0, 0.0},
     {-.6576711592166907658503022166430023351478   ,0.0, 0.0},
     {-.781514003896801406925230055520476050224    ,0.0, 0.0},
     {-.8802391537269859021229556944881556926234   ,0.0, 0.0},
     {-.9506755217687677612227169578958030214434   ,0.0, 0.0},
     {-.9905754753144173356754340199406652765078   ,0.0, 0.0}
    },
    { // weights
     .0241483028685479319601100262875653246917,
     .05545952937398720112944016535824466051285,
     .08503614831717918088353537019106207385049,
     .1118838471934039710947883856263559267358,
     .1351363684685254732863199817023501973721,
     .1540457610768102880814315948019586119405,
     .1680041021564500445099706637883231550212,
     .1765627053669926463252709901131972391509,
      .1794464703562065254582656442618856214488,
      .1765627053669926463252709901131972391509,
      .1680041021564500445099706637883231550212,
      .1540457610768102880814315948019586119405,
      .1351363684685254732863199817023501973721,
      .1118838471934039710947883856263559267358,
      .08503614831717918088353537019106207385049,
      .05545952937398720112944016535824466051285,
      .0241483028685479319601100262875653246917
    }
  },
  {
    18,
    { // points
     {.9915651684209309467300160047061507702526    ,0.0, 0.0},
     {.9558239495713977551811958929297763099728    ,0.0, 0.0},
     {.8926024664975557392060605911271455154079    ,0.0, 0.0},
     {.8037049589725231156824174550145907971033    ,0.0, 0.0},
     {.6916870430603532078748910812888483894523    ,0.0, 0.0},
     {.5597708310739475346078715485253291369276    ,0.0, 0.0},
     {.411751161462842646035931793833051637079     ,0.0, 0.0},
     {.2518862256915055095889728548779112301629    ,0.0, 0.0},
     {.08477501304173530124226185293578381173332   ,0.0, 0.0},
     {-.08477501304173530124226185293578381173332  ,0.0, 0.0},
     {-.2518862256915055095889728548779112301629   ,0.0, 0.0},
     {-.411751161462842646035931793833051637079    ,0.0, 0.0},
     {-.5597708310739475346078715485253291369276   ,0.0, 0.0},
     {-.6916870430603532078748910812888483894523   ,0.0, 0.0},
     {-.8037049589725231156824174550145907971033   ,0.0, 0.0},
     {-.8926024664975557392060605911271455154079   ,0.0, 0.0},
     {-.9558239495713977551811958929297763099728   ,0.0, 0.0},
     {-.9915651684209309467300160047061507702526   ,0.0, 0.0}
    },
    { // weights
     .02161601352648331031334271026645246938769,
     .04971454889496979645333494620263864168087,
     .07642573025488905652912967761663652560532,
     .1009420441062871655628139849248346070628,
     .1225552067114784601845191268002015552282,
     .1406429146706506512047313037519472280955,
     .1546846751262652449254180038363747721932,
     .1642764837458327229860537764659275904123,
      .1691423829631435918406564701349866103341,
      .1691423829631435918406564701349866103341,
      .1642764837458327229860537764659275904123,
      .1546846751262652449254180038363747721932,
      .1406429146706506512047313037519472280955,
      .1225552067114784601845191268002015552282,
      .1009420441062871655628139849248346070628,
      .07642573025488905652912967761663652560532,
      .04971454889496979645333494620263864168087,
      .02161601352648331031334271026645246938769
    }
  },
  {
    18,
    { // points
     {.9915651684209309467300160047061507702526    ,0.0, 0.0},
     {.9558239495713977551811958929297763099728    ,0.0, 0.0},
     {.8926024664975557392060605911271455154079    ,0.0, 0.0},
     {.8037049589725231156824174550145907971033    ,0.0, 0.0},
     {.6916870430603532078748910812888483894523    ,0.0, 0.0},
     {.5597708310739475346078715485253291369276    ,0.0, 0.0},
     {.411751161462842646035931793833051637079     ,0.0, 0.0},
     {.2518862256915055095889728548779112301629    ,0.0, 0.0},
     {.08477501304173530124226185293578381173332   ,0.0, 0.0},
     {-.08477501304173530124226185293578381173332  ,0.0, 0.0},
     {-.2518862256915055095889728548779112301629   ,0.0, 0.0},
     {-.411751161462842646035931793833051637079    ,0.0, 0.0},
     {-.5597708310739475346078715485253291369276   ,0.0, 0.0},
     {-.6916870430603532078748910812888483894523   ,0.0, 0.0},
     {-.8037049589725231156824174550145907971033   ,0.0, 0.0},
     {-.8926024664975557392060605911271455154079   ,0.0, 0.0},
     {-.9558239495713977551811958929297763099728   ,0.0, 0.0},
     {-.9915651684209309467300160047061507702526   ,0.0, 0.0}
    },
    { // weights
     .02161601352648331031334271026645246938769,
     .04971454889496979645333494620263864168087,
     .07642573025488905652912967761663652560532,
     .1009420441062871655628139849248346070628,
     .1225552067114784601845191268002015552282,
     .1406429146706506512047313037519472280955,
     .1546846751262652449254180038363747721932,
     .1642764837458327229860537764659275904123,
      .1691423829631435918406564701349866103341,
      .1691423829631435918406564701349866103341,
      .1642764837458327229860537764659275904123,
      .1546846751262652449254180038363747721932,
      .1406429146706506512047313037519472280955,
      .1225552067114784601845191268002015552282,
      .1009420441062871655628139849248346070628,
      .07642573025488905652912967761663652560532,
      .04971454889496979645333494620263864168087,
      .02161601352648331031334271026645246938769
    }
  },
  {
    19,
    { // points
     {.9924068438435844031890176702532604935893    ,0.0, 0.0},
     {.9602081521348300308527788406876515266151    ,0.0, 0.0},
     {.9031559036148179016426609285323124878094    ,0.0, 0.0},
     {.8227146565371428249789224867127139017745    ,0.0, 0.0},
     {.7209661773352293786170958608237816296571    ,0.0, 0.0},
     {.6005453046616810234696381649462392798683    ,0.0, 0.0},
     {.4645707413759609457172671481041023679763    ,0.0, 0.0},
     {.3165640999636298319901173288498449178923    ,0.0, 0.0},
     {.1603586456402253758680961157407435495049    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.1603586456402253758680961157407435495049   ,0.0, 0.0},
     {-.3165640999636298319901173288498449178923   ,0.0, 0.0},
     {-.4645707413759609457172671481041023679763   ,0.0, 0.0},
     {-.6005453046616810234696381649462392798683   ,0.0, 0.0},
     {-.7209661773352293786170958608237816296571   ,0.0, 0.0},
     {-.8227146565371428249789224867127139017745   ,0.0, 0.0},
     {-.9031559036148179016426609285323124878094   ,0.0, 0.0},
     {-.9602081521348300308527788406876515266151   ,0.0, 0.0},
     {-.9924068438435844031890176702532604935893   ,0.0, 0.0}
    },
    { // weights
     .01946178822972647703631204146443843575291,
     .04481422676569960033283815740199421195175,
     .06904454273764122658070825800601304496185,
     .09149002162244999946446209412383965266091,
     .1115666455473339947160239016817659974813,
     .1287539625393362276755157848568771170558,
     .1426067021736066117757461094419029724757,
     .152766042065859666778855400897662998461,
     .1589688433939543476499564394650472016788,
      .1610544498487836959791636253209167350399,
      .1589688433939543476499564394650472016788,
      .152766042065859666778855400897662998461,
      .1426067021736066117757461094419029724757,
      .1287539625393362276755157848568771170558,
      .1115666455473339947160239016817659974813,
      .09149002162244999946446209412383965266091,
      .06904454273764122658070825800601304496185,
      .04481422676569960033283815740199421195175,
      .01946178822972647703631204146443843575291
    }
  },
  {
    19,
    { // points
     {.9924068438435844031890176702532604935893    ,0.0, 0.0},
     {.9602081521348300308527788406876515266151    ,0.0, 0.0},
     {.9031559036148179016426609285323124878094    ,0.0, 0.0},
     {.8227146565371428249789224867127139017745    ,0.0, 0.0},
     {.7209661773352293786170958608237816296571    ,0.0, 0.0},
     {.6005453046616810234696381649462392798683    ,0.0, 0.0},
     {.4645707413759609457172671481041023679763    ,0.0, 0.0},
     {.3165640999636298319901173288498449178923    ,0.0, 0.0},
     {.1603586456402253758680961157407435495049    ,0.0, 0.0},
     { 0                                           ,0.0, 0.0},
     {-.1603586456402253758680961157407435495049   ,0.0, 0.0},
     {-.3165640999636298319901173288498449178923   ,0.0, 0.0},
     {-.4645707413759609457172671481041023679763   ,0.0, 0.0},
     {-.6005453046616810234696381649462392798683   ,0.0, 0.0},
     {-.7209661773352293786170958608237816296571   ,0.0, 0.0},
     {-.8227146565371428249789224867127139017745   ,0.0, 0.0},
     {-.9031559036148179016426609285323124878094   ,0.0, 0.0},
     {-.9602081521348300308527788406876515266151   ,0.0, 0.0},
     {-.9924068438435844031890176702532604935893   ,0.0, 0.0}
    },
    { // weights
     .01946178822972647703631204146443843575291,
     .04481422676569960033283815740199421195175,
     .06904454273764122658070825800601304496185,
     .09149002162244999946446209412383965266091,
     .1115666455473339947160239016817659974813,
     .1287539625393362276755157848568771170558,
     .1426067021736066117757461094419029724757,
     .152766042065859666778855400897662998461,
     .1589688433939543476499564394650472016788,
      .1610544498487836959791636253209167350399,
      .1589688433939543476499564394650472016788,
      .152766042065859666778855400897662998461,
      .1426067021736066117757461094419029724757,
      .1287539625393362276755157848568771170558,
      .1115666455473339947160239016817659974813,
      .09149002162244999946446209412383965266091,
      .06904454273764122658070825800601304496185,
      .04481422676569960033283815740199421195175,
      .01946178822972647703631204146443843575291
    }
  },
  {
    20,
    { // points
     {.9931285991850949247861223884713202782226    ,0.0, 0.0},
     {.9639719272779137912676661311972772219121    ,0.0, 0.0},
     {.9122344282513259058677524412032981130492    ,0.0, 0.0},
     {.8391169718222188233945290617015206853296    ,0.0, 0.0},
     {.7463319064601507926143050703556415903107    ,0.0, 0.0},
     {.6360536807265150254528366962262859367434    ,0.0, 0.0},
     {.5108670019508270980043640509552509984255    ,0.0, 0.0},
     {.3737060887154195606725481770249272373957    ,0.0, 0.0},
     {.2277858511416450780804961953685746247431    ,0.0, 0.0},
     {.0765265211334973337546404093988382110048    ,0.0, 0.0},
     {-.0765265211334973337546404093988382110048   ,0.0, 0.0},
     {-.2277858511416450780804961953685746247431   ,0.0, 0.0},
     {-.3737060887154195606725481770249272373957   ,0.0, 0.0},
     {-.5108670019508270980043640509552509984255   ,0.0, 0.0},
     {-.6360536807265150254528366962262859367434   ,0.0, 0.0},
     {-.7463319064601507926143050703556415903107   ,0.0, 0.0},
     {-.8391169718222188233945290617015206853296   ,0.0, 0.0},
     {-.9122344282513259058677524412032981130492   ,0.0, 0.0},
     {-.9639719272779137912676661311972772219121   ,0.0, 0.0},
     {-.9931285991850949247861223884713202782226   ,0.0, 0.0}
    },
    { // weights
     .01761400713915211831186196235185281636214,
     .04060142980038694133103995227493210987909,
     .0626720483341090635695065351870416063516,
     .08327674157670474872475814322204620610018,
     .1019301198172404350367501354803498761667,
     .118194531961518417312377377711382287005,
     .1316886384491766268984944997481631349161,
     .1420961093183820513292983250671649330345,
     .1491729864726037467878287370019694366927,
     .1527533871307258506980843319550975934919,
      .1527533871307258506980843319550975934919,
      .1491729864726037467878287370019694366927,
      .1420961093183820513292983250671649330345,
      .1316886384491766268984944997481631349161,
      .118194531961518417312377377711382287005,
      .1019301198172404350367501354803498761667,
      .08327674157670474872475814322204620610018,
      .0626720483341090635695065351870416063516,
      .04060142980038694133103995227493210987909,
      .01761400713915211831186196235185281636214
    }
  },
  {
    20,
    { // points
     {.9931285991850949247861223884713202782226    ,0.0, 0.0},
     {.9639719272779137912676661311972772219121    ,0.0, 0.0},
     {.9122344282513259058677524412032981130492    ,0.0, 0.0},
     {.8391169718222188233945290617015206853296    ,0.0, 0.0},
     {.7463319064601507926143050703556415903107    ,0.0, 0.0},
     {.6360536807265150254528366962262859367434    ,0.0, 0.0},
     {.5108670019508270980043640509552509984255    ,0.0, 0.0},
     {.3737060887154195606725481770249272373957    ,0.0, 0.0},
     {.2277858511416450780804961953685746247431    ,0.0, 0.0},
     {.0765265211334973337546404093988382110048    ,0.0, 0.0},
     {-.0765265211334973337546404093988382110048   ,0.0, 0.0},
     {-.2277858511416450780804961953685746247431   ,0.0, 0.0},
     {-.3737060887154195606725481770249272373957   ,0.0, 0.0},
     {-.5108670019508270980043640509552509984255   ,0.0, 0.0},
     {-.6360536807265150254528366962262859367434   ,0.0, 0.0},
     {-.7463319064601507926143050703556415903107   ,0.0, 0.0},
     {-.8391169718222188233945290617015206853296   ,0.0, 0.0},
     {-.9122344282513259058677524412032981130492   ,0.0, 0.0},
     {-.9639719272779137912676661311972772219121   ,0.0, 0.0},
     {-.9931285991850949247861223884713202782226   ,0.0, 0.0}
    },
    { // weights
     .01761400713915211831186196235185281636214,
     .04060142980038694133103995227493210987909,
     .0626720483341090635695065351870416063516,
     .08327674157670474872475814322204620610018,
     .1019301198172404350367501354803498761667,
     .118194531961518417312377377711382287005,
     .1316886384491766268984944997481631349161,
     .1420961093183820513292983250671649330345,
     .1491729864726037467878287370019694366927,
     .1527533871307258506980843319550975934919,
      .1527533871307258506980843319550975934919,
      .1491729864726037467878287370019694366927,
      .1420961093183820513292983250671649330345,
      .1316886384491766268984944997481631349161,
      .118194531961518417312377377711382287005,
      .1019301198172404350367501354803498761667,
      .08327674157670474872475814322204620610018,
      .0626720483341090635695065351870416063516,
      .04060142980038694133103995227493210987909,
      .01761400713915211831186196235185281636214
    }
  },
  {
    21,
    { // points
     {.9937521706203895002602420359379409291933    ,0.0, 0.0},
     {.9672268385663062943166222149076951614247    ,0.0, 0.0},
     {.9200993341504008287901871337149688941591    ,0.0, 0.0},
     {.8533633645833172836472506385875676702761    ,0.0, 0.0},
     {.768439963475677908615877851306228034821     ,0.0, 0.0},
     {.667138804197412319305966669990339162597     ,0.0, 0.0},
     {.5516188358872198070590187967243132866221    ,0.0, 0.0},
     {.4243421202074387835736688885437880520964    ,0.0, 0.0},
     {.288021316802401096600792516064600319909     ,0.0, 0.0},
     {.1455618541608950909370309823386863301163    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1455618541608950909370309823386863301163   ,0.0, 0.0},
     {-.288021316802401096600792516064600319909    ,0.0, 0.0},
     {-.4243421202074387835736688885437880520964   ,0.0, 0.0},
     {-.5516188358872198070590187967243132866221   ,0.0, 0.0},
     {-.667138804197412319305966669990339162597    ,0.0, 0.0},
     {-.768439963475677908615877851306228034821    ,0.0, 0.0},
     {-.8533633645833172836472506385875676702761   ,0.0, 0.0},
     {-.9200993341504008287901871337149688941591   ,0.0, 0.0},
     {-.9672268385663062943166222149076951614247   ,0.0, 0.0},
     {-.9937521706203895002602420359379409291933   ,0.0, 0.0}
    },
    { // weights
     .01601722825777433332422461685847101526589,
     .03695378977085249379995066829932966618894,
     .05713442542685720828363582647244795749129,
     .07610011362837930201705165330018317922615,
     .09344442345603386155328974111393208848353,
     .1087972991671483776634745780701056420337,
     .1218314160537285341953671771257335983563,
     .132268938633337461781052574496775604329,
     .1398873947910731547221334238675831108928,
     .1445244039899700590638271665537525436099,
      .1460811336496904271919851476833711882448,
     .1445244039899700590638271665537525436099,
     .1398873947910731547221334238675831108928,
     .132268938633337461781052574496775604329,
     .1218314160537285341953671771257335983563,
     .1087972991671483776634745780701056420337,
     .09344442345603386155328974111393208848353,
     .07610011362837930201705165330018317922615,
     .05713442542685720828363582647244795749129,
     .03695378977085249379995066829932966618894,
     .01601722825777433332422461685847101526589
    }
  },
  {
    21,
    { // points
     {.9937521706203895002602420359379409291933    ,0.0, 0.0},
     {.9672268385663062943166222149076951614247    ,0.0, 0.0},
     {.9200993341504008287901871337149688941591    ,0.0, 0.0},
     {.8533633645833172836472506385875676702761    ,0.0, 0.0},
     {.768439963475677908615877851306228034821     ,0.0, 0.0},
     {.667138804197412319305966669990339162597     ,0.0, 0.0},
     {.5516188358872198070590187967243132866221    ,0.0, 0.0},
     {.4243421202074387835736688885437880520964    ,0.0, 0.0},
     {.288021316802401096600792516064600319909     ,0.0, 0.0},
     {.1455618541608950909370309823386863301163    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1455618541608950909370309823386863301163   ,0.0, 0.0},
     {-.288021316802401096600792516064600319909    ,0.0, 0.0},
     {-.4243421202074387835736688885437880520964   ,0.0, 0.0},
     {-.5516188358872198070590187967243132866221   ,0.0, 0.0},
     {-.667138804197412319305966669990339162597    ,0.0, 0.0},
     {-.768439963475677908615877851306228034821    ,0.0, 0.0},
     {-.8533633645833172836472506385875676702761   ,0.0, 0.0},
     {-.9200993341504008287901871337149688941591   ,0.0, 0.0},
     {-.9672268385663062943166222149076951614247   ,0.0, 0.0},
     {-.9937521706203895002602420359379409291933   ,0.0, 0.0}
    },
    { // weights
     .01601722825777433332422461685847101526589,
     .03695378977085249379995066829932966618894,
     .05713442542685720828363582647244795749129,
     .07610011362837930201705165330018317922615,
     .09344442345603386155328974111393208848353,
     .1087972991671483776634745780701056420337,
     .1218314160537285341953671771257335983563,
     .132268938633337461781052574496775604329,
     .1398873947910731547221334238675831108928,
     .1445244039899700590638271665537525436099,
      .1460811336496904271919851476833711882448,
     .1445244039899700590638271665537525436099,
     .1398873947910731547221334238675831108928,
     .132268938633337461781052574496775604329,
     .1218314160537285341953671771257335983563,
     .1087972991671483776634745780701056420337,
     .09344442345603386155328974111393208848353,
     .07610011362837930201705165330018317922615,
     .05713442542685720828363582647244795749129,
     .03695378977085249379995066829932966618894,
     .01601722825777433332422461685847101526589
    }
  },
  {
    22,
    { // points
     {.9942945854823992920730314211612989803931    ,0.0, 0.0},
     {.970060497835428727123950986765268710806     ,0.0, 0.0},
     {.9269567721871740005206929392590531966353    ,0.0, 0.0},
     {.8658125777203001365364256370193787290848    ,0.0, 0.0},
     {.7878168059792081620042779554083515213882    ,0.0, 0.0},
     {.6944872631866827800506898357622567712673    ,0.0, 0.0},
     {.5876404035069115929588769276386473488776    ,0.0, 0.0},
     {.4693558379867570264063307109664063460953    ,0.0, 0.0},
     {.3419358208920842251581474204273796195592    ,0.0, 0.0},
     {.2078604266882212854788465339195457342157    ,0.0, 0.0},
     {.0697392733197222212138417961186280818223    ,0.0, 0.0},
     {-.0697392733197222212138417961186280818223   ,0.0, 0.0},
     {-.2078604266882212854788465339195457342157   ,0.0, 0.0},
     {-.3419358208920842251581474204273796195592   ,0.0, 0.0},
     {-.4693558379867570264063307109664063460953   ,0.0, 0.0},
     {-.5876404035069115929588769276386473488776   ,0.0, 0.0},
     {-.6944872631866827800506898357622567712673   ,0.0, 0.0},
     {-.7878168059792081620042779554083515213882   ,0.0, 0.0},
     {-.8658125777203001365364256370193787290848   ,0.0, 0.0},
     {-.9269567721871740005206929392590531966353   ,0.0, 0.0},
     {-.970060497835428727123950986765268710806    ,0.0, 0.0},
     {-.9942945854823992920730314211612989803931   ,0.0, 0.0}
    },
    { // weights
     .01462799529827220068499109804718544519021,
     .03377490158481415479330224686591290134917,
     .05229333515268328594031205127321125611215,
     .06979646842452048809496141893021765739878,
     .08594160621706772741444368137270286618914,
     .1004141444428809649320788378305362823508,
     .1129322960805392183934006074217843191143,
     .1232523768105124242855609861548144719594,
     .1311735047870623707329649925303074458757,
     .1365414983460151713525738312315173965864,
     .1392518728556319933754102483418099578739,
      .1392518728556319933754102483418099578739,
      .1365414983460151713525738312315173965864,
      .1311735047870623707329649925303074458757,
      .1232523768105124242855609861548144719594,
      .1129322960805392183934006074217843191143,
      .1004141444428809649320788378305362823508,
      .08594160621706772741444368137270286618914,
      .06979646842452048809496141893021765739878,
      .05229333515268328594031205127321125611215,
      .03377490158481415479330224686591290134917,
      .01462799529827220068499109804718544519021
    }
  },
  {
    22,
    { // points
     {.9942945854823992920730314211612989803931    ,0.0, 0.0},
     {.970060497835428727123950986765268710806     ,0.0, 0.0},
     {.9269567721871740005206929392590531966353    ,0.0, 0.0},
     {.8658125777203001365364256370193787290848    ,0.0, 0.0},
     {.7878168059792081620042779554083515213882    ,0.0, 0.0},
     {.6944872631866827800506898357622567712673    ,0.0, 0.0},
     {.5876404035069115929588769276386473488776    ,0.0, 0.0},
     {.4693558379867570264063307109664063460953    ,0.0, 0.0},
     {.3419358208920842251581474204273796195592    ,0.0, 0.0},
     {.2078604266882212854788465339195457342157    ,0.0, 0.0},
     {.0697392733197222212138417961186280818223    ,0.0, 0.0},
     {-.0697392733197222212138417961186280818223   ,0.0, 0.0},
     {-.2078604266882212854788465339195457342157   ,0.0, 0.0},
     {-.3419358208920842251581474204273796195592   ,0.0, 0.0},
     {-.4693558379867570264063307109664063460953   ,0.0, 0.0},
     {-.5876404035069115929588769276386473488776   ,0.0, 0.0},
     {-.6944872631866827800506898357622567712673   ,0.0, 0.0},
     {-.7878168059792081620042779554083515213882   ,0.0, 0.0},
     {-.8658125777203001365364256370193787290848   ,0.0, 0.0},
     {-.9269567721871740005206929392590531966353   ,0.0, 0.0},
     {-.970060497835428727123950986765268710806    ,0.0, 0.0},
     {-.9942945854823992920730314211612989803931   ,0.0, 0.0}
    },
    { // weights
     .01462799529827220068499109804718544519021,
     .03377490158481415479330224686591290134917,
     .05229333515268328594031205127321125611215,
     .06979646842452048809496141893021765739878,
     .08594160621706772741444368137270286618914,
     .1004141444428809649320788378305362823508,
     .1129322960805392183934006074217843191143,
     .1232523768105124242855609861548144719594,
     .1311735047870623707329649925303074458757,
     .1365414983460151713525738312315173965864,
     .1392518728556319933754102483418099578739,
      .1392518728556319933754102483418099578739,
      .1365414983460151713525738312315173965864,
      .1311735047870623707329649925303074458757,
      .1232523768105124242855609861548144719594,
      .1129322960805392183934006074217843191143,
      .1004141444428809649320788378305362823508,
      .08594160621706772741444368137270286618914,
      .06979646842452048809496141893021765739878,
      .05229333515268328594031205127321125611215,
      .03377490158481415479330224686591290134917,
      .01462799529827220068499109804718544519021
    }
  },
  {
    23,
    { // points
     {.9947693349975521235239257154455743605736    ,0.0, 0.0},
     {.9725424712181152319560240768207773751816    ,0.0, 0.0},
     {.9329710868260161023491969890384229782357    ,0.0, 0.0},
     {.8767523582704416673781568859341456716389    ,0.0, 0.0},
     {.8048884016188398921511184069967785579414    ,0.0, 0.0},
     {.7186613631319501944616244837486188483299    ,0.0, 0.0},
     {.6196098757636461563850973116495956533872    ,0.0, 0.0},
     {.5095014778460075496897930478668464305448    ,0.0, 0.0},
     {.3903010380302908314214888728806054585781    ,0.0, 0.0},
     {.264135680970344930533869538283309602979     ,0.0, 0.0},
     {.1332568242984661109317426822417661370104    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1332568242984661109317426822417661370104   ,0.0, 0.0},
     {-.264135680970344930533869538283309602979    ,0.0, 0.0},
     {-.3903010380302908314214888728806054585781   ,0.0, 0.0},
     {-.5095014778460075496897930478668464305448   ,0.0, 0.0},
     {-.6196098757636461563850973116495956533872   ,0.0, 0.0},
     {-.7186613631319501944616244837486188483299   ,0.0, 0.0},
     {-.8048884016188398921511184069967785579414   ,0.0, 0.0},
     {-.8767523582704416673781568859341456716389   ,0.0, 0.0},
     {-.9329710868260161023491969890384229782357   ,0.0, 0.0},
     {-.9725424712181152319560240768207773751816   ,0.0, 0.0},
     {-.9947693349975521235239257154455743605736   ,0.0, 0.0}
    },
    { // weights
     .01341185948714177208130949345861506497662,
     .03098800585697944431069421964188450538377,
     .04803767173108466857164107163203399656122,
     .06423242140852585212716961515891099803916,
     .07928141177671895492289252474204322691371,
     .09291576606003514747701861736976464860346,
     .1048920914645414100740861850147438548585,
     .1149966402224113649416435129339613014914,
     .1230490843067295304675784006720096548159,
     .1289057221880821499785953393997936532597,
     .132462039404696617371642464703316925805,
      .1336545721861061753514571105458443385832,
      .132462039404696617371642464703316925805,
      .1289057221880821499785953393997936532597,
      .1230490843067295304675784006720096548159,
      .1149966402224113649416435129339613014914,
      .1048920914645414100740861850147438548585,
      .09291576606003514747701861736976464860346,
      .07928141177671895492289252474204322691371,
      .06423242140852585212716961515891099803916,
      .04803767173108466857164107163203399656122,
      .03098800585697944431069421964188450538377,
      .01341185948714177208130949345861506497662
    }
  },
  {
    23,
    { // points
     {.9947693349975521235239257154455743605736    ,0.0, 0.0},
     {.9725424712181152319560240768207773751816    ,0.0, 0.0},
     {.9329710868260161023491969890384229782357    ,0.0, 0.0},
     {.8767523582704416673781568859341456716389    ,0.0, 0.0},
     {.8048884016188398921511184069967785579414    ,0.0, 0.0},
     {.7186613631319501944616244837486188483299    ,0.0, 0.0},
     {.6196098757636461563850973116495956533872    ,0.0, 0.0},
     {.5095014778460075496897930478668464305448    ,0.0, 0.0},
     {.3903010380302908314214888728806054585781    ,0.0, 0.0},
     {.264135680970344930533869538283309602979     ,0.0, 0.0},
     {.1332568242984661109317426822417661370104    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1332568242984661109317426822417661370104   ,0.0, 0.0},
     {-.264135680970344930533869538283309602979    ,0.0, 0.0},
     {-.3903010380302908314214888728806054585781   ,0.0, 0.0},
     {-.5095014778460075496897930478668464305448   ,0.0, 0.0},
     {-.6196098757636461563850973116495956533872   ,0.0, 0.0},
     {-.7186613631319501944616244837486188483299   ,0.0, 0.0},
     {-.8048884016188398921511184069967785579414   ,0.0, 0.0},
     {-.8767523582704416673781568859341456716389   ,0.0, 0.0},
     {-.9329710868260161023491969890384229782357   ,0.0, 0.0},
     {-.9725424712181152319560240768207773751816   ,0.0, 0.0},
     {-.9947693349975521235239257154455743605736   ,0.0, 0.0}
    },
    { // weights
     .01341185948714177208130949345861506497662,
     .03098800585697944431069421964188450538377,
     .04803767173108466857164107163203399656122,
     .06423242140852585212716961515891099803916,
     .07928141177671895492289252474204322691371,
     .09291576606003514747701861736976464860346,
     .1048920914645414100740861850147438548585,
     .1149966402224113649416435129339613014914,
     .1230490843067295304675784006720096548159,
     .1289057221880821499785953393997936532597,
     .132462039404696617371642464703316925805,
      .1336545721861061753514571105458443385832,
      .132462039404696617371642464703316925805,
      .1289057221880821499785953393997936532597,
      .1230490843067295304675784006720096548159,
      .1149966402224113649416435129339613014914,
      .1048920914645414100740861850147438548585,
      .09291576606003514747701861736976464860346,
      .07928141177671895492289252474204322691371,
      .06423242140852585212716961515891099803916,
      .04803767173108466857164107163203399656122,
      .03098800585697944431069421964188450538377,
      .01341185948714177208130949345861506497662
    }
  },
  {
    24,
    { // points
     {.9951872199970213601799974097007368118746    ,0.0, 0.0},
     {.9747285559713094981983919930081690617412    ,0.0, 0.0},
     {.9382745520027327585236490017087214496548    ,0.0, 0.0},
     {.8864155270044010342131543419821967550873    ,0.0, 0.0},
     {.8200019859739029219539498726697452080761    ,0.0, 0.0},
     {.7401241915785543642438281030999784255233    ,0.0, 0.0},
     {.6480936519369755692524957869107476266697    ,0.0, 0.0},
     {.5454214713888395356583756172183723700108    ,0.0, 0.0},
     {.4337935076260451384870842319133497124524    ,0.0, 0.0},
     {.3150426796961633743867932913198102407865    ,0.0, 0.0},
     {.1911188674736163091586398207570696318404    ,0.0, 0.0},
     {.064056892862605626085043082624745038591     ,0.0, 0.0},
     {-.064056892862605626085043082624745038591    ,0.0, 0.0},
     {-.1911188674736163091586398207570696318404   ,0.0, 0.0},
     {-.3150426796961633743867932913198102407865   ,0.0, 0.0},
     {-.4337935076260451384870842319133497124524   ,0.0, 0.0},
     {-.5454214713888395356583756172183723700108   ,0.0, 0.0},
     {-.6480936519369755692524957869107476266697   ,0.0, 0.0},
     {-.7401241915785543642438281030999784255233   ,0.0, 0.0},
     {-.8200019859739029219539498726697452080761   ,0.0, 0.0},
     {-.8864155270044010342131543419821967550873   ,0.0, 0.0},
     {-.9382745520027327585236490017087214496548   ,0.0, 0.0},
     {-.9747285559713094981983919930081690617412   ,0.0, 0.0},
     {-.9951872199970213601799974097007368118746   ,0.0, 0.0}
    },
    { // weights
     .01234122979998719954680566707003729157591,
     .0285313886289336631813078159518782864492,
     .04427743881741980616860274821133822885931,
     .0592985849154367807463677585001085845412,
     .07334648141108030573403361525311651811934,
     .08619016153195327591718520298374266718508,
     .09761865210411388826988066446424715442792,
     .1074442701159656347825773424466062227946,
     .1155056680537256013533444839067835598623,
     .121670472927803391204463153476262425607,
     .1258374563468282961213753825111836887264,
     .1279381953467521569740561652246953718517,
     .1279381953467521569740561652246953718517,
      .1258374563468282961213753825111836887264,
      .121670472927803391204463153476262425607,
      .1155056680537256013533444839067835598623,
      .1074442701159656347825773424466062227946,
      .09761865210411388826988066446424715442792,
      .08619016153195327591718520298374266718508,
      .07334648141108030573403361525311651811934,
      .0592985849154367807463677585001085845412,
      .04427743881741980616860274821133822885931,
      .0285313886289336631813078159518782864492,
      .01234122979998719954680566707003729157591
    }
  },
  {
    24,
    { // points
     {.9951872199970213601799974097007368118746    ,0.0, 0.0},
     {.9747285559713094981983919930081690617412    ,0.0, 0.0},
     {.9382745520027327585236490017087214496548    ,0.0, 0.0},
     {.8864155270044010342131543419821967550873    ,0.0, 0.0},
     {.8200019859739029219539498726697452080761    ,0.0, 0.0},
     {.7401241915785543642438281030999784255233    ,0.0, 0.0},
     {.6480936519369755692524957869107476266697    ,0.0, 0.0},
     {.5454214713888395356583756172183723700108    ,0.0, 0.0},
     {.4337935076260451384870842319133497124524    ,0.0, 0.0},
     {.3150426796961633743867932913198102407865    ,0.0, 0.0},
     {.1911188674736163091586398207570696318404    ,0.0, 0.0},
     {.064056892862605626085043082624745038591     ,0.0, 0.0},
     {-.064056892862605626085043082624745038591    ,0.0, 0.0},
     {-.1911188674736163091586398207570696318404   ,0.0, 0.0},
     {-.3150426796961633743867932913198102407865   ,0.0, 0.0},
     {-.4337935076260451384870842319133497124524   ,0.0, 0.0},
     {-.5454214713888395356583756172183723700108   ,0.0, 0.0},
     {-.6480936519369755692524957869107476266697   ,0.0, 0.0},
     {-.7401241915785543642438281030999784255233   ,0.0, 0.0},
     {-.8200019859739029219539498726697452080761   ,0.0, 0.0},
     {-.8864155270044010342131543419821967550873   ,0.0, 0.0},
     {-.9382745520027327585236490017087214496548   ,0.0, 0.0},
     {-.9747285559713094981983919930081690617412   ,0.0, 0.0},
     {-.9951872199970213601799974097007368118746   ,0.0, 0.0}
    },
    { // weights
     .01234122979998719954680566707003729157591,
     .0285313886289336631813078159518782864492,
     .04427743881741980616860274821133822885931,
     .0592985849154367807463677585001085845412,
     .07334648141108030573403361525311651811934,
     .08619016153195327591718520298374266718508,
     .09761865210411388826988066446424715442792,
     .1074442701159656347825773424466062227946,
     .1155056680537256013533444839067835598623,
     .121670472927803391204463153476262425607,
     .1258374563468282961213753825111836887264,
     .1279381953467521569740561652246953718517,
     .1279381953467521569740561652246953718517,
      .1258374563468282961213753825111836887264,
      .121670472927803391204463153476262425607,
      .1155056680537256013533444839067835598623,
      .1074442701159656347825773424466062227946,
      .09761865210411388826988066446424715442792,
      .08619016153195327591718520298374266718508,
      .07334648141108030573403361525311651811934,
      .0592985849154367807463677585001085845412,
      .04427743881741980616860274821133822885931,
      .0285313886289336631813078159518782864492,
      .01234122979998719954680566707003729157591
    }
  },
  {
    25,
    { // points
     {.9955569697904980979087849468939016172576    ,0.0, 0.0},
     {.9766639214595175114983153864795940677454    ,0.0, 0.0},
     {.9429745712289743394140111696584705319052    ,0.0, 0.0},
     {.8949919978782753688510420067828049541746    ,0.0, 0.0},
     {.833442628760834001421021108693569569461     ,0.0, 0.0},
     {.7592592630373576305772828652043609763875    ,0.0, 0.0},
     {.6735663684734683644851206332476221758834    ,0.0, 0.0},
     {.5776629302412229677236898416126540673957    ,0.0, 0.0},
     {.4730027314457149605221821150091920413318    ,0.0, 0.0},
     {.3611723058093878377358217301276406674221    ,0.0, 0.0},
     {.2438668837209884320451903627974515864056    ,0.0, 0.0},
     {.1228646926107103963873598188080368055322    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1228646926107103963873598188080368055322   ,0.0, 0.0},
     {-.2438668837209884320451903627974515864056   ,0.0, 0.0},
     {-.3611723058093878377358217301276406674221   ,0.0, 0.0},
     {-.4730027314457149605221821150091920413318   ,0.0, 0.0},
     {-.5776629302412229677236898416126540673957   ,0.0, 0.0},
     {-.6735663684734683644851206332476221758834   ,0.0, 0.0},
     {-.7592592630373576305772828652043609763875   ,0.0, 0.0},
     {-.833442628760834001421021108693569569461    ,0.0, 0.0},
     {-.8949919978782753688510420067828049541746   ,0.0, 0.0},
     {-.9429745712289743394140111696584705319052   ,0.0, 0.0},
     {-.9766639214595175114983153864795940677454   ,0.0, 0.0},
     {-.9955569697904980979087849468939016172576   ,0.0, 0.0}
    },
    { // weights
     .01139379850102628794790296411323477360332,
     .02635498661503213726190181529529914493596,
     .04093915670130631265562348771164595366085,
     .05490469597583519192593689154047332416011,
     .06803833381235691720718718565670796855471,
     .08014070033500101801323495966911130229023,
     .09102826198296364981149722070289165338099,
     .1005359490670506442022068903926858269885,
     .1085196244742636531160939570501166193401,
     .1148582591457116483393255458695558086409,
     .1194557635357847722281781265129010473902,
     .1222424429903100416889595189458515058351,
      .1231760537267154512039028730790501424382,
      .1222424429903100416889595189458515058351,
      .1194557635357847722281781265129010473902,
      .1148582591457116483393255458695558086409,
      .1085196244742636531160939570501166193401,
      .1005359490670506442022068903926858269885,
      .09102826198296364981149722070289165338099,
      .08014070033500101801323495966911130229023,
     .06803833381235691720718718565670796855471,
      .05490469597583519192593689154047332416011,
      .04093915670130631265562348771164595366085,
      .02635498661503213726190181529529914493596,
      .01139379850102628794790296411323477360332
    }
  },
  {
    25,
    { // points
     {.9955569697904980979087849468939016172576    ,0.0, 0.0},
     {.9766639214595175114983153864795940677454    ,0.0, 0.0},
     {.9429745712289743394140111696584705319052    ,0.0, 0.0},
     {.8949919978782753688510420067828049541746    ,0.0, 0.0},
     {.833442628760834001421021108693569569461     ,0.0, 0.0},
     {.7592592630373576305772828652043609763875    ,0.0, 0.0},
     {.6735663684734683644851206332476221758834    ,0.0, 0.0},
     {.5776629302412229677236898416126540673957    ,0.0, 0.0},
     {.4730027314457149605221821150091920413318    ,0.0, 0.0},
     {.3611723058093878377358217301276406674221    ,0.0, 0.0},
     {.2438668837209884320451903627974515864056    ,0.0, 0.0},
     {.1228646926107103963873598188080368055322    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1228646926107103963873598188080368055322   ,0.0, 0.0},
     {-.2438668837209884320451903627974515864056   ,0.0, 0.0},
     {-.3611723058093878377358217301276406674221   ,0.0, 0.0},
     {-.4730027314457149605221821150091920413318   ,0.0, 0.0},
     {-.5776629302412229677236898416126540673957   ,0.0, 0.0},
     {-.6735663684734683644851206332476221758834   ,0.0, 0.0},
     {-.7592592630373576305772828652043609763875   ,0.0, 0.0},
     {-.833442628760834001421021108693569569461    ,0.0, 0.0},
     {-.8949919978782753688510420067828049541746   ,0.0, 0.0},
     {-.9429745712289743394140111696584705319052   ,0.0, 0.0},
     {-.9766639214595175114983153864795940677454   ,0.0, 0.0},
     {-.9955569697904980979087849468939016172576   ,0.0, 0.0}
    },
    { // weights
     .01139379850102628794790296411323477360332,
     .02635498661503213726190181529529914493596,
     .04093915670130631265562348771164595366085,
     .05490469597583519192593689154047332416011,
     .06803833381235691720718718565670796855471,
     .08014070033500101801323495966911130229023,
     .09102826198296364981149722070289165338099,
     .1005359490670506442022068903926858269885,
     .1085196244742636531160939570501166193401,
     .1148582591457116483393255458695558086409,
     .1194557635357847722281781265129010473902,
     .1222424429903100416889595189458515058351,
      .1231760537267154512039028730790501424382,
      .1222424429903100416889595189458515058351,
      .1194557635357847722281781265129010473902,
      .1148582591457116483393255458695558086409,
      .1085196244742636531160939570501166193401,
      .1005359490670506442022068903926858269885,
      .09102826198296364981149722070289165338099,
      .08014070033500101801323495966911130229023,
     .06803833381235691720718718565670796855471,
      .05490469597583519192593689154047332416011,
      .04093915670130631265562348771164595366085,
      .02635498661503213726190181529529914493596,
      .01139379850102628794790296411323477360332
    }
  },
  {
    26,
    { // points
     {.9958857011456169290032169593229060259569    ,0.0, 0.0},
     {.9783854459564709911005803543119263054469    ,0.0, 0.0},
     {.9471590666617142501359152835180367545207    ,0.0, 0.0},
     {.9026378619843070742176655992312102513411    ,0.0, 0.0},
     {.8454459427884980187975070614678381615721    ,0.0, 0.0},
     {.7763859488206788561929672472422838669762    ,0.0, 0.0},
     {.6964272604199572648638139137294246978608    ,0.0, 0.0},
     {.6066922930176180632319787469168870544353    ,0.0, 0.0},
     {.5084407148245057176957030647255691753763    ,0.0, 0.0},
     {.4030517551234863064810773770988831036599    ,0.0, 0.0},
     {.2920048394859568951428353820778302968847    ,0.0, 0.0},
     {.1768588203568901839690577484183447499774    ,0.0, 0.0},
     {.05923009342931320709371857519840336079023   ,0.0, 0.0},
     {-.05923009342931320709371857519840336079023  ,0.0, 0.0},
     {-.1768588203568901839690577484183447499774   ,0.0, 0.0},
     {-.2920048394859568951428353820778302968847   ,0.0, 0.0},
     {-.4030517551234863064810773770988831036599   ,0.0, 0.0},
     {-.5084407148245057176957030647255691753763   ,0.0, 0.0},
     {-.6066922930176180632319787469168870544353   ,0.0, 0.0},
     {-.6964272604199572648638139137294246978608   ,0.0, 0.0},
     {-.7763859488206788561929672472422838669762   ,0.0, 0.0},
     {-.8454459427884980187975070614678381615721   ,0.0, 0.0},
     {-.9026378619843070742176655992312102513411   ,0.0, 0.0},
     {-.9471590666617142501359152835180367545207   ,0.0, 0.0},
     {-.9783854459564709911005803543119263054469   ,0.0, 0.0},
     {-.9958857011456169290032169593229060259569   ,0.0, 0.0}
    },
    { // weights
     .01055137261734300715565118768525197790435,
     .0244178510926319087896158275197884002405,
     .03796238329436276395030314124885049469077,
     .05097582529714781199831990072407337952588,
     .06327404632957483553945368990704509572222,
     .07468414976565974588707579610284833863875,
     .08504589431348523921044776507998169658392,
     .09421380035591414846366488306730319854917,
     .1020591610944254232384140702534307923272,
     .1084718405285765906565794267279975822894,
     .113361816546319666549440718442598145246,
     .116660443485296582044662507540362213632,
      .1183214152792622765163710857004686846499,
       .1183214152792622765163710857004686846499,
      .116660443485296582044662507540362213632,
      .113361816546319666549440718442598145246,
      .1084718405285765906565794267279975822894,
      .1020591610944254232384140702534307923272,
      .09421380035591414846366488306730319854917,
      .08504589431348523921044776507998169658392,
      .07468414976565974588707579610284833863875,
      .06327404632957483553945368990704509572222,
      .05097582529714781199831990072407337952588,
      .03796238329436276395030314124885049469077,
      .0244178510926319087896158275197884002405,
      .01055137261734300715565118768525197790435
    }
  },
  {
    26,
    { // points
     {.9958857011456169290032169593229060259569    ,0.0, 0.0},
     {.9783854459564709911005803543119263054469    ,0.0, 0.0},
     {.9471590666617142501359152835180367545207    ,0.0, 0.0},
     {.9026378619843070742176655992312102513411    ,0.0, 0.0},
     {.8454459427884980187975070614678381615721    ,0.0, 0.0},
     {.7763859488206788561929672472422838669762    ,0.0, 0.0},
     {.6964272604199572648638139137294246978608    ,0.0, 0.0},
     {.6066922930176180632319787469168870544353    ,0.0, 0.0},
     {.5084407148245057176957030647255691753763    ,0.0, 0.0},
     {.4030517551234863064810773770988831036599    ,0.0, 0.0},
     {.2920048394859568951428353820778302968847    ,0.0, 0.0},
     {.1768588203568901839690577484183447499774    ,0.0, 0.0},
     {.05923009342931320709371857519840336079023   ,0.0, 0.0},
     {-.05923009342931320709371857519840336079023  ,0.0, 0.0},
     {-.1768588203568901839690577484183447499774   ,0.0, 0.0},
     {-.2920048394859568951428353820778302968847   ,0.0, 0.0},
     {-.4030517551234863064810773770988831036599   ,0.0, 0.0},
     {-.5084407148245057176957030647255691753763   ,0.0, 0.0},
     {-.6066922930176180632319787469168870544353   ,0.0, 0.0},
     {-.6964272604199572648638139137294246978608   ,0.0, 0.0},
     {-.7763859488206788561929672472422838669762   ,0.0, 0.0},
     {-.8454459427884980187975070614678381615721   ,0.0, 0.0},
     {-.9026378619843070742176655992312102513411   ,0.0, 0.0},
     {-.9471590666617142501359152835180367545207   ,0.0, 0.0},
     {-.9783854459564709911005803543119263054469   ,0.0, 0.0},
     {-.9958857011456169290032169593229060259569   ,0.0, 0.0}
    },
    { // weights
     .01055137261734300715565118768525197790435,
     .0244178510926319087896158275197884002405,
     .03796238329436276395030314124885049469077,
     .05097582529714781199831990072407337952588,
     .06327404632957483553945368990704509572222,
     .07468414976565974588707579610284833863875,
     .08504589431348523921044776507998169658392,
     .09421380035591414846366488306730319854917,
     .1020591610944254232384140702534307923272,
     .1084718405285765906565794267279975822894,
     .113361816546319666549440718442598145246,
     .116660443485296582044662507540362213632,
      .1183214152792622765163710857004686846499,
       .1183214152792622765163710857004686846499,
      .116660443485296582044662507540362213632,
      .113361816546319666549440718442598145246,
      .1084718405285765906565794267279975822894,
      .1020591610944254232384140702534307923272,
      .09421380035591414846366488306730319854917,
      .08504589431348523921044776507998169658392,
      .07468414976565974588707579610284833863875,
      .06327404632957483553945368990704509572222,
      .05097582529714781199831990072407337952588,
      .03796238329436276395030314124885049469077,
      .0244178510926319087896158275197884002405,
      .01055137261734300715565118768525197790435
    }
  },
  {
    27,
    { // points
     {.9961792628889885669388872083812299902607    ,0.0, 0.0},
     {.9799234759615012228558733556610471845676    ,0.0, 0.0},
     {.9509005578147050068519080306438828930926    ,0.0, 0.0},
     {.909482320677491104300645018209689254948     ,0.0, 0.0},
     {.8562079080182944903027372227068427977643    ,0.0, 0.0},
     {.7917716390705082271443973441072707329194    ,0.0, 0.0},
     {.7170134737394236992948162116423395497507    ,0.0, 0.0},
     {.6329079719464951409277346376344886121031    ,0.0, 0.0},
     {.5405515645794568949003009415599828209592    ,0.0, 0.0},
     {.4411482517500268805859741556892814524477    ,0.0, 0.0},
     {.33599390363850889973031903420728454179      ,0.0, 0.0},
     {.2264593654395368588572391073602301711794    ,0.0, 0.0},
     {.1139725856095299669328949838699766271733    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1139725856095299669328949838699766271733   ,0.0, 0.0},
     {-.2264593654395368588572391073602301711794   ,0.0, 0.0},
     {-.33599390363850889973031903420728454179     ,0.0, 0.0},
     {-.4411482517500268805859741556892814524477   ,0.0, 0.0},
     {-.5405515645794568949003009415599828209592   ,0.0, 0.0},
     {-.6329079719464951409277346376344886121031   ,0.0, 0.0},
     {-.7170134737394236992948162116423395497507   ,0.0, 0.0},
     {-.7917716390705082271443973441072707329194   ,0.0, 0.0},
     {-.8562079080182944903027372227068427977643   ,0.0, 0.0},
     {-.909482320677491104300645018209689254948    ,0.0, 0.0},
     {-.9509005578147050068519080306438828930926   ,0.0, 0.0},
     {-.9799234759615012228558733556610471845676   ,0.0, 0.0},
     {-.9961792628889885669388872083812299902607   ,0.0, 0.0}
    },
    { // weights
     .009798996051294360261150055091259098098226,
     .02268623159618062319603420644676148881277,
     .03529705375741971102257828930471164113614,
     .04744941252061506270409671011418471822473,
     .05898353685983359911030083371953163316524,
     .06974882376624559298432288835666727613282,
     .07960486777305777126307495900984240297615,
     .08842315854375695019432280285374911105654,
     .09608872737002850756565264655810585916021,
     .1025016378177457986712477115326638093244,
     .107578285788533187212162984426657404819,
     .1112524883568451926721630960428486748329,
     .1134763461089651486203699480920992579205,
      .1142208673789569890450457369018352486807,
      .1134763461089651486203699480920992579205,
      .1112524883568451926721630960428486748329,
      .107578285788533187212162984426657404819,
      .1025016378177457986712477115326638093244,
      .09608872737002850756565264655810585916021,
      .08842315854375695019432280285374911105654,
      .07960486777305777126307495900984240297615,
      .06974882376624559298432288835666727613282,
      .05898353685983359911030083371953163316524,
      .04744941252061506270409671011418471822473,
      .03529705375741971102257828930471164113614,
      .02268623159618062319603420644676148881277,
      .009798996051294360261150055091259098098226
    }
  },
  {
    27,
    { // points
     {.9961792628889885669388872083812299902607    ,0.0, 0.0},
     {.9799234759615012228558733556610471845676    ,0.0, 0.0},
     {.9509005578147050068519080306438828930926    ,0.0, 0.0},
     {.909482320677491104300645018209689254948     ,0.0, 0.0},
     {.8562079080182944903027372227068427977643    ,0.0, 0.0},
     {.7917716390705082271443973441072707329194    ,0.0, 0.0},
     {.7170134737394236992948162116423395497507    ,0.0, 0.0},
     {.6329079719464951409277346376344886121031    ,0.0, 0.0},
     {.5405515645794568949003009415599828209592    ,0.0, 0.0},
     {.4411482517500268805859741556892814524477    ,0.0, 0.0},
     {.33599390363850889973031903420728454179      ,0.0, 0.0},
     {.2264593654395368588572391073602301711794    ,0.0, 0.0},
     {.1139725856095299669328949838699766271733    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1139725856095299669328949838699766271733   ,0.0, 0.0},
     {-.2264593654395368588572391073602301711794   ,0.0, 0.0},
     {-.33599390363850889973031903420728454179     ,0.0, 0.0},
     {-.4411482517500268805859741556892814524477   ,0.0, 0.0},
     {-.5405515645794568949003009415599828209592   ,0.0, 0.0},
     {-.6329079719464951409277346376344886121031   ,0.0, 0.0},
     {-.7170134737394236992948162116423395497507   ,0.0, 0.0},
     {-.7917716390705082271443973441072707329194   ,0.0, 0.0},
     {-.8562079080182944903027372227068427977643   ,0.0, 0.0},
     {-.909482320677491104300645018209689254948    ,0.0, 0.0},
     {-.9509005578147050068519080306438828930926   ,0.0, 0.0},
     {-.9799234759615012228558733556610471845676   ,0.0, 0.0},
     {-.9961792628889885669388872083812299902607   ,0.0, 0.0}
    },
    { // weights
     .009798996051294360261150055091259098098226,
     .02268623159618062319603420644676148881277,
     .03529705375741971102257828930471164113614,
     .04744941252061506270409671011418471822473,
     .05898353685983359911030083371953163316524,
     .06974882376624559298432288835666727613282,
     .07960486777305777126307495900984240297615,
     .08842315854375695019432280285374911105654,
     .09608872737002850756565264655810585916021,
     .1025016378177457986712477115326638093244,
     .107578285788533187212162984426657404819,
     .1112524883568451926721630960428486748329,
     .1134763461089651486203699480920992579205,
      .1142208673789569890450457369018352486807,
      .1134763461089651486203699480920992579205,
      .1112524883568451926721630960428486748329,
      .107578285788533187212162984426657404819,
      .1025016378177457986712477115326638093244,
      .09608872737002850756565264655810585916021,
      .08842315854375695019432280285374911105654,
      .07960486777305777126307495900984240297615,
      .06974882376624559298432288835666727613282,
      .05898353685983359911030083371953163316524,
      .04744941252061506270409671011418471822473,
      .03529705375741971102257828930471164113614,
      .02268623159618062319603420644676148881277,
      .009798996051294360261150055091259098098226
    }
  },
  {
    28,
    { // points
     {.9964424975739544499504363904833109917501    ,0.0, 0.0},
     {.9813031653708727536945599458078302764442    ,0.0, 0.0},
     {.9542592806289381972541018397052155890034    ,0.0, 0.0},
     {.9156330263921320738696894233299271463525    ,0.0, 0.0},
     {.8658925225743950489422545673796868043419    ,0.0, 0.0},
     {.8056413709171791714478859554252776586717    ,0.0, 0.0},
     {.7356108780136317720281445102925343680161    ,0.0, 0.0},
     {.6566510940388649612198981765067428215696    ,0.0, 0.0},
     {.5697204718114017193080032833564309331247    ,0.0, 0.0},
     {.4758742249551182610344118476674340426272    ,0.0, 0.0},
     {.3762515160890787102213572095560869828717    ,0.0, 0.0},
     {.2720616276351780776768263561257697348248    ,0.0, 0.0},
     {.1645692821333807712814717778911654571457    ,0.0, 0.0},
     {.0550792898840342704265165273418799778508    ,0.0, 0.0},
     {-.0550792898840342704265165273418799778508   ,0.0, 0.0},
     {-.1645692821333807712814717778911654571457   ,0.0, 0.0},
     {-.2720616276351780776768263561257697348248   ,0.0, 0.0},
     {-.3762515160890787102213572095560869828717   ,0.0, 0.0},
     {-.4758742249551182610344118476674340426272   ,0.0, 0.0},
     {-.5697204718114017193080032833564309331247   ,0.0, 0.0},
     {-.6566510940388649612198981765067428215696   ,0.0, 0.0},
     {-.7356108780136317720281445102925343680161   ,0.0, 0.0},
     {-.8056413709171791714478859554252776586717   ,0.0, 0.0},
     {-.8658925225743950489422545673796868043419   ,0.0, 0.0},
     {-.9156330263921320738696894233299271463525   ,0.0, 0.0},
     {-.9542592806289381972541018397052155890034   ,0.0, 0.0},
     {-.9813031653708727536945599458078302764442   ,0.0, 0.0},
     {-.9964424975739544499504363904833109917501   ,0.0, 0.0}
    },
    { // weights
     .009124282593094517738816153922951706240344,
     .02113211259277125975150038099326543445137,
     .03290142778230437997763081917053204598153,
     .04427293475900422783958787765320739061499,
     .05510734567571674543148291822694559121906,
     .06527292396699959579339756677550463635364,
     .07464621423456877902393188717302203802164,
     .08311341722890121839039649824433247986113,
     .09057174439303284094218603133678412982286,
     .09693065799792991585048900609544060176503,
     .1021129675780607698142166385057121350466,
     .1060557659228464179104164369968108287899,
     .1087111922582941352535715193036733678753,
     .1100470130164751962823762656018176139566,
      .1100470130164751962823762656018176139566,
      .1087111922582941352535715193036733678753,
      .1060557659228464179104164369968108287899,
      .1021129675780607698142166385057121350466,
      .09693065799792991585048900609544060176503,
      .09057174439303284094218603133678412982286,
      .08311341722890121839039649824433247986113,
      .07464621423456877902393188717302203802164,
      .06527292396699959579339756677550463635364,
      .05510734567571674543148291822694559121906,
      .04427293475900422783958787765320739061499,
      .03290142778230437997763081917053204598153,
      .02113211259277125975150038099326543445137,
      .009124282593094517738816153922951706240344
    }
  },
  {
    28,
    { // points
     {.9964424975739544499504363904833109917501    ,0.0, 0.0},
     {.9813031653708727536945599458078302764442    ,0.0, 0.0},
     {.9542592806289381972541018397052155890034    ,0.0, 0.0},
     {.9156330263921320738696894233299271463525    ,0.0, 0.0},
     {.8658925225743950489422545673796868043419    ,0.0, 0.0},
     {.8056413709171791714478859554252776586717    ,0.0, 0.0},
     {.7356108780136317720281445102925343680161    ,0.0, 0.0},
     {.6566510940388649612198981765067428215696    ,0.0, 0.0},
     {.5697204718114017193080032833564309331247    ,0.0, 0.0},
     {.4758742249551182610344118476674340426272    ,0.0, 0.0},
     {.3762515160890787102213572095560869828717    ,0.0, 0.0},
     {.2720616276351780776768263561257697348248    ,0.0, 0.0},
     {.1645692821333807712814717778911654571457    ,0.0, 0.0},
     {.0550792898840342704265165273418799778508    ,0.0, 0.0},
     {-.0550792898840342704265165273418799778508   ,0.0, 0.0},
     {-.1645692821333807712814717778911654571457   ,0.0, 0.0},
     {-.2720616276351780776768263561257697348248   ,0.0, 0.0},
     {-.3762515160890787102213572095560869828717   ,0.0, 0.0},
     {-.4758742249551182610344118476674340426272   ,0.0, 0.0},
     {-.5697204718114017193080032833564309331247   ,0.0, 0.0},
     {-.6566510940388649612198981765067428215696   ,0.0, 0.0},
     {-.7356108780136317720281445102925343680161   ,0.0, 0.0},
     {-.8056413709171791714478859554252776586717   ,0.0, 0.0},
     {-.8658925225743950489422545673796868043419   ,0.0, 0.0},
     {-.9156330263921320738696894233299271463525   ,0.0, 0.0},
     {-.9542592806289381972541018397052155890034   ,0.0, 0.0},
     {-.9813031653708727536945599458078302764442   ,0.0, 0.0},
     {-.9964424975739544499504363904833109917501   ,0.0, 0.0}
    },
    { // weights
     .009124282593094517738816153922951706240344,
     .02113211259277125975150038099326543445137,
     .03290142778230437997763081917053204598153,
     .04427293475900422783958787765320739061499,
     .05510734567571674543148291822694559121906,
     .06527292396699959579339756677550463635364,
     .07464621423456877902393188717302203802164,
     .08311341722890121839039649824433247986113,
     .09057174439303284094218603133678412982286,
     .09693065799792991585048900609544060176503,
     .1021129675780607698142166385057121350466,
     .1060557659228464179104164369968108287899,
     .1087111922582941352535715193036733678753,
     .1100470130164751962823762656018176139566,
      .1100470130164751962823762656018176139566,
      .1087111922582941352535715193036733678753,
      .1060557659228464179104164369968108287899,
      .1021129675780607698142166385057121350466,
      .09693065799792991585048900609544060176503,
      .09057174439303284094218603133678412982286,
      .08311341722890121839039649824433247986113,
      .07464621423456877902393188717302203802164,
      .06527292396699959579339756677550463635364,
      .05510734567571674543148291822694559121906,
      .04427293475900422783958787765320739061499,
      .03290142778230437997763081917053204598153,
      .02113211259277125975150038099326543445137,
      .009124282593094517738816153922951706240344
    }
  },
  {
    29,
    { // points
     {.9966794422605965861631915325493538856518    ,0.0, 0.0},
     {.9825455052614131748709260157863769561068    ,0.0, 0.0},
     {.9572855957780877257982080369808235637376    ,0.0, 0.0},
     {.9211802329530587850937534360831064254088    ,0.0, 0.0},
     {.8746378049201027904177934212565785469106    ,0.0, 0.0},
     {.81818548761525244498957221457878497563      ,0.0, 0.0},
     {.7524628517344771339126100772121381902126    ,0.0, 0.0},
     {.6782145376026865151561850053919859263883    ,0.0, 0.0},
     {.5962817971382278203795862111889897800782    ,0.0, 0.0},
     {.5075929551242276421026279196275201533512    ,0.0, 0.0},
     {.4131528881740086638907065860316168233234    ,0.0, 0.0},
     {.3140316378676399349481959231910474482584    ,0.0, 0.0},
     {.2113522861660010745063757289029374990567    ,0.0, 0.0},
     {.1062782301326792301709823924303769809172    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1062782301326792301709823924303769809172   ,0.0, 0.0},
     {-.2113522861660010745063757289029374990567   ,0.0, 0.0},
     {-.3140316378676399349481959231910474482584   ,0.0, 0.0},
     {-.4131528881740086638907065860316168233234   ,0.0, 0.0},
     {-.5075929551242276421026279196275201533512   ,0.0, 0.0},
     {-.5962817971382278203795862111889897800782   ,0.0, 0.0},
     {-.6782145376026865151561850053919859263883   ,0.0, 0.0},
     {-.7524628517344771339126100772121381902126   ,0.0, 0.0},
     {-.81818548761525244498957221457878497563     ,0.0, 0.0},
     {-.8746378049201027904177934212565785469106   ,0.0, 0.0},
     {-.9211802329530587850937534360831064254088   ,0.0, 0.0},
     {-.9572855957780877257982080369808235637376   ,0.0, 0.0},
     {-.9825455052614131748709260157863769561068   ,0.0, 0.0},
     {-.9966794422605965861631915325493538856518   ,0.0, 0.0}
    },
    { // weights
     .008516903878746409654263813302249803002399,
     .01973208505612270598385980164039563114961,
     .03074049220209362264440852537461674974712,
     .04140206251868283610483001011407692153349,
     .05159482690249792391259438117954259791962,
     .06120309065707913854210984802390704492406,
     .07011793325505127856958148694887917310239,
     .07823832713576378382814488865968033136688,
     .0854722573661725275453448492972080713817,
     .09173775713925876334796641107711080689823,
     .09696383409440860630190007488268875917636,
     .1010912737599149661218205469074973636476,
     .1040733100777293739133284712851200689107,
     .1058761550973209414065913278521878930748,
      .1064793817183142442465111269096775683302,
      .1058761550973209414065913278521878930748,
      .1040733100777293739133284712851200689107,
      .1010912737599149661218205469074973636476,
      .09696383409440860630190007488268875917636,
      .09173775713925876334796641107711080689823,
      .0854722573661725275453448492972080713817,
      .07823832713576378382814488865968033136688,
      .07011793325505127856958148694887917310239,
      .06120309065707913854210984802390704492406,
      .05159482690249792391259438117954259791962,
      .04140206251868283610483001011407692153349,
      .03074049220209362264440852537461674974712,
      .01973208505612270598385980164039563114961,
      .008516903878746409654263813302249803002399
    }
  },
  {
    29,
    { // points
     {.9966794422605965861631915325493538856518    ,0.0, 0.0},
     {.9825455052614131748709260157863769561068    ,0.0, 0.0},
     {.9572855957780877257982080369808235637376    ,0.0, 0.0},
     {.9211802329530587850937534360831064254088    ,0.0, 0.0},
     {.8746378049201027904177934212565785469106    ,0.0, 0.0},
     {.81818548761525244498957221457878497563      ,0.0, 0.0},
     {.7524628517344771339126100772121381902126    ,0.0, 0.0},
     {.6782145376026865151561850053919859263883    ,0.0, 0.0},
     {.5962817971382278203795862111889897800782    ,0.0, 0.0},
     {.5075929551242276421026279196275201533512    ,0.0, 0.0},
     {.4131528881740086638907065860316168233234    ,0.0, 0.0},
     {.3140316378676399349481959231910474482584    ,0.0, 0.0},
     {.2113522861660010745063757289029374990567    ,0.0, 0.0},
     {.1062782301326792301709823924303769809172    ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.1062782301326792301709823924303769809172   ,0.0, 0.0},
     {-.2113522861660010745063757289029374990567   ,0.0, 0.0},
     {-.3140316378676399349481959231910474482584   ,0.0, 0.0},
     {-.4131528881740086638907065860316168233234   ,0.0, 0.0},
     {-.5075929551242276421026279196275201533512   ,0.0, 0.0},
     {-.5962817971382278203795862111889897800782   ,0.0, 0.0},
     {-.6782145376026865151561850053919859263883   ,0.0, 0.0},
     {-.7524628517344771339126100772121381902126   ,0.0, 0.0},
     {-.81818548761525244498957221457878497563     ,0.0, 0.0},
     {-.8746378049201027904177934212565785469106   ,0.0, 0.0},
     {-.9211802329530587850937534360831064254088   ,0.0, 0.0},
     {-.9572855957780877257982080369808235637376   ,0.0, 0.0},
     {-.9825455052614131748709260157863769561068   ,0.0, 0.0},
     {-.9966794422605965861631915325493538856518   ,0.0, 0.0}
    },
    { // weights
     .008516903878746409654263813302249803002399,
     .01973208505612270598385980164039563114961,
     .03074049220209362264440852537461674974712,
     .04140206251868283610483001011407692153349,
     .05159482690249792391259438117954259791962,
     .06120309065707913854210984802390704492406,
     .07011793325505127856958148694887917310239,
     .07823832713576378382814488865968033136688,
     .0854722573661725275453448492972080713817,
     .09173775713925876334796641107711080689823,
     .09696383409440860630190007488268875917636,
     .1010912737599149661218205469074973636476,
     .1040733100777293739133284712851200689107,
     .1058761550973209414065913278521878930748,
      .1064793817183142442465111269096775683302,
      .1058761550973209414065913278521878930748,
      .1040733100777293739133284712851200689107,
      .1010912737599149661218205469074973636476,
      .09696383409440860630190007488268875917636,
      .09173775713925876334796641107711080689823,
      .0854722573661725275453448492972080713817,
      .07823832713576378382814488865968033136688,
      .07011793325505127856958148694887917310239,
      .06120309065707913854210984802390704492406,
      .05159482690249792391259438117954259791962,
      .04140206251868283610483001011407692153349,
      .03074049220209362264440852537461674974712,
      .01973208505612270598385980164039563114961,
      .008516903878746409654263813302249803002399
    }
  },
  {
    30,
    { // points
     {.9968934840746495402716300509186952833409    ,0.0, 0.0},
     {.9836681232797472099700325816056628019403    ,0.0, 0.0},
     {.9600218649683075122168710255817976629304    ,0.0, 0.0},
     {.9262000474292743258793242770804740040865    ,0.0, 0.0},
     {.8825605357920526815431164625302255900567    ,0.0, 0.0},
     {.8295657623827683974428981197325019164391    ,0.0, 0.0},
     {.7677774321048261949179773409745031316949    ,0.0, 0.0},
     {.6978504947933157969322923880266400683824    ,0.0, 0.0},
     {.6205261829892428611404775564311892992074    ,0.0, 0.0},
     {.5366241481420198992641697933110727941642    ,0.0, 0.0},
     {.4470337695380891767806099003228540001624    ,0.0, 0.0},
     {.3527047255308781134710372070893738606536    ,0.0, 0.0},
     {.2546369261678898464398051298178051078828    ,0.0, 0.0},
     {.1538699136085835469637946727432559204186    ,0.0, 0.0},
     {.05147184255531769583302521316672257374914   ,0.0, 0.0},
     {-.05147184255531769583302521316672257374914  ,0.0, 0.0},
     {-.1538699136085835469637946727432559204186   ,0.0, 0.0},
     {-.2546369261678898464398051298178051078828   ,0.0, 0.0},
     {-.3527047255308781134710372070893738606536   ,0.0, 0.0},
     {-.4470337695380891767806099003228540001624   ,0.0, 0.0},
     {-.5366241481420198992641697933110727941642   ,0.0, 0.0},
     {-.6205261829892428611404775564311892992074   ,0.0, 0.0},
     {-.6978504947933157969322923880266400683824   ,0.0, 0.0},
     {-.7677774321048261949179773409745031316949   ,0.0, 0.0},
     {-.8295657623827683974428981197325019164391   ,0.0, 0.0},
     {-.8825605357920526815431164625302255900567   ,0.0, 0.0},
     {-.9262000474292743258793242770804740040865   ,0.0, 0.0},
     {-.9600218649683075122168710255817976629304   ,0.0, 0.0},
     {-.9836681232797472099700325816056628019403   ,0.0, 0.0},
     {-.9968934840746495402716300509186952833409   ,0.0, 0.0}
    },
    { // weights
     .007968192496166605615465883474673622450481,
     .01846646831109095914230213191204726909621,
     .02878470788332336934971917961129204363959,
     .0387991925696270495968019364463476920332,
     .04840267283059405290293814042280751781527,
     .05749315621761906648172168940205612879712,
     .06597422988218049512812851511596236123744,
     .07375597473770520626824385002219073415377,
     .08075589522942021535469493846052973087589,
     .08689978720108297980238753071512570257675,
     .09212252223778612871763270708761876719691,
     .09636873717464425963946862635180986509641,
     .09959342058679526706278028210356947652987,
     .1017623897484055045964289521685540446327,
     .1028526528935588403412856367054150438684,
      .1028526528935588403412856367054150438684,
      .1017623897484055045964289521685540446327,
      .09959342058679526706278028210356947652987,
      .09636873717464425963946862635180986509641,
      .09212252223778612871763270708761876719691,
      .08689978720108297980238753071512570257675,
      .08075589522942021535469493846052973087589,
      .07375597473770520626824385002219073415377,
      .06597422988218049512812851511596236123744,
      .05749315621761906648172168940205612879712,
      .04840267283059405290293814042280751781527,
      .0387991925696270495968019364463476920332,
      .02878470788332336934971917961129204363959,
      .01846646831109095914230213191204726909621,
      .007968192496166605615465883474673622450481
    }
  },
  {
    30,
    { // points
     {.9968934840746495402716300509186952833409    ,0.0, 0.0},
     {.9836681232797472099700325816056628019403    ,0.0, 0.0},
     {.9600218649683075122168710255817976629304    ,0.0, 0.0},
     {.9262000474292743258793242770804740040865    ,0.0, 0.0},
     {.8825605357920526815431164625302255900567    ,0.0, 0.0},
     {.8295657623827683974428981197325019164391    ,0.0, 0.0},
     {.7677774321048261949179773409745031316949    ,0.0, 0.0},
     {.6978504947933157969322923880266400683824    ,0.0, 0.0},
     {.6205261829892428611404775564311892992074    ,0.0, 0.0},
     {.5366241481420198992641697933110727941642    ,0.0, 0.0},
     {.4470337695380891767806099003228540001624    ,0.0, 0.0},
     {.3527047255308781134710372070893738606536    ,0.0, 0.0},
     {.2546369261678898464398051298178051078828    ,0.0, 0.0},
     {.1538699136085835469637946727432559204186    ,0.0, 0.0},
     {.05147184255531769583302521316672257374914   ,0.0, 0.0},
     {-.05147184255531769583302521316672257374914  ,0.0, 0.0},
     {-.1538699136085835469637946727432559204186   ,0.0, 0.0},
     {-.2546369261678898464398051298178051078828   ,0.0, 0.0},
     {-.3527047255308781134710372070893738606536   ,0.0, 0.0},
     {-.4470337695380891767806099003228540001624   ,0.0, 0.0},
     {-.5366241481420198992641697933110727941642   ,0.0, 0.0},
     {-.6205261829892428611404775564311892992074   ,0.0, 0.0},
     {-.6978504947933157969322923880266400683824   ,0.0, 0.0},
     {-.7677774321048261949179773409745031316949   ,0.0, 0.0},
     {-.8295657623827683974428981197325019164391   ,0.0, 0.0},
     {-.8825605357920526815431164625302255900567   ,0.0, 0.0},
     {-.9262000474292743258793242770804740040865   ,0.0, 0.0},
     {-.9600218649683075122168710255817976629304   ,0.0, 0.0},
     {-.9836681232797472099700325816056628019403   ,0.0, 0.0},
     {-.9968934840746495402716300509186952833409   ,0.0, 0.0}
    },
    { // weights
     .007968192496166605615465883474673622450481,
     .01846646831109095914230213191204726909621,
     .02878470788332336934971917961129204363959,
     .0387991925696270495968019364463476920332,
     .04840267283059405290293814042280751781527,
     .05749315621761906648172168940205612879712,
     .06597422988218049512812851511596236123744,
     .07375597473770520626824385002219073415377,
     .08075589522942021535469493846052973087589,
     .08689978720108297980238753071512570257675,
     .09212252223778612871763270708761876719691,
     .09636873717464425963946862635180986509641,
     .09959342058679526706278028210356947652987,
     .1017623897484055045964289521685540446327,
     .1028526528935588403412856367054150438684,
      .1028526528935588403412856367054150438684,
      .1017623897484055045964289521685540446327,
      .09959342058679526706278028210356947652987,
      .09636873717464425963946862635180986509641,
      .09212252223778612871763270708761876719691,
      .08689978720108297980238753071512570257675,
      .08075589522942021535469493846052973087589,
      .07375597473770520626824385002219073415377,
      .06597422988218049512812851511596236123744,
      .05749315621761906648172168940205612879712,
      .04840267283059405290293814042280751781527,
      .0387991925696270495968019364463476920332,
      .02878470788332336934971917961129204363959,
      .01846646831109095914230213191204726909621,
      .007968192496166605615465883474673622450481
    }
  },
  {
    31,
    { // points
     {.9970874818194770740556265542231025082514    ,0.0, 0.0},
     {.9846859096651524840024651667346847109385    ,0.0, 0.0},
     {.9625039250929496617890524041058583018972    ,0.0, 0.0},
     {.9307569978966481649569457597292635128134    ,0.0, 0.0},
     {.8897600299482710433741920089821592615436    ,0.0, 0.0},
     {.8399203201462673400869045359401783560547    ,0.0, 0.0},
     {.7817331484166249404063600201946844910954    ,0.0, 0.0},
     {.7157767845868532839059708653664859558902    ,0.0, 0.0},
     {.6427067229242603461844182032325014539917    ,0.0, 0.0},
     {.5632491614071492627209449235951614279486    ,0.0, 0.0},
     {.4781937820449024804405940393564857484395    ,0.0, 0.0},
     {.3883859016082329430613514612875201019851    ,0.0, 0.0},
     {.2947180699817016166179038976717043387529    ,0.0, 0.0},
     {.1981211993355706287724129960328339377748    ,0.0, 0.0},
     {.099555312152341520325174790118940733839     ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.099555312152341520325174790118940733839    ,0.0, 0.0},
     {-.1981211993355706287724129960328339377748   ,0.0, 0.0},
     {-.2947180699817016166179038976717043387529   ,0.0, 0.0},
     {-.3883859016082329430613514612875201019851   ,0.0, 0.0},
     {-.4781937820449024804405940393564857484395   ,0.0, 0.0},
     {-.5632491614071492627209449235951614279486   ,0.0, 0.0},
     {-.6427067229242603461844182032325014539917   ,0.0, 0.0},
     {-.7157767845868532839059708653664859558902   ,0.0, 0.0},
     {-.7817331484166249404063600201946844910954   ,0.0, 0.0},
     {-.8399203201462673400869045359401783560547   ,0.0, 0.0},
     {-.8897600299482710433741920089821592615436   ,0.0, 0.0},
     {-.9307569978966481649569457597292635128134   ,0.0, 0.0},
     {-.9625039250929496617890524041058583018972   ,0.0, 0.0},
     {-.9846859096651524840024651667346847109385   ,0.0, 0.0},
     {-.9970874818194770740556265542231025082514   ,0.0, 0.0}
    },
    { // weights
     .007470831579248775858696875032212407022151,
     .01731862079031058246315799608682114136627,
     .02700901918497942180060870809193215614236,
     .03643227391238546402439201046780344232715,
     .04549370752720110290231585789462967726355,
     .05410308242491685371166625908662004272039,
     .06217478656102842691034354368669666715873,
     .06962858323541036616775612625512571051509,
     .07639038659877661642635767490133064492576,
     .08239299176158926390382336743196177061577,
     .08757674060847787612619806969533309222926,
     .09189011389364147821536287160715012549731,
     .09529024291231951280720419748759668454132,
     .09774333538632872509347401097899670383573,
     .09922501122667230787487551442861501401754,
     .09972054479342645142753383373434943964325,
     .09922501122667230787487551442861501401754,
      .09774333538632872509347401097899670383573,
      .09529024291231951280720419748759668454132,
      .09189011389364147821536287160715012549731,
      .08757674060847787612619806969533309222926,
      .08239299176158926390382336743196177061577,
      .07639038659877661642635767490133064492576,
      .06962858323541036616775612625512571051509,
      .06217478656102842691034354368669666715873,
      .05410308242491685371166625908662004272039,
      .04549370752720110290231585789462967726355,
      .03643227391238546402439201046780344232715,
      .02700901918497942180060870809193215614236,
      .01731862079031058246315799608682114136627,
      .007470831579248775858696875032212407022151
    }
  },
  {
    31,
    { // points
     {.9970874818194770740556265542231025082514    ,0.0, 0.0},
     {.9846859096651524840024651667346847109385    ,0.0, 0.0},
     {.9625039250929496617890524041058583018972    ,0.0, 0.0},
     {.9307569978966481649569457597292635128134    ,0.0, 0.0},
     {.8897600299482710433741920089821592615436    ,0.0, 0.0},
     {.8399203201462673400869045359401783560547    ,0.0, 0.0},
     {.7817331484166249404063600201946844910954    ,0.0, 0.0},
     {.7157767845868532839059708653664859558902    ,0.0, 0.0},
     {.6427067229242603461844182032325014539917    ,0.0, 0.0},
     {.5632491614071492627209449235951614279486    ,0.0, 0.0},
     {.4781937820449024804405940393564857484395    ,0.0, 0.0},
     {.3883859016082329430613514612875201019851    ,0.0, 0.0},
     {.2947180699817016166179038976717043387529    ,0.0, 0.0},
     {.1981211993355706287724129960328339377748    ,0.0, 0.0},
     {.099555312152341520325174790118940733839     ,0.0, 0.0},
     {0                                            ,0.0, 0.0},
     {-.099555312152341520325174790118940733839    ,0.0, 0.0},
     {-.1981211993355706287724129960328339377748   ,0.0, 0.0},
     {-.2947180699817016166179038976717043387529   ,0.0, 0.0},
     {-.3883859016082329430613514612875201019851   ,0.0, 0.0},
     {-.4781937820449024804405940393564857484395   ,0.0, 0.0},
     {-.5632491614071492627209449235951614279486   ,0.0, 0.0},
     {-.6427067229242603461844182032325014539917   ,0.0, 0.0},
     {-.7157767845868532839059708653664859558902   ,0.0, 0.0},
     {-.7817331484166249404063600201946844910954   ,0.0, 0.0},
     {-.8399203201462673400869045359401783560547   ,0.0, 0.0},
     {-.8897600299482710433741920089821592615436   ,0.0, 0.0},
     {-.9307569978966481649569457597292635128134   ,0.0, 0.0},
     {-.9625039250929496617890524041058583018972   ,0.0, 0.0},
     {-.9846859096651524840024651667346847109385   ,0.0, 0.0},
     {-.9970874818194770740556265542231025082514   ,0.0, 0.0}
    },
    { // weights
     .007470831579248775858696875032212407022151,
     .01731862079031058246315799608682114136627,
     .02700901918497942180060870809193215614236,
     .03643227391238546402439201046780344232715,
     .04549370752720110290231585789462967726355,
     .05410308242491685371166625908662004272039,
     .06217478656102842691034354368669666715873,
     .06962858323541036616775612625512571051509,
     .07639038659877661642635767490133064492576,
     .08239299176158926390382336743196177061577,
     .08757674060847787612619806969533309222926,
     .09189011389364147821536287160715012549731,
     .09529024291231951280720419748759668454132,
     .09774333538632872509347401097899670383573,
     .09922501122667230787487551442861501401754,
     .09972054479342645142753383373434943964325,
     .09922501122667230787487551442861501401754,
      .09774333538632872509347401097899670383573,
      .09529024291231951280720419748759668454132,
      .09189011389364147821536287160715012549731,
      .08757674060847787612619806969533309222926,
      .08239299176158926390382336743196177061577,
      .07639038659877661642635767490133064492576,
      .06962858323541036616775612625512571051509,
      .06217478656102842691034354368669666715873,
      .05410308242491685371166625908662004272039,
      .04549370752720110290231585789462967726355,
      .03643227391238546402439201046780344232715,
      .02700901918497942180060870809193215614236,
      .01731862079031058246315799608682114136627,
      .007470831579248775858696875032212407022151
    }
  } // end Gauss

};
    
} // end namespace Intrepid2
