/*
 * The source code contained herein is licensed under the IBM Public License
 * Version 1.0, which has been approved by the Open Source Initiative.
 * Copyright (C) 2001, International Business Machines Corporation
 * All Rights Reserved.
 *
 */

package org.uddi4j.datatype.binding;

import java.util.Vector;

import org.uddi4j.UDDIElement;
import org.uddi4j.UDDIException;
import org.uddi4j.datatype.Description;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Represents the tModelInstanceInfo element within the UDDI version 2.0 schema.
 * This class contains the following types of methods:
 * 
 * <ul>
 *   <li>Constructor passing required fields.
 *   <li>Constructor that will instantiate the object from an appropriate XML
 *       DOM element.
 *   <li>Get/set methods for each attribute that this element can contain.
 *   <li>A get/setVector method is provided for sets of attributes.
 *   <li>SaveToXML method. Serializes this class within a passed in element.
 * </ul>
 * 
 * Typically, this class is used to construct parameters for, or interpret
 * responses from methods in the UDDIProxy class.
 *
 * <p><b>Element description:</b>
 * <p>Support element used to contain implementation instance specific information
 * about compatible specifications (via tModel reference) and optional setting's
 * details.
 *
 * @author David Melgar (dmelgar@us.ibm.com)
 */
public class TModelInstanceInfo extends UDDIElement {

   public static final String UDDI_TAG = "tModelInstanceInfo";

   protected Element base = null;

   String tModelKey = null;
   InstanceDetails instanceDetails = null;
   // Vector of Description objects
   Vector description = new Vector();

   /**
    * Default constructor.
    * Avoid using the default constructor for validation. It does not validate
    * required fields. Instead, use the required fields constructor to perform
    * validation.
    */
   public TModelInstanceInfo() {
   }

   /**
    * Construct the object with required fields.
    *
    * @param tModelKey  String
    */
   public TModelInstanceInfo(String tModelKey) {
      this.tModelKey = tModelKey;
   }

   /**
    * Construct the object from a DOM tree. Used by
    * UDDIProxy to construct an object from a received UDDI
    * message.
    *
    * @param base   Element with the name appropriate for this class.
    *
    * @exception UDDIException Thrown if DOM tree contains a SOAP fault
    *  or a disposition report indicating a UDDI error.
    */
   public TModelInstanceInfo(Element base) throws UDDIException {
      // Check if it is a fault. Throw exception if it is.
      super(base);
      tModelKey = base.getAttribute("tModelKey");
      NodeList nl = null;
      nl = getChildElementsByTagName(base, InstanceDetails.UDDI_TAG);
      if (nl.getLength() > 0) {
         instanceDetails = new InstanceDetails((Element)nl.item(0));
      }
      nl = getChildElementsByTagName(base, Description.UDDI_TAG);
      for (int i=0; i < nl.getLength(); i++) {
         description.addElement(new Description((Element)nl.item(i)));
      }
   }

   public void setTModelKey(String s) {
      tModelKey = s;
   }

   public void setInstanceDetails(InstanceDetails s) {
      instanceDetails = s;
   }

   /**
    * Set description vector.
    *
    * @param s  Vector of <I>Description</I> objects.
    */
   public void setDescriptionVector(Vector s) {
      description = s;
   }

   /**
    * Set default (english) description string.
    *
    * @param s  String
    */
   public void setDefaultDescriptionString(String s) {
      if (description.size() > 0) {
         description.setElementAt(new Description(s), 0);
      } else {
         description.addElement(new Description(s));
      }
   }

   public String getTModelKey() {
      return tModelKey;
   }


   public InstanceDetails getInstanceDetails() {
      return instanceDetails;
   }


   /**
    * Get description.
    *
    * @return s Vector of <I>Description</I> objects.
    */
   public Vector getDescriptionVector() {
      return description;
   }

   /**
    * Get default description string.
    *
    * @return s String
    */
   public String getDefaultDescriptionString() {
      if ((description).size() > 0) {
         Description t = (Description)description.elementAt(0);
         return t.getText();
      } else {
         return null;
      }
   }

   /**
    * Save an object to the DOM tree. Used to serialize an object
    * to a DOM tree, usually to send a UDDI message.
    *
    * <BR>Used by UDDIProxy.
    *
    * @param parent Object will serialize as a child element under the
    *  passed in parent element.
    */
   public void saveToXML(Element parent) {
      base = parent.getOwnerDocument().createElementNS(UDDIElement.XMLNS, UDDIElement.XMLNS_PREFIX + UDDI_TAG);
      // Save attributes
      if (tModelKey!=null) {
         base.setAttribute("tModelKey", tModelKey);
      }
      if (description!=null) {
        for (int i=0; i < description.size(); i++) {
           ((Description)(description.elementAt(i))).saveToXML(base);
        }
      }
      if (instanceDetails!=null) {
         instanceDetails.saveToXML(base);
      }
      parent.appendChild(base);
   }
}
