/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qzionrectangle.h"

#include <QDebug>
#include <QPainter>

#include "qzionobject_p.h"
#include "qzionrectangle_p.h"


QZionRectangle::QZionRectangle(QZionAbstractCanvas *canvas)
    : QZionObject(canvas), _clipReferences(0)
{
    QZINIT(QZionRectangle, canvas);
}


QZionRectangle::QZionRectangle(QZionAbstractCanvas *canvas,
                               const QColor &color, const QSize &size)
    : QZionObject(canvas), _clipReferences(0)
{
    QZINIT(QZionRectangle, canvas);

    setColor(color);
    setSize(size);
}

QZionRectangle::~QZionRectangle()
{
    QZD(QZionRectangle);

    // Update clippees
    foreach (QZionObject *clippee, d->clippees) {
        clippee->setClipper(NULL);
    }

    QZDELETE(QZionRectangle);
}

QColor QZionRectangle::color() const
{
    QZD(QZionRectangle);
    return d->color;
}


void QZionRectangle::setColor(const QColor &color)
{
    QZD(QZionRectangle);

    if (this->color().rgb() != color.rgb())
        d->color = color.rgb();

    if (opacity() != color.alpha())
        setOpacity(color.alpha());
    else
        changed();
}


QSize QZionRectangle::size() const
{
    QZD(QZionRectangle);
    return d->rect().size().toSize();
}


void QZionRectangle::setSize(const QSize &size)
{
    QZD(QZionRectangle);
    QRectF orig = d->rect();
    d->setRect(orig.x(), orig.y(), size.width(), size.height());

    QZionObject::setSize(size);
}

QRect QZionRectangle::rect() const
{
    QZD(QZionRectangle);
    return d->rect().toRect().translated(pos());
}

void QZionRectangle::paint(QPainter * /*painter*/)
{

}

void QZionRectangle::addClippee(QZionObject* clippee)
{
    QZD(QZionRectangle);

    /* Rect became a clipper itself */
    if (d->clippees.isEmpty())
        changed();

    d->clippees.insert(clippee);
}

void QZionRectangle::removeClippee(QZionObject* clippee)
{
    QZD(QZionRectangle);

    d->clippees.remove(clippee);

    /* Rect used to be a clipper */
    if (d->clippees.isEmpty())
        changed();
}

QColor QZionRectangle::effectiveColor() const
{
    QZD(QZionRectangle);

    if (d->colorDirty)
        return QZionObject::effectiveColor();
    else
        return d->brush().color();
}

QRect QZionRectangle::effectiveRect() const
{
    QZD(QZionRectangle);

    if (d->rectDirty)
        return QZionObject::effectiveRect();
    else
        return d->effectiveRect;
}

void QZionRectangle::changed()
{
    QZD(QZionRectangle);

    d->updateEffectiveColor();
    d->updateEffectiveRect();
    if (_clipReferences)
        d->updateChildren();

    QZionObject::changed();
}
