/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Invertlev     invertlev       Invert level
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"

static void
invertLevDes(int vlistID)
{
  const auto nzaxis = vlistNzaxis(vlistID);
  for (int index = 0; index < nzaxis; index++)
    {
      const auto zaxisID1 = vlistZaxis(vlistID, index);
      const auto zaxisID2 = zaxisDuplicate(zaxisID1);
      const auto zaxistype = zaxisInqType(zaxisID1);

      const auto nlev = zaxisInqSize(zaxisID1);
      if (nlev <= 1) continue;

      if (zaxisInqLevels(zaxisID1, nullptr))
        {
          Varray<double> yv1(nlev);
          Varray<double> yv2(nlev);
          zaxisInqLevels(zaxisID1, yv1.data());
          for (int ilev = 0; ilev < nlev; ++ilev) yv2[nlev - ilev - 1] = yv1[ilev];
          zaxisDefLevels(zaxisID2, yv2.data());
        }

      if (zaxisInqLbounds(zaxisID1, nullptr) && zaxisInqUbounds(zaxisID1, nullptr))
        {
          Varray<double> yb1(nlev);
          Varray<double> yb2(nlev);
          zaxisInqLbounds(zaxisID1, yb1.data());
          for (int ilev = 0; ilev < nlev; ++ilev) yb2[nlev - ilev - 1] = yb1[ilev];
          zaxisDefLbounds(zaxisID2, yb2.data());

          zaxisInqUbounds(zaxisID1, yb1.data());
          for (int ilev = 0; ilev < nlev; ++ilev) yb2[nlev - ilev - 1] = yb1[ilev];
          zaxisDefUbounds(zaxisID2, yb2.data());
        }

      if (zaxistype == ZAXIS_HYBRID || zaxistype == ZAXIS_HYBRID_HALF)
        {
          const int vctsize = zaxisInqVctSize(zaxisID1);
          if (vctsize && vctsize % 2 == 0)
            {
              Varray<double> vct1(vctsize);
              Varray<double> vct2(vctsize);
              zaxisInqVct(zaxisID1, vct1.data());
              for (int i = 0; i < vctsize / 2; ++i)
                {
                  vct2[vctsize / 2 - 1 - i] = vct1[i];
                  vct2[vctsize - 1 - i] = vct1[vctsize / 2 + i];
                }
              zaxisDefVct(zaxisID2, vctsize, vct2.data());
            }
        }

      vlistChangeZaxis(vlistID, zaxisID1, zaxisID2);
    }
}

void *
Invertlev(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;
  int nlev, nlevel;
  int gridID, zaxisID;
  bool linvert = false;

  cdoInitialize(process);

  const auto lcopy = unchangedRecord();

  cdoOperatorAdd("invertlev", func_all, 0, nullptr);

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  if (operfunc == func_all || operfunc == func_hrd) invertLevDes(vlistID2);

  const auto streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> array(gridsizemax);

  auto nvars = vlistNvars(vlistID1);

  Varray2D<double> vardata(nvars);
  std::vector<std::vector<size_t>> varnmiss(nvars);

  for (varID = 0; varID < nvars; varID++)
    {
      gridID = vlistInqVarGrid(vlistID1, varID);
      zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const auto gridsize = gridInqSize(gridID);
      nlev = zaxisInqSize(zaxisID);

      if (nlev > 1)
        {
          linvert = true;
          vardata[varID].resize(gridsize * nlev);
          varnmiss[varID].resize(nlev);
        }
    }

  if (!linvert) cdoWarning("No variables with invertable levels found!");

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);

      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (vardata[varID].size())
            {
              gridID = vlistInqVarGrid(vlistID1, varID);
              const auto gridsize = gridInqSize(gridID);
              const auto offset = gridsize * levelID;

              cdoReadRecord(streamID1, &vardata[varID][offset], &nmiss);
              varnmiss[varID][levelID] = nmiss;
            }
          else
            {
              cdoDefRecord(streamID2, varID, levelID);
              if (lcopy)
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  cdoReadRecord(streamID1, array.data(), &nmiss);
                  cdoWriteRecord(streamID2, array.data(), nmiss);
                }
            }
        }

      for (varID = 0; varID < nvars; varID++)
        {
          if (vardata[varID].size())
            {
              gridID = vlistInqVarGrid(vlistID1, varID);
              zaxisID = vlistInqVarZaxis(vlistID1, varID);
              const auto gridsize = gridInqSize(gridID);
              nlevel = zaxisInqSize(zaxisID);
              for (levelID = 0; levelID < nlevel; levelID++)
                {
                  cdoDefRecord(streamID2, varID, levelID);

                  auto offset = gridsize * (nlevel - levelID - 1);
                  nmiss = varnmiss[varID][nlevel - levelID - 1];

                  cdoWriteRecord(streamID2, &vardata[varID][offset], nmiss);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
