/*
 * Copyright (c) 2009-2016, Albertas Vyšniauskas
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *     * Neither the name of the software author nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "GenerateScheme.h"
#include "ColorObject.h"
#include "ColorSourceManager.h"
#include "ColorSource.h"
#include "DragDrop.h"
#include "GlobalState.h"
#include "ToolColorNaming.h"
#include "uiUtilities.h"
#include "MathUtil.h"
#include "ColorRYB.h"
#include "ColorList.h"
#include "gtk/ColorWidget.h"
#include "gtk/ColorWheel.h"
#include "ColorWheelType.h"
#include "Converters.h"
#include "dynv/Map.h"
#include "I18N.h"
#include "Random.h"
#include "StandardEventHandler.h"
#include "IMenuExtension.h"
#include "color_names/ColorNames.h"
#include <gdk/gdkkeysyms.h>
#include <sstream>

const int MaxColors = 6;
const SchemeType types[] = {
	{ N_("Complementary"), 1, 1, { 180 } },
	{ N_("Analogous"), 5, 1, { 30 } },
	{ N_("Triadic"), 2, 1, { 120 } },
	{ N_("Split-Complementary"), 2, 2, { 150, 60 } },
	{ N_("Rectangle (tetradic)"), 3, 2, { 60, 120 } },
	{ N_("Square"), 3, 1, { 90 } },
	{ N_("Neutral"), 5, 1, { 15 } },
	{ N_("Clash"), 2, 2, { 90, 180 } },
	{ N_("Five-Tone"), 4, 4, { 115, 40, 50, 40 } },
	{ N_("Six-Tone"), 5, 2, { 30, 90 } },
};
struct GenerateSchemeColorNameAssigner: public ToolColorNameAssigner {
	GenerateSchemeColorNameAssigner(GlobalState *gs):
		ToolColorNameAssigner(gs) {
	}
	void assign(ColorObject *colorObject, const Color *color, int ident, int schemeType) {
		m_ident = ident;
		m_schemeType = schemeType;
		ToolColorNameAssigner::assign(colorObject, color);
	}
	virtual std::string getToolSpecificName(ColorObject *colorObject, const Color *color) {
		m_stream.str("");
		m_stream << _("scheme") << " " << _(generate_scheme_get_scheme_type(m_schemeType)->name) << " #" << m_ident << "[" << color_names_get(m_gs->getColorNames(), color, false) << "]";
		return m_stream.str();
	}
protected:
	std::stringstream m_stream;
	int m_ident;
	int m_schemeType;
};
struct GenerateSchemeArgs {
	ColorSource source;
	GtkWidget *main, *statusBar, *generationType, *wheelTypeCombo, *colorWheel, *hueRange, *saturationRange, *lightnessRange, *colorPreviews, *lastFocusedColor;
	struct {
		GtkWidget *widget;
		float colorHue, hueShift, saturationShift, valueShift, originalHue, originalSaturation, originalValue;
	} items[MaxColors];
	bool wheelLocked;
	int colorsVisible;
	dynv::Ref options;
	GlobalState *gs;
	void addToPalette() {
		color_list_add_color_object(gs->getColorList(), getColor(), true);
	}
	void addToPalette(GenerateSchemeColorNameAssigner &nameAssigner, Color &color, GtkWidget *widget) {
		gtk_color_get_color(GTK_COLOR(widget), &color);
		colorObject.setColor(color);
		int type = gtk_combo_box_get_active(GTK_COMBO_BOX(generationType));
		nameAssigner.assign(&colorObject, &color, identifyColorWidget(widget), type);
		color_list_add_color_object(gs->getColorList(), colorObject, true);
	}
	void addAllToPalette() {
		GenerateSchemeColorNameAssigner nameAssigner(gs);
		Color color;
		for (int i = 0; i < colorsVisible; ++i)
			addToPalette(nameAssigner, color, items[i].widget);
	}
	void setColor(const ColorObject &colorObject) {
		int index = 0;
		for (int i = 0; i < colorsVisible; ++i) {
			if (items[i].widget == lastFocusedColor) {
				index = i;
				break;
			}
		}
		Color color = colorObject.getColor();
		float hue, saturation, lightness, shiftedHue;
		Color hsl, hsv, hslResult;
		color_rgb_to_hsv(&color, &hsv);
		color_hsv_to_hsl(&hsv, &hsl);
		int wheelType = gtk_combo_box_get_active(GTK_COMBO_BOX(wheelTypeCombo));
		auto &wheel = color_wheel_types_get()[wheelType];
		double tmp;
		wheel.rgbhue_to_hue(hsl.hsl.hue, &tmp);
		hue = static_cast<float>(tmp);
		shiftedHue = wrap_float(hue - items[index].colorHue - items[index].hueShift);
		wheel.hue_to_hsl(hue, &hslResult);
		saturation = hsl.hsl.saturation * 1 / hslResult.hsl.saturation;
		lightness = hsl.hsl.lightness - hslResult.hsl.lightness;
		shiftedHue *= 360.0f;
		saturation *= 100.0f;
		lightness *= 100.0f;
		gtk_range_set_value(GTK_RANGE(hueRange), shiftedHue);
		gtk_range_set_value(GTK_RANGE(saturationRange), saturation);
		gtk_range_set_value(GTK_RANGE(lightnessRange), lightness);
		update();
	}
	ColorObject colorObject;
	const ColorObject &getColor() {
		Color color;
		gtk_color_get_color(GTK_COLOR(lastFocusedColor), &color);
		colorObject.setColor(color);
		GenerateSchemeColorNameAssigner nameAssigner(gs);
		int type = gtk_combo_box_get_active(GTK_COMBO_BOX(generationType));
		nameAssigner.assign(&colorObject, &color, identifyColorWidget(lastFocusedColor), type);
		return colorObject;
	}
	int identifyColorWidget(GtkWidget *widget) {
		for (int i = 0; i < MaxColors; ++i) {
			if (items[i].widget == widget) {
				return i + 1;
			}
		}
		return 0;
	}
	static gboolean onFocusEvent(GtkWidget *widget, GdkEventFocus *, GenerateSchemeArgs *args) {
		args->setActiveWidget(widget);
		return false;
	}
	static gchar *onSaturationFormat(GtkScale *scale, gdouble value) {
		return g_strdup_printf("%d%%", int(value));
	}
	static gchar *onLightnessFormat(GtkScale *scale, gdouble value) {
		if (value >= 0)
			return g_strdup_printf("+%d%%", int(value));
		else
			return g_strdup_printf("-%d%%", -int(value));
	}
	static void onLockToggle(GtkWidget *widget, GenerateSchemeArgs *args) {
		args->wheelLocked = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget));
		gtk_color_wheel_set_block_editable(GTK_COLOR_WHEEL(args->colorWheel), !args->wheelLocked);
	}
	void setActiveWidget(GtkWidget *widget) {
		lastFocusedColor = widget;
		for (int i = 0; i < MaxColors; ++i) {
			if (widget == items[i].widget) {
				gtk_color_wheel_set_selected(GTK_COLOR_WHEEL(colorWheel), i);
				break;
			}
		}
	}
	static void onReset(GtkWidget *, GenerateSchemeArgs *args) {
		for (int i = 0; i < args->colorsVisible; ++i) {
			if (args->items[i].widget == args->lastFocusedColor) {
				args->items[i].hueShift = 0;
				args->items[i].saturationShift = 0;
				args->items[i].valueShift = 0;
				args->update();
				break;
			}
		}
	}
	static void onResetAll(GtkWidget *, GenerateSchemeArgs *args) {
		for (int i = 0; i < MaxColors; ++i) {
			args->items[i].hueShift = 0;
			args->items[i].saturationShift = 0;
			args->items[i].valueShift = 0;
		}
		args->update();
	}
	static void onColorActivate(GtkWidget *, GenerateSchemeArgs *args) {
		args->addToPalette();
	}
	static void onHueChange(GtkWidget *widget, gint colorId, GenerateSchemeArgs *args) {
		if (args->wheelLocked) {
			float hue = static_cast<float>(gtk_range_get_value(GTK_RANGE(args->hueRange)) / 360.0f);
			hue = wrap_float(hue - args->items[colorId].hueShift + static_cast<float>(gtk_color_wheel_get_hue(GTK_COLOR_WHEEL(widget), colorId)) - args->items[colorId].originalHue);
			gtk_range_set_value(GTK_RANGE(args->hueRange), hue * 360.0f);
		} else {
			args->items[colorId].hueShift = static_cast<float>(gtk_color_wheel_get_hue(GTK_COLOR_WHEEL(widget), colorId)) - args->items[colorId].originalHue;
			onChange(widget, args);
		}
	}
	static void onSaturationChange(GtkWidget *widget, gint colorId, GenerateSchemeArgs *args) {
		if (args->wheelLocked) {
			double saturation = static_cast<float>(gtk_range_get_value(GTK_RANGE(args->saturationRange)) / 100.0f);
			double lightness = static_cast<float>(gtk_range_get_value(GTK_RANGE(args->lightnessRange)) / 100.0f);
			gtk_range_set_value(GTK_RANGE(args->saturationRange), saturation * 100.0f);
			gtk_range_set_value(GTK_RANGE(args->lightnessRange), lightness * 100.0f);
		} else {
			args->items[colorId].saturationShift = static_cast<float>(gtk_color_wheel_get_saturation(GTK_COLOR_WHEEL(widget), colorId)) - args->items[colorId].originalSaturation;
			args->items[colorId].valueShift = static_cast<float>(gtk_color_wheel_get_value(GTK_COLOR_WHEEL(widget), colorId)) - args->items[colorId].originalValue;
			onChange(widget, args);
		}
	}
	static void onChange(GtkWidget *, GenerateSchemeArgs *args) {
		args->update();
	}
	void update(bool saveSettings = false) {
		int type = gtk_combo_box_get_active(GTK_COMBO_BOX(generationType));
		int colorCount = generate_scheme_get_scheme_type(type)->colors + 1;
		int wheelType = gtk_combo_box_get_active(GTK_COMBO_BOX(wheelTypeCombo));
		gtk_color_wheel_set_color_wheel_type(GTK_COLOR_WHEEL(colorWheel), &color_wheel_types_get()[wheelType]);
		gtk_color_wheel_set_n_colors(GTK_COLOR_WHEEL(colorWheel), colorCount);
		float hue = static_cast<float>(gtk_range_get_value(GTK_RANGE(hueRange)));
		float saturation = static_cast<float>(gtk_range_get_value(GTK_RANGE(saturationRange)));
		float lightness = static_cast<float>(gtk_range_get_value(GTK_RANGE(lightnessRange)));
		if (saveSettings) {
			options->set("type", type);
			options->set("wheel_type", wheelType);
			options->set("hue", hue);
			options->set("saturation", saturation);
			options->set("lightness", lightness);
		}
		for (int i = colorsVisible; i > colorCount; --i)
			gtk_widget_hide(items[i - 1].widget);
		for (int i = colorsVisible; i < colorCount; ++i)
			gtk_widget_show(items[i].widget);
		colorsVisible = colorCount;
		hue /= 360.0f;
		saturation /= 100.0f;
		lightness /= 100.0f;
		Color hsl, r;
		auto &wheel = color_wheel_types_get()[wheelType];
		float chaos = 0;
		float hueOffset = 0;
		float hueStep;
		Color hsv;
		for (int i = 0; i < colorCount; ++i) {
			wheel.hue_to_hsl(wrap_float(hue + items[i].hueShift), &hsl);
			hsl.hsl.lightness = clamp_float(hsl.hsl.lightness + lightness, 0, 1);
			hsl.hsl.saturation = clamp_float(hsl.hsl.saturation * saturation, 0, 1);
			color_hsl_to_hsv(&hsl, &hsv);
			items[i].originalHue = hue;
			items[i].originalSaturation = hsv.hsv.saturation;
			items[i].originalValue = hsv.hsv.value;
			hsv.hsv.saturation = clamp_float(hsv.hsv.saturation + items[i].saturationShift, 0, 1);
			hsv.hsv.value = clamp_float(hsv.hsv.value + items[i].valueShift, 0, 1);
			color_hsv_to_rgb(&hsv, &r);
			auto text = gs->converters().serialize(r, Converters::Type::display);
			gtk_color_set_color(GTK_COLOR(items[i].widget), r, text);
			items[i].colorHue = hueOffset;
			gtk_color_wheel_set_hue(GTK_COLOR_WHEEL(colorWheel), i, wrap_float(hue + items[i].hueShift));
			gtk_color_wheel_set_saturation(GTK_COLOR_WHEEL(colorWheel), i, hsv.hsv.saturation);
			gtk_color_wheel_set_value(GTK_COLOR_WHEEL(colorWheel), i, hsv.hsv.value);
			hueStep = (generate_scheme_get_scheme_type(type)->turn[i % generate_scheme_get_scheme_type(type)->turn_types]) / (360.0f) + chaos * static_cast<float>(random_get_double(gs->getRandom()) - 0.5);
			hue = wrap_float(hue + hueStep);
			hueOffset = wrap_float(hueOffset + hueStep);
		}
	}
	struct Editable: IEditableColorsUI, IMenuExtension {
		Editable(GenerateSchemeArgs *args):
			args(args) {
		}
		virtual ~Editable() = default;
		virtual void addToPalette(const ColorObject &) override {
			args->addToPalette();
		}
		virtual void addAllToPalette() override {
			args->addAllToPalette();
		}
		virtual void setColor(const ColorObject &colorObject) override {
			args->setColor(colorObject.getColor());
		}
		virtual const ColorObject &getColor() override {
			return args->getColor();
		}
		virtual std::vector<ColorObject> getColors(bool selected) override {
			std::vector<ColorObject> colors;
			colors.push_back(getColor());
			return colors;
		}
		virtual bool isEditable() override {
			return true;
		}
		virtual bool hasColor() override {
			return true;
		}
		virtual bool hasSelectedColor() override {
			return true;
		}
		virtual void extendMenu(GtkWidget *menu, Position position) {
			if (position != Position::end)
				return;
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());
			auto item = newMenuItem(_("_Reset"), GTK_STOCK_CANCEL);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
			g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(GenerateSchemeArgs::onReset), args);
			item = newMenuItem(_("_Reset scheme"), GTK_STOCK_CANCEL);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
			g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(GenerateSchemeArgs::onResetAll), args);
		}
	private:
		GenerateSchemeArgs *args;
	};
	boost::optional<Editable> editable;
};
static void showMenu(GtkWidget *widget, GenerateSchemeArgs *args, GdkEventButton *event) {
	auto menu = gtk_menu_new();
	auto item = gtk_check_menu_item_new_with_mnemonic(_("_Linked"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), args->wheelLocked);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	g_signal_connect(G_OBJECT(item), "toggled", G_CALLBACK(GenerateSchemeArgs::onLockToggle), args);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());
	item = newMenuItem(_("_Reset scheme"), GTK_STOCK_CANCEL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(GenerateSchemeArgs::onResetAll), args);
	showContextMenu(menu, event);
}
static gboolean onButtonPress(GtkWidget *widget, GdkEventButton *event, GenerateSchemeArgs *args) {
	if (event->button == 3 && event->type == GDK_BUTTON_PRESS) {
		showMenu(widget, args, event);
	}
	return false;
}
static void onPopupMenu(GtkWidget *widget, GenerateSchemeArgs *args) {
	showMenu(widget, args, nullptr);
}
static int destroy(GenerateSchemeArgs *args) {
	gtk_widget_destroy(args->main);
	delete args;
	return 0;
}
static int getColor(GenerateSchemeArgs *args, ColorObject **color) {
	auto colorObject = args->getColor();
	*color = colorObject.copy();
	return 0;
}
static int setColor(GenerateSchemeArgs *args, ColorObject *colorObject) {
	args->setColor(*colorObject);
	return 0;
}
static int setNthColor(GenerateSchemeArgs *args, size_t index, ColorObject *colorObject) {
	if (index < 0 || index >= MaxColors)
		return -1;
	args->setActiveWidget(args->items[index].widget);
	args->setColor(*colorObject);
	return 0;
}
static int activate(GenerateSchemeArgs *args) {
	auto chain = args->gs->getTransformationChain();
	for (int i = 0; i < MaxColors; ++i) {
		gtk_color_set_transformation_chain(GTK_COLOR(args->items[i].widget), chain);
	}
	gtk_statusbar_push(GTK_STATUSBAR(args->statusBar), gtk_statusbar_get_context_id(GTK_STATUSBAR(args->statusBar), "empty"), "");
	return 0;
}
static int deactivate(GenerateSchemeArgs *args) {
	args->update(true);
	args->options->set("wheel_locked", args->wheelLocked);
	float hsvShifts[MaxColors * 3];
	for (uint32_t i = 0; i < MaxColors; ++i) {
		hsvShifts[i * 3 + 0] = args->items[i].hueShift;
		hsvShifts[i * 3 + 1] = args->items[i].saturationShift;
		hsvShifts[i * 3 + 2] = args->items[i].valueShift;
	}
	args->options->set("hsv_shift", common::Span<float>(hsvShifts, MaxColors * 3));
	return 0;
}
static ColorObject *getColorObject(struct DragDrop *dd) {
	auto *args = (GenerateSchemeArgs *)dd->userdata;
	return args->getColor().copy();
}
static int setColorObjectAt(struct DragDrop *dd, ColorObject *colorObject, int, int, bool, bool) {
	auto *args = static_cast<GenerateSchemeArgs *>(dd->userdata);
	args->setActiveWidget(dd->widget);
	args->setColor(*colorObject);
	return 0;
}
static int setColorObjectAtColorWheel(struct DragDrop *dd, ColorObject *colorObject, int x, int y, bool, bool) {
	int index = gtk_color_wheel_get_at(GTK_COLOR_WHEEL(dd->widget), x, y);
	if (!(index >= 0 && index < MaxColors))
		return -1;
	auto *args = static_cast<GenerateSchemeArgs *>(dd->userdata);
	Color color, hsl;
	float hue;
	color = colorObject->getColor();
	color_rgb_to_hsl(&color, &hsl);
	int wheelType = gtk_combo_box_get_active(GTK_COMBO_BOX(args->wheelTypeCombo));
	auto &wheel = color_wheel_types_get()[wheelType];
	double tmp;
	wheel.rgbhue_to_hue(hsl.hsl.hue, &tmp);
	hue = static_cast<float>(tmp);
	if (args->wheelLocked) {
		float hueShift = (hue - args->items[index].originalHue) - args->items[index].hueShift;
		hue = wrap_float(static_cast<float>(gtk_range_get_value(GTK_RANGE(args->hueRange))) / 360.0f + hueShift);
		gtk_range_set_value(GTK_RANGE(args->hueRange), hue * 360.0f);
	} else {
		args->items[index].hueShift = hue - args->items[index].originalHue;
		args->update();
	}
	return 0;
}
static bool testAtColorWheel(struct DragDrop *dd, int x, int y) {
	int index = gtk_color_wheel_get_at(GTK_COLOR_WHEEL(dd->widget), x, y);
	return index >= 0 && index < MaxColors;
}
static ColorSource *source_implement(ColorSource *source, GlobalState *gs, const dynv::Ref &options) {
	auto *args = new GenerateSchemeArgs;
	args->editable = GenerateSchemeArgs::Editable(args);
	args->options = options;
	args->statusBar = gs->getStatusBar();
	args->gs = gs;
	color_source_init(&args->source, source->identificator, source->hr_name);
	args->source.destroy = (int (*)(ColorSource *))destroy;
	args->source.get_color = (int (*)(ColorSource *, ColorObject **))getColor;
	args->source.set_color = (int (*)(ColorSource *, ColorObject *))setColor;
	args->source.set_nth_color = (int (*)(ColorSource *, size_t, ColorObject *))setNthColor;
	args->source.deactivate = (int (*)(ColorSource *))deactivate;
	args->source.activate = (int (*)(ColorSource *))activate;
	auto hsvShifts = args->options->getFloats("hsv_shift");
	auto hsvShiftCount = hsvShifts.size() / 3;
	for (uint32_t i = 0; i < MaxColors; ++i) {
		if (i < hsvShiftCount) {
			args->items[i].hueShift = hsvShifts[i * 3 + 0];
			args->items[i].saturationShift = hsvShifts[i * 3 + 1];
			args->items[i].valueShift = hsvShifts[i * 3 + 2];
		} else {
			args->items[i].hueShift = 0;
			args->items[i].saturationShift = 0;
			args->items[i].valueShift = 0;
		}
	}
	GtkWidget *table, *vbox, *hbox, *widget, *hbox2;
	hbox = gtk_hbox_new(false, 5);
	vbox = gtk_vbox_new(false, 5);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, true, true, 5);
	args->colorPreviews = gtk_table_new(3, 2, false);
	gtk_box_pack_start(GTK_BOX(vbox), args->colorPreviews, true, true, 0);
	struct DragDrop dd;
	dragdrop_init(&dd, gs);
	dd.converterType = Converters::Type::display;
	dd.userdata = args;
	dd.get_color_object = getColorObject;
	dd.set_color_object_at = setColorObjectAt;
	for (intptr_t i = 0; i < MaxColors; ++i) {
		widget = gtk_color_new();
		gtk_color_set_rounded(GTK_COLOR(widget), true);
		gtk_color_set_roundness(GTK_COLOR(widget), 5);
		gtk_color_set_hcenter(GTK_COLOR(widget), true);
		gtk_table_attach(GTK_TABLE(args->colorPreviews), widget, i % 2, (i % 2) + 1, i / 2, i / 2 + 1, GtkAttachOptions(GTK_FILL | GTK_EXPAND), GtkAttachOptions(GTK_FILL | GTK_EXPAND), 0, 0);
		args->items[i].widget = widget;
		g_signal_connect(G_OBJECT(widget), "activated", G_CALLBACK(GenerateSchemeArgs::onColorActivate), args);
		g_signal_connect(G_OBJECT(widget), "focus-in-event", G_CALLBACK(GenerateSchemeArgs::onFocusEvent), args);
		StandardEventHandler::forWidget(widget, args->gs, &*args->editable);
		//setup drag&drop
		gtk_drag_dest_set(widget, GtkDestDefaults(GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT), 0, 0, GDK_ACTION_COPY);
		gtk_drag_source_set(widget, GDK_BUTTON1_MASK, 0, 0, GDK_ACTION_COPY);
		dragdrop_widget_attach(widget, DragDropFlags(DRAGDROP_SOURCE | DRAGDROP_DESTINATION), &dd);
	}
	hbox2 = gtk_hbox_new(false, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox2, false, false, 0);
	args->colorWheel = gtk_color_wheel_new();
	gtk_box_pack_start(GTK_BOX(hbox2), args->colorWheel, false, false, 0);
	g_signal_connect(G_OBJECT(args->colorWheel), "hue_changed", G_CALLBACK(GenerateSchemeArgs::onHueChange), args);
	g_signal_connect(G_OBJECT(args->colorWheel), "saturation_value_changed", G_CALLBACK(GenerateSchemeArgs::onSaturationChange), args);
	g_signal_connect(G_OBJECT(args->colorWheel), "popup-menu", G_CALLBACK(onPopupMenu), args);
	g_signal_connect(G_OBJECT(args->colorWheel), "button-press-event", G_CALLBACK(onButtonPress), args);
	args->wheelLocked = options->getBool("wheel_locked", true);
	gtk_color_wheel_set_block_editable(GTK_COLOR_WHEEL(args->colorWheel), !args->wheelLocked);
	gtk_drag_dest_set(args->colorWheel, GtkDestDefaults(GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT), 0, 0, GDK_ACTION_COPY);
	dd.userdata = args;
	dd.set_color_object_at = setColorObjectAtColorWheel;
	dd.test_at = testAtColorWheel;
	dragdrop_widget_attach(args->colorWheel, DragDropFlags(DRAGDROP_DESTINATION), &dd);
	gint table_y;
	table = gtk_table_new(5, 2, false);
	gtk_box_pack_start(GTK_BOX(hbox2), table, true, true, 0);
	table_y = 0;
	gtk_table_attach(GTK_TABLE(table), gtk_label_aligned_new(_("Hue:"), 0, 0.5, 0, 0), 0, 1, table_y, table_y + 1, GtkAttachOptions(GTK_FILL), GTK_FILL, 5, 5);
	args->hueRange = widget = gtk_hscale_new_with_range(0, 360, 1);
	gtk_range_set_value(GTK_RANGE(widget), options->getFloat("hue", 180));
	g_signal_connect(G_OBJECT(widget), "value-changed", G_CALLBACK(GenerateSchemeArgs::onChange), args);
	gtk_table_attach(GTK_TABLE(table), widget, 1, 2, table_y, table_y + 1, GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_FILL, 5, 0);
	table_y++;
	gtk_table_attach(GTK_TABLE(table), gtk_label_aligned_new(_("Saturation:"), 0, 0.5, 0, 0), 0, 1, table_y, table_y + 1, GtkAttachOptions(GTK_FILL), GTK_FILL, 5, 5);
	args->saturationRange = widget = gtk_hscale_new_with_range(0, 120, 1);
	gtk_range_set_value(GTK_RANGE(widget), options->getFloat("saturation", 100));
	g_signal_connect(G_OBJECT(widget), "value-changed", G_CALLBACK(GenerateSchemeArgs::onChange), args);
	g_signal_connect(G_OBJECT(widget), "format-value", G_CALLBACK(GenerateSchemeArgs::onSaturationFormat), args);
	gtk_table_attach(GTK_TABLE(table), widget, 1, 2, table_y, table_y + 1, GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_FILL, 5, 0);
	table_y++;
	gtk_table_attach(GTK_TABLE(table), gtk_label_aligned_new(_("Lightness:"), 0, 0.5, 0, 0), 0, 1, table_y, table_y + 1, GtkAttachOptions(GTK_FILL), GTK_FILL, 5, 5);
	args->lightnessRange = widget = gtk_hscale_new_with_range(-50, 80, 1);
	gtk_range_set_value(GTK_RANGE(widget), options->getFloat("lightness", 0));
	g_signal_connect(G_OBJECT(widget), "value-changed", G_CALLBACK(GenerateSchemeArgs::onChange), args);
	g_signal_connect(G_OBJECT(widget), "format-value", G_CALLBACK(GenerateSchemeArgs::onLightnessFormat), args);
	gtk_table_attach(GTK_TABLE(table), widget, 1, 2, table_y, table_y + 1, GtkAttachOptions(GTK_FILL | GTK_EXPAND), GTK_FILL, 5, 0);
	table_y++;
	gtk_table_attach(GTK_TABLE(table), gtk_label_aligned_new(_("Type:"), 0, 0.5, 0, 0), 0, 1, table_y, table_y + 1, GTK_FILL, GTK_SHRINK, 5, 5);
	args->generationType = widget = gtk_combo_box_text_new();
	for (size_t i = 0; i < generate_scheme_get_n_scheme_types(); i++) {
		gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(widget), _(generate_scheme_get_scheme_type(i)->name));
	}
	gtk_combo_box_set_active(GTK_COMBO_BOX(widget), options->getInt32("type", 0));
	g_signal_connect(G_OBJECT(widget), "changed", G_CALLBACK(GenerateSchemeArgs::onChange), args);
	gtk_table_attach(GTK_TABLE(table), widget, 1, 2, table_y, table_y + 1, GTK_FILL, GTK_SHRINK, 5, 0);
	table_y++;
	gtk_table_attach(GTK_TABLE(table), gtk_label_aligned_new(_("Color wheel:"), 0, 0.5, 0, 0), 0, 1, table_y, table_y + 1, GTK_FILL, GTK_SHRINK, 5, 5);
	args->wheelTypeCombo = widget = gtk_combo_box_text_new();
	for (size_t i = 0; i < color_wheel_types_get_n(); i++) {
		gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(widget), _(color_wheel_types_get()[i].name));
	}
	gtk_combo_box_set_active(GTK_COMBO_BOX(widget), options->getInt32("wheel_type", 0));
	g_signal_connect(G_OBJECT(widget), "changed", G_CALLBACK(GenerateSchemeArgs::onChange), args);
	gtk_table_attach(GTK_TABLE(table), widget, 1, 2, table_y, table_y + 1, GTK_FILL, GTK_SHRINK, 5, 0);
	table_y++;
	args->colorsVisible = MaxColors;
	gtk_widget_show_all(hbox);
	args->update();
	args->main = hbox;
	args->source.widget = hbox;
	return (ColorSource *)args;
}
int generate_scheme_source_register(ColorSourceManager *csm) {
	ColorSource *color_source = new ColorSource;
	color_source_init(color_source, "generate_scheme", _("Scheme generation"));
	color_source->implement = source_implement;
	color_source->default_accelerator = GDK_KEY_g;
	color_source_manager_add_source(csm, color_source);
	return 0;
}
const SchemeType *generate_scheme_get_scheme_type(size_t index) {
	if (index >= 0 && index < generate_scheme_get_n_scheme_types())
		return &types[index];
	else
		return 0;
}
size_t generate_scheme_get_n_scheme_types() {
	return sizeof(types) / sizeof(SchemeType);
}
