/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_dummy_good.mod dummy OK plugin

    \section dummysection1 The dummy good plugin
    ALWAYS return OK !
*/

/*!
    \file   lcmaps_dummy_good.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains code for a dummy LCMAPS plugin that always returns OK.
    the plugin has to provide to the LCMAPS. The interface consists of the following
    functions:
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

#define _XOPEN_SOURCE	500

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_plugins_basic_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include <ctype.h>

/******************************************************************************
                                Definitions
******************************************************************************/

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static uid_t dummy_uid;
static int dummy_uid_found=0;
static gid_t dummy_gid;
static int dummy_gid_found=0;
static gid_t dummy_sgid;
static int dummy_sgid_found=0;


/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
/*!
    \fn plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
        )
    \brief Plugin asks for required arguments

    \param int * argc
    \param lcmaps_argument_t ** argv
    \retval LCMAPS_MOD_SUCCESS success
    \retval LCMAPS_MOD_FAIL    failure (will result in a lcmaps failure)
*/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    const char * logstr = "lcmaps_dummy_good-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        { NULL           , NULL               , -1,   NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: %p\n", logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
/*!
    \fn plugin_initialize(
        int argc,
        char ** argv
        )
    \brief initialize the plugin.
    
    Everything that is needed to initialize the plugin should be put inside this function.
    Arguments as read from the LCMAPS database (argc, argv) are passed to the plugin.

    \param argc number of passed arguments.
    \param argv argument list. argv[0] contains the name of the plugin.
    \retval LCMAPS_MOD_SUCCESS successful initialization
    \retval LCMAPS_MOD_FAIL    failure in the plugin initialization
    \retval LCMAPS_MOD_NOFILE  private plugin database could not be found (same effect as
                             LCMAPS_MOD_FAIL)
*/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    const char * logstr = "lcmaps_dummy_good-plugin_initialize()";
    int i;
    size_t j;
    long value;


    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
        lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr,
             i,argv[i]);
    }

    for (i=1; i < argc; i++)
    {
        /* Dummy UID */
        if ( !(strcasecmp ("--dummy-uid", argv[i]))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"--dummy-uid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         goto fail_dummy;
                     }
                 }
		 errno=0;
		 value=strtol(argv[i + 1], NULL, 10);
		 if (errno!=0 || value<0)	{
		     lcmaps_log(LOG_ERR,"%s: Error: cannot convert %s to dummy_uid.\n",
			     logstr,argv[i + 1]);
		     goto fail_dummy;
		 }
                 dummy_uid = (uid_t)value;
		 dummy_uid_found = 1;
            }
        }
        /* Dummy GID */
        else if ( !(strcasecmp ("--dummy-gid", argv[i]))
                  && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"--dummy-gid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         goto fail_dummy;
                     }
                 }
		 errno=0;
		 value=strtol(argv[i + 1], NULL, 10);
		 if (errno!=0 || value<0)	{
		     lcmaps_log(LOG_ERR,"%s: Error: cannot convert %s to dummy_gid.\n",
			     logstr,argv[i + 1]);
		     goto fail_dummy;
		 }
                 dummy_gid = (gid_t)value;
		 dummy_gid_found = 1;
            }
        }
        /* Dummy Secondary GID */
        else if ( !(strcasecmp ("--dummy-sec-gid", argv[i]))
                  && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"--dummy-sec-gid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         goto fail_dummy;
                     }
                 }
		 errno=0;
		 value=strtol(argv[i + 1], NULL, 10);
		 if (errno!=0 || value<0)	{
		     lcmaps_log(LOG_ERR,"%s: Error: cannot convert %s to dummy_sgid.\n",
			     logstr,argv[i + 1]);
		     goto fail_dummy;
		 }
                 dummy_sgid = (gid_t)value;
		 dummy_sgid_found = 1;
            }
        }
        /* Dummy Username */
        else if ( !(strcasecmp ("--dummy-username", argv[i]))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                struct passwd       *user_info      = NULL;

                lcmaps_log_debug(5,"%s: Checking if argument behind \"--dummy-username\" is an existing username\n", logstr);
                if ( ( user_info = getpwnam(argv[i + 1]) ) )
                {
                    dummy_uid = user_info->pw_uid;
		    dummy_uid_found = 1;
                } 
                else
                {
                    lcmaps_log(LOG_ERR,"%s: no valid user account found with the name \"%s\"\n", logstr, argv[i + 1]);
                    goto fail_dummy;
                }
            }
        }
        /* Dummy Group */
        else if ( !(strcasecmp ("--dummy-group", argv[i]))
                  && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                struct group * group_info = NULL;

                lcmaps_log_debug(5,"%s: Checking if argument behind \"--dummy-group\" is an existing groupname\n", logstr);
                if ( ( group_info = getgrnam(argv[i + 1])) )
                {
                    dummy_gid = group_info->gr_gid;
		    dummy_gid_found = 1;
                } 
                else
                {
                    lcmaps_log(LOG_ERR,"%s: no valid group found with the name \"%s\"\n", logstr, argv[i + 1]);
                    goto fail_dummy;
                }
            }
        }
        /* Dummy Sec Group */
        else if ( !(strcasecmp ("--dummy-sec-group", argv[i]))
                  && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                struct group * group_info = NULL;

                lcmaps_log_debug(5,"%s: Checking if argument behind \"--dummy-sec-group\" is an existing secondary groupname\n", logstr);
                if ( ( group_info = getgrnam(argv[i + 1])) )
                {
                    dummy_sgid = group_info->gr_gid;
		    dummy_sgid_found = 1;
                } 
                else
                {
                    lcmaps_log(LOG_ERR,"%s: no valid group found with the name \"%s\"\n", logstr, argv[i + 1]);
                    goto fail_dummy;
                }
            }
        }
    }
    return LCMAPS_MOD_SUCCESS;


fail_dummy:
    return LCMAPS_MOD_FAIL; 
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
/*!
    \fn plugin_run(
        int argc,
        lcmaps_argument_t * argv
        )
    \brief Gather credentials for user making use of the ordered arguments

    Ask for credentials by passing the arguments (like JDL, globus DN, VOMS roles etc.)
    that were ordered earlier by the plugin_introspect() function

    \param argc number of arguments
    \param argv list of arguments
    \retval LCMAPS_MOD_SUCCESS authorization succeeded
    \retval LCMAPS_MOD_FAIL    authorization failed
*/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    const char * logstr = "lcmaps_dummy_good-plugin_run()";

    lcmaps_log_debug(5,"%s:\n", logstr);


    /* 
     * Add this credential data to the credential data repository in the plugin manager
     */
    if (dummy_uid_found)
    {
        addCredentialData(UID, &dummy_uid);
    }

    if (dummy_gid_found)
    {
        addCredentialData(PRI_GID, &dummy_gid);
    }

    if (dummy_sgid_found)
    {
        addCredentialData(SEC_GID, &dummy_sgid);
    }

    if ((dummy_uid_found) && (dummy_gid_found))
    {
        char * encoded_dn = (char*)malloc (sizeof (char) * 42);
        snprintf (encoded_dn, (size_t)(42 - 1), "dummy%%20%d%%20%d", (int)dummy_uid, (int)dummy_gid);

        addCredentialData(POOL_INDEX, &encoded_dn);
	/* addCredentialData uses strdup() */
	free(encoded_dn);
    }


    lcmaps_log(LOG_INFO, "%s: good plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
/*!
    \fn plugin_verify(
        int argc,
        lcmaps_argument_t * argv
        )
    \brief Verify credentials for user making use of the ordered arguments

    Ask for credentials by passing the arguments (like JDL, globus DN, VOMS roles etc.)
    that were ordered earlier by the plugin_introspect() function and verify the right
    to use a certain account

    \param argc number of arguments
    \param argv list of arguments
    \retval LCMAPS_MOD_SUCCESS authorization succeeded
    \retval LCMAPS_MOD_FAIL    authorization failed
*/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    const char * logstr = "lcmaps_dummy_good-plugin_verify()";

    lcmaps_log_debug(5,"%s:\n", logstr);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
/*!
    \fn plugin_terminate()
    \brief Whatever is needed to terminate the plugin module goes in here.
    \retval LCMAPS_MOD_SUCCESS success
    \retval LCMAPS_MOD_FAIL    failure (will result in an authorization failure)
*/
int plugin_terminate(void)
{
    const char * logstr = "lcmaps_dummy_good-plugin_terminate()";

    lcmaps_log_debug(5,"%s: terminating\n", logstr);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-basic/src/dummy/lcmaps_dummy_good.c,v $
    $Date: 2010-02-19 06:08:47 $
    $Revision: 1.9 $
    $Author: okoeroo $
******************************************************************************/
