/* parser.y
   
   Bison grammar for a parser of numeric specifications

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

%{
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>

#include "xmalloc.h"
#include "xstrdup.h"
#include "numbers.h"
#include "util.h"

static int proto_or_service_number(const char *);
static int create_iprange(numspec_t *, char *, int);
static int create_iprand(numspec_t *, char *, int);
static uint16_t mask2bits(char *);
static void numspecerror(numspec_t *, int, const char *);

extern int numspeclex(void);
extern void *numspec_scan_string(const char *);
extern int numspec_delete_buffer(void *);
%}

%error-verbose

%parse-param {numspec_t *nspec}
%parse-param {int hint}

/*************************************************************************
 **				  Tokens                                **
 *************************************************************************/

%token OP_INCR OP_DECR OP_ADD OP_SUB OP_MUL OP_DIV OP_MOD OP_SHL OP_SHR 
%token OP_RANGE
%token RANDOM
%token <string> STRING
%token <string> IPSTRING
%token <string> HOSTNAME
%token <number> NUMBER
%left ',' /* Token used to separate members of a list */
%left '-' '+'
%left '*' '/'
%left NEG   /* negaction - unary minus */

%type <numspec> numdef
%type <numspec> ipdef
%type <number> exp
%union {
    char *string;
    uint32_t number;
    numspec_t numspec;
}

%%

numspecdef:	  common    /* Common definitions */
		| numdef    /* Number definition */
		| ipdef	    /* IPv4 definition */
;

common:		  STRING    /* Service name (NUM_FIXED) or hostname
			     * (IP_FIXED), depending on the value passed
			     * to the parser via the hint variable.
			     */
    {
	struct addr a;

	if (hint == HINT_NUM) {
	    nspec->type = NUM_FIXED;
	    nspec->num_u32.value = proto_or_service_number($1);
	    if (nspec->num_u32.value == (uint32_t) -1) {
		free($1);
		YYABORT;
	    }
	} else {
	    /* Treat symbols as hostnames */
	    if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
		|| a.addr_bits != IP_ADDR_BITS) {
		free($1);
		YYABORT;
	    }

	    nspec->type = IP_FIXED;
	    nspec->num_ip.value = a.addr_ip;
	}

	free($1);
    }
		/* Random number or IP address */
		| RANDOM
    {
	nspec->type = hint == HINT_NUM ? NUM_RANDOM : IP_RANDOM;
    }
;

/* Definition of a number */
numdef:		  '<' numlistdef '>'
    {
	nspec->type = NUM_LIST;
    }
		| exp
    {
	nspec->type = NUM_FIXED;
	nspec->num_u32.value = $1;
    }
		| exp OP_INCR
    {
	nspec->type = NUM_ADD;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = 1;
    }
		| exp OP_DECR
    {
	nspec->type = NUM_SUB;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = 1;
    }
		| exp OP_ADD exp
    {
	nspec->type = NUM_ADD;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		| exp OP_SUB exp
    {
	nspec->type = NUM_SUB;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		| exp OP_MUL exp
    {
	nspec->type = NUM_MUL;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		| exp OP_DIV exp
    {
	if ($3 == 0)
	    YYABORT;

	nspec->type = NUM_DIV;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		| exp OP_MOD exp
    {
	nspec->type = NUM_MOD;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		| exp OP_SHL exp
    {
	nspec->type = NUM_SHL;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		| exp OP_SHR exp
    {
	nspec->type = NUM_SHR;
	nspec->num_u32op.current = $1;
	nspec->num_u32op.operand = $3;
    }
		/* Range, e.g. "1..1024" */
		| exp OP_RANGE exp
    {
	nspec->type = NUM_RANGE;
	nspec->num_u32range.current = $1;
	nspec->num_u32range.start = $1;
	nspec->num_u32range.end = $3;
	nspec->nvalues = abs($1 - $3) + 1;
    }
		/* Pseudo-random number, e.g. "random(5, 75%)" */
		| RANDOM '(' NUMBER ',' NUMBER '%' ')'
    {
	nspec->type = NUM_RANDOM;
	nspec->num_u32rand.value = $3;
	nspec->num_u32rand.probability = $5/100.0;
    }
		/* Pseudo-random number, e.g. "random(telnet, 75%)" */
		| RANDOM '(' STRING ',' NUMBER '%' ')'
    {
	nspec->type = NUM_RANDOM;
	nspec->num_u32rand.probability = $5/100.0;
	nspec->num_u32rand.value = proto_or_service_number($3);
	if (nspec->num_u32rand.value == (uint32_t) -1) {
	    free($3);
	    YYABORT;
	}
    }
		/* Random number within a range */
		| RANDOM '(' NUMBER OP_RANGE NUMBER ')'
    {
	nspec->type = NUM_RANDOM;
	nspec->num_u32rand.start = $3;
	nspec->num_u32rand.end = $5;
    }
;

numlistdef:	  numlistdef ',' numlistdef
		| exp
    {
	nspec->nvalues++;
	nspec->num_u32list.values = xrealloc(nspec->num_u32list.values,
				 nspec->nvalues*sizeof(uint32_t) );
	nspec->num_u32list.values[nspec->nvalues - 1] = $1;
    }
		| STRING
    {
	struct addr a;

	if (hint == HINT_NUM) {
	    nspec->nvalues++;
	    nspec->num_u32list.values = xrealloc(nspec->num_u32list.values,
				     nspec->nvalues*sizeof(uint32_t) );
	    nspec->num_u32list.values[nspec->nvalues - 1] =
		    proto_or_service_number($1);
	    if (nspec->num_u32list.values[nspec->nvalues - 1] == (uint32_t) -1) {
		free($1);
		YYABORT;
	    }
	} else {
	    /* Treat symbols as hostnames */
	    if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
		|| a.addr_bits != IP_ADDR_BITS) {
		free($1);
		YYABORT;
	    }

	    nspec->nvalues++;
	    nspec->num_iplist.values = xrealloc(nspec->num_iplist.values,
				     nspec->nvalues*sizeof(ip_addr_t) );
	    nspec->num_iplist.values[nspec->nvalues - 1] = a.addr_ip;
	}

	free($1);
    }
;

/* Expression */
exp:		  NUMBER	    { $$ = $1;	    }
		| exp '+' exp	    { $$ = $1 + $3; }
		| exp '-' exp	    { $$ = $1 - $3; }
		| exp '*' exp	    { $$ = $1 * $3; }
		| exp '/' exp	    { $$ = $1 / $3; }
		| '-' exp %prec NEG { $$ = -$2;     } /* Unary minus */
		| '(' exp ')'       { $$ = $2;      }
;

ipdef:		  '<' iplistdef '>'
    {
	nspec->type = IP_LIST;
    }
		| IPSTRING
    {
	struct addr a;

	if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	    || a.addr_bits != IP_ADDR_BITS) {
	    free($1);
	    YYABORT;
	}

	nspec->type = IP_FIXED;
	nspec->num_ip.value = a.addr_ip;
	free($1);
    }
		| HOSTNAME
    {
	struct addr a;

	if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	    || a.addr_bits != IP_ADDR_BITS) {
	    free($1);
	    YYABORT;
	}

	nspec->type = IP_FIXED;
	nspec->num_ip.value = a.addr_ip;
	free($1);
    }
		  /* Expressions like "ssh/2" are not supported. If hint is
		     HINT_IP then we interpret this as an IP range like
		     "host/24" where "host" is a hostname that does not
		     specify a domain name. */
		| STRING '/' NUMBER
    {
	if (hint == HINT_NUM) {
	    free($1);
	    YYABORT;
	}

	if (create_iprange(nspec, $1, $3) == -1)
	    YYABORT;
    }
		| IPSTRING '/' NUMBER	/* e.g. 1.2.3.4/24 */
    {
	if (create_iprange(nspec, $1, $3) == -1)
	    YYABORT;
    }
		| HOSTNAME '/' NUMBER	/* e.g. host.example.com/24 */
    {
	if (create_iprange(nspec, $1, $3) == -1)
	    YYABORT;
    }
		| IPSTRING '/' IPSTRING	/* e.g. 1.2.3.4/255.255.255.240 */
    {
	uint16_t bits;

	bits = mask2bits($3);
	free($3);

	if (create_iprange(nspec, $1, bits) == -1)
	    YYABORT;
    }
		| HOSTNAME '/' IPSTRING	/* e.g. example.com/255.255.255.240 */
    {
	uint16_t bits;

	bits = mask2bits($3);
	free($3);

	if (create_iprange(nspec, $1, bits) == -1)
	    YYABORT;
    }
		| STRING '/' IPSTRING	/* e.g. hal9000/255.255.255.240 */
    {
	uint16_t bits;

	if (hint != HINT_IP) {
	    free($1);
	    free($3);
	    YYABORT;
	}

	bits = mask2bits($3);
	free($3);

	if (create_iprange(nspec, $1, bits) == -1)
	    YYABORT;
    }
		  /* Random IP address within a range, e.g.
		     "random(1.2.3.4/24)" */
		| RANDOM '(' IPSTRING '/' NUMBER ')'
    {
	if (create_iprand(nspec, $3, $5) == -1)
	    YYABORT;
    }
		  /* IP range, e.g. 1.2.3.4 .. 1.2.3.10 */
		| IPSTRING OP_RANGE IPSTRING
    {
	struct addr a;
	struct addr b;

	if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	    || a.addr_bits != IP_ADDR_BITS) {
	    free($1);
	    free($3);
	    YYABORT;
	}

	free($1);

	if (addr_pton($3, &b) == -1 || b.addr_type != ADDR_TYPE_IP
	    || b.addr_bits != IP_ADDR_BITS) {
	    free($3);
	    YYABORT;
	}

	free($3);

	if (a.addr_ip >= b.addr_ip)
	    YYABORT;

	nspec->type = IP_RANGE;
	nspec->num_iprange.start = a.addr_ip;
	nspec->num_iprange.end = b.addr_ip;
	nspec->num_iprange.current = nspec->num_iprange.start;
	nspec->nvalues = abs(ntohl(nspec->num_iprange.end)
			     - ntohl(nspec->num_iprange.start) ) + 1;
    }
		  /* Random IP address within a range, e.g.
		     "random(www.example.com/24)" */
		| RANDOM '(' HOSTNAME '/' NUMBER ')'
    {
	if (create_iprand(nspec, $3, $5) == -1)
	    YYABORT;
    }
		  /* Random IP address within a range, e.g.
		     "random(hal9000/24)" */
		| RANDOM '(' STRING '/' NUMBER ')'
    {
	if (create_iprand(nspec, $3, $5) == -1)
	    YYABORT;
    }
		  /* Incremented IP address, e.g. 192.168.1.1++ */
		| IPSTRING OP_INCR
    {
	struct addr a;

	if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	    || a.addr_bits != IP_ADDR_BITS) {
	    free($1);
	    YYABORT;
	}

	free($1);

	nspec->type = IP_ADD;
	nspec->num_ipop.current = a.addr_ip;
	nspec->num_ipop.operand = 1;
    }
;

iplistdef:	  iplistdef ',' iplistdef
		| IPSTRING
    {
	struct addr a;

	if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	    || a.addr_bits != IP_ADDR_BITS) {
	    free($1);
	    YYABORT;
	}

	nspec->nvalues++;
	nspec->num_iplist.values = xrealloc(nspec->num_iplist.values,
				 nspec->nvalues*sizeof(ip_addr_t) );
	nspec->num_iplist.values[nspec->nvalues - 1] = a.addr_ip;

	free($1);
    }
		| HOSTNAME
    {
	struct addr a;

	if (addr_pton($1, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	    || a.addr_bits != IP_ADDR_BITS) {
	    free($1);
	    YYABORT;
	}

	nspec->nvalues++;
	nspec->num_iplist.values = xrealloc(nspec->num_iplist.values,
				 nspec->nvalues*sizeof(ip_addr_t) );
	nspec->num_iplist.values[nspec->nvalues - 1] = a.addr_ip;

	free($1);
    }
;

%%

/*
 * Given a protocol or TCP or UDP service name, try to obtain the
 * corresponding protocol or service number by querying the
 * system protocols and services databases.
 */
static int
proto_or_service_number(const char *proto_or_service_name)
{
    struct servent *service;
    struct protoent *protocol;

    /* Try as a service first */
    if (!(service = getservbyname(proto_or_service_name, NULL) ) ) {
	/* Not a service; try as a protocol */
	if (!(protocol = getprotobyname(proto_or_service_name) ) )
	    /* Not a protocol either; give up */
	    return -1;

	return protocol->p_proto;
    }

    return ntohs(service->s_port);
}

static int
create_iprange(numspec_t *nspec, char *host, int addr_bits)
{
    struct addr a;

    if (addr_bits > IP_ADDR_BITS) {
	free(host);
	return 1;
    }

    if (addr_pton(host, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	|| a.addr_bits != IP_ADDR_BITS) {
	free(host);
	return -1;
    }

    nspec->type = IP_RANGE;
    nspec->num_iprange.start = a.addr_ip
	& htonl(~( (1 << (IP_ADDR_BITS - addr_bits) ) - 1) );
    nspec->num_iprange.end = nspec->num_iprange.start
			     + htonl( (1 << (IP_ADDR_BITS - addr_bits) ) - 1);
    nspec->num_iprange.current = nspec->num_iprange.start;
    nspec->nvalues = abs(ntohl(nspec->num_iprange.end)
		     - ntohl(nspec->num_iprange.start) ) + 1;

    free(host);

    return 0;
}

static int
create_iprand(numspec_t *nspec, char *host, int addr_bits)
{
    struct addr a;

    if (addr_bits > IP_ADDR_BITS) {
	free(host);
	return 1;
    }

    if (addr_pton(host, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	|| a.addr_bits != IP_ADDR_BITS) {
	free(host);
	return -1;
    }

    nspec->type = IP_RANDOM;
    nspec->num_iprand.start = a.addr_ip
	& htonl(~( (1 << (IP_ADDR_BITS - addr_bits) ) - 1) );
    nspec->num_iprand.end = nspec->num_iprand.start
	+ htonl( (1 << (IP_ADDR_BITS - addr_bits) ) - 1);

    free(host);

    return 0;
}

static uint16_t
mask2bits(char *netmask)
{
    struct addr a;
    uint16_t bits;

    if (addr_pton(netmask, &a) == -1 || a.addr_type != ADDR_TYPE_IP
	|| a.addr_bits != IP_ADDR_BITS)
	return 0;

    addr_mtob(&a.addr_ip, sizeof(a.addr_ip), &bits);

    return bits;
}

static void
numspecerror(numspec_t *n, int hint, const char *s)
{
    memset(n, 0, sizeof *n);

    fprintf(stderr, "%s; hint = %d\n", s, hint);
}

numspec_t *
num_init(const char *number_spec)
{
    void *yybuf;
    int retval;
    numspec_t *n;

    yybuf = numspec_scan_string(number_spec);

    n = xmalloc(sizeof(numspec_t) );
    memset(n, 0, sizeof *n);
    n->sequence = numspec_seq++;
    n->spec = xstrdup(number_spec);

    retval = numspecparse(n, HINT_NUM);	/* Parse the numeric specification */

    numspec_delete_buffer(yybuf);

    if (retval != 0) {
	free(n->spec);
	free(n);
	return NULL;
    }

    /*
     * Verify that we created an specification for a number.
     */
    switch (n->type) {
    case IP_FIXED:
    case IP_ADD:
    case IP_RANGE:
    case IP_RANDOM:
    case IP_LIST:
	free(n->spec);
	free(n);
	return NULL;
    default:
	;
    }

    return n;
}

/*
 * Same as num_init() but receives a number instead of a string.
 */
numspec_t *
num_initn(uint32_t number)
{
    numspec_t *spec;
    char str[256];

    spec = xmalloc(sizeof *spec);
    memset(spec, 0, sizeof *spec);
    spec->sequence = numspec_seq++;

    spec->type = NUM_FIXED;
    spec->num_u32.value = number;

    snprintf(str, sizeof str, "%u", number);
    spec->spec = xstrdup(str);

    return spec;
}

numspec_t *
ip_init(const char *number_spec)
{
    void *yybuf;
    int retval;
    numspec_t *n;

    yybuf = numspec_scan_string(number_spec);

    n = xmalloc(sizeof *n);
    memset(n, 0, sizeof *n);
    n->sequence = numspec_seq++;
    n->spec = xstrdup(number_spec);

    retval = numspecparse(n, HINT_IP);	/* Parse the numeric specification */

    numspec_delete_buffer(yybuf);

    if (retval != 0) {
	free(n->spec);
	free(n);
	return NULL;
    }

    /*
     * Verify that we created an specification for an IP address.
     */
    switch (n->type) {
    case IP_FIXED:
    case IP_ADD:
    case IP_RANGE:
    case IP_RANDOM:
    case IP_LIST:
	break;
    default:
	free(n->spec);
	free(n);
	return NULL;
    }

    return n;
}

numspec_t *
ip_initn(ip_addr_t ipaddr)
{
    numspec_t *spec;

    spec = xmalloc(sizeof *spec);
    memset(spec, 0, sizeof *spec);
    spec->sequence = numspec_seq++;

    spec->type = IP_FIXED;
    spec->num_ip.value = ipaddr;

    spec->spec = xstrdup(_num_ipaddr2str(ipaddr) );

    return spec;
}
