/* util.c
  
   Utility functions used by different parts of the libpbuild subsystem

   Copyright (C) 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dnet.h>
#include <glib.h>

#include "missing.h"
#include "util.h"
#include "pbuild-priv.h"

#ifndef CHAR_BIT
#define CHAR_BIT 8
#endif

unsigned short
_pb_cksum(void *data, int len)
{
    int counter = len;
    unsigned short *w = data;
    int sum = 0;
    unsigned short tmp = 0;

    while (counter > 1) {
	sum += *w++;
	counter -= 2;
    }

    if (counter == 1) {
	/* Length is odd */
	*(unsigned char *) &tmp = *(unsigned char *) w;
	sum += tmp;
    }

    /* We want a 1's complement sum, so we must add whatever was
       carried out from the low 16 bits to the low 16 bits */
    sum = (sum >> 16) + (unsigned short) sum;

    /* Add the carry from the previous sum */
    sum += sum >> 16;

    /* Now we take the 1's complement of the 1's complement sum */
    tmp = ~sum;

    return tmp;
}

/*
 * Used by dump_pdu(). Creates a prefix of whitespace that is used to
 * indent PDU dumps.
 */
char *
_pb_make_prefix(int level)
{
    static char prefix[512];
    int nspaces;

    nspaces = level*2;
    if (nspaces > (int) sizeof prefix)
	/* To prevent writing past the end of the prefix[] array should
	 * the integer "level" gets out of control.
	 */
	nspaces = sizeof(prefix) - 1;
    memset(prefix, ' ', nspaces);
    prefix[nspaces] = '\0';
    
    return prefix;
}

/*
 * Takes as input a MAC address and returns its ASCII representation.
 * The returned string lives in a static buffer that is overwritten
 * with each call.
 */
char *
_pb_mac_to_ascii(eth_addr_t *mac)
{
    static char mac_ascii[sizeof("xx:xx:xx:xx:xx:xx")];

    snprintf(mac_ascii, sizeof(mac_ascii), "%02x:%02x:%02x:%02x:%02x:%02x",
	     mac->data[0], mac->data[1],
	     mac->data[2], mac->data[3],
	     mac->data[4], mac->data[5]);

    return mac_ascii;
}

char *
_pb_ip6addr2str(ip6_addr_t ip6)
{
    struct addr a;
    static char addr[46];

    memset(&a, 0, sizeof(struct addr) );
    a.addr_type = ADDR_TYPE_IP6;
    a.addr_bits = IP6_ADDR_BITS;
    a.addr_ip6 = ip6;

    return addr_ntop(&a, addr, sizeof(addr) );
}

/*
 * Shifts an array of a specific size a certain number of bits
 * to the left. Inserted bits are 0. The shiftcount must be
 * less or equal to CHAR_BIT (usually 8).
 */
void
_pb_array_shift_left(unsigned char *array, size_t size, int shiftcount)
{
    int i;
    unsigned char byte, bits_out;

    for (i = size - 1, bits_out = 0; i >= 0; i--) {
	byte = array[i];
	array[i] <<= shiftcount;
	array[i] |= bits_out;
	bits_out = byte >> (CHAR_BIT - shiftcount);
    }
}

void
_pb_dump_bytes(const unsigned char *data, unsigned len)
{
    unsigned i, j;

    for (i = 0; i <= len/16; i++) {
	printf("%08x  ", i*16);

	for (j = 0; j < 16; j++) {
	    if (i*16 + j < len)
		printf("%02x", data[i*16 + j]);
	    else
		printf("  ");

	    if (j & 1)
		printf(" ");
	}

	for (j = 0; j < 16; j++)
	    if (i*16 + j < len)
		printf("%c", isprint( (int ) data[i*16 + j])
			     ? data[i*16 + j] : '.');

	printf("\n");
    }

    printf("\n");
}

/*
 * Converts the ASCII representation of an Ethernet MAC address
 * to its binary representation.
 *
 * Stolen from packit, with some enhancements for netexpect
 *
 * Returns 0 if ASCII representation of a MAC address is valid,
 * or -1 otherwise.
 *
 * Accepts a MAC address as xx:xx:xx:xx:xx:xx or as
 * xxxx.xxxx.xxxx (as used in Cisco IOS.)
 */
int
_pb_format_ethernet_addr(const char *ethstr, uint8_t u_eaddr[ETH_ADDR_LEN])
{
    int i;
    unsigned long base16;
    char *eptr;
    char o_ethstr[sizeof("xx:xx:xx:xx:xx:xx")]; /* will hold temp. copy of
						   input string since we
						   don't want to modify the
						   input string */

    if (ethstr) {
	if (   !strcasecmp("broadcast", ethstr)
	    || !strcasecmp("bcast", ethstr) ) {
	    /*
	     * MAC address is ff:ff:ff:ff:ff:ff.
	     */
	    memset(u_eaddr, 0xff, ETH_ADDR_LEN);
	} else if (!strcasecmp("random", ethstr) ) {
	    /*
	     * MAC address is random.
	     *
	     * We use IANA Ethernet address block for unicast use. See
	     * http://www.iana.org/assignments/ethernet-numbers.
	     */
	    u_eaddr[0] = 0x00;
	    u_eaddr[1] = 0x00;
	    u_eaddr[2] = 0x5e;
	    _pb_random_block(u_eaddr + 3, ETH_ADDR_LEN - 3);
	} else {
	    /*
	     * Not a special MAC address (broadcast or random) so we need
	     * to parse the ASCII representation of the MAC address.
	     */

	    /*
	     * We don't want to modify the input string ethstr (and
	     * that's why we declared it as "const char *") so we need
	     * to make a local copy of it because strtok() does modify
	     * the input string.
	     */
	    strlcpy(o_ethstr, ethstr, sizeof(o_ethstr) );

	    if (strchr(o_ethstr, '.') ) {
		/* Assume ffff.ffff.ffff format */

		for (i = 0, eptr = strtok( (char *) o_ethstr, ".");
		     eptr && i <= ETH_ADDR_LEN;
		     eptr = strtok(NULL, ":"), i += 2) {
		    if ( (base16 = strtoul(eptr, 0, 16) ) > 0xffff)
			return -1;

		    u_eaddr[i] = base16;
		}

		if (i != ETH_ADDR_LEN || eptr)
		    /*
		     * Not enough address bytes or there's still some input left.
		     * This is an invalid MAC address.
		     */
		    return -1;
	    } else {
		/* Assume ff:ff:ff:ff:ff:ff format */

		for (i = 0, eptr = strtok( (char *) o_ethstr, ":");
		     eptr && i <= ETH_ADDR_LEN;
		     eptr = strtok(NULL, ":"), i++) {
		    if ( (base16 = strtoul(eptr, 0, 16) ) > 0xff)
			return -1;

		    u_eaddr[i] = base16;
		}

		if (i != ETH_ADDR_LEN || eptr)
		    /*
		     * Not enough address bytes or there's still some input left.
		     * This is an invalid MAC address.
		     */
		    return -1;
	    }
	}
    } else
	/*
	 * We received NULL for the MAC address string. Assume a MAC
	 * address of 00:00:00:00:00:00.
	 */
	memset(u_eaddr, 0x00, ETH_ADDR_LEN);

    return 0;
}

/*
 * Takes as input a MAC address and returns its ASCII representation.
 * The returned string lives in a static buffer that is overwritten
 * with each call.
 */
char *
_pb_ipx_to_ascii(struct ipx_address *ipx)
{
    static char ipx_ascii[sizeof("AABBCCDD.aabbccddeeff")];

    snprintf(ipx_ascii, sizeof(ipx_ascii), "%08x.%02x%02x%02x%02x%02x%02x",
	     ntohl(ipx->net),
	     ipx->node[0], ipx->node[1],
	     ipx->node[2], ipx->node[3],
	     ipx->node[4], ipx->node[5]);

    return ipx_ascii;
}

/*
 * Converts an IPX address from its ASCII representation to a binary
 * representation. An IPX address has the form:
 *
 * net_addr.node_addr
 *
 * net_addr is 4 bytes and node_addr is 6 bytes (like an Ethernet MAC)
 *
 * A couple of examples of the addresses this function can parse:
 * 
 * 12345678.123456789012
 * 1234.123456
 *
 * Returns -1 if address is invalid or if there is a memory allocation
 * error, or 0 if everything's fine.
 */
int
_pb_format_ipx_addr(const char *ipxaddr, struct ipx_address *ipx)
{
    char *s, *net, *node, digit;

    s = strdup(ipxaddr);
    if (!s)
	return -1;

    node = strchr(s, '.');
    if (!node++ || strlen(node) > IPX_NODE_LEN*2)
	goto invalid;

    node[-1] = '\0'; /* Terminate network address string */
    net = s;
    if (strlen(net) > IPX_NET_LEN*2)
	goto invalid;

    /* Since the network address is 4-bytes long we can just use
     * strtoul() to convert the string to binary... */
    ipx->net = htonl(strtoul(net, NULL, 16) );

    /* Convert the node address. The node address is 6-bytes long
     * so we can't use something like strtoul(). It is a bit more
     * complicated - we handle it by shifting the array 4 bits at
     * a time to the left. */
    memset(&ipx->node, 0, IPX_NODE_LEN);
    while ( (digit = *node++) ) {
	if (!isxdigit(digit) )
	    goto invalid;

	/* Doesn't work. Compiler bug or invalid C?
	digit -= (digit = tolower(digit) ) >= 'a' ? 'a' - 10 : '0'; */
	digit = tolower(digit);
	digit -= digit >= 'a' ? 'a' - 10 : '0';

	/* Shift the node array 4 bits to the left */
	_pb_array_shift_left(ipx->node, IPX_NODE_LEN, 4);
	ipx->node[IPX_NODE_LEN - 1] += digit;
    }

    free(s);
    return 0;

invalid:
    free(s);
    return -1;
}

/*
 * Fills a block of memory with random data.
 */
void
_pb_random_block(uint8_t *dest, size_t len)
{
    while (len--)
	*dest++ = rand();
}

GHashTable *
_pb_new_hash(const struct pdu_dict *dict)
{
    GHashTable *hash;

    /*
     * Create hash table for protocol name to protocol value
     * mappings. This is used by the 1st pass PDU builder.
     */
    hash = g_hash_table_new(g_str_hash, g_str_equal);

    for (; dict->key; dict++)
	g_hash_table_insert(hash, dict->key, dict->data);

    return hash;
}
