"estimateTruncatedNormalParameters" <-
function(numericalSample, testLevel = 0.975){
  
  # Checking for NAs in sample
  numericalSample <- numericalSample[ ! is.na(numericalSample)]

  a <- min(numericalSample)
  b <- max(numericalSample)
  # First guess for the optimization step. Should find a better solution.
  mean <- mean(numericalSample)
  var <- var(numericalSample)
  mean0 <- mean
  var0 <- var
  for (i in 1:100) {
    sd <- sqrt(var)
    phiA <- dnorm(a, mean, sd)
    phiB <- dnorm(b, mean, sd)
    PhiA <- pnorm(a, mean, sd)
    PhiB <- pnorm(b, mean, sd)
    denom <- (PhiB - PhiA - (mean0 - a) * phiA - (b - mean0) * phiB)
    var <- var0 * (PhiB - PhiA) / denom
    mean <- mean0 + var * (phiB - phiA) / (PhiB - PhiA)
  }
  
  # For reasons of data types and calls, we need to use an auxiliary function. However, 
  # this is just an intermediary step, we still use fitdistr on the Beta Pdf.
  AuxiliaryTruncatedNormal <- function(numericalSample,...){
    # Call to fitdistr (MASS package) .
    estimation <- fitdistr(numericalSample, computeTruncatedNormalPdf, 
                           start = list(mu = mean, sigma = sqrt(var)), ...)
  }

  estimation <- NA
  try(estimation <- AuxiliaryTruncatedNormal(numericalSample, a = (a - .Machine$double.eps), b = (b + .Machine$double.eps)))
  if (is.na(estimation))
  {
    # If the first method of estimation fails, we try the L-BFGS-B method.
    estimation <- AuxiliaryTruncatedNormal(numericalSample, a = (a - .Machine$double.eps), b = (b + .Machine$double.eps), method = "L-BFGS-B", lower = c(-Inf,0))
  }

  # Checking whether loglikelihood is supported in the current R version. 
  if (length(estimation$loglik) == 0) {
    warning("Warning: the current R version does not support the loglikelihood computation")
    estimation$loglik = "Warning: the current R version does not support the loglikelihood computation"
  }

  # Computation of the t-value used in the CIs

  tValue <- qt(testLevel, length(numericalSample)-1)

  confidenceIntervalmu <- c(as.numeric(estimation$estimate[1]) - 
                              tValue * as.numeric(estimation$sd[1]),
                            as.numeric(estimation$estimate[1]) + 
                              tValue * as.numeric(estimation$sd[1]))

  confidenceIntervalsigma <- c(as.numeric(estimation$estimate[2]) - 
                                 tValue * as.numeric(estimation$sd[2]),
                               as.numeric(estimation$estimate[2]) + 
                                 tValue * as.numeric(estimation$sd[2]))
  
  estimatedTruncatedNormal <- list(distribution = "TruncatedNormal",
                        mu = as.numeric(estimation$estimate[1]),
                        sigma = as.numeric(estimation$estimate[2]),
                        a = a,
                        b = b,
                        confidenceIntervalmu = confidenceIntervalmu,
                        confidenceIntervalsigma = confidenceIntervalsigma,
                        logLikelihood = estimation$loglik)
}
