///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/rendering/FrameBuffer.h>
#include <core/rendering/RenderSettings.h>
#include <core/utilities/ProgressIndicator.h>
#include <core/reference/CloneHelper.h>
#include <core/plugins/PluginManager.h>
#include <core/plugins/Plugin.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/objects/SceneObject.h>
#include <core/data/DataSet.h>
#include <core/data/DataSetManager.h>

#include "TachyonRenderer.h"
#include "TachyonRendererEditor.h"

#if !(((TACHYON_MAJOR_VERSION >= 0) && (TACHYON_MINOR_VERSION >= 99)) || ((TACHYON_MAJOR_VERSION == 0) && (TACHYON_MINOR_VERSION == 99) && (TACHYON_PATCH_VERSION >= 0)))
#error "OVITO Tachyon plugin requires Tachyon version 0.99.0 or higher."
#endif

// Callback routines for the Tachyon library:

extern "C" {

static ProgressIndicator* currentIndicator = NULL;

void my_rt_ui_message(int a, char * msg) {
  MsgLogger() << "Tachyon: " << msg << endl;
}

void my_rt_ui_progress(int percent) {
	if(currentIndicator != NULL) {
		currentIndicator->setValue(percent);
		currentIndicator->isCanceled();
	}
	else
		VerboseLogger() << "Tachyon rendering progress: " << percent << "%% complete" << endl;
}

}

namespace TachyonPlugin {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(TachyonRenderer, PluginRenderer)
DEFINE_PROPERTY_FIELD(TachyonRenderer, "EnableAntialiasing", _enableAntialiasing)
DEFINE_PROPERTY_FIELD(TachyonRenderer, "RenderingMode", _renderingMode)
DEFINE_PROPERTY_FIELD(TachyonRenderer, "AntialiasingSamples", _antialiasingSamples)
DEFINE_PROPERTY_FIELD(TachyonRenderer, "AmbientOcclusionSamples", _ambientOcclusionSamples)
SET_PROPERTY_FIELD_LABEL(TachyonRenderer, _enableAntialiasing, "Enable anti-aliasing")
SET_PROPERTY_FIELD_LABEL(TachyonRenderer, _renderingMode, "Rendering mode")
SET_PROPERTY_FIELD_LABEL(TachyonRenderer, _antialiasingSamples, "Anti-aliasing samples")
SET_PROPERTY_FIELD_LABEL(TachyonRenderer, _ambientOcclusionSamples, "Ambient occlusion samples")

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(TachyonExportInterface, PluginClass)

/******************************************************************************
* Default constructor.
******************************************************************************/
TachyonRenderer::TachyonRenderer(bool isLoading)
	: PluginRenderer(isLoading),
	  _enableAntialiasing(true), _renderingMode(0),
	  _antialiasingSamples(12), _ambientOcclusionSamples(12)
{
	INIT_PROPERTY_FIELD(TachyonRenderer, _enableAntialiasing)
	INIT_PROPERTY_FIELD(TachyonRenderer, _renderingMode)
	INIT_PROPERTY_FIELD(TachyonRenderer, _antialiasingSamples)
	INIT_PROPERTY_FIELD(TachyonRenderer, _ambientOcclusionSamples)
}

/******************************************************************************
* Prepares the renderer for rendering of the given scene.
******************************************************************************/
bool TachyonRenderer::startRender(DataSet* dataset)
{
	this->dataset = dataset;

	loadCustomExporters();

	MsgLogger() << "Initializing Tachyon raytracer library." << endl;
	rt_initialize(0, NULL);
	rt_set_ui_message(my_rt_ui_message);
	rt_set_ui_progress(my_rt_ui_progress);

	return true;
}

/******************************************************************************
* Renders a single animation frame into the given frame buffer.
******************************************************************************/
bool TachyonRenderer::renderFrame(TimeTicks time, CameraViewDescription view, FrameBuffer* frameBuffer)
{
	ProgressIndicator progress(tr("Tachyon engine (no abort)"), 100);
	currentIndicator = &progress;

	// Create new scene and set up parameters.
	_rtscene = rt_newscene();
	rt_resolution(_rtscene, renderSettings()->outputImageWidth(), renderSettings()->outputImageHeight());
	if(_enableAntialiasing)
		rt_aa_maxsamples(_rtscene, _antialiasingSamples);

	// Install Tachyon framebuffer.
	QImage img(renderSettings()->outputImageWidth(), renderSettings()->outputImageHeight(), QImage::Format_RGB888);
	rt_rawimage_rgb24(_rtscene, img.bits());

	// Set background color.
	Color backgroundColor(renderSettings()->backgroundColorController()->getValueAtTime(time));
	rt_background(_rtscene, rt_color(backgroundColor.r, backgroundColor.g, backgroundColor.b));

	// Set equation used for rendering specular highlights.
	rt_phong_shader(_rtscene, RT_SHADER_BLINN_FAST);	// Fast version of Blinn's equation

	// Set up camera.
	if(view.isPerspective) {
		rt_camera_projection(_rtscene, RT_PROJECTION_PERSPECTIVE);

		// Calculate projection point and directions in camera space.
		Point3 p0 = view.inverseProjectionMatrix * Point3(0,0,0);
		Vector3 direction = view.inverseProjectionMatrix * Point3(0,0,0) - ORIGIN;
		Vector3 up = view.inverseProjectionMatrix * Point3(0,1,0) - p0;
		// Transform to world space.
		p0 = ORIGIN + view.inverseViewMatrix.getTranslation();
		direction = Normalize(view.inverseViewMatrix * direction);
		up = Normalize(view.inverseViewMatrix * up);
		rt_camera_position(_rtscene, rt_vector(p0.X, p0.Y, -p0.Z), rt_vector(direction.X, direction.Y, -direction.Z), rt_vector(up.X, up.Y, -up.Z));
		rt_camera_zoom(_rtscene, 0.5 / tan(view.fieldOfView * 0.5));
	}
	else {
		rt_camera_projection(_rtscene, RT_PROJECTION_ORTHOGRAPHIC);

		// Calculate projection point and directions in camera space.
		Point3 p0 = view.inverseProjectionMatrix * Point3(0,0,0);
		Vector3 direction = view.inverseProjectionMatrix * Point3(0,0,1) - p0;
		Vector3 up = view.inverseProjectionMatrix * Point3(0,1,0) - p0;
		// Transform to world space.
		p0 = view.inverseViewMatrix * p0;
		direction = Normalize(view.inverseViewMatrix * direction);
		up = Normalize(view.inverseViewMatrix * up);
		p0 += direction * view.znear;
		rt_camera_position(_rtscene, rt_vector(p0.X, p0.Y, -p0.Z), rt_vector(direction.X, direction.Y, -direction.Z), rt_vector(up.X, up.Y, -up.Z));
		rt_camera_zoom(_rtscene, 0.5 / view.fieldOfView / view.aspectRatio);
	}

	// Set up lights.
	apitexture lightTex;
	memset(&lightTex, 0, sizeof(lightTex));
	lightTex.col.r = 0.8;
	lightTex.col.g = 0.8;
	lightTex.col.b = 0.8;
	lightTex.ambient = 1.0;
	lightTex.opacity = 1.0;
	lightTex.diffuse = 1.0;
	void* lightTexPtr = rt_texture(_rtscene, &lightTex);
	rt_directional_light(_rtscene, lightTexPtr, rt_vector(1.2, -1.6, 1.9));
	rt_directional_light(_rtscene, lightTexPtr, rt_vector(-1.2, 1.6, -1.9));

	if(_renderingMode > 0) {
		// Full shading mode required
		rt_shadermode(_rtscene, RT_SHADER_FULL);
	}
	else {
		// This will turn off shadows.
		rt_shadermode(_rtscene, RT_SHADER_MEDIUM);
	}

	if(_renderingMode == 2) {
		apicolor skycol;
		skycol.r = 0.5;
		skycol.g = 0.5;
		skycol.b = 0.5;
		rt_rescale_lights(_rtscene, 0.5);
		rt_ambient_occlusion(_rtscene, _ambientOcclusionSamples, skycol);
	}

	TachyonWriter writer(dataset, time, view, this, _rtscene);

	// Export scene objects.
	SceneNodesIterator iter(dataset->sceneRoot());
	for(; !iter.finished(); iter.next()) {
		ObjectNode* node = dynamic_object_cast<ObjectNode>(iter.current());
		if(!node) continue;

		PipelineFlowState state = node->evalPipeline(time);
		if(!state.result()) continue;

		TimeInterval iv;
		AffineTransformation nodeTM = node->objectTransform() * node->getWorldTransform(time, iv);

		Q_FOREACH(const TachyonExportInterface::SmartPtr& iface, _exportInterfaces) {
			if(iface->exportSceneObject(state.result(), writer, node, nodeTM))
				break;
		}
	}

	// Render scene.
	rt_renderscene(_rtscene);

	// Clean up.
	rt_deletescene(_rtscene);
	currentIndicator = NULL;

	// Copy rendered image into Ovito's frame buffer.
	// Qt stores images with lines aligned to four byte memory addresses.
	// Tachyon does no alignment. That's why we have to move the pixel data for each scan line.
	if((renderSettings()->outputImageWidth() % 4) != 0) {
		int bperline = renderSettings()->outputImageWidth() * 3;
		for(int y = renderSettings()->outputImageHeight() - 1; y > 0; y--) {
			memmove(img.bits() + y*img.bytesPerLine(), img.bits() + y*bperline, bperline);
		}
	}
	// Flip image since Tachyon fills the buffer upside down.
	QPainter painter(&frameBuffer->image());
	painter.drawImage(0, 0, img.mirrored(false, true));

	return progress.isCanceled() == false;
}

/******************************************************************************
* Finishes the rendering pass. This is called after all animation frames have been rendered
* or when the rendering operation has been aborted.
******************************************************************************/
void TachyonRenderer::endRender()
{
	// Shut down Tachyon library.
	rt_finalize();
}

/******************************************************************************
* Loads all installed scene object export interfaces.
******************************************************************************/
void TachyonRenderer::loadCustomExporters()
{
	if(_exportInterfaces.empty()) {
		// Load custom object export interfaces (from other plugins only).
		Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(TachyonExportInterface))) {
			if(clazz->isAbstract()) continue;
			try {
				TachyonExportInterface::SmartPtr iface = static_object_cast<TachyonExportInterface>(clazz->createInstance());
				_exportInterfaces.push_back(iface);
				MsgLogger() << "Custom Tachyon export interface found:" << clazz->name() << endl;
			}
			catch(const Exception& ex) {
				qWarning() << QString("WARNING: Failed to load Tachyon object export interface %1 from plugin %2.").arg(clazz->name(), clazz->plugin()->pluginId());
				qWarning() << QString("Reason: %1").arg(ex.message());
			}
		}
	}
}

/******************************************************************************
* Creates a texture with the given color.
******************************************************************************/
void* TachyonWriter::getTexture(const Color& color)
{
	apitexture tex;
	void* voidtex;

	memset(&tex, 0, sizeof(tex));
	tex.ambient  = 0.2;
	tex.diffuse  = 0.8;
	tex.specular = 0.0;
	tex.opacity  = 1.0;
	tex.col.r = color.r;
	tex.col.g = color.g;
	tex.col.b = color.b;
	tex.texturefunc = RT_TEXTURE_CONSTANT;

	voidtex = rt_texture(_rtscene, &tex);

	//rt_tex_phong(voidtex, 0.1, 2.0, RT_PHONG_PLASTIC);

	return voidtex;
}

/******************************************************************************
* Renders a list of spheres.
******************************************************************************/
void TachyonWriter::sphereArray(int numberOfSpheres, Point3 const* centers, FloatType const* radii, Color const* colors)
{
	OVITO_ASSERT(centers != NULL);
	OVITO_ASSERT(colors != NULL);
	OVITO_ASSERT(radii != NULL);

	const Point3* p = centers;
	const FloatType* r = radii;
	const Color* c = colors;
	for(int i = 0; i < numberOfSpheres; i++, ++p, ++r, ++c) {
		void* tex = getTexture(*c);
		Point3 tp = transformation() * (*p);
		rt_sphere(_rtscene, tex, rt_vector(tp.X, tp.Y, -tp.Z), *r);
	}
}

/******************************************************************************
* Renders a cylinder (without caps).
******************************************************************************/
void TachyonWriter::cylinder(const Point3& from, const Point3& to, FloatType radius, const Color& color, bool topCap, bool bottomCap)
{
	void* tex = getTexture(color);
	Point3 tp = transformation() * from;
	Vector3 ta = transformation() * (to - from);
	rt_fcylinder(_rtscene, tex,
	               rt_vector(tp.X, tp.Y, -tp.Z),
	               rt_vector(ta.X, ta.Y, -ta.Z),
	               radius);

	if(topCap) {
		rt_ring(_rtscene, tex,
				rt_vector(tp.X+ta.X, tp.Y+ta.Y, -tp.Z-ta.Z),
				rt_vector(ta.X, ta.Y, -ta.Z), 0, radius);
	}

	if(bottomCap) {
		rt_ring(_rtscene, tex,
				rt_vector(tp.X, tp.Y, -tp.Z),
				rt_vector(-ta.X, -ta.Y, ta.Z), 0, radius);
	}
}

/******************************************************************************
* Renders a sphere.
******************************************************************************/
void TachyonWriter::sphere(const Point3& center, FloatType radius, const Color& color)
{
	void* tex = getTexture(color);
	Point3 tp = transformation() * center;
	rt_sphere(_rtscene, tex,
	               rt_vector(tp.X, tp.Y, -tp.Z),
	               radius);
}

};

