#ifndef _RHEO_MASS_S_H
#define _RHEO_MASS_S_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:mass_s
NAME: @code{mass_s} -- L2 scalar product on a surface
@bfindex mass_s
@apindex P1
@cindex level set method
SYNOPSIS:
    @example
        form(const space& Bh, const space& Bh, "mass_s", phi_h);
    @end example
DESCRIPTION:       
    @noindent
    On a surface Gamma defined as the zero of a level-set function,
    assembly the matrix associated to the L2 scalar product:
@iftex
@tex
    $$
   	m(u,v) = \int_\Gamma  u v \, dx
    $$
@end tex
@end iftex
@ifnottex
    @example
                 /
                 |
   	m(u,v) = | u v dx
                 |
                 / Gamma
    @end example
@end ifnottex
    
    @noindent
    The Bh space may be a @code{P1} finite element space on the band arround the zero level
    set.
    Notice that elements different from P1 are not yet supported.
EXAMPLE:
    @example
         space Vh (lambda, "P1");
         field phi_h = interpolate(Vh, phi);
         geo beta = banded_level_set (phi_h);
         space Bh (beta, "P1");
         Bh.block("isolated");
         field phi_bh = interpolate(Bh, phi);
         form m  (Bh, Bh, "mass_s", phi_bh);
    @end example
AUTHOR: Lara Aborm, Jocelyn Etienne, Pierre Saramito
DATE:
    18 april 2011
End:
*/

#include "rheolef/form_element_rep.h"
namespace rheolef { 

class mass_s: public form_element_rep {
public:
   typedef enum {Wh_Wh, Wh_Bh, Bh_Wh, Bh_Bh} XY_type;
   XY_type get_XY_type () const { return _XY_switch; }

// allocator:

    mass_s() : form_element_rep() {}

// virtual accessor:

    void operator() (const geo_element& K, ublas::matrix<Float>& m) const;
    void check_after_initialize () const;

// data:
protected:
    mutable XY_type _XY_switch;
};
}// namespace rheolef
#endif // _RHEO_MASS_S_H
