use crate::lex::lex;
use crate::types::*;
use crate::SyntaxKind;
use crate::SyntaxKind::*;
use crate::DEFAULT_VERSION;
use std::str::FromStr;

#[derive(Debug, Clone, PartialEq, Eq, Hash)]
pub struct ParseError(Vec<String>);

impl std::fmt::Display for ParseError {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        for err in &self.0 {
            writeln!(f, "{}", err)?;
        }
        Ok(())
    }
}

impl std::error::Error for ParseError {}

/// Second, implementing the `Language` trait teaches rowan to convert between
/// these two SyntaxKind types, allowing for a nicer SyntaxNode API where
/// "kinds" are values from our `enum SyntaxKind`, instead of plain u16 values.
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Hash)]
enum Lang {}
impl rowan::Language for Lang {
    type Kind = SyntaxKind;
    fn kind_from_raw(raw: rowan::SyntaxKind) -> Self::Kind {
        unsafe { std::mem::transmute::<u16, SyntaxKind>(raw.0) }
    }
    fn kind_to_raw(kind: Self::Kind) -> rowan::SyntaxKind {
        kind.into()
    }
}

/// GreenNode is an immutable tree, which is cheap to change,
/// but doesn't contain offsets and parent pointers.
use rowan::GreenNode;

/// You can construct GreenNodes by hand, but a builder
/// is helpful for top-down parsers: it maintains a stack
/// of currently in-progress nodes
use rowan::GreenNodeBuilder;

/// The parse results are stored as a "green tree".
/// We'll discuss working with the results later
struct Parse {
    green_node: GreenNode,
    #[allow(unused)]
    errors: Vec<String>,
    #[allow(unused)]
    version: i32,
}

fn parse(text: &str) -> Parse {
    struct Parser {
        /// input tokens, including whitespace,
        /// in *reverse* order.
        tokens: Vec<(SyntaxKind, String)>,
        /// the in-progress tree.
        builder: GreenNodeBuilder<'static>,
        /// the list of syntax errors we've accumulated
        /// so far.
        errors: Vec<String>,
    }

    impl Parser {
        fn parse_version(&mut self) -> Option<i32> {
            let mut version = None;
            if self.tokens.last() == Some(&(KEY, "version".to_string())) {
                self.builder.start_node(VERSION.into());
                self.bump();
                self.skip_ws();
                if self.current() != Some(EQUALS) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push("expected `=`".to_string());
                    self.bump();
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
                if self.current() != Some(VALUE) {
                    self.builder.start_node(ERROR.into());
                    self.errors
                        .push(format!("expected value, got {:?}", self.current()));
                    self.bump();
                    self.builder.finish_node();
                } else {
                    let version_str = self.tokens.last().unwrap().1.clone();
                    match version_str.parse() {
                        Ok(v) => {
                            version = Some(v);
                            self.bump();
                        }
                        Err(_) => {
                            self.builder.start_node(ERROR.into());
                            self.errors
                                .push(format!("invalid version: {}", version_str));
                            self.bump();
                            self.builder.finish_node();
                        }
                    }
                }
                if self.current() != Some(NEWLINE) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push("expected newline".to_string());
                    self.bump();
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
                self.builder.finish_node();
            }
            version
        }

        fn parse_watch_entry(&mut self) -> bool {
            self.skip_ws();
            if self.current().is_none() {
                return false;
            }
            if self.current() == Some(NEWLINE) {
                self.bump();
                return false;
            }
            self.builder.start_node(ENTRY.into());
            self.parse_options_list();
            for i in 0..4 {
                if self.current() == Some(NEWLINE) {
                    break;
                }
                if self.current() == Some(CONTINUATION) {
                    self.bump();
                    self.skip_ws();
                    continue;
                }
                if self.current() != Some(VALUE) && self.current() != Some(KEY) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push(format!(
                        "expected value, got {:?} (i={})",
                        self.current(),
                        i
                    ));
                    if self.current().is_some() {
                        self.bump();
                    }
                    self.builder.finish_node();
                } else {
                    // Wrap each field in its appropriate node
                    match i {
                        0 => {
                            // URL
                            self.builder.start_node(URL.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        1 => {
                            // Matching pattern
                            self.builder.start_node(MATCHING_PATTERN.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        2 => {
                            // Version policy
                            self.builder.start_node(VERSION_POLICY.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        3 => {
                            // Script
                            self.builder.start_node(SCRIPT.into());
                            self.bump();
                            self.builder.finish_node();
                        }
                        _ => {
                            self.bump();
                        }
                    }
                }
                self.skip_ws();
            }
            if self.current() != Some(NEWLINE) && self.current().is_some() {
                self.builder.start_node(ERROR.into());
                self.errors
                    .push(format!("expected newline, not {:?}", self.current()));
                if self.current().is_some() {
                    self.bump();
                }
                self.builder.finish_node();
            } else {
                self.bump();
            }
            self.builder.finish_node();
            true
        }

        fn parse_option(&mut self) -> bool {
            if self.current().is_none() {
                return false;
            }
            while self.current() == Some(CONTINUATION) {
                self.bump();
            }
            if self.current() == Some(WHITESPACE) {
                return false;
            }
            self.builder.start_node(OPTION.into());
            if self.current() != Some(KEY) {
                self.builder.start_node(ERROR.into());
                self.errors.push("expected key".to_string());
                self.bump();
                self.builder.finish_node();
            } else {
                self.bump();
            }
            if self.current() == Some(EQUALS) {
                self.bump();
                if self.current() != Some(VALUE) && self.current() != Some(KEY) {
                    self.builder.start_node(ERROR.into());
                    self.errors
                        .push(format!("expected value, got {:?}", self.current()));
                    self.bump();
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
            } else if self.current() == Some(COMMA) {
            } else {
                self.builder.start_node(ERROR.into());
                self.errors.push("expected `=`".to_string());
                if self.current().is_some() {
                    self.bump();
                }
                self.builder.finish_node();
            }
            self.builder.finish_node();
            true
        }

        fn parse_options_list(&mut self) {
            self.skip_ws();
            if self.tokens.last() == Some(&(KEY, "opts".to_string()))
                || self.tokens.last() == Some(&(KEY, "options".to_string()))
            {
                self.builder.start_node(OPTS_LIST.into());
                self.bump();
                self.skip_ws();
                if self.current() != Some(EQUALS) {
                    self.builder.start_node(ERROR.into());
                    self.errors.push("expected `=`".to_string());
                    if self.current().is_some() {
                        self.bump();
                    }
                    self.builder.finish_node();
                } else {
                    self.bump();
                }
                let quoted = if self.current() == Some(QUOTE) {
                    self.bump();
                    true
                } else {
                    false
                };
                loop {
                    if quoted {
                        if self.current() == Some(QUOTE) {
                            self.bump();
                            break;
                        }
                        self.skip_ws();
                    }
                    if !self.parse_option() {
                        break;
                    }
                    if self.current() == Some(COMMA) {
                        self.builder.start_node(OPTION_SEPARATOR.into());
                        self.bump();
                        self.builder.finish_node();
                    } else if !quoted {
                        break;
                    }
                }
                self.builder.finish_node();
                self.skip_ws();
            }
        }

        fn parse(mut self) -> Parse {
            let mut version = 1;
            // Make sure that the root node covers all source
            self.builder.start_node(ROOT.into());
            if let Some(v) = self.parse_version() {
                version = v;
            }
            // TODO: use version to influence parsing
            loop {
                if !self.parse_watch_entry() {
                    break;
                }
            }
            // Don't forget to eat *trailing* whitespace
            self.skip_ws();
            // Close the root node.
            self.builder.finish_node();

            // Turn the builder into a GreenNode
            Parse {
                green_node: self.builder.finish(),
                errors: self.errors,
                version,
            }
        }
        /// Advance one token, adding it to the current branch of the tree builder.
        fn bump(&mut self) {
            let (kind, text) = self.tokens.pop().unwrap();
            self.builder.token(kind.into(), text.as_str());
        }
        /// Peek at the first unprocessed token
        fn current(&self) -> Option<SyntaxKind> {
            self.tokens.last().map(|(kind, _)| *kind)
        }
        fn skip_ws(&mut self) {
            while self.current() == Some(WHITESPACE)
                || self.current() == Some(CONTINUATION)
                || self.current() == Some(COMMENT)
            {
                self.bump()
            }
        }
    }

    let mut tokens = lex(text);
    tokens.reverse();
    Parser {
        tokens,
        builder: GreenNodeBuilder::new(),
        errors: Vec::new(),
    }
    .parse()
}

/// To work with the parse results we need a view into the
/// green tree - the Syntax tree.
/// It is also immutable, like a GreenNode,
/// but it contains parent pointers, offsets, and
/// has identity semantics.

type SyntaxNode = rowan::SyntaxNode<Lang>;
#[allow(unused)]
type SyntaxToken = rowan::SyntaxToken<Lang>;
#[allow(unused)]
type SyntaxElement = rowan::NodeOrToken<SyntaxNode, SyntaxToken>;

impl Parse {
    fn syntax(&self) -> SyntaxNode {
        SyntaxNode::new_root_mut(self.green_node.clone())
    }

    fn root(&self) -> WatchFile {
        WatchFile::cast(self.syntax()).unwrap()
    }
}

macro_rules! ast_node {
    ($ast:ident, $kind:ident) => {
        #[derive(PartialEq, Eq, Hash)]
        #[repr(transparent)]
        /// A node in the syntax tree for $ast
        pub struct $ast(SyntaxNode);
        impl $ast {
            #[allow(unused)]
            fn cast(node: SyntaxNode) -> Option<Self> {
                if node.kind() == $kind {
                    Some(Self(node))
                } else {
                    None
                }
            }
        }

        impl ToString for $ast {
            fn to_string(&self) -> String {
                self.0.text().to_string()
            }
        }
    };
}

ast_node!(WatchFile, ROOT);
ast_node!(Version, VERSION);
ast_node!(Entry, ENTRY);
ast_node!(OptionList, OPTS_LIST);
ast_node!(_Option, OPTION);
ast_node!(Url, URL);
ast_node!(MatchingPattern, MATCHING_PATTERN);
ast_node!(VersionPolicyNode, VERSION_POLICY);
ast_node!(ScriptNode, SCRIPT);

impl WatchFile {
    /// Create a new watch file with specified version
    pub fn new(version: Option<u32>) -> WatchFile {
        let mut builder = GreenNodeBuilder::new();

        builder.start_node(ROOT.into());
        if let Some(version) = version {
            builder.start_node(VERSION.into());
            builder.token(KEY.into(), "version");
            builder.token(EQUALS.into(), "=");
            builder.token(VALUE.into(), version.to_string().as_str());
            builder.token(NEWLINE.into(), "\n");
            builder.finish_node();
        }
        builder.finish_node();
        WatchFile(SyntaxNode::new_root_mut(builder.finish()))
    }

    /// Returns the version of the watch file.
    pub fn version(&self) -> u32 {
        self.0
            .children()
            .find_map(Version::cast)
            .map(|it| it.version())
            .unwrap_or(DEFAULT_VERSION)
    }

    /// Returns an iterator over all entries in the watch file.
    pub fn entries(&self) -> impl Iterator<Item = Entry> + '_ {
        self.0.children().filter_map(Entry::cast)
    }

    /// Set the version of the watch file.
    pub fn set_version(&mut self, new_version: u32) {
        // Build the new version node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(VERSION.into());
        builder.token(KEY.into(), "version");
        builder.token(EQUALS.into(), "=");
        builder.token(VALUE.into(), new_version.to_string().as_str());
        builder.token(NEWLINE.into(), "\n");
        builder.finish_node();
        let new_version_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_version_node = SyntaxNode::new_root_mut(new_version_green);

        // Find existing version node if any
        let version_pos = self.0.children().position(|child| child.kind() == VERSION);

        if let Some(pos) = version_pos {
            // Replace existing version node
            self.0
                .splice_children(pos..pos + 1, vec![new_version_node.into()]);
        } else {
            // Insert version node at the beginning
            self.0.splice_children(0..0, vec![new_version_node.into()]);
        }
    }

    /// Discover releases for all entries in the watch file (async version)
    ///
    /// Fetches URLs and searches for version matches for all entries.
    /// Requires the 'discover' feature.
    ///
    /// # Examples
    ///
    /// ```ignore
    /// # use debian_watch::WatchFile;
    /// # async fn example() {
    /// let wf: WatchFile = r#"version=4
    /// https://example.com/releases/ .*/v?(\d+\.\d+)\.tar\.gz
    /// "#.parse().unwrap();
    /// let all_releases = wf.uscan(|| "mypackage".to_string()).await.unwrap();
    /// for (entry_idx, releases) in all_releases.iter().enumerate() {
    ///     println!("Entry {}: {} releases found", entry_idx, releases.len());
    /// }
    /// # }
    /// ```
    #[cfg(feature = "discover")]
    pub async fn uscan(
        &self,
        package: impl Fn() -> String,
    ) -> Result<Vec<Vec<crate::Release>>, Box<dyn std::error::Error>> {
        let mut all_releases = Vec::new();

        for entry in self.entries() {
            let releases = entry.discover(|| package()).await?;
            all_releases.push(releases);
        }

        Ok(all_releases)
    }

    /// Discover releases for all entries in the watch file (blocking version)
    ///
    /// Fetches URLs and searches for version matches for all entries.
    /// Requires both 'discover' and 'blocking' features.
    ///
    /// # Examples
    ///
    /// ```ignore
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// https://example.com/releases/ .*/v?(\d+\.\d+)\.tar\.gz
    /// "#.parse().unwrap();
    /// let all_releases = wf.uscan_blocking(|| "mypackage".to_string()).unwrap();
    /// for (entry_idx, releases) in all_releases.iter().enumerate() {
    ///     println!("Entry {}: {} releases found", entry_idx, releases.len());
    /// }
    /// ```
    #[cfg(all(feature = "discover", feature = "blocking"))]
    pub fn uscan_blocking(
        &self,
        package: impl Fn() -> String,
    ) -> Result<Vec<Vec<crate::Release>>, Box<dyn std::error::Error>> {
        let mut all_releases = Vec::new();

        for entry in self.entries() {
            let releases = entry.discover_blocking(|| package())?;
            all_releases.push(releases);
        }

        Ok(all_releases)
    }

    /// Add an entry to the watch file.
    ///
    /// Appends a new entry to the end of the watch file.
    ///
    /// # Examples
    ///
    /// ```
    /// use debian_watch::{WatchFile, EntryBuilder};
    ///
    /// let mut wf = WatchFile::new(Some(4));
    ///
    /// // Add an entry using EntryBuilder
    /// let entry = EntryBuilder::new("https://github.com/example/tags")
    ///     .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
    ///     .build();
    /// wf.add_entry(entry);
    ///
    /// // Or use the builder pattern directly
    /// wf.add_entry(
    ///     EntryBuilder::new("https://example.com/releases")
    ///         .matching_pattern(".*/(\\d+\\.\\d+)\\.tar\\.gz")
    ///         .opt("compression", "xz")
    ///         .version_policy("debian")
    ///         .build()
    /// );
    /// ```
    pub fn add_entry(&mut self, entry: Entry) {
        // Find the position to insert (after the last entry or after version)
        let insert_pos = self.0.children_with_tokens().count();

        // Detach the entry node from its current parent and get its green node
        let entry_green = entry.0.green().into_owned();
        let new_entry_node = SyntaxNode::new_root_mut(entry_green);

        // Insert the entry at the end
        self.0
            .splice_children(insert_pos..insert_pos, vec![new_entry_node.into()]);
    }
}

impl FromStr for WatchFile {
    type Err = ParseError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let parsed = parse(s);
        if parsed.errors.is_empty() {
            Ok(parsed.root())
        } else {
            Err(ParseError(parsed.errors))
        }
    }
}

impl Version {
    /// Returns the version of the watch file.
    pub fn version(&self) -> u32 {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE {
                        Some(token.text().parse().unwrap())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap_or(DEFAULT_VERSION)
    }
}

/// Builder for creating new watchfile entries.
///
/// Provides a fluent API for constructing entries with various components.
///
/// # Examples
///
/// ```
/// use debian_watch::EntryBuilder;
///
/// // Minimal entry with just URL and pattern
/// let entry = EntryBuilder::new("https://github.com/example/tags")
///     .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
///     .build();
///
/// // Entry with options
/// let entry = EntryBuilder::new("https://github.com/example/tags")
///     .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
///     .opt("compression", "xz")
///     .flag("repack")
///     .version_policy("debian")
///     .script("uupdate")
///     .build();
/// ```
#[derive(Debug, Clone, Default)]
pub struct EntryBuilder {
    url: Option<String>,
    matching_pattern: Option<String>,
    version_policy: Option<String>,
    script: Option<String>,
    opts: std::collections::HashMap<String, String>,
}

impl EntryBuilder {
    /// Create a new entry builder with the specified URL.
    pub fn new(url: impl Into<String>) -> Self {
        EntryBuilder {
            url: Some(url.into()),
            matching_pattern: None,
            version_policy: None,
            script: None,
            opts: std::collections::HashMap::new(),
        }
    }

    /// Set the matching pattern for the entry.
    pub fn matching_pattern(mut self, pattern: impl Into<String>) -> Self {
        self.matching_pattern = Some(pattern.into());
        self
    }

    /// Set the version policy for the entry.
    pub fn version_policy(mut self, policy: impl Into<String>) -> Self {
        self.version_policy = Some(policy.into());
        self
    }

    /// Set the script for the entry.
    pub fn script(mut self, script: impl Into<String>) -> Self {
        self.script = Some(script.into());
        self
    }

    /// Add an option to the entry.
    pub fn opt(mut self, key: impl Into<String>, value: impl Into<String>) -> Self {
        self.opts.insert(key.into(), value.into());
        self
    }

    /// Add a boolean flag option to the entry.
    ///
    /// Boolean options like "repack", "bare", "decompress" don't have values.
    pub fn flag(mut self, key: impl Into<String>) -> Self {
        self.opts.insert(key.into(), String::new());
        self
    }

    /// Build the entry.
    ///
    /// # Panics
    ///
    /// Panics if no URL was provided.
    pub fn build(self) -> Entry {
        let url = self.url.expect("URL is required for entry");

        let mut builder = GreenNodeBuilder::new();

        builder.start_node(ENTRY.into());

        // Add options list if provided
        if !self.opts.is_empty() {
            builder.start_node(OPTS_LIST.into());
            builder.token(KEY.into(), "opts");
            builder.token(EQUALS.into(), "=");

            let mut first = true;
            for (key, value) in self.opts.iter() {
                if !first {
                    builder.token(COMMA.into(), ",");
                }
                first = false;

                builder.start_node(OPTION.into());
                builder.token(KEY.into(), key);
                if !value.is_empty() {
                    builder.token(EQUALS.into(), "=");
                    builder.token(VALUE.into(), value);
                }
                builder.finish_node();
            }

            builder.finish_node();
            builder.token(WHITESPACE.into(), " ");
        }

        // Add URL (required)
        builder.start_node(URL.into());
        builder.token(VALUE.into(), &url);
        builder.finish_node();

        // Add matching pattern if provided
        if let Some(pattern) = self.matching_pattern {
            builder.token(WHITESPACE.into(), " ");
            builder.start_node(MATCHING_PATTERN.into());
            builder.token(VALUE.into(), &pattern);
            builder.finish_node();
        }

        // Add version policy if provided
        if let Some(policy) = self.version_policy {
            builder.token(WHITESPACE.into(), " ");
            builder.start_node(VERSION_POLICY.into());
            builder.token(VALUE.into(), &policy);
            builder.finish_node();
        }

        // Add script if provided
        if let Some(script_val) = self.script {
            builder.token(WHITESPACE.into(), " ");
            builder.start_node(SCRIPT.into());
            builder.token(VALUE.into(), &script_val);
            builder.finish_node();
        }

        builder.token(NEWLINE.into(), "\n");
        builder.finish_node();

        Entry(SyntaxNode::new_root_mut(builder.finish()))
    }
}

impl Entry {
    /// Create a new entry builder.
    ///
    /// This is a convenience method that returns an `EntryBuilder`.
    ///
    /// # Examples
    ///
    /// ```
    /// use debian_watch::Entry;
    ///
    /// let entry = Entry::builder("https://github.com/example/tags")
    ///     .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
    ///     .build();
    /// ```
    pub fn builder(url: impl Into<String>) -> EntryBuilder {
        EntryBuilder::new(url)
    }

    /// List of options
    pub fn option_list(&self) -> Option<OptionList> {
        self.0.children().find_map(OptionList::cast)
    }

    /// Get the value of an option
    pub fn get_option(&self, key: &str) -> Option<String> {
        self.option_list().and_then(|ol| ol.get_option(key))
    }

    /// Check if an option is set
    pub fn has_option(&self, key: &str) -> bool {
        self.option_list().map_or(false, |ol| ol.has_option(key))
    }

    /// The name of the secondary source tarball
    pub fn component(&self) -> Option<String> {
        self.get_option("component")
    }

    /// Component type
    pub fn ctype(&self) -> Result<Option<ComponentType>, ()> {
        self.get_option("ctype").map(|s| s.parse()).transpose()
    }

    /// Compression method
    pub fn compression(&self) -> Result<Option<Compression>, ()> {
        self.get_option("compression")
            .map(|s| s.parse())
            .transpose()
    }

    /// Repack the tarball
    pub fn repack(&self) -> bool {
        self.has_option("repack")
    }

    /// Repack suffix
    pub fn repacksuffix(&self) -> Option<String> {
        self.get_option("repacksuffix")
    }

    /// Retrieve the mode of the watch file entry.
    pub fn mode(&self) -> Result<Mode, ()> {
        Ok(self
            .get_option("mode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the git pretty mode
    pub fn pretty(&self) -> Result<Pretty, ()> {
        Ok(self
            .get_option("pretty")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Set the date string used by the pretty option to an arbitrary format as an optional
    /// opts argument when the matching-pattern is HEAD or heads/branch for git mode.
    pub fn date(&self) -> String {
        self.get_option("date")
            .unwrap_or_else(|| "%Y%m%d".to_string())
    }

    /// Return the git export mode
    pub fn gitexport(&self) -> Result<GitExport, ()> {
        Ok(self
            .get_option("gitexport")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the git mode
    pub fn gitmode(&self) -> Result<GitMode, ()> {
        Ok(self
            .get_option("gitmode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the pgp mode
    pub fn pgpmode(&self) -> Result<PgpMode, ()> {
        Ok(self
            .get_option("pgpmode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the search mode
    pub fn searchmode(&self) -> Result<SearchMode, ()> {
        Ok(self
            .get_option("searchmode")
            .map(|s| s.parse())
            .transpose()?
            .unwrap_or_default())
    }

    /// Return the decompression mode
    pub fn decompress(&self) -> bool {
        self.has_option("decompress")
    }

    /// Whether to disable all site specific special case code such as URL director uses and page
    /// content alterations.
    pub fn bare(&self) -> bool {
        self.has_option("bare")
    }

    /// Set the user-agent string used to contact the HTTP(S) server as user-agent-string. (persistent)
    pub fn user_agent(&self) -> Option<String> {
        self.get_option("user-agent")
    }

    /// Use PASV mode for the FTP connection.
    pub fn passive(&self) -> Option<bool> {
        if self.has_option("passive") || self.has_option("pasv") {
            Some(true)
        } else if self.has_option("active") || self.has_option("nopasv") {
            Some(false)
        } else {
            None
        }
    }

    /// Add the extra options to use with the unzip command, such as -a, -aa, and -b, when executed
    /// by mk-origtargz.
    pub fn unzipoptions(&self) -> Option<String> {
        self.get_option("unzipopt")
    }

    /// Normalize the downloaded web page string.
    pub fn dversionmangle(&self) -> Option<String> {
        self.get_option("dversionmangle")
            .or_else(|| self.get_option("versionmangle"))
    }

    /// Normalize the directory path string matching the regex in a set of parentheses of
    /// http://URL as the sortable version index string.  This is used
    /// as the directory path sorting index only.
    pub fn dirversionmangle(&self) -> Option<String> {
        self.get_option("dirversionmangle")
    }

    /// Normalize the downloaded web page string.
    pub fn pagemangle(&self) -> Option<String> {
        self.get_option("pagemangle")
    }

    /// Normalize the candidate upstream version strings extracted from hrefs in the
    /// source of the web page.  This is used as the version sorting index when selecting the
    /// latest upstream version.
    pub fn uversionmangle(&self) -> Option<String> {
        self.get_option("uversionmangle")
            .or_else(|| self.get_option("versionmangle"))
    }

    /// Syntactic shorthand for uversionmangle=rules, dversionmangle=rules
    pub fn versionmangle(&self) -> Option<String> {
        self.get_option("versionmangle")
    }

    /// Convert the selected upstream tarball href string from the percent-encoded hexadecimal
    /// string to the decoded normal URL  string  for  obfuscated
    /// web sites.  Only percent-encoding is available and it is decoded with
    /// s/%([A-Fa-f\d]{2})/chr hex $1/eg.
    pub fn hrefdecode(&self) -> bool {
        self.get_option("hrefdecode").is_some()
    }

    /// Convert the selected upstream tarball href string into the accessible URL for obfuscated
    /// web sites.  This is run after hrefdecode.
    pub fn downloadurlmangle(&self) -> Option<String> {
        self.get_option("downloadurlmangle")
    }

    /// Generate the upstream tarball filename from the selected href string if matching-pattern
    /// can extract the latest upstream version <uversion> from the  selected  href  string.
    /// Otherwise, generate the upstream tarball filename from its full URL string and set the
    /// missing <uversion> from the generated upstream tarball filename.
    ///
    /// Without this option, the default upstream tarball filename is generated by taking the last
    /// component of the URL and  removing everything  after any '?' or '#'.
    pub fn filenamemangle(&self) -> Option<String> {
        self.get_option("filenamemangle")
    }

    /// Generate the candidate upstream signature file URL string from the upstream tarball URL.
    pub fn pgpsigurlmangle(&self) -> Option<String> {
        self.get_option("pgpsigurlmangle")
    }

    /// Generate the version string <oversion> of the source tarball <spkg>_<oversion>.orig.tar.gz
    /// from <uversion>.  This should be used to add a suffix such as +dfsg to a MUT package.
    pub fn oversionmangle(&self) -> Option<String> {
        self.get_option("oversionmangle")
    }

    /// Apply uversionmangle to a version string
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=uversionmangle=s/\+ds// https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(entry.apply_uversionmangle("1.0+ds").unwrap(), "1.0");
    /// ```
    pub fn apply_uversionmangle(
        &self,
        version: &str,
    ) -> Result<String, crate::mangle::MangleError> {
        if let Some(vm) = self.uversionmangle() {
            crate::mangle::apply_mangle(&vm, version)
        } else {
            Ok(version.to_string())
        }
    }

    /// Apply dversionmangle to a version string
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=dversionmangle=s/\+dfsg$// https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(entry.apply_dversionmangle("1.0+dfsg").unwrap(), "1.0");
    /// ```
    pub fn apply_dversionmangle(
        &self,
        version: &str,
    ) -> Result<String, crate::mangle::MangleError> {
        if let Some(vm) = self.dversionmangle() {
            crate::mangle::apply_mangle(&vm, version)
        } else {
            Ok(version.to_string())
        }
    }

    /// Apply oversionmangle to a version string
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=oversionmangle=s/$/-1/ https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(entry.apply_oversionmangle("1.0").unwrap(), "1.0-1");
    /// ```
    pub fn apply_oversionmangle(
        &self,
        version: &str,
    ) -> Result<String, crate::mangle::MangleError> {
        if let Some(vm) = self.oversionmangle() {
            crate::mangle::apply_mangle(&vm, version)
        } else {
            Ok(version.to_string())
        }
    }

    /// Apply dirversionmangle to a directory path string
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=dirversionmangle=s/v(\d)/$1/ https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(entry.apply_dirversionmangle("v1.0").unwrap(), "1.0");
    /// ```
    pub fn apply_dirversionmangle(
        &self,
        version: &str,
    ) -> Result<String, crate::mangle::MangleError> {
        if let Some(vm) = self.dirversionmangle() {
            crate::mangle::apply_mangle(&vm, version)
        } else {
            Ok(version.to_string())
        }
    }

    /// Apply filenamemangle to a URL or filename string
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/mypackage-$1.tar.gz/ https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(
    ///     entry.apply_filenamemangle("https://example.com/v1.0.tar.gz").unwrap(),
    ///     "mypackage-1.0.tar.gz"
    /// );
    /// ```
    pub fn apply_filenamemangle(&self, url: &str) -> Result<String, crate::mangle::MangleError> {
        if let Some(vm) = self.filenamemangle() {
            crate::mangle::apply_mangle(&vm, url)
        } else {
            Ok(url.to_string())
        }
    }

    /// Apply pagemangle to page content bytes
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=pagemangle=s/&amp;/&/g https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(
    ///     entry.apply_pagemangle(b"foo &amp; bar").unwrap(),
    ///     b"foo & bar"
    /// );
    /// ```
    pub fn apply_pagemangle(&self, page: &[u8]) -> Result<Vec<u8>, crate::mangle::MangleError> {
        if let Some(vm) = self.pagemangle() {
            let page_str = String::from_utf8_lossy(page);
            let mangled = crate::mangle::apply_mangle(&vm, &page_str)?;
            Ok(mangled.into_bytes())
        } else {
            Ok(page.to_vec())
        }
    }

    /// Apply downloadurlmangle to a URL string
    ///
    /// # Examples
    ///
    /// ```
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// opts=downloadurlmangle=s|/archive/|/download/| https://example.com/ .*
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// assert_eq!(
    ///     entry.apply_downloadurlmangle("https://example.com/archive/file.tar.gz").unwrap(),
    ///     "https://example.com/download/file.tar.gz"
    /// );
    /// ```
    pub fn apply_downloadurlmangle(&self, url: &str) -> Result<String, crate::mangle::MangleError> {
        if let Some(vm) = self.downloadurlmangle() {
            crate::mangle::apply_mangle(&vm, url)
        } else {
            Ok(url.to_string())
        }
    }

    /// Discover releases for this entry (async version)
    ///
    /// Fetches the URL and searches for version matches.
    /// Requires the 'discover' feature.
    ///
    /// # Examples
    ///
    /// ```ignore
    /// # use debian_watch::WatchFile;
    /// # async fn example() {
    /// let wf: WatchFile = r#"version=4
    /// https://example.com/releases/ .*/v?(\d+\.\d+)\.tar\.gz
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// let releases = entry.discover(|| "mypackage".to_string()).await.unwrap();
    /// for release in releases {
    ///     println!("{}: {}", release.version, release.url);
    /// }
    /// # }
    /// ```
    #[cfg(feature = "discover")]
    pub async fn discover(
        &self,
        package: impl FnOnce() -> String,
    ) -> Result<Vec<crate::Release>, Box<dyn std::error::Error>> {
        let url = self.format_url(package);
        let user_agent = self
            .user_agent()
            .unwrap_or_else(|| crate::DEFAULT_USER_AGENT.to_string());
        let searchmode = self.searchmode().unwrap_or(crate::SearchMode::Html);

        let client = reqwest::Client::builder().user_agent(user_agent).build()?;

        let response = client.get(url.as_str()).send().await?;
        let body = response.bytes().await?;

        // Apply pagemangle if present
        let mangled_body = self.apply_pagemangle(&body)?;

        let matching_pattern = self
            .matching_pattern()
            .ok_or("matching_pattern is required")?;

        let package_name = String::new(); // Not used in search currently
        let results = crate::search::search(
            match searchmode {
                crate::SearchMode::Html => "html",
                crate::SearchMode::Plain => "plain",
            },
            std::io::Cursor::new(mangled_body.as_ref() as &[u8]),
            &subst(&matching_pattern, || package_name.clone()),
            &package_name,
            url.as_str(),
        )?;

        let mut releases = Vec::new();
        for (version, full_url) in results {
            // Apply uversionmangle
            let mangled_version = self.apply_uversionmangle(&version)?;

            // Apply downloadurlmangle
            let mangled_url = self.apply_downloadurlmangle(&full_url)?;

            // Apply pgpsigurlmangle if present
            let pgpsigurl = if let Some(mangle) = self.pgpsigurlmangle() {
                Some(crate::mangle::apply_mangle(&mangle, &mangled_url)?)
            } else {
                None
            };

            // Apply filenamemangle if present
            let target_filename = if self.filenamemangle().is_some() {
                Some(self.apply_filenamemangle(&mangled_url)?)
            } else {
                None
            };

            // Apply oversionmangle if present
            let package_version = if self.oversionmangle().is_some() {
                Some(self.apply_oversionmangle(&mangled_version)?)
            } else {
                None
            };

            releases.push(crate::Release::new_full(
                mangled_version,
                mangled_url,
                pgpsigurl,
                target_filename,
                package_version,
            ));
        }

        Ok(releases)
    }

    /// Discover releases for this entry (blocking version)
    ///
    /// Fetches the URL and searches for version matches.
    /// Requires both 'discover' and 'blocking' features.
    ///
    /// # Examples
    ///
    /// ```ignore
    /// # use debian_watch::WatchFile;
    /// let wf: WatchFile = r#"version=4
    /// https://example.com/releases/ .*/v?(\d+\.\d+)\.tar\.gz
    /// "#.parse().unwrap();
    /// let entry = wf.entries().next().unwrap();
    /// let releases = entry.discover_blocking(|| "mypackage".to_string()).unwrap();
    /// for release in releases {
    ///     println!("{}: {}", release.version, release.url);
    /// }
    /// ```
    #[cfg(all(feature = "discover", feature = "blocking"))]
    pub fn discover_blocking(
        &self,
        package: impl FnOnce() -> String,
    ) -> Result<Vec<crate::Release>, Box<dyn std::error::Error>> {
        let url = self.format_url(package);
        let user_agent = self
            .user_agent()
            .unwrap_or_else(|| crate::DEFAULT_USER_AGENT.to_string());
        let searchmode = self.searchmode().unwrap_or(crate::SearchMode::Html);

        let client = reqwest::blocking::Client::builder()
            .user_agent(user_agent)
            .build()?;

        let response = client.get(url.as_str()).send()?;
        let body = response.bytes()?;

        // Apply pagemangle if present
        let mangled_body = self.apply_pagemangle(&body)?;

        let matching_pattern = self
            .matching_pattern()
            .ok_or("matching_pattern is required")?;

        let package_name = String::new(); // Not used in search currently
        let results = crate::search::search(
            match searchmode {
                crate::SearchMode::Html => "html",
                crate::SearchMode::Plain => "plain",
            },
            std::io::Cursor::new(mangled_body.as_ref() as &[u8]),
            &subst(&matching_pattern, || package_name.clone()),
            &package_name,
            url.as_str(),
        )?;

        let mut releases = Vec::new();
        for (version, full_url) in results {
            // Apply uversionmangle
            let mangled_version = self.apply_uversionmangle(&version)?;

            // Apply downloadurlmangle
            let mangled_url = self.apply_downloadurlmangle(&full_url)?;

            // Apply pgpsigurlmangle if present
            let pgpsigurl = if let Some(mangle) = self.pgpsigurlmangle() {
                Some(crate::mangle::apply_mangle(&mangle, &mangled_url)?)
            } else {
                None
            };

            // Apply filenamemangle if present
            let target_filename = if self.filenamemangle().is_some() {
                Some(self.apply_filenamemangle(&mangled_url)?)
            } else {
                None
            };

            // Apply oversionmangle if present
            let package_version = if self.oversionmangle().is_some() {
                Some(self.apply_oversionmangle(&mangled_version)?)
            } else {
                None
            };

            releases.push(crate::Release::new_full(
                mangled_version,
                mangled_url,
                pgpsigurl,
                target_filename,
                package_version,
            ));
        }

        Ok(releases)
    }

    /// Returns options set
    pub fn opts(&self) -> std::collections::HashMap<String, String> {
        let mut options = std::collections::HashMap::new();

        if let Some(ol) = self.option_list() {
            for opt in ol.children() {
                let key = opt.key();
                let value = opt.value();
                if let (Some(key), Some(value)) = (key, value) {
                    options.insert(key.to_string(), value.to_string());
                }
            }
        }

        options
    }

    fn items(&self) -> impl Iterator<Item = String> + '_ {
        self.0.children_with_tokens().filter_map(|it| match it {
            SyntaxElement::Token(token) => {
                if token.kind() == VALUE || token.kind() == KEY {
                    Some(token.text().to_string())
                } else {
                    None
                }
            }
            SyntaxElement::Node(node) => {
                // Extract values from entry field nodes
                match node.kind() {
                    URL => Url::cast(node).map(|n| n.url()),
                    MATCHING_PATTERN => MatchingPattern::cast(node).map(|n| n.pattern()),
                    VERSION_POLICY => VersionPolicyNode::cast(node).map(|n| n.policy()),
                    SCRIPT => ScriptNode::cast(node).map(|n| n.script()),
                    _ => None,
                }
            }
        })
    }

    /// Returns the URL of the entry.
    pub fn url(&self) -> String {
        self.0
            .children()
            .find_map(Url::cast)
            .map(|it| it.url())
            .unwrap_or_else(|| {
                // Fallback for entries without URL node (shouldn't happen with new parser)
                self.items().next().unwrap()
            })
    }

    /// Returns the matching pattern of the entry.
    pub fn matching_pattern(&self) -> Option<String> {
        self.0
            .children()
            .find_map(MatchingPattern::cast)
            .map(|it| it.pattern())
            .or_else(|| {
                // Fallback for entries without MATCHING_PATTERN node
                self.items().nth(1)
            })
    }

    /// Returns the version policy
    pub fn version(&self) -> Result<Option<crate::VersionPolicy>, String> {
        self.0
            .children()
            .find_map(VersionPolicyNode::cast)
            .map(|it| it.policy().parse())
            .transpose()
            .or_else(|_e| {
                // Fallback for entries without VERSION_POLICY node
                self.items().nth(2).map(|it| it.parse()).transpose()
            })
    }

    /// Returns the script of the entry.
    pub fn script(&self) -> Option<String> {
        self.0
            .children()
            .find_map(ScriptNode::cast)
            .map(|it| it.script())
            .or_else(|| {
                // Fallback for entries without SCRIPT node
                self.items().nth(3)
            })
    }

    /// Replace all substitutions and return the resulting URL.
    pub fn format_url(&self, package: impl FnOnce() -> String) -> url::Url {
        subst(self.url().as_str(), package).parse().unwrap()
    }

    /// Set the URL of the entry.
    pub fn set_url(&mut self, new_url: &str) {
        // Build the new URL node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(URL.into());
        builder.token(VALUE.into(), new_url);
        builder.finish_node();
        let new_url_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_url_node = SyntaxNode::new_root_mut(new_url_green);

        // Find existing URL node position (need to use children_with_tokens for correct indexing)
        let url_pos = self
            .0
            .children_with_tokens()
            .position(|child| matches!(child, SyntaxElement::Node(node) if node.kind() == URL));

        if let Some(pos) = url_pos {
            // Replace existing URL node
            self.0
                .splice_children(pos..pos + 1, vec![new_url_node.into()]);
        }
    }

    /// Set the matching pattern of the entry.
    ///
    /// TODO: This currently only replaces an existing matching pattern.
    /// If the entry doesn't have a matching pattern, this method does nothing.
    /// Future implementation should insert the node at the correct position.
    pub fn set_matching_pattern(&mut self, new_pattern: &str) {
        // Build the new MATCHING_PATTERN node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(MATCHING_PATTERN.into());
        builder.token(VALUE.into(), new_pattern);
        builder.finish_node();
        let new_pattern_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_pattern_node = SyntaxNode::new_root_mut(new_pattern_green);

        // Find existing MATCHING_PATTERN node position
        let pattern_pos = self.0.children_with_tokens().position(
            |child| matches!(child, SyntaxElement::Node(node) if node.kind() == MATCHING_PATTERN),
        );

        if let Some(pos) = pattern_pos {
            // Replace existing MATCHING_PATTERN node
            self.0
                .splice_children(pos..pos + 1, vec![new_pattern_node.into()]);
        }
        // TODO: else insert new node after URL
    }

    /// Set the version policy of the entry.
    ///
    /// TODO: This currently only replaces an existing version policy.
    /// If the entry doesn't have a version policy, this method does nothing.
    /// Future implementation should insert the node at the correct position.
    pub fn set_version_policy(&mut self, new_policy: &str) {
        // Build the new VERSION_POLICY node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(VERSION_POLICY.into());
        // Version policy can be KEY (e.g., "debian") or VALUE
        builder.token(VALUE.into(), new_policy);
        builder.finish_node();
        let new_policy_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_policy_node = SyntaxNode::new_root_mut(new_policy_green);

        // Find existing VERSION_POLICY node position
        let policy_pos = self.0.children_with_tokens().position(
            |child| matches!(child, SyntaxElement::Node(node) if node.kind() == VERSION_POLICY),
        );

        if let Some(pos) = policy_pos {
            // Replace existing VERSION_POLICY node
            self.0
                .splice_children(pos..pos + 1, vec![new_policy_node.into()]);
        }
        // TODO: else insert new node after MATCHING_PATTERN (or URL if no pattern)
    }

    /// Set the script of the entry.
    ///
    /// TODO: This currently only replaces an existing script.
    /// If the entry doesn't have a script, this method does nothing.
    /// Future implementation should insert the node at the correct position.
    pub fn set_script(&mut self, new_script: &str) {
        // Build the new SCRIPT node
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(SCRIPT.into());
        // Script can be KEY (e.g., "uupdate") or VALUE
        builder.token(VALUE.into(), new_script);
        builder.finish_node();
        let new_script_green = builder.finish();

        // Create a syntax node (splice_children will detach and reattach it)
        let new_script_node = SyntaxNode::new_root_mut(new_script_green);

        // Find existing SCRIPT node position
        let script_pos = self
            .0
            .children_with_tokens()
            .position(|child| matches!(child, SyntaxElement::Node(node) if node.kind() == SCRIPT));

        if let Some(pos) = script_pos {
            // Replace existing SCRIPT node
            self.0
                .splice_children(pos..pos + 1, vec![new_script_node.into()]);
        }
        // TODO: else insert new node after VERSION_POLICY (or MATCHING_PATTERN/URL if no policy)
    }

    /// Set or update an option value.
    ///
    /// If the option already exists, it will be updated with the new value.
    /// If the option doesn't exist, it will be added to the options list.
    /// If there's no options list, one will be created.
    pub fn set_opt(&mut self, key: &str, value: &str) {
        // Find the OPTS_LIST position in Entry
        let opts_pos = self.0.children_with_tokens().position(
            |child| matches!(child, SyntaxElement::Node(node) if node.kind() == OPTS_LIST),
        );

        if let Some(_opts_idx) = opts_pos {
            if let Some(mut ol) = self.option_list() {
                // Find if the option already exists
                if let Some(mut opt) = ol.find_option(key) {
                    // Update the existing option's value
                    opt.set_value(value);
                    // Mutations should propagate automatically - no need to replace
                } else {
                    // Add new option
                    ol.add_option(key, value);
                    // Mutations should propagate automatically - no need to replace
                }
            }
        } else {
            // Create a new options list
            let mut builder = GreenNodeBuilder::new();
            builder.start_node(OPTS_LIST.into());
            builder.token(KEY.into(), "opts");
            builder.token(EQUALS.into(), "=");
            builder.start_node(OPTION.into());
            builder.token(KEY.into(), key);
            builder.token(EQUALS.into(), "=");
            builder.token(VALUE.into(), value);
            builder.finish_node();
            builder.finish_node();
            let new_opts_green = builder.finish();
            let new_opts_node = SyntaxNode::new_root_mut(new_opts_green);

            // Find position to insert (before URL if it exists, otherwise at start)
            let url_pos = self
                .0
                .children_with_tokens()
                .position(|child| matches!(child, SyntaxElement::Node(node) if node.kind() == URL));

            if let Some(url_idx) = url_pos {
                // Insert options list and a space before the URL
                // Build a parent node containing both space and whitespace to extract from
                let mut combined_builder = GreenNodeBuilder::new();
                combined_builder.start_node(ROOT.into()); // Temporary parent
                combined_builder.token(WHITESPACE.into(), " ");
                combined_builder.finish_node();
                let temp_green = combined_builder.finish();
                let temp_root = SyntaxNode::new_root_mut(temp_green);
                let space_element = temp_root.children_with_tokens().next().unwrap();

                self.0
                    .splice_children(url_idx..url_idx, vec![new_opts_node.into(), space_element]);
            } else {
                self.0.splice_children(0..0, vec![new_opts_node.into()]);
            }
        }
    }

    /// Delete an option.
    ///
    /// Removes the option with the specified key from the options list.
    /// If the option doesn't exist, this method does nothing.
    /// If deleting the option results in an empty options list, the entire
    /// opts= declaration is removed.
    pub fn del_opt(&mut self, key: &str) {
        if let Some(mut ol) = self.option_list() {
            let option_count = ol.0.children().filter(|n| n.kind() == OPTION).count();

            if option_count == 1 && ol.has_option(key) {
                // This is the last option, remove the entire OPTS_LIST from Entry
                let opts_pos = self.0.children().position(|node| node.kind() == OPTS_LIST);

                if let Some(opts_idx) = opts_pos {
                    // Remove the OPTS_LIST
                    self.0.splice_children(opts_idx..opts_idx + 1, vec![]);

                    // Remove any leading whitespace/continuation that was after the OPTS_LIST
                    while self.0.children_with_tokens().next().map_or(false, |e| {
                        matches!(
                            e,
                            SyntaxElement::Token(t) if t.kind() == WHITESPACE || t.kind() == CONTINUATION
                        )
                    }) {
                        self.0.splice_children(0..1, vec![]);
                    }
                }
            } else {
                // Defer to OptionList to remove the option
                ol.remove_option(key);
            }
        }
    }
}

const SUBSTITUTIONS: &[(&str, &str)] = &[
    // This is substituted with the source package name found in the first line
    // of the debian/changelog file.
    // "@PACKAGE@": None,
    // This is substituted by the legal upstream version regex (capturing).
    ("@ANY_VERSION@", r"[-_]?(\d[\-+\.:\~\da-zA-Z]*)"),
    // This is substituted by the typical archive file extension regex
    // (non-capturing).
    (
        "@ARCHIVE_EXT@",
        r"(?i)\.(?:tar\.xz|tar\.bz2|tar\.gz|zip|tgz|tbz|txz)",
    ),
    // This is substituted by the typical signature file extension regex
    // (non-capturing).
    (
        "@SIGNATURE_EXT@",
        r"(?i)\.(?:tar\.xz|tar\.bz2|tar\.gz|zip|tgz|tbz|txz)\.(?:asc|pgp|gpg|sig|sign)",
    ),
    // This is substituted by the typical Debian extension regexp (capturing).
    ("@DEB_EXT@", r"[\+~](debian|dfsg|ds|deb)(\.)?(\d+)?$"),
];

pub fn subst(text: &str, package: impl FnOnce() -> String) -> String {
    let mut substs = SUBSTITUTIONS.to_vec();
    let package_name;
    if text.contains("@PACKAGE@") {
        package_name = Some(package());
        substs.push(("@PACKAGE@", package_name.as_deref().unwrap()));
    }

    let mut text = text.to_string();

    for (k, v) in substs {
        text = text.replace(k, v);
    }

    text
}

#[test]
fn test_subst() {
    assert_eq!(
        subst("@ANY_VERSION@", || unreachable!()),
        r"[-_]?(\d[\-+\.:\~\da-zA-Z]*)"
    );
    assert_eq!(subst("@PACKAGE@", || "dulwich".to_string()), "dulwich");
}

impl std::fmt::Debug for OptionList {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("OptionList")
            .field("text", &self.0.text().to_string())
            .finish()
    }
}

impl OptionList {
    fn children(&self) -> impl Iterator<Item = _Option> + '_ {
        self.0.children().filter_map(_Option::cast)
    }

    pub fn has_option(&self, key: &str) -> bool {
        self.children().any(|it| it.key().as_deref() == Some(key))
    }

    pub fn get_option(&self, key: &str) -> Option<String> {
        for child in self.children() {
            if child.key().as_deref() == Some(key) {
                return child.value();
            }
        }
        None
    }

    /// Find an option by key.
    fn find_option(&self, key: &str) -> Option<_Option> {
        self.children()
            .find(|opt| opt.key().as_deref() == Some(key))
    }

    /// Add a new option to the end of the options list.
    fn add_option(&mut self, key: &str, value: &str) {
        let option_count = self.0.children().filter(|n| n.kind() == OPTION).count();

        // Build a structure containing separator (if needed) + option wrapped in a temporary parent
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(ROOT.into()); // Temporary parent

        if option_count > 0 {
            builder.start_node(OPTION_SEPARATOR.into());
            builder.token(COMMA.into(), ",");
            builder.finish_node();
        }

        builder.start_node(OPTION.into());
        builder.token(KEY.into(), key);
        builder.token(EQUALS.into(), "=");
        builder.token(VALUE.into(), value);
        builder.finish_node();

        builder.finish_node(); // Close temporary parent
        let combined_green = builder.finish();

        // Create a temporary root to extract children from
        let temp_root = SyntaxNode::new_root_mut(combined_green);
        let new_children: Vec<_> = temp_root.children_with_tokens().collect();

        let insert_pos = self.0.children_with_tokens().count();
        self.0.splice_children(insert_pos..insert_pos, new_children);
    }

    /// Remove an option by key. Returns true if an option was removed.
    fn remove_option(&mut self, key: &str) -> bool {
        if let Some(mut opt) = self.find_option(key) {
            opt.remove();
            true
        } else {
            false
        }
    }
}

impl _Option {
    /// Returns the key of the option.
    pub fn key(&self) -> Option<String> {
        self.0.children_with_tokens().find_map(|it| match it {
            SyntaxElement::Token(token) => {
                if token.kind() == KEY {
                    Some(token.text().to_string())
                } else {
                    None
                }
            }
            _ => None,
        })
    }

    /// Returns the value of the option.
    pub fn value(&self) -> Option<String> {
        self.0
            .children_with_tokens()
            .filter_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE || token.kind() == KEY {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .nth(1)
    }

    /// Set the value of the option.
    pub fn set_value(&mut self, new_value: &str) {
        let key = self.key().expect("Option must have a key");

        // Build a new OPTION node with the updated value
        let mut builder = GreenNodeBuilder::new();
        builder.start_node(OPTION.into());
        builder.token(KEY.into(), &key);
        builder.token(EQUALS.into(), "=");
        builder.token(VALUE.into(), new_value);
        builder.finish_node();
        let new_option_green = builder.finish();
        let new_option_node = SyntaxNode::new_root_mut(new_option_green);

        // Replace this option in the parent OptionList
        if let Some(parent) = self.0.parent() {
            let idx = self.0.index();
            parent.splice_children(idx..idx + 1, vec![new_option_node.into()]);
        }
    }

    /// Remove this option and its associated separator from the parent OptionList.
    pub fn remove(&mut self) {
        // Find adjacent separator to remove before detaching this node
        let next_sep = self
            .0
            .next_sibling()
            .filter(|n| n.kind() == OPTION_SEPARATOR);
        let prev_sep = self
            .0
            .prev_sibling()
            .filter(|n| n.kind() == OPTION_SEPARATOR);

        // Detach separator first if it exists
        if let Some(sep) = next_sep {
            sep.detach();
        } else if let Some(sep) = prev_sep {
            sep.detach();
        }

        // Now detach the option itself
        self.0.detach();
    }
}

impl Url {
    /// Returns the URL string.
    pub fn url(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

impl MatchingPattern {
    /// Returns the matching pattern string.
    pub fn pattern(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    if token.kind() == VALUE {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

impl VersionPolicyNode {
    /// Returns the version policy string.
    pub fn policy(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    // Can be KEY (e.g., "debian") or VALUE
                    if token.kind() == VALUE || token.kind() == KEY {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

impl ScriptNode {
    /// Returns the script string.
    pub fn script(&self) -> String {
        self.0
            .children_with_tokens()
            .find_map(|it| match it {
                SyntaxElement::Token(token) => {
                    // Can be KEY (e.g., "uupdate") or VALUE
                    if token.kind() == VALUE || token.kind() == KEY {
                        Some(token.text().to_string())
                    } else {
                        None
                    }
                }
                _ => None,
            })
            .unwrap()
    }
}

#[test]
fn test_entry_node_structure() {
    // Test that entries properly use the new node types
    let wf: super::WatchFile = r#"version=4
opts=compression=xz https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let entry = wf.entries().next().unwrap();

    // Verify URL node exists and works
    assert_eq!(entry.0.children().find(|n| n.kind() == URL).is_some(), true);
    assert_eq!(entry.url(), "https://example.com/releases");

    // Verify MATCHING_PATTERN node exists and works
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == MATCHING_PATTERN)
            .is_some(),
        true
    );
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );

    // Verify VERSION_POLICY node exists and works
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == VERSION_POLICY)
            .is_some(),
        true
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    // Verify SCRIPT node exists and works
    assert_eq!(
        entry.0.children().find(|n| n.kind() == SCRIPT).is_some(),
        true
    );
    assert_eq!(entry.script(), Some("uupdate".into()));
}

#[test]
fn test_entry_node_structure_partial() {
    // Test entry with only URL and pattern (no version or script)
    let wf: super::WatchFile = r#"version=4
https://github.com/example/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let entry = wf.entries().next().unwrap();

    // Should have URL and MATCHING_PATTERN nodes
    assert_eq!(entry.0.children().find(|n| n.kind() == URL).is_some(), true);
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == MATCHING_PATTERN)
            .is_some(),
        true
    );

    // Should NOT have VERSION_POLICY or SCRIPT nodes
    assert_eq!(
        entry
            .0
            .children()
            .find(|n| n.kind() == VERSION_POLICY)
            .is_some(),
        false
    );
    assert_eq!(
        entry.0.children().find(|n| n.kind() == SCRIPT).is_some(),
        false
    );

    // Verify accessors work correctly
    assert_eq!(entry.url(), "https://github.com/example/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);
}

#[test]
fn test_parse_v1() {
    const WATCHV1: &str = r#"version=4
opts=bare,filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/syncthing-gtk-$1\.tar\.gz/ \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#;
    let parsed = parse(WATCHV1);
    //assert_eq!(parsed.errors, Vec::<String>::new());
    let node = parsed.syntax();
    assert_eq!(
        format!("{:#?}", node),
        r#"ROOT@0..161
  VERSION@0..10
    KEY@0..7 "version"
    EQUALS@7..8 "="
    VALUE@8..9 "4"
    NEWLINE@9..10 "\n"
  ENTRY@10..161
    OPTS_LIST@10..86
      KEY@10..14 "opts"
      EQUALS@14..15 "="
      OPTION@15..19
        KEY@15..19 "bare"
      OPTION_SEPARATOR@19..20
        COMMA@19..20 ","
      OPTION@20..86
        KEY@20..34 "filenamemangle"
        EQUALS@34..35 "="
        VALUE@35..86 "s/.+\\/v?(\\d\\S+)\\.tar\\ ..."
    WHITESPACE@86..87 " "
    CONTINUATION@87..89 "\\\n"
    WHITESPACE@89..91 "  "
    URL@91..138
      VALUE@91..138 "https://github.com/sy ..."
    WHITESPACE@138..139 " "
    MATCHING_PATTERN@139..160
      VALUE@139..160 ".*/v?(\\d\\S+)\\.tar\\.gz"
    NEWLINE@160..161 "\n"
"#
    );

    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);

    assert_eq!(node.text(), WATCHV1);
}

#[test]
fn test_parse_v2() {
    let parsed = parse(
        r#"version=4
https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
# comment
"#,
    );
    assert_eq!(parsed.errors, Vec::<String>::new());
    let node = parsed.syntax();
    assert_eq!(
        format!("{:#?}", node),
        r###"ROOT@0..90
  VERSION@0..10
    KEY@0..7 "version"
    EQUALS@7..8 "="
    VALUE@8..9 "4"
    NEWLINE@9..10 "\n"
  ENTRY@10..80
    URL@10..57
      VALUE@10..57 "https://github.com/sy ..."
    WHITESPACE@57..58 " "
    MATCHING_PATTERN@58..79
      VALUE@58..79 ".*/v?(\\d\\S+)\\.tar\\.gz"
    NEWLINE@79..80 "\n"
  COMMENT@80..89 "# comment"
  NEWLINE@89..90 "\n"
"###
    );

    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );
    assert_eq!(
        entry.format_url(|| "syncthing-gtk".to_string()),
        "https://github.com/syncthing/syncthing-gtk/tags"
            .parse()
            .unwrap()
    );
}

#[test]
fn test_parse_v3() {
    let parsed = parse(
        r#"version=4
https://github.com/syncthing/@PACKAGE@/tags .*/v?(\d\S+)\.tar\.gz
# comment
"#,
    );
    assert_eq!(parsed.errors, Vec::<String>::new());
    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(entry.url(), "https://github.com/syncthing/@PACKAGE@/tags");
    assert_eq!(
        entry.format_url(|| "syncthing-gtk".to_string()),
        "https://github.com/syncthing/syncthing-gtk/tags"
            .parse()
            .unwrap()
    );
}

#[test]
fn test_parse_v4() {
    let cl: super::WatchFile = r#"version=4
opts=repack,compression=xz,dversionmangle=s/\+ds//,repacksuffix=+ds \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();
    assert_eq!(cl.version(), 4);
    let entries = cl.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(entry.url(), "https://github.com/example/example-cat/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert!(entry.repack());
    assert_eq!(entry.compression(), Ok(Some(Compression::Xz)));
    assert_eq!(entry.dversionmangle(), Some("s/\\+ds//".into()));
    assert_eq!(entry.repacksuffix(), Some("+ds".into()));
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert_eq!(
        entry.format_url(|| "example-cat".to_string()),
        "https://github.com/example/example-cat/tags"
            .parse()
            .unwrap()
    );
    assert_eq!(entry.version(), Ok(Some(VersionPolicy::Debian)));
}

#[test]
fn test_git_mode() {
    let text = r#"version=3
opts="mode=git, gitmode=shallow, pgpmode=gittag" \
https://git.kernel.org/pub/scm/linux/kernel/git/firmware/linux-firmware.git \
refs/tags/(.*) debian
"#;
    let parsed = parse(text);
    assert_eq!(parsed.errors, Vec::<String>::new());
    let cl = parsed.root();
    assert_eq!(cl.version(), 3);
    let entries = cl.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];
    assert_eq!(
        entry.url(),
        "https://git.kernel.org/pub/scm/linux/kernel/git/firmware/linux-firmware.git"
    );
    assert_eq!(entry.matching_pattern(), Some("refs/tags/(.*)".into()));
    assert_eq!(entry.version(), Ok(Some(VersionPolicy::Debian)));
    assert_eq!(entry.script(), None);
    assert_eq!(entry.gitmode(), Ok(GitMode::Shallow));
    assert_eq!(entry.pgpmode(), Ok(PgpMode::GitTag));
    assert_eq!(entry.mode(), Ok(Mode::Git));
}

#[test]
fn test_parse_quoted() {
    const WATCHV1: &str = r#"version=4
opts="bare, filenamemangle=blah" \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#;
    let parsed = parse(WATCHV1);
    //assert_eq!(parsed.errors, Vec::<String>::new());
    let node = parsed.syntax();

    let root = parsed.root();
    assert_eq!(root.version(), 4);
    let entries = root.entries().collect::<Vec<_>>();
    assert_eq!(entries.len(), 1);
    let entry = &entries[0];

    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);

    assert_eq!(node.text(), WATCHV1);
}

#[test]
fn test_set_url() {
    // Test setting URL on a simple entry without options
    let wf: super::WatchFile = r#"version=4
https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );

    entry.set_url("https://newurl.example.org/path");
    assert_eq!(entry.url(), "https://newurl.example.org/path");
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://newurl.example.org/path .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_url_with_options() {
    // Test setting URL on an entry with options
    let wf: super::WatchFile = r#"version=4
opts=foo=blah https://foo.com/bar .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.url(), "https://foo.com/bar");
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));

    entry.set_url("https://example.com/baz");
    assert_eq!(entry.url(), "https://example.com/baz");

    // Verify options are preserved
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=foo=blah https://example.com/baz .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_url_complex() {
    // Test with a complex watch file with multiple options and continuation
    let wf: super::WatchFile = r#"version=4
opts=bare,filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/syncthing-gtk-$1\.tar\.gz/ \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );

    entry.set_url("https://gitlab.com/newproject/tags");
    assert_eq!(entry.url(), "https://gitlab.com/newproject/tags");

    // Verify all options are preserved
    assert!(entry.bare());
    assert_eq!(
        entry.filenamemangle(),
        Some("s/.+\\/v?(\\d\\S+)\\.tar\\.gz/syncthing-gtk-$1\\.tar\\.gz/".into())
    );
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    // Verify the exact serialized output preserves structure
    assert_eq!(
        entry.to_string(),
        r#"opts=bare,filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/syncthing-gtk-$1\.tar\.gz/ \
  https://gitlab.com/newproject/tags .*/v?(\d\S+)\.tar\.gz
"#
    );
}

#[test]
fn test_set_url_with_all_fields() {
    // Test with all fields: options, URL, matching pattern, version, and script
    let wf: super::WatchFile = r#"version=4
opts=repack,compression=xz,dversionmangle=s/\+ds//,repacksuffix=+ds \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.url(), "https://github.com/example/example-cat/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));

    entry.set_url("https://gitlab.example.org/project/releases");
    assert_eq!(entry.url(), "https://gitlab.example.org/project/releases");

    // Verify all other fields are preserved
    assert!(entry.repack());
    assert_eq!(entry.compression(), Ok(Some(super::Compression::Xz)));
    assert_eq!(entry.dversionmangle(), Some("s/\\+ds//".into()));
    assert_eq!(entry.repacksuffix(), Some("+ds".into()));
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        r#"opts=repack,compression=xz,dversionmangle=s/\+ds//,repacksuffix=+ds \
    https://gitlab.example.org/project/releases \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    );
}

#[test]
fn test_set_url_quoted_options() {
    // Test with quoted options
    let wf: super::WatchFile = r#"version=4
opts="bare, filenamemangle=blah" \
  https://github.com/syncthing/syncthing-gtk/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.url(),
        "https://github.com/syncthing/syncthing-gtk/tags"
    );

    entry.set_url("https://example.org/new/path");
    assert_eq!(entry.url(), "https://example.org/new/path");

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        r#"opts="bare, filenamemangle=blah" \
  https://example.org/new/path .*/v?(\d\S+)\.tar\.gz
"#
    );
}

#[test]
fn test_set_opt_update_existing() {
    // Test updating an existing option
    let wf: super::WatchFile = r#"version=4
opts=foo=blah,bar=baz https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(entry.get_option("bar"), Some("baz".to_string()));

    entry.set_opt("foo", "updated");
    assert_eq!(entry.get_option("foo"), Some("updated".to_string()));
    assert_eq!(entry.get_option("bar"), Some("baz".to_string()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=foo=updated,bar=baz https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_opt_add_new() {
    // Test adding a new option to existing options
    let wf: super::WatchFile = r#"version=4
opts=foo=blah https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(entry.get_option("bar"), None);

    entry.set_opt("bar", "baz");
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(entry.get_option("bar"), Some("baz".to_string()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=foo=blah,bar=baz https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_opt_create_options_list() {
    // Test creating a new options list when none exists
    let wf: super::WatchFile = r#"version=4
https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.option_list(), None);

    entry.set_opt("compression", "xz");
    assert_eq!(entry.get_option("compression"), Some("xz".to_string()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=compression=xz https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_del_opt_remove_single() {
    // Test removing a single option from multiple options
    let wf: super::WatchFile = r#"version=4
opts=foo=blah,bar=baz,qux=quux https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(entry.get_option("bar"), Some("baz".to_string()));
    assert_eq!(entry.get_option("qux"), Some("quux".to_string()));

    entry.del_opt("bar");
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(entry.get_option("bar"), None);
    assert_eq!(entry.get_option("qux"), Some("quux".to_string()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=foo=blah,qux=quux https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_del_opt_remove_first() {
    // Test removing the first option
    let wf: super::WatchFile = r#"version=4
opts=foo=blah,bar=baz https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    entry.del_opt("foo");
    assert_eq!(entry.get_option("foo"), None);
    assert_eq!(entry.get_option("bar"), Some("baz".to_string()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=bar=baz https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_del_opt_remove_last() {
    // Test removing the last option
    let wf: super::WatchFile = r#"version=4
opts=foo=blah,bar=baz https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    entry.del_opt("bar");
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));
    assert_eq!(entry.get_option("bar"), None);

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=foo=blah https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_del_opt_remove_only_option() {
    // Test removing the only option (should remove entire opts list)
    let wf: super::WatchFile = r#"version=4
opts=foo=blah https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.get_option("foo"), Some("blah".to_string()));

    entry.del_opt("foo");
    assert_eq!(entry.get_option("foo"), None);
    assert_eq!(entry.option_list(), None);

    // Verify the exact serialized output (opts should be gone)
    assert_eq!(
        entry.to_string(),
        "https://example.com/releases .*/v?(\\d\\S+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_del_opt_nonexistent() {
    // Test deleting a non-existent option (should do nothing)
    let wf: super::WatchFile = r#"version=4
opts=foo=blah https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    let original = entry.to_string();

    entry.del_opt("nonexistent");
    assert_eq!(entry.to_string(), original);
}

#[test]
fn test_set_opt_multiple_operations() {
    // Test multiple set_opt operations
    let wf: super::WatchFile = r#"version=4
https://example.com/releases .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();

    entry.set_opt("compression", "xz");
    entry.set_opt("repack", "");
    entry.set_opt("dversionmangle", "s/\\+ds//");

    assert_eq!(entry.get_option("compression"), Some("xz".to_string()));
    assert_eq!(
        entry.get_option("dversionmangle"),
        Some("s/\\+ds//".to_string())
    );
}

#[test]
fn test_set_matching_pattern() {
    // Test setting matching pattern on a simple entry
    let wf: super::WatchFile = r#"version=4
https://github.com/example/tags .*/v?(\d\S+)\.tar\.gz
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz".into())
    );

    entry.set_matching_pattern("(?:.*?/)?v?([\\d.]+)\\.tar\\.gz");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?([\\d.]+)\\.tar\\.gz".into())
    );

    // Verify URL is preserved
    assert_eq!(entry.url(), "https://github.com/example/tags");

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://github.com/example/tags (?:.*?/)?v?([\\d.]+)\\.tar\\.gz\n"
    );
}

#[test]
fn test_set_matching_pattern_with_all_fields() {
    // Test with all fields present
    let wf: super::WatchFile = r#"version=4
opts=compression=xz https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );

    entry.set_matching_pattern(".*/version-([\\d.]+)\\.tar\\.xz");
    assert_eq!(
        entry.matching_pattern(),
        Some(".*/version-([\\d.]+)\\.tar\\.xz".into())
    );

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert_eq!(entry.compression(), Ok(Some(super::Compression::Xz)));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=compression=xz https://example.com/releases .*/version-([\\d.]+)\\.tar\\.xz debian uupdate\n"
    );
}

#[test]
fn test_set_version_policy() {
    // Test setting version policy
    let wf: super::WatchFile = r#"version=4
https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    entry.set_version_policy("previous");
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Previous)));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.script(), Some("uupdate".into()));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://example.com/releases (?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz previous uupdate\n"
    );
}

#[test]
fn test_set_version_policy_with_options() {
    // Test with options and continuation
    let wf: super::WatchFile = r#"version=4
opts=repack,compression=xz \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    entry.set_version_policy("ignore");
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Ignore)));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://github.com/example/example-cat/tags");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert!(entry.repack());

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        r#"opts=repack,compression=xz \
    https://github.com/example/example-cat/tags \
        (?:.*?/)?v?(\d[\d.]*)\.tar\.gz ignore uupdate
"#
    );
}

#[test]
fn test_set_script() {
    // Test setting script
    let wf: super::WatchFile = r#"version=4
https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.script(), Some("uupdate".into()));

    entry.set_script("uscan");
    assert_eq!(entry.script(), Some("uscan".into()));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "https://example.com/releases (?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz debian uscan\n"
    );
}

#[test]
fn test_set_script_with_options() {
    // Test with options
    let wf: super::WatchFile = r#"version=4
opts=compression=xz https://example.com/releases (?:.*?/)?v?(\d[\d.]*)\.tar\.gz debian uupdate
"#
    .parse()
    .unwrap();

    let mut entry = wf.entries().next().unwrap();
    assert_eq!(entry.script(), Some("uupdate".into()));

    entry.set_script("custom-script.sh");
    assert_eq!(entry.script(), Some("custom-script.sh".into()));

    // Verify all other fields are preserved
    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(
        entry.matching_pattern(),
        Some("(?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz".into())
    );
    assert_eq!(entry.version(), Ok(Some(super::VersionPolicy::Debian)));
    assert_eq!(entry.compression(), Ok(Some(super::Compression::Xz)));

    // Verify the exact serialized output
    assert_eq!(
        entry.to_string(),
        "opts=compression=xz https://example.com/releases (?:.*?/)?v?(\\d[\\d.]*)\\.tar\\.gz debian custom-script.sh\n"
    );
}

#[test]
fn test_apply_dversionmangle() {
    // Test basic dversionmangle
    let wf: super::WatchFile = r#"version=4
opts=dversionmangle=s/\+dfsg$// https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dversionmangle("1.0+dfsg").unwrap(), "1.0");
    assert_eq!(entry.apply_dversionmangle("1.0").unwrap(), "1.0");

    // Test with versionmangle (fallback)
    let wf: super::WatchFile = r#"version=4
opts=versionmangle=s/^v// https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dversionmangle("v1.0").unwrap(), "1.0");

    // Test with both dversionmangle and versionmangle (dversionmangle takes precedence)
    let wf: super::WatchFile = r#"version=4
opts=dversionmangle=s/\+ds//,versionmangle=s/^v// https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dversionmangle("1.0+ds").unwrap(), "1.0");

    // Test without any mangle options
    let wf: super::WatchFile = r#"version=4
https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dversionmangle("1.0+dfsg").unwrap(), "1.0+dfsg");
}

#[test]
fn test_apply_oversionmangle() {
    // Test basic oversionmangle - adding suffix
    let wf: super::WatchFile = r#"version=4
opts=oversionmangle=s/$/-1/ https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_oversionmangle("1.0").unwrap(), "1.0-1");
    assert_eq!(entry.apply_oversionmangle("2.5.3").unwrap(), "2.5.3-1");

    // Test oversionmangle for adding +dfsg suffix
    let wf: super::WatchFile = r#"version=4
opts=oversionmangle=s/$/.dfsg/ https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_oversionmangle("1.0").unwrap(), "1.0.dfsg");

    // Test without any mangle options
    let wf: super::WatchFile = r#"version=4
https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_oversionmangle("1.0").unwrap(), "1.0");
}

#[test]
fn test_apply_dirversionmangle() {
    // Test basic dirversionmangle - removing 'v' prefix
    let wf: super::WatchFile = r#"version=4
opts=dirversionmangle=s/^v// https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dirversionmangle("v1.0").unwrap(), "1.0");
    assert_eq!(entry.apply_dirversionmangle("v2.5.3").unwrap(), "2.5.3");

    // Test dirversionmangle with capture groups
    let wf: super::WatchFile = r#"version=4
opts=dirversionmangle=s/v(\d)/$1/ https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dirversionmangle("v1.0").unwrap(), "1.0");

    // Test without any mangle options
    let wf: super::WatchFile = r#"version=4
https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_dirversionmangle("v1.0").unwrap(), "v1.0");
}

#[test]
fn test_apply_filenamemangle() {
    // Test filenamemangle to generate tarball filename
    let wf: super::WatchFile = r#"version=4
opts=filenamemangle=s/.+\/v?(\d\S+)\.tar\.gz/mypackage-$1.tar.gz/ https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry
            .apply_filenamemangle("https://example.com/v1.0.tar.gz")
            .unwrap(),
        "mypackage-1.0.tar.gz"
    );
    assert_eq!(
        entry
            .apply_filenamemangle("https://example.com/2.5.3.tar.gz")
            .unwrap(),
        "mypackage-2.5.3.tar.gz"
    );

    // Test filenamemangle with different pattern
    let wf: super::WatchFile = r#"version=4
opts=filenamemangle=s/.*\/(.*)/$1/ https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry
            .apply_filenamemangle("https://example.com/path/to/file.tar.gz")
            .unwrap(),
        "file.tar.gz"
    );

    // Test without any mangle options
    let wf: super::WatchFile = r#"version=4
https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry
            .apply_filenamemangle("https://example.com/file.tar.gz")
            .unwrap(),
        "https://example.com/file.tar.gz"
    );
}

#[test]
fn test_apply_pagemangle() {
    // Test pagemangle to decode HTML entities
    let wf: super::WatchFile = r#"version=4
opts=pagemangle=s/&amp;/&/g https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.apply_pagemangle(b"foo &amp; bar").unwrap(),
        b"foo & bar"
    );
    assert_eq!(
        entry
            .apply_pagemangle(b"&amp; foo &amp; bar &amp;")
            .unwrap(),
        b"& foo & bar &"
    );

    // Test pagemangle with different pattern
    let wf: super::WatchFile = r#"version=4
opts=pagemangle=s/<[^>]+>//g https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(entry.apply_pagemangle(b"<div>text</div>").unwrap(), b"text");

    // Test without any mangle options
    let wf: super::WatchFile = r#"version=4
https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry.apply_pagemangle(b"foo &amp; bar").unwrap(),
        b"foo &amp; bar"
    );
}

#[test]
fn test_apply_downloadurlmangle() {
    // Test downloadurlmangle to change URL path
    let wf: super::WatchFile = r#"version=4
opts=downloadurlmangle=s|/archive/|/download/| https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry
            .apply_downloadurlmangle("https://example.com/archive/file.tar.gz")
            .unwrap(),
        "https://example.com/download/file.tar.gz"
    );

    // Test downloadurlmangle with different pattern
    let wf: super::WatchFile = r#"version=4
opts=downloadurlmangle=s/github\.com/raw.githubusercontent.com/ https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry
            .apply_downloadurlmangle("https://github.com/user/repo/file.tar.gz")
            .unwrap(),
        "https://raw.githubusercontent.com/user/repo/file.tar.gz"
    );

    // Test without any mangle options
    let wf: super::WatchFile = r#"version=4
https://example.com/ .*
"#
    .parse()
    .unwrap();
    let entry = wf.entries().next().unwrap();
    assert_eq!(
        entry
            .apply_downloadurlmangle("https://example.com/archive/file.tar.gz")
            .unwrap(),
        "https://example.com/archive/file.tar.gz"
    );
}

#[test]
fn test_entry_builder_minimal() {
    // Test creating a minimal entry with just URL and pattern
    let entry = super::EntryBuilder::new("https://github.com/example/tags")
        .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
        .build();

    assert_eq!(entry.url(), "https://github.com/example/tags");
    assert_eq!(
        entry.matching_pattern().as_deref(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz")
    );
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);
    assert!(entry.opts().is_empty());
}

#[test]
fn test_entry_builder_url_only() {
    // Test creating an entry with just URL
    let entry = super::EntryBuilder::new("https://example.com/releases").build();

    assert_eq!(entry.url(), "https://example.com/releases");
    assert_eq!(entry.matching_pattern(), None);
    assert_eq!(entry.version(), Ok(None));
    assert_eq!(entry.script(), None);
    assert!(entry.opts().is_empty());
}

#[test]
fn test_entry_builder_with_all_fields() {
    // Test creating an entry with all fields
    let entry = super::EntryBuilder::new("https://github.com/example/tags")
        .matching_pattern(".*/v?(\\d[\\d.]*)\\.tar\\.gz")
        .version_policy("debian")
        .script("uupdate")
        .opt("compression", "xz")
        .flag("repack")
        .build();

    assert_eq!(entry.url(), "https://github.com/example/tags");
    assert_eq!(
        entry.matching_pattern().as_deref(),
        Some(".*/v?(\\d[\\d.]*)\\.tar\\.gz")
    );
    assert_eq!(entry.version(), Ok(Some(VersionPolicy::Debian)));
    assert_eq!(entry.script(), Some("uupdate".into()));
    assert_eq!(entry.get_option("compression"), Some("xz".to_string()));
    assert!(entry.has_option("repack"));
    assert!(entry.repack());
}

#[test]
fn test_entry_builder_multiple_options() {
    // Test creating an entry with multiple options
    let entry = super::EntryBuilder::new("https://example.com/tags")
        .matching_pattern(".*/v?(\\d+\\.\\d+)\\.tar\\.gz")
        .opt("compression", "xz")
        .opt("dversionmangle", "s/\\+ds//")
        .opt("repacksuffix", "+ds")
        .build();

    assert_eq!(entry.get_option("compression"), Some("xz".to_string()));
    assert_eq!(
        entry.get_option("dversionmangle"),
        Some("s/\\+ds//".to_string())
    );
    assert_eq!(entry.get_option("repacksuffix"), Some("+ds".to_string()));
}

#[test]
fn test_entry_builder_via_entry() {
    // Test using Entry::builder() convenience method
    let entry = super::Entry::builder("https://github.com/example/tags")
        .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
        .version_policy("debian")
        .build();

    assert_eq!(entry.url(), "https://github.com/example/tags");
    assert_eq!(
        entry.matching_pattern().as_deref(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz")
    );
    assert_eq!(entry.version(), Ok(Some(VersionPolicy::Debian)));
}

#[test]
fn test_watchfile_add_entry_to_empty() {
    // Test adding an entry to an empty watchfile
    let mut wf = super::WatchFile::new(Some(4));

    let entry = super::EntryBuilder::new("https://github.com/example/tags")
        .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
        .build();

    wf.add_entry(entry);

    assert_eq!(wf.version(), 4);
    assert_eq!(wf.entries().count(), 1);

    let added_entry = wf.entries().next().unwrap();
    assert_eq!(added_entry.url(), "https://github.com/example/tags");
    assert_eq!(
        added_entry.matching_pattern().as_deref(),
        Some(".*/v?(\\d\\S+)\\.tar\\.gz")
    );
}

#[test]
fn test_watchfile_add_multiple_entries() {
    // Test adding multiple entries to a watchfile
    let mut wf = super::WatchFile::new(Some(4));

    wf.add_entry(
        super::EntryBuilder::new("https://github.com/example1/tags")
            .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
            .build(),
    );

    wf.add_entry(
        super::EntryBuilder::new("https://github.com/example2/releases")
            .matching_pattern(".*/(\\d+\\.\\d+)\\.tar\\.gz")
            .opt("compression", "xz")
            .build(),
    );

    assert_eq!(wf.entries().count(), 2);

    let entries: Vec<_> = wf.entries().collect();
    assert_eq!(entries[0].url(), "https://github.com/example1/tags");
    assert_eq!(entries[1].url(), "https://github.com/example2/releases");
    assert_eq!(entries[1].get_option("compression"), Some("xz".to_string()));
}

#[test]
fn test_watchfile_add_entry_to_existing() {
    // Test adding an entry to a watchfile that already has entries
    let mut wf: super::WatchFile = r#"version=4
https://example.com/old .*/v?(\\d\\S+)\\.tar\\.gz
"#
    .parse()
    .unwrap();

    assert_eq!(wf.entries().count(), 1);

    wf.add_entry(
        super::EntryBuilder::new("https://github.com/example/new")
            .matching_pattern(".*/v?(\\d+\\.\\d+)\\.tar\\.gz")
            .opt("compression", "xz")
            .version_policy("debian")
            .build(),
    );

    assert_eq!(wf.entries().count(), 2);

    let entries: Vec<_> = wf.entries().collect();
    assert_eq!(entries[0].url(), "https://example.com/old");
    assert_eq!(entries[1].url(), "https://github.com/example/new");
    assert_eq!(entries[1].version(), Ok(Some(VersionPolicy::Debian)));
}

#[test]
fn test_entry_builder_formatting() {
    // Test that the builder produces correctly formatted entries
    let entry = super::EntryBuilder::new("https://github.com/example/tags")
        .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
        .opt("compression", "xz")
        .flag("repack")
        .version_policy("debian")
        .script("uupdate")
        .build();

    let entry_str = entry.to_string();

    // Should start with opts=
    assert!(entry_str.starts_with("opts="));
    // Should contain the URL
    assert!(entry_str.contains("https://github.com/example/tags"));
    // Should contain the pattern
    assert!(entry_str.contains(".*/v?(\\d\\S+)\\.tar\\.gz"));
    // Should contain version policy
    assert!(entry_str.contains("debian"));
    // Should contain script
    assert!(entry_str.contains("uupdate"));
    // Should end with newline
    assert!(entry_str.ends_with('\n'));
}

#[test]
fn test_watchfile_add_entry_preserves_format() {
    // Test that adding entries preserves the watchfile format
    let mut wf = super::WatchFile::new(Some(4));

    wf.add_entry(
        super::EntryBuilder::new("https://github.com/example/tags")
            .matching_pattern(".*/v?(\\d\\S+)\\.tar\\.gz")
            .build(),
    );

    let wf_str = wf.to_string();

    // Should have version line
    assert!(wf_str.starts_with("version=4\n"));
    // Should have the entry
    assert!(wf_str.contains("https://github.com/example/tags"));

    // Parse it back and ensure it's still valid
    let reparsed: super::WatchFile = wf_str.parse().unwrap();
    assert_eq!(reparsed.version(), 4);
    assert_eq!(reparsed.entries().count(), 1);
}
