use ui;
use layout;
use progvis:net;
use core:geometry;

/**
 * Object containing callbacks that is called by the system as a response to different things the
 * program is doing. This is used to encapsulate the entire behavior of the front-end depending on
 * its current state.
 */
class Behavior on Ui {
	// Called when something went wrong in the running program.
	void onProgramError(ProgramError error) {}

	// Called when the user performed some action. If the string is recorded, the same sequence can
	// then be played back.
	void onUserAction(Str action) {}

	// Called when the program is restarted.
	void onRestart() {}

	// Called when this behavior is disposed.
	void onDispose() {}

	// Allow reloading this program?
	Str? allowReload() { null; }

	// Allow submitting this program?
	Str? allowSubmit() { null; }

	// Allow the user to step at their own pace.
	Bool allowStep() { true; }

	// Allow changing "track reads/writes". If this returns "false", track memory is forced on.
	Bool allowTrackMemory() { true; }

	// Get the panel to show during rendering.
	ProblemPanel? panel() { null; }
}


/**
 * Default callbacks for the main Progvis window.
 *
 * Simply displays any errors and ignores the user actions.
 */
class DefaultBehavior extends Behavior {
	protected MainWin window;

	init(MainWin window) {
		init { window = window; }
	}

	void onProgramError(ProgramError error) : override {
		if (error as ThreadError) {
			showMessage(window, "Thread ${error.threadId} crashed", "Thread ${error.threadId} crashed:\n" + error.message);
		} else if (error as DataRaceError) {
			showMessage(window, "Data race found", "You found data races:\n" + error.message);
		} else if (error as DeadlockError) {
			showMessage(window, "Deadlock found", error.message);
		} else if (error as NoError) {
			// Nothing needed, we just don't want to show "program error".
		} else {
			showMessage(window, "Program error: " + error.type, error.message);
		}
	}

}

/**
 * Behavior for revisiting an already solved problem.
 */
class RevisitBehavior extends DefaultBehavior {
	private Str title;

	init(MainWin window, Str title) {
		init(window) { title = title; }
	}

	Str? allowReload() : override { "Can not reload the current problem. Use Run -> Restart instead, or open a new problem."; }
	Str? allowSubmit() : override { "You can not re-submit this problem as your own. Open a new file and submit that instead."; }
	Bool allowTrackMemory() : override { false; }

	ProblemPanel? panel() : override {
		ProblemPanel("Find an error in \"${title}\" (can not submit a solution)");
	}
}


/**
 * Behavior for recording progress and saving when an error is found.
 */
class RecordBehavior extends Behavior {
	private MainWin window;
	private Client client;
	private problems:SolveProblem problem;
	private StrBuf operations;

	init(MainWin window, Client client, problems:SolveProblem problem) {
		init() {
			window = window;
			client = client;
			problem = problem;
		}
	}

	void onProgramError(ProgramError error) : override {
		SubmitErrorDlg dlg(error);
		if (dlg.show(window) < 0) {
			// Don't submit.
			window.updateBehavior(RevisitBehavior(window, problem.title));
			return;
		}

		try {
			Int solutionId = postSolution(error);

			if (error as NoError) {
				window.updateBehavior(RevisitBehavior(window, problem.title));
			} else {
				window.updateBehavior(ImproveBehavior(window, client, problem, solutionId));
			}

		} catch (Exception e) {
			showMessage(window, "Error", "Failed to submit your solution: ${e.message}");
		}
	}

	void onUserAction(Str action) : override {
		operations << action << ";";
	}

	void onRestart() : override {
		operations = StrBuf();
	}

	Str? allowReload() : override { "Can not reload the current problem. Use Run -> Restart instead, or open a new problem."; }
	Str? allowSubmit() : override { "You can not re-submit this problem as your own. Open a new file and submit that instead."; }
	Bool allowTrackMemory() : override { false; }

	ProblemPanel? panel() : override {
		ProblemPanel p("Find an error in \"${problem.title}\"!");
		p.button("Open in editor", &this.onOpenEditor());
		p.button("I think there are no errors", &this.onDone);
		p;
	}

	private void onOpenEditor() {
		(spawn window.onOpenEditor()).detach();
	}

	private void onDone() {
		// Note: this is needed due to a bug in the Gui library on Windows.
		(spawn onProgramError(NoError())).detach();
	}

	private Int postSolution(ProgramError e) {
		var req = if (e as NoError) {
			PostSolutionRequest(problem.problemId, e.type, null);
		} else {
			PostSolutionRequest(problem.problemId, e.type, operations.toS);
		};

		if (res = client.query(req) as PostSolutionResponse) {
			return res.id;
		} else {
			throw ProgvisError("Incorrect message received.");
		}
	}
}


/**
 * Behavior for playing back a solution.
 *
 * We inherit from default behavior to get default error messages.
 */
class ReplayBehavior extends DefaultBehavior {
	private Str[] steps;
	private Nat pos;

	// Auto step currently active.
	private Bool autoStep;

	// Stop the auto step?
	private Bool stopAuto;

	init(MainWin window, Str solution) {
		init(window) {}

		Char sep = ";".begin.v;
		Str:Iter last = solution.begin();
		do {
			Str:Iter next = solution.find(sep, last);
			if (last != next)
				steps << solution.substr(last, next);
		} while (next != solution.end) {
			last = next + 1;
		}
	}

	void onProgramError(ProgramError error) : override {
		stop();
		super:onProgramError(error);
	}

	void onRestart() : override {
		stop();
		pos = 0;
	}

	void onDispose() : override {
		stop();
	}

	ProblemPanel? panel() {
		ProblemPanel p("Showing the problem");
		p.button("Restart", &window.onRestart());
		p.button("Next step", &this.onStep());
		p.button("Play/Pause", &this.onAuto());
		p;
	}

	private void onStep() {
		if (!autoStep)
			applyStep();
	}

	private void onAuto() {
		if (autoStep) {
			stop();
		} else {
			stopAuto = false;
			(spawn autoStepFn()).detach();
		}
	}

	private void applyStep() {
		if (pos >= steps.count) {
			(spawn showMessage(window, "End", "No more steps are available. Press \"Restart\" to see it again.")).detach();
			return;
		}

		window.applyAction(steps[pos++]);
	}

	private void stop() {
		stopAuto = true;
		while (autoStep)
			yield();
	}

	// Function running in another UThread.
	// We're constantly monitoring the "main" thread. It is a bit wasteful, but OK.
	private void autoStepFn() {
		Moment next;

		autoStep = true;
		try {
			while (!stopAuto) {
				if (next < Moment()) {
					next += 1 s;

					applyStep();
				}

				if (pos >= steps.count)
					break;

				sleep(10 ms);
			}
		} catch (Exception e) {
			print("Internal error: ${e}");
		}
		autoStep = false;
	}

	Bool allowStep() : override { false; }
	Bool allowTrackMemory() : override { false; }
	Str? allowReload() : override { "You can not reload problems while showing a solution. Open another problem instead."; }
	Str? allowSubmit() : override { "You can not submit this problem. Please open a new problem that you made first."; }
}

/**
 * Dialog presented when the user has found a solution.
 */
dialog SubmitErrorDlg {
	layout Grid {
		Label type("") {}
		nextLine;
		Label details("") {}
		nextLine;
		Label("Do you wish to submit this solution? It will replace any previous solution of the same type.") {}
		nextLine;
		Grid {
			expandCol: 0, 1;
			expandCol: 3, 1;
			Button no("No") { col: 1; }
			Button yes("Yes") { col: 2; }
		}
	}

	init(ProgramError error) {
		init("You found a problem!", Size(50, 50)) {}

		if (error as NoError) {
			this.text = "Submit";
			type.text = "You arrived at the conclusion:";
		} else {
			type.text = "You found a problem of type: ${error.type}";
		}
		details.text = error.message;

		defaultChoice = yes;
		no.onClick = &this.close();
	}
}

/**
 * Behavior that lets the user modify the problem and submit an improved version.
 */
class ImproveBehavior extends DefaultBehavior {
	private Client client;
	private problems:SolveProblem problem;
	private Int solutionId;
	private Bool submitted;

	init(MainWin window, Client client, problems:SolveProblem problem, Int solutionId) {
		init(window) {
			client = client;
			problem = problem;
			solutionId = solutionId;
			submitted = false;
		}

		StrBuf msg;
		msg << "You may now attempt to solve the problem you found:\n";
		msg << "1. Select \"Open in editor\" to edit the code.\n";
		msg << "2. Save the file in your editor.\n";
		msg << "3. Select \"Reload\" to load your changes and see if they work.\n";
		msg << "4. Repeat 1-3 as necessary.\n";
		msg << "5. Click \"Submit this version!\" in the green box when you are done.";
		showMessage(window, "Improve the code", msg.toS);
	}

	// We encourage reloads!
	Str? allowReload() : override { null; }

	// But don't allow submitting through regular means.
	Str? allowSubmit() : override { "To submit your improved version, click the button in the green bar!"; }

	// Don't change 'track memory'.
	Bool allowTrackMemory() : override { false; }

	// Add a panel that explains it all.
	ProblemPanel? panel() : override {
		ProblemPanel p("Try to fix the problem you found!");
		p.button("Open in editor", &this.onOpenEditor());
		p.button("Reload", &this.onReload());
		p.button("Submit this version!", &this.onSubmit());
		p;
	}

	private void onSubmit() {
		// Avoid double submissions.
		if (submitted)
			return;
		submitted = true;

		(spawn onSubmitHelp()).detach();
	}

	private void onSubmitHelp() {
		Str source = window.currentSource();
		UploadDlg dlg(client, source, problem.source.ext, "Solution to " + problem.title, solutionId);
		if (dlg.show(window) >= 0) {
			window.updateBehavior(DefaultBehavior(window));
		}
	}

	private void onReload() {
		// Needed due to a bug in App on Windows.
		(spawn window.onReload()).detach();
	}

	private void onOpenEditor() {
		// Needed due to a bug in App on Windows.
		(spawn window.onOpenEditor()).detach();
	}
}
